/**
 * @file instanta_layer2vi_tcpdump.c
 * @brief Yusnic layer2 Rx catching messages. Output in .pcap file format.
 * @copyright Copyright (c) 2023 YUSUR Technology Co., Ltd. All Rights Reserved. Learn more at www.yusur.tech.
 * @author matianhao (math@yusur.tech)
 * @date 2023-06-16 14:37:54
 * @last_author: matianhao (math@yusur.tech)
 * @last_edit_time: 2023-06-16 14:37:54
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <time.h>

#include "instanta_layer2vi.h"

#define RECV_BUF_LEN    16000

#define BUFFER_NAME    "buffer1"

#define TCPDUMP_MAGIC      0xa1b2c3d4
#define NSEC_TCPDUMP_MAGIC 0xa1b23c4d
#define PCAP_VERSION_MAJOR 2
#define PCAP_VERSION_MINOR 4
#define DLT_EN10MB         1

struct pcap_file_header
{
        uint32_t magic;
        uint16_t version_major;
        uint16_t version_minor;
        uint32_t thiszone;   /* gmt to local correction */
        uint32_t sigfigs;    /* accuracy of timestamps */
        uint32_t snaplen;    /* max length saved portion of each pkt */
        uint32_t linktype;   /* data link type (LINKTYPE_*) */
};

struct pcap_pkthdr
{
        uint32_t ts_sec;     /* time stamp */
        uint32_t ts_usec;
        uint32_t caplen;     /* length of portion present */
        uint32_t len;        /* length this packet (off wire) */
};

static uint32_t write_pcap_header(FILE *fp, int nsec_pcap, int snaplen)
{
    struct pcap_file_header hdr;
    hdr.magic = nsec_pcap ? NSEC_TCPDUMP_MAGIC : TCPDUMP_MAGIC;
    hdr.version_major = PCAP_VERSION_MAJOR;
    hdr.version_minor = PCAP_VERSION_MINOR;
    hdr.thiszone = 0;
    hdr.sigfigs = 0; /* 9? libpcap always writes 0 */
    hdr.snaplen = snaplen;
    hdr.linktype = DLT_EN10MB;
    fwrite(&hdr, sizeof(hdr), 1, fp);
    return sizeof(hdr);
}

static uint32_t write_pcap_packet(char *data, ssize_t len, struct timespec *recv_timespec, int nsec_pcap, int snaplen, FILE *fp)
{
    struct pcap_pkthdr hdr;
    ssize_t caplen = (len > snaplen) ? snaplen : len;
    hdr.ts_sec = recv_timespec->tv_sec;
    hdr.ts_usec = nsec_pcap ? recv_timespec->tv_nsec : (recv_timespec->tv_nsec/1000);
    hdr.caplen = caplen;
    hdr.len = len;
    fwrite(&hdr, sizeof(hdr), 1, fp);
    fwrite(data, 1, caplen, fp);
    return sizeof(hdr) + caplen;
}

static int recv_tcpdump(const char *netdev_name, const char *savefile)
{
    if (NULL == netdev_name)
    {
        printf("netdev_name is NULL.\n");
        return -1;
    }

    if (NULL == savefile)
    {
        printf("savefile is NULL.\n");
        return -1;
    }

    int create_flag = 0;
    int recv_flag = 0;

    create_flag |= LAYER2VI_ATTR_CAPTURE_ALL;
    recv_flag |= LAYER2VI_HW_TIMESTAMP;
    recv_flag |= LAYER2VI_IGNORE_CRC_ERROR;

    // 创建设备
    char *buffer_name = BUFFER_NAME;
    LAYER2VI layer2vi = layer2vi_create_ex(netdev_name, buffer_name, create_flag);
    if (NULL == layer2vi)
    {
        printf("Create layer2vi fail!\n");
        return -1;
    }

    FILE *savefp = fopen(savefile, "w");
    if (!savefp)
    {
        printf("open %s failed.\n", savefile);
        return -1;
    }
    uint32_t file_size = write_pcap_header(savefp, 0, RECV_BUF_LEN);
    struct timespec recv_timespec;

    char recv_buf[RECV_BUF_LEN] = {0};
    int recv_len = 0;
    int recv_pkts = 0;
    // 收包
    while (1)
    {
        recv_len = layer2vi_receive_frame_nonblock_ex(layer2vi, recv_buf, RECV_BUF_LEN, recv_flag, &recv_timespec, sizeof(recv_timespec));
        if (recv_len > 0)
        {
            if (!(recv_flag & LAYER2VI_HW_TIMESTAMP))
            {
                clock_gettime(CLOCK_REALTIME, &recv_timespec);
            }

            file_size += write_pcap_packet(recv_buf, recv_len, &recv_timespec, 0, RECV_BUF_LEN, savefp);

            recv_pkts++;
        }
    }

    printf("recv %d packets\n", recv_pkts);

    fclose(savefp);
    layer2vi_destroy(layer2vi);

    return 0;
}

static void help_info()
{
    printf("\nexample:\n");
    printf("  instanta_layer2vi_tcpdump -i swift1f0 -w ./recv.pcap\n");
    printf("\noptions:\n");
    printf("  -i <interface>    - netdev interface\n");
    printf("  -w <write>        - write into savefile\n");
    printf("\n");
}

int main(int argc, char *argv[])
{
    char *netdev_name = NULL;
    char *savefile = NULL;

    static struct option long_opts[] = {
        {"interface", required_argument, NULL, 'i'},
        {"write", required_argument, NULL, 'w'},
        {0, 0, 0, 0}};

    int c = -1;
    while ((c = getopt_long(argc, argv, "i:w:", long_opts, NULL)) != -1)
    {
        switch (c)
        {
        case 'i':
            netdev_name = optarg;
            break;

        case 'w':
            savefile = optarg;
            break;

        default:
            printf("\nPlease check your options\n\n");
            help_info();
            return -1;
        }
    }

    if (argc < 5)
    {
        help_info();
        return -1;
    }

    if (recv_tcpdump(netdev_name, savefile) < 0)
    {
        printf("recv test failed.\n");
        return -1;
    }

    return 0;
}
