/**
 * @file instanta_layer2vi_multicast.c
 * @brief layer2vi组播接收示例
 * @copyright Copyright (c) 2022 YUSUR Technology Co., Ltd. All Rights Reserved. Learn more at www.yusur.tech.
 * @author Guo KH (guokh@yusur.tech)
 * @date 2023-09-04 19:23:54
 * @last_author: Kaihua Guo
 * @last_edit_time: 2023-09-05 17:29:55
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <sys/time.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <errno.h>

#include "instanta_layer2vi.h"

#define MAX_PACKET_SIZE 1514

static void help_info(const char *program)
{
    fprintf(stderr, "\nusage:\n");
    fprintf(stderr, "  %s [options] <interface addr> <multicast addr> <multicast port>\n", program);
    fprintf(stderr, "\noptions:\n");
    fprintf(stderr, "  -i <interface-name> - set interface name\n");
    fprintf(stderr, "  -q <queue-id>       - set rx queue id.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "\example:\n");
    fprintf(stderr, "  %s -i swift1f0 -q 1 170.170.50.50 227.1.1.50 50000\n", program); // 多个进程使用相同参数可以同时接收同一份报文
    fprintf(stderr, "\n");
}

int main(int argc, char *argv[])
{
    const char *program = argv[0];
    const char *if_name = NULL;
    int queue_id = -1;

    static struct option long_opts[] = {
        {"interface", required_argument, NULL, 'i'},
        {"queue", required_argument, NULL, 'q'},
        {0, 0, 0, 0}};

    int c = -1;
    while ((c = getopt_long(argc, argv, "i:q:", long_opts, NULL)) != -1)
    {
        switch (c)
        {
        case 'i':
            if_name = optarg;
            break;

        case 'q':
            queue_id = atoi(optarg);
            break;

        default:
            printf("\nPlease check your options\n\n");
            help_info(program);
            return -1;
        }
    }

    argc -= optind;
    argv += optind;

    if (!if_name || (queue_id < 0) || (argc != 3))
    {
        help_info(program);
        return -1;
    }
    
    const char *if_addr_s = argv[0];
    const char *mc_addr_s = argv[1];
    in_port_t mc_port = atoi(argv[2]);

    /* 通过内核协议栈socket加组 */

    int mc_sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (mc_sock < 0)
    {
        perror("create mc_sock failed: ");
        return -1;
    }

    uint32_t on = 1;
    if (setsockopt(mc_sock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0)
    {
        perror("setsockopt error: ");
        return -1;
    }

    struct sockaddr_in local_addr;
    local_addr.sin_family = AF_INET;
    local_addr.sin_addr.s_addr = inet_addr(mc_addr_s);
    local_addr.sin_port = htons(mc_port);
    if (bind(mc_sock, (struct sockaddr *)&local_addr, sizeof(local_addr)) < 0)
    {
        perror("bind error: ");
        return -1;
    }

    struct ip_mreq mreq;
    mreq.imr_multiaddr.s_addr = inet_addr(mc_addr_s);
    mreq.imr_interface.s_addr = inet_addr(if_addr_s);

    if (setsockopt(mc_sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0)
    {
        perror("setsockopt error: ");
        return -1;
    }

    printf("Join group succeed.\n");

    /* 创建layer2vi筛选并接收组播报文 */

    // 多个进程指定相同队列可同时接收数据报文
    LAYER2VI layer2vi = layer2vi_create_specify_queue(if_name, "mc_test", queue_id);
    if (NULL == layer2vi)
    {
        printf("create layer2vi fail!\n");
        return -1;
    }

    layer2vi_filter_t filter = {0};
    filter.rule_type = LAYER2VI_FILTER_TYPE_IP;
    filter.u.ip_filter.dst_addr = inet_addr(mc_addr_s);
    filter.u.ip_filter.dst_port = htons(mc_port);
    filter.u.ip_filter.src_addr = 0;
    filter.u.ip_filter.src_port = 0;
    filter.u.ip_filter.protocol = IPPROTO_UDP;  // UDP报文

    int filter_id = layer2vi_add_filter(layer2vi, filter);
    if (filter_id < 0)
    {
        printf("add filter fail\n");
        return -1;
    }

    printf("layer2 succeed.\n");
    printf("ready to recv messages.\n");

    while (1)
    {
        char buf[MAX_PACKET_SIZE] = {0};
        ssize_t recvd = layer2vi_receive_frame_nonblock(layer2vi, buf, MAX_PACKET_SIZE);
        if (recvd > 0)
        {
            printf("\nrecv: ");
            for (ssize_t i = 0; i < recvd; i++)
            {
                printf("%02X ", buf[i]);
            }
            printf("\n");
        }
        else if (recvd < 0 && errno != EAGAIN)
        {
            printf("layer2vi_receive_frame_nonblock failed: errno %d.\n", errno);
        }
    }

    if (setsockopt(mc_sock, IPPROTO_IP, IP_DROP_MEMBERSHIP, &mreq, sizeof(mreq)) < 0)
    {
        perror("setsockopt error: ");
        return -1;
    }

    layer2vi_destroy(layer2vi);
    close(mc_sock);

    return 0;
}
