/**
 * \file
 * \brief Yusur_sock private socket options and message flags
 */
#ifndef YUSUR_SOCK_API_SOCKET_H
#define YUSUR_SOCK_API_SOCKET_H

#include <errno.h>
#include <sys/socket.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Yusur_sock private socket option level
 *
 * To manipulate socket options at Yusur_sock level (i.e. manipulate an Yusur_sock
 * private socket option specified in this file) the level argument of
 * setsockopt()/getsockopt() needs to be specified as SOL_YUSUR_SOCK.
 */
#define SOL_YUSUR_SOCK 0x200

/**
 * \brief Yusur_sock socket option for disabling acceleration on the socket
 *
 * Disabling of acceleration on a socket is not allowed if the socket has
 * already been accelerated (either by binding it to an YUSNIC interface
 * or joining a multicast group with an YUSNIC interface). Once acceleration
 * has been disabled on the socket, it can no longer be re-enabled. If any of
 * above rules is not followed setsocketopt() fails with EPERM error.
 *
 * Using a zero value for this socket option will enable acceleration
 * when global acceleration is disabled. Note that even in this case,
 * manually disabling acceleration on the socket is permanent (as above).
 *
 * This is Yusur_sock private socket level option (the level argument of
 * setsockopt()/getsockopt() needs to be specified as SOL_YUSUR_SOCK).
 * This option takes an int value. This is a Boolean option.
 */
#define SO_YUS_NO_ACCEL     1

/**
 * \brief Yusur_sock socket option for passively listening to multicast data
 * arriving on an YUSNIC interface
 *
 * Setting SO_YUS_MCAST_LISTEN has a similar effect to setting the
 * IP_ADD_MEMBERSHIP socket option with an YUSNIC interface specified in
 * imr_address or imr_ifindex.  However the SO_YUS_MCAST_LISTEN option does not
 * result in multicast joins being sent.  Also, with SO_YUS_MCAST_LISTEN, it is
 * possible to allow receiving the multicast group's data from any local YUSNIC
 * interface (if imr_address is set to INADDR_ANY and imr_ifindex is set to 0).
 * If such a wildcard is specified to IP_ADD_MEMBERSHIP, the kernel will
 * arbitrarily choose one interface and kernel bypass acceleration will be lost.
 *
 * This option should not be used together with IP_ADD_MEMBERSHIP on the same
 * socket, but rather as an alternative, if needed.  If IGMP joins are still
 * required, additional steps need to be taken to make sure the multicast group
 * has been joined, for example joining the group in another process.
 *
 * Due to the design of yusur_sock, an application should not configure more than
 * one accelerated socket receiving the same multicast group from the same
 * interface.  (Only one socket will receive the data.)
 *
 * SO_YUS_MCAST_LISTEN is not allowed if acceleration has already been disabled
 * on the socket.  (setsockopt() will fail with EPERM.)
 *
 * This is Yusur_sock private socket level option (the level argument of
 * setsockopt() needs to be specified as SOL_YUSUR_SOCK).
 *
 * The argument is ip_mreqn structure (or alternatively ip_mreq structure), just
 * like in the case of IP_ADD_MEMBERSHIP socket option:
 * - imr_multiaddr: Contains the address of the multicast group. It must be a
 *                  valid multicast address (or setsockopt() fails with the
 *                  error EINVAL).
 * - imr_address:   The address of the local YUSNIC interface from which the
 *                  socket wants to receive the multicast data; if it is equal
 *                  to INADDR_ANY, the socket will receive the multicast data
 *                  arriving through any of local YUSNIC interfaces.
 * - imr_ifindex:   The interface index of the YUSNIC interface to receive data
 *                  from, or 0 to indicate any YUSNIC interface.
 */
#define SO_YUS_MCAST_LISTEN 2

#define SO_YUS_BLOCK        4

/**
 * \brief Disable acceleration on the socket
 *
 * This is a helper function for disabling acceleration on the socket. This
 * function can be used instead of calling directly setsockopt() with
 * \ref SO_YUS_NO_ACCEL Yusur_sock private socket option.
 * Disabling of acceleration on a socket is not allowed if the socket has
 * already been accelerated (either by binding it to an YUSNIC interface
 * or joining a multicast group with an YUSNIC interface). In such a case
 * this function fails with EPERM error.
 *
 * \param[in]   fd
 *      Yusur_sock socket to disable acceleration on
 *
 * \return 0 on success, or -1 if an error occurred
 */
static inline int yusur_sock_disable_acceleration(int fd)
{
    int disable = 1;

    return setsockopt(fd, SOL_YUSUR_SOCK, SO_YUS_NO_ACCEL, &disable,
                      sizeof(disable));
}

/**
 * \brief Yusur_sock message flag to mark the message to be sent as a fake one.
 *
 * Setting this flag in send(), sendto() or sendmsg() results in triggering
 * a dummy send, which purpose is solely to keep the send code path in cache.
 * A message passed to this call will be discarded in the end.
 * Calling a send() with MSG_YUS_WARM flag set on an accelerated socket shortly
 * before invoking this call for sending an actual data helps in minimizing
 * latency of getting the data on the wire.
 *
 * Ideally the length of the fake message should be equal (or at least similar)
 * to the length of a subsequent real message to be sent.
 *
 * An attempt to send a message with MSG_YUS_WARM flag on not accelerated socket
 * is not effective. Yusur_sock will return immediately without entering the send
 * code path at all.
 */
#define MSG_YUS_WARM    0x100000

#ifdef __cplusplus
}
#endif

#endif /* YUSUR_SOCK_API_SOCKET_H */
