/**
 * @file instanta_layer2vi.h
 * @brief Yusnic layer-2 Tx/Rx API on the buffer with filters(filters applied on Rx channel).
 * @copyright Copyright (c) 2022 YUSUR Technology Co., Ltd. All Rights Reserved. Learn more at www.yusur.tech.
 * @author yangpeng (yangp@yusur.tech)
 * @date 2022-05-19 11:14:54
 * @last_author: Kaihua Guo
 * @last_edit_time: 2023-09-04 16:40:52
 */

#ifndef INSTANTA_LAYER2VI_API_H
#define INSTANTA_LAYER2VI_API_H

#include <stdint.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

#define LAYER2VI_ATTR_CAPTURE_ALL (0x1 << 0) // 用于 layer2vi_create_ex 函数的 flag 参数

#define LAYER2VI_HW_TIMESTAMP     (0x1 << 0) // 用于 layer2vi_receive_frame_nonblock_ex 函数的 flag 参数
#define LAYER2VI_IGNORE_CRC_ERROR (0x1 << 1) // 用于 layer2vi_receive_frame_nonblock_ex 函数的 flag 参数

/**
 * 二层收发接口实例，用于管理NIC的一套Tx/Rx资源.
 */
typedef void *LAYER2VI;

// struct layer2vi;

/**
 * @brief hash 规则结构体
 *
 */
struct layer2vi_hash_filter
{
    int      hash_func; /* hash function identifiers */
    uint32_t num_cores; /* hash buf 数据量 不超过32 */
};

typedef enum
{
    ACCURATE_IP_FILTER = 0xa5,
    ROUGH_IP_FILTER    = 0x5a
} ip_filter_priority;


/**
 * @brief ip 规则结构体
 *
 */
struct layer2vi_ip_filter
{
    struct
    {
        uint32_t priority : 8; /* filter优先级 */
        uint32_t reserved : 24;
    } attr; /* 定义filter属性 */

    uint32_t src_addr; /**< Source IP address of packet */
    uint32_t dst_addr; /**< Destination IP address of packet */
    uint16_t src_port; /**< Source port of packet */
    uint16_t dst_port; /**< Destination port of packet */
    uint8_t  protocol; /**< IPPROTO_UDP or IPPROTO_TCP */
};

/**
 * @brief mac 规则结构体
 *
 */
struct layer2vi_mac_filter
{
    uint8_t  dst_mac[6];
    uint16_t ethertype;
    uint16_t vlan;
    int      vlan_match_method;
};

/**
 * @brief filter 过滤规则类型枚举
 *
 */
enum layer2vi_filter_type
{
    LAYER2VI_FILTER_TYPE_HASH = 0,
    LAYER2VI_FILTER_TYPE_MAC  = 1,
    LAYER2VI_FILTER_TYPE_IP   = 2,
    LAYER2VI_FILTER_TYPE_ERR  = 3
};

/**
 * @brief 规则下发结构体
 *
 */
typedef struct layer2vi_filter
{
    enum layer2vi_filter_type rule_type; /* 规则类型 */
    uint32_t filter_id; /* 仅查询时有效，表示规则优先级，数字越小优先级越高 */
    union
    {
        struct layer2vi_ip_filter  ip_filter;  /* ip 规则 */
        struct layer2vi_mac_filter mac_filter; /* mac 规则 */
        // struct layer2vi_hash_filter hash_filter; /* hash 规则 */
    } u;
} layer2vi_filter_t;

/**
 * @brief 统计信息结构体
 *
 */
typedef struct yusur_filter_statistics
{
    uint64_t pkg_count;
    uint64_t data_sum;
} yusur_filter_statistics_t;

/**
 * Wrap VLAN tag m_tatch methods for MAC filters.
 */
enum layer2vi_vlan_match_method
{
    /** Match on all frames, whether VLAN or not. */
    LAYER2VI_VLAN_MATCH_METHOD_ALL = 0,

    /** Only match on the VLAN given. */
    LAYER2VI_VLAN_MATCH_METHOD_SPECIFIC = 1,

    /** Only match if frame does not have a vlan tag. */
    LAYER2VI_VLAN_MATCH_METHOD_NOT_VLAN = 2,

    /** Match frames that have a VLAN tag, but not those that don't. */
    LAYER2VI_VLAN_MATCH_METHOD_ALL_VLAN = 3
};

/**
 * @brief 申请二层接口实例.
 *
 * @param[in] if_name  接口名称，通过ifconfig可查
 *
 * @param[in] layer2vi_name  实例名称，用于查询统计信息
 *
 * @return  成功返回 LAYER2VI， 失败返回NULL
 */
LAYER2VI layer2vi_create(const char *if_name, const char *layer2vi_name);

/**
 * @brief 申请二层接口实例。
 *
 * @param[in] if_name  接口名称，通过ifconfig可查
 *
 * @param[in] layer2vi_name  实例名称，用于查询统计信息
 *
 * @param[in] flag  创建不同类型实例的标志位，值为 LAYER2VI_ATTR_CAPTURE_ALL时，创建的二层实例可以接收所有报文
 *
 * @return  成功返回 LAYER2VI, 失败返回NULL
 */
LAYER2VI layer2vi_create_ex(const char *if_name, const char *layer2vi_name, uint32_t flag);

/**
 * @brief 指定rx通道申请二层接口实例。
 *
 * @param[in] if_name  接口名称，通过ifconfig可查
 *
 * @param[in] layer2vi_name  实例名称，用于查询统计信息
 *
 * @param[in] queue_id  使用指定的rx queue接收数据
 *
 * @return  成功返回 LAYER2VI, 失败返回NULL
 */
LAYER2VI layer2vi_create_specify_queue(const char *if_name, const char *layer2vi_name, int queue_id);

/**
 * @brief 二层接口获取rx buf id
 *
 * @param l2_if 要释放的实例指针
 *
 * @return buf_id 成功返回buf ID，失败返回 -1
 */
int layer2vi_get_rxbuf_id(LAYER2VI l2_if);

/**
 * @brief 二层接口实例释放
 *
 * @param l2_if 要释放的实例指针
 *
 * @return 无
 */
void layer2vi_destroy(LAYER2VI l2_if);

/**
 * @brief 添加filter到layer2vi实例，使网卡根据filter规则筛选收到的二层帧推送到layer2vi Rx buffer中.
 *
 * @param[in] l2_if 二层接口实例
 * @param[in] filter
 *
 * @return int 成功返回0，失败返回-1.
 */
int layer2vi_add_filter(LAYER2VI l2_if, layer2vi_filter_t filter);

/**
 * @brief 通过filter ID从layer2vi删除对应 filter.
 *
 * @param[in]   filter_id 规则下发返回filter ID，代表该条规则优先级
 *
 * @return 成功返回0，失败返回-1.
 */
int layer2vi_remove_filter(LAYER2VI l2_if, int filter_type, int filter_id);

/**
 * @brief 获取已下发规则数
 *
 * @param[in] l2_if
 *
 * @return 成功返回规则数，失败返回 -1
 */
int layer2vi_filter_count(LAYER2VI l2_if, int filter_type);

/**
 * @brief 查找filter id对应的filter内容
 *
 * @param[in] l2_if
 * @param[in] filter_id  查询规则id
 * @param[in] filter_type	查询规则类型
 * @param[out] filter 出参返回查询到的filter
 *
 * @return 成功返回0， 失败返回-1
 */
int layer2vi_query_filter_by_id(LAYER2VI l2_if, int filter_id, int filter_type, layer2vi_filter_t *filter);

/**
 * @brief 通过 LAYER2VI 收取二层帧数据包。阻塞模式。
 *
 * @param[in] l2_if         二层接口实例
 * @param[out] frame_buf     接收数据buf
 * @param[in] buf_len       buf长度
 *
 * @return 成功则返回收到的二层帧长度，失败返回-1, 置errno。
 */
ssize_t layer2vi_receive_frame(LAYER2VI l2_if, char *frame_buf, uint32_t buf_len);

/**
 * @brief 通过 LAYER2VI 收取二层帧数据包。非阻塞模式。
 *
 * @param[in] l2_if         二层接口实例
 * @param[out] frame_buf     接收数据buf
 * @param[in] buf_len       buf长度
 *
 * @return 成功则返回收到的二层帧长度；失败返回-1，置errno; 无数据返回-1，errno为EAGAIN。
 */
ssize_t layer2vi_receive_frame_nonblock(LAYER2VI l2_if, char *frame_buf, uint32_t buf_len);

/**
 * @brief 通过 LAYER2VI 非阻塞收取二层帧数据包，扩展.
 *
 * @param[in] l2_if         二层接口实例
 * @param[out] frame_buf     接收数据buf
 * @param[in] buf_len       buf长度
 * @param[in] flag 不同接收方式的标志位，值为LAYER2VI_HW_TIMESTAMP时，返回时间戳；值为LAYER2VI_IGNORE_CRC_ERROR时，接收CRC错包。
 * @param[out] out 根据flag不同传入不同类型的参数，flag为LAYER2VI_HW_TIMESTAMP时，该值类型需为struct timespec; flag为其他值时，该值需要置NULL。
 * @param[in] out_len 输出参数out的长度
 *
 * @return 成功则返回收到的二层帧长度，失败返回-1.
 */
ssize_t layer2vi_receive_frame_nonblock_ex(LAYER2VI l2_if, char *frame_buf, uint32_t buf_len, uint32_t flag, void *out, uint32_t out_len);

/**
 * @brief 通过 YUSURU_LAYER2VI 发送二层帧数据包
 *
 * @param[in] l2_if         二层接口实例
 * @param[in] frame_buf     发送数据buf
 * @param[in] frame_len     buf长度
 *
 * @return 成功则返回发送长度，失败返回-1.
 */
ssize_t layer2vi_transmit_frame(LAYER2VI l2_if, char *frame_buf, uint32_t frame_len);

/**
 * @brief 获取接收统计信息
 *
 * @param[in] l2_if             二层接口实例
 * @param[out] statistics        统计数据结构体
 *
 * @return 成功则返回0，失败返回-1.
 */
int layer2vi_rx_statistics(LAYER2VI l2_if, yusur_filter_statistics_t *statistics);

/**
 * @brief 获取发送统计信息
 *
 * @param[in] l2_if             二层接口实例
 * @param[out] statistics        统计数据结构体
 *
 * @return 成功则返回0，失败返回-1.
 */
int layer2vi_tx_statistics(LAYER2VI l2_if, yusur_filter_statistics_t *statistics);

#ifdef __cplusplus
}
#endif

#endif /* INSTANTA_LAYER2VI_API_H */
