require 'optparse'
require 'stringio'

module Spec
  module Runner
    class OptionParser
      BUILT_IN_FORMATTERS = {
        'specdoc'  => Formatter::SpecdocFormatter,
        's'        => Formatter::SpecdocFormatter,
        'html'     => Formatter::HtmlFormatter,
        'h'        => Formatter::HtmlFormatter,
        'rdoc'     => Formatter::RdocFormatter,
        'r'        => Formatter::RdocFormatter,
        'progress' => Formatter::ProgressBarFormatter,
        'p'        => Formatter::ProgressBarFormatter,
        'failing_examples' => Formatter::FailingExamplesFormatter,
        'e'        => Formatter::FailingExamplesFormatter,
        'failing_behaviours' => Formatter::FailingBehavioursFormatter,
        'b'        => Formatter::FailingBehavioursFormatter
      }

      COMMAND_LINE = {
        :diff =>    ["-D", "--diff [FORMAT]", "Show diff of objects that are expected to be equal when they are not",
                                             "Builtin formats: unified|u|context|c",
                                             "You can also specify a custom differ class",
                                             "(in which case you should also specify --require)"],
        :colour =>  ["-c", "--colour", "--color", "Show coloured (red/green) output"],
        :example => ["-e", "--example [NAME|FILE_NAME]",  "Execute example(s) with matching name(s). If the argument is",
                                                          "the path to an existing file (typically generated by a previous",
                                                          "run using --format failing_examples:file.txt), then the examples",
                                                          "on each line of thatfile will be executed. If the file is empty,",
                                                          "all examples will be run (as if --example was not specified).",
                                                          " ",
                                                          "If the argument is not an existing file, then it is treated as",
                                                          "an example name directly, causing RSpec to run just the example",
                                                          "matching that name"],
        :specification => ["-s", "--specification [NAME]", "DEPRECATED - use -e instead", "(This will be removed when autotest works with -e)"],
        :line => ["-l", "--line LINE_NUMBER", Integer, "Execute behaviout or specification at given line.",
                                                       "(does not work for dynamically generated specs)"],
        :format => ["-f", "--format FORMAT[:WHERE]",  "Specifies what format to use for output. Specify WHERE to tell",
                                                    "the formatter where to write the output. All built-in formats",
                                                    "expect WHERE to be a file name, and will write to STDOUT if it's",
                                                    "not specified. The --format option may be specified several times",
                                                    "if you want several outputs",
                                                    " ",
                                                    "Builtin formats: ",
                                                    "progress|p           : Text progress",
                                                    "specdoc|s            : Behaviour doc as text",
                                                    "rdoc|r               : Behaviour doc as RDoc",
                                                    "html|h               : A nice HTML report",
                                                    "failing_examples|e   : Write all failing examples - input for --example",
                                                    "failing_behaviours|b : Write all failing behaviours - input for --example",
                                                    " ",
                                                    "FORMAT can also be the name of a custom formatter class",
                                                    "(in which case you should also specify --require to load it)"],
        :require => ["-r", "--require FILE", "Require FILE before running specs",
                                          "Useful for loading custom formatters or other extensions.",
                                          "If this option is used it must come before the others"],
        :backtrace => ["-b", "--backtrace", "Output full backtrace"],
        :loadby => ["-L", "--loadby STRATEGY", "Specify the strategy by which spec files should be loaded.",
                                              "STRATEGY can currently only be 'mtime' (File modification time)",
                                              "By default, spec files are loaded in alphabetical order if --loadby",
                                              "is not specified."],
        :reverse => ["-R", "--reverse", "Run examples in reverse order"],
        :timeout => ["-t", "--timeout FLOAT", "Interrupt and fail each example that doesn't complete in the",
                                              "specified time"],
        :heckle => ["-H", "--heckle CODE", "If all examples pass, this will mutate the classes and methods",
                                           "identified by CODE little by little and run all the examples again",
                                           "for each mutation. The intent is that for each mutation, at least",
                                           "one example *should* fail, and RSpec will tell you if this is not the",
                                           "case. CODE should be either Some::Module, Some::Class or",
                                           "Some::Fabulous#method}"],
        :dry_run => ["-d", "--dry-run", "Invokes formatters without executing the examples."],
        :options_file => ["-O", "--options PATH", "Read options from a file"],
        :generate_options => ["-G", "--generate-options PATH", "Generate an options file for --options"],
        :runner => ["-U", "--runner RUNNER", "Use a custom BehaviourRunner."],
        :drb => ["-X", "--drb", "Run examples via DRb. (For example against script/spec_server)"],
        :version => ["-v", "--version", "Show version"],
        :help => ["-h", "--help", "You're looking at it"]
      }

      def initialize
        @spec_parser = SpecParser.new
        @file_factory = File
      end

      def create_behaviour_runner(args, err, out, warn_if_no_files)
        options = parse(args, err, out, warn_if_no_files)
        # Some exit points in parse (--generate-options, --drb) don't return the options, 
        # but hand over control. In that case we don't want to continue.
        return nil unless options.is_a?(Options)
        options.create_behaviour_runner
      end

      def parse(args, err, out, warn_if_no_files)
        options_file = nil
        args_copy = args.dup
        options = Options.new

        opts = ::OptionParser.new do |opts|
          opts.banner = "Usage: spec (FILE|DIRECTORY|GLOB)+ [options]"
          opts.separator ""

          def opts.rspec_on(name, &block)
            on(*COMMAND_LINE[name], &block)
          end

          opts.rspec_on(:diff) {|diff| options.parse_diff(diff, out, err)}

          opts.rspec_on(:colour) {options.colour = true}

          opts.rspec_on(:example) {|example| options.parse_example(example)}

          opts.rspec_on(:specification) {|example| options.parse_example(example)}

          opts.rspec_on(:line) {|line_number| options.line_number = line_number.to_i}

          opts.rspec_on(:format) {|format| options.parse_format(format, out, err)}

          opts.rspec_on(:require) {|req| options.parse_require(req)}

          opts.rspec_on(:backtrace) {options.backtrace_tweaker = NoisyBacktraceTweaker.new}

          opts.rspec_on(:loadby) {|loadby| options.loadby = loadby}

          opts.rspec_on(:reverse) {options.reverse = true}

          opts.rspec_on(:timeout) {|timeout| options.timeout = timeout.to_f}

          opts.rspec_on(:heckle) {|heckle| options.parse_heckle(heckle)}
          
          opts.rspec_on(:dry_run) {options.dry_run = true}

          opts.rspec_on(:options_file) do |options_file|
            return parse_options_file(options_file, out, err, args_copy, warn_if_no_files)
          end

          opts.rspec_on(:generate_options) do |options_file|
            options.parse_generate_options(options_file, args_copy, out)
          end

          opts.rspec_on(:runner) do |runner|
            options.parse_runner(runner, out, err)
          end

          opts.rspec_on(:drb) do
            return parse_drb(args_copy, out, err, warn_if_no_files)
          end

          opts.rspec_on(:version) {parse_version(out)}

          opts.on_tail(*COMMAND_LINE[:help]) {parse_help(opts, out)}
        end
        opts.parse!(args)

        if args.empty? && warn_if_no_files
          err.puts "No files specified."
          err.puts opts
          exit(6) if err == $stderr
        end

        if options.line_number
          set_spec_from_line_number(options, args, err)
        end

        if options.formatters.empty?
          options.formatters << Formatter::ProgressBarFormatter.new(out)
        end

        options
      end

      def parse_options_file(options_file, out_stream, error_stream, args_copy, warn_if_no_files)
        # Remove the --options option and the argument before writing to file
        index = args_copy.index("-O") || args_copy.index("--options")
        args_copy.delete_at(index)
        args_copy.delete_at(index)

        new_args = args_copy + IO.readlines(options_file).map {|l| l.chomp.split " "}.flatten
        return CommandLine.run(new_args, error_stream, out_stream, true, warn_if_no_files)
      end

      def parse_drb(args_copy, out_stream, error_stream, warn_if_no_files)
        # Remove the --drb option
        index = args_copy.index("-X") || args_copy.index("--drb")
        args_copy.delete_at(index)

        return DrbCommandLine.run(args_copy, error_stream, out_stream, true, warn_if_no_files)
      end

      def parse_version(out_stream)
        out_stream.puts ::Spec::VERSION::DESCRIPTION
        exit if out_stream == $stdout
      end

      def parse_help(opts, out_stream)
        out_stream.puts opts
        exit if out_stream == $stdout
      end      

      def set_spec_from_line_number(options, args, err)
        if options.examples.empty?
          if args.length == 1
            if @file_factory.file?(args[0])
              source = @file_factory.open(args[0])
              example = @spec_parser.spec_name_for(source, options.line_number)
              options.parse_example(example)
            elsif @file_factory.directory?(args[0])
              err.puts "You must specify one file, not a directory when using the --line option"
              exit(1) if err == $stderr
            else
              err.puts "#{args[0]} does not exist"
              exit(2) if err == $stderr
            end
          else
            err.puts "Only one file can be specified when using the --line option: #{args.inspect}"
            exit(3) if err == $stderr
          end
        else
          err.puts "You cannot use both --line and --example"
          exit(4) if err == $stderr
        end
      end
    end
  end
end
