/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.mapreduce;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.HFileOutputFormat;
import org.apache.hadoop.hbase.mapreduce.LoadIncrementalHFiles;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.jdbc.PhoenixEmbeddedDriver;
import org.apache.phoenix.job.JobManager;
import org.apache.phoenix.mapreduce.CsvBulkImportUtil;
import org.apache.phoenix.mapreduce.CsvToKeyValueMapper;
import org.apache.phoenix.schema.PTable;
import org.apache.phoenix.util.CSVCommonsLoader;
import org.apache.phoenix.util.ColumnInfo;
import org.apache.phoenix.util.PhoenixRuntime;
import org.apache.phoenix.util.QueryUtil;
import org.apache.phoenix.util.SchemaUtil;
import org.apache.phoenix.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CsvBulkLoadTool
extends Configured
implements Tool {
    private static final Logger LOG = LoggerFactory.getLogger(CsvBulkLoadTool.class);
    static final Option ZK_QUORUM_OPT = new Option("z", "zookeeper", true, "Supply zookeeper connection details (optional)");
    static final Option INPUT_PATH_OPT = new Option("i", "input", true, "Input CSV path (mandatory)");
    static final Option OUTPUT_PATH_OPT = new Option("o", "output", true, "Output path for temporary HFiles (optional)");
    static final Option SCHEMA_NAME_OPT = new Option("s", "schema", true, "Phoenix schema name (optional)");
    static final Option TABLE_NAME_OPT = new Option("t", "table", true, "Phoenix table name (mandatory)");
    static final Option INDEX_TABLE_NAME_OPT = new Option("it", "index-table", true, "Phoenix index table name when just loading this particualar index table");
    static final Option DELIMITER_OPT = new Option("d", "delimiter", true, "Input delimiter, defaults to comma");
    static final Option QUOTE_OPT = new Option("q", "quote", true, "Supply a custom phrase delimiter, defaults to double quote character");
    static final Option ESCAPE_OPT = new Option("e", "escape", true, "Supply a custom escape character, default is a backslash");
    static final Option ARRAY_DELIMITER_OPT = new Option("a", "array-delimiter", true, "Array element delimiter (optional)");
    static final Option IMPORT_COLUMNS_OPT = new Option("c", "import-columns", true, "Comma-separated list of columns to be imported");
    static final Option IGNORE_ERRORS_OPT = new Option("g", "ignore-errors", false, "Ignore input errors");
    static final Option HELP_OPT = new Option("h", "help", false, "Show this help and quit");

    public static void main(String[] args) throws Exception {
        ToolRunner.run(new CsvBulkLoadTool(), args);
    }

    CommandLine parseOptions(String[] args) {
        Options options = this.getOptions();
        PosixParser parser = new PosixParser();
        CommandLine cmdLine = null;
        try {
            cmdLine = parser.parse(options, args);
        }
        catch (ParseException e) {
            this.printHelpAndExit("Error parsing command line options: " + e.getMessage(), options);
        }
        if (cmdLine.hasOption(HELP_OPT.getOpt())) {
            this.printHelpAndExit(options, 0);
        }
        if (!cmdLine.hasOption(TABLE_NAME_OPT.getOpt())) {
            throw new IllegalStateException(TABLE_NAME_OPT.getLongOpt() + " is a mandatory parameter");
        }
        if (!cmdLine.getArgList().isEmpty()) {
            throw new IllegalStateException("Got unexpected extra parameters: " + cmdLine.getArgList());
        }
        if (!cmdLine.hasOption(INPUT_PATH_OPT.getOpt())) {
            throw new IllegalStateException(INPUT_PATH_OPT.getLongOpt() + " is a mandatory parameter");
        }
        return cmdLine;
    }

    private Options getOptions() {
        Options options = new Options();
        options.addOption(INPUT_PATH_OPT);
        options.addOption(TABLE_NAME_OPT);
        options.addOption(INDEX_TABLE_NAME_OPT);
        options.addOption(ZK_QUORUM_OPT);
        options.addOption(OUTPUT_PATH_OPT);
        options.addOption(SCHEMA_NAME_OPT);
        options.addOption(DELIMITER_OPT);
        options.addOption(QUOTE_OPT);
        options.addOption(ESCAPE_OPT);
        options.addOption(ARRAY_DELIMITER_OPT);
        options.addOption(IMPORT_COLUMNS_OPT);
        options.addOption(IGNORE_ERRORS_OPT);
        options.addOption(HELP_OPT);
        return options;
    }

    private void printHelpAndExit(String errorMessage, Options options) {
        System.err.println(errorMessage);
        this.printHelpAndExit(options, 1);
    }

    private void printHelpAndExit(Options options, int exitCode) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("help", options);
        System.exit(exitCode);
    }

    @Override
    public int run(String[] args) throws Exception {
        Configuration conf = HBaseConfiguration.create(this.getConf());
        CommandLine cmdLine = null;
        try {
            cmdLine = this.parseOptions(args);
        }
        catch (IllegalStateException e) {
            this.printHelpAndExit(e.getMessage(), this.getOptions());
        }
        return this.loadData(conf, cmdLine);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int loadData(Configuration conf, CommandLine cmdLine) throws SQLException, InterruptedException, ExecutionException, ClassNotFoundException {
        String tableName = cmdLine.getOptionValue(TABLE_NAME_OPT.getOpt());
        String schemaName = cmdLine.getOptionValue(SCHEMA_NAME_OPT.getOpt());
        String indexTableName = cmdLine.getOptionValue(INDEX_TABLE_NAME_OPT.getOpt());
        String qualifiedTableName = CsvBulkLoadTool.getQualifiedTableName(schemaName, tableName);
        String qualifiedIndexTableName = null;
        if (indexTableName != null) {
            qualifiedIndexTableName = CsvBulkLoadTool.getQualifiedTableName(schemaName, indexTableName);
        }
        if (cmdLine.hasOption(ZK_QUORUM_OPT.getOpt())) {
            String zkQuorum = cmdLine.getOptionValue(ZK_QUORUM_OPT.getOpt());
            PhoenixEmbeddedDriver.ConnectionInfo info = PhoenixEmbeddedDriver.ConnectionInfo.create(zkQuorum);
            LOG.info("Configuring HBase connection to {}", info);
            for (Map.Entry<String, String> entry : info.asProps()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Setting {} = {}", (Object)entry.getKey(), (Object)entry.getValue());
                }
                conf.set(entry.getKey(), entry.getValue());
            }
        }
        Connection conn = QueryUtil.getConnection(conf);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Reading columns from {} :: {}", (Object)((PhoenixConnection)conn).getURL(), (Object)qualifiedTableName);
        }
        List<ColumnInfo> importColumns = this.buildImportColumns(conn, cmdLine, qualifiedTableName);
        CsvBulkLoadTool.configureOptions(cmdLine, importColumns, conf);
        try {
            this.validateTable(conn, schemaName, tableName);
        }
        finally {
            conn.close();
        }
        Path inputPath = new Path(cmdLine.getOptionValue(INPUT_PATH_OPT.getOpt()));
        Path outputPath = cmdLine.hasOption(OUTPUT_PATH_OPT.getOpt()) ? new Path(cmdLine.getOptionValue(OUTPUT_PATH_OPT.getOpt())) : new Path("/tmp/" + UUID.randomUUID());
        ArrayList<TargetTableRef> tablesToBeLoaded = new ArrayList<TargetTableRef>();
        tablesToBeLoaded.add(new TargetTableRef(qualifiedTableName));
        tablesToBeLoaded.addAll(this.getIndexTables(conn, schemaName, qualifiedTableName));
        if (qualifiedIndexTableName != null) {
            TargetTableRef targetIndexRef = null;
            for (TargetTableRef tmpTable : tablesToBeLoaded) {
                if (tmpTable.getLogicalName().compareToIgnoreCase(qualifiedIndexTableName) != 0) continue;
                targetIndexRef = tmpTable;
                break;
            }
            if (targetIndexRef == null) {
                throw new IllegalStateException("CSV Bulk Loader error: index table " + qualifiedIndexTableName + " doesn't exist");
            }
            tablesToBeLoaded.clear();
            tablesToBeLoaded.add(targetIndexRef);
        }
        ArrayList<Future<Boolean>> runningJobs = new ArrayList<Future<Boolean>>();
        boolean useInstrumentedPool = conn.unwrap(PhoenixConnection.class).getQueryServices().getProps().getBoolean("phoenix.query.metrics.enabled", true);
        ThreadPoolExecutor executor = JobManager.createThreadPoolExec(Integer.MAX_VALUE, 5, 20, useInstrumentedPool);
        try {
            HashSet<String> physicalTables = new HashSet<String>();
            for (TargetTableRef targetTableRef : tablesToBeLoaded) {
                if (physicalTables.contains(targetTableRef.getPhysicalName())) continue;
                physicalTables.add(targetTableRef.getPhysicalName());
                Path tablePath = new Path(outputPath, targetTableRef.getPhysicalName());
                Configuration jobConf = new Configuration(conf);
                jobConf.set("phoenix.mapreduce.import.tablename", qualifiedTableName);
                if (qualifiedTableName.compareToIgnoreCase(targetTableRef.getLogicalName()) != 0) {
                    jobConf.set("phoenix.mapreduce.import.indextablename", targetTableRef.getPhysicalName());
                }
                TableLoader tableLoader = new TableLoader(jobConf, targetTableRef.getPhysicalName(), inputPath, tablePath);
                runningJobs.add(executor.submit(tableLoader));
            }
        }
        finally {
            executor.shutdown();
        }
        int retCode = 0;
        for (Future future : runningJobs) {
            if (((Boolean)future.get()).booleanValue() || retCode != 0) continue;
            retCode = -1;
        }
        return retCode;
    }

    List<ColumnInfo> buildImportColumns(Connection conn, CommandLine cmdLine, String qualifiedTableName) throws SQLException {
        ArrayList<String> userSuppliedColumnNames = null;
        if (cmdLine.hasOption(IMPORT_COLUMNS_OPT.getOpt())) {
            userSuppliedColumnNames = Lists.newArrayList(Splitter.on(",").trimResults().split(cmdLine.getOptionValue(IMPORT_COLUMNS_OPT.getOpt())));
        }
        return CSVCommonsLoader.generateColumnInfo(conn, qualifiedTableName, userSuppliedColumnNames, true);
    }

    @VisibleForTesting
    static String getQualifiedTableName(String schemaName, String tableName) {
        if (schemaName != null) {
            return String.format("%s.%s", SchemaUtil.normalizeIdentifier(schemaName), SchemaUtil.normalizeIdentifier(tableName));
        }
        return SchemaUtil.normalizeIdentifier(tableName);
    }

    private static void configureOptions(CommandLine cmdLine, List<ColumnInfo> importColumns, Configuration conf) throws SQLException {
        char delimiterChar = ',';
        if (cmdLine.hasOption(DELIMITER_OPT.getOpt())) {
            String delimString = cmdLine.getOptionValue(DELIMITER_OPT.getOpt());
            if (delimString.length() != 1) {
                throw new IllegalArgumentException("Illegal delimiter character: " + delimString);
            }
            delimiterChar = delimString.charAt(0);
        }
        char quoteChar = '\"';
        if (cmdLine.hasOption(QUOTE_OPT.getOpt())) {
            String quoteString = cmdLine.getOptionValue(QUOTE_OPT.getOpt());
            if (quoteString.length() != 1) {
                throw new IllegalArgumentException("Illegal quote character: " + quoteString);
            }
            quoteChar = quoteString.charAt(0);
        }
        char escapeChar = '\\';
        if (cmdLine.hasOption(ESCAPE_OPT.getOpt())) {
            String escapeString = cmdLine.getOptionValue(ESCAPE_OPT.getOpt());
            if (escapeString.length() != 1) {
                throw new IllegalArgumentException("Illegal escape character: " + escapeString);
            }
            escapeChar = escapeString.charAt(0);
        }
        CsvBulkImportUtil.initCsvImportJob(conf, CsvBulkLoadTool.getQualifiedTableName(cmdLine.getOptionValue(SCHEMA_NAME_OPT.getOpt()), cmdLine.getOptionValue(TABLE_NAME_OPT.getOpt())), delimiterChar, quoteChar, escapeChar, cmdLine.getOptionValue(ARRAY_DELIMITER_OPT.getOpt()), importColumns, cmdLine.hasOption(IGNORE_ERRORS_OPT.getOpt()));
    }

    private void validateTable(Connection conn, String schemaName, String tableName) throws SQLException {
        ResultSet rs = conn.getMetaData().getColumns(null, StringUtil.escapeLike(schemaName), StringUtil.escapeLike(tableName), null);
        while (rs.next()) {
            String familyName = rs.getString("COLUMN_FAMILY");
            if (familyName == null || !familyName.startsWith("_")) continue;
            if ("0".equals(familyName)) {
                throw new IllegalStateException("CSV Bulk Loader error: All column names that are not part of the primary key constraint must be prefixed with a column family name (i.e. f.my_column VARCHAR)");
            }
            throw new IllegalStateException("CSV Bulk Loader error: Column family name must not start with '_': " + familyName);
        }
        rs.close();
    }

    private List<TargetTableRef> getIndexTables(Connection conn, String schemaName, String qualifiedTableName) throws SQLException {
        PTable table = PhoenixRuntime.getTable(conn, qualifiedTableName);
        ArrayList<TargetTableRef> indexTables = new ArrayList<TargetTableRef>();
        for (PTable indexTable : table.getIndexes()) {
            if (indexTable.getIndexType() == PTable.IndexType.LOCAL) {
                indexTables.add(new TargetTableRef(CsvBulkLoadTool.getQualifiedTableName(schemaName, indexTable.getTableName().getString()), indexTable.getPhysicalName().getString()));
                continue;
            }
            indexTables.add(new TargetTableRef(CsvBulkLoadTool.getQualifiedTableName(schemaName, indexTable.getTableName().getString())));
        }
        return indexTables;
    }

    private static class TableLoader
    implements Callable<Boolean> {
        private Configuration conf;
        private String tableName;
        private Path inputPath;
        private Path outputPath;

        public TableLoader(Configuration conf, String qualifiedTableName, Path inputPath, Path outputPath) {
            this.conf = conf;
            this.tableName = qualifiedTableName;
            this.inputPath = inputPath;
            this.outputPath = outputPath;
        }

        @Override
        public Boolean call() {
            LOG.info("Configuring HFile output path to {}", this.outputPath);
            try {
                Job job = new Job(this.conf, "Phoenix MapReduce import for " + this.tableName);
                if (job.getJar() == null) {
                    job.setJarByClass(CsvToKeyValueMapper.class);
                }
                job.setInputFormatClass(TextInputFormat.class);
                FileInputFormat.addInputPath(job, this.inputPath);
                FileOutputFormat.setOutputPath(job, this.outputPath);
                job.setMapperClass(CsvToKeyValueMapper.class);
                job.setMapOutputKeyClass(ImmutableBytesWritable.class);
                job.setMapOutputValueClass(KeyValue.class);
                TableMapReduceUtil.initCredentials(job);
                HTable htable = new HTable(this.conf, this.tableName);
                HFileOutputFormat.configureIncrementalLoad(job, htable);
                LOG.info("Running MapReduce import job from {} to {}", this.inputPath, (Object)this.outputPath);
                boolean success = job.waitForCompletion(true);
                if (!success) {
                    LOG.error("Import job failed, check JobTracker for details");
                    htable.close();
                    return false;
                }
                LOG.info("Loading HFiles from {}", this.outputPath);
                LoadIncrementalHFiles loader = new LoadIncrementalHFiles(this.conf);
                loader.doBulkLoad(this.outputPath, htable);
                htable.close();
                LOG.info("Incremental load complete for table=" + this.tableName);
                LOG.info("Removing output directory {}", this.outputPath);
                if (!FileSystem.get(this.conf).delete(this.outputPath, true)) {
                    LOG.error("Removing output directory {} failed", this.outputPath);
                }
                return true;
            }
            catch (Exception ex) {
                LOG.error("Import job on table=" + this.tableName + " failed due to exception.", ex);
                return false;
            }
        }
    }

    private static class TargetTableRef {
        private final String logicalName;
        private final String physicalName;

        private TargetTableRef(String name) {
            this(name, name);
        }

        private TargetTableRef(String logicalName, String physicalName) {
            this.logicalName = logicalName;
            this.physicalName = physicalName;
        }

        public String getLogicalName() {
            return this.logicalName;
        }

        public String getPhysicalName() {
            return this.physicalName;
        }
    }
}

