/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.Charsets;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.AclException;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockStoragePolicy;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.FSLimitException;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelper;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguousUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockStoragePolicySuite;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.AclStorage;
import org.apache.hadoop.hdfs.server.namenode.DefaultINodeAttributesProvider;
import org.apache.hadoop.hdfs.server.namenode.DirectoryWithQuotaFeature;
import org.apache.hadoop.hdfs.server.namenode.EncryptionZoneManager;
import org.apache.hadoop.hdfs.server.namenode.FSDirStatAndListingOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirXAttrOp;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FSPermissionChecker;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeAttributeProvider;
import org.apache.hadoop.hdfs.server.namenode.INodeAttributes;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeId;
import org.apache.hadoop.hdfs.server.namenode.INodeMap;
import org.apache.hadoop.hdfs.server.namenode.INodeReference;
import org.apache.hadoop.hdfs.server.namenode.INodeWithAdditionalFields;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameCache;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.QuotaCounts;
import org.apache.hadoop.hdfs.server.namenode.XAttrFeature;
import org.apache.hadoop.hdfs.server.namenode.XAttrStorage;
import org.apache.hadoop.hdfs.util.ByteArray;
import org.apache.hadoop.hdfs.util.EnumCounters;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FSDirectory
implements Closeable {
    static final Logger LOG = LoggerFactory.getLogger(FSDirectory.class);
    @VisibleForTesting
    static boolean CHECK_RESERVED_FILE_NAMES = true;
    public static final String DOT_RESERVED_STRING = ".reserved";
    public static final String DOT_RESERVED_PATH_PREFIX = "/.reserved";
    public static final byte[] DOT_RESERVED = DFSUtil.string2Bytes(".reserved");
    private static final String RAW_STRING = "raw";
    private static final byte[] RAW = DFSUtil.string2Bytes("raw");
    public static final String DOT_INODES_STRING = ".inodes";
    public static final byte[] DOT_INODES = DFSUtil.string2Bytes(".inodes");
    INodeDirectory rootDir;
    private final FSNamesystem namesystem;
    private volatile boolean skipQuotaCheck = false;
    private final int maxComponentLength;
    private final int maxDirItems;
    private final int lsLimit;
    private final int contentCountLimit;
    private final long contentSleepMicroSec;
    private final INodeMap inodeMap;
    private long yieldCount = 0L;
    private final int inodeXAttrsLimit;
    private final SortedSet<String> protectedDirectories;
    private final ReentrantReadWriteLock dirLock = new ReentrantReadWriteLock(true);
    private final boolean isPermissionEnabled;
    private final boolean aclsEnabled;
    private final boolean xattrsEnabled;
    private final int xattrMaxSize;
    private final long accessTimePrecision;
    private final boolean storagePolicyEnabled;
    private final boolean quotaByStorageTypeEnabled;
    private final String fsOwnerShortUserName;
    private final String supergroup;
    private final INodeId inodeId = new INodeId();
    private final FSEditLog editLog;
    private INodeAttributeProvider attributeProvider;
    @VisibleForTesting
    public final EncryptionZoneManager ezManager;
    private final NameCache<ByteArray> nameCache;

    private static INodeDirectory createRoot(FSNamesystem namesystem) {
        INodeDirectory r = new INodeDirectory(16385L, INodeDirectory.ROOT_NAME, namesystem.createFsOwnerPermissions(new FsPermission(493)), 0L);
        r.addDirectoryWithQuotaFeature(new DirectoryWithQuotaFeature.Builder().nameSpaceQuota(Long.MAX_VALUE).storageSpaceQuota(-1L).build());
        r.addSnapshottableFeature();
        r.setSnapshotQuota(0);
        return r;
    }

    public void setINodeAttributeProvider(INodeAttributeProvider provider) {
        this.attributeProvider = provider;
    }

    void readLock() {
        this.dirLock.readLock().lock();
    }

    void readUnlock() {
        this.dirLock.readLock().unlock();
    }

    void writeLock() {
        this.dirLock.writeLock().lock();
    }

    void writeUnlock() {
        this.dirLock.writeLock().unlock();
    }

    boolean hasWriteLock() {
        return this.dirLock.isWriteLockedByCurrentThread();
    }

    boolean hasReadLock() {
        return this.dirLock.getReadHoldCount() > 0 || this.hasWriteLock();
    }

    public int getReadHoldCount() {
        return this.dirLock.getReadHoldCount();
    }

    public int getWriteHoldCount() {
        return this.dirLock.getWriteHoldCount();
    }

    FSDirectory(FSNamesystem ns, Configuration conf) throws IOException {
        this.rootDir = FSDirectory.createRoot(ns);
        this.inodeMap = INodeMap.newInstance(this.rootDir);
        this.isPermissionEnabled = conf.getBoolean("dfs.permissions.enabled", true);
        this.fsOwnerShortUserName = UserGroupInformation.getCurrentUser().getShortUserName();
        this.supergroup = conf.get("dfs.permissions.superusergroup", "supergroup");
        this.aclsEnabled = conf.getBoolean("dfs.namenode.acls.enabled", false);
        LOG.info("ACLs enabled? " + this.aclsEnabled);
        this.xattrsEnabled = conf.getBoolean("dfs.namenode.xattrs.enabled", true);
        LOG.info("XAttrs enabled? " + this.xattrsEnabled);
        this.xattrMaxSize = conf.getInt("dfs.namenode.fs-limits.max-xattr-size", 16384);
        Preconditions.checkArgument(this.xattrMaxSize >= 0, "Cannot set a negative value for the maximum size of an xattr (%s).", "dfs.namenode.fs-limits.max-xattr-size");
        String unlimited = this.xattrMaxSize == 0 ? " (unlimited)" : "";
        LOG.info("Maximum size of an xattr: " + this.xattrMaxSize + unlimited);
        this.accessTimePrecision = conf.getLong("dfs.namenode.accesstime.precision", 3600000L);
        this.storagePolicyEnabled = conf.getBoolean("dfs.storage.policy.enabled", true);
        this.quotaByStorageTypeEnabled = conf.getBoolean("dfs.quota.by.storage.type.enabled", true);
        int configuredLimit = conf.getInt("dfs.ls.limit", 1000);
        this.lsLimit = configuredLimit > 0 ? configuredLimit : 1000;
        this.contentCountLimit = conf.getInt("dfs.content-summary.limit", 5000);
        this.contentSleepMicroSec = conf.getLong("dfs.content-summary.sleep-microsec", 500L);
        this.maxComponentLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 255);
        this.maxDirItems = conf.getInt("dfs.namenode.fs-limits.max-directory-items", 0x100000);
        this.inodeXAttrsLimit = conf.getInt("dfs.namenode.fs-limits.max-xattrs-per-inode", 32);
        this.protectedDirectories = FSDirectory.parseProtectedDirectories(conf);
        Preconditions.checkArgument(this.inodeXAttrsLimit >= 0, "Cannot set a negative limit on the number of xattrs per inode (%s).", "dfs.namenode.fs-limits.max-xattrs-per-inode");
        int MAX_DIR_ITEMS = 6400000;
        Preconditions.checkArgument(this.maxDirItems > 0 && this.maxDirItems <= 6400000, "Cannot set dfs.namenode.fs-limits.max-directory-items to a value less than 1 or greater than 6400000");
        int threshold = conf.getInt("dfs.namenode.name.cache.threshold", 10);
        NameNode.LOG.info("Caching file names occuring more than " + threshold + " times");
        this.nameCache = new NameCache(threshold);
        this.namesystem = ns;
        this.editLog = ns.getEditLog();
        this.ezManager = new EncryptionZoneManager(this, conf);
    }

    FSNamesystem getFSNamesystem() {
        return this.namesystem;
    }

    @VisibleForTesting
    static SortedSet<String> parseProtectedDirectories(Configuration conf) {
        return new TreeSet<String>(FSDirectory.normalizePaths(conf.getTrimmedStringCollection("fs.protected.directories"), "fs.protected.directories"));
    }

    SortedSet<String> getProtectedDirectories() {
        return this.protectedDirectories;
    }

    private BlockManager getBlockManager() {
        return this.getFSNamesystem().getBlockManager();
    }

    public INodeDirectory getRoot() {
        return this.rootDir;
    }

    public BlockStoragePolicySuite getBlockStoragePolicySuite() {
        return this.getBlockManager().getStoragePolicySuite();
    }

    boolean isPermissionEnabled() {
        return this.isPermissionEnabled;
    }

    boolean isAclsEnabled() {
        return this.aclsEnabled;
    }

    boolean isXattrsEnabled() {
        return this.xattrsEnabled;
    }

    int getXattrMaxSize() {
        return this.xattrMaxSize;
    }

    boolean isStoragePolicyEnabled() {
        return this.storagePolicyEnabled;
    }

    boolean isAccessTimeSupported() {
        return this.accessTimePrecision > 0L;
    }

    boolean isQuotaByStorageTypeEnabled() {
        return this.quotaByStorageTypeEnabled;
    }

    int getLsLimit() {
        return this.lsLimit;
    }

    int getContentCountLimit() {
        return this.contentCountLimit;
    }

    long getContentSleepMicroSec() {
        return this.contentSleepMicroSec;
    }

    int getInodeXAttrsLimit() {
        return this.inodeXAttrsLimit;
    }

    FSEditLog getEditLog() {
        return this.editLog;
    }

    @Override
    public void close() throws IOException {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void markNameCacheInitialized() {
        this.writeLock();
        try {
            this.nameCache.initialized();
        }
        finally {
            this.writeUnlock();
        }
    }

    boolean shouldSkipQuotaChecks() {
        return this.skipQuotaCheck;
    }

    void enableQuotaChecks() {
        this.skipQuotaCheck = false;
    }

    void disableQuotaChecks() {
        this.skipQuotaCheck = true;
    }

    private static INodeFile newINodeFile(long id, PermissionStatus permissions, long mtime, long atime, short replication, long preferredBlockSize) {
        return FSDirectory.newINodeFile(id, permissions, mtime, atime, replication, preferredBlockSize, (byte)0);
    }

    private static INodeFile newINodeFile(long id, PermissionStatus permissions, long mtime, long atime, short replication, long preferredBlockSize, byte storagePolicyId) {
        return new INodeFile(id, null, permissions, mtime, atime, BlockInfoContiguous.EMPTY_ARRAY, replication, preferredBlockSize, storagePolicyId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodesInPath addFile(INodesInPath existing, String localName, PermissionStatus permissions, short replication, long preferredBlockSize, String clientName, String clientMachine) throws FileAlreadyExistsException, QuotaExceededException, UnresolvedLinkException, SnapshotAccessControlException, AclException {
        INodesInPath newiip;
        long modTime = Time.now();
        INodeFile newNode = FSDirectory.newINodeFile(this.allocateNewInodeId(), permissions, modTime, modTime, replication, preferredBlockSize);
        newNode.setLocalName(localName.getBytes(Charsets.UTF_8));
        newNode.toUnderConstruction(clientName, clientMachine);
        this.writeLock();
        try {
            newiip = this.addINode(existing, newNode);
        }
        finally {
            this.writeUnlock();
        }
        if (newiip == null) {
            NameNode.stateChangeLog.info("DIR* addFile: failed to add " + existing.getPath() + "/" + localName);
            return null;
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug("DIR* addFile: " + localName + " is added");
        }
        return newiip;
    }

    INodeFile addFileForEditLog(long id, INodesInPath existing, byte[] localName, PermissionStatus permissions, List<AclEntry> aclEntries, List<XAttr> xAttrs, short replication, long modificationTime, long atime, long preferredBlockSize, boolean underConstruction, String clientName, String clientMachine, byte storagePolicyId) {
        block8: {
            INodeFile newNode;
            assert (this.hasWriteLock());
            if (underConstruction) {
                newNode = FSDirectory.newINodeFile(id, permissions, modificationTime, modificationTime, replication, preferredBlockSize, storagePolicyId);
                newNode.toUnderConstruction(clientName, clientMachine);
            } else {
                newNode = FSDirectory.newINodeFile(id, permissions, modificationTime, atime, replication, preferredBlockSize, storagePolicyId);
            }
            newNode.setLocalName(localName);
            try {
                INodesInPath iip = this.addINode(existing, newNode);
                if (iip != null) {
                    if (aclEntries != null) {
                        AclStorage.updateINodeAcl(newNode, aclEntries, 0x7FFFFFFE);
                    }
                    if (xAttrs != null) {
                        XAttrStorage.updateINodeXAttrs(newNode, xAttrs, 0x7FFFFFFE);
                    }
                    return newNode;
                }
            }
            catch (IOException e) {
                if (!NameNode.stateChangeLog.isDebugEnabled()) break block8;
                NameNode.stateChangeLog.debug("DIR* FSDirectory.unprotectedAddFile: exception when add " + existing.getPath() + " to the file system", e);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BlockInfoContiguousUnderConstruction addBlock(String path, INodesInPath inodesInPath, Block block, DatanodeStorageInfo[] targets) throws IOException {
        this.writeLock();
        try {
            INodeFile fileINode = inodesInPath.getLastINode().asFile();
            Preconditions.checkState(fileINode.isUnderConstruction());
            this.updateCount(inodesInPath, 0L, fileINode.getPreferredBlockSize(), fileINode.getBlockReplication(), true);
            BlockInfoContiguousUnderConstruction blockInfo = new BlockInfoContiguousUnderConstruction(block, fileINode.getFileReplication(), HdfsServerConstants.BlockUCState.UNDER_CONSTRUCTION, targets);
            this.getBlockManager().addBlockCollection(blockInfo, fileINode);
            fileINode.addBlock(blockInfo);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug("DIR* FSDirectory.addBlock: " + path + " with " + block + " block is added to the in-memory " + "file system");
            }
            BlockInfoContiguousUnderConstruction blockInfoContiguousUnderConstruction = blockInfo;
            return blockInfoContiguousUnderConstruction;
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean removeBlock(String path, INodesInPath iip, INodeFile fileNode, Block block) throws IOException {
        Preconditions.checkArgument(fileNode.isUnderConstruction());
        this.writeLock();
        try {
            boolean bl = this.unprotectedRemoveBlock(path, iip, fileNode, block);
            return bl;
        }
        finally {
            this.writeUnlock();
        }
    }

    boolean unprotectedRemoveBlock(String path, INodesInPath iip, INodeFile fileNode, Block block) throws IOException {
        boolean removed = fileNode.removeLastBlock(block);
        if (!removed) {
            return false;
        }
        this.getBlockManager().removeBlockFromMap(block);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug("DIR* FSDirectory.removeBlock: " + path + " with " + block + " block is removed from the file system");
        }
        this.updateCount(iip, 0L, -fileNode.getPreferredBlockSize(), fileNode.getBlockReplication(), true);
        return true;
    }

    String resolvePath(FSPermissionChecker pc, String path, byte[][] pathComponents) throws FileNotFoundException, AccessControlException {
        if (FSDirectory.isReservedRawName(path) && this.isPermissionEnabled) {
            pc.checkSuperuserPrivilege();
        }
        return FSDirectory.resolvePath(path, pathComponents, this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isNonEmptyDirectory(INodesInPath inodesInPath) {
        this.readLock();
        try {
            INode inode = inodesInPath.getLastINode();
            if (inode == null || !inode.isDirectory()) {
                boolean bl = false;
                return bl;
            }
            int s = inodesInPath.getPathSnapshotId();
            boolean bl = !inode.asDirectory().getChildrenList(s).isEmpty();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    boolean isValidToCreate(String src, INodesInPath iip) throws SnapshotAccessControlException {
        String srcs = FSDirectory.normalizePath(src);
        return srcs.startsWith("/") && !srcs.endsWith("/") && iip.getLastINode() == null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isDir(String src) throws UnresolvedLinkException {
        src = FSDirectory.normalizePath(src);
        this.readLock();
        try {
            INode node = this.getINode(src, false);
            boolean bl = node != null && node.isDirectory();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void updateSpaceConsumed(INodesInPath iip, long nsDelta, long ssDelta, short replication) throws QuotaExceededException, FileNotFoundException, UnresolvedLinkException, SnapshotAccessControlException {
        this.writeLock();
        try {
            if (iip.getLastINode() == null) {
                throw new FileNotFoundException("Path not found: " + iip.getPath());
            }
            this.updateCount(iip, nsDelta, ssDelta, replication, true);
        }
        finally {
            this.writeUnlock();
        }
    }

    void updateCountForDelete(INode inode, INodesInPath iip) {
        if (this.getFSNamesystem().isImageLoaded() && !inode.isInLatestSnapshot(iip.getLatestSnapshotId())) {
            QuotaCounts counts = inode.computeQuotaUsage(this.getBlockStoragePolicySuite());
            FSDirectory.unprotectedUpdateCount(iip, iip.length() - 1, counts.negation());
        }
    }

    void updateCount(INodesInPath iip, long nsDelta, long ssDelta, short replication, boolean checkQuota) throws QuotaExceededException {
        INodeFile fileINode = iip.getLastINode().asFile();
        EnumCounters<StorageType> typeSpaceDeltas = this.getStorageTypeDeltas(fileINode.getStoragePolicyID(), ssDelta, replication, replication);
        this.updateCount(iip, iip.length() - 1, new QuotaCounts.Builder().nameSpace(nsDelta).storageSpace(ssDelta * (long)replication).typeSpaces(typeSpaceDeltas).build(), checkQuota);
    }

    void updateCount(INodesInPath iip, long nsDelta, long ssDelta, short oldRep, short newRep, boolean checkQuota) throws QuotaExceededException {
        INodeFile fileINode = iip.getLastINode().asFile();
        EnumCounters<StorageType> typeSpaceDeltas = this.getStorageTypeDeltas(fileINode.getStoragePolicyID(), ssDelta, oldRep, newRep);
        this.updateCount(iip, iip.length() - 1, new QuotaCounts.Builder().nameSpace(nsDelta).storageSpace(ssDelta * (long)(newRep - oldRep)).typeSpaces(typeSpaceDeltas).build(), checkQuota);
    }

    void updateCount(INodesInPath iip, int numOfINodes, QuotaCounts counts, boolean checkQuota) throws QuotaExceededException {
        assert (this.hasWriteLock());
        if (!this.namesystem.isImageLoaded()) {
            return;
        }
        if (numOfINodes > iip.length()) {
            numOfINodes = iip.length();
        }
        if (checkQuota && !this.skipQuotaCheck) {
            FSDirectory.verifyQuota(iip, numOfINodes, counts, null);
        }
        FSDirectory.unprotectedUpdateCount(iip, numOfINodes, counts);
    }

    void updateCountNoQuotaCheck(INodesInPath inodesInPath, int numOfINodes, QuotaCounts counts) {
        assert (this.hasWriteLock());
        try {
            this.updateCount(inodesInPath, numOfINodes, counts, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.error("BUG: unexpected exception ", e);
        }
    }

    static void unprotectedUpdateCount(INodesInPath inodesInPath, int numOfINodes, QuotaCounts counts) {
        for (int i = 0; i < numOfINodes; ++i) {
            if (!inodesInPath.getINode(i).isQuotaSet()) continue;
            inodesInPath.getINode(i).asDirectory().getDirectoryWithQuotaFeature().addSpaceConsumed2Cache(counts);
        }
    }

    public EnumCounters<StorageType> getStorageTypeDeltas(byte storagePolicyID, long dsDelta, short oldRep, short newRep) {
        EnumCounters<StorageType> typeSpaceDeltas = new EnumCounters<StorageType>(StorageType.class);
        if (dsDelta == 0L) {
            return typeSpaceDeltas;
        }
        if (storagePolicyID != 0) {
            BlockStoragePolicy storagePolicy = this.getBlockManager().getStoragePolicy(storagePolicyID);
            if (oldRep != newRep) {
                List<StorageType> oldChosenStorageTypes = storagePolicy.chooseStorageTypes(oldRep);
                for (StorageType t : oldChosenStorageTypes) {
                    if (!t.supportTypeQuota()) continue;
                    Preconditions.checkArgument(dsDelta > 0L);
                    typeSpaceDeltas.add(t, -dsDelta);
                }
            }
            List<StorageType> newChosenStorageTypes = storagePolicy.chooseStorageTypes(newRep);
            for (StorageType t : newChosenStorageTypes) {
                if (!t.supportTypeQuota()) continue;
                typeSpaceDeltas.add(t, dsDelta);
            }
        }
        return typeSpaceDeltas;
    }

    static String getFullPathName(INode[] inodes, int pos) {
        StringBuilder fullPathName = new StringBuilder();
        if (inodes[0].isRoot()) {
            if (pos == 0) {
                return "/";
            }
        } else {
            fullPathName.append(inodes[0].getLocalName());
        }
        for (int i = 1; i <= pos; ++i) {
            fullPathName.append('/').append(inodes[i].getLocalName());
        }
        return fullPathName.toString();
    }

    private static INode[] getRelativePathINodes(INode inode, INode ancestor) {
        int depth = 0;
        for (INode i = inode; i != null && !i.equals(ancestor); i = i.getParent()) {
            ++depth;
        }
        INode[] inodes = new INode[depth];
        for (int i = 0; i < depth; ++i) {
            if (inode == null) {
                NameNode.stateChangeLog.warn("Could not get full path. Corresponding file might have deleted already.");
                return null;
            }
            inodes[depth - i - 1] = inode;
            inode = inode.getParent();
        }
        return inodes;
    }

    private static INode[] getFullPathINodes(INode inode) {
        return FSDirectory.getRelativePathINodes(inode, null);
    }

    static String getFullPathName(INode inode) {
        INode[] inodes = FSDirectory.getFullPathINodes(inode);
        return inodes == null ? "" : FSDirectory.getFullPathName(inodes, inodes.length - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodesInPath addINode(INodesInPath existing, INode child) throws QuotaExceededException, UnresolvedLinkException {
        this.cacheName(child);
        this.writeLock();
        try {
            INodesInPath iNodesInPath = this.addLastINode(existing, child, true);
            return iNodesInPath;
        }
        finally {
            this.writeUnlock();
        }
    }

    static void verifyQuota(INodesInPath iip, int pos, QuotaCounts deltas, INode commonAncestor) throws QuotaExceededException {
        if (deltas.getNameSpace() <= 0L && deltas.getStorageSpace() <= 0L && deltas.getTypeSpaces().allLessOrEqual(0L)) {
            return;
        }
        for (int i = (pos > iip.length() ? iip.length() : pos) - 1; i >= 0; --i) {
            if (commonAncestor == iip.getINode(i)) {
                return;
            }
            DirectoryWithQuotaFeature q = iip.getINode(i).asDirectory().getDirectoryWithQuotaFeature();
            if (q == null) continue;
            try {
                q.verifyQuota(deltas);
                continue;
            }
            catch (QuotaExceededException e) {
                List<INode> inodes = iip.getReadOnlyINodes();
                String path = FSDirectory.getFullPathName(inodes.toArray(new INode[inodes.size()]), i);
                e.setPathName(path);
                throw e;
            }
        }
    }

    void verifyINodeName(byte[] childName) throws HadoopIllegalArgumentException {
        if (Arrays.equals(HdfsConstants.DOT_SNAPSHOT_DIR_BYTES, childName)) {
            String s = "\".snapshot\" is a reserved name.";
            if (!this.namesystem.isImageLoaded()) {
                s = s + "  Please rename it before upgrade.";
            }
            throw new HadoopIllegalArgumentException(s);
        }
    }

    void verifyMaxComponentLength(byte[] childName, String parentPath) throws FSLimitException.PathComponentTooLongException {
        if (this.maxComponentLength == 0) {
            return;
        }
        int length = childName.length;
        if (length > this.maxComponentLength) {
            FSLimitException.PathComponentTooLongException e = new FSLimitException.PathComponentTooLongException(this.maxComponentLength, length, parentPath, DFSUtil.bytes2String(childName));
            if (this.namesystem.isImageLoaded()) {
                throw e;
            }
            NameNode.LOG.error("ERROR in FSDirectory.verifyINodeName", e);
        }
    }

    void verifyMaxDirItems(INodeDirectory parent, String parentPath) throws FSLimitException.MaxDirectoryItemsExceededException {
        int count = parent.getChildrenList(0x7FFFFFFE).size();
        if (count >= this.maxDirItems) {
            FSLimitException.MaxDirectoryItemsExceededException e = new FSLimitException.MaxDirectoryItemsExceededException(this.maxDirItems, count);
            if (this.namesystem.isImageLoaded()) {
                e.setPathName(parentPath);
                throw e;
            }
            NameNode.LOG.error("FSDirectory.verifyMaxDirItems: " + e.getLocalizedMessage());
        }
    }

    @VisibleForTesting
    public INodesInPath addLastINode(INodesInPath existing, INode inode, boolean checkQuota) throws QuotaExceededException {
        boolean added;
        assert (existing.getLastINode() != null && existing.getLastINode().isDirectory());
        int pos = existing.length();
        if (pos == 1 && existing.getINode(0) == this.rootDir && FSDirectory.isReservedName(inode)) {
            throw new HadoopIllegalArgumentException("File name \"" + inode.getLocalName() + "\" is reserved and cannot " + "be created. If this is during upgrade change the name of the " + "existing file or directory to another name before upgrading " + "to the new release.");
        }
        INodeDirectory parent = existing.getINode(pos - 1).asDirectory();
        if (checkQuota) {
            String parentPath = existing.getPath();
            this.verifyMaxComponentLength(inode.getLocalNameBytes(), parentPath);
            this.verifyMaxDirItems(parent, parentPath);
        }
        this.verifyINodeName(inode.getLocalNameBytes());
        QuotaCounts counts = inode.computeQuotaUsage(this.getBlockStoragePolicySuite());
        this.updateCount(existing, pos, counts, checkQuota);
        boolean isRename = inode.getParent() != null;
        try {
            added = parent.addChild(inode, true, existing.getLatestSnapshotId());
        }
        catch (QuotaExceededException e) {
            this.updateCountNoQuotaCheck(existing, pos, counts.negation());
            throw e;
        }
        if (!added) {
            this.updateCountNoQuotaCheck(existing, pos, counts.negation());
            return null;
        }
        if (!isRename) {
            AclStorage.copyINodeDefaultAcl(inode);
        }
        this.addToInodeMap(inode);
        return INodesInPath.append(existing, inode, inode.getLocalNameBytes());
    }

    INodesInPath addLastINodeNoQuotaCheck(INodesInPath existing, INode i) {
        try {
            return this.addLastINode(existing, i, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn("FSDirectory.addChildNoQuotaCheck - unexpected", e);
            return null;
        }
    }

    @VisibleForTesting
    public long removeLastINode(INodesInPath iip) {
        int latestSnapshot = iip.getLatestSnapshotId();
        INode last = iip.getLastINode();
        INodeDirectory parent = iip.getINode(-2).asDirectory();
        if (!parent.removeChild(last, latestSnapshot)) {
            return -1L;
        }
        return !last.isInLatestSnapshot(latestSnapshot) && INodeReference.tryRemoveReference(last) > 0 ? 0L : 1L;
    }

    static Collection<String> normalizePaths(Collection<String> paths, String errorString) {
        if (paths.isEmpty()) {
            return paths;
        }
        ArrayList<String> normalized = new ArrayList<String>(paths.size());
        for (String dir : paths) {
            if (FSDirectory.isReservedName(dir)) {
                LOG.error("{} ignoring reserved path {}", (Object)errorString, (Object)dir);
                continue;
            }
            Path path = new Path(dir);
            if (!path.isAbsolute()) {
                LOG.error("{} ignoring relative path {}", (Object)errorString, (Object)dir);
                continue;
            }
            if (path.toUri().getScheme() != null) {
                LOG.error("{} ignoring path {} with scheme", (Object)errorString, (Object)dir);
                continue;
            }
            normalized.add(path.toString());
        }
        return normalized;
    }

    static String normalizePath(String src) {
        if (src.length() > 1 && src.endsWith("/")) {
            src = src.substring(0, src.length() - 1);
        }
        return src;
    }

    @VisibleForTesting
    public long getYieldCount() {
        return this.yieldCount;
    }

    void addYieldCount(long value) {
        this.yieldCount += value;
    }

    public INodeMap getINodeMap() {
        return this.inodeMap;
    }

    void unprotectedTruncate(String src, String clientName, String clientMachine, long newLength, long mtime, Block truncateBlock) throws UnresolvedLinkException, QuotaExceededException, SnapshotAccessControlException, IOException {
        INodesInPath iip = this.getINodesInPath(src, true);
        INodeFile file = iip.getLastINode().asFile();
        INode.BlocksMapUpdateInfo collectedBlocks = new INode.BlocksMapUpdateInfo();
        boolean onBlockBoundary = this.unprotectedTruncate(iip, newLength, collectedBlocks, mtime, null);
        if (!onBlockBoundary) {
            BlockInfoContiguous oldBlock = file.getLastBlock();
            Block tBlk = this.getFSNamesystem().prepareFileForTruncate(iip, clientName, clientMachine, file.computeFileSize() - newLength, truncateBlock);
            assert (Block.matchingIdAndGenStamp(tBlk, truncateBlock) && tBlk.getNumBytes() == truncateBlock.getNumBytes()) : "Should be the same block.";
            if (oldBlock.getBlockId() != tBlk.getBlockId() && !file.isBlockInLatestSnapshot(oldBlock)) {
                this.getBlockManager().removeBlockFromMap(oldBlock);
            }
        }
        assert (onBlockBoundary == (truncateBlock == null)) : "truncateBlock is null iff on block boundary: " + truncateBlock;
        this.getFSNamesystem().removeBlocksAndUpdateSafemodeTotal(collectedBlocks);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean truncate(INodesInPath iip, long newLength, INode.BlocksMapUpdateInfo collectedBlocks, long mtime, QuotaCounts delta) throws IOException {
        this.writeLock();
        try {
            boolean bl = this.unprotectedTruncate(iip, newLength, collectedBlocks, mtime, delta);
            return bl;
        }
        finally {
            this.writeUnlock();
        }
    }

    boolean unprotectedTruncate(INodesInPath iip, long newLength, INode.BlocksMapUpdateInfo collectedBlocks, long mtime, QuotaCounts delta) throws IOException {
        assert (this.hasWriteLock());
        INodeFile file = iip.getLastINode().asFile();
        int latestSnapshot = iip.getLatestSnapshotId();
        file.recordModification(latestSnapshot, true);
        this.verifyQuotaForTruncate(iip, file, newLength, delta);
        long remainingLength = file.collectBlocksBeyondMax(newLength, collectedBlocks);
        file.excludeSnapshotBlocks(latestSnapshot, collectedBlocks);
        file.setModificationTime(mtime);
        return remainingLength - newLength == 0L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyQuotaForTruncate(INodesInPath iip, INodeFile file, long newLength, QuotaCounts delta) throws QuotaExceededException {
        if (!this.getFSNamesystem().isImageLoaded() || this.shouldSkipQuotaChecks()) {
            return;
        }
        long diff = file.computeQuotaDeltaForTruncate(newLength);
        short repl = file.getBlockReplication();
        delta.addStorageSpace(diff * (long)repl);
        BlockStoragePolicy policy = this.getBlockStoragePolicySuite().getPolicy(file.getStoragePolicyID());
        List<StorageType> types = policy.chooseStorageTypes(repl);
        for (StorageType t : types) {
            if (!t.supportTypeQuota()) continue;
            delta.addTypeSpace(t, diff);
        }
        if (diff > 0L) {
            this.readLock();
            try {
                FSDirectory.verifyQuota(iip, iip.length() - 1, delta, null);
            }
            finally {
                this.readUnlock();
            }
        }
    }

    public final void addToInodeMap(INode inode) {
        if (inode instanceof INodeWithAdditionalFields) {
            XAttrFeature xaf;
            this.inodeMap.put(inode);
            if (!inode.isSymlink() && (xaf = inode.getXAttrFeature()) != null) {
                ImmutableList<XAttr> xattrs = xaf.getXAttrs();
                for (XAttr xattr : xattrs) {
                    String xaName = XAttrHelper.getPrefixName(xattr);
                    if (!"raw.hdfs.crypto.encryption.zone".equals(xaName)) continue;
                    try {
                        HdfsProtos.ZoneEncryptionInfoProto ezProto = HdfsProtos.ZoneEncryptionInfoProto.parseFrom(xattr.getValue());
                        this.ezManager.unprotectedAddEncryptionZone(inode.getId(), PBHelper.convert(ezProto.getSuite()), PBHelper.convert(ezProto.getCryptoProtocolVersion()), ezProto.getKeyName());
                    }
                    catch (InvalidProtocolBufferException e) {
                        NameNode.LOG.warn("Error parsing protocol buffer of EZ XAttr " + xattr.getName());
                    }
                }
            }
        }
    }

    public final void removeFromInodeMap(List<? extends INode> inodes) {
        if (inodes != null) {
            for (INode iNode : inodes) {
                if (iNode == null || !(iNode instanceof INodeWithAdditionalFields)) continue;
                this.inodeMap.remove(iNode);
                this.ezManager.removeEncryptionZone(iNode.getId());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public INode getInode(long id) {
        this.readLock();
        try {
            INode iNode = this.inodeMap.get(id);
            return iNode;
        }
        finally {
            this.readUnlock();
        }
    }

    @VisibleForTesting
    int getInodeMapSize() {
        return this.inodeMap.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long totalInodes() {
        this.readLock();
        try {
            long l = this.rootDir.getDirectoryWithQuotaFeature().getSpaceConsumed().getNameSpace();
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void reset() {
        this.writeLock();
        try {
            this.rootDir = FSDirectory.createRoot(this.getFSNamesystem());
            this.inodeMap.clear();
            this.addToInodeMap(this.rootDir);
            this.nameCache.reset();
            this.inodeId.setCurrentValue(16384L);
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isInAnEZ(INodesInPath iip) throws UnresolvedLinkException, SnapshotAccessControlException {
        this.readLock();
        try {
            boolean bl = this.ezManager.isInAnEZ(iip);
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    String getKeyName(INodesInPath iip) {
        this.readLock();
        try {
            String string = this.ezManager.getKeyName(iip);
            return string;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    XAttr createEncryptionZone(String src, CipherSuite suite, CryptoProtocolVersion version, String keyName) throws IOException {
        this.writeLock();
        try {
            XAttr xAttr = this.ezManager.createEncryptionZone(src, suite, version, keyName);
            return xAttr;
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    EncryptionZone getEZForPath(INodesInPath iip) {
        this.readLock();
        try {
            EncryptionZone encryptionZone = this.ezManager.getEZINodeForPath(iip);
            return encryptionZone;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BatchedRemoteIterator.BatchedListEntries<EncryptionZone> listEncryptionZones(long prevId) throws IOException {
        this.readLock();
        try {
            BatchedRemoteIterator.BatchedListEntries<EncryptionZone> batchedListEntries = this.ezManager.listEncryptionZones(prevId);
            return batchedListEntries;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setFileEncryptionInfo(String src, FileEncryptionInfo info) throws IOException {
        HdfsProtos.PerFileEncryptionInfoProto proto = PBHelper.convertPerFileEncInfo(info);
        byte[] protoBytes = proto.toByteArray();
        XAttr fileEncryptionAttr = XAttrHelper.buildXAttr("raw.hdfs.crypto.file.encryption.info", protoBytes);
        ArrayList<XAttr> xAttrs = Lists.newArrayListWithCapacity(1);
        xAttrs.add(fileEncryptionAttr);
        this.writeLock();
        try {
            FSDirXAttrOp.unprotectedSetXAttrs(this, src, xAttrs, EnumSet.of(XAttrSetFlag.CREATE));
        }
        finally {
            this.writeUnlock();
        }
    }

    FileEncryptionInfo getFileEncryptionInfo(INode inode, int snapshotId, INodesInPath iip) throws IOException {
        if (!inode.isFile()) {
            return null;
        }
        this.readLock();
        try {
            EncryptionZone encryptionZone = this.getEZForPath(iip);
            if (encryptionZone == null) {
                FileEncryptionInfo fileEncryptionInfo = null;
                return fileEncryptionInfo;
            }
            if ((encryptionZone.getPath() == null || encryptionZone.getPath().isEmpty()) && NameNode.LOG.isDebugEnabled()) {
                NameNode.LOG.debug("Encryption zone " + encryptionZone.getPath() + " does not have a valid path.");
            }
            CryptoProtocolVersion version = encryptionZone.getVersion();
            CipherSuite suite = encryptionZone.getSuite();
            String keyName = encryptionZone.getKeyName();
            XAttr fileXAttr = FSDirXAttrOp.unprotectedGetXAttrByName(inode, snapshotId, "raw.hdfs.crypto.file.encryption.info");
            if (fileXAttr == null) {
                NameNode.LOG.warn("Could not find encryption XAttr for file " + iip.getPath() + " in encryption zone " + encryptionZone.getPath());
                FileEncryptionInfo fileEncryptionInfo = null;
                return fileEncryptionInfo;
            }
            HdfsProtos.PerFileEncryptionInfoProto fileProto = HdfsProtos.PerFileEncryptionInfoProto.parseFrom(fileXAttr.getValue());
            FileEncryptionInfo fileEncryptionInfo = PBHelper.convert(fileProto, suite, version, keyName);
            return fileEncryptionInfo;
        }
        finally {
            this.readUnlock();
        }
    }

    static INode resolveLastINode(INodesInPath iip) throws FileNotFoundException {
        INode inode = iip.getLastINode();
        if (inode == null) {
            throw new FileNotFoundException("cannot find " + iip.getPath());
        }
        return inode;
    }

    void cacheName(INode inode) {
        if (!inode.isFile()) {
            return;
        }
        ByteArray name = new ByteArray(inode.getLocalNameBytes());
        if ((name = this.nameCache.put(name)) != null) {
            inode.setLocalName(name.getBytes());
        }
    }

    void shutdown() {
        this.nameCache.reset();
        this.inodeMap.clear();
    }

    public static byte[][] getPathComponents(INode inode) {
        ArrayList<byte[]> components = new ArrayList<byte[]>();
        components.add(0, inode.getLocalNameBytes());
        while (inode.getParent() != null) {
            components.add(0, inode.getParent().getLocalNameBytes());
            inode = inode.getParent();
        }
        return (byte[][])components.toArray((T[])new byte[components.size()][]);
    }

    static byte[][] getPathComponentsForReservedPath(String src) {
        return !FSDirectory.isReservedName(src) ? (byte[][])null : INode.getPathComponents(src);
    }

    public static boolean isReservedName(INode inode) {
        return CHECK_RESERVED_FILE_NAMES && Arrays.equals(inode.getLocalNameBytes(), DOT_RESERVED);
    }

    public static boolean isReservedName(String src) {
        return src.startsWith("/.reserved/");
    }

    static boolean isReservedRawName(String src) {
        return src.startsWith("/.reserved/raw");
    }

    static String resolvePath(String src, byte[][] pathComponents, FSDirectory fsd) throws FileNotFoundException {
        int nComponents;
        int n = nComponents = pathComponents == null ? 0 : pathComponents.length;
        if (nComponents <= 2) {
            return src;
        }
        if (!Arrays.equals(DOT_RESERVED, pathComponents[1])) {
            return src;
        }
        if (Arrays.equals(DOT_INODES, pathComponents[2])) {
            if (nComponents > 3) {
                return FSDirectory.resolveDotInodesPath(src, pathComponents, fsd);
            }
            return src;
        }
        if (Arrays.equals(RAW, pathComponents[2])) {
            if (nComponents == 3) {
                return "/";
            }
            return FSDirectory.constructRemainingPath("", pathComponents, 3);
        }
        return src;
    }

    private static String resolveDotInodesPath(String src, byte[][] pathComponents, FSDirectory fsd) throws FileNotFoundException {
        long id;
        String inodeId = DFSUtil.bytes2String(pathComponents[3]);
        try {
            id = Long.parseLong(inodeId);
        }
        catch (NumberFormatException e) {
            throw new FileNotFoundException("Invalid inode path: " + src);
        }
        if (id == 16385L && pathComponents.length == 4) {
            return "/";
        }
        INode inode = fsd.getInode(id);
        if (inode == null) {
            throw new FileNotFoundException("File for given inode path does not exist: " + src);
        }
        if (pathComponents.length > 4 && DFSUtil.bytes2String(pathComponents[4]).equals("..")) {
            INodeDirectory parent = inode.getParent();
            if (parent == null || ((INode)parent).getId() == 16385L) {
                return "/";
            }
            return parent.getFullPathName();
        }
        String path = "";
        if (id != 16385L) {
            path = inode.getFullPathName();
        }
        return FSDirectory.constructRemainingPath(path, pathComponents, 4);
    }

    private static String constructRemainingPath(String pathPrefix, byte[][] pathComponents, int startAt) {
        StringBuilder path = new StringBuilder(pathPrefix);
        for (int i = startAt; i < pathComponents.length; ++i) {
            path.append("/").append(DFSUtil.bytes2String(pathComponents[i]));
        }
        if (NameNode.LOG.isDebugEnabled()) {
            NameNode.LOG.debug("Resolved path is " + path);
        }
        return path.toString();
    }

    INode getINode4DotSnapshot(String src) throws UnresolvedLinkException {
        Preconditions.checkArgument(src.endsWith("/.snapshot"), "%s does not end with %s", src, "/.snapshot");
        String dirPath = FSDirectory.normalizePath(src.substring(0, src.length() - ".snapshot".length()));
        INode node = this.getINode(dirPath);
        if (node != null && node.isDirectory() && node.asDirectory().isSnapshottable()) {
            return node;
        }
        return null;
    }

    INodesInPath getExistingPathINodes(byte[][] components) throws UnresolvedLinkException {
        return INodesInPath.resolve(this.rootDir, components, false);
    }

    public INodesInPath getINodesInPath4Write(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        return this.getINodesInPath4Write(src, true);
    }

    public INode getINode4Write(String src) throws UnresolvedLinkException, SnapshotAccessControlException {
        return this.getINodesInPath4Write(src, true).getLastINode();
    }

    public INodesInPath getINodesInPath(String path, boolean resolveLink) throws UnresolvedLinkException {
        byte[][] components = INode.getPathComponents(path);
        return INodesInPath.resolve(this.rootDir, components, resolveLink);
    }

    INode getINode(String path, boolean resolveLink) throws UnresolvedLinkException {
        return this.getINodesInPath(path, resolveLink).getLastINode();
    }

    public INode getINode(String src) throws UnresolvedLinkException {
        return this.getINode(src, true);
    }

    INodesInPath getINodesInPath4Write(String src, boolean resolveLink) throws UnresolvedLinkException, SnapshotAccessControlException {
        byte[][] components = INode.getPathComponents(src);
        INodesInPath inodesInPath = INodesInPath.resolve(this.rootDir, components, resolveLink);
        if (inodesInPath.isSnapshot()) {
            throw new SnapshotAccessControlException("Modification on a read-only snapshot is disallowed");
        }
        return inodesInPath;
    }

    FSPermissionChecker getPermissionChecker() throws AccessControlException {
        try {
            return this.getPermissionChecker(this.fsOwnerShortUserName, this.supergroup, NameNode.getRemoteUser());
        }
        catch (IOException e) {
            throw new AccessControlException(e);
        }
    }

    @VisibleForTesting
    FSPermissionChecker getPermissionChecker(String fsOwner, String superGroup, UserGroupInformation ugi) throws AccessControlException {
        return new FSPermissionChecker(fsOwner, superGroup, ugi, this.attributeProvider == null ? DefaultINodeAttributesProvider.DEFAULT_PROVIDER : this.attributeProvider);
    }

    void checkOwner(FSPermissionChecker pc, INodesInPath iip) throws AccessControlException, FileNotFoundException {
        if (iip.getLastINode() == null) {
            throw new FileNotFoundException("Directory/File does not exist " + iip.getPath());
        }
        this.checkPermission(pc, iip, true, null, null, null, null);
    }

    void checkPathAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException {
        this.checkPermission(pc, iip, false, null, null, access, null);
    }

    void checkParentAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException {
        this.checkPermission(pc, iip, false, null, access, null, null);
    }

    void checkAncestorAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException {
        this.checkPermission(pc, iip, false, access, null, null, null);
    }

    void checkTraverse(FSPermissionChecker pc, INodesInPath iip) throws AccessControlException {
        this.checkPermission(pc, iip, false, null, null, null, null);
    }

    void checkPermission(FSPermissionChecker pc, INodesInPath iip, boolean doCheckOwner, FsAction ancestorAccess, FsAction parentAccess, FsAction access, FsAction subAccess) throws AccessControlException {
        this.checkPermission(pc, iip, doCheckOwner, ancestorAccess, parentAccess, access, subAccess, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void checkPermission(FSPermissionChecker pc, INodesInPath iip, boolean doCheckOwner, FsAction ancestorAccess, FsAction parentAccess, FsAction access, FsAction subAccess, boolean ignoreEmptyDir) throws AccessControlException {
        if (!pc.isSuperUser()) {
            this.readLock();
            try {
                pc.checkPermission(iip, doCheckOwner, ancestorAccess, parentAccess, access, subAccess, ignoreEmptyDir);
            }
            finally {
                this.readUnlock();
            }
        }
    }

    HdfsFileStatus getAuditFileInfo(INodesInPath iip) throws IOException {
        return this.namesystem.isAuditEnabled() && this.namesystem.isExternalInvocation() ? FSDirStatAndListingOp.getFileInfo(this, iip.getPath(), iip, false, false) : null;
    }

    void verifyParentDir(INodesInPath iip, String src) throws FileNotFoundException, ParentNotDirectoryException {
        Path parent = new Path(src).getParent();
        if (parent != null) {
            INode parentNode = iip.getINode(-2);
            if (parentNode == null) {
                throw new FileNotFoundException("Parent directory doesn't exist: " + parent);
            }
            if (!parentNode.isDirectory() && !parentNode.isSymlink()) {
                throw new ParentNotDirectoryException("Parent path is not a directory: " + parent);
            }
        }
    }

    long allocateNewInodeId() {
        return this.inodeId.nextValue();
    }

    public long getLastInodeId() {
        return this.inodeId.getCurrentValue();
    }

    void resetLastInodeId(long newValue) throws IOException {
        try {
            this.inodeId.skipTo(newValue);
        }
        catch (IllegalStateException ise) {
            throw new IOException(ise);
        }
    }

    void resetLastInodeIdWithoutChecking(long newValue) {
        this.inodeId.setCurrentValue(newValue);
    }

    INodeAttributes getAttributes(String fullPath, byte[] path, INode node, int snapshot) {
        INodeAttributes nodeAttrs = node;
        if (this.attributeProvider != null) {
            nodeAttrs = node.getSnapshotINode(snapshot);
            fullPath = fullPath + (fullPath.endsWith("/") ? "" : "/") + DFSUtil.bytes2String(path);
            nodeAttrs = this.attributeProvider.getAttributes(fullPath, nodeAttrs);
        } else {
            nodeAttrs = node.getSnapshotINode(snapshot);
        }
        return nodeAttrs;
    }
}

