/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure;

import com.google.common.annotations.VisibleForTesting;
import com.microsoft.azure.storage.StorageException;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.BufferedFSInputStream;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.azure.AzureException;
import org.apache.hadoop.fs.azure.AzureNativeFileSystemStore;
import org.apache.hadoop.fs.azure.BlobMaterialization;
import org.apache.hadoop.fs.azure.FileMetadata;
import org.apache.hadoop.fs.azure.NativeAzureFileSystemHelper;
import org.apache.hadoop.fs.azure.NativeFileSystemStore;
import org.apache.hadoop.fs.azure.PartialListing;
import org.apache.hadoop.fs.azure.SelfRenewingLease;
import org.apache.hadoop.fs.azure.metrics.AzureFileSystemInstrumentation;
import org.apache.hadoop.fs.azure.metrics.AzureFileSystemMetricsSystem;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Progressable;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.JsonParser;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
@InterfaceStability.Stable
public class NativeAzureFileSystem
extends FileSystem {
    private static final int USER_WX_PERMISION = 192;
    private static final String TRAILING_PERIOD_PLACEHOLDER = "[[.]]";
    private static final Pattern TRAILING_PERIOD_PLACEHOLDER_PATTERN = Pattern.compile("\\[\\[\\.\\]\\](?=$|/)");
    private static final Pattern TRAILING_PERIOD_PATTERN = Pattern.compile("\\.(?=$|/)");
    public static final Logger LOG = LoggerFactory.getLogger(NativeAzureFileSystem.class);
    static final String AZURE_BLOCK_SIZE_PROPERTY_NAME = "fs.azure.block.size";
    static final String AZURE_TEMP_EXPIRY_PROPERTY_NAME = "fs.azure.fsck.temp.expiry.seconds";
    private static final int AZURE_TEMP_EXPIRY_DEFAULT = 3600;
    static final String PATH_DELIMITER = "/";
    static final String AZURE_TEMP_FOLDER = "_$azuretmpfolder$";
    private static final int AZURE_LIST_ALL = -1;
    private static final int AZURE_UNBOUNDED_DEPTH = -1;
    private static final long MAX_AZURE_BLOCK_SIZE = 0x20000000L;
    private static final String AZURE_DEFAULT_GROUP_PROPERTY_NAME = "fs.azure.permissions.supergroup";
    static final String AZURE_DEFAULT_GROUP_DEFAULT = "supergroup";
    static final String AZURE_BLOCK_LOCATION_HOST_PROPERTY_NAME = "fs.azure.block.location.impersonatedhost";
    private static final String AZURE_BLOCK_LOCATION_HOST_DEFAULT = "localhost";
    static final String AZURE_RINGBUFFER_CAPACITY_PROPERTY_NAME = "fs.azure.ring.buffer.capacity";
    static final String AZURE_OUTPUT_STREAM_BUFFER_SIZE_PROPERTY_NAME = "fs.azure.output.stream.buffer.size";
    public static final String SKIP_AZURE_METRICS_PROPERTY_NAME = "fs.azure.skip.metrics";
    public static final String APPEND_SUPPORT_ENABLE_PROPERTY_NAME = "fs.azure.enable.append.support";
    private URI uri;
    private NativeFileSystemStore store;
    private AzureNativeFileSystemStore actualStore;
    private Path workingDir;
    private long blockSize = 0x20000000L;
    private AzureFileSystemInstrumentation instrumentation;
    private String metricsSourceName;
    private boolean isClosed = false;
    private static boolean suppressRetryPolicy = false;
    private static AtomicInteger metricsSourceNameCounter = new AtomicInteger();
    private boolean appendSupportEnabled = false;

    @Override
    public String getScheme() {
        return "wasb";
    }

    public NativeAzureFileSystem() {
    }

    public NativeAzureFileSystem(NativeFileSystemStore store) {
        this.store = store;
    }

    @VisibleForTesting
    static void suppressRetryPolicy() {
        suppressRetryPolicy = true;
    }

    @VisibleForTesting
    static void resumeRetryPolicy() {
        suppressRetryPolicy = false;
    }

    @VisibleForTesting
    public static String newMetricsSourceName() {
        int number = metricsSourceNameCounter.incrementAndGet();
        String baseName = "AzureFileSystemMetrics";
        if (number == 1) {
            return "AzureFileSystemMetrics";
        }
        return "AzureFileSystemMetrics" + number;
    }

    private static boolean isWasbScheme(String scheme) {
        return scheme != null && (scheme.equalsIgnoreCase("asv") || scheme.equalsIgnoreCase("asvs") || scheme.equalsIgnoreCase("wasb") || scheme.equalsIgnoreCase("wasbs"));
    }

    private static URI reconstructAuthorityIfNeeded(URI uri, Configuration conf) {
        URI defaultUri;
        if (null == uri.getAuthority() && (defaultUri = FileSystem.getDefaultUri(conf)) != null && NativeAzureFileSystem.isWasbScheme(defaultUri.getScheme())) {
            try {
                return new URI(uri.getScheme(), defaultUri.getAuthority(), uri.getPath(), uri.getQuery(), uri.getFragment());
            }
            catch (URISyntaxException e) {
                throw new Error("Bad URI construction", e);
            }
        }
        return uri;
    }

    @Override
    protected void checkPath(Path path) {
        super.checkPath(new Path(NativeAzureFileSystem.reconstructAuthorityIfNeeded(path.toUri(), this.getConf())));
    }

    @Override
    public void initialize(URI uri, Configuration conf) throws IOException, IllegalArgumentException {
        if (null == (uri = NativeAzureFileSystem.reconstructAuthorityIfNeeded(uri, conf)).getAuthority()) {
            String errMsg = String.format("Cannot initialize WASB file system, URI authority not recognized.", new Object[0]);
            throw new IllegalArgumentException(errMsg);
        }
        super.initialize(uri, conf);
        if (this.store == null) {
            this.store = this.createDefaultStore(conf);
        }
        this.instrumentation = new AzureFileSystemInstrumentation(conf);
        if (!conf.getBoolean(SKIP_AZURE_METRICS_PROPERTY_NAME, false)) {
            AzureFileSystemMetricsSystem.fileSystemStarted();
            this.metricsSourceName = NativeAzureFileSystem.newMetricsSourceName();
            String sourceDesc = "Azure Storage Volume File System metrics";
            AzureFileSystemMetricsSystem.registerSource(this.metricsSourceName, sourceDesc, this.instrumentation);
        }
        this.store.initialize(uri, conf, this.instrumentation);
        this.setConf(conf);
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.workingDir = new Path("/user", UserGroupInformation.getCurrentUser().getShortUserName()).makeQualified(this.getUri(), this.getWorkingDirectory());
        this.blockSize = conf.getLong(AZURE_BLOCK_SIZE_PROPERTY_NAME, 0x20000000L);
        this.appendSupportEnabled = conf.getBoolean(APPEND_SUPPORT_ENABLE_PROPERTY_NAME, false);
        LOG.debug("NativeAzureFileSystem. Initializing.");
        LOG.debug("  blockSize  = {}", conf.getLong(AZURE_BLOCK_SIZE_PROPERTY_NAME, 0x20000000L));
    }

    private NativeFileSystemStore createDefaultStore(Configuration conf) {
        this.actualStore = new AzureNativeFileSystemStore();
        if (suppressRetryPolicy) {
            this.actualStore.suppressRetryPolicy();
        }
        return this.actualStore;
    }

    private static String encodeTrailingPeriod(String toEncode) {
        Matcher matcher = TRAILING_PERIOD_PATTERN.matcher(toEncode);
        return matcher.replaceAll(TRAILING_PERIOD_PLACEHOLDER);
    }

    private static String decodeTrailingPeriod(String toDecode) {
        Matcher matcher = TRAILING_PERIOD_PLACEHOLDER_PATTERN.matcher(toDecode);
        return matcher.replaceAll(".");
    }

    @VisibleForTesting
    public String pathToKey(Path path) {
        URI tmpUri = path.toUri();
        String pathUri = tmpUri.getPath();
        Path newPath = path;
        if ("".equals(pathUri)) {
            newPath = new Path(tmpUri.toString() + PATH_DELIMITER);
        }
        if (!newPath.isAbsolute()) {
            throw new IllegalArgumentException("Path must be absolute: " + path);
        }
        String key = null;
        key = newPath.toUri().getPath();
        key = NativeAzureFileSystem.removeTrailingSlash(key);
        if ((key = NativeAzureFileSystem.encodeTrailingPeriod(key)).length() == 1) {
            return key;
        }
        return key.substring(1);
    }

    private static String removeTrailingSlash(String key) {
        if (key.length() == 0 || key.length() == 1) {
            return key;
        }
        if (key.charAt(key.length() - 1) == '/') {
            return key.substring(0, key.length() - 1);
        }
        return key;
    }

    private static Path keyToPath(String key) {
        if (key.equals(PATH_DELIMITER)) {
            return new Path(PATH_DELIMITER);
        }
        return new Path(PATH_DELIMITER + NativeAzureFileSystem.decodeTrailingPeriod(key));
    }

    @VisibleForTesting
    public Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    @VisibleForTesting
    public AzureNativeFileSystemStore getStore() {
        return this.actualStore;
    }

    NativeFileSystemStore getStoreInterface() {
        return this.store;
    }

    public AzureFileSystemInstrumentation getInstrumentation() {
        return this.instrumentation;
    }

    @Override
    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        if (!this.appendSupportEnabled) {
            throw new UnsupportedOperationException("Append Support not enabled");
        }
        LOG.debug("Opening file: {} for append", f);
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        FileMetadata meta = null;
        try {
            meta = this.store.retrieveMetadata(key);
        }
        catch (Exception ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", key));
            }
            throw ex;
        }
        if (meta == null) {
            throw new FileNotFoundException(f.toString());
        }
        if (meta.isDir()) {
            throw new FileNotFoundException(f.toString() + " is a directory not a file.");
        }
        if (this.store.isPageBlobKey(key)) {
            throw new IOException("Append not supported for Page Blobs");
        }
        DataOutputStream appendStream = null;
        try {
            appendStream = this.store.retrieveAppendStream(key, bufferSize);
        }
        catch (Exception ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", key));
            }
            throw ex;
        }
        return new FSDataOutputStream(appendStream, this.statistics);
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(f, permission, overwrite, true, bufferSize, replication, blockSize, progress, null);
    }

    public SelfRenewingLease acquireLease(Path path) throws AzureException {
        String fullKey = this.pathToKey(this.makeAbsolute(path));
        return this.getStore().acquireLease(fullKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path parent = f.getParent();
        SelfRenewingLease lease = null;
        if (this.store.isAtomicRenameKey(this.pathToKey(f))) {
            try {
                lease = this.acquireLease(parent);
            }
            catch (AzureException e) {
                String errorCode = "";
                try {
                    StorageException e2 = (StorageException)e.getCause();
                    errorCode = e2.getErrorCode();
                }
                catch (Exception e3) {
                    // empty catch block
                }
                if (errorCode.equals("BlobNotFound")) {
                    throw new FileNotFoundException("Cannot create file " + f.getName() + " because parent folder does not exist.");
                }
                LOG.warn("Got unexpected exception trying to get lease on {} . {}", (Object)this.pathToKey(parent), (Object)e.getMessage());
                throw e;
            }
        }
        if (!this.exists(parent)) {
            try {
                lease.free();
                throw new FileNotFoundException("Cannot create file " + f.getName() + " because parent folder does not exist.");
            }
            catch (Exception e) {
                LOG.warn("Unable to free lease because: {}", (Object)e.getMessage());
            }
            throw new FileNotFoundException("Cannot create file " + f.getName() + " because parent folder does not exist.");
        }
        FSDataOutputStream out = null;
        try {
            out = this.create(f, permission, overwrite, false, bufferSize, replication, blockSize, progress, lease);
        }
        catch (Throwable throwable) {
            try {
                if (lease == null) throw throwable;
                lease.free();
                throw throwable;
            }
            catch (Exception e) {
                NativeAzureFileSystemHelper.cleanup(LOG, out);
                String msg = "Unable to free lease on " + parent.toUri();
                LOG.error(msg);
                throw new IOException(msg, e);
            }
        }
        try {
            if (lease == null) return out;
            lease.free();
            return out;
        }
        catch (Exception e) {
            NativeAzureFileSystemHelper.cleanup(LOG, out);
            String msg = "Unable to free lease on " + parent.toUri();
            LOG.error(msg);
            throw new IOException(msg, e);
        }
    }

    @Override
    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        EnumSet<CreateFlag> createflags = EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE);
        boolean overwrite = flags.containsAll(createflags);
        return this.createNonRecursive(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    @Override
    public FSDataOutputStream createNonRecursive(Path f, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.createNonRecursive(f, FsPermission.getFileDefault(), overwrite, bufferSize, replication, blockSize, progress);
    }

    private FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, boolean createParent, int bufferSize, short replication, long blockSize, Progressable progress, SelfRenewingLease parentFolderLease) throws FileAlreadyExistsException, IOException {
        OutputStream bufOutStream;
        Path parentFolder;
        LOG.debug("Creating file: {}", (Object)f.toString());
        if (this.containsColon(f)) {
            throw new IOException("Cannot create file " + f + " through WASB that has colons in the name");
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        FileMetadata existingMetadata = this.store.retrieveMetadata(key);
        if (existingMetadata != null) {
            if (existingMetadata.isDir()) {
                throw new FileAlreadyExistsException("Cannot create file " + f + "; already exists as a directory.");
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException("File already exists:" + f);
            }
        }
        if ((parentFolder = absolutePath.getParent()) != null && parentFolder.getParent() != null) {
            String parentKey = this.pathToKey(parentFolder);
            FileMetadata parentMetadata = this.store.retrieveMetadata(parentKey);
            if (parentMetadata != null && parentMetadata.isDir() && parentMetadata.getBlobMaterialization() == BlobMaterialization.Explicit) {
                if (parentFolderLease != null) {
                    this.store.updateFolderLastModifiedTime(parentKey, parentFolderLease);
                } else {
                    this.updateParentFolderLastModifiedTime(key);
                }
            } else {
                Path firstExisting = parentFolder.getParent();
                FileMetadata metadata = this.store.retrieveMetadata(this.pathToKey(firstExisting));
                while (metadata == null) {
                    firstExisting = firstExisting.getParent();
                    metadata = this.store.retrieveMetadata(this.pathToKey(firstExisting));
                }
                this.mkdirs(parentFolder, metadata.getPermissionStatus().getPermission(), true);
            }
        }
        FsPermission masked = this.applyUMask(permission, UMaskApplyMode.NewFile);
        PermissionStatus permissionStatus = this.createPermissionStatus(masked);
        if (this.store.isPageBlobKey(key)) {
            bufOutStream = this.store.storefile(key, permissionStatus);
        } else {
            String keyEncoded = NativeAzureFileSystem.encodeKey(key);
            this.store.storeEmptyLinkFile(key, keyEncoded, permissionStatus);
            bufOutStream = new NativeAzureFsOutputStream(this.store.storefile(keyEncoded, permissionStatus), key, keyEncoded);
        }
        FSDataOutputStream fsOut = new FSDataOutputStream(bufOutStream, this.statistics);
        this.instrumentation.fileCreated();
        return fsOut;
    }

    @Override
    @Deprecated
    public boolean delete(Path path) throws IOException {
        return this.delete(path, true);
    }

    @Override
    public boolean delete(Path f, boolean recursive) throws IOException {
        return this.delete(f, recursive, false);
    }

    public boolean delete(Path f, boolean recursive, boolean skipParentFolderLastModifidedTimeUpdate) throws IOException {
        LOG.debug("Deleting file: {}", (Object)f.toString());
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        FileMetadata metaFile = null;
        try {
            metaFile = this.store.retrieveMetadata(key);
        }
        catch (IOException e) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                return false;
            }
            throw e;
        }
        if (null == metaFile) {
            return false;
        }
        if (!metaFile.isDir()) {
            Path parentPath = absolutePath.getParent();
            if (parentPath.getParent() != null) {
                String parentKey = this.pathToKey(parentPath);
                FileMetadata parentMetadata = null;
                try {
                    parentMetadata = this.store.retrieveMetadata(parentKey);
                }
                catch (IOException e) {
                    Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                    if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                        throw new IOException("File " + f + " has a parent directory " + parentPath + " whose metadata cannot be retrieved. Can't resolve");
                    }
                    throw e;
                }
                if (parentMetadata == null) {
                    throw new IOException("File " + f + " has a parent directory " + parentPath + " whose metadata cannot be retrieved. Can't resolve");
                }
                if (!parentMetadata.isDir()) {
                    throw new AzureException("File " + f + " has a parent directory " + parentPath + " which is also a file. Can't resolve.");
                }
                if (parentMetadata.getBlobMaterialization() == BlobMaterialization.Implicit) {
                    LOG.debug("Found an implicit parent directory while trying to delete the file {}. Creating the directory blob for it in {}.", f, (Object)parentKey);
                    this.store.storeEmptyFolder(parentKey, this.createPermissionStatus(FsPermission.getDefault()));
                } else if (!skipParentFolderLastModifidedTimeUpdate) {
                    this.updateParentFolderLastModifiedTime(key);
                }
            }
            try {
                this.store.delete(key);
                this.instrumentation.fileDeleted();
            }
            catch (IOException e) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    return false;
                }
                throw e;
            }
        }
        LOG.debug("Directory Delete encountered: {}", (Object)f.toString());
        Path parentPath = absolutePath.getParent();
        if (parentPath.getParent() != null) {
            String parentKey = this.pathToKey(parentPath);
            FileMetadata parentMetadata = null;
            try {
                parentMetadata = this.store.retrieveMetadata(parentKey);
            }
            catch (IOException e) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    throw new IOException("File " + f + " has a parent directory " + parentPath + " whose metadata cannot be retrieved. Can't resolve");
                }
                throw e;
            }
            if (parentMetadata == null) {
                throw new IOException("File " + f + " has a parent directory " + parentPath + " whose metadata cannot be retrieved. Can't resolve");
            }
            if (parentMetadata.getBlobMaterialization() == BlobMaterialization.Implicit) {
                LOG.debug("Found an implicit parent directory while trying to delete the directory {}. Creating the directory blob for it in {}. ", f, (Object)parentKey);
                this.store.storeEmptyFolder(parentKey, this.createPermissionStatus(FsPermission.getDefault()));
            }
        }
        String priorLastKey = null;
        PartialListing listing = null;
        try {
            listing = this.store.listAll(key, -1, 1, priorLastKey);
        }
        catch (IOException e) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                return false;
            }
            throw e;
        }
        if (listing == null) {
            return false;
        }
        FileMetadata[] contents = listing.getFiles();
        if (!recursive && contents.length > 0) {
            throw new IOException("Non-recursive delete of non-empty directory " + f.toString());
        }
        for (FileMetadata p : contents) {
            String suffix = p.getKey().substring(p.getKey().lastIndexOf(PATH_DELIMITER));
            if (!p.isDir()) {
                try {
                    this.store.delete(key + suffix);
                    this.instrumentation.fileDeleted();
                    continue;
                }
                catch (IOException e) {
                    Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                    if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                        return false;
                    }
                    throw e;
                }
            }
            if (this.delete(new Path(f.toString() + suffix), true)) continue;
            return false;
        }
        try {
            this.store.delete(key);
        }
        catch (IOException e) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                return false;
            }
            throw e;
        }
        Path parent = absolutePath.getParent();
        if (parent != null && parent.getParent() != null && !skipParentFolderLastModifidedTimeUpdate) {
            this.updateParentFolderLastModifiedTime(key);
        }
        this.instrumentation.directoryDeleted();
        LOG.debug("Delete Successful for : {}", (Object)f.toString());
        return true;
    }

    @Override
    public FileStatus getFileStatus(Path f) throws FileNotFoundException, IOException {
        LOG.debug("Getting the file status for {}", (Object)f.toString());
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        if (key.length() == 0) {
            return this.newDirectory(null, absolutePath);
        }
        FileMetadata meta = null;
        try {
            meta = this.store.retrieveMetadata(key);
        }
        catch (Exception ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", key));
            }
            throw ex;
        }
        if (meta != null) {
            if (meta.isDir()) {
                LOG.debug("Path {} is a folder.", (Object)f.toString());
                if (this.conditionalRedoFolderRename(f)) {
                    throw new FileNotFoundException(absolutePath + ": No such file or directory.");
                }
                return this.newDirectory(meta, absolutePath);
            }
            LOG.debug("Found the path: {} as a file.", (Object)f.toString());
            return this.newFile(meta, absolutePath);
        }
        throw new FileNotFoundException(absolutePath + ": No such file or directory.");
    }

    private boolean conditionalRedoFolderRename(Path f) throws IOException {
        if (f.getName().equals("")) {
            return false;
        }
        Path absoluteRenamePendingFile = this.renamePendingFilePath(f);
        if (this.exists(absoluteRenamePendingFile)) {
            FolderRenamePending pending = new FolderRenamePending(absoluteRenamePendingFile, this);
            pending.redo();
            return true;
        }
        return false;
    }

    private Path renamePendingFilePath(Path f) {
        Path absPath = this.makeAbsolute(f);
        String key = this.pathToKey(absPath);
        key = key + "-RenamePending.json";
        return NativeAzureFileSystem.keyToPath(key);
    }

    @Override
    public URI getUri() {
        return this.uri;
    }

    @Override
    public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
        LOG.debug("Listing status for {}", (Object)f.toString());
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        TreeSet<FileStatus> status = new TreeSet<FileStatus>();
        FileMetadata meta = null;
        try {
            meta = this.store.retrieveMetadata(key);
        }
        catch (IOException ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", f));
            }
            throw ex;
        }
        if (meta != null) {
            if (!meta.isDir()) {
                LOG.debug("Found path as a file");
                return new FileStatus[]{this.newFile(meta, absolutePath)};
            }
            String partialKey = null;
            PartialListing listing = null;
            try {
                listing = this.store.list(key, -1, 1, partialKey);
            }
            catch (IOException ex) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    throw new FileNotFoundException(String.format("%s is not found", key));
                }
                throw ex;
            }
            boolean renamed = this.conditionalRedoFolderRenames(listing);
            if (renamed) {
                listing = null;
                try {
                    listing = this.store.list(key, -1, 1, partialKey);
                }
                catch (IOException ex) {
                    Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
                    if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                        throw new FileNotFoundException(String.format("%s is not found", key));
                    }
                    throw ex;
                }
            }
            for (FileMetadata fileMetadata : listing.getFiles()) {
                Path subpath = NativeAzureFileSystem.keyToPath(fileMetadata.getKey());
                if (fileMetadata.isDir()) {
                    if (fileMetadata.getKey().equals(AZURE_TEMP_FOLDER)) continue;
                    status.add(this.newDirectory(fileMetadata, subpath));
                    continue;
                }
                status.add(this.newFile(fileMetadata, subpath));
            }
        } else {
            LOG.debug("Did not find any metadata for path: {}", (Object)key);
            throw new FileNotFoundException("File" + f + " does not exist.");
        }
        LOG.debug("Found path as a directory with {} files in it.", status.size());
        return status.toArray(new FileStatus[0]);
    }

    private boolean conditionalRedoFolderRenames(PartialListing listing) throws IllegalArgumentException, IOException {
        boolean renamed = false;
        for (FileMetadata fileMetadata : listing.getFiles()) {
            Path subpath = NativeAzureFileSystem.keyToPath(fileMetadata.getKey());
            if (!this.isRenamePendingFile(subpath)) continue;
            FolderRenamePending pending = new FolderRenamePending(subpath, this);
            pending.redo();
            renamed = true;
        }
        return renamed;
    }

    private boolean isRenamePendingFile(Path path) {
        return path.toString().endsWith("-RenamePending.json");
    }

    private FileStatus newFile(FileMetadata meta, Path path) {
        return new FileStatus(meta.getLength(), false, 1, this.blockSize, meta.getLastModified(), 0L, meta.getPermissionStatus().getPermission(), meta.getPermissionStatus().getUserName(), meta.getPermissionStatus().getGroupName(), path.makeQualified(this.getUri(), this.getWorkingDirectory()));
    }

    private FileStatus newDirectory(FileMetadata meta, Path path) {
        return new FileStatus(0L, true, 1, this.blockSize, meta == null ? 0L : meta.getLastModified(), 0L, meta == null ? FsPermission.getDefault() : meta.getPermissionStatus().getPermission(), meta == null ? "" : meta.getPermissionStatus().getUserName(), meta == null ? "" : meta.getPermissionStatus().getGroupName(), path.makeQualified(this.getUri(), this.getWorkingDirectory()));
    }

    private FsPermission applyUMask(FsPermission permission, UMaskApplyMode applyMode) {
        FsPermission newPermission = new FsPermission(permission);
        if (applyMode == UMaskApplyMode.NewFile || applyMode == UMaskApplyMode.NewDirectory) {
            newPermission = newPermission.applyUMask(FsPermission.getUMask(this.getConf()));
        }
        return newPermission;
    }

    @VisibleForTesting
    PermissionStatus createPermissionStatus(FsPermission permission) throws IOException {
        return new PermissionStatus(UserGroupInformation.getCurrentUser().getShortUserName(), this.getConf().get(AZURE_DEFAULT_GROUP_PROPERTY_NAME, AZURE_DEFAULT_GROUP_DEFAULT), permission);
    }

    @Override
    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        return this.mkdirs(f, permission, false);
    }

    public boolean mkdirs(Path f, FsPermission permission, boolean noUmask) throws IOException {
        LOG.debug("Creating directory: {}", (Object)f.toString());
        if (this.containsColon(f)) {
            throw new IOException("Cannot create directory " + f + " through WASB that has colons in the name");
        }
        Path absolutePath = this.makeAbsolute(f);
        PermissionStatus permissionStatus = null;
        permissionStatus = noUmask ? this.createPermissionStatus(this.applyUMask(FsPermission.createImmutable((short)(permission.toShort() | 0xC0)), UMaskApplyMode.NewDirectoryNoUmask)) : this.createPermissionStatus(this.applyUMask(permission, UMaskApplyMode.NewDirectory));
        ArrayList<String> keysToCreateAsFolder = new ArrayList<String>();
        ArrayList<String> keysToUpdateAsFolder = new ArrayList<String>();
        boolean childCreated = false;
        Path current = absolutePath;
        Path parent = current.getParent();
        while (parent != null) {
            String currentKey = this.pathToKey(current);
            FileMetadata currentMetadata = this.store.retrieveMetadata(currentKey);
            if (currentMetadata != null && !currentMetadata.isDir()) {
                throw new FileAlreadyExistsException("Cannot create directory " + f + " because " + current + " is an existing file.");
            }
            if (currentMetadata == null) {
                keysToCreateAsFolder.add(currentKey);
                childCreated = true;
            } else {
                if (childCreated) {
                    keysToUpdateAsFolder.add(currentKey);
                }
                childCreated = false;
            }
            current = parent;
            parent = current.getParent();
        }
        for (String currentKey : keysToCreateAsFolder) {
            this.store.storeEmptyFolder(currentKey, permissionStatus);
        }
        this.instrumentation.directoryCreated();
        return true;
    }

    @Override
    public FSDataInputStream open(Path f, int bufferSize) throws FileNotFoundException, IOException {
        LOG.debug("Opening file: {}", (Object)f.toString());
        Path absolutePath = this.makeAbsolute(f);
        String key = this.pathToKey(absolutePath);
        FileMetadata meta = null;
        try {
            meta = this.store.retrieveMetadata(key);
        }
        catch (Exception ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", key));
            }
            throw ex;
        }
        if (meta == null) {
            throw new FileNotFoundException(f.toString());
        }
        if (meta.isDir()) {
            throw new FileNotFoundException(f.toString() + " is a directory not a file.");
        }
        DataInputStream inputStream = null;
        try {
            inputStream = this.store.retrieve(key);
        }
        catch (Exception ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("%s is not found", key));
            }
            throw ex;
        }
        return new FSDataInputStream(new BufferedFSInputStream(new NativeAzureFsInputStream(inputStream, key, meta.getLength()), bufferSize));
    }

    @Override
    public boolean rename(Path src, Path dst) throws FileNotFoundException, IOException {
        FolderRenamePending renamePending = null;
        LOG.debug("Moving {} to {}", src, (Object)dst);
        if (this.containsColon(dst)) {
            throw new IOException("Cannot rename to file " + dst + " through WASB that has colons in the name");
        }
        String srcKey = this.pathToKey(this.makeAbsolute(src));
        if (srcKey.length() == 0) {
            return false;
        }
        Path absoluteDst = this.makeAbsolute(dst);
        String dstKey = this.pathToKey(absoluteDst);
        FileMetadata dstMetadata = null;
        try {
            dstMetadata = this.store.retrieveMetadata(dstKey);
        }
        catch (IOException ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException) {
                if (NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    LOG.debug("BlobNotFound exception encountered for Destination key : {}. Swallowin the exception to handle race condition gracefully", (Object)dstKey);
                }
            }
            throw ex;
        }
        if (dstMetadata != null && dstMetadata.isDir()) {
            dstKey = this.pathToKey(this.makeAbsolute(new Path(dst, src.getName())));
            LOG.debug("Destination {}  is a directory, adjusted the destination to be {}", dst, (Object)dstKey);
        } else {
            if (dstMetadata != null) {
                LOG.debug("Destination {} is an already existing file, failing the rename.", dst);
                return false;
            }
            FileMetadata parentOfDestMetadata = null;
            try {
                parentOfDestMetadata = this.store.retrieveMetadata(this.pathToKey(absoluteDst.getParent()));
            }
            catch (IOException ex) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    LOG.debug("Parent of destination {} doesn't exists. Failing rename", dst);
                    return false;
                }
                throw ex;
            }
            if (parentOfDestMetadata == null) {
                LOG.debug("Parent of the destination {} doesn't exist, failing the rename.", dst);
                return false;
            }
            if (!parentOfDestMetadata.isDir()) {
                LOG.debug("Parent of the destination {} is a file, failing the rename.", dst);
                return false;
            }
        }
        FileMetadata srcMetadata = null;
        try {
            srcMetadata = this.store.retrieveMetadata(srcKey);
        }
        catch (IOException ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                LOG.debug("Source {} doesn't exists. Failing rename", src);
                return false;
            }
            throw ex;
        }
        if (srcMetadata == null) {
            LOG.debug("Source {} doesn't exist, failing the rename.", src);
            return false;
        }
        if (!srcMetadata.isDir()) {
            LOG.debug("Source {} found as a file, renaming.", src);
            try {
                this.store.rename(srcKey, dstKey);
            }
            catch (IOException ex) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    LOG.debug("BlobNotFoundException encountered. Failing rename", src);
                    return false;
                }
                throw ex;
            }
        } else {
            renamePending = this.prepareAtomicFolderRename(srcKey, dstKey);
            renamePending.execute();
            LOG.debug("Renamed {} to {} successfully.", src, (Object)dst);
            renamePending.cleanup();
            return true;
        }
        this.updateParentFolderLastModifiedTime(srcKey);
        this.updateParentFolderLastModifiedTime(dstKey);
        LOG.debug("Renamed {} to {} successfully.", src, (Object)dst);
        return true;
    }

    private void updateParentFolderLastModifiedTime(String key) throws IOException {
        String parentKey;
        FileMetadata parentMetadata;
        Path parent = this.makeAbsolute(NativeAzureFileSystem.keyToPath(key)).getParent();
        if (parent != null && parent.getParent() != null && (parentMetadata = this.store.retrieveMetadata(parentKey = this.pathToKey(parent))) != null) {
            if (parentMetadata.isDir() && parentMetadata.getBlobMaterialization() == BlobMaterialization.Implicit) {
                this.store.storeEmptyFolder(parentKey, this.createPermissionStatus(FsPermission.getDefault()));
            }
            if (this.store.isAtomicRenameKey(parentKey)) {
                SelfRenewingLease lease = null;
                try {
                    lease = this.leaseSourceFolder(parentKey);
                    this.store.updateFolderLastModifiedTime(parentKey, lease);
                }
                catch (AzureException e) {
                    String errorCode = "";
                    try {
                        StorageException e2 = (StorageException)e.getCause();
                        errorCode = e2.getErrorCode();
                    }
                    catch (Exception e3) {
                        // empty catch block
                    }
                    if (errorCode.equals("BlobNotFound")) {
                        throw new FileNotFoundException("Folder does not exist: " + parentKey);
                    }
                    LOG.warn("Got unexpected exception trying to get lease on {}. {}", (Object)parentKey, (Object)e.getMessage());
                    throw e;
                }
                finally {
                    try {
                        if (lease != null) {
                            lease.free();
                        }
                    }
                    catch (Exception e) {
                        LOG.error("Unable to free lease on {}", (Object)parentKey, (Object)e);
                    }
                }
            }
            this.store.updateFolderLastModifiedTime(parentKey, null);
        }
    }

    private FolderRenamePending prepareAtomicFolderRename(String srcKey, String dstKey) throws IOException {
        if (this.store.isAtomicRenameKey(srcKey)) {
            SelfRenewingLease lease = this.leaseSourceFolder(srcKey);
            FolderRenamePending renamePending = new FolderRenamePending(srcKey, dstKey, lease, this);
            renamePending.writeFile(this);
            return renamePending;
        }
        FolderRenamePending renamePending = new FolderRenamePending(srcKey, dstKey, null, this);
        return renamePending;
    }

    private SelfRenewingLease leaseSourceFolder(String srcKey) throws AzureException {
        return this.store.acquireLease(srcKey);
    }

    @Override
    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
        if (file == null) {
            return null;
        }
        if (start < 0L || len < 0L) {
            throw new IllegalArgumentException("Invalid start or len parameter");
        }
        if (file.getLen() < start) {
            return new BlockLocation[0];
        }
        String blobLocationHost = this.getConf().get(AZURE_BLOCK_LOCATION_HOST_PROPERTY_NAME, AZURE_BLOCK_LOCATION_HOST_DEFAULT);
        String[] name = new String[]{blobLocationHost};
        String[] host = new String[]{blobLocationHost};
        long blockSize = file.getBlockSize();
        if (blockSize <= 0L) {
            throw new IllegalArgumentException("The block size for the given file is not a positive number: " + blockSize);
        }
        int numberOfLocations = (int)(len / blockSize) + (len % blockSize == 0L ? 0 : 1);
        BlockLocation[] locations = new BlockLocation[numberOfLocations];
        for (int i = 0; i < locations.length; ++i) {
            long currentOffset = start + (long)i * blockSize;
            long currentLength = Math.min(blockSize, start + len - currentOffset);
            locations[i] = new BlockLocation(name, host, currentOffset, currentLength);
        }
        return locations;
    }

    @Override
    public void setWorkingDirectory(Path newDir) {
        this.workingDir = this.makeAbsolute(newDir);
    }

    @Override
    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    @Override
    public void setPermission(Path p, FsPermission permission) throws FileNotFoundException, IOException {
        Path absolutePath = this.makeAbsolute(p);
        String key = this.pathToKey(absolutePath);
        FileMetadata metadata = null;
        try {
            metadata = this.store.retrieveMetadata(key);
        }
        catch (IOException ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("File %s doesn't exists.", p));
            }
            throw ex;
        }
        if (metadata == null) {
            throw new FileNotFoundException("File doesn't exist: " + p);
        }
        permission = this.applyUMask(permission, metadata.isDir() ? UMaskApplyMode.ChangeExistingDirectory : UMaskApplyMode.ChangeExistingFile);
        if (metadata.getBlobMaterialization() == BlobMaterialization.Implicit) {
            this.store.storeEmptyFolder(key, this.createPermissionStatus(permission));
        } else if (!metadata.getPermissionStatus().getPermission().equals(permission)) {
            this.store.changePermissionStatus(key, new PermissionStatus(metadata.getPermissionStatus().getUserName(), metadata.getPermissionStatus().getGroupName(), permission));
        }
    }

    @Override
    public void setOwner(Path p, String username, String groupname) throws IOException {
        Path absolutePath = this.makeAbsolute(p);
        String key = this.pathToKey(absolutePath);
        FileMetadata metadata = null;
        try {
            metadata = this.store.retrieveMetadata(key);
        }
        catch (IOException ex) {
            Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(ex);
            if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                throw new FileNotFoundException(String.format("File %s doesn't exists.", p));
            }
            throw ex;
        }
        if (metadata == null) {
            throw new FileNotFoundException("File doesn't exist: " + p);
        }
        PermissionStatus newPermissionStatus = new PermissionStatus(username == null ? metadata.getPermissionStatus().getUserName() : username, groupname == null ? metadata.getPermissionStatus().getGroupName() : groupname, metadata.getPermissionStatus().getPermission());
        if (metadata.getBlobMaterialization() == BlobMaterialization.Implicit) {
            this.store.storeEmptyFolder(key, newPermissionStatus);
        } else {
            this.store.changePermissionStatus(key, newPermissionStatus);
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.isClosed) {
            return;
        }
        super.close();
        this.store.close();
        long startTime = System.currentTimeMillis();
        if (!this.getConf().getBoolean(SKIP_AZURE_METRICS_PROPERTY_NAME, false)) {
            AzureFileSystemMetricsSystem.unregisterSource(this.metricsSourceName);
            AzureFileSystemMetricsSystem.fileSystemClosed();
        }
        LOG.debug("Submitting metrics when file system closed took {} ms.", System.currentTimeMillis() - startTime);
        this.isClosed = true;
    }

    private boolean containsColon(Path p) {
        return p.toUri().getPath().toString().contains(":");
    }

    private void handleFilesWithDanglingTempData(Path root, DanglingFileHandler handler) throws IOException {
        PartialListing listing;
        long cutoffForDangling = new Date().getTime() - (long)(this.getConf().getInt(AZURE_TEMP_EXPIRY_PROPERTY_NAME, 3600) * 1000);
        String priorLastKey = null;
        do {
            listing = this.store.listAll(this.pathToKey(root), -1, -1, priorLastKey);
            for (FileMetadata file : listing.getFiles()) {
                FileMetadata linkMetadata;
                String link;
                if (file.isDir() || (link = this.store.getLinkInFileMetadata(file.getKey())) == null || (linkMetadata = this.store.retrieveMetadata(link)) == null || linkMetadata.getLastModified() < cutoffForDangling) continue;
                handler.handleFile(file, linkMetadata);
            }
        } while ((priorLastKey = listing.getPriorLastKey()) != null);
    }

    public void recoverFilesWithDanglingTempData(Path root, Path destination) throws IOException {
        LOG.debug("Recovering files with dangling temp data in {}", root);
        this.handleFilesWithDanglingTempData(root, new DanglingFileRecoverer(destination));
    }

    public void deleteFilesWithDanglingTempData(Path root) throws IOException {
        LOG.debug("Deleting files with dangling temp data in {}", root);
        this.handleFilesWithDanglingTempData(root, new DanglingFileDeleter());
    }

    protected void finalize() throws Throwable {
        LOG.debug("finalize() called.");
        this.close();
        super.finalize();
    }

    private static String encodeKey(String aKey) {
        String fileName = aKey.substring(aKey.lastIndexOf(PATH_DELIMITER) + 1, aKey.length());
        String filePrefix = "_$azuretmpfolder$/" + UUID.randomUUID().toString();
        String randomizedKey = filePrefix + fileName;
        return randomizedKey;
    }

    private class DanglingFileRecoverer
    extends DanglingFileHandler {
        private final Path destination;

        DanglingFileRecoverer(Path destination) {
            this.destination = destination;
        }

        @Override
        void handleFile(FileMetadata file, FileMetadata tempFile) throws IOException {
            LOG.debug("Recovering {}", (Object)file.getKey());
            String finalDestinationKey = NativeAzureFileSystem.this.pathToKey(new Path(this.destination, file.getKey()));
            NativeAzureFileSystem.this.store.rename(tempFile.getKey(), finalDestinationKey);
            if (!finalDestinationKey.equals(file.getKey())) {
                NativeAzureFileSystem.this.store.delete(file.getKey());
            }
        }
    }

    private class DanglingFileDeleter
    extends DanglingFileHandler {
        private DanglingFileDeleter() {
        }

        @Override
        void handleFile(FileMetadata file, FileMetadata tempFile) throws IOException {
            LOG.debug("Deleting dangling file {}", (Object)file.getKey());
            NativeAzureFileSystem.this.store.delete(file.getKey());
            NativeAzureFileSystem.this.store.delete(tempFile.getKey());
        }
    }

    private abstract class DanglingFileHandler {
        private DanglingFileHandler() {
        }

        abstract void handleFile(FileMetadata var1, FileMetadata var2) throws IOException;
    }

    private static enum UMaskApplyMode {
        NewFile,
        NewDirectory,
        NewDirectoryNoUmask,
        ChangeExistingFile,
        ChangeExistingDirectory;

    }

    private class NativeAzureFsOutputStream
    extends OutputStream {
        private String key;
        private String keyEncoded;
        private OutputStream out;

        public NativeAzureFsOutputStream(OutputStream out, String aKey, String anEncodedKey) throws IOException {
            if (null == out) {
                throw new IllegalArgumentException("Illegal argument: the output stream is null.");
            }
            if (null == aKey || 0 == aKey.length()) {
                throw new IllegalArgumentException("Illegal argument the key string is null or empty");
            }
            if (null == anEncodedKey || 0 == anEncodedKey.length()) {
                throw new IllegalArgumentException("Illegal argument the encoded key string is null or empty");
            }
            this.out = out;
            this.setKey(aKey);
            this.setEncodedKey(anEncodedKey);
        }

        @Override
        public synchronized void close() throws IOException {
            if (this.out != null) {
                this.out.close();
                this.restoreKey();
                this.out = null;
            }
        }

        @Override
        public void write(int b) throws IOException {
            try {
                this.out.write(b);
            }
            catch (IOException e) {
                if (e.getCause() instanceof StorageException) {
                    StorageException storageExcp = (StorageException)e.getCause();
                    LOG.error("Encountered Storage Exception for write on Blob : {} Exception details: {} Error Code : {}", new Object[]{this.key, e.getMessage(), storageExcp.getErrorCode()});
                }
                throw e;
            }
        }

        @Override
        public void write(byte[] b) throws IOException {
            try {
                this.out.write(b);
            }
            catch (IOException e) {
                if (e.getCause() instanceof StorageException) {
                    StorageException storageExcp = (StorageException)e.getCause();
                    LOG.error("Encountered Storage Exception for write on Blob : {} Exception details: {} Error Code : {}", new Object[]{this.key, e.getMessage(), storageExcp.getErrorCode()});
                }
                throw e;
            }
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            try {
                this.out.write(b, off, len);
            }
            catch (IOException e) {
                if (e.getCause() instanceof StorageException) {
                    StorageException storageExcp = (StorageException)e.getCause();
                    LOG.error("Encountered Storage Exception for write on Blob : {} Exception details: {} Error Code : {}", new Object[]{this.key, e.getMessage(), storageExcp.getErrorCode()});
                }
                throw e;
            }
        }

        public String getKey() {
            return this.key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public String getEncodedKey() {
            return this.keyEncoded;
        }

        public void setEncodedKey(String anEncodedKey) {
            this.keyEncoded = anEncodedKey;
        }

        private void restoreKey() throws IOException {
            NativeAzureFileSystem.this.store.rename(this.getEncodedKey(), this.getKey());
        }
    }

    private class NativeAzureFsInputStream
    extends FSInputStream {
        private InputStream in;
        private final String key;
        private long pos = 0L;
        private boolean closed = false;
        private boolean isPageBlob;
        private long fileLength;

        public NativeAzureFsInputStream(DataInputStream in, String key, long fileLength) {
            this.in = in;
            this.key = key;
            this.isPageBlob = NativeAzureFileSystem.this.store.isPageBlobKey(key);
            this.fileLength = fileLength;
        }

        @Override
        public synchronized int available() throws IOException {
            if (this.isPageBlob) {
                return this.in.available();
            }
            if (this.closed) {
                throw new IOException("Stream closed");
            }
            long remaining = this.fileLength - this.pos;
            return remaining <= Integer.MAX_VALUE ? (int)remaining : Integer.MAX_VALUE;
        }

        @Override
        public synchronized int read() throws FileNotFoundException, IOException {
            try {
                int result = 0;
                result = this.in.read();
                if (result != -1) {
                    ++this.pos;
                    if (NativeAzureFileSystem.this.statistics != null) {
                        NativeAzureFileSystem.this.statistics.incrementBytesRead(1L);
                    }
                }
                return result;
            }
            catch (IOException e) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                if (innerException instanceof StorageException) {
                    LOG.error("Encountered Storage Exception for read on Blob : {} Exception details: {} Error Code : {}", new Object[]{this.key, e, ((StorageException)innerException).getErrorCode()});
                    if (NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                        throw new FileNotFoundException(String.format("%s is not found", this.key));
                    }
                }
                throw e;
            }
        }

        @Override
        public synchronized int read(byte[] b, int off, int len) throws FileNotFoundException, IOException {
            try {
                int result = 0;
                result = this.in.read(b, off, len);
                if (result > 0) {
                    this.pos += (long)result;
                }
                if (null != NativeAzureFileSystem.this.statistics) {
                    NativeAzureFileSystem.this.statistics.incrementBytesRead(result);
                }
                return result;
            }
            catch (IOException e) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                if (innerException instanceof StorageException) {
                    LOG.error("Encountered Storage Exception for read on Blob : {} Exception details: {} Error Code : {}", new Object[]{this.key, e, ((StorageException)innerException).getErrorCode()});
                    if (NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                        throw new FileNotFoundException(String.format("%s is not found", this.key));
                    }
                }
                throw e;
            }
        }

        @Override
        public synchronized void close() throws IOException {
            if (!this.closed) {
                this.closed = true;
                IOUtils.closeStream(this.in);
                this.in = null;
            }
        }

        @Override
        public synchronized void seek(long pos) throws FileNotFoundException, EOFException, IOException {
            try {
                this.checkNotClosed();
                if (pos < 0L) {
                    throw new EOFException("Cannot seek to a negative offset");
                }
                IOUtils.closeStream(this.in);
                this.in = NativeAzureFileSystem.this.store.retrieve(this.key);
                this.pos = this.in.skip(pos);
                LOG.debug("Seek to position {}. Bytes skipped {}", pos, (Object)this.pos);
            }
            catch (IOException e) {
                Throwable innerException = NativeAzureFileSystemHelper.checkForAzureStorageException(e);
                if (innerException instanceof StorageException && NativeAzureFileSystemHelper.isFileNotFoundException((StorageException)innerException)) {
                    throw new FileNotFoundException(String.format("%s is not found", this.key));
                }
                throw e;
            }
            catch (IndexOutOfBoundsException e) {
                throw new EOFException("Attempted to seek or read past the end of the file");
            }
        }

        @Override
        public synchronized long getPos() throws IOException {
            return this.pos;
        }

        @Override
        public boolean seekToNewSource(long targetPos) throws IOException {
            return false;
        }

        private void checkNotClosed() throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed!");
            }
        }
    }

    public static class Secure
    extends NativeAzureFileSystem {
        @Override
        public String getScheme() {
            return "wasbs";
        }
    }

    public static class FolderRenamePending {
        private SelfRenewingLease folderLease;
        private String srcKey;
        private String dstKey;
        private FileMetadata[] fileMetadata = null;
        private ArrayList<String> fileStrings = null;
        private NativeAzureFileSystem fs;
        private static final int MAX_RENAME_PENDING_FILE_SIZE = 10000000;
        private static final int FORMATTING_BUFFER = 10000;
        private boolean committed;
        public static final String SUFFIX = "-RenamePending.json";

        public FolderRenamePending(String srcKey, String dstKey, SelfRenewingLease lease, NativeAzureFileSystem fs) throws IOException {
            PartialListing listing;
            this.srcKey = srcKey;
            this.dstKey = dstKey;
            this.folderLease = lease;
            this.fs = fs;
            ArrayList<FileMetadata> fileMetadataList = new ArrayList<FileMetadata>();
            String priorLastKey = null;
            do {
                listing = fs.getStoreInterface().listAll(srcKey, -1, -1, priorLastKey);
                for (FileMetadata file : listing.getFiles()) {
                    fileMetadataList.add(file);
                }
            } while ((priorLastKey = listing.getPriorLastKey()) != null);
            this.fileMetadata = fileMetadataList.toArray(new FileMetadata[fileMetadataList.size()]);
            this.committed = true;
        }

        public FolderRenamePending(Path redoFile, NativeAzureFileSystem fs) throws IllegalArgumentException, IOException {
            this.fs = fs;
            Path f = redoFile;
            FSDataInputStream input = fs.open(f);
            byte[] bytes = new byte[10000000];
            int l = input.read(bytes);
            if (l <= 0) {
                LOG.error("Deleting empty rename pending file " + redoFile + " -- no data available");
                this.deleteRenamePendingFile(fs, redoFile);
                return;
            }
            if (l == 10000000) {
                throw new IOException("Error reading pending rename file contents -- maximum file size exceeded");
            }
            String contents = new String(bytes, 0, l, Charset.forName("UTF-8"));
            ObjectMapper objMapper = new ObjectMapper();
            objMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
            JsonNode json = null;
            try {
                json = (JsonNode)objMapper.readValue(contents, JsonNode.class);
                this.committed = true;
            }
            catch (JsonMappingException e) {
                this.committed = false;
            }
            catch (JsonParseException e) {
                this.committed = false;
            }
            catch (IOException e) {
                this.committed = false;
            }
            if (!this.committed) {
                LOG.error("Deleting corruped rename pending file {} \n {}", redoFile, (Object)contents);
                this.deleteRenamePendingFile(fs, redoFile);
                return;
            }
            ArrayList<String> fileStrList = new ArrayList<String>();
            JsonNode oldFolderName = json.get("OldFolderName");
            JsonNode newFolderName = json.get("NewFolderName");
            if (oldFolderName == null || newFolderName == null) {
                this.committed = false;
            } else {
                this.srcKey = oldFolderName.getTextValue();
                this.dstKey = newFolderName.getTextValue();
                if (this.srcKey == null || this.dstKey == null) {
                    this.committed = false;
                } else {
                    JsonNode fileList = json.get("FileList");
                    if (fileList == null) {
                        this.committed = false;
                    } else {
                        for (int i = 0; i < fileList.size(); ++i) {
                            fileStrList.add(fileList.get(i).getTextValue());
                        }
                    }
                }
            }
            this.fileStrings = fileStrList;
        }

        public FileMetadata[] getFiles() {
            return this.fileMetadata;
        }

        public SelfRenewingLease getFolderLease() {
            return this.folderLease;
        }

        @VisibleForTesting
        void deleteRenamePendingFile(FileSystem fs, Path redoFile) throws IOException {
            try {
                fs.delete(redoFile, false);
            }
            catch (IOException e) {
                Throwable t = e.getCause();
                if (t != null && t instanceof StorageException && "BlobNotFound".equals(((StorageException)t).getErrorCode())) {
                    LOG.warn("rename pending file " + redoFile + " is already deleted");
                }
                throw e;
            }
        }

        public void writeFile(FileSystem fs) throws IOException {
            Path path = this.getRenamePendingFilePath();
            LOG.debug("Preparing to write atomic rename state to {}", (Object)path.toString());
            FSDataOutputStream output = null;
            String contents = this.makeRenamePendingFileContents();
            try {
                output = fs.create(path);
                ((OutputStream)output).write(contents.getBytes(Charset.forName("UTF-8")));
            }
            catch (IOException e) {
                throw new IOException("Unable to write RenamePending file for folder rename from " + this.srcKey + " to " + this.dstKey, e);
            }
            finally {
                NativeAzureFileSystemHelper.cleanup(LOG, output);
            }
        }

        public String makeRenamePendingFileContents() {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
            sdf.setTimeZone(TimeZone.getTimeZone("UTC"));
            String time = sdf.format(new Date());
            StringBuilder builder = new StringBuilder();
            builder.append("[\n");
            for (int i = 0; i != this.fileMetadata.length; ++i) {
                if (i > 0) {
                    builder.append(",\n");
                }
                builder.append("    ");
                String noPrefix = StringUtils.removeStart(this.fileMetadata[i].getKey(), this.srcKey + NativeAzureFileSystem.PATH_DELIMITER);
                builder.append(this.quote(noPrefix));
                if (builder.length() < 9990000) continue;
                LOG.error("Internal error: Exceeded maximum rename pending file size of {} bytes.", 10000000);
                return "exceeded maximum rename pending file size";
            }
            builder.append("\n  ]");
            String fileList = builder.toString();
            String contents = "{\n  FormatVersion: \"1.0\",\n  OperationUTCTime: \"" + time + "\",\n" + "  OldFolderName: " + this.quote(this.srcKey) + ",\n" + "  NewFolderName: " + this.quote(this.dstKey) + ",\n" + "  FileList: " + fileList + "\n" + "}\n";
            return contents;
        }

        private String quote(String string) {
            if (string == null || string.length() == 0) {
                return "\"\"";
            }
            char c = '\u0000';
            int len = string.length();
            StringBuilder sb = new StringBuilder(len + 4);
            sb.append('\"');
            block9: for (int i = 0; i < len; ++i) {
                c = string.charAt(i);
                switch (c) {
                    case '\"': 
                    case '\\': {
                        sb.append('\\');
                        sb.append(c);
                        continue block9;
                    }
                    case '/': {
                        sb.append('\\');
                        sb.append(c);
                        continue block9;
                    }
                    case '\b': {
                        sb.append("\\b");
                        continue block9;
                    }
                    case '\t': {
                        sb.append("\\t");
                        continue block9;
                    }
                    case '\n': {
                        sb.append("\\n");
                        continue block9;
                    }
                    case '\f': {
                        sb.append("\\f");
                        continue block9;
                    }
                    case '\r': {
                        sb.append("\\r");
                        continue block9;
                    }
                    default: {
                        if (c < ' ') {
                            String t = "000" + Integer.toHexString(c);
                            sb.append("\\u" + t.substring(t.length() - 4));
                            continue block9;
                        }
                        sb.append(c);
                    }
                }
            }
            sb.append('\"');
            return sb.toString();
        }

        public String getSrcKey() {
            return this.srcKey;
        }

        public String getDstKey() {
            return this.dstKey;
        }

        public FileMetadata getSourceMetadata() throws IOException {
            return this.fs.getStoreInterface().retrieveMetadata(this.srcKey);
        }

        public void execute() throws IOException {
            for (FileMetadata file : this.getFiles()) {
                if (file.getBlobMaterialization() != BlobMaterialization.Explicit) continue;
                String srcName = file.getKey();
                String suffix = srcName.substring(this.getSrcKey().length());
                String dstName = this.getDstKey() + suffix;
                boolean acquireLease = this.fs.getStoreInterface().isAtomicRenameKey(srcName);
                this.fs.getStoreInterface().rename(srcName, dstName, acquireLease, null);
            }
            FileMetadata srcMetadata2 = this.getSourceMetadata();
            if (srcMetadata2.getBlobMaterialization() == BlobMaterialization.Explicit) {
                this.fs.getStoreInterface().rename(this.getSrcKey(), this.getDstKey(), false, this.folderLease);
            }
            this.fs.updateParentFolderLastModifiedTime(this.srcKey);
            this.fs.updateParentFolderLastModifiedTime(this.dstKey);
        }

        public void cleanup() throws IOException {
            if (this.fs.getStoreInterface().isAtomicRenameKey(this.srcKey)) {
                this.fs.delete(this.getRenamePendingFilePath(), false);
            }
        }

        private Path getRenamePendingFilePath() {
            String fileName = this.srcKey + SUFFIX;
            Path fileNamePath = NativeAzureFileSystem.keyToPath(fileName);
            Path path = this.fs.makeAbsolute(fileNamePath);
            return path;
        }

        public void redo() throws IOException {
            if (!this.committed) {
                return;
            }
            SelfRenewingLease lease = null;
            boolean sourceFolderGone = false;
            try {
                lease = this.fs.leaseSourceFolder(this.srcKey);
            }
            catch (AzureException e) {
                String errorCode = "";
                try {
                    StorageException se = (StorageException)e.getCause();
                    errorCode = se.getErrorCode();
                }
                catch (Exception e2) {
                    // empty catch block
                }
                if (errorCode.equals("BlobNotFound")) {
                    sourceFolderGone = true;
                }
                throw new IOException("Unexpected error when trying to lease source folder name during folder rename redo", e);
            }
            if (!sourceFolderGone) {
                Path dst = this.fullPath(this.dstKey);
                if (!this.fs.exists(dst)) {
                    this.fs.mkdirs(dst);
                }
                for (String fileName : this.fileStrings) {
                    this.finishSingleFileRename(fileName);
                }
                try {
                    this.fs.getStoreInterface().delete(this.srcKey, lease);
                }
                catch (Exception e) {
                    LOG.info("Unable to delete source folder during folder rename redo. If the source folder is already gone, this is not an error condition. Continuing with redo.", e);
                }
                this.fs.updateParentFolderLastModifiedTime(this.srcKey);
                this.fs.updateParentFolderLastModifiedTime(this.dstKey);
            }
            this.fs.delete(this.getRenamePendingFilePath(), false);
        }

        private void finishSingleFileRename(String fileName) throws IOException {
            Path srcFile = this.fullPath(this.srcKey, fileName);
            Path dstFile = this.fullPath(this.dstKey, fileName);
            String srcName = this.fs.pathToKey(srcFile);
            String dstName = this.fs.pathToKey(dstFile);
            boolean srcExists = this.fs.getStoreInterface().explicitFileExists(srcName);
            boolean dstExists = this.fs.getStoreInterface().explicitFileExists(dstName);
            if (srcExists) {
                this.fs.getStoreInterface().rename(srcName, dstName, true, null);
            } else if (srcExists || !dstExists) {
                throw new IOException("Attempting to complete rename of file " + this.srcKey + NativeAzureFileSystem.PATH_DELIMITER + fileName + " during folder rename redo, and file was not found in source " + "or destination.");
            }
        }

        private Path fullPath(String folderKey, String fileName) {
            return new Path(new Path(this.fs.getUri()), NativeAzureFileSystem.PATH_DELIMITER + folderKey + NativeAzureFileSystem.PATH_DELIMITER + fileName);
        }

        private Path fullPath(String fileKey) {
            return new Path(new Path(this.fs.getUri()), NativeAzureFileSystem.PATH_DELIMITER + fileKey);
        }
    }
}

