/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.zookeeper.cli;

import org.apache.commons.cli.*;
import org.apache.zookeeper.KeeperException;

/**
 * delete command for cli
 */
public class DeleteCommand extends CliCommand {

    private static Options options = new Options();
    private String[] args;
    private CommandLine cl;

    {
        options.addOption("v", true, "version");
    }

    public DeleteCommand() {
        super("delete", "[-v version] path");
    }

    @Override
    public CliCommand parse(String[] cmdArgs) throws ParseException {
        Parser parser = new PosixParser();
        cl = parser.parse(options, cmdArgs);
        args = cl.getArgs();
        if (args.length < 2) {
            throw new ParseException(getUsageStr());
        }
        
        retainCompatibility(cmdArgs);

        return this;
    }

    private void retainCompatibility(String[] cmdArgs) throws ParseException {
        // delete path [version]
        if (args.length > 2) {
            // rewrite to option
            String [] newCmd = new String[4];
            newCmd[0] = cmdArgs [0];
            newCmd[1] = "-v";
            newCmd[2] = cmdArgs[2]; // version
            newCmd[3] = cmdArgs[1]; // path            
            err.println("'delete path [version]' has been deprecated. "
                    + "Please use 'delete [-v version] path' instead.");
            Parser parser = new PosixParser();
            cl = parser.parse(options, newCmd);
            args = cl.getArgs();
        }
    }

    @Override
    public boolean exec() throws KeeperException, InterruptedException {
        String path = args[1];
        int version;
        if (cl.hasOption("v")) {
            version = Integer.parseInt(cl.getOptionValue("v"));
        } else {
            version = -1;
        }
        
        try {
        zk.delete(path, version);
        } catch(KeeperException.BadVersionException ex) {
            err.println(ex.getMessage());
        }
        return false;
    }
}
