/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.ha;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.hdfs.protocolPB.NamenodeProtocolPB;
import org.apache.hadoop.hdfs.protocolPB.NamenodeProtocolTranslatorPB;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputException;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocol;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.ipc.StandbyException;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class EditLogTailer {
    public static final Log LOG = LogFactory.getLog(EditLogTailer.class);
    private final EditLogTailerThread tailerThread = new EditLogTailerThread();
    private final Configuration conf;
    private final FSNamesystem namesystem;
    private FSEditLog editLog;
    private InetSocketAddress activeAddr;
    private NamenodeProtocol cachedActiveProxy = null;
    private long lastRollTriggerTxId = -12345L;
    private long lastLoadedTxnId = -12345L;
    private long lastLoadTimestamp;
    private final long logRollPeriodMs;
    private final long rollEditsTimeoutMs;
    private final ExecutorService rollEditsRpcExecutor;
    private final long sleepTimeMs;

    public EditLogTailer(FSNamesystem namesystem, Configuration conf) {
        this.conf = conf;
        this.namesystem = namesystem;
        this.editLog = namesystem.getEditLog();
        this.lastLoadTimestamp = Time.now();
        this.logRollPeriodMs = conf.getInt("dfs.ha.log-roll.period", 120) * 1000;
        if (this.logRollPeriodMs >= 0L) {
            this.activeAddr = this.getActiveNodeAddress();
            Preconditions.checkArgument((this.activeAddr.getPort() > 0 ? 1 : 0) != 0, (String)"Active NameNode must have an IPC port configured. Got address '%s'", (Object[])new Object[]{this.activeAddr});
            LOG.info((Object)("Will roll logs on active node at " + this.activeAddr + " every " + this.logRollPeriodMs / 1000L + " seconds."));
        } else {
            LOG.info((Object)"Not going to trigger log rolls on active node because dfs.ha.log-roll.period is negative.");
        }
        this.sleepTimeMs = conf.getInt("dfs.ha.tail-edits.period", 60) * 1000;
        this.rollEditsTimeoutMs = conf.getInt("dfs.ha.tail-edits.rolledits.timeout", 60) * 1000;
        this.rollEditsRpcExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).build());
        LOG.debug((Object)("logRollPeriodMs=" + this.logRollPeriodMs + " sleepTime=" + this.sleepTimeMs));
    }

    private InetSocketAddress getActiveNodeAddress() {
        Configuration activeConf = HAUtil.getConfForOtherNode(this.conf);
        return NameNode.getServiceAddress(activeConf, true);
    }

    private NamenodeProtocol getActiveNodeProxy() throws IOException {
        if (this.cachedActiveProxy == null) {
            int rpcTimeout = this.conf.getInt("dfs.ha.log-roll.rpc.timeout", 20000);
            NamenodeProtocolPB proxy = (NamenodeProtocolPB)RPC.waitForProxy(NamenodeProtocolPB.class, (long)RPC.getProtocolVersion(NamenodeProtocolPB.class), (InetSocketAddress)this.activeAddr, (Configuration)this.conf, (int)rpcTimeout, (long)Long.MAX_VALUE);
            this.cachedActiveProxy = new NamenodeProtocolTranslatorPB(proxy);
        }
        assert (this.cachedActiveProxy != null);
        return this.cachedActiveProxy;
    }

    public void start() {
        this.tailerThread.start();
    }

    public void stop() throws IOException {
        this.rollEditsRpcExecutor.shutdown();
        this.tailerThread.setShouldRun(false);
        this.tailerThread.interrupt();
        try {
            this.tailerThread.join();
        }
        catch (InterruptedException e) {
            LOG.warn((Object)"Edit log tailer thread exited with an exception");
            throw new IOException(e);
        }
    }

    @VisibleForTesting
    FSEditLog getEditLog() {
        return this.editLog;
    }

    @VisibleForTesting
    public void setEditLog(FSEditLog editLog) {
        this.editLog = editLog;
    }

    public void catchupDuringFailover() throws IOException {
        Preconditions.checkState((this.tailerThread == null || !this.tailerThread.isAlive() ? 1 : 0) != 0, (Object)"Tailer thread should not be running once failover starts");
        SecurityUtil.doAsLoginUser((PrivilegedExceptionAction)new PrivilegedExceptionAction<Void>(){

            @Override
            public Void run() throws Exception {
                try {
                    EditLogTailer.this.doTailEdits();
                }
                catch (InterruptedException e) {
                    throw new IOException(e);
                }
                return null;
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void doTailEdits() throws IOException, InterruptedException {
        this.namesystem.writeLockInterruptibly();
        try {
            Collection<EditLogInputStream> streams;
            FSImage image = this.namesystem.getFSImage();
            long lastTxnId = image.getLastAppliedTxId();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("lastTxnId: " + lastTxnId));
            }
            try {
                streams = this.editLog.selectInputStreams(lastTxnId + 1L, 0L, null, false);
            }
            catch (IOException ioe) {
                LOG.warn((Object)"Edits tailer failed to find any streams. Will try again later.", (Throwable)ioe);
                this.namesystem.writeUnlock();
                return;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("edit streams to load from: " + streams.size()));
            }
            long editsLoaded = 0L;
            try {
                editsLoaded = image.loadEdits(streams, this.namesystem);
            }
            catch (EditLogInputException elie) {
                try {
                    editsLoaded = elie.getNumEditsLoaded();
                    throw elie;
                }
                catch (Throwable throwable) {
                    if (editsLoaded > 0L || LOG.isDebugEnabled()) {
                        LOG.info((Object)String.format("Loaded %d edits starting from txid %d ", editsLoaded, lastTxnId));
                    }
                    throw throwable;
                }
            }
            if (editsLoaded > 0L || LOG.isDebugEnabled()) {
                LOG.info((Object)String.format("Loaded %d edits starting from txid %d ", editsLoaded, lastTxnId));
            }
            if (editsLoaded > 0L) {
                this.lastLoadTimestamp = Time.now();
            }
            this.lastLoadedTxnId = image.getLastAppliedTxId();
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    public long getLastLoadTimestamp() {
        return this.lastLoadTimestamp;
    }

    private boolean tooLongSinceLastLoad() {
        return this.logRollPeriodMs >= 0L && Time.now() - this.lastLoadTimestamp > this.logRollPeriodMs;
    }

    @VisibleForTesting
    Callable<Void> getRollEditsTask() {
        return new Callable<Void>(){

            @Override
            public Void call() throws IOException {
                EditLogTailer.this.getActiveNodeProxy().rollEditLog();
                return null;
            }
        };
    }

    @VisibleForTesting
    void triggerActiveLogRoll() {
        LOG.info((Object)"Triggering log roll on remote NameNode");
        Future<Void> future = null;
        try {
            future = this.rollEditsRpcExecutor.submit(this.getRollEditsTask());
            future.get(this.rollEditsTimeoutMs, TimeUnit.MILLISECONDS);
            this.lastRollTriggerTxId = this.lastLoadedTxnId;
        }
        catch (ExecutionException e) {
            IOException ioe;
            Throwable cause = e.getCause();
            if (cause instanceof RemoteException && (ioe = ((RemoteException)cause).unwrapRemoteException()) instanceof StandbyException) {
                LOG.info((Object)("Skipping log roll. Remote node is not in Active state: " + ioe.getMessage().split("\n")[0]));
                return;
            }
            LOG.warn((Object)"Unable to trigger a roll of the active NN", (Throwable)e);
        }
        catch (TimeoutException e) {
            if (future != null) {
                future.cancel(true);
            }
            LOG.warn((Object)String.format("Unable to finish rolling edits in %d ms", this.rollEditsTimeoutMs));
        }
        catch (InterruptedException e) {
            LOG.warn((Object)"Unable to trigger a roll of the active NN", (Throwable)e);
        }
    }

    private class EditLogTailerThread
    extends Thread {
        private volatile boolean shouldRun;

        private EditLogTailerThread() {
            super("Edit log tailer");
            this.shouldRun = true;
        }

        private void setShouldRun(boolean shouldRun) {
            this.shouldRun = shouldRun;
        }

        @Override
        public void run() {
            SecurityUtil.doAsLoginUserOrFatal((PrivilegedAction)new PrivilegedAction<Object>(){

                @Override
                public Object run() {
                    EditLogTailerThread.this.doWork();
                    return null;
                }
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void doWork() {
            while (this.shouldRun) {
                try {
                    if (EditLogTailer.this.tooLongSinceLastLoad() && EditLogTailer.this.lastRollTriggerTxId < EditLogTailer.this.lastLoadedTxnId) {
                        EditLogTailer.this.triggerActiveLogRoll();
                    }
                    if (!this.shouldRun) break;
                    EditLogTailer.this.namesystem.cpLockInterruptibly();
                    try {
                        EditLogTailer.this.doTailEdits();
                    }
                    finally {
                        EditLogTailer.this.namesystem.cpUnlock();
                    }
                }
                catch (EditLogInputException elie) {
                    LOG.warn((Object)"Error while reading edits from disk. Will try again.", (Throwable)elie);
                }
                catch (InterruptedException ie) {
                    continue;
                }
                catch (Throwable t) {
                    LOG.fatal((Object)"Unknown error encountered while tailing edits. Shutting down standby NN.", t);
                    ExitUtil.terminate((int)1, (Throwable)t);
                }
                try {
                    Thread.sleep(EditLogTailer.this.sleepTimeMs);
                }
                catch (InterruptedException e) {
                    LOG.warn((Object)"Edit log tailer interrupted", (Throwable)e);
                }
            }
        }
    }
}

