/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicyInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicyState;
import org.apache.hadoop.hdfs.protocol.SystemErasureCodingPolicies;
import org.apache.hadoop.io.erasurecode.CodecUtil;
import org.apache.hadoop.io.erasurecode.ECSchema;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"HDFS"})
public final class ErasureCodingPolicyManager {
    public static Logger LOG = LoggerFactory.getLogger(ErasureCodingPolicyManager.class);
    private int maxCellSize = 0x400000;
    private static final byte[] SUITABLE_STORAGE_POLICIES_FOR_EC_STRIPED_MODE = new byte[]{7, 2, 12};
    private Map<String, ErasureCodingPolicyInfo> policiesByName;
    private Map<Byte, ErasureCodingPolicyInfo> policiesByID;
    private ErasureCodingPolicyInfo[] allPolicies;
    private Map<String, ErasureCodingPolicy> enabledPoliciesByName;
    private ErasureCodingPolicy[] enabledPolicies;
    private static volatile ErasureCodingPolicyManager instance = null;

    public static ErasureCodingPolicyManager getInstance() {
        if (instance == null) {
            instance = new ErasureCodingPolicyManager();
        }
        return instance;
    }

    private ErasureCodingPolicyManager() {
    }

    public void init(Configuration conf) {
        String defaultPolicyName = conf.getTrimmed("dfs.namenode.ec.system.default.policy", "RS-6-3-1024k");
        this.policiesByName = new TreeMap<String, ErasureCodingPolicyInfo>();
        this.policiesByID = new TreeMap<Byte, ErasureCodingPolicyInfo>();
        this.enabledPoliciesByName = new TreeMap<String, ErasureCodingPolicy>();
        for (ErasureCodingPolicy policy : SystemErasureCodingPolicies.getPolicies()) {
            ErasureCodingPolicyInfo info = new ErasureCodingPolicyInfo(policy);
            this.policiesByName.put(policy.getName(), info);
            this.policiesByID.put(policy.getId(), info);
        }
        if (!defaultPolicyName.isEmpty()) {
            ErasureCodingPolicyInfo info = this.policiesByName.get(defaultPolicyName);
            if (info == null) {
                String names = this.policiesByName.values().stream().map(pi -> pi.getPolicy().getName()).collect(Collectors.joining(", "));
                String msg = String.format("EC policy '%s' specified at %s is not a valid policy. Please choose from list of available policies: [%s]", defaultPolicyName, "dfs.namenode.ec.system.default.policy", names);
                throw new HadoopIllegalArgumentException(msg);
            }
            info.setState(ErasureCodingPolicyState.ENABLED);
            this.enabledPoliciesByName.put(info.getPolicy().getName(), info.getPolicy());
        }
        this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicyInfo[0]);
        this.maxCellSize = conf.getInt("dfs.namenode.ec.policies.max.cellsize", 0x400000);
    }

    public ErasureCodingPolicy[] getEnabledPolicies() {
        return this.enabledPolicies;
    }

    public ErasureCodingPolicy getEnabledPolicyByName(String name) {
        ErasureCodingPolicy ecPolicy = this.enabledPoliciesByName.get(name);
        if (ecPolicy == null && name.equalsIgnoreCase("replication")) {
            ecPolicy = SystemErasureCodingPolicies.getReplicationPolicy();
        }
        return ecPolicy;
    }

    public static boolean checkStoragePolicySuitableForECStripedMode(byte storagePolicyID) {
        boolean isPolicySuitable = false;
        for (byte suitablePolicy : SUITABLE_STORAGE_POLICIES_FOR_EC_STRIPED_MODE) {
            if (storagePolicyID != suitablePolicy) continue;
            isPolicySuitable = true;
            break;
        }
        return isPolicySuitable;
    }

    public ErasureCodingPolicyInfo[] getPolicies() {
        return this.allPolicies;
    }

    public ErasureCodingPolicy getByID(byte id) {
        ErasureCodingPolicyInfo ecpi = this.getPolicyInfoByID(id);
        if (ecpi == null) {
            return null;
        }
        return ecpi.getPolicy();
    }

    private ErasureCodingPolicyInfo getPolicyInfoByID(byte id) {
        return this.policiesByID.get(id);
    }

    public ErasureCodingPolicy getByName(String name) {
        ErasureCodingPolicyInfo ecpi = this.getPolicyInfoByName(name);
        if (ecpi == null) {
            return null;
        }
        return ecpi.getPolicy();
    }

    private ErasureCodingPolicyInfo getPolicyInfoByName(String name) {
        return this.policiesByName.get(name);
    }

    public void clear() {
    }

    public synchronized ErasureCodingPolicy addPolicy(ErasureCodingPolicy policy) {
        if (!CodecUtil.hasCodec((String)policy.getCodecName())) {
            throw new HadoopIllegalArgumentException("Codec name " + policy.getCodecName() + " is not supported");
        }
        if (policy.getCellSize() > this.maxCellSize) {
            throw new HadoopIllegalArgumentException("Cell size " + policy.getCellSize() + " should not exceed maximum " + this.maxCellSize + " bytes");
        }
        String assignedNewName = ErasureCodingPolicy.composePolicyName((ECSchema)policy.getSchema(), (int)policy.getCellSize());
        for (ErasureCodingPolicyInfo info : this.getPolicies()) {
            ErasureCodingPolicy p = info.getPolicy();
            if (p.getName().equals(assignedNewName)) {
                LOG.info("The policy name " + assignedNewName + " already exists");
                return p;
            }
            if (!p.getSchema().equals((Object)policy.getSchema()) || p.getCellSize() != policy.getCellSize()) continue;
            LOG.info("A policy with same schema " + policy.getSchema().toString() + " and cell size " + p.getCellSize() + " already exists");
            return p;
        }
        if (this.getCurrentMaxPolicyID() == 127) {
            throw new HadoopIllegalArgumentException("Adding erasure coding policy failed because the number of policies stored in the system already reached the threshold, which is 127");
        }
        policy = new ErasureCodingPolicy(assignedNewName, policy.getSchema(), policy.getCellSize(), this.getNextAvailablePolicyID());
        ErasureCodingPolicyInfo pi = new ErasureCodingPolicyInfo(policy);
        this.policiesByName.put(policy.getName(), pi);
        this.policiesByID.put(policy.getId(), pi);
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicyInfo[0]);
        return policy;
    }

    private byte getCurrentMaxPolicyID() {
        return this.policiesByID.keySet().stream().max(Byte::compareTo).orElse((byte)0);
    }

    private byte getNextAvailablePolicyID() {
        byte nextPolicyID = (byte)(this.getCurrentMaxPolicyID() + 1);
        return nextPolicyID > 64 ? nextPolicyID : (byte)64;
    }

    public synchronized void removePolicy(String name) {
        ErasureCodingPolicyInfo info = this.policiesByName.get(name);
        if (info == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        ErasureCodingPolicy ecPolicy = info.getPolicy();
        if (ecPolicy.isSystemPolicy()) {
            throw new HadoopIllegalArgumentException("System erasure coding policy " + name + " cannot be removed");
        }
        if (this.enabledPoliciesByName.containsKey(name)) {
            this.enabledPoliciesByName.remove(name);
            this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        }
        info.setState(ErasureCodingPolicyState.REMOVED);
        LOG.info("Remove erasure coding policy " + name);
    }

    @VisibleForTesting
    public List<ErasureCodingPolicy> getRemovedPolicies() {
        ArrayList<ErasureCodingPolicy> removedPolicies = new ArrayList<ErasureCodingPolicy>();
        for (ErasureCodingPolicyInfo info : this.policiesByName.values()) {
            ErasureCodingPolicy ecPolicy = info.getPolicy();
            if (!info.isRemoved()) continue;
            removedPolicies.add(ecPolicy);
        }
        return removedPolicies;
    }

    public synchronized boolean disablePolicy(String name) {
        ErasureCodingPolicyInfo info = this.policiesByName.get(name);
        if (info == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        if (this.enabledPoliciesByName.containsKey(name)) {
            this.enabledPoliciesByName.remove(name);
            this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
            info.setState(ErasureCodingPolicyState.DISABLED);
            LOG.info("Disable the erasure coding policy " + name);
            return true;
        }
        return false;
    }

    public synchronized boolean enablePolicy(String name) {
        ErasureCodingPolicyInfo info = this.policiesByName.get(name);
        if (info == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        if (this.enabledPoliciesByName.containsKey(name)) {
            return false;
        }
        ErasureCodingPolicy ecPolicy = info.getPolicy();
        this.enabledPoliciesByName.put(name, ecPolicy);
        info.setState(ErasureCodingPolicyState.ENABLED);
        this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        LOG.info("Enable the erasure coding policy " + name);
        return true;
    }

    private void loadPolicy(ErasureCodingPolicyInfo info) {
        Preconditions.checkNotNull((Object)info);
        ErasureCodingPolicy policy = info.getPolicy();
        if (!CodecUtil.hasCodec((String)policy.getCodecName()) || policy.getCellSize() > this.maxCellSize) {
            info.setState(ErasureCodingPolicyState.DISABLED);
        }
        this.policiesByName.put(policy.getName(), info);
        this.policiesByID.put(policy.getId(), info);
        if (info.isEnabled()) {
            this.enablePolicy(policy.getName());
        }
    }

    public synchronized void loadPolicies(List<ErasureCodingPolicyInfo> ecPolicies) {
        Preconditions.checkNotNull(ecPolicies);
        for (ErasureCodingPolicyInfo p : ecPolicies) {
            this.loadPolicy(p);
        }
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicyInfo[0]);
    }
}

