/*
 * Decompiled with CFR 0.152.
 */
package org.apache.falcon.entity.parser;

import java.io.IOException;
import java.net.URI;
import java.util.HashSet;
import java.util.List;
import javax.jms.ConnectionFactory;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang3.StringUtils;
import org.apache.falcon.FalconException;
import org.apache.falcon.catalog.CatalogServiceFactory;
import org.apache.falcon.entity.ClusterHelper;
import org.apache.falcon.entity.EntityUtil;
import org.apache.falcon.entity.parser.EntityParser;
import org.apache.falcon.entity.parser.ValidationException;
import org.apache.falcon.entity.v0.EntityType;
import org.apache.falcon.entity.v0.cluster.ACL;
import org.apache.falcon.entity.v0.cluster.Cluster;
import org.apache.falcon.entity.v0.cluster.ClusterLocationType;
import org.apache.falcon.entity.v0.cluster.Interface;
import org.apache.falcon.entity.v0.cluster.Interfacetype;
import org.apache.falcon.entity.v0.cluster.Location;
import org.apache.falcon.entity.v0.cluster.Properties;
import org.apache.falcon.entity.v0.cluster.Property;
import org.apache.falcon.hadoop.HadoopClientFactory;
import org.apache.falcon.util.StartupProperties;
import org.apache.falcon.workflow.WorkflowEngineFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authorize.AuthorizationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterEntityParser
extends EntityParser<Cluster> {
    private static final Logger LOG = LoggerFactory.getLogger(ClusterEntityParser.class);

    public ClusterEntityParser() {
        super(EntityType.CLUSTER);
    }

    @Override
    public void validate(Cluster cluster) throws ValidationException {
        this.validateScheme(cluster, Interfacetype.READONLY);
        this.validateScheme(cluster, Interfacetype.WRITE);
        this.validateScheme(cluster, Interfacetype.WORKFLOW);
        if (ClusterHelper.getInterface(cluster, Interfacetype.MESSAGING) != null) {
            this.validateScheme(cluster, Interfacetype.MESSAGING);
        }
        if (CatalogServiceFactory.isEnabled() && ClusterHelper.getInterface(cluster, Interfacetype.REGISTRY) != null) {
            this.validateScheme(cluster, Interfacetype.REGISTRY);
        }
        this.validateACL(cluster);
        if (!EntityUtil.responsibleFor(cluster.getColo())) {
            return;
        }
        this.validateReadInterface(cluster);
        this.validateWriteInterface(cluster);
        this.validateExecuteInterface(cluster);
        this.validateWorkflowInterface(cluster);
        this.validateMessagingInterface(cluster);
        this.validateRegistryInterface(cluster);
        this.validateLocations(cluster);
        this.validateProperties(cluster);
        this.validateSparkMasterInterface(cluster);
    }

    private void validateScheme(Cluster cluster, Interfacetype interfacetype) throws ValidationException {
        String endpoint = ClusterHelper.getInterface(cluster, interfacetype).getEndpoint();
        URI uri = new Path(endpoint).toUri();
        if (uri.getScheme() == null) {
            if (Interfacetype.WORKFLOW == interfacetype && uri.toString().equals("localoozie")) {
                return;
            }
            throw new ValidationException("Cannot get valid scheme for interface: " + (Object)((Object)interfacetype) + " of cluster: " + cluster.getName());
        }
    }

    private void validateReadInterface(Cluster cluster) throws ValidationException {
        String readOnlyStorageUrl = ClusterHelper.getReadOnlyStorageUrl(cluster);
        LOG.info("Validating read interface: {}", (Object)readOnlyStorageUrl);
        this.validateFileSystem(cluster, readOnlyStorageUrl);
    }

    private void validateWriteInterface(Cluster cluster) throws ValidationException {
        String writeStorageUrl = ClusterHelper.getStorageUrl(cluster);
        LOG.info("Validating write interface: {}", (Object)writeStorageUrl);
        this.validateFileSystem(cluster, writeStorageUrl);
    }

    private void validateFileSystem(Cluster cluster, String storageUrl) throws ValidationException {
        try {
            Configuration conf = new Configuration();
            conf.set("fs.defaultFS", storageUrl);
            conf.setInt("ipc.client.connect.max.retries", 10);
            if (UserGroupInformation.isSecurityEnabled()) {
                String nameNodePrincipal = ClusterHelper.getPropertyValue(cluster, "dfs.namenode.kerberos.principal");
                Validate.notEmpty((String)nameNodePrincipal, (String)"Cluster definition missing required namenode credential property: dfs.namenode.kerberos.principal");
                conf.set("dfs.namenode.kerberos.principal", nameNodePrincipal);
            }
            FileSystem fs = HadoopClientFactory.get().createProxiedFileSystem(conf);
            fs.exists(new Path("/"));
        }
        catch (Exception e) {
            throw new ValidationException("Invalid storage server or port: " + storageUrl + ", " + e.getMessage(), e);
        }
    }

    private void validateExecuteInterface(Cluster cluster) throws ValidationException {
        String executeUrl = ClusterHelper.getMREndPoint(cluster);
        LOG.info("Validating execute interface: {}", (Object)executeUrl);
        try {
            String rmPrincipal = ClusterHelper.getPropertyValue(cluster, "yarn.resourcemanager.principal");
            HadoopClientFactory.get().validateJobClient(executeUrl, rmPrincipal);
        }
        catch (IOException e) {
            throw new ValidationException("Invalid Execute server or port: " + executeUrl, e);
        }
    }

    protected void validateWorkflowInterface(Cluster cluster) throws ValidationException {
        String workflowUrl = ClusterHelper.getOozieUrl(cluster);
        LOG.info("Validating workflow interface: {}", (Object)workflowUrl);
        if ("localoozie".equals(workflowUrl)) {
            return;
        }
        try {
            if (!WorkflowEngineFactory.getWorkflowEngine().isAlive(cluster)) {
                throw new ValidationException("Unable to reach Workflow server:" + workflowUrl);
            }
        }
        catch (FalconException e) {
            throw new ValidationException("Invalid Workflow server or port: " + workflowUrl, e);
        }
    }

    protected void validateMessagingInterface(Cluster cluster) throws ValidationException {
        Interface messagingInterface = ClusterHelper.getInterface(cluster, Interfacetype.MESSAGING);
        if (messagingInterface == null) {
            LOG.info("Messaging service is not enabled for cluster: {}", (Object)cluster.getName());
            return;
        }
        String messagingUrl = ClusterHelper.getMessageBrokerUrl(cluster);
        String implementation = StartupProperties.get().getProperty("broker.impl.class", "org.apache.activemq.ActiveMQConnectionFactory");
        LOG.info("Validating messaging interface: {}, implementation: {}", (Object)messagingUrl, (Object)implementation);
        try {
            Class<?> clazz = this.getClass().getClassLoader().loadClass(implementation);
            ConnectionFactory connectionFactory = (ConnectionFactory)clazz.getConstructor(String.class, String.class, String.class).newInstance("", "", messagingUrl);
            connectionFactory.createConnection();
        }
        catch (Exception e) {
            throw new ValidationException("Invalid Messaging server or port: " + messagingUrl + " for: " + implementation, e);
        }
    }

    protected void validateRegistryInterface(Cluster cluster) throws ValidationException {
        boolean isCatalogRegistryEnabled = CatalogServiceFactory.isEnabled();
        if (!isCatalogRegistryEnabled) {
            return;
        }
        Interface catalogInterface = ClusterHelper.getInterface(cluster, Interfacetype.REGISTRY);
        if (catalogInterface == null) {
            LOG.info("Catalog service is not enabled for cluster: {}", (Object)cluster.getName());
            return;
        }
        String catalogUrl = catalogInterface.getEndpoint();
        LOG.info("Validating catalog registry interface: {}", (Object)catalogUrl);
        try {
            Configuration clusterConf = ClusterHelper.getConfiguration(cluster);
            if (UserGroupInformation.isSecurityEnabled()) {
                String metaStorePrincipal = clusterConf.get("hive.metastore.kerberos.principal");
                Validate.notEmpty((String)metaStorePrincipal, (String)"Cluster definition missing required metastore credential property: hive.metastore.kerberos.principal");
            }
            if (!CatalogServiceFactory.getCatalogService().isAlive(clusterConf, catalogUrl)) {
                throw new ValidationException("Unable to reach Catalog server:" + catalogUrl);
            }
        }
        catch (FalconException e) {
            throw new ValidationException("Invalid Catalog server or port: " + catalogUrl, e);
        }
    }

    protected void validateSparkMasterInterface(Cluster cluster) throws ValidationException {
        String sparkMasterEndPoint = ClusterHelper.getSparkMasterEndPoint(cluster);
        LOG.info("Validating spark interface: {}", (Object)sparkMasterEndPoint);
        if (StringUtils.isNotEmpty(sparkMasterEndPoint) && !"yarn-cluster".equalsIgnoreCase(sparkMasterEndPoint) && !"yarn-client".equalsIgnoreCase(sparkMasterEndPoint) && !"local".equalsIgnoreCase(sparkMasterEndPoint)) {
            throw new ValidationException("Invalid Spark Interface End Point:" + sparkMasterEndPoint);
        }
    }

    private void validateACL(Cluster cluster) throws ValidationException {
        if (this.isAuthorizationDisabled) {
            return;
        }
        ACL clusterACL = cluster.getACL();
        if (clusterACL == null) {
            throw new ValidationException("Cluster ACL cannot be empty for:  " + cluster.getName());
        }
        this.validateACLOwnerAndGroup(clusterACL);
        try {
            this.authorize(cluster.getName(), clusterACL);
        }
        catch (AuthorizationException e) {
            throw new ValidationException((Exception)((Object)e));
        }
    }

    protected void validateLocations(Cluster cluster) throws ValidationException {
        FileSystem fs;
        Configuration conf = ClusterHelper.getConfiguration(cluster);
        try {
            fs = HadoopClientFactory.get().createFalconFileSystem(conf);
        }
        catch (FalconException e) {
            throw new ValidationException("Unable to get file system handle for cluster " + cluster.getName(), e);
        }
        Location stagingLocation = ClusterHelper.getLocation(cluster, ClusterLocationType.STAGING);
        if (stagingLocation == null) {
            throw new ValidationException("Unable to find the mandatory location of name: " + ClusterLocationType.STAGING.value() + " for cluster " + cluster.getName());
        }
        this.checkPathOwnerAndPermission(cluster.getName(), stagingLocation.getPath(), fs, HadoopClientFactory.ALL_PERMISSION);
        if (!ClusterHelper.checkWorkingLocationExists(cluster)) {
            this.createWorkingDirUnderStaging(fs, cluster, stagingLocation);
        } else {
            Location workingLocation = ClusterHelper.getLocation(cluster, ClusterLocationType.WORKING);
            if (stagingLocation.getPath().equals(workingLocation.getPath())) {
                throw new ValidationException("Location with name: " + stagingLocation.getName().value() + " and " + workingLocation.getName().value() + " cannot have same path: " + stagingLocation.getPath() + " for cluster :" + cluster.getName());
            }
            this.checkPathOwnerAndPermission(cluster.getName(), workingLocation.getPath(), fs, HadoopClientFactory.READ_EXECUTE_PERMISSION);
        }
        this.createStagingSubdirs(fs, cluster, stagingLocation, "falcon/workflows/feed", HadoopClientFactory.ALL_PERMISSION);
        this.createStagingSubdirs(fs, cluster, stagingLocation, "falcon/workflows/process", HadoopClientFactory.ALL_PERMISSION);
        this.createStagingSubdirs(fs, cluster, stagingLocation, "EMPTY_DIR_DONT_DELETE", HadoopClientFactory.READ_ONLY_PERMISSION);
    }

    private void createWorkingDirUnderStaging(FileSystem fs, Cluster cluster, Location stagingLocation) throws ValidationException {
        block5: {
            Path workingDirPath = new Path(stagingLocation.getPath(), "working");
            try {
                if (!fs.exists(workingDirPath)) {
                    HadoopClientFactory.mkdirs(fs, workingDirPath, HadoopClientFactory.READ_EXECUTE_PERMISSION);
                    break block5;
                }
                if (fs.isDirectory(workingDirPath)) {
                    FsPermission workingPerms = fs.getFileStatus(workingDirPath).getPermission();
                    if (!workingPerms.equals((Object)HadoopClientFactory.READ_EXECUTE_PERMISSION)) {
                        throw new ValidationException("Falcon needs subdir working inside staging dir:" + stagingLocation.getPath() + " when staging location not specified with " + HadoopClientFactory.READ_EXECUTE_PERMISSION.toString() + " got " + workingPerms.toString());
                    }
                    break block5;
                }
                throw new ValidationException("Falcon needs subdir working inside staging dir:" + stagingLocation.getPath() + " when staging location not specified. Got a file at " + workingDirPath.toString());
            }
            catch (IOException e) {
                throw new ValidationException("Unable to create path for " + workingDirPath.toString() + " with path: " + workingDirPath.toString() + " for cluster " + cluster.getName(), e);
            }
        }
    }

    private void createStagingSubdirs(FileSystem fs, Cluster cluster, Location stagingLocation, String path, FsPermission permission) throws ValidationException {
        Path subdirPath = new Path(stagingLocation.getPath(), path);
        try {
            HadoopClientFactory.mkdirs(fs, subdirPath, permission);
        }
        catch (IOException e) {
            throw new ValidationException("Unable to create path " + subdirPath.toString() + " for cluster " + cluster.getName(), e);
        }
    }

    protected void validateProperties(Cluster cluster) throws ValidationException {
        Properties properties = cluster.getProperties();
        if (properties == null) {
            return;
        }
        List<Property> propertyList = cluster.getProperties().getProperties();
        HashSet<String> propertyKeys = new HashSet<String>();
        for (Property prop : propertyList) {
            if (StringUtils.isBlank(prop.getName())) {
                throw new ValidationException("Property name and value cannot be empty for Cluster: " + cluster.getName());
            }
            if (propertyKeys.add(prop.getName())) continue;
            throw new ValidationException("Multiple properties with same name found for Cluster: " + cluster.getName());
        }
    }

    private void checkPathOwnerAndPermission(String clusterName, String location, FileSystem fs, FsPermission expectedPermission) throws ValidationException {
        Path locationPath = new Path(location);
        try {
            if (!fs.exists(locationPath)) {
                throw new ValidationException("Location " + location + " for cluster " + clusterName + " must exist.");
            }
            String loginUser = UserGroupInformation.getLoginUser().getShortUserName();
            FileStatus fileStatus = fs.getFileStatus(locationPath);
            String locationOwner = fileStatus.getOwner();
            if (!locationOwner.equals(loginUser)) {
                LOG.error("Owner of the location {} is {} for cluster {}. Current user {} is not the owner of the location.", new Object[]{locationPath, locationOwner, clusterName, loginUser});
                throw new ValidationException("Path [" + locationPath + "] on the cluster [" + clusterName + "] has owner [" + locationOwner + "]. Current user [" + loginUser + "] is not the owner of the path");
            }
            String errorMessage = "Path " + locationPath + " has permissions: " + fileStatus.getPermission().toString() + ", should be " + expectedPermission;
            if (fileStatus.getPermission().toShort() != expectedPermission.toShort()) {
                LOG.error(errorMessage);
                throw new ValidationException(errorMessage);
            }
            fs.listStatus(locationPath);
        }
        catch (IOException e) {
            throw new ValidationException("Unable to validate the location with path: " + location + " for cluster:" + clusterName + " due to transient failures ", e);
        }
    }
}

