/*
 * Decompiled with CFR 0.152.
 */
package io.specto.hoverfly.junit.core;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import io.specto.hoverfly.junit.core.HoverflyConfig;
import io.specto.hoverfly.junit.core.HoverflyMode;
import io.specto.hoverfly.junit.core.HoverflySslUtils;
import io.specto.hoverfly.junit.core.HoverflyUtils;
import io.specto.hoverfly.junit.core.SimulationSource;
import io.specto.hoverfly.junit.core.model.Simulation;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.ProcessExecutor;
import org.zeroturnaround.exec.StartedProcess;
import org.zeroturnaround.exec.stream.slf4j.Slf4jStream;

public class Hoverfly {
    private static final Logger LOGGER = LoggerFactory.getLogger(Hoverfly.class);
    private static final int BOOT_TIMEOUT_SECONDS = 10;
    private static final int RETRY_BACKOFF_INTERVAL_MS = 100;
    private static final String HEALTH_CHECK_PATH = "/api/stats";
    private static final String SIMULATION_PATH = "/api/v2/simulation";
    private static final int DEFAULT_PROXY_PORT = 8500;
    private static final int DEFAULT_ADMIN_PORT = 8888;
    private final HoverflyConfig hoverflyConfig;
    private final HoverflyMode hoverflyMode;
    private final Integer proxyPort;
    private final Integer adminPort;
    private final WebResource hoverflyResource;
    private StartedProcess startedProcess;
    private Path binaryPath;

    public Hoverfly(HoverflyConfig hoverflyConfig, HoverflyMode hoverflyMode) {
        this.hoverflyConfig = hoverflyConfig;
        this.hoverflyMode = hoverflyMode;
        if (hoverflyConfig.isRemoteInstance()) {
            this.proxyPort = hoverflyConfig.getProxyPort() == 0 ? 8500 : hoverflyConfig.getProxyPort();
            this.adminPort = hoverflyConfig.getAdminPort() == 0 ? 8888 : hoverflyConfig.getAdminPort();
        } else {
            this.proxyPort = hoverflyConfig.getProxyPort() == 0 ? HoverflyUtils.findUnusedPort() : hoverflyConfig.getProxyPort();
            this.adminPort = hoverflyConfig.getAdminPort() == 0 ? HoverflyUtils.findUnusedPort() : hoverflyConfig.getAdminPort();
        }
        this.hoverflyResource = Client.create().resource(UriBuilder.fromUri((String)(HoverflyConfig.configs().isRemoteInstance() ? HoverflyConfig.configs().getRemoteHost() : "http://localhost")).port(this.adminPort.intValue()).build(new Object[0]));
    }

    public Hoverfly(HoverflyMode hoverflyMode) {
        this(HoverflyConfig.configs(), hoverflyMode);
    }

    public void start() {
        if (!this.hoverflyConfig.isRemoteInstance()) {
            this.startHoverflyProcess();
        }
        this.waitForHoverflyToBecomeHealthy();
        HoverflySslUtils.setTrustStore();
        this.setProxySystemProperties();
    }

    private void startHoverflyProcess() {
        HoverflyUtils.checkPortInUse(this.proxyPort);
        HoverflyUtils.checkPortInUse(this.adminPort);
        try {
            this.binaryPath = this.extractBinary(HoverflyUtils.getBinaryName());
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not excecute binary", e);
        }
        LOGGER.info("Executing binary at {}", (Object)this.binaryPath);
        ArrayList<String> commands = new ArrayList<String>();
        commands.add(this.binaryPath.toString());
        commands.add("-db");
        commands.add("memory");
        commands.add("-pp");
        commands.add(this.proxyPort.toString());
        commands.add("-ap");
        commands.add(this.adminPort.toString());
        if (this.hoverflyMode == HoverflyMode.CAPTURE) {
            commands.add("-capture");
        }
        try {
            this.startedProcess = new ProcessExecutor().command(commands).redirectOutput((OutputStream)Slf4jStream.of((Logger)LOGGER).asInfo()).directory(this.binaryPath.getParent().toFile()).start();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not start Hoverfly process", e);
        }
    }

    public void stop() {
        LOGGER.info("Destroying hoverfly process");
        if (this.startedProcess != null) {
            this.startedProcess.getProcess().destroy();
        }
        if (this.binaryPath != null) {
            try {
                Files.deleteIfExists(this.binaryPath);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to delete hoverfly binary", (Throwable)e);
                this.binaryPath.toFile().deleteOnExit();
            }
        }
    }

    public void importSimulation(SimulationSource simulationSource) {
        LOGGER.info("Importing simulation data to Hoverfly");
        simulationSource.getSimulation().ifPresent(s -> this.hoverflyResource.path(SIMULATION_PATH).type(MediaType.APPLICATION_JSON_TYPE).put(s));
    }

    public void exportSimulation(Path path) {
        LOGGER.info("Exporting simulation data from Hoverfly");
        try {
            Files.deleteIfExists(path);
            Files.write(path, ((String)this.hoverflyResource.path(SIMULATION_PATH).get(String.class)).getBytes(), new OpenOption[0]);
        }
        catch (Exception e) {
            LOGGER.error("Failed to export simulation data", (Throwable)e);
        }
    }

    public Simulation getSimulation() {
        return (Simulation)this.hoverflyResource.path(SIMULATION_PATH).get(Simulation.class);
    }

    public int getProxyPort() {
        return this.proxyPort;
    }

    private boolean isHealthy() {
        try (ClientResponse response = null;){
            response = (ClientResponse)this.hoverflyResource.path(HEALTH_CHECK_PATH).get(ClientResponse.class);
            LOGGER.debug("Hoverfly health check status code is: {}", (Object)response.getStatus());
            boolean bl = response.getStatus() == ClientResponse.Status.OK.getStatusCode();
            return bl;
        }
        return false;
    }

    private void setProxySystemProperties() {
        LOGGER.info("Setting proxy host to {}", (Object)"localhost");
        System.setProperty("http.proxyHost", "localhost");
        System.setProperty("https.proxyHost", "localhost");
        if (this.hoverflyConfig.isProxyLocalHost()) {
            System.setProperty("http.nonProxyHosts", "");
        } else {
            System.setProperty("http.nonProxyHosts", "local|*.local|169.254/16|*.169.254/16");
        }
        LOGGER.info("Setting proxy proxyPort to {}", (Object)this.hoverflyConfig.getProxyPort());
        System.setProperty("http.proxyPort", this.proxyPort.toString());
        System.setProperty("https.proxyPort", this.proxyPort.toString());
    }

    private void waitForHoverflyToBecomeHealthy() {
        Instant now = Instant.now();
        while (Duration.between(now, Instant.now()).getSeconds() < 10L) {
            if (this.isHealthy()) {
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        throw new IllegalStateException("Hoverfly has not become healthy in 10 seconds");
    }

    private Path extractBinary(String binaryName) throws IOException {
        LOGGER.info("Selecting the following binary based on the current operating system: {}", (Object)binaryName);
        URL sourceHoverflyUrl = HoverflyUtils.findResourceOnClasspath("binaries/" + binaryName);
        Path temporaryHoverflyPath = Files.createTempFile(binaryName, "", new FileAttribute[0]);
        LOGGER.info("Storing binary in temporary directory {}", (Object)temporaryHoverflyPath);
        File temporaryHoverflyFile = temporaryHoverflyPath.toFile();
        FileUtils.copyURLToFile((URL)sourceHoverflyUrl, (File)temporaryHoverflyFile);
        if (SystemUtils.IS_OS_WINDOWS) {
            temporaryHoverflyFile.setExecutable(true);
            temporaryHoverflyFile.setReadable(true);
            temporaryHoverflyFile.setWritable(true);
        } else {
            Files.setPosixFilePermissions(temporaryHoverflyPath, new HashSet<PosixFilePermission>(Arrays.asList(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.OWNER_READ)));
        }
        return temporaryHoverflyPath;
    }

    public int getAdminPort() {
        return this.adminPort;
    }
}

