/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.codec.spdy;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.spdy.SpdyFrameDecoder;
import io.netty.handler.codec.spdy.SpdyFrameDecoderDelegate;
import io.netty.handler.codec.spdy.SpdyVersion;
import java.util.ArrayDeque;
import java.util.Queue;
import java.util.Random;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class SpdyFrameDecoderTest {
    private static final Random RANDOM = new Random();
    private final SpdyFrameDecoderDelegate delegate = (SpdyFrameDecoderDelegate)Mockito.mock(SpdyFrameDecoderDelegate.class);
    private final TestSpdyFrameDecoderDelegate testDelegate = new TestSpdyFrameDecoderDelegate();
    private SpdyFrameDecoder decoder;

    @Before
    public void createDecoder() {
        this.decoder = new SpdyFrameDecoder(SpdyVersion.SPDY_3_1, (SpdyFrameDecoderDelegate)this.testDelegate);
    }

    @After
    public void releaseBuffers() {
        this.testDelegate.releaseAll();
    }

    private static void encodeDataFrameHeader(ByteBuf buffer, int streamId, byte flags, int length) {
        buffer.writeInt(streamId & Integer.MAX_VALUE);
        buffer.writeByte((int)flags);
        buffer.writeMedium(length);
    }

    private static void encodeControlFrameHeader(ByteBuf buffer, short type, byte flags, int length) {
        buffer.writeShort(0x8000 | SpdyVersion.SPDY_3_1.getVersion());
        buffer.writeShort((int)type);
        buffer.writeByte((int)flags);
        buffer.writeMedium(length);
    }

    @Test
    public void testSpdyDataFrame() throws Exception {
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        byte flags = 0;
        int length = 1024;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId, flags, length);
        for (int i = 0; i < 256; ++i) {
            buf.writeInt(RANDOM.nextInt());
        }
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId, false, buf.slice(8, length));
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testEmptySpdyDataFrame() throws Exception {
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        byte flags = 0;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId, false, Unpooled.EMPTY_BUFFER);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testLastSpdyDataFrame() throws Exception {
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        byte flags = 1;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId, true, Unpooled.EMPTY_BUFFER);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdyDataFrameFlags() throws Exception {
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        byte flags = -2;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId, false, Unpooled.EMPTY_BUFFER);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdyDataFrameStreamId() throws Exception {
        int streamId = 0;
        byte flags = 0;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError((String)Mockito.any());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testPipelinedSpdyDataFrames() throws Exception {
        int streamId1 = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int streamId2 = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        byte flags = 0;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(2 * (8 + length)));
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId1, flags, length);
        SpdyFrameDecoderTest.encodeDataFrameHeader(buf, streamId2, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId1, false, Unpooled.EMPTY_BUFFER);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readDataFrame(streamId2, false, Unpooled.EMPTY_BUFFER);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySynStreamFrame() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testLastSpdySynStreamFrame() throws Exception {
        short type = 1;
        byte flags = 1;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, true, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnidirectionalSpdySynStreamFrame() throws Exception {
        short type = 1;
        byte flags = 2;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, true);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIndependentSpdySynStreamFrame() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = 0;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdySynStreamFrameFlags() throws Exception {
        short type = 1;
        byte flags = -4;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdySynStreamFrameBits() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 10;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId | Integer.MIN_VALUE);
        buf.writeInt(associatedToStreamId | Integer.MIN_VALUE);
        buf.writeByte(priority << 5 | 0x1F);
        buf.writeByte(255);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdySynStreamFrameLength() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdySynStreamFrameStreamId() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 10;
        int streamId = 0;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySynStreamFrameHeaderBlock() throws Exception {
        short type = 1;
        byte flags = 0;
        int length = 10;
        int headerBlockLength = 1024;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int associatedToStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        byte priority = (byte)(RANDOM.nextInt() & 7);
        ByteBuf buf = Unpooled.buffer((int)(8 + length + headerBlockLength));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length + headerBlockLength);
        buf.writeInt(streamId);
        buf.writeInt(associatedToStreamId);
        buf.writeByte(priority << 5);
        buf.writeByte(0);
        ByteBuf headerBlock = Unpooled.buffer((int)headerBlockLength);
        for (int i = 0; i < 256; ++i) {
            headerBlock.writeInt(RANDOM.nextInt());
        }
        this.decoder.decode(buf);
        this.decoder.decode(headerBlock);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynStreamFrame(streamId, associatedToStreamId, priority, false, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlock(headerBlock.slice(0, headerBlock.writerIndex()));
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        Assert.assertFalse((boolean)headerBlock.isReadable());
        buf.release();
        headerBlock.release();
    }

    @Test
    public void testSpdySynReplyFrame() throws Exception {
        short type = 2;
        byte flags = 0;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynReplyFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testLastSpdySynReplyFrame() throws Exception {
        short type = 2;
        byte flags = 1;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynReplyFrame(streamId, true);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdySynReplyFrameFlags() throws Exception {
        short type = 2;
        byte flags = -2;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynReplyFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdySynReplyFrameBits() throws Exception {
        short type = 2;
        byte flags = 0;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId | Integer.MIN_VALUE);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynReplyFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdySynReplyFrameLength() throws Exception {
        short type = 2;
        byte flags = 0;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdySynReplyFrameStreamId() throws Exception {
        short type = 2;
        byte flags = 0;
        int length = 4;
        int streamId = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySynReplyFrameHeaderBlock() throws Exception {
        short type = 2;
        byte flags = 0;
        int length = 4;
        int headerBlockLength = 1024;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length + headerBlockLength));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length + headerBlockLength);
        buf.writeInt(streamId);
        ByteBuf headerBlock = Unpooled.buffer((int)headerBlockLength);
        for (int i = 0; i < 256; ++i) {
            headerBlock.writeInt(RANDOM.nextInt());
        }
        this.decoder.decode(buf);
        this.decoder.decode(headerBlock);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSynReplyFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlock(headerBlock.slice(0, headerBlock.writerIndex()));
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        Assert.assertFalse((boolean)headerBlock.isReadable());
        buf.release();
        headerBlock.release();
    }

    @Test
    public void testSpdyRstStreamFrame() throws Exception {
        short type = 3;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readRstStreamFrame(streamId, statusCode);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdyRstStreamFrameBits() throws Exception {
        short type = 3;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId | Integer.MIN_VALUE);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readRstStreamFrame(streamId, statusCode);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyRstStreamFrameFlags() throws Exception {
        short type = 3;
        byte flags = -1;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyRstStreamFrameLength() throws Exception {
        short type = 3;
        byte flags = 0;
        int length = 12;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdyRstStreamFrameStreamId() throws Exception {
        short type = 3;
        byte flags = 0;
        int length = 8;
        int streamId = 0;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdyRstStreamFrameStatusCode() throws Exception {
        short type = 3;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        int statusCode = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySettingsFrame() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 2;
        int length = 8 * numSettings + 4;
        int idFlags = 0;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.delegate.readSettingsEnd();
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate, (VerificationMode)Mockito.times((int)numSettings))).readSetting(id, value, false, false);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testEmptySpdySettingsFrame() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 0;
        int length = 8 * numSettings + 4;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySettingsFrameClearFlag() throws Exception {
        short type = 4;
        byte flags = 1;
        int numSettings = 0;
        int length = 8 * numSettings + 4;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(true);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySettingsPersistValues() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 1;
        int length = 8 * numSettings + 4;
        int idFlags = 1;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.delegate.readSettingsEnd();
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate, (VerificationMode)Mockito.times((int)numSettings))).readSetting(id, value, true, false);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdySettingsPersistedValues() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 1;
        int length = 8 * numSettings + 4;
        int idFlags = 2;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.delegate.readSettingsEnd();
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate, (VerificationMode)Mockito.times((int)numSettings))).readSetting(id, value, false, true);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdySettingsFrameFlags() throws Exception {
        short type = 4;
        byte flags = -2;
        int numSettings = 0;
        int length = 8 * numSettings + 4;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdySettingsFlags() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 1;
        int length = 8 * numSettings + 4;
        int idFlags = -4;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.delegate.readSettingsEnd();
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readSettingsFrame(false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate, (VerificationMode)Mockito.times((int)numSettings))).readSetting(id, value, false, false);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdySettingsFrameLength() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 2;
        int length = 8 * numSettings + 8;
        int idFlags = 0;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(numSettings);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdySettingsFrameNumSettings() throws Exception {
        short type = 4;
        byte flags = 0;
        int numSettings = 2;
        int length = 8 * numSettings + 4;
        int idFlags = 0;
        int id = RANDOM.nextInt() & 0xFFFFFF;
        int value = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(0);
        for (int i = 0; i < numSettings; ++i) {
            buf.writeByte(idFlags);
            buf.writeMedium(id);
            buf.writeInt(value);
        }
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testDiscardUnknownFrame() throws Exception {
        short type = 5;
        byte flags = -1;
        int length = 8;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeLong(RANDOM.nextLong());
        this.decoder.decode(buf);
        Mockito.verifyZeroInteractions((Object[])new Object[]{this.delegate});
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testDiscardUnknownEmptyFrame() throws Exception {
        short type = 5;
        byte flags = -1;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        this.decoder.decode(buf);
        Mockito.verifyZeroInteractions((Object[])new Object[]{this.delegate});
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testProgressivelyDiscardUnknownEmptyFrame() throws Exception {
        short type = 5;
        byte flags = -1;
        int segment = 4;
        int length = 2 * segment;
        ByteBuf header = Unpooled.buffer((int)8);
        ByteBuf segment1 = Unpooled.buffer((int)segment);
        ByteBuf segment2 = Unpooled.buffer((int)segment);
        SpdyFrameDecoderTest.encodeControlFrameHeader(header, type, flags, length);
        segment1.writeInt(RANDOM.nextInt());
        segment2.writeInt(RANDOM.nextInt());
        this.decoder.decode(header);
        this.decoder.decode(segment1);
        this.decoder.decode(segment2);
        Mockito.verifyZeroInteractions((Object[])new Object[]{this.delegate});
        Assert.assertFalse((boolean)header.isReadable());
        Assert.assertFalse((boolean)segment1.isReadable());
        Assert.assertFalse((boolean)segment2.isReadable());
        header.release();
        segment1.release();
        segment2.release();
    }

    @Test
    public void testSpdyPingFrame() throws Exception {
        short type = 6;
        byte flags = 0;
        int length = 4;
        int id = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(id);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readPingFrame(id);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdyPingFrameFlags() throws Exception {
        short type = 6;
        byte flags = -1;
        int length = 4;
        int id = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(id);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readPingFrame(id);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyPingFrameLength() throws Exception {
        short type = 6;
        byte flags = 0;
        int length = 8;
        int id = RANDOM.nextInt();
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(id);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdyGoAwayFrame() throws Exception {
        short type = 7;
        byte flags = 0;
        int length = 8;
        int lastGoodStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(lastGoodStreamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readGoAwayFrame(lastGoodStreamId, statusCode);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdyGoAwayFrameFlags() throws Exception {
        short type = 7;
        byte flags = -1;
        int length = 8;
        int lastGoodStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(lastGoodStreamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readGoAwayFrame(lastGoodStreamId, statusCode);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdyGoAwayFrameBits() throws Exception {
        short type = 7;
        byte flags = 0;
        int length = 8;
        int lastGoodStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(lastGoodStreamId | Integer.MIN_VALUE);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readGoAwayFrame(lastGoodStreamId, statusCode);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyGoAwayFrameLength() throws Exception {
        short type = 7;
        byte flags = 0;
        int length = 12;
        int lastGoodStreamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int statusCode = RANDOM.nextInt() | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(lastGoodStreamId);
        buf.writeInt(statusCode);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdyHeadersFrame() throws Exception {
        short type = 8;
        byte flags = 0;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeadersFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testLastSpdyHeadersFrame() throws Exception {
        short type = 8;
        byte flags = 1;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeadersFrame(streamId, true);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testUnknownSpdyHeadersFrameFlags() throws Exception {
        short type = 8;
        byte flags = -2;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeadersFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdyHeadersFrameBits() throws Exception {
        short type = 8;
        byte flags = 0;
        int length = 4;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId | Integer.MIN_VALUE);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeadersFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyHeadersFrameLength() throws Exception {
        short type = 8;
        byte flags = 0;
        int length = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyHeadersFrameStreamId() throws Exception {
        short type = 8;
        byte flags = 0;
        int length = 4;
        int streamId = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testSpdyHeadersFrameHeaderBlock() throws Exception {
        short type = 8;
        byte flags = 0;
        int length = 4;
        int headerBlockLength = 1024;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length + headerBlockLength);
        buf.writeInt(streamId);
        ByteBuf headerBlock = Unpooled.buffer((int)headerBlockLength);
        for (int i = 0; i < 256; ++i) {
            headerBlock.writeInt(RANDOM.nextInt());
        }
        this.decoder.decode(buf);
        this.decoder.decode(headerBlock);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeadersFrame(streamId, false);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlock(headerBlock.slice(0, headerBlock.writerIndex()));
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readHeaderBlockEnd();
        Assert.assertFalse((boolean)buf.isReadable());
        Assert.assertFalse((boolean)headerBlock.isReadable());
        buf.release();
        headerBlock.release();
    }

    @Test
    public void testSpdyWindowUpdateFrame() throws Exception {
        short type = 9;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int deltaWindowSize = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(deltaWindowSize);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readWindowUpdateFrame(streamId, deltaWindowSize);
        Assert.assertFalse((boolean)buf.isReadable());
    }

    @Test
    public void testUnknownSpdyWindowUpdateFrameFlags() throws Exception {
        short type = 9;
        byte flags = -1;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int deltaWindowSize = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(deltaWindowSize);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readWindowUpdateFrame(streamId, deltaWindowSize);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testReservedSpdyWindowUpdateFrameBits() throws Exception {
        short type = 9;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int deltaWindowSize = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId | Integer.MIN_VALUE);
        buf.writeInt(deltaWindowSize | Integer.MIN_VALUE);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readWindowUpdateFrame(streamId, deltaWindowSize);
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testInvalidSpdyWindowUpdateFrameLength() throws Exception {
        short type = 9;
        byte flags = 0;
        int length = 12;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int deltaWindowSize = RANDOM.nextInt() & Integer.MAX_VALUE | 1;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(deltaWindowSize);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    @Test
    public void testIllegalSpdyWindowUpdateFrameDeltaWindowSize() throws Exception {
        short type = 9;
        byte flags = 0;
        int length = 8;
        int streamId = RANDOM.nextInt() & Integer.MAX_VALUE;
        int deltaWindowSize = 0;
        ByteBuf buf = Unpooled.buffer((int)(8 + length));
        SpdyFrameDecoderTest.encodeControlFrameHeader(buf, type, flags, length);
        buf.writeInt(streamId);
        buf.writeInt(deltaWindowSize);
        this.decoder.decode(buf);
        ((SpdyFrameDecoderDelegate)Mockito.verify((Object)this.delegate)).readFrameError(Mockito.anyString());
        Assert.assertFalse((boolean)buf.isReadable());
        buf.release();
    }

    private final class TestSpdyFrameDecoderDelegate
    implements SpdyFrameDecoderDelegate {
        private final Queue<ByteBuf> buffers = new ArrayDeque<ByteBuf>();

        private TestSpdyFrameDecoderDelegate() {
        }

        public void readDataFrame(int streamId, boolean last, ByteBuf data) {
            SpdyFrameDecoderTest.this.delegate.readDataFrame(streamId, last, data);
            this.buffers.add(data);
        }

        public void readSynStreamFrame(int streamId, int associatedToStreamId, byte priority, boolean last, boolean unidirectional) {
            SpdyFrameDecoderTest.this.delegate.readSynStreamFrame(streamId, associatedToStreamId, priority, last, unidirectional);
        }

        public void readSynReplyFrame(int streamId, boolean last) {
            SpdyFrameDecoderTest.this.delegate.readSynReplyFrame(streamId, last);
        }

        public void readRstStreamFrame(int streamId, int statusCode) {
            SpdyFrameDecoderTest.this.delegate.readRstStreamFrame(streamId, statusCode);
        }

        public void readSettingsFrame(boolean clearPersisted) {
            SpdyFrameDecoderTest.this.delegate.readSettingsFrame(clearPersisted);
        }

        public void readSetting(int id, int value, boolean persistValue, boolean persisted) {
            SpdyFrameDecoderTest.this.delegate.readSetting(id, value, persistValue, persisted);
        }

        public void readSettingsEnd() {
            SpdyFrameDecoderTest.this.delegate.readSettingsEnd();
        }

        public void readPingFrame(int id) {
            SpdyFrameDecoderTest.this.delegate.readPingFrame(id);
        }

        public void readGoAwayFrame(int lastGoodStreamId, int statusCode) {
            SpdyFrameDecoderTest.this.delegate.readGoAwayFrame(lastGoodStreamId, statusCode);
        }

        public void readHeadersFrame(int streamId, boolean last) {
            SpdyFrameDecoderTest.this.delegate.readHeadersFrame(streamId, last);
        }

        public void readWindowUpdateFrame(int streamId, int deltaWindowSize) {
            SpdyFrameDecoderTest.this.delegate.readWindowUpdateFrame(streamId, deltaWindowSize);
        }

        public void readHeaderBlock(ByteBuf headerBlock) {
            SpdyFrameDecoderTest.this.delegate.readHeaderBlock(headerBlock);
            this.buffers.add(headerBlock);
        }

        public void readHeaderBlockEnd() {
            SpdyFrameDecoderTest.this.delegate.readHeaderBlockEnd();
        }

        public void readFrameError(String message) {
            SpdyFrameDecoderTest.this.delegate.readFrameError(message);
        }

        void releaseAll() {
            ByteBuf buf;
            while ((buf = this.buffers.poll()) != null) {
                buf.release();
            }
            return;
        }
    }
}

