# -*- coding: utf-8 -*
from __future__ import print_function
import xlrd
import re
import os
import json
import sys
import logging as logger

CONTENT_START=6
MAX_IP = 4

LOG_FILE = "/var/log/ceph_install.log"
#LOG_FILE = "ceph_install.log"

logger.basicConfig(filename=LOG_FILE,
                  level=logger.DEBUG,
                  filemode='a',
                  format='input parse %(asctime)s %(levelname)s %(message)s')

# 获取当前EXCEL中的sheetname下的文件内容
class excelReader(object):
    def __init__(self, file):
        self.filename = file

    def getExcelData(self, sheetname):
        try:
            with xlrd.open_workbook(self.filename) as r_file:
                sheet_names_list = r_file.sheet_names()
                is_sheetname_exists = False
                sheet = None
                if sheetname not in sheet_names_list:
                    for i, sheet_name in enumerate(sheet_names_list):
                        nu_list1 = re.findall(r'[0-9]+.[0-9]+|[0-9]+', sheet_name)
                        nu_list2 = re.findall(r'[0-9]+.[0-9]+|[0-9]+', sheetname)
                        if len(nu_list1) > 0:
                            sheet_name_new = sheet_name[len(nu_list1[0]):]
                        else:
                            sheet_name_new = sheet_name
                        if len(nu_list2) > 0:
                            sheetname_new = sheetname[len(nu_list2[0]):]
                        else:
                            sheetname_new = sheetname

                        if sheet_name_new.strip() == sheetname_new.strip():
                            index = i
                            sheet = r_file.sheet_by_index(index)
                            is_sheetname_exists = True
                            break
                else:
                    sheet = r_file.sheet_by_name(sheetname)
                    is_sheetname_exists = True
            if not is_sheetname_exists:
                return [], ''
            return [self.getcellvalue(sheet, i) for i in range(sheet.nrows)], ""

        except Exception as why:
            logger.error(why.__traceback__)
            logger.error("get the excel of %s content data failed,the reason is %s" % (sheetname, why))
            return [], why

    def getcelltype(self, cell, sheet, nrow, ncolmn):
        content = None
        if cell.ctype == xlrd.XL_CELL_TEXT or cell.ctype == xlrd.XL_CELL_EMPTY:
            content = cell.value
        elif cell.ctype == xlrd.XL_CELL_NUMBER:
            value = str(cell.value)
            # 如果是.0结尾的浮点数，转成整数
            if value.endswith('.0'):
                content = value[:-2]
            else:
                content = cell.value
        elif cell.ctype == xlrd.XL_CELL_BOOLEAN:
            content = 'TRUE' if cell.value else 'FALSE'
        else:
            content = None

        if not content:
            merged = sheet.merged_cells
            for (rlow, rhigh, clow, chigh) in merged:
                if (nrow >= rlow and nrow < rhigh):
                    if (ncolmn >= clow and ncolmn < chigh):
                        cell_value = sheet.cell_value(rlow, clow)
                        # logger.error('该单元格[%d,%d]属于合并单元格，值为[%s]' % (row_index, col_index, cell_value))
                        content = cell_value
                        break
        return content

    def getcellvalue(self, sheet, nrow):
        return [self.getcelltype(sheet.cell(nrow, i), sheet, nrow, i) for i in range(sheet.ncols)]

class excelParser(object):
    def __init__(self, filename, output_path):
        self.filename = filename
        self.dir_path = os.path.abspath(os.path.dirname(__file__))
        if output_path[-1] != "/":
            output_path += "/"
        self.output_path = self.dir_path + "/" + output_path
        self.reader = excelReader(self.dir_path + "/" + self.filename)

        self.sheetnames = ["Ceph Configuration", "Server Configuration", "Client Configuration"]
        self.pcardkey = ["Public1_netcard", "Public2_netcard", "Public3_netcard", "Public4_netcard"]
        self.ccardkey = ["Cluster1_netcard", "Cluster2_netcard", "Cluster3_netcard", "Cluster4_netcard"]
        self.pipkey = ["Public1_IP", "Public2_IP", "Public3_IP", "Public4_IP"]
        self.cipkey = ["Cluster1_IP", "Cluster2_IP", "Cluster3_IP", "Cluster4_IP"]
        self.normalconfig = {}
        self.osd = set()
        self.mgr = set()
        self.mon = set()
        self.mds = set()
        self.rgw = set()
        self.servers = {}
        self.clients = {}
        self.managerips = set()
        self.clusterips = set()
        self.publicips = set()

    def content_parse(self):
        if self.parse_normal_config() and self.parse_server_config() and self.parse_client_config():
            return True
        return False

    def check_config(self, server_config):
        for item in server_config:
            configkey = item.get("Key", "")
            content = item.get("Value", "")
            if configkey.lower().startswith("public") or configkey.lower().startswith("cluster"):
                if not self.check_IP(content):
                    logger.error("字段 %s 内容 %s, IP格式不合法" % (configkey, content))
                    print("字段 %s 内容 %s, IP格式不合法" % (configkey, content))
                    return False
            if configkey.lower() == "db_size" or configkey.lower() == "wal_size":
                if len(self.normalconfig.get("db_wal_disk", {})) > 0:
                    if content == "0":
                        logger.error("字段 db or wal need 配置了 %s 必须配置内容" % (configkey))
                        print("字段 db or wal need 配置了 %s 必须配置内容" % (configkey))
                        continue
                    content = int(content)
            #增加网段以及网关地址的校验
            if configkey.lower() == "data_disk" or configkey.lower() == "db_wal_disk":
                if content != "":
                    content = content.split(",")
                else:
                    content = []
            self.normalconfig[configkey] = content
        return True

    def parse_normal_config(self):
        sheet_name = self.sheetnames[0]
        data_content = list(self.reader.getExcelData(sheet_name)[0])
        if not data_content:
            logger.error("%s 获取数据内容失败" % sheet_name)
            print("%s 获取数据内容失败" % sheet_name)
            return False

        data_list = [dict(zip(data_content[5], item)) for item in data_content[6:]]

        ret = self.check_config(data_list)
        if not ret:
            logger.error("%s 数据格式检查错误" % sheet_name)
            print("%s 数据格式检查错误" % sheet_name)
            return False
        return True

    def parse_server_config(self):
        sheet_name = self.sheetnames[1]
        data_content = list(self.reader.getExcelData(sheet_name)[0])
        if not data_content:
            logger.error("get the excel of device_information sheet content failed!")
            return False

        data_list = [dict(zip(data_content[5], item)) for item in data_content[6:]]
        ret = self.check_server_config(data_list)
        if not ret:
            logger.error("%s 数据格式检查错误" % sheet_name)
            print("%s 数据格式检查错误" % sheet_name)
            return False

        return True

    def check_server_config(self, data_list):
        if len(data_list) <= 0:
            logger.error("服务端列表不能配置为空")
            print("服务端列表不能配置为空")
            return False
        #获取网卡配置信息， public1和cluster1必须配置，public2-4 和cluster 2-4可选
        # 以第一个server为准
        firstitem = data_list[0]
        cnet = [True, False, False, False]
        pnet = [True, False, False, False]
        for i in range(0, MAX_IP):
            cnet[i] = firstitem.get(self.ccardkey[i], "").strip() != ""
            pnet[i] = firstitem.get(self.pcardkey[i], "").strip() != ""

        for item in data_list:
            if not self.parse_server_cell(item, cnet, pnet):
                return False
        return True

    def parse_server_cell(self, serverinfo, cnet, pnet):
        servername = serverinfo.get("Hostname")
        managerip = serverinfo.get("ManagerIp")
        bmcip = serverinfo.get("BmcIP")
        server_detail = {}
        publicips = []
        clusterips = []
        publicnetcards = []
        clusternetcards = []
        if servername in self.servers.keys():
            logger.error("Hostname %s, 有重复请检查结果" %servername)
            print("Hostname %s, 有重复请检查结果" % servername)
            return False

        if not managerip or not self.check_IP(managerip) or managerip in self.managerips:
            logger.error("Hostname %s, 管理ip配置有问题请检查输入" % servername)
            print("Hostname %s, 管理ip配置有问题请检查输入" % servername)
            return False

        #判断网络配置同第一台服务器是否一致
        for i in range(0, MAX_IP):
            if cnet[i]:
                card = serverinfo.get(self.ccardkey[i], "")
                ip = serverinfo.get(self.cipkey[i], "")
                if not card or not ip:
                    logger.error("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % servername)
                    print("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % servername)
                    return False
                # 判断主机内ip是否有重复
                if not self.check_IP(ip) or ip in clusterips:
                    logger.error("Hostname %s, ip %s 地址配置重复， 请校验结果" % (servername, ip))
                    print("Hostname %s, ip %s 地址配置重复， 请校验结果" % (servername, ip))
                    return False
                # 判断主机内网卡是否有重复
                if not self.check_IP(ip) or card in clusternetcards:
                    logger.error("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (servername, card))
                    print("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (servername, card))
                    return False
                clusterips.append(ip)
                clusternetcards.append(card)

            if pnet[i]:
                card = serverinfo.get(self.pcardkey[i], "")
                ip = serverinfo.get(self.pipkey[i], "")
                if not card or not ip:
                    logger.error("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" %servername)
                    print("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % servername)
                    return False
                if ip in publicips:
                    logger.error("Hostname %s, ip %s 地址配置重复， 请校验结果" % (servername, ip))
                    print("Hostname %s, ip %s 地址配置重复， 请校验结果" % (servername, ip))
                    return False
                if card in publicnetcards:
                    logger.error("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (servername, card))
                    print("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (servername, card))
                    return False
                publicips.append(ip)
                publicnetcards.append(card)

        #判断主机内ip是否已经在其他主机上配置
        if set(publicips) & self.publicips:
            logger.error("Hostname %s, public ip地址 %s 同其他主机地址配置重复， 请校验结果" % (servername, publicips))
            print("Hostname %s, public ip地址 %s 同其他主机地址配置重复， 请校验结果" % (servername, publicips))
            return False
        if set(clusterips) & self.clusterips:
            logger.error("Hostname %s, cluster ip地址 %s 同其他主机地址配置重复， 请校验结果" % (servername, clusterips))
            print("Hostname %s, cluster ip地址 %s 同其他主机地址配置重复， 请校验结果" % (servername, clusterips))
            return False

        bond_index = 0
        pubcarddict, bond_index = self.parse_netcard(publicnetcards, bond_index)
        clusterdict, bond_index = self.parse_netcard(clusternetcards, bond_index)

        server_detail["public_ip"] = publicips
        server_detail["cluster_ip"] = clusterips
        server_detail["public_netcard"] = pubcarddict
        server_detail["cluster_netcard"] = clusterdict
        server_detail["manager_ip"] = managerip
        server_detail["bmc_ip"] = bmcip
        self.servers[servername] = server_detail
        self.clusterips = self.clusterips | set(clusterips)
        self.publicips = self.publicips | set(publicips)
        self.managerips.add(managerip)
        if serverinfo.get("Mgr", "") == "Y":
            self.mgr.add(servername)
        if serverinfo.get("Mon", "") == "Y":
            self.mon.add(servername)
        if serverinfo.get("Osd", "") == "Y":
            self.osd.add(servername)
        if serverinfo.get("Mds", "") == "Y":
            self.mds.add(servername)
        if serverinfo.get("Rgw", "") == "Y":
            self.rgw.add(servername)
        return True

    def parse_netcard(self, cards, bond_index):
        carddict = {}
        if len(cards) == 0:
            return {}
        for cardinfo in cards:
            if cardinfo and cardinfo.find(",") >=0:
                carddict["bond" + str(bond_index)] = cardinfo.split(",")
                bond_index += 1
            else:
                carddict[cardinfo] = [cardinfo]

        return carddict, bond_index

    def check_IP(self, ip):
        sep = ip.split('.')
        if len(sep) != 4:
            return False
        for i, x in enumerate(sep):
            int_x = int(x)
            if int_x < 0 or int_x > 255:
                return False
        return True

    def parse_client_config(self):
        sheet_name = self.sheetnames[2]
        data_content = list(self.reader.getExcelData(sheet_name)[0])
        if not data_content:
            logger.error("%s 获取数据表内容失败" % sheet_name)
            print("%s 获取数据表内容失败" % sheet_name)
            return False

        data_list = [dict(zip(data_content[5], item)) for item in data_content[6:]]
        ret = self.check_client_config(data_list)
        if not ret:
            logger.error("%s 数据格式检查错误" %sheet_name)
            print("%s 数据格式检查错误" %sheet_name)
            return False
        return True

    def check_client_config(self, data_list):
        if len(data_list) <= 0:
            logger.error("客户端列表为空")
            return True
        #获取网卡配置信息， public1必须配置，public2-4 可选
        # 以第一个server为准
        firstitem = data_list[0]
        pnet = [True, False, False, False]
        for i in range(0, MAX_IP):
            pnet[i] = firstitem.get(self.pcardkey[i], "").strip() != ""

        for item in data_list:
            if not self.parse_client_cell(item, pnet):
                logger.error("解析客户端信息失败")
                print("解析客户端信息失败")
                return False
        return True

    def parse_client_cell(self, serverinfo, pnet):
        clientname = serverinfo.get("Hostname")
        managerip = serverinfo.get("ManagerIp")
        bmcip = serverinfo.get("BmcIP")
        client_detail = {}
        publicips = []
        publicnetcards = []
        if clientname in self.clients.keys():
            logger.error("Hostname %s, 有重复请检查结果" % clientname)
            print("Hostname %s, 有重复请检查结果" % clientname)
            return False

        if not managerip or not self.check_IP(managerip) or managerip in self.managerips:
            logger.error("Hostname %s, 管理ip配置有问题请检查输入。" % clientname)
            print("Hostname %s, 管理ip配置有问题请检查输入。" % clientname)
            return False

        # 判断网络配置同第一台服务器是否一致
        for i in range(0, MAX_IP):
            if pnet[i]:
                card = serverinfo.get(self.pcardkey[i], "")
                ip = serverinfo.get(self.pipkey[i], "")
                if not card or not ip:
                    logger.error("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % clientname)
                    print("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % clientname)
                    return False
                if ip in publicips:
                    logger.error("Hostname %s, ip %s 地址配置重复， 请校验结果" % (clientname,ip))
                    print("Hostname %s, 同其他列Cluster网络配置不一致， 请校验结果" % clientname)
                    return False
                if card in publicnetcards:
                    logger.error("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (clientname,card))
                    print("Hostname %s, 网卡 %s 地址配置重复， 请校验结果" % (clientname, card))
                    return False
                publicips.append(ip)
                publicnetcards.append(card)

        # 判断主机内ip是否已经在其他主机上配置
        if set(publicips) & self.publicips:
            logger.error("Hostname %s, ip地址 %s 同其他主机地址配置重复， 请校验结果" % (clientname, publicips))
            print("Hostname %s, ip地址 %s 同其他主机地址配置重复， 请校验结果" % (clientname, publicips))
            return False

        bond_index = 0
        pubcards_dict, bond_index = self.parse_netcard(publicnetcards, bond_index)
        client_detail["public_ip"] = publicips
        client_detail["public_netcard"] = pubcards_dict
        client_detail["manager_ip"] = managerip
        client_detail["bmc_ip"] = bmcip
        self.clients[clientname] = client_detail
        self.publicips = self.publicips | set(publicips)
        self.managerips.add(managerip)
        return True

    def exchange_mask(self, mask):
        count_bit = lambda bin_str: len([i for i in bin_str if i == '1'])
        mask_splited = mask.split('.')
        mask_count = [count_bit(bin(int(i))) for i in mask_splited]
        return sum(mask_count)

    #####################################################################
    def output_config_print(self):
        filename = "common.cfg"
        client_ips = []
        server_ips = []
        client_names = []
        server_names = []
        manager_ips = []
        hostnames = []

        for key, item in self.servers.items():
            server_names.append(key)
            server_ips.append(item.get("public_ip")[0])
            manager_ips.append(item.get("manager_ip"))
            hostnames.append(key)
            with open(self.output_path + key, 'w+') as f:
                print("manager_ip={0}".format(item.get("manager_ip")), file=f)
                print("hostname={0}".format(key), file=f)
                print("public_ip=({0})".format(" ".join(item.get("public_ip"))), file=f)
                print("cluster_ip=({0})".format(" ".join(item.get("cluster_ip"))), file=f)
                print("public_ethlist=({0})".format(" ".join(item.get("public_netcard").keys())), file=f)
                print("cluster_ethlist=({0})".format(" ".join(item.get("cluster_netcard").keys())), file=f)
                bondinfo = []
                for card, detail in item.get("public_netcard").items():
                    if len(detail) >= 2:
                        bondinfo.append("'{0} {1}'".format(card, " ".join(detail)))
                for card, detail in item.get("cluster_netcard").items():
                    if len(detail) >= 2:
                        bondinfo.append("'{0} {1}'".format(card, " ".join(detail)))
                if len(bondinfo) > 0:
                    print("bond=({0})".format(" ".join(bondinfo)), file=f)

                print("dataDevList='{0}'".format(" ".join(self.normalconfig.get("data_disk", ""))), file=f)
                print("dataDevSize={0}".format(self.normalconfig.get("data_disk_size", "")), file=f)
                print("dbwalDevList='{0}'".format(" ".join(self.normalconfig.get("db_wal_disk", ""))), file=f)
                #print("dbwalDevSize={0}".format(self.normalconfig.get("db_wal_disk_size", "")), file=f)
                print("dbSize={0}".format(self.normalconfig.get("db_size", 0)), file=f)
                print("walSize={0}".format(self.normalconfig.get("wal_size", 0)), file=f)
                print("all_node_ssd={0}".format(1 if self.normalconfig.get("all_node_ssd", "").lower() == "y" else 0), file=f)
                print("firewall={0}".format(1 if self.normalconfig.get("firewall_open", "").lower() == "y" else 0), file=f)
                print("osd_per_dev={0}".format(self.normalconfig.get("osd_per_dev", "")), file=f)
                print("", file=f)

        for key, item in self.clients.items():
            client_names.append(key)
            client_ips.append(item.get("public_ip")[0])
            manager_ips.append(item.get("manager_ip"))
            hostnames.append(key)
            with open(self.output_path + key, 'w+') as f:
                print("manager_ip={0}".format(item.get("manager_ip")), file=f)
                print("hostname={0}".format(key), file=f)
                print("public_ip=({0})".format(" ".join(item.get("public_ip"))), file=f)
                print("public_ethlist=({0})".format(" ".join(item.get("public_netcard").keys())), file=f)
                bondinfo = []
                for card, detail in item.get("public_netcard").items():
                    if len(detail) >= 2:
                        bondinfo.append("'{0} {1})'".format(card, " ".join(detail)))
                if len(bondinfo) > 0:
                    print("bond=({0})".format(" ".join(bondinfo)), file=f)
                print("firewall={0}".format(1 if self.normalconfig.get("firewall_open", "").lower() == "y" else 0),
                      file=f)
                print("", file=f)

        with open(self.output_path + filename, 'w+') as f:
            print("root_pwd={0}".format(self.normalconfig.get("root_password")), file=f)
            print("ntp_server={0}".format(self.normalconfig.get("ntp_server")), file=f)

            print("public_net={0}/{1}".format(self.normalconfig.get("public_net"),
                                              self.exchange_mask(self.normalconfig.get("public_mask"))), file=f)
            print("public_gateway={0}".format(self.normalconfig.get("public_gateway")), file=f)
            print("public_mask={0}".format(self.normalconfig.get("public_mask")), file=f)

            print("cluster_net={0}/{1}".format(self.normalconfig.get("cluster_net"),
                                                self.exchange_mask(self.normalconfig.get("cluster_mask"))), file=f)
            print("cluster_gateway={0}".format(self.normalconfig.get("cluster_gateway")), file=f)
            print("cluster_mask={0}".format(self.normalconfig.get("cluster_mask")), file=f)
            print("server=({0})".format(" ".join(server_ips)), file=f)
            print("servername=({0})".format(" ".join(server_names)), file=f)
            print("client=({0})".format(" ".join(client_ips)), file=f)
            print("clientname=({0})".format(" ".join(client_names)), file=f)
            print("managerips=({0})".format(" ".join(manager_ips)), file=f)
            print("hostnames=({0})".format(" ".join(hostnames)), file=f)

    def get_config_info(self):
        import copy
        cfg = copy.deepcopy(self.normalconfig)
        cfg["public_net"] = "{0}/{1}".format(self.normalconfig.get("public_net"),
                                              self.exchange_mask(self.normalconfig.get("public_mask")))
        cfg["cluster_net"] = "{0}/{1}".format(self.normalconfig.get("cluster_net"),
                                                        self.exchange_mask(self.normalconfig.get("cluster_mask")))
        #cfg["db_wal_disk"] = self.normalconfig.get("db_wal_disk").split(",")
        cfg["mons"] = list(self.mon)
        cfg["osds"] = list(self.osd)
        cfg["mdss"] = list(self.mds)
        cfg["rgws"] = list(self.rgw)
        cfg["mgrs"] = list(self.mgr)
        cfg["clients"] = list(self.clients.keys())
        cfg["osd_per_dev"] = int(self.normalconfig.get("osd_per_dev"))
        monitor_ip = {}
        for name in self.servers.keys():
            monitor_ip[name] = self.servers[name].get("public_ip")[0]
        cfg["host_ip"] = monitor_ip
        with open(self.output_path + 'ceph_install.cfg', 'w') as f:
            json.dump(cfg, f)
        return cfg


if __name__ == "__main__":

    name = "input/ceph_input.xlsx"
    path = "scripts/conf/"
    parser = excelParser(name, path)
    re = parser.content_parse()
    if re:
        parser.output_config_print()
        logger.info(parser.servers)
        logger.info(parser.clients)
        logger.info(parser.normalconfig)
        logger.info(parser.osd)
        logger.info(parser.mgr)
        logger.info(parser.mds)
        logger.info(parser.mon)
        logger.info(parser.rgw)
        logger.info(parser.get_config_info())
        print("解析 %s 成功" % name)
        sys.exit(0)
    print("解析 %s 失败" % name)
    sys.exit(1)
