#!/usr/bin/env bash

# Copyright (C) 2022 NETINT Technologies
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

SCRIPT_VERSION="v3.6"

# Global variables and default settings
SCRIPT_PATH=$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" 2>&1 > /dev/null; pwd)
CUSTOM_PATH="${SCRIPT_PATH}/FL_BIN"           # Arg for --path
UPGRADE_LOG="/tmp/netint_fw_upgrade_log.txt"  # Path to background upgrade processes (cold_upgrade(), warm_upgrade(), warm_upgrade_check()) log
FLAG_ERROR=false         # Error occured
FLAG_COLD_UPGRADE=false  # Prefer cold upgrade over warm upgrade
FLAG_ASSUME_YES=false    # Assume Yes to prompt
FLAG_SKIP_SAME_VER_UPGRADES=false # Skip upgrade for cards with same FWrev as FW binary file's FWrev
FLAG_WARM_UPGRADE_REINIT=false # Attempt to handle ni_rsrc re-initialization for warm upgraded cards
FLAG_LOG_TO_DMESG=false  # Write some high level logs to dmesg
NI_RSRC_PERMISSION_PREFIX="" # Will be "sudo " if user does not have write access to /dev/shm/NI* files; else ""
FW_BIN=""                # Path to FW bin to use for upgrade
FW_BIN_FWREV=""          # 8 character FW rev of FW_BIN
FW_IMG=""                # Path to FW img to use for warm upgrade if applicable
FW_IMG_FWREV=""          # 8 character FW rev of FW_IMG

# Global arrays holding card info and sharing same index order
# Below associative arrays are all indexed by NVMe device path (eg. /dev/nvme0)
declare -a DEVICES=()        # List of NVMe device paths (eg. /dev/nvme0)
declare -A NODES=()          # List of NVMe block paths (eg. /dev/nvme0n1)
declare -A SERIAL_NUMS=()    # Card serial numbers
declare -A UPGRADE_TYPE=()   # Single letters for upgrade type to use: w-warm, c-cold, s-skip
declare -A MODEL_NUMS=()     # Card model numbers
declare -A FW_REVISION=()    # Card FW revs
declare -A UPGRADE_RESULT=() # Integers for result of upgrade: -1-skip, 0-good, 1- reset_error, 2-download_error, 3-commit_error, 4-activation_error, 5-fw_rev_check_error

# Configure variables for terminal color text
function setup_terminal_colors() {
    if [[ $SHELL =~ .*zsh ]]; then
        cRst="\x1b[0m"
        cRed="\x1b[31m"
        cGrn="\x1b[32m"
        cYlw="\x1b[33m"
        cBlu="\x1b[34m"
        cMag="\x1b[35m"
        cCyn="\x1b[36m"
    else
        cRst="\e[0m"
        cRed="\e[31m"
        cGrn="\e[32m"
        cYlw="\e[33m"
        cBlu="\e[34m"
        cMag="\e[35m"
        cCyn="\e[36m"
    fi
}

# Write a log message to multiple places at same time. Note, writing to dmesg
# only allowed if $FLAG_LOG_TO_DMESG is true
# $1 - log message text
# $2 - bitmask for log destination (0:stdout, 1:stderr, 2:dmesg, 3:upgrade_log.txt)
function log() {
    LOG_DEST=$2
    if (( (LOG_DEST & 0x1) == 0x1 )); then
        echo -e "$1"
    fi
    if (( (LOG_DEST & 0x2) == 0x2 )); then
        echo -e "$1" 1>&2
    fi
    # Not user friendly, but conforming with verbatim requirement
    if $FLAG_LOG_TO_DMESG && (( ((LOG_DEST & 0x4) == 0x4) || ((LOG_DEST & 0x1) == 0x1) || ((LOG_DEST & 0x2) == 0x2) )); then
        # TODO: on some server/OS(ubuntu22) messages containing color codes are mistakenly converted by /dev/kmsg to zsh color codes on write. For now, just remove color codes from messages going to kmsg
        sudo sh -c "echo 'quadra_auto_upgrade.sh: ${1}' | perl -pe 's/\e\[[0-9;]*m//g' > /dev/kmsg"
        # sudo sh -c "echo 'quadra_auto_upgrade.sh: ${1}' > /dev/kmsg"
    fi
    # if $FLAG_LOG_TO_DMESG && (( (LOG_DEST & 0x4) == 0x4 )); then
        # sudo sh -c "echo 'quadra_auto_upgrade.sh: ${1}' > /dev/kmsg"
    # fi
    if (( (LOG_DEST & 0x8) == 0x8 )); then
        # Note: multiple processes will write to same $UPGRADE_LOG concurrently. This may have intra-line interleaving on NFS/HDFS
        echo -e "$1" >> $UPGRADE_LOG
    fi
}

# Prints usage information for the shell script
# $1 - if true, print full help text
function print_help_text() {
    echo "-------- quadra_auto_upgrade.sh --------"
    echo "Perform firmware upgrade to Quadra cards on system."
    echo "This script will automatically detect Quadra cards and select upgrade method."
    echo ""
    echo "Usage: ./quadra_auto_upgrade.sh [OPTION]"
    echo "Example: ./quadra_auto_upgrade.sh -p FL_BIN/nor_flash_v2.2.0_RC1.bin"
    echo ""
    echo "Options:"
    echo "-p, --path=PATH"
    echo "    Set PATH to firmware binary file (eg. nor_flash.bin) to use for upgrade; or,"
    echo "    set PATH to folder containing official release firmware binary files with"
    echo "    official name format (nor_flash_v*.*.*-Quadra*.bin) where file with largest"
    echo "    release version will be auto-selected."
    echo "    (Default: <quadra_auto_upgrade.sh path>/FL_BIN/)"
    echo "-c, --cold"
    echo "    Prefer cold upgrade to warm upgrade. If this option is not selected, warm"
    echo "    upgrade will be used when firmware loader component is same between firmware"
    echo "    binary file and card's current firmware."
    echo "-y, --yes"
    echo "    Automatically answer yes to all prompts. Requires password-less sudo access."
    echo "--skip_same_ver_upgrades"
    echo "    Skip upgrade for cards which have same firmware revision and firmware loader"
    echo "    component as the firmware binary file to use for firmware upgrade."
    echo "--reinit_after_warm_upgrade"
    echo "    Handle ni_rsrc re-initialization (init_rsrc) for warm upgraded cards."
    echo "--log_to_dmesg"
    # Not user friendly, but conforming with verbatim requirement
    echo "    Write log messages going to stdout/stderr to dmesg as well"
    # echo "    Write high level log messages to dmesg in addition to stdout/stderr"
    echo "-h, --help"
    echo "    Display short help text and exit."
    echo "--help_full"
    echo "    Display full text and exit."
    echo "-v, --version"
    echo "    Output version information and exit."
    if $1; then
        echo ""
        echo "Description:"
        echo "quadra_auto_upgrade.sh performs automated FW upgrade for Netint Quadra hardware."
        echo "It detects availabe Quadra devices on system, can automatically select a viable"
        echo "release binary, perform FW upgrades, then produce a summary table of results."
        echo ""
        echo "FW upgrade is prevented when Netint device is running video operations as it"
        echo "could intefere with FW upgrade success."
        echo ""
        echo "The core FW upgrade process involves a few steps:"
        echo "    1. Reset:    Perform initial controller reset to ensure good state."
        echo "    2. Download: Write FW binary file to Netint device's DDR memory."
        echo "    3. Commit:   Instruct the Netint device to move the downloaded FW binary to"
        echo "                 NOR flash memory and use it after next reset. If commit fails,"
        echo "                 the current FW on the Netint device will still be used."
        echo "    4. Activate: Reset to activate the new FW binary. For cold upgrade the reset"
        echo "                 will be to power-cycle the card (hotplug, host reboot, etc.)."
        echo "                 User will need to trigger the power-cycle. For warm upgrade the"
        echo "                 reset will be an NVMe controller reset issued by this script."
        echo "    5. Check:    After upgrade, check new FW is being run. This script will"
        echo "                 check result of warm upgrade; but not cold upgrade, as its"
        echo "                 activation involves host reboot."
        echo "The core FW upgrade process uses the nvme-cli application to run its steps. The"
        echo "nvme-cli application requires root permissions (use of sudo) as it interacts"
        echo "with the kernel's NVMe driver."
        echo ""
        echo "Trouble-shooting:"
        echo "If the error message 'Failed to generate FW image from FW binary' appears, check"
        echo "user has permissions and disk space to write a 5MB file to /tmp/."
        echo ""
        echo "If FW commit fails due to 'firmware image specified for activation is invalid'"
        echo "this means that the signature of the FW binary does not match its contents,"
        echo "possibly due to file corruption. Try re-downloading the FW release from Netint."
        echo ""
        echo "If no Netint devices are detected in ni_rsrc_mon after FW ugprade, check"
        echo "ownership of Netint shared memory files (/dev/shm/NI*) and run ni_rsrc_mon with"
        echo "sudo if the shared memory files are owned by root user."
        echo "This script can be configured to re-init the shared memory files after warm"
        echo "upgrade. This will result in root user's ownership of the shared memory files if"
        echo "root user owned the shared memory files at the start of quadra_auto_upgrade.sh."
        echo ""
        echo "If no Netint devices are detected by quadra_auto_upgrade.sh check that the"
        echo "kernel's NVMe driver is loaded. On Linux, run 'sudo lsmod | grep nvme' and check"
        echo "for 'nvme' and 'nvme_core'. Use 'sudo modprobe -a nvme_core nvme' to start nvme"
        echo "driver."
    fi
    return 0
}

# Parse Command line arguments
function parse_command_line_args() {
    while [ "$1" != "" ]; do
        case $1 in
            -p | --path)                 CUSTOM_PATH=${2}; shift
            ;;
            -c | --cold)                 FLAG_COLD_UPGRADE=true
            ;;
            -y | --yes)                  FLAG_ASSUME_YES=true
            ;;
            --skip_same_ver_upgrades)    FLAG_SKIP_SAME_VER_UPGRADES=true
            ;;
            --reinit_after_warm_upgrade) FLAG_WARM_UPGRADE_REINIT=true
            ;;
            --log_to_dmesg)              FLAG_LOG_TO_DMESG=true
            ;;
            -h | --help)                 print_help_text false; exit 0
            ;;
            --help_full)                 print_help_text true; exit 0
            ;;
            -v | --version)              echo $SCRIPT_VERSION; exit 0
            ;;
            *)                           echo "quadra_auto_upgrade.sh: invalid option -- '${1}'" 1>&2;
                                         echo "Try './quadra_auto_upgrade.sh -h' for more information." 1>&2;
                                         exit 1
            ;;
        esac
        shift
    done
}

# Check for passwordless sudo access
# return 0 if true, 124 if false
function sudo_check() {
    timeout -k 1 1 sudo whoami &> /dev/null
    return $?
}

# Determine whether to use sudo for accessing ni_rsrc_mon and ni_rsrc_update
# Set global variable NI_RSRC_PERMISSION_PREFIX to "sudo " or ""
function get_ni_rsrc_permission_prefix() {
    if [ ! -f /dev/shm/NI_SHM_CODERS ]; then
        NI_RSRC_PERMISSION_PREFIX=""
    elif [ -w /dev/shm/NI_SHM_CODERS ]; then
        NI_RSRC_PERMISSION_PREFIX=""
    else
        NI_RSRC_PERMISSION_PREFIX="sudo "
    fi
}

# Present a yes/no question, read answer
# $1 - prompt to use. Note, ' Press [y/n]: ' will be auto-inserted
# return 1 if yes, 0 if no
function prompt_yn() {
    if ! $FLAG_ASSUME_YES; then
        while true; do
            read -p "${1} Press [y/n]: " -n  1 -r
            log "" 1
            if [[ $REPLY =~ ^[Yy1]$ ]]; then
                return 1
            elif [[ $REPLY =~ ^[Nn0]$ ]]; then
                return 0
            else
                log "${cYlw}Warning${cRst}: Unrecognized input. Please try again." 1
            fi
        done
    else
        # Assume yes case
        log "${1} Press [y/n]: y" 1
        return 1
    fi
}

# Check nvme-cli is installed on system
# return 0 if installed, 1 if failed
function check_nvme_cli() {
    if (! [[ -x "$(command -v nvme)" ]]) && (! [[ -x "$(sudo which nvme)" ]]); then
        # ERROR: nvme-cli not installed
        log "${cRed}Error${cRst}: NVMe-CLI is not installed. Please install it and try again!" 6
        return 1
    fi
    return 0
}

# Initialize upgrade script log file for individual card upgrade functions run
# in parallel (reset_device(), cold_upgrade(), warm_upgrade_check(), and warm_upgrade())
# return 0 if successful, 1 if failed
function create_upgrade_log() {
    sudo rm $UPGRADE_LOG &> /dev/null
    touch $UPGRADE_LOG
    return 0
}

# Update the global arrays $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS with
# latest information of card/device
# return 0 if successful, 1 if failed
function get_cards_info() {
    # Local arrays that will be updated and used
    UNSORTED_DEVICES=()
    UNSORTED_NODES=()
    UNSORTED_SERIAL_NUMS=()
    UNSORTED_MODEL_NUMS=()
    UNSORTED_FW_REVISION=()
    INDEX_SEQUENCE=()       # Index sequence for sorting

    # Reset global arrays for this function's output
    DEVICES=()
    NODES=()
    SERIAL_NUMS=()
    FW_REVISION=()
    MODEL_NUMS=()

    # Get the unsorted information of the cards/devices from sudo nvme list
    XCOD_DATA=(`sudo nvme list | sed -e '1,2d' | grep -P "Quadra" | sed 's/\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\).*\(........\)/\1 \2 \3 \4/'`)
    # Check all devices found are physical functions
    PFs=`ls -l /sys/class/block/ | grep 00.0/nvme`
    for (( i=0; i<${#XCOD_DATA[@]}; i+=4 )); do
        NODE=$(echo ${XCOD_DATA[$i]} | perl -nle'print $& while m{(?<=/dev/).*$}g' 2> /dev/null)
        if echo $PFs | grep -q $NODE; then
            # Get device path for a given block path
            if which udevadm &> /dev/null && \
               udevadm info -q path -n ${XCOD_DATA[$i]} &> /dev/null && \
               udevadm info -q path -n ${XCOD_DATA[$i]} 2> /dev/null | perl -nle'print $& while m{(?<=/)nvme\d+(?=(/|$))}g' &> /dev/null; then
                # Check udevadm to find connection from block name to device name
                DEVID="/dev/$(udevadm info -q path -n ${XCOD_DATA[$i]} | perl -nle'print $& while m{(?<=/)nvme\d+(?=(/|$))}g')"
            else
                # Guess device path as contraction of block path
                DEVID=`echo ${XCOD_DATA[$i]} | cut -c 6-`
            fi
            UNSORTED_DEVICES+=(${DEVID})
            UNSORTED_NODES+=(${XCOD_DATA[$i]})
            UNSORTED_SERIAL_NUMS+=(${XCOD_DATA[$i+1]})
            UNSORTED_MODEL_NUMS+=(${XCOD_DATA[$i+2]})
            UNSORTED_FW_REVISION+=(${XCOD_DATA[$i+3]})
        fi
    done

    # Set up indices for sorting
    INDEX_SEQUENCE=( $(IFS=$'\n'; sort -V <<< "${UNSORTED_DEVICES[*]}" | perl -nle'print $& while m{(?<=/dev/nvme)\d+}g') )
    if [ -z $INDEX_SEQUENCE ]; then
        log "${cRed}Error${cRst}: No Quadra Transcoder device found" 6
        return 1
    fi

    # Sorts and updates $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS
    for INDEX in ${INDEX_SEQUENCE[@]}; do
        for DEV_IDX in ${!UNSORTED_DEVICES[@]}; do
            if [[ ${UNSORTED_DEVICES[$DEV_IDX]} == /dev/nvme${INDEX} ]]; then
                DEVICES+=(${UNSORTED_DEVICES[$DEV_IDX]})
                NODES[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_NODES[$DEV_IDX]}
                SERIAL_NUMS[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_SERIAL_NUMS[$DEV_IDX]}
                MODEL_NUMS[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_MODEL_NUMS[$DEV_IDX]}
                FW_REVISION[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_FW_REVISION[$DEV_IDX]}
            fi
        done
    done

    return 0
}

# Check for some simple errors after get_cards_info() but before start of upgrade()
function error_checking() {
    if [[ ${#DEVICES[@]} == 0 ]]; then
        # ERROR: no devices found
        log "${cRed}Error${cRst}: No Quadra Transcoder device found" 6
        return 1
    elif [[ ${#SERIAL_NUMS[@]} != ${#DEVICES[@]} ]]; then
        # ERROR: not all serial numbers for all cards retrieved
        log "${cRed}Error${cRst}: Not all cards have proper serial number format" 6
        return 1
    elif [ -n ${CUSTOM_PATH} ] && [ ! -f ${CUSTOM_PATH} ] && [ ! -d ${CUSTOM_PATH} ]; then
        # ERROR: invalid path specified for --path
        log "${cRed}Error${cRst}: Cannot find specified FW file or folder at ${CUSTOM_PATH}" 6
        return 1
    elif ${NI_RSRC_PERMISSION_PREFIX}which ni_rsrc_mon &> /dev/null && \
         ${NI_RSRC_PERMISSION_PREFIX}timeout -s SIGKILL 2 ni_rsrc_mon -S 2> /dev/null | \
         awk 'match($0, "([0-9]+[ ]+){3}([0-9]+)[ ]+([0-9]+[ ]+){3}", m) { print m[2] }' | \
         grep -q [^0]; then
        # ERROR: found running dec/enc instances
        log "${cRed}Error${cRst}: Please stop all transcoding processes before running firmware upgrade" 6
        return 1
    fi
    return 0
}

# Select FW binary file to use; set FW_BIN. Select warm or cold upgrade
# depending on FL version on card vs on file; set UPGRADE_TYPE.
# return 0 if successful, 1 if failed
function determine_upgrade_type() {
    # Determine FW binary file to use
    if [ -n ${CUSTOM_PATH} ] && [ -f ${CUSTOM_PATH} ]; then
        # user specified FW binary path
        FW_BIN=$CUSTOM_PATH
    elif [ -n ${CUSTOM_PATH} ] && [ -d ${CUSTOM_PATH} ]; then
        # auto select FW binary in <quadra_auto_upgrade.sh path>/FL_BIN/
        # official releases must be named with format nor_flash_v*.*.*-Quadra*.bin for auto-discovery
        FW_BIN=`ls ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra*.bin 2> /dev/null | sort -V | tail -n 1 | xargs -n 1 realpath 2> /dev/null`
        # Check FW found
        if [ -z "$FW_BIN" ]; then
            log "${cRed}Error${cRst}: Cannot find a valid FW file at ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra*.bin" 6
            return 1
        fi
        FW_BIN_SIZE=`stat -c %b ${FW_BIN}`
        if [[ $FW_BIN_SIZE < 1048704 ]]; then  # 1048704 is offset to end of header of FW img in nor_flash.bin
            log "${cRed}Error${cRst}: Suspicious file size for FW file at ${FW_BIN}. File possibly corrupt" 6
            return 1
        fi
    else
        log "${cRed}Error${cRst}: Cannot find specified FW file or folder at ${CUSTOM_PATH}" 6
        return 1
    fi

    FW_BIN_FWREV=`dd if=${FW_BIN} ibs=1 skip=1048580 count=8 2>/dev/null` # 1048580 is offset for FW rev in header of FW img in nor_flash.bin

    # Determine upgrade type for each card
    # Store output in UPGRADE_TYPE as single letters for upgrade type to use:
    #     w-warm, c-cold, s-skip
    for DEVICE in ${DEVICES[@]}; do
        # default upgrade type is cold
        UPGRADE_TYPE[$DEVICE]='c'

        # compare card FL2 version with FW binary FL2 version
        CARD_FL_VERSION=`sudo nvme get-log ${DEVICE} -i 194 -l 9 2>/dev/null | grep -Poh '(?<= ").{5}'`
        BIN_FL_VERSION=`dd if=${FW_BIN} ibs=1 skip=131076 count=5 2>/dev/null`
        if [[ $CARD_FL_VERSION == $BIN_FL_VERSION ]]; then
            UPGRADE_TYPE[$DEVICE]='w'
        fi

        # if FL2 version is not of expected format. The input FW binary file may be corrupt
        if ! [[ $BIN_FL_VERSION =~ [0-9A-Z]\.[0-9A-Z]\.[0-9A-Z] ]]; then
            log "${cRed}Error${cRst}: Invalid firmware loader version format (${BIN_FL_VERSION}) for FW file at ${FW_BIN}. Possibly incorrect file format or corruption" 6
            return 1
        fi

        # compare FW rev on card and in binary
        if $FLAG_SKIP_SAME_VER_UPGRADES && [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]] && \
           [[ "${FW_REVISION[$DEVICE]}" == "${FW_BIN_FWREV}" ]]; then
            UPGRADE_TYPE[$DEVICE]='s'
        fi

        # prefer cold-upgrade to warm-upgrade but not skip-upgrade
        if $FLAG_COLD_UPGRADE && [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]]; then
            UPGRADE_TYPE[$DEVICE]='c'
        fi
    done
    return 0
}

# List information about cards detected and the FW they will be upgraded to
# return 0 if successful, 1 if failed
function list_quadra_devices() {
    if [ $FLAG_COLD_UPGRADE == true ]; then
        log "Prefer Cold Upgrade Selected" 1
    fi

    # State which binary is being used and the upgrade type
    log "Number of Quadra Transcoders found on $(hostname): ${#DEVICE[@]}" 1
    log "FW upgrade file: $FW_BIN" 1
    log "#   Device         Block              Serial Number        Card FW Rev File FW Rev Action" 1
    log "--- -------------- ------------------ -------------------- ----------- ----------- ------------" 1
    INDEX=0
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]]; then
            ACTION="warm upgrade"
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then
            ACTION="cold upgrade"
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
            ACTION="skip upgrade"
        fi
        log "$(printf "%-3s %-14s %-18s %-20s %-11s %-11s %-12s\n" "${INDEX}" "${DEVICE}" "${NODES[$DEVICE]}" "${SERIAL_NUMS[$DEVICE]}" "${FW_REVISION[$DEVICE]}" "${FW_BIN_FWREV}" "${ACTION}")" 1
        (( INDEX+=1 ))
    done

    return 0
}

# From a FW binary, generate a FW img without bootloader/firmwareloader for warm
# upgrade. The FW img will be output to /tmp/
# $1 - path to FW binary file
# return 0 if successful, 1 if failed
function create_fw_img_from_bin() {
    FW_IMG="/tmp/$(basename ${1} .bin).img"
    sudo dd if=${1} ibs=1 skip=1048576 of=${FW_IMG} 2> /dev/null  # 1048576 is offset for FW img in nor_flash.bin
    if [ $? -ne 0 ]; then
        log "${cRed}Error${cRst}: Failed to generate FW image from FW binary" 6
        return 1
    fi
    FW_IMG_FWREV=`dd if=${FW_BIN} ibs=1 skip=4 count=8 2>/dev/null` # 8 is offset for FW rev in header of FW img in fw.img
    return 0
}

# Reset device so that its read queue is cleared
# $1 - target nvme device path (eg. /dev/nvme0)
function reset_device() {
    log "${1} -- Initial reset of device." 8
    sudo nvme reset $1 &> /dev/null
    rc=$?
    log "${1} -- Return value: ${rc}" 8
    if [ $rc -ne 0 ]; then
        log "${1} -- Error: Initial reset failed!" 8
        return 1
    fi
    return 0
}

# Issue cold upgrade nvme commands to the selected card. Note, device still
# requires hard reset (eg. powercycle/hotplug) to activate new FW.
# $1 - target nvme device path (eg. /dev/nvme0)
# $2 - fw binary path (eg. FL_BIN/nor_flash.bin)
# return 0 if successful, non-0 if failed
function cold_upgrade() {
    DEVICE=$1
    BINARY=$2

    log "${DEVICE} -- Upgrading with $BINARY." 8
    # Download FW binary onto the card
    log "${DEVICE} -- Downloading firmware binary." 8
    OUTPUT=$(sudo nvme fw-download ${DEVICE} -f ${BINARY} -x 65536 2>&1)
    log "${DEVICE} -- Return value: $?" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if ! [ "$OUTPUT" == "Firmware download success" ]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Download failed!" 6
        log "${DEVICE} -- Error: Download failed!" 8
        return 1
    fi

    # Commit FW binary for activation
    log "${DEVICE} -- Committing firmware binary." 8
    OUTPUT=$(sudo nvme admin-passthru ${DEVICE} -o 0xc7 -t 180000 2>&1) # opcode 0xC7 is burn_nor
    rc=$?
    log "${DEVICE} -- Return value: ${rc}" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if [ $rc -eq 0 ] && [[ $OUTPUT =~ 00000000$ ]]; then
        log "${DEVICE} -- Upgrade complete!" 8
    else
        # log "${DEVICE} -- ${cRed}Error${cRst}: Commit failed!" 6
        log "${DEVICE} -- Error: Commit failed!" 8
        return 2
    fi

    log "${DEVICE} -- Finished FW cold upgrade download and commit." 8
    return 0
}

# Issue warm upgrade nvme commands to the selected card and attempt to activate
# the new FW through NVMe soft reset.
# $1 - target nvme device path (eg. /dev/nvme0)
# $2 - fw image path (eg. fw.img)
# return 0 if successful, non-0 if failed
function warm_upgrade() {
    DEVICE=$1
    BINARY=$2

    log "${DEVICE} -- Upgrading with $BINARY." 8

    # Download FW binary onto the card
    log "${DEVICE} -- Downloading firmware image." 8
    OUTPUT=$(sudo nvme fw-download ${DEVICE} -f ${BINARY} 2>&1)
    log "${DEVICE} -- Return value: $?" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if ! [ "$OUTPUT" == "Firmware download success" ]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Download failed!" 6
        log "${DEVICE} -- Error: Download failed!" 8
        return 1
    fi

    # Commit FW binary for activation
    log "${DEVICE} -- Activating firmware image." 8
    OUTPUT=$(sudo nvme admin-passthru ${DEVICE} -o 0x10 --cdw10=0xa -t 180000 2>&1) # opcode 0x10 is fw_commit, cdw10=0xa is action 1 slot 2
    log "${DEVICE} -- Return value: $?" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if [[ ! $OUTPUT =~ "the image specified does not support being activated without a reset" ]] && \
       [[ ! $OUTPUT =~ "FW_NEEDS_RESET" ]] || [[ ! $OUTPUT =~ [^0-9]"111"[^0-9] ]]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Commit failed!" 6
        log "${DEVICE} -- Error: Commit failed!" 8
        return 2
    fi

    # Reset card to activate new FW
    log "${DEVICE} -- Resetting device." 8
    OUTPUT=$(sudo nvme reset ${DEVICE} 2>&1)
    rc=$?
    log "${DEVICE} -- Return value: ${rc}" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if [ $rc -ne 0 ]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Reset failed!" 6
        log "${DEVICE} -- Error: Reset failed! Require reboot." 8
        return 3
    fi

    log "${DEVICE} -- Finished FW warm upgrade download, commit, and reset." 8
    return 0
}

# Check warm upgrade sucessfully activated new FW rev
# the new FW.
# $1 - target nvme device path (eg. /dev/nvme0)
# return 0 if successful, non-0 if failed
function warm_upgrade_check() {
    DEVICE=$1
    FW_REV=`sudo nvme id-ctrl ${DEVICE} | perl -nle'print $& while m{fr +: \K.*}g'`

    # Check that the current FW Rev on the card matches what is in image file used for upgrade
    if [[ $FW_BIN_FWREV != ${FW_REV} ]]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Warm upgrade FW not properly activated! May require reboot." 6
        log "${DEVICE} -- Error: Warm upgrade FW not properly activated! May require reboot." 8
        return 4
    fi
    return 0
}

# Re-init resources for warm upgraded cards
# return 0 if successful, non-0 if failed
function warm_upgrade_reinit() {
    rc=0
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]]; then
            ${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_update -d ${DEVICE} &> /dev/null
            (( rc|=$? ))
        fi
    done
    ${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_mon &> /dev/null
    (( rc|=$? ))

    if [ $rc -ne 0 ]; then
        return 1
    else
        return 0
    fi
}

# Parse $UPGRADE_LOG for upgrade results of each card to update $UPGRADE_RESULT. Can be called
# multiple times and before completion of upgrade process
# Note: log message formats referenced here have to match where they are written
function check_upgrade_results() {
    for DEVICE in ${DEVICES[@]}; do
        if grep -q "${DEVICE} -- Finished FW" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=0
        fi
        if grep -q "${DEVICE} -- Error: Initial reset failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=1
        elif grep -q "${DEVICE} -- Error: Download failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=2
        elif grep -q "${DEVICE} -- Error: Commit failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=3
        elif grep -q "${DEVICE} -- Error: Reset failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=4
        elif grep -q "${DEVICE} -- Error: Warm upgrade FW not properly activated" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=5
        fi
    done
}

# Convert $UPGRADE_RESULT integer to summary string
function upgrade_result_code_to_message() {
    case $1 in
        -1) UPGRADE_RESULT_STR="Skipped same ver upgrade"
        ;;
        0) if [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then UPGRADE_RESULT_STR="Successful cold upgrade";
           elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]]; then UPGRADE_RESULT_STR="Successful warm upgrade";
           else UPGRADE_RESULT_STR="Script error, unknown \$UPGRADE_TYPE"; fi
        ;;
        1) UPGRADE_RESULT_STR="Failed initial reset"
        ;;
        2) UPGRADE_RESULT_STR="Failed FW download"
        ;;
        3) UPGRADE_RESULT_STR="Failed FW commit"
        ;;
        4) UPGRADE_RESULT_STR="Failed FW activation"
        ;;
        5) UPGRADE_RESULT_STR="Failed activation check"
        ;;
        *) UPGRADE_RESULT_STR="Script error, unknown \$UPGRADE_RESULT"
        ;;
    esac
}

# List upgrade status after upgrade is complete and successful
function list_upgrade_status() {
    log "#   Device         Block              Serial Number        Card FW Rev Result" 1
    log "--- -------------- ------------------ -------------------- ----------- ------------------------" 1
    INDEX=0
    for DEVICE in ${DEVICES[@]}; do
        upgrade_result_code_to_message ${UPGRADE_RESULT[$DEVICE]}
        log "$(printf "%-3s %-14s %-18s %-20s %-11s %s\n" "${INDEX}" "${DEVICE}" "${NODES[$DEVICE]}" "${SERIAL_NUMS[$DEVICE]}" "${FW_REVISION[$DEVICE]}" "${UPGRADE_RESULT_STR}")" 1
        (( INDEX+=1 ))
    done
    return 0
}

# Perform upgrade with information from global arrays
# return 0 if successful, 1 if failed
function upgrade() {
    log "Starting to upgrade. This process may take 1 minute..." 1
    create_upgrade_log

    # Wait for FW reads caused by ni_rsrc_mon in error_checking() to complete lest they timeout
    PIDS=()
    for DEVICE in ${DEVICES[@]}; do
        UPGRADE_RESULT[${DEVICE}]=0
        reset_device $DEVICE &
    done
    for PID in ${PIDS[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results
    sleep 1

    # If a card requires warm upgrade, create a temporary FW img file
    if [[ "${UPGRADE_TYPE[*]}" =~ "w" ]]; then
        create_fw_img_from_bin $FW_BIN
        if [ $? -ne 0 ]; then return 1; fi
    fi
    for PID in ${PIDS[@]}; do
        wait $PID > /dev/null 2>&1
    done

    # Perform upgrades in parallel
    PIDS=()
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_RESULT[${DEVICE}]} != 0 ]]; then
            continue
        fi
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]]; then
            warm_upgrade ${DEVICE} ${FW_IMG} &
            PIDS+=($!)
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then
            cold_upgrade ${DEVICE} ${FW_BIN} &
            PIDS+=($!)
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
            UPGRADE_RESULT[${DEVICE}]=-1
        fi
    done
    for PID in ${PIDS[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results

    # Remove temporary FW img file
    if [ -f $FW_IMG ]; then
        sudo rm $FW_IMG 2> /dev/null
    fi

    # Check warm upgraded cards are sucessfully activated
    PIDS=()
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'w' ]] && [[ ${UPGRADE_RESULT[${DEVICE}]} == 0 ]]; then
            warm_upgrade_check ${DEVICE} &
            PIDS+=($!)
        fi
    done
    for PID in ${PIDS[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results

    # For reduced confusion, in results table show sucessfully upgraded cards 'Card FW Rev' as $FW_BIN_FWREV
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_RESULT[${DEVICE}]} == 0 && ( ${UPGRADE_TYPE[$DEVICE]} == 'c' || ${UPGRADE_TYPE[$DEVICE]} == 'w' ) ]]; then
            FW_REVISION[$DEVICE]=$FW_BIN_FWREV
        fi
    done

    # Upgrade completed
    if grep -q "Error" $UPGRADE_LOG; then
        mv $UPGRADE_LOG ./$(basename ${UPGRADE_LOG})
        if [ $? -ne 0 ]; then
            log "${cRed}Error${cRst}: Firmware upgrade failed! See errors in ${UPGRADE_LOG} and try again." 6
        else
            log "${cRed}Error${cRst}: Firmware upgrade failed! See errors in $(basename ${UPGRADE_LOG}) and try again." 6
        fi
        list_upgrade_status
        return 1
    else
        log "Firmware upgrade complete!" 1
        list_upgrade_status
        if [[ "${UPGRADE_TYPE[*]}" =~ "c" ]]; then
            log "Cards that underwent cold upgrade require system reboot to activate new FW." 1
        fi
        # Use ni_rsrc_update to remove warm upgraded cards and re-init resources
        if [[ "${UPGRADE_TYPE[*]}" =~ "w" ]] && $FLAG_WARM_UPGRADE_REINIT; then
            warm_upgrade_reinit
            if [ $? -eq 0 ]; then
                log "Cards that underwent warm upgrade ready for use." 1
            else
                log "${cYlw}Warning${cRst}: Cards that underwent warm upgrade have new FW activated but still require ni_rsrc initialization (${NI_RSRC_PERMISSION_PREFIX}init_rsrc)." 1
            fi
        elif [[ "${UPGRADE_TYPE[*]}" =~ "w" ]]; then
            log "Cards that underwent warm upgrade have new FW activated but still require ni_rsrc initialization (${NI_RSRC_PERMISSION_PREFIX}init_rsrc)." 1
        fi
        # rm $UPGRADE_LOG # No need to remove $UPGRADE_LOG if its only in /tmp/
        return 0
    fi
}

function exit_caused_by_script() {
    trap - EXIT
    log "quadra_auto_upgrade.sh finished" 1
    exit $1
}

function exit_caused_by_signal() {
    log "${cYlw}Warning${cRst}: quadra_auto_upgrade.sh received external signal to exit" 2
    exit_caused_by_script $1
}

####### MAIN #######
shopt -u nullglob
setup_terminal_colors           # Setup variables for terminal color printouts
parse_command_line_args "$@"    # Parse user args

log "Welcome to the Quadra Transcoder Firmware Upgrade Utility!" 1
trap exit_caused_by_signal EXIT

# Check for conditions to run without user intervention
if $FLAG_ASSUME_YES; then
    sudo_check
    if [ $? -ne 0 ]; then
        log "${cRed}Error${cRst}: need to configure passwordless use of sudo before using -y|--yes" 2
        exit_caused_by_script 1
    fi
fi

# Set global variable $NI_RSRC_PERMISSION_PREFIX
get_ni_rsrc_permission_prefix

# check nvme-cli installed on system
check_nvme_cli
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Initialize and sort arrays with card attributes: $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS
get_cards_info
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Check for simple errors in environment
error_checking
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# For each Quadra card, select FW file to upgrade with
determine_upgrade_type
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# List Quadra devices and corresponding serial numbers and auto-select binary files
list_quadra_devices
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Check if cards have the same FW Rev as the target binary
INDEX=0
IDX_CARDS_SAME_FW_AS_TARGET=()
for DEVICE in ${DEVICES[@]}; do
    if [[ $FW_BIN_FWREV == ${FW_REVISION[$DEVICE]} ]]; then
        IDX_CARDS_SAME_FW_AS_TARGET+=($INDEX)
    fi
    (( INDEX+=1 ))
done
if [[ ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} == ${#DEVICES[@]} ]]; then
    log "${cYlw}Warning${cRst}: All cards have the same FW as the target FW upgrade file" 1
    if $FLAG_SKIP_SAME_VER_UPGRADES; then
        exit_caused_by_script 0
    fi
elif [[ ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} > 0 ]]; then
    log "${cYlw}Warning${cRst}: ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} cards have the same FW as the target FW upgrade file" 1
    log "Indices of devices with same FW:" $( IFS="," ; echo "${!IDX_CARDS_SAME_FW_AS_TARGET[*]}" ) 1
fi

# Prompt for user to begin upgrade
prompt_yn "Do you want to upgrade ALL of the above Quadra Transcoder(s)?"
if [ $? -eq 0 ]; then
    log "Upgrade cancelled" 1
    exit_caused_by_script 0;
fi

# Perform upgrade
log "${cYlw}Warning${cRst}: User must not interrupt the upgrade process, doing so may lead to device malfunction!" 1
upgrade;
if [ $? -eq 0 ]; then
    exit_caused_by_script 0;    # Success
else
    exit_caused_by_script 1;    # Fail
fi
