#!/usr/bin/env bash

# HELP FUNCTION
# Prints up the usage information for the shell script
function usage() {
    echo "-------- t4xx_auto_upgrade.sh --------"
    echo "Perform firmware upgrade to T4XX cards."
    echo "This script will automatically detect T4XX cards and select upgrade method."
    echo ""
    echo "Usage: ./t4xx_auto_upgrade.sh [OPTION]"
    echo "Example: ./t4xx_auto_upgrade.sh -p FL_BIN/nor_flash_v2.0.0.bin"
    echo
    echo "Options:"
    echo "-h, --help                display this help and exit"
    echo "-d, --default             run the default configuration to install FL_BIN/nor_flash.bin"
    echo "-p, --path                specify an input firmware binary file path"
    echo "                          (ex. -p FL_BIN/custom_nor_flash.bin)"
    echo "-c, --cold                force cold upgrade"
    echo "-w, --warm                force warm upgrade"
    echo "-y, --yes                 automatic yes to prompts"
    echo "--prompt_at_card          require a prompt before loading each individual card"
    echo "                          (can be used to upgrade specific cards)"
    echo "--skip_same_ver_upgrades  skip upgrades on cards with same FW version as FW file"

    return 0
}

# Function that updates the arrays with latest information of the cards/devices
function get_card_info() {
    # Updates NODES, SERIAL_NUMS, FW_REVISION, MODEL_NUMS arrays with latest information of card/device
    # NODES: /dev/nvme*n1
    # SERIAL_NUM: Card S/N
    # FW_REVISION: FW Rev active on the card (can be checked with sudo nvme list)
    # MODEL_NUMS: Card Model number
    # These arrays are used in the other functions and in the script
    # The updated arrays are indexed by card/device index.
    # i.e. /dev/nvme0n1 -> Card/device 0

    # Refreshes/unset the array variables to grab new information
    UNSORTED_NODES=()
    UNSORTED_SERIAL_NUMS=()
    UNSORTED_MODEL_NUMS=()
    UNSORTED_FW_REVISION=()

    # Index sequence for sorting
    INDEX_SEQUENCE=()

    # Arrays that will be updated and used
    NODES=()
    SERIAL_NUMS=()
    MODEL_NUMS=()
    FW_REVISION=()

    # Get ths unsorted information of the cards/devices from sudo nvme list
    XCOD_DATA=(`sudo nvme list | sed -e '1,2d' | grep -P "T4(08|32)-" | sed 's/\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\).*\(........\)/\1 \2 \3 \4/'`)
    for (( i=0; i<${#XCOD_DATA[@]}; i+=4 )); do
        UNSORTED_NODES+=(${XCOD_DATA[$i]})
        UNSORTED_SERIAL_NUMS+=(${XCOD_DATA[$i+1]})
        UNSORTED_MODEL_NUMS+=(${XCOD_DATA[$i+2]})
        UNSORTED_FW_REVISION+=(${XCOD_DATA[$i+3]})
    done

    # Set up indices for sorting
    INDICES=(${UNSORTED_NODES[@]#/dev/nvme})
    INDICES=(${INDICES[@]%n1})

    MAX_INDEX=${INDICES[0]}
    for INDEX in ${INDICES[@]}; do
        if (( $INDEX > $MAX_INDEX )); then
        MAX_INDEX=$INDEX
        fi
    done

    for (( i=0; i<=$MAX_INDEX; i++ )); do
        INDEX_SEQUENCE=(${INDEX_SEQUENCE[@]} ${i})
    done

    # Sorts and updates NODES, SERIAL_NUMS, FW_REVISION, MODEL_NUMS
    for INDEX in ${INDEX_SEQUENCE[@]}; do
        for DEVICE in ${!UNSORTED_NODES[@]}; do
            if [[ ${UNSORTED_NODES[$DEVICE]} == /dev/nvme${INDEX}n1 ]]; then
                NODES+=(${UNSORTED_NODES[$DEVICE]})
                SERIAL_NUMS+=(${UNSORTED_SERIAL_NUMS[$DEVICE]})
                MODEL_NUMS+=(${UNSORTED_MODEL_NUMS[$DEVICE]})
                FW_REVISION+=(${UNSORTED_FW_REVISION[$DEVICE]})
            fi
        done
    done

    return 0
}

# COLD UPGRADE FUNCTION
function cold_upgrade() {
    # INPUTS: $1(T4XX_NODE) , $2(BINARY)
    # T4XX_NODE: /dev/nvme*n1
    # BINARY: nor_flash*.bin

    # Issues cold upgrade nvme commands to the cards/devices in order
    # Writes the steps and output status code for each step/command
    # for the cold upgrade into the UPGRADE_LOG for parsing/logging

    # RETURN:  1 if error occurs else 0
    # NOTE: It does not issue the reboot command, that has to be done post execution
    T4XX_NODE=$1
    BINARY=$2

    echo [$(date +%T)] "${T4XX_NODE} -- Upgrading with $BINARY." >> $UPGRADE_LOG
    # Download FW binary onto the card
    echo [$(date +%T)] "${T4XX_NODE} -- Downloading firmware binaries." >> $UPGRADE_LOG
    OUTPUT=$(sudo nvme fw-download ${T4XX_NODE} -f ${BINARY} -x 65536 2>&1)
    rc=$?
    if ! [ "$OUTPUT" == "Firmware download success" ]; then
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- Download failed!" >> $UPGRADE_LOG
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- rc=$rc rs=${OUTPUT}" >> $UPGRADE_LOG
        return 1
    fi
    # Activate the FW binary on the card
    echo [$(date +%T)] "${T4XX_NODE} -- Committing firmware binaries." >> $UPGRADE_LOG
    OUTPUT=$(sudo nvme admin-passthru ${T4XX_NODE} -o 0xc7 -t 120000 2>&1) # opcode 0xC7 is burn_nor
    if [ "$OUTPUT" == "NVMe command result:00000000" ]; then
        echo [$(date +%T)] "${T4XX_NODE} -- Upgrade completed!" >> $UPGRADE_LOG
    else
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- Commit failed!" >> $UPGRADE_LOG
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- rc=$rc rs=${OUTPUT}" >> $UPGRADE_LOG
        return 1
    fi

    return 0
}

# WARM UPGRADE FUNCTION
function warm_upgrade() {
    # Inputs: $1(T4XX_NODE) , $2(BINARY) , $3(DEV_IDX)
    # T4XX_NODE: /dev/nvme*n1
    # BINARY: nor_flash*.bin
    # DEV_IDX: index of the card being upgraded/downgraded

    # Issues warm upgrade nvme commands to the cards/devices in order
    # Writes the steps and output status code for each step/command for
    # the warm upgrade into the UPGRADE_LOG for parsing/logging

    # RETURN: 1 if error occurs else 0
    T4XX_NODE=$1
    BINARY=$2
    DEV_IDX=$3
    
    echo [$(date +%T)] "${T4XX_NODE} -- Upgrading with $BINARY." >> $UPGRADE_LOG

    # Download FW binary onto the card
    echo [$(date +%T)] "${T4XX_NODE} -- Downloading firmware image." >> $UPGRADE_LOG
    OUTPUT=$(sudo nvme fw-download ${T4XX_NODE} -f ${BINARY} -x 8192 -o 0 2>&1)
    rc=$?
    if ! [ "$OUTPUT" == "Firmware download success" ]; then
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- Download failed!" >> $UPGRADE_LOG
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- rc=$rc rs=${OUTPUT}" >> $UPGRADE_LOG
        return 1
    fi
    # Activate the FW binary on the card
    echo [$(date +%T)] "${T4XX_NODE} -- Activating firmware image." >> $UPGRADE_LOG
    OUTPUT=$(sudo nvme admin-passthru ${T4XX_NODE} -o 0x10 --cdw10=0xa -t 120000 2>&1) # opcode 0x10 is fw_commit, cdw10=0xa is action 1 slot 2
    rc=$?
    if [[ $OUTPUT =~ FW_NEEDS_RESET ]] && [[ $OUTPUT =~ 111 ]]; then
        # Get NVMe device path from namespace
        # linux <5.4
        T4XX_DEVICE=$(udevadm info -q path -n ${T4XX_NODE} | grep -Poh "nvme\d{1,5}\b")
        if [[ -z $T4XX_DEVICE ]]; then
            # linux >=5.4
            T4XX_DEVICE=$(ls /sys/`udevadm info -q path -n ${T4XX_NODE}`/device | grep -m 1 -Poh "nvme\d{1,5}\b")
        fi
        if [[ -z $T4XX_DEVICE ]]; then
            echo "NVMe device path not found! Will reset default device ${T4XX_NODE%n*}"
            T4XX_DEVICE=${T4XX_NODE%n*}
        else
            T4XX_DEVICE="/dev/${T4XX_DEVICE%/*}"
        fi
        echo [$(date +%T)] "${T4XX_NODE} -- Resetting device" >> $UPGRADE_LOG
        `sudo nvme reset ${T4XX_DEVICE} 2>&1`
    else
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- Activation failed!" >> $UPGRADE_LOG
        echo [$(date +%T)] "Error: ${T4XX_NODE} -- rc=$rc rs=${OUTPUT}" >> $UPGRADE_LOG
        return 1
    fi

    return 0
}

####### MAIN #######

# Default values
shopt -u nullglob
# Assume Yes to prompt flag
flag_assume_yes=false
flag_default=false

# Upgrade type flags
flag_default=false
flag_warm=false
flag_cold=false

# Error occured flag
flag_error=false

# Reboot prompt flag
flag_reboot=false

# Prompt to upgrade cards individually flag
flag_prompt_per_card=false

# Skip upgrade for same FW Rev (current/target) flag
flag_skip_same_upgrades=false

declare -A NEW_FW_REVISION=()

# Initialize upgrade script log file
UPGRADE_LOG="upgrade_log.txt"
if ! [ -f $UPGRADE_LOG ]; then
    touch $UPGRADE_LOG
else
    echo -n "" > $UPGRADE_LOG
fi
exec &> >(tee -a $UPGRADE_LOG)
sleep 0.001 # Workaround for log file pipe hang

# Command line arguments
while [ "$1" != "" ]; do
    case $1 in
        -h | --help)    usage; exit 0
        ;;
        -d | --default) flag_default=true
        ;;
        -p | --path)    CUSTOM_IMG=${2}; shift
        ;;
        -c | --cold)    flag_cold=true
        ;;
        -w | --warm)    flag_warm=true
        ;;
        -y | --yes)          flag_assume_yes=true
        ;;
        --prompt_at_card)    flag_prompt_per_card=true
        ;;
        --skip_same_ver_upgrades)     flag_skip_same_upgrades=true
        ;;
        *)              echo "Usage: ./t4xx_auto_upgrade.sh [OPTION]..."; echo "Try './t4xx_auto_upgrade.sh --help' for more information"; exit 1
        ;;
    esac
    shift
done

# Custom firmware image is set to the default firmware image
if $flag_default; then
    if $flag_warm; then
        CUSTOM_IMG="FL_BIN/fw.img"
    else
        CUSTOM_IMG="FL_BIN/nor_flash.bin"
    fi
fi

# Warm and cold upgrade forcing is mutually exclusive
if $flag_cold && $flag_warm; then
    echo "Error - warm and cold upgrade forcing is mutually exclusive."
    exit 1;
fi

# Initialize and sorts card attributes: NODES, SERIAL_NUMS, FW_REVISION, MODEL_NUMS
# NODES, SERIAL_NUMS, FW_REVISION, MODEL_NUMS are arrays which store the information of each card in their respective indexes
get_card_info
WARM_UPGRADED_DEV_IDX=()
XCOD_NODES=(${NODES[@]})
XCOD_SERIAL_NUMS=(${SERIAL_NUMS[@]})
XCOD_MODEL_NUMS=(${MODEL_NUMS[@]})

# Copyright message
echo "Copyright (C) 2020 NETINT Technologies. All rights reserved."
echo

# Welcome message
echo "Welcome to the Codensity Transcoder Firmware Upgrade Utility!"
echo

# Error checking
if (! [[ -x "$(command -v nvme)" ]]) && (! [[ -x "$(sudo which nvme)" ]]); then
    # ERROR: nvme command
    echo "Try: command -v nvme"
    echo "Error: NVMe-CLI is not installed. Please install it and try again!" >&2
    exit 1
elif [[ ${#XCOD_NODES[@]} == 0 ]]; then
    # ERROR: no devices found
    echo "Try: sudo nvme list"
    echo "Error: No Codensity Transcoder device found! Exiting." >&2
    exit 1
elif [[ ${#XCOD_SERIAL_NUMS[@]} != ${#XCOD_NODES[@]} ]]; then
    # ERROR: serial number cannot be found
    echo "Try: sudo nvme list"
    echo "Error: Invalid serial number! Exiting." >&2
    exit 1
elif (! [[ -z ${CUSTOM_IMG} ]]) && (! [[ -f ${CUSTOM_IMG} ]]); then
    # ERROR: invalid path-specified binary
    echo "Error: Cannot find path-specified binary ${CUSTOM_IMG}! Exiting." >&2
    exit 1
fi

# For each T4XX card, select FW file to upgrade with
for DEV_IDX in ${!XCOD_SERIAL_NUMS[@]}; do
    # Determine model of card
    if [[ ${XCOD_MODEL_NUMS[$DEV_IDX]} =~ "T408-" ]]; then
        model_type_bin_suffix="-T408"
    elif [[ ${XCOD_MODEL_NUMS[$DEV_IDX]} =~ "T432-" ]]; then
        model_type_bin_suffix="-T432"
    else
        model_type_bin_suffix=""
    fi

    # Select FW file to use for upgrade method
    # official releases must be named with format nor_flash_v*.*.*-T4*.bin for auto-discovery
    if [[ -n ${CUSTOM_IMG} ]]; then
        XCOD_BINS[${DEV_IDX}]=$CUSTOM_IMG
    elif $flag_warm; then
        if ls FL_BIN/fw_v*.*.*$model_type_bin_suffix*.img 1> /dev/null 2>&1; then
            XCOD_BINS[${DEV_IDX}]=`ls FL_BIN/fw_v*.*.*${model_type_bin_suffix}*.img | sort -V | tail -n 1`
        else
            echo "No valid release T4XX FW .img found in FL_BIN/"
        fi
        continue
    elif $flag_cold; then
        if ls FL_BIN/nor_flash_v*.*.*$model_type_bin_suffix.bin 1> /dev/null 2>&1; then
            XCOD_BINS[${DEV_IDX}]=`ls FL_BIN/nor_flash_v*.*.*${model_type_bin_suffix}*.bin | sort -V | tail -n 1`
        else
            echo "No valid release T4XX FW .bin found in FL_BIN/"
        fi
        continue
    else
        # Auto-select warm/cold upgrade
        # If can evaluate FL versions and they match, then warm upgrade; else, cold upgrade is default
        if ls FL_BIN/fw_v*.*.*$model_type_bin_suffix*.img 1> /dev/null 2>&1 && \
           ls FL_BIN/nor_flash_v*.*.*$model_type_bin_suffix.bin 1> /dev/null 2>&1; then
            CURRENT_FL_VERSION=`sudo nvme get-log ${XCOD_NODES[$DEV_IDX]} -i 194 -l 9 2>/dev/null | grep -Poh '(?<= ").{5}'`
            bin_path=`ls FL_BIN/nor_flash_v*.*.*${model_type_bin_suffix}*.bin | sort -V | tail -n 1`
            NEW_FL_VERSION=`dd if=${bin_path} ibs=1 skip=65540 count=5 2>/dev/null`
            if [[ $CURRENT_FL_VERSION == $NEW_FL_VERSION ]]; then
                XCOD_BINS[${DEV_IDX}]=`ls FL_BIN/fw_v*.*.*${model_type_bin_suffix}*.img | sort -V | tail -n 1`
                continue
            fi
        fi
        if ls FL_BIN/nor_flash_v*.*.*$model_type_bin_suffix.bin 1> /dev/null 2>&1; then
            XCOD_BINS[${DEV_IDX}]=`ls FL_BIN/nor_flash_v*.*.*${model_type_bin_suffix}*.bin | sort -V | tail -n 1`
        else
            echo "No valid release T4XX FW .img/.bin found in FL_BIN/ for card ${DEV_IDX}"
        fi
        continue
    fi
done

if [ -z $XCOD_BINS ]; then
    echo "no valid binaries found for any detected cards, exiting..."
    exit 1
fi

# Array to store devices with same FW REV as the target binary
declare -A SAME_REV=()
# List Codensity devices and corresponding serial numbers and auto-select binary files
echo "Number of Codensity Transcoders found on $(hostname): ${#XCOD_NODES[@]}"
echo "Index Device           SN                   Binary File                      Card FW Rev File FW Rev"
echo "----- ---------------- -------------------- -------------------------------- ----------- -----------"
for DEV_IDX in ${!XCOD_SERIAL_NUMS[@]}; do
    if [[ ${XCOD_BINS[$DEV_IDX]} =~ .*.img ]]; then
        NEW_FW_REVISION[$DEV_IDX]=`dd if=${XCOD_BINS[$DEV_IDX]} ibs=1 skip=0 count=8 2>/dev/null`
    else
        NEW_FW_REVISION[$DEV_IDX]=`dd if=${XCOD_BINS[$DEV_IDX]} ibs=1 skip=524288 count=8 2>/dev/null`
    fi
    # Check and store (mark) devices with the same FW Rev as the target binary
    if [[ ${FW_REVISION[$DEV_IDX]} =~  ${NEW_FW_REVISION[$DEV_IDX]} ]]; then
        SAME_REV[$DEV_IDX]=${XCOD_SERIAL_NUMS[$DEV_IDX]}
    fi
    printf "%-5s %-16s %-20s %-32s %-11s %-12s\n" ${DEV_IDX} ${XCOD_NODES[$DEV_IDX]} ${XCOD_SERIAL_NUMS[$DEV_IDX]} ${XCOD_BINS[$DEV_IDX]} ${FW_REVISION[$DEV_IDX]} ${NEW_FW_REVISION[$DEV_IDX]}

    BIN_SIZE=`stat -c %b ${XCOD_BINS[$DEV_IDX]}`
    if [[ $BIN_SIZE == 0 ]]; then
        echo
        echo "Error: Binary file is empty!. Exiting." >&2
        exit 1
    fi
done
echo ""
 
echo "WARNING!"
echo "User must not interrupt the upgrade process, doing so may lead to device malfunction!"
echo

# Check if all cards have the same FW Rev as the target binary
if [[ ${#XCOD_SERIAL_NUMS[@]} -eq ${#SAME_REV[@]} ]]; then
    echo 'All cards have the same FW as the designated binary file to upgrade to!'
    if $flag_skip_same_upgrades; then
        echo 'Skipping all upgrades!'; exit 0
    fi
# Else if only some cards have the same FW REV
elif [[ ${#SAME_REV[@]} > 0 ]]; then
    echo 'Some cards have the same FW as the designated binary file to upgrade to!'
    printf 'Index of devices with same FW: [%s] \n' $( IFS=", " ; echo "${!SAME_REV[*]}" )
fi

# Prompt for user to begin upgrade
if $flag_skip_same_upgrades; then
    echo 'Do you want to upgrade the above Codensity Transcoder(s) with different FW Rev?'
else
    echo 'Do you want to upgrade ALL of the above Codensity Transcoder(s)? (skip same FW Rev upgrades with --skip_same_ver_upgrades flag)'
fi
if $flag_assume_yes; then
    echo 'Press [Y/y] to begin. y'
    REPLY=y
else
    read -p "Press [Y/y] to begin. " -n 1 -r
    printf '\n'
fi

# Array of devices (by index) to upgrade and those to skip
UPGRADE_IDX=()
if [[ $REPLY =~ ^[Yy]$ ]]; then
    # Check for cards to upgrades and put them into an array (includes prompt check)
    for DEV_IDX in ${!XCOD_NODES[@]}; do
        # If skip upgrade flag is selected
        if [ "${SAME_REV[$DEV_IDX]+abc}" ] && $flag_skip_same_upgrades; then
            echo "Skipping upgrade on device - $DEV_IDX, S/N - ${SAME_REV[$DEV_IDX]}"
            continue
        fi
        if $flag_prompt_per_card && $flag_assume_yes; then
            echo "Do you wish to upgrade ${XCOD_NODES[$DEV_IDX]}?. Press [Y/y] to begin. y"
            REPLY=y
        elif $flag_prompt_per_card && ! $flag_assume_yes; then
            read -p "Do you wish to upgrade ${XCOD_NODES[$DEV_IDX]}?. Press [Y/y] to begin. " -n 1 -r 
            echo
        else
            REPLY=y
        fi
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            UPGRADE_IDX+=($DEV_IDX)
        fi
    done
    # Start the upgrade process
    echo
    echo "Starting to upgrade. This process may take 1 minute."
    for DEV_IDX in ${UPGRADE_IDX[@]}; do
        # Perform upgrade
        # Warm upgrade if binary is fw.img
        if [[ ${XCOD_BINS[$DEV_IDX]} =~ .*.img ]]; then
            warm_upgrade ${XCOD_NODES[$DEV_IDX]} ${XCOD_BINS[$DEV_IDX]} ${DEV_IDX} &
            PIDS[${DEV_IDX}]=$!
            WARM_UPGRADED_DEV_IDX+=($DEV_IDX)
        # Cold upgrade if binary is nor_flash.bin
        elif [[ ${XCOD_BINS[$DEV_IDX]} =~ .*.bin ]]; then
            cold_upgrade ${XCOD_NODES[$DEV_IDX]} ${XCOD_BINS[$DEV_IDX]} &
            PIDS[${DEV_IDX}]=$!
            flag_reboot=true
        else
            echo "Invalid binary file used for T4xx upgrade. Please use valid binary files."; exit 1
        fi
    done

    # Wait for all background processes to finish
    for PID in ${PIDS[@]}; do
        wait $PID >/dev/null 2>&1
    done
    
    if ((${#WARM_UPGRADED_DEV_IDX[@]})); then
        get_card_info
        for DEV_IDX in ${WARM_UPGRADED_DEV_IDX[@]}; do
          # Checks that the current FW Rev on the card/device matches what is in the binary it upgraded/downgrade to
          if [[ ${FW_REVISION[$DEV_IDX]} =~ ${NEW_FW_REVISION[$DEV_IDX]} ]]; then # Note: this check does not work for upgrade to same version number
              echo [$(date +%T)] "${XCOD_NODES[$DEV_IDX]} -- Upgrade completed!" >> $UPGRADE_LOG
          else
              echo [$(date +%T)] "Error: ${XCOD_NODES[$DEV_IDX]} -- Warm upgrade failed! New FW revision not activated. May require reboot." >> $UPGRADE_LOG
          fi
        done
    fi

    echo
    echo

    # Error: NVME command failure
    if grep -q "Error" upgrade_log.txt; then
        echo 'Error: Firmware Upgrade Failed! See error(s) in upgrade_log.txt and try again.' >&2
        exit 1
    fi

    # Upgrade completed
    if $flag_error; then
        echo "Firmware Upgrade Failed! One or more devices were not upgraded properly. Please address "
        echo "errors and try again." >&2
    elif $flag_reboot; then
        echo "Firmware Upgrade Completed! You may now reboot the system to activate the firmware."
    else
        echo "Firmware Upgrade Completed! Firmware has been activated."
        echo "Libxcoder connection to card must be reinitializaed."
        echo "Please run the following command when T4XX cards on system are not in use for dec/enc:"
        echo "sudo rm -f /dev/shm/NI_*; init_rsrc_logan"
    fi
else
    # Upgrade cancelled
    echo
    echo
    echo "Update Cancelled. Exiting."
fi
