#!/bin/bash
# 本脚本将基于kbox基础云手机镜像和目标GPU的二进制驱动合并制作云手机镜像，方便用户使用
# 操作命令
# ./make_images.sh [gpu驱动包路径及名称] [kbox基础云手机镜像名称] [目标GPU Kbox云手机镜像名称]
# 参数可选，默认为kbox基础云手机镜像名称：kbox:latest，目标GPU Kbox云手机镜像名称：kbox:latest_{GPU_NAME}
# 例：
# ./make_image.sh xxxx.tar.gz
# ./make_image.sh xxxx.tar.gz kbox:latest
# ./make_image.sh xxxx.tar.gz kbox:latest kbox:latest_

set -e

CUR_PATH=$(cd $(dirname "${0}");pwd)

# kbox基础云手机和视频流云手机默认镜像名称，可根据需要自行修改
DOCKER_IMAGE_IN=kbox:latest
DOCKER_IMAGE_OUT=kbox:latest_

GPUTYPE=""
TMP_PACKAGE_DIR=${CUR_PATH}/GpuDriver
DOCKER_FILE_DIR=${CUR_PATH}/Dockerfiles
DOCKER_FILE_BAK=${CUR_PATH}/Dockerfile.bak
DOCKER_FILE=${CUR_PATH}/Dockerfile

function gen_docker_file()
{
    local target_docker_file="${DOCKER_FILE_DIR}/Dockerfile_"
    if [ $1 = "awm" ]; then
        target_docker_file="${target_docker_file}awm"
        DOCKER_IMAGE_OUT="${DOCKER_IMAGE_OUT}awm"
    elif [ $1 = "vastai" ]; then
        target_docker_file="${target_docker_file}vastai"
        DOCKER_IMAGE_OUT="${DOCKER_IMAGE_OUT}vastai"
    fi

    # 判断Dockerfile是否存在
    if [ ! -f "${target_docker_file}" ]; then
        echo -e "\033[1;31m[ERROR] ${target_docker_file} is not exist. \033[0m" && return -1
    fi

    if [ -f "${DOCKER_FILE}" ]; then
        mv ${DOCKER_FILE} ${DOCKER_FILE_BAK}
    fi

    cp ${target_docker_file} ${DOCKER_FILE}
}

function gen_vastaigpu_package()
{
    mkdir -p ${TMP_PACKAGE_DIR}/system
    cp -rp ${TMP_PACKAGE_DIR}/vastai_driver/umd/vendor/ ${TMP_PACKAGE_DIR}/system/
}

function gen_awmgpu_package()
{
    mkdir -p ${TMP_PACKAGE_DIR}/system/etc/firmware/awmgpu/
    mkdir -p ${TMP_PACKAGE_DIR}/system/lib/
    mkdir -p ${TMP_PACKAGE_DIR}/system/lib64/
    mkdir -p ${TMP_PACKAGE_DIR}/system/vendor/lib/hw/
    mkdir -p ${TMP_PACKAGE_DIR}/system/vendor/lib64/hw/
    mkdir -p ${TMP_PACKAGE_DIR}/system/vendor/lib/egl/
    mkdir -p ${TMP_PACKAGE_DIR}/system/vendor/lib64/egl/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/powervr.ini ${TMP_PACKAGE_DIR}/system/etc/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/codec_api/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/codec_api/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/codec_api/firmware/release/* ${TMP_PACKAGE_DIR}/system/etc/firmware/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm/libEGL_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm64/libEGL_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm/libIMGegl.so ${TMP_PACKAGE_DIR}/system/vendor/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm64/libIMGegl.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm/libpvrANDROID_WSEGL.so ${TMP_PACKAGE_DIR}/system/vendor/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/egl/arm64/libpvrANDROID_WSEGL.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm/libGLESv1_CM_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm64/libGLESv1_CM_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm/libGLESv2_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm64/libGLESv2_powervr.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/egl/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm/libsrv_um.so ${TMP_PACKAGE_DIR}/system/vendor/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm64/libsrv_um.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm/libufwriter.so ${TMP_PACKAGE_DIR}/system/vendor/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm64/libufwriter.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm/libusc.so ${TMP_PACKAGE_DIR}/system/vendor/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/arm64/libusc.so ${TMP_PACKAGE_DIR}/system/vendor/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/opengles/firmware/* ${TMP_PACKAGE_DIR}/system/etc/firmware/awmgpu/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/gralloc/arm/* ${TMP_PACKAGE_DIR}/system/vendor/lib/hw/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/gralloc/arm64/* ${TMP_PACKAGE_DIR}/system/vendor/lib64/hw/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/ifbc/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/ifbc/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/isr/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/isr/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/isr/wbc/ ${TMP_PACKAGE_DIR}/system/etc/firmware/awmgpu/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/libva/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/libva/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/va-api/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/va-api/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/openmax/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/openmax/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/vulkan/arm/* ${TMP_PACKAGE_DIR}/system/vendor/lib/hw/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/vulkan/arm64/* ${TMP_PACKAGE_DIR}/system/vendor/lib64/hw/

    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/c-utils/arm/* ${TMP_PACKAGE_DIR}/system/lib/
    cp -pr ${TMP_PACKAGE_DIR}/awmgpu/c-utils/arm64/* ${TMP_PACKAGE_DIR}/system/lib64/
}

function gen_gpu_driver()
{
    # 判断驱动包目录是否存在，若存在则删除重建
    if [ -d "${TMP_PACKAGE_DIR}" ]; then
        rm -rf ${TMP_PACKAGE_DIR}
    fi
    mkdir -p ${TMP_PACKAGE_DIR}
    # 解压GPU驱动包（注意：通过--no-same-owner保证解压后的文件属组为当前操作用户（root）文件属组）
    cmd="tar --no-same-owner -zxvf $1 -C ${TMP_PACKAGE_DIR}"
    echo -e "\033[1;36m[INFO] "$cmd" \033[0m"
    $cmd > /dev/null

    if [ $GPUTYPE = "awm" ]; then
        gen_awmgpu_package
    elif [ $GPUTYPE = "vastai" ]; then
        gen_vastaigpu_package
    fi

    gen_docker_file "$GPUTYPE"
}

# 前置条件验证和准备
function prepare_kbox()
{
    if [[ $1 == awmgpu* ]]; then
        GPUTYPE="awm"
    elif [[ $1 == vastai* ]]; then
        GPUTYPE="vastai"
    else
        echo -e "\033[1;31m[ERROR] unsupport package type. \033[0m" && exit -1
    fi

    # 判断Dockerfile是否存在
    if [ ! -d "${DOCKER_FILE_DIR}" ]; then
        echo -e "\033[1;31m[ERROR] ${DOCKER_FILE_DIR} is not exist. \033[0m" && return -1
    fi

    # 判断GPU驱动包是否存在
    if [ ! -f "$1" ]; then
        echo -e "\033[1;31m[ERROR] driver package $1 is not exist. \033[0m" && return -1
    fi

    gen_gpu_driver $1
}

# 制作镜像
function make_image()
{
    cmd="docker build --build-arg KBOX_IMAGE=${DOCKER_IMAGE_IN} -t ${DOCKER_IMAGE_OUT} ."
    echo -e "\033[1;36m[INFO] "$cmd" \033[0m"
    $cmd
    wait
}

function main()
{
    echo -e "\033[1;36m[INFO] Begin to make image \033[0m"
    # 构建镜像
    local start_time=$(date +%s)
    local end_time=0

    prepare_kbox "$@"

    if [ -n "$2" ]; then
        DOCKER_IMAGE_IN="$2"
    fi
    if [ -n "$3" ]; then
        DOCKER_IMAGE_OUT="$3"
    fi

    make_image "$@"

    [ ${?} != 0 ] && echo -e "\033[1;31m[ERROR] Failed to make image. \033[0m" && exit -1

    # 制作镜像时间统计
    end_time=$(date +%s)
    time_interval=$((${end_time}-${start_time}))
    echo -e "\033[1;36m[INFO] End to make image from ${DOCKER_IMAGE_IN} to ${DOCKER_IMAGE_OUT} \033[0m"
    echo -e "\033[1;36m[INFO] Make image takes ${time_interval} seconds. \033[0m"
}

main "$@"
