#!/bin/bash

# ******************************************************************************** #
# Copyright Kbox Technologies Co., Ltd. 2023-2023. All rights reserved.
# File Name: kbox11_deploy.sh
# Description: ubuntu kbox11环境部署脚本。
# Usage: 
# 1.请确认脚本位于：Kbox-AOSP11/deploy_scripts/ubuntu_android11_deploy/
# 2.在运行脚本前，请提前上传以下文件到同一目录下：
#   [1] android.tar
#   [2] NETINT.tar.gz [硬解模式需上传]
#   若使用本地方式获取源码包，则以下文件也需要上传：(自动下载模式则无需上传)
#   [3] Exagear-ARM32-ARM64.tar.gz
# 3.脚本运行格式：
#   ./kbox11_deploy.sh ${packages_dir} 
#   ${packages_dir}需替换成文件上传目录的路径
#   示例： ./kbox11_deploy.sh /home/kbox_packages/
# ******************************************************************************** #

#set -x
# 脚本解释器 强制设置为 bash
if [ "$BASH" != "/bin/bash" ] && [ "$BASH" != "/usr/bin/bash" ]
then
   bash "$0" "$@"
   exit $?
fi

function error(){
    echo -e "\033[1;31m$1\033[0m"
    exit 1
}

# 操作系统版本
OS_VERSION="ubuntu_20.04"
# linux kernel版本
KERNEL_VERSION="5.15.98"
kernel_r=$(uname -r)

exagear_src=https://kunpeng-repo.obs.cn-north-4.myhuaweicloud.com/Exagear%20ARM32-ARM64/Exagear%20ARM32-ARM64%202.0.0.2/Exagear-ARM32-ARM64.tar.gz

# 当前目录
CURRENT_DIR=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)
cd "${CURRENT_DIR}" || exit
new_file_dir="/usr/new_file"
# 依赖目录
mkdir -p ~/dependency/
work_dir=~/dependency/
# Kbox_AOSP根目录
kbox_dir=$(cd "${CURRENT_DIR}"/../../ && pwd)
# 本地依赖包目录
packages_dir=$1

################################################################################
# Function Name: choose_run_mode
# Description  : 选择源码包获取方式
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function choose_run_mode(){
    read -p "请选择源码包获取方式([0]自动下载-需联网 [1]本地):" run_mode
    while [[ ${run_mode} != "0" ]] && [[ ${run_mode} != "1" ]]
    do
        read -p "输入无效, 请重新输入:" run_mode
    done
}

################################################################################
# Function Name: choose_decode_method
# Description  : 选择解码方式
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function choose_decode_method(){
    read -p "请选择解码方式([0]软解 [1]硬解):" decode_method
    while [[ ${decode_method} != "0" ]] && [[ ${decode_method} != "1" ]]
    do
        read -p "输入无效, 请重新输入:" decode_method
    done
}

################################################################################
# Function Name: check_local_package
# Description  : 检查本地依赖包
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function check_local_package(){
    [ -e "${packages_dir}/${1}" ] || error "指定路径下无法找到${packages_dir}/${1}"
}

################################################################################
# Function Name: check_local_packages
# Description  : 检查本地依赖包
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function check_local_packages(){
    check_local_package "android.tar"
    if [ ${decode_method} = "1" ]
    then
        check_local_package "NETINT.tar.gz"
    fi
    if [ ${run_mode} = "1" ]
    then
        check_local_package "Exagear-ARM32-ARM64.tar.gz"
    fi
}

################################################################################
# Function Name: get_nic_name
# Description  : 获取网口名称
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function get_nic_name(){
read -p "请输入网口名称(如enp125s0f1):" nic_name
    pci_id=$(ethtool -i ${nic_name} | grep bus-info | awk '{print $2}')
    while [ -z ${pci_id} ]
    do
        read -p "网口名称无效, 请重新输入:" nic_name
        pci_id=$(ethtool -i ${nic_name} | grep bus-info | awk '{print $2}')
    done
}

################################################################################
# Function Name: show_kernel_version
# Description  : 检查内核版本是否正确。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function show_kernel_version(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 1 检查内核版本 ]]]]]]]]]]]]]]]]]]]]"
    echo "Linux kernel version: ${kernel_r}"
    if [ ${kernel_r} != ${KERNEL_VERSION} ]
    then
        error "内核版本不符, 默认版本为${KERNEL_VERSION}"
    fi
}

################################################################################
# Function Name: set_amdgpu_performance
# Description  : 设置AMD显卡性能模式。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function set_amdgpu_performance(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 2 设置AMD显卡性能模式 ]]]]]]]]]]]]]]]]]]]]"
    find /sys -name power_dpm_force_performance_level | xargs -I {} sh -c "echo high > '{}'"
    [ $? -ne 0 ] && error "显卡性能模式配置失败"
    echo "------------显卡已设置为性能模式-----------"
}

################################################################################
# Function Name: start_docker
# Description  : 启动docker服务。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function start_docker(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 3 启动docker服务 ]]]]]]]]]]]]]]]]]]]]"
    systemctl daemon-reload
    systemctl restart docker
    systemctl enable docker
    echo "------------docker服务已启动-----------"
}

################################################################################
# Function Name: set_a32a64
# Description  : 转码使能。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function set_a32a64(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 4 Exagear转码使能 ]]]]]]]]]]]]]]]]]]]]"
    if [ -e "/proc/sys/fs/binfmt_misc/ubt_a32a64" ]
    then
        < /proc/sys/fs/binfmt_misc/ubt_a32a64 grep "enabled" > /dev/null 2>&1
        [ $? -eq 0 ] && return 0
    fi
    [ -e "/proc/sys/fs/binfmt_misc/ubt_a32a64" ] && echo -1 > /proc/sys/fs/binfmt_misc/ubt_a32a64
    rm -rf /opt/exagear/ubt_a32a64
    mount |grep "binfmt_misc on" > /dev/null 2>&1
    [ $? -ne 0 ] && mount -t binfmt_misc none /proc/sys/fs/binfmt_misc
    mkdir -p /opt/exagear
    chmod -R 700 /opt/exagear
    cd ${work_dir} || exit
    if [ ${run_mode} = "0" ]
    then
        echo "------------下载Exagear转码包转码------------"
        wget ${exagear_src} --no-check-certificate || error "Exagear转码包下载失败"
    else
        cp "${packages_dir}/Exagear-ARM32-ARM64.tar.gz" . || exit
    fi
    [ -e "output" ] && rm -rf patchForKernel
    tar -xzvf Exagear-ARM32-ARM64.tar.gz
    chown -R root:root output
    chmod -R 700 output
    cp output/ubt_a32a64 /opt/exagear/ || exit
    chmod +x /opt/exagear/ubt_a32a64
    echo "------------配置Exagear转码------------"
    translator_name=ubt_a32a64
    echo ":${translator_name}:M::\x7fELF\x01\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x28\x00:\xff\xff\xff\xff\xff\xff\xff\x00\xff\xff\xff\xff\xff\xff\xff\xff\xfe\xff\xff\xff:/opt/exagear/ubt_a32a64:POCF" > /proc/sys/fs/binfmt_misc/register
    < /proc/sys/fs/binfmt_misc/ubt_a32a64 grep "enabled"
    [ $? -ne 0 ] && error "Exagear规则注册失败"
    echo "------------Exagear转码已开启------------"
}

################################################################################
# Function Name: insmod_ashmem_binder
# Description  : 安装ashmem、binder两个内核模块。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function insmod_ashmem_binder(){
    local count_time=0
    local binder_num
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 5 安装ashmem binder模块 ]]]]]]]]]]]]]]]]]]]]"
    cd /lib/modules/${KERNEL_VERSION}/kernel/lib/ || exit
    lsmod |grep -w aosp_binder_linux
    if [ $? -ne 0 ]
    then
        echo "--------安装binder模块--------"
        insmod aosp_binder_linux.ko num_devices=400
        lsmod |grep -w aosp_binder_linux
        [ $? -ne 0 ] && error "binder安装失败"
    fi
    cd /lib/modules/${KERNEL_VERSION}/kernel/lib/ || exit
    lsmod |grep -w ashmem_linux
    if [ $? -ne 0 ]
    then
        echo "--------安装ashmem模块--------"
        insmod ashmem_linux.ko
        lsmod |grep -w ashmem_linux
        [ $? -ne 0 ] && error "ashmem安装失败"
    fi
    while true
    do
        sleep 1
        binder_num=$(ls /dev/|grep -c "^aosp_binder[0-9]\{1,3\}$")
        [ "${binder_num}" -eq 400 ] && break
        if [ ${count_time} -gt 15 ]
        then
            echo -e "\033[1;31m insmod aosp_binder failed\033[0m"
            break
        fi
        (( count_time++ )) || true
    done
    echo "--------配置dev可执行权限--------"
    chmod 600 /dev/aosp_binder*
    chmod 600 /dev/ashmem
    chmod 600 /dev/dri/*
    chmod 600 /dev/input
}

################################################################################
# Function Name: nic_bind_cpu
# Description  : 网卡绑定cpu。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function nic_bind_cpu(){
    local interrupts_ids
    local numa_node_id
    local cpu_id=0
    local cpu_id_start
    local cpu_bind_num=2  # 绑定NIC的CPU核数暂时定为2
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 6 网卡绑定CPU ]]]]]]]]]]]]]]]]]]]]"
    numa_node_id=$(lspci -vvvs ${pci_id} | grep "NUMA node" | awk '{print $3}')
    cpu_id_start=$(lscpu | grep "NUMA node${numa_node_id}" | awk '{print $4}' | awk -F - '{print $1}')
    interrupts_ids=("$(cat /proc/interrupts | grep "${pci_id}" | awk -F: '{print $1}')")
    for interrupts_id in ${interrupts_ids[*]}
    do
        echo $((cpu_id_start+cpu_id%cpu_bind_num)) > /proc/irq/${interrupts_id}/smp_affinity_list
        let "cpu_id++"
    done
    echo "--------已绑定完毕--------"
}

################################################################################
# Function Name: import_android_image
# Description  : 导入android镜像到docker。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function import_android_image(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 7 导入android镜像 ]]]]]]]]]]]]]]]]]]]]"
    cd ${packages_dir} || exit
    docker images|awk '{print $1" "$2}'|grep -w "kbox"|grep -w "origin" > /dev/null 2>&1
    if [ $? -ne 0 ]
    then
        if [ -f "android.tar" ]
        then
            echo "--------正在导入镜像--------"
            docker import android.tar kbox:origin || exit
        fi
    fi

    if [[ ${decode_method} = "1" ]]
    then
        echo "--------制作硬解镜像--------"
        local netint_file="${packages_dir}/NETINT.tar.gz"
        cp ${netint_file} ${kbox_dir}/make_img_sample/decode_iso_build/ || exit
        cd ${kbox_dir}/make_img_sample/decode_iso_build/ || exit
        chmod +x Dockerfile make_image.sh
        ./make_image.sh kbox:origin kbox:latest
    fi
}

################################################################################
# Function Name: modify_config
# Description  : config文件修改。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function modify_config(){
    < ${3} grep -v "#"|grep -w "${1}"
    if [ $? -eq 0 ]
    then
        if [ "${2}" = "is not set" ]
        then
            sed -i "s|${1}=.*|\# ${1} is not set|g" ${3}
        else
            sed -i "s|${1}=.*|${1}=${2}|g" ${3}
        fi
    else
        if [ "${2}" != "is not set" ]
        then
            echo "${1}=${2}" >> ${3}
        fi
    fi
    echo "${3} >> $(< ${3} grep -w "${1}")"
}

################################################################################
# Function Name: start_kbox_default
# Description  : 尝试启动一个kbox11，判断环境是否已经配置好。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function start_kbox_default(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 8 启动Kbox ]]]]]]]]]]]]]]]]]]]]"
    local kbox_name
    local count_time=0
    local boot_state
    local nvme_ids
    local bus_id
    local numa_id
    local netint0_str
    local netint1_str
    cd ${kbox_dir}/deploy_scripts || exit
    chmod +x android11_kbox.sh
    docker ps -a|awk '{print $(NF-1)" "$NF}'|grep -E -w "8501|kbox_1" > /dev/null 2>&1
    [ $? -eq 0 ] && return 0

    if [ ${decode_method} = "0" ]
    then
        ./android11_kbox.sh start kbox:origin 1 > /dev/null 2>&1
    else
        modify_config "ENABLE_HARD_DECODE" "1" kbox_config.cfg
        nvme_ids=$(nvme list | grep "T432-8" | grep -o "nvme." | grep -o "[0-9]")
        for nvme_id in ${nvme_ids[*]}
        do
            bus_id=$(find /sys/devices/ -name nvme${nvme_id} | grep "pci" | awk -F '/' '{print $6}')
            numa_id=$(lspci -vvvs ${bus_id} | grep "NUMA" | grep -o [0-9])
            if [ ${numa_id} \< "2" ]
            then
                netint0_str=${netint0_str}"/dev/nvme${nvme_id},/dev/nvme${nvme_id}n1,"
            else
                netint1_str=${netint1_str}"/dev/nvme${nvme_id},/dev/nvme${nvme_id}n1,"
            fi
        done
        if [ -z netint0_str ]
        then
            netint0_str=netint1_str
        fi
        if [ -z netint1_str ]
        then
            netint1_str=netint0_str
        fi
        netint0_str=${netint0_str%%,} || exit
        netint1_str=${netint1_str%%,} || exit
        modify_config "NETINT0" "${netint0_str}" kbox_config.cfg
        modify_config "NETINT1" "${netint1_str}" kbox_config.cfg

        ./android11_kbox.sh start kbox:latest 1 > /dev/null 2>&1
    fi

    kbox_name=kbox_1
    docker exec -i ${kbox_name} sh -c "getprop sys.boot_completed"|grep 1
    if [ $? -ne 0 ]
    then
        echo "${kbox_name} 启动失败, 请重新检查环境。"
    else
        echo "${kbox_name} 已启动, 环境部署完毕。"
    fi
}

main(){
    choose_run_mode
    choose_decode_method
    get_nic_name
    show_kernel_version
    set_amdgpu_performance
    start_docker
    set_a32a64
    insmod_ashmem_binder
    nic_bind_cpu
    import_android_image
    start_kbox_default
}

main "$@"
exit 0
