#!/bin/bash
#******************************************************************************
# Copyright (C), 2015-2017, Huawei Tech. Co., Ltd.
# File : uninstall.sh
# 1.uninstall iBMA2.0 software
#
#  History       :
#  1.Date        : 2016/12/26
#    Modification: Created file
#******************************************************************************

if [ -L ${0} ]; then
    REALFILE=$(readlink ${0})
else
    REALFILE=${0}
fi

CUR_DIR=$(dirname ${REALFILE})
cd ${CUR_DIR}
cd ..

# Paths to programs
ECHO=/bin/echo
OLD_BMA_DIR=$(pwd)
SERVICE_NAME=iBMA
DIR_NAME=ibma
SRRVICE_EXIST=false
SILENT_MODE=false
UNINSTALL_DRIVER=false
SERVICE_SCRIPT=/etc/init.d/${SERVICE_NAME}
IBMA_REBOOT_NEEDED=false

BOBSERVICE_NAME=bob
BOBSYSTEMCTL_SERVICE=bob.service
BOBSERVICE_SCRIPT=/etc/init.d/${BOBSERVICE_NAME}

NETWORK_SETUP_TOOL="${OLD_BMA_DIR}/${DIR_NAME}/bin/manager"

OS_SUPPROT_SYSTEMCTL=false
SYSTEMCTL_SERVICE=iBMA.service
SYSTEMD_DIR='/usr/lib/systemd/system/'
if [ ! -e "${SYSTEMD_DIR}" ]; then
    # Ubuntu systemctl services save in the follow directory
    SYSTEMD_DIR='/lib/systemd/system/'
fi
SYSTEMD_EXEC_DIR='/etc/systemd/system/multi-user.target.wants/'

source "${OLD_BMA_DIR}/${DIR_NAME}/lib/Linux/log.sh"

#*****************************************************************************
# Prototype    : print_help
# Description  : show help information
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/03/03
#    Modification : Created function
#
#*****************************************************************************
function print_help()
{
    LOG_INFO "${SERVICE_NAME} Uninstaller                               "
    LOG_INFO "Usage:                                                    "
    LOG_INFO "$0          Uninstall ${SERVICE_NAME}, recommended        "
    LOG_INFO "$0 -s       Silent uninstallation mode                    "
    LOG_INFO "$0 -a       Uninstall ${SERVICE_NAME} and driver          "
    LOG_INFO "$0 -h       Show this help                                "
}

#*****************************************************************************
# Prototype    : stop_service
# Description  : stop the service which is specify by parameter.
# Parameter:
#   input:  $1: service name,  $2: service script path.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/11/19
#    Modification : Created function
#
#*****************************************************************************
function stop_service()
{
    local service_running=true

    stop_output=$(${2} stop 2>&1)
    if [ $? -eq 0 ]; then
        service_running=false
    fi

    LOG_INFO "${stop_output}"

    if [ false != ${service_running} ]; then
        LOG_ERROR "Exit uninstallation process."
        exit 1
    fi

    if [ -r "${SYSTEMD_DIR}/${1}.service" ]; then
        systemctl stop ${1} >& /dev/null
    fi
}

#*****************************************************************************
# Prototype    : stop_ibma_service
# Description  : try to stop iBMA service if it's running.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2018/07/26
#    Modification : Stop bob service
#  3.Date         : 2019/08/10
#    Modification : Stop all network service
#*****************************************************************************
function stop_ibma_service()
{
    local service_running=false
    local bobservice_running=true
    local bma_script_path=""
    local bob_script_path=""
    local i=0
    local running_flag=false

    # "/etc/init.d/iBMA" is the default path
    if [ -r "${SERVICE_SCRIPT}" ]; then
        SRRVICE_EXIST=true
        service_running=true
        # get running status.

        local _path=$(grep ^ROOT_PATH= "${SERVICE_SCRIPT}")
        if [ "$_path" != "" ]; then
            IFS='=';
            arr=($_path);
            BMA_DIR=$(echo ${arr[1]}|sed 's/\"//g')
            LOG_INFO "Found ${SERVICE_NAME} installed in the path <${BMA_DIR}>" "HIDE"
        else
            BMA_DIR=""
            LOG_WARN "Found ${SERVICE_NAME} file is invalid"
        fi

        if [ "${BMA_DIR}" != "" ]; then
            bma_script_path="${BMA_DIR}/ibma/iBMA.sh"
            bob_script_path="${BMA_DIR}/ibma/bob.sh"
        fi
    elif [ -r "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ]; then
        SRRVICE_EXIST=true
        # get running status.
        local section=Service
        local item=ExecStart
        service_running=true
        exec_start_value=$(read_from_file "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ${section} ${item})
        bma_script_path=$(echo $exec_start_value| awk '{print $1}')
        BMA_DIR=$(echo "${bma_script_path%/*}")
        BMA_DIR=$(echo "${BMA_DIR%/*}")
        bob_script_path="${BMA_DIR}/ibma/bob.sh"
    fi

    "${bma_script_path}" status  >& /dev/null || service_running=false
    "${bob_script_path}" status  >& /dev/null || bobservice_running=false

    service_list=("${BOBSERVICE_NAME}" "${bob_script_path}" "${bobservice_running}"
                  "${SERVICE_NAME}" "${bma_script_path}" "${service_running}")
    local service_item_num=3

    # service is running, exit if in quiet mode. else, prompt.
    while [ ${i} -le $(expr ${#service_list[@]} - ${service_item_num}) ]
    do
        if [ true == ${service_list[$(expr ${i} + 2)]} ]; then
            if [ true == ${SILENT_MODE} ]; then
                LOG_INFO "${service_list[${i}]} service is running. Stop it first."
                running_flag=true
                i=$(expr ${i} + ${service_item_num})
                continue
            fi

            LOG_INFO "${service_list[${i}]} service is running."
            PS3="Enter your choice:"
            options=("Continue" "Exit")
            select opt in "${options[@]}"; do
                case $opt in
                    "Continue")
                        LOG_INFO "Trying to stop ${service_list[${i}]} service."
                        break;
                        ;;
                    "Exit")
                        LOG_INFO "You chose to cancel." "HIDE"
                        exit 1;
                        ;;
                    *)
                        LOG_INFO "Invalid option."
                        ;;
                esac
            done

            # try stop service.
            stop_service "${service_list[${i}]}" "${service_list[$(expr ${i} + 1)]}"
        fi

        i=$(expr ${i} + ${service_item_num})
    done

    if [ false != ${running_flag} ]; then
        LOG_ERROR "Failed to uninstall ${SERVICE_NAME} software."
        exit 3
    fi

    if [ -x ${NETWORK_SETUP_TOOL} ]; then
        ${NETWORK_SETUP_TOOL} stop all
    fi
}

#*****************************************************************************
# Prototype    : remove_ibma_files
# Description  : remove iBMA files.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function remove_ibma_files()
{
    LOG_INFO "Delete ${SERVICE_NAME} directory."

    if [ -d "${OLD_BMA_DIR}/${DIR_NAME}" ]; then
        rm -rf "${OLD_BMA_DIR}/${DIR_NAME}"
    fi

    if [ -d "${OLD_BMA_DIR}" ] && [ "${OLD_BMA_DIR}" != "/opt" ]; then
        rmdir "${OLD_BMA_DIR}" >& /dev/null
    fi
}

#*****************************************************************************
# Prototype    : remove_related_files
# Description  : remove files related to iBMA driver.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/07/10
#    Modification : Created function
#  2.Date         : 2018/03/07
#    Modification : Only match the string before -.
#                   Solve the problem of deleting weak ko failed
#*****************************************************************************
function remove_related_files()
{
    local kernel_version=$(uname -r)
    local first_version=$(echo ${kernel_version/-*})
    local driver_path=$1
    local driver_path_weak="/lib/modules/${first_version}*/weak-updates/"

    LOG_INFO "delete all files related to kernel ${driver_path}" "HIDE"

    if [ -n "${driver_path}" ]; then
        # delete all weak updates related to this module.
        ko_files=$(find ${driver_path_weak} -lname ${driver_path}*.ko 2>/dev/null)

        if [ -z "${ko_files}" ]; then
            return 0
        fi

        ko_count=$(echo "${ko_files}" | wc -l)
        for ((ko_cnt=1;ko_cnt<=ko_count;ko_cnt++ )); do
            ko_file=$(echo "${ko_files}" | awk '(NR=='$ko_cnt')')
            if [ -z "${ko_file}" ]; then
                continue;
            fi

            res=$(rm -f ${ko_file} 2>&1)
            if [ -n "${res}" ]; then
                LOG_INFO "delete ${ko_file}, res:${res}" "HIDE"
            fi
        done
    fi

    res=$(depmod -a 2>&1)
    LOG_INFO "depmod:${res}" "HIDE"

    return 0
}

#*****************************************************************************
# Prototype    : sles_driver_post
# Description  : action after sles drvier uninstall .
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2021/1/17
#    Modification : Created function
#
#*****************************************************************************
function driver_uninstall_post()
{
    local ibmadrivers=("host_cdev_drv" "host_kbox_drv" "host_veth_drv" "cdev_veth_drv" "host_edma_drv")
    local drivernum=0
    while [ ${drivernum} -lt ${#ibmadrivers[@]} ]; do
        mods=( $(/sbin/lsmod | grep -wo "${ibmadrivers[${drivernum}]}") )
        if [ "${mods}" == "${ibmadrivers[${drivernum}]}" ]; then
            /sbin/rmmod "${ibmadrivers[${drivernum}]}" >/dev/null 2>&1
            if [ "$?" -ne 0 ]; then
                LOG_INFO "Warning: failed to uninstall the installed ${ibmadrivers[${drivernum}]} kernel module."
                LOG_INFO "Please uninstall the installed ${ibmadrivers[${drivernum}]} kernel module manually or reboot the system."
            else
                LOG_INFO "Uninstall the installed ${ibmadrivers[${drivernum}]} kernel module successfully." "HIDE"
            fi
        fi
        let drivernum++
    done
}

#*****************************************************************************
# Prototype    : uninstall_driver
# Description  : uninstall iBMA driver.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2017/07/10
#    Modification : add function remove_related_files.
#  3.Date         : 2019/08/10
#    Modification : unload 'cdev_veth_drv' module after uninstallation,
#                   ignore error.
#  3.Date         : 2021/1/17
#    Modification : rmmod driver after uninstall.
#*****************************************************************************
function uninstall_driver()
{
    local res
    local packages
    local driver_path
    local reboot_pattern="reboot the system"
    local veth_cfg="/etc/sysconfig/network-scripts/ifcfg-veth"

    if [ false != ${UNINSTALL_DRIVER} ]; then

        driver_path=$(modinfo host_edma_drv 2>/dev/null | grep ^filename: | awk '{print $2}')
        packages=$(rpm -qa 2>/dev/null | grep -v "iBMA_Driver-dkms" | grep "iBMA_driver")
        if [ $? -eq 0 ]; then
            LOG_INFO "Uninstalling ${SERVICE_NAME} driver ..."
            pkg_count=$(echo "${packages}" | wc -l)
            for ((pkg_cnt=1;pkg_cnt<=pkg_count;pkg_cnt++ )); do
                package=$(echo "${packages}" | awk '(NR=='$pkg_cnt')')
                if [ -z "${package}" ]; then
                    continue;
                fi

                res=$(rpm -e "${package}" 2>&1)
                if [ $? -ne 0 ] || [ $(rpm -qa |grep ${package} |wc -l) -eq 1 ]; then
                    LOG_ERROR "Failed to uninstall ${SERVICE_NAME} driver,${res}"
                    exit 1
                else
                    remove_ibma_driver_from_initramfs
                fi
                LOG_INFO "Uninstall ${package} result:${res}" "HIDE"
                if [[ ${res} == *${reboot_pattern}* ]]; then
                    IBMA_REBOOT_NEEDED=true
                fi
                driver_uninstall_post
            done
        else
            dpkg -l 2>/dev/null | grep ibmadriver  > /dev/null 2>&1
            if [ 0 -ne $? ]; then
                LOG_INFO "Driver package is already uninstalled."
                # delete veth config file.
                res=$(rm -f "${veth_cfg}")
                LOG_INFO "delete ${veth_cfg}:${res}" "HIDE"
                return 0
            fi

            LOG_INFO "Uninstalling ${SERVICE_NAME} driver ..."
            res=$(dpkg -P "ibmadriver" 2>&1)
            LOG_INFO "Uninstall ibmadriver result:${res}" "HIDE"
            if [[ ${res} == *${reboot_pattern}* ]]; then
                IBMA_REBOOT_NEEDED=true
            fi
        fi

        [ -L "${driver_path}" ] && driver_path=$(readlink "${driver_path}")
        driver_path=${driver_path%/*}

        remove_related_files ${driver_path}
        res=$(rm -f "${veth_cfg}")
        LOG_INFO "delete ${veth_cfg}:${res}" "HIDE"

        LOG_INFO "Uninstall ${SERVICE_NAME} driver successfully."
    else
        LOG_INFO "Do not uninstall ${SERVICE_NAME} driver." "HIDE"
        rmmod cdev_veth_drv >& /dev/null
    fi
}

#*****************************************************************************
# Prototype    : remove_ibma_driver_from_initramfs
# Description  : from initramfs remove iBMA driver ko.
# Parameter:
#   input:  $1 iBMA driver ko list
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2018/06/22
#    Modification : Created function
#*****************************************************************************
function remove_ibma_driver_from_initramfs()
{
    local res=""

    cd /opt

    # get initramfs first, ignore the exception
    initramfs_list=$(ls /boot/initramfs-*{x86_64,aarch64}.img 2>/dev/null)
    initramfs_count=$(echo "${initramfs_list}" | wc -l)
    for ((initramfs_cnt=1;initramfs_cnt<=initramfs_count;initramfs_cnt++ )); do
        initramfs_file=$(echo "${initramfs_list}" | awk '(NR=='$initramfs_cnt')')
        if [ -z "${initramfs_file}" ]; then
            continue;
        fi

        ko_path_list=$(lsinitrd ${initramfs_file} 2>&1 | grep -i "host_edma_drv.ko")
        if [ $? -ne 0 ]; then
            continue
        fi

        ko_path_count=$(echo "${ko_path_list}" | wc -l)
        for (( ko_cnt=1;ko_cnt<=ko_path_count;ko_cnt++ )); do
            ko_path=$(echo "${ko_path_list}" | awk '(NR=='$ko_cnt')' | awk '{print $9}')
            res=$(printf "%s\n" "${ko_path}" | /sbin/weak-modules --remove-modules ${initramfs_file} 2>&1)
            if [ $? -ne 0 ]; then
                LOG_ERROR "Failed to remove iBMA driver modules, ${res}." "HIDE"
            fi
        done
    done

    return 0
}

#*****************************************************************************
# Prototype    : uninstall_package
# Description  : uninstall package.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/12/30
#    Modification : Created function
#
#*****************************************************************************
function uninstall_package()
{
    local res
    LOG_INFO "Uninstall ${SERVICE_NAME} package."
    # rpm package.
    packages=$(rpm -qa 2>/dev/null | grep "iBMA2.0" | grep -v "iBMA2.0-package")
    if [ $? -eq 0 ]; then
        res=$(rpm -e ${packages} 2>&1)
        if [ $? -ne 0 ] || [ $(rpm -qa |grep ${packages} |wc -l) -eq 1 ]; then
            LOG_ERROR "Failed to uninstall ${packages}, ${res}"
            return 1
        fi
    fi

    packages=$(dpkg -l 2>/dev/null | grep "ibma2.0" | grep -v "ibma2.0-package")
    if [ $? -eq 0 ]; then
        res=$(dpkg -P ibma2.0 2>&1)
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to uninstall ibma2.0,${res}"
            return 1
        fi
    fi

    return 0
}

case $1 in
    '')
        LOG_INFO "Starting to uninstall ${SERVICE_NAME}."
        ;;
    '-s')
        SILENT_MODE=true
        LOG_INFO "Starting to uninstall ${SERVICE_NAME} in silent mode."
        ;;
    '-h')
        print_help
        exit 0
        ;;
    '-a')
        UNINSTALL_DRIVER=true
        LOG_INFO "Starting to uninstall ${SERVICE_NAME} and driver."
        ;;
    *)
        print_help
        exit 1
        ;;
esac

# try stop service.
stop_ibma_service

#uninstall iBMA package.
uninstall_package
if [ $? -ne 0 ]; then
    exit 1
fi

# remove iBMA dir, and try to remove installation path.
remove_ibma_files

# uninstall iBMA driver.
uninstall_driver

# log uninstall successfully.
LOG_INFO "${SERVICE_NAME} uninstalled successfully."
if ${IBMA_REBOOT_NEEDED}; then
    LOG_INFO "You need to reboot the OS to take effect."
fi

exit 0
