#!/bin/bash
#******************************************************************************
# Copyright (C), 2015-2017, Huawei Tech. Co., Ltd.
# File : install.sh
# 1.install iBMA2.0 software
#
#  History       :
#  1.Date        : 2016/12/26
#    Modification: Created file
#******************************************************************************

# get and cd to project dir
if [ -L ${0} ]; then
    REALFILE=$(readlink ${0})
else
    REALFILE=${0}
fi

CUR_DIR=$(dirname ${REALFILE})

cd ${CUR_DIR}

# Paths to programs
G_WORK_DIR=$(pwd)
ECHO=/bin/echo

BMA_DIR=/opt/huawei
DIR_NAME=ibma

INSTALL_LOG_FILE=${BMA_DIR}/${DIR_NAME}/log/runlog
IBMA_UPGRADE_LOG_PATH=${BMA_DIR}/${DIR_NAME}/log/iBMAUpgradeLog

SERVICE_NAME=iBMA
SYSTEMCTL_SERVICE=iBMA.service
SERVICE_SCRIPT=/etc/init.d/${SERVICE_NAME}

BOBSERVICE_NAME=bob
BOBSYSTEMCTL_SERVICE=bob.service
BOBSERVICE_SCRIPT=/etc/init.d/${BOBSERVICE_NAME}

IBMACLI_LINK_PATH=/usr/bin/ibmacli

PRODUCT_ID=""

OS_NAME=""
VERSION_ID_BASE=0
SYSTEMD_DIR='/usr/lib/systemd/system/'
if [ ! -e "${SYSTEMD_DIR}" ]; then
    # Ubuntu systemctl services save in the follow directory
    SYSTEMD_DIR='/lib/systemd/system/'
fi
SYSTEMD_EXEC_DIR='/etc/systemd/system/multi-user.target.wants/'

# Version of SUSE system is stored in /etc/os-release starting from SLES 15
readonly ETC_FILE=( "/etc/bclinux-release" "bclinux" \
                    "/etc/openEuler-release" "openEuler" \
                    "/etc/euleros-release" "EulerOS" \
                    "/etc/centos-release" "CentOS" \
                    "/etc/redhat-release" "RedHat" \
                    "/etc/SuSE-release" "SUSE" \
                    "/etc/os-release" "SUSE")

readonly OTHER_OS_FILE="/etc/os-release"

# Black list of os which not support systemctl cmd
OS_SUPPROT_SYSTEMCTL_BLACK_LIST=("SUSE" "RedHat" "CentOS")

OS_SUPPROT_SYSTEMCTL=false

source "${BMA_DIR}/${DIR_NAME}/lib/Linux/log.sh"

#*****************************************************************************
# Prototype    : get_os_version_id_base
# Description  : get OS name, save to OS_NAME.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#
#*****************************************************************************
function get_os_version_id_base()
{
    # check os-specific file.
    local index=0
    local centos_versionid_base=7
    local redhat_versionid_base=7
    local suse_versionid_base=12

    #search ETC_FILE list
    while [ ${index} -lt ${#ETC_FILE[@]} ]; do

        #ETC_FILE[index] is exist
        if [ -e "${ETC_FILE[${index}]}" ]; then
            OS_NAME=${ETC_FILE[${index}+1]}
            case ${OS_NAME} in
                "CentOS")
                    VERSION_ID_BASE=$centos_versionid_base
                    ;;
                "RedHat")
                    VERSION_ID_BASE=$redhat_versionid_base
                    ;;
                "SUSE")
                    cat "${ETC_FILE[${index}]}" |grep -wi ${OS_NAME} -q
                    if [ $? -eq 0 ]; then
                        VERSION_ID_BASE=$suse_versionid_base
                    fi
                    ;;
                *)
                    ;;
            esac

            LOG_INFO "ETC OS is [${OS_NAME}]" "HIDE"
            return 0;
        fi

        #index offset 2
        let index+=2;
    done
}

#*****************************************************************************
# Prototype    : cmp_os_version_id
# Description  : compare os version id.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#
#*****************************************************************************
function cmp_os_version_id()
{
    # check other os file.
    local cmp_return
    local version_id_key="VERSION_ID="
    local content=$(cat "${OTHER_OS_FILE}" 2>/dev/null |grep -i "${version_id_key}");
    if [ -n "${content}" ]; then
        version_id_str=${content##*=}
        version_id=$(echo ${version_id_str}|sed 's/\"//g')
        local cmp_return=$(echo ${version_id} ${VERSION_ID_BASE} \
                            | awk '{if($1<$2) {printf"less"} \
                            else {printf"greater"}}' 2> /dev/null)
        local ret=$(echo $cmp_return|grep "greater")
        if [ -n "${ret}" ]; then
            LOG_INFO "Current OS need use systemctl" "HIDE"
            return 0;
        else
            return 1;
        fi
    else
        LOG_INFO "Failed to get the system version id." "HIDE"
        return 1;
    fi
}

#*****************************************************************************
# Prototype    : is_support_systemctl
# Description  : is support systemctl.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#  2.Date         : 2020/10/23
#    Modification : Support systemctl on uos
#*****************************************************************************
function is_support_systemctl()
{
    local ret=0
    local res1=0
    local res2=1

    get_os_version_id_base

    for os_type in ${OS_SUPPROT_SYSTEMCTL_BLACK_LIST[@]}
    do
        if [[ "${OS_NAME}" =~ "${os_type}" ]]; then
            ret=1
            break
        fi
    done

    # Only SUSE, Redhat and CentOS may not support systemctl cmd
    if [ ${VERSION_ID_BASE} -ne 0 ] && [ $ret -eq 1 ]; then
        ret=$(cmp_os_version_id 2>/dev/null)
        if [ $? -ne 0 ] ; then
            res1=1
        fi
    fi

    ret=$(systemctl get-default 2>/dev/null)
    if [ $? -eq 0 ] ; then
        res2=0
    fi

    if [ $res1 -eq 0 ] && [ $res2 -eq 0 ] ; then
        OS_SUPPROT_SYSTEMCTL=true
    fi
}

#*****************************************************************************
# Prototype    : system_ctl_proc
# Description  : is use systemctl, system process.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#
#*****************************************************************************
function system_ctl_proc()
{
    local res_src_link=1
    local res_dst_link=1
    local res_bob_src_link=1
    local res_bob_dst_link=1
    if [ -e "${SYSTEMD_EXEC_DIR}${SYSTEMCTL_SERVICE}" ]; then
        # remove old link.
        res=$(rm -rf  "${SYSTEMD_EXEC_DIR}${SYSTEMCTL_SERVICE}" 2> /dev/null)
        res_src_link=0
    fi

    if [ -e "${SYSTEMD_DIR}${SYSTEMCTL_SERVICE}" ]; then
        # remove old link.
        res=$(rm -rf  "${SYSTEMD_DIR}${SYSTEMCTL_SERVICE}" 2> /dev/null)
        res_dst_link=0
    fi

    if [ -e "${SYSTEMD_EXEC_DIR}${BOBSYSTEMCTL_SERVICE}" ]; then
        # remove old link.
        res=$(rm -rf  "${SYSTEMD_EXEC_DIR}${BOBSYSTEMCTL_SERVICE}" 2> /dev/null)
        res_bob_src_link=0
    fi

    if [ -e "${SYSTEMD_DIR}${BOBSYSTEMCTL_SERVICE}" ]; then
        # remove old link.
        res=$(rm -rf  "${SYSTEMD_DIR}${BOBSYSTEMCTL_SERVICE}" 2> /dev/null)
        res_bob_dst_link=0
    fi

    if [ 0 == $res_src_link ] || [ 0 == $res_dst_link ] \
        || [ 0 == $res_bob_src_link ] || [ 0 == $res_bob_dst_link ] ; then
        res=$(systemctl daemon-reload 2> /dev/null)
        if [ 0 != $? ] ; then
            LOG_ERROR "Systemctl daemon-reload failed, return ${res}."
            return 1;
        fi
    fi

    # Set permissions of service scripts under SYSTEMD_DIR to 644.
    # Otherwise, "/usr/lib/systemd/system/iBMA.service is marked world-inaccessible." will be logged in /var/log/messages.
    cp -a "${BMA_DIR}/${DIR_NAME}/script/${SYSTEMCTL_SERVICE}" "${SYSTEMD_DIR}"
    chmod 644 ${SYSTEMD_DIR}${SYSTEMCTL_SERVICE}

    cp -a "${BMA_DIR}/${DIR_NAME}/script/${BOBSYSTEMCTL_SERVICE}" "${SYSTEMD_DIR}"
    chmod 644 ${SYSTEMD_DIR}${BOBSYSTEMCTL_SERVICE}

    res=$(systemctl list-unit-files|grep iBMA)
    if [ 0 -eq $? ] ; then
        res=$(systemctl enable ${SYSTEMCTL_SERVICE} 2>/dev/null 1>/dev/null)
        if [ 0 != $? ] ; then
            LOG_ERROR "Systemctl enable ${SYSTEMCTL_SERVICE} failed, return ${res}."
            return 1;
        fi
    fi

    res=$(systemctl daemon-reload 2> /dev/null)
    if [ 0 != $? ] ; then
        LOG_WARN "Systemctl daemon-reload failed, return ${res}." "HIDE"
    fi

    NEED_SYSTEMCTL_START_IBMA=true
}

#*****************************************************************************
# Prototype    : handle_oam_function
# Description  : set related permissions for OAM files.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/08/16
#    Modification : Created function
#  2.Date         : 2018/08/20
#    Modification : support OAM on all servers when only support on E9000 before
#*****************************************************************************
function handle_oam_function()
{
    LOG_INFO "Setting OAM related permissions." "HIDE"
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/OAM.ini
    ln -sf "${BMA_DIR}/${DIR_NAME}/bin/ReloadOAM" \
           "${BMA_DIR}/${DIR_NAME}/config/ReloadOAM"
}

#*****************************************************************************
# Prototype    : handle_dile
# Description  : handle file.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/12/31
#    Modification : Created function
#  2.Date         : 2020/08/12
#    Modification : change the permission of MeshDetect.ini to 640 if it exist
#  3.Date         : 2020/12/03
#    Modification : change the permission of EthernetInterface.ini to 640
#*****************************************************************************
function handle_dile()
{
    cd ${BMA_DIR}/${DIR_NAME}

    umask 177

    chmod 540 -Rf ${BMA_DIR}/${DIR_NAME}/bin
    chmod 540 -Rf ${BMA_DIR}/${DIR_NAME}/lib
    chmod 440 -Rf ${BMA_DIR}/${DIR_NAME}/lib64
    chmod 540 -Rf ${BMA_DIR}/${DIR_NAME}/script
    chmod 440 -Rf ${BMA_DIR}/${DIR_NAME}/config
    chmod 640 -Rf ${BMA_DIR}/${DIR_NAME}/log
    chmod 540 -Rf ${BMA_DIR}/${DIR_NAME}/tools

    find . -name "*.sh" -exec chmod 540 {} \; 2>/dev/null
    find . -name "*.py" -exec chmod 540 {} \; 2>/dev/null
    find . -name "*.ini" -exec chmod 440 {} \; 2>/dev/null
    find . -name "*.cfg" -exec chmod 440 {} \; 2>/dev/null
    find . -name "*.xml" -exec chmod 440 {} \; 2>/dev/null
    find . -name "*.json" -exec chmod 440 {} \; 2>/dev/null

    handle_oam_function
    
    # Create link to ${BMA_DIR}/${DIR_NAME}/bin/ibmacli in directory /usr/bin
    ln -sf "${BMA_DIR}/${DIR_NAME}/bin/ibmacli" "${IBMACLI_LINK_PATH}"

    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/iBMA.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/Monitor.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/archive.cfg
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/syslog.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/HarddiskIOAnalysis.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/SenseCode.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/EthernetInterface.ini
    chmod 640 ${BMA_DIR}/${DIR_NAME}/lib/Linux/config/WatchDog.ini

    if [ -e "${BMA_DIR}/${DIR_NAME}/lib/Linux/config/DSVerification.ini" ]; then
        chmod 640 ${BMA_DIR}/${DIR_NAME}/lib/Linux/config/DSVerification.ini
    fi

    if [ -e "${BMA_DIR}/${DIR_NAME}/lib/Linux/config/Product.ini" ]; then
        chmod 640 ${BMA_DIR}/${DIR_NAME}/lib/Linux/config/Product.ini
    fi

    chmod +x ${BMA_DIR}/${DIR_NAME}/lib/Linux/upgrade/CMSVerify

    if [ -e "${BMA_DIR}/${DIR_NAME}/bin/hwkbox" ]; then
        chmod 550 ${BMA_DIR}/${DIR_NAME}/bin/hwkbox
    fi
}

#*****************************************************************************
# Prototype    : fix_file_contexts
# Description  : fix file contexts.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/10/30
#    Modification : Created function
#*****************************************************************************
function fix_file_contexts()
{
    local ret=""

    ret=$(restorecon -R ${BMA_DIR}/${DIR_NAME} 2>&1)
    if [ 0 -ne $? ]; then
        LOG_ERROR "Failed to fix contexts" "HIDE"
        LOG_ERROR "${ret}" "HIDE"
    fi
}

#*****************************************************************************
# Prototype    : register_service
# Description  : register iBMA service.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function register_service()
{
    if [ false == ${OS_SUPPROT_SYSTEMCTL} ] ; then
        cp -rf ${BMA_DIR}/${DIR_NAME}/${SERVICE_NAME}.sh  ${SERVICE_SCRIPT}
        chmod 540 ${SERVICE_SCRIPT}

        local res1 res2
        # set startup level.
        res1=$(chkconfig --level 2345 ${SERVICE_NAME} on 2>&1) || \
        res2=$(update-rc.d ${SERVICE_NAME} defaults 2>&1)
        if [ $? == 0 ]; then
            LOG_INFO "The ${SERVICE_NAME} service registered successfully."
        else
            LOG_ERROR "res1=${res1}" "HIDE"
            LOG_ERROR "res2=${res2}" "HIDE"
            LOG_ERROR "Failed to register ${SERVICE_NAME} service. Exiting ..."
            return 1
        fi

        cp -rf ${BMA_DIR}/${DIR_NAME}/${BOBSERVICE_NAME}.sh  ${BOBSERVICE_SCRIPT}
        chmod 540 ${BOBSERVICE_SCRIPT}
        # set startup level.
        res1=$(chkconfig --level 2345 ${BOBSERVICE_NAME} off 2>&1) || \
        res2=$(update-rc.d ${BOBSERVICE_NAME} defaults 2>&1)
        if [ $? == 0 ]; then
            LOG_INFO "The ${BOBSERVICE_NAME} service registered successfully."
        else
            LOG_ERROR "res1=${res1}" "HIDE"
            LOG_ERROR "res2=${res2}" "HIDE"
            LOG_ERROR "Failed to register ${BOBSERVICE_NAME} service. Exiting ..."
            return 1
        fi

        return 0
    else
        #is use systemctl, system process
        system_ctl_proc
        if [ $? == 0 ]; then
            LOG_INFO "The ${SERVICE_NAME} service registered successfully."
            return 0
        else
            LOG_ERROR "Failed to register ${SERVICE_NAME} service. Exiting ..."
            return 1
        fi
    fi
}

#*****************************************************************************
# Prototype    : record_upgrade_log
# Description  : record upgrade log.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/09/11
#    Modification : Created function
#
#*****************************************************************************
function record_upgrade_log()
{
    local system_date=$(date -d today +"%Y-%m-%d %H:%M:%S")
    echo "[ ${system_date} ] $1" 2> /dev/null > "${IBMA_UPGRADE_LOG_PATH}"
}

#*****************************************************************************
# Prototype    : update_config
# Description  : update config file.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/12/30
#    Modification : Created function
#
#*****************************************************************************
function update_config()
{
    if [ -d "${BMA_DIR}/${DIR_NAME}/log/backup/config" ]; then
        # copy old WatchDog.ini to new config directory if old WatchDog.ini exists
        if [ -e "${BMA_DIR}/${DIR_NAME}/log/backup/config/WatchDog.ini" ] && [ -d ${BMA_DIR}/${DIR_NAME}/lib/Linux/config ] ; then
            cp -f ${BMA_DIR}/${DIR_NAME}/log/backup/config/WatchDog.ini ${BMA_DIR}/${DIR_NAME}/lib/Linux/config/
        fi

        # copy old bob.ini to new config directory if old bob.ini exists
        if [ -e "${BMA_DIR}/${DIR_NAME}/log/backup/config/bob.ini" ] && [ -d "${BMA_DIR}/${DIR_NAME}/tools/bob" ] ; then
            cp -f ${BMA_DIR}/${DIR_NAME}/log/backup/config/bob.ini ${BMA_DIR}/${DIR_NAME}/tools/bob/
        fi

        # 3 param, $1 is old version iBMA log directory, $2 is new version iBMA log directory
        res=$(${BMA_DIR}/${DIR_NAME}/bin/UpdateCfg "${BMA_DIR}/${DIR_NAME}/log/backup/config" ${BMA_DIR}/${DIR_NAME}/config 0 2>&1)
        if [ $? -eq 0 ]; then
            LOG_INFO "Update config return=${res}" "HIDE"
            ret=$(echo "$res"|grep -i "iBMA Version has no change" -q 2>/dev/null)
            if [ $? -eq 0 ]; then
                LOG_INFO "The same version of iBMA is already installed."
                record_upgrade_log "The same version of iBMA is already installed."
                exit 0
            else
                ret=$(echo "$res"|grep -i "The iBMA version is not supported" -q 2>/dev/null)
                if [ $? -eq 0 ]; then
                    LOG_INFO "The iBMA version is not supported."
                    record_upgrade_log "The iBMA version is not supported."
                    exit 1
                fi

                ret=$(echo "$res"|grep -i "Old version config not exist" -q 2>/dev/null)
                if [ $? -eq 0 ]; then
                    LOG_INFO "Old version config not exist." "HIDE"
                fi
            fi
        else
            rm -rf "${BMA_DIR}/${DIR_NAME}/log/backup"
            LOG_ERROR "res1=${res}"
            LOG_ERROR "Failed to update config. Exiting ..."
            LOG_ERROR "${SERVICE_NAME} upgraded failed."
            record_upgrade_log "${SERVICE_NAME} upgraded failed."
            exit 1
        fi
    fi
}

#*****************************************************************************
# Prototype    : update_log_path
# Description  : update log path.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/07/19
#    Modification : Created function
#
#*****************************************************************************
function update_log_path()
{
    local cfg_file="${BMA_DIR}/${DIR_NAME}/config/archive.cfg"
    local log_path=$(read_from_file ${cfg_file} "LogConfig" "logpath")
    # delete spaces before and after
    log_path=$(echo ${log_path} | awk '$1=$1')
    if [ "${log_path}" != "" ]; then
        mkdir -p ${log_path}
        [ -d "${log_path}" ] && rm -rf ${log_path}
        mv ${BMA_DIR}/${DIR_NAME}/log ${log_path}
        ln -sf ${log_path} ${BMA_DIR}/${DIR_NAME}/log
    fi
}

# is support systemctl
is_support_systemctl

# register service
register_service

# update config file
update_config

# update log path
update_log_path

# handle file.
handle_dile

# create file named auto_select_https when need not to keep the user's choice
# Redfish service will decide according to this file whether to use HTTPS protocol when iBMC supports HTTPS
AUTO_SELECT_HTTPS_FILE="${BMA_DIR}/${DIR_NAME}/log/auto_select_https"

if [ -e ${AUTO_SELECT_HTTPS_FILE} ]; then
    rm -f ${AUTO_SELECT_HTTPS_FILE}
fi

# ensure that iBMA can also automatically switch to HTTPS when the rpm package is installed separately.
# $1 is 1 when install rpm package, $1 is 2 when upgrade rpm package
if [ "1" == "$1" ]; then
    touch ${AUTO_SELECT_HTTPS_FILE}
    chmod 440 ${AUTO_SELECT_HTTPS_FILE}
fi

# success.
exit 0
