#!/bin/bash
#
# chkconfig: - 90 10
# description: Activates/Deactivates restful api agent service
#
### BEGIN INIT INFO
# Provides: iBMA 2.0
# Default-Start: 2 3 4 5
# Default-Stop: -
# Required-Start: $network
### END INIT INFO

#******************************************************************************
# Copyright (C), 2015-2021, Huawei Tech. Co., Ltd.
# File : iBMA.sh
# 1.iBMA service script
#
#  History       :
#  1.Date        : 2016/12/26
#    Modification: Created file
#  2.Date        : 2017/01/12
#    Modification: customize all exit values.
#                  0        success.
#                  1        failed.
#                  2        parameter is invalid.
#  3.Date        : 2017/01/16
#    Modification: check parameter only when doing start.
#  4.Date        : 2021/06/07
#    Modification: Add the iBMC_veth_ip attribute to the iBMA.ini file.
#******************************************************************************

# [GLOBAL]

RET_VAL=0
ROOT_PATH=/opt/huawei
DIR_NAME="ibma"
CONFIG_FILE="${ROOT_PATH}/${DIR_NAME}/config/iBMA.ini"
MANAGER_CFG="${ROOT_PATH}/${DIR_NAME}/config/Manager.cfg"
CHAR_DEVICE=""
SERVICE_NAME="iBMA"
HTTP_USER=""
HTTP_PORT=0
SOCKET_PORT=0
MANAGER_PATH=""
MONITOR_PATH=""
HTTPD_PATH=""
INTERPRETER=""
LOGFILE_PATH="log/iBMA_manager.log"
ENABLE_KBOX=false
KBOX_DEV="/dev/kbox"
USB_DEV="/dev/lcd0"
NIC=""
NETWORK_TYPE=veth
NETWORK_TYPE_VETH=veth
NETWORK_TYPE_CDEV=cdev
NETWORK_TYPE_USB=usb
NETWORK_TYPE_AUTO=auto
NETWORK_CDEV_NAME="/dev/net_cdev"
NETWORK_SETUP_TOOL="${ROOT_PATH}/${DIR_NAME}/bin/manager"
SUPPORT_SYSTEMCTL_PROCESS_RESTART=false
OWNER_FILE_PATH="${ROOT_PATH}/${DIR_NAME}/log/owner"
USER_ITEM="iBMA_user"

IBMC_IP="iBMC_ip"
IBMC_VETH_IP="iBMC_veth_ip"
IBMA_SECTION="iBMA_System"

LOG_DIR="${ROOT_PATH}/${DIR_NAME}/log"

# Solve Monitor panic error
export STORELIB_INI_FILE_PATH="${ROOT_PATH}/${DIR_NAME}/config"
# [GLOBAL END]

if [ ! -d "${LOG_DIR}" ]; then
    mkdir -p ${LOG_DIR}
    #Set directory permissions=640
    chmod 640 ${LOG_DIR}
fi

while [[ $ROOT_PATH == */ ]] ; do
    ROOT_PATH=${ROOT_PATH%/}
done

# check if this is the root
if [ $(/usr/bin/id -u) != 0 ] ; then
    echo "$0: This script must be run as root."
    exit 1
fi

if [ ! -f "${ROOT_PATH}/${DIR_NAME}/lib/Linux/log.sh" ]; then
    echo "Could not find 'log.sh'"
    exit 127
fi

cd "${ROOT_PATH}/${DIR_NAME}"

source "${ROOT_PATH}/${DIR_NAME}/lib/Linux/log.sh"

function usage ()
{
    LOG_INFO $"Usage: $0 {start|stop|status|restart}"
}

#*****************************************************************************
# Prototype    : init_default_param
# Description  : read config file, init global variables.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function init_default_param()
{
    HTTP_USER=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_user")
    HTTP_PORT=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_http_server_port")
    SOCKET_PORT=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_socket_port")
    CHAR_DEVICE=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_cdev")
    ENABLE_KBOX=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_kbox")
    NETWORK_TYPE=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_network_type")
    NIC=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_nic")
    MANAGER_PATH=$(read_from_file ${MANAGER_CFG} "Manager" "manager")
    MONITOR_PATH=$(read_from_file ${MANAGER_CFG} "Manager" "monitor")
    HTTPD_PATH=$(read_from_file ${MANAGER_CFG} "Manager" "httpd")
    INTERPRETER=$(read_from_file ${MANAGER_CFG} "Manager" "interpreter")
}

#*****************************************************************************
# Prototype    : check_script_file
# Description  : check the if a script exists and executable:
# Parameter:
#   input:  $1 script path, $2 if check executable is needed.
#   output: NA
# Return Value : 1 on error, 0 success.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function check_script_file()
{
    if [ ! -e "${1}" ]; then
        LOG_ERROR "File ${1} does not exist."
        return 1;
    fi

    if [ false != ${2} ]; then
        if [ ! -x "${1}" ]; then
            LOG_ERROR "File ${1} is not executable."
            return 1;
        fi
    fi

    return 0;
}

#*****************************************************************************
# Prototype    : save_to_config_file
# Description  : save user choice to config file.
# Parameter:
#   input:  $1: which section,  $2: which item,  $3: new value.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2021/06/04
#    Modification : Created function
#
#*****************************************************************************
function save_to_config_file()
{
    section=$1; item=$2; value=$3

    sed -i "/^${item}=/s/=.*/=${value}/" ${CONFIG_FILE}
    # recheck
    old_value=$(read_from_file ${CONFIG_FILE} ${section} ${item})
    if [ ${old_value} == ${value} ] ; then
        LOG_DEBUG "Success to set the ${item} to ${value}." "HIDE"
    else
        LOG_ERROR "Set '${item}' to '${value}' failed." "HIDE"
        LOG_ERROR "Failed to save the configuration. Exiting ..."
        exit 1
    fi
}

#*****************************************************************************
# Prototype    : kill_all_remaining_process
# Description  : kill all remaining process using:
#                        MANAGER_PATH   MONITOR_PATH   HTTPD_PATH.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2017/10/09
#    Modification : Created function
#*****************************************************************************
function kill_all_remaining_process()
{
    local ret1=""
    local ret2=""

    ret1=$(pkill -KILL -f "${ROOT_PATH}/${DIR_NAME}/${HTTPD_PATH}" 2>&1) || \
    ret2=$(pkill -SIGKILL -f "${ROOT_PATH}/${DIR_NAME}/${HTTPD_PATH}" 2>&1)
    LOG_INFO "kill ${HTTPD_PATH}, ret1=${ret1}, ret2=${ret2}." "HIDE"

    ret1=$(pkill -KILL -f "${ROOT_PATH}/${DIR_NAME}/${MONITOR_PATH}" 2>&1) || \
    ret2=$(pkill -SIGKILL -f "${ROOT_PATH}/${DIR_NAME}/${MONITOR_PATH}" 2>&1)
    LOG_INFO "kill ${MONITOR_PATH}, ret1=${ret1}, ret2=${ret2}." "HIDE"

    return 0
}

#*****************************************************************************
# Prototype    : check_config_param
# Description  : check the following parameters:
#                        HTTP_USER HTTP_PORT SOCKET_PORT
#                        NIC CHAR_DEVICE
#                        MANAGER_PATH MONITOR_PATH HTTPD_PATH.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2021/06/07
#    Modification : Integrate the iBMC_ip and iBMC_veth_ip attributes.
#
#*****************************************************************************
function check_config_param()
{
    local ret1=""
    local ret2=""
    # check scripts.
    local test_file_exec=false
    if [ -z "${INTERPRETER}" ]; then
        # no interpreter found, run script directly, should test if it's executable.
        test_file_exec=true
    fi

    check_script_file "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" ${test_file_exec}
    if [ $? -ne 0 ]; then
        return 1;
    fi

    check_script_file "${ROOT_PATH}/${DIR_NAME}/${MONITOR_PATH}" ${test_file_exec}
    if [ $? -ne 0 ]; then
        return 1;
    fi

    check_script_file "${ROOT_PATH}/${DIR_NAME}/${HTTPD_PATH}" ${test_file_exec}
    if [ $? -ne 0 ]; then
        return 1;
    fi

    kill_all_remaining_process

    iBMC_veth_ip=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} ${IBMC_VETH_IP})
    iBMC_ip=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} ${IBMC_IP})
    if [ "${iBMC_veth_ip}" != "${iBMC_ip}" ]; then
        save_to_config_file ${IBMA_SECTION} ${IBMC_IP} ${iBMC_veth_ip}
    fi

    # user should not be empty.
    if [ -z "${HTTP_USER}" ]; then
        LOG_ERROR "Redfish server user name is empty."
        return 1
    fi

    # check if HTTP_USER exist.
    id "${HTTP_USER}" 2>/dev/null | grep "(${HTTP_USER})" -q
    if [ $? -ne 0 ]; then
        LOG_ERROR "Redfish server user ${HTTP_USER} does not exist."
        return 1
    fi

    # HTTP_PORT should between 1024~65535.
    if [ ! "${HTTP_PORT}" -ge 1024 ] >& /dev/null || \
       [ ! "${HTTP_PORT}" -le 65535 ] >& /dev/null; then
        LOG_ERROR "Redfish server port ${HTTP_PORT} is invalid."
        return 1
    fi

    # check if HTTP_PORT is in use.
    ret1=$(lsof -i TCP:"${HTTP_PORT}" 2>&1)
    if [ $? -eq 0 ]; then
        LOG_ERROR "Redfish server port ${HTTP_PORT} is in use."
        LOG_ERROR "${ret1}." "HIDE"
        return 1
    fi

    # check SOCKET_PORT
    if [ ! "${SOCKET_PORT}" -ge 1024 ] >& /dev/null || \
       [ ! "${SOCKET_PORT}" -le 65535 ] >& /dev/null; then
        LOG_ERROR "Socket server port ${SOCKET_PORT} is invalid."
        return 1
    fi

    # check if SOCKET_PORT is in use.
    ret1=$(lsof -i TCP:"${SOCKET_PORT}" 2>&1)
    if [ $? -eq 0 ]; then
        LOG_ERROR "Socket server port ${SOCKET_PORT} is in use."
        LOG_ERROR "${ret1}." "HIDE"
        return 1
    fi

    # nic name non-empty.
    if [ -z "${NIC}" ]; then
        LOG_ERROR "NIC name is empty."
        return 1
    fi

    if [ "${NETWORK_TYPE}" == "${NETWORK_TYPE_USB}" ]; then
        return 0
    fi

    if [ "${NETWORK_TYPE_VETH}" == "${NETWORK_TYPE}" ]; then
        # nic should exist.
        if [ ! -e "/sys/class/net/${NIC}" ]; then
            LOG_ERROR "NIC ${NIC} does not exist."
            return 1
        fi

        # set nic up.
        ret1=$(ip link set "${NIC}" up 2>&1) || ret2=$(ifconfig "${NIC}" up 2>&1)
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to bring up ${NIC}."
            LOG_ERROR "ret1=${ret1},ret2=${ret2}" "HIDE"
            return 1
        fi
    elif [ ! -c "${NETWORK_CDEV_NAME}" ]; then
        LOG_ERROR "Char device ${NETWORK_CDEV_NAME} does not exist."
        return 1
    fi

    # char device name non-empty.
    if [ -z "${CHAR_DEVICE}" ]; then
        LOG_ERROR "Char device name is empty."
        return 1
    fi

    # check if char device exists.
    if [ ! -e "${CHAR_DEVICE}" ]; then
        LOG_ERROR "Char device ${CHAR_DEVICE} does not exist."
        return 1
    fi

    # check if char device is in use.
    if lsof "${CHAR_DEVICE}" >& /dev/null; then
        LOG_ERROR "Char device ${CHAR_DEVICE} is in use."
        return 1
    fi
}

#*****************************************************************************
# Prototype    : check_pcie_device
# Description  : check if PCIe device is in use.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 success, otherwise failed.
#  History        :
#  1.Date         : 2017/09/09
#    Modification : Created function, add device driver check.
#*****************************************************************************
function check_pcie_device()
{
    local drv_name=""
    local pcie_device_dir=""
    local device_id="0x1710"
    local vendor_id="0x19e5"

    LOG_INFO "Checking PCIe device ..." "HIDE"

    device=$(grep "${device_id}" /sys/bus/pci/devices/*/device 2>/dev/null)
    if [ -z "${device}" ]; then
        LOG_INFO "DID ${device} not found." "HIDE"
        return 1
    fi

    pcie_device_dir=${device%/device:*}
    grep "${vendor_id}" ${pcie_device_dir}/vendor >& /dev/null
    if [ 0 != $? ]; then
        LOG_INFO "VID ${vendor_id} not found." "HIDE"
        return 1
    fi

    LOG_INFO "Checking if PCIe device is in use." "HIDE"

    drv_name=$(readlink "${pcie_device_dir}/driver")
    LOG_INFO "PCIe device driver is [${drv_name}]" "HIDE"
    drv_name=${drv_name##*/}

    if [ -z "${drv_name}" ] || [[ ${drv_name} == *edma_drv ]]; then
        return 0
    fi

    LOG_ERROR "PCIe device is being used by ${drv_name} driver."
    LOG_ERROR "You need to unload it first."
    exit 1
}

#*****************************************************************************
# Prototype    : try_create_driver_links
# Description  : try to create driver links.
#                just like    weak-modules --add-kernel `uname -r`   do.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2017/07/08
#    Modification : Created function
#  2.Date         : 2018/03/07
#    Modification : Support SUSE and Ubuntu opearting system
#  3.Date         : 2019/07/29
#    Modification : Modify the driver directory lookup method
#*****************************************************************************
function try_create_driver_links()
{
    local driver_list=( "host_edma_drv.ko" \
                        "host_cdev_drv.ko" \
                        "host_veth_drv.ko" \
                        "cdev_veth_drv.ko" \
                        "host_kbox_drv.ko" )

    local kernel_version=$(uname -r)
    local module_path="/lib/modules/${kernel_version}/"
    local driver_weak_path="/lib/modules/${kernel_version}/weak-updates"

    host_cdev_drv=$(find ${module_path} -name "host_cdev_drv.ko" 2>/dev/null)
    if [ -n "${host_cdev_drv}" ]; then
        # already exists.
        return 0
    fi

    major_version=$(echo ${kernel_version/-*})
    host_edma_drv_file=$((ls /lib/modules/${major_version}*/extra/iBMA_driver/host_edma_drv.ko 2>/dev/null || \
                         ls /lib/modules/${major_version}*/updates/iBMA_driver/host_edma_drv.ko 2>/dev/null || \
                         ls /lib/modules/${major_version}*/updates/ibmadriver/host_edma_drv.ko 2>/dev/null) | head -n 1)

    if [ -z "${host_edma_drv_file}" ]; then
        LOG_INFO "No driver modules found." "HIDE"
        return 1
    fi

    driver_path=${host_edma_drv_file%/host_edma_drv.ko}

    if [ ! -e ${driver_weak_path} ]; then
        mkdir -p ${driver_weak_path}
    fi

    for drv in ${driver_list[*]}; do
        rlt=$(ln -sf ${driver_path}/${drv} ${driver_weak_path}/${drv} 2>&1)
        LOG_INFO "${driver_weak_path}/${drv} -> ${driver_path}/${drv},rlt:${rlt}" "HIDE"
    done

    rlt=$(depmod -a 2>&1)
    LOG_INFO "depmod result:${rlt}" "HIDE"

    return 0
}

#*****************************************************************************
# Prototype    : modprobe_driver
# Description  : modprobe driver needed by iBMA.
# Parameter:
#   input:  driver.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2018/2/7
#    Modification : Created function
#*****************************************************************************
function modprobe_driver()
{
    local not_found_pattern="not found."
    local drv="${1}"

    if [ -e /etc/modprobe.d/unsupported-modules ] || [ -e /etc/modprobe.d/10-unsupported-modules.conf ] ; then
        res=$(modprobe "${drv}" --allow-unsupported-modules 2>&1)
        if [ $? -ne 0 ]; then
            res=$(modprobe "${drv}" 2>&1)
        fi
    else
        res=$(modprobe "${drv}" 2>&1)
    fi

    if [ $? -ne 0 ]; then
        if [[ ${res} == *${not_found_pattern}* ]]; then
            LOG_ERROR "${SERVICE_NAME} driver ${drv} is missing."
        else
            LOG_ERROR "modprobe driver ${drv} Failed."
        fi
        LOG_ERROR "res=${res}" "HIDE"
        return 1
    fi

    return 0
}

#*****************************************************************************
# Prototype    : load_drivers
# Description  : load drivers needed by iBMA.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2017/07/08
#    Modification : add function try_create_driver_links.
#  3.Date         : 2017/07/19
#    Modification : remove modules if load failed, then try again.
#  4.Date         : 2019/09/23
#    Modification : Move the action of creating a soft link to the driver files
#                   to the start function.
#*****************************************************************************
function load_drivers()
{
    local load_successfully=true
    local driver_list=( "host_edma_drv" \
                        "host_cdev_drv" )

    # check device.
    check_pcie_device
    if [ 0 != $? ]; then
        LOG_ERROR "The PCIe device is disabled."
        LOG_ERROR "Enable Black Box on the iBMC first."
        exit 1
    fi

    for drv in ${driver_list[*]}; do
        modprobe_driver "${drv}"
        if [ $? -ne 0 ]; then
            load_successfully=false
            break
        fi
    done

    if ${load_successfully}; then
        return 0
    fi

    LOG_INFO "Failed to load drivers, try again after remove them." "HIDE"

    local count=${#driver_list[@]}
    while [ $count -gt 0 ]; do
        let count--
        res=$(rmmod "${driver_list[$count]}" 2>&1)
        LOG_INFO "rmmod ${driver_list[$count]}:${res}" "HIDE"
    done

    for drv in ${driver_list[*]}; do
        modprobe_driver "${drv}"
        if [ $? -ne 0 ]; then
            return 1
        fi
    done

    return 0
}

#*****************************************************************************
# Prototype    : setup_network_driver
# Description  : setup network driver.
# Parameter: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2019/05/10
#    Modification : if not configurable, use 'veth'; if it's auto, check modules
#  2.Date         : 2019/09/11
#    Modification : add executable permission for owner of manager
#  3.Date         : 2019/09/20
#    Modification : load host_veth_drv directly if manager does not exists.
#  4.Date         : 2020/02/19
#    Modification : if there is no cdev_veth_drv module in the system, use veth
#                   mode. Add notice in the log
#*****************************************************************************
function setup_network_driver()
{
    local res;
    # auto mode, load veth first.
    if [ -z "${NETWORK_TYPE}" ]; then
        NETWORK_TYPE=${NETWORK_TYPE_VETH}
    elif [ "${NETWORK_TYPE_AUTO}" == "${NETWORK_TYPE}" ]; then
        res=$(modinfo "cdev_veth_drv" 2>&1)
        if [ $? -ne 0 ]; then
            LOG_INFO "Cdev mode is not supported. Use veth mode."
            NETWORK_TYPE=${NETWORK_TYPE_VETH}
        elif [ -e "/sys/module/host_veth_drv" ]; then
            NETWORK_TYPE=${NETWORK_TYPE_VETH}
        else
            NETWORK_TYPE=${NETWORK_TYPE_CDEV}
        fi
    elif [ "${NETWORK_TYPE_VETH}" != "${NETWORK_TYPE}" -a \
           "${NETWORK_TYPE_CDEV}" != "${NETWORK_TYPE}"  -a \
           "${NETWORK_TYPE_USB}" != "${NETWORK_TYPE}" ]; then
        LOG_ERROR "Invalid network type ${NETWORK_TYPE}"
        return 1
    fi

    # send ipmi to bmc
    if [ ${NETWORK_TYPE} == ${NETWORK_TYPE_CDEV} ]; then
        ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x51 0x01 &>/dev/null
    elif [ ${NETWORK_TYPE} == ${NETWORK_TYPE_VETH} ]; then
        ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x51 0x00 &>/dev/null
    fi

    if [ -e ${NETWORK_SETUP_TOOL} ]; then
        # error log should be printed to screen.
        chmod u+x ${NETWORK_SETUP_TOOL}
        ${NETWORK_SETUP_TOOL} switch ${NETWORK_TYPE}
        return $?
    else
        modprobe_driver "host_veth_drv"
    fi
}

#*****************************************************************************
# Prototype    : handle_kbox
# Description  : handle kbox configuration: load driver if enabled.
# Parameter:
#   input:  current EDMA loaded module string.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2019/05/08
#    Modification : fix /dev/kbox conflicts in Euler OS.
#  3.Date         : 2019/08/26
#    Modification : if hwkbox does not exists, we do not handle kbox.
#*****************************************************************************
function handle_kbox()
{
    local res;
    local drv_str="${1}"

    if [ ! -e "${ROOT_PATH}/${DIR_NAME}/bin/hwkbox" ]; then
        return 0
    fi

    chmod 550 "${ROOT_PATH}/${DIR_NAME}/bin/hwkbox"

    if [[ ${drv_str} == *kbox_drv* ]] && ! ${ENABLE_KBOX} ; then
        res=$(rmmod "host_kbox_drv" 2>&1)
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to stop Black Box, try again later."
            LOG_ERROR "res=${res}" "HIDE"
            return 1;
        else
            LOG_INFO "Stop Black Box successfully."
        fi
    elif [[ ! ${drv_str} == *kbox_drv* ]] && ${ENABLE_KBOX} ; then
        service kdump status >& /dev/null
        if [ $? -eq 0 ]; then
            LOG_ERROR "Failed to start Black Box."
            LOG_ERROR "You need to stop kdump service first."
            return 1;
        fi

        if [ -e "${KBOX_DEV}" ]; then
            LOG_ERROR "Failed to start Black Box."
            LOG_ERROR "You need to remove ${KBOX_DEV} first."
            return 1;
        fi

        modprobe_driver "host_kbox_drv"
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to start Black Box, try again later."
            return 1;
        else
            LOG_INFO "Start Black Box successfully."
        fi
    fi

    return 0;
}

#*****************************************************************************
# Prototype    : record_manager_pid
# Description  : record manager process pid to /opt/huawei/ibma/manager.pid.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2018/05/12
#    Modification : Created function
#*****************************************************************************
function record_manager_pid()
{
    local ret=1
    local manager_pid=$1
    if [ ! -z "${manager_pid}" ]; then
        if [ ! -f "${ROOT_PATH}/${DIR_NAME}/lib/Linux/manager.pid" ]; then
            touch "${ROOT_PATH}/${DIR_NAME}/lib/Linux/manager.pid"
            chmod 600 "${ROOT_PATH}/${DIR_NAME}/lib/Linux/manager.pid"
        fi
        echo ${manager_pid} > "${ROOT_PATH}/${DIR_NAME}/lib/Linux/manager.pid"
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to record manager pid."
            return $ret
        fi

        ret=0
    fi

    return $ret
}

#*****************************************************************************
# Prototype    : check_systemctl_process_restart
# Description  : check systemctl process restart.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2018/05/17
#    Modification : Created function
#*****************************************************************************
function check_systemctl_process_restart()
{
    local restart_flag=""
    local system_dir='/usr/lib/systemd/system/'
    if [ ! -e "${system_dir}" ]; then
        # Ubuntu systemctl services save in the follow directory
        system_dir='/lib/systemd/system/'
    fi

    if [ -e "${system_dir}iBMA.service" ]; then
        restart_flag=$(cat "${system_dir}iBMA.service" | grep "#Restart=on-failure")
        if [ $? -eq 0 ]; then
            SUPPORT_SYSTEMCTL_PROCESS_RESTART=false
        else
            restart_flag=$(cat "${system_dir}iBMA.service" | grep "Restart=on-failure")
            if [ $? -eq 0 ]; then
                SUPPORT_SYSTEMCTL_PROCESS_RESTART=true
            fi
        fi
    fi

    return 0
}

#*****************************************************************************
# Prototype    : set_permission_for_user
# Description  : set permission of installed path for the specified user.
# Parameter:
#   input:  $1 user;$2 path.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2019/06/21
#    Modification : Created function
#  2.Date         : 2019/09/11
#    Modification : Don't change permissions of backup folder
#  3.Date         : 2019/09/11
#    Modification : Change owner of /opt/huawei to non-root user
#                   when flag file exists
#  4.Date         : 2020/08/26
#    Modification : Change the permission of the configuration file.
#  5.Date         : 2020/10/14
#    Modification : The chown command add -h argument.
#  6.Date         : 2020/10/29
#    Modification : Change the permission of the txt and img file.
#*****************************************************************************
function set_permission_for_user()
{
    local user=$1
    local path=$2

    # we can change owner of /opt/huawei to non-root user when flag file exists
    if [ -f "${OWNER_FILE_PATH}" ];then
        path="${ROOT_PATH}"
    fi

    setfacl --version >& /dev/null
    if [ 0 -ne $? ]; then
        LOG_ERROR "There is no 'setfacl' tool exist." "HIDE"

        # try to change owner of /opt/huawei/ibma from root to non-root user
        # when setfacl was not installed.
        res=$(chown -R -h "${user}" "${path}" 2>&1)
        if [ 0 -ne $? ]; then
            LOG_ERROR "Failed to change owner of ${path} to ${user}, ${res}." "HIDE"
            LOG_ERROR "You need to install 'setfacl' tool to run ${SERVICE_NAME} service."
            return 1
        fi

        find "${ROOT_PATH}/${DIR_NAME}/" -type d -name "socket" -exec chown -R -h root {} \; 2>/dev/null
        find "${ROOT_PATH}/${DIR_NAME}/" -type d -name "https" -exec chown -R -h root {} \; 2>/dev/null
        return 0
    fi

    setfacl -m u:${HTTP_USER}:rx "${ROOT_PATH}/" >/dev/null 2>&1
    res=$(setfacl -R -m u:${HTTP_USER}:r "${ROOT_PATH}/${DIR_NAME}/" 2>&1)
    if [ 0 -ne $? ]; then
        LOG_ERROR "Failed to set ACL rules for Redfish server user ${HTTP_USER}, ${res}" "HIDE"

        # create flag file /opt/huawei/ibma/log/owner to
        # indicate the /opt/huawei is created by iBMA
        OWNER_FILE_PATH="${BMA_DIR}/${DIR_NAME}/log/owner"

        # try to change owner of /opt/huawei/ibma from root to non-root user
        # when setfacl was not installed.
        res=$(chown -R -h "${user}" "${path}" 2>&1)
        if [ 0 -ne $? ]; then
            LOG_ERROR "Failed to change owner of ${path} to ${user}, ${res}." "HIDE"
            return 1
        fi

        find "${ROOT_PATH}/${DIR_NAME}/" -type d -name "socket" -exec chown -R -h root {} \; 2>/dev/null
        find "${ROOT_PATH}/${DIR_NAME}/" -type d -name "https" -exec chown -R -h root {} \; 2>/dev/null
        return 0
    fi

    mv "${ROOT_PATH}/${DIR_NAME}/log/backup" "${ROOT_PATH}/${DIR_NAME}/backup" 2>/dev/null

    find ${ROOT_PATH}/${DIR_NAME}/ -type d -exec setfacl -m u:${HTTP_USER}:rx {} \; 2>/dev/null
    find bin -type f -exec setfacl -m u:${HTTP_USER}:rx {} \; 2>/dev/null

    find . -name "*.ini" -exec setfacl -R -m u:${HTTP_USER}:r {} \; 2>/dev/null
    find . -name "*.xml" -exec setfacl -R -m u:${HTTP_USER}:r {} \; 2>/dev/null
    find . -name "*.json" -exec setfacl -R -m u:${HTTP_USER}:r {} \; 2>/dev/null
    find . -name "*.log.*" -exec setfacl -R -m u:${HTTP_USER}:- {} \; 2>/dev/null
    find . -name "Arc*.zip" -exec setfacl -R -m u:${HTTP_USER}:- {} \; 2>/dev/null
    find . -name "*.cfg" -exec setfacl -R -m u:${HTTP_USER}:r {} \; 2>/dev/null
    find . -name "*.txt" -exec setfacl -R -m u:${HTTP_USER}:- {} \; 2>/dev/null
    find . -name "*.img" -exec setfacl -R -m u:${HTTP_USER}:- {} \; 2>/dev/null

    # remove the acl permissions.
    find . -name "*.crt" -exec setfacl -x u:${HTTP_USER} {} \; 2>/dev/null
    find . -name "*.key" -exec setfacl -x u:${HTTP_USER} {} \; 2>/dev/null
    find . -name "socket" -exec setfacl -x u:${HTTP_USER} {} \; 2>/dev/null
    find . -name "https" -exec setfacl -x u:${HTTP_USER} {} \; 2>/dev/null

    mv "${ROOT_PATH}/${DIR_NAME}/backup" "${ROOT_PATH}/${DIR_NAME}/log/backup" 2>/dev/null

    return 0
}

function handle_ifcfg_veth_file()
{
    veth_cfg=$1
    config_network_type=$2
    old_onboot_option=$(cat ${veth_cfg} | grep "ONBOOT" | awk -F '=' '{print $2}')

    if [[ "${config_network_type}" == "cdev" && "${old_onboot_option}" == "yes" ]]; then
        sed -i "/^ONBOOT=/c ONBOOT=no" ${eth_cfg_dir}/${veth_cfg}
        sync
    elif [[ "${config_network_type}" == "veth" && "${old_onboot_option}" == "no" ]]; then
        sed -i "/^ONBOOT=/c ONBOOT=yes" ${eth_cfg_dir}/${veth_cfg}
        sync
    fi
}

# start function entry.
function start()
{
    local res;
    local drv_str;
    local counter=0

    create_default_user

    ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep -q
    if [ $? -eq 0 ]; then
        LOG_INFO "${SERVICE_NAME} service is already running"
        return 0
    fi

    RET_VAL=1

    if [[ ${NETWORK_TYPE} != "${NETWORK_TYPE_USB}" ]]; then
        # update weak modules before loading drivers.
        try_create_driver_links

        # check if char device driver installed.
        drv_str=$(cat /proc/modules | grep host_edma_drv)
        if [[ ! ${drv_str} == *cdev_drv* ]]; then
            LOG_INFO "Loading ${SERVICE_NAME} driver modules ..."
            load_drivers
        fi

        setup_network_driver
        if [ 0 -ne $? ]; then
            return 1
        fi

        drv_str=$(cat /proc/modules | grep host_edma_drv)
        if [[ ! ${drv_str} == *cdev_drv* ]] || [[ ! ${drv_str} == *veth_drv* ]]; then
            LOG_ERROR "Failed to load ${SERVICE_NAME} driver modules."
            return 1
        fi

        # if use cdev mode, don't load veth on boot
        veth_cfg="/etc/sysconfig/network-scripts/ifcfg-veth"
        config_network_type=$(read_from_file ${CONFIG_FILE} ${IBMA_SECTION} "iBMA_network_type")

        if [[ -f "${veth_cfg}" && "${config_network_type}" != "${NETWORK_TYPE_AUTO}" ]]; then
            handle_ifcfg_veth_file "${veth_cfg}" "${config_network_type}"
        fi
    elif [[ ${NETWORK_TYPE} == "${NETWORK_TYPE_USB}" ]]; then
        ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x51 0x02 &>/dev/null
        rmmod host_cdev_drv >& /dev/null
        if [ ! -e "${USB_DEV}" ];then
            LOG_ERROR "BMC not load ${NETWORK_TYPE_USB} driver ."
            return 1
        fi
    fi

    # check configuration parameters.
    check_config_param
    if [ $? -ne 0 ]; then
        return 1
    fi

    # config firewall rules
    if [ -f "${ROOT_PATH}/${DIR_NAME}/script/config_firewall_rules.sh" ]; then
        "${ROOT_PATH}/${DIR_NAME}/script/config_firewall_rules.sh"
    fi

    # we do not handle the return value.
    handle_kbox "${drv_str}"

    # set user access policy for iBMA files.
    if [ "root" != "${HTTP_USER}" ]; then
        LOG_INFO "Redfish server user is ${HTTP_USER}." "HIDE"

        set_permission_for_user "${HTTP_USER}" "${ROOT_PATH}/${DIR_NAME}"
        if [ 0 -ne $? ]; then
            return 1
        fi
    else
        res=$(setfacl -R -b "${ROOT_PATH}/" 2>&1)
        if [ 0 -ne $? ]; then
            LOG_ERROR "Failed to set ACL rules, ${res}" "HIDE"

            # try to change owner of /opt/huawei/ibma from root to non-root user
            # when set ACL rules for Redfish server user failed.
            res=$(chown -R -h root "${ROOT_PATH}/${DIR_NAME}" 2>&1)
            if [ 0 -ne $? ]; then
                LOG_ERROR "Failed to change owner of ${ROOT_PATH}/${DIR_NAME} to root, ${res}." "HIDE"
            fi
        fi
    fi

    if [ -e "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}" ]; then
        rm -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}"
    fi

    #ckeck SOCKET_PORT is shut down.
    while [ $counter -lt 60 ]; do
        res=$(netstat -tunlp 2>/dev/null | grep ":${SOCKET_PORT} ")
        if [ $? -eq 0 ]; then
            LOG_ERROR "TCP link is not completely shut down." "HIDE"
            sleep 2
            let counter++
            continue
        fi
        break
    done

    #check systemctl process restart
    check_systemctl_process_restart

    local start_log_file="${LOG_DIR}/iBMA_start.log"
    ${INTERPRETER} ${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH} >& ${start_log_file} &
    chmod 640 ${start_log_file}
    local manager_pid=$!
    # if has GPU resource, record manager process pid.
    if $SUPPORT_SYSTEMCTL_PROCESS_RESTART; then
        record_manager_pid $manager_pid
        if [ $? -ne 0 ]; then
            return 1;
        fi
    fi

    # wait for 30 seconds at most.
    counter=0
    while [ $counter -lt 60 ]; do
        sleep 0.5
        # check if it's still running.
        ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep -q
        local running_status=$?
        # still running, wait for log from manager.
        if [ -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}" ]; then
            RUNNING_LOG=$(cat "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}")
            if echo "${RUNNING_LOG}" | grep -i 'success' -q; then
                LOG_INFO "${SERVICE_NAME} service started successfully."
                rm -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}"
                RET_VAL=0
                return ${RET_VAL}
            elif [ $running_status -eq 0 ]; then
                # iBMA is still running, do not check failure status.
                let counter++
                continue
            elif echo "${RUNNING_LOG}" | grep -i 'failed\|error\|invalid\|exception' -q; then
                LOG_INFO "${RUNNING_LOG}"
                rm -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}"
                RET_VAL=1

                # when iBMA service start failed with non-root user
                echo "${RUNNING_LOG}" | grep -i 'Permission denied' -q
                if [ $? -eq 0 ] && [ "${HTTP_USER}" != root ]; then
                    LOG_HINT "Failed to start the Redfish service by user ${HTTP_USER}."
                    LOG_HINT "You can change ${USER_ITEM} in ${CONFIG_FILE} file to root or"
                    LOG_HINT "change the permission on the ${ROOT_PATH} directory "
                    LOG_HINT "so that user ${HTTP_USER} has the access permission "
                    LOG_HINT "and restart iBMA service for the modification to take effect."
                    LOG_INFO "-----------------------------------------------------------------"
                fi
                return ${RET_VAL}
            else
                LOG_INFO "current content: ${RUNNING_LOG}" "HIDE"
            fi
        fi

        # if exit status is 0, it's still running.
        if [ $running_status -ne 0 ]; then
            START_FAILED_LOG=$(cat "${start_log_file}")
            LOG_ERROR "Failed to start ${SERVICE_NAME} service."
            LOG_ERROR "${START_FAILED_LOG}"
            return ${RET_VAL}
        fi

        let counter++
    done

    # failed, timeout or just exited.
    LOG_ERROR "Failed to start ${SERVICE_NAME} service."
    LOG_ERROR "Timeout waiting for response." "HIDE"

    return ${RET_VAL}
}

#*****************************************************************************
# Prototype    : resume_process
# Description  : resume process by pid.
# Parameter:
#   input:  $1 pid list (separated with ' ').
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2017/10/09
#    Modification : Created function
#*****************************************************************************
function resume_process()
{
    local pids=$1
    for pid in ${pids}; do
        if [ -z "${pid}" -o ! -f "/proc/${pid}/stat" ]; then
            LOG_ERROR "Process id ${pid} is invalid."
            continue
        fi

        status=$(cat "/proc/${pid}/stat" 2>/dev/null | awk '{print $3}')
        if [ -z "${status}" ]; then
            LOG_ERROR "Failed to get process ${pid} status."
            continue
        fi

        LOG_INFO "PID ${pid} status is ${status}." "HIDE"
        if [[ "T" == "${status}" ]]; then
            LOG_ERROR "process ${pid} is in abnormal state ${status}, resume it."
            res1=$(kill -CONT ${pid} 2>&1) || res2=$(kill -SIGCONT ${pid} 2>&1)
            if [ 0 -ne $? ]; then
                LOG_ERROR "Failed to resume process ${pid}."
                LOG_ERROR "res1=${res1},res2=${res2}."
            else
                LOG_INFO "Resume process ${pid} successfully."
            fi
        fi
    done

    return 0
}

#*****************************************************************************
# Prototype    : systemctl_reset_failed
# Description  : iBMA status reset failed.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : return 1 on error, otherwise 0.
#
#  History        :
#  1.Date         : 2018/05/17
#    Modification : Created function
#*****************************************************************************
function systemctl_reset_failed()
{
    local ret=1
    local res=""
    local ibma_status=""

    ibma_status=$(systemctl status iBMA | grep "Active:")
    if [ $? -eq 0 ]; then
        res=$(systemctl reset-failed iBMA 2>&1)
        if [ $? -eq 0 ]; then
            LOG_INFO "${SERVICE_NAME} service reset-failed successfully." "HIDE"
            ret=0
        else
            LOG_ERROR "Failed to reset-failed ${SERVICE_NAME} service, ${res}." "HIDE"
        fi
    fi

    return $ret
}

# stop function entry.
function stop()
{
    PID=$(ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep | awk '{print $2}')

    if [ -z "${PID}" ]; then
        LOG_INFO "${SERVICE_NAME} service is not running."
        return 0
    fi
    resume_process "${PID}"

    # try to delete old log file.
    if [ -e "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}" ]; then
        LOG_INFO "Delete log file ${LOGFILE_PATH}" "HIDE"
        rm -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}"
    fi

    # send SIGTERM signal, and wait.
    res1=$(kill -TERM ${PID} 2>&1) || res2=$(kill -SIGTERM ${PID} 2>&1)
    if [ $? -eq 0 ]; then
        LOG_INFO "Sent stop signal to ${SERVICE_NAME} process, now wait." "HIDE"
    else
        LOG_ERROR "Failed to send stop signal to ${SERVICE_NAME} process, try again later."
        LOG_ERROR "res1=${res1},res2=${res2}" "HIDE"
        RET_VAL=1
        return 1;
    fi

    #check systemctl process restart
    check_systemctl_process_restart

    # wait for 40 seconds at most.
    counter=0
    while [ $counter -lt 40 ]; do
        # check if it's still running.
        ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep -q

        # no process found, ie, stopped.
        if [ $? -ne 0 ] ; then
            LOG_INFO "${SERVICE_NAME} service stopped successfully."
            kill_all_remaining_process

            # iBMA status reset failed
            if $SUPPORT_SYSTEMCTL_PROCESS_RESTART; then
                systemctl_reset_failed
            fi
            return 0
        else
            # still running, try error log.
            if [ -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}" ]; then
                RET_VAL=1
                FAIL_LOG=$(cat "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}")
                # wait till we get a "." from the log, which means the end of file.
                if echo ${FAIL_LOG} | grep '\.' -q; then
                    LOG_ERROR "Failed to stop ${SERVICE_NAME} service. ${FAIL_LOG}"
                    rm -f "${ROOT_PATH}/${DIR_NAME}/${LOGFILE_PATH}"
                    return ${RET_VAL}
                else
                    LOG_INFO "Current content: ${FAIL_LOG}" "HIDE"
                fi
            fi
        fi
        let counter++
        sleep 1
    done

    RET_VAL=1
    LOG_ERROR "Failed to stop ${SERVICE_NAME} service, try again later."
    return ${RET_VAL}
}

# restart function entry.
function restart()
{
    local ret_code=1
    stop
    if [ $? -ne 0 ]; then
        LOG_ERROR "Failed to restart ${SERVICE_NAME} service."
        return
    fi

    start
    if [ $? -ne 0 ]; then
        LOG_ERROR "Failed to restart ${SERVICE_NAME} service."
        return
    fi

    ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep -q
    if [ $? -eq 0 ] ;then
        LOG_INFO "Restart ${SERVICE_NAME} service successfully."
    else
        RET_VAL=1
        LOG_ERROR "Failed to restart ${SERVICE_NAME} service."
    fi
}

# status function entry.
function status()
{
    ps -ef | grep -w "${ROOT_PATH}/${DIR_NAME}/${MANAGER_PATH}" | grep -v grep -q
    if [ $? -eq 0 ] ;then
        LOG_INFO "${SERVICE_NAME} service is running."
    else
        RET_VAL=3
        LOG_INFO "${SERVICE_NAME} service is not running."
    fi
}

#*****************************************************************************
# Prototype    : create_default_user
# Description  : create an ibma user, if already exists, exit the installation.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/04/24
#    Modification : Created function
#  2.Date         : 2019/09/20
#    Modification : Create ibma user with the shell mode /sbin/nologin
#
#*****************************************************************************
function create_default_user()
{
    local ibma_section="iBMA_System"
    local ibma_default_user_name="ibma"
    local ibma_user_desc="This is an ibma user"
    local user_file="/etc/passwd"

    HTTP_USER=$(read_from_file ${CONFIG_FILE} ${ibma_section} ${USER_ITEM})
    if [ "${ibma_default_user_name}" != "${HTTP_USER}" ]; then
        LOG_INFO "Redfish service user is ${HTTP_USER}, not ${ibma_default_user_name} user." "HIDE"
        return 0
    fi

    id "${ibma_default_user_name}" 2>/dev/null | grep "(${ibma_default_user_name})" -q
    if [ $? -eq 0 ]; then
        # Check if the ibma user was created by iBMA and the nologin property.
        cat "${user_file}" |grep -w "${ibma_user_desc}" | grep "/sbin/nologin" -q
        if [ $? -ne 0 ]; then
            LOG_ERROR "The user ${ibma_default_user_name} already exists, but it is not the default iBMA user."
            exit 2
        fi

        # If the ibma user was created by iBMA, then continue to use it
        LOG_INFO "${ibma_default_user_name} user already exists, not need to create." "HIDE"
        return 0
    fi

    # Create ibma user with the shell mode /sbin/nologin
    useradd -M -c "${ibma_user_desc}" "${ibma_default_user_name}" -s /sbin/nologin
    if [ $? -ne 0 ]; then
        LOG_ERROR "Failed to create ${ibma_default_user_name} user."
        exit 2
    fi

    LOG_INFO "Create ${ibma_default_user_name} user successfully."
    return 0
}

if [ ! -f "${CONFIG_FILE}" ]
then
    LOG_ERROR "The ${CONFIG_FILE} file does not exist."
    exit 1
fi

init_default_param

case "$1" in
    "stop")
        stop
        ;;
    "start")
        start
        ;;
    "status")
        status
        ;;
    "restart")
        restart
        ;;
    *)
        usage
        RET_VAL=2
        ;;
esac

exit $RET_VAL
