#!/bin/bash
#******************************************************************************
# Copyright (C), 2015-2017, Huawei Tech. Co., Ltd.
# File : install.sh
# 1.install iBMA2.0 driver.
#
#  History       :
#  1.Date        : 2017/01/25
#    Modification: Created file
#******************************************************************************

# get and cd to project dir
if [ -L ${0} ]; then
    REALFILE=$(readlink ${0})
else
    REALFILE=${0}
fi

CUR_DIR=$(dirname ${REALFILE})

cd ${CUR_DIR}
cd ..

G_WORK_DIR=$(pwd)
FORCE_MODE=false
SILENT_MODE=false
REBOOT_NEEDED=false
IPV4_MODE=false
DRIVER_DIR="${G_WORK_DIR}/drivers"
source "${G_WORK_DIR}/script/log.sh"
BMA_DRIVER_VERSION=""
PCIE_DEVICE_DIR=""

OS_NAME=""
PACKAGE_FILE=""
PACKAGE_MANAGER=0
IBMA_DRIVER_MODULES=""

readonly DRIVER_LIST=( "host_edma_drv" \
                       "host_cdev_drv" )

readonly DRIVER_LIST_ALL=( "host_edma_drv" \
                           "host_cdev_drv" \
                           "cdev_veth_drv" \
                           "host_veth_drv" \
                           "host_kbox_drv")

readonly OS_NAME_LIST="CentOS,RedHat,SUSE,Ubuntu,XenServer,OracleServer"

readonly DEB_OS_LIST=("Ubuntu" "Debian" "uos")

readonly ETC_FILE=( "/etc/neokylin-release" "NeoKylin" \
                    "/etc/bclinux-release" "bclinux" \
                    "/etc/openEuler-release" "openEuler" \
                    "/etc/euleros-release" "EulerOS" \
                    "/etc/oracle-release" "Oracle" \
                    "/etc/centos-release" "CentOS" \
                    "/etc/redhat-release" "RedHat" \
                    "/etc/SuSE-release" "SUSE" )

readonly OTHER_OS_FILE="/etc/os-release"

# Version of SUSE system is stored in /etc/os-release starting from SLES 15
readonly OTHER_OS_NAME_LIST=("Ubuntu" "Debian" "SUSE" "Kylin" "uos")

#*****************************************************************************
# Prototype    : is_pcie_detected
# Description  : detect PCIe device.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : return 0 if detected.
#
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#
#*****************************************************************************
function is_pcie_detected()
{
    local device_id="0x1710";
    local vendor_id="0x19e5";

    device=$(grep "${device_id}" /sys/bus/pci/devices/*/device 2>/dev/null)

    if [ -z "${device}" ]; then
        return 1;
    fi

    PCIE_DEVICE_DIR=${device%/device:*}
    grep "${vendor_id}" ${PCIE_DEVICE_DIR}/vendor >& /dev/null

    return $?;
}

#*****************************************************************************
# Prototype    : open_pcie_device
# Description  : Send IPMI command to iBMC, open PCIe device.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#
#*****************************************************************************
function open_pcie_device()
{
    local ret=0
    local ipmi_result=""
    local ipmi_module_loaded=true
    local ipmi_module_name="ipmi_si"
    local res_success="db 07 00 00 01"

    LOG_INFO "Enabling PCIe device ..."

    cat /proc/modules | grep "^${ipmi_module_name}" -q
    if [ $? -ne 0 ]; then
        ipmi_module_loaded=false
        ipmi_result=$(modprobe "${ipmi_module_name}" 2>&1)
        LOG_INFO "Loaded ${ipmi_module_name} module." "HIDE"
        LOG_INFO "ipmi_result=${ipmi_result}" "HIDE"
    fi

    ipmi_result=$(ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x0f 0x0c 0x01 0x01 0x00 0x00 0x01 2>&1)
    ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x10 0x0c 2>/dev/null | grep "${res_success}" -q
    ret=$?

    if [ "${ipmi_module_loaded}" == "false" ]; then
        LOG_INFO "Remove ${ipmi_module_name} module." "HIDE"
        rmmod "${ipmi_module_name}" >& /dev/null
    fi

    if [ ${ret} -ne 0 ]; then
        LOG_ERROR "Failed to enable the PCIe device."
        LOG_ERROR "Enable Black Box on the iBMC first."
        LOG_ERROR "ipmi result:${ipmi_result}" "HIDE"
        exit 1;
    else
        LOG_INFO "The PCIe device is enabled successfully."
        return 0;
    fi
}

#*****************************************************************************
# Prototype    : check_pcie_device
# Description  : check if PCIe device is in use.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 success, otherwise failed.
#  History        :
#  1.Date         : 2017/09/09
#    Modification : Created function, add device driver check.
#*****************************************************************************
function check_pcie_device()
{
    local drv_name=""

    LOG_INFO "Checking if PCIe device is in use." "HIDE"

    if [ ! -d "${PCIE_DEVICE_DIR}" ]; then
        LOG_ERROR "Failed to find PCIe device directory."
        exit 1
    fi

    drv_name=$(readlink "${PCIE_DEVICE_DIR}/driver")
    LOG_INFO "PCIe device driver is [${drv_name}]" "HIDE"
    drv_name=${drv_name##*/}

    if [ -z "${drv_name}" ] || [[ "${drv_name}" == *edma_drv ]]; then
        return 0
    fi

    LOG_ERROR "PCIe device is being used by ${drv_name} driver."
    LOG_ERROR "You need to unload it first."
    exit 1
}

#*****************************************************************************
# Prototype    : get_package_manager
# Description  : get driver package manager.
# Parameter:
#   input:  $1 package name
#   output: NA
# Return Value : exit on error.
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#*****************************************************************************
function get_package_manager()
{
    local package="$1"

    if [[ "${package}" == *.rpm ]]; then
        PACKAGE_MANAGER=0;
    elif [[ "${package}" == *.deb ]]; then
        PACKAGE_MANAGER=1;
    else
        LOG_ERROR "Driver package file name is invalid."
        exit 1;
    fi
}

#*****************************************************************************
# Prototype    : get_package_version
# Description  : Get driver version.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : exit on error.
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#*****************************************************************************
function get_package_version()
{
    local package="$1"

    # ***-$(uname -r)-version-OS.rpm
    BMA_DRIVER_VERSION=$(echo "${package##*$(uname -r)}" | awk -F "-" '{print $2}')

    if [ -z "${BMA_DRIVER_VERSION}" ]; then
        LOG_ERROR "Driver package file name is invalid."
        exit 1;
    fi
}

#*****************************************************************************
# Prototype    : get_sles_driver_package
# Description  : get sles driver by kernel
# Parameter:
#   input:  NA
#   output: NA
# Return Value : if no driver found return 1, else 0.
#
#  History        :
#  1.Date         : 2018/03/10
#    Modification : Created function
#  2.Date         : 2018/05/09
#    Modification : Driver support xen model
#  3.Date         : 2019/07/27
#    Modification : Only select the first driver if more than one are available.
#*****************************************************************************
function get_sles_driver_package()
{
    PACKAGE_MANAGER=0
    local kernel_version=$(uname -r)

    # match this version first
    minor_version=$(echo ${kernel_version} | grep -P ".*-\d+" -o)
    # get the first string before -
    major_version=$(echo ${kernel_version/-*})
    type_version=$(echo ${kernel_version/*-})

    # check if it is a UVP system
    if [ ! -f "/etc/uvp_version" ]; then
        # found driver by os version first
        os_info=$(cat /etc/SuSE-release 2>/dev/null || cat /etc/os-release 2>/dev/null)
        os_version=$(echo "${os_info}" | grep -i "VERSION *= *" | awk -F"=" '{print $2}' | tr -d '\r\n "-' | tr 'A-Z' 'a-z')
        patchlevel_version=$(echo "${os_info}" | grep -i "PATCHLEVEL *= *" | awk -F"=" '{print $2}' | tr -d '\r\n ')
        os_aarch=$(uname -p)

        if [ -z "${patchlevel_version}" ] || [ "${patchlevel_version}" == "0" ]; then
            last_str=".${os_aarch}.rpm"
        else
            last_str="sp${patchlevel_version}.${os_aarch}.rpm"
        fi

        match_str="${os_version}${last_str}"

        # use the standard version if possible.
        PACKAGE_FILE=$((ls ${DRIVER_DIR}/${OS_NAME}/*-${type_version}*${match_str} 2>/dev/null) | head -n 1)
    else
        OS_NAME="UVP"
        PACKAGE_FILE=$(ls ${DRIVER_DIR}/${OS_NAME}/*-${type_version}*.rpm 2>/dev/null)
    fi

    if [ ! -f "${PACKAGE_FILE}" ]; then
        return 1
    fi

    LOG_INFO "Selected driver package:[${PACKAGE_FILE}]" "HIDE"

    driver_name=$(basename "${PACKAGE_FILE}")
    # delete string before type version
    driver_version=$(echo "${driver_name##*${type_version}-}")
    # delete string after _k
    BMA_DRIVER_VERSION=$(echo "${driver_version%_k*}")

    return 0;
}

#*****************************************************************************
# Prototype    : get_available_driver_package
# Description  : get driver by kernel
# Parameter:
#   input:  NA
#   output: NA
# Return Value : if no driver found return 1, else 0.
#
#  History        :
#  1.Date         : 2017/07/15
#    Modification : Created function
#  2.Date         : 2018/03/07
#    Modification : delete the sub functions
#                   major_version match the first string before -.
#                   type_version match the least string after -.
#  3.Date         : 2019/07/27
#    Modification : Support for the Euler aarch64 version,
#                   which minor version may be empty.
#  4.Date         : 2020/12/10
#    Modification : Remove the major version make the kernel match more accuracy.
#  5.Date         : 2020/12/10
#    Modification : Modify the kernel matching rule of the EulerOS.
#*****************************************************************************
function get_available_driver_package()
{
    local kernel_version=$(uname -r)
    LOG_INFO "Kernel version is [${kernel_version}]" "HIDE"

    # match this version first
    minor_version=$(echo ${kernel_version} | grep -P ".*-[\da-zA-Z]+" -o)
    # get the first string before -
    major_version=$(echo ${kernel_version/-*})
    if [ -z "${minor_version}" ]; then
        minor_version=${major_version}
    fi
    type_version=""

    case "${OS_NAME}" in
        "CentOS"|"RedHat")
            # the kernel version like: 3.10.0-123.27.1.el7.x86_64
            # get the last two string after .
            type_version=$(echo ${kernel_version} | grep -P "\.[^.]*\.[^.]*$" -o)
            ;;
        "SUSE")
            # Special treatment for SUSE
            get_sles_driver_package
            return $?
            ;;
        "EulerOS")
            # the kernel version like: 3.10.0-327.55.58.94_14.x86_64
            # get the last string after .
            type_version=$(echo ${kernel_version} | grep -P "\.[^.]*$" -o)
            if [[ ${minor_version} =~ "vhulk" ]];then
                minor_version=${major_version}
            fi
            ;;
        *)
            # the kernel version like: 4.4.0-87-generic
            # get the last string after -
            type_version=$(echo ${kernel_version/*-})
            ;;
    esac

    # use the standard version if possible.
    PACKAGE_FILE=$((ls ${DRIVER_DIR}/${OS_NAME}/*-${minor_version}${type_version}-* 2>/dev/null || \
                   ls ${DRIVER_DIR}/${OS_NAME}/*-${minor_version}*${type_version}-* 2>/dev/null) | head -n 1)

    if [ ! -f "${PACKAGE_FILE}" ]; then
        return 1
    fi

    LOG_INFO "Selected driver package:[${PACKAGE_FILE}]" "HIDE"

    PACKAGE_MANAGER=0
    if [[ "${PACKAGE_FILE}" == *.deb ]]; then
        PACKAGE_MANAGER=1
    fi

    driver_name=$(basename "${PACKAGE_FILE}")
    if [ "${OS_NAME}" == "EulerOS" ]; then
        BMA_DRIVER_VERSION=$(echo "${driver_name#*${type_version}}" | awk -F "-" '{print $2}')
    else
        BMA_DRIVER_VERSION=$(echo "${driver_name##*${type_version}}" | awk -F "-" '{print $2}')
    fi

    return 0;
}

#*****************************************************************************
# Prototype    : check_pre_installed_driver
# Description  : check whether OS already has a pre-installed iBMA driver.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : if no pre-installed ibma driver found return 1, else 0.
#
#  History        :
#  1.Date         : 2021/07/19
#    Modification : Created function
#*****************************************************************************
function check_pre_installed_driver()
{
    modinfo "host_edma_drv" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        return 1
    fi

    case ${PACKAGE_MANAGER} in
        0)
            rpm -qa 2>/dev/null | grep "iBMA_driver" > /dev/null 2>&1
            if [ $? -ne 0 ]; then
                LOG_INFO "Your environment already has a pre-installed iBMA driver"
                return 0
            fi
            ;;
        1)
            dpkg -l 2>/dev/null | grep "ibmadriver" > /dev/null 2>&1
            if [ $? -ne 0 ]; then
                LOG_INFO "Your environment already has a pre-installed iBMA driver"
                return 0
            fi
            ;;
        *)
            LOG_ERROR "Failed to get package type."
            ;;
        esac

    return 1
}

#*****************************************************************************
# Prototype    : record_dkms_driver_installed
# Description  : record log about dkms driver installed
# Parameter:
#   input:  NA
#   output: NA
# Return Value : if no dkms driver found return 1, else 0.
#
#  History        :
#  1.Date         : 2021/02/14
#    Modification : Created function
#*****************************************************************************
function record_dkms_driver_installed()
{
    case ${PACKAGE_MANAGER} in
        0)
            ret=$(rpm -qa 2>/dev/null | grep iBMA_Driver-dkms)
            if [ $? -eq 0 ]; then
                # Get rpm DKMS driver version: iBMA_Driver-dkms-0.3.5-src.aarch64
                version=$(echo ${ret#*iBMA_Driver-dkms-})
                BMA_DRIVER_VERSION=$(echo ${version%-src*})
                LOG_INFO "The DKMS format driver is already installed!"
                return 0
            fi
            ;;
        1)
            ret=$(dpkg -l 2>/dev/null | grep ibmasrc)
            if [ $? -eq 0 ]; then
                # Get deb DKMS driver version: ii ibmasrc 0.3.5 arm64 Kernnel Module Huawei iBMA Driver Source.
                BMA_DRIVER_VERSION=$(echo $ret | awk '{print $3}')
                LOG_INFO "The DKMS format driver is already installed!"
                return 0
            fi
            ;;
        *)
            LOG_ERROR "Failed to get package type."
            ;;
        esac

    return 1
}

#*****************************************************************************
# Prototype    : get_driver_package_file
# Description  : find driver package name, save to PACKAGE_FILE.
#                get driver version driver package name (BMA_DRIVER_VERSION)
#                get package manager (PACKAGE_MANAGER)
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#  2.Date         : 2017/06/15
#    Modification : compatiable modification.
#  3.Date         : 2020/12/07
#    Modification : Added the function of determining whether the driver is installed.
#*****************************************************************************
function get_driver_package_file()
{
    # specially for the simple version, change OS name.
    if [ "${OS_NAME}" == "RedHat" ] && [ ! -d "${DRIVER_DIR}/RedHat" ]; then
        PACKAGE_FILE=$(ls ${DRIVER_DIR}/CentOS/*-$(uname -r)-* 2>/dev/null | head -n1)
    else
        PACKAGE_FILE=$(ls ${DRIVER_DIR}/${OS_NAME}/*-$(uname -r)-* 2>/dev/null | head -n1)
    fi

    if [[ "${DEB_OS_LIST[@]}" =~ "${OS_NAME}" ]]; then
        PACKAGE_MANAGER=1
    fi

    if [ ! -f "${PACKAGE_FILE}" ]; then
        # try to get an available one.
        get_available_driver_package
        if [ $? -eq 0 ]; then
            return 0;
        fi

        record_dkms_driver_installed
        if [ $? -eq 0 ]; then
            return 0
        fi

        modinfo "host_edma_drv" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            BMA_DRIVER_VERSION=$(modinfo host_edma_drv 2>/dev/null | grep "^version:" | awk '{print $2}')
            return 0
        fi

        # no available, failed.
        LOG_HINT "Kernel version $(uname -r) does not support."
        LOG_ERROR "Failed to get driver package for the system."
        exit 1;
    fi

    LOG_INFO "Driver package:[${PACKAGE_FILE}]" "HIDE"

    get_package_manager "${PACKAGE_FILE}"
    get_package_version "${PACKAGE_FILE}"

    return 0;
}

#*****************************************************************************
# Prototype    : get_citrix_or_redhat
# Description  : get Citrix or Redhat, save to OS_NAME.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/08/16
#    Modification : Created function
#
#*****************************************************************************
function get_citrix_or_redhat()
{
    if [[ "$1" == "/etc/redhat-release" ]] ;then
        os_sys=$(cat $1 | grep XenServer)
        if [ -z "$os_sys" ] ;then
            OS_NAME=RedHat
        else
            OS_NAME=Citrix
        fi
    fi
}

#*****************************************************************************
# Prototype    : get_os_name
# Description  : get OS name, save to OS_NAME.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#
#*****************************************************************************
function get_os_name()
{
    # check lsb_release command.
    OS_NAME=$(lsb_release -i -s 2>/dev/null)
    if [ -n "${OS_NAME}" ] &&
        [[ "${OS_NAME_LIST}" == *${OS_NAME}* ]]; then
        if [ "${OS_NAME}" == "XenServer" ]; then
            OS_NAME=Citrix
        elif [ "${OS_NAME}" == "OracleServer" ]; then
            OS_NAME=Oracle
        fi
        LOG_INFO "LSB OS is [${OS_NAME}]" "HIDE"
        return 0;
    fi

    # check os-specific file.
    local index=0
    while [ ${index} -lt ${#ETC_FILE[@]} ]; do
        if [ -e "${ETC_FILE[${index}]}" ]; then
            OS_NAME=${ETC_FILE[${index}+1]}
            get_citrix_or_redhat ${ETC_FILE[${index}]}
            LOG_INFO "ETC OS is [${OS_NAME}]" "HIDE"
            return 0;
        fi
        let index+=2;
    done

    # check other os file.
    local content=$(cat "${OTHER_OS_FILE}" 2> /dev/null);
    for os in ${OTHER_OS_NAME_LIST[*]}; do
        echo "${content}" | grep -i "${os}" -q
        if [ $? -eq 0 ]; then
            OS_NAME=${os}
            LOG_INFO "OTHER OS is [${OS_NAME}]" "HIDE"
            return 0;
        fi
    done

    LOG_HINT "System version does not support."
    LOG_ERROR "Failed to get the system version."

    exit 1;
}

#*****************************************************************************
# Prototype    : uninstall_old_driver
# Description  : uninstall old driver package.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#  2.Date         : 2018/03/07
#    Modification : Modify the log records
#
#*****************************************************************************
function uninstall_old_driver()
{
    local res="";

    # rmmod driver first.
    local count=${#DRIVER_LIST_ALL[@]}

    while [ $count -gt 0 ]; do
        let count--
        res=$(rmmod "${DRIVER_LIST_ALL[$count]}" 2>&1)
        if [ $? -ne 0 ] || [ -n "${res}" ]; then
            LOG_INFO "rmmod ${DRIVER_LIST_ALL[$count]}:${res}" "HIDE"
        fi
    done

    LOG_INFO "Uninstalling the installed driver package ..."
    case ${PACKAGE_MANAGER} in
        0)
            # rpm package.
            local packages=$(rpm -qa 2>/dev/null | grep -v "iBMA_driver-dkms" | grep "iBMA_driver")
            pkg_count=$(echo "${packages}" | wc -l)
            for ((pkg_cnt=1;pkg_cnt<=pkg_count;pkg_cnt++ )); do
                package=$(echo "${packages}" | awk '(NR=='$pkg_cnt')')
                if [ -z "${package}" ]; then
                    continue;
                fi

                res=$(rpm -e "${package}" 2>&1)
                if [ $? -ne 0 ]; then
                    LOG_ERROR "Failed to uninstall driver ${package}."
                    LOG_ERROR "res=${res}" "HIDE"
                    return 1;
                else
                    remove_ibma_driver_from_initramfs
                fi
            done
            ;;
        1)
            # dpkg package.
            res=$(dpkg -P "ibmadriver" 2>&1)
            if [ $? -ne 0 ]; then
                LOG_ERROR "Failed to uninstall driver package."
                LOG_ERROR "res=${res}" "HIDE"
                return 1;
            fi
            ;;
        *)
            LOG_ERROR "Failed to get package installer."
            return 127
            ;;
    esac

    LOG_INFO "Uninstall the installed driver package successfully."
    return 0;
}

function is_newer_version()
{
    local cur_ver="${BMA_DRIVER_VERSION}"
    echo -e "${1}\n${cur_ver}" | sort -V | tail -n 1 | grep "${1}" -q
    return $?
}

#*****************************************************************************
# Prototype    : remove_ibma_driver_from_initramfs
# Description  : from initramfs remove iBMA driver ko.
# Parameter:
#   input:  $1 iBMA driver ko list
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2018/05/30
#    Modification : Created function
#  2.Date         : 2018/06/22
#    Modification : add check initramfs and delete driver version check
#*****************************************************************************
function remove_ibma_driver_from_initramfs()
{
    local res=""

    cd /opt

    # get initramfs first, ignore the exception
    initramfs_list=$(ls /boot/initramfs-*{x86_64,aarch64}.img 2>/dev/null)
    initramfs_count=$(echo "${initramfs_list}" | wc -l)
    for ((initramfs_cnt=1;initramfs_cnt<=initramfs_count;initramfs_cnt++ )); do
        initramfs_file=$(echo "${initramfs_list}" | awk '(NR=='$initramfs_cnt')')
        if [ -z "${initramfs_file}" ]; then
            continue;
        fi

        ko_path_list=$(lsinitrd ${initramfs_file} 2>&1 | grep -i "host_edma_drv.ko")
        if [ $? -ne 0 ]; then
            continue
        fi

        ko_path_count=$(echo "${ko_path_list}" | wc -l)
        for (( ko_cnt=1;ko_cnt<=ko_path_count;ko_cnt++ )); do
            ko_path=$(echo "${ko_path_list}" | awk '(NR=='$ko_cnt')' | awk '{print $9}')
            res=$(printf "%s\n" "${ko_path}" | /sbin/weak-modules --remove-modules ${initramfs_file} 2>&1)
            if [ $? -ne 0 ]; then
                LOG_ERROR "Failed to remove iBMA driver modules, ${res}." "HIDE"
            fi
        done
    done

    return 0
}

#*****************************************************************************
# Prototype    : handle_old_version
# Description  : handle old driver package.
# Parameter:
#   input:  $1 old version.
#   output: NA
# Return Value : 0 install. 1 failed. 2 continue, do not install driver.
#
#  History        :
#  1.Date         : 2017/02/14
#    Modification : Created function
#  2.Date         : 2018/03/07
#    Modification : Modify the code structure
#                   SLES old driver must be delete
#  3.Date         : 2018/05/12
#    Modification : Modify the information when kernel has changed
#  4.Date         : 2018/05/30
#    Modification : from initramfs remove iBMA driver ko
#
#*****************************************************************************
function handle_old_version()
{
    local packages=""
    local kernel_version=""
    local old_driver_packages=""
    local package_find_xen_ret=1
    local kernel_find_xen_ret=1
    local running_status=1
    local need_proc_initramfs=1
    local version=$(modinfo host_edma_drv 2>/dev/null | grep "^version:" | awk '{print $2}')
    if [ -z "${version}" ]; then
        packages=$(rpm -qa 2>/dev/null | grep "iBMA_driver" | grep -v "iBMA_Driver-dkms")
        if [ $? -eq 0 ]; then
            old_driver_packages=$(echo $packages | grep -i xen)
            package_find_xen_ret=$?
            kernel_version=$(echo $(uname -r) | grep -i xen)
            kernel_find_xen_ret=$?
            if [ ${kernel_find_xen_ret} -eq 0 ] && [ ${package_find_xen_ret} -ne 0 ]; then
                LOG_INFO "The kernel is changed from default to xen, the driver needs to be reinstalled."
            elif [ ${kernel_find_xen_ret} -ne 0 ] && [ ${package_find_xen_ret} -eq 0 ]; then
                LOG_INFO "The kernel is changed from xen to default, the driver needs to be reinstalled."
            elif [[ ${kernel_find_xen_ret} -eq 0 && ${package_find_xen_ret} -eq 0 ]] \
                || [[ ${kernel_find_xen_ret} -ne 0 && ${package_find_xen_ret} -ne 0 ]]; then
                LOG_INFO "Old driver package is installed, need uninstall it first."
            fi

            uninstall_old_driver
            return $?;
        fi

        LOG_INFO "No driver package installed before." "HIDE"
        return 0;
    elif [ "${version}" == "${BMA_DRIVER_VERSION}" ]; then
        record_dkms_driver_installed
        if [ $? -ne 0 ]; then
            LOG_INFO "Driver package is already installed."
        fi
        return 2;
    fi

    if ! is_newer_version "${version}" ; then
        LOG_INFO "A driver package of an older version has been installed."

        uninstall_old_driver
        running_status=$?

        return $running_status;
    fi

    if ${SILENT_MODE}; then
        LOG_INFO "A driver package of a newer version has been installed."
        LOG_INFO "Retain the current version."
        return 2;
    fi

    LOG_INFO "A driver package of a newer version has been installed."
    PS3="Enter your choice:"
    options=("Retain the current version." "Install an earlier version.")
    select opt in "${options[@]}"; do
        case $opt in
            "Retain the current version.")
                LOG_INFO "Use old version, no need to install." "HIDE"
                return 2;
                ;;
            "Install an earlier version.")
                uninstall_old_driver
                return $?;
                ;;
            *)
                LOG_INFO "invalid option."
                ;;
        esac
    done

    return 0;
}


#*****************************************************************************
# Prototype    : sles_driver_pre
# Description  : action before sles drvier install .
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 if success, otherwise failed.
#
#  History        :
#  1.Date         : 2021/1/17
#    Modification : Created function
#
#*****************************************************************************
function sles_driver_pre()
{
    local ibmadrivers=("host_cdev_drv" "host_kbox_drv" "host_veth_drv" "cdev_veth_drv" "host_edma_drv")
    local drivernum=0
    while [ ${drivernum} -lt ${#ibmadrivers[@]} ]; do
        mods=( $(/sbin/lsmod | grep -wo "${ibmadrivers[${drivernum}]}") )
        if [ "${mods}" == "${ibmadrivers[${drivernum}]}" ]; then
            /sbin/rmmod "${ibmadrivers[${drivernum}]}" >/dev/null 2>&1
            if [ $? -ne 0 ]; then
                LOG_INFO "Warning: fail to uninstall the old ${ibmadrivers[${drivernum}]} kernel module."
                LOG_INFO "Please uninstall the old ${ibmadrivers[${drivernum}]} kernel module manually or reboot the system."
                return 1
            else
                LOG_INFO "Uninstall the old ${ibmadrivers[${drivernum}]} kernel module successfully." "HIDE"
            fi
        fi
        let drivernum++
    done

    return 0
}


#*****************************************************************************
# Prototype    : install_driver_package
# Description  : install driver.
#                1. install if no previous version found.
#                2. update if we have a newer version, uninstall old version.
#                3. ask if we had an older version or fail in SILENT_MODE.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 if success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#  2.Date         : 2021/1/7
#    Modification : rmmod old drvier for sles.
#
#*****************************************************************************
function install_driver_package()
{
    handle_old_version
    case $? in
        0)
            # OK, not install or version is lower.
            ;;
        2)
            # OK, same version install or version is newer, no need to install.
            return 0
            ;;
        *)
            # NOK, return 1.
            return 1;
            ;;
    esac

    if [ -e "${PACKAGE_FILE}" ]; then
        LOG_INFO "Installing driver package ..."
        case ${PACKAGE_MANAGER} in
            0)
                if [ "${OS_NAME}" == "SUSE" ]; then
                    sles_driver_pre
                    if [ $? -ne 0 ]; then
                        return 1
                    fi
                fi

                res=$(rpm -i --force "${PACKAGE_FILE}" 2>&1)
                if [ 0 == $? ]; then
                    LOG_INFO "Messages returned after the driver package is installed:${res}" "HIDE"
                    LOG_INFO "Install driver package successfully."
                    res=$(depmod -a 2>&1)
                    LOG_INFO "depmod $?, result ${res}" "HIDE"
                    return 0
                else
                    LOG_ERROR "Failed to install driver package."
                    LOG_ERROR "res=${res}" "HIDE"
                    return 1
                fi
                ;;
            1)
                res=$(dpkg -i "${PACKAGE_FILE}" 2>&1)
                if [ $? -eq 0 ]; then
                    LOG_INFO "Messages returned after the driver package is installed:${res}" "HIDE"
                    LOG_INFO "Install driver package successfully."
                    res=$(depmod -a 2>&1)
                    LOG_INFO "depmod $?, result ${res}" "HIDE"
                    return 0
                else
                    LOG_ERROR "Failed to install driver package."
                    LOG_ERROR "res=${res}" "HIDE"
                    return 1
                fi
                ;;
            *)
                LOG_ERROR "Failed to get package installer."
                return 127;
                ;;
        esac
    else
        LOG_ERROR "Driver ${rpm} is not found."
        return 127;
    fi
}

function print_system_info()
{
    LOG_INFO "System is ${OS_NAME}"
    LOG_INFO "Kernel version is $(uname -r)"
    LOG_INFO "Driver package version is ${BMA_DRIVER_VERSION}"
}

#*****************************************************************************
# Prototype    : install_driver
# Description  : install driver.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : 0 if success, otherwise failed.
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#  2.Date         : 2017/07/20
#    Modification : remove modules before install.
#  2.Date         : 2021/07/19
#    Modification : check whether OS already has a pre-installed iBMA driver.
#*****************************************************************************
function install_driver()
{
    local res=""

    local nmcli_ret=$(nmcli c show 2>/dev/null | grep -cE "^veth ")
    if [ ${nmcli_ret} -gt 0 ]; then
        nmcli c delete veth >& /dev/null
    fi

    record_dkms_driver_installed
    if [ $? -eq 0 ]; then
        return 0
    fi

    check_pre_installed_driver
    if [ $? -eq 0 ]; then
        return 0
    fi

    install_driver_package
    if [ $? -ne 0 ]; then
        exit 1
    fi

    # do not insert driver modules.
    if ${REBOOT_NEEDED}; then
        return 0;
    fi

    for drv in ${DRIVER_LIST[*]}; do
        res=$(modprobe "${drv}" 2>&1)
        if [ $? -ne 0 ]; then
            echo "${res}" | grep -i "required key not available" -q
            if [ $? -eq 0 ]; then
                LOG_ERROR "Secure boot is enabled, but no Huawei certificate is intalled."
                LOG_ERROR "modprobe ${drv}:${modprobe_res}" "HIDE"
                uninstall_old_driver
                return 1
            fi
        fi
        [ -n "${res}" ] && LOG_INFO "modprobe ${drv}:${res}" "HIDE"
    done

    return 0;
}

function update_key_value()
{
    local key=$1
    local value=$2
    local file=$3
    local res=$(sed -i -e "/^${key}=/d" \
                       -e "\$a${key}=${value}" "${file}" \
                )
    [ -n "${res}" ] && LOG_INFO "SED:${res}"
}

#*****************************************************************************
# Prototype    : handle_config_files
# Description  : copy config files, or modify them.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA.
#
#  History        :
#  1.Date         : 2017/06/29
#    Modification : Created function
#  2.Date         : 2018/07/25
#    Modification : Check veth ipv6 addr set
#  3.Date         : 2019/08/25
#    Modification : Check veth link using '/sys/class/net/veth'.
#*****************************************************************************
function handle_config_files()
{
    local res=""
    local counter=0
    local veth_cfg="ifcfg-veth"
    local eth_cfg_dir="/etc/sysconfig/network-scripts"
    local target_cfg=""
    local drv1="host_veth_drv"
    local drv2="cdev_veth_drv"

    mac=$(grep "^HWADDR=" "${DRIVER_DIR}/${veth_cfg}" | awk -F "=" '{print $2}')
    name=$(grep "^NAME=" "${DRIVER_DIR}/${veth_cfg}" | awk -F "=" '{print $2}')
    addr=$(grep "^IPV6ADDR=" "${DRIVER_DIR}/${veth_cfg}" | awk -F "=" '{print $2}')

    if [ ! -e "/sys/module/${drv2}" ]; then
        res=$(modprobe "${drv1}" 2>&1)
        if [ $? -ne 0 ]; then
            LOG_ERROR "Failed to load ${drv1}, result: ${res}" "HIDE"
            return
        fi

        if [ "false" == "${IPV4_MODE}" ]; then
            res=$(cat /proc/sys/net/ipv6/conf/"${name}"/disable_ipv6 2>/dev/null)
            if [ -n "${res}" ]; then
                if [[ "${res}" != "0" ]]; then
                    LOG_HINT "IPv6 is disabled for ${name} in this host."
                    LOG_HINT "You need to change your settings or use IPv4 instead."
                fi
            fi
        fi

        while [ $counter -lt 10 ]; do
            sleep 1

            if [ ! -e "/sys/class/net/${name}" ]; then
                continue;
            fi

            res=$(ifconfig "${name}" inet6 add "${addr}" up 2>&1 || \
                  (ip addr add "${addr}" dev "${name}" 2>&1 && ip link set "${name}" up))
            LOG_INFO "set IPv6 address for ${name}: code $?, result ${res}" "HIDE"

            (ifconfig "${name}" 2>/dev/null | grep -i inet6 -q)|| \
            (ip addr show "${name}" 2>/dev/null | grep -i inet6 -q)
            if [ $? -eq 0 ]; then
                break
            fi

            LOG_ERROR "Failed to set IPv6 address for ${name}, try again later." "HIDE"

            let counter++
        done
    fi

    if [ ! -d "${eth_cfg_dir}" ]; then
        LOG_INFO "${eth_cfg_dir} does not exist." "HIDE"
        return
    fi

    if [ -e "${eth_cfg_dir}/${veth_cfg}" ]; then
        LOG_INFO "${veth_cfg} already exists." "HIDE"
        chmod 644 "${eth_cfg_dir}/${target_cfg}"

        mac2=$(grep "^HWADDR=" "${eth_cfg_dir}/${veth_cfg}" | awk -F "=" '{print $2}')
        name2=$(grep "^NAME=" "${eth_cfg_dir}/${veth_cfg}" | awk -F "=" '{print $2}')

        LOG_INFO "mac2:${mac2}, name2:${name2}" "HIDE"

        if [ "${mac2}" == "${mac}" -a "${name2}" == "${name}" ]; then
            update_key_value "IPV6ADDR" "${addr}" "${eth_cfg_dir}/${veth_cfg}"
            update_key_value "NM_CONTROLLED" "no" "${eth_cfg_dir}/${veth_cfg}"
            LOG_INFO "update ${veth_cfg} done." "HIDE"
            target_cfg=${veth_cfg}
        elif [ -n "${mac2}" -a -n "${name2}" -a \
               ! -f "${eth_cfg_dir}/ifcfg-${name2}" ]; then
            res=$(mv -f "${eth_cfg_dir}/${veth_cfg}" "${eth_cfg_dir}/ifcfg-${name2}" 2>&1)
            LOG_INFO "rename old ${veth_cfg} to ifcfg-${name2}: ${res}" "HIDE"
        else
            res=$(rm -f "${eth_cfg_dir}/${veth_cfg}" 2>&1)
            LOG_INFO "delete old ${veth_cfg}: ${res}" "HIDE"
        fi
    fi

    cfg_files=$(grep -i "${mac}" ${eth_cfg_dir}/ifcfg-* 2>/dev/null | \
               awk -F ":" '{print $1}')
    for file in ${cfg_files[*]}; do
        if [ "${file}" == "${eth_cfg_dir}/${target_cfg}" ]; then
            continue
        fi
        res=$(rm -f "${file}" 2>&1)
        LOG_INFO "delete ${file}: ${res}" "HIDE"
    done

    if [ -z "${target_cfg}" ]; then
        cp -a "${DRIVER_DIR}/${veth_cfg}" "${eth_cfg_dir}/${veth_cfg}"
        LOG_INFO "copy new ifcfg-veth." "HIDE"
        target_cfg=${veth_cfg}
    fi

    chmod 644 "${eth_cfg_dir}/${target_cfg}"
    res=$(restorecon -v "${eth_cfg_dir}/${target_cfg}" 2>&1)
    [ -n "${res}" ] && LOG_INFO "restorecon:${res}" "HIDE"
}

# check if this is the root
if [ $(/usr/bin/id -u) != 0 ] ; then
    LOG_ERROR "$0: This script must be run as root."
    exit 1
fi

if [ "${1}" == "-s" ] || [ "${1}" == "-u" ] ; then
    SILENT_MODE=true;
fi

if [ "${2}" == "true" ]; then
    FORCE_MODE=true;
fi

if [ "${3}" == "IPv4" ]; then
    IPV4_MODE=true;
fi

get_os_name
get_driver_package_file
print_system_info

is_pcie_detected
if [ $? -ne 0 ] ; then
    if [ "${FORCE_MODE}" == "false" ] &&
       [ "${SILENT_MODE}" != "false" ]; then
        LOG_ERROR "The PCIe device is disabled."
        LOG_ERROR "Enable Black Box on the iBMC first."
        exit 1;
    fi
    open_pcie_device
    REBOOT_NEEDED=true
else
    check_pcie_device
    if [ $? -ne 0 ]; then
        exit 1
    fi
fi

install_driver
if [ $? -ne 0 ]; then
    exit 1
fi

handle_config_files

if ${REBOOT_NEEDED}; then
    exit 3
fi

# success.
exit 0