#!/bin/bash
#******************************************************************************
# Copyright (C), 2015-2017, Huawei Tech. Co., Ltd.
# File : log.sh
# 1. simple log interface
#
#  History       :
#  1.Date        : 2016/12/26
#    Modification: Created file
#******************************************************************************

# bash color.
COLOR_NORMAL='\033[0m'
COLOR_RED='\033[0;31m'

LOG_LEVEL=1
LOG_STDOUT=1
LOG_LEVEL_HINT=4

# alias log function with log level.
alias LOG_DEBUG='log 0 "[DEBUG] [${BASH_SOURCE} ${LINENO}]"'
alias LOG_INFO='log 1 "[INFO ] [${BASH_SOURCE} ${LINENO}]"'
alias LOG_WARN='log 2 "[WARN ] [${BASH_SOURCE} ${LINENO}]"'
alias LOG_ERROR='log 3 "[ERROR] [${BASH_SOURCE} ${LINENO}]"'
alias LOG_HINT='log 4 "[HINT ] [${BASH_SOURCE} ${LINENO}]"'
shopt -s expand_aliases

LOG_DIR="log"
LOG_NAME="runlog"

ibma_syslog=false
SYSLOG_TAG="iBMA ${LOG_NAME}"
LOG_TYPE_ARRAY=(auth cron daemon kernel mail syslog lpr)
SYSLOG_TYPE=user

PROJECT_PATH=$(pwd)
if [ ! -e "${PROJECT_PATH}/config/iBMA.ini" ]; then
    PROJECT_PATH="/opt/huawei/ibma"
fi

LOG_PATH="/opt/huawei/ibma/${LOG_DIR}/runlog"

if [ ! -e "${LOG_PATH}" ]; then
    touch "${LOG_PATH}" >& /dev/null
    chmod 640 "${LOG_PATH}" >& /dev/null
fi

#*****************************************************************************
# Prototype    : read_from_file
# Description  : read item value of a section in configuration file.
# Parameter:
#   input:  $1 file name,  $2 section,  $3 item
#   output: value
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/05/04
#    Modification : Created function
#
#*****************************************************************************
function read_from_file()
{
    local ini_file="$1"
    local section="$2"
    local item="$3"
    local value

    value=$(awk -F '=' '/\['$section'\]/{a=1}a==1&&$1~/'$item'/{print $2;exit}' $ini_file)

    echo "${value}" | sed -e 's/[\r\n]//g'
}

#*****************************************************************************
# Prototype    : get_syslog_config
# Description  : get syslog configuration information.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/05/04
#    Modification : Created function
#
#*****************************************************************************
function get_syslog_config()
{
    # update ibma_syslog
    local ibma_ini="${PROJECT_PATH}/config/iBMA.ini"
    if [ -f "${ibma_ini}" ]; then
        ibma_syslog=$(read_from_file ${ibma_ini} "iBMA_System" "iBMA_syslog")
    fi
    
    # syslog type
    local syslog_ini="${PROJECT_PATH}/config/syslog.ini"
    if [ -f "${syslog_ini}" ]; then
        local syslog_type_tmp=$(read_from_file ${syslog_ini} "iBMA_syslog" "type")
        if [[ "${LOG_TYPE_ARRAY[@]}" =~ "$syslog_type_tmp" ]] ; then
            SYSLOG_TYPE=${syslog_type_tmp}
        fi
    fi
}

#*****************************************************************************
# Prototype    : log
# Description  : record log.
# Parameter:
#   input:  $1 log level,  $2 location,  $3 message
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/05/04
#    Modification : Created function
#
#*****************************************************************************
function log()
{
    # 3 arguments, or with a 'HIDE'.
    if [ $# -ne 3 -a $# -ne 4 ] ; then
        echo "Log error, invalid parameters."
        return
    fi

    # log to file and syslog.
    if [ $1 -ge ${LOG_LEVEL} ]; then
        local system_date=$(date -d today +"%Y-%m-%d %H:%M:%S")
        echo "[ ${system_date} ] $2 $3" 2> /dev/null >> "${LOG_PATH}"
        
        # log to syslog.
        if [ "true" == "${ibma_syslog}" ]; then
            if [ $1 -eq 0 ]; then
                logger -t ${SYSLOG_TAG} -p ${SYSLOG_TYPE}.debug "$2" "$3" 2> /dev/null
            elif [ $1 -eq 1 ]; then
                logger -t ${SYSLOG_TAG} -p ${SYSLOG_TYPE}.info "$2" "$3" 2> /dev/null
            elif [ $1 -eq 2 ]; then
                logger -t ${SYSLOG_TAG} -p ${SYSLOG_TYPE}.warn "$2" "$3" 2> /dev/null
            elif [ $1 -eq 3 ]; then
                logger -t ${SYSLOG_TAG} -p ${SYSLOG_TYPE}.error "$2" "$3" 2> /dev/null
            fi
        fi
    fi

    # log to stdout.
    if [ $# -eq 3 ]; then
        if [ $1 -eq ${LOG_LEVEL_HINT} ]; then
            echo -e ${COLOR_RED}"$3"${COLOR_NORMAL}
        elif [ $1 -ge ${LOG_STDOUT} ]; then
            echo "$3"
        fi
    fi
}

# get syslog configuration information.
get_syslog_config

