#!/bin/bash
#******************************************************************************
# Copyright (C), 2015-2021, Huawei Tech. Co., Ltd.
# File : install.sh
# 1.install iBMA2.0 software
#
#  History       :
#  1.Date        : 2016/12/26
#    Modification: Created file
#  2.Date        : 2019/08/28
#    Modification: Remove check of server type and
#                  execute install script of drivers with a parser
#  3.Date        : 2020/05/18
#    Modification: Modify install path that replacing the "huawei"
#                  by a custom vendor name; Delete storcli64 under Kunpeng build
#  4.Date        : 2021/06/07
#    Modification: Add the iBMC_veth_ip attribute to the iBMA.ini file.
#******************************************************************************

# get and cd to project dir
if [ -L ${0} ]; then
    REALFILE=$(readlink ${0})
else
    REALFILE=${0}
fi

CUR_DIR=$(dirname ${REALFILE})

cd ${CUR_DIR}

IBMA_VERSION=2.2.0.3.001
OLD_IBMA_VER=0
MANUFACTURER_NAME=/huawei
# Paths to programs
G_WORK_DIR=$(pwd)
ECHO=/bin/echo
OLD_BMA_DIR=/opt${MANUFACTURER_NAME}
BMA_DIR=/opt${MANUFACTURER_NAME}
HTTP_USER=ibma
HTTP_PORT=8090
BMA_NIC="veth"
SOCKET_PORT=8091
HTTP_PROTOCOL_TYPE=""
IBMA_REBOOT_NEEDED=false
IBMA_RUN_IMMEDIATELY=true
SERVICE_NAME=iBMA
SYSTEMCTL_SERVICE=iBMA.service
DIR_NAME=ibma
SERVICE_SCRIPT=/etc/init.d/${SERVICE_NAME}
IBMA_SECTION="iBMA_System"
USER_ITEM="iBMA_user"
NIC_ITEM="iBMA_nic"
SERVER_PORT_ITEM="iBMA_http_server_port"
SOCKET_PORT_ITEM="iBMA_socket_port"
BMC_EVENT_ITEM="iBMC_event"
SUPPORT_HTTPS_ITEM="iBMA_support_https"
NETWORK_TYPE_ITEM="iBMA_network_type"
IBMC_IP="iBMC_ip"
IBMC_VETH_IP="iBMC_veth_ip"
NETWORK_SETUP_TOOL=${BMA_DIR}/${DIR_NAME}/bin/manager

TOTAL_DRIVER_FILES=0

IBMA_UPGRADE_LOG_PATH=${BMA_DIR}/${DIR_NAME}/log/iBMAUpgradeLog
CONFIG=${BMA_DIR}/${DIR_NAME}/config/iBMA.ini
INSTALL_LOG_FILE=${BMA_DIR}/${DIR_NAME}/log/runlog
OLD_CONFIG_PATH=${BMA_DIR}/${DIR_NAME}/config_old

SILENT_MODE=false
custom_mode=false
FORCE_MODE=false
IP_MODE="IPv6"
DRIVER_INSTALL_MODE=""
IBMC_EVENT=false
CONFIG_IBMC_EVENT=false
IBMA_SUPPORT_HTTPS=true
CONFIG_IBMA_SUPPORT_HTTPS=false
KEEP_SUPPORT_HTTPS_SELECT=false

ALREADY_IN_IBMA_PKG=false

OS_NAME=""
VERSION_ID_BASE=0
SYSTEMD_DIR='/usr/lib/systemd/system/'
if [ ! -e "${SYSTEMD_DIR}" ]; then
    # Ubuntu systemctl services save in the follow directory
    SYSTEMD_DIR='/lib/systemd/system/'
fi
SYSTEMD_EXEC_DIR='/etc/systemd/system/multi-user.target.wants/'
readonly ETC_FILE=( "/etc/neokylin-release" "NeoKylin" \
                    "/etc/bclinux-release" "bclinux" \
                    "/etc/openEuler-release" "openEuler" \
                    "/etc/euleros-release" "EulerOS" \
                    "/etc/oracle-release" "Oracle" \
                    "/etc/centos-release" "CentOS" \
                    "/etc/redhat-release" "RedHat" \
                    "/etc/SuSE-release" "SUSE" )

readonly OTHER_OS_FILE="/etc/os-release"

readonly OS_NAME_LIST="CentOS,RedHat,SUSE,Ubuntu,XenServer,OracleServer"

# Version of SUSE system is stored in /etc/os-release starting from SLES 15
readonly OTHER_OS_NAME_LIST=("Debian" "Ubuntu" "SUSE" "Kylin" "uos")

NEED_SYSTEMCTL_START_IBMA=false
OS_SUPPROT_SYSTEMCTL=false
IS_UPGRADE=false
IS_RESUME=false
SYSTEMCTL_SERVICE_SCRIPT=""
PRODUCT_ID=""
ibma_upgrade_exit=false

OLD_BMA_SH_PATH=""
unset LD_LIBRARY_PATH
source "${G_WORK_DIR}/script/log.sh"

BOBSERVICE_NAME=bob
BOBSYSTEMCTL_SERVICE=bob.service
BOBSERVICE_SCRIPT=/etc/init.d/${BOBSERVICE_NAME}
OLD_BOB_SH_PATH=""

IBMA_IS_HUAWEI_OWNER=false

# set IBMA_IS_HUAWEI_OWNER to true when BMA_DIR not exists
# or only ibma under BMA_DIR
if [ ! -d "${BMA_DIR}" ];then
    IBMA_IS_HUAWEI_OWNER=true
else
    if [ -d "${BMA_DIR}/${DIR_NAME}" ] && \
       [ 1 == $(ls "${BMA_DIR}" |wc -l) ];then
        IBMA_IS_HUAWEI_OWNER=true
    fi
fi

if [ ! -d "${BMA_DIR}/${DIR_NAME}/log" ]; then
    mkdir -pm 640 ${BMA_DIR}/${DIR_NAME}/log
fi

if [ ! -e "${INSTALL_LOG_FILE}" ]; then
    touch "${INSTALL_LOG_FILE}"
    if [ $? -ne 0 ]; then
        LOG_ERROR "Failed to create ${INSTALL_LOG_FILE}."
    fi
    chmod 640 ${INSTALL_LOG_FILE}
fi

if [ -d "${OLD_CONFIG_PATH}" ]; then
    rm -rf "${OLD_CONFIG_PATH}"
fi

#*****************************************************************************
# Prototype    : print_help
# Description  : show help information
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function print_help()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "${SERVICE_NAME} Installer                                          "
    LOG_INFO "Usage: $0 {-s [options...]|-c|-u|-h} [-f]                          "
    LOG_INFO "arguments:                                                         "
    LOG_INFO " -s  Silent installation mode, recommended                         "
    LOG_INFO "     options:--enable-iBMC_event=true|false                        "
    LOG_INFO "                 Enable/Disable the feature of logging iBMC events "
    LOG_INFO "             --enable-iBMA_https=true|false                        "
    LOG_INFO "                 Redfish Server starts with HTTPS/HTTP protocol    "
    LOG_INFO "                 true indicates that HTTPS is selected             "
    LOG_INFO "                 false indicates that HTTP is selected             "
    LOG_INFO " -c  Custom installation mode                                      "
    LOG_INFO " -f  Force installation mode, used with -s or -c,                  "
    LOG_INFO "     overwrite the configuration files                             "
    LOG_INFO " -u  Upgrade ${SERVICE_NAME}, reserve the configuration files      "
    LOG_INFO " -h  Show this help                                                "
    LOG_INFO "-----------------------------------------------------------------"
}

#*****************************************************************************
# Prototype    : read_from_buff
# Description  : read item value of a section in Buffer.
# Parameter:
#   input:  $1 string buffer,  $2 section,  $3 item
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/8/25
#    Modification : Created function
#
#*****************************************************************************
function read_from_buff()
{
    local string_buf="$1"
    local section="$2"
    local item="$3"
    local _read_ini

    _read_ini=$(echo -e ${string_buf} | awk -F '=' '/\['$section'\]/{a=1}a==1&&$1~/'$item'/{print $2;exit}')

    $ECHO "${_read_ini}" | sed -e 's/[\r\n]//g'
}

#*****************************************************************************
# Prototype    : is_valid_port_num
# Description  : check if a port is valid.
# Parameter:
#   input:  $1 port number.
#   output: NA
# Return Value : 0 OK, 1 out-of-range, 2 not-a-number.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function is_valid_port_num()
{
    local port="$1"

    if [ "${port}" -ge 0 ] 2>/dev/null ; then
        if [ "${port}" -ge 1024 ] && [ "${port}" -le 65535 ] ; then
            return 0;
        else
            return 1;
        fi
    else
        return 2;
    fi
}

#*****************************************************************************
# Prototype    : is_valid_install_path
# Description  : check if a path is valid for installation.
# Parameter:
#   input:  $1 path.
#   output: NA
# Return Value : 0 OK, 1 pattern no match,
#                2 not writeable, 3 generic error.
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function is_valid_install_path()
{
    local path="$1";
    local pattern="^[/][a-zA-Z0-9_/-]{0,127}$";

    if [[ ! ${path} =~ ${pattern} ]] || [[ ${path} == *//* ]]; then
        return 1;
    fi

    # this should success, otherwise, not writeable.
    if mkdir -p "${path}" 2>/dev/null; then
        if mkdir -p "${path}/${DIR_NAME}" 2>/dev/null; then
            return 0;
        fi
        return 2;
    fi

    return 3;
}

#*****************************************************************************
# Prototype    : check_default_user
# Description  : if the ibma user already exists, exit the installation.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/04/24
#    Modification : Created function
#
#*****************************************************************************
function check_default_user()
{
    local ibma_default_user_name="ibma"
    local ibma_user_desc="This is an ibma user"
    local user_file="/etc/passwd"

    id "${ibma_default_user_name}" 2>/dev/null |grep "(${ibma_default_user_name})" -q
    if [ $? -eq 0 ]; then
        # Check if the ibma user was created by iBMA
        cat "${user_file}" |grep -w "${ibma_user_desc}" -q
        if [ $? -ne 0 ]; then
            LOG_ERROR "The user ${ibma_default_user_name} already exists, but it is not the default iBMA user."
            LOG_ERROR "Failed to install iBMA."
            exit 2
        fi

        # If the ibma user was created by iBMA, then continue to use it
        LOG_INFO "The user ${ibma_default_user_name} already exists, not need to create." "HIDE"
        return 0
    fi

    LOG_INFO "The user ${ibma_default_user_name} not exists." "HIDE"
    return 0
}

#*****************************************************************************
# Prototype    : set_redfish_server_user
# Description  : ask user to input Redfish server user.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function set_redfish_server_user()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 1: Set the Redfish server user            "
    LOG_INFO "-----------------------------------------------------------------"

    PS3='Enter your choice:'
    options=("Use ibma user (recommended)         "
             "Enter an existing non-root user"
             "Use root user")
    select opt in "${options[@]}"; do
        case $opt in
            "Use ibma user (recommended)         ")
                LOG_INFO "STEP 1: you choose choice 1 to use ibma user." "HIDE"
                check_default_user
                return
                ;;
            "Enter an existing non-root user")
                LOG_INFO "STEP 1: you choose choice 2 to set an existing non-root user." "HIDE"
                break
                ;;
            "Use root user")
                LOG_INFO "STEP 1: you choose choice 3 to use root user." "HIDE"
                HTTP_USER=root
                return
                ;;
            *)
                LOG_INFO "Invalid option."
                continue
                ;;
        esac
    done

    # input user name.
    while true; do
        read -p "User name:" INPUT
        # check the user if exist. if user input "-u root", the test should fail.
        id ${INPUT} 2>/dev/null | grep "(${INPUT})" -q
        if [ 0 -eq $? ]; then
            HTTP_USER=${INPUT}
            LOG_INFO "User <${INPUT}> is valid. Continue."
            return
        else
            LOG_ERROR "User <${INPUT}> does not exist. Enter a existing user."
            continue;
        fi
    done
}

#*****************************************************************************
# Prototype    : set_redfish_server_port
# Description  : ask user to input Redfish server port.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function set_redfish_server_port()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 2: Set the Redfish server port            "
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "The default port number is <${HTTP_PORT}>"

    PS3='Enter your choice:'
    options=("Use the default port number (recommended)"
             "Set a port number")
    select opt in "${options[@]}"; do
        case $opt in
            "Use the default port number (recommended)")
                LOG_INFO "STEP 2: you choose choice 1 to use the default port <${HTTP_PORT}>." "HIDE"
                return
                ;;
            "Set a port number")
                LOG_INFO "STEP 2: you choose choice 2 to set a new port number." "HIDE"
                break
                ;;
            *)
                LOG_INFO "Invalid option."
                continue
                ;;
        esac
    done

    # input port number.
    while true; do
        LOG_INFO "Enter a number ranging from 1024 to 65535."
        read -p "Port number:" INPUT

        # check if port is valid
        is_valid_port_num "${INPUT}"
        case $? in
            0)
                INPUT=$(expr "${INPUT}" + 0)
                HTTP_PORT=${INPUT}
                LOG_INFO "Port number <${HTTP_PORT}> is valid. Continue."
                return
                ;;
            1)
                LOG_ERROR "Port number is out of range."
                continue
                ;;
            *)
                LOG_ERROR "Port number is invalid."
                continue
                ;;
        esac
    done
}

#*****************************************************************************
# Prototype    : set_socket_server_port
# Description  : ask user to input socket server port.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function set_socket_server_port()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 3: Set the socket server port             "
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "The default port number is <${SOCKET_PORT}>"

    if [ ${HTTP_PORT} != ${SOCKET_PORT} ]; then
        PS3='Enter your choice:'
        options=("Use the default port number (recommended)"
                 "Set a port number")
        select opt in "${options[@]}"; do
            case $opt in
                "Use the default port number (recommended)")
                    LOG_INFO "STEP 3: you choose choice 1 to use the default port number." "HIDE"
                    return
                    ;;
                "Set a port number")
                    LOG_INFO "STEP 3: you choose choice 2 to set a new port number." "HIDE"
                    break
                    ;;
                *)
                    LOG_INFO "Invalid option."
                    continue
                    ;;
            esac
        done
    else
        LOG_INFO "Port number <${SOCKET_PORT}> is used by Redfish server."
    fi

    # input port number.
    while true; do
        LOG_INFO "Enter a number ranging from 1024 to 65535."
        read -p "Port number:" INPUT

        # check if port is valid
        is_valid_port_num "${INPUT}"
        case $? in
            0)
                INPUT=$(expr "${INPUT}" + 0)
                if [ ${INPUT} != ${HTTP_PORT} ]; then
                    SOCKET_PORT=${INPUT}
                    LOG_INFO "Port number <${SOCKET_PORT}> is valid. Continue."
                    return
                else
                    LOG_ERROR "Port number <${INPUT}> is used by Redfish server."
                    continue;
                fi
                ;;
            1)
                LOG_ERROR "Port number is out of range."
                continue
                ;;
            *)
                LOG_ERROR "Port number is invalid."
                continue
                ;;
        esac
    done
}

#*****************************************************************************
# Prototype    : set_redfish_server_protocol
# Description  : ask user to select Redfish Server protocol type.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/09/05
#    Modification : Created function
#
#*****************************************************************************
function set_redfish_server_protocol()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 4: Set the Redfish server protocol        "
    LOG_INFO "-----------------------------------------------------------------"

    PS3='Enter your choice:'
    options=("Use the HTTP protocol" "Use the HTTPS protocol")
    select opt in "${options[@]}"; do
        case $opt in
            "Use the HTTP protocol")
                LOG_INFO "STEP 4: you choose choice 1 to use the HTTP protocol." "HIDE"
                HTTP_PROTOCOL_TYPE="HTTP"
                IBMA_SUPPORT_HTTPS=false
                return
                ;;
            "Use the HTTPS protocol")
                LOG_INFO "STEP 4: you choose choice 2 to use HTTPS protocol." "HIDE"
                HTTP_PROTOCOL_TYPE="HTTPS"
                IBMA_SUPPORT_HTTPS=true
                return
                ;;
            *)
                LOG_INFO "Invalid option."
                continue
                ;;
        esac
    done
}

#*****************************************************************************
# Prototype    : set_if_ibma_start_right_now
# Description  : ask user if start iBMA immediately.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function set_if_ibma_start_right_now()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 5: Start iBMA service immediately         "
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "iBMA will start after installation"

    PS3='Enter your choice:'
    options=("Start after installation (recommended)"
             "Start iBMA manually")
    select opt in "${options[@]}"; do
        case $opt in
            "Start after installation (recommended)")
                LOG_INFO "STEP 5: start iBMA automatically after installation." "HIDE"
                return
                ;;
            "Start iBMA manually")
                LOG_INFO "STEP 5: start iBMA service manually." "HIDE"
                IBMA_RUN_IMMEDIATELY=false
                return
                ;;
            *)
                LOG_INFO "Invalid option."
                continue
                ;;
        esac
    done
}

#*****************************************************************************
# Prototype    : confirm_settings
# Description  : ask user to confirm iBMA settings.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function confirm_settings()
{
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "                  STEP 6: Confirm settings                       "
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "Settings summary:"
    LOG_INFO "1. Redfish server user: ${HTTP_USER}"
    LOG_INFO "2. Redfish server port: ${HTTP_PORT}"
    LOG_INFO "3. Socket server port: ${SOCKET_PORT}"
    LOG_INFO "4. Redfish server protocol type: ${HTTP_PROTOCOL_TYPE}"

    if [ $IBMA_RUN_IMMEDIATELY = true ] ; then
        LOG_INFO "5. Start iBMA immediately: Yes"
    else
        LOG_INFO "5. Start iBMA immediately: No"
    fi

    LOG_INFO ""

    PS3='Enter your choice:'
    options=("Continue to install (recommended)"
             "Quit installation process")
    select opt in "${options[@]}"; do
        case "$opt" in
            "Continue to install (recommended)")
                LOG_DEBUG "Continue to install." "HIDE"
                return
                ;;
            "Quit installation process")
                LOG_DEBUG "You choose to quit." "HIDE"
                exit 1;
                ;;
            *)
                LOG_INFO "Invalid option."
                continue
                ;;
        esac
    done
}

#*****************************************************************************
# Prototype    : resume_ibma
# Description  : resume iBMA version.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function resume_ibma()
{
    resume_ibma_file

    # start iBMA Service.
    start_service
}

#*****************************************************************************
# Prototype    : resume_ibma_file
# Description  : resume iBMA File.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function resume_ibma_file()
{
    ver_dir=""
    ver_dir=$OLD_BMA_DIR/${DIR_NAME}

    cd ${ver_dir}/log/backup
    cp -rf ${ver_dir}/log/backup/bin/*  ${ver_dir}/bin/
    rm -rf ${ver_dir}/config/*
    cp -rf ${OLD_CONFIG_PATH}/*  ${ver_dir}/config/
    cp -rf ${ver_dir}/log/backup/lib/*  ${ver_dir}/lib/
    cp -rf ${ver_dir}/log/backup/lib64/*  ${ver_dir}/lib64/
    copy ${ver_dir}/log/backup/uninstall.sh  ${ver_dir}/
    copy ${ver_dir}/log/backup/${SERVICE_NAME}.sh  ${ver_dir}/

    handle_oam_function

    # create soft link
    cd ${BMA_DIR}/${DIR_NAME}/lib64
    chmod 500 *

    cd - > /dev/null

    # full access for user to access log.
    mkdir -p ${BMA_DIR}/${DIR_NAME}/log
    chmod 640 ${BMA_DIR}/${DIR_NAME}/log

    # hwkbox is not executable for other users.
    if [ -e "${BMA_DIR}/${DIR_NAME}/bin/hwkbox" ]; then
        chmod 550 ${BMA_DIR}/${DIR_NAME}/bin/hwkbox
    fi
}

#*****************************************************************************
# Prototype    : old_ibma_files_backup
# Description  : old iBMA file backup.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function old_ibma_files_backup()
{
    ver_dir=$OLD_BMA_DIR/${DIR_NAME}

    if [ ! -d "${ver_dir}/config" ]; then
        LOG_INFO "No need back iBMA files." "HIDE"
        return 0
    fi

    if [ ! -d "${ver_dir}/log/backup" ]; then
        mkdir -p ${ver_dir}/log/backup
    else
        cd ${ver_dir}/log/backup
        rm -rf *
    fi

    cd ${ver_dir}
    copy ${ver_dir}/bin ${ver_dir}/log/backup/
    copy ${ver_dir}/config ${ver_dir}/log/backup/
    copy ${ver_dir}/lib ${ver_dir}/log/backup/
    copy ${ver_dir}/lib64 ${ver_dir}/log/backup/

    copy ${ver_dir}/uninstall.sh ${ver_dir}/log/backup/
    copy ${ver_dir}/${SERVICE_NAME}.sh ${ver_dir}/log/backup/

    if [ true == ${OS_SUPPROT_SYSTEMCTL} ] && [ -e "${SYSTEMD_DIR}${SYSTEMCTL_SERVICE}" ] ; then
        cp -rf  ${SYSTEMD_DIR}${SYSTEMCTL_SERVICE} ${ver_dir}/log/backup/
    fi

    return 0
}

#*****************************************************************************
# Prototype    : erase_ibma_rpm
# Description  : erase iBMA rpm package.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/12/30
#    Modification : Created function
#
#*****************************************************************************
function erase_ibma_rpm()
{
    if [[ ${OS_NAME} == Ubuntu ]] || [[ ${OS_NAME} == uos ]]; then
        res=$(dpkg --force-all -P ibma2.0 2>&1)
        if [ 0 -ne $? ]; then
            LOG_ERROR "Failed to uninstall iBMA2.0."
            LOG_ERROR "res=${res}" "HIDE"
            return 1;
        fi
    else
        packages=$(rpm -qa 2>/dev/null | grep "iBMA2.0" | grep -v "iBMA2.0-package")
        if [ $? -eq 0 ]; then
            res=$(rpm -e ${packages} 2>&1)
            if [ $? -ne 0 ]; then
                LOG_ERROR "Failed to uninstall iBMA2.0, ${res}"
                return 1
            fi
        fi
    fi
}

#*****************************************************************************
# Prototype    : handle_old_ibma_files
# Description  : copy old settings, delete old install files.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2019/08/15
#    Modification : Change back up path to BMA_DIR/ibma, and don't delete
#                   log/runlog and config_old in BMA_DIR/ibma/ during
#                   silent and custom installation
#
#*****************************************************************************
function handle_old_ibma_files()
{
    local res=""
    local res_query_deb=0
    local res_query_rpm=0

    if [ "" != "${OLD_BMA_DIR}" ]; then
        LOG_INFO "Back up configuration files."
        res=$(mkdir -pm 640 ${OLD_CONFIG_PATH} 2>/dev/null)
        if [ $? -ne 0 ]; then
            LOG_WARN "Create old iBMA version config directory has error."
            return 1
        fi

        # check old bob.ini
        if [ -e "${OLD_BMA_DIR}/${DIR_NAME}/tools/bob/bob.ini" ]; then
            cp -f "${OLD_BMA_DIR}/${DIR_NAME}/tools/bob/bob.ini" "${OLD_CONFIG_PATH}"
        fi

        # check old watchdog.ini
        if [ -e "${OLD_BMA_DIR}/${DIR_NAME}/lib/Linux/config/WatchDog.ini" ]; then
            cp -f "${OLD_BMA_DIR}/${DIR_NAME}/lib/Linux/config/WatchDog.ini" "${OLD_CONFIG_PATH}"
        fi

        res=$(rpm -qa 2>/dev/null | grep "iBMA2.0" | grep -v "iBMA2.0-package")
        res_query_rpm=$?

        res=$(dpkg -l 2>/dev/null | grep "iBMA2.0" | grep -v "ibma2.0-package")
        res_query_deb=$?

        if [ 0 -eq ${res_query_rpm} ] || [ 0 -eq ${res_query_deb} ] ; then
            LOG_INFO "iBMA2.0 package is already installed!"  "HIDE"
            ALREADY_IN_IBMA_PKG=true

            if [ true == ${IS_UPGRADE} ]; then
                LOG_INFO "Back up the old config directory when upgrade iBMA2.0 package."
                old_ibma_files_backup
            fi

            erase_ibma_rpm
            if [ 0 -ne $? ]; then
                return 1;
            fi
        fi

        if [ true == ${IS_UPGRADE} ]; then
            if [ false == ${ALREADY_IN_IBMA_PKG} ]; then
                LOG_INFO "Back up the old version file before installation."
                old_ibma_files_backup
                LOG_INFO "Delete the old version file before installation."
                cd "${OLD_BMA_DIR}/${DIR_NAME}"
                ls |grep -v log |grep -v config_old |xargs rm -rf
            fi
        else
            LOG_INFO "Delete the old version before installation."
            # keep log/runlog and config_old
            cd "${OLD_BMA_DIR}/${DIR_NAME}/log"
            ls |grep -v runlog |xargs rm -rf
            cd "${OLD_BMA_DIR}/${DIR_NAME}"
            ls |grep -v log |grep -v config_old |xargs rm -rf
        fi
    fi
}

#*****************************************************************************
# Prototype    : handle_oam_function
# Description  : set related permissions for OAM files.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/08/16
#    Modification : Created function
#  2.Date         : 2018/08/20
#    Modification : support OAM on all servers when only support on E9000 before
#*****************************************************************************
function handle_oam_function()
{
    LOG_INFO "Setting OAM related permissions." "HIDE"
    chmod 640 ${BMA_DIR}/${DIR_NAME}/config/OAM.ini 2>/dev/null
    ln -sf "${BMA_DIR}/${DIR_NAME}/bin/ReloadOAM" \
           "${BMA_DIR}/${DIR_NAME}/config/ReloadOAM"
}

#*****************************************************************************
# Prototype    : install_ibma
# Description  : install iBMA software.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2017/07/31
#    Modification : Delete Crypto install
#  3.Date         : 2019/09/16
#    Modification : Present the error message to the user when the iBMA deb
#                   package is installed failed.
#*****************************************************************************
function install_ibma()
{
    if [[ ${OS_NAME} == Ubuntu ]] || [[ ${OS_NAME} == uos || ${OS_NAME} == Debian ]]; then
        cd ${G_WORK_DIR}/app
        res=$(dpkg -i iBMA2.0*.deb 2>&1)
        if [ 0 == $? ]; then
            LOG_INFO "Install iBMA2.0 deb package successfully." "HIDE"
            LOG_INFO "res=${res}" "HIDE"
            return 0
        else
            LOG_ERROR "Failed to install iBMA2.0 deb package."
            LOG_ERROR "res=${res}" "HIDE"
            return 1
        fi
    else

        local op_flag=Install
        local op_cmd=ivh

        if [ "true" == "${IS_UPGRADE}" ]; then
            op_flag=Upgrade
            op_cmd=Uvh
        fi

        cd ${G_WORK_DIR}/app
        res=$(rpm -${op_cmd} --force "iBMA2.0*.rpm" --nodeps 2>&1)
        if [ 0 == $? ]; then
            LOG_INFO "${op_flag} iBMA2.0 package successfully." "HIDE"
            LOG_INFO "res=${res}" "HIDE"
            return 0
        else
            LOG_ERROR "Failed to ${op_flag} iBMA2.0 package."
            LOG_ERROR "res=${res}" "HIDE"
            return 1
        fi
    fi
}

#*****************************************************************************
# Prototype    : save_to_config_file
# Description  : save user choice to config file.
# Parameter:
#   input:  $1: which section,  $2: which item,  $3: new value.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function save_to_config_file()
{
    section=$1; item=$2; value=$3

    sed -i "/^${item}=/s/=.*/=${value}/" ${CONFIG}
    # recheck
    old_value=$(read_from_file ${CONFIG} ${section} ${item})
    if [ ${old_value} == ${value} ] ; then
        LOG_INFO "Success to set the ${item} to ${value}." "HIDE"
    else
        LOG_ERROR "Set '${item}' to '${value}' failed." "HIDE"
        LOG_ERROR "Failed to save the configuration. Exiting ..."
        exit 1
    fi
}

#*****************************************************************************
# Prototype    : start_service
# Description  : start iBMA service.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#  2.Date         : 2020/11/04
#    Modification : To avoid iBMA status chaos, stop iBMA first if the user 
#                   does not want to start iBMA immediately
#*****************************************************************************
function start_service()
{
    if [ $IBMA_RUN_IMMEDIATELY = true ]; then
        local start_failed=0
        local service_log
        local service_running=true

        LOG_INFO "-----------------------------------------------------------------"
        LOG_INFO "Starting ${SERVICE_NAME} service."

        # start do not always give systemd the right hint, use restart instead.
        if [[ -r "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ]] ; then
            # try stop service.
            local section=Service
            local item=ExecStart

            if [ -e "${SERVICE_SCRIPT}" ]; then
                rm -f "${SERVICE_SCRIPT}"
            fi

            exec_start_value=$(read_from_file "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ${section} ${item})
            SYSTEMCTL_SERVICE_SCRIPT=$(echo $exec_start_value| awk '{print $1}')
            service_log=$(systemctl restart ${SERVICE_NAME} 2>&1)
            res=$(${SYSTEMCTL_SERVICE_SCRIPT} status 2>&1) || start_failed=1
            if [ 0 == ${start_failed} ]; then
                LOG_INFO "Start ${SERVICE_NAME} service successfully."
            else
                LOG_INFO "${service_log}"
                LOG_INFO "Start ${SERVICE_NAME} service failed."
                LOG_INFO "${res}" "HIDE"
                return 1
            fi
        else
            service ${SERVICE_NAME} restart >& /dev/null;
            service_log=$(${SERVICE_SCRIPT} start 2>&1) || start_failed=1
            if [ 0 != ${start_failed} ]; then
                LOG_INFO "${service_log}"
                return 1
            else
                LOG_INFO "Start ${SERVICE_NAME} service successfully."
                # let systemd know iBMA already started.
                service ${SERVICE_NAME} start >& /dev/null;
            fi
        fi

        LOG_INFO "-----------------------------------------------------------------"
        return 0
    elif ${IBMA_REBOOT_NEEDED}; then
        LOG_HINT "You need to reboot the OS to run ${SERVICE_NAME} service."
        LOG_INFO ""
        return 0
    elif [ $IBMA_RUN_IMMEDIATELY = false ]; then
        service iBMA stop &>/dev/null
        return 0
    fi
}

# copy file, exit on error.
function copy() {
    local res
    res=$(cp -a "$1" "$2" 2>&1)
    if [ $? -ne 0 ] ; then
        LOG_ERROR "Copy $1 to $2 failed, abort!"
        LOG_ERROR "Error message:${res}" "HIDE"
        LOG_INFO "-----------------------------------------------------------------"
        LOG_ERROR "Failed to install ${SERVICE_NAME} software."
        exit 1
    fi
}

#*****************************************************************************
# Prototype    : find_old_ibma_path
# Description  : find old iBMA install path.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#
#*****************************************************************************
function find_old_ibma_path()
{
    # "/etc/init.d/iBMA" is the default path
    if [[ ! -r ${SERVICE_SCRIPT} ]] && [[ ! -r "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ]] ; then
        OLD_BMA_DIR=""
        return
    fi

    local service_running=true
    if [[ -r "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ]] ; then
        local section=Service
        local item=ExecStart
        exec_start_value=$(read_from_file "${SYSTEMD_DIR}/${SYSTEMCTL_SERVICE}" ${section} ${item})
        SYSTEMCTL_SERVICE_SCRIPT=$(echo $exec_start_value| awk '{print $1}')
        OLD_BMA_DIR=$(echo "${SYSTEMCTL_SERVICE_SCRIPT%/*}")
        OLD_BMA_DIR=$(echo "${OLD_BMA_DIR%/*}")
        OLD_BMA_SH_PATH=${SYSTEMCTL_SERVICE_SCRIPT}

        LOG_INFO "Found old ${SERVICE_NAME} installed in the path <${OLD_BMA_DIR}>" "HIDE"

        "${SYSTEMCTL_SERVICE_SCRIPT}" status >& /dev/null || service_running=false
    elif [[ -r ${SERVICE_SCRIPT} ]] ; then
        local _path=$(grep ^ROOT_PATH= "${SERVICE_SCRIPT}")
        if [ "$_path" != "" ]; then
            IFS='=';
            arr=($_path);
            OLD_BMA_DIR=$(echo ${arr[1]}|sed 's/\"//g')
            LOG_INFO "Found old ${SERVICE_NAME} installed in the path <${OLD_BMA_DIR}>" "HIDE"
        else
            OLD_BMA_DIR=""
            LOG_WARN "Found old ${SERVICE_NAME} file is invalid"
        fi

        if [ "${OLD_BMA_DIR}" != "" ]; then
            OLD_BMA_SH_PATH=${OLD_BMA_DIR}/ibma/iBMA.sh
        fi
    fi

    if [ "${OLD_BMA_DIR}" != "" ]; then
        OLD_BOB_SH_PATH=${OLD_BMA_DIR}/ibma/bob.sh
    fi
}

#*****************************************************************************
# Prototype    : check_ip_conflicts
# Description  : check config file IP conflicts.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/10/12
#    Modification : Created function
#
#*****************************************************************************
function check_ip_conflicts()
{
    # res1 is new ip config, res2 is old ip config
    local res1="IPv6"
    local res2="IPv6"

    local old_config_file=""
    old_config_file=${OLD_BMA_DIR}/${DIR_NAME}/config/iBMA.ini
    if [ ! -e "${old_config_file}" ]; then
        IP_MODE=${res1}
        LOG_INFO "iBMA config file does not exist, no need check iBMA config file." "HIDE"
        return 0
    fi

    local old_ibma_ip=$(read_from_file ${old_config_file} ${IBMA_SECTION} "iBMA_http_server_ip")
    echo ${old_ibma_ip} | grep : > /dev/null
    if [ 0 -ne $? ] ; then
        res2="IPv4"
    fi

    if [ "true" == "${IS_UPGRADE}" ]; then
        IP_MODE=${res2}
    elif [ "true" == "${FORCE_MODE}" ]; then
        IP_MODE=${res1}
    else
        if [ "${res1}" == "${res2}" ]; then
            IP_MODE=${res1}
        else
            LOG_ERROR "Old iBMA ip ${old_ibma_ip} conflicts with new iBMA ip ${new_ibma_ip}" "HIDE"
            LOG_ERROR "Current configuration(${res2}) conflicts with target configuration(${res1})."
            LOG_ERROR "You need to add \"-f\" to overwrite the configuration file,"
            LOG_ERROR "or enter \"-u\" to reserve the configuration file."
            LOG_INFO "-----------------------------------------------------------------"
            LOG_ERROR "Failed to install ${SERVICE_NAME} software."
            return 1
        fi
    fi

    return 0
}

#*****************************************************************************
# Prototype    : stop_service
# Description  : stop the service which is specify by parameter.
# Parameter:
#   input:  $1: service name,  $2: service script path.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2018/11/19
#    Modification : Created function
#
#*****************************************************************************
function stop_service()
{
    local service_running=true

    stop_output=$(${2} stop 2>&1)
    if [ $? -eq 0 ]; then
        service_running=false
    fi

    LOG_INFO "${stop_output}"

    if [ false != ${service_running} ]; then
        LOG_INFO "-----------------------------------------------------------------"
        LOG_ERROR "Failed to stop ${1} software." "HIDE"
        LOG_ERROR "Failed to install iBMA software."
        exit 1
    else
        if [ false == ${OS_SUPPROT_SYSTEMCTL} ]; then
            # let systemd know service already exited.
            service ${1} stop >& /dev/null
        else
            # let systemd know service already exited.
            systemctl stop ${1} >& /dev/null
        fi
    fi
}

#*****************************************************************************
# Prototype    : old_ibma_service_proc
# Description  : old iBMA service process.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/10/12
#    Modification : Created function
#
#*****************************************************************************
function old_ibma_service_proc()
{
    local service_running=false
    local bob_service_running=false
    local service_list
    local i=0
    local running_flag=false

    if [[ ! -r "${OLD_BMA_SH_PATH}" ]] ; then
        return
    fi

    "${OLD_BMA_SH_PATH}" status >& /dev/null && service_running=true
    "${OLD_BOB_SH_PATH}" status >& /dev/null && bob_service_running=true

    service_list=("${SERVICE_NAME}" "${OLD_BMA_SH_PATH}" "${service_running}"
                  "${BOBSERVICE_NAME}" "${OLD_BOB_SH_PATH}" "${bob_service_running}")
    local service_item_num=3

    # service is running, exit if in quiet mode, stop service if in upgrade mode, else, prompt.
    while [ ${i} -le $(expr ${#service_list[@]} - ${service_item_num}) ]
    do
        if [ true == ${service_list[$(expr ${i} + 2)]} ]; then
            if [ false == ${IS_UPGRADE} ]; then
                if [ true == ${SILENT_MODE} ]; then
                    LOG_INFO "${service_list[${i}]} service is running. Stop it first."
                    running_flag=true
                    i=$(expr ${i} + ${service_item_num})
                    continue
                fi

                LOG_INFO "${service_list[${i}]} service is running."
                PS3="Enter your choice:"
                options=("Stop service" "Exit")
                select opt in "${options[@]}"; do
                    case $opt in
                        "Stop service")
                            LOG_INFO "Stopping ${service_list[${i}]} service."
                            break;
                            ;;
                        "Exit")
                            LOG_DEBUG "You chose to exit." "HIDE"
                            exit 1;
                            ;;
                        *)
                            LOG_INFO "Invalid option."
                            ;;
                    esac
                done
            fi

            # try stop service.
            stop_service "${service_list[${i}]}" "${service_list[$(expr ${i} + 1)]}"
        fi

        i=$(expr ${i} + ${service_item_num})
    done

    if [ false != ${running_flag} ]; then
        LOG_INFO "-----------------------------------------------------------------"
        LOG_ERROR "Failed to install ${SERVICE_NAME} software."
        exit 3
    fi

    if [[ -r "${SERVICE_SCRIPT}" ]] && [[ false != ${OS_SUPPROT_SYSTEMCTL} ]]; then
        rm -f "${SERVICE_SCRIPT}"
    fi

    if [[ -r "${BOBSERVICE_SCRIPT}" ]] && [[ false != ${OS_SUPPROT_SYSTEMCTL} ]]; then
        rm -f "${BOBSERVICE_SCRIPT}"
    fi
}

#*****************************************************************************
# Prototype    : sh_close_fd
# Description  : close fd [3,1023).
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/09/11
#    Modification : Created function
#
#*****************************************************************************
function sh_close_fd()
{
    count=3
    while [ ${count} -lt 1023 ]
    do
        eval "exec ${count}>&-"
        eval "exec ${count}<&-"
        count=$((count + 1))
    done
}

#*****************************************************************************
# Prototype    : record_upgrade_log
# Description  : record upgrade log.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/09/11
#    Modification : Created function
#
#*****************************************************************************
function record_upgrade_log()
{
    if [ false == ${IS_UPGRADE} ]; then
        return
    fi

    local system_date=$(date -d today +"%Y-%m-%d %H:%M:%S")
    echo "[ ${system_date} ] $1" 2> /dev/null >> "${IBMA_UPGRADE_LOG_PATH}"
}

#*****************************************************************************
# Prototype    : backup_old_ibma_config
# Description  : backup old iBMA version config dir.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 1-error, 0-success
#
#  History        :
#  1.Date         : 2017/09/11
#    Modification : Created function
#
#*****************************************************************************
function backup_old_ibma_config()
{
    ver_dir=${OLD_BMA_DIR}/${DIR_NAME}

    if [ -d "${ver_dir}" ]; then
        res=$(mkdir -pm 640 ${OLD_CONFIG_PATH} 2>/dev/null)
        if [ $? -ne 0 ]; then
            LOG_WARN "Create old iBMA version config directory has error."
            return 1
        fi

        if [ -d "${ver_dir}/config" ]; then
            cp -rf ${ver_dir}/config/* ${OLD_CONFIG_PATH}/ 2>/dev/null
            find ${ver_dir}/config/* -type d | xargs rm -rf
        fi
    fi

    return 0
}

#*****************************************************************************
# Prototype    : check_old_modules
# Description  : check if
#                      1) old iBMA is installed.
#                      2) old Black Box module is loaded.
# Parameter:
#   input:  NA
#   output: NA
# Return Value : NA. exit if old iBMA is installed, or Black Box is loaded.
#                    skip check if in force mode.
#
#  History        :
#  1.Date         : 2017/02/15
#    Modification : Created function
#  2.Date         : 2017/06/29
#    Modification : add check for old iBMA versions.
#
#*****************************************************************************
function check_old_modules()
{
    local hwdiag="${OLD_BMA_DIR}/bma/bin/hwdiag"

    if ${FORCE_MODE}; then
        LOG_INFO "Skip to check old iBMA, continue ..." "HIDE"
        return
    fi

    if [ -f "${hwdiag}" ]; then
        LOG_ERROR "An older version of iBMA (iBMA1.0) has been installed."
        LOG_ERROR "You need to uninstall it first."
        LOG_INFO "-----------------------------------------------------------------"
        exit 1;
    fi

    LOG_INFO "Old iBMA check passed." "HIDE"
}

#*****************************************************************************
# Prototype    : del_temp_files
# Description  : delete temp files.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2016/12/26
#    Modification : Created function
#    Modification : Add python Crypto
#  2.Date         : 2019/8/12
#    Modification : Delete the processing of iBMAUpgradeLog
#  3.Date         : 2019/8/30
#    Modification : Delete the config_old folder after install completed
#
#*****************************************************************************
function del_temp_files()
{
    if [ -d "${BMA_DIR}/${DIR_NAME}/log/backup" ]; then
        rm -rf "${BMA_DIR}/${DIR_NAME}/log/backup" 2>/dev/null
    fi

    if [ -d "${OLD_CONFIG_PATH}" ]; then
        rm -rf "${OLD_CONFIG_PATH}" 2>/dev/null
    fi
}

#*****************************************************************************
# Prototype    : print_firewall_warning
# Description  : print firewall warning.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/10/26
#    Modification : Created function
#  2.Date         : 2019/09/02
#    Modification : do not print if not veth mode.
#*****************************************************************************
function print_firewall_warning()
{
    local config="${BMA_DIR}/${DIR_NAME}/config/iBMA.ini"

    if [ ! -e "/sys/module/host_veth_drv" ]; then
        return
    fi

    if [ -e "${config}" ]; then
        BMA_NIC=$(read_from_file ${config} ${IBMA_SECTION} "${NIC_ITEM}")
        HTTP_PORT=$(read_from_file ${config} ${IBMA_SECTION} "${SERVER_PORT_ITEM}")
    fi

    LOG_INFO "Manually enable port ${HTTP_PORT} via the ${BMA_NIC} interface "
    LOG_INFO "on the firewall (TCP/${IP_MODE})."
}

#*****************************************************************************
# Prototype    : get_citrix_or_redhat
# Description  : get Citrix or Redhat, save to OS_NAME.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/08/16
#    Modification : Created function
#
#*****************************************************************************
function get_citrix_or_redhat()
{
    if [[ $1 == "/etc/redhat-release" ]] ;then
        os_sys=$(cat $1 | grep XenServer)
        if [ -z "$os_sys" ] ;then
            OS_NAME=RedHat
        else
            OS_NAME=Citrix
        fi
    fi
}

#*****************************************************************************
# Prototype    : get_os_name
# Description  : get OS name, save to OS_NAME.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2017/03/15
#    Modification : Created function
#
#*****************************************************************************
function get_os_name()
{
    # check lsb_release command.
    OS_NAME=$(lsb_release -i -s 2>/dev/null)
    if [ -n "${OS_NAME}" ] &&
        [[ ${OS_NAME_LIST} == *${OS_NAME}* ]]; then
        if [ ${OS_NAME} == "XenServer" ]; then
            OS_NAME=Citrix
        elif [ ${OS_NAME} == "OracleServer" ]; then
            OS_NAME=Oracle
        fi
        LOG_INFO "LSB OS is [${OS_NAME}]" "HIDE"
        return 0;
    fi

    # check os-specific file.
    local index=0
    while [ ${index} -lt ${#ETC_FILE[@]} ]; do
        if [ -e "${ETC_FILE[${index}]}" ]; then
            OS_NAME=${ETC_FILE[${index}+1]}
            get_citrix_or_redhat ${ETC_FILE[${index}]}
            LOG_INFO "ETC OS is [${OS_NAME}]" "HIDE"
            return 0;
        fi
        let index+=2;
    done

    # check other os file.
    local content=$(cat "${OTHER_OS_FILE}" 2> /dev/null);
    for os in ${OTHER_OS_NAME_LIST[*]}; do
        echo "${content}" | grep -i "${os}" -q
        if [ 0 == $? ]; then
            OS_NAME=${os}
            LOG_INFO "OTHER OS is [${OS_NAME}]" "HIDE"
            return 0;
        fi
    done

    LOG_HINT "System version does not support."
    LOG_ERROR "Failed to get the system version."

    exit 1;
}

#*****************************************************************************
# Prototype    : check_platform
# Description  : check the platform of the current system, exit if package
#                platform does not match the system
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, otherwise failed.
#
#  History        :
#  1.Date         : 2020/02/04
#    Modification : Created function
#  2.Date         : 2020/9/19
#    Modification : Add support for UOS, fix the problem that res_sys may return "unknown"
#*****************************************************************************
function check_platform()
{
    local res_sys=$(uname -p)
    if [[ "${res_sys}" == "unknown" ]]; then
        res_sys=$(uname -m)
    fi

    if [[ "${OS_NAME}" != "Ubuntu" ]] && [[ "${OS_NAME}" != "uos" ]] && [[ "${OS_NAME}" != "Debian" ]]; then
        local res_pkg=$(rpm -qpl ${G_WORK_DIR}/app/iBMA2.0-*.rpm | grep -i aarch64 | wc -l)
        if [[ ${res_pkg} != 0 ]] && [[ "${res_sys}" == "aarch64" ]]; then
            LOG_INFO "The platform of the package matches the current system(aarch64)" "HIDE"
            return 0
        elif [[ ${res_pkg} == 0 ]] && [[ "${res_sys}" =~ "x86" ]]; then
            LOG_INFO "The platform of the package matches the current system(x86)" "HIDE"
            return 0
        fi
    else
        local res_pkg=$(file ${G_WORK_DIR}/app/iBMA2.0-*.rpm | grep x86)
        if [[ -z "${res_pkg}" ]] && [[ "${res_sys}" == "aarch64" ]]; then
            LOG_INFO "The platform of the package matches the current system(aarch64)" "HIDE"
            return 0
        elif [[ -n "${res_pkg}" ]] && [[ "${res_sys}" =~ "x86" ]]; then
            LOG_INFO "The platform of the package matches the current system(x86)" "HIDE"
            return 0
        fi
    fi

    LOG_ERROR "The platform of the package does not match the current system(${res_sys})"
    exit 1
}

#*****************************************************************************
# Prototype    : handle_cdev_function
# Description  : # SUSE/Citrix does not support CDEV mode.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/8/22
#    Modification : Created function
#  2.Date         : 2019/10/17
#    Modification : Delete configuration if the setup tool does not exist.
#  3.Date         : 2020/02/17
#    Modification : Add arch check for SUSE, support veth for SUSE aarch64
#*****************************************************************************
function handle_cdev_function()
{
    if [ "${OS_NAME}" == "SUSE" -a "$(uname -p)" != "aarch64" ] \
       || [ "${OS_NAME}" == "Citrix" ] \
       || [ ! -x "${NETWORK_SETUP_TOOL}" ];then
        sed -i -e "/^${NETWORK_TYPE_ITEM}=/d" ${CONFIG}
        LOG_INFO "Delete CDEV function for ${OS_NAME}" "HIDE"
    fi
}

#*****************************************************************************
# Prototype    : handle_kbox_function
# Description  : delete kbox related files and configuration.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/08/26
#    Modification : delete hwkbox files if there is no hwkbox.
#*****************************************************************************
function handle_kbox_function()
{
    local kbox_item="iBMA_kbox"

    if [ -e "${BMA_DIR}/${DIR_NAME}/bin/hwkbox" ]; then
        return
    fi

    sed -i -e "/^${kbox_item}=/d" ${CONFIG}

    LOG_INFO "Delete all kbox related files and configuration successfully" "HIDE"
}

#*****************************************************************************
# Prototype    : update_config
# Description  : update config file.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2017/12/30
#    Modification : Created function
#  4.Date        : 2021/06/07
#    Modification: Add the iBMC_veth_ip attribute to the iBMA.ini file.
#
#*****************************************************************************
function update_config()
{
    if [ -d "${OLD_CONFIG_PATH}" ] && [ true == ${IS_UPGRADE} ] ; then
        # copy old WatchDog.ini to new config directory if old WatchDog.ini exists
        if [ -e ${OLD_CONFIG_PATH}/WatchDog.ini ] && [ -d ${BMA_DIR}/${DIR_NAME}/lib/Linux/config ] ; then
            cp -f ${OLD_CONFIG_PATH}/WatchDog.ini ${BMA_DIR}/${DIR_NAME}/lib/Linux/config/
        fi

        # copy old bob.ini to new config directory if old bob.ini exists
        if [ -e "${OLD_CONFIG_PATH}/bob.ini" ] && [ -d "${BMA_DIR}/${DIR_NAME}/tools/bob" ] ; then
            cp -f ${OLD_CONFIG_PATH}/bob.ini ${BMA_DIR}/${DIR_NAME}/tools/bob/
        fi

        # 3 param, $1 is old version iBMA log directory, $2 is new version iBMA log directory
        res=$(${BMA_DIR}/${DIR_NAME}/bin/UpdateCfg "${OLD_CONFIG_PATH}" ${BMA_DIR}/${DIR_NAME}/config 0 2>&1)
        if [ $? -eq 0 ]; then
            LOG_INFO "Update config return=${res}" "HIDE"
            ret=$(echo "$res"|grep -i "iBMA Version has no change" -q 2>/dev/null)
            if [ $? -eq 0 ]; then
                LOG_INFO "The same version of iBMA is already installed."
                record_upgrade_log "The same version of iBMA is already installed."
                exit 0
            else
                ret=$(echo "$res"|grep -i "The iBMA version is not supported" -q 2>/dev/null)
                if [ $? -eq 0 ]; then
                    LOG_INFO "The iBMA version is not supported."
                    record_upgrade_log "The iBMA version is not supported."
                    exit 1
                fi

                ret=$(echo "$res"|grep -i "Old version config not exist" -q 2>/dev/null)
                if [ $? -eq 0 ]; then
                    LOG_INFO "Old version config not exist." "HIDE"
                    return 0
                fi
            fi

            # get http type from the configuration file
            SUPPORT_HTTPS=$(read_from_file ${CONFIG} ${IBMA_SECTION} ${SUPPORT_HTTPS_ITEM})
            if [ "false" == "${SUPPORT_HTTPS}" ]; then
                LOG_INFO "-----------------------------------------------------------------"
                LOG_HINT "Redfish service can be started using HTTPS protocol"
                LOG_HINT "(iBMC version must be V3.16 or later)."
                LOG_HINT "You can modify ${SUPPORT_HTTPS_ITEM} in ${CONFIG}"
                LOG_HINT "and restart iBMA service for the modification to take effect."
                LOG_INFO "-----------------------------------------------------------------"
            fi

            iBMC_ip=$(read_from_file ${CONFIG} ${IBMA_SECTION} ${IBMC_IP})
            iBMC_veth_ip=$(read_from_file ${CONFIG} ${IBMA_SECTION} ${IBMC_VETH_IP})
            if [ "${iBMC_veth_ip}" != "${iBMC_ip}" ]; then
                save_to_config_file ${IBMA_SECTION} ${IBMC_VETH_IP} ${iBMC_ip}
            fi
        else
            LOG_ERROR "res1=${res}"
            LOG_ERROR "Failed to update config. Exiting ..."
            LOG_ERROR "${SERVICE_NAME} upgraded failed."
            record_upgrade_log "${SERVICE_NAME} upgraded failed."
            exit 1
        fi
    fi
}

#*****************************************************************************
# Prototype    : check_driver_files
# Description  : check driver file counts.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : 0 success, else failure.
#
#  History        :
#  1.Date         : 2019/04/02
#    Modification : Created function
#  2.Date         : 2020/12/07
#    Modification : Added the function of determining whether the driver is installed.
#*****************************************************************************
function check_driver_files()
{
    driver_count=$(find ./drivers/ -name "*" 2>/dev/null | wc -l 2>/dev/null)

    LOG_INFO "driver count=${driver_count}, total=${TOTAL_DRIVER_FILES}" "HIDE"

    [ "${driver_count}" -ge "${TOTAL_DRIVER_FILES}" ]
    return $?
}

#*****************************************************************************
# Prototype    : check_device
# Description  : Check whether the device is the virtual iBMA USB device.
# Parameter:
#   input:  $1 dev_path; $2 mount_path
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/9/3
#    Modification : Created function
#
#*****************************************************************************
function check_device()
{
    local dev_path=$1
    local mount_path=$2

    # consider an abnormal scenario that multiple devices are mounted to the same directory
    dev_array=(${dev_path// /})
    for dev in ${dev_array[@]}; do
        # get model of the device
        dev_name=${dev##*/}
        dev_link=$(readlink "/sys/block/${dev_name}")
        if [ -n "${dev_link}" ]; then
            model_path="$(realpath /sys/block/${dev_link})/../../model"
            cat "${model_path}" 2>/dev/null |grep "^${IBMA_USB_MODEL_NAME}" -q
            if [ $? -eq 0 ]; then
                echo "${mount_path},${dev}" > "${PARTITION_INFO_PATH}"
                chmod 400 "${PARTITION_INFO_PATH}"
                LOG_INFO "Record ${mount_path},${dev} in ${PARTITION_INFO_PATH}." "HIDE"
                return
            fi

            LOG_INFO "${dev} is not iBMA USB Device." "HIDE"
        else
            LOG_INFO "${dev} does not exist." "HIDE"
        fi
    done
}

#*****************************************************************************
# Prototype    : check_partition_info
# Description  : Check whether the directory of the installation script is in
#                the iBMA partition mounted by the virtual USB device.
# Parameter:
#   input:  NA.
#   output: NA
# Return Value : NA
#
#  History        :
#  1.Date         : 2019/9/3
#    Modification : Created function
#
#*****************************************************************************
function check_partition_info()
{
    # If the directory is Linux, and the corresponding device can be found in
    # /proc/mounts, create file BMA_DIR/ibma/log/partition_info to record
    # iBMA partition mount point and device path so that iBMA can use them to
    # unload the partition after starting.
    local cur_dir=$(basename ${G_WORK_DIR})

    if [ "${cur_dir}" == "Linux" ]; then
        mount_path=$(dirname ${G_WORK_DIR})
        if [ "${mount_path}" != '/' ]; then
            dev_path=$(cat "/proc/mounts" |grep "${mount_path}" |awk '{print $1}')
            if [ -n "${dev_path}" ]; then
                check_device "${dev_path}" "${mount_path}"
            fi
        fi
    fi
}

# parameter check
for optarg in "$@"; do
    case "${optarg}" in
        --enable-iBMC_event=*)
            IBMC_EVENT=$(echo "$optarg" | cut -d"=" -f2- 2>/dev/null)
            if [ false != "${IBMC_EVENT}" ] && [ true != "${IBMC_EVENT}" ]; then
                print_help
                exit 1
            fi
            CONFIG_IBMC_EVENT=true
            ;;
        --enable-iBMA_https=*)
            IBMA_SUPPORT_HTTPS=$(echo "$optarg" | cut -d"=" -f2- 2>/dev/null)
            if [ false != "${IBMA_SUPPORT_HTTPS}" ] && [ true != "${IBMA_SUPPORT_HTTPS}" ]; then
                print_help
                exit 1
            fi
            CONFIG_IBMA_SUPPORT_HTTPS=true
            # need to keep user's choice
            KEEP_SUPPORT_HTTPS_SELECT=true
            ;;
        '-s')
            SILENT_MODE=true
            DRIVER_INSTALL_MODE='-s'
            ;;
        '-c')
            custom_mode=true
            # need to keep user's choice
            KEEP_SUPPORT_HTTPS_SELECT=true
            DRIVER_INSTALL_MODE='-c'
            ;;
        '-u')
            FORCE_MODE=true
            IS_UPGRADE=true
            DRIVER_INSTALL_MODE='-u'
            # need to keep user's choice
            KEEP_SUPPORT_HTTPS_SELECT=true
            if [ -z $(read_from_file "${BMA_DIR}/${DIR_NAME}/config/iBMA.ini" ${IBMA_SECTION} ${SUPPORT_HTTPS_ITEM}) ]; then
                KEEP_SUPPORT_HTTPS_SELECT=false
            fi
            trap "" 1 2 3 11 15
            ;;
        '-f')
            FORCE_MODE=true
            ;;
        '-h')
            print_help
            exit 0
            ;;
        *)
            print_help
            exit 1
            ;;
   esac
done

if [[ ${custom_mode} == true && ${SILENT_MODE} == true ]] \
  || [[ ${custom_mode} == true && ${IS_UPGRADE} == true ]] \
  || [[ ${IS_UPGRADE} == true && ${SILENT_MODE} == true ]] \
  || [[ ${IS_UPGRADE} == false && ${SILENT_MODE} == false && ${custom_mode} == false ]] \
  || [[ ${SILENT_MODE} == false && ${CONFIG_IBMC_EVENT} == true ]] \
  || [[ ${SILENT_MODE} == false && ${CONFIG_IBMA_SUPPORT_HTTPS} == true ]]; then
    print_help
    exit 1
elif ${IS_UPGRADE} ; then
    SILENT_MODE=true
    # clean iBMAUpgradeLog before upgrade
    : > "${IBMA_UPGRADE_LOG_PATH}"
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "Starting to upgrade ${SERVICE_NAME}."
elif ${custom_mode} ; then
    # clean iBMAUpgradeLog before install in custom mode
    : > "${INSTALL_LOG_FILE}"
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "Starting to install ${SERVICE_NAME} in custom mode."
elif ${SILENT_MODE} ; then
    # clean iBMAUpgradeLog before install in silent mode
    : > "${INSTALL_LOG_FILE}"
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "Starting to install ${SERVICE_NAME} in silent mode."
fi

# check old iBMA or kernel modules.
check_old_modules

# get os name
get_os_name

# check if current platform is valid
check_platform

# find old version
find_old_ibma_path

# check driver files.
check_driver_files
if [ 0 -ne $? ]; then
    LOG_ERROR "The installation package is damaged."
    LOG_ERROR "You need to download the complete installation package and try again."
    exit 1
fi

# cmp old and new version config files ip conflicts
check_ip_conflicts
if [ 0 -ne $? ]; then
    exit 1
fi

if [ true == ${custom_mode} ]; then
    LOG_HINT "Press Ctrl + C to abort the installation."
    LOG_INFO ""
fi

if [ false == ${IS_UPGRADE} ]; then
    # stop old iBMA service
    old_ibma_service_proc
else
    if [ -e "${OLD_BMA_DIR}/ibma/config/iBMA.ini" ]; then
        OLD_IBMA_VER=$(read_from_file "${OLD_BMA_DIR}/ibma/config/iBMA.ini" ${IBMA_SECTION} "iBMA_version")
    fi

    if [ "${IBMA_VERSION}" == "${OLD_IBMA_VER}" ] ; then
        LOG_INFO "The same version of iBMA is already installed."
        record_upgrade_log "The same version of iBMA is already installed."
        ibma_upgrade_exit=true
        del_temp_files
        exit 0
    fi

    # stop old iBMA service
    old_ibma_service_proc

    # close fd
    sh_close_fd

    # backup old iBMA version config dir
    backup_old_ibma_config
fi

readlink "/bin/sh" |grep -w "dash" -q
if [ $? -eq 0 ]; then
    SH_CMD="bash"
else
    SH_CMD="sh"
fi

${SH_CMD} "${G_WORK_DIR}/drivers/install.sh" "${DRIVER_INSTALL_MODE}" "${FORCE_MODE}" "${IP_MODE}"
case $? in
    0)
        LOG_INFO "Install ${SERVICE_NAME} driver successfully." "HIDE"
        ;;
    3)
        IBMA_REBOOT_NEEDED=true;
        IBMA_RUN_IMMEDIATELY=false;
        ;;
    *)
        LOG_INFO "-----------------------------------------------------------------"
        LOG_ERROR "Failed to install ${SERVICE_NAME} driver."
        record_upgrade_log "${SERVICE_NAME} upgraded failed."
        ibma_upgrade_exit=true
        del_temp_files
        exit 1;
        ;;
esac

# set parameter
if [ true == ${custom_mode} ]; then
    # set up Redfish Server user
    set_redfish_server_user

    # set up Redfish Server port
    set_redfish_server_port

    # set up Socket Server port
    set_socket_server_port

    # set up Redfish Server protocol type
    set_redfish_server_protocol

    if ! ${IBMA_REBOOT_NEEDED}; then
        # start iBMA now
        set_if_ibma_start_right_now
    fi

    # confirm settings
    confirm_settings
fi

# remove last '/'
while [[ $BMA_DIR == /*/ ]] ; do
    BMA_DIR=${BMA_DIR%/}
done

LOG_INFO "Installing ${SERVICE_NAME} ..."

# delete old iBMA files, backup if need.
handle_old_ibma_files
if [ 0 -ne $? ]; then
    exit 1
fi

# check default ibma user in silent mode
if [ ${SILENT_MODE} == true ] && [ ${IS_UPGRADE} == false ]; then
    check_default_user
fi

# install iBMA.
install_ibma
if [ 0 -ne $? ]; then
    exit 1
fi

# Use Monitor_TCE.ini only on TCE server.
if [ -f ${BMA_DIR}/${DIR_NAME}/config/Monitor_TCE.ini ]; then
    cur_server=$(dmidecode -t2 | grep "Product Name:" | awk '$1=$1')
    cur_server=${cur_server#*"Product Name: "}
    if echo "$cur_server" | grep "CN9" > /dev/null; then
        LOG_INFO "Installation on TCE server: $cur_server, use Monitor_TCE.ini." "HIDE"
        mv -f ${BMA_DIR}/${DIR_NAME}/config/Monitor_TCE.ini ${BMA_DIR}/${DIR_NAME}/config/Monitor.ini
        rm -rf ${BMA_DIR}/${DIR_NAME}/lib64/*storelib* 2>/dev/null
    else
        LOG_INFO "Installation on $cur_server, delete the Monitor_TCE.ini for TCE." "HIDE"
        rm -f ${BMA_DIR}/${DIR_NAME}/config/Monitor_TCE.ini
        rm -f ${BMA_DIR}/${DIR_NAME}/config/MeshDetect.ini
    fi
fi

# move storelib64 to bin
if [ -d ${G_WORK_DIR}/tools ] && [ -d ${BMA_DIR}/${DIR_NAME}/bin ]; then
    chmod 550 ${G_WORK_DIR}/tools/* 2>/dev/null
    cp -a ${G_WORK_DIR}/tools/* ${BMA_DIR}/${DIR_NAME}/bin 2>/dev/null
fi

# SUSE/Citrix does not support CDEV mode.
handle_cdev_function

#update config
update_config

# delete kbox if not needed.
handle_kbox_function

if [ false == ${IS_UPGRADE} ]; then
    LOG_INFO "-----------------------------------------------------------------"
    LOG_INFO "${SERVICE_NAME} installed successfully."
fi

record_upgrade_log "${SERVICE_NAME} upgraded successfully."

# save current config to file
if [ false == ${IS_UPGRADE} ]; then
    if [ "" != "${HTTP_USER}" ]; then
        save_to_config_file ${IBMA_SECTION} ${USER_ITEM} ${HTTP_USER}
    fi

    if [ "0" != "${HTTP_PORT}" ]; then
        save_to_config_file ${IBMA_SECTION} ${SERVER_PORT_ITEM} ${HTTP_PORT}
    fi

    if [ "0" != "${SOCKET_PORT}" ]; then
        save_to_config_file ${IBMA_SECTION} ${SOCKET_PORT_ITEM} ${SOCKET_PORT}
    fi

    # the default value in the configuration file is false
    # need to be saved to the configuration file only when it is set to false
    if [ true == ${IBMC_EVENT} ]; then
        save_to_config_file ${IBMA_SECTION} ${BMC_EVENT_ITEM} ${IBMC_EVENT}
    fi

    if [ false == ${IBMA_SUPPORT_HTTPS} ]; then
        save_to_config_file ${IBMA_SECTION} ${SUPPORT_HTTPS_ITEM} ${IBMA_SUPPORT_HTTPS}
    fi
fi

# get user from the configuration file
HTTP_USER=$(read_from_file ${CONFIG} ${IBMA_SECTION} ${USER_ITEM})
if [ "root" == "${HTTP_USER}" ]; then
    LOG_HINT "Redfish service can be run by a non-root user."
    LOG_HINT "You can modify ${USER_ITEM} in ${CONFIG} file "
    LOG_HINT "and restart iBMA service for the modification to take effect."
fi

# create file named auto_select_https when need not to keep the user's choice
# Redfish service will decide according to this file whether to use HTTPS protocol when iBMC supports HTTPS
AUTO_SELECT_HTTPS_FILE="${BMA_DIR}/${DIR_NAME}/log/auto_select_https"

if [ -e ${AUTO_SELECT_HTTPS_FILE} ]; then
    rm -f ${AUTO_SELECT_HTTPS_FILE}
fi

if [ false == ${KEEP_SUPPORT_HTTPS_SELECT} ]; then
    touch ${AUTO_SELECT_HTTPS_FILE}
    chmod 440 ${AUTO_SELECT_HTTPS_FILE}
fi

# Check whether the directory of the installation script is in the iBMA partition
# mounted by the virtual USB device.
# If the directory is Linux, and the corresponding device can be found in
# /proc/mounts, create file BMA_DIR/ibma/log/partition_info to record
# iBMA partition mount point and device path so that iBMA can use them to
# unload the partition after starting.
PARTITION_INFO_NAME="partition_info"
PARTITION_INFO_PATH="${BMA_DIR}/${DIR_NAME}/log/${PARTITION_INFO_NAME}"
IBMA_USB_MODEL_NAME="iBMA USB Device"
check_partition_info

# create flag file BMA_DIR/ibma/log/owner to
# indicate the BMA_DIR is created by iBMA
OWNER_FILE_PATH="${BMA_DIR}/${DIR_NAME}/log/owner"
if [ true == ${IBMA_IS_HUAWEI_OWNER} ]; then
    touch ${OWNER_FILE_PATH}
    chmod 440 ${OWNER_FILE_PATH}
fi

# start iBMA Service.
start_service
if [[ $? -ne 0 ]]; then
    if [ true == ${IS_UPGRADE} ] && [ -d "${OLD_BMA_DIR}" ]; then
        ibma_upgrade_exit=true
        LOG_ERROR "${SERVICE_NAME} upgraded failed."
        record_upgrade_log "${SERVICE_NAME} upgraded failed."
        resume_ibma
        if [ $? == 0 ]; then
            LOG_INFO "Resume iBMA successfully." "HIDE"
            del_temp_files
            exit 0
        else
            LOG_ERROR "Resume iBMA failed." "HIDE"
            del_temp_files
            exit 1
        fi
    fi
else
    if [ true == ${IS_UPGRADE} ]; then
        LOG_INFO "${SERVICE_NAME} upgraded successfully."
    fi
fi

print_firewall_warning

# delete temporary files before the installation completes
del_temp_files

# success.
exit 0
