/*
 * Decompiled with CFR 0.152.
 */
package org.kitesdk.data.spi.filesystem;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.avro.Schema;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.kitesdk.compat.DynMethods;
import org.kitesdk.data.DatasetDescriptor;
import org.kitesdk.data.DatasetIOException;
import org.kitesdk.data.Format;
import org.kitesdk.data.Formats;
import org.kitesdk.data.IncompatibleSchemaException;
import org.kitesdk.data.PartitionStrategy;
import org.kitesdk.data.ValidationException;
import org.kitesdk.data.spi.Pair;
import org.kitesdk.data.spi.SchemaUtil;
import org.kitesdk.data.spi.Schemas;
import org.kitesdk.data.spi.filesystem.PathFilters;
import org.kitesdk.data.spi.partition.ProvidedFieldPartitioner;
import org.kitesdk.shaded.com.google.common.base.Preconditions;
import org.kitesdk.shaded.com.google.common.base.Splitter;
import org.kitesdk.shaded.com.google.common.collect.Iterables;
import org.kitesdk.shaded.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemUtil {
    private static final Logger LOG;
    private static final List<Format> SUPPORTED_FORMATS;
    private static final DynMethods.UnboundMethod IS_SYMLINK;
    private static final Splitter DOT;

    public static void ensureLocationExists(DatasetDescriptor descriptor, Configuration conf) {
        Preconditions.checkNotNull(descriptor.getLocation(), "Cannot get FileSystem for a descriptor with no location");
        Path dataPath = new Path(descriptor.getLocation().toString());
        FileSystem fs = null;
        try {
            fs = dataPath.getFileSystem(conf);
        }
        catch (IOException e) {
            throw new DatasetIOException("Cannot get FileSystem for descriptor: " + descriptor, e);
        }
        try {
            if (!fs.exists(dataPath)) {
                fs.mkdirs(dataPath);
            }
        }
        catch (IOException e) {
            throw new DatasetIOException("Cannot access data location", e);
        }
    }

    static List<Pair<Path, Path>> stageMove(FileSystem fs, Path src, Path dest, String ext) {
        ArrayList<Pair<Path, Path>> staged;
        try {
            if (!fs.exists(dest)) {
                fs.mkdirs(dest);
            }
            FileStatus[] stats = fs.listStatus(src);
            staged = Lists.newArrayList();
            for (FileStatus stat : stats) {
                if (stat.isDir()) continue;
                Path srcFile = stat.getPath();
                Path dotFile = new Path(dest, "." + srcFile.getName() + "." + ext);
                Path destFile = new Path(dest, srcFile.getName());
                if (fs.rename(srcFile, dotFile)) {
                    staged.add(Pair.of(dotFile, destFile));
                    continue;
                }
                throw new IOException("Failed to rename " + srcFile + " to " + dotFile);
            }
        }
        catch (IOException e) {
            throw new DatasetIOException("Could not move contents of " + src + " to " + dest, e);
        }
        return staged;
    }

    static void finishMove(FileSystem fs, List<Pair<Path, Path>> staged) {
        try {
            for (Pair<Path, Path> pair : staged) {
                if (fs.rename(pair.first(), pair.second())) continue;
                throw new IOException("Failed to rename " + pair.first() + " to " + pair.second());
            }
        }
        catch (IOException e) {
            throw new DatasetIOException("Could not finish replacement", e);
        }
    }

    static void replace(FileSystem fs, Path root, Path destination, Path replacement, List<Path> removals) {
        try {
            Path staged;
            if (!fs.exists(destination)) {
                fs.mkdirs(destination);
            }
            if (!(staged = new Path(destination.getParent(), "." + destination.getName() + ".replacement")).equals((Object)replacement) && !fs.rename(replacement, staged)) {
                throw new IOException("Failed to rename " + replacement + " to " + staged);
            }
            for (Path toRemove : removals) {
                if (toRemove.equals((Object)destination)) continue;
                FileSystemUtil.cleanlyDelete(fs, root, toRemove, false);
            }
            fs.delete(destination, true);
            if (!fs.rename(staged, destination)) {
                throw new IOException("Failed to rename " + staged + " to " + destination);
            }
        }
        catch (IOException e) {
            throw new DatasetIOException("Could not replace " + destination + " with " + replacement, e);
        }
    }

    static boolean cleanlyDelete(FileSystem fs, Path root, Path path) {
        return FileSystemUtil.cleanlyDelete(fs, root, path, false);
    }

    static boolean cleanlyMoveToTrash(FileSystem fs, Path root, Path path) {
        return FileSystemUtil.cleanlyDelete(fs, root, path, true);
    }

    private static boolean cleanlyDelete(FileSystem fs, Path root, Path path, boolean useTrash) {
        Preconditions.checkNotNull(fs, "File system cannot be null");
        Preconditions.checkNotNull(root, "Root path cannot be null");
        Preconditions.checkNotNull(path, "Path to delete cannot be null");
        try {
            boolean deleted;
            Path relativePath = path.isAbsolute() ? new Path(root.toUri().relativize(path.toUri())) : path;
            if (relativePath.isAbsolute()) {
                LOG.debug("Deleting path {}", path);
                deleted = useTrash ? Trash.moveToAppropriateTrash((FileSystem)fs, (Path)path, (Configuration)fs.getConf()) : fs.delete(path, true);
            } else {
                Path absolute = new Path(root, relativePath);
                LOG.debug("Deleting path {}", absolute);
                deleted = useTrash ? Trash.moveToAppropriateTrash((FileSystem)fs, (Path)absolute, (Configuration)fs.getConf()) : fs.delete(absolute, true);
                deleted |= FileSystemUtil.deleteParentDirectoriesIfEmpty(fs, root, absolute);
            }
            return deleted;
        }
        catch (IOException ex) {
            throw new DatasetIOException("Could not cleanly delete path:" + path, ex);
        }
    }

    static boolean deleteParentDirectoriesIfEmpty(FileSystem fs, Path root, Path path) throws IOException {
        boolean deleted = false;
        try {
            FileStatus[] stats;
            Path current = path.getParent();
            while (!(current.equals((Object)root) || current.getParent() == null || (stats = fs.listStatus(current)) != null && stats.length != 0)) {
                LOG.debug("Deleting empty path {}", current);
                deleted = fs.delete(current, true) || deleted;
                current = current.getParent();
            }
        }
        catch (FileNotFoundException e) {
            LOG.debug("Path does not exist it may have been deleted by another process.", e);
        }
        return deleted;
    }

    public static Schema schema(String name, FileSystem fs, Path location) throws IOException {
        if (!fs.exists(location)) {
            return null;
        }
        return FileSystemUtil.visit(new GetSchema(name), fs, location);
    }

    public static PartitionStrategy strategy(FileSystem fs, Path location) throws IOException {
        if (!fs.exists(location)) {
            return null;
        }
        List<Pair<String, Class<? extends Comparable>>> pairs = FileSystemUtil.visit(new GetPartitionInfo(), fs, location);
        if (pairs == null || pairs.isEmpty() || pairs.size() <= 1) {
            return null;
        }
        PartitionStrategy.Builder builder = new PartitionStrategy.Builder();
        for (int i = 1; i < pairs.size(); ++i) {
            Pair<String, Class<? extends Comparable>> pair = pairs.get(i);
            builder.provided(pair.first() == null ? "partition_" + i : pair.first(), ProvidedFieldPartitioner.valuesString(pair.second()));
        }
        return builder.build();
    }

    public static Format format(FileSystem fs, Path location) throws IOException {
        if (!fs.exists(location)) {
            return null;
        }
        return FileSystemUtil.visit(new GetFormat(), fs, location);
    }

    private static <T> T visit(PathVisitor<T> visitor, FileSystem fs, Path path) throws IOException {
        return FileSystemUtil.visit(visitor, fs, path, Lists.newArrayList());
    }

    private static <T> T visit(PathVisitor<T> visitor, FileSystem fs, Path path, List<Path> followedLinks) throws IOException {
        FileStatus[] statuses;
        if (fs.isFile(path)) {
            return visitor.file(fs, path);
        }
        if (IS_SYMLINK != null && ((Boolean)IS_SYMLINK.invoke(fs.getFileStatus(path), new Object[0])).booleanValue()) {
            Preconditions.checkArgument(!followedLinks.contains(path), "Encountered recursive path structure at link: " + path);
            followedLinks.add(path);
            return FileSystemUtil.visit(visitor, fs, fs.getLinkTarget(path), followedLinks);
        }
        ArrayList<T> children = Lists.newArrayList();
        for (FileStatus stat : statuses = fs.listStatus(path, PathFilters.notHidden())) {
            children.add(FileSystemUtil.visit(visitor, fs, stat.getPath()));
        }
        return visitor.directory(fs, path, children);
    }

    public static Collection<DatasetDescriptor> findPotentialDatasets(FileSystem fs, Path path) throws IOException {
        ArrayList<DatasetDescriptor> descriptors = Lists.newArrayList();
        Result result = FileSystemUtil.visit(new FindDatasets(), fs, path);
        if (result instanceof Result.Table) {
            descriptors.add(FileSystemUtil.descriptor(fs, (Result.Table)result));
        } else if (result instanceof Result.Group) {
            for (Result.Table table : ((Result.Group)result).tables) {
                descriptors.add(FileSystemUtil.descriptor(fs, table));
            }
        }
        return descriptors;
    }

    private static DatasetDescriptor descriptor(FileSystem fs, Result.Table table) throws IOException {
        PartitionStrategy strategy = FileSystemUtil.strategy(fs, table.location);
        DatasetDescriptor.Builder builder = new DatasetDescriptor.Builder().format(table.format).schema(table.schema).partitionStrategy(strategy).location(table.location);
        if (table.depth < 0) {
            builder.property("kite.filesystem.mixed-depth", "true");
        }
        return builder.build();
    }

    private static Format formatFromExt(Path path) {
        String filename = path.getName();
        String ext = Iterables.getLast(DOT.split(filename));
        for (Format format : SUPPORTED_FORMATS) {
            if (!ext.equals(format.getExtension())) continue;
            return format;
        }
        return null;
    }

    private static Schema merge(@Nullable Schema left, @Nullable Schema right) {
        if (left == null) {
            return right;
        }
        if (right == null) {
            return left;
        }
        return SchemaUtil.merge(left, right);
    }

    public static boolean supportsRename(URI fsUri, Configuration conf) {
        String fsUriScheme = fsUri.getScheme();
        return conf.getBoolean("kite.writer.fs-supports-rename", !fsUriScheme.equalsIgnoreCase("s3n") && !fsUriScheme.equalsIgnoreCase("s3a"));
    }

    static {
        DynMethods.UnboundMethod isSymlink;
        LOG = LoggerFactory.getLogger(FileSystemUtil.class);
        SUPPORTED_FORMATS = Lists.newArrayList(Formats.AVRO, Formats.PARQUET, Formats.JSON, Formats.CSV);
        try {
            isSymlink = new DynMethods.Builder("isSymlink").impl(FileStatus.class, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            isSymlink = null;
        }
        IS_SYMLINK = isSymlink;
        DOT = Splitter.on('.');
    }

    private static class FindDatasets
    extends PathVisitor<Result> {
        private FindDatasets() {
        }

        @Override
        Result directory(FileSystem fs, Path path, List<Result> children) throws IOException {
            boolean allCompatible = true;
            boolean containsUnknown = false;
            Schema mergedSchema = null;
            Format onlyFormat = null;
            int depth = 0;
            ArrayList<Result.Table> tables = Lists.newArrayList();
            for (Result child : children) {
                if (child instanceof Result.Unknown) {
                    allCompatible = false;
                    containsUnknown = true;
                    continue;
                }
                if (child instanceof Result.Group) {
                    Result.Group group = (Result.Group)child;
                    if ((containsUnknown |= group.containsUnknown) || !group.tables.isEmpty()) {
                        allCompatible = false;
                    }
                    tables.addAll(group.tables);
                    continue;
                }
                Result.Table table = (Result.Table)child;
                tables.add(table);
                if (!allCompatible) continue;
                try {
                    mergedSchema = FileSystemUtil.merge(mergedSchema, table.schema);
                }
                catch (IncompatibleSchemaException e) {
                    allCompatible = false;
                }
                if (onlyFormat == null) {
                    onlyFormat = table.format;
                } else if (onlyFormat != table.format) {
                    allCompatible = false;
                }
                if (depth == 0) {
                    depth = table.depth;
                    continue;
                }
                if (depth == table.depth) continue;
                depth = -1;
            }
            if (allCompatible && tables.size() > 0) {
                if (tables.size() == 1) {
                    return (Result)tables.get(0);
                }
                return new Result.Table(path, onlyFormat, mergedSchema, depth);
            }
            return new Result.Group(tables, containsUnknown);
        }

        @Override
        Result file(FileSystem fs, Path path) throws IOException {
            Format format = FileSystemUtil.formatFromExt(path);
            Schema schema = null;
            if (format == Formats.AVRO) {
                schema = Schemas.fromAvro(fs, path);
            } else if (format == Formats.PARQUET) {
                schema = Schemas.fromParquet(fs, path);
            } else if (format == Formats.JSON) {
                schema = Schemas.fromJSON("record", fs, path);
            }
            if (schema == null) {
                return new Result.Unknown();
            }
            return new Result.Table(path, format, schema, path.depth());
        }
    }

    private static interface Result {

        public static class Group
        implements Result {
            private final List<Table> tables;
            private final boolean containsUnknown;

            public Group(List<Table> tables, boolean containsUnknown) {
                this.tables = tables;
                this.containsUnknown = containsUnknown;
            }
        }

        public static class Table
        implements Result {
            private static final int UNKNOWN_DEPTH = 0;
            private static final int MIXED_DEPTH = -1;
            private final Path location;
            private final Format format;
            private final Schema schema;
            private final int depth;

            public Table(Path location, Format format, Schema schema, int depth) {
                this.location = location;
                this.format = format;
                this.schema = schema;
                this.depth = depth;
            }
        }

        public static class Unknown
        implements Result {
        }
    }

    private static class GetSchema
    extends PathVisitor<Schema> {
        private final String name;

        public GetSchema(String name) {
            this.name = name;
        }

        @Override
        Schema directory(FileSystem fs, Path path, List<Schema> schemas) {
            Schema merged = null;
            for (Schema schema : schemas) {
                merged = FileSystemUtil.merge(merged, schema);
            }
            return merged;
        }

        @Override
        Schema file(FileSystem fs, Path path) throws IOException {
            String filename = path.getName();
            if (filename.endsWith(Formats.AVRO.getExtension())) {
                return Schemas.fromAvro(fs, path);
            }
            if (filename.endsWith(Formats.PARQUET.getExtension())) {
                return Schemas.fromParquet(fs, path);
            }
            if (filename.endsWith(Formats.JSON.getExtension())) {
                return Schemas.fromJSON(this.name, fs, path);
            }
            return null;
        }
    }

    private static class GetFormat
    extends PathVisitor<Format> {
        private GetFormat() {
        }

        @Override
        Format directory(FileSystem fs, Path path, List<Format> formats) throws IOException {
            Format format = null;
            for (Format otherFormat : formats) {
                if (format == null) {
                    format = otherFormat;
                    continue;
                }
                if (format.equals(otherFormat)) continue;
                throw new ValidationException(String.format("Path contains multiple formats (%s, %s): %s", format, otherFormat, path));
            }
            return format;
        }

        @Override
        Format file(FileSystem fs, Path path) throws IOException {
            return FileSystemUtil.formatFromExt(path);
        }
    }

    private static class GetPartitionInfo
    extends PathVisitor<List<Pair<String, Class<? extends Comparable>>>> {
        private static final Splitter EQUALS = Splitter.on('=').limit(2).trimResults();

        private GetPartitionInfo() {
        }

        @Override
        List<Pair<String, Class<? extends Comparable>>> directory(FileSystem fs, Path path, List<List<Pair<String, Class<? extends Comparable>>>> children) throws IOException {
            String value;
            String name;
            ArrayList<Pair<String, Class<? extends Comparable>>> accumulated = Lists.newArrayList();
            for (List<Pair<String, Class<? extends Comparable>>> child : children) {
                if (child == null) continue;
                for (int i = 0; i < child.size(); ++i) {
                    if (accumulated.size() > i) {
                        Pair<String, Class<? extends Comparable>> pair = this.merge((Pair)accumulated.get(i), child.get(i));
                        accumulated.set(i, pair);
                        continue;
                    }
                    if (child.get(i) == null) continue;
                    accumulated.add(child.get(i));
                }
            }
            ArrayList<String> parts = Lists.newArrayList(EQUALS.split(path.getName()));
            if (parts.size() == 2) {
                name = (String)parts.get(0);
                value = (String)parts.get(1);
            } else {
                name = null;
                value = (String)parts.get(0);
            }
            accumulated.add(0, new Pair<String, Class<? extends Comparable>>(name, this.dataClass(value)));
            return accumulated;
        }

        @Override
        List<Pair<String, Class<? extends Comparable>>> file(FileSystem fs, Path path) throws IOException {
            return null;
        }

        public Pair<String, Class<? extends Comparable>> merge(Pair<String, Class<? extends Comparable>> left, Pair<String, Class<? extends Comparable>> right) {
            String name = left.first();
            if (name == null || name.isEmpty()) {
                name = right.first();
            }
            if (left.second() == String.class) {
                return new Pair<String, Class<? extends Comparable>>(name, String.class);
            }
            if (right.second() == String.class) {
                return new Pair<String, Class<? extends Comparable>>(name, String.class);
            }
            if (left.second() == Long.class) {
                return new Pair<String, Class<? extends Comparable>>(name, Long.class);
            }
            if (right.second() == Long.class) {
                return new Pair<String, Class<? extends Comparable>>(name, Long.class);
            }
            return new Pair<String, Class<? extends Comparable>>(name, Integer.class);
        }

        public Class<? extends Comparable> dataClass(String value) {
            try {
                Integer.parseInt(value);
                return Integer.class;
            }
            catch (NumberFormatException numberFormatException) {
                try {
                    Long.parseLong(value);
                    return Long.class;
                }
                catch (NumberFormatException numberFormatException2) {
                    return String.class;
                }
            }
        }
    }

    private static abstract class PathVisitor<T> {
        private PathVisitor() {
        }

        abstract T directory(FileSystem var1, Path var2, List<T> var3) throws IOException;

        abstract T file(FileSystem var1, Path var2) throws IOException;
    }
}

