/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.snapshot;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.protocol.SnapshotException;
import org.apache.hadoop.hdfs.server.namenode.Content;
import org.apache.hadoop.hdfs.server.namenode.ContentSummaryComputationContext;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeMap;
import org.apache.hadoop.hdfs.server.namenode.INodeReference;
import org.apache.hadoop.hdfs.server.namenode.Quota;
import org.apache.hadoop.hdfs.server.namenode.snapshot.DirectoryWithSnapshotFeature;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotDiffInfo;
import org.apache.hadoop.hdfs.util.Diff;
import org.apache.hadoop.hdfs.util.ReadOnlyList;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class INodeDirectorySnapshottable
extends INodeDirectory {
    static final int SNAPSHOT_LIMIT = 65536;
    private final List<Snapshot> snapshotsByNames = new ArrayList<Snapshot>();
    private int snapshotQuota = 65536;

    public static INodeDirectorySnapshottable valueOf(INode inode, String src) throws IOException {
        INodeDirectory dir = INodeDirectory.valueOf(inode, src);
        if (!dir.isSnapshottable()) {
            throw new SnapshotException("Directory is not a snapshottable directory: " + src);
        }
        return (INodeDirectorySnapshottable)dir;
    }

    ReadOnlyList<Snapshot> getSnapshotsByNames() {
        return ReadOnlyList.Util.asReadOnlyList(this.snapshotsByNames);
    }

    public INodeDirectorySnapshottable(INodeDirectory dir) {
        super(dir, true, dir.getFeatures());
        if (!this.isWithSnapshot()) {
            this.addSnapshotFeature(null);
        }
    }

    public int getNumSnapshots() {
        return this.snapshotsByNames.size();
    }

    private int searchSnapshot(byte[] snapshotName) {
        return Collections.binarySearch(this.snapshotsByNames, snapshotName);
    }

    public Snapshot getSnapshot(byte[] snapshotName) {
        int i = this.searchSnapshot(snapshotName);
        return i < 0 ? null : this.snapshotsByNames.get(i);
    }

    public Snapshot getSnapshotById(int sid) {
        for (Snapshot s : this.snapshotsByNames) {
            if (s.getId() != sid) continue;
            return s;
        }
        return null;
    }

    public ReadOnlyList<Snapshot> getSnapshotList() {
        return ReadOnlyList.Util.asReadOnlyList(this.snapshotsByNames);
    }

    void renameSnapshot(String path, String oldName, String newName) throws SnapshotException {
        if (newName.equals(oldName)) {
            return;
        }
        int indexOfOld = this.searchSnapshot(DFSUtil.string2Bytes(oldName));
        if (indexOfOld < 0) {
            throw new SnapshotException("The snapshot " + oldName + " does not exist for directory " + path);
        }
        byte[] newNameBytes = DFSUtil.string2Bytes(newName);
        int indexOfNew = this.searchSnapshot(newNameBytes);
        if (indexOfNew >= 0) {
            throw new SnapshotException("The snapshot " + newName + " already exists for directory " + path);
        }
        Snapshot snapshot = this.snapshotsByNames.remove(indexOfOld);
        Snapshot.Root ssRoot = snapshot.getRoot();
        ssRoot.setLocalName(newNameBytes);
        indexOfNew = -indexOfNew - 1;
        if (indexOfNew <= indexOfOld) {
            this.snapshotsByNames.add(indexOfNew, snapshot);
        } else {
            this.snapshotsByNames.add(indexOfNew - 1, snapshot);
        }
    }

    public int getSnapshotQuota() {
        return this.snapshotQuota;
    }

    public void setSnapshotQuota(int snapshotQuota) {
        if (snapshotQuota < 0) {
            throw new HadoopIllegalArgumentException("Cannot set snapshot quota to " + snapshotQuota + " < 0");
        }
        this.snapshotQuota = snapshotQuota;
    }

    @Override
    public boolean isSnapshottable() {
        return true;
    }

    void addSnapshot(Snapshot snapshot) {
        this.snapshotsByNames.add(snapshot);
    }

    Snapshot addSnapshot(int id, String name) throws SnapshotException, QuotaExceededException {
        int n = this.getNumSnapshots();
        if (n + 1 > this.snapshotQuota) {
            throw new SnapshotException("Failed to add snapshot: there are already " + n + " snapshot(s) and the snapshot quota is " + this.snapshotQuota);
        }
        Snapshot s = new Snapshot(id, name, this);
        byte[] nameBytes = s.getRoot().getLocalNameBytes();
        int i = this.searchSnapshot(nameBytes);
        if (i >= 0) {
            throw new SnapshotException("Failed to add snapshot: there is already a snapshot with the same name \"" + Snapshot.getSnapshotName(s) + "\".");
        }
        DirectoryWithSnapshotFeature.DirectoryDiff d = (DirectoryWithSnapshotFeature.DirectoryDiff)this.getDiffs().addDiff(id, this);
        d.setSnapshotRoot(s.getRoot());
        this.snapshotsByNames.add(-i - 1, s);
        this.updateModificationTime(Time.now(), 0x7FFFFFFE);
        s.getRoot().setModificationTime(this.getModificationTime(), 0x7FFFFFFE);
        return s;
    }

    Snapshot removeSnapshot(String snapshotName, INode.BlocksMapUpdateInfo collectedBlocks, List<INode> removedINodes) throws SnapshotException {
        int i = this.searchSnapshot(DFSUtil.string2Bytes(snapshotName));
        if (i < 0) {
            throw new SnapshotException("Cannot delete snapshot " + snapshotName + " from path " + this.getFullPathName() + ": the snapshot does not exist.");
        }
        Snapshot snapshot = this.snapshotsByNames.get(i);
        int prior = Snapshot.findLatestSnapshot(this, snapshot.getId());
        try {
            Quota.Counts counts = this.cleanSubtree(snapshot.getId(), prior, collectedBlocks, removedINodes, true);
            INodeDirectory parent = this.getParent();
            if (parent != null) {
                parent.addSpaceConsumed(-counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE), true);
            }
        }
        catch (QuotaExceededException e) {
            LOG.error("BUG: removeSnapshot increases namespace usage.", e);
        }
        this.snapshotsByNames.remove(i);
        return snapshot;
    }

    @Override
    public ContentSummaryComputationContext computeContentSummary(ContentSummaryComputationContext summary) {
        super.computeContentSummary(summary);
        summary.getCounts().add(Content.SNAPSHOT, this.snapshotsByNames.size());
        summary.getCounts().add(Content.SNAPSHOTTABLE_DIRECTORY, 1L);
        return summary;
    }

    SnapshotDiffInfo computeDiff(String from, String to) throws SnapshotException {
        Snapshot fromSnapshot = this.getSnapshotByName(from);
        Snapshot toSnapshot = this.getSnapshotByName(to);
        if (from.equals(to)) {
            return null;
        }
        SnapshotDiffInfo diffs = new SnapshotDiffInfo(this, fromSnapshot, toSnapshot);
        this.computeDiffRecursively(this, new ArrayList<byte[]>(), diffs);
        return diffs;
    }

    private Snapshot getSnapshotByName(String snapshotName) throws SnapshotException {
        Snapshot s = null;
        if (snapshotName != null && !snapshotName.isEmpty()) {
            int index = this.searchSnapshot(DFSUtil.string2Bytes(snapshotName));
            if (index < 0) {
                throw new SnapshotException("Cannot find the snapshot of directory " + this.getFullPathName() + " with name " + snapshotName);
            }
            s = this.snapshotsByNames.get(index);
        }
        return s;
    }

    private void computeDiffRecursively(INode node, List<byte[]> parentPath, SnapshotDiffInfo diffReport) {
        INodeFile file;
        boolean change;
        Snapshot earlierSnapshot = diffReport.isFromEarlier() ? diffReport.getFrom() : diffReport.getTo();
        Snapshot laterSnapshot = diffReport.isFromEarlier() ? diffReport.getTo() : diffReport.getFrom();
        byte[][] relativePath = (byte[][])parentPath.toArray((T[])new byte[parentPath.size()][]);
        if (node.isDirectory()) {
            boolean change2;
            DirectoryWithSnapshotFeature.ChildrenDiff diff = new DirectoryWithSnapshotFeature.ChildrenDiff();
            INodeDirectory dir = node.asDirectory();
            DirectoryWithSnapshotFeature sf = dir.getDirectoryWithSnapshotFeature();
            if (sf != null && (change2 = sf.computeDiffBetweenSnapshots(earlierSnapshot, laterSnapshot, diff, dir))) {
                diffReport.addDirDiff(dir, relativePath, diff);
            }
            ReadOnlyList<INode> children = dir.getChildrenList(earlierSnapshot.getId());
            for (INode child : children) {
                byte[][] renameTargetPath;
                boolean toProcess;
                byte[] name = child.getLocalNameBytes();
                boolean bl = toProcess = diff.searchIndex(Diff.ListType.DELETED, name) < 0;
                if (!toProcess && child instanceof INodeReference.WithName && (renameTargetPath = this.findRenameTargetPath((INodeReference.WithName)child, laterSnapshot == null ? 0x7FFFFFFE : laterSnapshot.getId())) != null) {
                    toProcess = true;
                    diffReport.setRenameTarget(child.getId(), renameTargetPath);
                }
                if (!toProcess) continue;
                parentPath.add(name);
                this.computeDiffRecursively(child, parentPath, diffReport);
                parentPath.remove(parentPath.size() - 1);
            }
        } else if (node.isFile() && node.asFile().isWithSnapshot() && (change = (file = node.asFile()).getFileWithSnapshotFeature().changedBetweenSnapshots(file, earlierSnapshot, laterSnapshot))) {
            diffReport.addFileDiff(file, relativePath);
        }
    }

    private byte[][] findRenameTargetPath(INodeReference.WithName wn, int snapshotId) {
        INode inode = wn.getReferredINode();
        LinkedList<byte[]> ancestors = Lists.newLinkedList();
        while (inode != null) {
            int sid;
            INode parent;
            if (inode == this) {
                return (byte[][])ancestors.toArray((T[])new byte[ancestors.size()][]);
            }
            if (inode instanceof INodeReference.WithCount) {
                inode = ((INodeReference.WithCount)inode).getParentRef(snapshotId);
                continue;
            }
            INode iNode = parent = inode.getParentReference() != null ? inode.getParentReference() : inode.getParent();
            if (parent != null && parent instanceof INodeDirectory && (sid = parent.asDirectory().searchChild(inode)) < snapshotId) {
                return null;
            }
            if (!(parent instanceof INodeReference.WithCount)) {
                ancestors.addFirst(inode.getLocalNameBytes());
            }
            inode = parent;
        }
        return null;
    }

    INodeDirectory replaceSelf(int latestSnapshotId, INodeMap inodeMap) throws QuotaExceededException {
        if (latestSnapshotId == 0x7FFFFFFE) {
            Preconditions.checkState(this.getDirectoryWithSnapshotFeature().getLastSnapshotId() == 0x7FFFFFFE, "this=%s", this);
        }
        INodeDirectory dir = this.replaceSelf4INodeDirectory(inodeMap);
        if (latestSnapshotId != 0x7FFFFFFE) {
            dir.recordModification(latestSnapshotId);
        }
        return dir;
    }

    @Override
    public String toDetailString() {
        return super.toDetailString() + ", snapshotsByNames=" + this.snapshotsByNames;
    }

    @Override
    public void dumpTreeRecursively(PrintWriter out, StringBuilder prefix, int snapshot) {
        super.dumpTreeRecursively(out, prefix, snapshot);
        if (snapshot == 0x7FFFFFFE) {
            out.println();
            out.print(prefix);
            out.print("Snapshot of ");
            String name = this.getLocalName();
            out.print(name.isEmpty() ? "/" : name);
            out.print(": quota=");
            out.print(this.getSnapshotQuota());
            int n = 0;
            for (DirectoryWithSnapshotFeature.DirectoryDiff diff : this.getDiffs()) {
                if (!diff.isSnapshotRoot()) continue;
                ++n;
            }
            Preconditions.checkState(n == this.snapshotsByNames.size(), "#n=" + n + ", snapshotsByNames.size()=" + this.snapshotsByNames.size());
            out.print(", #snapshot=");
            out.println(n);
            INodeDirectorySnapshottable.dumpTreeRecursively(out, prefix, new Iterable<INodeDirectory.SnapshotAndINode>(){

                @Override
                public Iterator<INodeDirectory.SnapshotAndINode> iterator() {
                    return new Iterator<INodeDirectory.SnapshotAndINode>(){
                        final Iterator<DirectoryWithSnapshotFeature.DirectoryDiff> i;
                        private DirectoryWithSnapshotFeature.DirectoryDiff next;
                        {
                            this.i = INodeDirectorySnapshottable.this.getDiffs().iterator();
                            this.next = this.findNext();
                        }

                        private DirectoryWithSnapshotFeature.DirectoryDiff findNext() {
                            while (this.i.hasNext()) {
                                DirectoryWithSnapshotFeature.DirectoryDiff diff = this.i.next();
                                if (!diff.isSnapshotRoot()) continue;
                                return diff;
                            }
                            return null;
                        }

                        @Override
                        public boolean hasNext() {
                            return this.next != null;
                        }

                        @Override
                        public INodeDirectory.SnapshotAndINode next() {
                            INodeDirectory.SnapshotAndINode pair = new INodeDirectory.SnapshotAndINode(this.next.getSnapshotId(), INodeDirectorySnapshottable.this.getSnapshotById(this.next.getSnapshotId()).getRoot());
                            this.next = this.findNext();
                            return pair;
                        }

                        @Override
                        public void remove() {
                            throw new UnsupportedOperationException();
                        }
                    };
                }
            });
        }
    }
}

