/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.NavigableSet;
import java.util.TreeSet;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceExistException;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZKNamespaceManager;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.constraint.ConstraintException;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.handler.CreateTableHandler;
import org.apache.hadoop.hbase.master.procedure.CreateTableProcedure;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
public class TableNamespaceManager {
    private static final Log LOG = LogFactory.getLog(TableNamespaceManager.class);
    private Configuration conf;
    private MasterServices masterServices;
    private Table nsTable;
    private ZKNamespaceManager zkNamespaceManager;
    private boolean initialized;
    public static final String KEY_MAX_REGIONS = "hbase.namespace.quota.maxregions";
    public static final String KEY_MAX_TABLES = "hbase.namespace.quota.maxtables";
    static final String NS_INIT_TIMEOUT = "hbase.master.namespace.init.timeout";
    static final int DEFAULT_NS_INIT_TIMEOUT = 300000;

    public TableNamespaceManager(MasterServices masterServices) {
        this.masterServices = masterServices;
        this.conf = masterServices.getConfiguration();
    }

    public void start() throws IOException {
        if (!MetaTableAccessor.tableExists(this.masterServices.getConnection(), TableName.NAMESPACE_TABLE_NAME)) {
            LOG.info("Namespace table not found. Creating...");
            this.createNamespaceTable(this.masterServices);
        }
        try {
            long startTime = EnvironmentEdgeManager.currentTime();
            int timeout = this.conf.getInt(NS_INIT_TIMEOUT, 300000);
            while (!this.isTableAssigned()) {
                if (EnvironmentEdgeManager.currentTime() - startTime + 100L > (long)timeout) {
                    throw new IOException("Timedout " + timeout + "ms waiting for namespace table to " + "be assigned");
                }
                Thread.sleep(100L);
            }
        }
        catch (InterruptedException e) {
            throw (InterruptedIOException)new InterruptedIOException().initCause(e);
        }
        this.isTableAvailableAndInitialized();
    }

    private synchronized Table getNamespaceTable() throws IOException {
        if (!this.isTableAvailableAndInitialized()) {
            throw new IOException(this.getClass().getName() + " isn't ready to serve");
        }
        return this.nsTable;
    }

    public synchronized NamespaceDescriptor get(String name) throws IOException {
        if (!this.isTableAvailableAndInitialized()) {
            return null;
        }
        return this.zkNamespaceManager.get(name);
    }

    public synchronized void create(NamespaceDescriptor ns) throws IOException {
        this.create(this.getNamespaceTable(), ns);
    }

    public synchronized void update(NamespaceDescriptor ns) throws IOException {
        Table table = this.getNamespaceTable();
        if (this.get(table, ns.getName()) == null) {
            throw new NamespaceNotFoundException(ns.getName());
        }
        this.upsert(table, ns);
    }

    private NamespaceDescriptor get(Table table, String name) throws IOException {
        Result res = table.get(new Get(Bytes.toBytes(name)));
        if (res.isEmpty()) {
            return null;
        }
        byte[] val = CellUtil.cloneValue(res.getColumnLatestCell(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
        return ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val));
    }

    private void create(Table table, NamespaceDescriptor ns) throws IOException {
        if (this.get(table, ns.getName()) != null) {
            throw new NamespaceExistException(ns.getName());
        }
        this.validateTableAndRegionCount(ns);
        FileSystem fs = this.masterServices.getMasterFileSystem().getFileSystem();
        fs.mkdirs(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), ns.getName()));
        this.upsert(table, ns);
        if (this.masterServices.isInitialized()) {
            this.masterServices.getMasterQuotaManager().setNamespaceQuota(ns);
        }
    }

    private void upsert(Table table, NamespaceDescriptor ns) throws IOException {
        this.validateTableAndRegionCount(ns);
        Put p = new Put(Bytes.toBytes(ns.getName()));
        p.addImmutable(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES, ProtobufUtil.toProtoNamespaceDescriptor(ns).toByteArray());
        table.put(p);
        try {
            this.zkNamespaceManager.update(ns);
        }
        catch (IOException ex) {
            String msg = "Failed to update namespace information in ZK. Aborting.";
            LOG.fatal(msg, ex);
            this.masterServices.abort(msg, ex);
        }
    }

    public synchronized void remove(String name) throws IOException {
        int tableCount;
        if (this.get(name) == null) {
            throw new NamespaceNotFoundException(name);
        }
        if (NamespaceDescriptor.RESERVED_NAMESPACES.contains(name)) {
            throw new ConstraintException("Reserved namespace " + name + " cannot be removed.");
        }
        try {
            tableCount = this.masterServices.listTableDescriptorsByNamespace(name).size();
        }
        catch (FileNotFoundException fnfe) {
            throw new NamespaceNotFoundException(name);
        }
        if (tableCount > 0) {
            throw new ConstraintException("Only empty namespaces can be removed. Namespace " + name + " has " + tableCount + " tables");
        }
        Delete d = new Delete(Bytes.toBytes(name));
        this.getNamespaceTable().delete(d);
        this.zkNamespaceManager.remove(name);
        FileSystem fs = this.masterServices.getMasterFileSystem().getFileSystem();
        for (FileStatus status : fs.listStatus(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), name))) {
            if (HConstants.HBASE_NON_TABLE_DIRS.contains(status.getPath().getName())) continue;
            throw new IOException("Namespace directory contains table dir: " + status.getPath());
        }
        if (!fs.delete(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), name), true)) {
            throw new IOException("Failed to remove namespace: " + name);
        }
        this.masterServices.getMasterQuotaManager().removeNamespaceQuota(name);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized NavigableSet<NamespaceDescriptor> list() throws IOException {
        TreeSet<NamespaceDescriptor> ret = Sets.newTreeSet(NamespaceDescriptor.NAMESPACE_DESCRIPTOR_COMPARATOR);
        try (ResultScanner scanner = this.getNamespaceTable().getScanner(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES);){
            for (Result r : scanner) {
                byte[] val = CellUtil.cloneValue(r.getColumnLatestCell(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
                ret.add(ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val)));
            }
        }
        return ret;
    }

    private void createNamespaceTable(MasterServices masterServices) throws IOException {
        HRegionInfo[] newRegions = new HRegionInfo[]{new HRegionInfo(HTableDescriptor.NAMESPACE_TABLEDESC.getTableName(), null, null)};
        if (masterServices.isMasterProcedureExecutorEnabled()) {
            masterServices.getMasterProcedureExecutor().submitProcedure(new CreateTableProcedure(masterServices.getMasterProcedureExecutor().getEnvironment(), HTableDescriptor.NAMESPACE_TABLEDESC, newRegions));
        } else {
            masterServices.getExecutorService().submit(new CreateTableHandler(masterServices, masterServices.getMasterFileSystem(), HTableDescriptor.NAMESPACE_TABLEDESC, masterServices.getConfiguration(), newRegions, masterServices).prepare());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized boolean isTableAvailableAndInitialized() throws IOException {
        if (this.initialized) {
            this.nsTable = this.masterServices.getConnection().getTable(TableName.NAMESPACE_TABLE_NAME);
            return true;
        }
        if (this.isTableAssigned()) {
            try {
                this.nsTable = this.masterServices.getConnection().getTable(TableName.NAMESPACE_TABLE_NAME);
                this.zkNamespaceManager = new ZKNamespaceManager(this.masterServices.getZooKeeper());
                this.zkNamespaceManager.start();
                if (this.get(this.nsTable, NamespaceDescriptor.DEFAULT_NAMESPACE.getName()) == null) {
                    this.create(this.nsTable, NamespaceDescriptor.DEFAULT_NAMESPACE);
                }
                if (this.get(this.nsTable, NamespaceDescriptor.SYSTEM_NAMESPACE.getName()) == null) {
                    this.create(this.nsTable, NamespaceDescriptor.SYSTEM_NAMESPACE);
                }
                try (ResultScanner scanner = this.nsTable.getScanner(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES);){
                    for (Result result : scanner) {
                        byte[] val = CellUtil.cloneValue(result.getColumnLatest(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
                        NamespaceDescriptor ns = ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val));
                        this.zkNamespaceManager.update(ns);
                    }
                }
                this.initialized = true;
                return true;
            }
            catch (IOException ie) {
                LOG.warn("Caught exception in initializing namespace table manager", ie);
                if (this.nsTable != null) {
                    this.nsTable.close();
                }
                throw ie;
            }
        }
        return false;
    }

    private boolean isTableAssigned() {
        return !this.masterServices.getAssignmentManager().getRegionStates().getRegionsOfTable(TableName.NAMESPACE_TABLE_NAME).isEmpty();
    }

    void validateTableAndRegionCount(NamespaceDescriptor desc) throws IOException {
        if (TableNamespaceManager.getMaxRegions(desc) <= 0L) {
            throw new ConstraintException("The max region quota for " + desc.getName() + " is less than or equal to zero.");
        }
        if (TableNamespaceManager.getMaxTables(desc) <= 0L) {
            throw new ConstraintException("The max tables quota for " + desc.getName() + " is less than or equal to zero.");
        }
    }

    public static long getMaxTables(NamespaceDescriptor ns) throws IOException {
        String value = ns.getConfigurationValue(KEY_MAX_TABLES);
        long maxTables = 0L;
        if (StringUtils.isNotEmpty(value)) {
            try {
                maxTables = Long.parseLong(value);
            }
            catch (NumberFormatException exp) {
                throw new DoNotRetryIOException("NumberFormatException while getting max tables.", exp);
            }
        } else {
            maxTables = Long.MAX_VALUE;
        }
        return maxTables;
    }

    public static long getMaxRegions(NamespaceDescriptor ns) throws IOException {
        String value = ns.getConfigurationValue(KEY_MAX_REGIONS);
        long maxRegions = 0L;
        if (StringUtils.isNotEmpty(value)) {
            try {
                maxRegions = Long.parseLong(value);
            }
            catch (NumberFormatException exp) {
                throw new DoNotRetryIOException("NumberFormatException while getting max regions.", exp);
            }
        } else {
            maxRegions = Long.MAX_VALUE;
        }
        return maxRegions;
    }
}

