/*
 * Decompiled with CFR 0.152.
 */
package io.specto.hoverfly.junit.core;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import io.specto.hoverfly.junit.api.HoverflyClient;
import io.specto.hoverfly.junit.api.OkHttpHoverflyClient;
import io.specto.hoverfly.junit.core.HoverflyConfig;
import io.specto.hoverfly.junit.core.HoverflyConfiguration;
import io.specto.hoverfly.junit.core.HoverflyMode;
import io.specto.hoverfly.junit.core.HoverflyUtils;
import io.specto.hoverfly.junit.core.ProxyConfigurer;
import io.specto.hoverfly.junit.core.SimulationSource;
import io.specto.hoverfly.junit.core.SslConfigurer;
import io.specto.hoverfly.junit.core.SystemConfig;
import io.specto.hoverfly.junit.core.SystemConfigFactory;
import io.specto.hoverfly.junit.core.TempFileManager;
import io.specto.hoverfly.junit.core.model.HoverflyInfo;
import io.specto.hoverfly.junit.core.model.Simulation;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.ProcessExecutor;
import org.zeroturnaround.exec.StartedProcess;
import org.zeroturnaround.exec.stream.slf4j.Slf4jStream;

public class Hoverfly
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Hoverfly.class);
    private static final int BOOT_TIMEOUT_SECONDS = 10;
    private static final int RETRY_BACKOFF_INTERVAL_MS = 100;
    private final HoverflyConfiguration hoverflyConfig;
    private final HoverflyMode hoverflyMode;
    private final ProxyConfigurer proxyConfigurer;
    private final SslConfigurer sslConfigurer = new SslConfigurer();
    private final HoverflyClient hoverflyClient;
    private final TempFileManager tempFileManager = new TempFileManager();
    private StartedProcess startedProcess;
    private boolean useDefaultSslCert = true;

    public Hoverfly(HoverflyConfig hoverflyConfigBuilder, HoverflyMode hoverflyMode) {
        this.hoverflyConfig = hoverflyConfigBuilder.build();
        this.proxyConfigurer = new ProxyConfigurer(this.hoverflyConfig);
        this.hoverflyClient = new OkHttpHoverflyClient(this.hoverflyConfig);
        this.hoverflyMode = hoverflyMode;
    }

    public Hoverfly(HoverflyMode hoverflyMode) {
        this(HoverflyConfig.configs(), hoverflyMode);
    }

    public void start() {
        Runtime.getRuntime().addShutdownHook(new Thread(this::close));
        if (this.startedProcess != null) {
            LOGGER.warn("Local Hoverfly is already running.");
            return;
        }
        if (!this.hoverflyConfig.isRemoteInstance()) {
            this.startHoverflyProcess();
        }
        this.waitForHoverflyToBecomeHealthy();
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getDestination())) {
            this.setDestination(this.hoverflyConfig.getDestination());
        }
        this.setMode(this.hoverflyMode);
        if (this.hoverflyConfig.getProxyCaCertificate().isPresent()) {
            this.sslConfigurer.setDefaultSslContext(this.hoverflyConfig.getProxyCaCertificate().get());
        } else if (this.useDefaultSslCert) {
            this.sslConfigurer.setDefaultSslContext();
        }
        this.proxyConfigurer.setProxySystemProperties();
    }

    private void startHoverflyProcess() {
        HoverflyUtils.checkPortInUse(this.hoverflyConfig.getProxyPort());
        HoverflyUtils.checkPortInUse(this.hoverflyConfig.getAdminPort());
        SystemConfig systemConfig = new SystemConfigFactory().createSystemConfig();
        Path binaryPath = this.tempFileManager.copyHoverflyBinary(systemConfig);
        LOGGER.info("Executing binary at {}", (Object)binaryPath);
        ArrayList<String> commands = new ArrayList<String>();
        commands.add(binaryPath.toString());
        commands.add("-db");
        commands.add("memory");
        commands.add("-pp");
        commands.add(String.valueOf(this.hoverflyConfig.getProxyPort()));
        commands.add("-ap");
        commands.add(String.valueOf(this.hoverflyConfig.getAdminPort()));
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getSslCertificatePath())) {
            this.tempFileManager.copyClassPathResource(this.hoverflyConfig.getSslCertificatePath(), "ca.crt");
            commands.add("-cert");
            commands.add("ca.crt");
        }
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getSslKeyPath())) {
            this.tempFileManager.copyClassPathResource(this.hoverflyConfig.getSslKeyPath(), "ca.key");
            commands.add("-key");
            commands.add("ca.key");
            this.useDefaultSslCert = false;
        }
        try {
            this.startedProcess = new ProcessExecutor().command(commands).redirectOutput((OutputStream)Slf4jStream.of((Logger)LOGGER).asInfo()).directory(this.tempFileManager.getTempDirectory().toFile()).start();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not start Hoverfly process", e);
        }
    }

    @Override
    public void close() {
        this.cleanUp();
    }

    public void importSimulation(SimulationSource simulationSource) {
        LOGGER.info("Importing simulation data to Hoverfly");
        Simulation simulation = simulationSource.getSimulation();
        this.hoverflyClient.setSimulation(simulation);
    }

    public void reset() {
        this.importSimulation(SimulationSource.empty());
    }

    public void exportSimulation(Path path) {
        if (path == null) {
            return;
        }
        LOGGER.info("Exporting simulation data from Hoverfly");
        try {
            Files.deleteIfExists(path);
            Simulation simulation = this.hoverflyClient.getSimulation();
            this.persistSimulation(path, simulation);
        }
        catch (Exception e) {
            LOGGER.error("Failed to export simulation data", (Throwable)e);
        }
    }

    public Simulation getSimulation() {
        return this.hoverflyClient.getSimulation();
    }

    public HoverflyInfo getHoverflyInfo() {
        return this.hoverflyClient.getConfigInfo();
    }

    public void setDestination(String destination) {
        this.hoverflyClient.setDestination(destination);
    }

    public void setMode(HoverflyMode mode) {
        this.hoverflyClient.setMode(mode);
    }

    public HoverflyConfiguration getHoverflyConfig() {
        return this.hoverflyConfig;
    }

    public HoverflyMode getMode() {
        return HoverflyMode.valueOf(this.hoverflyClient.getConfigInfo().getMode().toUpperCase());
    }

    private void persistSimulation(Path path, Simulation simulation) throws IOException {
        ObjectWriter objectWriter = new ObjectMapper().writerWithDefaultPrettyPrinter();
        Files.createDirectories(path.getParent(), new FileAttribute[0]);
        objectWriter.writeValue(path.toFile(), (Object)simulation);
    }

    private void waitForHoverflyToBecomeHealthy() {
        Instant now = Instant.now();
        while (Duration.between(now, Instant.now()).getSeconds() < 10L) {
            if (this.hoverflyClient.getHealth()) {
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        throw new IllegalStateException("Hoverfly has not become healthy in 10 seconds");
    }

    private void cleanUp() {
        LOGGER.info("Destroying hoverfly process");
        if (this.startedProcess != null) {
            Process process = this.startedProcess.getProcess();
            process.destroy();
            ExecutorService executorService = Executors.newSingleThreadExecutor();
            Future<Integer> future = executorService.submit(process::waitFor);
            try {
                future.get(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException | ExecutionException | TimeoutException e) {
                LOGGER.warn("Timeout when waiting for hoverfly process to terminate.");
            }
            executorService.shutdownNow();
        }
        this.proxyConfigurer.restoreProxySystemProperties();
        this.tempFileManager.purge();
    }
}

