/*
 * Decompiled with CFR 0.152.
 */
package io.specto.hoverfly.junit.core;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import io.specto.hoverfly.junit.core.HoverflyConfig;
import io.specto.hoverfly.junit.core.HoverflyConfigValidator;
import io.specto.hoverfly.junit.core.HoverflyMode;
import io.specto.hoverfly.junit.core.HoverflyUtils;
import io.specto.hoverfly.junit.core.ProxyConfigurer;
import io.specto.hoverfly.junit.core.SimulationSource;
import io.specto.hoverfly.junit.core.SslConfigurer;
import io.specto.hoverfly.junit.core.SystemConfig;
import io.specto.hoverfly.junit.core.SystemConfigFactory;
import io.specto.hoverfly.junit.core.TempFileManager;
import io.specto.hoverfly.junit.core.model.HoverflyInfo;
import io.specto.hoverfly.junit.core.model.Simulation;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import okhttp3.Call;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.ProcessExecutor;
import org.zeroturnaround.exec.StartedProcess;
import org.zeroturnaround.exec.stream.slf4j.Slf4jStream;

public class Hoverfly
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Hoverfly.class);
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final int BOOT_TIMEOUT_SECONDS = 10;
    private static final int RETRY_BACKOFF_INTERVAL_MS = 100;
    private static final String HEALTH_CHECK_PATH = "/api/stats";
    private static final String SIMULATION_PATH = "/api/v2/simulation";
    private static final String INFO_PATH = "/api/v2/hoverfly";
    private static final String DESTINATION_PATH = "/api/v2/hoverfly/destination";
    private static final String MODE_PATH = "/api/v2/hoverfly/mode";
    private final HoverflyConfig hoverflyConfig;
    private final HoverflyMode hoverflyMode;
    private final ProxyConfigurer proxyConfigurer;
    private final SslConfigurer sslConfigurer = new SslConfigurer();
    private final OkHttpClient client = new OkHttpClient();
    private final TempFileManager tempFileManager = new TempFileManager();
    private StartedProcess startedProcess;
    private boolean useDefaultSslCert = true;

    public Hoverfly(HoverflyConfig hoverflyConfig, HoverflyMode hoverflyMode) {
        this.hoverflyConfig = new HoverflyConfigValidator().validate(hoverflyConfig);
        this.proxyConfigurer = new ProxyConfigurer(hoverflyConfig);
        this.hoverflyMode = hoverflyMode;
    }

    public Hoverfly(HoverflyMode hoverflyMode) {
        this(HoverflyConfig.configs(), hoverflyMode);
    }

    public void start() {
        Runtime.getRuntime().addShutdownHook(new Thread(this::close));
        if (this.startedProcess != null) {
            LOGGER.warn("Local Hoverfly is already running.");
            return;
        }
        if (!this.hoverflyConfig.isRemoteInstance()) {
            this.startHoverflyProcess();
        }
        this.waitForHoverflyToBecomeHealthy();
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getDestination())) {
            this.setDestination(this.hoverflyConfig.getDestination());
        }
        if (this.hoverflyMode == HoverflyMode.CAPTURE) {
            this.setMode(this.hoverflyMode);
        }
        if (this.useDefaultSslCert) {
            this.sslConfigurer.setTrustStore();
        }
        this.proxyConfigurer.setProxySystemProperties();
    }

    private void startHoverflyProcess() {
        HoverflyUtils.checkPortInUse(this.hoverflyConfig.getProxyPort());
        HoverflyUtils.checkPortInUse(this.hoverflyConfig.getAdminPort());
        SystemConfig systemConfig = new SystemConfigFactory().createSystemConfig();
        Path binaryPath = this.tempFileManager.copyHoverflyBinary(systemConfig);
        LOGGER.info("Executing binary at {}", (Object)binaryPath);
        ArrayList<String> commands = new ArrayList<String>();
        commands.add(binaryPath.toString());
        commands.add("-db");
        commands.add("memory");
        commands.add("-pp");
        commands.add(String.valueOf(this.hoverflyConfig.getProxyPort()));
        commands.add("-ap");
        commands.add(String.valueOf(this.hoverflyConfig.getAdminPort()));
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getSslCertificatePath())) {
            this.tempFileManager.copyClassPathResource(this.hoverflyConfig.getSslCertificatePath(), "ca.crt");
            commands.add("-cert");
            commands.add("ca.crt");
        }
        if (StringUtils.isNotBlank((CharSequence)this.hoverflyConfig.getSslKeyPath())) {
            this.tempFileManager.copyClassPathResource(this.hoverflyConfig.getSslKeyPath(), "ca.key");
            commands.add("-key");
            commands.add("ca.key");
            this.useDefaultSslCert = false;
        }
        try {
            this.startedProcess = new ProcessExecutor().command(commands).redirectOutput((OutputStream)Slf4jStream.of((Logger)LOGGER).asInfo()).directory(this.tempFileManager.getTempDirectory().toFile()).start();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not start Hoverfly process", e);
        }
    }

    @Override
    public void close() {
        this.cleanUp();
    }

    public void importSimulation(SimulationSource simulationSource) {
        LOGGER.info("Importing simulation data to Hoverfly");
        Simulation simulation = simulationSource.getSimulation();
        try {
            byte[] jsonContent = OBJECT_MAPPER.writeValueAsBytes((Object)simulation);
            RequestBody body = RequestBody.create((MediaType)JSON, (byte[])jsonContent);
            Request.Builder builder = this.createRequestBuilderWithUrl(SIMULATION_PATH);
            Request request = builder.put(body).build();
            Call call = this.client.newCall(request);
            try (Response response = call.execute();){
                if (!response.isSuccessful()) {
                    throw new IOException("Unexpected code " + response);
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to import simulation data", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    public void reset() {
        this.importSimulation(SimulationSource.empty());
    }

    public void exportSimulation(Path path) {
        if (path == null) {
            return;
        }
        LOGGER.info("Exporting simulation data from Hoverfly");
        try {
            Files.deleteIfExists(path);
            Request.Builder builder = this.createRequestBuilderWithUrl(SIMULATION_PATH);
            Request request = builder.get().build();
            Call call = this.client.newCall(request);
            Simulation simulation = this.readSimulation(call);
            this.persistSimulation(path, simulation);
        }
        catch (Exception e) {
            LOGGER.error("Failed to export simulation data", (Throwable)e);
        }
    }

    public Simulation getSimulation() {
        Request.Builder builder = this.createRequestBuilderWithUrl(SIMULATION_PATH);
        Request request = builder.get().build();
        try {
            Call call = this.client.newCall(request);
            return this.readSimulation(call);
        }
        catch (IOException e) {
            LOGGER.error("Failed to get simulation data", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public HoverflyInfo getHoverflyInfo() {
        Request.Builder builder = this.createRequestBuilderWithUrl(INFO_PATH);
        Request request = builder.get().build();
        try (Response response = this.client.newCall(request).execute();){
            if (!response.isSuccessful()) {
                throw new IOException("Unexpected code " + response);
            }
            HoverflyInfo hoverflyInfo = (HoverflyInfo)OBJECT_MAPPER.readValue(response.body().charStream(), HoverflyInfo.class);
            return hoverflyInfo;
        }
        catch (IOException e) {
            LOGGER.error("Failed to get Hoverfly info", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    public void setDestination(String destination) {
        try {
            HoverflyInfo hoverflyInfo = new HoverflyInfo(destination, null, null, null);
            byte[] jsonContent = OBJECT_MAPPER.writeValueAsBytes((Object)hoverflyInfo);
            RequestBody body = RequestBody.create((MediaType)JSON, (byte[])jsonContent);
            Request.Builder builder = this.createRequestBuilderWithUrl(DESTINATION_PATH);
            Request request = builder.put(body).build();
            try (Response response = this.client.newCall(request).execute();){
                if (!response.isSuccessful()) {
                    throw new IOException("Unexpected code " + response);
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to get Hoverfly info", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    public void setMode(HoverflyMode mode) {
        try {
            HoverflyInfo hoverflyInfo = new HoverflyInfo(null, mode.name().toLowerCase(), null, null);
            byte[] jsonContent = OBJECT_MAPPER.writeValueAsBytes((Object)hoverflyInfo);
            RequestBody body = RequestBody.create((MediaType)JSON, (byte[])jsonContent);
            Request.Builder builder = this.createRequestBuilderWithUrl(MODE_PATH);
            Request request = builder.put(body).build();
            try (Response response = this.client.newCall(request).execute();){
                if (!response.isSuccessful()) {
                    throw new IOException("Unexpected code " + response);
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to get Hoverfly info", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    public HoverflyConfig getHoverflyConfig() {
        return this.hoverflyConfig;
    }

    public HoverflyMode getMode() {
        return this.hoverflyMode;
    }

    private void persistSimulation(Path path, Simulation simulation) throws IOException {
        ObjectWriter objectWriter = new ObjectMapper().writerWithDefaultPrettyPrinter();
        Files.createDirectories(path.getParent(), new FileAttribute[0]);
        objectWriter.writeValue(path.toFile(), (Object)simulation);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isHealthy() {
        Request.Builder builder = this.createRequestBuilderWithUrl(HEALTH_CHECK_PATH);
        Request request = builder.get().build();
        try {
            Call call = this.client.newCall(request);
            try (Response response = call.execute();){
                LOGGER.debug("Hoverfly health check status code is: {}", (Object)response.code());
                boolean bl = response.isSuccessful();
                return bl;
            }
        }
        catch (IOException e) {
            LOGGER.debug("Not yet healthy", (Throwable)e);
            return false;
        }
    }

    private void waitForHoverflyToBecomeHealthy() {
        Instant now = Instant.now();
        while (Duration.between(now, Instant.now()).getSeconds() < 10L) {
            if (this.isHealthy()) {
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        throw new IllegalStateException("Hoverfly has not become healthy in 10 seconds");
    }

    private void cleanUp() {
        LOGGER.info("Destroying hoverfly process");
        if (this.startedProcess != null) {
            Process process = this.startedProcess.getProcess();
            process.destroy();
            ExecutorService executorService = Executors.newSingleThreadExecutor();
            Future<Integer> future = executorService.submit(process::waitFor);
            try {
                future.get(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException | ExecutionException | TimeoutException e) {
                LOGGER.warn("Timeout when waiting for hoverfly process to terminate.");
            }
            executorService.shutdownNow();
        }
        this.proxyConfigurer.restoreProxySystemProperties();
        this.tempFileManager.purge();
    }

    private Request.Builder createRequestBuilderWithUrl(String path) {
        Request.Builder builder;
        try {
            builder = new Request.Builder().url(new URL("http", this.hoverflyConfig.getHost(), this.hoverflyConfig.getAdminPort(), path));
        }
        catch (MalformedURLException e) {
            throw new IllegalArgumentException(e);
        }
        return builder;
    }

    private Simulation readSimulation(Call call) throws IOException {
        try (Response response = call.execute();){
            if (!response.isSuccessful()) {
                throw new IOException("Unexpected code " + response);
            }
            Simulation simulation = (Simulation)OBJECT_MAPPER.readValue(response.body().charStream(), Simulation.class);
            return simulation;
        }
    }
}

