/*
 * Decompiled with CFR 0.152.
 */
package io.netty.resolver.dns;

import io.netty.util.internal.PlatformDependent;
import io.netty.util.internal.ThreadLocalRandom;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;

public final class DnsServerAddresses {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(DnsServerAddresses.class);
    private static final List<InetSocketAddress> DEFAULT_NAME_SERVER_LIST;
    private static final InetSocketAddress[] DEFAULT_NAME_SERVER_ARRAY;

    public static List<InetSocketAddress> defaultAddresses() {
        return DEFAULT_NAME_SERVER_LIST;
    }

    public static Iterable<InetSocketAddress> sequential(Iterable<? extends InetSocketAddress> addresses) {
        return DnsServerAddresses.sequential0(DnsServerAddresses.sanitize(addresses));
    }

    public static Iterable<InetSocketAddress> sequential(InetSocketAddress ... addresses) {
        return DnsServerAddresses.sequential0(DnsServerAddresses.sanitize(addresses));
    }

    private static Iterable<InetSocketAddress> sequential0(final InetSocketAddress[] addresses) {
        return new Iterable<InetSocketAddress>(){

            @Override
            public Iterator<InetSocketAddress> iterator() {
                return new SequentialAddressIterator(addresses, 0);
            }
        };
    }

    public static Iterable<InetSocketAddress> shuffled(Iterable<? extends InetSocketAddress> addresses) {
        return DnsServerAddresses.shuffled0(DnsServerAddresses.sanitize(addresses));
    }

    public static Iterable<InetSocketAddress> shuffled(InetSocketAddress ... addresses) {
        return DnsServerAddresses.shuffled0(DnsServerAddresses.sanitize(addresses));
    }

    private static Iterable<InetSocketAddress> shuffled0(final InetSocketAddress[] addresses) {
        if (addresses.length == 1) {
            return DnsServerAddresses.singleton(addresses[0]);
        }
        return new Iterable<InetSocketAddress>(){

            @Override
            public Iterator<InetSocketAddress> iterator() {
                return new ShuffledAddressIterator(addresses);
            }
        };
    }

    public static Iterable<InetSocketAddress> rotational(Iterable<? extends InetSocketAddress> addresses) {
        return DnsServerAddresses.rotational0(DnsServerAddresses.sanitize(addresses));
    }

    public static Iterable<InetSocketAddress> rotational(InetSocketAddress ... addresses) {
        return DnsServerAddresses.rotational0(DnsServerAddresses.sanitize(addresses));
    }

    private static Iterable<InetSocketAddress> rotational0(InetSocketAddress[] addresses) {
        return new RotationalAddresses(addresses);
    }

    public static Iterable<InetSocketAddress> singleton(final InetSocketAddress address) {
        if (address == null) {
            throw new NullPointerException("address");
        }
        if (address.isUnresolved()) {
            throw new IllegalArgumentException("cannot use an unresolved DNS server address: " + address);
        }
        return new Iterable<InetSocketAddress>(){
            private final Iterator<InetSocketAddress> iterator = new Iterator<InetSocketAddress>(){

                @Override
                public boolean hasNext() {
                    return true;
                }

                @Override
                public InetSocketAddress next() {
                    return address;
                }

                @Override
                public void remove() {
                    throw new UnsupportedOperationException();
                }
            };

            @Override
            public Iterator<InetSocketAddress> iterator() {
                return this.iterator;
            }
        };
    }

    private static InetSocketAddress[] sanitize(Iterable<? extends InetSocketAddress> addresses) {
        if (addresses == null) {
            throw new NullPointerException("addresses");
        }
        ArrayList<InetSocketAddress> list = addresses instanceof Collection ? new ArrayList(((Collection)addresses).size()) : new ArrayList<InetSocketAddress>(4);
        for (InetSocketAddress inetSocketAddress : addresses) {
            if (inetSocketAddress == null) break;
            if (inetSocketAddress.isUnresolved()) {
                throw new IllegalArgumentException("cannot use an unresolved DNS server address: " + inetSocketAddress);
            }
            list.add(inetSocketAddress);
        }
        if (list.isEmpty()) {
            return DEFAULT_NAME_SERVER_ARRAY;
        }
        return list.toArray(new InetSocketAddress[list.size()]);
    }

    private static InetSocketAddress[] sanitize(InetSocketAddress[] addresses) {
        if (addresses == null) {
            throw new NullPointerException("addresses");
        }
        ArrayList<InetSocketAddress> list = new ArrayList<InetSocketAddress>(addresses.length);
        for (InetSocketAddress a : addresses) {
            if (a == null) break;
            if (a.isUnresolved()) {
                throw new IllegalArgumentException("cannot use an unresolved DNS server address: " + a);
            }
            list.add(a);
        }
        if (list.isEmpty()) {
            return DEFAULT_NAME_SERVER_ARRAY;
        }
        return list.toArray(new InetSocketAddress[list.size()]);
    }

    private DnsServerAddresses() {
    }

    static {
        int DNS_PORT = 53;
        ArrayList<InetSocketAddress> defaultNameServers = new ArrayList<InetSocketAddress>(2);
        try {
            Class<?> configClass = Class.forName("sun.net.dns.ResolverConfiguration");
            Method open = configClass.getMethod("open", new Class[0]);
            Method nameservers = configClass.getMethod("nameservers", new Class[0]);
            Object instance = open.invoke(null, new Object[0]);
            List list = (List)nameservers.invoke(instance, new Object[0]);
            int size = list.size();
            for (int i = 0; i < size; ++i) {
                String dnsAddr = (String)list.get(i);
                if (dnsAddr == null) continue;
                defaultNameServers.add(new InetSocketAddress(InetAddress.getByName(dnsAddr), 53));
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (!defaultNameServers.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Default DNS servers: {} (sun.net.dns.ResolverConfiguration)", defaultNameServers);
            }
        } else {
            Collections.addAll(defaultNameServers, new InetSocketAddress("8.8.8.8", 53), new InetSocketAddress("8.8.4.4", 53));
            if (logger.isWarnEnabled()) {
                logger.warn("Default DNS servers: {} (Google Public DNS as a fallback)", defaultNameServers);
            }
        }
        DEFAULT_NAME_SERVER_LIST = Collections.unmodifiableList(defaultNameServers);
        DEFAULT_NAME_SERVER_ARRAY = defaultNameServers.toArray(new InetSocketAddress[defaultNameServers.size()]);
    }

    private static final class RotationalAddresses
    implements Iterable<InetSocketAddress> {
        private static final AtomicIntegerFieldUpdater<RotationalAddresses> startIdxUpdater;
        private final InetSocketAddress[] addresses;
        private volatile int startIdx;

        RotationalAddresses(InetSocketAddress[] addresses) {
            this.addresses = addresses;
        }

        @Override
        public Iterator<InetSocketAddress> iterator() {
            int nextStartIdx;
            int curStartIdx;
            do {
                if ((nextStartIdx = (curStartIdx = this.startIdx) + 1) < this.addresses.length) continue;
                nextStartIdx = 0;
            } while (!startIdxUpdater.compareAndSet(this, curStartIdx, nextStartIdx));
            return new SequentialAddressIterator(this.addresses, curStartIdx);
        }

        static {
            AtomicIntegerFieldUpdater<RotationalAddresses> updater = PlatformDependent.newAtomicIntegerFieldUpdater(RotationalAddresses.class, (String)"startIdx");
            if (updater == null) {
                updater = AtomicIntegerFieldUpdater.newUpdater(RotationalAddresses.class, "startIdx");
            }
            startIdxUpdater = updater;
        }
    }

    private static final class ShuffledAddressIterator
    implements Iterator<InetSocketAddress> {
        private final InetSocketAddress[] addresses;
        private int i;

        ShuffledAddressIterator(InetSocketAddress[] addresses) {
            this.addresses = (InetSocketAddress[])addresses.clone();
            this.shuffle();
        }

        private void shuffle() {
            InetSocketAddress[] addresses = this.addresses;
            ThreadLocalRandom r = ThreadLocalRandom.current();
            for (int i = addresses.length - 1; i >= 0; --i) {
                InetSocketAddress tmp = addresses[i];
                int j = r.nextInt(i + 1);
                addresses[i] = addresses[j];
                addresses[j] = tmp;
            }
        }

        @Override
        public boolean hasNext() {
            return true;
        }

        @Override
        public InetSocketAddress next() {
            int i = this.i;
            InetSocketAddress next = this.addresses[i];
            if (++i < this.addresses.length) {
                this.i = i;
            } else {
                this.i = 0;
                this.shuffle();
            }
            return next;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private static final class SequentialAddressIterator
    implements Iterator<InetSocketAddress> {
        private final InetSocketAddress[] addresses;
        private int i;

        SequentialAddressIterator(InetSocketAddress[] addresses, int startIdx) {
            this.addresses = addresses;
            this.i = startIdx;
        }

        @Override
        public boolean hasNext() {
            return true;
        }

        @Override
        public InetSocketAddress next() {
            int i = this.i;
            InetSocketAddress next = this.addresses[i];
            this.i = ++i < this.addresses.length ? i : 0;
            return next;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

