/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.logging;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.buffer.DefaultByteBufHolder;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelMetadata;
import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.handler.logging.ByteBufFormat;
import io.netty.handler.logging.LogLevel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.util.CharsetUtil;
import io.netty.util.internal.StringUtil;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.ArgumentMatcher;
import org.mockito.Mockito;
import org.slf4j.LoggerFactory;

public class LoggingHandlerTest {
    private static final String LOGGER_NAME = LoggingHandler.class.getName();
    private static final Logger rootLogger = (Logger)LoggerFactory.getLogger((String)"ROOT");
    private static final Logger logger = (Logger)LoggerFactory.getLogger((String)LOGGER_NAME);
    private static final List<Appender<ILoggingEvent>> oldAppenders = new ArrayList<Appender<ILoggingEvent>>();
    private Appender<ILoggingEvent> appender;

    @BeforeClass
    public static void beforeClass() {
        Iterator i = rootLogger.iteratorForAppenders();
        while (i.hasNext()) {
            Appender a = (Appender)i.next();
            oldAppenders.add((Appender<ILoggingEvent>)a);
            rootLogger.detachAppender(a);
        }
        Unpooled.buffer();
    }

    @AfterClass
    public static void afterClass() {
        for (Appender<ILoggingEvent> a : oldAppenders) {
            rootLogger.addAppender(a);
        }
    }

    @Before
    public void setup() {
        this.appender = (Appender)Mockito.mock(Appender.class);
        logger.addAppender(this.appender);
    }

    @After
    public void teardown() {
        logger.detachAppender(this.appender);
    }

    @Test(expected=NullPointerException.class)
    public void shouldNotAcceptNullLogLevel() {
        LogLevel level = null;
        new LoggingHandler(level);
    }

    @Test
    public void shouldApplyCustomLogLevel() {
        LoggingHandler handler = new LoggingHandler(LogLevel.INFO);
        Assert.assertEquals((Object)LogLevel.INFO, (Object)handler.level());
    }

    @Test
    public void shouldLogChannelActive() {
        new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+ACTIVE$")));
    }

    @Test
    public void shouldLogChannelWritabilityChanged() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.config().setWriteBufferLowWaterMark(5);
        channel.config().setWriteBufferHighWaterMark(10);
        channel.write((Object)"hello", channel.newPromise());
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+WRITABILITY CHANGED$")));
    }

    @Test
    public void shouldLogChannelRegistered() {
        new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+REGISTERED$")));
    }

    @Test
    public void shouldLogChannelClose() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.close().await();
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+CLOSE$")));
    }

    @Test
    public void shouldLogChannelConnect() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.connect((SocketAddress)new InetSocketAddress(80)).await();
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+CONNECT: 0.0.0.0/0.0.0.0:80$")));
    }

    @Test
    public void shouldLogChannelConnectWithLocalAddress() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.connect((SocketAddress)new InetSocketAddress(80), (SocketAddress)new InetSocketAddress(81)).await();
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher("^\\[id: 0xembedded, L:embedded - R:embedded\\] CONNECT: 0.0.0.0/0.0.0.0:80, 0.0.0.0/0.0.0.0:81$")));
    }

    @Test
    public void shouldLogChannelDisconnect() throws Exception {
        DisconnectingEmbeddedChannel channel = new DisconnectingEmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.connect(new InetSocketAddress(80)).await();
        channel.disconnect().await();
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+DISCONNECT$")));
    }

    @Test
    public void shouldLogChannelInactive() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.pipeline().fireChannelInactive();
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+INACTIVE$")));
    }

    @Test
    public void shouldLogChannelBind() throws Exception {
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.bind((SocketAddress)new InetSocketAddress(80));
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+BIND: 0.0.0.0/0.0.0.0:80$")));
    }

    @Test
    public void shouldLogChannelUserEvent() throws Exception {
        String userTriggered = "iAmCustom!";
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.pipeline().fireUserEventTriggered((Object)new String(userTriggered));
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+USER_EVENT: " + userTriggered + '$')));
    }

    @Test
    public void shouldLogChannelException() throws Exception {
        String msg = "illegalState";
        IllegalStateException cause = new IllegalStateException(msg);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.pipeline().fireExceptionCaught((Throwable)cause);
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+EXCEPTION: " + cause.getClass().getCanonicalName() + ": " + msg + '$')));
    }

    @Test
    public void shouldLogDataWritten() throws Exception {
        String msg = "hello";
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeOutbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+WRITE: " + msg + '$')));
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+FLUSH$")));
    }

    @Test
    public void shouldLogNonByteBufDataRead() throws Exception {
        String msg = "hello";
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ: " + msg + '$')));
        String handledMsg = (String)channel.readInbound();
        MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.sameInstance((Object)handledMsg)));
        MatcherAssert.assertThat((Object)channel.readInbound(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
    }

    @Test
    public void shouldLogByteBufDataRead() throws Exception {
        ByteBuf msg = Unpooled.copiedBuffer((CharSequence)"hello", (Charset)CharsetUtil.UTF_8);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ: " + msg.readableBytes() + "B$", true)));
        ByteBuf handledMsg = (ByteBuf)channel.readInbound();
        MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.sameInstance((Object)handledMsg)));
        handledMsg.release();
        MatcherAssert.assertThat((Object)channel.readInbound(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
    }

    @Test
    public void shouldLogByteBufDataReadWithSimpleFormat() throws Exception {
        ByteBuf msg = Unpooled.copiedBuffer((CharSequence)"hello", (Charset)CharsetUtil.UTF_8);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler(LogLevel.DEBUG, ByteBufFormat.SIMPLE)});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ: " + msg.readableBytes() + "B$", false)));
        ByteBuf handledMsg = (ByteBuf)channel.readInbound();
        MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.sameInstance((Object)handledMsg)));
        handledMsg.release();
        MatcherAssert.assertThat((Object)channel.readInbound(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
    }

    @Test
    public void shouldLogEmptyByteBufDataRead() throws Exception {
        ByteBuf msg = Unpooled.EMPTY_BUFFER;
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ: 0B$", false)));
        ByteBuf handledMsg = (ByteBuf)channel.readInbound();
        MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.sameInstance((Object)handledMsg)));
        MatcherAssert.assertThat((Object)channel.readInbound(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
    }

    @Test
    public void shouldLogByteBufHolderDataRead() throws Exception {
        DefaultByteBufHolder msg = new DefaultByteBufHolder(Unpooled.copiedBuffer((CharSequence)"hello", (Charset)CharsetUtil.UTF_8)){

            public String toString() {
                return "foobar";
            }
        };
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ: foobar, 5B$", true)));
        ByteBufHolder handledMsg = (ByteBufHolder)channel.readInbound();
        MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.sameInstance((Object)handledMsg)));
        handledMsg.release();
        MatcherAssert.assertThat((Object)channel.readInbound(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
    }

    @Test
    public void shouldLogChannelReadComplete() throws Exception {
        ByteBuf msg = Unpooled.EMPTY_BUFFER;
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{new LoggingHandler()});
        channel.writeInbound(new Object[]{msg});
        ((Appender)Mockito.verify(this.appender)).doAppend(Mockito.argThat((ArgumentMatcher)new RegexLogMatcher(".+READ COMPLETE$")));
    }

    private static final class DisconnectingEmbeddedChannel
    extends EmbeddedChannel {
        private DisconnectingEmbeddedChannel(ChannelHandler ... handlers) {
            super(handlers);
        }

        public ChannelMetadata metadata() {
            return new ChannelMetadata(true);
        }
    }

    private static final class RegexLogMatcher
    implements ArgumentMatcher<ILoggingEvent> {
        private final String expected;
        private final boolean shouldContainNewline;
        private String actualMsg;

        RegexLogMatcher(String expected) {
            this(expected, false);
        }

        RegexLogMatcher(String expected, boolean shouldContainNewline) {
            this.expected = expected;
            this.shouldContainNewline = shouldContainNewline;
        }

        public boolean matches(ILoggingEvent actual) {
            this.actualMsg = actual.getMessage().split("(?s)[\\r\\n]+")[0];
            if (this.actualMsg.matches(this.expected)) {
                return actual.getMessage().contains(StringUtil.NEWLINE) == this.shouldContainNewline;
            }
            return false;
        }
    }
}

