/*
 * Copyright 2012 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package io.netty.logging;

import org.osgi.service.log.LogService;

/**
 * <a href="http://www.osgi.org/">OSGi</a> {@link LogService} logger.
 */
class OsgiLogger extends AbstractInternalLogger {

    private final OsgiLoggerFactory parent;
    private final InternalLogger fallback;
    private final String name;
    private final String prefix;

    OsgiLogger(OsgiLoggerFactory parent, String name, InternalLogger fallback) {
        this.parent = parent;
        this.name = name;
        this.fallback = fallback;
        prefix = '[' + name + "] ";
    }

    @Override
    public void trace(String msg) {
        // This logger doesn't have TRACE level
    }

    @Override
    public void trace(String msg, Throwable cause) {
        // This logger doesn't have TRACE level
    }

    @Override
    public void debug(String msg) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_DEBUG, prefix + msg);
        } else {
            fallback.debug(msg);
        }
    }

    @Override
    public void debug(String msg, Throwable cause) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_DEBUG, prefix + msg, cause);
        } else {
            fallback.debug(msg, cause);
        }
    }

    @Override
    public void error(String msg) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_ERROR, prefix + msg);
        } else {
            fallback.error(msg);
        }
    }

    @Override
    public void error(String msg, Throwable cause) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_ERROR, prefix + msg, cause);
        } else {
            fallback.error(msg, cause);
        }
    }

    @Override
    public void info(String msg) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_INFO, prefix + msg);
        } else {
            fallback.info(msg);
        }
    }

    @Override
    public void info(String msg, Throwable cause) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_INFO, prefix + msg, cause);
        } else {
            fallback.info(msg, cause);
        }
    }

    @Override
    public boolean isTraceEnabled() {
        return false;
    }

    @Override
    public boolean isDebugEnabled() {
        return true;
    }

    @Override
    public boolean isErrorEnabled() {
        return true;
    }

    @Override
    public boolean isInfoEnabled() {
        return true;
    }

    @Override
    public boolean isWarnEnabled() {
        return true;
    }

    @Override
    public void warn(String msg) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_WARNING, prefix + msg);
        } else {
            fallback.warn(msg);
        }
    }

    @Override
    public void warn(String msg, Throwable cause) {
        LogService logService = parent.getLogService();
        if (logService != null) {
            logService.log(LogService.LOG_WARNING, prefix + msg, cause);
        } else {
            fallback.warn(msg, cause);
        }
    }

    @Override
    public String toString() {
        return name;
    }
}
