#!/system/bin/sh

# args: driver
setup_vulkan() {
    echo "setup vulkan for driver: $1"
    case "$1" in
        i915)
            setprop ro.hardware.vulkan intel
            ;;
        amdgpu)
            setprop ro.hardware.vulkan radeon
            ;;
        virtio_gpu)
            setprop ro.hardware.vulkan virtio
            ;;
        v3d|vc4)
            setprop ro.hardware.vulkan broadcom
            ;;
        msm_drm)
            setprop ro.hardware.vulkan freedreno
            ;;
        panfrost)
            setprop ro.hardware.vulkan panfrost
            ;;
        *)
            echo "not supported driver: $1"
            ;;
    esac
}

setup_render_node() {
    node=`getprop ro.kernel.redroid.gpu.node`
    if [ ! -z "$node" ]; then
        echo "force render node: $node"

        setprop gralloc.gbm.device $node
        chmod 666 $node

        # setup vulkan
        cd /sys/kernel/debug/dri
        driver="`cat ${node: -3}/name | cut -d' ' -f1`"
        setup_vulkan $driver
        return 0
    fi

    cd /sys/kernel/debug/dri
    for d in * ; do
        if [ "$d" -ge "128" ]; then
            driver="`cat $d/name | cut -d' ' -f1`"
            echo "DRI node exists, driver: $driver"
            setup_vulkan $driver
            case $driver in
                i915|amdgpu|virtio_gpu|v3d|vc4|msm_drm|panfrost)
                    node="/dev/dri/renderD$d"
                    echo "use render node: $node"
                    setprop gralloc.gbm.device $node
                    chmod 666 $node
                    return 0
                    ;;
            esac
        fi
    done

    echo "NO qualified render node found"
    return 1
}

find_va_gpu_node() {
    cd /dev/dri
    for d in * ; do
        node="`echo $d | grep renderD`"
        if [ ! -z $node ]; then
            id="`echo $d | grep -Eo '[0-9]+'`"
            if [ ! -z $id ]; then
                vendor="`cat /sys/kernel/debug/dri/$id/name | cut -d' ' -f1`"
                case $vendor in
                    va)
                        echo "va render node exists: /dev/dri/$node" >> /data/gpu_config.txt
                        echo "use render node: /dev/dri/$node" >> /data/gpu_config.txt
                        setprop ro.kernel.va.gpu.node /dev/dri/$node
                        return 0
                        ;;
                esac
            fi
        fi
    done

    echo "NO qualified render node found"
    return 1
}

find_va_video_node() {
    cd /dev
    for d in * ; do
        node="`echo $d | grep va_video`"
        if [ ! -z $node ]; then
            echo "va video node exists: /dev/$node" >> /data/gpu_config.txt
            id="`echo $d | grep -Eo '[0-9]+'`"
            if [ ! -z $id ]; then
                echo "use gpu id: $id" >> /data/gpu_config.txt
                setprop ro.kernel.va.gpu.id $id
                return 0
            fi
        fi
    done

    echo "NO qualified video node found"
    return 1
}

setup_va_gpu_node() {
    echo "setup_va_gpu_node start" >> /data/gpu_config.txt
    gpuid=`getprop ro.kernel.va.gpu.id`
    if [ ! -z "$gpuid" ]; then
        echo "force use va gpu id: $gpuid" >> /data/gpu_config.txt
        return 0
    fi

    ifs_old=$IFS
    IFS=$','
    for d in ${VASTAI_ALLOCATE_DEVICES} ; do
        echo $d >> /data/gpu_config.txt
        dd=$(echo $d|grep ^/dev/vacc)
        if [ ! -z "$d" ] && [ "$d" == "$dd" ]; then
            gpuid=$(echo $d|awk -F'vacc' '{print $2}')
            if [ "$gpuid" -ge "0" ]; then
                setprop ro.kernel.va.gpu.id $gpuid
            fi
        fi
        dd=$(echo $d|grep ^/dev/dri/renderD)
        if [ ! -z "$d" ] && [ "$d" == "$dd" ]; then
            setprop ro.kernel.va.gpu.node $d
        fi
    done

    gpuid=`getprop ro.kernel.va.gpu.id`
    if [ -z "$gpuid" ]; then
        find_va_gpu_node
        find_va_video_node
    fi

    echo "setup_va_gpu_node end" >> /data/gpu_config.txt
}

setup_va_gpu_prop() {
    setup_va_gpu_node
}

gpu_setup_host() {
    echo "use GPU host mode"

    setprop ro.hardware.egl mesa
    setprop ro.hardware.gralloc gbm
    setprop ro.kernel.redroid.fps 30
}

gpu_setup_guest() {
    echo "use GPU guest mode"

    setprop ro.hardware.egl swiftshader
    setprop ro.hardware.gralloc redroid
    setprop ro.hardware.vulkan pastel
}

gpu_setup_sgpu100() {
    echo "use GPU host mode sgpu100 device"

    setprop debug.sf.vsync_reactor false
    # set OpenGL ES version to 3.2
    setprop ro.opengles.version 196610
    
    setprop ro.hardware.vulkan va
    setprop ro.hardware.egl va
    setprop ro.hardware.gralloc sgpu100

    #default value as below, if not set by docker
    setprop ro.hardware.fps 30
    setprop ro.hardware.width 1920
    setprop ro.hardware.height 1080

    width=`getprop ro.hardware.width 1920`
    height=`getprop ro.hardware.height 1080`
    size=$width*$height

    if (( $size > 1920*1080 )); then  #2k
        setprop ro.sf.lcd_density 640
    elif (( $size > 1520*720 )); then #1080p
        setprop ro.sf.lcd_density 440
    else                            #720p
        setprop ro.sf.lcd_density 320
    fi

    setup_va_gpu_prop

    chmod 666 /dev/ion
    chmod 666 /dev/va_sync
    chmod 666 /dev/va$(getprop ro.kernel.va.gpu.id)_ctl
    chmod 666 /dev/va_video$(getprop ro.kernel.va.gpu.id)
    chmod 666 /dev/vacc$(getprop ro.kernel.va.gpu.id)
    chmod 666 /sys/kernel/debug/sync/sw_sync
    chmod 666 /dev/pts/ptmx

}

gpu_setup() {
    ## redroid.gpu.mode=(auto, host, guest)
    ## redroid.gpu.node= (/dev/dri/renderDxxx)
    ## TODO: we will selete gpu node with this property in multi-GPU platform.

    mode=`getprop ro.kernel.redroid.gpu.mode host_sgpu100`
    if [ "$mode" = "host" ]; then
        setup_render_node
        gpu_setup_host
    elif [ "$mode" = "guest" ]; then
        gpu_setup_guest
    elif [ "$mode" = "host_sgpu100" ]; then
        gpu_setup_sgpu100
    elif [ "$mode" = "auto" ]; then
         echo "use GPU auto mode"
         if setup_render_node; then
            gpu_setup_host
         else
            gpu_setup_guest
         fi
    else
        echo "unknown mode: $mode"
    fi
}

# ?? workaround for ueventd
chmod 0666 /dev/dri/*

codec2_setup() {
    ## target=(redroid_arm64, redroid_arm64_only)
    target=`getprop ro.build.product redroid_arm64`

    if [ "$target" = "redroid_arm64_only" ]; then
        setprop debug.stagefright.ccodec 4
        setprop debug.stagefright.c2-poolmask 0xfc0000
    elif [ "$target" = "redroid_arm64" ]; then
        setprop debug.stagefright.ccodec 0
    fi
}

gpu_setup
codec2_setup
