#!/bin/bash

# ******************************************************************************** #
# Copyright Kbox Technologies Co., Ltd. 2023-2023. All rights reserved.
# File Name: kbox11_install_kernel.sh
# Description: ubuntu kbox11内核编译构建安装。
# Usage: 
# 1.请确认脚本位于：Kbox-AOSP11/deploy_scripts/ubuntu_android11_deploy/
# 2.在运行脚本前，请提前上传以下文件到同一目录下：
#   [1] T4XX_V3.1.0.tar.gz
#   若使用本地方式获取源码包，则以下文件也需要上传：(自动下载模式则无需上传)
#   [2] linux-firmware-20210919.tar.gz
#   [3] linux-5.15.98.tar.gz
#   [4] anbox-modules-master.zip
#   [5] docker-19.03.15.tgz
# 3.脚本运行格式：
#   ./kbox11_install_kernel.sh ${packages_dir} 
#   ${packages_dir}需替换成文件上传目录的路径
#   示例： ./kbox11_install_kernel.sh /home/kbox_packages/
# ******************************************************************************** #

#set -x
# 脚本解释器 强制设置为 bash
if [ "$BASH" != "/bin/bash" ] && [ "$BASH" != "/usr/bin/bash" ]
then
   bash "$0" "$@"
   exit $?
fi

function error(){
    echo -e "\033[1;31m$1\033[0m"
    exit 1
}

# 操作系统版本
OS_VERSION="ubuntu_20.04"
# linux kernel版本
KERNEL_VERSION="5.15.98"

docker_src="https://download.docker.com/linux/static/stable/aarch64/docker-19.03.15.tgz"
firmware_src="https://mirrors.edge.kernel.org/pub/linux/kernel/firmware/linux-firmware-20210919.tar.gz"
firmware_verify_src="https://mirrors.edge.kernel.org/pub/linux/kernel/firmware/sha256sums.asc"
kernel_src="https://mirrors.edge.kernel.org/pub/linux/kernel/v5.x/linux-${KERNEL_VERSION}.tar.gz"
kernel_verify_src="https://mirrors.edge.kernel.org/pub/linux/kernel/v5.x/sha256sums.asc"
anbox_src="https://github.com/anbox/anbox-modules/archive/refs/heads/master.zip"

# 内核源码存放目录
kernel_dir="/usr/src/"
# 当前目录
CURRENT_DIR=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)
cd "${CURRENT_DIR}" || exit
# 依赖目录
mkdir -p ~/dependency/
work_dir=~/dependency/
# Kbox_AOSP根目录
kbox_dir=$(cd "${CURRENT_DIR}"/../../ && pwd)
# 本地依赖包目录
packages_dir=$1

cpu_num=$(< /proc/cpuinfo grep -c "processor")

################################################################################
# Function Name: check_local_package
# Description  : 检查本地依赖包
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function check_local_package(){
    [ -e "${packages_dir}/${1}" ] || error "指定路径下无法找到${packages_dir}/${1}"
}

################################################################################
# Function Name: check_local_packages
# Description  : 检查本地依赖包
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function check_local_packages(){
    check_local_package "T4XX_V3.1.0.tar.gz"
    if [ ${run_mode} = "1" ]
    then
        check_local_package "linux-firmware-20210919.tar.gz"
        check_local_package "linux-${KERNEL_VERSION}.tar.gz"
        check_local_package "anbox-modules-master.zip"
        check_local_package "docker-19.03.15.tgz"
    fi
}

################################################################################
# Function Name: choose_run_mode
# Description  : 选择源码包获取方式
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function choose_run_mode(){
    read -p "请选择源码包获取方式([0]自动下载-需联网 [1]本地):" run_mode
    while [[ ${run_mode} != "0" ]] && [[ ${run_mode} != "1" ]]
    do
        read -p "输入无效, 请重新输入:" run_mode
    done
}

################################################################################
# Function Name: check_kernel_version
# Description  : 检查内核是否已更新
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function check_kernel_version(){
    if [ $(uname -r) = ${KERNEL_VERSION} ]
    then
        echo "内核版本已是${KERNEL_VERSION}, 内核无需更新"
        exit 0
    fi
}

################################################################################
# Function Name: modify_config
# Description  : config文件修改。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function modify_config(){
    < ${3} grep -v "#"|grep -w "${1}"
    if [ $? -eq 0 ]
    then
        if [ "${2}" = "is not set" ]
        then
            sed -i "s|${1}=.*|\# ${1} is not set|g" ${3}
        else
            sed -i "s|${1}=.*|${1}=${2}|g" ${3}
        fi
    else
        if [ "${2}" != "is not set" ]
        then
            echo "${1}=${2}" >> ${3}
        fi
    fi
    echo "${3} >> $(< ${3} grep -w "${1}")"
}

################################################################################
# Function Name: download_copy_packages
# Description  : 下载/拷贝源码包
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function download_copy_packages(){
    local verify_sha256sums
    local file_sha256sums
    cd "${work_dir}" || exit
    if [ ${run_mode} = "0" ]
    then
        echo "[[[[[[[[[[[[[[[[[[[[ STEP 1 源码包下载 ]]]]]]]]]]]]]]]]]]]]"

        [ -e "sha256sums.asc" ] && rm -rf sha256sums.asc
        wget ${firmware_verify_src} --no-check-certificate || error "固件校验文件下载失败"
        verify_sha256sums=$(grep linux-firmware-20210919.tar.gz sha256sums.asc)
        file_sha256sums=$(sha256sum linux-firmware-20210919.tar.gz)
        if [ ! -e "linux-firmware-20210919.tar.gz" ] || [ "${verify_sha256sums}" != "${file_sha256sums}" ]
        then
            [ -e "linux-firmware-20210919.tar.gz" ] && rm -rf linux-firmware-20210919.tar.gz
            echo "---------下载固件---------"
            wget "${firmware_src}" --no-check-certificate || error "固件下载失败"
        fi

        [ -e "sha256sums.asc" ] && rm -rf sha256sums.asc
        wget ${kernel_verify_src} --no-check-certificate || error "kernel校验文件下载失败"
        verify_sha256sums=$(grep linux-${KERNEL_VERSION}.tar.gz sha256sums.asc)
        file_sha256sums=$(sha256sum linux-${KERNEL_VERSION}.tar.gz)
        if [ ! -e "linux-${KERNEL_VERSION}.tar.gz" ] || [ "${verify_sha256sums}" != "${file_sha256sums}" ]
        then
            [ -e "linux-${KERNEL_VERSION}.tar.gz" ] && rm -rf linux-${KERNEL_VERSION}.tar.gz
            echo "---------下载kernel源码---------"
            wget "${kernel_src}" --no-check-certificate || error "kernel源码下载失败"
        fi

        echo "---------下载docker---------"
        [ -e "docker-19.03.15.tgz" ] && rm -rf docker-19.03.15.tgz
        wget "${docker_src}" --no-check-certificate || error "docker下载失败"

        echo "---------下载anbox---------"
        [ -e "anbox-modules-master.zip" ] && rm -rf anbox-modules-master.zip
        wget "${anbox_src}" --no-check-certificate || error "anbox下载失败"
        mv master.zip anbox-modules-master.zip || exit
    else
        echo "[[[[[[[[[[[[[[[[[[[[ STEP 1 源码包准备 ]]]]]]]]]]]]]]]]]]]]"
        cp "${packages_dir}/docker-19.03.15.tgz" . || exit
        cp "${packages_dir}/linux-firmware-20210919.tar.gz" . || exit
        cp "${packages_dir}/linux-${KERNEL_VERSION}.tar.gz" . || exit
        cp "${packages_dir}/anbox-modules-master.zip" . || exit
    fi
    cp "${packages_dir}/T4XX_V3.1.0.tar.gz" . || exit
}

################################################################################
# Function Name: install_dependency
# Description  : 安装编译构建所需依赖。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function install_dependency(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 2 安装依赖 ]]]]]]]]]]]]]]]]]]]]"
    apt update || exit
    apt install -y nvme-cli
    apt install -y dpkg dpkg-dev libncurses5-dev libssl-dev libpciaccess0
    apt install -y libdrm-amdgpu1 xserver-xorg-video-amdgpu lxc
    apt install -y build-essential
    apt install -y libncurses5-dev openssl libssl-dev
    apt install -y pkg-config
    apt install -y bison
    apt install -y flex
    apt install -y libelf-dev
}

################################################################################
# Function Name: install_docker
# Description  : 安装预置的docker-*.tgz包。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function install_docker(){
    local docker_version=$(docker -v | grep "Docker version")
    if [[ -n ${docker_version} ]]
    then
        echo "docker已安装"
        return 0
    fi

    cd ${work_dir} || exit
    [ -e "docker" ] && rm -rf docker
    tar xvpf docker-19.03.15.tgz || exit
    cp -p docker/* /usr/bin || exit
    cat >/usr/lib/systemd/system/docker.service <<EOF
[Unit]
Description=Docker Application Container Engine
Documentation=http://docs.docker.com
After=network.target docker.socket
[Service]
Type=notify
EnvironmentFile=-/run/flannel/docker
WorkingDirectory=/usr/local/bin
ExecStart=/usr/bin/dockerd -H tcp://0.0.0.0:4243 -H unix:///var/run/docker.sock --selinux-enabled=false --log-opt max-size=1g
ExecReload=/bin/kill -s HUP $MAINPID
# Having non-zero Limit*s causes performance problems due to accounting overhead
# in the kernel. We recommend using cgroups to do container-local accounting.
LimitNOFILE=infinity
LimitNPROC=infinity
LimitCORE=infinity
# Uncomment TasksMax if your systemd version supports it.
# Only systemd 226 and above support this version.
#TasksMax=infinity
TimeoutStartSec=0
# set delegate yes so that systemd does not reset the cgroups of docker containers
Delegate=yes
# kill only the docker process, not all processes in the cgroup
KillMode=process
Restart=on-failure
[Install]
WantedBy=multi-user.target
EOF
    systemctl daemon-reload
    systemctl restart docker
    systemctl enable docker
}

################################################################################
# Function Name: update_nvme
# Description  : 更新nvme。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function update_nvme(){
    local nvme_version=$(nvme list | grep -o "[0-9A-F]\{3\}X[0-9A-F]\{4\}$")
    if [ -z ${nvme_version} ] || [ ${nvme_version} \< "310X2013" ]
    then
        cd ${work_dir} || exit
        [ -e "T4XX_V3.1.0" ] && rm -rf docker
        tar -zxvf T4XX_V3.1.0.tar.gz
        cd T4XX_V3.1.0/
        echo "---------开始更新NVME固件---------"
        ./t4xx_auto_upgrade.sh
        echo "---------NVME固件更新完毕---------"
    else
        echo "---------NVME固件无需更新---------"
    fi
}

################################################################################
# Function Name: prepare_environment
# Description  : 环境准备。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function prepare_environment(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 3 环境准备 ]]]]]]]]]]]]]]]]]]]]"
    echo "---------安装docker---------"
    install_docker

    echo "---------更新NVME固件---------"
    update_nvme

    echo "---------更新gpu固件---------"
    cd "${work_dir}" || exit
    [ -e "linux-firmware-20210919" ] && rm -rf linux-firmware-20210919
    tar -xvpf linux-firmware-20210919.tar.gz || exit
    cp -ar linux-firmware-20210919/*gpu /usr/lib/firmware/ || exit
    echo "---------gpu固件更新完毕---------"

    echo "---------文件配置---------"
    modify_config "fs.inotify.max_user_instances" "8192" /etc/sysctl.conf
    if [ ! -e "/etc/selinux/config" ]
    then
        touch /etc/selinux/config
        echo "SELINUX=disabled" >> /etc/selinux/config
        echo "/etc/selinux/config >> $(< /etc/selinux/config grep -v "#"|grep -w "SELINUX")"
    else
        modify_config "SELINUX" "disabled" /etc/selinux/config
    fi
    echo "---------以上配置在重启后生效---------"
}

################################################################################
# Function Name: apply_patch
# Description  : 转码及内核补丁合入。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function apply_patch(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 4 合入kernel补丁 ]]]]]]]]]]]]]]]]]]]]"

    cd "${work_dir}" || exit
    [ -e "linux-${KERNEL_VERSION}" ] && rm -rf linux-${KERNEL_VERSION}
    [ -e "linux-kernel-${KERNEL_VERSION}" ] && rm -rf linux-kernel-${KERNEL_VERSION}
    echo "---------解压kernel源码---------"
    tar -zxvf "linux-${KERNEL_VERSION}.tar.gz" || exit
    mv linux-${KERNEL_VERSION} linux-kernel-${KERNEL_VERSION}
    cp -r linux-kernel-${KERNEL_VERSION} ${kernel_dir} || exit
    touch ${kernel_dir}/linux-kernel-${KERNEL_VERSION}/.scmversion

    cd "${kbox_dir}"/patchForExagear/hostOS/"${OS_VERSION}"/ || exit
    cp 0001-exagear-adapt-kernel-5.15.98.patch "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"/ || exit
    cd "${kbox_dir}"/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/kernel || exit
    cp 0001.pid_max_limit.patch "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}" || exit
    cp 0002.5.15.98_mmap.patch  "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}" || exit
    cp 0003.enable_AMD6800_6600_5.15.98.patch "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}" || exit
    cp 0004.binder_kallsyms_and_wake_up_pollfree.patch "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}" || exit

    cd "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"/ || exit
    echo "*--patching 0001-exagear-adapt-kernel-5.15.98.patch"
    patch -p1 < 0001-exagear-adapt-kernel-5.15.98.patch || exit
    echo "*--patching 0001.pid_max_limit.patch"
    patch -p1 < 0001.pid_max_limit.patch || exit
    echo "*--patching 0002.5.15.98_mmap.patch"
    patch -p1 < 0002.5.15.98_mmap.patch || exit
    echo "*--patching 0003.enable_AMD6800_6600_5.15.98.patch"
    patch -p1 < 0003.enable_AMD6800_6600_5.15.98.patch || exit
    echo "*--patching 0004.binder_kallsyms_and_wake_up_pollfree.patch"
    patch -p1 < 0004.binder_kallsyms_and_wake_up_pollfree.patch || exit
}

################################################################################
# Function Name: build_install_kernel
# Description  : 内核及内核模块编译。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function build_install_kernel(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 5  编译&安装kernel ]]]]]]]]]]]]]]]]]]]]"
    echo "---------内核编译---------"
    cd "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"/ || exit
    cp /boot/config-$(uname -r) ./config || exit
    (echo -e \'\\0x65\'; echo -e \'\\0x79\') | make menuconfig
    sleep 1
    [ ! -e ".config" ] && error "config file not found"
    modify_config "CONFIG_BINFMT_MISC" "y" .config
    modify_config "CONFIG_EXAGEAR_BT" "y" .config
    modify_config "CONFIG_CHECKPOINT_RESTORE" "y" .config
    modify_config "CONFIG_PROC_CHILDREN" "y" .config
    modify_config "CONFIG_VFAT_FS" "y" .config
    modify_config "CONFIG_INPUT_UINPUT" "y" .config
    modify_config "CONFIG_HISI_PMU" "y" .config
    modify_config "CONFIG_SYSTEM_TRUSTED_KEYS" "\"\"" .config
    modify_config "CONFIG_SYSTEM_REVOCATION_KEYS" "\"\"" .config
    modify_config "CONFIG_DEBUG_INFO" "is not set" .config

    echo "---------内核编译---------"
    make clean && make -j"${cpu_num}"
    [ $? -ne 0 ] && error "内核编译失败"
    echo "---------安装模块---------"
    make modules_install
    [ $? -ne 0 ] && error "模块安装失败"
    echo "---------安装内核---------"
    make install
    [ $? -ne 0 ] && error "内核安装失败"
    echo "---------更新启动项---------"
    grub_default=$(< /boot/grub/grub.cfg grep "Ubuntu, with Linux"|awk -F \' '{print i++ ":"$2}'|grep -v "recovery mode"|grep -w ""${KERNEL_VERSION}"$"|awk -F ':' '{print $1}')
    sed -i "s/GRUB_DEFAULT=.*/GRUB_DEFAULT=\"1\> ${grub_default}\"/g" /etc/default/grub
    sed -i "s/GRUB_CMDLINE_LINUX=.*/GRUB_CMDLINE_LINUX=\"cgroup_enable=memory swapaccount=1\"/g" /etc/default/grub
    update-grub2
    [ $? -ne 0 ] && error "启动项更新失败"
}

################################################################################
# Function Name: build_ashmem_binder
# Description  : ashmem和binder两个内核模块编译。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function build_ashmem_binder(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 6 编译ashmem binder ]]]]]]]]]]]]]]]]]]]]"
    cd "${work_dir}" || exit
    [ -e "anbox-modules-master" ] && rm -rf anbox-modules-master
    echo "---------解压ashmem源码---------"
    unzip anbox-modules-master.zip || exit
    echo "---------拷贝ashmem源码---------"
    cd ${work_dir}/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/ || exit
    [ -e "ashmem" ] && rm -rf ashmem
    mkdir -p ashmem
    chmod -R 700 ashmem
    cp -r ${work_dir}/anbox-modules-master/ashmem/* ./ashmem/ || exit

    echo "---------拷贝binder源码---------"
    [ -e "binder" ] && rm -rf binder
    mkdir -p binder
    chmod -R 700 binder
    cd binder || exit
    mkdir -p uapi
    chmod -R 700 uapi
    cd "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"/drivers/android || exit
    cp binder_alloc.c binder_alloc.h binder.c binder_internal.h binder_trace.h Makefile\
     ${work_dir}/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/binder/ || exit
    cd "${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"/include/uapi/linux/android/ || exit
    cp binder.h ${work_dir}/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/binder/uapi/ || exit

    cd "${work_dir}"/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/ || exit
    echo "--patching ashmem.patch"
    patch -p1 < ashmem.patch || exit
    echo "--patching binder.patch"
    patch -p1 < binder.patch || exit

    echo "--------编译ashmem--------"
    cd ashmem || exit
    make clean
    make KERNEL_SRC="${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"\
     V=0 M=~/dependency/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/ashmem/
    [ $? -ne 0 ] && error "asdmem编译失败"
    cp ashmem_linux.ko /lib/modules/"${KERNEL_VERSION}"/kernel/lib/ || exit
    cd ..

    echo "---------编译binder--------"
    cd binder || exit
    make clean
    make KERNEL_SRC="${kernel_dir}"/linux-kernel-"${KERNEL_VERSION}"\
     V=0 M=~/dependency/patchForKernel/"${OS_VERSION}"/kernel_"${KERNEL_VERSION}"/ashmem_binder/binder/
    [ $? -ne 0 ] && error "binder编译失败"
    cp aosp_binder_linux.ko /lib/modules/"${KERNEL_VERSION}"/kernel/lib/
}

################################################################################
# Function Name: system_reboot
# Description  : 系统重启。
# Parameter    :
# Returns      : 0 on success, otherwise on fail
################################################################################
function system_reboot(){
    echo "[[[[[[[[[[[[[[[[[[[[ STEP 7 重启系统 ]]]]]]]]]]]]]]]]]]]]"
    local reboot_val
    read -p "内核安装与参数配置已完成, 是否重启以生效 [Y/y]" reboot_val

    if [ ${reboot_val} = "Y" ] || [ ${reboot_val} = "y" ]
    then
        echo "系统即将重启"
        reboot
    else
        echo "请稍后手动重启系统"
    fi
}

main(){
    choose_run_mode
    check_local_packages
    download_copy_packages
    install_dependency
    prepare_environment
    check_kernel_version
    apply_patch
    build_install_kernel
    build_ashmem_binder
    system_reboot
}

main "$@"
exit 0
