var documenterSearchIndex = {"docs":
[{"title":"Tasks","location":"base/parallel.html#Tasks","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Core.Task\nBase.@task\nBase.@async\nBase.asyncmap\nBase.asyncmap!\nBase.current_task\nBase.istaskdone\nBase.istaskstarted\nBase.istaskfailed\nBase.task_local_storage(::Any)\nBase.task_local_storage(::Any, ::Any)\nBase.task_local_storage(::Function, ::Any, ::Any)","page":"Tasks"},{"title":"Core.Task","location":"base/parallel.html#Core.Task","category":"type","text":"Task(func)\n\nCreate a Task (i.e. coroutine) to execute the given function func (which must be callable with no arguments). The task exits when this function returns. The task will run in the \"world age\" from the parent at construction when scheduled.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@task","location":"base/parallel.html#Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only creates a task, and does not run it.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@async","location":"base/parallel.html#Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nwarning: Warning\nIt is strongly encouraged to favor Threads.@spawn over @async always even when no parallelism is required especially in publicly distributed libraries.  This is because a use of @async disables the migration of the parent task across worker threads in the current implementation of Julia.  Thus, seemingly innocent use of @async in a library function can have a large impact on the performance of very different parts of user applications.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap","location":"base/parallel.html#Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple equal length collections). For multiple collection arguments, f is applied elementwise.\n\nntasks specifies the number of tasks to run concurrently. Depending on the length of the collections, if ntasks is unspecified, up to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the number of tasks to run in parallel is checked before processing every element and a new task started if the value of ntasks_func is greater than the current number of tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must then be a function that must accept a Vector of argument tuples and must return a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning the objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Array{UInt64,1}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Array{UInt64,1}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array of argument tuples and return an array of results. map is used in the modified mapping function to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Array{String,1}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\nnote: Note\nCurrently, all tasks in Julia are executed in a single OS thread co-operatively. Consequently, asyncmap is beneficial only when the mapping function involves any I/O - disk, network, remote worker invocation, etc.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap!","location":"base/parallel.html#Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than returning a collection.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.current_task","location":"base/parallel.html#Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskdone","location":"base/parallel.html#Base.istaskdone","category":"function","text":"istaskdone(t::Task) -> Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskstarted","location":"base/parallel.html#Base.istaskstarted","category":"function","text":"istaskstarted(t::Task) -> Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskfailed","location":"base/parallel.html#Base.istaskfailed","category":"function","text":"istaskfailed(t::Task) -> Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to key; the previous value of key, or lack thereof, is restored afterwards. Useful for emulating dynamic scoping.\n\n\n\n\n\n","page":"Tasks"},{"title":"Scheduling","location":"base/parallel.html#Scheduling","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.yield\nBase.yieldto\nBase.sleep\nBase.schedule","page":"Tasks"},{"title":"Base.yield","location":"base/parallel.html#Base.yield","category":"function","text":"yield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this function is still runnable, and will be restarted immediately if there are no other runnable tasks.\n\n\n\n\n\nyield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which immediately yields to t before calling the scheduler.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.yieldto","location":"base/parallel.html#Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is called with no arguments. On subsequent switches, arg is returned from the task's last call to yieldto. This is a low-level call that only switches tasks, not considering states or scheduling in any way. Its use is discouraged.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.sleep","location":"base/parallel.html#Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1 millisecond or input of 0.001.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.schedule","location":"base/parallel.html#Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system is otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of yieldto) when it runs again. If error is true, the value is raised as an exception in the woken task.\n\nwarning: Warning\nIt is incorrect to use schedule on an arbitrary Task that has already been started. See the API reference for more information.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Synchronization","location":"base/parallel.html#lib-task-sync","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.errormonitor\nBase.@sync\nBase.wait\nBase.fetch(t::Task)\nBase.fetch(x::Any)\nBase.timedwait\n\nBase.Condition\nBase.Threads.Condition\nBase.Threads.Event\nBase.notify\nBase.reset(::Base.Threads.Event)\n\nBase.Semaphore\nBase.acquire\nBase.release\n\nBase.AbstractLock\nBase.lock\nBase.unlock\nBase.trylock\nBase.islocked\nBase.ReentrantLock","page":"Tasks"},{"title":"Base.errormonitor","location":"base/parallel.html#Base.errormonitor","category":"function","text":"errormonitor(t::Task)\n\nPrint an error log to stderr if task t fails.\n\nExamples\n\njulia> Base._wait(errormonitor(Threads.@spawn error(\"task failed\")))\nUnhandled Task ERROR: task failed\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@sync","location":"base/parallel.html#Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn, @spawnat and @distributed are complete. All exceptions thrown by enclosed async operations are collected and thrown as a CompositeException.\n\nExamples\n\njulia> Threads.nthreads()\n4\n\njulia> @sync begin\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 1\")\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 2\")\n       end;\nThread-id 3, task 1\nThread-id 1, task 2\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.wait","location":"base/parallel.html#Base.wait","category":"function","text":"Special note for Threads.Condition:\n\nThe caller must be holding the lock that owns a Threads.Condition before calling this method. The calling task will be blocked until some other task wakes it, usually by calling notify on the same Threads.Condition object. The lock will be atomically released when blocking (even if it was locked recursively), and will be reacquired before returning.\n\n\n\n\n\nwait([x])\n\nBlock the current task until some event occurs, depending on the type of the argument:\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition and return the val parameter passed to notify. Waiting on a condition additionally allows passing first=true which results in the waiter being put first in line to wake up on notify instead of the usual first-in-first-out behavior.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process can be used to determine success or failure.\nTask: Wait for a Task to finish. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be restarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before proceeding.\n\n\n\n\n\nwait(c::Channel)\n\nBlocks until the Channel isready.\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> task = Task(() -> wait(c));\n\njulia> schedule(task);\n\njulia> istaskdone(task)  # task is blocked because channel is not ready\nfalse\n\njulia> put!(c, 1);\n\njulia> istaskdone(task)  # task is now unblocked\ntrue\n\n\n\n\n\nwait(r::Future)\n\nWait for a value to become available for the specified Future.\n\n\n\n\n\nwait(r::RemoteChannel, args...)\n\nWait for a value to become available on the specified RemoteChannel.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Task}","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Any}","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.timedwait","location":"base/parallel.html#Base.timedwait","category":"function","text":"timedwait(testcb, timeout::Real; pollint::Real=0.1)\n\nWaits until testcb() returns true or timeout seconds have passed, whichever is earlier. The test function is polled every pollint seconds. The minimum value for pollint is 0.001 seconds, that is, 1 millisecond.\n\nReturn :ok or :timed_out.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Condition","location":"base/parallel.html#Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a Condition are suspended and queued. Tasks are woken up when notify is later called on the Condition. Edge triggering means that only tasks waiting at the time notify is called can be woken up. For level-triggered notifications, you must keep extra state to keep track of whether a notification has happened. The Channel and Threads.Event types do this, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Threads.Condition","location":"base/parallel.html#Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call lock on it. When wait is called, the lock is atomically released during blocking, and will be reacquired before wait returns. Therefore idiomatic use of a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Event","location":"base/parallel.html#Base.Event","category":"type","text":"Event([autoreset=false])\n\nCreate a level-triggered event source. Tasks that call wait on an Event are suspended and queued until notify is called on the Event. After notify is called, the Event remains in a signaled state and tasks will no longer block when waiting for it, until reset is called.\n\nIf autoreset is true, at most one task will be released from wait for each call to notify.\n\nThis provides an acquire & release memory ordering on notify/wait.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\ncompat: Julia 1.8\nThe autoreset functionality and memory ordering guarantee requires at least Julia 1.8.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.notify","location":"base/parallel.html#Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default), all waiting tasks are woken, otherwise only one is. If error is true, the passed value is raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.reset","location":"base/parallel.html#Base.reset-Tuple{Base.Event}","category":"method","text":"reset(::Event)\n\nReset an Event back into an un-set state. Then any future calls to wait will block until notify is called again.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Semaphore","location":"base/parallel.html#Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size acquires to be in use at any time. Each acquire must be matched with a release.\n\nThis provides a acquire & release memory ordering on acquire/release calls.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.acquire","location":"base/parallel.html#Base.acquire","category":"function","text":"acquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available, blocking until one can be acquired.\n\n\n\n\n\nacquire(f, s::Semaphore)\n\nExecute f after acquiring from Semaphore s, and release on completion or error.\n\nFor example, a do-block form that ensures only 2 calls of foo will be active at the same time:\n\ns = Base.Semaphore(2)\n@sync for _ in 1:100\n    Threads.@spawn begin\n        Base.acquire(s) do\n            foo()\n        end\n    end\nend\n\ncompat: Julia 1.8\nThis method requires at least Julia 1.8.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.release","location":"base/parallel.html#Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool, possibly allowing another task to acquire it and resume execution.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.AbstractLock","location":"base/parallel.html#Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that implement the synchronization primitives: lock, trylock, unlock, and islocked.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.lock","location":"base/parallel.html#Base.lock","category":"function","text":"lock(lock)\n\nAcquire the lock when it becomes available. If the lock is already locked by a different task/thread, wait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f returns. If the lock is already locked by a different task/thread, wait for it to become available.\n\nWhen this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\ncompat: Julia 1.7\nUsing a Channel as the second argument requires Julia 1.7 or later.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.unlock","location":"base/parallel.html#Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an internal counter and return immediately.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.trylock","location":"base/parallel.html#Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available, and return true if successful. If the lock is already locked by a different task/thread, return false.\n\nEach successful trylock must be matched by an unlock.\n\nFunction trylock combined with islocked can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.islocked","location":"base/parallel.html#Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread. This function alone should not be used for synchronization. However, islocked combined with trylock can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\nExtended help\n\nFor example, an exponential backoff can be implemented as follows if the lock implementation satisfied the properties documented below.\n\nnspins = 0\nwhile true\n    while islocked(lock)\n        GC.safepoint()\n        nspins += 1\n        nspins > LIMIT && error(\"timeout\")\n    end\n    trylock(lock) && break\n    backoff()\nend\n\nImplementation\n\nA lock implementation is advised to define islocked with the following properties and note it in its docstring.\n\nislocked(lock) is data-race-free.\nIf islocked(lock) returns false, an immediate invocation of trylock(lock) must succeed (returns true) if there is no interference from other tasks.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.ReentrantLock","location":"base/parallel.html#Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreates a re-entrant lock for synchronizing Tasks. The same task can acquire the lock as many times as required. Each lock must be matched with an unlock.\n\nCalling 'lock' will also inhibit running of finalizers on that thread until the corresponding 'unlock'. Use of the standard lock pattern illustrated below should naturally be supported, but beware of inverting the try/lock order or missing the try block entirely (e.g. attempting to return with the lock still held):\n\nThis provides a acquire/release memory ordering on lock/unlock calls.\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\nIf !islocked(lck::ReentrantLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Channels","location":"base/parallel.html#Channels","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.AbstractChannel\nBase.Channel\nBase.Channel(::Function)\nBase.put!(::Channel, ::Any)\nBase.take!(::Channel)\nBase.isready(::Channel)\nBase.fetch(::Channel)\nBase.close(::Channel)\nBase.bind(c::Channel, task::Task)","page":"Tasks"},{"title":"Base.AbstractChannel","location":"base/parallel.html#Base.AbstractChannel","category":"type","text":"AbstractChannel{T}\n\nRepresentation of a channel passing objects of type T.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstructs a Channel with an internal buffer that can hold a maximum of size objects of type T. put! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called. And vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel-Tuple{Function}","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false)\n\nCreate a new task from func, bind it to a new channel of type T and size size, and schedule the task, all in a single call. The channel is automatically closed when the task terminates.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via the keyword argument taskref.\n\nIf spawn = true, the Task created for func may be scheduled on another thread in parallel, equivalent to creating a task via Threads.@spawn.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3. In earlier versions of Julia, Channel used keyword arguments to set size and T, but those constructors are deprecated.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end\nChannel{Char}(1) (2 items available)\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.put!","location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different task.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.take!","location":"base/parallel.html#Base.take!-Tuple{Channel}","category":"method","text":"take!(c::Channel)\n\nRemoves and returns a value from a Channel in order. Blocks until data is available. For unbuffered channels, blocks until a put! is performed by a different task.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(1);\n\njulia> put!(c, 1);\n\njulia> take!(c)\n1\n\nUnbuffered channel:\n\njulia> c = Channel(0);\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);\n\njulia> take!(c)\n1\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.isready","location":"base/parallel.html#Base.isready-Tuple{Channel}","category":"method","text":"isready(c::Channel)\n\nDetermines whether a Channel has a value stored in it. Returns immediately, does not block.\n\nFor unbuffered channels returns true if there are tasks waiting on a put!.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isready(c)\ntrue\n\nUnbuffered channel:\n\njulia> c = Channel();\n\njulia> isready(c)  # no tasks waiting to put!\nfalse\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);  # schedule a put! task\n\njulia> isready(c)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Channel}","category":"method","text":"fetch(c::Channel)\n\nWaits for and returns (without removing) the first available item from the Channel. Note: fetch is unsupported on an unbuffered (0-size) Channel.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(3) do ch\n           foreach(i -> put!(ch, i), 1:3)\n       end;\n\njulia> fetch(c)\n1\n\njulia> collect(c)  # item is not removed\n3-element Vector{Any}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.close","location":"base/parallel.html#Base.close-Tuple{Channel}","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.bind","location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task. Channel chnl is automatically closed when the task terminates. Any uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination. Terminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will close the channel. When multiple channels are bound to the same task, termination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n","page":"Tasks"},{"title":"Low-level synchronization using schedule and wait","location":"base/parallel.html#low-level-schedule-wait","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"The easiest correct use of schedule is on a Task that is not started (scheduled) yet.  However, it is possible to use schedule and wait as a very low-level building block for constructing synchronization interfaces.  A crucial pre-condition of calling schedule(task) is that the caller must \"own\" the task; i.e., it must know that the call to wait in the given task is happening at the locations known to the code calling schedule(task).  One strategy for ensuring such pre-condition is to use atomics, as demonstrated in the following example:","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"@enum OWEState begin\n    OWE_EMPTY\n    OWE_WAITING\n    OWE_NOTIFYING\nend\n\nmutable struct OneWayEvent\n    @atomic state::OWEState\n    task::Task\n    OneWayEvent() = new(OWE_EMPTY)\nend\n\nfunction Base.notify(ev::OneWayEvent)\n    state = @atomic ev.state\n    while state !== OWE_NOTIFYING\n        # Spin until we successfully update the state to OWE_NOTIFYING:\n        state, ok = @atomicreplace(ev.state, state => OWE_NOTIFYING)\n        if ok\n            if state == OWE_WAITING\n                # OWE_WAITING -> OWE_NOTIFYING transition means that the waiter task is\n                # already waiting or about to call `wait`. The notifier task must wake up\n                # the waiter task.\n                schedule(ev.task)\n            else\n                @assert state == OWE_EMPTY\n                # Since we are assuming that there is only one notifier task (for\n                # simplicity), we know that the other possible case here is OWE_EMPTY.\n                # We do not need to do anything because we know that the waiter task has\n                # not called `wait(ev::OneWayEvent)` yet.\n            end\n            break\n        end\n    end\n    return\nend\n\nfunction Base.wait(ev::OneWayEvent)\n    ev.task = current_task()\n    state, ok = @atomicreplace(ev.state, OWE_EMPTY => OWE_WAITING)\n    if ok\n        # OWE_EMPTY -> OWE_WAITING transition means that the notifier task is guaranteed to\n        # invoke OWE_WAITING -> OWE_NOTIFYING transition.  The waiter task must call\n        # `wait()` immediately.  In particular, it MUST NOT invoke any function that may\n        # yield to the scheduler at this point in code.\n        wait()\n    else\n        @assert state == OWE_NOTIFYING\n        # Otherwise, the `state` must have already been moved to OWE_NOTIFYING by the\n        # notifier task.\n    end\n    return\nend\n\nev = OneWayEvent()\n@sync begin\n    @async begin\n        wait(ev)\n        println(\"done\")\n    end\n    println(\"notifying...\")\n    notify(ev)\nend\n\n# output\nnotifying...\ndone","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"OneWayEvent lets one task to wait for another task's notify.  It is a limited communication interface since wait can only be used once from a single task (note the non-atomic assignment of ev.task)","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"In this example, notify(ev::OneWayEvent) is allowed to call schedule(ev.task) if and only if it modifies the state from OWE_WAITING to OWE_NOTIFYING.  This lets us know that the task executing wait(ev::OneWayEvent) is now in the ok branch and that there cannot be other tasks that tries to schedule(ev.task) since their @atomicreplace(ev.state, state => OWE_NOTIFYING) will fail.","page":"Tasks"},{"title":"Missing Values","location":"manual/missing.html#missing","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Julia provides support for representing missing values in the statistical sense. This is for situations where no value is available for a variable in an observation, but a valid value theoretically exists. Missing values are represented via the missing object, which is the singleton instance of the type Missing. missing is equivalent to NULL in SQL and NA in R, and behaves like them in most situations.","page":"Missing Values"},{"title":"Propagation of Missing Values","location":"manual/missing.html#Propagation-of-Missing-Values","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"missing values propagate automatically when passed to standard mathematical operators and functions. For these functions, uncertainty about the value of one of the operands induces uncertainty about the result. In practice, this means a math operation involving a missing value generally returns missing:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> missing + 1\nmissing\n\njulia> \"a\" * missing\nmissing\n\njulia> abs(missing)\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Since missing is a normal Julia object, this propagation rule only works for functions which have opted in to implement this behavior. This can be achieved by:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"adding a specific method defined for arguments of type Missing,\naccepting arguments of this type, and passing them to functions which propagate them (like standard math operators).","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Packages should consider whether it makes sense to propagate missing values when defining new functions, and define methods appropriately if this is the case. Passing a missing value to a function which does not have a method accepting arguments of type Missing throws a MethodError, just like for any other type.","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Functions that do not propagate missing values can be made to do so by wrapping them in the passmissing function provided by the Missings.jl package. For example, f(x) becomes passmissing(f)(x).","page":"Missing Values"},{"title":"Equality and Comparison Operators","location":"manual/missing.html#Equality-and-Comparison-Operators","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Standard equality and comparison operators follow the propagation rule presented above: if any of the operands is missing, the result is missing. Here are a few examples:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> missing == 1\nmissing\n\njulia> missing == missing\nmissing\n\njulia> missing < 1\nmissing\n\njulia> 2 >= missing\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"In particular, note that missing == missing returns missing, so == cannot be used to test whether a value is missing. To test whether x is missing, use ismissing(x).","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Special comparison operators isequal and === are exceptions to the propagation rule. They will always return a Bool value, even in the presence of missing values, considering missing as equal to missing and as different from any other value. They can therefore be used to test whether a value is missing:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> missing === 1\nfalse\n\njulia> isequal(missing, 1)\nfalse\n\njulia> missing === missing\ntrue\n\njulia> isequal(missing, missing)\ntrue","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"The isless operator is another exception: missing is considered as greater than any other value. This operator is used by sort, which therefore places missing values after all other values:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> isless(1, missing)\ntrue\n\njulia> isless(missing, Inf)\nfalse\n\njulia> isless(missing, missing)\nfalse","page":"Missing Values"},{"title":"Logical operators","location":"manual/missing.html#Logical-operators","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Logical (or boolean) operators |, & and xor are another special case since they only propagate missing values when it is logically required. For these operators, whether or not the result is uncertain, depends on the particular operation. This follows the well-established rules of three-valued logic which are implemented by e.g. NULL in SQL and NA in R. This abstract definition corresponds to a relatively natural behavior which is best explained via concrete examples.","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Let us illustrate this principle with the logical \"or\" operator |. Following the rules of boolean logic, if one of the operands is true, the value of the other operand does not have an influence on the result, which will always be true:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> true | true\ntrue\n\njulia> true | false\ntrue\n\njulia> false | true\ntrue","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Based on this observation, we can conclude if one of the operands is true and the other missing, we know that the result is true in spite of the uncertainty about the actual value of one of the operands. If we had been able to observe the actual value of the second operand, it could only be true or false, and in both cases the result would be true. Therefore, in this particular case, missingness does not propagate:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> true | missing\ntrue\n\njulia> missing | true\ntrue","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"On the contrary, if one of the operands is false, the result could be either true or false depending on the value of the other operand. Therefore, if that operand is missing, the result has to be missing too:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> false | true\ntrue\n\njulia> true | false\ntrue\n\njulia> false | false\nfalse\n\njulia> false | missing\nmissing\n\njulia> missing | false\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"The behavior of the logical \"and\" operator & is similar to that of the | operator, with the difference that missingness does not propagate when one of the operands is false. For example, when that is the case of the first operand:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> false & false\nfalse\n\njulia> false & true\nfalse\n\njulia> false & missing\nfalse","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"On the other hand, missingness propagates when one of the operands is true, for example the first one:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> true & true\ntrue\n\njulia> true & false\nfalse\n\njulia> true & missing\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Finally, the \"exclusive or\" logical operator xor always propagates missing values, since both operands always have an effect on the result. Also note that the negation operator ! returns missing when the operand is missing, just like other unary operators.","page":"Missing Values"},{"title":"Control Flow and Short-Circuiting Operators","location":"manual/missing.html#Control-Flow-and-Short-Circuiting-Operators","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Control flow operators including if, while and the ternary operator x ? y : z do not allow for missing values. This is because of the uncertainty about whether the actual value would be true or false if we could observe it. This implies we do not know how the program should behave. In this case, a TypeError is thrown as soon as a missing value is encountered in this context:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> if missing\n           println(\"here\")\n       end\nERROR: TypeError: non-boolean (Missing) used in boolean context","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"For the same reason, contrary to logical operators presented above, the short-circuiting boolean operators && and || do not allow for missing values in situations where the value of the operand determines whether the next operand is evaluated or not. For example:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> missing || false\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\njulia> missing && false\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\njulia> true && missing && false\nERROR: TypeError: non-boolean (Missing) used in boolean context","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"In contrast, there is no error thrown when the result can be determined without the missing values. This is the case when the code short-circuits before evaluating the missing operand, and when the missing operand is the last one:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> true && missing\nmissing\n\njulia> false && missing\nfalse","page":"Missing Values"},{"title":"Arrays With Missing Values","location":"manual/missing.html#Arrays-With-Missing-Values","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Arrays containing missing values can be created like other arrays:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> [1, missing]\n2-element Vector{Union{Missing, Int64}}:\n 1\n  missing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"As this example shows, the element type of such arrays is Union{Missing, T}, with T the type of the non-missing values. This reflects the fact that array entries can be either of type T (here, Int64) or of type Missing. This kind of array uses an efficient memory storage equivalent to an Array{T} holding the actual values combined with an Array{UInt8} indicating the type of the entry (i.e. whether it is Missing or T).","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Arrays allowing for missing values can be constructed with the standard syntax. Use Array{Union{Missing, T}}(missing, dims) to create arrays filled with missing values:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> Array{Union{Missing, String}}(missing, 2, 3)\n2×3 Matrix{Union{Missing, String}}:\n missing  missing  missing\n missing  missing  missing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"note: Note\nUsing undef or similar may currently give an array filled with missing, but this is not the correct way to obtain such an array. Use a missing constructor as shown above instead.","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"An array with element type allowing missing entries (e.g. Vector{Union{Missing, T}}) which does not contain any missing entries can be converted to an array type that does not allow for missing entries (e.g. Vector{T}) using convert. If the array contains missing values, a MethodError is thrown during conversion:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> x = Union{Missing, String}[\"a\", \"b\"]\n2-element Vector{Union{Missing, String}}:\n \"a\"\n \"b\"\n\njulia> convert(Array{String}, x)\n2-element Vector{String}:\n \"a\"\n \"b\"\n\njulia> y = Union{Missing, String}[missing, \"b\"]\n2-element Vector{Union{Missing, String}}:\n missing\n \"b\"\n\njulia> convert(Array{String}, y)\nERROR: MethodError: Cannot `convert` an object of type Missing to an object of type String","page":"Missing Values"},{"title":"Skipping Missing Values","location":"manual/missing.html#Skipping-Missing-Values","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Since missing values propagate with standard mathematical operators, reduction functions return missing when called on arrays which contain missing values:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> sum([1, missing])\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"In this situation, use the skipmissing function to skip missing values:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> sum(skipmissing([1, missing]))\n1","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"This convenience function returns an iterator which filters out missing values efficiently. It can therefore be used with any function which supports iterators:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> x = skipmissing([3, missing, 2, 1])\nskipmissing(Union{Missing, Int64}[3, missing, 2, 1])\n\njulia> maximum(x)\n3\n\njulia> sum(x)\n6\n\njulia> mapreduce(sqrt, +, x)\n4.146264369941973","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Objects created by calling skipmissing on an array can be indexed using indices from the parent array. Indices corresponding to missing values are not valid for these objects, and an error is thrown when trying to use them (they are also skipped by keys and eachindex):","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> x[1]\n3\n\njulia> x[2]\nERROR: MissingException: the value at index (2,) is missing\n[...]","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"This allows functions which operate on indices to work in combination with skipmissing. This is notably the case for search and find functions. These functions return indices valid for the object returned by skipmissing, and are also the indices of the matching entries in the parent array:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> findall(==(1), x)\n1-element Vector{Int64}:\n 4\n\njulia> findfirst(!iszero, x)\n1\n\njulia> argmax(x)\n1","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Use collect to extract non-missing values and store them in an array:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> collect(x)\n3-element Vector{Int64}:\n 3\n 2\n 1","page":"Missing Values"},{"title":"Logical Operations on Arrays","location":"manual/missing.html#Logical-Operations-on-Arrays","category":"section","text":"","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"The three-valued logic described above for logical operators is also used by logical functions applied to arrays. Thus, array equality tests using the == operator return missing whenever the result cannot be determined without knowing the actual value of the missing entry. In practice, this means missing is returned if all non-missing values of the compared arrays are equal, but one or both arrays contain missing values (possibly at different positions):","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> [1, missing] == [2, missing]\nfalse\n\njulia> [1, missing] == [1, missing]\nmissing\n\njulia> [1, 2, missing] == [1, missing, 2]\nmissing","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"As for single values, use isequal to treat missing values as equal to other missing values, but different from non-missing values:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> isequal([1, missing], [1, missing])\ntrue\n\njulia> isequal([1, 2, missing], [1, missing, 2])\nfalse","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"Functions any and all also follow the rules of three-valued logic. Thus, returning missing when the result cannot be determined:","page":"Missing Values"},{"title":"Missing Values","location":"manual/missing.html","category":"page","text":"julia> all([true, missing])\nmissing\n\njulia> all([false, missing])\nfalse\n\njulia> any([true, missing])\ntrue\n\njulia> any([false, missing])\nmissing","page":"Missing Values"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Using-Valgrind-with-Julia","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind is a tool for memory debugging, memory leak detection, and profiling.  This section describes things to keep in mind when using Valgrind to debug memory issues with Julia.","page":"Using Valgrind with Julia"},{"title":"General considerations","location":"devdocs/valgrind.html#General-considerations","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"By default, Valgrind assumes that there is no self modifying code in the programs it runs.  This assumption works fine in most instances but fails miserably for a just-in-time compiler like julia.  For this reason it is crucial to pass --smc-check=all-non-file to valgrind, else code may crash or behave unexpectedly (often in subtle ways).","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"In some cases, to better detect memory errors using Valgrind it can help to compile julia with memory pools disabled.  The compile-time flag MEMDEBUG disables memory pools in Julia, and MEMDEBUG2 disables memory pools in FemtoLisp.  To build julia with both flags, add the following line to Make.user:","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"CFLAGS = -DMEMDEBUG -DMEMDEBUG2","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Another thing to note: if your program uses multiple workers processes, it is likely that you want all such worker processes to run under Valgrind, not just the parent process.  To do this, pass --trace-children=yes to valgrind.","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Yet another thing to note: if using valgrind errors with Unable to find compatible target in system image, try rebuilding the sysimage with target generic or julia with JULIA_CPU_TARGET=generic.","page":"Using Valgrind with Julia"},{"title":"Suppressions","location":"devdocs/valgrind.html#Suppressions","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind will typically display spurious warnings as it runs.  To reduce the number of such warnings, it helps to provide a suppressions file to Valgrind.  A sample suppressions file is included in the Julia source distribution at contrib/valgrind-julia.supp.","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"The suppressions file can be used from the julia/ source directory as follows:","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"$ valgrind --smc-check=all-non-file --suppressions=contrib/valgrind-julia.supp ./julia progname.jl","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Any memory errors that are displayed should either be reported as bugs or contributed as additional suppressions.  Note that some versions of Valgrind are shipped with insufficient default suppressions, so that may be one thing to consider before submitting any bugs.","page":"Using Valgrind with Julia"},{"title":"Running the Julia test suite under Valgrind","location":"devdocs/valgrind.html#Running-the-Julia-test-suite-under-Valgrind","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"It is possible to run the entire Julia test suite under Valgrind, but it does take quite some time (typically several hours).  To do so, run the following command from the julia/test/ directory:","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"valgrind --smc-check=all-non-file --trace-children=yes --suppressions=$PWD/../contrib/valgrind-julia.supp ../julia runtests.jl all","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"If you would like to see a report of \"definite\" memory leaks, pass the flags --leak-check=full --show-leak-kinds=definite to valgrind as well.","page":"Using Valgrind with Julia"},{"title":"Additional spurious warnings","location":"devdocs/valgrind.html#Additional-spurious-warnings","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"This section covers Valgrind warnings which cannot be added to the suppressions file yet are nonetheless safe to ignore.","page":"Using Valgrind with Julia"},{"title":"Unhandled rr system calls","location":"devdocs/valgrind.html#Unhandled-rr-system-calls","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind will emit a warning if it encounters any of the system calls that are specific to rr, the Record and Replay Framework.  In particular, a warning about an unhandled 1008 syscall will be shown when julia tries to detect whether it is running under rr:","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"--xxxxxx-- WARNING: unhandled amd64-linux syscall: 1008\n--xxxxxx-- You may be able to write your own handler.\n--xxxxxx-- Read the file README_MISSING_SYSCALL_OR_IOCTL.\n--xxxxxx-- Nevertheless we consider this a bug.  Please report\n--xxxxxx-- it at http://valgrind.org/support/bug_reports.html.","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"This issue has been reported to the Valgrind developers as they have requested.","page":"Using Valgrind with Julia"},{"title":"Caveats","location":"devdocs/valgrind.html#Caveats","category":"section","text":"","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind currently does not support multiple rounding modes, so code that adjusts the rounding mode will behave differently when run under Valgrind.","page":"Using Valgrind with Julia"},{"title":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"In general, if after setting --smc-check=all-non-file you find that your program behaves differently when run under Valgrind, it may help to pass --tool=none to valgrind as you investigate further.  This will enable the minimal Valgrind machinery but will also run much faster than when the full memory checker is enabled.","page":"Using Valgrind with Julia"},{"title":"Windows","location":"devdocs/build/windows.html#Windows","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"This file describes how to install, or build, and use Julia on Windows.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"For more general information about Julia, please see the main README or the documentation.","page":"Windows"},{"title":"General Information for Windows","location":"devdocs/build/windows.html#General-Information-for-Windows","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"We highly recommend running Julia using a modern terminal application, in particular Windows Terminal, which can be installed from the Microsoft Store.","page":"Windows"},{"title":"Line endings","location":"devdocs/build/windows.html#Line-endings","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Julia uses binary-mode files exclusively. Unlike many other Windows programs, if you write \\n to a file, you get a \\n in the file, not some other bit pattern. This matches the behavior exhibited by other operating systems. If you have installed Git for Windows, it is suggested, but not required, that you configure your system Git to use the same convention:","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"git config --global core.eol lf\ngit config --global core.autocrlf input","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"or edit %USERPROFILE%\\.gitconfig and add/edit the lines:","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"[core]\n    eol = lf\n    autocrlf = input","page":"Windows"},{"title":"Binary distribution","location":"devdocs/build/windows.html#Binary-distribution","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"For the binary distribution installation notes on Windows please see the instructions at https://julialang.org/downloads/platform/#windows.","page":"Windows"},{"title":"Source distribution","location":"devdocs/build/windows.html#Source-distribution","category":"section","text":"","page":"Windows"},{"title":"Cygwin-to-MinGW cross-compiling","location":"devdocs/build/windows.html#Cygwin-to-MinGW-cross-compiling","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"The recommended way of compiling Julia from source on Windows is by cross compiling from Cygwin, using versions of the MinGW-w64 compilers available through Cygwin's package manager.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Download and run Cygwin setup for 32 bit or 64 bit. Note, that you can compile either 32 or 64 bit Julia from either 32 or 64 bit Cygwin. 64 bit Cygwin has a slightly smaller but often more up-to-date selection of packages.\nAdvanced: you may skip steps 2-4 by running:\nsetup-x86_64.exe -s <url> -q -P cmake,gcc-g++,git,make,patch,curl,m4,python3,p7zip,mingw64-i686-gcc-g++,mingw64-i686-gcc-fortran,mingw64-x86_64-gcc-g++,mingw64-x86_64-gcc-fortran\n:: replace <url> with a site from https://cygwin.com/mirrors.html\n:: or run setup manually first and select a mirror\nSelect installation location and download mirror.\nAt the 'Select Packages' step, select the following:\nFrom the Devel category: cmake, gcc-g++, git, make, patch\nFrom the Net category: curl\nFrom Interpreters (or Python) category: m4, python3\nFrom the Archive category: p7zip\nFor 32 bit Julia, and also from the Devel category:  mingw64-i686-gcc-g++ and mingw64-i686-gcc-fortran\nFor 64 bit Julia, and also from the Devel category:  mingw64-x86_64-gcc-g++ and mingw64-x86_64-gcc-fortran\nAllow Cygwin installation to finish, then start from the installed shortcut a 'Cygwin Terminal', or 'Cygwin64 Terminal', respectively.\nBuild Julia and its dependencies from source:\nGet the Julia sources\ngit clone https://github.com/JuliaLang/julia.git\ncd julia\nTip: If you get an error: cannot fork() for fetch-pack: Resource temporarily unavailable from git, add alias git=\"env PATH=/usr/bin git\" to ~/.bashrc and restart Cygwin.\nSet the XC_HOST variable in Make.user to indicate MinGW-w64 cross compilation\necho 'XC_HOST = i686-w64-mingw32' > Make.user     # for 32 bit Julia\n# or\necho 'XC_HOST = x86_64-w64-mingw32' > Make.user   # for 64 bit Julia\nStart the build\nmake -j 4   # Adjust the number of threads (4) to match your build environment.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"> Protip: build both!\n> ```sh\n> make O=julia-win32 configure\n> make O=julia-win64 configure\n> echo 'XC_HOST = i686-w64-mingw32' > julia-win32/Make.user\n> echo 'XC_HOST = x86_64-w64-mingw32' > julia-win64/Make.user\n> echo 'ifeq ($(BUILDROOT),$(JULIAHOME))\n>         $(error \"in-tree build disabled\")\n>       endif' >> Make.user\n> make -C julia-win32  # build for Windows x86 in julia-win32 folder\n> make -C julia-win64  # build for Windows x86-64 in julia-win64 folder\n> ```","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Run Julia using the Julia executables directly\nusr/bin/julia.exe\nusr/bin/julia-debug.exe","page":"Windows"},{"title":"Compiling with MinGW/MSYS2","location":"devdocs/build/windows.html#Compiling-with-MinGW/MSYS2","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"MSYS2 provides a robust MSYS experience.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Note: MSYS2 requires 64 bit Windows 7 or newer.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Install and configure MSYS2, Software Distribution and Building Platform for Windows.\nDownload and run the latest installer for the  64-bit distribution.  The installer will have a name like msys2-x86_64-yyyymmdd.exe.\nOpen MSYS2. Update package database and base packages:  sh  pacman -Syu\nExit and restart MSYS2, Update the rest of the base packages:  sh  pacman -Syu\nThen install tools required to build julia:  ```sh\ntools\npacman -S cmake diffutils git m4 make patch tar p7zip curl python\nFor 64 bit Julia, install x86_64\npacman -S mingw-w64-x86_64-gcc\nFor 32 bit Julia, install i686\npacman -S mingw-w64-i686-gcc  ```\nConfiguration of MSYS2 is complete. Now exit the MSYS2 shell.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Build Julia and its dependencies with pre-build dependencies.\nOpen a new MINGW64/MINGW32 shell.  Currently we can't use both mingw32 and mingw64,  so if you want to build the x86_64 and i686 versions,  you'll need to build them in each environment separately.\nand clone the Julia sources  sh  git clone https://github.com/JuliaLang/julia.git  cd julia\nStart the build  sh  make -j$(nproc)\nProtip: build in dirmake O=julia-mingw-w64 configure\necho 'ifeq ($(BUILDROOT),$(JULIAHOME))\n        $(error \"in-tree build disabled\")\n      endif' >> Make.user\nmake -C julia-mingw-w64","page":"Windows"},{"title":"Cross-compiling from Unix (Linux/Mac/WSL)","location":"devdocs/build/windows.html#Cross-compiling-from-Unix-(Linux/Mac/WSL)","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"You can also use MinGW-w64 cross compilers to build a Windows version of Julia from Linux, Mac, or the Windows Subsystem for Linux (WSL).","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"First, you will need to ensure your system has the required dependencies. We need wine (>=1.7.5), a system compiler, and some downloaders. Note: a cygwin install might interfere with this method if using WSL.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"On Ubuntu (on other Linux systems the dependency names are likely to be similar):","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"apt-get install wine-stable gcc wget p7zip-full winbind mingw-w64 gfortran-mingw-w64\ndpkg --add-architecture i386 && apt-get update && apt-get install wine32 # add sudo to each if needed\n# switch all of the following to their \"-posix\" variants (interactively):\nfor pkg in i686-w64-mingw32-g++ i686-w64-mingw32-gcc i686-w64-mingw32-gfortran x86_64-w64-mingw32-g++ x86_64-w64-mingw32-gcc x86_64-w64-mingw32-gfortran; do sudo update-alternatives --config $pkg; done","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"On Mac: Install XCode, XCode command line tools, X11 (now XQuartz), and MacPorts or Homebrew.  Then run port install wine wget mingw-w64, or brew install wine wget mingw-w64, as appropriate.","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Then run the build:","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"git clone https://github.com/JuliaLang/julia.git julia-win32\ncd julia-win32\necho override XC_HOST = i686-w64-mingw32 >> Make.user\nmake\nmake win-extras (Necessary before running make binary-dist)\nmake binary-dist then make exe to create the Windows installer.\nmove the julia-*.exe installer to the target machine","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"If you are building for 64-bit windows, the steps are essentially the same. Just replace i686 in XC_HOST with x86_64. (note: on Mac, wine only runs in 32-bit mode).","page":"Windows"},{"title":"Debugging a cross-compiled build under wine","location":"devdocs/build/windows.html#Debugging-a-cross-compiled-build-under-wine","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"The most effective way to debug a cross-compiled version of Julia on the cross-compilation host is to install a windows version of gdb and run it under wine as usual. The pre-built packages available as part of the MSYS2 project are known to work. Apart from the GDB package you may also need the python and termcap packages. Finally, GDB's prompt may not work when launch from the command line. This can be worked around by prepending wineconsole to the regular GDB invocation.","page":"Windows"},{"title":"After compiling","location":"devdocs/build/windows.html#After-compiling","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Compiling using one of the options above creates a basic Julia build, but not some extra components that are included if you run the full Julia binary installer. If you need these components, the easiest way to get them is to build the installer yourself using make win-extras followed by make binary-dist and make exe. Then running the resulting installer.","page":"Windows"},{"title":"Windows Build Debugging","location":"devdocs/build/windows.html#Windows-Build-Debugging","category":"section","text":"","page":"Windows"},{"title":"GDB hangs with cygwin mintty","location":"devdocs/build/windows.html#GDB-hangs-with-cygwin-mintty","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Run gdb under the windows console (cmd) instead. gdb may not function properly under mintty with non- cygwin applications. You can use cmd /c start to start the windows console from mintty if necessary.","page":"Windows"},{"title":"GDB not attaching to the right process","location":"devdocs/build/windows.html#GDB-not-attaching-to-the-right-process","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Use the PID from the windows task manager or WINPID from the ps command instead of the PID from unix style command line tools (e.g. pgrep).  You may need to add the PID column if it is not shown by default in the windows task manager.","page":"Windows"},{"title":"GDB not showing the right backtrace","location":"devdocs/build/windows.html#GDB-not-showing-the-right-backtrace","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"When attaching to the julia process, GDB may not be attaching to the right thread.  Use info threads command to show all the threads and thread <threadno> to switch threads.\nBe sure to use a 32 bit version of GDB to debug a 32 bit build of Julia, or a 64 bit version of GDB to debug a 64 bit build of Julia.","page":"Windows"},{"title":"Build process is slow/eats memory/hangs my computer","location":"devdocs/build/windows.html#Build-process-is-slow/eats-memory/hangs-my-computer","category":"section","text":"","page":"Windows"},{"title":"Windows","location":"devdocs/build/windows.html","category":"page","text":"Disable the Windows Superfetch and Program Compatibility Assistant services, as they are known to have spurious interactions with MinGW/Cygwin.\nAs mentioned in the link above: excessive memory use by svchost specifically may be investigated in the Task Manager by clicking on the high-memory svchost.exe process and selecting Go to Services. Disable child services one-by-one until a culprit is found.\nBeware of BLODA. The vmmap tool is indispensable for identifying such software conflicts. Use vmmap to inspect the list of loaded DLLs for bash, mintty, or another persistent process used to drive the build. Essentially any DLL outside of the Windows System directory is potential BLODA.","page":"Windows"},{"title":"Metaprogramming","location":"manual/metaprogramming.html#Metaprogramming","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp, Julia represents its own code as a data structure of the language itself. Since code is represented by objects that can be created and manipulated from within the language, it is possible for a program to transform and generate its own code. This allows sophisticated code generation without extra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees. In contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation and substitution before any actual parsing or interpretation occurs. Because all data types and code in Julia are represented by Julia data structures, powerful reflection capabilities are available to explore the internals of a program and its types just like any other data.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"warning: Warning\nMetaprogramming is a powerful tool, but it introduces complexity that can make code more difficult to understand. For example, it can be surprisingly hard to get scope rules correct. Metaprogramming should typically be used only when other approaches such as higher order functions and closures cannot be applied.eval and defining new macros should be typically used as a last resort. It is almost never a good idea to use Meta.parse or convert an arbitrary string into Julia code. For manipulating Julia code, use the Expr data structure directly to avoid the complexity of how Julia syntax is parsed.The best uses of metaprogramming often implement most of their functionality in runtime helper functions, striving to minimize the amount of code they generate.","page":"Metaprogramming"},{"title":"Program representation","location":"manual/metaprogramming.html#Program-representation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every Julia program starts life as a string:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> prog = \"1 + 1\"\n\"1 + 1\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens next?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The next step is to parse each string into an object called an expression, represented by the Julia type Expr:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = Meta.parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects contain two parts:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"a Symbol identifying the kind of expression. A symbol is an interned string identifier (more discussion below).","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.head\n:call","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the expression arguments, which may be symbols, other expressions, or literal values:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.args\n3-element Vector{Any}:\n  :+\n 1\n 1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions may also be constructed directly in prefix notation:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The two expressions constructed above – by parsing and by direct construction – are equivalent:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 == ex2\ntrue","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The key point here is that Julia code is internally represented as a data structure that is accessible from the language itself.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The dump function provides indented and annotated display of Expr objects:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects may also be nested:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex3 = Meta.parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to view expressions is with Meta.show_sexpr, which displays the S-expression form of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating the display on a nested Expr:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)","page":"Metaprogramming"},{"title":"Symbols","location":"manual/metaprogramming.html#Symbols","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The : character has two syntactic purposes in Julia. The first form creates a Symbol, an interned string used as one building-block of expressions, from valid identifiers:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> s = :foo\n:foo\n\njulia> typeof(s)\nSymbol","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The Symbol constructor takes any number of arguments and creates a new symbol by concatenating their string representations together:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :foo === Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"1foo\") # `:1foo` would not work, as `1foo` is not a valid identifier\nSymbol(\"1foo\")\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_sym","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the context of an expression, symbols are used to indicate access to variables; when an expression is evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Sometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :(:)\n:(:)\n\njulia> :(::)\n:(::)","page":"Metaprogramming"},{"title":"Expressions and evaluation","location":"manual/metaprogramming.html#Expressions-and-evaluation","category":"section","text":"","page":"Metaprogramming"},{"title":"Quoting","location":"manual/metaprogramming.html#Quoting","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The second syntactic purpose of the : character is to create expression objects without using the explicit Expr constructor. This is referred to as quoting. The : character, followed by paired parentheses around a single statement of Julia code, produces an Expr object based on the enclosed code. Here is an example of the short form used to quote an arithmetic expression:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(to view the structure of this expression, try ex.head and ex.args, or use dump as above or Meta.@dump)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that equivalent expressions may be constructed using Meta.parse or the direct Expr form:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia>      :(a + b*c + 1)       ==\n       Meta.parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrue","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions provided by the parser generally only have symbols, other expressions, and literal values as their args, whereas expressions constructed by Julia code can have arbitrary run-time values without literal forms as args. In this specific example, + and a are symbols, *(b,c) is a subexpression, and 1 is a literal 64-bit signed integer.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is a second syntactic form of quoting for multiple expressions: blocks of code enclosed in quote ... end.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote\n    #= none:2 =#\n    x = 1\n    #= none:3 =#\n    y = 2\n    #= none:4 =#\n    x + y\nend\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Interpolation","location":"manual/metaprogramming.html#man-expression-interpolation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Direct construction of Expr objects with value arguments is powerful, but Expr constructors can be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows interpolation of literals or expressions into quoted expressions. Interpolation is indicated by a prefix $.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the value of variable a is interpolated:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Interpolating into an unquoted expression is not supported and will cause a compile-time error:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> $a + b\nERROR: syntax: \"$\" expression outside quote","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The use of $ for expression interpolation is intentionally reminiscent of string interpolation and command interpolation. Expression interpolation allows convenient, readable programmatic construction of complex Julia expressions.","page":"Metaprogramming"},{"title":"Splatting interpolation","location":"manual/metaprogramming.html#Splatting-interpolation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the $ interpolation syntax allows inserting only a single expression into an enclosing expression. Occasionally, you have an array of expressions and need them all to become arguments of the surrounding expression. This can be done with the syntax $(xs...). For example, the following code generates a function call where the number of arguments is determined programmatically:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> args = [:x, :y, :z];\n\njulia> :(f(1, $(args...)))\n:(f(1, x, y, z))","page":"Metaprogramming"},{"title":"Nested quote","location":"manual/metaprogramming.html#Nested-quote","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Naturally, it is possible for quote expressions to contain other quote expressions. Understanding how interpolation works in these cases can be a bit tricky. Consider this example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = :(1 + 2);\n\njulia> e = quote quote $x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :x))\nend))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the result contains $x, which means that x has not been evaluated yet. In other words, the $ expression \"belongs to\" the inner quote expression, and so its argument is only evaluated when the inner quote expression is:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    1 + 2\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, the outer quote expression is able to interpolate values inside the $ in the inner quote. This is done with multiple $s:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> e = quote quote $$x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :(1 + 2)))\nend))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that (1 + 2) now appears in the result instead of the symbol x. Evaluating this expression yields an interpolated 3:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    3\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The intuition behind this behavior is that x is evaluated once for each $: one $ works similarly to eval(:x), giving x's value, while two $s do the equivalent of eval(eval(:x)).","page":"Metaprogramming"},{"title":"QuoteNode","location":"manual/metaprogramming.html#man-quote-node","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The usual representation of a quote form in an AST is an Expr with head :quote:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":(1+2)\"))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Expr\n      head: Symbol call\n      args: Array{Any}((3,))\n        1: Symbol +\n        2: Int64 1\n        3: Int64 2","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As we have seen, such expressions support interpolation with $. However, in some situations it is necessary to quote code without performing interpolation. This kind of quoting does not yet have syntax, but is represented internally as an object of type QuoteNode:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(Meta.quot(Expr(:$, :(1+2))))\n3\n\njulia> eval(QuoteNode(Expr(:$, :(1+2))))\n:($(Expr(:$, :(1 + 2))))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The parser yields QuoteNodes for simple quoted items like symbols:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":x\"))\nQuoteNode\n  value: Symbol x","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"QuoteNode can also be used for certain advanced metaprogramming tasks.","page":"Metaprogramming"},{"title":"Evaluating expressions","location":"manual/metaprogramming.html#Evaluating-expressions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Given an expression object, one can cause Julia to evaluate (execute) it at global scope using eval:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = :(1 + 2)\n:(1 + 2)\n\njulia> eval(ex1)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: `b` not defined\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every module has its own eval function that evaluates expressions in its global scope. Expressions passed to eval are not limited to returning values – they can also have side-effects that alter the state of the enclosing module's environment:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: `x` not defined\n\njulia> eval(ex)\n1\n\njulia> x\n1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, the evaluation of an expression object causes a value to be assigned to the global variable x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since expressions are just Expr objects which can be constructed programmatically and then evaluated, it is possible to dynamically generate arbitrary code which can then be run using eval. Here is a simple example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of a is used to construct the expression ex which applies the + function to the value 1 and the variable b. Note the important distinction between the way a and b are used:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of the variable a at expression construction time is used as an immediate value in the expression. Thus, the value of a when the expression is evaluated no longer matters: the value in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol :b is used in the expression construction, so the value of the variable b at that time is irrelevant – :b is just a symbol and the variable b need not even be defined. At expression evaluation time, however, the value of the symbol :b is resolved by looking up the value of the variable b.","page":"Metaprogramming"},{"title":"Functions on Expressions","location":"manual/metaprogramming.html#Functions-on-Expressions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate Julia code within Julia itself. We have already seen one example of a function returning Expr objects: the parse function, which takes a string of Julia code and returns the corresponding Expr. A function can also take one or more Expr objects as arguments, and return another Expr. Here is a simple, motivating example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As another example, here is a function that doubles any numeric argument, but leaves expressions alone:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42","page":"Metaprogramming"},{"title":"Macros","location":"manual/metaprogramming.html#man-macros","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros provide a mechanism to include generated code in the final body of a program. A macro maps a tuple of arguments to a returned expression, and the resulting expression is compiled directly rather than requiring a runtime eval call. Macro arguments may include expressions, literal values, and symbols.","page":"Metaprogramming"},{"title":"Basics","location":"manual/metaprogramming.html#Basics","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is an extraordinarily simple macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique name declared in a macro NAME ... end block. In this example, the compiler will replace all instances of @sayhello with:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":":( println(\"Hello, world!\") )","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When @sayhello is entered in the REPL, the expression executes immediately, thus we only see the evaluation result:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello()\nHello, world!","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now, consider a slightly more complex macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro takes one argument: name. When @sayhello is encountered, the quoted expression is expanded to interpolate the value of the argument into the final expression:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello(\"human\")\nHello, human","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can view the quoted return expression using the function macroexpand (important note: this is an extremely useful tool for debugging macros):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = macroexpand(Main, :(@sayhello(\"human\")) )\n:(Main.println(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can see that the \"human\" literal has been interpolated into the expression.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @sayhello \"human\"\n:(println(\"Hello, \", \"human\"))","page":"Metaprogramming"},{"title":"Hold up: why macros?","location":"manual/metaprogramming.html#Hold-up:-why-macros?","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand is also such a function. So, why do macros exist?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are necessary because they execute when code is parsed, therefore, macros allow the programmer to generate and include fragments of customized code before the full program is run. To illustrate the difference, consider the following example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand(Main, :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: :((1, 2, 3))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The first call to println is executed when macroexpand is called. The resulting expression contains only the second println:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(ex)\nExpr\n\njulia> ex\n:(println(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)","page":"Metaprogramming"},{"title":"Macro invocation","location":"manual/metaprogramming.html#Macro-invocation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are invoked with the following general syntax:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name expr1 expr2 ...\n@name(expr1, expr2, ...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note the distinguishing @ before the macro name and the lack of commas between the argument expressions in the first form, and the lack of whitespace after @name in the second form. The two styles should not be mixed. For example, the following syntax is different from the examples above; it passes the tuple (expr1, expr2, ...) as one argument to the macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name (expr1, expr2, ...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An alternative way to invoke a macro over an array literal (or comprehension) is to juxtapose both without using parentheses. In this case, the array will be the only expression fed to the macro. The following syntax is equivalent (and different from @name [a b] * v):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name[a b] * v\n@name([a b]) * v","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is important to emphasize that macros receive their arguments as expressions, literals, or symbols. One way to explore macro arguments is to call the show function within the macro body:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In addition to the given argument list, every macro is passed extra arguments named __source__ and __module__.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __source__ provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. This allows macros to include better error diagnostic information, and is commonly used by logging, string-parser macros, and docs, for example, as well as to implement the @__LINE__, @__FILE__, and @__DIR__ macros.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The location information can be accessed by referencing __source__.line and __source__.file:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro __LOCATION__(); return QuoteNode(__source__); end\n@__LOCATION__ (macro with 1 method)\n\njulia> dump(\n            @__LOCATION__(\n       ))\nLineNumberNode\n  line: Int64 2\n  file: Symbol none","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __module__ provides information (in the form of a Module object) about the expansion context of the macro invocation. This allows macros to look up contextual information, such as existing bindings, or to insert the value as an extra argument to a runtime function call doing self-reflection in the current module.","page":"Metaprogramming"},{"title":"Building an advanced macro","location":"manual/metaprogramming.html#Building-an-advanced-macro","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is a simplified definition of Julia's @assert macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can be used like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In place of the written syntax, the macro call is expanded at parse time to its returned result. This is equivalent to writing:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot, while the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire expression, thus constructed, is placed into the syntax tree where the @assert macro call occurs. Then at execution time, if the test expression evaluates to true, then nothing is returned, whereas if the test is false, an error is raised indicating the asserted expression that was false. Notice that it would not be possible to write this as a function, since only the value of the condition is available and it would be impossible to display the expression that computed it in the error message.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The actual definition of @assert in Julia Base is more complicated. It allows the user to optionally specify their own error message, instead of just printing the failed expression. Just like in functions with a variable number of arguments (Varargs Functions), this is specified with an ellipses following the last argument:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now @assert has two modes of operation, depending upon the number of arguments it receives! If there's only one argument, the tuple of expressions captured by msgs will be empty and it will behave the same as the simpler definition above. But now if the user specifies a second argument, it is printed in the message body instead of the failing expression. You can inspect the result of a macro expansion with the aptly named @macroexpand macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @assert a == b\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a == b\"))\n    end)\n\njulia> @macroexpand @assert a==b \"a should equal b!\"\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a should equal b!\"))\n    end)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is yet another case that the actual @assert macro handles: what if, in addition to printing \"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation in the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work as expected with the above macro. Can you see why? Recall from string interpolation that an interpolated string is rewritten to a call to string. Compare:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So now instead of getting a plain string in msg_body, the macro is receiving a full expression that will need to be evaluated in order to display as expected. This can be spliced directly into the returned expression as an argument to the string call; see error.jl for the complete implementation.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @assert macro makes great use of splicing into quoted expressions to simplify the manipulation of expressions inside the macro body.","page":"Metaprogramming"},{"title":"Hygiene","location":"manual/metaprogramming.html#Hygiene","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An issue that arises in more complex macros is that of hygiene. In short, macros must ensure that the variables they introduce in their returned expressions do not accidentally clash with existing variables in the surrounding code they expand into. Conversely, the expressions that are passed into a macro as arguments are often expected to evaluate in the context of the surrounding code, interacting with and modifying the existing variables. Another concern arises from the fact that a macro may be called in a different module from where it was defined. In this case we need to ensure that all global variables are resolved to the correct module. Julia already has a major advantage over languages with textual macro expansion (like C) in that it only needs to consider the returned expression. All the other variables (such as msg in @assert above) follow the normal scoping block behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate these issues, let us consider writing a @time macro that takes an expression as its argument, records the time, evaluates the expression, records the time again, prints the difference between the before and after times, and then has the value of the expression as its final value. The macro might look like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    return quote\n        local t0 = time_ns()\n        local val = $ex\n        local t1 = time_ns()\n        println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n        val\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, we want t0, t1, and val to be private temporary variables, and we want time_ns to refer to the time_ns function in Julia Base, not to any time_ns variable the user might have (the same applies to println). Imagine the problems that could occur if the user expression ex also contained assignments to a variable called t0, or defined its own time_ns variable. We might get errors, or mysteriously incorrect behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's macro expander solves these problems in the following way. First, variables within a macro result are classified as either local or global. A variable is considered local if it is assigned to (and not declared global), declared local, or used as a function argument name. Otherwise, it is considered global. Local variables are then renamed to be unique (using the gensym function, which generates new symbols), and global variables are resolved within the macro definition environment. Therefore both of the above concerns are handled; the macro's locals will not conflict with any user variables, and time_ns and println will refer to the Julia Base definitions.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"One problem remains however. Consider the following use of this macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"module MyModule\nimport Base.@time\n\ntime_ns() = ... # compute something\n\n@time time_ns()\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here the user expression ex is a call to time_ns, but not the same time_ns function that the macro uses. It clearly refers to MyModule.time_ns. Therefore we must arrange for the code in ex to be resolved in the macro call environment. This is done by \"escaping\" the expression with esc:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An expression wrapped in this manner is left alone by the macro expander and simply pasted into the output verbatim. Therefore it will be resolved in the macro call environment.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce or manipulate user variables. For example, the following macro sets x to zero in the call environment:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This kind of manipulation of variables should be used judiciously, but is occasionally quite handy.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Getting the hygiene rules correct can be a formidable challenge. Before using a macro, you might want to consider whether a function closure would be sufficient. Another useful strategy is to defer as much work as possible to runtime. For example, many macros simply wrap their arguments in a QuoteNode or other similar Expr. Some examples of this include @task body which simply returns schedule(Task(() -> $body)), and @eval expr, which simply returns eval(QuoteNode(expr)).","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate, we might rewrite the @time example above as:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(expr)\n    return :(timeit(() -> $(esc(expr))))\nend\nfunction timeit(f)\n    t0 = time_ns()\n    val = f()\n    t1 = time_ns()\n    println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n    return val\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, we don't do this for a good reason: wrapping the expr in a new scope block (the anonymous function) also slightly changes the meaning of the expression (the scope of any variables in it), while we want @time to be usable with minimum impact on the wrapped code.","page":"Metaprogramming"},{"title":"Macros and dispatch","location":"manual/metaprogramming.html#Macros-and-dispatch","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros, just like Julia functions, are generic. This means they can also have multiple method definitions, thanks to multiple dispatch:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m end\n@m (macro with 0 methods)\n\njulia> macro m(args...)\n           println(\"$(length(args)) arguments\")\n       end\n@m (macro with 1 method)\n\njulia> macro m(x,y)\n           println(\"Two arguments\")\n       end\n@m (macro with 2 methods)\n\njulia> @m \"asd\"\n1 arguments\n\njulia> @m 1 2\nTwo arguments","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However one should keep in mind, that macro dispatch is based on the types of AST that are handed to the macro, not the types that the AST evaluates to at runtime:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m(::Int)\n           println(\"An Integer\")\n       end\n@m (macro with 3 methods)\n\njulia> @m 2\nAn Integer\n\njulia> x = 2\n2\n\njulia> @m x\n1 arguments","page":"Metaprogramming"},{"title":"Code Generation","location":"manual/metaprogramming.html#Code-Generation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When a significant amount of repetitive boilerplate code is required, it is common to generate it programmatically to avoid redundancy. In most languages, this requires an extra build step, and a separate program to generate the repetitive code. In Julia, expression interpolation and eval allow such code generation to take place in the normal course of program execution. For example, consider the following custom type","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"struct MyNumber\n    x::Float64\nend\n# output\n","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for which we want to add a number of methods to. We can do this programmatically in the following loop:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(quote\n        Base.$op(a::MyNumber) = MyNumber($op(a.x))\n    end)\nend\n# output\n","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and we can now use those functions with our custom type:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = MyNumber(π)\nMyNumber(3.141592653589793)\n\njulia> sin(x)\nMyNumber(1.2246467991473532e-16)\n\njulia> cos(x)\nMyNumber(-1.0)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this manner, Julia acts as its own preprocessor, and allows code generation from inside the language. The above code could be written slightly more tersely using the : prefix quoting form:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(:(Base.$op(a::MyNumber) = MyNumber($op(a.x))))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This sort of in-language code generation, however, using the eval(quote(...)) pattern, is common enough that Julia comes with a macro to abbreviate this pattern:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    @eval Base.$op(a::MyNumber) = MyNumber($op(a.x))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @eval macro rewrites this call to be precisely equivalent to the above longer versions. For longer blocks of generated code, the expression argument given to @eval can be a block:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@eval begin\n    # multiple lines\nend","page":"Metaprogramming"},{"title":"Non-Standard String Literals","location":"manual/metaprogramming.html#meta-non-standard-string-literals","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Recall from Strings that string literals prefixed by an identifier are called non-standard string literals, and can have different semantics than un-prefixed string literals. For example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"r\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128].","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Perhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead, they are custom behaviors provided by a general mechanism that anyone can use: prefixed string literals are parsed as calls to specially-named macros. For example, the regular expression macro is just the following:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro r_str(p)\n    Regex(p)\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should be passed to the @r_str macro and the result of that expansion should be placed in the syntax tree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent to placing the following object directly into the syntax tree:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Regex(\"^\\\\s*(?:#|\\$)\")","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Not only is the string literal form shorter and far more convenient, but it is also more efficient: since the regular expression is compiled and the Regex object is actually created when the code is compiled, the compilation occurs only once, rather than every time the code is executed. Consider if the regular expression occurs in a loop:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when this code is parsed, the expression is only compiled once instead of each time the loop is executed. In order to accomplish this without macros, one would have to write this loop like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"re = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Moreover, if the compiler could not determine that the regex object was constant over all loops, certain optimizations might not be possible, making this version still less efficient than the more convenient literal form above. Of course, there are still situations where the non-literal form is more convenient: if one needs to interpolate a variable into the regular expression, one must take this more verbose approach; in cases where the regular expression pattern itself is dynamic, potentially changing upon each loop iteration, a new regular expression object must be constructed on each iteration. In the vast majority of use cases, however, regular expressions are not constructed based on run-time data. In this majority of cases, the ability to write regular expressions as compile-time values is invaluable.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's non-standard literals implemented using it, but the command literal syntax (`echo \"Hello, $person\"`) is also implemented using the following innocuous-looking macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Of course, a large amount of complexity is hidden in the functions used in this macro definition, but they are just functions, written entirely in Julia. You can read their source and see precisely what they do – and all they do is construct expression objects to be inserted into your program's syntax tree.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Like string literals, command literals can also be prefixed by an identifier to form what are called non-standard command literals. These command literals are parsed as calls to specially-named macros. For example, the syntax custom`literal` is parsed as @custom_cmd \"literal\". Julia itself does not contain any non-standard command literals, but packages can make use of this syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix, non-standard command literals behave exactly like non-standard string literals.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the event that two modules provide non-standard string or command literals with the same name, it is possible to qualify the string or command literal with a module name. For instance, if both Foo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\" or Bar.x\"literal\" to disambiguate between the two.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to define a macro would be like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro foo_str(str, flag)\n    # do stuff\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can then be called with the following syntax:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"foo\"str\"flag","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The type of flag in the above mentioned syntax would be a String with contents of whatever trails after the string literal.","page":"Metaprogramming"},{"title":"Generated functions","location":"manual/metaprogramming.html#Generated-functions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"A very special macro is @generated, which allows you to define so-called generated functions. These have the capability to generate specialized code depending on the types of their arguments with more flexibility and/or less code than what can be achieved with multiple dispatch. While macros work with expressions at parse time and cannot access the types of their inputs, a generated function gets expanded at a time when the types of the arguments are known, but the function is not yet compiled.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Instead of performing some calculation or action, a generated function declaration returns a quoted expression which then forms the body for the method corresponding to the types of the arguments. When a generated function is called, the expression it returns is compiled and then run. To make this efficient, the result is usually cached. And to make this inferable, only a limited subset of the language is usable. Thus, generated functions provide a flexible way to move work from run time to compile time, at the expense of greater restrictions on allowed constructs.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When defining generated functions, there are five main differences to ordinary functions:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"You annotate the function declaration with the @generated macro. This adds some information to the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments – not their values.\nInstead of calculating something or performing some action, you return a quoted expression which, when evaluated, does what you want.\nGenerated functions are only permitted to call functions that were defined before the definition of the generated function. (Failure to follow this may result in getting MethodErrors referring to functions from a future world-age.)\nGenerated functions must not mutate or observe any non-constant global state (including, for example, IO, locks, non-local dictionaries, or using hasmethod). This means they can only read global constants, and cannot have any side effects. In other words, they must be completely pure. Due to an implementation limitation, this also means that they currently cannot define a closure or generator.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It's easiest to illustrate this with an example. We can declare a generated function foo as","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the body returns a quoted expression, namely :(x * x), rather than just the value of x * x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"From the caller's perspective, this is identical to a regular function; in fact, you don't have to know whether you're calling a regular or generated function. Let's see how foo behaves:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, we see that in the body of the generated function, x is the type of the passed argument, and the value returned by the generated function, is the result of evaluating the quoted expression we returned from the definition, now with the value of x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens if we evaluate foo again with a type that we have already used?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> foo(4)\n16","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that there is no printout of Int64. We can see that the body of the generated function was only executed once here, for the specific set of argument types, and the result was cached. After that, for this example, the expression returned from the generated function on the first invocation was re-used as the method body. However, the actual caching behavior is an implementation-defined performance optimization, so it is invalid to depend too closely on this behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The number of times a generated function is generated might be only once, but it might also be more often, or appear to not happen at all. As a consequence, you should never write a generated function with side effects - when, and how often, the side effects occur is undefined. (This is true for macros too - and just like for macros, the use of eval in a generated function is a sign that you're doing something the wrong way.) However, unlike macros, the runtime system cannot correctly handle a call to eval, so it is disallowed.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is also important to see how @generated functions interact with method redefinition. Following the principle that a correct @generated function must not observe any mutable state or cause any mutation of global state, we see the following behavior. Observe that the generated function cannot call any method that was not defined prior to the definition of the generated function itself.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Initially f(x) has one definition","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x) = \"original definition\";","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Define other operations that use f(x):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We now add some new definitions for f(x):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and compare how these results differ:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Each method of a generated function has its own view of defined functions:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The example generated function foo above did not do anything a normal function foo(x) = x * x could not do (except printing the type on the first invocation, and incurring higher overhead). However, the power of a generated function lies in its ability to compute different quoted expressions depending on the types passed to it:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(although of course this contrived example would be more easily implemented using multiple dispatch...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Abusing this will corrupt the runtime system and cause undefined behavior:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code is undefined.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Don't copy these examples!","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"These examples are hopefully helpful to illustrate how generated functions work, both in the definition end and at the call site; however, don't copy them, for the following reasons:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the foo function has side-effects (the call to Core.println), and it is undefined exactly when, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x and bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathological","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the set of operations that should not be attempted in a generated function is unbounded, and the runtime system can currently only detect a subset of the invalid operations. There are many other operations that will simply corrupt the runtime system without notification, usually in subtle ways not obviously connected to the bad definition. Because the function generator is run during inference, it must respect all of the limitations of that code.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Some operations that should not be attempted include:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Caching of native pointers.\nInteracting with the contents or methods of Core.Compiler in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting to observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic to call malloc, even though most implementations require locks internally) but don't attempt to hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition is relaxed for incrementally-loaded precompiled modules to allow calling any function in the module.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Alright, now that we have a better understanding of how generated functions work, let's use them to build some more advanced (and valid) functionality...","page":"Metaprogramming"},{"title":"An advanced example","location":"manual/metaprogramming.html#An-advanced-example","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's base library has an internal sub2ind function to calculate a linear index into an n-dimensional array, based on a set of n multilinear indices - in other words, to calculate the index i that can be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation is the following:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end\nsub2ind_loop (generic function with 1 method)\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The same thing can be done using recursion:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Both these implementations, although different, do essentially the same thing: a runtime loop over the dimensions of the array, collecting the offset in each dimension into the final index.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, all the information we need for the loop is embedded in the type information of the arguments. This allows the compiler to move the iteration to compile time and eliminate the runtime loops altogether. We can utilize generated functions to achieve a similar effect; in compiler parlance, we use generated functions to manually unroll the loop. The body becomes almost identical, but instead of calculating the linear index, we build up an expression that calculates the index:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What code will this generate?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An easy way to find out is to extract the body into another (regular) function:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen_impl (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can now execute sub2ind_gen_impl and examine the expression it returns:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, the method body that will be used here doesn't include a loop at all - just indexing into the two tuples, multiplication and addition/subtraction. All the looping is performed compile-time, and we avoid looping during execution entirely. Thus, we only loop once per type, in this case once per N (except in edge cases where the function is generated more than once - see disclaimer above).","page":"Metaprogramming"},{"title":"Optionally-generated functions","location":"manual/metaprogramming.html#Optionally-generated-functions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Generated functions can achieve high efficiency at run time, but come with a compile time cost: a new function body must be generated for every combination of concrete argument types. Typically, Julia is able to compile \"generic\" versions of functions that will work for any arguments, but with generated functions this is impossible. This means that programs making heavy use of generated functions might be impossible to statically compile.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To solve this problem, the language provides syntax for writing normal, non-generated alternative implementations of generated functions. Applied to the sub2ind example above, it would look like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n    if N != length(I)\n        throw(ArgumentError(\"Number of dimensions must match number of indices.\"))\n    end\n    if @generated\n        ex = :(I[$N] - 1)\n        for i = (N - 1):-1:1\n            ex = :(I[$i] - 1 + dims[$i] * $ex)\n        end\n        return :($ex + 1)\n    else\n        ind = I[N] - 1\n        for i = (N - 1):-1:1\n            ind = I[i] - 1 + dims[i]*ind\n        end\n        return ind + 1\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Internally, this code creates two implementations of the function: a generated one where the first block in if @generated is used, and a normal one where the else block is used. Inside the then part of the if @generated block, code has the same semantics as other generated functions: argument names refer to types, and the code should return an expression. Multiple if @generated blocks may occur, in which case the generated implementation uses all of the then blocks and the alternate implementation uses all of the else blocks.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that we added an error check to the top of the function. This code will be common to both versions, and is run-time code in both versions (it will be quoted and returned as an expression from the generated version). That means that the values and types of local variables are not available at code generation time –- the code-generation code can only see the types of arguments.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this style of definition, the code generation feature is essentially an optional optimization. The compiler will use it if convenient, but otherwise may choose to use the normal implementation instead. This style is preferred, since it allows the compiler to make more decisions and compile programs in more ways, and since normal code is more readable than code-generating code. However, which implementation is used depends on compiler implementation details, so it is essential for the two implementations to behave identically.","page":"Metaprogramming"},{"title":"NetworkOptions","location":"stdlib/NetworkOptions.html#NetworkOptions","category":"section","text":"","page":"NetworkOptions"},{"title":"NetworkOptions","location":"stdlib/NetworkOptions.html","category":"page","text":"NetworkOptions.ca_roots\nNetworkOptions.ca_roots_path\nNetworkOptions.ssh_dir\nNetworkOptions.ssh_key_pass\nNetworkOptions.ssh_key_name\nNetworkOptions.ssh_key_path\nNetworkOptions.ssh_pub_key_path\nNetworkOptions.ssh_known_hosts_files\nNetworkOptions.ssh_known_hosts_file\nNetworkOptions.verify_host","page":"NetworkOptions"},{"title":"NetworkOptions.ca_roots","location":"stdlib/NetworkOptions.html#NetworkOptions.ca_roots","category":"function","text":"ca_roots() :: Union{Nothing, String}\n\nThe ca_roots() function tells the caller where, if anywhere, to find a file or directory of PEM-encoded certificate authority roots. By default, on systems like Windows and macOS where the built-in TLS engines know how to verify hosts using the system's built-in certificate verification mechanism, this function will return nothing. On classic UNIX systems (excluding macOS), root certificates are typically stored in a file in /etc: the common places for the current UNIX system will be searched and if one of these paths exists, it will be returned; if none of these typical root certificate paths exist, then the path to the set of root certificates that are bundled with Julia is returned.\n\nThe default value returned by ca_roots() may be overridden by setting the JULIA_SSL_CA_ROOTS_PATH, SSL_CERT_DIR, or SSL_CERT_FILE environment variables, in which case this function will always return the value of the first of these variables that is set (whether the path exists or not). If JULIA_SSL_CA_ROOTS_PATH is set to the empty string, then the other variables are ignored (as if unset); if the other variables are set to the empty string, they behave is if they are not set.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ca_roots_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ca_roots_path","category":"function","text":"ca_roots_path() :: String\n\nThe ca_roots_path() function is similar to the ca_roots() function except that it always returns a path to a file or directory of PEM-encoded certificate authority roots. When called on a system like Windows or macOS, where system root certificates are not stored in the file system, it will currently return the path to the set of root certificates that are bundled with Julia. (In the future, this function may instead extract the root certificates from the system and save them to a file whose path would be returned.)\n\nIf it is possible to configure a library that uses TLS to use the system certificates that is generally preferable: i.e. it is better to use ca_roots() which returns nothing to indicate that the system certs should be used. The ca_roots_path() function should only be used when configuring libraries which require a path to a file or directory for root certificates.\n\nThe default value returned by ca_roots_path() may be overridden by setting the JULIA_SSL_CA_ROOTS_PATH, SSL_CERT_DIR, or SSL_CERT_FILE environment variables, in which case this function will always return the value of the first of these variables that is set (whether the path exists or not). If JULIA_SSL_CA_ROOTS_PATH is set to the empty string, then the other variables are ignored (as if unset); if the other variables are set to the empty string, they behave is if they are not set.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_dir","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_dir","category":"function","text":"ssh_dir() :: String\n\nThe ssh_dir() function returns the location of the directory where the ssh program keeps/looks for configuration files. By default this is ~/.ssh but this can be overridden by setting the environment variable SSH_DIR.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_key_pass","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_pass","category":"function","text":"ssh_key_pass() :: String\n\nThe ssh_key_pass() function returns the value of the environment variable SSH_KEY_PASS if it is set or nothing if it is not set. In the future, this may be able to find a password by other means, such as secure system storage, so packages that need a password to decrypt an SSH private key should use this API instead of directly checking the environment variable so that they gain such capabilities automatically when they are added.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_key_name","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_name","category":"function","text":"ssh_key_name() :: String\n\nThe ssh_key_name() function returns the base name of key files that SSH should use for when establishing a connection. There is usually no reason that this function should be called directly and libraries should generally use the ssh_key_path and ssh_pub_key_path functions to get full paths. If the environment variable SSH_KEY_NAME is set then this function returns that; otherwise it returns id_rsa by default.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_key_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_path","category":"function","text":"ssh_key_path() :: String\n\nThe ssh_key_path() function returns the path of the SSH private key file that should be used for SSH connections. If the SSH_KEY_PATH environment variable is set then it will return that value. Otherwise it defaults to returning\n\njoinpath(ssh_dir(), ssh_key_name())\n\nThis default value in turn depends on the SSH_DIR and SSH_KEY_NAME environment variables.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_pub_key_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_pub_key_path","category":"function","text":"ssh_pub_key_path() :: String\n\nThe ssh_pub_key_path() function returns the path of the SSH public key file that should be used for SSH connections. If the SSH_PUB_KEY_PATH environment variable is set then it will return that value. If that isn't set but SSH_KEY_PATH is set, it will return that path with the .pub suffix appended. If neither is set, it defaults to returning\n\njoinpath(ssh_dir(), ssh_key_name() * \".pub\")\n\nThis default value in turn depends on the SSH_DIR and SSH_KEY_NAME environment variables.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_known_hosts_files","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_known_hosts_files","category":"function","text":"ssh_known_hosts_files() :: Vector{String}\n\nThe ssh_known_hosts_files() function returns a vector of paths of SSH known hosts files that should be used when establishing the identities of remote servers for SSH connections. By default this function returns\n\n[joinpath(ssh_dir(), \"known_hosts\"), bundled_known_hosts]\n\nwhere bundled_known_hosts is the path of a copy of a known hosts file that is bundled with this package (containing known hosts keys for github.com and gitlab.com). If the environment variable SSH_KNOWN_HOSTS_FILES is set, however, then its value is split into paths on the : character (or on ; on Windows) and this vector of paths is returned instead. If any component of this vector is empty, it is expanded to the default known hosts paths.\n\nPackages that use ssh_known_hosts_files() should ideally look for matching entries by comparing the host name and key types, considering the first entry in any of the files which matches to be the definitive identity of the host. If the caller cannot compare the key type (e.g. because it has been hashes) then it must approximate the above algorithm by looking for all matching entries for a host in each file: if a file has any entries for a host then one of them must match; the caller should only continue to search further known hosts files if there are no entries for the host in question in an earlier file.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.ssh_known_hosts_file","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_known_hosts_file","category":"function","text":"ssh_known_hosts_file() :: String\n\nThe ssh_known_hosts_file() function returns a single path of an SSH known hosts file that should be used when establishing the identities of remote servers for SSH connections. It returns the first path returned by ssh_known_hosts_files that actually exists. Callers who can look in more than one known hosts file should use ssh_known_hosts_files instead and look for host matches in all the files returned as described in that function's docs.\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"NetworkOptions.verify_host","location":"stdlib/NetworkOptions.html#NetworkOptions.verify_host","category":"function","text":"verify_host(url::AbstractString, [transport::AbstractString]) :: Bool\n\nThe verify_host function tells the caller whether the identity of a host should be verified when communicating over secure transports like TLS or SSH. The url argument may be:\n\na proper URL staring with proto://\nan ssh-style bare host name or host name prefixed with user@\nan scp-style host as above, followed by : and a path location\n\nIn each case the host name part is parsed out and the decision about whether to verify or not is made based solely on the host name, not anything else about the input URL. In particular, the protocol of the URL does not matter (more below).\n\nThe transport argument indicates the kind of transport that the query is about. The currently known values are SSL/ssl (alias TLS/tls) and SSH/ssh. If the transport is omitted, the query will return true only if the host name should not be verified regardless of transport.\n\nThe host name is matched against the host patterns in the relevant environment variables depending on whether transport is supplied and what its value is:\n\nJULIA_NO_VERIFY_HOSTS — hosts that should not be verified for any transport\nJULIA_SSL_NO_VERIFY_HOSTS — hosts that should not be verified for SSL/TLS\nJULIA_SSH_NO_VERIFY_HOSTS — hosts that should not be verified for SSH\nJULIA_ALWAYS_VERIFY_HOSTS — hosts that should always be verified\n\nThe values of each of these variables is a comma-separated list of host name patterns with the following syntax — each pattern is split on . into parts and each part must one of:\n\nA literal domain name component consisting of one or more ASCII letter, digit, hyphen or underscore (technically not part of a legal host name, but sometimes used). A literal domain name component matches only itself.\nA **, which matches zero or more domain name components.\nA *, which match any one domain name component.\n\nWhen matching a host name against a pattern list in one of these variables, the host name is split on . into components and that sequence of words is matched against the pattern: a literal pattern matches exactly one host name component with that value; a * pattern matches exactly one host name component with any value; a ** pattern matches any number of host name components. For example:\n\n** matches any host name\n**.org matches any host name in the .org top-level domain\nexample.com matches only the exact host name example.com\n*.example.com matches api.example.com but not example.com or v1.api.example.com\n**.example.com matches any domain under example.com, including example.com itself, api.example.com and v1.api.example.com\n\n\n\n\n\n","page":"NetworkOptions"},{"title":"Serialization","location":"stdlib/Serialization.html#Serialization","category":"section","text":"","page":"Serialization"},{"title":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"Provides serialization of Julia objects.","page":"Serialization"},{"title":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"Serialization.serialize\nSerialization.deserialize\nSerialization.writeheader","page":"Serialization"},{"title":"Serialization.serialize","location":"stdlib/Serialization.html#Serialization.serialize","category":"function","text":"serialize(stream::IO, value)\n\nWrite an arbitrary value to a stream in an opaque format, such that it can be read back by deserialize. The read-back value will be as identical as possible to the original, but note that Ptr values are serialized as all-zero bit patterns (NULL).\n\nAn 8-byte identifying header is written to the stream first. To avoid writing the header, construct a Serializer and use it as the first argument to serialize instead. See also Serialization.writeheader.\n\nThe data format can change in minor (1.x) Julia releases, but files written by prior 1.x versions will remain readable. The main exception to this is when the definition of a type in an external package changes. If that occurs, it may be necessary to specify an explicit compatible version of the affected package in your environment. Renaming functions, even private functions, inside packages can also put existing files out of sync. Anonymous functions require special care: because their names are automatically generated, minor code changes can cause them to be renamed. Serializing anonymous functions should be avoided in files intended for long-term storage.\n\nIn some cases, the word size (32- or 64-bit) of the reading and writing machines must match. In rarer cases the OS or architecture must also match, for example when using packages that contain platform-dependent code.\n\n\n\n\n\nserialize(filename::AbstractString, value)\n\nOpen a file and serialize the given value to it.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n","page":"Serialization"},{"title":"Serialization.deserialize","location":"stdlib/Serialization.html#Serialization.deserialize","category":"function","text":"deserialize(stream)\n\nRead a value written by serialize. deserialize assumes the binary data read from stream is correct and has been serialized by a compatible implementation of serialize. deserialize is designed for simplicity and performance, and so does not validate the data read. Malformed data can result in process termination. The caller must ensure the integrity and correctness of data read from stream.\n\n\n\n\n\ndeserialize(filename::AbstractString)\n\nOpen a file and deserialize its contents.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n","page":"Serialization"},{"title":"Serialization.writeheader","location":"stdlib/Serialization.html#Serialization.writeheader","category":"function","text":"Serialization.writeheader(s::AbstractSerializer)\n\nWrite an identifying header to the specified serializer. The header consists of 8 bytes as follows:\n\nOffset Description\n0 tag byte (0x37)\n1-2 signature bytes \"JL\"\n3 protocol version\n4 bits 0-1: endianness: 0 = little, 1 = big\n4 bits 2-3: platform: 0 = 32-bit, 1 = 64-bit\n5-7 reserved\n\n\n\n\n\n","page":"Serialization"},{"title":"File Events","location":"stdlib/FileWatching.html#lib-filewatching","category":"section","text":"","page":"File Events"},{"title":"File Events","location":"stdlib/FileWatching.html","category":"page","text":"FileWatching.poll_fd\nFileWatching.poll_file\nFileWatching.watch_file\nFileWatching.watch_folder\nFileWatching.unwatch_folder","page":"File Events"},{"title":"FileWatching.poll_fd","location":"stdlib/FileWatching.html#FileWatching.poll_fd","category":"function","text":"poll_fd(fd, timeout_s::Real=-1; readable=false, writable=false)\n\nMonitor a file descriptor fd for changes in the read or write availability, and with a timeout given by timeout_s seconds.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.poll_file","location":"stdlib/FileWatching.html#FileWatching.poll_file","category":"function","text":"poll_file(path::AbstractString, interval_s::Real=5.007, timeout_s::Real=-1) -> (previous::StatStruct, current)\n\nMonitor a file for changes by polling every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds.\n\nReturns a pair of status objects (previous, current) when a change is detected. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (indicating the timeout elapsed), or some other Exception subtype (if the stat operation failed - for example, if the path does not exist).\n\nTo determine when a file was modified, compare current isa StatStruct && mtime(prev) != mtime(current) to detect notification of changes. However, using watch_file for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.watch_file","location":"stdlib/FileWatching.html#FileWatching.watch_file","category":"function","text":"watch_file(path::AbstractString, timeout_s::Real=-1)\n\nWatch file or directory path for changes until a change occurs or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThe returned value is an object with boolean fields renamed, changed, and timedout, giving the result of watching the file.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.watch_folder","location":"stdlib/FileWatching.html#FileWatching.watch_folder","category":"function","text":"watch_folder(path::AbstractString, timeout_s::Real=-1)\n\nWatches a file or directory path for changes until a change has occurred or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis will continuing tracking changes for path in the background until unwatch_folder is called on the same path.\n\nThe returned value is an pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed, changed, and timedout, giving the event.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.unwatch_folder","location":"stdlib/FileWatching.html#FileWatching.unwatch_folder","category":"function","text":"unwatch_folder(path::AbstractString)\n\nStop background tracking of changes for path. It is not recommended to do this while another task is waiting for watch_folder to return on the same path, as the result may be unpredictable.\n\n\n\n\n\n","page":"File Events"},{"title":"Pidfile","location":"stdlib/FileWatching.html#Pidfile","category":"section","text":"","page":"File Events"},{"title":"File Events","location":"stdlib/FileWatching.html","category":"page","text":"CurrentModule = FileWatching.Pidfile","page":"File Events"},{"title":"File Events","location":"stdlib/FileWatching.html","category":"page","text":"A simple utility tool for creating advisory pidfiles (lock files).","page":"File Events"},{"title":"Primary Functions","location":"stdlib/FileWatching.html#Primary-Functions","category":"section","text":"","page":"File Events"},{"title":"File Events","location":"stdlib/FileWatching.html","category":"page","text":"mkpidlock\nclose(lock::LockMonitor)","page":"File Events"},{"title":"FileWatching.Pidfile.mkpidlock","location":"stdlib/FileWatching.html#FileWatching.Pidfile.mkpidlock","category":"function","text":"mkpidlock([f::Function], at::String, [pid::Cint, proc::Process]; kwopts...)\n\nCreate a pidfile lock for the path \"at\" for the current process or the process identified by pid or proc. Can take a function to execute once locked, for usage in do blocks, after which the lock will be automatically closed. If the lock fails and wait is false, then an error is thrown.\n\nThe lock will be released by either close, a finalizer, or shortly after proc exits. Make sure the return value is live through the end of the critical section of your program, so the finalizer does not reclaim it early.\n\nOptional keyword arguments:\n\nmode: file access mode (modified by the process umask). Defaults to world-readable.\npoll_interval: Specify the maximum time to between attempts (if watch_file doesn't work)\nstale_age: Delete an existing pidfile (ignoring the lock) if its mtime is older than this.   The file won't be deleted until 25x longer than this if the pid in the file appears that it may be valid.   By default this is disabled (stale_age = 0), but a typical recommended value would be about 3-5x an   estimated normal completion time.\nrefresh: Keeps a lock from becoming stale by updating the mtime every interval of time that passes.   By default, this is set to stale_age/2, which is the recommended value.\nwait: If true, block until we get the lock, if false, raise error if lock fails.\n\n\n\n\n\n","page":"File Events"},{"title":"Base.close","location":"stdlib/FileWatching.html#Base.close-Tuple{FileWatching.Pidfile.LockMonitor}","category":"method","text":"close(lock::LockMonitor)\n\nRelease a pidfile lock.\n\n\n\n\n\n","page":"File Events"},{"title":"Helper Functions","location":"stdlib/FileWatching.html#Helper-Functions","category":"section","text":"","page":"File Events"},{"title":"File Events","location":"stdlib/FileWatching.html","category":"page","text":"Pidfile.open_exclusive\nPidfile.tryopen_exclusive\nPidfile.write_pidfile\nPidfile.parse_pidfile\nPidfile.stale_pidfile\nPidfile.isvalidpid\nBase.touch(::Pidfile.LockMonitor)","page":"File Events"},{"title":"FileWatching.Pidfile.open_exclusive","location":"stdlib/FileWatching.html#FileWatching.Pidfile.open_exclusive","category":"function","text":"open_exclusive(path::String; mode, poll_interval, stale_age) :: File\n\nCreate a new a file for read-write advisory-exclusive access. If wait is false then error out if the lock files exist otherwise block until we get the lock.\n\nFor a description of the keyword arguments, see mkpidlock.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.Pidfile.tryopen_exclusive","location":"stdlib/FileWatching.html#FileWatching.Pidfile.tryopen_exclusive","category":"function","text":"tryopen_exclusive(path::String, mode::Integer = 0o444) :: Union{Void, File}\n\nTry to create a new file for read-write advisory-exclusive access, return nothing if it already exists.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.Pidfile.write_pidfile","location":"stdlib/FileWatching.html#FileWatching.Pidfile.write_pidfile","category":"function","text":"write_pidfile(io, pid)\n\nWrite our pidfile format to an open IO descriptor.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.Pidfile.parse_pidfile","location":"stdlib/FileWatching.html#FileWatching.Pidfile.parse_pidfile","category":"function","text":"parse_pidfile(file::Union{IO, String}) => (pid, hostname, age)\n\nAttempt to parse our pidfile format, replaced an element with (0, \"\", 0.0), respectively, for any read that failed.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.Pidfile.stale_pidfile","location":"stdlib/FileWatching.html#FileWatching.Pidfile.stale_pidfile","category":"function","text":"stale_pidfile(path::String, stale_age::Real) :: Bool\n\nHelper function for open_exclusive for deciding if a pidfile is stale.\n\n\n\n\n\n","page":"File Events"},{"title":"FileWatching.Pidfile.isvalidpid","location":"stdlib/FileWatching.html#FileWatching.Pidfile.isvalidpid","category":"function","text":"isvalidpid(hostname::String, pid::Cuint) :: Bool\n\nAttempt to conservatively estimate whether pid is a valid process id.\n\n\n\n\n\n","page":"File Events"},{"title":"Base.Filesystem.touch","location":"stdlib/FileWatching.html#Base.Filesystem.touch-Tuple{FileWatching.Pidfile.LockMonitor}","category":"method","text":"Base.touch(::Pidfile.LockMonitor)\n\nUpdate the mtime on the lock, to indicate it is still fresh.\n\nSee also the refresh keyword in the mkpidlock constructor.\n\n\n\n\n\n","page":"File Events"},{"title":"Iteration utilities","location":"base/iterators.html#Iteration-utilities","category":"section","text":"","page":"Iteration utilities"},{"title":"Iteration utilities","location":"base/iterators.html","category":"page","text":"Base.Iterators.Stateful\nBase.Iterators.zip\nBase.Iterators.enumerate\nBase.Iterators.rest\nBase.Iterators.countfrom\nBase.Iterators.take\nBase.Iterators.takewhile\nBase.Iterators.drop\nBase.Iterators.dropwhile\nBase.Iterators.cycle\nBase.Iterators.repeated\nBase.Iterators.product\nBase.Iterators.flatten\nBase.Iterators.flatmap\nBase.Iterators.partition\nBase.Iterators.map\nBase.Iterators.filter\nBase.Iterators.accumulate\nBase.Iterators.reverse\nBase.Iterators.only\nBase.Iterators.peel","page":"Iteration utilities"},{"title":"Base.Iterators.Stateful","location":"base/iterators.html#Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.zip","location":"base/iterators.html#Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.enumerate","location":"base/iterators.html#Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a different element. This will happen if iter has indices that do not start at 1, and may happen for strings, dictionaries, etc. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.rest","location":"base/iterators.html#Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state.\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> collect(Iterators.rest([1,2,3,4], 2))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.countfrom","location":"base/iterators.html#Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.take","location":"base/iterators.html#Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.takewhile","location":"base/iterators.html#Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.drop","location":"base/iterators.html#Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.dropwhile","location":"base/iterators.html#Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.cycle","location":"base/iterators.html#Base.Iterators.cycle","category":"function","text":"cycle(iter)\n\nAn iterator that cycles through iter forever. If iter is empty, so is cycle(iter).\n\nSee also: Iterators.repeated, Base.repeat.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.repeated","location":"base/iterators.html#Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nSee also: Iterators.cycle, Base.repeat.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.product","location":"base/iterators.html#Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.flatten","location":"base/iterators.html#Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.flatmap","location":"base/iterators.html#Base.Iterators.flatmap","category":"function","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.partition","location":"base/iterators.html#Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.map","location":"base/iterators.html#Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.filter","location":"base/iterators.html#Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.accumulate","location":"base/iterators.html#Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.reverse","location":"base/iterators.html#Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order. This iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns an Iterators.Reverse object wrapping itr, which is iterable if the corresponding iterate methods are defined, but some itr types may implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr; more generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.only","location":"base/iterators.html#Base.Iterators.only","category":"function","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the collection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.peel","location":"base/iterators.html#Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Bounds checking","location":"devdocs/boundscheck.html#Bounds-checking","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection.","page":"Bounds checking"},{"title":"Eliding bounds checks","location":"devdocs/boundscheck.html#Eliding-bounds-checks","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"function sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"With a custom array-like type MyArray having:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A,i); A.data[to_index(i)])","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Then when getindex is inlined into sum, the call to checkbounds(A,i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack.","page":"Bounds checking"},{"title":"Caution!","location":"devdocs/boundscheck.html#Caution!","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"It is easy to accidentally expose unsafe operations with @inbounds. You might be tempted to write the above example as","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"function sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Which quietly assumes 1-based indexing and therefore exposes unsafe memory access when used with OffsetArrays:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"julia> using OffsetArrays\n\njulia> sum(OffsetArray([1,2,3], -10))\n9164911648 # inconsistent results or segfault","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"While the original source of the error here is 1:length(A), the use of @inbounds increases the consequences from a bounds error to a less easily caught and debugged unsafe memory access. It is often difficult or impossible to prove that a method which uses @inbounds is safe, so one must weigh the benefits of performance improvements against the risk of segfaults and silent misbehavior, especially in public facing APIs.","page":"Bounds checking"},{"title":"Propagating inbounds","location":"devdocs/boundscheck.html#Propagating-inbounds","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"To override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining.","page":"Bounds checking"},{"title":"The bounds checking call hierarchy","location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"The overall hierarchy is:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Here A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"so checkindex checks a single dimension.  All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"If you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"If you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array.  If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Note this hierarchy has been designed to reduce the likelihood of method ambiguities.  We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument).","page":"Bounds checking"},{"title":"Emit bounds checks","location":"devdocs/boundscheck.html#Emit-bounds-checks","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations.","page":"Bounds checking"},{"title":"Interfaces","location":"manual/interfaces.html#Interfaces","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A lot of the power and extensibility in Julia comes from a collection of informal interfaces.  By extending a few specific methods to work for a custom type, objects of that type not only receive those functionalities, but they are also able to be used in other methods that are written to generically build upon those behaviors.","page":"Interfaces"},{"title":"Iteration","location":"manual/interfaces.html#man-interface-iteration","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Required methods  Brief description\niterate(iter)  Returns either a tuple of the first item and initial state or nothing if empty\niterate(iter, state)  Returns either a tuple of the next item and next state or nothing if no items remain\nImportant optional methods Default definition Brief description\nBase.IteratorSize(IterType) Base.HasLength() One of Base.HasLength(), Base.HasShape{N}(), Base.IsInfinite(), or Base.SizeUnknown() as appropriate\nBase.IteratorEltype(IterType) Base.HasEltype() Either Base.EltypeUnknown() or Base.HasEltype() as appropriate\neltype(IterType) Any The type of the first entry of the tuple returned by iterate()\nlength(iter) (undefined) The number of items, if known\nsize(iter, [dim]) (undefined) The number of items in each dimension, if known\nBase.isdone(iter[, state]) missing Fast-path hint for iterator completion. Should be defined for stateful iterators, or else isempty(iter) may call iterate(iter[, state]) and mutate the iterator.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Value returned by IteratorSize(IterType) Required Methods\nBase.HasLength() length(iter)\nBase.HasShape{N}() length(iter)  and size(iter, [dim])\nBase.IsInfinite() (none)\nBase.SizeUnknown() (none)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Value returned by IteratorEltype(IterType) Required Methods\nBase.HasEltype() eltype(IterType)\nBase.EltypeUnknown() (none)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Sequential iteration is implemented by the iterate function. Instead of mutating objects as they are iterated over, Julia iterators may keep track of the iteration state externally from the object. The return value from iterate is always either a tuple of a value and a state, or nothing if no elements remain. The state object will be passed back to the iterate function on the next iteration and is generally considered an implementation detail private to the iterable object.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Any object that defines this function is iterable and can be used in the many functions that rely upon iteration. It can also be used directly in a for loop since the syntax:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"for item in iter   # or  \"for item = iter\"\n    # body\nend","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"is translated into:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"next = iterate(iter)\nwhile next !== nothing\n    (item, state) = next\n    # body\n    next = iterate(iter, state)\nend","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A simple example is an iterable sequence of square numbers with a defined length:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct Squares\n           count::Int\n       end\n\njulia> Base.iterate(S::Squares, state=1) = state > S.count ? nothing : (state*state, state+1)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"With only iterate definition, the Squares type is already pretty powerful. We can iterate over all the elements:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> for item in Squares(7)\n           println(item)\n       end\n1\n4\n9\n16\n25\n36\n49","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"We can use many of the builtin methods that work with iterables, like in or sum:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> 25 in Squares(10)\ntrue\n\njulia> sum(Squares(100))\n338350","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"There are a few more methods we can extend to give Julia more information about this iterable collection.  We know that the elements in a Squares sequence will always be Int. By extending the eltype method, we can give that information to Julia and help it make more specialized code in the more complicated methods. We also know the number of elements in our sequence, so we can extend length, too:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.eltype(::Type{Squares}) = Int # Note that this is defined for the type\n\njulia> Base.length(S::Squares) = S.count","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Now, when we ask Julia to collect all the elements into an array it can preallocate a Vector{Int} of the right size instead of naively push!ing each element into a Vector{Any}:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> collect(Squares(4))\n4-element Vector{Int64}:\n  1\n  4\n  9\n 16","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"While we can rely upon generic implementations, we can also extend specific methods where we know there is a simpler algorithm. For example, there's a formula to compute the sum of squares, so we can override the generic iterative version with a more performant solution:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.sum(S::Squares) = (n = S.count; return n*(n+1)*(2n+1)÷6)\n\njulia> sum(Squares(1803))\n1955361914","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This is a very common pattern throughout Julia Base: a small set of required methods define an informal interface that enable many fancier behaviors. In some cases, types will want to additionally specialize those extra behaviors when they know a more efficient algorithm can be used in their specific case.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"It is also often useful to allow iteration over a collection in reverse order by iterating over Iterators.reverse(iterator).  To actually support reverse-order iteration, however, an iterator type T needs to implement iterate for Iterators.Reverse{T}. (Given r::Iterators.Reverse{T}, the underling iterator of type T is r.itr.) In our Squares example, we would implement Iterators.Reverse{Squares} methods:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.iterate(rS::Iterators.Reverse{Squares}, state=rS.itr.count) = state < 1 ? nothing : (state*state, state-1)\n\njulia> collect(Iterators.reverse(Squares(4)))\n4-element Vector{Int64}:\n 16\n  9\n  4\n  1","page":"Interfaces"},{"title":"Indexing","location":"manual/interfaces.html#Indexing","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement Brief description\ngetindex(X, i) X[i], indexed element access\nsetindex!(X, v, i) X[i] = v, indexed assignment\nfirstindex(X) The first index, used in X[begin]\nlastindex(X) The last index, used in X[end]","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For the Squares iterable above, we can easily compute the ith element of the sequence by squaring it.  We can expose this as an indexing expression S[i]. To opt into this behavior, Squares simply needs to define getindex:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> function Base.getindex(S::Squares, i::Int)\n           1 <= i <= S.count || throw(BoundsError(S, i))\n           return i*i\n       end\n\njulia> Squares(100)[23]\n529","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Additionally, to support the syntax S[begin] and S[end], we must define firstindex and lastindex to specify the first and last valid indices, respectively:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.firstindex(S::Squares) = 1\n\njulia> Base.lastindex(S::Squares) = length(S)\n\njulia> Squares(23)[end]\n529","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For multi-dimensional begin/end indexing as in a[3, begin, 7], for example, you should define firstindex(a, dim) and lastindex(a, dim) (which default to calling first and last on axes(a, dim), respectively).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Note, though, that the above only defines getindex with one integer index. Indexing with anything other than an Int will throw a MethodError saying that there was no matching method. In order to support indexing with ranges or vectors of Ints, separate methods must be written:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.getindex(S::Squares, i::Number) = S[convert(Int, i)]\n\njulia> Base.getindex(S::Squares, I) = [S[i] for i in I]\n\njulia> Squares(10)[[3,4.,5]]\n3-element Vector{Int64}:\n  9\n 16\n 25","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"While this is starting to support more of the indexing operations supported by some of the builtin types, there's still quite a number of behaviors missing. This Squares sequence is starting to look more and more like a vector as we've added behaviors to it. Instead of defining all these behaviors ourselves, we can officially define it as a subtype of an AbstractArray.","page":"Interfaces"},{"title":"Abstract Arrays","location":"manual/interfaces.html#man-interface-array","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement  Brief description\nsize(A)  Returns a tuple containing the dimensions of A\ngetindex(A, i::Int)  (if IndexLinear) Linear scalar indexing\ngetindex(A, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexing\nOptional methods Default definition Brief description\nIndexStyle(::Type) IndexCartesian() Returns either IndexLinear() or IndexCartesian(). See the description below.\nsetindex!(A, v, i::Int)  (if IndexLinear) Scalar indexed assignment\nsetindex!(A, v, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexed assignment\ngetindex(A, I...) defined in terms of scalar getindex Multidimensional and nonscalar indexing\nsetindex!(A, X, I...) defined in terms of scalar setindex! Multidimensional and nonscalar indexed assignment\niterate defined in terms of scalar getindex Iteration\nlength(A) prod(size(A)) Number of elements\nsimilar(A) similar(A, eltype(A), size(A)) Return a mutable array with the same shape and element type\nsimilar(A, ::Type{S}) similar(A, S, size(A)) Return a mutable array with the same shape and the specified element type\nsimilar(A, dims::Dims) similar(A, eltype(A), dims) Return a mutable array with the same element type and size dims\nsimilar(A, ::Type{S}, dims::Dims) Array{S}(undef, dims) Return a mutable array with the specified element type and size\nNon-traditional indices Default definition Brief description\naxes(A) map(OneTo, size(A)) Return a tuple of AbstractUnitRange{<:Integer} of valid indices\nsimilar(A, ::Type{S}, inds) similar(A, S, Base.to_shape(inds)) Return a mutable array with the specified indices inds (see below)\nsimilar(T::Union{Type,Function}, inds) T(Base.to_shape(inds)) Return an array similar to T with the specified indices inds (see below)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If a type is defined as a subtype of AbstractArray, it inherits a very large set of rich behaviors including iteration and multidimensional indexing built on top of single-element access.  See the arrays manual page and the Julia Base section for more supported methods.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A key part in defining an AbstractArray subtype is IndexStyle. Since indexing is such an important part of an array and often occurs in hot loops, it's important to make both indexing and indexed assignment as efficient as possible.  Array data structures are typically defined in one of two ways: either it most efficiently accesses its elements using just one index (linear indexing) or it intrinsically accesses the elements with indices specified for every dimension.  These two modalities are identified by Julia as IndexLinear() and IndexCartesian().  Converting a linear index to multiple indexing subscripts is typically very expensive, so this provides a traits-based mechanism to enable efficient generic code for all array types.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This distinction determines which scalar indexing methods the type must define. IndexLinear() arrays are simple: just define getindex(A::ArrayType, i::Int).  When the array is subsequently indexed with a multidimensional set of indices, the fallback getindex(A::AbstractArray, I...) efficiently converts the indices into one linear index and then calls the above method. IndexCartesian() arrays, on the other hand, require methods to be defined for each supported dimensionality with ndims(A) Int indices. For example, SparseMatrixCSC from the SparseArrays standard library module, only supports two dimensions, so it just defines getindex(A::SparseMatrixCSC, i::Int, j::Int). The same holds for setindex!.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Returning to the sequence of squares from above, we could instead define it as a subtype of an AbstractArray{Int, 1}:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct SquaresVector <: AbstractArray{Int, 1}\n           count::Int\n       end\n\njulia> Base.size(S::SquaresVector) = (S.count,)\n\njulia> Base.IndexStyle(::Type{<:SquaresVector}) = IndexLinear()\n\njulia> Base.getindex(S::SquaresVector, i::Int) = i*i","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Note that it's very important to specify the two parameters of the AbstractArray; the first defines the eltype, and the second defines the ndims. That supertype and those three methods are all it takes for SquaresVector to be an iterable, indexable, and completely functional array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> s = SquaresVector(4)\n4-element SquaresVector:\n  1\n  4\n  9\n 16\n\njulia> s[s .> 8]\n2-element Vector{Int64}:\n  9\n 16\n\njulia> s + s\n4-element Vector{Int64}:\n  2\n  8\n 18\n 32\n\njulia> sin.(s)\n4-element Vector{Float64}:\n  0.8414709848078965\n -0.7568024953079282\n  0.4121184852417566\n -0.2879033166650653","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"As a more complicated example, let's define our own toy N-dimensional sparse-like array type built on top of Dict:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct SparseArray{T,N} <: AbstractArray{T,N}\n           data::Dict{NTuple{N,Int}, T}\n           dims::NTuple{N,Int}\n       end\n\njulia> SparseArray(::Type{T}, dims::Int...) where {T} = SparseArray(T, dims);\n\njulia> SparseArray(::Type{T}, dims::NTuple{N,Int}) where {T,N} = SparseArray{T,N}(Dict{NTuple{N,Int}, T}(), dims);\n\njulia> Base.size(A::SparseArray) = A.dims\n\njulia> Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where {T} = SparseArray(T, dims)\n\njulia> Base.getindex(A::SparseArray{T,N}, I::Vararg{Int,N}) where {T,N} = get(A.data, I, zero(T))\n\njulia> Base.setindex!(A::SparseArray{T,N}, v, I::Vararg{Int,N}) where {T,N} = (A.data[I] = v)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Notice that this is an IndexCartesian array, so we must manually define getindex and setindex! at the dimensionality of the array. Unlike the SquaresVector, we are able to define setindex!, and so we can mutate the array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A = SparseArray(Float64, 3, 3)\n3×3 SparseArray{Float64, 2}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\njulia> fill!(A, 2)\n3×3 SparseArray{Float64, 2}:\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n\njulia> A[:] = 1:length(A); A\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The result of indexing an AbstractArray can itself be an array (for instance when indexing by an AbstractRange). The AbstractArray fallback methods use similar to allocate an Array of the appropriate size and element type, which is filled in using the basic indexing method described above. However, when implementing an array wrapper you often want the result to be wrapped as well:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A[1:2,:]\n2×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In this example it is accomplished by defining Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where T to create the appropriate wrapped array. (Note that while similar supports 1- and 2-argument forms, in most case you only need to specialize the 3-argument form.) For this to work it's important that SparseArray is mutable (supports setindex!). Defining similar, getindex and setindex! for SparseArray also makes it possible to copy the array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> copy(A)\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In addition to all the iterable and indexable methods from above, these types can also interact with each other and use most of the methods defined in Julia Base for AbstractArrays:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A[SquaresVector(3)]\n3-element SparseArray{Float64, 1}:\n 1.0\n 4.0\n 9.0\n\njulia> sum(A)\n45.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If you are defining an array type that allows non-traditional indexing (indices that start at something other than 1), you should specialize axes. You should also specialize similar so that the dims argument (ordinarily a Dims size-tuple) can accept AbstractUnitRange objects, perhaps range-types Ind of your own design. For more information, see Arrays with custom indices.","page":"Interfaces"},{"title":"Strided Arrays","location":"manual/interfaces.html#man-interface-strided-arrays","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement  Brief description\nstrides(A)  Return the distance in memory (in number of elements) between adjacent elements in each dimension as a tuple. If A is an AbstractArray{T,0}, this should return an empty tuple.\nBase.unsafe_convert(::Type{Ptr{T}}, A)  Return the native address of an array.\nBase.elsize(::Type{<:A})  Return the stride between consecutive elements in the array.\nOptional methods Default definition Brief description\nstride(A, i::Int) strides(A)[i] Return the distance in memory (in number of elements) between adjacent elements in dimension k.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A strided array is a subtype of AbstractArray whose entries are stored in memory with fixed strides. Provided the element type of the array is compatible with BLAS, a strided array can utilize BLAS and LAPACK routines for more efficient linear algebra routines.  A typical example of a user-defined strided array is one that wraps a standard Array with additional structure.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Warning: do not implement these methods if the underlying storage is not actually strided, as it may lead to incorrect results or segmentation faults.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Here are some examples to demonstrate which type of arrays are strided and which are not:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"1:5   # not strided (there is no storage associated with this array.)\nVector(1:5)  # is strided with strides (1,)\nA = [1 5; 2 6; 3 7; 4 8]  # is strided with strides (1,4)\nV = view(A, 1:2, :)   # is strided with strides (1,4)\nV = view(A, 1:2:3, 1:2)   # is strided with strides (2,4)\nV = view(A, [1,2,4], :)   # is not strided, as the spacing between rows is not fixed.","page":"Interfaces"},{"title":"Customizing broadcasting","location":"manual/interfaces.html#man-interfaces-broadcasting","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement Brief description\nBase.BroadcastStyle(::Type{SrcType}) = SrcStyle() Broadcasting behavior of SrcType\nBase.similar(bc::Broadcasted{DestStyle}, ::Type{ElType}) Allocation of output container\nOptional methods \nBase.BroadcastStyle(::Style1, ::Style2) = Style12() Precedence rules for mixing styles\nBase.axes(x) Declaration of the indices of x, as per axes(x).\nBase.broadcastable(x) Convert x to an object that has axes and supports indexing\nBypassing default machinery \nBase.copy(bc::Broadcasted{DestStyle}) Custom implementation of broadcast\nBase.copyto!(dest, bc::Broadcasted{DestStyle}) Custom implementation of broadcast!, specializing on DestStyle\nBase.copyto!(dest::DestType, bc::Broadcasted{Nothing}) Custom implementation of broadcast!, specializing on DestType\nBase.Broadcast.broadcasted(f, args...) Override the default lazy behavior within a fused expression\nBase.Broadcast.instantiate(bc::Broadcasted{DestStyle}) Override the computation of the lazy broadcast's axes","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Broadcasting is triggered by an explicit call to broadcast or broadcast!, or implicitly by \"dot\" operations like A .+ b or f.(x, y). Any object that has axes and supports indexing can participate as an argument in broadcasting, and by default the result is stored in an Array. This basic framework is extensible in three major ways:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Ensuring that all arguments support broadcast\nSelecting an appropriate output array for the given set of arguments\nSelecting an efficient implementation for the given set of arguments","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Not all types support axes and indexing, but many are convenient to allow in broadcast. The Base.broadcastable function is called on each argument to broadcast, allowing it to return something different that supports axes and indexing. By default, this is the identity function for all AbstractArrays and Numbers — they already support axes and indexing.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If a type is intended to act like a \"0-dimensional scalar\" (a single object) rather than as a container for broadcasting, then the following method should be defined:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.broadcastable(o::MyType) = Ref(o)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"that returns the argument wrapped in a 0-dimensional Ref container.   For example, such a wrapper method is defined for types themselves, functions, special singletons like missing and nothing, and dates.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Custom array-like types can specialize Base.broadcastable to define their shape, but they should follow the convention that collect(Base.broadcastable(x)) == collect(x). A notable exception is AbstractString; strings are special-cased to behave as scalars for the purposes of broadcast even though they are iterable collections of their characters (see Strings for more).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The next two steps (selecting the output array and implementation) are dependent upon determining a single answer for a given set of arguments. Broadcast must take all the varied types of its arguments and collapse them down to just one output array and one implementation. Broadcast calls this single answer a \"style\". Every broadcastable object each has its own preferred style, and a promotion-like system is used to combine these styles into a single answer — the \"destination style\".","page":"Interfaces"},{"title":"Broadcast Styles","location":"manual/interfaces.html#Broadcast-Styles","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle is the abstract type from which all broadcast styles are derived. When used as a function it has two possible forms, unary (single-argument) and binary. The unary variant states that you intend to implement specific broadcasting behavior and/or output type, and do not wish to rely on the default fallback Broadcast.DefaultArrayStyle.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"To override these defaults, you can define a custom BroadcastStyle for your object:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct MyStyle <: Broadcast.BroadcastStyle end\nBase.BroadcastStyle(::Type{<:MyType}) = MyStyle()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In some cases it might be convenient not to have to define MyStyle, in which case you can leverage one of the general broadcast wrappers:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Type{<:MyType}) = Broadcast.Style{MyType}() can be used for arbitrary types.\nBase.BroadcastStyle(::Type{<:MyType}) = Broadcast.ArrayStyle{MyType}() is preferred if MyType is an AbstractArray.\nFor AbstractArrays that only support a certain dimensionality, create a subtype of Broadcast.AbstractArrayStyle{N} (see below).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"When your broadcast operation involves several arguments, individual argument styles get combined to determine a single DestStyle that controls the type of the output container. For more details, see below.","page":"Interfaces"},{"title":"Selecting an appropriate output array","location":"manual/interfaces.html#Selecting-an-appropriate-output-array","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The broadcast style is computed for every broadcasting operation to allow for dispatch and specialization. The actual allocation of the result array is handled by similar, using the Broadcasted object as its first argument.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.similar(bc::Broadcasted{DestStyle}, ::Type{ElType})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The fallback definition is","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"similar(bc::Broadcasted{DefaultArrayStyle{N}}, ::Type{ElType}) where {N,ElType} =\n    similar(Array{ElType}, axes(bc))","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"However, if needed you can specialize on any or all of these arguments. The final argument bc is a lazy representation of a (potentially fused) broadcast operation, a Broadcasted object.  For these purposes, the most important fields of the wrapper are f and args, describing the function and argument list, respectively.  Note that the argument list can — and often does — include other nested Broadcasted wrappers.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For a complete example, let's say you have created a type, ArrayAndChar, that stores an array and a single character:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct ArrayAndChar{T,N} <: AbstractArray{T,N}\n    data::Array{T,N}\n    char::Char\nend\nBase.size(A::ArrayAndChar) = size(A.data)\nBase.getindex(A::ArrayAndChar{T,N}, inds::Vararg{Int,N}) where {T,N} = A.data[inds...]\nBase.setindex!(A::ArrayAndChar{T,N}, val, inds::Vararg{Int,N}) where {T,N} = A.data[inds...] = val\nBase.showarg(io::IO, A::ArrayAndChar, toplevel) = print(io, typeof(A), \" with char '\", A.char, \"'\")","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"You might want broadcasting to preserve the char \"metadata\". First we define","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Type{<:ArrayAndChar}) = Broadcast.ArrayStyle{ArrayAndChar}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This means we must also define a corresponding similar method:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"function Base.similar(bc::Broadcast.Broadcasted{Broadcast.ArrayStyle{ArrayAndChar}}, ::Type{ElType}) where ElType\n    # Scan the inputs for the ArrayAndChar:\n    A = find_aac(bc)\n    # Use the char field of A to create the output\n    ArrayAndChar(similar(Array{ElType}, axes(bc)), A.char)\nend\n\n\"`A = find_aac(As)` returns the first ArrayAndChar among the arguments.\"\nfind_aac(bc::Base.Broadcast.Broadcasted) = find_aac(bc.args)\nfind_aac(args::Tuple) = find_aac(find_aac(args[1]), Base.tail(args))\nfind_aac(x) = x\nfind_aac(::Tuple{}) = nothing\nfind_aac(a::ArrayAndChar, rest) = a\nfind_aac(::Any, rest) = find_aac(rest)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"From these definitions, one obtains the following behavior:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> a = ArrayAndChar([1 2; 3 4], 'x')\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 1  2\n 3  4\n\njulia> a .+ 1\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 2  3\n 4  5\n\njulia> a .+ [5,10]\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n  6   7\n 13  14","page":"Interfaces"},{"title":"Extending broadcast with custom implementations","location":"manual/interfaces.html#extending-in-place-broadcast","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In general, a broadcast operation is represented by a lazy Broadcasted container that holds onto the function to be applied alongside its arguments. Those arguments may themselves be more nested Broadcasted containers, forming a large expression tree to be evaluated. A nested tree of Broadcasted containers is directly constructed by the implicit dot syntax; 5 .+ 2.*x is transiently represented by Broadcasted(+, 5, Broadcasted(*, 2, x)), for example. This is invisible to users as it is immediately realized through a call to copy, but it is this container that provides the basis for broadcast's extensibility for authors of custom types. The built-in broadcast machinery will then determine the result type and size based upon the arguments, allocate it, and then finally copy the realization of the Broadcasted object into it with a default copyto!(::AbstractArray, ::Broadcasted) method. The built-in fallback broadcast and broadcast! methods similarly construct a transient Broadcasted representation of the operation so they can follow the same codepath. This allows custom array implementations to provide their own copyto! specialization to customize and optimize broadcasting. This is again determined by the computed broadcast style. This is such an important part of the operation that it is stored as the first type parameter of the Broadcasted type, allowing for dispatch and specialization.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For some types, the machinery to \"fuse\" operations across nested levels of broadcasting is not available or could be done more efficiently incrementally. In such cases, you may need or want to evaluate x .* (x .+ 1) as if it had been written broadcast(*, x, broadcast(+, x, 1)), where the inner operation is evaluated before tackling the outer operation. This sort of eager operation is directly supported by a bit of indirection; instead of directly constructing Broadcasted objects, Julia lowers the fused expression x .* (x .+ 1) to Broadcast.broadcasted(*, x, Broadcast.broadcasted(+, x, 1)). Now, by default, broadcasted just calls the Broadcasted constructor to create the lazy representation of the fused expression tree, but you can choose to override it for a particular combination of function and arguments.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"As an example, the builtin AbstractRange objects use this machinery to optimize pieces of broadcasted expressions that can be eagerly evaluated purely in terms of the start, step, and length (or stop) instead of computing every single element. Just like all the other machinery, broadcasted also computes and exposes the combined broadcast style of its arguments, so instead of specializing on broadcasted(f, args...), you can specialize on broadcasted(::DestStyle, f, args...) for any combination of style, function, and arguments.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For example, the following definition supports the negation of ranges:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"broadcasted(::DefaultArrayStyle{1}, ::typeof(-), r::OrdinalRange) = range(-first(r), step=-step(r), length=length(r))","page":"Interfaces"},{"title":"Extending in-place broadcasting","location":"manual/interfaces.html#extending-in-place-broadcast-2","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In-place broadcasting can be supported by defining the appropriate copyto!(dest, bc::Broadcasted) method. Because you might want to specialize either on dest or the specific subtype of bc, to avoid ambiguities between packages we recommend the following convention.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If you wish to specialize on a particular style DestStyle, define a method for","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"copyto!(dest, bc::Broadcasted{DestStyle})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Optionally, with this form you can also specialize on the type of dest.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If instead you want to specialize on the destination type DestType without specializing on DestStyle, then you should define a method with the following signature:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"copyto!(dest::DestType, bc::Broadcasted{Nothing})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This leverages a fallback implementation of copyto! that converts the wrapper into a Broadcasted{Nothing}. Consequently, specializing on DestType has lower precedence than methods that specialize on DestStyle.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Similarly, you can completely override out-of-place broadcasting with a copy(::Broadcasted) method.","page":"Interfaces"},{"title":"Working with Broadcasted objects","location":"manual/interfaces.html#Working-with-Broadcasted-objects","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In order to implement such a copy or copyto!, method, of course, you must work with the Broadcasted wrapper to compute each element. There are two main ways of doing so:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Broadcast.flatten recomputes the potentially nested operation into a single function and flat list of arguments. You are responsible for implementing the broadcasting shape rules yourself, but this may be helpful in limited situations.\nIterating over the CartesianIndices of the axes(::Broadcasted) and using indexing with the resulting CartesianIndex object to compute the result.","page":"Interfaces"},{"title":"Writing binary broadcasting rules","location":"manual/interfaces.html#writing-binary-broadcasting-rules","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The precedence rules are defined by binary BroadcastStyle calls:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Style1, ::Style2) = Style12()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"where Style12 is the BroadcastStyle you want to choose for outputs involving arguments of Style1 and Style2. For example,","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Broadcast.Style{Tuple}, ::Broadcast.AbstractArrayStyle{0}) = Broadcast.Style{Tuple}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"indicates that Tuple \"wins\" over zero-dimensional arrays (the output container will be a tuple). It is worth noting that you do not need to (and should not) define both argument orders of this call; defining one is sufficient no matter what order the user supplies the arguments in.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For AbstractArray types, defining a BroadcastStyle supersedes the fallback choice, Broadcast.DefaultArrayStyle. DefaultArrayStyle and the abstract supertype, AbstractArrayStyle, store the dimensionality as a type parameter to support specialized array types that have fixed dimensionality requirements.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"DefaultArrayStyle \"loses\" to any other AbstractArrayStyle that has been defined because of the following methods:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"BroadcastStyle(a::AbstractArrayStyle{Any}, ::DefaultArrayStyle) = a\nBroadcastStyle(a::AbstractArrayStyle{N}, ::DefaultArrayStyle{N}) where N = a\nBroadcastStyle(a::AbstractArrayStyle{M}, ::DefaultArrayStyle{N}) where {M,N} =\n    typeof(a)(Val(max(M, N)))","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"You do not need to write binary BroadcastStyle rules unless you want to establish precedence for two or more non-DefaultArrayStyle types.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If your array type does have fixed dimensionality requirements, then you should subtype AbstractArrayStyle. For example, the sparse array code has the following definitions:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct SparseVecStyle <: Broadcast.AbstractArrayStyle{1} end\nstruct SparseMatStyle <: Broadcast.AbstractArrayStyle{2} end\nBase.BroadcastStyle(::Type{<:SparseVector}) = SparseVecStyle()\nBase.BroadcastStyle(::Type{<:SparseMatrixCSC}) = SparseMatStyle()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Whenever you subtype AbstractArrayStyle, you also need to define rules for combining dimensionalities, by creating a constructor for your style that takes a Val(N) argument. For example:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"SparseVecStyle(::Val{0}) = SparseVecStyle()\nSparseVecStyle(::Val{1}) = SparseVecStyle()\nSparseVecStyle(::Val{2}) = SparseMatStyle()\nSparseVecStyle(::Val{N}) where N = Broadcast.DefaultArrayStyle{N}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"These rules indicate that the combination of a SparseVecStyle with 0- or 1-dimensional arrays yields another SparseVecStyle, that its combination with a 2-dimensional array yields a SparseMatStyle, and anything of higher dimensionality falls back to the dense arbitrary-dimensional framework. These rules allow broadcasting to keep the sparse representation for operations that result in one or two dimensional outputs, but produce an Array for any other dimensionality.","page":"Interfaces"},{"title":"Instance Properties","location":"manual/interfaces.html#man-instance-properties","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement Default definition Brief description\npropertynames(x::ObjType, private::Bool=false) fieldnames(typeof(x)) Return a tuple of the properties (x.property) of an object x. If private=true, also return property names intended to be kept as private\ngetproperty(x::ObjType, s::Symbol) getfield(x, s) Return property s of x. x.s calls getproperty(x, :s).\nsetproperty!(x::ObjType, s::Symbol, v) setfield!(x, s, v) Set property s of x to v. x.s = v calls setproperty!(x, :s, v). Should return v.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Sometimes, it is desirable to change how the end-user interacts with the fields of an object. Instead of granting direct access to type fields, an extra layer of abstraction between the user and the code can be provided by overloading object.field. Properties are what the user sees of the object, fields what the object actually is.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"By default, properties and fields are the same. However, this behavior can be changed. For example, take this representation of a point in a plane in polar coordinates:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> mutable struct Point\n           r::Float64\n           ϕ::Float64\n       end\n\njulia> p = Point(7.0, pi/4)\nPoint(7.0, 0.7853981633974483)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"As described in the table above dot access p.r is the same as getproperty(p, :r) which is by default the same as getfield(p, :r):","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> propertynames(p)\n(:r, :ϕ)\n\njulia> getproperty(p, :r), getproperty(p, :ϕ)\n(7.0, 0.7853981633974483)\n\njulia> p.r, p.ϕ\n(7.0, 0.7853981633974483)\n\njulia> getfield(p, :r), getproperty(p, :ϕ)\n(7.0, 0.7853981633974483)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"However, we may want users to be unaware that Point stores the coordinates as r and ϕ (fields), and instead interact with x and y (properties). The methods in the first column can be defined to add new functionality:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.propertynames(::Point, private::Bool=false) = private ? (:x, :y, :r, :ϕ) : (:x, :y)\n\njulia> function Base.getproperty(p::Point, s::Symbol)\n           if s === :x\n               return getfield(p, :r) * cos(getfield(p, :ϕ))\n           elseif s === :y\n               return getfield(p, :r) * sin(getfield(p, :ϕ))\n           else\n               # This allows accessing fields with p.r and p.ϕ\n               return getfield(p, s)\n           end\n       end\n\njulia> function Base.setproperty!(p::Point, s::Symbol, f)\n           if s === :x\n               y = p.y\n               setfield!(p, :r, sqrt(f^2 + y^2))\n               setfield!(p, :ϕ, atan(y, f))\n               return f\n           elseif s === :y\n               x = p.x\n               setfield!(p, :r, sqrt(x^2 + f^2))\n               setfield!(p, :ϕ, atan(f, x))\n               return f\n           else\n               # This allow modifying fields with p.r and p.ϕ\n               return setfield!(p, s, f)\n           end\n       end","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"It is important that getfield and setfield are used inside getproperty and setproperty! instead of the dot syntax, since the dot syntax would make the functions recursive which can lead to type inference issues. We can now try out the new functionality:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> propertynames(p)\n(:x, :y)\n\njulia> p.x\n4.949747468305833\n\njulia> p.y = 4.0\n4.0\n\njulia> p.r\n6.363961030678928","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Finally, it is worth noting that adding instance properties like this is quite rarely done in Julia and should in general only be done if there is a good reason for doing so.","page":"Interfaces"},{"title":"Linux","location":"devdocs/build/linux.html#Linux","category":"section","text":"","page":"Linux"},{"title":"Linux","location":"devdocs/build/linux.html","category":"page","text":"GCC version 4.7 or later is required to build Julia.\nTo use external shared libraries not in the system library search path, set USE_SYSTEM_XXX=1 and LDFLAGS=-Wl,-rpath,/path/to/dir/contains/libXXX.so in Make.user.\nInstead of setting LDFLAGS, putting the library directory into the environment variable LD_LIBRARY_PATH (at both compile and run time) also works.\nThe USE_SYSTEM_* flags should be used with caution. These are meant only for troubleshooting, porting, and packaging, where package maintainers work closely with the Julia developers to make sure that Julia is built correctly. Production use cases should use the officially provided binaries. Issues arising from the use of these flags will generally not be accepted.\nSee also the external dependencies.","page":"Linux"},{"title":"Architecture Customization","location":"devdocs/build/linux.html#Architecture-Customization","category":"section","text":"","page":"Linux"},{"title":"Linux","location":"devdocs/build/linux.html","category":"page","text":"Julia can be built for a non-generic architecture by configuring the ARCH Makefile variable in a Make.user file. See the appropriate section of Make.inc for additional customization options, such as MARCH and JULIA_CPU_TARGET.","page":"Linux"},{"title":"Linux","location":"devdocs/build/linux.html","category":"page","text":"For example, to build for Pentium 4, set MARCH=pentium4 and install the necessary system libraries for linking. On Ubuntu, these may include lib32gfortran-6-dev, lib32gcc1, and lib32stdc++6, among others.","page":"Linux"},{"title":"Linux","location":"devdocs/build/linux.html","category":"page","text":"You can also set MARCH=native in Make.user for a maximum-performance build customized for the current machine CPU.","page":"Linux"},{"title":"Linux Build Troubleshooting","location":"devdocs/build/linux.html#Linux-Build-Troubleshooting","category":"section","text":"","page":"Linux"},{"title":"Linux","location":"devdocs/build/linux.html","category":"page","text":"Problem Possible Solution\nOpenBLAS build failure Set one of the following build options in Make.user and build again: <ul><li> OPENBLAS_TARGET_ARCH=BARCELONA (AMD CPUs) or OPENBLAS_TARGET_ARCH=NEHALEM (Intel CPUs)<ul>Set OPENBLAS_DYNAMIC_ARCH = 0 to disable compiling multiple architectures in a single binary.</ul></li><li> OPENBLAS_NO_AVX2 = 1 disables AVX2 instructions, allowing OpenBLAS to compile with OPENBLAS_DYNAMIC_ARCH = 1 using old versions of binutils </li><li> USE_SYSTEM_BLAS=1 uses the system provided libblas <ul><li>Set LIBBLAS=-lopenblas and LIBBLASNAME=libopenblas to force the use of the system provided OpenBLAS when multiple BLAS versions are installed. </li></ul></li></ul><p> If you get an error that looks like ../kernel/x86_64/dgemm_kernel_4x4_haswell.S:1709: Error: no such instruction: `vpermpd $ 0xb1,%ymm0,%ymm0', then you need to set OPENBLAS_DYNAMIC_ARCH = 0 or OPENBLAS_NO_AVX2 = 1, or you need a newer version of binutils (2.18 or newer). (Issue #7653)</p><p> If the linker cannot find gfortran and you get an error like julia /usr/bin/x86_64-linux-gnu-ld: cannot find -lgfortran, check the path with gfortran -print-file-name=libgfortran.so and use the output to export something similar to this: export LDFLAGS=-L/usr/lib/gcc/x86_64-linux-gnu/8/. See Issue #6150.</p>\nIllegal Instruction error Check if your CPU supports AVX while your OS does not (e.g. through virtualization, as described in this issue).","page":"Linux"},{"title":"Julia ASTs","location":"devdocs/ast.html#Julia-ASTs","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser (e.g. the Meta.parse function), and manipulated by macros. It is a structured representation of code as it is written, constructed by julia-parser.scm from a character stream. Next there is a lowered form, or IR (intermediate representation), which is used by type inference and code generation. In the lowered form there are fewer types of nodes, all macros are expanded, and all control flow is converted to explicit branches and sequences of statements. The lowered form is constructed by julia-syntax.scm.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"First we will focus on the AST, since it is needed to write macros.","page":"Julia ASTs"},{"title":"Surface syntax AST","location":"devdocs/ast.html#Surface-syntax-AST","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers). There is generally a different expression head for each visually distinct syntactic form. Examples will be given in s-expression syntax. Each parenthesized list corresponds to an Expr, where the first element is the head. For example (call f x) corresponds to Expr(:call, :f, :x) in Julia.","page":"Julia ASTs"},{"title":"Calls","location":"devdocs/ast.html#Calls","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"do syntax:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"f(x) do a,b\n    body\nend","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as (do (call f x) (-> (tuple a b) (block body))).","page":"Julia ASTs"},{"title":"Operators","location":"devdocs/ast.html#Operators","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Most uses of operators are just function calls, so they are parsed with the head call. However some operators are special forms (not necessarily function calls), and in those cases the operator itself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\". Some operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument call. Finally, chains of comparisons have their own special expression structure.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))","page":"Julia ASTs"},{"title":"Bracketed forms","location":"devdocs/ast.html#Bracketed-forms","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)","page":"Julia ASTs"},{"title":"Macros","location":"devdocs/ast.html#Macros","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)","page":"Julia ASTs"},{"title":"Strings","location":"devdocs/ast.html#Strings","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Doc string syntax:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"\"some docs\"\nf(x) = x","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x))).","page":"Julia ASTs"},{"title":"Imports and such","location":"devdocs/ast.html#Imports-and-such","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"using has the same representation as import, but with expression head :using instead of :import.","page":"Julia ASTs"},{"title":"Numbers","location":"devdocs/ast.html#Numbers","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented directly as scheme numbers in the AST.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")","page":"Julia ASTs"},{"title":"Block forms","location":"devdocs/ast.html#Block-forms","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A block of statements is parsed as (block stmt1 stmt2 ...).","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"If statement:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"if a\n    b\nelseif c\n    d\nelse\n    e\nend","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6 e))))","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A while loop parses as (while condition body).","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A for loop parses as (for (= var iter) body). If there is more than one iteration specification, they are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body).","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"break and continue are parsed as 0-argument expressions (break) and (continue).","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"let is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body), like for loops.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A basic function definition is parsed as (function (call f x) body). A more complex example:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"function f(x::T; k = 1) where T\n    return x+1\nend","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Type definition:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"mutable struct Foo{T<:S}\n    x::T\nend","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"The first argument is a boolean telling whether the type is mutable.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"try blocks parse as (try try_block var catch_block finally_block). If no variable is present after catch, var is #f. If there is no finally clause, then the last argument is not present.","page":"Julia ASTs"},{"title":"Quote expressions","location":"devdocs/ast.html#Quote-expressions","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $. In Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms. Internally, there is also a need for code quoting without interpolation. In Julia's scheme code, non-interpolating quote is represented with the expression head inert.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"inert expressions are converted to Julia QuoteNode objects. These objects wrap a single value of any type, and when evaluated simply return that value.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A quote expression whose argument is an atom also gets converted to a QuoteNode.","page":"Julia ASTs"},{"title":"Line numbers","location":"devdocs/ast.html#Line-numbers","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Source location information is represented as (line line_num file_name) where the third component is optional (and omitted when the current line number, but not file name, changes).","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"These expressions are represented as LineNumberNodes in Julia.","page":"Julia ASTs"},{"title":"Macros","location":"devdocs/ast.html#Macros-2","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope. The result of a macro expansion is automatically wrapped in (hygienic-scope block module), to represent the result of the new scope. The user can insert (escape block) inside to interpolate code from the caller.","page":"Julia ASTs"},{"title":"Lowered form","location":"devdocs/ast.html#Lowered-form","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference, optimizations like inlining, and code generation. It is also less obvious to the human, since it results from a significant rearrangement of the input syntax.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"In addition to Symbols and some number types, the following data types exist in lowered form:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Expr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of subexpressions. While almost every part of a surface AST is represented by an Expr, the IR uses only a limited number of Exprs, mostly for calls and some top-level-only forms.\nSlot\nIdentifies arguments and local variables by consecutive numbering. Slot is an abstract type with subtypes SlotNumber and TypedSlot. Both types have an integer-valued id field giving the slot index. Most slots have the same type at all uses, and so are represented with SlotNumber. The types of these slots are found in the slottypes field of their CodeInfo object. Slots that require per-use type annotations are represented with TypedSlot, which has a typ field.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the referenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in the code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified by the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function. If the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a QuoteNode whose value field is the symbol a, in order to return the symbol itself instead of evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted by the compiler. The number (id) of an SSAValue is the code array index of the expression whose value it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined.","page":"Julia ASTs"},{"title":"Expr types","location":"devdocs/ast.html#Expr-types","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"These symbols appear in the head field of Exprs in lowered form.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"call\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the arguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a Slot or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end. In the 1-argument form, the argument is a symbol. If this symbol already names a function in the current scope, nothing happens. If the symbol is undefined, a new function is created and assigned to the identifier specified by the symbol. If the symbol is defined but names a non-function, an error is raised. The definition of \"names a function\" is that the binding is constant, and refers to an object of singleton type. The rationale for this is that an instance of a singleton type uniquely identifies the type to add the method to. When the type has fields, it wouldn't be clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression first behaves like the 1-argument form above. This argument is ignored from then on. It can be nothing when methods are added strictly by type, (::T)(x) = x, or when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the argument types, and args[2][2] is a SimpleVector of type variables corresponding to the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a method to a function that also has methods defined in different scopes) this is an expression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer, GlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number of fields, or the minimum number of fields called by an inner constructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The arguments are the same as arguments 1, 2, and 4 of struct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4 are the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5 and replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered to this, and the type is always inserted by the compiler.  This is very much an internal-only feature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to splat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has already been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception().\nundefcheck\nTemporary node inserted by the compiler and will be processed in type_lift_pass!.\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on error.  Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a catch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this expression is true or false (true means bounds checks are disabled), it is pushed onto the stack. If the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds, otherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark the inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply copied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the arguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{Symbol} : calling convention\nThe calling convention for the call.\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call. See Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : isva\nIndicates whether the closure accepts varargs.\nargs[3] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[4] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7","page":"Julia ASTs"},{"title":"Method","location":"devdocs/ast.html#ast-lowered-method","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A unique'd container describing the shared metadata for a single method.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"name, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness. Uniqueness is required for efficiency, especially for incremental precompile and tracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by compression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, isstaged, pure\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method.","page":"Julia ASTs"},{"title":"MethodInstance","location":"devdocs/ast.html#MethodInstance","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A unique'd container describing a single callable signature for a Method. See especially Proper maintenance and care of multi-threading locks for important details on how to modify these fields safely.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"specTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a def.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module, if this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes indexed by def.sparam_syms. For the MethodInstance at Method.unspecialized, this is the empty SimpleVector. But for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nuninferred\nThe uncompressed source code for a toplevel thunk. Additionally, for a generated function, this is one of many places that the source code might be found.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions. This works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance. Those optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation. Thus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation.","page":"Julia ASTs"},{"title":"CodeInstance","location":"devdocs/ast.html#CodeInstance","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"def\nThe MethodInstance that this cache entry is derived from.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"rettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is also the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function, or it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JLCALLABLE `jlvaluet ()(jlfunctiont *f, jlvaluet *args[nargs], uint32t nargs)`\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_function_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_function_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called. If max_world is the special token value -1, the value is not yet known. It may continue to be used until we encounter a backedge that requires us to reconsider.","page":"Julia ASTs"},{"title":"CodeInfo","location":"devdocs/ast.html#CodeInfo","category":"section","text":"","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A (usually temporary) container for holding lowered source code.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"code\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n2  - assigned (only false if there are no assignment statements with this var on the left)\n8  - const (currently unused for local variables)\n16 - statically assigned once\n32 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the function (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level flags for each expression in the function. Many of these are reserved, but not yet implemented:\n0x01 << 0 = statement is marked as @inbounds\n0x01 << 1 = statement is marked as @inline\n0x01 << 2 = statement is marked as @noinline\n0x01 << 3 = statement is within a block that leads to throw call\n0x01 << 4 = statement may be removed if its result is unused, in particular it is thus be both pure and effect free\n0x01 << 5-6 = <unused>\n0x01 << 7 = <reserved> has out-of-band info\nlinetable\nAn array of source location objects\ncodelocs\nAn array of integer indices into the linetable, giving the location associated with each statement.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Optional Fields:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"slottypes\nAn array of types for the slots.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if necessary during inference.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Boolean properties:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"inferred\nWhether this has been produced by type inference.\ninlineable\nWhether this should be eligible for inlining.\npropagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding @boundscheck blocks.\npure\nWhether this is known to be a pure function of its arguments, without respect to the state of the method caches or other mutable global state.","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"UInt8 settings:","page":"Julia ASTs"},{"title":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"constprop\n0 = use heuristic\n1 = aggressive\n2 = none\npurity Constructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details.","page":"Julia ASTs"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html#man-custom-indices","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Conventionally, Julia's arrays are indexed starting at 1, whereas some other languages start numbering at 0, and yet others (e.g., Fortran) allow you to specify arbitrary starting indices.  While there is much merit in picking a standard (i.e., 1 for Julia), there are some algorithms which simplify considerably if you can index outside the range 1:size(A,d) (and not just 0:size(A,d)-1, either). To facilitate such computations, Julia supports arrays with arbitrary indices.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"The purpose of this page is to address the question, \"what do I have to do to support such arrays in my own code?\"  First, let's address the simplest case: if you know that your code will never need to handle arrays with unconventional indexing, hopefully the answer is \"nothing.\" Old code, on conventional arrays, should function essentially without alteration as long as it was using the exported interfaces of Julia. If you find it more convenient to just force your users to supply traditional arrays where indexing starts at one, you can add","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Base.require_one_based_indexing(arrays...)","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"where arrays... is a list of the array objects that you wish to check for anything that violates 1-based indexing.","page":"Arrays with custom indices"},{"title":"Generalizing existing code","location":"devdocs/offset-arrays.html#Generalizing-existing-code","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"As an overview, the steps are:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"replace many uses of size with axes\nreplace 1:length(A) with eachindex(A), or in some cases LinearIndices(A)\nreplace explicit allocations like Array{Int}(undef, size(B)) with similar(Array{Int}, axes(B))","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"These are described in more detail below.","page":"Arrays with custom indices"},{"title":"Things to watch out for","location":"devdocs/offset-arrays.html#Things-to-watch-out-for","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Because unconventional indexing breaks many people's assumptions that all arrays start indexing with 1, there is always the chance that using such arrays will trigger errors. The most frustrating bugs would be incorrect results or segfaults (total crashes of Julia). For example, consider the following function:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"function mycopy!(dest::AbstractVector, src::AbstractVector)\n    length(dest) == length(src) || throw(DimensionMismatch(\"vectors must match\"))\n    # OK, now we're safe to use @inbounds, right? (not anymore!)\n    for i = 1:length(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"This code implicitly assumes that vectors are indexed from 1; if dest starts at a different index than src, there is a chance that this code would trigger a segfault. (If you do get segfaults, to help locate the cause try running julia with the option --check-bounds=yes.)","page":"Arrays with custom indices"},{"title":"Using axes for bounds checks and loop iteration","location":"devdocs/offset-arrays.html#Using-axes-for-bounds-checks-and-loop-iteration","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"axes(A) (reminiscent of size(A)) returns a tuple of AbstractUnitRange{<:Integer} objects, specifying the range of valid indices along each dimension of A.  When A has unconventional indexing, the ranges may not start at 1.  If you just want the range for a particular dimension d, there is axes(A, d).","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Base implements a custom range type, OneTo, where OneTo(n) means the same thing as 1:n but in a form that guarantees (via the type system) that the lower index is 1. For any new AbstractArray type, this is the default returned by axes, and it indicates that this array type uses \"conventional\" 1-based indexing.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"For bounds checking, note that there are dedicated functions checkbounds and checkindex which can sometimes simplify such tests.","page":"Arrays with custom indices"},{"title":"Linear indexing (LinearIndices)","location":"devdocs/offset-arrays.html#Linear-indexing-(LinearIndices)","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Some algorithms are most conveniently (or efficiently) written in terms of a single linear index, A[i] even if A is multi-dimensional. Regardless of the array's native indices, linear indices always range from 1:length(A). However, this raises an ambiguity for one-dimensional arrays (a.k.a., AbstractVector): does v[i] mean linear indexing , or Cartesian indexing with the array's native indices?","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"For this reason, your best option may be to iterate over the array with eachindex(A), or, if you require the indices to be sequential integers, to get the index range by calling LinearIndices(A). This will return axes(A, 1) if A is an AbstractVector, and the equivalent of 1:length(A) otherwise.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"By this definition, 1-dimensional arrays always use Cartesian indexing with the array's native indices. To help enforce this, it's worth noting that the index conversion functions will throw an error if shape indicates a 1-dimensional array with unconventional indexing (i.e., is a Tuple{UnitRange} rather than a tuple of OneTo). For arrays with conventional indexing, these functions continue to work the same as always.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Using axes and LinearIndices, here is one way you could rewrite mycopy!:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"function mycopy!(dest::AbstractVector, src::AbstractVector)\n    axes(dest) == axes(src) || throw(DimensionMismatch(\"vectors must match\"))\n    for i in LinearIndices(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend","page":"Arrays with custom indices"},{"title":"Allocating storage using generalizations of similar","location":"devdocs/offset-arrays.html#Allocating-storage-using-generalizations-of-similar","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Storage is often allocated with Array{Int}(undef, dims) or similar(A, args...). When the result needs to match the indices of some other array, this may not always suffice. The generic replacement for such patterns is to use similar(storagetype, shape).  storagetype indicates the kind of underlying \"conventional\" behavior you'd like, e.g., Array{Int} or BitArray or even dims->zeros(Float32, dims) (which would allocate an all-zeros array). shape is a tuple of Integer or AbstractUnitRange values, specifying the indices that you want the result to use. Note that a convenient way of producing an all-zeros array that matches the indices of A is simply zeros(A).","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Let's walk through a couple of explicit examples. First, if A has conventional indices, then similar(Array{Int}, axes(A)) would end up calling Array{Int}(undef, size(A)), and thus return an array.  If A is an AbstractArray type with unconventional indexing, then similar(Array{Int}, axes(A)) should return something that \"behaves like\" an Array{Int} but with a shape (including indices) that matches A.  (The most obvious implementation is to allocate an Array{Int}(undef, size(A)) and then \"wrap\" it in a type that shifts the indices.)","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Note also that similar(Array{Int}, (axes(A, 2),)) would allocate an AbstractVector{Int} (i.e., 1-dimensional array) that matches the indices of the columns of A.","page":"Arrays with custom indices"},{"title":"Writing custom array types with non-1 indexing","location":"devdocs/offset-arrays.html#Writing-custom-array-types-with-non-1-indexing","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Most of the methods you'll need to define are standard for any AbstractArray type, see Abstract Arrays. This page focuses on the steps needed to define unconventional indexing.","page":"Arrays with custom indices"},{"title":"Custom AbstractUnitRange types","location":"devdocs/offset-arrays.html#Custom-AbstractUnitRange-types","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"If you're writing a non-1 indexed array type, you will want to specialize axes so it returns a UnitRange, or (perhaps better) a custom AbstractUnitRange.  The advantage of a custom type is that it \"signals\" the allocation type for functions like similar. If we're writing an array type for which indexing will start at 0, we likely want to begin by creating a new AbstractUnitRange, ZeroRange, where ZeroRange(n) is equivalent to 0:n-1.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"In general, you should probably not export ZeroRange from your package: there may be other packages that implement their own ZeroRange, and having multiple distinct ZeroRange types is (perhaps counterintuitively) an advantage: ModuleA.ZeroRange indicates that similar should create a ModuleA.ZeroArray, whereas ModuleB.ZeroRange indicates a ModuleB.ZeroArray type.  This design allows peaceful coexistence among many different custom array types.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Note that the Julia package CustomUnitRanges.jl can sometimes be used to avoid the need to write your own ZeroRange type.","page":"Arrays with custom indices"},{"title":"Specializing axes","location":"devdocs/offset-arrays.html#Specializing-axes","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Once you have your AbstractUnitRange type, then specialize axes:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Base.axes(A::ZeroArray) = map(n->ZeroRange(n), A.size)","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"where here we imagine that ZeroArray has a field called size (there would be other ways to implement this).","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"In some cases, the fallback definition for axes(A, d):","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"axes(A::AbstractArray{T,N}, d) where {T,N} = d <= N ? axes(A)[d] : OneTo(1)","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"may not be what you want: you may need to specialize it to return something other than OneTo(1) when d > ndims(A).  Likewise, in Base there is a dedicated function axes1 which is equivalent to axes(A, 1) but which avoids checking (at runtime) whether ndims(A) > 0. (This is purely a performance optimization.)  It is defined as:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"axes1(A::AbstractArray{T,0}) where {T} = OneTo(1)\naxes1(A::AbstractArray) = axes(A)[1]","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"If the first of these (the zero-dimensional case) is problematic for your custom array type, be sure to specialize it appropriately.","page":"Arrays with custom indices"},{"title":"Specializing similar","location":"devdocs/offset-arrays.html#Specializing-similar","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Given your custom ZeroRange type, then you should also add the following two specializations for similar:","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"function Base.similar(A::AbstractArray, T::Type, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nfunction Base.similar(f::Union{Function,DataType}, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Both of these should allocate your custom array type.","page":"Arrays with custom indices"},{"title":"Specializing reshape","location":"devdocs/offset-arrays.html#Specializing-reshape","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Optionally, define a method","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Base.reshape(A::AbstractArray, shape::Tuple{ZeroRange,Vararg{ZeroRange}}) = ...","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"and you can reshape an array so that the result has custom indices.","page":"Arrays with custom indices"},{"title":"For objects that mimic AbstractArray but are not subtypes","location":"devdocs/offset-arrays.html#For-objects-that-mimic-AbstractArray-but-are-not-subtypes","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"has_offset_axes depends on having axes defined for the objects you call it on. If there is some reason you don't have an axes method defined for your object, consider defining a method","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"Base.has_offset_axes(obj::MyNon1IndexedArraylikeObject) = true","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"This will allow code that assumes 1-based indexing to detect a problem and throw a helpful error, rather than returning incorrect results or segfaulting julia.","page":"Arrays with custom indices"},{"title":"Catching errors","location":"devdocs/offset-arrays.html#Catching-errors","category":"section","text":"","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"If your new array type triggers errors in other code, one helpful debugging step can be to comment out @boundscheck in your getindex and setindex! implementation. This will ensure that every element access checks bounds. Or, restart julia with --check-bounds=yes.","page":"Arrays with custom indices"},{"title":"Arrays with custom indices","location":"devdocs/offset-arrays.html","category":"page","text":"In some cases it may also be helpful to temporarily disable size and length for your new array type, since code that makes incorrect assumptions frequently uses these functions.","page":"Arrays with custom indices"},{"title":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray.  This page documents some of the design principles and implementation of SubArrays.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"One of the major design goals is to ensure high performance for views of both IndexLinear and IndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be IndexLinear to the extent that it is possible.","page":"SubArrays"},{"title":"Index replacement","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Consider making 2d slices of a 3d array:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"DocTestSetup = :(import Random; Random.seed!(1234))","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.967143  0.103929\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.789764  0.806704\n 0.566704  0.962715","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"DocTestSetup = nothing","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"view drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2 are two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j]. To extract the value from the parent array A, the natural approach is to replace S1[i,j] with A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]].","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The key feature of the design of SubArrays is that this index replacement can be performed without any runtime overhead.","page":"SubArrays"},{"title":"SubArray design","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":"","page":"SubArrays"},{"title":"Type parameters and fields","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The strategy adopted is first and foremost expressed in the definition of the type:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"struct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"SubArray has 5 type parameters.  The first two are the standard element type and dimensionality.  The next is the type of the parent AbstractArray.  The most heavily-used is the fourth parameter, a Tuple of the types of the indices for each dimension. The final one, L, is only provided as a convenience for dispatch; it's a boolean that represents whether the index types support fast linear indexing. More on that later.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"If in our example above A is a Array{Float64, 3}, our S1 case above would be a SubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}. Note in particular the tuple parameter, which stores the types of the indices used to create S1. Likewise,","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Storing these values allows index replacement, and having the types encoded as parameters allows one to dispatch to efficient algorithms.","page":"SubArrays"},{"title":"Index translation","location":"devdocs/subarrays.html#Index-translation","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Performing index translation requires that you do different things for different concrete SubArray types.  For example, for S1, one needs to apply the i,j indices to the first and third dimensions of the parent array, whereas for S2 one needs to apply them to the second and third.  The simplest approach to indexing would be to do the type-analysis at runtime:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"parentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Unfortunately, this would be disastrous in terms of performance: each element access would allocate memory, and involves the running of a lot of poorly-typed code.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The better approach is to dispatch to specific methods to handle each type of stored index. That's what reindex does: it dispatches on the type of the first stored index and consumes the appropriate number of input indices, and then it recurses on the remaining indices. In the case of S1, this expands to","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Base.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"for any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below).","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is the core of a SubArray; indexing methods depend upon reindex to do this index translation. Sometimes, though, we can avoid the indirection and make it even faster.","page":"SubArrays"},{"title":"Linear indexing","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that separates successive elements, starting from some offset. This means that we can pre-compute these values and represent linear indexing simply as an addition and multiplication, avoiding the indirection of reindex and (more importantly) the slow computation of the cartesian coordinates entirely.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"For SubArray types, the availability of efficient linear indexing is based purely on the types of the indices, and does not depend on values like the size of the parent array. You can ask whether a given set of indices supports fast linear indexing with the internal Base.viewindexing function:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is computed during construction of the SubArray and stored in the L type parameter as a boolean that encodes fast linear indexing support. While not strictly necessary, it means that we can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Since this computation doesn't depend on runtime values, it can miss some cases in which the stride happens to be uniform:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"A view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear indexing indeed could be performed efficiently.  However, success in this case depends on the size of the array: if the first dimension instead were odd,","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"then A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.  Since we have to base this decision based purely on types encoded in the parameters of the SubArray, S = view(A, 2:2:4, :) cannot implement efficient linear indexing.","page":"SubArrays"},{"title":"A few details","location":"devdocs/subarrays.html#A-few-details","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply determines how and where the stored indices should be reindexed. It not only supports integer indices, but it supports non-scalar indexing, too. This means that views of views don't need two levels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given by the parameter N, is not necessarily equal to the dimensionality of the parent array or the length of the indices tuple.  Neither do user-supplied indices necessarily line up with entries in the indices tuple (e.g., the second user-supplied index might correspond to the third dimension of the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal to the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting this dramatically complicates the reindexing process, especially for views of views. Not only do you need to dispatch on the types of the stored indices, but you need to examine whether a given index is the final one and \"merge\" any remaining stored indices together. This is not an easy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly if possible. Consequently, view ensures that the parent array is the appropriate dimensionality for the given indices by reshaping it if needed. The inner SubArray constructor ensures that this invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that reindex simply dispatches on the type of the stored indices in order to determine how many passed indices should be used and where they should go. But with CartesianIndex, there's no longer a one-to-one correspondence between the number of passed arguments and the number of dimensions that they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)), you can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1). It should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across all combinations of dimensionalities in an intractable manner. As such, reindex must never be called with CartesianIndex indices. Fortunately, the scalar case is easily handled by first flattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however, cannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view must ensure that there are no arrays of CartesianIndex in the argument list. If there are, it can simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray with two levels of indirection instead.","page":"SubArrays"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html#Networking-and-Streams","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Julia provides a rich interface to deal with streaming I/O objects such as terminals, pipes and TCP sockets. This interface, though asynchronous at the system level, is presented in a synchronous manner to the programmer and it is usually unnecessary to think about the underlying asynchronous operation. This is achieved by making heavy use of Julia cooperative threading (coroutine) functionality.","page":"Networking and Streams"},{"title":"Basic Stream I/O","location":"manual/networking-and-streams.html#Basic-Stream-I/O","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"All Julia streams expose at least a read and a write method, taking the stream as their first argument, e.g.:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> write(stdout, \"Hello World\");  # suppress return value 11 with ;\nHello World\njulia> read(stdin, Char)\n\n'\\n': ASCII/Unicode U+000a (category Cc: Other, control)","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Note that write returns 11, the number of bytes (in \"Hello World\") written to stdout, but this return value is suppressed with the ;.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Here Enter was pressed again so that Julia would read the newline. Now, as you can see from this example, write takes the data to write as its second argument, while read takes the type of the data to be read as the second argument.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"For example, to read a simple byte array, we could do:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> x = zeros(UInt8, 4)\n4-element Array{UInt8,1}:\n 0x00\n 0x00\n 0x00\n 0x00\n\njulia> read!(stdin, x)\nabcd\n4-element Array{UInt8,1}:\n 0x61\n 0x62\n 0x63\n 0x64","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"However, since this is slightly cumbersome, there are several convenience methods provided. For example, we could have written the above as:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> read(stdin, 4)\nabcd\n4-element Array{UInt8,1}:\n 0x61\n 0x62\n 0x63\n 0x64","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"or if we had wanted to read the entire line instead:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> readline(stdin)\nabcd\n\"abcd\"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Note that depending on your terminal settings, your TTY may be line buffered and might thus require an additional enter before the data is sent to Julia.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"To read every line from stdin you can use eachline:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"for line in eachline(stdin)\n    print(\"Found $line\")\nend","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"or read if you wanted to read by character instead:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"while !eof(stdin)\n    x = read(stdin, Char)\n    println(\"Found: $x\")\nend","page":"Networking and Streams"},{"title":"Text I/O","location":"manual/networking-and-streams.html#Text-I/O","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Note that the write method mentioned above operates on binary streams. In particular, values do not get converted to any canonical text representation but are written out as is:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> write(stdout, 0x61);  # suppress return value 1 with ;\na","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Note that a is written to stdout by the write function and that the returned value is 1 (since 0x61 is one byte).","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"For text I/O, use the print or show methods, depending on your needs (see the documentation for these two methods for a detailed discussion of the difference between them):","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> print(stdout, 0x61)\n97","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"See Custom pretty-printing for more information on how to implement display methods for custom types.","page":"Networking and Streams"},{"title":"IO Output Contextual Properties","location":"manual/networking-and-streams.html#IO-Output-Contextual-Properties","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Sometimes IO output can benefit from the ability to pass contextual information into show methods. The IOContext object provides this framework for associating arbitrary metadata with an IO object. For example, :compact => true adds a hinting parameter to the IO object that the invoked show method should print a shorter output (if applicable). See the IOContext documentation for a list of common properties.","page":"Networking and Streams"},{"title":"Working with Files","location":"manual/networking-and-streams.html#Working-with-Files","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Like many other environments, Julia has an open function, which takes a filename and returns an IOStream object that you can use to read and write things from the file. For example, if we have a file, hello.txt, whose contents are Hello, World!:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> f = open(\"hello.txt\")\nIOStream(<file hello.txt>)\n\njulia> readlines(f)\n1-element Array{String,1}:\n \"Hello, World!\"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"If you want to write to a file, you can open it with the write (\"w\") flag:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> f = open(\"hello.txt\",\"w\")\nIOStream(<file hello.txt>)\n\njulia> write(f,\"Hello again.\")\n12","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"If you examine the contents of hello.txt at this point, you will notice that it is empty; nothing has actually been written to disk yet. This is because the IOStream must be closed before the write is actually flushed to disk:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> close(f)","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Examining hello.txt again will show its contents have been changed.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Opening a file, doing something to its contents, and closing it again is a very common pattern. To make this easier, there exists another invocation of open which takes a function as its first argument and filename as its second, opens the file, calls the function with the file as an argument, and then closes it again. For example, given a function:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"function read_and_capitalize(f::IOStream)\n    return uppercase(read(f, String))\nend","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"You can call:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> open(read_and_capitalize, \"hello.txt\")\n\"HELLO AGAIN.\"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"to open hello.txt, call read_and_capitalize on it, close hello.txt and return the capitalized contents.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"To avoid even having to define a named function, you can use the do syntax, which creates an anonymous function on the fly:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> open(\"hello.txt\") do f\n           uppercase(read(f, String))\n       end\n\"HELLO AGAIN.\"","page":"Networking and Streams"},{"title":"A simple TCP example","location":"manual/networking-and-streams.html#A-simple-TCP-example","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Let's jump right in with a simple example involving TCP sockets. This functionality is in a standard library package called Sockets. Let's first create a simple server:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> using Sockets\n\njulia> errormonitor(@async begin\n           server = listen(2000)\n           while true\n               sock = accept(server)\n               println(\"Hello World\\n\")\n           end\n       end)\nTask (runnable) @0x00007fd31dc11ae0","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"To those familiar with the Unix socket API, the method names will feel familiar, though their usage is somewhat simpler than the raw Unix socket API. The first call to listen will create a server waiting for incoming connections on the specified port (2000) in this case. The same function may also be used to create various other kinds of servers:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> listen(2000) # Listens on localhost:2000 (IPv4)\nSockets.TCPServer(active)\n\njulia> listen(ip\"127.0.0.1\",2000) # Equivalent to the first\nSockets.TCPServer(active)\n\njulia> listen(ip\"::1\",2000) # Listens on localhost:2000 (IPv6)\nSockets.TCPServer(active)\n\njulia> listen(IPv4(0),2001) # Listens on port 2001 on all IPv4 interfaces\nSockets.TCPServer(active)\n\njulia> listen(IPv6(0),2001) # Listens on port 2001 on all IPv6 interfaces\nSockets.TCPServer(active)\n\njulia> listen(\"testsocket\") # Listens on a UNIX domain socket\nSockets.PipeServer(active)\n\njulia> listen(\"\\\\\\\\.\\\\pipe\\\\testsocket\") # Listens on a Windows named pipe\nSockets.PipeServer(active)","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Note that the return type of the last invocation is different. This is because this server does not listen on TCP, but rather on a named pipe (Windows) or UNIX domain socket. Also note that Windows named pipe format has to be a specific pattern such that the name prefix (\\\\.\\pipe\\) uniquely identifies the file type. The difference between TCP and named pipes or UNIX domain sockets is subtle and has to do with the accept and connect methods. The accept method retrieves a connection to the client that is connecting on the server we just created, while the connect function connects to a server using the specified method. The connect function takes the same arguments as listen, so, assuming the environment (i.e. host, cwd, etc.) is the same you should be able to pass the same arguments to connect as you did to listen to establish the connection. So let's try that out (after having created the server above):","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> connect(2000)\nTCPSocket(open, 0 bytes waiting)\n\njulia> Hello World","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"As expected we saw \"Hello World\" printed. So, let's actually analyze what happened behind the scenes. When we called connect, we connect to the server we had just created. Meanwhile, the accept function returns a server-side connection to the newly created socket and prints \"Hello World\" to indicate that the connection was successful.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"A great strength of Julia is that since the API is exposed synchronously even though the I/O is actually happening asynchronously, we didn't have to worry about callbacks or even making sure that the server gets to run. When we called connect the current task waited for the connection to be established and only continued executing after that was done. In this pause, the server task resumed execution (because a connection request was now available), accepted the connection, printed the message and waited for the next client. Reading and writing works in the same way. To see this, consider the following simple echo server:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> errormonitor(@async begin\n           server = listen(2001)\n           while true\n               sock = accept(server)\n               @async while isopen(sock)\n                   write(sock, readline(sock, keep=true))\n               end\n           end\n       end)\nTask (runnable) @0x00007fd31dc12e60\n\njulia> clientside = connect(2001)\nTCPSocket(RawFD(28) open, 0 bytes waiting)\n\njulia> errormonitor(@async while isopen(clientside)\n           write(stdout, readline(clientside, keep=true))\n       end)\nTask (runnable) @0x00007fd31dc11870\n\njulia> println(clientside,\"Hello World from the Echo Server\")\nHello World from the Echo Server","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"As with other streams, use close to disconnect the socket:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> close(clientside)","page":"Networking and Streams"},{"title":"Resolving IP Addresses","location":"manual/networking-and-streams.html#Resolving-IP-Addresses","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"One of the connect methods that does not follow the listen methods is connect(host::String,port), which will attempt to connect to the host given by the host parameter on the port given by the port parameter. It allows you to do things like:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> connect(\"google.com\", 80)\nTCPSocket(RawFD(30) open, 0 bytes waiting)","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"At the base of this functionality is getaddrinfo, which will do the appropriate address resolution:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> getaddrinfo(\"google.com\")\nip\"74.125.226.225\"","page":"Networking and Streams"},{"title":"Asynchronous I/O","location":"manual/networking-and-streams.html#Asynchronous-I/O","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"All I/O operations exposed by Base.read and Base.write can be performed asynchronously through the use of coroutines. You can create a new coroutine to read from or write to a stream using the @async macro:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> task = @async open(\"foo.txt\", \"w\") do io\n           write(io, \"Hello, World!\")\n       end;\n\njulia> wait(task)\n\njulia> readlines(\"foo.txt\")\n1-element Array{String,1}:\n \"Hello, World!\"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"It's common to run into situations where you want to perform multiple asynchronous operations concurrently and wait until they've all completed. You can use the @sync macro to cause your program to block until all of the coroutines it wraps around have exited:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"julia> using Sockets\n\njulia> @sync for hostname in (\"google.com\", \"github.com\", \"julialang.org\")\n           @async begin\n               conn = connect(hostname, 80)\n               write(conn, \"GET / HTTP/1.1\\r\\nHost:$(hostname)\\r\\n\\r\\n\")\n               readline(conn, keep=true)\n               println(\"Finished connection to $(hostname)\")\n           end\n       end\nFinished connection to google.com\nFinished connection to julialang.org\nFinished connection to github.com","page":"Networking and Streams"},{"title":"Multicast","location":"manual/networking-and-streams.html#Multicast","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Julia supports multicast over IPv4 and IPv6 using the User Datagram Protocol (UDP) as transport.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Unlike the Transmission Control Protocol (TCP), UDP makes almost no assumptions about the needs of the application. TCP provides flow control (it accelerates and decelerates to maximize throughput), reliability (lost or corrupt packets are automatically retransmitted), sequencing (packets are ordered by the operating system before they are given to the application), segment size, and session setup and teardown. UDP provides no such features.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"A common use for UDP is in multicast applications. TCP is a stateful protocol for communication between exactly two devices. UDP can use special multicast addresses to allow simultaneous communication between many devices.","page":"Networking and Streams"},{"title":"Receiving IP Multicast Packets","location":"manual/networking-and-streams.html#Receiving-IP-Multicast-Packets","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"To transmit data over UDP multicast, simply recv on the socket, and the first packet received will be returned. Note that it may not be the first packet that you sent however!","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"using Sockets\ngroup = ip\"228.5.6.7\"\nsocket = Sockets.UDPSocket()\nbind(socket, ip\"0.0.0.0\", 6789)\njoin_multicast_group(socket, group)\nprintln(String(recv(socket)))\nleave_multicast_group(socket, group)\nclose(socket)","page":"Networking and Streams"},{"title":"Sending IP Multicast Packets","location":"manual/networking-and-streams.html#Sending-IP-Multicast-Packets","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"To transmit data over UDP multicast, simply send to the socket. Notice that it is not necessary for a sender to join the multicast group.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"using Sockets\ngroup = ip\"228.5.6.7\"\nsocket = Sockets.UDPSocket()\nsend(socket, group, 6789, \"Hello over IPv4\")\nclose(socket)","page":"Networking and Streams"},{"title":"IPv6 Example","location":"manual/networking-and-streams.html#IPv6-Example","category":"section","text":"","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"This example gives the same functionality as the previous program, but uses IPv6 as the network-layer protocol.","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Listener:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"using Sockets\ngroup = Sockets.IPv6(\"ff05::5:6:7\")\nsocket = Sockets.UDPSocket()\nbind(socket, Sockets.IPv6(\"::\"), 6789)\njoin_multicast_group(socket, group)\nprintln(String(recv(socket)))\nleave_multicast_group(socket, group)\nclose(socket)","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"Sender:","page":"Networking and Streams"},{"title":"Networking and Streams","location":"manual/networking-and-streams.html","category":"page","text":"using Sockets\ngroup = Sockets.IPv6(\"ff05::5:6:7\")\nsocket = Sockets.UDPSocket()\nsend(socket, group, 6789, \"Hello over IPv6\")\nclose(socket)","page":"Networking and Streams"},{"title":"SHA","location":"stdlib/SHA.html#SHA","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"DocTestSetup = quote\n    using SHA\n    using InteractiveUtils\nend","page":"SHA"},{"title":"SHA functions","location":"stdlib/SHA.html#SHA-functions","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"Usage is very straightforward:","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> bytes2hex(sha256(\"test\"))\n\"9f86d081884c7d659a2feaa0c55ad015a3bf4f1b2b0b822cd15d6c15b0f00a08\"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"Each exported function (at the time of this writing, SHA-1, SHA-2 224, 256, 384 and 512, and SHA-3 224, 256, 384 and 512 functions are implemented) takes in either an AbstractVector{UInt8}, an AbstractString or an IO object.  This makes it trivial to checksum a file:","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"shell> cat /tmp/test.txt\ntest\njulia> using SHA\n\njulia> open(\"/tmp/test.txt\") do f\n           sha2_256(f)\n       end\n32-element Array{UInt8,1}:\n 0x9f\n 0x86\n 0xd0\n 0x81\n 0x88\n 0x4c\n 0x7d\n 0x65\n    ⋮\n 0x5d\n 0x6c\n 0x15\n 0xb0\n 0xf0\n 0x0a\n 0x08","page":"SHA"},{"title":"All SHA functions","location":"stdlib/SHA.html#All-SHA-functions","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"Due to the colloquial usage of sha256 to refer to sha2_256, convenience functions are provided, mapping shaxxx() function calls to sha2_xxx(). For SHA-3, no such colloquialisms exist and the user must use the full sha3_xxx() names.","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"shaxxx() takes AbstractString and array-like objects (NTuple and Array) with elements of type UInt8.","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"sha1","page":"SHA"},{"title":"SHA.sha1","location":"stdlib/SHA.html#SHA.sha1","category":"function","text":"sha1(data)\n\nHash data using the sha1 algorithm and return the resulting digest. See also SHA1_CTX.\n\n\n\n\n\nsha1(io::IO)\n\nHash data from io using sha1 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"sha224\nsha256\nsha384\nsha512\nsha2_224\nsha2_256\nsha2_384\nsha2_512","page":"SHA"},{"title":"SHA.sha224","location":"stdlib/SHA.html#SHA.sha224","category":"function","text":"sha224(data)\n\nHash data using the sha224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha224(io::IO)\n\nHash data from io using sha224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha256","location":"stdlib/SHA.html#SHA.sha256","category":"function","text":"sha256(data)\n\nHash data using the sha256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha256(io::IO)\n\nHash data from io using sha256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha384","location":"stdlib/SHA.html#SHA.sha384","category":"function","text":"sha384(data)\n\nHash data using the sha384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha384(io::IO)\n\nHash data from io using sha384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha512","location":"stdlib/SHA.html#SHA.sha512","category":"function","text":"sha512(data)\n\nHash data using the sha512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha512(io::IO)\n\nHash data from io using sha512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha2_224","location":"stdlib/SHA.html#SHA.sha2_224","category":"function","text":"sha2_224(data)\n\nHash data using the sha2_224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha2_224(io::IO)\n\nHash data from io using sha2_224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha2_256","location":"stdlib/SHA.html#SHA.sha2_256","category":"function","text":"sha2_256(data)\n\nHash data using the sha2_256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha2_256(io::IO)\n\nHash data from io using sha2_256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha2_384","location":"stdlib/SHA.html#SHA.sha2_384","category":"function","text":"sha2_384(data)\n\nHash data using the sha2_384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha2_384(io::IO)\n\nHash data from io using sha2_384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha2_512","location":"stdlib/SHA.html#SHA.sha2_512","category":"function","text":"sha2_512(data)\n\nHash data using the sha2_512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha2_512(io::IO)\n\nHash data from io using sha2_512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"sha3_224\nsha3_256\nsha3_384\nsha3_512","page":"SHA"},{"title":"SHA.sha3_224","location":"stdlib/SHA.html#SHA.sha3_224","category":"function","text":"sha3_224(data)\n\nHash data using the sha3_224 algorithm and return the resulting digest. See also SHA3_224_CTX.\n\n\n\n\n\nsha3_224(io::IO)\n\nHash data from io using sha3_224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha3_256","location":"stdlib/SHA.html#SHA.sha3_256","category":"function","text":"sha3_256(data)\n\nHash data using the sha3_256 algorithm and return the resulting digest. See also SHA3_256_CTX.\n\n\n\n\n\nsha3_256(io::IO)\n\nHash data from io using sha3_256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha3_384","location":"stdlib/SHA.html#SHA.sha3_384","category":"function","text":"sha3_384(data)\n\nHash data using the sha3_384 algorithm and return the resulting digest. See also SHA3_384_CTX.\n\n\n\n\n\nsha3_384(io::IO)\n\nHash data from io using sha3_384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.sha3_512","location":"stdlib/SHA.html#SHA.sha3_512","category":"function","text":"sha3_512(data)\n\nHash data using the sha3_512 algorithm and return the resulting digest. See also SHA3_512_CTX.\n\n\n\n\n\nsha3_512(io::IO)\n\nHash data from io using sha3_512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"Working with context","location":"stdlib/SHA.html#Working-with-context","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"To create a hash from multiple items the SHAX_XXX_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> ctx = SHA2_256_CTX()\nSHA2 256-bit hash state\n\njulia> update!(ctx, b\"some data\")\n0x0000000000000009\n\njulia> update!(ctx, b\"some more data\")\n0x0000000000000017\n\njulia> digest!(ctx)\n32-element Vector{UInt8}:\n 0xbe\n 0xcf\n 0x23\n 0xda\n 0xaf\n 0x02\n 0xf7\n 0xa3\n 0x57\n 0x92\n    ⋮\n 0x89\n 0x4f\n 0x59\n 0xd8\n 0xb3\n 0xb4\n 0x81\n 0x8b\n 0xc5","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"Note that, at the time of this writing, the SHA3 code is not optimized, and as such is roughly an order of magnitude slower than SHA2.","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"update!\ndigest!","page":"SHA"},{"title":"SHA.update!","location":"stdlib/SHA.html#SHA.update!","category":"function","text":"update!(context, data[, datalen])\n\nUpdate the SHA context with the bytes in data. See also digest! for finalizing the hash.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.digest!","location":"stdlib/SHA.html#SHA.digest!","category":"function","text":"digest!(context)\n\nFinalize the SHA context and return the hash as array of bytes (Array{Uint8, 1}).\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\njulia> digest!(ctx)\n20-element Array{UInt8,1}:\n 0x83\n 0xe4\n ⋮\n 0x89\n 0xf5\n\n\n\n\n\n","page":"SHA"},{"title":"All SHA context types","location":"stdlib/SHA.html#All-SHA-context-types","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA1_CTX","page":"SHA"},{"title":"SHA.SHA1_CTX","location":"stdlib/SHA.html#SHA.SHA1_CTX","category":"type","text":"SHA1_CTX()\n\nConstruct an empty SHA1 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"Convenience types are also provided, where SHAXXX_CTX is a type alias for SHA2_XXX_CTX.","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA224_CTX\nSHA256_CTX\nSHA384_CTX\nSHA512_CTX\nSHA2_224_CTX\nSHA2_256_CTX\nSHA2_384_CTX\nSHA2_512_CTX","page":"SHA"},{"title":"SHA.SHA224_CTX","location":"stdlib/SHA.html#SHA.SHA224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA256_CTX","location":"stdlib/SHA.html#SHA.SHA256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA384_CTX","location":"stdlib/SHA.html#SHA.SHA384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA512_CTX","location":"stdlib/SHA.html#SHA.SHA512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA2_224_CTX","location":"stdlib/SHA.html#SHA.SHA2_224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA2_256_CTX","location":"stdlib/SHA.html#SHA.SHA2_256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA2_384_CTX","location":"stdlib/SHA.html#SHA.SHA2_384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA2_512_CTX","location":"stdlib/SHA.html#SHA.SHA2_512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA3_224_CTX\nSHA3_256_CTX\nSHA3_384_CTX\nSHA3_512_CTX","page":"SHA"},{"title":"SHA.SHA3_224_CTX","location":"stdlib/SHA.html#SHA.SHA3_224_CTX","category":"type","text":"SHA3_224_CTX()\n\nConstruct an empty SHA3_224 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA3_256_CTX","location":"stdlib/SHA.html#SHA.SHA3_256_CTX","category":"type","text":"SHA3_256_CTX()\n\nConstruct an empty SHA3_256 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA3_384_CTX","location":"stdlib/SHA.html#SHA.SHA3_384_CTX","category":"type","text":"SHA3_384_CTX()\n\nConstruct an empty SHA3_384 context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.SHA3_512_CTX","location":"stdlib/SHA.html#SHA.SHA3_512_CTX","category":"type","text":"SHA3_512_CTX()\n\nConstruct an empty SHA3_512 context.\n\n\n\n\n\n","page":"SHA"},{"title":"HMAC functions","location":"stdlib/SHA.html#HMAC-functions","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> bytes2hex(hmac_sha3_256(key, \"test-message\"))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"To create a hash from multiple items, the HMAC_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!.","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> ctx = HMAC_CTX(SHA3_256_CTX(), key);\n\njulia> update!(ctx, b\"test-\")\n0x0000000000000000000000000000008d\n\njulia> update!(ctx, b\"message\")\n0x00000000000000000000000000000094\n\njulia> bytes2hex(digest!(ctx))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\"","page":"SHA"},{"title":"All HMAC functions","location":"stdlib/SHA.html#All-HMAC-functions","category":"section","text":"","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"HMAC context type","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"HMAC_CTX","page":"SHA"},{"title":"SHA.HMAC_CTX","location":"stdlib/SHA.html#SHA.HMAC_CTX","category":"type","text":"HMAC_CTX(ctx::CTX, key::Vector{UInt8}) where {CTX<:SHA_CTX}\n\nConstruct an empty HMAC_CTX context.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"hmac_sha1","page":"SHA"},{"title":"SHA.hmac_sha1","location":"stdlib/SHA.html#SHA.hmac_sha1","category":"function","text":"hmac_sha1(key, data)\n\nHash data using the sha1 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha1(key, io::IO)\n\nHash data from io with the passed key using sha1 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"hmac_sha224\nhmac_sha256\nhmac_sha384\nhmac_sha512\nhmac_sha2_224\nhmac_sha2_256\nhmac_sha2_384\nhmac_sha2_512","page":"SHA"},{"title":"SHA.hmac_sha224","location":"stdlib/SHA.html#SHA.hmac_sha224","category":"function","text":"hmac_sha224(key, data)\n\nHash data using the sha224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha224(key, io::IO)\n\nHash data from io with the passed key using sha224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha256","location":"stdlib/SHA.html#SHA.hmac_sha256","category":"function","text":"hmac_sha256(key, data)\n\nHash data using the sha256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha256(key, io::IO)\n\nHash data from io with the passed key using sha256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha384","location":"stdlib/SHA.html#SHA.hmac_sha384","category":"function","text":"hmac_sha384(key, data)\n\nHash data using the sha384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha384(key, io::IO)\n\nHash data from io with the passed key using sha384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha512","location":"stdlib/SHA.html#SHA.hmac_sha512","category":"function","text":"hmac_sha512(key, data)\n\nHash data using the sha512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha512(key, io::IO)\n\nHash data from io with the passed key using sha512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha2_224","location":"stdlib/SHA.html#SHA.hmac_sha2_224","category":"function","text":"hmac_sha2_224(key, data)\n\nHash data using the sha2_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_224(key, io::IO)\n\nHash data from io with the passed key using sha2_224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha2_256","location":"stdlib/SHA.html#SHA.hmac_sha2_256","category":"function","text":"hmac_sha2_256(key, data)\n\nHash data using the sha2_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_256(key, io::IO)\n\nHash data from io with the passed key using sha2_256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha2_384","location":"stdlib/SHA.html#SHA.hmac_sha2_384","category":"function","text":"hmac_sha2_384(key, data)\n\nHash data using the sha2_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_384(key, io::IO)\n\nHash data from io with the passed key using sha2_384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha2_512","location":"stdlib/SHA.html#SHA.hmac_sha2_512","category":"function","text":"hmac_sha2_512(key, data)\n\nHash data using the sha2_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_512(key, io::IO)\n\nHash data from io with the passed key using sha2_512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3","page":"SHA"},{"title":"SHA","location":"stdlib/SHA.html","category":"page","text":"hmac_sha3_224\nhmac_sha3_256\nhmac_sha3_384\nhmac_sha3_512","page":"SHA"},{"title":"SHA.hmac_sha3_224","location":"stdlib/SHA.html#SHA.hmac_sha3_224","category":"function","text":"hmac_sha3_224(key, data)\n\nHash data using the sha3_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_224(key, io::IO)\n\nHash data from io with the passed key using sha3_224 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha3_256","location":"stdlib/SHA.html#SHA.hmac_sha3_256","category":"function","text":"hmac_sha3_256(key, data)\n\nHash data using the sha3_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_256(key, io::IO)\n\nHash data from io with the passed key using sha3_256 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha3_384","location":"stdlib/SHA.html#SHA.hmac_sha3_384","category":"function","text":"hmac_sha3_384(key, data)\n\nHash data using the sha3_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_384(key, io::IO)\n\nHash data from io with the passed key using sha3_384 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"SHA.hmac_sha3_512","location":"stdlib/SHA.html#SHA.hmac_sha3_512","category":"function","text":"hmac_sha3_512(key, data)\n\nHash data using the sha3_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_512(key, io::IO)\n\nHash data from io with the passed key using sha3_512 algorithm.\n\n\n\n\n\n","page":"SHA"},{"title":"TOML","location":"stdlib/TOML.html#TOML","category":"section","text":"","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"TOML.jl is a Julia standard library for parsing and writing TOML v1.0 files.","page":"TOML"},{"title":"Parsing TOML data","location":"stdlib/TOML.html#Parsing-TOML-data","category":"section","text":"","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> data = \"\"\"\n           [database]\n           server = \"192.168.1.1\"\n           ports = [ 8001, 8001, 8002 ]\n       \"\"\";\n\njulia> TOML.parse(data)\nDict{String, Any} with 1 entry:\n  \"database\" => Dict{String, Any}(\"server\"=>\"192.168.1.1\", \"ports\"=>[8001, 8001…","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"To parse a file, use TOML.parsefile. If the file has a syntax error, an exception is thrown:","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> TOML.parse(\"\"\"\n           value = 0.0.0\n       \"\"\")\nERROR: TOML Parser error:\nnone:1:16 error: failed to parse value\n      value = 0.0.0\n                 ^\n[...]","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"There are other versions of the parse functions (TOML.tryparse and [TOML.tryparsefile]) that instead of throwing exceptions on parser error returns a TOML.ParserError with information:","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> err = TOML.tryparse(\"\"\"\n           value = 0.0.0\n       \"\"\");\n\njulia> err.type\nErrGenericValueError::ErrorType = 14\n\njulia> err.line\n1\n\njulia> err.column\n16","page":"TOML"},{"title":"Exporting data to TOML file","location":"stdlib/TOML.html#Exporting-data-to-TOML-file","category":"section","text":"","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"The TOML.print function is used to print (or serialize) data into TOML format.","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> data = Dict(\n          \"names\" => [\"Julia\", \"Julio\"],\n          \"age\" => [10, 20],\n       );\n\njulia> TOML.print(data)\nnames = [\"Julia\", \"Julio\"]\nage = [10, 20]\n\njulia> fname = tempname();\n\njulia> open(fname, \"w\") do io\n           TOML.print(io, data)\n       end\n\njulia> TOML.parsefile(fname)\nDict{String, Any} with 2 entries:\n  \"names\" => [\"Julia\", \"Julio\"]\n  \"age\"   => [10, 20]","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"Keys can be sorted according to some value","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> TOML.print(Dict(\n       \"abc\"  => 1,\n       \"ab\"   => 2,\n       \"abcd\" => 3,\n       ); sorted=true, by=length)\nab = 2\nabc = 1\nabcd = 3","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"For custom structs, pass a function that converts the struct to a supported type","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> struct MyStruct\n           a::Int\n           b::String\n       end\n\njulia> TOML.print(Dict(\"foo\" => MyStruct(5, \"bar\"))) do x\n           x isa MyStruct && return [x.a, x.b]\n           error(\"unhandled type $(typeof(x))\")\n       end\nfoo = [5, \"bar\"]","page":"TOML"},{"title":"References","location":"stdlib/TOML.html#References","category":"section","text":"","page":"TOML"},{"title":"TOML","location":"stdlib/TOML.html","category":"page","text":"TOML.parse\nTOML.parsefile\nTOML.tryparse\nTOML.tryparsefile\nTOML.print\nTOML.Parser\nTOML.ParserError","page":"TOML"},{"title":"TOML.parse","location":"stdlib/TOML.html#TOML.parse","category":"function","text":"parse(x::Union{AbstractString, IO})\nparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string  or stream x, and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparse.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.parsefile","location":"stdlib/TOML.html#TOML.parsefile","category":"function","text":"parsefile(f::AbstractString)\nparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparsefile.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.tryparse","location":"stdlib/TOML.html#TOML.tryparse","category":"function","text":"tryparse(x::Union{AbstractString, IO})\ntryparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string or stream x, and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parse.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.tryparsefile","location":"stdlib/TOML.html#TOML.tryparsefile","category":"function","text":"tryparsefile(f::AbstractString)\ntryparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parsefile.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.print","location":"stdlib/TOML.html#TOML.print","category":"function","text":"print([to_toml::Function], io::IO [=stdout], data::AbstractDict; sorted=false, by=identity)\n\nWrite data as TOML syntax to the stream io. If the keyword argument sorted is set to true, sort tables according to the function given by the keyword argument by.\n\nThe following data types are supported: AbstractDict, AbstractVector, AbstractString, Integer, AbstractFloat, Bool, Dates.DateTime, Dates.Time, Dates.Date. Note that the integers and floats need to be convertible to Float64 and Int64 respectively. For other data types, pass the function to_toml that takes the data types and returns a value of a supported type.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.Parser","location":"stdlib/TOML.html#TOML.Parser","category":"type","text":"Parser()\n\nConstructor for a TOML Parser.  Note that in most cases one does not need to explicitly create a Parser but instead one directly use use TOML.parsefile or TOML.parse.  Using an explicit parser will however reuse some internal data structures which can be beneficial for performance if a larger number of small files are parsed.\n\n\n\n\n\n","page":"TOML"},{"title":"TOML.ParserError","location":"stdlib/TOML.html#TOML.ParserError","category":"type","text":"ParserError\n\nType that is returned from tryparse and tryparsefile when parsing fails. It contains (among others) the following fields:\n\npos, the position in the string when the error happened\ntable, the result that so far was successfully parsed\ntype, an error type, different for different types of errors\n\n\n\n\n\n","page":"TOML"},{"title":"FreeBSD","location":"devdocs/build/freebsd.html#FreeBSD","category":"section","text":"","page":"FreeBSD"},{"title":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"Clang is the default compiler on FreeBSD 11.0-RELEASE and above. The remaining build tools are available from the Ports Collection, and can be installed using pkg install git gcc gmake cmake pkgconf. To build Julia, simply run gmake. (Note that gmake must be used rather than make, since make on FreeBSD corresponds to the incompatible BSD Make rather than GNU Make.)","page":"FreeBSD"},{"title":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"As mentioned above, it is important to note that the USE_SYSTEM_* flags should be used with caution on FreeBSD. This is because many system libraries, and even libraries from the Ports Collection, link to the system's libgcc_s.so.1, or to another library which links to the system libgcc_s. This library declares its GCC version to be 4.6, which is too old to build Julia, and conflicts with other libraries when linking. Thus it is highly recommended to simply allow Julia to build all of its dependencies. If you do choose to use the USE_SYSTEM_* flags, note that /usr/local is not on the compiler path by default, so you may need to add LDFLAGS=-L/usr/local/lib and CPPFLAGS=-I/usr/local/include to your Make.user, though doing so may interfere with other dependencies.","page":"FreeBSD"},{"title":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"Note that the x86 architecture does not support threading due to lack of compiler runtime library support, so you may need to set JULIA_THREADS=0 in your Make.user if you're on a 32-bit system.","page":"FreeBSD"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html#Static-analyzer-annotations-for-GC-correctness-in-C-code","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Running the analysis","location":"devdocs/gc-sa.html#Running-the-analysis","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer plugin that drives the analysis ships with julia. Its source code can be found in src/clangsa. Running it requires the clang dependency to be build. Set the BUILD_LLVM_CLANG variable in your Make.user in order to build an appropriate version of clang. You may also want to use the prebuilt binaries using the USE_BINARYBUILDER_LLVM options.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Alternatively (or if these do not suffice), try","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"make -C src install-analysis-deps","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"from Julia's toplevel directory.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Afterwards, running the analysis over the source tree is as simple as running make -C src analyzegc.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"General Overview","location":"devdocs/gc-sa.html#General-Overview","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Since Julia's GC is precise, it needs to maintain correct rooting information for any value that may be referenced at any time GC may occur. These places are known as safepoints and in the function local context, we extend this designation to any function call that may recursively end up at a safepoint.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"In generated code, this is taken care of automatically by the GC root placement pass (see the chapter on GC rooting in the LLVM codegen devdocs). However, in C code, we need to inform the runtime of any GC roots manually. This is done using the following macros:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"// The value assigned to any slot passed as an argument to these\n// is rooted for the duration of this GC frame.\nJL_GC_PUSH{1,...,6}(args...)\n// The values assigned into the size `n` array `rts` are rooted\n// for the duration of this GC frame.\nJL_GC_PUSHARGS(rts, n)\n// Pop a GC frame\nJL_GC_POP","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"If these macros are not used where they need to be, or they are used incorrectly, the result is silent memory corruption. As such it is very important that they are placed correctly in all applicable code.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"As such, we employ static analysis (and in particular the clang static analyzer) to help ensure that these macros are used correctly. The remainder of this document gives an overview of this static analysis and describes the support needed in the julia code base to make things work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"GC Invariants","location":"devdocs/gc-sa.html#GC-Invariants","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"There is two simple invariants correctness:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"All GC_PUSH calls need to be followed by an appropriate GC_POP (in practice we enforce this at the function level)\nIf a value was previously not rooted at any safepoint, it may no longer be referenced afterwards","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Of course the devil is in the details here. In particular to satisfy the second of the above conditions, we need to know:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Which calls are safepoints and which are not\nWhich values are rooted at any given safepoint and which are not\nWhen is a value referenced","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"For the second point in particular, we need to know which memory locations will be considered rooting at runtime (i.e. values assigned to such locations are rooted). This includes locations explicitly designated as such by passing them to one of the GC_PUSH macros, globally rooted locations and values, as well as any location recursively reachable from one of those locations.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static Analysis Algorithm","location":"devdocs/gc-sa.html#Static-Analysis-Algorithm","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The idea itself is very simple, although the implementation is quite a bit more complicated (mainly due to a large number of special cases and intricacies of C and C++). In essence, we keep track of all locations that are rooting, all values that are rootable and any expression (assignments, allocations, etc) affect the rootedness of any rootable values. Then, at any safepoint, we perform a \"symbolic GC\" and poison any values that are not rooted at said location. If these values are later referenced, we emit an error.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The clang static analyzer works by constructing a graph of states and exploring this graph for sources of errors. Several nodes in this graph are generated by the analyzer itself (e.g. for control flow), but the definitions above augment this graph with our own state.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The static analyzer is interprocedural and can analyze control flow across function boundaries. However, the static analyzer is not fully recursive and makes heuristic decisions about which calls to explore (additionally some calls are cross-translation unit and invisible to the analyzer). In our case, our definition of correctness requires total information. As such, we need to annotate the prototypes of all function calls with whatever information the analysis required, even if that information would otherwise be available by interprocedural static analysis.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Luckily however, we can still use this interprocedural analysis to ensure that the annotations we place on a given function are indeed correct given the implementation of said function.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"The analyzer annotations","location":"devdocs/gc-sa.html#The-analyzer-annotations","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"These annotations are found in src/support/analyzer_annotations.h. The are only active when the analyzer is being used and expand either to nothing (for prototype annotations) or to no-ops (for function like annotations).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_NOTSAFEPOINT","location":"devdocs/gc-sa.html#JL_NOTSAFEPOINT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is perhaps the most common annotation, and should be placed on any function that is known not to possibly lead to reaching a GC safepoint. In general, it is only safe for such a function to perform arithmetic, memory accesses and calls to functions either annotated JL_NOTSAFEPOINT or otherwise known not to be safepoints (e.g. function in the C standard library, which are hardcoded as such in the analyzer)","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"It is valid to keep values unrooted across calls to any function annotated with this attribute:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_get_one() JL_NOTSAFEPOINT {\n  return 1;\n}\n\njl_value_t *example() {\n  jl_value_t *val = jl_alloc_whatever();\n  // This is valid, even though `val` is unrooted, because\n  // jl_get_one is not a safepoint\n  jl_get_one();\n  return val;\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","location":"devdocs/gc-sa.html#JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"When JL_MAYBE_UNROOTED is annotated as an argument on a function, indicates that said argument may be passed, even if it is not rooted. In the ordinary course of events, the julia ABI guarantees that callers root values before passing them to callees. However, some functions do not follow this ABI and allow values to be passed to them even though they are not rooted. Note however, that this does not automatically imply that said argument will be preserved. The ROOTS_TEMPORARILY annotation provides the stronger guarantee that, not only may the value be unrooted when passed, it will also be preserved across any internal safepoints by the callee.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Note that JL_NOTSAFEPOINT essentially implies JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY, because the rootedness of an argument is irrelevant if the function contains no safepoints.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"One additional point to note is that these annotations apply on both the caller and the callee side. On the caller side, they lift rootedness restrictions that are normally required for julia ABI functions. On the callee side, they have the reverse effect of preventing these arguments from being considered implicitly rooted.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"If either of these annotations is applied to the function as a whole, it applies to all arguments of the function. This should generally only be necessary for varargs functions.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"JL_DLLEXPORT void JL_NORETURN jl_throw(jl_value_t *e JL_MAYBE_UNROOTED);\njl_value_t *jl_alloc_error();\n\nvoid example() {\n  // The return value of the allocation is unrooted. This would normally\n  // be an error, but is allowed because of the above annotation.\n  jl_throw(jl_alloc_error());\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_PROPAGATES_ROOT","location":"devdocs/gc-sa.html#JL_PROPAGATES_ROOT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation is commonly found on accessor functions that return one rootable object stored within another. When annotated on a function argument, it tells the analyzer that the root for that argument also applies to the value returned by the function.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"jl_value_t *jl_svecref(jl_svec_t *t JL_PROPAGATES_ROOT, size_t i) JL_NOTSAFEPOINT;\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_svecref(svec, 1)\n  // This is valid, because, as annotated by the PROPAGATES_ROOT annotation,\n  // jl_svecref propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","location":"devdocs/gc-sa.html#JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is essentially the assignment counterpart to JL_PROPAGATES_ROOT. When assigning a value to a field of another value that is already rooted, the assigned value will inherit the root of the value it is assigned into.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_svecset(void *t JL_ROOTING_ARGUMENT, size_t i, void *x JL_ROOTED_ARGUMENT) JL_NOTSAFEPOINT\n\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_box_long(10000);\n  jl_svecset(svec, val);\n  // This is valid, because the annotations imply that the\n  // jl_svecset propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_DISABLED","location":"devdocs/gc-sa.html#JL_GC_DISABLED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation implies that this function is only called with the GC runtime-disabled. Functions of this kind are most often encountered during startup and in the GC code itself. Note that this annotation is checked against the runtime enable/disable calls, so clang will know if you lie. This is not a good way to disable processing of a given function if the GC is not actually disabled (use ifdef __clang_analyzer__ for that if you must).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_do_magic() JL_GC_DISABLED {\n  // Wildly allocate here with no regard for roots\n}\n\nvoid example() {\n  int en = jl_gc_enable(0);\n  jl_do_magic();\n  jl_gc_enable(en);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_REQUIRE_ROOTED_SLOT","location":"devdocs/gc-sa.html#JL_REQUIRE_ROOTED_SLOT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation requires the caller to pass in a slot that is rooted (i.e. values assigned to this slot will be rooted).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_do_processing(jl_value_t **slot JL_REQUIRE_ROOTED_SLOT) {\n  *slot = jl_box_long(1);\n  // Ok, only, because the slot was annotated as rooting\n  jl_gc_safepoint();\n}\n\nvoid example() {\n  jl_value_t *slot = NULL;\n  JL_GC_PUSH1(&slot);\n  jl_do_processing(&slot);\n  JL_GC_POP();\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GLOBALLY_ROOTED","location":"devdocs/gc-sa.html#JL_GLOBALLY_ROOTED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation implies that a given value is always globally rooted. It can be applied to global variable declarations, in which case it will apply to the value of those variables (or values if the declaration if for an array), or to functions, in which case it will apply to the return value of such functions (e.g. for functions that always return some private, globally rooted value).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"extern JL_DLLEXPORT jl_datatype_t *jl_any_type JL_GLOBALLY_ROOTED;\njl_ast_context_t *jl_ast_ctx(fl_context_t *fl) JL_GLOBALLY_ROOTED;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ALWAYS_LEAFTYPE","location":"devdocs/gc-sa.html#JL_ALWAYS_LEAFTYPE","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotations is essentially equivalent to JL_GLOBALLY_ROOTED, except that is should only be used if those values are globally rooted by virtue of being a leaftype. The rooting of leaftypes is a bit complicated. They are generally rooted through cache field of the corresponding TypeName, which itself is rooted by the containing module (so they're rooted as long as the containing module is ok) and we can generally assume that leaftypes are rooted where they are used, but we may refine this property in the future, so the separate annotation helps split out the reason for being globally rooted.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer also automatically detects checks for leaftype-ness and will not complain about missing GC roots on these paths.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"JL_DLLEXPORT jl_value_t *jl_apply_array_type(jl_value_t *type, size_t dim) JL_ALWAYS_LEAFTYPE;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_PROMISE_ROOTED","location":"devdocs/gc-sa.html#JL_GC_PROMISE_ROOTED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is a function-like annotation. Any value passed to this annotation will be considered rooted for the scope of the current function. It is designed as an escape hatch for analyzer inadequacy or complicated situations. However, it should be used sparingly, in favor of improving the analyzer itself.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void example() {\n  jl_value_t *val = jl_alloc_something();\n  if (some_condition) {\n    // We happen to know for complicated external reasons\n    // that val is rooted under these conditions\n    JL_GC_PROMISE_ROOTED(val);\n  }\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Completeness of analysis","location":"devdocs/gc-sa.html#Completeness-of-analysis","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer only looks at local information. In particular, e.g. in the PROPAGATES_ROOT case above, it assumes that such memory is only modified in ways it can see, not in any called functions (unless it happens to decide to consider them in its analysis) and not in any concurrently running threads. As such, it may miss a few problematic cases, though in practice such concurrent modification is fairly rare. Improving the analyzer to handle more such cases may be an interesting topic for future work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Collections and Data Structures","location":"base/collections.html#Collections-and-Data-Structures","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Iteration","location":"base/collections.html#lib-collections-iteration","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Sequential iteration is implemented by the iterate function. The general for loop:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"for i in iter   # or  \"for i = iter\"\n    # body\nend","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"is translated into:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"next = iterate(iter)\nwhile next !== nothing\n    (i, state) = next\n    # body\n    next = iterate(iter, state)\nend","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"The state object may be anything, and should be chosen appropriately for each iterable type. See the manual section on the iteration interface for more details about defining a custom iterable type.","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.iterate\nBase.IteratorSize\nBase.IteratorEltype","page":"Collections and Data Structures"},{"title":"Base.iterate","location":"base/collections.html#Base.iterate","category":"function","text":"iterate(iter [, state]) -> Union{Nothing, Tuple{Any, Any}}\n\nAdvance the iterator to obtain the next element. If no elements remain, nothing should be returned. Otherwise, a 2-tuple of the next element and the new iteration state should be returned.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.IteratorSize","location":"base/collections.html#Base.IteratorSize","category":"type","text":"IteratorSize(itertype::Type) -> IteratorSize\n\nGiven the type of an iterator, return one of the following values:\n\nSizeUnknown() if the length (number of elements) cannot be determined in advance.\nHasLength() if there is a fixed, finite length.\nHasShape{N}() if there is a known length plus a notion of multidimensional shape (as for an array).  In this case N should give the number of dimensions, and the axes function is valid  for the iterator.\nIsInfinite() if the iterator yields values forever.\n\nThe default value (for iterators that do not define this function) is HasLength(). This means that most iterators are assumed to implement length.\n\nThis trait is generally used to select between algorithms that pre-allocate space for their result, and algorithms that resize their result incrementally.\n\njulia> Base.IteratorSize(1:5)\nBase.HasShape{1}()\n\njulia> Base.IteratorSize((2,3))\nBase.HasLength()\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.IteratorEltype","location":"base/collections.html#Base.IteratorEltype","category":"type","text":"IteratorEltype(itertype::Type) -> IteratorEltype\n\nGiven the type of an iterator, return one of the following values:\n\nEltypeUnknown() if the type of elements yielded by the iterator is not known in advance.\nHasEltype() if the element type is known, and eltype would return a meaningful value.\n\nHasEltype() is the default, since iterators are assumed to implement eltype.\n\nThis trait is generally used to select between algorithms that pre-allocate a specific type of result, and algorithms that pick a result type based on the types of yielded values.\n\njulia> Base.IteratorEltype(1:5)\nBase.HasEltype()\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"AbstractRange\nUnitRange\nTuple\nNumber\nAbstractArray\nBitSet\nIdDict\nDict\nWeakKeyDict\nEachLine\nAbstractString\nSet\nPair\nNamedTuple","page":"Collections and Data Structures"},{"title":"Constructors and Types","location":"base/collections.html#Constructors-and-Types","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.AbstractRange\nBase.OrdinalRange\nBase.AbstractUnitRange\nBase.StepRange\nBase.UnitRange\nBase.LinRange","page":"Collections and Data Structures"},{"title":"Base.AbstractRange","location":"base/collections.html#Base.AbstractRange","category":"type","text":"AbstractRange{T}\n\nSupertype for ranges with elements of type T. UnitRange and other types are subtypes of this.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.OrdinalRange","location":"base/collections.html#Base.OrdinalRange","category":"type","text":"OrdinalRange{T, S} <: AbstractRange{T}\n\nSupertype for ordinal ranges with elements of type T with spacing(s) of type S. The steps should be always-exact multiples of oneunit, and T should be a \"discrete\" type, which cannot have values smaller than oneunit. For example, Integer or Date types would qualify, whereas Float64 would not (since this type can represent values smaller than oneunit(Float64). UnitRange, StepRange, and other types are subtypes of this.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.AbstractUnitRange","location":"base/collections.html#Base.AbstractUnitRange","category":"type","text":"AbstractUnitRange{T} <: OrdinalRange{T, T}\n\nSupertype for ranges with a step size of oneunit(T) with elements of type T. UnitRange and other types are subtypes of this.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.StepRange","location":"base/collections.html#Base.StepRange","category":"type","text":"StepRange{T, S} <: OrdinalRange{T, S}\n\nRanges with elements of type T with spacing of type S. The step between each element is constant, and the range is defined in terms of a start and stop of type T and a step of type S. Neither T nor S should be floating point types. The syntax a:b:c with b != 0 and a, b, and c all integers creates a StepRange.\n\nExamples\n\njulia> collect(StepRange(1, Int8(2), 10))\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\njulia> typeof(StepRange(1, Int8(2), 10))\nStepRange{Int64, Int8}\n\njulia> typeof(1:3:6)\nStepRange{Int64, Int64}\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.UnitRange","location":"base/collections.html#Base.UnitRange","category":"type","text":"UnitRange{T<:Real}\n\nA range parameterized by a start and stop of type T, filled with elements spaced by 1 from start until stop is exceeded. The syntax a:b with a and b both Integers creates a UnitRange.\n\nExamples\n\njulia> collect(UnitRange(2.3, 5.2))\n3-element Vector{Float64}:\n 2.3\n 3.3\n 4.3\n\njulia> typeof(1:10)\nUnitRange{Int64}\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.LinRange","location":"base/collections.html#Base.LinRange","category":"type","text":"LinRange{T,L}\n\nA range with len linearly spaced elements between its start and stop. The size of the spacing is controlled by len, which must be an Integer.\n\nExamples\n\njulia> LinRange(1.5, 5.5, 9)\n9-element LinRange{Float64, Int64}:\n 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0, 5.5\n\nCompared to using range, directly constructing a LinRange should have less overhead but won't try to correct for floating point errors:\n\njulia> collect(range(-0.1, 0.3, length=5))\n5-element Vector{Float64}:\n -0.1\n  0.0\n  0.1\n  0.2\n  0.3\n\njulia> collect(LinRange(-0.1, 0.3, 5))\n5-element Vector{Float64}:\n -0.1\n -1.3877787807814457e-17\n  0.09999999999999999\n  0.19999999999999998\n  0.3\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"General Collections","location":"base/collections.html#General-Collections","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.isempty\nBase.empty!\nBase.length\nBase.checked_length","page":"Collections and Data Structures"},{"title":"Base.isempty","location":"base/collections.html#Base.isempty","category":"function","text":"isempty(collection) -> Bool\n\nDetermine whether a collection is empty (has no elements).\n\nwarning: Warning\nisempty(itr) may consume the next element of a stateful iterator itr unless an appropriate Base.isdone(itr) or isempty method is defined. Use of isempty should therefore be avoided when writing generic code which should support any iterator type.\n\nExamples\n\njulia> isempty([])\ntrue\n\njulia> isempty([1 2 3])\nfalse\n\n\n\n\n\nisempty(condition)\n\nReturn true if no tasks are waiting on the condition, false otherwise.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.empty!","location":"base/collections.html#Base.empty!","category":"function","text":"empty!(collection) -> collection\n\nRemove all elements from a collection.\n\nExamples\n\njulia> A = Dict(\"a\" => 1, \"b\" => 2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> empty!(A);\n\njulia> A\nDict{String, Int64}()\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.length","location":"base/collections.html#Base.length","category":"function","text":"length(collection) -> Integer\n\nReturn the number of elements in the collection.\n\nUse lastindex to get the last valid index of an indexable collection.\n\nSee also: size, ndims, eachindex.\n\nExamples\n\njulia> length(1:5)\n5\n\njulia> length([1, 2, 3, 4])\n4\n\njulia> length([1 2; 3 4])\n4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.checked_length","location":"base/collections.html#Base.checked_length","category":"function","text":"Base.checked_length(r)\n\nCalculates length(r), but may check for overflow errors where applicable when the result doesn't fit into Union{Integer(eltype(r)),Int}.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"AbstractRange\nUnitRange\nTuple\nNumber\nAbstractArray\nBitSet\nIdDict\nDict\nWeakKeyDict\nAbstractString\nSet\nNamedTuple","page":"Collections and Data Structures"},{"title":"Iterable Collections","location":"base/collections.html#Iterable-Collections","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.in\nBase.:∉\nBase.eltype\nBase.indexin\nBase.unique\nBase.unique!\nBase.allunique\nBase.allequal\nBase.reduce(::Any, ::Any)\nBase.reduce(::Any, ::AbstractArray)\nBase.foldl(::Any, ::Any)\nBase.foldr(::Any, ::Any)\nBase.maximum\nBase.maximum!\nBase.minimum\nBase.minimum!\nBase.extrema\nBase.extrema!\nBase.argmax\nBase.argmin\nBase.findmax\nBase.findmin\nBase.findmax!\nBase.findmin!\nBase.sum\nBase.sum!\nBase.prod\nBase.prod!\nBase.any(::Any)\nBase.any(::AbstractArray, ::Any)\nBase.any!\nBase.all(::Any)\nBase.all(::AbstractArray, ::Any)\nBase.all!\nBase.count\nBase.foreach\nBase.map\nBase.map!\nBase.mapreduce(::Any, ::Any, ::Any)\nBase.mapfoldl(::Any, ::Any, ::Any)\nBase.mapfoldr(::Any, ::Any, ::Any)\nBase.first\nBase.last\nBase.front\nBase.tail\nBase.step\nBase.collect(::Any)\nBase.collect(::Type, ::Any)\nBase.filter\nBase.filter!\nBase.replace(::Any, ::Pair...)\nBase.replace(::Base.Callable, ::Any)\nBase.replace!\nBase.rest\nBase.split_rest","page":"Collections and Data Structures"},{"title":"Base.in","location":"base/collections.html#Base.in","category":"function","text":"in(item, collection) -> Bool\n∈(item, collection) -> Bool\n\nDetermine whether an item is in the given collection, in the sense that it is == to one of the values generated by iterating over the collection. Returns a Bool value, except if item is missing or collection contains missing but not item, in which case missing is returned (three-valued logic, matching the behavior of any and ==).\n\nSome collections follow a slightly different definition. For example, Sets check whether the item isequal to one of the elements; Dicts look for key=>value pairs, and the key is compared using isequal.\n\nTo test for the presence of a key in a dictionary, use haskey or k in keys(dict). For the collections mentioned above, the result is always a Bool.\n\nWhen broadcasting with in.(items, collection) or items .∈ collection, both item and collection are broadcasted over, which is often not what is intended. For example, if both arguments are vectors (and the dimensions match), the result is a vector indicating whether each value in collection items is in the value at the corresponding position in collection. To get a vector indicating whether each value in items is in collection, wrap collection in a tuple or a Ref like this: in.(items, Ref(collection)) or items .∈ Ref(collection).\n\nSee also: ∉.\n\nExamples\n\njulia> a = 1:3:20\n1:3:19\n\njulia> 4 in a\ntrue\n\njulia> 5 in a\nfalse\n\njulia> missing in [1, 2]\nmissing\n\njulia> 1 in [2, missing]\nmissing\n\njulia> 1 in [1, missing]\ntrue\n\njulia> missing in Set([1, 2])\nfalse\n\njulia> (1=>missing) in Dict(1=>10, 2=>20)\nmissing\n\njulia> [1, 2] .∈ [2, 3]\n2-element BitVector:\n 0\n 0\n\njulia> [1, 2] .∈ ([2, 3],)\n2-element BitVector:\n 0\n 1\n\nSee also: insorted, contains, occursin, issubset.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.:∉","location":"base/collections.html#Base.:∉","category":"function","text":"∉(item, collection) -> Bool\n∌(collection, item) -> Bool\n\nNegation of ∈ and ∋, i.e. checks that item is not in collection.\n\nWhen broadcasting with items .∉ collection, both item and collection are broadcasted over, which is often not what is intended. For example, if both arguments are vectors (and the dimensions match), the result is a vector indicating whether each value in collection items is not in the value at the corresponding position in collection. To get a vector indicating whether each value in items is not in collection, wrap collection in a tuple or a Ref like this: items .∉ Ref(collection).\n\nExamples\n\njulia> 1 ∉ 2:4\ntrue\n\njulia> 1 ∉ 1:3\nfalse\n\njulia> [1, 2] .∉ [2, 3]\n2-element BitVector:\n 1\n 1\n\njulia> [1, 2] .∉ ([2, 3],)\n2-element BitVector:\n 1\n 0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.eltype","location":"base/collections.html#Base.eltype","category":"function","text":"eltype(type)\n\nDetermine the type of the elements generated by iterating a collection of the given type. For dictionary types, this will be a Pair{KeyType,ValType}. The definition eltype(x) = eltype(typeof(x)) is provided for convenience so that instances can be passed instead of types. However the form that accepts a type argument should be defined for new types.\n\nSee also: keytype, typeof.\n\nExamples\n\njulia> eltype(fill(1f0, (2,2)))\nFloat32\n\njulia> eltype(fill(0x1, (2,2)))\nUInt8\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.indexin","location":"base/collections.html#Base.indexin","category":"function","text":"indexin(a, b)\n\nReturn an array containing the first index in b for each value in a that is a member of b. The output array contains nothing wherever a is not a member of b.\n\nSee also: sortperm, findfirst.\n\nExamples\n\njulia> a = ['a', 'b', 'c', 'b', 'd', 'a'];\n\njulia> b = ['a', 'b', 'c'];\n\njulia> indexin(a, b)\n6-element Vector{Union{Nothing, Int64}}:\n 1\n 2\n 3\n 2\n  nothing\n 1\n\njulia> indexin(b, a)\n3-element Vector{Union{Nothing, Int64}}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.unique","location":"base/collections.html#Base.unique","category":"function","text":"unique(itr)\n\nReturn an array containing only the unique elements of collection itr, as determined by isequal, in the order that the first of each set of equivalent elements originally appears. The element type of the input is preserved.\n\nSee also: unique!, allunique, allequal.\n\nExamples\n\njulia> unique([1, 2, 6, 2])\n3-element Vector{Int64}:\n 1\n 2\n 6\n\njulia> unique(Real[1, 1.0, 2])\n2-element Vector{Real}:\n 1\n 2\n\n\n\n\n\nunique(f, itr)\n\nReturn an array containing one value from itr for each unique value produced by f applied to elements of itr.\n\nExamples\n\njulia> unique(x -> x^2, [1, -1, 3, -3, 4])\n3-element Vector{Int64}:\n 1\n 3\n 4\n\nThis functionality can also be used to extract the indices of the first occurrences of unique elements in an array:\n\njulia> a = [3.1, 4.2, 5.3, 3.1, 3.1, 3.1, 4.2, 1.7];\n\njulia> i = unique(i -> a[i], eachindex(a))\n4-element Vector{Int64}:\n 1\n 2\n 3\n 8\n\njulia> a[i]\n4-element Vector{Float64}:\n 3.1\n 4.2\n 5.3\n 1.7\n\njulia> a[i] == unique(a)\ntrue\n\n\n\n\n\nunique(A::AbstractArray; dims::Int)\n\nReturn unique regions of A along dimension dims.\n\nExamples\n\njulia> A = map(isodd, reshape(Vector(1:8), (2,2,2)))\n2×2×2 Array{Bool, 3}:\n[:, :, 1] =\n 1  1\n 0  0\n\n[:, :, 2] =\n 1  1\n 0  0\n\njulia> unique(A)\n2-element Vector{Bool}:\n 1\n 0\n\njulia> unique(A, dims=2)\n2×1×2 Array{Bool, 3}:\n[:, :, 1] =\n 1\n 0\n\n[:, :, 2] =\n 1\n 0\n\njulia> unique(A, dims=3)\n2×2×1 Array{Bool, 3}:\n[:, :, 1] =\n 1  1\n 0  0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.unique!","location":"base/collections.html#Base.unique!","category":"function","text":"unique!(f, A::AbstractVector)\n\nSelects one value from A for each unique value produced by f applied to elements of A, then return the modified A.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\nExamples\n\njulia> unique!(x -> x^2, [1, -1, 3, -3, 4])\n3-element Vector{Int64}:\n 1\n 3\n 4\n\njulia> unique!(n -> n%3, [5, 1, 8, 9, 3, 4, 10, 7, 2, 6])\n3-element Vector{Int64}:\n 5\n 1\n 9\n\njulia> unique!(iseven, [2, 3, 5, 7, 9])\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\nunique!(A::AbstractVector)\n\nRemove duplicate items as determined by isequal, then return the modified A. unique! will return the elements of A in the order that they occur. If you do not care about the order of the returned data, then calling (sort!(A); unique!(A)) will be much more efficient as long as the elements of A can be sorted.\n\nExamples\n\njulia> unique!([1, 1, 1])\n1-element Vector{Int64}:\n 1\n\njulia> A = [7, 3, 2, 3, 7, 5];\n\njulia> unique!(A)\n4-element Vector{Int64}:\n 7\n 3\n 2\n 5\n\njulia> B = [7, 6, 42, 6, 7, 42];\n\njulia> sort!(B);  # unique! is able to process sorted data much more efficiently.\n\njulia> unique!(B)\n3-element Vector{Int64}:\n  6\n  7\n 42\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.allunique","location":"base/collections.html#Base.allunique","category":"function","text":"allunique(itr) -> Bool\n\nReturn true if all values from itr are distinct when compared with isequal.\n\nSee also: unique, issorted, allequal.\n\nExamples\n\njulia> allunique([1, 2, 3])\ntrue\n\njulia> allunique([1, 2, 1, 2])\nfalse\n\njulia> allunique(Real[1, 1.0, 2])\nfalse\n\njulia> allunique([NaN, 2.0, NaN, 4.0])\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.allequal","location":"base/collections.html#Base.allequal","category":"function","text":"allequal(itr) -> Bool\n\nReturn true if all values from itr are equal when compared with isequal.\n\nSee also: unique, allunique.\n\ncompat: Julia 1.8\nThe allequal function requires at least Julia 1.8.\n\nExamples\n\njulia> allequal([])\ntrue\n\njulia> allequal([1])\ntrue\n\njulia> allequal([1, 1])\ntrue\n\njulia> allequal([1, 2])\nfalse\n\njulia> allequal(Dict(:a => 1, :b => 1))\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.reduce","location":"base/collections.html#Base.reduce-Tuple{Any, Any}","category":"method","text":"reduce(op, itr; [init])\n\nReduce the given collection itr with the given binary operator op. If provided, the initial value init must be a neutral element for op that will be returned for empty collections. It is unspecified whether init is used for non-empty collections.\n\nFor empty collections, providing init will be necessary, except for some special cases (e.g. when op is one of +, *, max, min, &, |) when Julia can determine the neutral element of op.\n\nReductions for certain commonly-used operators may have special implementations, and should be used instead: maximum(itr), minimum(itr), sum(itr), prod(itr), any(itr), all(itr). There are efficient methods for concatenating certain arrays of arrays by calling reduce(vcat, arr) or reduce(hcat, arr).\n\nThe associativity of the reduction is implementation dependent. This means that you can't use non-associative operations like - because it is undefined whether reduce(-,[1,2,3]) should be evaluated as (1-2)-3 or 1-(2-3). Use foldl or foldr instead for guaranteed left or right associativity.\n\nSome operations accumulate error. Parallelism will be easier if the reduction can be executed in groups. Future versions of Julia might change the algorithm. Note that the elements are not reordered if you use an ordered collection.\n\nExamples\n\njulia> reduce(*, [2; 3; 4])\n24\n\njulia> reduce(*, [2; 3; 4]; init=-1)\n-24\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.reduce","location":"base/collections.html#Base.reduce-Tuple{Any, AbstractArray}","category":"method","text":"reduce(f, A::AbstractArray; dims=:, [init])\n\nReduce 2-argument function f along dimensions of A. dims is a vector specifying the dimensions to reduce, and the keyword argument init is the initial value to use in the reductions. For +, *, max and min the init argument is optional.\n\nThe associativity of the reduction is implementation-dependent; if you need a particular associativity, e.g. left-to-right, you should write your own loop or consider using foldl or foldr. See documentation for reduce.\n\nExamples\n\njulia> a = reshape(Vector(1:16), (4,4))\n4×4 Matrix{Int64}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> reduce(max, a, dims=2)\n4×1 Matrix{Int64}:\n 13\n 14\n 15\n 16\n\njulia> reduce(max, a, dims=1)\n1×4 Matrix{Int64}:\n 4  8  12  16\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.foldl","location":"base/collections.html#Base.foldl-Tuple{Any, Any}","category":"method","text":"foldl(op, itr; [init])\n\nLike reduce, but with guaranteed left associativity. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\nSee also mapfoldl, foldr, accumulate.\n\nExamples\n\njulia> foldl(=>, 1:4)\n((1 => 2) => 3) => 4\n\njulia> foldl(=>, 1:4; init=0)\n(((0 => 1) => 2) => 3) => 4\n\njulia> accumulate(=>, (1,2,3,4))\n(1, 1 => 2, (1 => 2) => 3, ((1 => 2) => 3) => 4)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.foldr","location":"base/collections.html#Base.foldr-Tuple{Any, Any}","category":"method","text":"foldr(op, itr; [init])\n\nLike reduce, but with guaranteed right associativity. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\nExamples\n\njulia> foldr(=>, 1:4)\n1 => (2 => (3 => 4))\n\njulia> foldr(=>, 1:4; init=0)\n1 => (2 => (3 => (4 => 0)))\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.maximum","location":"base/collections.html#Base.maximum","category":"function","text":"maximum(f, itr; [init])\n\nReturn the largest result of calling function f on each element of itr.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for max (i.e. which is less than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> maximum(length, [\"Julion\", \"Julia\", \"Jule\"])\n6\n\njulia> maximum(length, []; init=-1)\n-1\n\njulia> maximum(sin, Real[]; init=-1.0)  # good, since output of sin is >= -1\n-1.0\n\n\n\n\n\nmaximum(itr; [init])\n\nReturn the largest element in a collection.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for max (i.e. which is less than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> maximum(-20.5:10)\n9.5\n\njulia> maximum([1,2,3])\n3\n\njulia> maximum(())\nERROR: MethodError: reducing over an empty collection is not allowed; consider supplying `init` to the reducer\nStacktrace:\n[...]\n\njulia> maximum((); init=-Inf)\n-Inf\n\n\n\n\n\nmaximum(A::AbstractArray; dims)\n\nCompute the maximum value of an array over the given dimensions. See also the max(a,b) function to take the maximum of two or more arguments, which can be applied elementwise to arrays via max.(a,b).\n\nSee also: maximum!, extrema, findmax, argmax.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum(A, dims=1)\n1×2 Matrix{Int64}:\n 3  4\n\njulia> maximum(A, dims=2)\n2×1 Matrix{Int64}:\n 2\n 4\n\n\n\n\n\nmaximum(f, A::AbstractArray; dims)\n\nCompute the maximum value by calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 9  16\n\njulia> maximum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n  4\n 16\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.maximum!","location":"base/collections.html#Base.maximum!","category":"function","text":"maximum!(r, A)\n\nCompute the maximum value of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum!([1; 1], A)\n2-element Vector{Int64}:\n 2\n 4\n\njulia> maximum!([1 1], A)\n1×2 Matrix{Int64}:\n 3  4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.minimum","location":"base/collections.html#Base.minimum","category":"function","text":"minimum(f, itr; [init])\n\nReturn the smallest result of calling function f on each element of itr.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for min (i.e. which is greater than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> minimum(length, [\"Julion\", \"Julia\", \"Jule\"])\n4\n\njulia> minimum(length, []; init=typemax(Int64))\n9223372036854775807\n\njulia> minimum(sin, Real[]; init=1.0)  # good, since output of sin is <= 1\n1.0\n\n\n\n\n\nminimum(itr; [init])\n\nReturn the smallest element in a collection.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for min (i.e. which is greater than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> minimum(-20.5:10)\n-20.5\n\njulia> minimum([1,2,3])\n1\n\njulia> minimum([])\nERROR: MethodError: reducing over an empty collection is not allowed; consider supplying `init` to the reducer\nStacktrace:\n[...]\n\njulia> minimum([]; init=Inf)\nInf\n\n\n\n\n\nminimum(A::AbstractArray; dims)\n\nCompute the minimum value of an array over the given dimensions. See also the min(a,b) function to take the minimum of two or more arguments, which can be applied elementwise to arrays via min.(a,b).\n\nSee also: minimum!, extrema, findmin, argmin.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum(A, dims=1)\n1×2 Matrix{Int64}:\n 1  2\n\njulia> minimum(A, dims=2)\n2×1 Matrix{Int64}:\n 1\n 3\n\n\n\n\n\nminimum(f, A::AbstractArray; dims)\n\nCompute the minimum value by calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 1  4\n\njulia> minimum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n 1\n 9\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.minimum!","location":"base/collections.html#Base.minimum!","category":"function","text":"minimum!(r, A)\n\nCompute the minimum value of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum!([1; 1], A)\n2-element Vector{Int64}:\n 1\n 3\n\njulia> minimum!([1 1], A)\n1×2 Matrix{Int64}:\n 1  2\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.extrema","location":"base/collections.html#Base.extrema","category":"function","text":"extrema(itr; [init]) -> (mn, mx)\n\nCompute both the minimum mn and maximum mx element in a single pass, and return them as a 2-tuple.\n\nThe value returned for empty itr can be specified by init. It must be a 2-tuple whose first and second elements are neutral elements for min and max respectively (i.e. which are greater/less than or equal to any other element). As a consequence, when itr is empty the returned (mn, mx) tuple will satisfy mn ≥ mx. When init is specified it may be used even for non-empty itr.\n\ncompat: Julia 1.8\nKeyword argument init requires Julia 1.8 or later.\n\nExamples\n\njulia> extrema(2:10)\n(2, 10)\n\njulia> extrema([9,pi,4.5])\n(3.141592653589793, 9.0)\n\njulia> extrema([]; init = (Inf, -Inf))\n(Inf, -Inf)\n\n\n\n\n\nextrema(f, itr; [init]) -> (mn, mx)\n\nCompute both the minimum mn and maximum mx of f applied to each element in itr and return them as a 2-tuple. Only one pass is made over itr.\n\nThe value returned for empty itr can be specified by init. It must be a 2-tuple whose first and second elements are neutral elements for min and max respectively (i.e. which are greater/less than or equal to any other element). It is used for non-empty collections. Note: it implies that, for empty itr, the returned value (mn, mx) satisfies mn ≥ mx even though for non-empty itr it  satisfies mn ≤ mx.  This is a \"paradoxical\" but yet expected result.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\ncompat: Julia 1.8\nKeyword argument init requires Julia 1.8 or later.\n\nExamples\n\njulia> extrema(sin, 0:π)\n(0.0, 0.9092974268256817)\n\njulia> extrema(sin, Real[]; init = (1.0, -1.0))  # good, since -1 ≤ sin(::Real) ≤ 1\n(1.0, -1.0)\n\n\n\n\n\nextrema(A::AbstractArray; dims) -> Array{Tuple}\n\nCompute the minimum and maximum elements of an array over the given dimensions.\n\nSee also: minimum, maximum, extrema!.\n\nExamples\n\njulia> A = reshape(Vector(1:2:16), (2,2,2))\n2×2×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  5\n 3  7\n\n[:, :, 2] =\n  9  13\n 11  15\n\njulia> extrema(A, dims = (1,2))\n1×1×2 Array{Tuple{Int64, Int64}, 3}:\n[:, :, 1] =\n (1, 7)\n\n[:, :, 2] =\n (9, 15)\n\n\n\n\n\nextrema(f, A::AbstractArray; dims) -> Array{Tuple}\n\nCompute the minimum and maximum of f applied to each element in the given dimensions of A.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.extrema!","location":"base/collections.html#Base.extrema!","category":"function","text":"extrema!(r, A)\n\nCompute the minimum and maximum value of A over the singleton dimensions of r, and write results to r.\n\ncompat: Julia 1.8\nThis method requires Julia 1.8 or later.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> extrema!([(1, 1); (1, 1)], A)\n2-element Vector{Tuple{Int64, Int64}}:\n (1, 2)\n (3, 4)\n\njulia> extrema!([(1, 1);; (1, 1)], A)\n1×2 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (2, 4)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.argmax","location":"base/collections.html#Base.argmax","category":"function","text":"argmax(r::AbstractRange)\n\nRanges can have multiple maximal elements. In that case argmax will return a maximal index, but not necessarily the first one.\n\n\n\n\n\nargmax(f, domain)\n\nReturn a value x from domain for which f(x) is maximised. If there are multiple maximal values for f(x) then the first one will be found.\n\ndomain must be a non-empty iterable.\n\nValues are compared with isless.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nSee also argmin, findmax.\n\nExamples\n\njulia> argmax(abs, -10:5)\n-10\n\njulia> argmax(cos, 0:π/2:2π)\n0.0\n\n\n\n\n\nargmax(itr)\n\nReturn the index or key of the maximal element in a collection. If there are multiple maximal elements, then the first one will be returned.\n\nThe collection must not be empty.\n\nValues are compared with isless.\n\nSee also: argmin, findmax.\n\nExamples\n\njulia> argmax([8, 0.1, -9, pi])\n1\n\njulia> argmax([1, 7, 7, 6])\n2\n\njulia> argmax([1, 7, 7, NaN])\n4\n\n\n\n\n\nargmax(A; dims) -> indices\n\nFor an array input, return the indices of the maximum elements over the given dimensions. NaN is treated as greater than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> argmax(A, dims=1)\n1×2 Matrix{CartesianIndex{2}}:\n CartesianIndex(2, 1)  CartesianIndex(2, 2)\n\njulia> argmax(A, dims=2)\n2×1 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 2)\n CartesianIndex(2, 2)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.argmin","location":"base/collections.html#Base.argmin","category":"function","text":"argmin(r::AbstractRange)\n\nRanges can have multiple minimal elements. In that case argmin will return a minimal index, but not necessarily the first one.\n\n\n\n\n\nargmin(f, domain)\n\nReturn a value x from domain for which f(x) is minimised. If there are multiple minimal values for f(x) then the first one will be found.\n\ndomain must be a non-empty iterable.\n\nNaN is treated as less than all other values except missing.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nSee also argmax, findmin.\n\nExamples\n\njulia> argmin(sign, -10:5)\n-10\n\njulia> argmin(x -> -x^3 + x^2 - 10, -5:5)\n5\n\njulia> argmin(acos, 0:0.1:1)\n1.0\n\n\n\n\n\nargmin(itr)\n\nReturn the index or key of the minimal element in a collection. If there are multiple minimal elements, then the first one will be returned.\n\nThe collection must not be empty.\n\nNaN is treated as less than all other values except missing.\n\nSee also: argmax, findmin.\n\nExamples\n\njulia> argmin([8, 0.1, -9, pi])\n3\n\njulia> argmin([7, 1, 1, 6])\n2\n\njulia> argmin([7, 1, 1, NaN])\n4\n\n\n\n\n\nargmin(A; dims) -> indices\n\nFor an array input, return the indices of the minimum elements over the given dimensions. NaN is treated as less than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> argmin(A, dims=1)\n1×2 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 1)  CartesianIndex(1, 2)\n\njulia> argmin(A, dims=2)\n2×1 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 1)\n CartesianIndex(2, 1)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.findmax","location":"base/collections.html#Base.findmax","category":"function","text":"findmax(f, domain) -> (f(x), index)\n\nReturn a pair of a value in the codomain (outputs of f) and the index of the corresponding value in the domain (inputs to f) such that f(x) is maximised. If there are multiple maximal points, then the first one will be returned.\n\ndomain must be a non-empty iterable.\n\nValues are compared with isless.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nExamples\n\njulia> findmax(identity, 5:9)\n(9, 5)\n\njulia> findmax(-, 1:10)\n(-1, 1)\n\njulia> findmax(first, [(1, :a), (3, :b), (3, :c)])\n(3, 2)\n\njulia> findmax(cos, 0:π/2:2π)\n(1.0, 1)\n\n\n\n\n\nfindmax(itr) -> (x, index)\n\nReturn the maximal element of the collection itr and its index or key. If there are multiple maximal elements, then the first one will be returned. Values are compared with isless.\n\nSee also: findmin, argmax, maximum.\n\nExamples\n\njulia> findmax([8, 0.1, -9, pi])\n(8.0, 1)\n\njulia> findmax([1, 7, 7, 6])\n(7, 2)\n\njulia> findmax([1, 7, 7, NaN])\n(NaN, 4)\n\n\n\n\n\nfindmax(A; dims) -> (maxval, index)\n\nFor an array input, returns the value and index of the maximum over the given dimensions. NaN is treated as greater than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> findmax(A, dims=1)\n([3.0 4.0], CartesianIndex{2}[CartesianIndex(2, 1) CartesianIndex(2, 2)])\n\njulia> findmax(A, dims=2)\n([2.0; 4.0;;], CartesianIndex{2}[CartesianIndex(1, 2); CartesianIndex(2, 2);;])\n\n\n\n\n\nfindmax(f, A; dims) -> (f(x), index)\n\nFor an array input, returns the value in the codomain and index of the corresponding value which maximize f over the given dimensions.\n\nExamples\n\njulia> A = [-1.0 1; -0.5 2]\n2×2 Matrix{Float64}:\n -1.0  1.0\n -0.5  2.0\n\njulia> findmax(abs2, A, dims=1)\n([1.0 4.0], CartesianIndex{2}[CartesianIndex(1, 1) CartesianIndex(2, 2)])\n\njulia> findmax(abs2, A, dims=2)\n([1.0; 4.0;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 2);;])\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.findmin","location":"base/collections.html#Base.findmin","category":"function","text":"findmin(f, domain) -> (f(x), index)\n\nReturn a pair of a value in the codomain (outputs of f) and the index of the corresponding value in the domain (inputs to f) such that f(x) is minimised. If there are multiple minimal points, then the first one will be returned.\n\ndomain must be a non-empty iterable.\n\nNaN is treated as less than all other values except missing.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nExamples\n\njulia> findmin(identity, 5:9)\n(5, 1)\n\njulia> findmin(-, 1:10)\n(-10, 10)\n\njulia> findmin(first, [(2, :a), (2, :b), (3, :c)])\n(2, 1)\n\njulia> findmin(cos, 0:π/2:2π)\n(-1.0, 3)\n\n\n\n\n\nfindmin(itr) -> (x, index)\n\nReturn the minimal element of the collection itr and its index or key. If there are multiple minimal elements, then the first one will be returned. NaN is treated as less than all other values except missing.\n\nSee also: findmax, argmin, minimum.\n\nExamples\n\njulia> findmin([8, 0.1, -9, pi])\n(-9.0, 3)\n\njulia> findmin([1, 7, 7, 6])\n(1, 1)\n\njulia> findmin([1, 7, 7, NaN])\n(NaN, 4)\n\n\n\n\n\nfindmin(A; dims) -> (minval, index)\n\nFor an array input, returns the value and index of the minimum over the given dimensions. NaN is treated as less than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> findmin(A, dims=1)\n([1.0 2.0], CartesianIndex{2}[CartesianIndex(1, 1) CartesianIndex(1, 2)])\n\njulia> findmin(A, dims=2)\n([1.0; 3.0;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 1);;])\n\n\n\n\n\nfindmin(f, A; dims) -> (f(x), index)\n\nFor an array input, returns the value in the codomain and index of the corresponding value which minimize f over the given dimensions.\n\nExamples\n\njulia> A = [-1.0 1; -0.5 2]\n2×2 Matrix{Float64}:\n -1.0  1.0\n -0.5  2.0\n\njulia> findmin(abs2, A, dims=1)\n([0.25 1.0], CartesianIndex{2}[CartesianIndex(2, 1) CartesianIndex(1, 2)])\n\njulia> findmin(abs2, A, dims=2)\n([1.0; 0.25;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 1);;])\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.findmax!","location":"base/collections.html#Base.findmax!","category":"function","text":"findmax!(rval, rind, A) -> (maxval, index)\n\nFind the maximum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind. NaN is treated as greater than all other values except missing.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.findmin!","location":"base/collections.html#Base.findmin!","category":"function","text":"findmin!(rval, rind, A) -> (minval, index)\n\nFind the minimum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind. NaN is treated as less than all other values except missing.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.sum","location":"base/collections.html#Base.sum","category":"function","text":"sum(f, itr; [init])\n\nSum the results of calling function f on each element of itr.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the additive identity (i.e. zero) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> sum(abs2, [2; 3; 4])\n29\n\nNote the important difference between sum(A) and reduce(+, A) for arrays with small integer eltype:\n\njulia> sum(Int8[100, 28])\n128\n\njulia> reduce(+, Int8[100, 28])\n-128\n\nIn the former case, the integers are widened to system word size and therefore the result is 128. In the latter case, no such widening happens and integer overflow results in -128.\n\n\n\n\n\nsum(itr; [init])\n\nReturn the sum of all elements in a collection.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the additive identity (i.e. zero) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nSee also: reduce, mapreduce, count, union.\n\nExamples\n\njulia> sum(1:20)\n210\n\njulia> sum(1:20; init = 0.0)\n210.0\n\n\n\n\n\nsum(A::AbstractArray; dims)\n\nSum elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum(A, dims=1)\n1×2 Matrix{Int64}:\n 4  6\n\njulia> sum(A, dims=2)\n2×1 Matrix{Int64}:\n 3\n 7\n\n\n\n\n\nsum(f, A::AbstractArray; dims)\n\nSum the results of calling function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 10  20\n\njulia> sum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n  5\n 25\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.sum!","location":"base/collections.html#Base.sum!","category":"function","text":"sum!(r, A)\n\nSum elements of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum!([1; 1], A)\n2-element Vector{Int64}:\n 3\n 7\n\njulia> sum!([1 1], A)\n1×2 Matrix{Int64}:\n 4  6\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.prod","location":"base/collections.html#Base.prod","category":"function","text":"prod(f, itr; [init])\n\nReturn the product of f applied to each element of itr.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the multiplicative identity (i.e. one) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> prod(abs2, [2; 3; 4])\n576\n\n\n\n\n\nprod(itr; [init])\n\nReturn the product of all elements of a collection.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the multiplicative identity (i.e. one) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nSee also: reduce, cumprod, any.\n\nExamples\n\njulia> prod(1:5)\n120\n\njulia> prod(1:5; init = 1.0)\n120.0\n\n\n\n\n\nprod(A::AbstractArray; dims)\n\nMultiply elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod(A, dims=1)\n1×2 Matrix{Int64}:\n 3  8\n\njulia> prod(A, dims=2)\n2×1 Matrix{Int64}:\n  2\n 12\n\n\n\n\n\nprod(f, A::AbstractArray; dims)\n\nMultiply the results of calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 9  64\n\njulia> prod(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n   4\n 144\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.prod!","location":"base/collections.html#Base.prod!","category":"function","text":"prod!(r, A)\n\nMultiply elements of A over the singleton dimensions of r, and write results to r.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod!([1; 1], A)\n2-element Vector{Int64}:\n  2\n 12\n\njulia> prod!([1 1], A)\n1×2 Matrix{Int64}:\n 3  8\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.any","location":"base/collections.html#Base.any-Tuple{Any}","category":"method","text":"any(itr) -> Bool\n\nTest whether any elements of a boolean collection are true, returning true as soon as the first true value in itr is encountered (short-circuiting). To short-circuit on false, use all.\n\nIf the input contains missing values, return missing if all non-missing values are false (or equivalently, if the input contains no true value), following three-valued logic.\n\nSee also: all, count, sum, |, , ||.\n\nExamples\n\njulia> a = [true,false,false,true]\n4-element Vector{Bool}:\n 1\n 0\n 0\n 1\n\njulia> any(a)\ntrue\n\njulia> any((println(i); v) for (i, v) in enumerate(a))\n1\ntrue\n\njulia> any([missing, true])\ntrue\n\njulia> any([false, missing])\nmissing\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.any","location":"base/collections.html#Base.any-Tuple{AbstractArray, Any}","category":"method","text":"any(p, itr) -> Bool\n\nDetermine whether predicate p returns true for any elements of itr, returning true as soon as the first item in itr for which p returns true is encountered (short-circuiting). To short-circuit on false, use all.\n\nIf the input contains missing values, return missing if all non-missing values are false (or equivalently, if the input contains no true value), following three-valued logic.\n\nExamples\n\njulia> any(i->(4<=i<=6), [3,5,7])\ntrue\n\njulia> any(i -> (println(i); i > 3), 1:10)\n1\n2\n3\n4\ntrue\n\njulia> any(i -> i > 0, [1, missing])\ntrue\n\njulia> any(i -> i > 0, [-1, missing])\nmissing\n\njulia> any(i -> i > 0, [-1, 0])\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.any!","location":"base/collections.html#Base.any!","category":"function","text":"any!(r, A)\n\nTest whether any values in A along the singleton dimensions of r are true, and write results to r.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Matrix{Bool}:\n 1  0\n 1  0\n\njulia> any!([1; 1], A)\n2-element Vector{Int64}:\n 1\n 1\n\njulia> any!([1 1], A)\n1×2 Matrix{Int64}:\n 1  0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.all","location":"base/collections.html#Base.all-Tuple{Any}","category":"method","text":"all(itr) -> Bool\n\nTest whether all elements of a boolean collection are true, returning false as soon as the first false value in itr is encountered (short-circuiting). To short-circuit on true, use any.\n\nIf the input contains missing values, return missing if all non-missing values are true (or equivalently, if the input contains no false value), following three-valued logic.\n\nSee also: all!, any, count, &, , &&, allunique.\n\nExamples\n\njulia> a = [true,false,false,true]\n4-element Vector{Bool}:\n 1\n 0\n 0\n 1\n\njulia> all(a)\nfalse\n\njulia> all((println(i); v) for (i, v) in enumerate(a))\n1\n2\nfalse\n\njulia> all([missing, false])\nfalse\n\njulia> all([true, missing])\nmissing\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.all","location":"base/collections.html#Base.all-Tuple{AbstractArray, Any}","category":"method","text":"all(p, itr) -> Bool\n\nDetermine whether predicate p returns true for all elements of itr, returning false as soon as the first item in itr for which p returns false is encountered (short-circuiting). To short-circuit on true, use any.\n\nIf the input contains missing values, return missing if all non-missing values are true (or equivalently, if the input contains no false value), following three-valued logic.\n\nExamples\n\njulia> all(i->(4<=i<=6), [4,5,6])\ntrue\n\njulia> all(i -> (println(i); i < 3), 1:10)\n1\n2\n3\nfalse\n\njulia> all(i -> i > 0, [1, missing])\nmissing\n\njulia> all(i -> i > 0, [-1, missing])\nfalse\n\njulia> all(i -> i > 0, [1, 2])\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.all!","location":"base/collections.html#Base.all!","category":"function","text":"all!(r, A)\n\nTest whether all values in A along the singleton dimensions of r are true, and write results to r.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Matrix{Bool}:\n 1  0\n 1  0\n\njulia> all!([1; 1], A)\n2-element Vector{Int64}:\n 0\n 0\n\njulia> all!([1 1], A)\n1×2 Matrix{Int64}:\n 1  0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.count","location":"base/collections.html#Base.count","category":"function","text":"count([f=identity,] itr; init=0) -> Integer\n\nCount the number of elements in itr for which the function f returns true. If f is omitted, count the number of true elements in itr (which should be a collection of boolean values). init optionally specifies the value to start counting from and therefore also determines the output type.\n\ncompat: Julia 1.6\ninit keyword was added in Julia 1.6.\n\nSee also: any, sum.\n\nExamples\n\njulia> count(i->(4<=i<=6), [2,3,4,5,6])\n3\n\njulia> count([true, false, true, true])\n3\n\njulia> count(>(3), 1:7, init=0x03)\n0x07\n\n\n\n\n\ncount(\n    pattern::Union{AbstractChar,AbstractString,AbstractPattern},\n    string::AbstractString;\n    overlap::Bool = false,\n)\n\nReturn the number of matches for pattern in string. This is equivalent to calling length(findall(pattern, string)) but more efficient.\n\nIf overlap=true, the matching sequences are allowed to overlap indices in the original string, otherwise they must be from disjoint character ranges.\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\ncompat: Julia 1.7\nUsing a character as the pattern requires at least Julia 1.7.\n\nExamples\n\njulia> count('a', \"JuliaLang\")\n2\n\njulia> count(r\"a(.)a\", \"cabacabac\", overlap=true)\n3\n\njulia> count(r\"a(.)a\", \"cabacabac\")\n2\n\n\n\n\n\ncount([f=identity,] A::AbstractArray; dims=:)\n\nCount the number of elements in A for which f returns true over the given dimensions.\n\ncompat: Julia 1.5\ndims keyword was added in Julia 1.5.\n\ncompat: Julia 1.6\ninit keyword was added in Julia 1.6.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> count(<=(2), A, dims=1)\n1×2 Matrix{Int64}:\n 1  1\n\njulia> count(<=(2), A, dims=2)\n2×1 Matrix{Int64}:\n 2\n 0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.foreach","location":"base/collections.html#Base.foreach","category":"function","text":"foreach(f, c...) -> Nothing\n\nCall function f on each element of iterable c. For multiple iterable arguments, f is called elementwise, and iteration stops when any iterator is finished.\n\nforeach should be used instead of map when the results of f are not needed, for example in foreach(println, array).\n\nExamples\n\njulia> tri = 1:3:7; res = Int[];\n\njulia> foreach(x -> push!(res, x^2), tri)\n\njulia> res\n3-element Vector{Int32}:\n  1\n 16\n 49\n\njulia> foreach((x, y) -> println(x, \" with \", y), tri, 'a':'z')\n1 with a\n4 with b\n7 with c\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.map","location":"base/collections.html#Base.map","category":"function","text":"map(f, c...) -> collection\n\nTransform collection c by applying f to each element. For multiple collection arguments, apply f elementwise, and stop when when any of them is exhausted.\n\nSee also map!, foreach, mapreduce, mapslices, zip, Iterators.map.\n\nExamples\n\njulia> map(x -> x * 2, [1, 2, 3])\n3-element Vector{Int64}:\n 2\n 4\n 6\n\njulia> map(+, [1, 2, 3], [10, 20, 30, 400, 5000])\n3-element Vector{Int64}:\n 11\n 22\n 33\n\n\n\n\n\nmap(f, A::AbstractArray...) -> N-array\n\nWhen acting on multi-dimensional arrays of the same ndims, they must all have the same axes, and the answer will too.\n\nSee also broadcast, which allows mismatched sizes.\n\nExamples\n\njulia> map(//, [1 2; 3 4], [4 3; 2 1])\n2×2 Matrix{Rational{Int32}}:\n 1//4  2//3\n 3//2  4//1\n\njulia> map(+, [1 2; 3 4], zeros(2,1))\nERROR: DimensionMismatch\n\njulia> map(+, [1 2; 3 4], [1,10,100,1000], zeros(3,1))  # iterates until 3rd is exhausted\n3-element Vector{Float64}:\n   2.0\n  13.0\n 102.0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.map!","location":"base/collections.html#Base.map!","category":"function","text":"map!(function, destination, collection...)\n\nLike map, but stores the result in destination rather than a new collection. destination must be at least as large as the smallest collection.\n\nSee also: map, foreach, zip, copyto!.\n\nExamples\n\njulia> a = zeros(3);\n\njulia> map!(x -> x * 2, a, [1, 2, 3]);\n\njulia> a\n3-element Vector{Float64}:\n 2.0\n 4.0\n 6.0\n\njulia> map!(+, zeros(Int, 5), 100:999, 1:3)\n5-element Vector{Int32}:\n 101\n 103\n 105\n   0\n   0\n\n\n\n\n\nmap!(f, values(dict::AbstractDict))\n\nModifies dict by transforming each value from val to f(val). Note that the type of dict cannot be changed: if f(val) is not an instance of the value type of dict then it will be converted to the value type if possible and otherwise raise an error.\n\ncompat: Julia 1.2\nmap!(f, values(dict::AbstractDict)) requires Julia 1.2 or later.\n\nExamples\n\njulia> d = Dict(:a => 1, :b => 2)\nDict{Symbol, Int64} with 2 entries:\n  :a => 1\n  :b => 2\n\njulia> map!(v -> v-1, values(d))\nValueIterator for a Dict{Symbol, Int64} with 2 entries. Values:\n  0\n  1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.mapreduce","location":"base/collections.html#Base.mapreduce-Tuple{Any, Any, Any}","category":"method","text":"mapreduce(f, op, itrs...; [init])\n\nApply function f to each element(s) in itrs, and then reduce the result using the binary function op. If provided, init must be a neutral element for op that will be returned for empty collections. It is unspecified whether init is used for non-empty collections. In general, it will be necessary to provide init to work with empty collections.\n\nmapreduce is functionally equivalent to calling reduce(op, map(f, itr); init=init), but will in general execute faster since no intermediate collection needs to be created. See documentation for reduce and map.\n\ncompat: Julia 1.2\nmapreduce with multiple iterators requires Julia 1.2 or later.\n\nExamples\n\njulia> mapreduce(x->x^2, +, [1:3;]) # == 1 + 4 + 9\n14\n\nThe associativity of the reduction is implementation-dependent. Additionally, some implementations may reuse the return value of f for elements that appear multiple times in itr. Use mapfoldl or mapfoldr instead for guaranteed left or right associativity and invocation of f for every value.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.mapfoldl","location":"base/collections.html#Base.mapfoldl-Tuple{Any, Any, Any}","category":"method","text":"mapfoldl(f, op, itr; [init])\n\nLike mapreduce, but with guaranteed left associativity, as in foldl. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.mapfoldr","location":"base/collections.html#Base.mapfoldr-Tuple{Any, Any, Any}","category":"method","text":"mapfoldr(f, op, itr; [init])\n\nLike mapreduce, but with guaranteed right associativity, as in foldr. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.first","location":"base/collections.html#Base.first","category":"function","text":"first(coll)\n\nGet the first element of an iterable collection. Return the start point of an AbstractRange even if it is empty.\n\nSee also: only, firstindex, last.\n\nExamples\n\njulia> first(2:2:10)\n2\n\njulia> first([1; 2; 3; 4])\n1\n\n\n\n\n\nfirst(itr, n::Integer)\n\nGet the first n elements of the iterable collection itr, or fewer elements if itr is not long enough.\n\nSee also: startswith, Iterators.take.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\nExamples\n\njulia> first([\"foo\", \"bar\", \"qux\"], 2)\n2-element Vector{String}:\n \"foo\"\n \"bar\"\n\njulia> first(1:6, 10)\n1:6\n\njulia> first(Bool[], 1)\nBool[]\n\n\n\n\n\nfirst(s::AbstractString, n::Integer)\n\nGet a string consisting of the first n characters of s.\n\nExamples\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 1)\n\"∀\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 3)\n\"∀ϵ≠\"\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.last","location":"base/collections.html#Base.last","category":"function","text":"last(coll)\n\nGet the last element of an ordered collection, if it can be computed in O(1) time. This is accomplished by calling lastindex to get the last index. Return the end point of an AbstractRange even if it is empty.\n\nSee also first, endswith.\n\nExamples\n\njulia> last(1:2:10)\n9\n\njulia> last([1; 2; 3; 4])\n4\n\n\n\n\n\nlast(itr, n::Integer)\n\nGet the last n elements of the iterable collection itr, or fewer elements if itr is not long enough.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\nExamples\n\njulia> last([\"foo\", \"bar\", \"qux\"], 2)\n2-element Vector{String}:\n \"bar\"\n \"qux\"\n\njulia> last(1:6, 10)\n1:6\n\njulia> last(Float64[], 1)\nFloat64[]\n\n\n\n\n\nlast(s::AbstractString, n::Integer)\n\nGet a string consisting of the last n characters of s.\n\nExamples\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 1)\n\"0\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 3)\n\"²>0\"\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.front","location":"base/collections.html#Base.front","category":"function","text":"front(x::Tuple)::Tuple\n\nReturn a Tuple consisting of all but the last component of x.\n\nSee also: first, tail.\n\nExamples\n\njulia> Base.front((1,2,3))\n(1, 2)\n\njulia> Base.front(())\nERROR: ArgumentError: Cannot call front on an empty tuple.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.tail","location":"base/collections.html#Base.tail","category":"function","text":"tail(x::Tuple)::Tuple\n\nReturn a Tuple consisting of all but the first component of x.\n\nSee also: front, rest, first, Iterators.peel.\n\nExamples\n\njulia> Base.tail((1,2,3))\n(2, 3)\n\njulia> Base.tail(())\nERROR: ArgumentError: Cannot call tail on an empty tuple.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.step","location":"base/collections.html#Base.step","category":"function","text":"step(r)\n\nGet the step size of an AbstractRange object.\n\nExamples\n\njulia> step(1:10)\n1\n\njulia> step(1:2:10)\n2\n\njulia> step(2.5:0.3:10.9)\n0.3\n\njulia> step(range(2.5, stop=10.9, length=85))\n0.1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.collect","location":"base/collections.html#Base.collect-Tuple{Any}","category":"method","text":"collect(collection)\n\nReturn an Array of all items in a collection or iterator. For dictionaries, returns Pair{KeyType, ValType}. If the argument is array-like or is an iterator with the HasShape trait, the result will have the same shape and number of dimensions as the argument.\n\nUsed by comprehensions to turn a generator into an Array.\n\nExamples\n\njulia> collect(1:2:13)\n7-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n 13\n\njulia> [x^2 for x in 1:8 if isodd(x)]\n4-element Vector{Int64}:\n  1\n  9\n 25\n 49\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.collect","location":"base/collections.html#Base.collect-Tuple{Type, Any}","category":"method","text":"collect(element_type, collection)\n\nReturn an Array with the given element type of all items in a collection or iterable. The result has the same shape and number of dimensions as collection.\n\nExamples\n\njulia> collect(Float64, 1:2:5)\n3-element Vector{Float64}:\n 1.0\n 3.0\n 5.0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.filter","location":"base/collections.html#Base.filter","category":"function","text":"filter(f, a)\n\nReturn a copy of collection a, removing elements for which f is false. The function f is passed one argument.\n\ncompat: Julia 1.4\nSupport for a as a tuple requires at least Julia 1.4.\n\nSee also: filter!, Iterators.filter.\n\nExamples\n\njulia> a = 1:10\n1:10\n\njulia> filter(isodd, a)\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\n\n\n\n\nfilter(f)\n\nCreate a function that filters its arguments with function f using filter, i.e. a function equivalent to x -> filter(f, x).\n\nThe returned function is of type Base.Fix1{typeof(filter)}, which can be used to implement specialized methods.\n\nExamples\n\njulia> (1, 2, Inf, 4, NaN, 6) |> filter(isfinite)\n(1, 2, 4, 6)\n\njulia> map(filter(iseven), [1:3, 2:4, 3:5])\n3-element Vector{Vector{Int64}}:\n [2]\n [2, 4]\n [4]\n\ncompat: Julia 1.9\nThis method requires at least Julia 1.9.\n\n\n\n\n\nfilter(f, d::AbstractDict)\n\nReturn a copy of d, removing elements for which f is false. The function f is passed key=>value pairs.\n\nExamples\n\njulia> d = Dict(1=>\"a\", 2=>\"b\")\nDict{Int64, String} with 2 entries:\n  2 => \"b\"\n  1 => \"a\"\n\njulia> filter(p->isodd(p.first), d)\nDict{Int64, String} with 1 entry:\n  1 => \"a\"\n\n\n\n\n\nfilter(f, itr::SkipMissing{<:AbstractArray})\n\nReturn a vector similar to the array wrapped by the given SkipMissing iterator but with all missing elements and those for which f returns false removed.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\nExamples\n\njulia> x = [1 2; missing 4]\n2×2 Matrix{Union{Missing, Int64}}:\n 1         2\n  missing  4\n\njulia> filter(isodd, skipmissing(x))\n1-element Vector{Int64}:\n 1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.filter!","location":"base/collections.html#Base.filter!","category":"function","text":"filter!(f, a)\n\nUpdate collection a, removing elements for which f is false. The function f is passed one argument.\n\nExamples\n\njulia> filter!(isodd, Vector(1:10))\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\n\n\n\n\nfilter!(f, d::AbstractDict)\n\nUpdate d, removing elements for which f is false. The function f is passed key=>value pairs.\n\nExample\n\njulia> d = Dict(1=>\"a\", 2=>\"b\", 3=>\"c\")\nDict{Int64, String} with 3 entries:\n  2 => \"b\"\n  3 => \"c\"\n  1 => \"a\"\n\njulia> filter!(p->isodd(p.first), d)\nDict{Int64, String} with 2 entries:\n  3 => \"c\"\n  1 => \"a\"\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.replace","location":"base/collections.html#Base.replace-Tuple{Any, Vararg{Pair}}","category":"method","text":"replace(A, old_new::Pair...; [count::Integer])\n\nReturn a copy of collection A where, for each pair old=>new in old_new, all occurrences of old are replaced by new. Equality is determined using isequal. If count is specified, then replace at most count occurrences in total.\n\nThe element type of the result is chosen using promotion (see promote_type) based on the element type of A and on the types of the new values in pairs. If count is omitted and the element type of A is a Union, the element type of the result will not include singleton types which are replaced with values of a different type: for example, Union{T,Missing} will become T if missing is replaced.\n\nSee also replace!, splice!, delete!, insert!.\n\ncompat: Julia 1.7\nVersion 1.7 is required to replace elements of a Tuple.\n\nExamples\n\njulia> replace([1, 2, 1, 3], 1=>0, 2=>4, count=2)\n4-element Vector{Int64}:\n 0\n 4\n 1\n 3\n\njulia> replace([1, missing], missing=>0)\n2-element Vector{Int64}:\n 1\n 0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.replace","location":"base/collections.html#Base.replace-Tuple{Union{Function, Type}, Any}","category":"method","text":"replace(new::Function, A; [count::Integer])\n\nReturn a copy of A where each value x in A is replaced by new(x). If count is specified, then replace at most count values in total (replacements being defined as new(x) !== x).\n\ncompat: Julia 1.7\nVersion 1.7 is required to replace elements of a Tuple.\n\nExamples\n\njulia> replace(x -> isodd(x) ? 2x : x, [1, 2, 3, 4])\n4-element Vector{Int64}:\n 2\n 2\n 6\n 4\n\njulia> replace(Dict(1=>2, 3=>4)) do kv\n           first(kv) < 3 ? first(kv)=>3 : kv\n       end\nDict{Int64, Int64} with 2 entries:\n  3 => 4\n  1 => 3\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.replace!","location":"base/collections.html#Base.replace!","category":"function","text":"replace!(A, old_new::Pair...; [count::Integer])\n\nFor each pair old=>new in old_new, replace all occurrences of old in collection A by new. Equality is determined using isequal. If count is specified, then replace at most count occurrences in total. See also replace.\n\nExamples\n\njulia> replace!([1, 2, 1, 3], 1=>0, 2=>4, count=2)\n4-element Vector{Int64}:\n 0\n 4\n 1\n 3\n\njulia> replace!(Set([1, 2, 3]), 1=>0)\nSet{Int64} with 3 elements:\n  0\n  2\n  3\n\n\n\n\n\nreplace!(new::Function, A; [count::Integer])\n\nReplace each element x in collection A by new(x). If count is specified, then replace at most count values in total (replacements being defined as new(x) !== x).\n\nExamples\n\njulia> replace!(x -> isodd(x) ? 2x : x, [1, 2, 3, 4])\n4-element Vector{Int64}:\n 2\n 2\n 6\n 4\n\njulia> replace!(Dict(1=>2, 3=>4)) do kv\n           first(kv) < 3 ? first(kv)=>3 : kv\n       end\nDict{Int64, Int64} with 2 entries:\n  3 => 4\n  1 => 3\n\njulia> replace!(x->2x, Set([3, 6]))\nSet{Int64} with 2 elements:\n  6\n  12\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.rest","location":"base/collections.html#Base.rest","category":"function","text":"Base.rest(collection[, itr_state])\n\nGeneric function for taking the tail of collection, starting from a specific iteration state itr_state. Return a Tuple, if collection itself is a Tuple, a subtype of AbstractVector, if collection is an AbstractArray, a subtype of AbstractString if collection is an AbstractString, and an arbitrary iterator, falling back to Iterators.rest(collection[, itr_state]), otherwise.\n\nCan be overloaded for user-defined collection types to customize the behavior of slurping in assignments in final position, like a, b... = collection.\n\ncompat: Julia 1.6\nBase.rest requires at least Julia 1.6.\n\nSee also: first, Iterators.rest, Base.split_rest.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> first, state = iterate(a)\n(1, 2)\n\njulia> first, Base.rest(a, state)\n(1, [3, 2, 4])\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.split_rest","location":"base/collections.html#Base.split_rest","category":"function","text":"Base.split_rest(collection, n::Int[, itr_state]) -> (rest_but_n, last_n)\n\nGeneric function for splitting the tail of collection, starting from a specific iteration state itr_state. Returns a tuple of two new collections. The first one contains all elements of the tail but the n last ones, which make up the second collection.\n\nThe type of the first collection generally follows that of Base.rest, except that the fallback case is not lazy, but is collected eagerly into a vector.\n\nCan be overloaded for user-defined collection types to customize the behavior of slurping in assignments in non-final position, like a, b..., c = collection.\n\ncompat: Julia 1.9\nBase.split_rest requires at least Julia 1.9.\n\nSee also: Base.rest.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> first, state = iterate(a)\n(1, 2)\n\njulia> first, Base.split_rest(a, 1, state)\n(1, ([3, 2], [4]))\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Indexable Collections","location":"base/collections.html#Indexable-Collections","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.getindex\nBase.setindex!\nBase.firstindex\nBase.lastindex","page":"Collections and Data Structures"},{"title":"Base.getindex","location":"base/collections.html#Base.getindex","category":"function","text":"getindex(collection, key...)\n\nRetrieve the value(s) stored at the given key or index within a collection. The syntax a[i,j,...] is converted by the compiler to getindex(a, i, j, ...).\n\nSee also get, keys, eachindex.\n\nExamples\n\njulia> A = Dict(\"a\" => 1, \"b\" => 2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> getindex(A, \"a\")\n1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.setindex!","location":"base/collections.html#Base.setindex!","category":"function","text":"setindex!(collection, value, key...)\n\nStore the given value at the given key or index within a collection. The syntax a[i,j,...] = x is converted by the compiler to (setindex!(a, x, i, j, ...); x).\n\nExamples\n\njulia> a = Dict(\"a\"=>1)\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\njulia> setindex!(a, 2, \"b\")\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.firstindex","location":"base/collections.html#Base.firstindex","category":"function","text":"firstindex(collection) -> Integer\nfirstindex(collection, d) -> Integer\n\nReturn the first index of collection. If d is given, return the first index of collection along dimension d.\n\nThe syntaxes A[begin] and A[1, begin] lower to A[firstindex(A)] and A[1, firstindex(A, 2)], respectively.\n\nSee also: first, axes, lastindex, nextind.\n\nExamples\n\njulia> firstindex([1,2,4])\n1\n\njulia> firstindex(rand(3,4,5), 2)\n1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.lastindex","location":"base/collections.html#Base.lastindex","category":"function","text":"lastindex(collection) -> Integer\nlastindex(collection, d) -> Integer\n\nReturn the last index of collection. If d is given, return the last index of collection along dimension d.\n\nThe syntaxes A[end] and A[end, end] lower to A[lastindex(A)] and A[lastindex(A, 1), lastindex(A, 2)], respectively.\n\nSee also: axes, firstindex, eachindex, prevind.\n\nExamples\n\njulia> lastindex([1,2,4])\n3\n\njulia> lastindex(rand(3,4,5), 2)\n4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Array\nBitArray\nAbstractArray\nSubArray","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Partially implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"AbstractRange\nUnitRange\nTuple\nAbstractString\nDict\nIdDict\nWeakKeyDict\nNamedTuple","page":"Collections and Data Structures"},{"title":"Dictionaries","location":"base/collections.html#Dictionaries","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Dict is the standard dictionary. Its implementation uses hash as the hashing function for the key, and isequal to determine equality. Define these two functions for custom types to override how they are stored in a hash table.","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"IdDict is a special hash table where the keys are always object identities.","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"WeakKeyDict is a hash table implementation where the keys are weak references to objects, and thus may be garbage collected even when referenced in a hash table. Like Dict it uses hash for hashing and isequal for equality, unlike Dict it does not convert keys on insertion.","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Dicts can be created by passing pair objects constructed with => to a Dict constructor: Dict(\"A\"=>1, \"B\"=>2). This call will attempt to infer type information from the keys and values (i.e. this example creates a Dict{String, Int64}). To explicitly specify types use the syntax Dict{KeyType,ValueType}(...). For example, Dict{String,Int32}(\"A\"=>1, \"B\"=>2).","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Dictionaries may also be created with generators. For example, Dict(i => f(i) for i = 1:10).","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Given a dictionary D, the syntax D[x] returns the value of key x (if it exists) or throws an error, and D[x] = y stores the key-value pair x => y in D (replacing any existing value for the key x).  Multiple arguments to D[...] are converted to tuples; for example, the syntax D[x,y]  is equivalent to D[(x,y)], i.e. it refers to the value keyed by the tuple (x,y).","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.AbstractDict\nBase.Dict\nBase.IdDict\nBase.WeakKeyDict\nBase.ImmutableDict\nBase.haskey\nBase.get\nBase.get!\nBase.getkey\nBase.delete!\nBase.pop!(::Any, ::Any, ::Any)\nBase.keys\nBase.values\nBase.pairs\nBase.merge\nBase.mergewith\nBase.merge!\nBase.mergewith!\nBase.sizehint!\nBase.keytype\nBase.valtype","page":"Collections and Data Structures"},{"title":"Base.AbstractDict","location":"base/collections.html#Base.AbstractDict","category":"type","text":"AbstractDict{K, V}\n\nSupertype for dictionary-like types with keys of type K and values of type V. Dict, IdDict and other types are subtypes of this. An AbstractDict{K, V} should be an iterator of Pair{K, V}.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.Dict","location":"base/collections.html#Base.Dict","category":"type","text":"Dict([itr])\n\nDict{K,V}() constructs a hash table with keys of type K and values of type V. Keys are compared with isequal and hashed with hash.\n\nGiven a single iterable argument, constructs a Dict whose key-value pairs are taken from 2-tuples (key,value) generated by the argument.\n\nExamples\n\njulia> Dict([(\"A\", 1), (\"B\", 2)])\nDict{String, Int64} with 2 entries:\n  \"B\" => 2\n  \"A\" => 1\n\nAlternatively, a sequence of pair arguments may be passed.\n\njulia> Dict(\"A\"=>1, \"B\"=>2)\nDict{String, Int64} with 2 entries:\n  \"B\" => 2\n  \"A\" => 1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.IdDict","location":"base/collections.html#Base.IdDict","category":"type","text":"IdDict([itr])\n\nIdDict{K,V}() constructs a hash table using objectid as hash and === as equality with keys of type K and values of type V.\n\nSee Dict for further help. In the example below, The Dict keys are all isequal and therefore get hashed the same, so they get overwritten. The IdDict hashes by object-id, and thus preserves the 3 different keys.\n\nExamples\n\njulia> Dict(true => \"yes\", 1 => \"no\", 1.0 => \"maybe\")\nDict{Real, String} with 1 entry:\n  1.0 => \"maybe\"\n\njulia> IdDict(true => \"yes\", 1 => \"no\", 1.0 => \"maybe\")\nIdDict{Any, String} with 3 entries:\n  true => \"yes\"\n  1.0  => \"maybe\"\n  1    => \"no\"\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.WeakKeyDict","location":"base/collections.html#Base.WeakKeyDict","category":"type","text":"WeakKeyDict([itr])\n\nWeakKeyDict() constructs a hash table where the keys are weak references to objects which may be garbage collected even when referenced in a hash table.\n\nSee Dict for further help.  Note, unlike Dict, WeakKeyDict does not convert keys on insertion, as this would imply the key object was unreferenced anywhere before insertion.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.ImmutableDict","location":"base/collections.html#Base.ImmutableDict","category":"type","text":"ImmutableDict\n\nImmutableDict is a dictionary implemented as an immutable linked list, which is optimal for small dictionaries that are constructed over many individual insertions. Note that it is not possible to remove a value, although it can be partially overridden and hidden by inserting a new value with the same key.\n\nImmutableDict(KV::Pair)\n\nCreate a new entry in the ImmutableDict for a key => value pair\n\nuse (key => value) in dict to see if this particular combination is in the properties set\nuse get(dict, key, default) to retrieve the most recent value for a particular key\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.haskey","location":"base/collections.html#Base.haskey","category":"function","text":"haskey(collection, key) -> Bool\n\nDetermine whether a collection has a mapping for a given key.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> haskey(D, 'a')\ntrue\n\njulia> haskey(D, 'c')\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.get","location":"base/collections.html#Base.get","category":"function","text":"get(collection, key, default)\n\nReturn the value stored for the given key, or the given default value if no mapping for the key is present.\n\ncompat: Julia 1.7\nFor tuples and numbers, this function requires at least Julia 1.7.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2);\n\njulia> get(d, \"a\", 3)\n1\n\njulia> get(d, \"c\", 3)\n3\n\n\n\n\n\nget(f::Function, collection, key)\n\nReturn the value stored for the given key, or if no mapping for the key is present, return f().  Use get! to also store the default value in the dictionary.\n\nThis is intended to be called using do block syntax\n\nget(dict, key) do\n    # default value calculated here\n    time()\nend\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.get!","location":"base/collections.html#Base.get!","category":"function","text":"get!(collection, key, default)\n\nReturn the value stored for the given key, or if no mapping for the key is present, store key => default, and return default.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> get!(d, \"a\", 5)\n1\n\njulia> get!(d, \"d\", 4)\n4\n\njulia> d\nDict{String, Int64} with 4 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n  \"d\" => 4\n\n\n\n\n\nget!(f::Function, collection, key)\n\nReturn the value stored for the given key, or if no mapping for the key is present, store key => f(), and return f().\n\nThis is intended to be called using do block syntax.\n\nExamples\n\njulia> squares = Dict{Int, Int}();\n\njulia> function get_square!(d, i)\n           get!(d, i) do\n               i^2\n           end\n       end\nget_square! (generic function with 1 method)\n\njulia> get_square!(squares, 2)\n4\n\njulia> squares\nDict{Int64, Int64} with 1 entry:\n  2 => 4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.getkey","location":"base/collections.html#Base.getkey","category":"function","text":"getkey(collection, key, default)\n\nReturn the key matching argument key if one exists in collection, otherwise return default.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> getkey(D, 'a', 1)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> getkey(D, 'd', 'a')\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.delete!","location":"base/collections.html#Base.delete!","category":"function","text":"delete!(collection, key)\n\nDelete the mapping for the given key in a collection, if any, and return the collection.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> delete!(d, \"b\")\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\njulia> delete!(d, \"b\") # d is left unchanged\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.pop!","location":"base/collections.html#Base.pop!-Tuple{Any, Any, Any}","category":"method","text":"pop!(collection, key[, default])\n\nDelete and return the mapping for key if it exists in collection, otherwise return default, or throw an error if default is not specified.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> pop!(d, \"a\")\n1\n\njulia> pop!(d, \"d\")\nERROR: KeyError: key \"d\" not found\nStacktrace:\n[...]\n\njulia> pop!(d, \"e\", 4)\n4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.keys","location":"base/collections.html#Base.keys","category":"function","text":"keys(iterator)\n\nFor an iterator or collection that has keys and values (e.g. arrays and dictionaries), return an iterator over the keys.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.values","location":"base/collections.html#Base.values","category":"function","text":"values(iterator)\n\nFor an iterator or collection that has keys and values, return an iterator over the values. This function simply returns its argument by default, since the elements of a general iterator are normally considered its \"values\".\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2);\n\njulia> values(d)\nValueIterator for a Dict{String, Int64} with 2 entries. Values:\n  2\n  1\n\njulia> values([2])\n1-element Vector{Int64}:\n 2\n\n\n\n\n\nvalues(a::AbstractDict)\n\nReturn an iterator over all values in a collection. collect(values(a)) returns an array of values. When the values are stored internally in a hash table, as is the case for Dict, the order in which they are returned may vary. But keys(a) and values(a) both iterate a and return the elements in the same order.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> collect(values(D))\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.pairs","location":"base/collections.html#Base.pairs","category":"function","text":"pairs(collection)\n\nReturn an iterator over key => value pairs for any collection that maps a set of keys to a set of values. This includes arrays, where the keys are the array indices.\n\nExamples\n\njulia> a = Dict(zip([\"a\", \"b\", \"c\"], [1, 2, 3]))\nDict{String, Int64} with 3 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n\njulia> pairs(a)\nDict{String, Int64} with 3 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n\njulia> foreach(println, pairs([\"a\", \"b\", \"c\"]))\n1 => \"a\"\n2 => \"b\"\n3 => \"c\"\n\njulia> (;a=1, b=2, c=3) |> pairs |> collect\n3-element Vector{Pair{Symbol, Int64}}:\n :a => 1\n :b => 2\n :c => 3\n\njulia> (;a=1, b=2, c=3) |> collect\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\npairs(IndexLinear(), A)\npairs(IndexCartesian(), A)\npairs(IndexStyle(A), A)\n\nAn iterator that accesses each element of the array A, returning i => x, where i is the index for the element and x = A[i]. Identical to pairs(A), except that the style of index can be selected. Also similar to enumerate(A), except i will be a valid index for A, while enumerate always counts from 1 regardless of the indices of A.\n\nSpecifying IndexLinear() ensures that i will be an integer; specifying IndexCartesian() ensures that i will be a Base.CartesianIndex; specifying IndexStyle(A) chooses whichever has been defined as the native indexing style for array A.\n\nMutation of the bounds of the underlying array will invalidate this iterator.\n\nExamples\n\njulia> A = [\"a\" \"d\"; \"b\" \"e\"; \"c\" \"f\"];\n\njulia> for (index, value) in pairs(IndexStyle(A), A)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n4 d\n5 e\n6 f\n\njulia> S = view(A, 1:2, :);\n\njulia> for (index, value) in pairs(IndexStyle(S), S)\n           println(\"$index $value\")\n       end\nCartesianIndex(1, 1) a\nCartesianIndex(2, 1) b\nCartesianIndex(1, 2) d\nCartesianIndex(2, 2) e\n\nSee also IndexStyle, axes.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.merge","location":"base/collections.html#Base.merge","category":"function","text":"merge(d::AbstractDict, others::AbstractDict...)\n\nConstruct a merged collection from the given collections. If necessary, the types of the resulting collection will be promoted to accommodate the types of the merged collections. If the same key is present in another collection, the value for that key will be the value it has in the last collection listed. See also mergewith for custom handling of values with the same key.\n\nExamples\n\njulia> a = Dict(\"foo\" => 0.0, \"bar\" => 42.0)\nDict{String, Float64} with 2 entries:\n  \"bar\" => 42.0\n  \"foo\" => 0.0\n\njulia> b = Dict(\"baz\" => 17, \"bar\" => 4711)\nDict{String, Int64} with 2 entries:\n  \"bar\" => 4711\n  \"baz\" => 17\n\njulia> merge(a, b)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 4711.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\njulia> merge(b, a)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 42.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\n\n\n\n\nmerge(a::NamedTuple, bs::NamedTuple...)\n\nConstruct a new named tuple by merging two or more existing ones, in a left-associative manner. Merging proceeds left-to-right, between pairs of named tuples, and so the order of fields present in both the leftmost and rightmost named tuples take the same position as they are found in the leftmost named tuple. However, values are taken from matching fields in the rightmost named tuple that contains that field. Fields present in only the rightmost named tuple of a pair are appended at the end. A fallback is implemented for when only a single named tuple is supplied, with signature merge(a::NamedTuple).\n\ncompat: Julia 1.1\nMerging 3 or more NamedTuple requires at least Julia 1.1.\n\nExamples\n\njulia> merge((a=1, b=2, c=3), (b=4, d=5))\n(a = 1, b = 4, c = 3, d = 5)\n\njulia> merge((a=1, b=2), (b=3, c=(d=1,)), (c=(d=2,),))\n(a = 1, b = 3, c = (d = 2,))\n\n\n\n\n\nmerge(a::NamedTuple, iterable)\n\nInterpret an iterable of key-value pairs as a named tuple, and perform a merge.\n\njulia> merge((a=1, b=2, c=3), [:b=>4, :d=>5])\n(a = 1, b = 4, c = 3, d = 5)\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.mergewith","location":"base/collections.html#Base.mergewith","category":"function","text":"mergewith(combine, d::AbstractDict, others::AbstractDict...)\nmergewith(combine)\nmerge(combine, d::AbstractDict, others::AbstractDict...)\n\nConstruct a merged collection from the given collections. If necessary, the types of the resulting collection will be promoted to accommodate the types of the merged collections. Values with the same key will be combined using the combiner function.  The curried form mergewith(combine) returns the function (args...) -> mergewith(combine, args...).\n\nMethod merge(combine::Union{Function,Type}, args...) as an alias of mergewith(combine, args...) is still available for backward compatibility.\n\ncompat: Julia 1.5\nmergewith requires Julia 1.5 or later.\n\nExamples\n\njulia> a = Dict(\"foo\" => 0.0, \"bar\" => 42.0)\nDict{String, Float64} with 2 entries:\n  \"bar\" => 42.0\n  \"foo\" => 0.0\n\njulia> b = Dict(\"baz\" => 17, \"bar\" => 4711)\nDict{String, Int64} with 2 entries:\n  \"bar\" => 4711\n  \"baz\" => 17\n\njulia> mergewith(+, a, b)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 4753.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\njulia> ans == mergewith(+)(a, b)\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.merge!","location":"base/collections.html#Base.merge!","category":"function","text":"merge!(d::AbstractDict, others::AbstractDict...)\n\nUpdate collection with pairs from the other collections. See also merge.\n\nExamples\n\njulia> d1 = Dict(1 => 2, 3 => 4);\n\njulia> d2 = Dict(1 => 4, 4 => 5);\n\njulia> merge!(d1, d2);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 4\n  1 => 4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.mergewith!","location":"base/collections.html#Base.mergewith!","category":"function","text":"mergewith!(combine, d::AbstractDict, others::AbstractDict...) -> d\nmergewith!(combine)\nmerge!(combine, d::AbstractDict, others::AbstractDict...) -> d\n\nUpdate collection with pairs from the other collections. Values with the same key will be combined using the combiner function.  The curried form mergewith!(combine) returns the function (args...) -> mergewith!(combine, args...).\n\nMethod merge!(combine::Union{Function,Type}, args...) as an alias of mergewith!(combine, args...) is still available for backward compatibility.\n\ncompat: Julia 1.5\nmergewith! requires Julia 1.5 or later.\n\nExamples\n\njulia> d1 = Dict(1 => 2, 3 => 4);\n\njulia> d2 = Dict(1 => 4, 4 => 5);\n\njulia> mergewith!(+, d1, d2);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 4\n  1 => 6\n\njulia> mergewith!(-, d1, d1);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 0\n  3 => 0\n  1 => 0\n\njulia> foldl(mergewith!(+), [d1, d2]; init=Dict{Int64, Int64}())\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 0\n  1 => 4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.sizehint!","location":"base/collections.html#Base.sizehint!","category":"function","text":"sizehint!(s, n) -> s\n\nSuggest that collection s reserve capacity for at least n elements. That is, if you expect that you're going to have to push a lot of values onto s, you can avoid the cost of incremental reallocation by doing it once up front; this can improve performance.\n\nSee also resize!.\n\nNotes on the performance model\n\nFor types that support sizehint!,\n\npush! and append! methods generally may (but are not required to) preallocate extra storage. For types implemented in Base, they typically do, using a heuristic optimized for a general use case.\nsizehint! may control this preallocation. Again, it typically does this for types in Base.\nempty! is nearly costless (and O(1)) for types that support this kind of preallocation.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.keytype","location":"base/collections.html#Base.keytype","category":"function","text":"keytype(T::Type{<:AbstractArray})\nkeytype(A::AbstractArray)\n\nReturn the key type of an array. This is equal to the eltype of the result of keys(...), and is provided mainly for compatibility with the dictionary interface.\n\nExamples\n\njulia> keytype([1, 2, 3]) == Int\ntrue\n\njulia> keytype([1 2; 3 4])\nCartesianIndex{2}\n\ncompat: Julia 1.2\nFor arrays, this function requires at least Julia 1.2.\n\n\n\n\n\nkeytype(type)\n\nGet the key type of a dictionary type. Behaves similarly to eltype.\n\nExamples\n\njulia> keytype(Dict(Int32(1) => \"foo\"))\nInt32\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.valtype","location":"base/collections.html#Base.valtype","category":"function","text":"valtype(T::Type{<:AbstractArray})\nvaltype(A::AbstractArray)\n\nReturn the value type of an array. This is identical to eltype and is provided mainly for compatibility with the dictionary interface.\n\nExamples\n\njulia> valtype([\"one\", \"two\", \"three\"])\nString\n\ncompat: Julia 1.2\nFor arrays, this function requires at least Julia 1.2.\n\n\n\n\n\nvaltype(type)\n\nGet the value type of a dictionary type. Behaves similarly to eltype.\n\nExamples\n\njulia> valtype(Dict(Int32(1) => \"foo\"))\nString\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"IdDict\nDict\nWeakKeyDict","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Partially implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"BitSet\nSet\nEnvDict\nArray\nBitArray\nImmutableDict\nIterators.Pairs","page":"Collections and Data Structures"},{"title":"Set-Like Collections","location":"base/collections.html#Set-Like-Collections","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.AbstractSet\nBase.Set\nBase.BitSet\nBase.union\nBase.union!\nBase.intersect\nBase.setdiff\nBase.setdiff!\nBase.symdiff\nBase.symdiff!\nBase.intersect!\nBase.issubset\nBase.:⊈\nBase.:⊊\nBase.issetequal\nBase.isdisjoint","page":"Collections and Data Structures"},{"title":"Base.AbstractSet","location":"base/collections.html#Base.AbstractSet","category":"type","text":"AbstractSet{T}\n\nSupertype for set-like types whose elements are of type T. Set, BitSet and other types are subtypes of this.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.Set","location":"base/collections.html#Base.Set","category":"type","text":"Set{T} <: AbstractSet{T}\n\nSets are mutable containers that provide fast membership testing.\n\nSets have efficient implementations of set operations such as in, union and intersect. Elements in a Set are unique, as determined by the elements' definition of isequal. The order of elements in a Set is an implementation detail and cannot be relied on.\n\nSee also: AbstractSet, BitSet, Dict, push!, empty!, union!, in, isequal\n\nExamples\n\njulia> s = Set(\"aaBca\")\nSet{Char} with 3 elements:\n  'a'\n  'c'\n  'B'\n\njulia> push!(s, 'b')\nSet{Char} with 4 elements:\n  'a'\n  'c'\n  'b'\n  'B'\n\njulia> s = Set([NaN, 0.0, 1.0, 2.0]);\n\njulia> -0.0 in s # isequal(0.0, -0.0) is false\nfalse\n\njulia> NaN in s # isequal(NaN, NaN) is true\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.BitSet","location":"base/collections.html#Base.BitSet","category":"type","text":"BitSet([itr])\n\nConstruct a sorted set of Ints generated by the given iterable object, or an empty set. Implemented as a bit string, and therefore designed for dense integer sets. If the set will be sparse (for example, holding a few very large integers), use Set instead.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.union","location":"base/collections.html#Base.union","category":"function","text":"union(s, itrs...)\n∪(s, itrs...)\n\nConstruct an object containing all distinct elements from all of the arguments.\n\nThe first argument controls what kind of container is returned. If this is an array, it maintains the order in which elements first appear.\n\nUnicode ∪ can be typed by writing \\cup then pressing tab in the Julia REPL, and in many editors. This is an infix operator, allowing s ∪ itr.\n\nSee also unique, intersect, isdisjoint, vcat, Iterators.flatten.\n\nExamples\n\njulia> union([1, 2], [3])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> union([4 2 3 4 4], 1:3, 3.0)\n4-element Vector{Float64}:\n 4.0\n 2.0\n 3.0\n 1.0\n\njulia> (0, 0.0) ∪ (-0.0, NaN)\n3-element Vector{Real}:\n   0\n  -0.0\n NaN\n\njulia> union(Set([1, 2]), 2:3)\nSet{Int64} with 3 elements:\n  2\n  3\n  1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.union!","location":"base/collections.html#Base.union!","category":"function","text":"union!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nConstruct the union of passed in sets and overwrite s with the result. Maintain order with arrays.\n\nExamples\n\njulia> a = Set([3, 4, 5]);\n\njulia> union!(a, 1:2:7);\n\njulia> a\nSet{Int64} with 5 elements:\n  5\n  4\n  7\n  3\n  1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.intersect","location":"base/collections.html#Base.intersect","category":"function","text":"intersect(s, itrs...)\n∩(s, itrs...)\n\nConstruct the set containing those elements which appear in all of the arguments.\n\nThe first argument controls what kind of container is returned. If this is an array, it maintains the order in which elements first appear.\n\nUnicode ∩ can be typed by writing \\cap then pressing tab in the Julia REPL, and in many editors. This is an infix operator, allowing s ∩ itr.\n\nSee also setdiff, isdisjoint, issubset, issetequal.\n\ncompat: Julia 1.8\nAs of Julia 1.8 intersect returns a result with the eltype of the type-promoted eltypes of the two inputs\n\nExamples\n\njulia> intersect([1, 2, 3], [3, 4, 5])\n1-element Vector{Int64}:\n 3\n\njulia> intersect([1, 4, 4, 5, 6], [6, 4, 6, 7, 8])\n2-element Vector{Int64}:\n 4\n 6\n\njulia> intersect(1:16, 7:99)\n7:16\n\njulia> (0, 0.0) ∩ (-0.0, 0)\n1-element Vector{Real}:\n 0\n\njulia> intersect(Set([1, 2]), BitSet([2, 3]), 1.0:10.0)\nSet{Float64} with 1 element:\n  2.0\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.setdiff","location":"base/collections.html#Base.setdiff","category":"function","text":"setdiff(s, itrs...)\n\nConstruct the set of elements in s but not in any of the iterables in itrs. Maintain order with arrays.\n\nSee also setdiff!, union and intersect.\n\nExamples\n\njulia> setdiff([1,2,3], [3,4,5])\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.setdiff!","location":"base/collections.html#Base.setdiff!","category":"function","text":"setdiff!(s, itrs...)\n\nRemove from set s (in-place) each element of each iterable from itrs. Maintain order with arrays.\n\nExamples\n\njulia> a = Set([1, 3, 4, 5]);\n\njulia> setdiff!(a, 1:2:6);\n\njulia> a\nSet{Int64} with 1 element:\n  4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.symdiff","location":"base/collections.html#Base.symdiff","category":"function","text":"symdiff(s, itrs...)\n\nConstruct the symmetric difference of elements in the passed in sets. When s is not an AbstractSet, the order is maintained.\n\nSee also symdiff!, setdiff, union and intersect.\n\nExamples\n\njulia> symdiff([1,2,3], [3,4,5], [4,5,6])\n3-element Vector{Int64}:\n 1\n 2\n 6\n\njulia> symdiff([1,2,1], [2, 1, 2])\nInt64[]\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.symdiff!","location":"base/collections.html#Base.symdiff!","category":"function","text":"symdiff!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nConstruct the symmetric difference of the passed in sets, and overwrite s with the result. When s is an array, the order is maintained. Note that in this case the multiplicity of elements matters.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.intersect!","location":"base/collections.html#Base.intersect!","category":"function","text":"intersect!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nIntersect all passed in sets and overwrite s with the result. Maintain order with arrays.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.issubset","location":"base/collections.html#Base.issubset","category":"function","text":"issubset(a, b) -> Bool\n⊆(a, b) -> Bool\n⊇(b, a) -> Bool\n\nDetermine whether every element of a is also in b, using in.\n\nSee also ⊊, ⊈, ∩, ∪, contains.\n\nExamples\n\njulia> issubset([1, 2], [1, 2, 3])\ntrue\n\njulia> [1, 2, 3] ⊆ [1, 2]\nfalse\n\njulia> [1, 2, 3] ⊇ [1, 2]\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.:⊈","location":"base/collections.html#Base.:⊈","category":"function","text":"⊈(a, b) -> Bool\n⊉(b, a) -> Bool\n\nNegation of ⊆ and ⊇, i.e. checks that a is not a subset of b.\n\nSee also issubset (⊆), ⊊.\n\nExamples\n\njulia> (1, 2) ⊈ (2, 3)\ntrue\n\njulia> (1, 2) ⊈ (1, 2, 3)\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.:⊊","location":"base/collections.html#Base.:⊊","category":"function","text":"⊊(a, b) -> Bool\n⊋(b, a) -> Bool\n\nDetermines if a is a subset of, but not equal to, b.\n\nSee also issubset (⊆), ⊈.\n\nExamples\n\njulia> (1, 2) ⊊ (1, 2, 3)\ntrue\n\njulia> (1, 2) ⊊ (1, 2)\nfalse\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.issetequal","location":"base/collections.html#Base.issetequal","category":"function","text":"issetequal(a, b) -> Bool\n\nDetermine whether a and b have the same elements. Equivalent to a ⊆ b && b ⊆ a but more efficient when possible.\n\nSee also: isdisjoint, union.\n\nExamples\n\njulia> issetequal([1, 2], [1, 2, 3])\nfalse\n\njulia> issetequal([1, 2], [2, 1])\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.isdisjoint","location":"base/collections.html#Base.isdisjoint","category":"function","text":"isdisjoint(a, b) -> Bool\n\nDetermine whether the collections a and b are disjoint. Equivalent to isempty(a ∩ b) but more efficient when possible.\n\nSee also: intersect, isempty, issetequal.\n\ncompat: Julia 1.5\nThis function requires at least Julia 1.5.\n\nExamples\n\njulia> isdisjoint([1, 2], [2, 3, 4])\nfalse\n\njulia> isdisjoint([3, 1], [2, 4])\ntrue\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"BitSet\nSet","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Partially implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Array","page":"Collections and Data Structures"},{"title":"Dequeues","location":"base/collections.html#Dequeues","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.push!\nBase.pop!\nBase.popat!\nBase.pushfirst!\nBase.popfirst!\nBase.insert!\nBase.deleteat!\nBase.keepat!\nBase.splice!\nBase.resize!\nBase.append!\nBase.prepend!","page":"Collections and Data Structures"},{"title":"Base.push!","location":"base/collections.html#Base.push!","category":"function","text":"push!(collection, items...) -> collection\n\nInsert one or more items in collection. If collection is an ordered container, the items are inserted at the end (in the given order).\n\nExamples\n\njulia> push!([1, 2, 3], 4, 5, 6)\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\nIf collection is ordered, use append! to add all the elements of another collection to it. The result of the preceding example is equivalent to append!([1, 2, 3], [4, 5, 6]). For AbstractSet objects, union! can be used instead.\n\nSee sizehint! for notes about the performance model.\n\nSee also pushfirst!.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.pop!","location":"base/collections.html#Base.pop!","category":"function","text":"pop!(collection) -> item\n\nRemove an item in collection and return it. If collection is an ordered container, the last item is returned; for unordered containers, an arbitrary element is returned.\n\nSee also: popfirst!, popat!, delete!, deleteat!, splice!, and push!.\n\nExamples\n\njulia> A=[1, 2, 3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> pop!(A)\n3\n\njulia> A\n2-element Vector{Int64}:\n 1\n 2\n\njulia> S = Set([1, 2])\nSet{Int64} with 2 elements:\n  2\n  1\n\njulia> pop!(S)\n2\n\njulia> S\nSet{Int64} with 1 element:\n  1\n\njulia> pop!(Dict(1=>2))\n1 => 2\n\n\n\n\n\npop!(collection, key[, default])\n\nDelete and return the mapping for key if it exists in collection, otherwise return default, or throw an error if default is not specified.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> pop!(d, \"a\")\n1\n\njulia> pop!(d, \"d\")\nERROR: KeyError: key \"d\" not found\nStacktrace:\n[...]\n\njulia> pop!(d, \"e\", 4)\n4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.popat!","location":"base/collections.html#Base.popat!","category":"function","text":"popat!(a::Vector, i::Integer, [default])\n\nRemove the item at the given i and return it. Subsequent items are shifted to fill the resulting gap. When i is not a valid index for a, return default, or throw an error if default is not specified.\n\nSee also: pop!, popfirst!, deleteat!, splice!.\n\ncompat: Julia 1.5\nThis function is available as of Julia 1.5.\n\nExamples\n\njulia> a = [4, 3, 2, 1]; popat!(a, 2)\n3\n\njulia> a\n3-element Vector{Int64}:\n 4\n 2\n 1\n\njulia> popat!(a, 4, missing)\nmissing\n\njulia> popat!(a, 4)\nERROR: BoundsError: attempt to access 3-element Vector{Int64} at index [4]\n[...]\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.pushfirst!","location":"base/collections.html#Base.pushfirst!","category":"function","text":"pushfirst!(collection, items...) -> collection\n\nInsert one or more items at the beginning of collection.\n\nThis function is called unshift in many other programming languages.\n\nExamples\n\njulia> pushfirst!([1, 2, 3, 4], 5, 6)\n6-element Vector{Int64}:\n 5\n 6\n 1\n 2\n 3\n 4\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.popfirst!","location":"base/collections.html#Base.popfirst!","category":"function","text":"popfirst!(collection) -> item\n\nRemove the first item from collection.\n\nThis function is called shift in many other programming languages.\n\nSee also: pop!, popat!, delete!.\n\nExamples\n\njulia> A = [1, 2, 3, 4, 5, 6]\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\njulia> popfirst!(A)\n1\n\njulia> A\n5-element Vector{Int64}:\n 2\n 3\n 4\n 5\n 6\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.insert!","location":"base/collections.html#Base.insert!","category":"function","text":"insert!(a::Vector, index::Integer, item)\n\nInsert an item into a at the given index. index is the index of item in the resulting a.\n\nSee also: push!, replace, popat!, splice!.\n\nExamples\n\njulia> insert!(Any[1:6;], 3, \"here\")\n7-element Vector{Any}:\n 1\n 2\n  \"here\"\n 3\n 4\n 5\n 6\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.deleteat!","location":"base/collections.html#Base.deleteat!","category":"function","text":"deleteat!(a::Vector, i::Integer)\n\nRemove the item at the given i and return the modified a. Subsequent items are shifted to fill the resulting gap.\n\nSee also: keepat!, delete!, popat!, splice!.\n\nExamples\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], 2)\n5-element Vector{Int64}:\n 6\n 4\n 3\n 2\n 1\n\n\n\n\n\ndeleteat!(a::Vector, inds)\n\nRemove the items at the indices given by inds, and return the modified a. Subsequent items are shifted to fill the resulting gap.\n\ninds can be either an iterator or a collection of sorted and unique integer indices, or a boolean vector of the same length as a with true indicating entries to delete.\n\nExamples\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], 1:2:5)\n3-element Vector{Int64}:\n 5\n 3\n 1\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], [true, false, true, false, true, false])\n3-element Vector{Int64}:\n 5\n 3\n 1\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], (2, 2))\nERROR: ArgumentError: indices must be unique and sorted\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.keepat!","location":"base/collections.html#Base.keepat!","category":"function","text":"keepat!(a::Vector, inds)\nkeepat!(a::BitVector, inds)\n\nRemove the items at all the indices which are not given by inds, and return the modified a. Items which are kept are shifted to fill the resulting gaps.\n\ninds must be an iterator of sorted and unique integer indices. See also deleteat!.\n\ncompat: Julia 1.7\nThis function is available as of Julia 1.7.\n\nExamples\n\njulia> keepat!([6, 5, 4, 3, 2, 1], 1:2:5)\n3-element Vector{Int64}:\n 6\n 4\n 2\n\n\n\n\n\nkeepat!(a::Vector, m::AbstractVector{Bool})\nkeepat!(a::BitVector, m::AbstractVector{Bool})\n\nThe in-place version of logical indexing a = a[m]. That is, keepat!(a, m) on vectors of equal length a and m will remove all elements from a for which m at the corresponding index is false.\n\nExamples\n\njulia> a = [:a, :b, :c];\n\njulia> keepat!(a, [true, false, true])\n2-element Vector{Symbol}:\n :a\n :c\n\njulia> a\n2-element Vector{Symbol}:\n :a\n :c\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.splice!","location":"base/collections.html#Base.splice!","category":"function","text":"splice!(a::Vector, index::Integer, [replacement]) -> item\n\nRemove the item at the given index, and return the removed item. Subsequent items are shifted left to fill the resulting gap. If specified, replacement values from an ordered collection will be spliced in place of the removed item.\n\nSee also: replace, delete!, deleteat!, pop!, popat!.\n\nExamples\n\njulia> A = [6, 5, 4, 3, 2, 1]; splice!(A, 5)\n2\n\njulia> A\n5-element Vector{Int64}:\n 6\n 5\n 4\n 3\n 1\n\njulia> splice!(A, 5, -1)\n1\n\njulia> A\n5-element Vector{Int64}:\n  6\n  5\n  4\n  3\n -1\n\njulia> splice!(A, 1, [-1, -2, -3])\n6\n\njulia> A\n7-element Vector{Int64}:\n -1\n -2\n -3\n  5\n  4\n  3\n -1\n\nTo insert replacement before an index n without removing any items, use splice!(collection, n:n-1, replacement).\n\n\n\n\n\nsplice!(a::Vector, indices, [replacement]) -> items\n\nRemove items at specified indices, and return a collection containing the removed items. Subsequent items are shifted left to fill the resulting gaps. If specified, replacement values from an ordered collection will be spliced in place of the removed items; in this case, indices must be a AbstractUnitRange.\n\nTo insert replacement before an index n without removing any items, use splice!(collection, n:n-1, replacement).\n\ncompat: Julia 1.5\nPrior to Julia 1.5, indices must always be a UnitRange.\n\ncompat: Julia 1.8\nPrior to Julia 1.8, indices must be a UnitRange if splicing in replacement values.\n\nExamples\n\njulia> A = [-1, -2, -3, 5, 4, 3, -1]; splice!(A, 4:3, 2)\nInt64[]\n\njulia> A\n8-element Vector{Int64}:\n -1\n -2\n -3\n  2\n  5\n  4\n  3\n -1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.resize!","location":"base/collections.html#Base.resize!","category":"function","text":"resize!(a::Vector, n::Integer) -> Vector\n\nResize a to contain n elements. If n is smaller than the current collection length, the first n elements will be retained. If n is larger, the new elements are not guaranteed to be initialized.\n\nExamples\n\njulia> resize!([6, 5, 4, 3, 2, 1], 3)\n3-element Vector{Int64}:\n 6\n 5\n 4\n\njulia> a = resize!([6, 5, 4, 3, 2, 1], 8);\n\njulia> length(a)\n8\n\njulia> a[1:6]\n6-element Vector{Int64}:\n 6\n 5\n 4\n 3\n 2\n 1\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.append!","location":"base/collections.html#Base.append!","category":"function","text":"append!(collection, collections...) -> collection.\n\nFor an ordered container collection, add the elements of each collections to the end of it.\n\ncompat: Julia 1.6\nSpecifying multiple collections to be appended requires at least Julia 1.6.\n\nExamples\n\njulia> append!([1], [2, 3])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> append!([1, 2, 3], [4, 5], [6])\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\nUse push! to add individual items to collection which are not already themselves in another collection. The result of the preceding example is equivalent to push!([1, 2, 3], 4, 5, 6).\n\nSee sizehint! for notes about the performance model.\n\nSee also vcat for vectors, union! for sets, and prepend! and pushfirst! for the opposite order.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.prepend!","location":"base/collections.html#Base.prepend!","category":"function","text":"prepend!(a::Vector, collections...) -> collection\n\nInsert the elements of each collections to the beginning of a.\n\nWhen collections specifies multiple collections, order is maintained: elements of collections[1] will appear leftmost in a, and so on.\n\ncompat: Julia 1.6\nSpecifying multiple collections to be prepended requires at least Julia 1.6.\n\nExamples\n\njulia> prepend!([3], [1, 2])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> prepend!([6], [1, 2], [3, 4, 5])\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Fully implemented by:","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Vector (a.k.a. 1-dimensional Array)\nBitVector (a.k.a. 1-dimensional BitArray)","page":"Collections and Data Structures"},{"title":"Utility Collections","location":"base/collections.html#Utility-Collections","category":"section","text":"","page":"Collections and Data Structures"},{"title":"Collections and Data Structures","location":"base/collections.html","category":"page","text":"Base.Pair\nIterators.Pairs","page":"Collections and Data Structures"},{"title":"Core.Pair","location":"base/collections.html#Core.Pair","category":"type","text":"Pair(x, y)\nx => y\n\nConstruct a Pair object with type Pair{typeof(x), typeof(y)}. The elements are stored in the fields first and second. They can also be accessed via iteration (but a Pair is treated as a single \"scalar\" for broadcasting operations).\n\nSee also Dict.\n\nExamples\n\njulia> p = \"foo\" => 7\n\"foo\" => 7\n\njulia> typeof(p)\nPair{String, Int64}\n\njulia> p.first\n\"foo\"\n\njulia> for x in p\n           println(x)\n       end\nfoo\n7\n\njulia> replace.([\"xops\", \"oxps\"], \"x\" => \"o\")\n2-element Vector{String}:\n \"oops\"\n \"oops\"\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Base.Pairs","location":"base/collections.html#Base.Pairs","category":"type","text":"Iterators.Pairs(values, keys) <: AbstractDict{eltype(keys), eltype(values)}\n\nTransforms an indexable container into a Dictionary-view of the same data. Modifying the key-space of the underlying data may invalidate this object.\n\n\n\n\n\n","page":"Collections and Data Structures"},{"title":"Random Numbers","location":"stdlib/Random.html#Random-Numbers","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"DocTestSetup = :(using Random)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random number generation in Julia uses the Xoshiro256++ algorithm by default, with per-Task state. Other RNG types can be plugged in by inheriting the AbstractRNG type; they can then be used to obtain multiple streams of random numbers.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The PRNGs (pseudorandom number generators) exported by the Random package are:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"TaskLocalRNG: a token that represents use of the currently active Task-local stream, deterministically seeded from the parent task, or by RandomDevice (with system randomness) at program start\nXoshiro: generates a high-quality stream of random numbers with a small state vector and high performance using the Xoshiro256++ algorithm\nRandomDevice: for OS-provided entropy. This may be used for cryptographically secure random numbers (CS(P)RNG).\nMersenneTwister: an alternate high-quality PRNG which was the default in older versions of Julia, and is also quite fast, but requires much more space to store the state vector and generate a random sequence.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Most functions related to random generation accept an optional AbstractRNG object as first argument. Some also accept dimension specifications dims... (which can also be given as a tuple) to generate arrays of random values. In a multi-threaded program, you should generally use different RNG objects from different threads or tasks in order to be thread-safe. However, the default RNG is thread-safe as of Julia 1.3 (using a per-thread RNG up to version 1.6, and per-task thereafter).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The provided RNGs can generate uniform random numbers of the following types: Float16, Float32, Float64, BigFloat, Bool, Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, BigInt (or complex numbers of those types). Random floating point numbers are generated uniformly in 0 1). As BigInt represents unbounded integers, the interval must be specified (e.g. rand(big.(1:6))).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Additionally, normal and exponential distributions are implemented for some AbstractFloat and Complex types, see randn and randexp for details.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"To generate random numbers from other distributions, see the Distributions.jl package.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"warning: Warning\nBecause the precise way in which random numbers are generated is considered an implementation detail, bug fixes and speed improvements may change the stream of numbers that are generated after a version change. Relying on a specific seed or generated stream of numbers during unit testing is thus discouraged - consider testing properties of the methods in question instead.","page":"Random Numbers"},{"title":"Random numbers module","location":"stdlib/Random.html#Random-numbers-module","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.Random","page":"Random Numbers"},{"title":"Random.Random","location":"stdlib/Random.html#Random.Random","category":"module","text":"Random\n\nSupport for generating random numbers. Provides rand, randn, AbstractRNG, MersenneTwister, and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random generation functions","location":"stdlib/Random.html#Random-generation-functions","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.rand\nRandom.rand!\nRandom.bitrand\nRandom.randn\nRandom.randn!\nRandom.randexp\nRandom.randexp!\nRandom.randstring","page":"Random Numbers"},{"title":"Base.rand","location":"stdlib/Random.html#Base.rand","category":"function","text":"rand([rng=default_rng()], [S], [dims...])\n\nPick a random element or array of random elements from the set of values specified by S; S can be\n\nan indexable collection (for example 1:9 or ('x', \"y\", :z)),\nan AbstractDict or AbstractSet object,\na string (considered as a collection of characters), or\na type: the set of values to pick from is then equivalent to typemin(S):typemax(S) for integers (this is not applicable to BigInt), to 0 1) for floating point numbers and to 0 1)+i0 1) for complex floating point numbers;\n\nS defaults to Float64. When only one argument is passed besides the optional rng and is a Tuple, it is interpreted as a collection of values (S) and not as dims.\n\ncompat: Julia 1.1\nSupport for S as a tuple requires at least Julia 1.1.\n\nExamples\n\njulia> rand(Int, 2)\n2-element Array{Int64,1}:\n 1339893410598768192\n 1575814717733606317\n\njulia> using Random\n\njulia> rand(MersenneTwister(0), Dict(1=>2, 3=>4))\n1=>2\n\njulia> rand((2, 3))\n3\n\njulia> rand(Float64, (2, 3))\n2×3 Array{Float64,2}:\n 0.999717  0.0143835  0.540787\n 0.696556  0.783855   0.938235\n\nnote: Note\nThe complexity of rand(rng, s::Union{AbstractDict,AbstractSet}) is linear in the length of s, unless an optimized method with constant complexity is available, which is the case for Dict, Set and dense BitSets. For more than a few calls, use rand(rng, collect(s)) instead, or either rand(rng, Dict(s)) or rand(rng, Set(s)) as appropriate.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.rand!","location":"stdlib/Random.html#Random.rand!","category":"function","text":"rand!([rng=default_rng()], A, [S=eltype(A)])\n\nPopulate the array A with random values. If S is specified (S can be a type or a collection, cf. rand for details), the values are picked randomly from S. This is equivalent to copyto!(A, rand(rng, S, size(A))) but without allocating a new array.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> rand!(rng, zeros(5))\n5-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n 0.5662374165061859\n 0.4600853424625171\n 0.7940257103317943\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.bitrand","location":"stdlib/Random.html#Random.bitrand","category":"function","text":"bitrand([rng=default_rng()], [dims...])\n\nGenerate a BitArray of random boolean values.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> bitrand(rng, 10)\n10-element BitVector:\n 0\n 0\n 0\n 0\n 1\n 0\n 0\n 0\n 1\n 1\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Base.randn","location":"stdlib/Random.html#Base.randn","category":"function","text":"randn([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a normally-distributed random number of type T with mean 0 and standard deviation 1. Optionally generate an array of normally-distributed random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default), and their Complex counterparts. When the type argument is complex, the values are drawn from the circularly symmetric complex normal distribution of variance 1 (corresponding to real and imaginary part having independent normal distribution with mean zero and variance 1/2).\n\nExamples\n\njulia> using Random\n\njulia> rng = MersenneTwister(1234);\n\njulia> randn(rng, ComplexF64)\n0.6133070881429037 - 0.6376291670853887im\n\njulia> randn(rng, ComplexF32, (2, 3))\n2×3 Matrix{ComplexF32}:\n -0.349649-0.638457im  0.376756-0.192146im  -0.396334-0.0136413im\n  0.611224+1.56403im   0.355204-0.365563im  0.0905552+1.31012im\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randn!","location":"stdlib/Random.html#Random.randn!","category":"function","text":"randn!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with normally-distributed (mean 0, standard deviation 1) random numbers. Also see the rand function.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randn!(rng, zeros(5))\n5-element Vector{Float64}:\n  0.8673472019512456\n -0.9017438158568171\n -0.4944787535042339\n -0.9029142938652416\n  0.8644013132535154\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp","location":"stdlib/Random.html#Random.randexp","category":"function","text":"randexp([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a random number of type T according to the exponential distribution with scale 1. Optionally generate an array of such random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randexp(rng, Float32)\n2.4835055f0\n\njulia> randexp(rng, 3, 3)\n3×3 Matrix{Float64}:\n 1.5167    1.30652   0.344435\n 0.604436  2.78029   0.418516\n 0.695867  0.693292  0.643644\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp!","location":"stdlib/Random.html#Random.randexp!","category":"function","text":"randexp!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with random numbers following the exponential distribution (with scale 1).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randexp!(rng, zeros(5))\n5-element Vector{Float64}:\n 2.4835053723904896\n 1.516703605376473\n 0.6044364871025417\n 0.6958665886385867\n 1.3065196315496677\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randstring","location":"stdlib/Random.html#Random.randstring","category":"function","text":"randstring([rng=default_rng()], [chars], [len=8])\n\nCreate a random string of length len, consisting of characters from chars, which defaults to the set of upper- and lower-case letters and the digits 0-9. The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> Random.seed!(3); randstring()\n\"Lxz5hUwn\"\n\njulia> randstring(MersenneTwister(3), 'a':'z', 6)\n\"ocucay\"\n\njulia> randstring(\"ACGT\")\n\"TGCTCCTC\"\n\nnote: Note\nchars can be any collection of characters, of type Char or UInt8 (more efficient), provided rand can randomly pick characters from it.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Subsequences, permutations and shuffling","location":"stdlib/Random.html#Subsequences,-permutations-and-shuffling","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.randsubseq\nRandom.randsubseq!\nRandom.randperm\nRandom.randperm!\nRandom.randcycle\nRandom.randcycle!\nRandom.shuffle\nRandom.shuffle!","page":"Random Numbers"},{"title":"Random.randsubseq","location":"stdlib/Random.html#Random.randsubseq","category":"function","text":"randsubseq([rng=default_rng(),] A, p) -> Vector\n\nReturn a vector consisting of a random subsequence of the given array A, where each element of A is included (in order) with independent probability p. (Complexity is linear in p*length(A), so this function is efficient even if p is small and A is large.) Technically, this process is known as \"Bernoulli sampling\" of A.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randsubseq(rng, 1:8, 0.3)\n2-element Vector{Int64}:\n 7\n 8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randsubseq!","location":"stdlib/Random.html#Random.randsubseq!","category":"function","text":"randsubseq!([rng=default_rng(),] S, A, p)\n\nLike randsubseq, but the results are stored in S (which is resized as needed).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> S = Int64[];\n\njulia> randsubseq!(rng, S, 1:8, 0.3)\n2-element Vector{Int64}:\n 7\n 8\n\njulia> S\n2-element Vector{Int64}:\n 7\n 8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm","location":"stdlib/Random.html#Random.randperm","category":"function","text":"randperm([rng=default_rng(),] n::Integer)\n\nConstruct a random permutation of length n. The optional rng argument specifies a random number generator (see Random Numbers). The element type of the result is the same as the type of n.\n\nTo randomly permute an arbitrary vector, see shuffle or shuffle!.\n\ncompat: Julia 1.1\nIn Julia 1.1 randperm returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randperm(MersenneTwister(1234), 4)\n4-element Vector{Int64}:\n 2\n 1\n 4\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm!","location":"stdlib/Random.html#Random.randperm!","category":"function","text":"randperm!([rng=default_rng(),] A::Array{<:Integer})\n\nConstruct in A a random permutation of length length(A). The optional rng argument specifies a random number generator (see Random Numbers). To randomly permute an arbitrary vector, see shuffle or shuffle!.\n\nExamples\n\njulia> randperm!(MersenneTwister(1234), Vector{Int}(undef, 4))\n4-element Vector{Int64}:\n 2\n 1\n 4\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle","location":"stdlib/Random.html#Random.randcycle","category":"function","text":"randcycle([rng=default_rng(),] n::Integer)\n\nConstruct a random cyclic permutation of length n. The optional rng argument specifies a random number generator, see Random Numbers. The element type of the result is the same as the type of n.\n\ncompat: Julia 1.1\nIn Julia 1.1 randcycle returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randcycle(MersenneTwister(1234), 6)\n6-element Vector{Int64}:\n 3\n 5\n 4\n 6\n 1\n 2\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle!","location":"stdlib/Random.html#Random.randcycle!","category":"function","text":"randcycle!([rng=default_rng(),] A::Array{<:Integer})\n\nConstruct in A a random cyclic permutation of length length(A). The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> randcycle!(MersenneTwister(1234), Vector{Int}(undef, 6))\n6-element Vector{Int64}:\n 3\n 5\n 4\n 6\n 1\n 2\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle","location":"stdlib/Random.html#Random.shuffle","category":"function","text":"shuffle([rng=default_rng(),] v::AbstractArray)\n\nReturn a randomly permuted copy of v. The optional rng argument specifies a random number generator (see Random Numbers). To permute v in-place, see shuffle!. To obtain randomly permuted indices, see randperm.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> shuffle(rng, Vector(1:10))\n10-element Vector{Int64}:\n  6\n  1\n 10\n  2\n  3\n  9\n  5\n  7\n  4\n  8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle!","location":"stdlib/Random.html#Random.shuffle!","category":"function","text":"shuffle!([rng=default_rng(),] v::AbstractArray)\n\nIn-place version of shuffle: randomly permute v in-place, optionally supplying the random-number generator rng.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> shuffle!(rng, Vector(1:16))\n16-element Vector{Int64}:\n  2\n 15\n  5\n 14\n  1\n  9\n 10\n  6\n 11\n  3\n 16\n  7\n  4\n 12\n  8\n 13\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Generators (creation and seeding)","location":"stdlib/Random.html#Generators-(creation-and-seeding)","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.default_rng\nRandom.seed!\nRandom.AbstractRNG\nRandom.TaskLocalRNG\nRandom.Xoshiro\nRandom.MersenneTwister\nRandom.RandomDevice","page":"Random Numbers"},{"title":"Random.default_rng","location":"stdlib/Random.html#Random.default_rng","category":"function","text":"default_rng() -> rng\n\nReturn the default global random number generator (RNG).\n\nnote: Note\nWhat the default RNG is is an implementation detail.  Across different versions of Julia, you should not expect the default RNG to be always the same, nor that it will return the same stream of random numbers for a given seed.\n\ncompat: Julia 1.3\nThis function was introduced in Julia 1.3.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.seed!","location":"stdlib/Random.html#Random.seed!","category":"function","text":"seed!([rng=default_rng()], seed) -> rng\nseed!([rng=default_rng()]) -> rng\n\nReseed the random number generator: rng will give a reproducible sequence of numbers if and only if a seed is provided. Some RNGs don't accept a seed, like RandomDevice. After the call to seed!, rng is equivalent to a newly created object initialized with the same seed.\n\nIf rng is not specified, it defaults to seeding the state of the shared task-local generator.\n\nExamples\n\njulia> Random.seed!(1234);\n\njulia> x1 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> Random.seed!(1234);\n\njulia> x2 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\njulia> rng = Xoshiro(1234); rand(rng, 2) == x1\ntrue\n\njulia> Xoshiro(1) == Random.seed!(rng, 1)\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible either\nfalse\n\njulia> rand(Xoshiro(), Bool) # not reproducible either\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.AbstractRNG","location":"stdlib/Random.html#Random.AbstractRNG","category":"type","text":"AbstractRNG\n\nSupertype for random number generators such as MersenneTwister and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.TaskLocalRNG","location":"stdlib/Random.html#Random.TaskLocalRNG","category":"type","text":"TaskLocalRNG\n\nThe TaskLocalRNG has state that is local to its task, not its thread. It is seeded upon task creation, from the state of its parent task. Therefore, task creation is an event that changes the parent's RNG state.\n\nAs an upside, the TaskLocalRNG is pretty fast, and permits reproducible multithreaded simulations (barring race conditions), independent of scheduler decisions. As long as the number of threads is not used to make decisions on task creation, simulation results are also independent of the number of available threads / CPUs. The random stream should not depend on hardware specifics, up to endianness and possibly word size.\n\nUsing or seeding the RNG of any other task than the one returned by current_task() is undefined behavior: it will work most of the time, and may sometimes fail silently.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.Xoshiro","location":"stdlib/Random.html#Random.Xoshiro","category":"type","text":"Xoshiro(seed)\nXoshiro()\n\nXoshiro256++ is a fast pseudorandom number generator described by David Blackman and Sebastiano Vigna in \"Scrambled Linear Pseudorandom Number Generators\", ACM Trans. Math. Softw., 2021. Reference implementation is available at http://prng.di.unimi.it\n\nApart from the high speed, Xoshiro has a small memory footprint, making it suitable for applications where many different random states need to be held for long time.\n\nJulia's Xoshiro implementation has a bulk-generation mode; this seeds new virtual PRNGs from the parent, and uses SIMD to generate in parallel (i.e. the bulk stream consists of multiple interleaved xoshiro instances). The virtual PRNGs are discarded once the bulk request has been serviced (and should cause no heap allocations).\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(1234);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> rng = Xoshiro(1234);\n\njulia> x2 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.MersenneTwister","location":"stdlib/Random.html#Random.MersenneTwister","category":"type","text":"MersenneTwister(seed)\nMersenneTwister()\n\nCreate a MersenneTwister RNG object. Different RNG objects can have their own seeds, which may be useful for generating different streams of random numbers. The seed may be a non-negative integer or a vector of UInt32 integers. If no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing MersenneTwister object.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n\njulia> rng = MersenneTwister(1234);\n\njulia> x2 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.RandomDevice","location":"stdlib/Random.html#Random.RandomDevice","category":"type","text":"RandomDevice()\n\nCreate a RandomDevice RNG object. Two such objects will always generate different streams of random numbers. The entropy is obtained from the operating system.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Hooking into the Random API","location":"stdlib/Random.html#Hooking-into-the-Random-API","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"There are two mostly orthogonal ways to extend Random functionalities:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"generating random values of custom types\ncreating new generators","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API for 1) is quite functional, but is relatively recent so it may still have to evolve in subsequent releases of the Random module. For example, it's typically sufficient to implement one rand method in order to have all other usual methods work automatically.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API for 2) is still rudimentary, and may require more work than strictly necessary from the implementor, in order to support usual types of generated values.","page":"Random Numbers"},{"title":"Generating random values of custom types","location":"stdlib/Random.html#Generating-random-values-of-custom-types","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Generating random values for some distributions may involve various trade-offs. Pre-computed values, such as an alias table for discrete distributions, or “squeezing” functions for univariate distributions, can speed up sampling considerably. How much information should be pre-computed can depend on the number of values we plan to draw from a distribution. Also, some random number generators can have certain properties that various algorithms may want to exploit.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The Random module defines a customizable framework for obtaining random values that can address these issues. Each invocation of rand generates a sampler which can be customized with the above trade-offs in mind, by adding methods to Sampler, which in turn can dispatch on the random number generator, the object that characterizes the distribution, and a suggestion for the number of repetitions. Currently, for the latter, Val{1} (for a single sample) and Val{Inf} (for an arbitrary number) are used, with Random.Repetition an alias for both.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The object returned by Sampler is then used to generate the random values. When implementing the random generation interface for a value X that can be sampled from, the implementor should define the method","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"rand(rng, sampler)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"for the particular sampler returned by Sampler(rng, X, repetition).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Samplers can be arbitrary values that implement rand(rng, sampler), but for most applications the following predefined samplers may be sufficient:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"SamplerType{T}() can be used for implementing samplers that draw from type T (e.g. rand(Int)). This is the default returned by Sampler for types.\nSamplerTrivial(self) is a simple wrapper for self, which can be accessed with []. This is the recommended sampler when no pre-computed information is needed (e.g. rand(1:3)), and is the default returned by Sampler for values.\nSamplerSimple(self, data) also contains the additional data field, which can be used to store arbitrary pre-computed values, which should be computed in a custom method of Sampler.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"We provide examples for each of these. We assume here that the choice of algorithm is independent of the RNG, so we use AbstractRNG in our signatures.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.Sampler\nRandom.SamplerType\nRandom.SamplerTrivial\nRandom.SamplerSimple","page":"Random Numbers"},{"title":"Random.Sampler","location":"stdlib/Random.html#Random.Sampler","category":"type","text":"Sampler(rng, x, repetition = Val(Inf))\n\nReturn a sampler object that can be used to generate random values from rng for x.\n\nWhen sp = Sampler(rng, x, repetition), rand(rng, sp) will be used to draw random values, and should be defined accordingly.\n\nrepetition can be Val(1) or Val(Inf), and should be used as a suggestion for deciding the amount of precomputation, if applicable.\n\nRandom.SamplerType and Random.SamplerTrivial are default fallbacks for types and values, respectively. Random.SamplerSimple can be used to store pre-computed values without defining extra types for only this purpose.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerType","location":"stdlib/Random.html#Random.SamplerType","category":"type","text":"SamplerType{T}()\n\nA sampler for types, containing no other information. The default fallback for Sampler when called with types.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerTrivial","location":"stdlib/Random.html#Random.SamplerTrivial","category":"type","text":"SamplerTrivial(x)\n\nCreate a sampler that just wraps the given value x. This is the default fall-back for values. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values without precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerSimple","location":"stdlib/Random.html#Random.SamplerSimple","category":"type","text":"SamplerSimple(x, data)\n\nCreate a sampler that wraps the given value x and the data. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values with precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Decoupling pre-computation from actually generating the values is part of the API, and is also available to the user. As an example, assume that rand(rng, 1:20) has to be called repeatedly in a loop: the way to take advantage of this decoupling is as follows:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"rng = MersenneTwister()\nsp = Random.Sampler(rng, 1:20) # or Random.Sampler(MersenneTwister, 1:20)\nfor x in X\n    n = rand(rng, sp) # similar to n = rand(rng, 1:20)\n    # use n\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"This is the mechanism that is also used in the standard library, e.g. by the default implementation of random array generation (like in rand(1:20, 10)).","page":"Random Numbers"},{"title":"Generating values from a type","location":"stdlib/Random.html#Generating-values-from-a-type","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Given a type T, it's currently assumed that if rand(T) is defined, an object of type T will be produced. SamplerType is the default sampler for types. In order to define random generation of values of type T, the rand(rng::AbstractRNG, ::Random.SamplerType{T}) method should be defined, and should return values what rand(rng, T) is expected to return.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Let's take the following example: we implement a Die type, with a variable number n of sides, numbered from 1 to n. We want rand(Die) to produce a Die with a random number of up to 20 sides (and at least 4):","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"struct Die\n    nsides::Int # number of sides\nend\n\nRandom.rand(rng::AbstractRNG, ::Random.SamplerType{Die}) = Die(rand(rng, 4:20))\n\n# output\n","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Scalar and array methods for Die now work as expected:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"julia> rand(Die)\nDie(5)\n\njulia> rand(MersenneTwister(0), Die)\nDie(11)\n\njulia> rand(Die, 3)\n3-element Vector{Die}:\n Die(9)\n Die(15)\n Die(14)\n\njulia> a = Vector{Die}(undef, 3); rand!(a)\n3-element Vector{Die}:\n Die(19)\n Die(7)\n Die(17)","page":"Random Numbers"},{"title":"A simple sampler without pre-computed data","location":"stdlib/Random.html#A-simple-sampler-without-pre-computed-data","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Here we define a sampler for a collection. If no pre-computed data is required, it can be implemented with a SamplerTrivial sampler, which is in fact the default fallback for values.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In order to define random generation out of objects of type S, the following method should be defined: rand(rng::AbstractRNG, sp::Random.SamplerTrivial{S}). Here, sp simply wraps an object of type S, which can be accessed via sp[]. Continuing the Die example, we want now to define rand(d::Die) to produce an Int corresponding to one of d's sides:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"julia> Random.rand(rng::AbstractRNG, d::Random.SamplerTrivial{Die}) = rand(rng, 1:d[].nsides);\n\njulia> rand(Die(4))\n1\n\njulia> rand(Die(4), 3)\n3-element Vector{Any}:\n 2\n 3\n 3","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Given a collection type S, it's currently assumed that if rand(::S) is defined, an object of type eltype(S) will be produced. In the last example, a Vector{Any} is produced; the reason is that eltype(Die) == Any. The remedy is to define Base.eltype(::Type{Die}) = Int.","page":"Random Numbers"},{"title":"Generating values for an AbstractFloat type","location":"stdlib/Random.html#Generating-values-for-an-AbstractFloat-type","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"AbstractFloat types are special-cased, because by default random values are not produced in the whole type domain, but rather in [0,1). The following method should be implemented for T <: AbstractFloat: Random.rand(::AbstractRNG, ::Random.SamplerTrivial{Random.CloseOpen01{T}})","page":"Random Numbers"},{"title":"An optimized sampler with pre-computed data","location":"stdlib/Random.html#An-optimized-sampler-with-pre-computed-data","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Consider a discrete distribution, where numbers 1:n are drawn with given probabilities that sum to one. When many values are needed from this distribution, the fastest method is using an alias table. We don't provide the algorithm for building such a table here, but suppose it is available in make_alias_table(probabilities) instead, and draw_number(rng, alias_table) can be used to draw a random number from it.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Suppose that the distribution is described by","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"struct DiscreteDistribution{V <: AbstractVector}\n    probabilities::V\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"and that we always want to build an alias table, regardless of the number of values needed (we learn how to customize this below). The methods","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.eltype(::Type{<:DiscreteDistribution}) = Int\n\nfunction Random.Sampler(::Type{<:AbstractRNG}, distribution::DiscreteDistribution, ::Repetition)\n    SamplerSimple(disribution, make_alias_table(distribution.probabilities))\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"should be defined to return a sampler with pre-computed data, then","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"function rand(rng::AbstractRNG, sp::SamplerSimple{<:DiscreteDistribution})\n    draw_number(rng, sp.data)\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"will be used to draw the values.","page":"Random Numbers"},{"title":"Custom sampler types","location":"stdlib/Random.html#Custom-sampler-types","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The SamplerSimple type is sufficient for most use cases with precomputed data. However, in order to demonstrate how to use custom sampler types, here we implement something similar to SamplerSimple.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Going back to our Die example: rand(::Die) uses random generation from a range, so there is an opportunity for this optimization. We call our custom sampler SamplerDie.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"import Random: Sampler, rand\n\nstruct SamplerDie <: Sampler{Int} # generates values of type Int\n    die::Die\n    sp::Sampler{Int} # this is an abstract type, so this could be improved\nend\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerDie(die, Sampler(RNG, 1:die.nsides, r))\n# the `r` parameter will be explained later on\n\nrand(rng::AbstractRNG, sp::SamplerDie) = rand(rng, sp.sp)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"It's now possible to get a sampler with sp = Sampler(rng, die), and use sp instead of die in any rand call involving rng. In the simplistic example above, die doesn't need to be stored in SamplerDie but this is often the case in practice.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Of course, this pattern is so frequent that the helper type used above, namely Random.SamplerSimple, is available, saving us the definition of SamplerDie: we could have implemented our decoupling with:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerSimple(die, Sampler(RNG, 1:die.nsides, r))\n\nrand(rng::AbstractRNG, sp::SamplerSimple{Die}) = rand(rng, sp.data)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Here, sp.data refers to the second parameter in the call to the SamplerSimple constructor (in this case equal to Sampler(rng, 1:die.nsides, r)), while the Die object can be accessed via sp[].","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Like SamplerDie, any custom sampler must be a subtype of Sampler{T} where T is the type of the generated values. Note that SamplerSimple(x, data) isa Sampler{eltype(x)}, so this constrains what the first argument to SamplerSimple can be (it's recommended to use SamplerSimple like in the Die example, where x is simply forwarded while defining a Sampler method). Similarly, SamplerTrivial(x) isa Sampler{eltype(x)}.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Another helper type is currently available for other cases, Random.SamplerTag, but is considered as internal API, and can break at any time without proper deprecations.","page":"Random Numbers"},{"title":"Using distinct algorithms for scalar or array generation","location":"stdlib/Random.html#Using-distinct-algorithms-for-scalar-or-array-generation","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In some cases, whether one wants to generate only a handful of values or a large number of values will have an impact on the choice of algorithm. This is handled with the third parameter of the Sampler constructor. Let's assume we defined two helper types for Die, say SamplerDie1 which should be used to generate only few random values, and SamplerDieMany for many values. We can use those types as follows:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{1}) = SamplerDie1(...)\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{Inf}) = SamplerDieMany(...)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Of course, rand must also be defined on those types (i.e. rand(::AbstractRNG, ::SamplerDie1) and rand(::AbstractRNG, ::SamplerDieMany)). Note that, as usual, SamplerTrivial and SamplerSimple can be used if custom types are not necessary.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Note: Sampler(rng, x) is simply a shorthand for Sampler(rng, x, Val(Inf)), and Random.Repetition is an alias for Union{Val{1}, Val{Inf}}.","page":"Random Numbers"},{"title":"Creating new generators","location":"stdlib/Random.html#Creating-new-generators","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API is not clearly defined yet, but as a rule of thumb:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"any rand method producing \"basic\" types (isbitstype integer and floating types in Base) should be defined for this specific RNG, if they are needed;\nother documented rand methods accepting an AbstractRNG should work out of the box, (provided the methods from 1) what are relied on are implemented), but can of course be specialized for this RNG if there is room for optimization;\ncopy for pseudo-RNGs should return an independent copy that generates the exact same random sequence as the original from that point when called in the same way. When this is not feasible (e.g. hardware-based RNGs), copy must not be implemented.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Concerning 1), a rand method may happen to work automatically, but it's not officially supported and may break without warnings in a subsequent release.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"To define a new rand method for an hypothetical MyRNG generator, and a value specification s (e.g. s == Int, or s == 1:10) of type S==typeof(s) or S==Type{s} if s is a type, the same two methods as we saw before must be defined:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(::Type{MyRNG}, ::S, ::Repetition), which returns an object of type say SamplerS\nrand(rng::MyRNG, sp::SamplerS)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"It can happen that Sampler(rng::AbstractRNG, ::S, ::Repetition) is already defined in the Random module. It would then be possible to skip step 1) in practice (if one wants to specialize generation for this particular RNG type), but the corresponding SamplerS type is considered as internal detail, and may be changed without warning.","page":"Random Numbers"},{"title":"Specializing array generation","location":"stdlib/Random.html#Specializing-array-generation","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In some cases, for a given RNG type, generating an array of random values can be more efficient with a specialized method than by merely using the decoupling technique explained before. This is for example the case for MersenneTwister, which natively writes random values in an array.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"To implement this specialization for MyRNG and for a specification s, producing elements of type S, the following method can be defined: rand!(rng::MyRNG, a::AbstractArray{S}, ::SamplerS), where SamplerS is the type of the sampler returned by Sampler(MyRNG, s, Val(Inf)). Instead of AbstractArray, it's possible to implement the functionality only for a subtype, e.g. Array{S}. The non-mutating array method of rand will automatically call this specialization internally.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"DocTestSetup = nothing","page":"Random Numbers"},{"title":"Reproducibility","location":"stdlib/Random.html#Reproducibility","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"By using an RNG parameter initialized with a given seed, you can reproduce the same pseudorandom number sequence when running your program multiple times. However, a minor release of Julia (e.g. 1.3 to 1.4) may change the sequence of pseudorandom numbers generated from a specific seed, in particular if MersenneTwister is used. (Even if the sequence produced by a low-level function like rand does not change, the output of higher-level functions like randsubseq may change due to algorithm updates.) Rationale: guaranteeing that pseudorandom streams never change prohibits many algorithmic improvements.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"If you need to guarantee exact reproducibility of random data, it is advisable to simply save the data (e.g. as a supplementary attachment in a scientific publication). (You can also, of course, specify a particular Julia version and package manifest, especially if you require bit reproducibility.)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Software tests that rely on specific \"random\" data should also generally either save the data, embed it into the test code, or use third-party packages like StableRNGs.jl. On the other hand, tests that should pass for most random data (e.g. testing A \\ (A*x) ≈ x for a random matrix A = randn(n,n)) can use an RNG with a fixed seed to ensure that simply running the test many times does not encounter a failure due to very improbable data (e.g. an extremely ill-conditioned matrix).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The statistical distribution from which random samples are drawn is guaranteed to be the same across any minor Julia releases.","page":"Random Numbers"},{"title":"Arrays","location":"base/arrays.html#lib-arrays","category":"section","text":"","page":"Arrays"},{"title":"Constructors and Types","location":"base/arrays.html#Constructors-and-Types","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Core.AbstractArray\nBase.AbstractVector\nBase.AbstractMatrix\nBase.AbstractVecOrMat\nCore.Array\nCore.Array(::UndefInitializer, ::Any)\nCore.Array(::Nothing, ::Any)\nCore.Array(::Missing, ::Any)\nCore.UndefInitializer\nCore.undef\nBase.Vector\nBase.Vector(::UndefInitializer, ::Any)\nBase.Vector(::Nothing, ::Any)\nBase.Vector(::Missing, ::Any)\nBase.Matrix\nBase.Matrix(::UndefInitializer, ::Any, ::Any)\nBase.Matrix(::Nothing, ::Any, ::Any)\nBase.Matrix(::Missing, ::Any, ::Any)\nBase.VecOrMat\nCore.DenseArray\nBase.DenseVector\nBase.DenseMatrix\nBase.DenseVecOrMat\nBase.StridedArray\nBase.StridedVector\nBase.StridedMatrix\nBase.StridedVecOrMat\nBase.Slices\nBase.RowSlices\nBase.ColumnSlices\nBase.getindex(::Type, ::Any...)\nBase.zeros\nBase.ones\nBase.BitArray\nBase.BitArray(::UndefInitializer, ::Integer...)\nBase.BitArray(::Any)\nBase.trues\nBase.falses\nBase.fill\nBase.fill!\nBase.empty\nBase.similar","page":"Arrays"},{"title":"Core.AbstractArray","location":"base/arrays.html#Core.AbstractArray","category":"type","text":"AbstractArray{T,N}\n\nSupertype for N-dimensional arrays (or array-like types) with elements of type T. Array and other types are subtypes of this. See the manual section on the AbstractArray interface.\n\nSee also: AbstractVector, AbstractMatrix, eltype, ndims.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.AbstractVector","location":"base/arrays.html#Base.AbstractVector","category":"type","text":"AbstractVector{T}\n\nSupertype for one-dimensional arrays (or array-like types) with elements of type T. Alias for AbstractArray{T,1}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.AbstractMatrix","location":"base/arrays.html#Base.AbstractMatrix","category":"type","text":"AbstractMatrix{T}\n\nSupertype for two-dimensional arrays (or array-like types) with elements of type T. Alias for AbstractArray{T,2}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.AbstractVecOrMat","location":"base/arrays.html#Base.AbstractVecOrMat","category":"type","text":"AbstractVecOrMat{T}\n\nUnion type of AbstractVector{T} and AbstractMatrix{T}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.Array","location":"base/arrays.html#Core.Array","category":"type","text":"Array{T,N} <: AbstractArray{T,N}\n\nN-dimensional dense array with elements of type T.\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.Array","location":"base/arrays.html#Core.Array-Tuple{UndefInitializer, Any}","category":"method","text":"Array{T}(undef, dims)\nArray{T,N}(undef, dims)\n\nConstruct an uninitialized N-dimensional Array containing elements of type T. N can either be supplied explicitly, as in Array{T,N}(undef, dims), or be determined by the length or number of dims. dims may be a tuple or a series of integer arguments corresponding to the lengths in each dimension. If the rank N is supplied explicitly, then it must match the length or number of dims. Here undef is the UndefInitializer.\n\nExamples\n\njulia> A = Array{Float64, 2}(undef, 2, 3) # N given explicitly\n2×3 Matrix{Float64}:\n 6.90198e-310  6.90198e-310  6.90198e-310\n 6.90198e-310  6.90198e-310  0.0\n\njulia> B = Array{Float64}(undef, 4) # N determined by the input\n4-element Vector{Float64}:\n   2.360075077e-314\n NaN\n   2.2671131793e-314\n   2.299821756e-314\n\njulia> similar(B, 2, 4, 1) # use typeof(B), and the given size\n2×4×1 Array{Float64, 3}:\n[:, :, 1] =\n 2.26703e-314  2.26708e-314  0.0           2.80997e-314\n 0.0           2.26703e-314  2.26708e-314  0.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.Array","location":"base/arrays.html#Core.Array-Tuple{Nothing, Any}","category":"method","text":"Array{T}(nothing, dims)\nArray{T,N}(nothing, dims)\n\nConstruct an N-dimensional Array containing elements of type T, initialized with nothing entries. Element type T must be able to hold these values, i.e. Nothing <: T.\n\nExamples\n\njulia> Array{Union{Nothing, String}}(nothing, 2)\n2-element Vector{Union{Nothing, String}}:\n nothing\n nothing\n\njulia> Array{Union{Nothing, Int}}(nothing, 2, 3)\n2×3 Matrix{Union{Nothing, Int64}}:\n nothing  nothing  nothing\n nothing  nothing  nothing\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.Array","location":"base/arrays.html#Core.Array-Tuple{Missing, Any}","category":"method","text":"Array{T}(missing, dims)\nArray{T,N}(missing, dims)\n\nConstruct an N-dimensional Array containing elements of type T, initialized with missing entries. Element type T must be able to hold these values, i.e. Missing <: T.\n\nExamples\n\njulia> Array{Union{Missing, String}}(missing, 2)\n2-element Vector{Union{Missing, String}}:\n missing\n missing\n\njulia> Array{Union{Missing, Int}}(missing, 2, 3)\n2×3 Matrix{Union{Missing, Int64}}:\n missing  missing  missing\n missing  missing  missing\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.UndefInitializer","location":"base/arrays.html#Core.UndefInitializer","category":"type","text":"UndefInitializer\n\nSingleton type used in array initialization, indicating the array-constructor-caller would like an uninitialized array. See also undef, an alias for UndefInitializer().\n\nExamples\n\njulia> Array{Float64, 1}(UndefInitializer(), 3)\n3-element Array{Float64, 1}:\n 2.2752528595e-314\n 2.202942107e-314\n 2.275252907e-314\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.undef","location":"base/arrays.html#Core.undef","category":"constant","text":"undef\n\nAlias for UndefInitializer(), which constructs an instance of the singleton type UndefInitializer, used in array initialization to indicate the array-constructor-caller would like an uninitialized array.\n\nSee also: missing, similar.\n\nExamples\n\njulia> Array{Float64, 1}(undef, 3)\n3-element Vector{Float64}:\n 2.2752528595e-314\n 2.202942107e-314\n 2.275252907e-314\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Vector","location":"base/arrays.html#Base.Vector","category":"type","text":"Vector{T} <: AbstractVector{T}\n\nOne-dimensional dense array with elements of type T, often used to represent a mathematical vector. Alias for Array{T,1}.\n\nSee also empty, similar and zero for creating vectors.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Vector","location":"base/arrays.html#Base.Vector-Tuple{UndefInitializer, Any}","category":"method","text":"Vector{T}(undef, n)\n\nConstruct an uninitialized Vector{T} of length n.\n\nExamples\n\njulia> Vector{Float64}(undef, 3)\n3-element Array{Float64, 1}:\n 6.90966e-310\n 6.90966e-310\n 6.90966e-310\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Vector","location":"base/arrays.html#Base.Vector-Tuple{Nothing, Any}","category":"method","text":"Vector{T}(nothing, m)\n\nConstruct a Vector{T} of length m, initialized with nothing entries. Element type T must be able to hold these values, i.e. Nothing <: T.\n\nExamples\n\njulia> Vector{Union{Nothing, String}}(nothing, 2)\n2-element Vector{Union{Nothing, String}}:\n nothing\n nothing\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Vector","location":"base/arrays.html#Base.Vector-Tuple{Missing, Any}","category":"method","text":"Vector{T}(missing, m)\n\nConstruct a Vector{T} of length m, initialized with missing entries. Element type T must be able to hold these values, i.e. Missing <: T.\n\nExamples\n\njulia> Vector{Union{Missing, String}}(missing, 2)\n2-element Vector{Union{Missing, String}}:\n missing\n missing\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Matrix","location":"base/arrays.html#Base.Matrix","category":"type","text":"Matrix{T} <: AbstractMatrix{T}\n\nTwo-dimensional dense array with elements of type T, often used to represent a mathematical matrix. Alias for Array{T,2}.\n\nSee also fill, zeros, undef and similar for creating matrices.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Matrix","location":"base/arrays.html#Base.Matrix-Tuple{UndefInitializer, Any, Any}","category":"method","text":"Matrix{T}(undef, m, n)\n\nConstruct an uninitialized Matrix{T} of size m×n.\n\nExamples\n\njulia> Matrix{Float64}(undef, 2, 3)\n2×3 Array{Float64, 2}:\n 2.36365e-314  2.28473e-314    5.0e-324\n 2.26704e-314  2.26711e-314  NaN\n\njulia> similar(ans, Int32, 2, 2)\n2×2 Matrix{Int32}:\n 490537216  1277177453\n         1  1936748399\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Matrix","location":"base/arrays.html#Base.Matrix-Tuple{Nothing, Any, Any}","category":"method","text":"Matrix{T}(nothing, m, n)\n\nConstruct a Matrix{T} of size m×n, initialized with nothing entries. Element type T must be able to hold these values, i.e. Nothing <: T.\n\nExamples\n\njulia> Matrix{Union{Nothing, String}}(nothing, 2, 3)\n2×3 Matrix{Union{Nothing, String}}:\n nothing  nothing  nothing\n nothing  nothing  nothing\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Matrix","location":"base/arrays.html#Base.Matrix-Tuple{Missing, Any, Any}","category":"method","text":"Matrix{T}(missing, m, n)\n\nConstruct a Matrix{T} of size m×n, initialized with missing entries. Element type T must be able to hold these values, i.e. Missing <: T.\n\nExamples\n\njulia> Matrix{Union{Missing, String}}(missing, 2, 3)\n2×3 Matrix{Union{Missing, String}}:\n missing  missing  missing\n missing  missing  missing\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.VecOrMat","location":"base/arrays.html#Base.VecOrMat","category":"type","text":"VecOrMat{T}\n\nUnion type of Vector{T} and Matrix{T} which allows functions to accept either a Matrix or a Vector.\n\nExamples\n\njulia> Vector{Float64} <: VecOrMat{Float64}\ntrue\n\njulia> Matrix{Float64} <: VecOrMat{Float64}\ntrue\n\njulia> Array{Float64, 3} <: VecOrMat{Float64}\nfalse\n\n\n\n\n\n","page":"Arrays"},{"title":"Core.DenseArray","location":"base/arrays.html#Core.DenseArray","category":"type","text":"DenseArray{T, N} <: AbstractArray{T,N}\n\nN-dimensional dense array with elements of type T. The elements of a dense array are stored contiguously in memory.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.DenseVector","location":"base/arrays.html#Base.DenseVector","category":"type","text":"DenseVector{T}\n\nOne-dimensional DenseArray with elements of type T. Alias for DenseArray{T,1}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.DenseMatrix","location":"base/arrays.html#Base.DenseMatrix","category":"type","text":"DenseMatrix{T}\n\nTwo-dimensional DenseArray with elements of type T. Alias for DenseArray{T,2}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.DenseVecOrMat","location":"base/arrays.html#Base.DenseVecOrMat","category":"type","text":"DenseVecOrMat{T}\n\nUnion type of DenseVector{T} and DenseMatrix{T}.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.StridedArray","location":"base/arrays.html#Base.StridedArray","category":"type","text":"StridedArray{T, N}\n\nA hard-coded Union of common array types that follow the strided array interface, with elements of type T and N dimensions.\n\nIf A is a StridedArray, then its elements are stored in memory with offsets, which may vary between dimensions but are constant within a dimension. For example, A could have stride 2 in dimension 1, and stride 3 in dimension 2. Incrementing A along dimension d jumps in memory by [strides(A, d)] slots. Strided arrays are particularly important and useful because they can sometimes be passed directly as pointers to foreign language libraries like BLAS.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.StridedVector","location":"base/arrays.html#Base.StridedVector","category":"type","text":"StridedVector{T}\n\nOne dimensional StridedArray with elements of type T.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.StridedMatrix","location":"base/arrays.html#Base.StridedMatrix","category":"type","text":"StridedMatrix{T}\n\nTwo dimensional StridedArray with elements of type T.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.StridedVecOrMat","location":"base/arrays.html#Base.StridedVecOrMat","category":"type","text":"StridedVecOrMat{T}\n\nUnion type of StridedVector and StridedMatrix with elements of type T.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Slices","location":"base/arrays.html#Base.Slices","category":"type","text":"Slices{P,SM,AX,S,N} <: AbstractSlices{S,N}\n\nAn AbstractArray of slices into a parent array over specified dimension(s), returning views that select all the data from the other dimension(s).\n\nThese should typically be constructed by eachslice, eachcol or eachrow.\n\nparent(s::Slices) will return the parent array.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.RowSlices","location":"base/arrays.html#Base.RowSlices","category":"type","text":"RowSlices{M,AX,S}\n\nA special case of Slices that is a vector of row slices of a matrix, as constructed by eachrow.\n\nparent can be used to get the underlying matrix.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.ColumnSlices","location":"base/arrays.html#Base.ColumnSlices","category":"type","text":"ColumnSlices{M,AX,S}\n\nA special case of Slices that is a vector of column slices of a matrix, as constructed by eachcol.\n\nparent can be used to get the underlying matrix.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.getindex","location":"base/arrays.html#Base.getindex-Tuple{Type, Vararg{Any}}","category":"method","text":"getindex(type[, elements...])\n\nConstruct a 1-d array of the specified type. This is usually called with the syntax Type[]. Element values can be specified using Type[a,b,c,...].\n\nExamples\n\njulia> Int8[1, 2, 3]\n3-element Vector{Int8}:\n 1\n 2\n 3\n\njulia> getindex(Int8, 1, 2, 3)\n3-element Vector{Int8}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.zeros","location":"base/arrays.html#Base.zeros","category":"function","text":"zeros([T=Float64,] dims::Tuple)\nzeros([T=Float64,] dims...)\n\nCreate an Array, with element type T, of all zeros with size specified by dims. See also fill, ones, zero.\n\nExamples\n\njulia> zeros(1)\n1-element Vector{Float64}:\n 0.0\n\njulia> zeros(Int8, 2, 3)\n2×3 Matrix{Int8}:\n 0  0  0\n 0  0  0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.ones","location":"base/arrays.html#Base.ones","category":"function","text":"ones([T=Float64,] dims::Tuple)\nones([T=Float64,] dims...)\n\nCreate an Array, with element type T, of all ones with size specified by dims. See also fill, zeros.\n\nExamples\n\njulia> ones(1,2)\n1×2 Matrix{Float64}:\n 1.0  1.0\n\njulia> ones(ComplexF64, 2, 3)\n2×3 Matrix{ComplexF64}:\n 1.0+0.0im  1.0+0.0im  1.0+0.0im\n 1.0+0.0im  1.0+0.0im  1.0+0.0im\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.BitArray","location":"base/arrays.html#Base.BitArray","category":"type","text":"BitArray{N} <: AbstractArray{Bool, N}\n\nSpace-efficient N-dimensional boolean array, using just one bit for each boolean value.\n\nBitArrays pack up to 64 values into every 8 bytes, resulting in an 8x space efficiency over Array{Bool, N} and allowing some operations to work on 64 values at once.\n\nBy default, Julia returns BitArrays from broadcasting operations that generate boolean elements (including dotted-comparisons like .==) as well as from the functions trues and falses.\n\nnote: Note\nDue to its packed storage format, concurrent access to the elements of a BitArray where at least one of them is a write is not thread-safe.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.BitArray","location":"base/arrays.html#Base.BitArray-Tuple{UndefInitializer, Vararg{Integer}}","category":"method","text":"BitArray(undef, dims::Integer...)\nBitArray{N}(undef, dims::NTuple{N,Int})\n\nConstruct an undef BitArray with the given dimensions. Behaves identically to the Array constructor. See undef.\n\nExamples\n\njulia> BitArray(undef, 2, 2)\n2×2 BitMatrix:\n 0  0\n 0  0\n\njulia> BitArray(undef, (3, 1))\n3×1 BitMatrix:\n 0\n 0\n 0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.BitArray","location":"base/arrays.html#Base.BitArray-Tuple{Any}","category":"method","text":"BitArray(itr)\n\nConstruct a BitArray generated by the given iterable object. The shape is inferred from the itr object.\n\nExamples\n\njulia> BitArray([1 0; 0 1])\n2×2 BitMatrix:\n 1  0\n 0  1\n\njulia> BitArray(x+y == 3 for x = 1:2, y = 1:3)\n2×3 BitMatrix:\n 0  1  0\n 1  0  0\n\njulia> BitArray(x+y == 3 for x = 1:2 for y = 1:3)\n6-element BitVector:\n 0\n 1\n 0\n 1\n 0\n 0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.trues","location":"base/arrays.html#Base.trues","category":"function","text":"trues(dims)\n\nCreate a BitArray with all values set to true.\n\nExamples\n\njulia> trues(2,3)\n2×3 BitMatrix:\n 1  1  1\n 1  1  1\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.falses","location":"base/arrays.html#Base.falses","category":"function","text":"falses(dims)\n\nCreate a BitArray with all values set to false.\n\nExamples\n\njulia> falses(2,3)\n2×3 BitMatrix:\n 0  0  0\n 0  0  0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.fill","location":"base/arrays.html#Base.fill","category":"function","text":"fill(value, dims::Tuple)\nfill(value, dims...)\n\nCreate an array of size dims with every location set to value.\n\nFor example, fill(1.0, (5,5)) returns a 5×5 array of floats, with 1.0 in every location of the array.\n\nThe dimension lengths dims may be specified as either a tuple or a sequence of arguments. An N-length tuple or N arguments following the value specify an N-dimensional array. Thus, a common idiom for creating a zero-dimensional array with its only location set to x is fill(x).\n\nEvery location of the returned array is set to (and is thus === to) the value that was passed; this means that if the value is itself modified, all elements of the filled array will reflect that modification because they're still that very value. This is of no concern with fill(1.0, (5,5)) as the value 1.0 is immutable and cannot itself be modified, but can be unexpected with mutable values like — most commonly — arrays.  For example, fill([], 3) places the very same empty array in all three locations of the returned vector:\n\njulia> v = fill([], 3)\n3-element Vector{Vector{Any}}:\n []\n []\n []\n\njulia> v[1] === v[2] === v[3]\ntrue\n\njulia> value = v[1]\nAny[]\n\njulia> push!(value, 867_5309)\n1-element Vector{Any}:\n 8675309\n\njulia> v\n3-element Vector{Vector{Any}}:\n [8675309]\n [8675309]\n [8675309]\n\nTo create an array of many independent inner arrays, use a comprehension instead. This creates a new and distinct array on each iteration of the loop:\n\njulia> v2 = [[] for _ in 1:3]\n3-element Vector{Vector{Any}}:\n []\n []\n []\n\njulia> v2[1] === v2[2] === v2[3]\nfalse\n\njulia> push!(v2[1], 8675309)\n1-element Vector{Any}:\n 8675309\n\njulia> v2\n3-element Vector{Vector{Any}}:\n [8675309]\n []\n []\n\nSee also: fill!, zeros, ones, similar.\n\nExamples\n\njulia> fill(1.0, (2,3))\n2×3 Matrix{Float64}:\n 1.0  1.0  1.0\n 1.0  1.0  1.0\n\njulia> fill(42)\n0-dimensional Array{Int64, 0}:\n42\n\njulia> A = fill(zeros(2), 2) # sets both elements to the same [0.0, 0.0] vector\n2-element Vector{Vector{Float64}}:\n [0.0, 0.0]\n [0.0, 0.0]\n\njulia> A[1][1] = 42; # modifies the filled value to be [42.0, 0.0]\n\njulia> A # both A[1] and A[2] are the very same vector\n2-element Vector{Vector{Float64}}:\n [42.0, 0.0]\n [42.0, 0.0]\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.fill!","location":"base/arrays.html#Base.fill!","category":"function","text":"fill!(A, x)\n\nFill array A with the value x. If x is an object reference, all elements will refer to the same object. fill!(A, Foo()) will return A filled with the result of evaluating Foo() once.\n\nExamples\n\njulia> A = zeros(2,3)\n2×3 Matrix{Float64}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\njulia> fill!(A, 2.)\n2×3 Matrix{Float64}:\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n\njulia> a = [1, 1, 1]; A = fill!(Vector{Vector{Int}}(undef, 3), a); a[1] = 2; A\n3-element Vector{Vector{Int64}}:\n [2, 1, 1]\n [2, 1, 1]\n [2, 1, 1]\n\njulia> x = 0; f() = (global x += 1; x); fill!(Vector{Int}(undef, 3), f())\n3-element Vector{Int64}:\n 1\n 1\n 1\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.empty","location":"base/arrays.html#Base.empty","category":"function","text":"empty(x::Tuple)\n\nReturn an empty tuple, ().\n\n\n\n\n\nempty(v::AbstractVector, [eltype])\n\nCreate an empty vector similar to v, optionally changing the eltype.\n\nSee also: empty!, isempty, isassigned.\n\nExamples\n\njulia> empty([1.0, 2.0, 3.0])\nFloat64[]\n\njulia> empty([1.0, 2.0, 3.0], String)\nString[]\n\n\n\n\n\nempty(a::AbstractDict, [index_type=keytype(a)], [value_type=valtype(a)])\n\nCreate an empty AbstractDict container which can accept indices of type index_type and values of type value_type. The second and third arguments are optional and default to the input's keytype and valtype, respectively. (If only one of the two types is specified, it is assumed to be the value_type, and the index_type we default to keytype(a)).\n\nCustom AbstractDict subtypes may choose which specific dictionary type is best suited to return for the given index and value types, by specializing on the three-argument signature. The default is to return an empty Dict.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.similar","location":"base/arrays.html#Base.similar","category":"function","text":"similar(array, [element_type=eltype(array)], [dims=size(array)])\n\nCreate an uninitialized mutable array with the given element type and size, based upon the given source array. The second and third arguments are both optional, defaulting to the given array's eltype and size. The dimensions may be specified either as a single tuple argument or as a series of integer arguments.\n\nCustom AbstractArray subtypes may choose which specific array type is best-suited to return for the given element type and dimensionality. If they do not specialize this method, the default is an Array{element_type}(undef, dims...).\n\nFor example, similar(1:10, 1, 4) returns an uninitialized Array{Int,2} since ranges are neither mutable nor support 2 dimensions:\n\njulia> similar(1:10, 1, 4)\n1×4 Matrix{Int64}:\n 4419743872  4374413872  4419743888  0\n\nConversely, similar(trues(10,10), 2) returns an uninitialized BitVector with two elements since BitArrays are both mutable and can support 1-dimensional arrays:\n\njulia> similar(trues(10,10), 2)\n2-element BitVector:\n 0\n 0\n\nSince BitArrays can only store elements of type Bool, however, if you request a different element type it will create a regular Array instead:\n\njulia> similar(falses(10), Float64, 2, 4)\n2×4 Matrix{Float64}:\n 2.18425e-314  2.18425e-314  2.18425e-314  2.18425e-314\n 2.18425e-314  2.18425e-314  2.18425e-314  2.18425e-314\n\nSee also: undef, isassigned.\n\n\n\n\n\nsimilar(storagetype, axes)\n\nCreate an uninitialized mutable array analogous to that specified by storagetype, but with axes specified by the last argument.\n\nExamples:\n\nsimilar(Array{Int}, axes(A))\n\ncreates an array that \"acts like\" an Array{Int} (and might indeed be backed by one), but which is indexed identically to A. If A has conventional indexing, this will be identical to Array{Int}(undef, size(A)), but if A has unconventional indexing then the indices of the result will match A.\n\nsimilar(BitArray, (axes(A, 2),))\n\nwould create a 1-dimensional logical array whose indices match those of the columns of A.\n\n\n\n\n\nsimilar(A::AbstractSparseMatrixCSC{Tv,Ti}, [::Type{TvNew}, ::Type{TiNew}, m::Integer, n::Integer]) where {Tv,Ti}\n\nCreate an uninitialized mutable array with the given element type, index type, and size, based upon the given source SparseMatrixCSC. The new sparse matrix maintains the structure of the original sparse matrix, except in the case where dimensions of the output matrix are different from the output.\n\nThe output matrix has zeros in the same locations as the input, but unititialized values for the nonzero locations.\n\n\n\n\n\n","page":"Arrays"},{"title":"Basic functions","location":"base/arrays.html#Basic-functions","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.ndims\nBase.size\nBase.axes(::Any)\nBase.axes(::AbstractArray, ::Any)\nBase.length(::AbstractArray)\nBase.keys(::AbstractArray)\nBase.eachindex\nBase.IndexStyle\nBase.IndexLinear\nBase.IndexCartesian\nBase.conj!\nBase.stride\nBase.strides","page":"Arrays"},{"title":"Base.ndims","location":"base/arrays.html#Base.ndims","category":"function","text":"ndims(A::AbstractArray) -> Integer\n\nReturn the number of dimensions of A.\n\nSee also: size, axes.\n\nExamples\n\njulia> A = fill(1, (3,4,5));\n\njulia> ndims(A)\n3\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.size","location":"base/arrays.html#Base.size","category":"function","text":"size(A::AbstractArray, [dim])\n\nReturn a tuple containing the dimensions of A. Optionally you can specify a dimension to just get the length of that dimension.\n\nNote that size may not be defined for arrays with non-standard indices, in which case axes may be useful. See the manual chapter on arrays with custom indices.\n\nSee also: length, ndims, eachindex, sizeof.\n\nExamples\n\njulia> A = fill(1, (2,3,4));\n\njulia> size(A)\n(2, 3, 4)\n\njulia> size(A, 2)\n3\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.axes","location":"base/arrays.html#Base.axes-Tuple{Any}","category":"method","text":"axes(A)\n\nReturn the tuple of valid indices for array A.\n\nSee also: size, keys, eachindex.\n\nExamples\n\njulia> A = fill(1, (5,6,7));\n\njulia> axes(A)\n(Base.OneTo(5), Base.OneTo(6), Base.OneTo(7))\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.axes","location":"base/arrays.html#Base.axes-Tuple{AbstractArray, Any}","category":"method","text":"axes(A, d)\n\nReturn the valid range of indices for array A along dimension d.\n\nSee also size, and the manual chapter on arrays with custom indices.\n\nExamples\n\njulia> A = fill(1, (5,6,7));\n\njulia> axes(A, 2)\nBase.OneTo(6)\n\njulia> axes(A, 4) == 1:1  # all dimensions d > ndims(A) have size 1\ntrue\n\nUsage note\n\nEach of the indices has to be an AbstractUnitRange{<:Integer}, but at the same time can be a type that uses custom indices. So, for example, if you need a subset, use generalized indexing constructs like begin/end or firstindex/lastindex:\n\nix = axes(v, 1)\nix[2:end]          # will work for eg Vector, but may fail in general\nix[(begin+1):end]  # works for generalized indexes\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.length","location":"base/arrays.html#Base.length-Tuple{AbstractArray}","category":"method","text":"length(A::AbstractArray)\n\nReturn the number of elements in the array, defaults to prod(size(A)).\n\nExamples\n\njulia> length([1, 2, 3, 4])\n4\n\njulia> length([1 2; 3 4])\n4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.keys","location":"base/arrays.html#Base.keys-Tuple{AbstractArray}","category":"method","text":"keys(a::AbstractArray)\n\nReturn an efficient array describing all valid indices for a arranged in the shape of a itself.\n\nThe keys of 1-dimensional arrays (vectors) are integers, whereas all other N-dimensional arrays use CartesianIndex to describe their locations.  Often the special array types LinearIndices and CartesianIndices are used to efficiently represent these arrays of integers and CartesianIndexes, respectively.\n\nNote that the keys of an array might not be the most efficient index type; for maximum performance use  eachindex instead.\n\nExamples\n\njulia> keys([4, 5, 6])\n3-element LinearIndices{1, Tuple{Base.OneTo{Int64}}}:\n 1\n 2\n 3\n\njulia> keys([4 5; 6 7])\nCartesianIndices((2, 2))\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.eachindex","location":"base/arrays.html#Base.eachindex","category":"function","text":"eachindex(A...)\neachindex(::IndexStyle, A::AbstractArray...)\n\nCreate an iterable object for visiting each index of an AbstractArray A in an efficient manner. For array types that have opted into fast linear indexing (like Array), this is simply the range 1:length(A) if they use 1-based indexing. For array types that have not opted into fast linear indexing, a specialized Cartesian range is typically returned to efficiently index into the array with indices specified for every dimension.\n\nIn general eachindex accepts arbitrary iterables, including strings and dictionaries, and returns an iterator object supporting arbitrary index types (e.g. unevenly spaced or non-integer indices).\n\nIf A is AbstractArray it is possible to explicitly specify the style of the indices that should be returned by eachindex by passing a value having IndexStyle type as its first argument (typically IndexLinear() if linear indices are required or IndexCartesian() if Cartesian range is wanted).\n\nIf you supply more than one AbstractArray argument, eachindex will create an iterable object that is fast for all arguments (typically a UnitRange if all inputs have fast linear indexing, a CartesianIndices otherwise). If the arrays have different sizes and/or dimensionalities, a DimensionMismatch exception will be thrown.\n\nSee also pairs(A) to iterate over indices and values together, and axes(A, 2) for valid indices along one dimension.\n\nExamples\n\njulia> A = [10 20; 30 40];\n\njulia> for i in eachindex(A) # linear indexing\n           println(\"A[\", i, \"] == \", A[i])\n       end\nA[1] == 10\nA[2] == 30\nA[3] == 20\nA[4] == 40\n\njulia> for i in eachindex(view(A, 1:2, 1:1)) # Cartesian indexing\n           println(i)\n       end\nCartesianIndex(1, 1)\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.IndexStyle","location":"base/arrays.html#Base.IndexStyle","category":"type","text":"IndexStyle(A)\nIndexStyle(typeof(A))\n\nIndexStyle specifies the \"native indexing style\" for array A. When you define a new AbstractArray type, you can choose to implement either linear indexing (with IndexLinear) or cartesian indexing. If you decide to only implement linear indexing, then you must set this trait for your array type:\n\nBase.IndexStyle(::Type{<:MyArray}) = IndexLinear()\n\nThe default is IndexCartesian().\n\nJulia's internal indexing machinery will automatically (and invisibly) recompute all indexing operations into the preferred style. This allows users to access elements of your array using any indexing style, even when explicit methods have not been provided.\n\nIf you define both styles of indexing for your AbstractArray, this trait can be used to select the most performant indexing style. Some methods check this trait on their inputs, and dispatch to different algorithms depending on the most efficient access pattern. In particular, eachindex creates an iterator whose type depends on the setting of this trait.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.IndexLinear","location":"base/arrays.html#Base.IndexLinear","category":"type","text":"IndexLinear()\n\nSubtype of IndexStyle used to describe arrays which are optimally indexed by one linear index.\n\nA linear indexing style uses one integer index to describe the position in the array (even if it's a multidimensional array) and column-major ordering is used to efficiently access the elements. This means that requesting eachindex from an array that is IndexLinear will return a simple one-dimensional range, even if it is multidimensional.\n\nA custom array that reports its IndexStyle as IndexLinear only needs to implement indexing (and indexed assignment) with a single Int index; all other indexing expressions — including multidimensional accesses — will be recomputed to the linear index.  For example, if A were a 2×3 custom matrix with linear indexing, and we referenced A[1, 3], this would be recomputed to the equivalent linear index and call A[5] since 2*1 + 3 = 5.\n\nSee also IndexCartesian.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.IndexCartesian","location":"base/arrays.html#Base.IndexCartesian","category":"type","text":"IndexCartesian()\n\nSubtype of IndexStyle used to describe arrays which are optimally indexed by a Cartesian index. This is the default for new custom AbstractArray subtypes.\n\nA Cartesian indexing style uses multiple integer indices to describe the position in a multidimensional array, with exactly one index per dimension. This means that requesting eachindex from an array that is IndexCartesian will return a range of CartesianIndices.\n\nA N-dimensional custom array that reports its IndexStyle as IndexCartesian needs to implement indexing (and indexed assignment) with exactly N Int indices; all other indexing expressions — including linear indexing — will be recomputed to the equivalent Cartesian location.  For example, if A were a 2×3 custom matrix with cartesian indexing, and we referenced A[5], this would be recomputed to the equivalent Cartesian index and call A[1, 3] since 5 = 2*1 + 3.\n\nIt is significantly more expensive to compute Cartesian indices from a linear index than it is to go the other way.  The former operation requires division — a very costly operation — whereas the latter only uses multiplication and addition and is essentially free. This asymmetry means it is far more costly to use linear indexing with an IndexCartesian array than it is to use Cartesian indexing with an IndexLinear array.\n\nSee also IndexLinear.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.conj!","location":"base/arrays.html#Base.conj!","category":"function","text":"conj!(A)\n\nTransform an array to its complex conjugate in-place.\n\nSee also conj.\n\nExamples\n\njulia> A = [1+im 2-im; 2+2im 3+im]\n2×2 Matrix{Complex{Int64}}:\n 1+1im  2-1im\n 2+2im  3+1im\n\njulia> conj!(A);\n\njulia> A\n2×2 Matrix{Complex{Int64}}:\n 1-1im  2+1im\n 2-2im  3-1im\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.stride","location":"base/arrays.html#Base.stride","category":"function","text":"stride(A, k::Integer)\n\nReturn the distance in memory (in number of elements) between adjacent elements in dimension k.\n\nSee also: strides.\n\nExamples\n\njulia> A = fill(1, (3,4,5));\n\njulia> stride(A,2)\n3\n\njulia> stride(A,3)\n12\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.strides","location":"base/arrays.html#Base.strides","category":"function","text":"strides(A)\n\nReturn a tuple of the memory strides in each dimension.\n\nSee also: stride.\n\nExamples\n\njulia> A = fill(1, (3,4,5));\n\njulia> strides(A)\n(1, 3, 12)\n\n\n\n\n\n","page":"Arrays"},{"title":"Broadcast and vectorization","location":"base/arrays.html#Broadcast-and-vectorization","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"See also the dot syntax for vectorizing functions; for example, f.(args...) implicitly calls broadcast(f, args...). Rather than relying on \"vectorized\" methods of functions like sin to operate on arrays, you should use sin.(a) to vectorize via broadcast.","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.broadcast\nBase.Broadcast.broadcast!\nBase.@__dot__","page":"Arrays"},{"title":"Base.Broadcast.broadcast","location":"base/arrays.html#Base.Broadcast.broadcast","category":"function","text":"broadcast(f, As...)\n\nBroadcast the function f over the arrays, tuples, collections, Refs and/or scalars As.\n\nBroadcasting applies the function f over the elements of the container arguments and the scalars themselves in As. Singleton and missing dimensions are expanded to match the extents of the other arguments by virtually repeating the value. By default, only a limited number of types are considered scalars, including Numbers, Strings, Symbols, Types, Functions and some common singletons like missing and nothing. All other arguments are iterated over or indexed into elementwise.\n\nThe resulting container type is established by the following rules:\n\nIf all the arguments are scalars or zero-dimensional arrays, it returns an unwrapped scalar.\nIf at least one argument is a tuple and all others are scalars or zero-dimensional arrays, it returns a tuple.\nAll other combinations of arguments default to returning an Array, but custom container types can define their own implementation and promotion-like rules to customize the result when they appear as arguments.\n\nA special syntax exists for broadcasting: f.(args...) is equivalent to broadcast(f, args...), and nested f.(g.(args...)) calls are fused into a single broadcast loop.\n\nExamples\n\njulia> A = [1, 2, 3, 4, 5]\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> B = [1 2; 3 4; 5 6; 7 8; 9 10]\n5×2 Matrix{Int64}:\n 1   2\n 3   4\n 5   6\n 7   8\n 9  10\n\njulia> broadcast(+, A, B)\n5×2 Matrix{Int64}:\n  2   3\n  5   6\n  8   9\n 11  12\n 14  15\n\njulia> parse.(Int, [\"1\", \"2\"])\n2-element Vector{Int64}:\n 1\n 2\n\njulia> abs.((1, -2))\n(1, 2)\n\njulia> broadcast(+, 1.0, (0, -2.0))\n(1.0, -1.0)\n\njulia> (+).([[0,2], [1,3]], Ref{Vector{Int}}([1,-1]))\n2-element Vector{Vector{Int64}}:\n [1, 1]\n [2, 2]\n\njulia> string.((\"one\",\"two\",\"three\",\"four\"), \": \", 1:4)\n4-element Vector{String}:\n \"one: 1\"\n \"two: 2\"\n \"three: 3\"\n \"four: 4\"\n\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.broadcast!","location":"base/arrays.html#Base.Broadcast.broadcast!","category":"function","text":"broadcast!(f, dest, As...)\n\nLike broadcast, but store the result of broadcast(f, As...) in the dest array. Note that dest is only used to store the result, and does not supply arguments to f unless it is also listed in the As, as in broadcast!(f, A, A, B) to perform A[:] = broadcast(f, A, B).\n\nExamples\n\njulia> A = [1.0; 0.0]; B = [0.0; 0.0];\n\njulia> broadcast!(+, B, A, (0, -2.0));\n\njulia> B\n2-element Vector{Float64}:\n  1.0\n -2.0\n\njulia> A\n2-element Vector{Float64}:\n 1.0\n 0.0\n\njulia> broadcast!(+, A, A, (0, -2.0));\n\njulia> A\n2-element Vector{Float64}:\n  1.0\n -2.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.@__dot__","location":"base/arrays.html#Base.Broadcast.@__dot__","category":"macro","text":"@. expr\n\nConvert every function call or operator in expr into a \"dot call\" (e.g. convert f(x) to f.(x)), and convert every assignment in expr to a \"dot assignment\" (e.g. convert += to .+=).\n\nIf you want to avoid adding dots for selected function calls in expr, splice those function calls in with $.  For example, @. sqrt(abs($sort(x))) is equivalent to sqrt.(abs.(sort(x))) (no dot for sort).\n\n(@. is equivalent to a call to @__dot__.)\n\nExamples\n\njulia> x = 1.0:3.0; y = similar(x);\n\njulia> @. y = x + 3 * sin(x)\n3-element Vector{Float64}:\n 3.5244129544236893\n 4.727892280477045\n 3.4233600241796016\n\n\n\n\n\n","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"For specializing broadcast on custom types, see","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.BroadcastStyle\nBase.Broadcast.AbstractArrayStyle\nBase.Broadcast.ArrayStyle\nBase.Broadcast.DefaultArrayStyle\nBase.Broadcast.broadcastable\nBase.Broadcast.combine_axes\nBase.Broadcast.combine_styles\nBase.Broadcast.result_style","page":"Arrays"},{"title":"Base.Broadcast.BroadcastStyle","location":"base/arrays.html#Base.Broadcast.BroadcastStyle","category":"type","text":"BroadcastStyle is an abstract type and trait-function used to determine behavior of objects under broadcasting. BroadcastStyle(typeof(x)) returns the style associated with x. To customize the broadcasting behavior of a type, one can declare a style by defining a type/method pair\n\nstruct MyContainerStyle <: BroadcastStyle end\nBase.BroadcastStyle(::Type{<:MyContainer}) = MyContainerStyle()\n\nOne then writes method(s) (at least similar) operating on Broadcasted{MyContainerStyle}. There are also several pre-defined subtypes of BroadcastStyle that you may be able to leverage; see the Interfaces chapter for more information.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.AbstractArrayStyle","location":"base/arrays.html#Base.Broadcast.AbstractArrayStyle","category":"type","text":"Broadcast.AbstractArrayStyle{N} <: BroadcastStyle is the abstract supertype for any style associated with an AbstractArray type. The N parameter is the dimensionality, which can be handy for AbstractArray types that only support specific dimensionalities:\n\nstruct SparseMatrixStyle <: Broadcast.AbstractArrayStyle{2} end\nBase.BroadcastStyle(::Type{<:SparseMatrixCSC}) = SparseMatrixStyle()\n\nFor AbstractArray types that support arbitrary dimensionality, N can be set to Any:\n\nstruct MyArrayStyle <: Broadcast.AbstractArrayStyle{Any} end\nBase.BroadcastStyle(::Type{<:MyArray}) = MyArrayStyle()\n\nIn cases where you want to be able to mix multiple AbstractArrayStyles and keep track of dimensionality, your style needs to support a Val constructor:\n\nstruct MyArrayStyleDim{N} <: Broadcast.AbstractArrayStyle{N} end\n(::Type{<:MyArrayStyleDim})(::Val{N}) where N = MyArrayStyleDim{N}()\n\nNote that if two or more AbstractArrayStyle subtypes conflict, broadcasting machinery will fall back to producing Arrays. If this is undesirable, you may need to define binary BroadcastStyle rules to control the output type.\n\nSee also Broadcast.DefaultArrayStyle.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.ArrayStyle","location":"base/arrays.html#Base.Broadcast.ArrayStyle","category":"type","text":"Broadcast.ArrayStyle{MyArrayType}() is a BroadcastStyle indicating that an object behaves as an array for broadcasting. It presents a simple way to construct Broadcast.AbstractArrayStyles for specific AbstractArray container types. Broadcast styles created this way lose track of dimensionality; if keeping track is important for your type, you should create your own custom Broadcast.AbstractArrayStyle.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.DefaultArrayStyle","location":"base/arrays.html#Base.Broadcast.DefaultArrayStyle","category":"type","text":"Broadcast.DefaultArrayStyle{N}() is a BroadcastStyle indicating that an object behaves as an N-dimensional array for broadcasting. Specifically, DefaultArrayStyle is used for any AbstractArray type that hasn't defined a specialized style, and in the absence of overrides from other broadcast arguments the resulting output type is Array. When there are multiple inputs to broadcast, DefaultArrayStyle \"loses\" to any other Broadcast.ArrayStyle.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.broadcastable","location":"base/arrays.html#Base.Broadcast.broadcastable","category":"function","text":"Broadcast.broadcastable(x)\n\nReturn either x or an object like x such that it supports axes, indexing, and its type supports ndims.\n\nIf x supports iteration, the returned value should have the same axes and indexing behaviors as collect(x).\n\nIf x is not an AbstractArray but it supports axes, indexing, and its type supports ndims, then broadcastable(::typeof(x)) may be implemented to just return itself. Further, if x defines its own BroadcastStyle, then it must define its broadcastable method to return itself for the custom style to have any effect.\n\nExamples\n\njulia> Broadcast.broadcastable([1,2,3]) # like `identity` since arrays already support axes and indexing\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> Broadcast.broadcastable(Int) # Types don't support axes, indexing, or iteration but are commonly used as scalars\nBase.RefValue{Type{Int64}}(Int64)\n\njulia> Broadcast.broadcastable(\"hello\") # Strings break convention of matching iteration and act like a scalar instead\nBase.RefValue{String}(\"hello\")\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.combine_axes","location":"base/arrays.html#Base.Broadcast.combine_axes","category":"function","text":"combine_axes(As...) -> Tuple\n\nDetermine the result axes for broadcasting across all values in As.\n\njulia> Broadcast.combine_axes([1], [1 2; 3 4; 5 6])\n(Base.OneTo(3), Base.OneTo(2))\n\njulia> Broadcast.combine_axes(1, 1, 1)\n()\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.combine_styles","location":"base/arrays.html#Base.Broadcast.combine_styles","category":"function","text":"combine_styles(cs...) -> BroadcastStyle\n\nDecides which BroadcastStyle to use for any number of value arguments. Uses BroadcastStyle to get the style for each argument, and uses result_style to combine styles.\n\nExamples\n\njulia> Broadcast.combine_styles([1], [1 2; 3 4])\nBase.Broadcast.DefaultArrayStyle{2}()\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Broadcast.result_style","location":"base/arrays.html#Base.Broadcast.result_style","category":"function","text":"result_style(s1::BroadcastStyle[, s2::BroadcastStyle]) -> BroadcastStyle\n\nTakes one or two BroadcastStyles and combines them using BroadcastStyle to determine a common BroadcastStyle.\n\nExamples\n\njulia> Broadcast.result_style(Broadcast.DefaultArrayStyle{0}(), Broadcast.DefaultArrayStyle{3}())\nBase.Broadcast.DefaultArrayStyle{3}()\n\njulia> Broadcast.result_style(Broadcast.Unknown(), Broadcast.DefaultArrayStyle{1}())\nBase.Broadcast.DefaultArrayStyle{1}()\n\n\n\n\n\n","page":"Arrays"},{"title":"Indexing and assignment","location":"base/arrays.html#Indexing-and-assignment","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.getindex(::AbstractArray, ::Any...)\nBase.setindex!(::AbstractArray, ::Any, ::Any...)\nBase.copyto!(::AbstractArray, ::CartesianIndices, ::AbstractArray, ::CartesianIndices)\nBase.copy!\nBase.isassigned\nBase.Colon\nBase.CartesianIndex\nBase.CartesianIndices\nBase.Dims\nBase.LinearIndices\nBase.to_indices\nBase.checkbounds\nBase.checkindex\nBase.elsize","page":"Arrays"},{"title":"Base.getindex","location":"base/arrays.html#Base.getindex-Tuple{AbstractArray, Vararg{Any}}","category":"method","text":"getindex(A, inds...)\n\nReturn a subset of array A as specified by inds, where each ind may be, for example, an Int, an AbstractRange, or a Vector. See the manual section on array indexing for details.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> getindex(A, 1)\n1\n\njulia> getindex(A, [2, 1])\n2-element Vector{Int64}:\n 3\n 1\n\njulia> getindex(A, 2:4)\n3-element Vector{Int64}:\n 3\n 2\n 4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.setindex!","location":"base/arrays.html#Base.setindex!-Tuple{AbstractArray, Any, Vararg{Any}}","category":"method","text":"setindex!(A, X, inds...)\nA[inds...] = X\n\nStore values from array X within some subset of A as specified by inds. The syntax A[inds...] = X is equivalent to (setindex!(A, X, inds...); X).\n\nExamples\n\njulia> A = zeros(2,2);\n\njulia> setindex!(A, [10, 20], [1, 2]);\n\njulia> A[[3, 4]] = [30, 40];\n\njulia> A\n2×2 Matrix{Float64}:\n 10.0  30.0\n 20.0  40.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.copyto!","location":"base/arrays.html#Base.copyto!-Tuple{AbstractArray, CartesianIndices, AbstractArray, CartesianIndices}","category":"method","text":"copyto!(dest, Rdest::CartesianIndices, src, Rsrc::CartesianIndices) -> dest\n\nCopy the block of src in the range of Rsrc to the block of dest in the range of Rdest. The sizes of the two regions must match.\n\nExamples\n\njulia> A = zeros(5, 5);\n\njulia> B = [1 2; 3 4];\n\njulia> Ainds = CartesianIndices((2:3, 2:3));\n\njulia> Binds = CartesianIndices(B);\n\njulia> copyto!(A, Ainds, B, Binds)\n5×5 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0\n 0.0  1.0  2.0  0.0  0.0\n 0.0  3.0  4.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.copy!","location":"base/arrays.html#Base.copy!","category":"function","text":"copy!(dst, src) -> dst\n\nIn-place copy of src into dst, discarding any pre-existing elements in dst. If dst and src are of the same type, dst == src should hold after the call. If dst and src are multidimensional arrays, they must have equal axes.\n\nSee also copyto!.\n\ncompat: Julia 1.1\nThis method requires at least Julia 1.1. In Julia 1.0 this method is available from the Future standard library as Future.copy!.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.isassigned","location":"base/arrays.html#Base.isassigned","category":"function","text":"isassigned(array, i) -> Bool\n\nTest whether the given array has a value associated with index i. Return false if the index is out of bounds, or has an undefined reference.\n\nExamples\n\njulia> isassigned(rand(3, 3), 5)\ntrue\n\njulia> isassigned(rand(3, 3), 3 * 3 + 1)\nfalse\n\njulia> mutable struct Foo end\n\njulia> v = similar(rand(3), Foo)\n3-element Vector{Foo}:\n #undef\n #undef\n #undef\n\njulia> isassigned(v, 1)\nfalse\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Colon","location":"base/arrays.html#Base.Colon","category":"type","text":"Colon()\n\nColons (:) are used to signify indexing entire objects or dimensions at once.\n\nVery few operations are defined on Colons directly; instead they are converted by to_indices to an internal vector type (Base.Slice) to represent the collection of indices they span before being used.\n\nThe singleton instance of Colon is also a function used to construct ranges; see :.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.IteratorsMD.CartesianIndex","location":"base/arrays.html#Base.IteratorsMD.CartesianIndex","category":"type","text":"CartesianIndex(i, j, k...)   -> I\nCartesianIndex((i, j, k...)) -> I\n\nCreate a multidimensional index I, which can be used for indexing a multidimensional array A.  In particular, A[I] is equivalent to A[i,j,k...].  One can freely mix integer and CartesianIndex indices; for example, A[Ipre, i, Ipost] (where Ipre and Ipost are CartesianIndex indices and i is an Int) can be a useful expression when writing algorithms that work along a single dimension of an array of arbitrary dimensionality.\n\nA CartesianIndex is sometimes produced by eachindex, and always when iterating with an explicit CartesianIndices.\n\nExamples\n\njulia> A = reshape(Vector(1:16), (2, 2, 2, 2))\n2×2×2×2 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1  3\n 2  4\n\n[:, :, 2, 1] =\n 5  7\n 6  8\n\n[:, :, 1, 2] =\n  9  11\n 10  12\n\n[:, :, 2, 2] =\n 13  15\n 14  16\n\njulia> A[CartesianIndex((1, 1, 1, 1))]\n1\n\njulia> A[CartesianIndex((1, 1, 1, 2))]\n9\n\njulia> A[CartesianIndex((1, 1, 2, 1))]\n5\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.IteratorsMD.CartesianIndices","location":"base/arrays.html#Base.IteratorsMD.CartesianIndices","category":"type","text":"CartesianIndices(sz::Dims) -> R\nCartesianIndices((istart:[istep:]istop, jstart:[jstep:]jstop, ...)) -> R\n\nDefine a region R spanning a multidimensional rectangular range of integer indices. These are most commonly encountered in the context of iteration, where for I in R ... end will return CartesianIndex indices I equivalent to the nested loops\n\nfor j = jstart:jstep:jstop\n    for i = istart:istep:istop\n        ...\n    end\nend\n\nConsequently these can be useful for writing algorithms that work in arbitrary dimensions.\n\nCartesianIndices(A::AbstractArray) -> R\n\nAs a convenience, constructing a CartesianIndices from an array makes a range of its indices.\n\ncompat: Julia 1.6\nThe step range method CartesianIndices((istart:istep:istop, jstart:[jstep:]jstop, ...)) requires at least Julia 1.6.\n\nExamples\n\njulia> foreach(println, CartesianIndices((2, 2, 2)))\nCartesianIndex(1, 1, 1)\nCartesianIndex(2, 1, 1)\nCartesianIndex(1, 2, 1)\nCartesianIndex(2, 2, 1)\nCartesianIndex(1, 1, 2)\nCartesianIndex(2, 1, 2)\nCartesianIndex(1, 2, 2)\nCartesianIndex(2, 2, 2)\n\njulia> CartesianIndices(fill(1, (2,3)))\nCartesianIndices((2, 3))\n\nConversion between linear and cartesian indices\n\nLinear index to cartesian index conversion exploits the fact that a CartesianIndices is an AbstractArray and can be indexed linearly:\n\njulia> cartesian = CartesianIndices((1:3, 1:2))\nCartesianIndices((1:3, 1:2))\n\njulia> cartesian[4]\nCartesianIndex(1, 2)\n\njulia> cartesian = CartesianIndices((1:2:5, 1:2))\nCartesianIndices((1:2:5, 1:2))\n\njulia> cartesian[2, 2]\nCartesianIndex(3, 2)\n\nBroadcasting\n\nCartesianIndices support broadcasting arithmetic (+ and -) with a CartesianIndex.\n\ncompat: Julia 1.1\nBroadcasting of CartesianIndices requires at least Julia 1.1.\n\njulia> CIs = CartesianIndices((2:3, 5:6))\nCartesianIndices((2:3, 5:6))\n\njulia> CI = CartesianIndex(3, 4)\nCartesianIndex(3, 4)\n\njulia> CIs .+ CI\nCartesianIndices((5:6, 9:10))\n\nFor cartesian to linear index conversion, see LinearIndices.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.Dims","location":"base/arrays.html#Base.Dims","category":"type","text":"Dims{N}\n\nAn NTuple of N Ints used to represent the dimensions of an AbstractArray.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.LinearIndices","location":"base/arrays.html#Base.LinearIndices","category":"type","text":"LinearIndices(A::AbstractArray)\n\nReturn a LinearIndices array with the same shape and axes as A, holding the linear index of each entry in A. Indexing this array with cartesian indices allows mapping them to linear indices.\n\nFor arrays with conventional indexing (indices start at 1), or any multidimensional array, linear indices range from 1 to length(A). However, for AbstractVectors linear indices are axes(A, 1), and therefore do not start at 1 for vectors with unconventional indexing.\n\nCalling this function is the \"safe\" way to write algorithms that exploit linear indexing.\n\nExamples\n\njulia> A = fill(1, (5,6,7));\n\njulia> b = LinearIndices(A);\n\njulia> extrema(b)\n(1, 210)\n\nLinearIndices(inds::CartesianIndices) -> R\nLinearIndices(sz::Dims) -> R\nLinearIndices((istart:istop, jstart:jstop, ...)) -> R\n\nReturn a LinearIndices array with the specified shape or axes.\n\nExample\n\nThe main purpose of this constructor is intuitive conversion from cartesian to linear indexing:\n\njulia> linear = LinearIndices((1:3, 1:2))\n3×2 LinearIndices{2, Tuple{UnitRange{Int64}, UnitRange{Int64}}}:\n 1  4\n 2  5\n 3  6\n\njulia> linear[1,2]\n4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.to_indices","location":"base/arrays.html#Base.to_indices","category":"function","text":"to_indices(A, I::Tuple)\n\nConvert the tuple I to a tuple of indices for use in indexing into array A.\n\nThe returned tuple must only contain either Ints or AbstractArrays of scalar indices that are supported by array A. It will error upon encountering a novel index type that it does not know how to process.\n\nFor simple index types, it defers to the unexported Base.to_index(A, i) to process each index i. While this internal function is not intended to be called directly, Base.to_index may be extended by custom array or index types to provide custom indexing behaviors.\n\nMore complicated index types may require more context about the dimension into which they index. To support those cases, to_indices(A, I) calls to_indices(A, axes(A), I), which then recursively walks through both the given tuple of indices and the dimensional indices of A in tandem. As such, not all index types are guaranteed to propagate to Base.to_index.\n\nExamples\n\njulia> A = zeros(1,2,3,4);\n\njulia> to_indices(A, (1,1,2,2))\n(1, 1, 2, 2)\n\njulia> to_indices(A, (1,1,2,20)) # no bounds checking\n(1, 1, 2, 20)\n\njulia> to_indices(A, (CartesianIndex((1,)), 2, CartesianIndex((3,4)))) # exotic index\n(1, 2, 3, 4)\n\njulia> to_indices(A, ([1,1], 1:2, 3, 4))\n([1, 1], 1:2, 3, 4)\n\njulia> to_indices(A, (1,2)) # no shape checking\n(1, 2)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.checkbounds","location":"base/arrays.html#Base.checkbounds","category":"function","text":"checkbounds(Bool, A, I...)\n\nReturn true if the specified indices I are in bounds for the given array A. Subtypes of AbstractArray should specialize this method if they need to provide custom bounds checking behaviors; however, in many cases one can rely on A's indices and checkindex.\n\nSee also checkindex.\n\nExamples\n\njulia> A = rand(3, 3);\n\njulia> checkbounds(Bool, A, 2)\ntrue\n\njulia> checkbounds(Bool, A, 3, 4)\nfalse\n\njulia> checkbounds(Bool, A, 1:3)\ntrue\n\njulia> checkbounds(Bool, A, 1:3, 2:4)\nfalse\n\n\n\n\n\ncheckbounds(A, I...)\n\nThrow an error if the specified indices I are not in bounds for the given array A.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.checkindex","location":"base/arrays.html#Base.checkindex","category":"function","text":"checkindex(Bool, inds::AbstractUnitRange, index)\n\nReturn true if the given index is within the bounds of inds. Custom types that would like to behave as indices for all arrays can extend this method in order to provide a specialized bounds checking implementation.\n\nSee also checkbounds.\n\nExamples\n\njulia> checkindex(Bool, 1:20, 8)\ntrue\n\njulia> checkindex(Bool, 1:20, 21)\nfalse\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.elsize","location":"base/arrays.html#Base.elsize","category":"function","text":"elsize(type)\n\nCompute the memory stride in bytes between consecutive elements of eltype stored inside the given type, if the array elements are stored densely with a uniform linear stride.\n\nExamples\n\njulia> Base.elsize(rand(Float32, 10))\n4\n\n\n\n\n\n","page":"Arrays"},{"title":"Views (SubArrays and other view types)","location":"base/arrays.html#Views-(SubArrays-and-other-view-types)","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"A “view” is a data structure that acts like an array (it is a subtype of AbstractArray), but the underlying data is actually part of another array.","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"For example, if x is an array and v = @view x[1:10], then v acts like a 10-element array, but its data is actually accessing the first 10 elements of x. Writing to a view, e.g. v[3] = 2, writes directly to the underlying array x (in this case modifying x[3]).","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Slicing operations like x[1:10] create a copy by default in Julia. @view x[1:10] changes it to make a view. The @views macro can be used on a whole block of code (e.g. @views function foo() .... end or @views begin ... end) to change all the slicing operations in that block to use views.  Sometimes making a copy of the data is faster and sometimes using a view is faster, as described in the performance tips.","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.view\nBase.@view\nBase.@views\nBase.parent\nBase.parentindices\nBase.selectdim\nBase.reinterpret\nBase.reshape\nBase.dropdims\nBase.vec\nBase.SubArray","page":"Arrays"},{"title":"Base.view","location":"base/arrays.html#Base.view","category":"function","text":"view(A, inds...)\n\nLike getindex, but returns a lightweight array that lazily references (or is effectively a view into) the parent array A at the given index or indices inds instead of eagerly extracting elements or constructing a copied subset. Calling getindex or setindex! on the returned value (often a SubArray) computes the indices to access or modify the parent array on the fly.  The behavior is undefined if the shape of the parent array is changed after view is called because there is no bound check for the parent array; e.g., it may cause a segmentation fault.\n\nSome immutable parent arrays (like ranges) may choose to simply recompute a new array in some circumstances instead of returning a SubArray if doing so is efficient and provides compatible semantics.\n\ncompat: Julia 1.6\nIn Julia 1.6 or later, view can be called on an AbstractString, returning a SubString.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> b = view(A, :, 1)\n2-element view(::Matrix{Int64}, :, 1) with eltype Int64:\n 1\n 3\n\njulia> fill!(b, 0)\n2-element view(::Matrix{Int64}, :, 1) with eltype Int64:\n 0\n 0\n\njulia> A # Note A has changed even though we modified b\n2×2 Matrix{Int64}:\n 0  2\n 0  4\n\njulia> view(2:5, 2:3) # returns a range as type is immutable\n3:4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.@view","location":"base/arrays.html#Base.@view","category":"macro","text":"@view A[inds...]\n\nTransform the indexing expression A[inds...] into the equivalent view call.\n\nThis can only be applied directly to a single indexing expression and is particularly helpful for expressions that include the special begin or end indexing syntaxes like A[begin, 2:end-1] (as those are not supported by the normal view function).\n\nNote that @view cannot be used as the target of a regular assignment (e.g., @view(A[1, 2:end]) = ...), nor would the un-decorated indexed assignment (A[1, 2:end] = ...) or broadcasted indexed assignment (A[1, 2:end] .= ...) make a copy.  It can be useful, however, for updating broadcasted assignments like @view(A[1, 2:end]) .+= 1 because this is a simple syntax for @view(A[1, 2:end]) .= @view(A[1, 2:end]) + 1, and the indexing expression on the right-hand side would otherwise make a copy without the @view.\n\nSee also @views to switch an entire block of code to use views for non-scalar indexing.\n\ncompat: Julia 1.5\nUsing begin in an indexing expression to refer to the first index requires at least Julia 1.5.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> b = @view A[:, 1]\n2-element view(::Matrix{Int64}, :, 1) with eltype Int64:\n 1\n 3\n\njulia> fill!(b, 0)\n2-element view(::Matrix{Int64}, :, 1) with eltype Int64:\n 0\n 0\n\njulia> A\n2×2 Matrix{Int64}:\n 0  2\n 0  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.@views","location":"base/arrays.html#Base.@views","category":"macro","text":"@views expression\n\nConvert every array-slicing operation in the given expression (which may be a begin/end block, loop, function, etc.) to return a view. Scalar indices, non-array types, and explicit getindex calls (as opposed to array[...]) are unaffected.\n\nSimilarly, @views converts string slices into SubString views.\n\nnote: Note\nThe @views macro only affects array[...] expressions that appear explicitly in the given expression, not array slicing that occurs in functions called by that code.\n\ncompat: Julia 1.5\nUsing begin in an indexing expression to refer to the first index requires at least Julia 1.5.\n\nExamples\n\njulia> A = zeros(3, 3);\n\njulia> @views for row in 1:3\n           b = A[row, :]\n           b[:] .= row\n       end\n\njulia> A\n3×3 Matrix{Float64}:\n 1.0  1.0  1.0\n 2.0  2.0  2.0\n 3.0  3.0  3.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.parent","location":"base/arrays.html#Base.parent","category":"function","text":"parent(A)\n\nReturn the underlying \"parent array”. This parent array of objects of types SubArray, ReshapedArray or LinearAlgebra.Transpose is what was passed as an argument to view, reshape, transpose, etc. during object creation. If the input is not a wrapped object, return the input itself. If the input is wrapped multiple times, only the outermost wrapper will be removed.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> V = view(A, 1:2, :)\n2×2 view(::Matrix{Int64}, 1:2, :) with eltype Int64:\n 1  2\n 3  4\n\njulia> parent(V)\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.parentindices","location":"base/arrays.html#Base.parentindices","category":"function","text":"parentindices(A)\n\nReturn the indices in the parent which correspond to the array view A.\n\nExamples\n\njulia> A = [1 2; 3 4];\n\njulia> V = view(A, 1, :)\n2-element view(::Matrix{Int64}, 1, :) with eltype Int64:\n 1\n 2\n\njulia> parentindices(V)\n(1, Base.Slice(Base.OneTo(2)))\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.selectdim","location":"base/arrays.html#Base.selectdim","category":"function","text":"selectdim(A, d::Integer, i)\n\nReturn a view of all the data of A where the index for dimension d equals i.\n\nEquivalent to view(A,:,:,...,i,:,:,...) where i is in position d.\n\nSee also: eachslice.\n\nExamples\n\njulia> A = [1 2 3 4; 5 6 7 8]\n2×4 Matrix{Int64}:\n 1  2  3  4\n 5  6  7  8\n\njulia> selectdim(A, 2, 3)\n2-element view(::Matrix{Int64}, :, 3) with eltype Int64:\n 3\n 7\n\njulia> selectdim(A, 2, 3:4)\n2×2 view(::Matrix{Int64}, :, 3:4) with eltype Int64:\n 3  4\n 7  8\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.reinterpret","location":"base/arrays.html#Base.reinterpret","category":"function","text":"reinterpret(type, A)\n\nChange the type-interpretation of the binary data in the primitive type A to that of the primitive type type. The size of type has to be the same as that of the type of A. For example, reinterpret(Float32, UInt32(7)) interprets the 4 bytes corresponding to UInt32(7) as a Float32.\n\nExamples\n\njulia> reinterpret(Float32, UInt32(7))\n1.0f-44\n\n\n\n\n\nreinterpret(reshape, T, A::AbstractArray{S}) -> B\n\nChange the type-interpretation of A while consuming or adding a \"channel dimension.\"\n\nIf sizeof(T) = n*sizeof(S) for n>1, A's first dimension must be of size n and B lacks A's first dimension. Conversely, if sizeof(S) = n*sizeof(T) for n>1, B gets a new first dimension of size n. The dimensionality is unchanged if sizeof(T) == sizeof(S).\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int32}:\n 1  2\n 3  4\n\njulia> reinterpret(reshape, Complex{Int}, A)    # the result is a vector\n2-element reinterpret(reshape, Complex{Int32}, ::Matrix{Int32}) with eltype Complex{Int32}:\n 1 + 3im\n 2 + 4im\n\njulia> a = [(1,2,3), (4,5,6)]\n2-element Vector{Tuple{Int32, Int32, Int32}}:\n (1, 2, 3)\n (4, 5, 6)\n\njulia> reinterpret(reshape, Int, a)             # the result is a matrix\n3×2 reinterpret(reshape, Int32, ::Vector{Tuple{Int32, Int32, Int32}}) with eltype Int32:\n 1  4\n 2  5\n 3  6\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.reshape","location":"base/arrays.html#Base.reshape","category":"function","text":"reshape(A, dims...) -> AbstractArray\nreshape(A, dims) -> AbstractArray\n\nReturn an array with the same data as A, but with different dimension sizes or number of dimensions. The two arrays share the same underlying data, so that the result is mutable if and only if A is mutable, and setting elements of one alters the values of the other.\n\nThe new dimensions may be specified either as a list of arguments or as a shape tuple. At most one dimension may be specified with a :, in which case its length is computed such that its product with all the specified dimensions is equal to the length of the original array A. The total number of elements must not change.\n\nExamples\n\njulia> A = Vector(1:16)\n16-element Vector{Int64}:\n  1\n  2\n  3\n  4\n  5\n  6\n  7\n  8\n  9\n 10\n 11\n 12\n 13\n 14\n 15\n 16\n\njulia> reshape(A, (4, 4))\n4×4 Matrix{Int64}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> reshape(A, 2, :)\n2×8 Matrix{Int64}:\n 1  3  5  7   9  11  13  15\n 2  4  6  8  10  12  14  16\n\njulia> reshape(1:6, 2, 3)\n2×3 reshape(::UnitRange{Int64}, 2, 3) with eltype Int64:\n 1  3  5\n 2  4  6\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.dropdims","location":"base/arrays.html#Base.dropdims","category":"function","text":"dropdims(A; dims)\n\nReturn an array with the same data as A, but with the dimensions specified by dims removed. size(A,d) must equal 1 for every d in dims, and repeated dimensions or numbers outside 1:ndims(A) are forbidden.\n\nThe result shares the same underlying data as A, such that the result is mutable if and only if A is mutable, and setting elements of one alters the values of the other.\n\nSee also: reshape, vec.\n\nExamples\n\njulia> a = reshape(Vector(1:4),(2,2,1,1))\n2×2×1×1 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1  3\n 2  4\n\njulia> b = dropdims(a; dims=3)\n2×2×1 Array{Int64, 3}:\n[:, :, 1] =\n 1  3\n 2  4\n\njulia> b[1,1,1] = 5; a\n2×2×1×1 Array{Int64, 4}:\n[:, :, 1, 1] =\n 5  3\n 2  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.vec","location":"base/arrays.html#Base.vec","category":"function","text":"vec(a::AbstractArray) -> AbstractVector\n\nReshape the array a as a one-dimensional column vector. Return a if it is already an AbstractVector. The resulting array shares the same underlying data as a, so it will only be mutable if a is mutable, in which case modifying one will also modify the other.\n\nExamples\n\njulia> a = [1 2 3; 4 5 6]\n2×3 Matrix{Int64}:\n 1  2  3\n 4  5  6\n\njulia> vec(a)\n6-element Vector{Int64}:\n 1\n 4\n 2\n 5\n 3\n 6\n\njulia> vec(1:3)\n1:3\n\nSee also reshape, dropdims.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.SubArray","location":"base/arrays.html#Base.SubArray","category":"type","text":"SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n\nN-dimensional view into a parent array (of type P) with an element type T, restricted by a tuple of indices (of type I). L is true for types that support fast linear indexing, and false otherwise.\n\nConstruct SubArrays using the view function.\n\n\n\n\n\n","page":"Arrays"},{"title":"Concatenation and permutation","location":"base/arrays.html#Concatenation-and-permutation","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.cat\nBase.vcat\nBase.hcat\nBase.hvcat\nBase.hvncat\nBase.stack\nBase.vect\nBase.circshift\nBase.circshift!\nBase.circcopy!\nBase.findall(::Any)\nBase.findall(::Function, ::Any)\nBase.findfirst(::Any)\nBase.findfirst(::Function, ::Any)\nBase.findlast(::Any)\nBase.findlast(::Function, ::Any)\nBase.findnext(::Any, ::Integer)\nBase.findnext(::Function, ::Any, ::Integer)\nBase.findprev(::Any, ::Integer)\nBase.findprev(::Function, ::Any, ::Integer)\nBase.permutedims\nBase.permutedims!\nBase.PermutedDimsArray\nBase.promote_shape","page":"Arrays"},{"title":"Base.cat","location":"base/arrays.html#Base.cat","category":"function","text":"cat(A...; dims)\n\nConcatenate the input arrays along the dimensions specified in dims.\n\nAlong a dimension d in dims, the size of the output array is sum(size(a,d) for a in A). Along other dimensions, all input arrays should have the same size, which will also be the size of the output array along those dimensions.\n\nIf dims is a single number, the different arrays are tightly packed along that dimension. If dims is an iterable containing several dimensions, the positions along these dimensions are increased simultaneously for each input array, filling with zero elsewhere. This allows one to construct block-diagonal matrices as cat(matrices...; dims=(1,2)), and their higher-dimensional analogues.\n\nThe special case dims=1 is vcat, and dims=2 is hcat. See also hvcat, hvncat, stack, repeat.\n\nThe keyword also accepts Val(dims).\n\ncompat: Julia 1.8\nFor multiple dimensions dims = Val(::Tuple) was added in Julia 1.8.\n\nExamples\n\njulia> cat([1 2; 3 4], [pi, pi], fill(10, 2,3,1); dims=2)  # same as hcat\n2×6×1 Array{Float64, 3}:\n[:, :, 1] =\n 1.0  2.0  3.14159  10.0  10.0  10.0\n 3.0  4.0  3.14159  10.0  10.0  10.0\n\njulia> cat(true, trues(2,2), trues(4)', dims=(1,2))  # block-diagonal\n4×7 Matrix{Bool}:\n 1  0  0  0  0  0  0\n 0  1  1  0  0  0  0\n 0  1  1  0  0  0  0\n 0  0  0  1  1  1  1\n\njulia> cat(1, [2], [3;;]; dims=Val(2))\n1×3 Matrix{Int64}:\n 1  2  3\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.vcat","location":"base/arrays.html#Base.vcat","category":"function","text":"vcat(A...)\n\nConcatenate arrays or numbers vertically. Equivalent to cat(A...; dims=1), and to the syntax [a; b; c].\n\nTo concatenate a large vector of arrays, reduce(vcat, A) calls an efficient method when A isa AbstractVector{<:AbstractVecOrMat}, rather than working pairwise.\n\nSee also hcat, Iterators.flatten, stack.\n\nExamples\n\njulia> v = vcat([1,2], [3,4])\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> v == vcat(1, 2, [3,4])  # accepts numbers\ntrue\n\njulia> v == [1; 2; [3,4]]  # syntax for the same operation\ntrue\n\njulia> summary(ComplexF64[1; 2; [3,4]])  # syntax for supplying the element type\n\"4-element Vector{ComplexF64}\"\n\njulia> vcat(range(1, 2, length=3))  # collects lazy ranges\n3-element Vector{Float64}:\n 1.0\n 1.5\n 2.0\n\njulia> two = ([10, 20, 30]', Float64[4 5 6; 7 8 9])  # row vector and a matrix\n([10 20 30], [4.0 5.0 6.0; 7.0 8.0 9.0])\n\njulia> vcat(two...)\n3×3 Matrix{Float64}:\n 10.0  20.0  30.0\n  4.0   5.0   6.0\n  7.0   8.0   9.0\n\njulia> vs = [[1, 2], [3, 4], [5, 6]];\n\njulia> reduce(vcat, vs)  # more efficient than vcat(vs...)\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\njulia> ans == collect(Iterators.flatten(vs))\ntrue\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.hcat","location":"base/arrays.html#Base.hcat","category":"function","text":"hcat(A...)\n\nConcatenate arrays or numbers horizontally. Equivalent to cat(A...; dims=2), and to the syntax [a b c] or [a;; b;; c].\n\nFor a large vector of arrays, reduce(hcat, A) calls an efficient method when A isa AbstractVector{<:AbstractVecOrMat}. For a vector of vectors, this can also be written stack(A).\n\nSee also vcat, hvcat.\n\nExamples\n\njulia> hcat([1,2], [3,4], [5,6])\n2×3 Matrix{Int64}:\n 1  3  5\n 2  4  6\n\njulia> hcat(1, 2, [30 40], [5, 6, 7]')  # accepts numbers\n1×7 Matrix{Int64}:\n 1  2  30  40  5  6  7\n\njulia> ans == [1 2 [30 40] [5, 6, 7]']  # syntax for the same operation\ntrue\n\njulia> Float32[1 2 [30 40] [5, 6, 7]']  # syntax for supplying the eltype\n1×7 Matrix{Float32}:\n 1.0  2.0  30.0  40.0  5.0  6.0  7.0\n\njulia> ms = [zeros(2,2), [1 2; 3 4], [50 60; 70 80]];\n\njulia> reduce(hcat, ms)  # more efficient than hcat(ms...)\n2×6 Matrix{Float64}:\n 0.0  0.0  1.0  2.0  50.0  60.0\n 0.0  0.0  3.0  4.0  70.0  80.0\n\njulia> stack(ms) |> summary  # disagrees on a vector of matrices\n\"2×2×3 Array{Float64, 3}\"\n\njulia> hcat(Int[], Int[], Int[])  # empty vectors, each of size (0,)\n0×3 Matrix{Int64}\n\njulia> hcat([1.1, 9.9], Matrix(undef, 2, 0))  # hcat with empty 2×0 Matrix\n2×1 Matrix{Any}:\n 1.1\n 9.9\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.hvcat","location":"base/arrays.html#Base.hvcat","category":"function","text":"hvcat(blocks_per_row::Union{Tuple{Vararg{Int}}, Int}, values...)\n\nHorizontal and vertical concatenation in one call. This function is called for block matrix syntax. The first argument specifies the number of arguments to concatenate in each block row. If the first argument is a single integer n, then all block rows are assumed to have n block columns.\n\nExamples\n\njulia> a, b, c, d, e, f = 1, 2, 3, 4, 5, 6\n(1, 2, 3, 4, 5, 6)\n\njulia> [a b c; d e f]\n2×3 Matrix{Int64}:\n 1  2  3\n 4  5  6\n\njulia> hvcat((3,3), a,b,c,d,e,f)\n2×3 Matrix{Int64}:\n 1  2  3\n 4  5  6\n\njulia> [a b; c d; e f]\n3×2 Matrix{Int64}:\n 1  2\n 3  4\n 5  6\n\njulia> hvcat((2,2,2), a,b,c,d,e,f)\n3×2 Matrix{Int64}:\n 1  2\n 3  4\n 5  6\njulia> hvcat((2,2,2), a,b,c,d,e,f) == hvcat(2, a,b,c,d,e,f)\ntrue\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.hvncat","location":"base/arrays.html#Base.hvncat","category":"function","text":"hvncat(dim::Int, row_first, values...)\nhvncat(dims::Tuple{Vararg{Int}}, row_first, values...)\nhvncat(shape::Tuple{Vararg{Tuple}}, row_first, values...)\n\nHorizontal, vertical, and n-dimensional concatenation of many values in one call.\n\nThis function is called for block matrix syntax. The first argument either specifies the shape of the concatenation, similar to hvcat, as a tuple of tuples, or the dimensions that specify the key number of elements along each axis, and is used to determine the output dimensions. The dims form is more performant, and is used by default when the concatenation operation has the same number of elements along each axis (e.g., [a b; c d;;; e f ; g h]). The shape form is used when the number of elements along each axis is unbalanced (e.g., [a b ; c]). Unbalanced syntax needs additional validation overhead. The dim form is an optimization for concatenation along just one dimension. row_first indicates how values are ordered. The meaning of the first and second elements of shape are also swapped based on row_first.\n\nExamples\n\njulia> a, b, c, d, e, f = 1, 2, 3, 4, 5, 6\n(1, 2, 3, 4, 5, 6)\n\njulia> [a b c;;; d e f]\n1×3×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  2  3\n\n[:, :, 2] =\n 4  5  6\n\njulia> hvncat((2,1,3), false, a,b,c,d,e,f)\n2×1×3 Array{Int64, 3}:\n[:, :, 1] =\n 1\n 2\n\n[:, :, 2] =\n 3\n 4\n\n[:, :, 3] =\n 5\n 6\n\njulia> [a b;;; c d;;; e f]\n1×2×3 Array{Int64, 3}:\n[:, :, 1] =\n 1  2\n\n[:, :, 2] =\n 3  4\n\n[:, :, 3] =\n 5  6\n\njulia> hvncat(((3, 3), (3, 3), (6,)), true, a, b, c, d, e, f)\n1×3×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  2  3\n\n[:, :, 2] =\n 4  5  6\n\nExamples for construction of the arguments\n\n[a b c ; d e f ;;;\n g h i ; j k l ;;;\n m n o ; p q r ;;;\n s t u ; v w x]\n⇒ dims = (2, 3, 4)\n\n[a b ; c ;;; d ;;;;]\n ___   _     _\n 2     1     1 = elements in each row (2, 1, 1)\n _______     _\n 3           1 = elements in each column (3, 1)\n _____________\n 4             = elements in each 3d slice (4,)\n _____________\n 4             = elements in each 4d slice (4,)\n⇒ shape = ((2, 1, 1), (3, 1), (4,), (4,)) with `row_first` = true\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.stack","location":"base/arrays.html#Base.stack","category":"function","text":"stack(iter; [dims])\n\nCombine a collection of arrays (or other iterable objects) of equal size into one larger array, by arranging them along one or more new dimensions.\n\nBy default the axes of the elements are placed first, giving size(result) = (size(first(iter))..., size(iter)...). This has the same order of elements as Iterators.flatten(iter).\n\nWith keyword dims::Integer, instead the ith element of iter becomes the slice selectdim(result, dims, i), so that size(result, dims) == length(iter). In this case stack reverses the action of eachslice with the same dims.\n\nThe various cat functions also combine arrays. However, these all extend the arrays' existing (possibly trivial) dimensions, rather than placing the arrays along new dimensions. They also accept arrays as separate arguments, rather than a single collection.\n\ncompat: Julia 1.9\nThis function requires at least Julia 1.9.\n\nExamples\n\njulia> vecs = (1:2, [30, 40], Float32[500, 600]);\n\njulia> mat = stack(vecs)\n2×3 Matrix{Float32}:\n 1.0  30.0  500.0\n 2.0  40.0  600.0\n\njulia> mat == hcat(vecs...) == reduce(hcat, collect(vecs))\ntrue\n\njulia> vec(mat) == vcat(vecs...) == reduce(vcat, collect(vecs))\ntrue\n\njulia> stack(zip(1:4, 10:99))  # accepts any iterators of iterators\n2×4 Matrix{Int64}:\n  1   2   3   4\n 10  11  12  13\n\njulia> vec(ans) == collect(Iterators.flatten(zip(1:4, 10:99)))\ntrue\n\njulia> stack(vecs; dims=1)  # unlike any cat function, 1st axis of vecs[1] is 2nd axis of result\n3×2 Matrix{Float32}:\n   1.0    2.0\n  30.0   40.0\n 500.0  600.0\n\njulia> x = rand(3,4);\n\njulia> x == stack(eachcol(x)) == stack(eachrow(x), dims=1)  # inverse of eachslice\ntrue\n\nHigher-dimensional examples:\n\njulia> A = rand(5, 7, 11);\n\njulia> E = eachslice(A, dims=2);  # a vector of matrices\n\njulia> (element = size(first(E)), container = size(E))\n(element = (5, 11), container = (7,))\n\njulia> stack(E) |> size\n(5, 11, 7)\n\njulia> stack(E) == stack(E; dims=3) == cat(E...; dims=3)\ntrue\n\njulia> A == stack(E; dims=2)\ntrue\n\njulia> M = (fill(10i+j, 2, 3) for i in 1:5, j in 1:7);\n\njulia> (element = size(first(M)), container = size(M))\n(element = (2, 3), container = (5, 7))\n\njulia> stack(M) |> size  # keeps all dimensions\n(2, 3, 5, 7)\n\njulia> stack(M; dims=1) |> size  # vec(container) along dims=1\n(35, 2, 3)\n\njulia> hvcat(5, M...) |> size  # hvcat puts matrices next to each other\n(14, 15)\n\n\n\n\n\nstack(f, args...; [dims])\n\nApply a function to each element of a collection, and stack the result. Or to several collections, zipped together.\n\nThe function should return arrays (or tuples, or other iterators) all of the same size. These become slices of the result, each separated along dims (if given) or by default along the last dimensions.\n\nSee also mapslices, eachcol.\n\nExamples\n\njulia> stack(c -> (c, c-32), \"julia\")\n2×5 Matrix{Char}:\n 'j'  'u'  'l'  'i'  'a'\n 'J'  'U'  'L'  'I'  'A'\n\njulia> stack(eachrow([1 2 3; 4 5 6]), (10, 100); dims=1) do row, n\n         vcat(row, row .* n, row ./ n)\n       end\n2×9 Matrix{Float64}:\n 1.0  2.0  3.0   10.0   20.0   30.0  0.1   0.2   0.3\n 4.0  5.0  6.0  400.0  500.0  600.0  0.04  0.05  0.06\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.vect","location":"base/arrays.html#Base.vect","category":"function","text":"vect(X...)\n\nCreate a Vector with element type computed from the promote_typeof of the argument, containing the argument list.\n\nExamples\n\njulia> a = Base.vect(UInt8(1), 2.5, 1//2)\n3-element Vector{Float64}:\n 1.0\n 2.5\n 0.5\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.circshift","location":"base/arrays.html#Base.circshift","category":"function","text":"circshift(A, shifts)\n\nCircularly shift, i.e. rotate, the data in an array. The second argument is a tuple or vector giving the amount to shift in each dimension, or an integer to shift only in the first dimension.\n\nSee also: circshift!, circcopy!, bitrotate, <<.\n\nExamples\n\njulia> b = reshape(Vector(1:16), (4,4))\n4×4 Matrix{Int64}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> circshift(b, (0,2))\n4×4 Matrix{Int64}:\n  9  13  1  5\n 10  14  2  6\n 11  15  3  7\n 12  16  4  8\n\njulia> circshift(b, (-1,0))\n4×4 Matrix{Int64}:\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n 1  5   9  13\n\njulia> a = BitArray([true, true, false, false, true])\n5-element BitVector:\n 1\n 1\n 0\n 0\n 1\n\njulia> circshift(a, 1)\n5-element BitVector:\n 1\n 1\n 1\n 0\n 0\n\njulia> circshift(a, -1)\n5-element BitVector:\n 1\n 0\n 0\n 1\n 1\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.circshift!","location":"base/arrays.html#Base.circshift!","category":"function","text":"circshift!(dest, src, shifts)\n\nCircularly shift, i.e. rotate, the data in src, storing the result in dest. shifts specifies the amount to shift in each dimension.\n\nThe dest array must be distinct from the src array (they cannot alias each other).\n\nSee also circshift.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.circcopy!","location":"base/arrays.html#Base.circcopy!","category":"function","text":"circcopy!(dest, src)\n\nCopy src to dest, indexing each dimension modulo its length. src and dest must have the same size, but can be offset in their indices; any offset results in a (circular) wraparound. If the arrays have overlapping indices, then on the domain of the overlap dest agrees with src.\n\nSee also: circshift.\n\nExamples\n\njulia> src = reshape(Vector(1:16), (4,4))\n4×4 Array{Int64,2}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> dest = OffsetArray{Int}(undef, (0:3,2:5))\n\njulia> circcopy!(dest, src)\nOffsetArrays.OffsetArray{Int64,2,Array{Int64,2}} with indices 0:3×2:5:\n 8  12  16  4\n 5   9  13  1\n 6  10  14  2\n 7  11  15  3\n\njulia> dest[1:3,2:4] == src[1:3,2:4]\ntrue\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findall","location":"base/arrays.html#Base.findall-Tuple{Any}","category":"method","text":"findall(A)\n\nReturn a vector I of the true indices or keys of A. If there are no such elements of A, return an empty array. To search for other kinds of values, pass a predicate as the first argument.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nSee also: findfirst, searchsorted.\n\nExamples\n\njulia> A = [true, false, false, true]\n4-element Vector{Bool}:\n 1\n 0\n 0\n 1\n\njulia> findall(A)\n2-element Vector{Int64}:\n 1\n 4\n\njulia> A = [true false; false true]\n2×2 Matrix{Bool}:\n 1  0\n 0  1\n\njulia> findall(A)\n2-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 1)\n CartesianIndex(2, 2)\n\njulia> findall(falses(3))\nInt64[]\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findall","location":"base/arrays.html#Base.findall-Tuple{Function, Any}","category":"method","text":"findall(f::Function, A)\n\nReturn a vector I of the indices or keys of A where f(A[I]) returns true. If there are no such elements of A, return an empty array.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> x = [1, 3, 4]\n3-element Vector{Int64}:\n 1\n 3\n 4\n\njulia> findall(isodd, x)\n2-element Vector{Int64}:\n 1\n 2\n\njulia> A = [1 2 0; 3 4 0]\n2×3 Matrix{Int64}:\n 1  2  0\n 3  4  0\njulia> findall(isodd, A)\n2-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 1)\n CartesianIndex(2, 1)\n\njulia> findall(!iszero, A)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 1)\n CartesianIndex(2, 1)\n CartesianIndex(1, 2)\n CartesianIndex(2, 2)\n\njulia> d = Dict(:A => 10, :B => -1, :C => 0)\nDict{Symbol, Int64} with 3 entries:\n  :A => 10\n  :B => -1\n  :C => 0\n\njulia> findall(x -> x >= 0, d)\n2-element Vector{Symbol}:\n :A\n :C\n\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findfirst","location":"base/arrays.html#Base.findfirst-Tuple{Any}","category":"method","text":"findfirst(A)\n\nReturn the index or key of the first true value in A. Return nothing if no such value is found. To search for other kinds of values, pass a predicate as the first argument.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nSee also: findall, findnext, findlast, searchsortedfirst.\n\nExamples\n\njulia> A = [false, false, true, false]\n4-element Vector{Bool}:\n 0\n 0\n 1\n 0\n\njulia> findfirst(A)\n3\n\njulia> findfirst(falses(3)) # returns nothing, but not printed in the REPL\n\njulia> A = [false false; true false]\n2×2 Matrix{Bool}:\n 0  0\n 1  0\n\njulia> findfirst(A)\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findfirst","location":"base/arrays.html#Base.findfirst-Tuple{Function, Any}","category":"method","text":"findfirst(predicate::Function, A)\n\nReturn the index or key of the first element of A for which predicate returns true. Return nothing if there is no such element.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> A = [1, 4, 2, 2]\n4-element Vector{Int64}:\n 1\n 4\n 2\n 2\n\njulia> findfirst(iseven, A)\n2\n\njulia> findfirst(x -> x>10, A) # returns nothing, but not printed in the REPL\n\njulia> findfirst(isequal(4), A)\n2\n\njulia> A = [1 4; 2 2]\n2×2 Matrix{Int64}:\n 1  4\n 2  2\n\njulia> findfirst(iseven, A)\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findlast","location":"base/arrays.html#Base.findlast-Tuple{Any}","category":"method","text":"findlast(A)\n\nReturn the index or key of the last true value in A. Return nothing if there is no true value in A.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nSee also: findfirst, findprev, findall.\n\nExamples\n\njulia> A = [true, false, true, false]\n4-element Vector{Bool}:\n 1\n 0\n 1\n 0\n\njulia> findlast(A)\n3\n\njulia> A = falses(2,2);\n\njulia> findlast(A) # returns nothing, but not printed in the REPL\n\njulia> A = [true false; true false]\n2×2 Matrix{Bool}:\n 1  0\n 1  0\n\njulia> findlast(A)\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findlast","location":"base/arrays.html#Base.findlast-Tuple{Function, Any}","category":"method","text":"findlast(predicate::Function, A)\n\nReturn the index or key of the last element of A for which predicate returns true. Return nothing if there is no such element.\n\nIndices or keys are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> A = [1, 2, 3, 4]\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> findlast(isodd, A)\n3\n\njulia> findlast(x -> x > 5, A) # returns nothing, but not printed in the REPL\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> findlast(isodd, A)\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findnext","location":"base/arrays.html#Base.findnext-Tuple{Any, Integer}","category":"method","text":"findnext(A, i)\n\nFind the next index after or including i of a true element of A, or nothing if not found.\n\nIndices are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> A = [false, false, true, false]\n4-element Vector{Bool}:\n 0\n 0\n 1\n 0\n\njulia> findnext(A, 1)\n3\n\njulia> findnext(A, 4) # returns nothing, but not printed in the REPL\n\njulia> A = [false false; true false]\n2×2 Matrix{Bool}:\n 0  0\n 1  0\n\njulia> findnext(A, CartesianIndex(1, 1))\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findnext","location":"base/arrays.html#Base.findnext-Tuple{Function, Any, Integer}","category":"method","text":"findnext(predicate::Function, A, i)\n\nFind the next index after or including i of an element of A for which predicate returns true, or nothing if not found.\n\nIndices are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> A = [1, 4, 2, 2];\n\njulia> findnext(isodd, A, 1)\n1\n\njulia> findnext(isodd, A, 2) # returns nothing, but not printed in the REPL\n\njulia> A = [1 4; 2 2];\n\njulia> findnext(isodd, A, CartesianIndex(1, 1))\nCartesianIndex(1, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findprev","location":"base/arrays.html#Base.findprev-Tuple{Any, Integer}","category":"method","text":"findprev(A, i)\n\nFind the previous index before or including i of a true element of A, or nothing if not found.\n\nIndices are of the same type as those returned by keys(A) and pairs(A).\n\nSee also: findnext, findfirst, findall.\n\nExamples\n\njulia> A = [false, false, true, true]\n4-element Vector{Bool}:\n 0\n 0\n 1\n 1\n\njulia> findprev(A, 3)\n3\n\njulia> findprev(A, 1) # returns nothing, but not printed in the REPL\n\njulia> A = [false false; true true]\n2×2 Matrix{Bool}:\n 0  0\n 1  1\n\njulia> findprev(A, CartesianIndex(2, 1))\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.findprev","location":"base/arrays.html#Base.findprev-Tuple{Function, Any, Integer}","category":"method","text":"findprev(predicate::Function, A, i)\n\nFind the previous index before or including i of an element of A for which predicate returns true, or nothing if not found.\n\nIndices are of the same type as those returned by keys(A) and pairs(A).\n\nExamples\n\njulia> A = [4, 6, 1, 2]\n4-element Vector{Int64}:\n 4\n 6\n 1\n 2\n\njulia> findprev(isodd, A, 1) # returns nothing, but not printed in the REPL\n\njulia> findprev(isodd, A, 3)\n3\n\njulia> A = [4 6; 1 2]\n2×2 Matrix{Int64}:\n 4  6\n 1  2\n\njulia> findprev(isodd, A, CartesianIndex(1, 2))\nCartesianIndex(2, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.permutedims","location":"base/arrays.html#Base.permutedims","category":"function","text":"permutedims(A::AbstractArray, perm)\n\nPermute the dimensions of array A. perm is a vector or a tuple of length ndims(A) specifying the permutation.\n\nSee also permutedims!, PermutedDimsArray, transpose, invperm.\n\nExamples\n\njulia> A = reshape(Vector(1:8), (2,2,2))\n2×2×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  3\n 2  4\n\n[:, :, 2] =\n 5  7\n 6  8\n\njulia> perm = (3, 1, 2); # put the last dimension first\n\njulia> B = permutedims(A, perm)\n2×2×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  2\n 5  6\n\n[:, :, 2] =\n 3  4\n 7  8\n\njulia> A == permutedims(B, invperm(perm)) # the inverse permutation\ntrue\n\nFor each dimension i of B = permutedims(A, perm), its corresponding dimension of A will be perm[i]. This means the equality size(B, i) == size(A, perm[i]) holds.\n\njulia> A = randn(5, 7, 11, 13);\n\njulia> perm = [4, 1, 3, 2];\n\njulia> B = permutedims(A, perm);\n\njulia> size(B)\n(13, 5, 11, 7)\n\njulia> size(A)[perm] == ans\ntrue\n\n\n\n\n\npermutedims(m::AbstractMatrix)\n\nPermute the dimensions of the matrix m, by flipping the elements across the diagonal of the matrix. Differs from LinearAlgebra's transpose in that the operation is not recursive.\n\nExamples\n\njulia> a = [1 2; 3 4];\n\njulia> b = [5 6; 7 8];\n\njulia> c = [9 10; 11 12];\n\njulia> d = [13 14; 15 16];\n\njulia> X = [[a] [b]; [c] [d]]\n2×2 Matrix{Matrix{Int64}}:\n [1 2; 3 4]     [5 6; 7 8]\n [9 10; 11 12]  [13 14; 15 16]\n\njulia> permutedims(X)\n2×2 Matrix{Matrix{Int64}}:\n [1 2; 3 4]  [9 10; 11 12]\n [5 6; 7 8]  [13 14; 15 16]\n\njulia> transpose(X)\n2×2 transpose(::Matrix{Matrix{Int64}}) with eltype Transpose{Int64, Matrix{Int64}}:\n [1 3; 2 4]  [9 11; 10 12]\n [5 7; 6 8]  [13 15; 14 16]\n\n\n\n\n\npermutedims(v::AbstractVector)\n\nReshape vector v into a 1 × length(v) row matrix. Differs from LinearAlgebra's transpose in that the operation is not recursive.\n\nExamples\n\njulia> permutedims([1, 2, 3, 4])\n1×4 Matrix{Int64}:\n 1  2  3  4\n\njulia> V = [[[1 2; 3 4]]; [[5 6; 7 8]]]\n2-element Vector{Matrix{Int64}}:\n [1 2; 3 4]\n [5 6; 7 8]\n\njulia> permutedims(V)\n1×2 Matrix{Matrix{Int64}}:\n [1 2; 3 4]  [5 6; 7 8]\n\njulia> transpose(V)\n1×2 transpose(::Vector{Matrix{Int64}}) with eltype Transpose{Int64, Matrix{Int64}}:\n [1 3; 2 4]  [5 7; 6 8]\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.permutedims!","location":"base/arrays.html#Base.permutedims!","category":"function","text":"permutedims!(dest, src, perm)\n\nPermute the dimensions of array src and store the result in the array dest. perm is a vector specifying a permutation of length ndims(src). The preallocated array dest should have size(dest) == size(src)[perm] and is completely overwritten. No in-place permutation is supported and unexpected results will happen if src and dest have overlapping memory regions.\n\nSee also permutedims.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.PermutedDimsArrays.PermutedDimsArray","location":"base/arrays.html#Base.PermutedDimsArrays.PermutedDimsArray","category":"type","text":"PermutedDimsArray(A, perm) -> B\n\nGiven an AbstractArray A, create a view B such that the dimensions appear to be permuted. Similar to permutedims, except that no copying occurs (B shares storage with A).\n\nSee also permutedims, invperm.\n\nExamples\n\njulia> A = rand(3,5,4);\n\njulia> B = PermutedDimsArray(A, (3,1,2));\n\njulia> size(B)\n(4, 3, 5)\n\njulia> B[3,1,2] == A[1,2,3]\ntrue\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.promote_shape","location":"base/arrays.html#Base.promote_shape","category":"function","text":"promote_shape(s1, s2)\n\nCheck two array shapes for compatibility, allowing trailing singleton dimensions, and return whichever shape has more dimensions.\n\nExamples\n\njulia> a = fill(1, (3,4,1,1,1));\n\njulia> b = fill(1, (3,4));\n\njulia> promote_shape(a,b)\n(Base.OneTo(3), Base.OneTo(4), Base.OneTo(1), Base.OneTo(1), Base.OneTo(1))\n\njulia> promote_shape((2,3,1,4), (2, 3, 1, 4, 1))\n(2, 3, 1, 4, 1)\n\n\n\n\n\n","page":"Arrays"},{"title":"Array functions","location":"base/arrays.html#Array-functions","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.accumulate\nBase.accumulate!\nBase.cumprod\nBase.cumprod!\nBase.cumsum\nBase.cumsum!\nBase.diff\nBase.repeat\nBase.rot180\nBase.rotl90\nBase.rotr90\nBase.mapslices\nBase.eachrow\nBase.eachcol\nBase.eachslice","page":"Arrays"},{"title":"Base.accumulate","location":"base/arrays.html#Base.accumulate","category":"function","text":"accumulate(op, A; dims::Integer, [init])\n\nCumulative operation op along the dimension dims of A (providing dims is optional for vectors). An initial value init may optionally be provided by a keyword argument. See also accumulate! to use a preallocated output array, both for performance and to control the precision of the output (e.g. to avoid overflow).\n\nFor common operations there are specialized variants of accumulate, see cumsum, cumprod. For a lazy version, see Iterators.accumulate.\n\ncompat: Julia 1.5\naccumulate on a non-array iterator requires at least Julia 1.5.\n\nExamples\n\njulia> accumulate(+, [1,2,3])\n3-element Vector{Int64}:\n 1\n 3\n 6\n\njulia> accumulate(min, (1, -2, 3, -4, 5), init=0)\n(0, -2, -2, -4, -4)\n\njulia> accumulate(/, (2, 4, Inf), init=100)\n(50.0, 12.5, 0.0)\n\njulia> accumulate(=>, i^2 for i in 1:3)\n3-element Vector{Any}:\n          1\n        1 => 4\n (1 => 4) => 9\n\njulia> accumulate(+, fill(1, 3, 4))\n3×4 Matrix{Int64}:\n 1  4  7  10\n 2  5  8  11\n 3  6  9  12\n\njulia> accumulate(+, fill(1, 2, 5), dims=2, init=100.0)\n2×5 Matrix{Float64}:\n 101.0  102.0  103.0  104.0  105.0\n 101.0  102.0  103.0  104.0  105.0\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.accumulate!","location":"base/arrays.html#Base.accumulate!","category":"function","text":"accumulate!(op, B, A; [dims], [init])\n\nCumulative operation op on A along the dimension dims, storing the result in B. Providing dims is optional for vectors.  If the keyword argument init is given, its value is used to instantiate the accumulation.\n\nSee also accumulate, cumsum!, cumprod!.\n\nExamples\n\njulia> x = [1, 0, 2, 0, 3];\n\njulia> y = rand(5);\n\njulia> accumulate!(+, y, x);\n\njulia> y\n5-element Vector{Float64}:\n 1.0\n 1.0\n 3.0\n 3.0\n 6.0\n\njulia> A = [1 2 3; 4 5 6];\n\njulia> B = similar(A);\n\njulia> accumulate!(-, B, A, dims=1)\n2×3 Matrix{Int64}:\n  1   2   3\n -3  -3  -3\n\njulia> accumulate!(*, B, A, dims=2, init=10)\n2×3 Matrix{Int64}:\n 10   20    60\n 40  200  1200\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.cumprod","location":"base/arrays.html#Base.cumprod","category":"function","text":"cumprod(A; dims::Integer)\n\nCumulative product along the dimension dim. See also cumprod! to use a preallocated output array, both for performance and to control the precision of the output (e.g. to avoid overflow).\n\nExamples\n\njulia> a = Int8[1 2 3; 4 5 6];\n\njulia> cumprod(a, dims=1)\n2×3 Matrix{Int64}:\n 1   2   3\n 4  10  18\n\njulia> cumprod(a, dims=2)\n2×3 Matrix{Int64}:\n 1   2    6\n 4  20  120\n\n\n\n\n\ncumprod(itr)\n\nCumulative product of an iterator.\n\nSee also cumprod!, accumulate, cumsum.\n\ncompat: Julia 1.5\ncumprod on a non-array iterator requires at least Julia 1.5.\n\nExamples\n\njulia> cumprod(fill(1//2, 3))\n3-element Vector{Rational{Int64}}:\n 1//2\n 1//4\n 1//8\n\njulia> cumprod((1, 2, 1, 3, 1))\n(1, 2, 2, 6, 6)\n\njulia> cumprod(\"julia\")\n5-element Vector{String}:\n \"j\"\n \"ju\"\n \"jul\"\n \"juli\"\n \"julia\"\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.cumprod!","location":"base/arrays.html#Base.cumprod!","category":"function","text":"cumprod!(B, A; dims::Integer)\n\nCumulative product of A along the dimension dims, storing the result in B. See also cumprod.\n\n\n\n\n\ncumprod!(y::AbstractVector, x::AbstractVector)\n\nCumulative product of a vector x, storing the result in y. See also cumprod.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.cumsum","location":"base/arrays.html#Base.cumsum","category":"function","text":"cumsum(A; dims::Integer)\n\nCumulative sum along the dimension dims. See also cumsum! to use a preallocated output array, both for performance and to control the precision of the output (e.g. to avoid overflow).\n\nExamples\n\njulia> a = [1 2 3; 4 5 6]\n2×3 Matrix{Int64}:\n 1  2  3\n 4  5  6\n\njulia> cumsum(a, dims=1)\n2×3 Matrix{Int64}:\n 1  2  3\n 5  7  9\n\njulia> cumsum(a, dims=2)\n2×3 Matrix{Int64}:\n 1  3   6\n 4  9  15\n\nnote: Note\nThe return array's eltype is Int for signed integers of less than system word size  and UInt for unsigned integers of less than system word size. To preserve eltype of arrays with small signed or unsigned integer accumulate(+, A) should be used.julia> cumsum(Int8[100, 28])\n2-element Vector{Int64}:\n 100\n 128\n\njulia> accumulate(+,Int8[100, 28])\n2-element Vector{Int8}:\n  100\n -128In the former case, the integers are widened to system word size and therefore the result is Int64[100, 128]. In the latter case, no such widening happens and integer overflow results in Int8[100, -128].\n\n\n\n\n\ncumsum(itr)\n\nCumulative sum of an iterator.\n\nSee also accumulate to apply functions other than +.\n\ncompat: Julia 1.5\ncumsum on a non-array iterator requires at least Julia 1.5.\n\nExamples\n\njulia> cumsum(1:3)\n3-element Vector{Int64}:\n 1\n 3\n 6\n\njulia> cumsum((true, false, true, false, true))\n(1, 1, 2, 2, 3)\n\njulia> cumsum(fill(1, 2) for i in 1:3)\n3-element Vector{Vector{Int64}}:\n [1, 1]\n [2, 2]\n [3, 3]\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.cumsum!","location":"base/arrays.html#Base.cumsum!","category":"function","text":"cumsum!(B, A; dims::Integer)\n\nCumulative sum of A along the dimension dims, storing the result in B. See also cumsum.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.diff","location":"base/arrays.html#Base.diff","category":"function","text":"diff(A::AbstractVector)\ndiff(A::AbstractArray; dims::Integer)\n\nFinite difference operator on a vector or a multidimensional array A. In the latter case the dimension to operate on needs to be specified with the dims keyword argument.\n\ncompat: Julia 1.1\ndiff for arrays with dimension higher than 2 requires at least Julia 1.1.\n\nExamples\n\njulia> a = [2 4; 6 16]\n2×2 Matrix{Int64}:\n 2   4\n 6  16\n\njulia> diff(a, dims=2)\n2×1 Matrix{Int64}:\n  2\n 10\n\njulia> diff(vec(a))\n3-element Vector{Int64}:\n  4\n -2\n 12\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.repeat","location":"base/arrays.html#Base.repeat","category":"function","text":"repeat(A::AbstractArray, counts::Integer...)\n\nConstruct an array by repeating array A a given number of times in each dimension, specified by counts.\n\nSee also: fill, Iterators.repeated, Iterators.cycle.\n\nExamples\n\njulia> repeat([1, 2, 3], 2)\n6-element Vector{Int64}:\n 1\n 2\n 3\n 1\n 2\n 3\n\njulia> repeat([1, 2, 3], 2, 3)\n6×3 Matrix{Int64}:\n 1  1  1\n 2  2  2\n 3  3  3\n 1  1  1\n 2  2  2\n 3  3  3\n\n\n\n\n\nrepeat(A::AbstractArray; inner=ntuple(Returns(1), ndims(A)), outer=ntuple(Returns(1), ndims(A)))\n\nConstruct an array by repeating the entries of A. The i-th element of inner specifies the number of times that the individual entries of the i-th dimension of A should be repeated. The i-th element of outer specifies the number of times that a slice along the i-th dimension of A should be repeated. If inner or outer are omitted, no repetition is performed.\n\nExamples\n\njulia> repeat(1:2, inner=2)\n4-element Vector{Int64}:\n 1\n 1\n 2\n 2\n\njulia> repeat(1:2, outer=2)\n4-element Vector{Int64}:\n 1\n 2\n 1\n 2\n\njulia> repeat([1 2; 3 4], inner=(2, 1), outer=(1, 3))\n4×6 Matrix{Int64}:\n 1  2  1  2  1  2\n 1  2  1  2  1  2\n 3  4  3  4  3  4\n 3  4  3  4  3  4\n\n\n\n\n\nrepeat(s::AbstractString, r::Integer)\n\nRepeat a string r times. This can be written as s^r.\n\nSee also ^.\n\nExamples\n\njulia> repeat(\"ha\", 3)\n\"hahaha\"\n\n\n\n\n\nrepeat(c::AbstractChar, r::Integer) -> String\n\nRepeat a character r times. This can equivalently be accomplished by calling c^r.\n\nExamples\n\njulia> repeat('A', 3)\n\"AAA\"\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.rot180","location":"base/arrays.html#Base.rot180","category":"function","text":"rot180(A)\n\nRotate matrix A 180 degrees.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rot180(a)\n2×2 Matrix{Int64}:\n 4  3\n 2  1\n\n\n\n\n\nrot180(A, k)\n\nRotate matrix A 180 degrees an integer k number of times. If k is even, this is equivalent to a copy.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rot180(a,1)\n2×2 Matrix{Int64}:\n 4  3\n 2  1\n\njulia> rot180(a,2)\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.rotl90","location":"base/arrays.html#Base.rotl90","category":"function","text":"rotl90(A)\n\nRotate matrix A left 90 degrees.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rotl90(a)\n2×2 Matrix{Int64}:\n 2  4\n 1  3\n\n\n\n\n\nrotl90(A, k)\n\nLeft-rotate matrix A 90 degrees counterclockwise an integer k number of times. If k is a multiple of four (including zero), this is equivalent to a copy.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rotl90(a,1)\n2×2 Matrix{Int64}:\n 2  4\n 1  3\n\njulia> rotl90(a,2)\n2×2 Matrix{Int64}:\n 4  3\n 2  1\n\njulia> rotl90(a,3)\n2×2 Matrix{Int64}:\n 3  1\n 4  2\n\njulia> rotl90(a,4)\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.rotr90","location":"base/arrays.html#Base.rotr90","category":"function","text":"rotr90(A)\n\nRotate matrix A right 90 degrees.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rotr90(a)\n2×2 Matrix{Int64}:\n 3  1\n 4  2\n\n\n\n\n\nrotr90(A, k)\n\nRight-rotate matrix A 90 degrees clockwise an integer k number of times. If k is a multiple of four (including zero), this is equivalent to a copy.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> rotr90(a,1)\n2×2 Matrix{Int64}:\n 3  1\n 4  2\n\njulia> rotr90(a,2)\n2×2 Matrix{Int64}:\n 4  3\n 2  1\n\njulia> rotr90(a,3)\n2×2 Matrix{Int64}:\n 2  4\n 1  3\n\njulia> rotr90(a,4)\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.mapslices","location":"base/arrays.html#Base.mapslices","category":"function","text":"mapslices(f, A; dims)\n\nTransform the given dimensions of array A by applying a function f on each slice of the form A[..., :, ..., :, ...], with a colon at each d in dims. The results are concatenated along the remaining dimensions.\n\nFor example, if dims = [1,2] and A is 4-dimensional, then f is called on x = A[:,:,i,j] for all i and j, and f(x) becomes R[:,:,i,j] in the result R.\n\nSee also eachcol or eachslice, used with map or stack.\n\nExamples\n\njulia> A = reshape(1:30,(2,5,3))\n2×5×3 reshape(::UnitRange{Int32}, 2, 5, 3) with eltype Int32:\n[:, :, 1] =\n 1  3  5  7   9\n 2  4  6  8  10\n\n[:, :, 2] =\n 11  13  15  17  19\n 12  14  16  18  20\n\n[:, :, 3] =\n 21  23  25  27  29\n 22  24  26  28  30\n\njulia> f(x::Matrix) = fill(x[1,1], 1,4);  # returns a 1×4 matrix\n\njulia> B = mapslices(f, A, dims=(1,2))\n1×4×3 Array{Int32, 3}:\n[:, :, 1] =\n 1  1  1  1\n\n[:, :, 2] =\n 11  11  11  11\n\n[:, :, 3] =\n 21  21  21  21\n\njulia> f2(x::AbstractMatrix) = fill(x[1,1], 1,4);\n\njulia> B == stack(f2, eachslice(A, dims=3))\ntrue\n\njulia> g(x) = x[begin] // x[end-1];  # returns a number\n\njulia> mapslices(g, A, dims=[1,3])\n1×5×1 Array{Rational{Int32}, 3}:\n[:, :, 1] =\n 1//21  3//23  1//5  7//27  9//29\n\njulia> map(g, eachslice(A, dims=2))\n5-element Vector{Rational{Int32}}:\n 1//21\n 3//23\n 1//5\n 7//27\n 9//29\n\njulia> mapslices(sum, A; dims=(1,3)) == sum(A; dims=(1,3))\ntrue\n\nNotice that in eachslice(A; dims=2), the specified dimension is the one without a colon in the slice. This is view(A,:,i,:), whereas mapslices(f, A; dims=(1,3)) uses A[:,i,:]. The function f may mutate values in the slice without affecting A.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.eachrow","location":"base/arrays.html#Base.eachrow","category":"function","text":"eachrow(A::AbstractVecOrMat) <: AbstractVector\n\nCreate a RowSlices object that is a vector of rows of matrix or vector A. Row slices are returned as AbstractVector views of A.\n\nFor the inverse, see stack(rows; dims=1).\n\nSee also eachcol, eachslice and mapslices.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this returned an iterator.\n\nExample\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> s = eachrow(a)\n2-element RowSlices{Matrix{Int64}, Tuple{Base.OneTo{Int64}}, SubArray{Int64, 1, Matrix{Int64}, Tuple{Int64, Base.Slice{Base.OneTo{Int64}}}, true}}:\n [1, 2]\n [3, 4]\n\njulia> s[1]\n2-element view(::Matrix{Int64}, 1, :) with eltype Int64:\n 1\n 2\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.eachcol","location":"base/arrays.html#Base.eachcol","category":"function","text":"eachcol(A::AbstractVecOrMat) <: AbstractVector\n\nCreate a ColumnSlices object that is a vector of columns of matrix or vector A. Column slices are returned as AbstractVector views of A.\n\nFor the inverse, see stack(cols) or reduce(hcat, cols).\n\nSee also eachrow, eachslice and mapslices.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this returned an iterator.\n\nExample\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> s = eachcol(a)\n2-element ColumnSlices{Matrix{Int64}, Tuple{Base.OneTo{Int64}}, SubArray{Int64, 1, Matrix{Int64}, Tuple{Base.Slice{Base.OneTo{Int64}}, Int64}, true}}:\n [1, 3]\n [2, 4]\n\njulia> s[1]\n2-element view(::Matrix{Int64}, :, 1) with eltype Int64:\n 1\n 3\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.eachslice","location":"base/arrays.html#Base.eachslice","category":"function","text":"eachslice(A::AbstractArray; dims, drop=true)\n\nCreate a Slices object that is an array of slices over dimensions dims of A, returning views that select all the data from the other dimensions in A. dims can either by an integer or a tuple of integers.\n\nIf drop = true (the default), the outer Slices will drop the inner dimensions, and the ordering of the dimensions will match those in dims. If drop = false, then the Slices will have the same dimensionality as the underlying array, with inner dimensions having size 1.\n\nSee stack(slices; dims) for the inverse of eachslice(A; dims::Integer).\n\nSee also eachrow, eachcol, mapslices and selectdim.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this returned an iterator, and only a single dimension dims was supported.\n\nExample\n\njulia> m = [1 2 3; 4 5 6; 7 8 9]\n3×3 Matrix{Int64}:\n 1  2  3\n 4  5  6\n 7  8  9\n\njulia> s = eachslice(m, dims=1)\n3-element RowSlices{Matrix{Int64}, Tuple{Base.OneTo{Int64}}, SubArray{Int64, 1, Matrix{Int64}, Tuple{Int64, Base.Slice{Base.OneTo{Int64}}}, true}}:\n [1, 2, 3]\n [4, 5, 6]\n [7, 8, 9]\n\njulia> s[1]\n3-element view(::Matrix{Int64}, 1, :) with eltype Int64:\n 1\n 2\n 3\n\njulia> eachslice(m, dims=1, drop=false)\n3×1 Slices{Matrix{Int64}, Tuple{Int64, Colon}, Tuple{Base.OneTo{Int64}, Base.OneTo{Int64}}, SubArray{Int64, 1, Matrix{Int64}, Tuple{Int64, Base.Slice{Base.OneTo{Int64}}}, true}, 2}:\n [1, 2, 3]\n [4, 5, 6]\n [7, 8, 9]\n\n\n\n\n\n","page":"Arrays"},{"title":"Combinatorics","location":"base/arrays.html#Combinatorics","category":"section","text":"","page":"Arrays"},{"title":"Arrays","location":"base/arrays.html","category":"page","text":"Base.invperm\nBase.isperm\nBase.permute!(::Any, ::AbstractVector)\nBase.invpermute!\nBase.reverse(::AbstractVector; kwargs...)\nBase.reverseind\nBase.reverse!","page":"Arrays"},{"title":"Base.invperm","location":"base/arrays.html#Base.invperm","category":"function","text":"invperm(v)\n\nReturn the inverse permutation of v. If B = A[v], then A == B[invperm(v)].\n\nSee also sortperm, invpermute!, isperm, permutedims.\n\nExamples\n\njulia> p = (2, 3, 1);\n\njulia> invperm(p)\n(3, 1, 2)\n\njulia> v = [2; 4; 3; 1];\n\njulia> invperm(v)\n4-element Vector{Int64}:\n 4\n 1\n 3\n 2\n\njulia> A = ['a','b','c','d'];\n\njulia> B = A[v]\n4-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> B[invperm(v)]\n4-element Vector{Char}:\n 'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.isperm","location":"base/arrays.html#Base.isperm","category":"function","text":"isperm(v) -> Bool\n\nReturn true if v is a valid permutation.\n\nExamples\n\njulia> isperm([1; 2])\ntrue\n\njulia> isperm([1; 3])\nfalse\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.permute!","location":"base/arrays.html#Base.permute!-Tuple{Any, AbstractVector}","category":"method","text":"permute!(v, p)\n\nPermute vector v in-place, according to permutation p. No checking is done to verify that p is a permutation.\n\nTo return a new permutation, use v[p]. Note that this is faster than permute!(v, p).\n\nSee also invpermute!.\n\nExamples\n\njulia> A = [1, 1, 3, 4];\n\njulia> perm = [2, 4, 3, 1];\n\njulia> permute!(A, perm);\n\njulia> A\n4-element Vector{Int64}:\n 1\n 4\n 3\n 1\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.invpermute!","location":"base/arrays.html#Base.invpermute!","category":"function","text":"invpermute!(v, p)\n\nLike permute!, but the inverse of the given permutation is applied.\n\nExamples\n\njulia> A = [1, 1, 3, 4];\n\njulia> perm = [2, 4, 3, 1];\n\njulia> invpermute!(A, perm);\n\njulia> A\n4-element Vector{Int64}:\n 4\n 1\n 3\n 1\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.reverse","location":"base/arrays.html#Base.reverse-Tuple{AbstractVector}","category":"method","text":"reverse(A; dims=:)\n\nReverse A along dimension dims, which can be an integer (a single dimension), a tuple of integers (a tuple of dimensions) or : (reverse along all the dimensions, the default).  See also reverse! for in-place reversal.\n\nExamples\n\njulia> b = Int64[1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> reverse(b, dims=2)\n2×2 Matrix{Int64}:\n 2  1\n 4  3\n\njulia> reverse(b)\n2×2 Matrix{Int64}:\n 4  3\n 2  1\n\ncompat: Julia 1.6\nPrior to Julia 1.6, only single-integer dims are supported in reverse.\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.reverseind","location":"base/arrays.html#Base.reverseind","category":"function","text":"reverseind(v, i)\n\nGiven an index i in reverse(v), return the corresponding index in v so that v[reverseind(v,i)] == reverse(v)[i]. (This can be nontrivial in cases where v contains non-ASCII characters.)\n\nExamples\n\njulia> s = \"Julia🚀\"\n\"Julia🚀\"\n\njulia> r = reverse(s)\n\"🚀ailuJ\"\n\njulia> for i in eachindex(s)\n           print(r[reverseind(r, i)])\n       end\nJulia🚀\n\n\n\n\n\n","page":"Arrays"},{"title":"Base.reverse!","location":"base/arrays.html#Base.reverse!","category":"function","text":"reverse!(v [, start=firstindex(v) [, stop=lastindex(v) ]]) -> v\n\nIn-place version of reverse.\n\nExamples\n\njulia> A = Vector(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> reverse!(A);\n\njulia> A\n5-element Vector{Int64}:\n 5\n 4\n 3\n 2\n 1\n\n\n\n\n\nreverse!(A; dims=:)\n\nLike reverse, but operates in-place in A.\n\ncompat: Julia 1.6\nMultidimensional reverse! requires Julia 1.6.\n\n\n\n\n\n","page":"Arrays"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html#Julia-SSA-form-IR","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Background","location":"devdocs/ssair.html#Background","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Beginning in Julia 0.7, parts of the compiler use a new SSA-form intermediate representation (IR). Historically, the compiler would directly generate LLVM IR from a lowered form of the Julia AST. This form had most syntactic abstractions removed, but still looked a lot like an abstract syntax tree. Over time, in order to facilitate optimizations, SSA values were introduced to this IR and the IR was linearized (i.e. turned into a form where function arguments could only be SSA values or constants). However, non-SSA values (slots) remained in the IR due to the lack of Phi nodes in the IR (necessary for back-edges and re-merging of conditional control flow). This negated much of the usefulness of SSA form representation when performing middle end optimizations. Some heroic effort was put into making these optimizations work without a complete SSA form representation, but the lack of such a representation ultimately proved prohibitive.","page":"Julia SSA-form IR"},{"title":"New IR nodes","location":"devdocs/ssair.html#New-IR-nodes","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"With the new IR representation, the compiler learned to handle four new IR nodes, Phi nodes, Pi nodes as well as PhiC nodes and Upsilon nodes (the latter two are only used for exception handling).","page":"Julia SSA-form IR"},{"title":"Phi nodes and Pi nodes","location":"devdocs/ssair.html#Phi-nodes-and-Pi-nodes","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Phi nodes are part of generic SSA abstraction (see the link above if you're not familiar with the concept). In the Julia IR, these nodes are represented as:","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"struct PhiNode\n    edges::Vector{Int32}\n    values::Vector{Any}\nend","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"where we ensure that both vectors always have the same length. In the canonical representation (the one handled by codegen and the interpreter), the edge values indicate come-from statement numbers (i.e. if edge has an entry of 15, there must be a goto, gotoifnot or implicit fall through from statement 15 that targets this phi node). Values are either SSA values or constants. It is also possible for a value to be unassigned if the variable was not defined on this path. However, undefinedness checks get explicitly inserted and represented as booleans after middle end optimizations, so code generators may assume that any use of a Phi node will have an assigned value in the corresponding slot. It is also legal for the mapping to be incomplete, i.e. for a Phi node to have missing incoming edges. In that case, it must be dynamically guaranteed that the corresponding value will not be used.","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"PiNodes encode statically proven information that may be implicitly assumed in basic blocks dominated by a given pi node. They are conceptually equivalent to the technique introduced in the paper ABCD: Eliminating Array Bounds Checks on Demand or the predicate info nodes in LLVM. To see how they work, consider, e.g.","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"%x::Union{Int, Float64} # %x is some Union{Int, Float64} typed ssa value\nif isa(x, Int)\n    # use x\nelse\n    # use x\nend","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"We can perform predicate insertion and turn this into:","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"%x::Union{Int, Float64} # %x is some Union{Int, Float64} typed ssa value\nif isa(x, Int)\n    %x_int = PiNode(x, Int)\n    # use %x_int\nelse\n    %x_float = PiNode(x, Float64)\n    # use %x_float\nend","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Pi nodes are generally ignored in the interpreter, since they don't have any effect on the values, but they may sometimes lead to code generation in the compiler (e.g. to change from an implicitly union split representation to a plain unboxed representation). The main usefulness of PiNodes stems from the fact that path conditions of the values can be accumulated simply by def-use chain walking that is generally done for most optimizations that care about these conditions anyway.","page":"Julia SSA-form IR"},{"title":"PhiC nodes and Upsilon nodes","location":"devdocs/ssair.html#PhiC-nodes-and-Upsilon-nodes","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Exception handling complicates the SSA story moderately, because exception handling introduces additional control flow edges into the IR across which values must be tracked. One approach to do so, which is followed by LLVM, is to make calls which may throw exceptions into basic block terminators and add an explicit control flow edge to the catch handler:","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"invoke @function_that_may_throw() to label %regular unwind to %catch\n\nregular:\n# Control flow continues here\n\ncatch:\n# Exceptions go here","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"However, this is problematic in a language like Julia, where at the start of the optimization pipeline, we do not know which calls throw. We would have to conservatively assume that every call (which in Julia is every statement) throws. This would have several negative effects. On the one hand, it would essentially reduce the scope of every basic block to a single call, defeating the purpose of having operations be performed at the basic block level. On the other hand, every catch basic block would have n*m phi node arguments (n, the number of statements in the critical region, m the number of live values through the catch block).","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"To work around this, we use a combination of Upsilon and PhiC nodes (the C standing for catch, written φᶜ in the IR pretty printer, because unicode subscript c is not available). There are several ways to think of these nodes, but perhaps the easiest is to think of each PhiC as a load from a unique store-many, read-once slot, with Upsilon being the corresponding store operation. The PhiC has an operand list of all the upsilon nodes that store to its implicit slot. The Upsilon nodes however, do not record which PhiC node they store to. This is done for more natural integration with the rest of the SSA IR. E.g. if there are no more uses of a PhiC node, it is safe to delete it, and the same is true of an Upsilon node. In most IR passes, PhiC nodes can be treated like Phi nodes. One can follow use-def chains through them, and they can be lifted to new PhiC nodes and new Upsilon nodes (in the same places as the original Upsilon nodes). The result of this scheme is that the number of Upsilon nodes (and PhiC arguments) is proportional to the number of assigned values to a particular variable (before SSA conversion), rather than the number of statements in the critical region.","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"To see this scheme in action, consider the function","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"@noinline opaque() = invokelatest(identity, nothing) # Something opaque\nfunction foo()\n    local y\n    x = 1\n    try\n        y = 2\n        opaque()\n        y = 3\n        error()\n    catch\n    end\n    (x, y)\nend","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"The corresponding IR (with irrelevant types stripped) is:","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"1 ─       nothing::Nothing\n2 ─ %2  = $(Expr(:enter, #4))\n3 ─ %3  = ϒ (false)\n│   %4  = ϒ (#undef)\n│   %5  = ϒ (1)\n│   %6  = ϒ (true)\n│   %7  = ϒ (2)\n│         invoke Main.opaque()::Any\n│   %9  = ϒ (true)\n│   %10 = ϒ (3)\n│         invoke Main.error()::Union{}\n└──       $(Expr(:unreachable))::Union{}\n4 ┄ %13 = φᶜ (%3, %6, %9)::Bool\n│   %14 = φᶜ (%4, %7, %10)::Core.Compiler.MaybeUndef(Int64)\n│   %15 = φᶜ (%5)::Core.Const(1)\n└──       $(Expr(:leave, 1))\n5 ─       $(Expr(:pop_exception, :(%2)))::Any\n│         $(Expr(:throw_undef_if_not, :y, :(%13)))::Any\n│   %19 = Core.tuple(%15, %14)\n└──       return %19","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Note in particular that every value live into the critical region gets an upsilon node at the top of the critical region. This is because catch blocks are considered to have an invisible control flow edge from outside the function. As a result, no SSA value dominates the catch blocks, and all incoming values have to come through a φᶜ node.","page":"Julia SSA-form IR"},{"title":"Main SSA data structure","location":"devdocs/ssair.html#Main-SSA-data-structure","category":"section","text":"","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"The main SSAIR data structure is worthy of discussion. It draws inspiration from LLVM and Webkit's B3 IR. The core of the data structure is a flat vector of statements. Each statement is implicitly assigned an SSA value based on its position in the vector (i.e. the result of the statement at idx 1 can be accessed using SSAValue(1) etc). For each SSA value, we additionally maintain its type. Since, SSA values are definitionally assigned only once, this type is also the result type of the expression at the corresponding index. However, while this representation is rather efficient (since the assignments don't need to be explicitly encoded), it of course carries the drawback that order is semantically significant, so reorderings and insertions change statement numbers. Additionally, we do not keep use lists (i.e. it is impossible to walk from a def to all its uses without explicitly computing this map–def lists however are trivial since you can look up the corresponding statement from the index), so the LLVM-style RAUW (replace-all-uses-with) operation is unavailable.","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"Instead, we do the following:","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"We keep a separate buffer of nodes to insert (including the position to insert them at, the type of the corresponding value and the node itself). These nodes are numbered by their occurrence in the insertion buffer, allowing their values to be immediately used elsewhere in the IR (i.e. if there are 12 statements in the original statement list, the first new statement will be accessible as SSAValue(13)).\nRAUW style operations are performed by setting the corresponding statement index to the replacement value.\nStatements are erased by setting the corresponding statement to nothing (this is essentially just a special-case convention of the above.\nIf there are any uses of the statement being erased, they will be set to nothing.","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"There is a compact! function that compacts the above data structure by performing the insertion of nodes in the appropriate place, trivial copy propagation, and renaming of uses to any changed SSA values. However, the clever part of this scheme is that this compaction can be done lazily as part of the subsequent pass. Most optimization passes need to walk over the entire list of statements, performing analysis or modifications along the way. We provide an IncrementalCompact iterator that can be used to iterate over the statement list. It will perform any necessary compaction and return the new index of the node, as well as the node itself. It is legal at this point to walk def-use chains, as well as make any modifications or deletions to the IR (insertions are disallowed however).","page":"Julia SSA-form IR"},{"title":"Julia SSA-form IR","location":"devdocs/ssair.html","category":"page","text":"The idea behind this arrangement is that, since the optimization passes need to touch the corresponding memory anyway and incur the corresponding memory access penalty, performing the extra housekeeping should have comparatively little overhead (and save the overhead of maintaining these data structures during IR modification).","page":"Julia SSA-form IR"},{"title":"StackTraces","location":"base/stacktraces.html#StackTraces","category":"section","text":"","page":"StackTraces"},{"title":"StackTraces","location":"base/stacktraces.html","category":"page","text":"Base.StackTraces.StackFrame\nBase.StackTraces.StackTrace\nBase.StackTraces.stacktrace","page":"StackTraces"},{"title":"Base.StackTraces.StackFrame","location":"base/stacktraces.html#Base.StackTraces.StackFrame","category":"type","text":"StackFrame\n\nStack information representing execution context, with the following fields:\n\nfunc::Symbol\nThe name of the function containing the execution context.\nlinfo::Union{Core.MethodInstance, CodeInfo, Nothing}\nThe MethodInstance containing the execution context (if it could be found).\nfile::Symbol\nThe path to the file containing the execution context.\nline::Int\nThe line number in the file containing the execution context.\nfrom_c::Bool\nTrue if the code is from C.\ninlined::Bool\nTrue if the code is from an inlined frame.\npointer::UInt64\nRepresentation of the pointer to the execution context as returned by backtrace.\n\n\n\n\n\n","page":"StackTraces"},{"title":"Base.StackTraces.StackTrace","location":"base/stacktraces.html#Base.StackTraces.StackTrace","category":"type","text":"StackTrace\n\nAn alias for Vector{StackFrame} provided for convenience; returned by calls to stacktrace.\n\n\n\n\n\n","page":"StackTraces"},{"title":"Base.StackTraces.stacktrace","location":"base/stacktraces.html#Base.StackTraces.stacktrace","category":"function","text":"stacktrace([trace::Vector{Ptr{Cvoid}},] [c_funcs::Bool=false]) -> StackTrace\n\nReturn a stack trace in the form of a vector of StackFrames. (By default stacktrace doesn't return C functions, but this can be enabled.) When called without specifying a trace, stacktrace first calls backtrace.\n\n\n\n\n\n","page":"StackTraces"},{"title":"StackTraces","location":"base/stacktraces.html","category":"page","text":"The following methods and types in Base.StackTraces are not exported and need to be called e.g. as StackTraces.lookup(ptr).","page":"StackTraces"},{"title":"StackTraces","location":"base/stacktraces.html","category":"page","text":"Base.StackTraces.lookup\nBase.StackTraces.remove_frames!","page":"StackTraces"},{"title":"Base.StackTraces.lookup","location":"base/stacktraces.html#Base.StackTraces.lookup","category":"function","text":"lookup(pointer::Ptr{Cvoid}) -> Vector{StackFrame}\n\nGiven a pointer to an execution context (usually generated by a call to backtrace), looks up stack frame context information. Returns an array of frame information for all functions inlined at that point, innermost function first.\n\n\n\n\n\n","page":"StackTraces"},{"title":"Base.StackTraces.remove_frames!","location":"base/stacktraces.html#Base.StackTraces.remove_frames!","category":"function","text":"remove_frames!(stack::StackTrace, name::Symbol)\n\nTakes a StackTrace (a vector of StackFrames) and a function name (a Symbol) and removes the StackFrame specified by the function name from the StackTrace (also removing all frames above the specified function). Primarily used to remove StackTraces functions from the StackTrace prior to returning it.\n\n\n\n\n\nremove_frames!(stack::StackTrace, m::Module)\n\nReturn the StackTrace with all StackFrames from the provided Module removed.\n\n\n\n\n\n","page":"StackTraces"},{"title":"Julia Functions","location":"devdocs/functions.html#Julia-Functions","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"This document will explain how functions, method definitions, and method tables work.","page":"Julia Functions"},{"title":"Method Tables","location":"devdocs/functions.html#Method-Tables","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Every function in Julia is a generic function. A generic function is conceptually a single function, but consists of many definitions, or methods. The methods of a generic function are stored in a method table. Method tables (type MethodTable) are associated with TypeNames. A TypeName describes a family of parameterized types. For example Complex{Float32} and Complex{Float64} share the same Complex type name object.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"All objects in Julia are potentially callable, because every object has a type, which in turn has a TypeName.","page":"Julia Functions"},{"title":"Function calls","location":"devdocs/functions.html#Function-calls","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Given the call f(x,y), the following steps are performed: first, the method table to use is accessed as typeof(f).name.mt. Second, an argument tuple type is formed, Tuple{typeof(f), typeof(x), typeof(y)}. Note that the type of the function itself is the first element. This is because the type might have parameters, and so needs to take part in dispatch. This tuple type is looked up in the method table.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"This dispatch process is performed by jl_apply_generic, which takes two arguments: a pointer to an array of the values f, x, and y, and the number of values (in this case 3).","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Throughout the system, there are two kinds of APIs that handle functions and argument lists: those that accept the function and arguments separately, and those that accept a single argument structure. In the first kind of API, the \"arguments\" part does not contain information about the function, since that is passed separately. In the second kind of API, the function is the first element of the argument structure.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"For example, the following function for performing a call accepts just an args pointer, so the first element of the args array will be the function to call:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"jl_value_t *jl_apply(jl_value_t **args, uint32_t nargs)","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"This entry point for the same functionality accepts the function separately, so the args array does not contain the function:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"jl_value_t *jl_call(jl_function_t *f, jl_value_t **args, int32_t nargs);","page":"Julia Functions"},{"title":"Adding methods","location":"devdocs/functions.html#Adding-methods","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Given the above dispatch process, conceptually all that is needed to add a new method is (1) a tuple type, and (2) code for the body of the method. jl_method_def implements this operation. jl_method_table_for is called to extract the relevant method table from what would be the type of the first argument. This is much more complicated than the corresponding procedure during dispatch, since the argument tuple type might be abstract. For example, we can define:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"(::Union{Foo{Int},Foo{Int8}})(x) = 0","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"which works since all possible matching methods would belong to the same method table.","page":"Julia Functions"},{"title":"Creating generic functions","location":"devdocs/functions.html#Creating-generic-functions","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Since every object is callable, nothing special is needed to create a generic function. Therefore jl_new_generic_function simply creates a new singleton (0 size) subtype of Function and returns its instance. A function can have a mnemonic \"display name\" which is used in debug info and when printing objects. For example the name of Base.sin is sin. By convention, the name of the created type is the same as the function name, with a # prepended. So typeof(sin) is Base.#sin.","page":"Julia Functions"},{"title":"Closures","location":"devdocs/functions.html#Closures","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"A closure is simply a callable object with field names corresponding to captured variables. For example, the following code:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function adder(x)\n    return y->x+y\nend","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"is lowered to (roughly):","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"struct ##1{T}\n    x::T\nend\n\n(_::##1)(y) = _.x + y\n\nfunction adder(x)\n    return ##1(x)\nend","page":"Julia Functions"},{"title":"Constructors","location":"devdocs/functions.html#Constructors","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"A constructor call is just a call to a type. The method table for Type contains all constructor definitions. All subtypes of Type (Type, UnionAll, Union, and DataType) currently share a method table via special arrangement.","page":"Julia Functions"},{"title":"Builtins","location":"devdocs/functions.html#Builtins","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The \"builtin\" functions, defined in the Core module, are:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function lines(words)\n    io = IOBuffer()\n    n = 0\n    for w in words\n        if n+length(w) > 80\n            print(io, '\\n', w)\n            n = length(w)\n        elseif n == 0\n            print(io, w);\n            n += length(w)\n        else\n            print(io, ' ', w);\n            n += length(w)+1\n        end\n    end\n    String(take!(io))\nend\nimport Markdown\n[string(n) for n in names(Core;all=true)\n    if getfield(Core,n) isa Core.Builtin && nameof(getfield(Core,n)) === n] |>\n    lines |>\n    s ->  \"```\\n$s\\n```\" |>\n    Markdown.parse","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"These are all singleton objects whose types are subtypes of Builtin, which is a subtype of Function. Their purpose is to expose entry points in the run time that use the \"jlcall\" calling convention:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"jl_value_t *(jl_value_t*, jl_value_t**, uint32_t)","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The method tables of builtins are empty. Instead, they have a single catch-all method cache entry (Tuple{Vararg{Any}}) whose jlcall fptr points to the correct function. This is kind of a hack but works reasonably well.","page":"Julia Functions"},{"title":"Keyword arguments","location":"devdocs/functions.html#Keyword-arguments","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Keyword arguments work by adding methods to the kwcall function. This function is usually the \"keyword argument sorter\" or \"keyword sorter\", which then calls the inner body of the function (defined anonymously). Every definition in the kwsorter function has the same arguments as some definition in the normal method table, except with a single NamedTuple argument prepended, which gives the names and values of passed keyword arguments. The kwsorter's job is to move keyword arguments into their canonical positions based on name, plus evaluate and substitute any needed default value expressions. The result is a normal positional argument list, which is then passed to yet another compiler-generated function.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The easiest way to understand the process is to look at how a keyword argument method definition is lowered. The code:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function circle(center, radius; color = black, fill::Bool = true, options...)\n    # draw\nend","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"actually produces three method definitions. The first is a function that accepts all arguments (including keyword arguments) as positional arguments, and includes the code for the method body. It has an auto-generated name:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function #circle#1(color, fill::Bool, options, circle, center, radius)\n    # draw\nend","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The second method is an ordinary definition for the original circle function, which handles the case where no keyword arguments are passed:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function circle(center, radius)\n    #circle#1(black, true, pairs(NamedTuple()), circle, center, radius)\nend","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"This simply dispatches to the first method, passing along default values. pairs is applied to the named tuple of rest arguments to provide key-value pair iteration. Note that if the method doesn't accept rest keyword arguments then this argument is absent.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Finally there is the kwsorter definition:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"function (::Core.kwftype(typeof(circle)))(kws, circle, center, radius)\n    if haskey(kws, :color)\n        color = kws.color\n    else\n        color = black\n    end\n    # etc.\n\n    # put remaining kwargs in `options`\n    options = structdiff(kws, NamedTuple{(:color, :fill)})\n\n    # if the method doesn't accept rest keywords, throw an error\n    # unless `options` is empty\n\n    #circle#1(color, fill, pairs(options), circle, center, radius)\nend","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The function Core.kwftype(t) creates the field t.name.mt.kwsorter (if it hasn't been created yet), and returns the type of that function.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"This design has the feature that call sites that don't use keyword arguments require no special handling; everything works as if they were not part of the language at all. Call sites that do use keyword arguments are dispatched directly to the called function's kwsorter. For example the call:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"circle((0,0), 1.0, color = red; other...)","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"is lowered to:","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"kwcall(merge((color = red,), other), circle, (0,0), 1.0)","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"kwcall (also inCore) denotes a kwcall signature and dispatch. The keyword splatting operation (written as other...) calls the named tuple merge function. This function further unpacks each element of other, expecting each one to contain two values (a symbol and a value). Naturally, a more efficient implementation is available if all splatted arguments are named tuples. Notice that the original circle function is passed through, to handle closures.","page":"Julia Functions"},{"title":"Compiler efficiency issues","location":"devdocs/functions.html#compiler-efficiency-issues","category":"section","text":"","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"Generating a new type for every function has potentially serious consequences for compiler resource use when combined with Julia's \"specialize on all arguments by default\" design. Indeed, the initial implementation of this design suffered from much longer build and test times, higher memory use, and a system image nearly 2x larger than the baseline. In a naive implementation, the problem is bad enough to make the system nearly unusable. Several significant optimizations were needed to make the design practical.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The first issue is excessive specialization of functions for different values of function-valued arguments. Many functions simply \"pass through\" an argument to somewhere else, e.g. to another function or to a storage location. Such functions do not need to be specialized for every closure that might be passed in. Fortunately this case is easy to distinguish by simply considering whether a function calls one of its arguments (i.e. the argument appears in \"head position\" somewhere). Performance-critical higher-order functions like map certainly call their argument function and so will still be specialized as expected. This optimization is implemented by recording which arguments are called during the analyze-variables pass in the front end. When cache_method sees an argument in the Function type hierarchy passed to a slot declared as Any or Function, it behaves as if the @nospecialize annotation were applied. This heuristic seems to be extremely effective in practice.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The next issue concerns the structure of method cache hash tables. Empirical studies show that the vast majority of dynamically-dispatched calls involve one or two arguments. In turn, many of these cases can be resolved by considering only the first argument. (Aside: proponents of single dispatch would not be surprised by this at all. However, this argument means \"multiple dispatch is easy to optimize in practice\", and that we should therefore use it, not \"we should use single dispatch\"!) So the method cache uses the type of the first argument as its primary key. Note, however, that this corresponds to the second element of the tuple type for a function call (the first element being the type of the function itself). Typically, type variation in head position is extremely low – indeed, the majority of functions belong to singleton types with no parameters. However, this is not the case for constructors, where a single method table holds constructors for every type. Therefore the Type method table is special-cased to use the first tuple type element instead of the second.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The front end generates type declarations for all closures. Initially, this was implemented by generating normal type declarations. However, this produced an extremely large number of constructors, all of which were trivial (simply passing all arguments through to new). Since methods are partially ordered, inserting all of these methods is O(n^2), plus there are just too many of them to keep around. This was optimized by generating struct_type expressions directly (bypassing default constructor generation), and using new directly to create closure instances. Not the prettiest thing ever, but you do what you gotta do.","page":"Julia Functions"},{"title":"Julia Functions","location":"devdocs/functions.html","category":"page","text":"The next problem was the @test macro, which generated a 0-argument closure for each test case. This is not really necessary, since each test case is simply run once in place. Therefore, @test was modified to expand to a try-catch block that records the test result (true, false, or exception raised) and calls the test suite handler on it.","page":"Julia Functions"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/NEWS.md\"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html#Julia-v1.9-Release-Notes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"New language features","location":"NEWS.html#New-language-features","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"It is now possible to assign to bindings in another module using setproperty!(::Module, ::Symbol, x) (#44137) (#44231).\nSlurping in assignments is now also allowed in non-final position. This is handled via Base.split_rest (#42902).\nCharacter literals now support the same syntax allowed in string literals; i.e. the syntax can represent invalid UTF-8 sequences as allowed by the Char type (#44989).\nSupport for Unicode 15 (#47392).\nNested combinations of tuples and named tuples of symbols are now allowed as type parameters (#46300).\nNew builtins getglobal(::Module, ::Symbol[, order]) and setglobal!(::Module, ::Symbol, x[, order]) for reading from and writing to globals. getglobal should now be preferred for accessing globals over getfield (#44137) (#44231).","page":"Julia v1.9 Release Notes"},{"title":"Language changes","location":"NEWS.html#Language-changes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"The @invoke macro introduced in 1.7 is now exported. Additionally, it now uses Core.Typeof(x) rather than Any when a type annotation is omitted for an argument x so that types passed as arguments are handled correctly (#45807).\nThe invokelatest function and @invokelatest macro introduced in 1.7 are now exported (#45831).","page":"Julia v1.9 Release Notes"},{"title":"Compiler/Runtime improvements","location":"NEWS.html#Compiler/Runtime-improvements","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Time to first execution (TTFX, sometimes called time to first plot) is greatly reduced. Package precompilation now saves native code into a \"pkgimage\", meaning that code generated during the precompilation process will not require compilation after package load. Use of pkgimages can be disabled via --pkgimages=no (#44527) (#47184).\nThe known quadratic behavior of type inference is now fixed and inference uses less memory in general. Certain edge cases with auto-generated long functions (e.g. ModelingToolkit.jl with partial differential equations and large causal models) should see significant compile-time improvements (#45276, #45404).\nNon-concrete call sites can now be union-split to be inlined or statically resolved even if there are multiple dispatch candidates. This may improve runtime performance in certain situations where object types are not fully known statically, by statically resolving @nospecialize-d call sites and avoiding excessive compilation (#44512).\nAll uses of the @pure macro in Base have been replaced with the now-preferred Base.@assume_effects (#44776).\ninvoke(f, invokesig, args...) calls to a less-specific method than would normally be chosen for f(args...) are no longer spuriously invalidated when loading package precompile files (#46010).","page":"Julia v1.9 Release Notes"},{"title":"Command-line option changes","location":"NEWS.html#Command-line-option-changes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"In Linux and Windows, --threads=auto now tries to infer the usable number of CPUs from the process affinity which is set typically in HPC and cloud environments (#42340).\n--math-mode=fast is now a no-op (#41638). Users are encouraged to use the @fastmath macro instead, which has more well-defined semantics.\nThe --threads command-line option now accepts auto|N[,auto|M] where M specifies the number of interactive threads to create (auto currently means 1) (#42302).\nNew option --heap-size-hint=<size> suggests a size limit to invoke garbage collection more eagerly. The size may be specified in bytes, kilobytes (1000k), megabytes (300M), or gigabytes (1.5G) (#45369).","page":"Julia v1.9 Release Notes"},{"title":"Multi-threading changes","location":"NEWS.html#Multi-threading-changes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Threads.@spawn now accepts an optional first argument: :default or :interactive. An interactive task desires low latency and implicitly agrees to be short duration or to yield frequently. Interactive tasks will run on interactive threads, if any are specified when Julia is started (#42302).\nThreads started outside the Julia runtime (e.g. from C or Java) can now become able to call into Julia code by calling jl_adopt_thread. This is done automatically when entering Julia code via cfunction or a @ccallable entry point. As a consequence, the number of threads can now change during execution (#46609).","page":"Julia v1.9 Release Notes"},{"title":"Build system changes","location":"NEWS.html#Build-system-changes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"New library functions","location":"NEWS.html#New-library-functions","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"New function Iterators.flatmap (#44792).\nNew pkgversion(m::Module) function to get the version of the package that loaded a given module, similar to pkgdir(m::Module) (#45607).\nNew function stack(x) which generalises reduce(hcat, x::Vector{<:Vector}) to any dimensionality, and allows any iterator of iterators. Method stack(f, x) generalises mapreduce(f, hcat, x) and is more efficient (#43334).\nNew macro @allocations which is similar to @allocated except reporting the total number of allocations rather than the total size of memory allocated (#47367).","page":"Julia v1.9 Release Notes"},{"title":"New library features","location":"NEWS.html#New-library-features","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"RoundFromZero now works for non-BigFloat types (#41246).\nDict can be now shrunk manually by sizehint! (#45004).\n@time now separates out % time spent recompiling invalidated methods (#45015).","page":"Julia v1.9 Release Notes"},{"title":"Standard library changes","location":"NEWS.html#Standard-library-changes","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"A known concurrency issue in iterate methods on Dict and other derived objects such as keys(::Dict), values(::Dict), and Set is fixed. These methods of iterate can now be called on a dictionary or set shared by arbitrary tasks provided that there are no tasks mutating the dictionary or set (#44534).\nPredicate function negation !f now returns a composed function (!) ∘ f instead of an anonymous function (#44752).\neachslice now works over multiple dimensions; eachslice, eachrow and eachcol return a Slices object, which allows dispatching to provide more efficient methods (#32310).\n@kwdef is now exported and added to the public API (#46273).\nAn issue with order of operations in fld1 is now fixed (#28973).\nSorting is now always stable by default, as QuickSort was stabilized (#45222).\nBase.splat is now exported. The return value is now a Base.Splat instead of an anonymous function, which allows for pretty printing (#42717).","page":"Julia v1.9 Release Notes"},{"title":"Package Manager","location":"NEWS.html#Package-Manager","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"\"Package Extensions\": support for loading a piece of code based on other packages being loaded in the Julia session. This has similar applications as the Requires.jl package but also supports precompilation and setting compatibility.","page":"Julia v1.9 Release Notes"},{"title":"LinearAlgebra","location":"NEWS.html#LinearAlgebra","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"The methods a / b and b \\ a with a a scalar and b a vector, which were equivalent to a * pinv(b), have been removed due to the risk of confusion with elementwise division (#44358).\nWe are now wholly reliant on libblastrampoline (LBT) for calling BLAS and LAPACK. OpenBLAS is shipped by default, but building the system image with other BLAS/LAPACK libraries is not supported. Instead, it is recommended that the LBT mechanism be used for swapping BLAS/LAPACK with vendor provided ones (#44360).\nlu supports a new pivoting strategy RowNonZero() that chooses the first non-zero pivot element, for use with new arithmetic types and for pedagogy (#44571).\nnormalize(x, p=2) now supports any normed vector space x, including scalars (#44925).\nThe default number of BLAS threads is now set to the number of CPU threads on ARM CPUs, and half the number of CPU threads on other architectures (#45412, #46085).","page":"Julia v1.9 Release Notes"},{"title":"Printf","location":"NEWS.html#Printf","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Error messages for bad format strings have been improved, to make it clearer what and where in the format string is wrong (#45366).","page":"Julia v1.9 Release Notes"},{"title":"Profile","location":"NEWS.html#Profile","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"New function Profile.take_heap_snapshot(file) that writes a file in Chrome's JSON-based .heapsnapshot format (#46862).","page":"Julia v1.9 Release Notes"},{"title":"Random","location":"NEWS.html#Random","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"randn and randexp now work for any AbstractFloat type defining rand (#44714).","page":"Julia v1.9 Release Notes"},{"title":"REPL","location":"NEWS.html#REPL","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Alt-e now opens the current input in an editor. The content (if modified) will be executed upon exiting the editor (#33759).\nThe contextual module which is active in the REPL can be changed (it is Main by default), via the REPL.activate(::Module) function or via typing the module in the REPL and pressing the keybinding Alt-m (#33872).\nA \"numbered prompt\" mode which prints numbers for each input and output and stores evaluated results in Out can be activated with REPL.numbered_prompt!(). See the manual for how to enable this at startup (#46474).\nTab completion displays available keyword arguments (#43536).","page":"Julia v1.9 Release Notes"},{"title":"SuiteSparse","location":"NEWS.html#SuiteSparse","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Code for the SuiteSparse solver wrappers has been moved to SparseArrays.jl. Solvers are now re-exported by SuiteSparse.jl.","page":"Julia v1.9 Release Notes"},{"title":"SparseArrays","location":"NEWS.html#SparseArrays","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"SuiteSparse solvers are now available as submodules of SparseArrays (https://github.com/JuliaSparse/SparseArrays.jl/pull/95).\nUMFPACK (https://github.com/JuliaSparse/SparseArrays.jl/pull/179) and CHOLMOD (https://github.com/JuliaSparse/SparseArrays.jl/pull/206) thread safety are improved by avoiding globals and using locks. Multithreaded ldiv! of UMFPACK objects may now be performed safely.\nAn experimental function SparseArrays.allowscalar(::Bool) allows scalar indexing of sparse arrays to be disabled or enabled. This function is intended to help find accidental scalar indexing of SparseMatrixCSC objects, which is a common source of performance issues (https://github.com/JuliaSparse/SparseArrays.jl/pull/200).","page":"Julia v1.9 Release Notes"},{"title":"Test","location":"NEWS.html#Test","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"New fail-fast mode for testsets that will terminate the test run early if a failure or error occurs. Set either via the @testset kwarg failfast=true or by setting env var JULIA_TEST_FAILFAST to \"true\" i.e. in CI runs to request the job failure be posted eagerly when issues occur (#45317)","page":"Julia v1.9 Release Notes"},{"title":"Dates","location":"NEWS.html#Dates","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Empty strings are no longer incorrectly parsed as valid DateTimes, Dates or Times and instead throw an ArgumentError in constructors and parse, while nothing is returned by tryparse (#47117).","page":"Julia v1.9 Release Notes"},{"title":"Distributed","location":"NEWS.html#Distributed","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"The package environment (active project, LOAD_PATH, DEPOT_PATH) is now propagated when adding local workers (e.g. with addprocs(N::Int) or through the --procs=N command line flag) (#43270).\naddprocs for local workers now accepts the env keyword argument for passing environment variables to worker processes. This was already supported for remote workers (#43270).","page":"Julia v1.9 Release Notes"},{"title":"Unicode","location":"NEWS.html#Unicode","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"graphemes(s, m:n) returns a substring of the m-th to n-th graphemes in s (#44266).","page":"Julia v1.9 Release Notes"},{"title":"DelimitedFiles","location":"NEWS.html#DelimitedFiles","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"DelimitedFiles has been moved out as a separate package. It now has to be explicitly installed to be used.","page":"Julia v1.9 Release Notes"},{"title":"Deprecated or removed","location":"NEWS.html#Deprecated-or-removed","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"External dependencies","location":"NEWS.html#External-dependencies","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"On Linux, now autodetects the system libstdc++ version, and automatically loads the system library if it is newer. The old behavior of loading the bundled libstdc++ regardless of the system version can be restored by setting the environment variable JULIA_PROBE_LIBSTDCXX=0 (#46976).\nRemoved RPATH from the julia binary. On Linux this may break libraries that have failed to set RUNPATH.","page":"Julia v1.9 Release Notes"},{"title":"Tooling Improvements","location":"NEWS.html#Tooling-Improvements","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"Printing of MethodError and methods (such as from methods(my_func)) is now prettified and colored consistently with printing of methods in stacktraces (#45069).","page":"Julia v1.9 Release Notes"},{"title":"Regressions","location":"NEWS.html#Regressions","category":"section","text":"","page":"Julia v1.9 Release Notes"},{"title":"Julia v1.9 Release Notes","location":"NEWS.html","category":"page","text":"A list of known regressions in the v1.9 release can be found here","page":"Julia v1.9 Release Notes"},{"title":"Environment Variables","location":"manual/environment-variables.html#Environment-Variables","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Julia can be configured with a number of environment variables, set either in the usual way for each operating system, or in a portable way from within Julia. Supposing that you want to set the environment variable JULIA_EDITOR to vim, you can type ENV[\"JULIA_EDITOR\"] = \"vim\" (for instance, in the REPL) to make this change on a case by case basis, or add the same to the user configuration file ~/.julia/config/startup.jl in the user's home directory to have a permanent effect. The current value of the same environment variable can be determined by evaluating ENV[\"JULIA_EDITOR\"].","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The environment variables that Julia uses generally start with JULIA. If InteractiveUtils.versioninfo is called with the keyword verbose=true, then the output will list any defined environment variables relevant for Julia, including those which include JULIA in their names.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nSome variables, such as JULIA_NUM_THREADS and JULIA_PROJECT, need to be set before Julia starts, therefore adding these to ~/.julia/config/startup.jl is too late in the startup process. In Bash, environment variables can either be set manually by running, e.g., export JULIA_NUM_THREADS=4 before starting Julia, or by adding the same command to ~/.bashrc or ~/.bash_profile to set the variable each time Bash is started.","page":"Environment Variables"},{"title":"File locations","location":"manual/environment-variables.html#File-locations","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_BINDIR","location":"manual/environment-variables.html#JULIA_BINDIR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path of the directory containing the Julia executable, which sets the global variable Sys.BINDIR. If $JULIA_BINDIR is not set, then Julia determines the value Sys.BINDIR at run-time.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The executable itself is one of","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/julia\n$JULIA_BINDIR/julia-debug","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"by default.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The global variable Base.DATAROOTDIR determines a relative path from Sys.BINDIR to the data directory associated with Julia. Then the path","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/$DATAROOTDIR/julia/base","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"determines the directory in which Julia initially searches for source files (via Base.find_source_file()).","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Likewise, the global variable Base.SYSCONFDIR determines a relative path to the configuration file directory. Then Julia searches for a startup.jl file at","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/$SYSCONFDIR/julia/startup.jl\n$JULIA_BINDIR/../etc/julia/startup.jl","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"by default (via Base.load_julia_startup()).","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"For example, a Linux installation with a Julia executable located at /bin/julia, a DATAROOTDIR of ../share, and a SYSCONFDIR of ../etc will have JULIA_BINDIR set to /bin, a source-file search path of","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"/share/julia/base","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"and a global configuration search path of","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"/etc/julia/startup.jl","page":"Environment Variables"},{"title":"JULIA_PROJECT","location":"manual/environment-variables.html#JULIA_PROJECT","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A directory path that indicates which project should be the initial active project. Setting this environment variable has the same effect as specifying the --project start-up option, but --project has higher precedence. If the variable is set to @. then Julia tries to find a project directory that contains Project.toml or JuliaProject.toml file from the current directory and its parents. See also the chapter on Code Loading.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_PROJECT must be defined before starting julia; defining it in startup.jl is too late in the startup process.","page":"Environment Variables"},{"title":"JULIA_LOAD_PATH","location":"manual/environment-variables.html#JULIA_LOAD_PATH","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The JULIA_LOAD_PATH environment variable is used to populate the global Julia LOAD_PATH variable, which determines which packages can be loaded via import and using (see Code Loading).","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Unlike the shell PATH variable, empty entries in JULIA_LOAD_PATH are expanded to the default value of LOAD_PATH, [\"@\", \"@v#.#\", \"@stdlib\"] when populating LOAD_PATH. This allows easy appending, prepending, etc. of the load path value in shell scripts regardless of whether JULIA_LOAD_PATH is already set or not. For example, to prepend the directory /foo/bar to LOAD_PATH just do","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"export JULIA_LOAD_PATH=\"/foo/bar:$JULIA_LOAD_PATH\"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If the JULIA_LOAD_PATH environment variable is already set, its old value will be prepended with /foo/bar. On the other hand, if JULIA_LOAD_PATH is not set, then it will be set to /foo/bar: which will expand to a LOAD_PATH value of [\"/foo/bar\", \"@\", \"@v#.#\", \"@stdlib\"]. If JULIA_LOAD_PATH is set to the empty string, it expands to an empty LOAD_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty load path via the environment variable. If you want the default load path, either unset the environment variable or if it must have a value, set it to the string :.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph.","page":"Environment Variables"},{"title":"JULIA_DEPOT_PATH","location":"manual/environment-variables.html#JULIA_DEPOT_PATH","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The JULIA_DEPOT_PATH environment variable is used to populate the global Julia DEPOT_PATH variable, which controls where the package manager, as well as Julia's code loading mechanisms, look for package registries, installed packages, named environments, repo clones, cached compiled package images, configuration files, and the default location of the REPL's history file.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Unlike the shell PATH variable but similar to JULIA_LOAD_PATH, empty entries in JULIA_DEPOT_PATH are expanded to the default value of DEPOT_PATH. This allows easy appending, prepending, etc. of the depot path value in shell scripts regardless of whether JULIA_DEPOT_PATH is already set or not. For example, to prepend the directory /foo/bar to DEPOT_PATH just do","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"export JULIA_DEPOT_PATH=\"/foo/bar:$JULIA_DEPOT_PATH\"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If the JULIA_DEPOT_PATH environment variable is already set, its old value will be prepended with /foo/bar. On the other hand, if JULIA_DEPOT_PATH is not set, then it will be set to /foo/bar: which will have the effect of prepending /foo/bar to the default depot path. If JULIA_DEPOT_PATH is set to the empty string, it expands to an empty DEPOT_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty depot path via the environment variable. If you want the default depot path, either unset the environment variable or if it must have a value, set it to the string :.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_DEPOT_PATH must be defined before starting julia; defining it in startup.jl is too late in the startup process; at that point you can instead directly modify the DEPOT_PATH array, which is populated from the environment variable.","page":"Environment Variables"},{"title":"JULIA_HISTORY","location":"manual/environment-variables.html#JULIA_HISTORY","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path REPL.find_hist_file() of the REPL's history file. If $JULIA_HISTORY is not set, then REPL.find_hist_file() defaults to","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$(DEPOT_PATH[1])/logs/repl_history.jl","page":"Environment Variables"},{"title":"JULIA_MAX_NUM_PRECOMPILE_FILES","location":"manual/environment-variables.html#env-max-num-precompile-files","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Sets the maximum number of different instances of a single package that are to be stored in the precompile cache (default = 10).","page":"Environment Variables"},{"title":"JULIA_VERBOSE_LINKING","location":"manual/environment-variables.html#JULIA_VERBOSE_LINKING","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, linker commands will be displayed during precompilation.","page":"Environment Variables"},{"title":"Pkg.jl","location":"manual/environment-variables.html#Pkg.jl","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_CI","location":"manual/environment-variables.html#JULIA_CI","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this indicates to the package server that any package operations are part of a continuous integration (CI) system for the purposes of gathering package usage statistics.","page":"Environment Variables"},{"title":"JULIA_NUM_PRECOMPILE_TASKS","location":"manual/environment-variables.html#JULIA_NUM_PRECOMPILE_TASKS","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The number of parallel tasks to use when precompiling packages. See Pkg.precompile.","page":"Environment Variables"},{"title":"JULIA_PKG_DEVDIR","location":"manual/environment-variables.html#JULIA_PKG_DEVDIR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The default directory used by Pkg.develop for downloading packages.","page":"Environment Variables"},{"title":"JULIA_PKG_IGNORE_HASHES","location":"manual/environment-variables.html#JULIA_PKG_IGNORE_HASHES","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to 1, this will ignore incorrect hashes in artifacts. This should be used carefully, as it disables verification of downloads, but can resolve issues when moving files across different types of file systems. See Pkg.jl issue #2317 for more details.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.6\nThis is only supported in Julia 1.6 and above.","page":"Environment Variables"},{"title":"JULIA_PKG_OFFLINE","location":"manual/environment-variables.html#JULIA_PKG_OFFLINE","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this will enable offline mode: see Pkg.offline.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.5\nPkg's offline mode requires Julia 1.5 or later.","page":"Environment Variables"},{"title":"JULIA_PKG_PRECOMPILE_AUTO","location":"manual/environment-variables.html#JULIA_PKG_PRECOMPILE_AUTO","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to 0, this will disable automatic precompilation by package actions which change the manifest. See Pkg.precompile.","page":"Environment Variables"},{"title":"JULIA_PKG_SERVER","location":"manual/environment-variables.html#JULIA_PKG_SERVER","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specifies the URL of the package registry to use. By default, Pkg uses https://pkg.julialang.org to fetch Julia packages. In addition, you can disable the use of the PkgServer protocol, and instead access the packages directly from their hosts (GitHub, GitLab, etc.) by setting: export JULIA_PKG_SERVER=\"\"","page":"Environment Variables"},{"title":"JULIA_PKG_SERVER_REGISTRY_PREFERENCE","location":"manual/environment-variables.html#JULIA_PKG_SERVER_REGISTRY_PREFERENCE","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specifies the preferred registry flavor. Currently supported values are conservative (the default), which will only publish resources that have been processed by the storage server (and thereby have a higher probability of being available from the PkgServers), whereas eager will publish registries whose resources have not necessarily been processed by the storage servers.  Users behind restrictive firewalls that do not allow downloading from arbitrary servers should not use the eager flavor.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThis only affects Julia 1.7 and above.","page":"Environment Variables"},{"title":"JULIA_PKG_UNPACK_REGISTRY","location":"manual/environment-variables.html#JULIA_PKG_UNPACK_REGISTRY","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this will unpack the registry instead of storing it as a compressed tarball.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThis only affects Julia 1.7 and above. Earlier versions will always unpack the registry.","page":"Environment Variables"},{"title":"JULIA_PKG_USE_CLI_GIT","location":"manual/environment-variables.html#JULIA_PKG_USE_CLI_GIT","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, Pkg operations which use the git protocol will use an external git executable instead of the default libgit2 library.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nUse of the git executable is only supported on Julia 1.7 and above.","page":"Environment Variables"},{"title":"JULIA_PKGRESOLVE_ACCURACY","location":"manual/environment-variables.html#JULIA_PKGRESOLVE_ACCURACY","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The accuracy of the package resolver. This should be a positive integer, the default is 1.","page":"Environment Variables"},{"title":"Network transport","location":"manual/environment-variables.html#Network-transport","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_NO_VERIFY_HOSTS / JULIA_SSL_NO_VERIFY_HOSTS / JULIA_SSH_NO_VERIFY_HOSTS / JULIA_ALWAYS_VERIFY_HOSTS","location":"manual/environment-variables.html#JULIA_NO_VERIFY_HOSTS-/-JULIA_SSL_NO_VERIFY_HOSTS-/-JULIA_SSH_NO_VERIFY_HOSTS-/-JULIA_ALWAYS_VERIFY_HOSTS","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specify hosts whose identity should or should not be verified for specific transport layers. See NetworkOptions.verify_host","page":"Environment Variables"},{"title":"JULIA_SSL_CA_ROOTS_PATH","location":"manual/environment-variables.html#JULIA_SSL_CA_ROOTS_PATH","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specify the file or directory containing the certificate authority roots. See NetworkOptions.ca_roots","page":"Environment Variables"},{"title":"External applications","location":"manual/environment-variables.html#External-applications","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_SHELL","location":"manual/environment-variables.html#JULIA_SHELL","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path of the shell with which Julia should execute external commands (via Base.repl_cmd()). Defaults to the environment variable $SHELL, and falls back to /bin/sh if $SHELL is unset.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, this environment variable is ignored, and external commands are executed directly.","page":"Environment Variables"},{"title":"JULIA_EDITOR","location":"manual/environment-variables.html#JULIA_EDITOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The editor returned by InteractiveUtils.editor() and used in, e.g., InteractiveUtils.edit, referring to the command of the preferred editor, for instance vim.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_EDITOR takes precedence over $VISUAL, which in turn takes precedence over $EDITOR. If none of these environment variables is set, then the editor is taken to be open on Windows and OS X, or /etc/alternatives/editor if it exists, or emacs otherwise.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"To use Visual Studio Code on Windows, set $JULIA_EDITOR to code.cmd.","page":"Environment Variables"},{"title":"Parallelization","location":"manual/environment-variables.html#Parallelization","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_CPU_THREADS","location":"manual/environment-variables.html#JULIA_CPU_THREADS","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Overrides the global variable Base.Sys.CPU_THREADS, the number of logical CPU cores available.","page":"Environment Variables"},{"title":"JULIA_WORKER_TIMEOUT","location":"manual/environment-variables.html#JULIA_WORKER_TIMEOUT","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A Float64 that sets the value of Distributed.worker_timeout() (default: 60.0). This function gives the number of seconds a worker process will wait for a master process to establish a connection before dying.","page":"Environment Variables"},{"title":"JULIA_NUM_THREADS","location":"manual/environment-variables.html#JULIA_NUM_THREADS","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"An unsigned 64-bit integer (uint64_t) that sets the maximum number of threads available to Julia.  If $JULIA_NUM_THREADS is not positive or is not set, or if the number of CPU threads cannot be determined through system calls, then the number of threads is set to 1.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If $JULIA_NUM_THREADS is set to auto, then the number of threads will be set to the number of CPU threads.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_NUM_THREADS must be defined before starting julia; defining it in startup.jl is too late in the startup process.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.5\nIn Julia 1.5 and above the number of threads can also be specified on startup using the -t/--threads command line argument.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThe auto value for $JULIA_NUM_THREADS requires Julia 1.7 or above.","page":"Environment Variables"},{"title":"JULIA_THREAD_SLEEP_THRESHOLD","location":"manual/environment-variables.html#JULIA_THREAD_SLEEP_THRESHOLD","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to a string that starts with the case-insensitive substring \"infinite\", then spinning threads never sleep. Otherwise, $JULIA_THREAD_SLEEP_THRESHOLD is interpreted as an unsigned 64-bit integer (uint64_t) and gives, in nanoseconds, the amount of time after which spinning threads should sleep.","page":"Environment Variables"},{"title":"JULIA_EXCLUSIVE","location":"manual/environment-variables.html#JULIA_EXCLUSIVE","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then Julia's thread policy is consistent with running on a dedicated machine: the master thread is on proc 0, and threads are affinitized. Otherwise, Julia lets the operating system handle thread policy.","page":"Environment Variables"},{"title":"REPL formatting","location":"manual/environment-variables.html#REPL-formatting","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Environment variables that determine how REPL output should be formatted at the terminal. Generally, these variables should be set to ANSI terminal escape sequences. Julia provides a high-level interface with much of the same functionality; see the section on The Julia REPL.","page":"Environment Variables"},{"title":"JULIA_ERROR_COLOR","location":"manual/environment-variables.html#JULIA_ERROR_COLOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.error_color() (default: light red, \"\\033[91m\") that errors should have at the terminal.","page":"Environment Variables"},{"title":"JULIA_WARN_COLOR","location":"manual/environment-variables.html#JULIA_WARN_COLOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.warn_color() (default: yellow, \"\\033[93m\") that warnings should have at the terminal.","page":"Environment Variables"},{"title":"JULIA_INFO_COLOR","location":"manual/environment-variables.html#JULIA_INFO_COLOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.info_color() (default: cyan, \"\\033[36m\") that info should have at the terminal.","page":"Environment Variables"},{"title":"JULIA_INPUT_COLOR","location":"manual/environment-variables.html#JULIA_INPUT_COLOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.input_color() (default: normal, \"\\033[0m\") that input should have at the terminal.","page":"Environment Variables"},{"title":"JULIA_ANSWER_COLOR","location":"manual/environment-variables.html#JULIA_ANSWER_COLOR","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.answer_color() (default: normal, \"\\033[0m\") that output should have at the terminal.","page":"Environment Variables"},{"title":"Debugging and profiling","location":"manual/environment-variables.html#Debugging-and-profiling","category":"section","text":"","page":"Environment Variables"},{"title":"JULIA_DEBUG","location":"manual/environment-variables.html#JULIA_DEBUG","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Enable debug logging for a file or module, see Logging for more information.","page":"Environment Variables"},{"title":"JULIA_GC_ALLOC_POOL, JULIA_GC_ALLOC_OTHER, JULIA_GC_ALLOC_PRINT","location":"manual/environment-variables.html#JULIA_GC_ALLOC_POOL,-JULIA_GC_ALLOC_OTHER,-JULIA_GC_ALLOC_PRINT","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set, these environment variables take strings that optionally start with the character 'r', followed by a string interpolation of a colon-separated list of three signed 64-bit integers (int64_t). This triple of integers a:b:c represents the arithmetic sequence a, a + b, a + 2*b, ... c.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If it's the nth time that jl_gc_pool_alloc() has been called, and n   belongs to the arithmetic sequence represented by $JULIA_GC_ALLOC_POOL,   then garbage collection is forced.\nIf it's the nth time that maybe_collect() has been called, and n belongs   to the arithmetic sequence represented by $JULIA_GC_ALLOC_OTHER, then garbage   collection is forced.\nIf it's the nth time that jl_gc_collect() has been called, and n belongs   to the arithmetic sequence represented by $JULIA_GC_ALLOC_PRINT, then counts   for the number of calls to jl_gc_pool_alloc() and maybe_collect() are   printed.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If the value of the environment variable begins with the character 'r', then the interval between garbage collection events is randomized.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThese environment variables only have an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration).","page":"Environment Variables"},{"title":"JULIA_GC_NO_GENERATIONAL","location":"manual/environment-variables.html#JULIA_GC_NO_GENERATIONAL","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then the Julia garbage collector never performs \"quick sweeps\" of memory.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration).","page":"Environment Variables"},{"title":"JULIA_GC_WAIT_FOR_DEBUGGER","location":"manual/environment-variables.html#JULIA_GC_WAIT_FOR_DEBUGGER","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then the Julia garbage collector will wait for a debugger to attach instead of aborting whenever there's a critical error.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration).","page":"Environment Variables"},{"title":"ENABLE_JITPROFILING","location":"manual/environment-variables.html#ENABLE_JITPROFILING","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then the compiler will create and register an event listener for just-in-time (JIT) profiling.","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThis environment variable only has an effect if Julia was compiled with JIT profiling support, using eitherIntel's VTune™ Amplifier (USE_INTEL_JITEVENTS set to 1 in the build configuration), or\nOProfile (USE_OPROFILE_JITEVENTS set to 1 in the build configuration).\nPerf (USE_PERF_JITEVENTS set to 1 in the build configuration). This integration is enabled by default.","page":"Environment Variables"},{"title":"ENABLE_GDBLISTENER","location":"manual/environment-variables.html#ENABLE_GDBLISTENER","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0 enables GDB registration of Julia code on release builds. On debug builds of Julia this is always enabled. Recommended to use with -g 2.","page":"Environment Variables"},{"title":"JULIA_LLVM_ARGS","location":"manual/environment-variables.html#JULIA_LLVM_ARGS","category":"section","text":"","page":"Environment Variables"},{"title":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Arguments to be passed to the LLVM backend.","page":"Environment Variables"},{"title":"Future","location":"stdlib/Future.html#Future","category":"section","text":"","page":"Future"},{"title":"Future","location":"stdlib/Future.html","category":"page","text":"The Future module implements future behavior of already existing functions, which will replace the current version in a future release of Julia.","page":"Future"},{"title":"Future","location":"stdlib/Future.html","category":"page","text":"Future.copy!\nFuture.randjump","page":"Future"},{"title":"Future.copy!","location":"stdlib/Future.html#Future.copy!","category":"function","text":"Future.copy!(dst, src) -> dst\n\nCopy src into dst.\n\ncompat: Julia 1.1\nThis function has moved to Base with Julia 1.1, consider using copy!(dst, src) instead. Future.copy! will be deprecated in the future.\n\n\n\n\n\n","page":"Future"},{"title":"Future.randjump","location":"stdlib/Future.html#Future.randjump","category":"function","text":"randjump(r::MersenneTwister, steps::Integer) -> MersenneTwister\n\nCreate an initialized MersenneTwister object, whose state is moved forward (without generating numbers) from r by steps steps. One such step corresponds to the generation of two Float64 numbers. For each different value of steps, a large polynomial has to be generated internally. One is already pre-computed for steps=big(10)^20.\n\n\n\n\n\n","page":"Future"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-and-Rational-Numbers","category":"section","text":"","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Julia includes predefined types for both complex and rational numbers, and supports all the standard Mathematical Operations and Elementary Functions on them. Conversion and Promotion are defined so that operations on any combination of predefined numeric types, whether primitive or composite, behave as expected.","page":"Complex and Rational Numbers"},{"title":"Complex Numbers","location":"manual/complex-and-rational-numbers.html#Complex-Numbers","category":"section","text":"","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The global constant im is bound to the complex number i, representing the principal square root of -1. (Using mathematicians' i or engineers' j for this global constant was rejected since they are such popular index variable names.) Since Julia allows numeric literals to be juxtaposed with identifiers as coefficients, this binding suffices to provide convenient syntax for complex numbers, similar to the traditional mathematical notation:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 1+2im\n1 + 2im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"You can perform all the standard arithmetic operations with complex numbers:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> (1 + 2im)*(2 - 3im)\n8 + 1im\n\njulia> (1 + 2im)/(1 - 2im)\n-0.6 + 0.8im\n\njulia> (1 + 2im) + (1 - 2im)\n2 + 0im\n\njulia> (-3 + 2im) - (5 - 1im)\n-8 + 3im\n\njulia> (-1 + 2im)^2\n-3 - 4im\n\njulia> (-1 + 2im)^2.5\n2.729624464784009 - 6.9606644595719im\n\njulia> (-1 + 2im)^(1 + 1im)\n-0.27910381075826657 + 0.08708053414102428im\n\njulia> 3(2 - 5im)\n6 - 15im\n\njulia> 3(2 - 5im)^2\n-63 - 60im\n\njulia> 3(2 - 5im)^-1.0\n0.20689655172413793 + 0.5172413793103449im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The promotion mechanism ensures that combinations of operands of different types just work:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2(1 - 1im)\n2 - 2im\n\njulia> (2 + 3im) - 1\n1 + 3im\n\njulia> (1 + 2im) + 0.5\n1.5 + 2.0im\n\njulia> (2 + 3im) - 0.5im\n2.0 + 2.5im\n\njulia> 0.75(1 + 2im)\n0.75 + 1.5im\n\njulia> (2 + 3im) / 2\n1.0 + 1.5im\n\njulia> (1 - 3im) / (2 + 2im)\n-0.5 - 1.0im\n\njulia> 2im^2\n-2 + 0im\n\njulia> 1 + 3/4im\n1.0 - 0.75im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Note that 3/4im == 3/(4*im) == -(3/4*im), since a literal coefficient binds more tightly than division.","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Standard functions to manipulate complex values are provided:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> z = 1 + 2im\n1 + 2im\n\njulia> real(1 + 2im) # real part of z\n1\n\njulia> imag(1 + 2im) # imaginary part of z\n2\n\njulia> conj(1 + 2im) # complex conjugate of z\n1 - 2im\n\njulia> abs(1 + 2im) # absolute value of z\n2.23606797749979\n\njulia> abs2(1 + 2im) # squared absolute value\n5\n\njulia> angle(1 + 2im) # phase angle in radians\n1.1071487177940904","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"As usual, the absolute value (abs) of a complex number is its distance from zero. abs2 gives the square of the absolute value, and is of particular use for complex numbers since it avoids taking a square root. angle returns the phase angle in radians (also known as the argument or arg function). The full gamut of other Elementary Functions is also defined for complex numbers:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> sqrt(1im)\n0.7071067811865476 + 0.7071067811865475im\n\njulia> sqrt(1 + 2im)\n1.272019649514069 + 0.7861513777574233im\n\njulia> cos(1 + 2im)\n2.0327230070196656 - 3.0518977991517997im\n\njulia> exp(1 + 2im)\n-1.1312043837568135 + 2.4717266720048188im\n\njulia> sinh(1 + 2im)\n-0.4890562590412937 + 1.4031192506220405im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Note that mathematical functions typically return real values when applied to real numbers and complex values when applied to complex numbers. For example, sqrt behaves differently when applied to -1 versus -1 + 0im even though -1 == -1 + 0im:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> sqrt(-1 + 0im)\n0.0 + 1.0im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The literal numeric coefficient notation does not work when constructing a complex number from variables. Instead, the multiplication must be explicitly written out:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2; a + b*im\n1 + 2im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"However, this is not recommended. Instead, use the more efficient complex function to construct a complex value directly from its real and imaginary parts:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2; complex(a, b)\n1 + 2im","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"This construction avoids the multiplication and addition operations.","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Inf and NaN propagate through complex numbers in the real and imaginary parts of a complex number as described in the Special floating-point values section:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 1 + Inf*im\n1.0 + Inf*im\n\njulia> 1 + NaN*im\n1.0 + NaN*im","page":"Complex and Rational Numbers"},{"title":"Rational Numbers","location":"manual/complex-and-rational-numbers.html#Rational-Numbers","category":"section","text":"","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Julia has a rational number type to represent exact ratios of integers. Rationals are constructed using the // operator:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2//3\n2//3","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"If the numerator and denominator of a rational have common factors, they are reduced to lowest terms such that the denominator is non-negative:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 6//9\n2//3\n\njulia> -4//8\n-1//2\n\njulia> 5//-15\n-1//3\n\njulia> -4//-12\n1//3","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"This normalized form for a ratio of integers is unique, so equality of rational values can be tested by checking for equality of the numerator and denominator. The standardized numerator and denominator of a rational value can be extracted using the numerator and denominator functions:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> numerator(2//3)\n2\n\njulia> denominator(2//3)\n3","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Direct comparison of the numerator and denominator is generally not necessary, since the standard arithmetic and comparison operations are defined for rational values:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2//3 == 6//9\ntrue\n\njulia> 2//3 == 9//27\nfalse\n\njulia> 3//7 < 1//2\ntrue\n\njulia> 3//4 > 2//3\ntrue\n\njulia> 2//4 + 1//6\n2//3\n\njulia> 5//12 - 1//4\n1//6\n\njulia> 5//8 * 3//12\n5//32\n\njulia> 6//5 / 10//7\n21//25","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Rationals can easily be converted to floating-point numbers:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> float(3//4)\n0.75","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Conversion from rational to floating-point respects the following identity for any integral values of a and b, with the exception of the two cases b == 0 and a == 0 && b < 0:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2;\n\njulia> isequal(float(a//b), a/b)\ntrue","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Constructing infinite rational values is acceptable:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 5//0\n1//0\n\njulia> x = -3//0\n-1//0\n\njulia> typeof(x)\nRational{Int64}","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Trying to construct a NaN rational value, however, is invalid:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 0//0\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"As usual, the promotion system makes interactions with other numeric types effortless:","page":"Complex and Rational Numbers"},{"title":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 3//5 + 1\n8//5\n\njulia> 3//5 - 0.5\n0.09999999999999998\n\njulia> 2//7 * (1 + 2im)\n2//7 + 4//7*im\n\njulia> 2//7 * (1.5 + 2im)\n0.42857142857142855 + 0.5714285714285714im\n\njulia> 3//2 / (1 + 2im)\n3//10 - 3//5*im\n\njulia> 1//2 + 2im\n1//2 + 2//1*im\n\njulia> 1 + 2//3im\n1//1 - 2//3*im\n\njulia> 0.5 == 1//2\ntrue\n\njulia> 0.33 == 1//3\nfalse\n\njulia> 0.33 < 1//3\ntrue\n\njulia> 1//3 - 0.33\n0.0033333333333332993","page":"Complex and Rational Numbers"},{"title":"More about types","location":"devdocs/types.html#More-about-types","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"If you've used Julia for a while, you understand the fundamental role that types play.  Here we try to get under the hood, focusing particularly on Parametric Types.","page":"More about types"},{"title":"Types and sets (and Any and Union{}/Bottom)","location":"devdocs/types.html#Types-and-sets-(and-Any-and-Union{}/Bottom)","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"It's perhaps easiest to conceive of Julia's type system in terms of sets. While programs manipulate individual values, a type refers to a set of values. This is not the same thing as a collection; for example a Set of values is itself a single Set value. Rather, a type describes a set of possible values, expressing uncertainty about which value we have.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"A concrete type T describes the set of values whose direct tag, as returned by the typeof function, is T. An abstract type describes some possibly-larger set of values.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Any describes the entire universe of possible values. Integer is a subset of Any that includes Int, Int8, and other concrete types. Internally, Julia also makes heavy use of another type known as Bottom, which can also be written as Union{}. This corresponds to the empty set.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Julia's types support the standard operations of set theory: you can ask whether T1 is a \"subset\" (subtype) of T2 with T1 <: T2. Likewise, you intersect two types using typeintersect, take their union with Union, and compute a type that contains their union with typejoin:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> typeintersect(Int, Float64)\nUnion{}\n\njulia> Union{Int, Float64}\nUnion{Float64, Int64}\n\njulia> typejoin(Int, Float64)\nReal\n\njulia> typeintersect(Signed, Union{UInt8, Int8})\nInt8\n\njulia> Union{Signed, Union{UInt8, Int8}}\nUnion{UInt8, Signed}\n\njulia> typejoin(Signed, Union{UInt8, Int8})\nInteger\n\njulia> typeintersect(Tuple{Integer, Float64}, Tuple{Int, Real})\nTuple{Int64, Float64}\n\njulia> Union{Tuple{Integer, Float64}, Tuple{Int, Real}}\nUnion{Tuple{Int64, Real}, Tuple{Integer, Float64}}\n\njulia> typejoin(Tuple{Integer, Float64}, Tuple{Int, Real})\nTuple{Integer, Real}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"While these operations may seem abstract, they lie at the heart of Julia.  For example, method dispatch is implemented by stepping through the items in a method list until reaching one for which the type of the argument tuple is a subtype of the method signature. For this algorithm to work, it's important that methods be sorted by their specificity, and that the search begins with the most specific methods. Consequently, Julia also implements a partial order on types; this is achieved by functionality that is similar to <:, but with differences that will be discussed below.","page":"More about types"},{"title":"UnionAll types","location":"devdocs/types.html#UnionAll-types","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Julia's type system can also express an iterated union of types: a union of types over all values of some variable. This is needed to describe parametric types where the values of some parameters are not known.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"For example, Array has two parameters as in Array{Int,2}. If we did not know the element type, we could write Array{T,2} where T, which is the union of Array{T,2} for all values of T: Union{Array{Int8,2}, Array{Int16,2}, ...}.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Such a type is represented by a UnionAll object, which contains a variable (T in this example, of type TypeVar), and a wrapped type (Array{T,2} in this example).","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Consider the following methods:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f1(A::Array) = 1\nf2(A::Array{Int}) = 2\nf3(A::Array{T}) where {T<:Any} = 3\nf4(A::Array{Any}) = 4","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The signature - as described in Function calls - of f3 is a UnionAll type wrapping a tuple type: Tuple{typeof(f3), Array{T}} where T. All but f4 can be called with a = [1,2]; all but f2 can be called with b = Any[1,2].","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Let's look at these types a little more closely:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> dump(Array)\nUnionAll\n  var: TypeVar\n    name: Symbol T\n    lb: Union{}\n    ub: Any\n  body: UnionAll\n    var: TypeVar\n      name: Symbol N\n      lb: Union{}\n      ub: Any\n    body: Array{T, N} <: DenseArray{T, N}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"This indicates that Array actually names a UnionAll type. There is one UnionAll type for each parameter, nested. The syntax Array{Int,2} is equivalent to Array{Int}{2}; internally each UnionAll is instantiated with a particular variable value, one at a time, outermost-first. This gives a natural meaning to the omission of trailing type parameters; Array{Int} gives a type equivalent to Array{Int,N} where N.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"A TypeVar is not itself a type, but rather should be considered part of the structure of a UnionAll type. Type variables have lower and upper bounds on their values (in the fields lb and ub). The symbol name is purely cosmetic. Internally, TypeVars are compared by address, so they are defined as mutable types to ensure that \"different\" type variables can be distinguished. However, by convention they should not be mutated.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"One can construct TypeVars manually:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> TypeVar(:V, Signed, Real)\nSigned<:V<:Real","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"There are convenience versions that allow you to omit any of these arguments except the name symbol.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The syntax Array{T} where T<:Integer is lowered to","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"let T = TypeVar(:T,Integer)\n    UnionAll(T, Array{T})\nend","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"so it is seldom necessary to construct a TypeVar manually (indeed, this is to be avoided).","page":"More about types"},{"title":"Free variables","location":"devdocs/types.html#Free-variables","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The concept of a free type variable is extremely important in the type system. We say that a variable V is free in type T if T does not contain the UnionAll that introduces variable V. For example, the type Array{Array{V} where V<:Integer} has no free variables, but the Array{V} part inside of it does have a free variable, V.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"A type with free variables is, in some sense, not really a type at all. Consider the type Array{Array{T}} where T, which refers to all homogeneous arrays of arrays. The inner type Array{T}, seen by itself, might seem to refer to any kind of array. However, every element of the outer array must have the same array type, so Array{T} cannot refer to just any old array. One could say that Array{T} effectively \"occurs\" multiple times, and T must have the same value each \"time\".","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"For this reason, the function jl_has_free_typevars in the C API is very important. Types for which it returns true will not give meaningful answers in subtyping and other type functions.","page":"More about types"},{"title":"TypeNames","location":"devdocs/types.html#TypeNames","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The following two Array types are functionally equivalent, yet print differently:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> TV, NV = TypeVar(:T), TypeVar(:N)\n(T, N)\n\njulia> Array\nArray\n\njulia> Array{TV, NV}\nArray{T, N}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"These can be distinguished by examining the name field of the type, which is an object of type TypeName:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> dump(Array{Int,1}.name)\nTypeName\n  name: Symbol Array\n  module: Module Core\n  names: empty SimpleVector\n  wrapper: UnionAll\n    var: TypeVar\n      name: Symbol T\n      lb: Union{}\n      ub: Any\n    body: UnionAll\n      var: TypeVar\n        name: Symbol N\n        lb: Union{}\n        ub: Any\n      body: Array{T, N} <: DenseArray{T, N}\n  cache: SimpleVector\n    ...\n\n  linearcache: SimpleVector\n    ...\n\n  hash: Int64 -7900426068641098781\n  mt: MethodTable\n    name: Symbol Array\n    defs: Nothing nothing\n    cache: Nothing nothing\n    max_args: Int64 0\n    module: Module Core\n    : Int64 0\n    : Int64 0","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"In this case, the relevant field is wrapper, which holds a reference to the top-level type used to make new Array types.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> pointer_from_objref(Array)\nPtr{Cvoid} @0x00007fcc7de64850\n\njulia> pointer_from_objref(Array.body.body.name.wrapper)\nPtr{Cvoid} @0x00007fcc7de64850\n\njulia> pointer_from_objref(Array{TV,NV})\nPtr{Cvoid} @0x00007fcc80c4d930\n\njulia> pointer_from_objref(Array{TV,NV}.name.wrapper)\nPtr{Cvoid} @0x00007fcc7de64850","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The wrapper field of Array points to itself, but for Array{TV,NV} it points back to the original definition of the type.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"What about the other fields? hash assigns an integer to each type.  To examine the cache field, it's helpful to pick a type that is less heavily used than Array. Let's first create our own type:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> struct MyType{T,N} end\n\njulia> MyType{Int,2}\nMyType{Int64, 2}\n\njulia> MyType{Float32, 5}\nMyType{Float32, 5}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"When you instantiate a parametric type, each concrete type gets saved in a type cache (MyType.body.body.name.cache). However, instances containing free type variables are not cached.","page":"More about types"},{"title":"Tuple types","location":"devdocs/types.html#Tuple-types","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Tuple types constitute an interesting special case.  For dispatch to work on declarations like x::Tuple, the type has to be able to accommodate any tuple.  Let's check the parameters:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> Tuple\nTuple\n\njulia> Tuple.parameters\nsvec(Vararg{Any})","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Unlike other types, tuple types are covariant in their parameters, so this definition permits Tuple to match any type of tuple:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> typeintersect(Tuple, Tuple{Int,Float64})\nTuple{Int64, Float64}\n\njulia> typeintersect(Tuple{Vararg{Any}}, Tuple{Int,Float64})\nTuple{Int64, Float64}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"However, if a variadic (Vararg) tuple type has free variables it can describe different kinds of tuples:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> typeintersect(Tuple{Vararg{T} where T}, Tuple{Int,Float64})\nTuple{Int64, Float64}\n\njulia> typeintersect(Tuple{Vararg{T}} where T, Tuple{Int,Float64})\nUnion{}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Notice that when T is free with respect to the Tuple type (i.e. its binding UnionAll type is outside the Tuple type), only one T value must work over the whole type. Therefore a heterogeneous tuple does not match.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Finally, it's worth noting that Tuple{} is distinct:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> Tuple{}\nTuple{}\n\njulia> Tuple{}.parameters\nsvec()\n\njulia> typeintersect(Tuple{}, Tuple{Int})\nUnion{}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"What is the \"primary\" tuple-type?","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> pointer_from_objref(Tuple)\nPtr{Cvoid} @0x00007f5998a04370\n\njulia> pointer_from_objref(Tuple{})\nPtr{Cvoid} @0x00007f5998a570d0\n\njulia> pointer_from_objref(Tuple.name.wrapper)\nPtr{Cvoid} @0x00007f5998a04370\n\njulia> pointer_from_objref(Tuple{}.name.wrapper)\nPtr{Cvoid} @0x00007f5998a04370","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"so Tuple == Tuple{Vararg{Any}} is indeed the primary type.","page":"More about types"},{"title":"Diagonal types","location":"devdocs/types.html#Diagonal-types","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Consider the type Tuple{T,T} where T. A method with this signature would look like:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f(x::T, y::T) where {T} = ...","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"According to the usual interpretation of a UnionAll type, this T ranges over all types, including Any, so this type should be equivalent to Tuple{Any,Any}. However, this interpretation causes some practical problems.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"First, a value of T needs to be available inside the method definition. For a call like f(1, 1.0), it's not clear what T should be. It could be Union{Int,Float64}, or perhaps Real. Intuitively, we expect the declaration x::T to mean T === typeof(x). To make sure that invariant holds, we need typeof(x) === typeof(y) === T in this method. That implies the method should only be called for arguments of the exact same type.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"It turns out that being able to dispatch on whether two values have the same type is very useful (this is used by the promotion system for example), so we have multiple reasons to want a different interpretation of Tuple{T,T} where T. To make this work we add the following rule to subtyping: if a variable occurs more than once in covariant position, it is restricted to ranging over only concrete types. (\"Covariant position\" means that only Tuple and Union types occur between an occurrence of a variable and the UnionAll type that introduces it.) Such variables are called \"diagonal variables\" or \"concrete variables\".","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"So for example, Tuple{T,T} where T can be seen as Union{Tuple{Int8,Int8}, Tuple{Int16,Int16}, ...}, where T ranges over all concrete types. This gives rise to some interesting subtyping results. For example Tuple{Real,Real} is not a subtype of Tuple{T,T} where T, because it includes some types like Tuple{Int8,Int16} where the two elements have different types. Tuple{Real,Real} and Tuple{T,T} where T have the non-trivial intersection Tuple{T,T} where T<:Real. However, Tuple{Real} is a subtype of Tuple{T} where T, because in that case T occurs only once and so is not diagonal.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Next consider a signature like the following:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f(a::Array{T}, x::T, y::T) where {T} = ...","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"In this case, T occurs in invariant position inside Array{T}. That means whatever type of array is passed unambiguously determines the value of T – we say T has an equality constraint on it. Therefore in this case the diagonal rule is not really necessary, since the array determines T and we can then allow x and y to be of any subtypes of T. So variables that occur in invariant position are never considered diagonal. This choice of behavior is slightly controversial – some feel this definition should be written as","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f(a::Array{T}, x::S, y::S) where {T, S<:T} = ...","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"to clarify whether x and y need to have the same type. In this version of the signature they would, or we could introduce a third variable for the type of y if x and y can have different types.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The next complication is the interaction of unions and diagonal variables, e.g.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f(x::Union{Nothing,T}, y::T) where {T} = ...","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Consider what this declaration means. y has type T. x then can have either the same type T, or else be of type Nothing. So all of the following calls should match:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"f(1, 1)\nf(\"\", \"\")\nf(2.0, 2.0)\nf(nothing, 1)\nf(nothing, \"\")\nf(nothing, 2.0)","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"These examples are telling us something: when x is nothing::Nothing, there are no extra constraints on y. It is as if the method signature had y::Any. Indeed, we have the following type equivalence:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"(Tuple{Union{Nothing,T},T} where T) == Union{Tuple{Nothing,Any}, Tuple{T,T} where T}","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The general rule is: a concrete variable in covariant position acts like it's not concrete if the subtyping algorithm only uses it once. When x has type Nothing, we don't need to use the T in Union{Nothing,T}; we only use it in the second slot. This arises naturally from the observation that in Tuple{T} where T restricting T to concrete types makes no difference; the type is equal to Tuple{Any} either way.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"However, appearing in invariant position disqualifies a variable from being concrete whether that appearance of the variable is used or not. Otherwise types can behave differently depending on which other types they are compared to, making subtyping not transitive. For example, consider","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Tuple{Int,Int8,Vector{Integer}} <: Tuple{T,T,Vector{Union{Integer,T}}} where T","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"If the T inside the Union is ignored, then T is concrete and the answer is \"false\" since the first two types aren't the same. But consider instead","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Tuple{Int,Int8,Vector{Any}} <: Tuple{T,T,Vector{Union{Integer,T}}} where T","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Now we cannot ignore the T in the Union (we must have T == Any), so T is not concrete and the answer is \"true\". That would make the concreteness of T depend on the other type, which is not acceptable since a type must have a clear meaning on its own. Therefore the appearance of T inside Vector is considered in both cases.","page":"More about types"},{"title":"Subtyping diagonal variables","location":"devdocs/types.html#Subtyping-diagonal-variables","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The subtyping algorithm for diagonal variables has two components: (1) identifying variable occurrences, and (2) ensuring that diagonal variables range over concrete types only.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The first task is accomplished by keeping counters occurs_inv and occurs_cov (in src/subtype.c) for each variable in the environment, tracking the number of invariant and covariant occurrences, respectively. A variable is diagonal when occurs_inv == 0 && occurs_cov > 1.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The second task is accomplished by imposing a condition on a variable's lower bound. As the subtyping algorithm runs, it narrows the bounds of each variable (raising lower bounds and lowering upper bounds) to keep track of the range of variable values for which the subtype relation would hold. When we are done evaluating the body of a UnionAll type whose variable is diagonal, we look at the final values of the bounds. Since the variable must be concrete, a contradiction occurs if its lower bound could not be a subtype of a concrete type. For example, an abstract type like AbstractArray cannot be a subtype of a concrete type, but a concrete type like Int can be, and the empty type Bottom can be as well. If a lower bound fails this test the algorithm stops with the answer false.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"For example, in the problem Tuple{Int,String} <: Tuple{T,T} where T, we derive that this would be true if T were a supertype of Union{Int,String}. However, Union{Int,String} is an abstract type, so the relation does not hold.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"This concreteness test is done by the function is_leaf_bound. Note that this test is slightly different from jl_is_leaf_type, since it also returns true for Bottom. Currently this function is heuristic, and does not catch all possible concrete types. The difficulty is that whether a lower bound is concrete might depend on the values of other type variable bounds. For example, Vector{T} is equivalent to the concrete type Vector{Int} only if both the upper and lower bounds of T equal Int. We have not yet worked out a complete algorithm for this.","page":"More about types"},{"title":"Introduction to the internal machinery","location":"devdocs/types.html#Introduction-to-the-internal-machinery","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Most operations for dealing with types are found in the files jltypes.c and subtype.c. A good way to start is to watch subtyping in action. Build Julia with make debug and fire up Julia within a debugger. gdb debugging tips has some tips which may be useful.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"Because the subtyping code is used heavily in the REPL itself – and hence breakpoints in this code get triggered often – it will be easiest if you make the following definition:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"julia> function mysubtype(a,b)\n           ccall(:jl_breakpoint, Cvoid, (Any,), nothing)\n           a <: b\n       end","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"and then set a breakpoint in jl_breakpoint.  Once this breakpoint gets triggered, you can set breakpoints in other functions.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"As a warm-up, try the following:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"mysubtype(Tuple{Int, Float64}, Tuple{Integer, Real})","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"We can make it more interesting by trying a more complex case:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"mysubtype(Tuple{Array{Int,2}, Int8}, Tuple{Array{T}, T} where T)","page":"More about types"},{"title":"Subtyping and method sorting","location":"devdocs/types.html#Subtyping-and-method-sorting","category":"section","text":"","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"The type_morespecific functions are used for imposing a partial order on functions in method tables (from most-to-least specific). Specificity is strict; if a is more specific than b, then a does not equal b and b is not more specific than a.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"If a is a strict subtype of b, then it is automatically considered more specific. From there, type_morespecific employs some less formal rules. For example, subtype is sensitive to the number of arguments, but type_morespecific may not be. In particular, Tuple{Int,AbstractFloat} is more specific than Tuple{Integer}, even though it is not a subtype.  (Of Tuple{Int,AbstractFloat} and Tuple{Integer,Float64}, neither is more specific than the other.)  Likewise, Tuple{Int,Vararg{Int}} is not a subtype of Tuple{Integer}, but it is considered more specific. However, morespecific does get a bonus for length: in particular, Tuple{Int,Int} is more specific than Tuple{Int,Vararg{Int}}.","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"If you're debugging how methods get sorted, it can be convenient to define the function:","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"type_morespecific(a, b) = ccall(:jl_type_morespecific, Cint, (Any,Any), a, b)","page":"More about types"},{"title":"More about types","location":"devdocs/types.html","category":"page","text":"which allows you to test whether tuple type a is more specific than tuple type b.","page":"More about types"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html#Handling-Operating-System-Variation","category":"section","text":"","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"When writing cross-platform applications or libraries, it is often necessary to allow for differences between operating systems. The variable Sys.KERNEL can be used to handle such cases. There are several functions in the Sys module intended to make this easier, such as isunix, islinux, isapple, isbsd, isfreebsd, and iswindows. These may be used as follows:","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"if Sys.iswindows()\n    windows_specific_thing(a)\nend","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"Note that islinux, isapple, and isfreebsd are mutually exclusive subsets of isunix. Additionally, there is a macro @static which makes it possible to use these functions to conditionally hide invalid code, as demonstrated in the following examples.","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"Simple blocks:","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"ccall((@static Sys.iswindows() ? :_fopen : :fopen), ...)","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"Complex blocks:","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"@static if Sys.islinux()\n    linux_specific_thing(a)\nelseif Sys.isapple()\n    apple_specific_thing(a)\nelse\n    generic_thing(a)\nend","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"When nesting conditionals, the @static must be repeated for each level (parentheses optional, but recommended for readability):","page":"Handling Operating System Variation"},{"title":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html","category":"page","text":"@static Sys.iswindows() ? :a : (@static Sys.isapple() ? :b : :c)","page":"Handling Operating System Variation"},{"title":"SIMD Support","location":"base/simd-types.html#SIMD-Support","category":"section","text":"","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"Type VecElement{T} is intended for building libraries of SIMD operations. Practical use of it requires using llvmcall. The type is defined as:","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"struct VecElement{T}\n    value::T\nend","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"It has a special compilation rule: a homogeneous tuple of VecElement{T} maps to an LLVM vector type when T is a primitive bits type.","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"At -O3, the compiler might automatically vectorize operations on such tuples. For example, the following program, when compiled with julia -O3 generates two SIMD addition instructions (addps) on x86 systems:","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"const m128 = NTuple{4,VecElement{Float32}}\n\nfunction add(a::m128, b::m128)\n    (VecElement(a[1].value+b[1].value),\n     VecElement(a[2].value+b[2].value),\n     VecElement(a[3].value+b[3].value),\n     VecElement(a[4].value+b[4].value))\nend\n\ntriple(c::m128) = add(add(c,c),c)\n\ncode_native(triple,(m128,))","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"However, since the automatic vectorization cannot be relied upon, future use will mostly be via libraries that use llvmcall.","page":"SIMD Support"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html#gdb-debugging-tips","category":"section","text":"","page":"gdb debugging tips"},{"title":"Displaying Julia variables","location":"devdocs/debuggingtips.html#Displaying-Julia-variables","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Within gdb, any jl_value_t* object obj can be displayed using","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) call jl_(obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The object will be displayed in the julia session, not in the gdb session. This is a useful way to discover the types and values of objects being manipulated by Julia's C code.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Similarly, if you're debugging some of Julia's internals (e.g., compiler.jl), you can print obj using","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"ccall(:jl_, Cvoid, (Any,), obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"This is a good way to circumvent problems that arise from the order in which julia's output streams are initialized.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia's flisp interpreter uses value_t objects; these can be displayed with call fl_print(fl_ctx, ios_stdout, obj).","page":"gdb debugging tips"},{"title":"Useful Julia variables for Inspecting","location":"devdocs/debuggingtips.html#Useful-Julia-variables-for-Inspecting","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"While the addresses of many variables, like singletons, can be useful to print for many failures, there are a number of additional variables (see julia.h for a complete list) that are even more useful.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(when in jl_apply_generic) mfunc and jl_uncompress_ast(mfunc->def, mfunc->code) :: for figuring out a bit about the call-stack\njl_lineno and jl_filename :: for figuring out what line in a test to go start debugging from (or figure out how far into a file has been parsed)\n$1 :: not really a variable, but still a useful shorthand for referring to the result of the last gdb command (such as print)\njl_options :: sometimes useful, since it lists all of the command line options that were successfully parsed\njl_uv_stderr :: because who doesn't like to be able to interact with stdio","page":"gdb debugging tips"},{"title":"Useful Julia functions for Inspecting those variables","location":"devdocs/debuggingtips.html#Useful-Julia-functions-for-Inspecting-those-variables","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"jl_gdblookup($rip) :: For looking up the current function and line. (use $eip on i686 platforms)\njlbacktrace() :: For dumping the current Julia backtrace stack to stderr. Only usable after record_backtrace() has been called.\njl_dump_llvm_value(Value*) :: For invoking Value->dump() in gdb, where it doesn't work natively. For example, f->linfo->functionObject, f->linfo->specFunctionObject, and to_function(f->linfo).\nType->dump() :: only works in lldb. Note: add something like ;1 to prevent lldb from printing its prompt over the output\njl_eval_string(\"expr\") :: for invoking side-effects to modify the current state or to lookup symbols\njl_typeof(jl_value_t*) :: for extracting the type tag of a Julia value (in gdb, call macro define jl_typeof jl_typeof first, or pick something short like ty for the first arg to define a shorthand)","page":"gdb debugging tips"},{"title":"Inserting breakpoints for inspection from gdb","location":"devdocs/debuggingtips.html#Inserting-breakpoints-for-inspection-from-gdb","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"In your gdb session, set a breakpoint in jl_breakpoint like so:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_breakpoint","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then within your Julia code, insert a call to jl_breakpoint by adding","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"ccall(:jl_breakpoint, Cvoid, (Any,), obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"where obj can be any variable or tuple you want to be accessible in the breakpoint.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"It's particularly helpful to back up to the jl_apply frame, from which you can display the arguments to a function using, e.g.,","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) call jl_(args[0])","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Another useful frame is to_function(jl_method_instance_t *li, bool cstyle). The jl_method_instance_t* argument is a struct with a reference to the final AST sent into the compiler. However, the AST at this point will usually be compressed; to view the AST, call jl_uncompress_ast and then pass the result to jl_:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"#2  0x00007ffff7928bf7 in to_function (li=0x2812060, cstyle=false) at codegen.cpp:584\n584          abort();\n(gdb) p jl_(jl_uncompress_ast(li, li->ast))","page":"gdb debugging tips"},{"title":"Inserting breakpoints upon certain conditions","location":"devdocs/debuggingtips.html#Inserting-breakpoints-upon-certain-conditions","category":"section","text":"","page":"gdb debugging tips"},{"title":"Loading a particular file","location":"devdocs/debuggingtips.html#Loading-a-particular-file","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Let's say the file is sysimg.jl:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_load if strcmp(fname, \"sysimg.jl\")==0","page":"gdb debugging tips"},{"title":"Calling a particular method","location":"devdocs/debuggingtips.html#Calling-a-particular-method","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_apply_generic if strcmp((char*)(jl_symbol_name)(jl_gf_mtable(F)->name), \"method_to_break\")==0","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Since this function is used for every call, you will make everything 1000x slower if you do this.","page":"gdb debugging tips"},{"title":"Dealing with signals","location":"devdocs/debuggingtips.html#Dealing-with-signals","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia requires a few signals to function properly. The profiler uses SIGUSR2 for sampling and the garbage collector uses SIGSEGV for threads synchronization. If you are debugging some code that uses the profiler or multiple threads, you may want to let the debugger ignore these signals since they can be triggered very often during normal operations. The command to do this in GDB is (replace SIGSEGV with SIGUSR2 or other signals you want to ignore):","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) handle SIGSEGV noprint nostop pass","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The corresponding LLDB command is (after the process is started):","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(lldb) pro hand -p true -s false -n false SIGSEGV","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"If you are debugging a segfault with threaded code, you can set a breakpoint on jl_critical_error (sigdie_handler should also work on Linux and BSD) in order to only catch the actual segfault rather than the GC synchronization points.","page":"gdb debugging tips"},{"title":"Debugging during Julia's build process (bootstrap)","location":"devdocs/debuggingtips.html#Debugging-during-Julia's-build-process-(bootstrap)","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Errors that occur during make need special handling. Julia is built in two stages, constructing sys0 and sys.ji. To see what commands are running at the time of failure, use make VERBOSE=1.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"At the time of this writing, you can debug build errors during the sys0 phase from the base directory using:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys0 sysimg.jl","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"You might need to delete all the files in usr/lib/julia/ to get this to work.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"You can debug the sys.ji phase using:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys -J ../usr/lib/julia/sys0.ji sysimg.jl","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"By default, any errors will cause Julia to exit, even under gdb. To catch an error \"in the act\", set a breakpoint in jl_error (there are several other useful spots, for specific kinds of failures, including: jl_too_few_args, jl_too_many_args, and jl_throw).","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Once an error is caught, a useful technique is to walk up the stack and examine the function by inspecting the related call to jl_apply. To take a real-world example:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Breakpoint 1, jl_throw (e=0x7ffdf42de400) at task.c:802\n802 {\n(gdb) p jl_(e)\nErrorException(\"auto_unbox: unable to determine argument type\")\n$2 = void\n(gdb) bt 10\n#0  jl_throw (e=0x7ffdf42de400) at task.c:802\n#1  0x00007ffff65412fe in jl_error (str=0x7ffde56be000 <_j_str267> \"auto_unbox:\n   unable to determine argument type\")\n   at builtins.c:39\n#2  0x00007ffde56bd01a in julia_convert_16886 ()\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n...","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The most recent jl_apply is at frame #3, so we can go back there and look at the AST for the function julia_convert_16886. This is the uniqued name for some method of convert. f in this frame is a jl_function_t*, so we can look at the type signature, if any, from the specTypes field:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) f 3\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n1281            return f->fptr((jl_value_t*)f, args, nargs);\n(gdb) p f->linfo->specTypes\n$4 = (jl_tupletype_t *) 0x7ffdf39b1030\n(gdb) p jl_( f->linfo->specTypes )\nTuple{Type{Float32}, Float64}           # <-- type signature for julia_convert_16886","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then, we can look at the AST for this function:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p jl_( jl_uncompress_ast(f->linfo, f->linfo->ast) )\nExpr(:lambda, Array{Any, 1}[:#s29, :x], Array{Any, 1}[Array{Any, 1}[], Array{Any, 1}[Array{Any, 1}[:#s29, :Any, 0], Array{Any, 1}[:x, :Any, 0]], Array{Any, 1}[], 0], Expr(:body,\nExpr(:line, 90, :float.jl)::Any,\nExpr(:return, Expr(:call, :box, :Float32, Expr(:call, :fptrunc, :Float32, :x)::Any)::Any)::Any)::Any)::Any","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Finally, and perhaps most usefully, we can force the function to be recompiled in order to step through the codegen process. To do this, clear the cached functionObject from the jl_lamdbda_info_t*:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p f->linfo->functionObject\n$8 = (void *) 0x1289d070\n(gdb) set f->linfo->functionObject = NULL","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then, set a breakpoint somewhere useful (e.g. emit_function, emit_expr, emit_call, etc.), and run codegen:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p jl_compile(f)\n... # your breakpoint here","page":"gdb debugging tips"},{"title":"Debugging precompilation errors","location":"devdocs/debuggingtips.html#Debugging-precompilation-errors","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Module precompilation spawns a separate Julia process to precompile each module. Setting a breakpoint or catching failures in a precompile worker requires attaching a debugger to the worker. The easiest approach is to set the debugger watch for new process launches matching a given name. For example:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) attach -w -n julia-debug","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"or:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(lldb) process attach -w -n julia-debug","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then run a script/command to start precompilation. As described earlier, use conditional breakpoints in the parent process to catch specific file-loading events and narrow the debugging window. (some operating systems may require alternative approaches, such as following each fork from the parent process)","page":"gdb debugging tips"},{"title":"Mozilla's Record and Replay Framework (rr)","location":"devdocs/debuggingtips.html#Mozilla's-Record-and-Replay-Framework-(rr)","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia now works out of the box with rr, the lightweight recording and deterministic debugging framework from Mozilla. This allows you to replay the trace of an execution deterministically.  The replayed execution's address spaces, register contents, syscall data etc are exactly the same in every run.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"A recent version of rr (3.1.0 or higher) is required.","page":"gdb debugging tips"},{"title":"Reproducing concurrency bugs with rr","location":"devdocs/debuggingtips.html#Reproducing-concurrency-bugs-with-rr","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"rr simulates a single-threaded machine by default. In order to debug concurrent code you can use rr record --chaos which will cause rr to simulate between one to eight cores, chosen randomly. You might therefore want to set JULIA_NUM_THREADS=8 and rerun your code under rr until you have caught your bug.","page":"gdb debugging tips"},{"title":"LibGit2","location":"stdlib/LibGit2.html#LibGit2","category":"section","text":"","page":"LibGit2"},{"title":"LibGit2","location":"stdlib/LibGit2.html","category":"page","text":"The LibGit2 module provides bindings to libgit2, a portable C library that implements core functionality for the Git version control system. These bindings are currently used to power Julia's package manager. It is expected that this module will eventually be moved into a separate package.","page":"LibGit2"},{"title":"Functionality","location":"stdlib/LibGit2.html#Functionality","category":"section","text":"","page":"LibGit2"},{"title":"LibGit2","location":"stdlib/LibGit2.html","category":"page","text":"Some of this documentation assumes some prior knowledge of the libgit2 API. For more information on some of the objects and methods referenced here, consult the upstream libgit2 API reference.","page":"LibGit2"},{"title":"LibGit2","location":"stdlib/LibGit2.html","category":"page","text":"LibGit2.Buffer\nLibGit2.CheckoutOptions\nLibGit2.CloneOptions\nLibGit2.DescribeOptions\nLibGit2.DescribeFormatOptions\nLibGit2.DiffDelta\nLibGit2.DiffFile\nLibGit2.DiffOptionsStruct\nLibGit2.FetchHead\nLibGit2.FetchOptions\nLibGit2.GitAnnotated\nLibGit2.GitBlame\nLibGit2.GitBlob\nLibGit2.GitCommit\nLibGit2.GitConfig\nLibGit2.GitHash\nLibGit2.GitObject\nLibGit2.GitRemote\nLibGit2.GitRemoteAnon\nLibGit2.GitRepo\nLibGit2.GitRepoExt\nLibGit2.GitRevWalker\nLibGit2.GitShortHash\nLibGit2.GitSignature\nLibGit2.GitStatus\nLibGit2.GitTag\nLibGit2.GitTree\nLibGit2.IndexEntry\nLibGit2.IndexTime\nLibGit2.BlameOptions\nLibGit2.MergeOptions\nLibGit2.ProxyOptions\nLibGit2.PushOptions\nLibGit2.RebaseOperation\nLibGit2.RebaseOptions\nLibGit2.RemoteCallbacks\nLibGit2.SignatureStruct\nLibGit2.StatusEntry\nLibGit2.StatusOptions\nLibGit2.StrArrayStruct\nLibGit2.TimeStruct\nLibGit2.addfile\nLibGit2.add!\nLibGit2.add_fetch!\nLibGit2.add_push!\nLibGit2.addblob!\nLibGit2.author\nLibGit2.authors\nLibGit2.branch\nLibGit2.branch!\nLibGit2.checkout!\nLibGit2.clone\nLibGit2.commit\nLibGit2.committer\nLibGit2.count\nLibGit2.counthunks\nLibGit2.create_branch\nLibGit2.credentials_callback\nLibGit2.credentials_cb\nLibGit2.default_signature\nLibGit2.delete_branch\nLibGit2.diff_files\nLibGit2.entryid\nLibGit2.entrytype\nLibGit2.fetch\nLibGit2.fetchheads\nLibGit2.fetch_refspecs\nLibGit2.fetchhead_foreach_cb\nLibGit2.merge_base\nLibGit2.merge!(::LibGit2.GitRepo; ::Any...)\nLibGit2.merge!(::LibGit2.GitRepo, ::Vector{LibGit2.GitAnnotated}; ::LibGit2.MergeOptions, ::LibGit2.CheckoutOptions)\nLibGit2.merge!(::LibGit2.GitRepo, ::Vector{LibGit2.GitAnnotated}, ::Bool; ::LibGit2.MergeOptions, ::LibGit2.CheckoutOptions)\nLibGit2.ffmerge!\nLibGit2.fullname\nLibGit2.features\nLibGit2.filename\nLibGit2.filemode\nLibGit2.gitdir\nLibGit2.git_url\nLibGit2.@githash_str\nLibGit2.head\nLibGit2.head!\nLibGit2.head_oid\nLibGit2.headname\nLibGit2.init\nLibGit2.is_ancestor_of\nLibGit2.isbinary\nLibGit2.iscommit\nLibGit2.isdiff\nLibGit2.isdirty\nLibGit2.isorphan\nLibGit2.isset\nLibGit2.iszero\nLibGit2.lookup_branch\nLibGit2.map\nLibGit2.mirror_callback\nLibGit2.mirror_cb\nLibGit2.message\nLibGit2.merge_analysis\nLibGit2.name\nLibGit2.need_update\nLibGit2.objtype\nLibGit2.path\nLibGit2.peel\nLibGit2.posixpath\nLibGit2.push\nLibGit2.push!(::LibGit2.GitRevWalker, ::LibGit2.GitHash)\nLibGit2.push_head!\nLibGit2.push_refspecs\nLibGit2.raw\nLibGit2.read_tree!\nLibGit2.rebase!\nLibGit2.ref_list\nLibGit2.reftype\nLibGit2.remotes\nLibGit2.remove!\nLibGit2.reset\nLibGit2.reset!\nLibGit2.restore\nLibGit2.revcount\nLibGit2.set_remote_url\nLibGit2.shortname\nLibGit2.snapshot\nLibGit2.split_cfg_entry\nLibGit2.status\nLibGit2.stage\nLibGit2.tag_create\nLibGit2.tag_delete\nLibGit2.tag_list\nLibGit2.target\nLibGit2.toggle\nLibGit2.transact\nLibGit2.treewalk\nLibGit2.upstream\nLibGit2.update!\nLibGit2.url\nLibGit2.version\nLibGit2.with\nLibGit2.with_warn\nLibGit2.workdir\nLibGit2.GitObject(::LibGit2.GitTreeEntry)\nLibGit2.UserPasswordCredential\nLibGit2.SSHCredential\nLibGit2.isfilled\nLibGit2.CachedCredentials\nLibGit2.CredentialPayload\nLibGit2.approve\nLibGit2.reject\nLibGit2.Consts.GIT_CONFIG","page":"LibGit2"},{"title":"LibGit2.Buffer","location":"stdlib/LibGit2.html#LibGit2.Buffer","category":"type","text":"LibGit2.Buffer\n\nA data buffer for exporting data from libgit2. Matches the git_buf struct.\n\nWhen fetching data from LibGit2, a typical usage would look like:\n\nbuf_ref = Ref(Buffer())\n@check ccall(..., (Ptr{Buffer},), buf_ref)\n# operation on buf_ref\nfree(buf_ref)\n\nIn particular, note that LibGit2.free should be called afterward on the Ref object.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.CheckoutOptions","location":"stdlib/LibGit2.html#LibGit2.CheckoutOptions","category":"type","text":"LibGit2.CheckoutOptions\n\nMatches the git_checkout_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncheckout_strategy: determine how to handle conflicts and whether to force the  checkout/recreate missing files.\ndisable_filters: if nonzero, do not apply filters like CLRF (to convert file newlines between UNIX and DOS).\ndir_mode: read/write/access mode for any directories involved in the checkout. Default is 0755.\nfile_mode: read/write/access mode for any files involved in the checkout.  Default is 0755 or 0644, depending on the blob.\nfile_open_flags: bitflags used to open any files during the checkout.\nnotify_flags: Flags for what sort of conflicts the user should be notified about.\nnotify_cb: An optional callback function to notify the user if a checkout conflict occurs.  If this function returns a non-zero value, the checkout will be cancelled.\nnotify_payload: Payload for the notify callback function.\nprogress_cb: An optional callback function to display checkout progress.\nprogress_payload: Payload for the progress callback.\npaths: If not empty, describes which paths to search during the checkout.  If empty, the checkout will occur over all files in the repository.\nbaseline: Expected content of the workdir, captured in a (pointer to a)  GitTree. Defaults to the state of the tree at HEAD.\nbaseline_index: Expected content of the workdir, captured in a (pointer to a)  GitIndex. Defaults to the state of the index at HEAD.\ntarget_directory: If not empty, checkout to this directory instead of the workdir.\nancestor_label: In case of conflicts, the name of the common ancestor side.\nour_label: In case of conflicts, the name of \"our\" side.\ntheir_label: In case of conflicts, the name of \"their\" side.\nperfdata_cb: An optional callback function to display performance data.\nperfdata_payload: Payload for the performance callback.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.CloneOptions","location":"stdlib/LibGit2.html#LibGit2.CloneOptions","category":"type","text":"LibGit2.CloneOptions\n\nMatches the git_clone_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncheckout_opts: The options for performing the checkout of the remote as part of the clone.\nfetch_opts: The options for performing the pre-checkout fetch of the remote as part of the clone.\nbare: If 0, clone the full remote repository. If non-zero, perform a bare clone, in which  there is no local copy of the source files in the repository and the gitdir and workdir  are the same.\nlocalclone: Flag whether to clone a local object database or do a fetch. The default is to let git decide.  It will not use the git-aware transport for a local clone, but will use it for URLs which begin with file://.\ncheckout_branch: The name of the branch to checkout. If an empty string, the default branch of the  remote will be checked out.\nrepository_cb: An optional callback which will be used to create the new repository into which  the clone is made.\nrepository_cb_payload: The payload for the repository callback.\nremote_cb: An optional callback used to create the GitRemote before making the clone from it.\nremote_cb_payload: The payload for the remote callback.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.DescribeOptions","location":"stdlib/LibGit2.html#LibGit2.DescribeOptions","category":"type","text":"LibGit2.DescribeOptions\n\nMatches the git_describe_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nmax_candidates_tags: consider this many most recent tags in refs/tags to describe a commit.  Defaults to 10 (so that the 10 most recent tags would be examined to see if they describe a commit).\ndescribe_strategy: whether to consider all entries in refs/tags (equivalent to git-describe --tags)  or all entries in refs/ (equivalent to git-describe --all). The default is to only show annotated tags.  If Consts.DESCRIBE_TAGS is passed, all tags, annotated or not, will be considered.  If Consts.DESCRIBE_ALL is passed, any ref in refs/ will be considered.\npattern: only consider tags which match pattern. Supports glob expansion.\nonly_follow_first_parent: when finding the distance from a matching reference to the described  object, only consider the distance from the first parent.\nshow_commit_oid_as_fallback: if no matching reference can be found which describes a commit, show the  commit's GitHash instead of throwing an error (the default behavior).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.DescribeFormatOptions","location":"stdlib/LibGit2.html#LibGit2.DescribeFormatOptions","category":"type","text":"LibGit2.DescribeFormatOptions\n\nMatches the git_describe_format_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nabbreviated_size: lower bound on the size of the abbreviated GitHash to use, defaulting to 7.\nalways_use_long_format: set to 1 to use the long format for strings even if a short format can be used.\ndirty_suffix: if set, this will be appended to the end of the description string if the workdir is dirty.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.DiffDelta","location":"stdlib/LibGit2.html#LibGit2.DiffDelta","category":"type","text":"LibGit2.DiffDelta\n\nDescription of changes to one entry. Matches the git_diff_delta struct.\n\nThe fields represent:\n\nstatus: One of Consts.DELTA_STATUS, indicating whether the file has been added/modified/deleted.\nflags: Flags for the delta and the objects on each side. Determines whether to treat the file(s)  as binary/text, whether they exist on each side of the diff, and whether the object ids are known  to be correct.\nsimilarity: Used to indicate if a file has been renamed or copied.\nnfiles: The number of files in the delta (for instance, if the delta  was run on a submodule commit id, it may contain more than one file).\nold_file: A DiffFile containing information about the file(s) before the changes.\nnew_file: A DiffFile containing information about the file(s) after the changes.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.DiffFile","location":"stdlib/LibGit2.html#LibGit2.DiffFile","category":"type","text":"LibGit2.DiffFile\n\nDescription of one side of a delta. Matches the git_diff_file struct.\n\nThe fields represent:\n\nid: the GitHash of the item in the diff. If the item is empty on this  side of the diff (for instance, if the diff is of the removal of a file), this will  be GitHash(0).\npath: a NULL terminated path to the item relative to the working directory of the repository.\nsize: the size of the item in bytes.\nflags: a combination of the git_diff_flag_t  flags. The ith bit of this integer sets the ith flag.\nmode: the stat mode for the item.\nid_abbrev: only present in LibGit2 versions newer than or equal to 0.25.0.  The length of the id field when converted using string. Usually equal to OID_HEXSZ (40).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.DiffOptionsStruct","location":"stdlib/LibGit2.html#LibGit2.DiffOptionsStruct","category":"type","text":"LibGit2.DiffOptionsStruct\n\nMatches the git_diff_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: flags controlling which files will appear in the diff. Defaults to DIFF_NORMAL.\nignore_submodules: whether to look at files in submodules or not. Defaults to SUBMODULE_IGNORE_UNSPECIFIED, which means the submodule's configuration will control  whether it appears in the diff or not.\npathspec: path to files to include in the diff. Default is to use all files in the repository.\nnotify_cb: optional callback which will notify the user of changes to the diff as file deltas are  added to it.\nprogress_cb: optional callback which will display diff progress. Only relevant on libgit2 versions  at least as new as 0.24.0.\npayload: the payload to pass to notify_cb and progress_cb.\ncontext_lines: the number of unchanged lines used to define the edges of a hunk.  This is also the number of lines which will be shown before/after a hunk to provide  context. Default is 3.\ninterhunk_lines: the maximum number of unchanged lines between two separate  hunks allowed before the hunks will be combined. Default is 0.\nid_abbrev: sets the length of the abbreviated GitHash to print.  Default is 7.\nmax_size: the maximum file size of a blob. Above this size, it will be treated  as a binary blob. The default is 512 MB.\nold_prefix: the virtual file directory in which to place old files on one side  of the diff. Default is \"a\".\nnew_prefix: the virtual file directory in which to place new files on one side  of the diff. Default is \"b\".\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.FetchHead","location":"stdlib/LibGit2.html#LibGit2.FetchHead","category":"type","text":"LibGit2.FetchHead\n\nContains the information about HEAD during a fetch, including the name and URL of the branch fetched from, the oid of the HEAD, and whether the fetched HEAD has been merged locally.\n\nThe fields represent:\n\nname: The name in the local reference database of the fetch head, for example,  \"refs/heads/master\".\nurl: The URL of the fetch head.\noid: The GitHash of the tip of the fetch head.\nismerge: Boolean flag indicating whether the changes at the  remote have been merged into the local copy yet or not. If true, the local  copy is up to date with the remote fetch head.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.FetchOptions","location":"stdlib/LibGit2.html#LibGit2.FetchOptions","category":"type","text":"LibGit2.FetchOptions\n\nMatches the git_fetch_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncallbacks: remote callbacks to use during the fetch.\nprune: whether to perform a prune after the fetch or not. The default is to  use the setting from the GitConfig.\nupdate_fetchhead: whether to update the FetchHead after the fetch.  The default is to perform the update, which is the normal git behavior.\ndownload_tags: whether to download tags present at the remote or not. The default  is to request the tags for objects which are being downloaded anyway from the server.\nproxy_opts: options for connecting to the remote through a proxy. See ProxyOptions.  Only present on libgit2 versions newer than or equal to 0.25.0.\ncustom_headers: any extra headers needed for the fetch. Only present on libgit2 versions  newer than or equal to 0.24.0.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitAnnotated","location":"stdlib/LibGit2.html#LibGit2.GitAnnotated","category":"type","text":"GitAnnotated(repo::GitRepo, commit_id::GitHash)\nGitAnnotated(repo::GitRepo, ref::GitReference)\nGitAnnotated(repo::GitRepo, fh::FetchHead)\nGitAnnotated(repo::GitRepo, committish::AbstractString)\n\nAn annotated git commit carries with it information about how it was looked up and why, so that rebase or merge operations have more information about the context of the commit. Conflict files contain information about the source/target branches in the merge which are conflicting, for instance. An annotated commit can refer to the tip of a remote branch, for instance when a FetchHead is passed, or to a branch head described using GitReference.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitBlame","location":"stdlib/LibGit2.html#LibGit2.GitBlame","category":"type","text":"GitBlame(repo::GitRepo, path::AbstractString; options::BlameOptions=BlameOptions())\n\nConstruct a GitBlame object for the file at path, using change information gleaned from the history of repo. The GitBlame object records who changed which chunks of the file when, and how. options controls how to separate the contents of the file and which commits to probe - see BlameOptions for more information.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitBlob","location":"stdlib/LibGit2.html#LibGit2.GitBlob","category":"type","text":"GitBlob(repo::GitRepo, hash::AbstractGitHash)\nGitBlob(repo::GitRepo, spec::AbstractString)\n\nReturn a GitBlob object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitCommit","location":"stdlib/LibGit2.html#LibGit2.GitCommit","category":"type","text":"GitCommit(repo::GitRepo, hash::AbstractGitHash)\nGitCommit(repo::GitRepo, spec::AbstractString)\n\nReturn a GitCommit object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitConfig","location":"stdlib/LibGit2.html#LibGit2.GitConfig","category":"type","text":"GitConfig(path::AbstractString, level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_APP, force::Bool=false)\n\nCreate a new GitConfig by loading configuration information from the file at path. See addfile for more information about the level, repo and force options.\n\n\n\n\n\nGitConfig(repo::GitRepo)\n\nGet the stored configuration for the git repository repo. If repo does not have a specific configuration file set, the default git configuration will be used.\n\n\n\n\n\nGitConfig(level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_DEFAULT)\n\nGet the default git configuration by loading the global and system configuration files into a prioritized configuration. This can be used to access default configuration options outside a specific git repository.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitHash","location":"stdlib/LibGit2.html#LibGit2.GitHash","category":"type","text":"GitHash\n\nA git object identifier, based on the sha-1 hash. It is a 20 byte string (40 hex digits) used to identify a GitObject in a repository.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitObject","location":"stdlib/LibGit2.html#LibGit2.GitObject","category":"type","text":"GitObject(repo::GitRepo, hash::AbstractGitHash)\nGitObject(repo::GitRepo, spec::AbstractString)\n\nReturn the specified object (GitCommit, GitBlob, GitTree or GitTag) from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitRemote","location":"stdlib/LibGit2.html#LibGit2.GitRemote","category":"type","text":"GitRemote(repo::GitRepo, rmt_name::AbstractString, rmt_url::AbstractString) -> GitRemote\n\nLook up a remote git repository using its name and URL. Uses the default fetch refspec.\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nremote = LibGit2.GitRemote(repo, \"upstream\", repo_url)\n\n\n\n\n\nGitRemote(repo::GitRepo, rmt_name::AbstractString, rmt_url::AbstractString, fetch_spec::AbstractString) -> GitRemote\n\nLook up a remote git repository using the repository's name and URL, as well as specifications for how to fetch from the remote (e.g. which remote branch to fetch from).\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nrefspec = \"+refs/heads/mybranch:refs/remotes/origin/mybranch\"\nremote = LibGit2.GitRemote(repo, \"upstream\", repo_url, refspec)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitRemoteAnon","location":"stdlib/LibGit2.html#LibGit2.GitRemoteAnon","category":"function","text":"GitRemoteAnon(repo::GitRepo, url::AbstractString) -> GitRemote\n\nLook up a remote git repository using only its URL, not its name.\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nremote = LibGit2.GitRemoteAnon(repo, repo_url)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitRepo","location":"stdlib/LibGit2.html#LibGit2.GitRepo","category":"type","text":"LibGit2.GitRepo(path::AbstractString)\n\nOpen a git repository at path.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitRepoExt","location":"stdlib/LibGit2.html#LibGit2.GitRepoExt","category":"function","text":"LibGit2.GitRepoExt(path::AbstractString, flags::Cuint = Cuint(Consts.REPOSITORY_OPEN_DEFAULT))\n\nOpen a git repository at path with extended controls (for instance, if the current user must be a member of a special access group to read path).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitRevWalker","location":"stdlib/LibGit2.html#LibGit2.GitRevWalker","category":"type","text":"GitRevWalker(repo::GitRepo)\n\nA GitRevWalker walks through the revisions (i.e. commits) of a git repository repo. It is a collection of the commits in the repository, and supports iteration and calls to LibGit2.map and LibGit2.count (for instance, LibGit2.count could be used to determine what percentage of commits in a repository were made by a certain author).\n\ncnt = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.count((oid,repo)->(oid == commit_oid1), walker, oid=commit_oid1, by=LibGit2.Consts.SORT_TIME)\nend\n\nHere, LibGit2.count finds the number of commits along the walk with a certain GitHash. Since the GitHash is unique to a commit, cnt will be 1.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitShortHash","location":"stdlib/LibGit2.html#LibGit2.GitShortHash","category":"type","text":"GitShortHash(hash::GitHash, len::Integer)\n\nA shortened git object identifier, which can be used to identify a git object when it is unique, consisting of the initial len hexadecimal digits of hash (the remaining digits are ignored).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitSignature","location":"stdlib/LibGit2.html#LibGit2.GitSignature","category":"type","text":"LibGit2.GitSignature\n\nThis is a Julia wrapper around a pointer to a git_signature object.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitStatus","location":"stdlib/LibGit2.html#LibGit2.GitStatus","category":"type","text":"LibGit2.GitStatus(repo::GitRepo; status_opts=StatusOptions())\n\nCollect information about the status of each file in the git repository repo (e.g. is the file modified, staged, etc.). status_opts can be used to set various options, for instance whether or not to look at untracked files or whether to include submodules or not. See StatusOptions for more information.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitTag","location":"stdlib/LibGit2.html#LibGit2.GitTag","category":"type","text":"GitTag(repo::GitRepo, hash::AbstractGitHash)\nGitTag(repo::GitRepo, spec::AbstractString)\n\nReturn a GitTag object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitTree","location":"stdlib/LibGit2.html#LibGit2.GitTree","category":"type","text":"GitTree(repo::GitRepo, hash::AbstractGitHash)\nGitTree(repo::GitRepo, spec::AbstractString)\n\nReturn a GitTree object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.IndexEntry","location":"stdlib/LibGit2.html#LibGit2.IndexEntry","category":"type","text":"LibGit2.IndexEntry\n\nIn-memory representation of a file entry in the index. Matches the git_index_entry struct.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.IndexTime","location":"stdlib/LibGit2.html#LibGit2.IndexTime","category":"type","text":"LibGit2.IndexTime\n\nMatches the git_index_time struct.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.BlameOptions","location":"stdlib/LibGit2.html#LibGit2.BlameOptions","category":"type","text":"LibGit2.BlameOptions\n\nMatches the git_blame_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: one of Consts.BLAME_NORMAL or Consts.BLAME_FIRST_PARENT (the other blame flags  are not yet implemented by libgit2).\nmin_match_characters: the minimum number of alphanumeric characters which much change in a commit in order for the change to be associated with that commit. The default is 20. Only takes effect if one of the Consts.BLAME_*_COPIES flags are used, which libgit2 does not implement yet.\nnewest_commit: the GitHash of the newest commit from which to look at changes.\noldest_commit: the GitHash of the oldest commit from which to look at changes.\nmin_line: the first line of the file from which to starting blaming. The default is 1.\nmax_line: the last line of the file to which to blame. The default is 0, meaning the last line of the file.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.MergeOptions","location":"stdlib/LibGit2.html#LibGit2.MergeOptions","category":"type","text":"LibGit2.MergeOptions\n\nMatches the git_merge_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: an enum for flags describing merge behavior.  Defined in git_merge_flag_t.  The corresponding Julia enum is GIT_MERGE and has values:\nMERGE_FIND_RENAMES: detect if a file has been renamed between the common ancestor and the \"ours\" or \"theirs\" side of the merge. Allows merges where a file has been renamed.\nMERGE_FAIL_ON_CONFLICT: exit immediately if a conflict is found rather than trying to resolve it.\nMERGE_SKIP_REUC: do not write the REUC extension on the index resulting from the merge.\nMERGE_NO_RECURSIVE: if the commits being merged have multiple merge bases, use the first one, rather than trying to recursively merge the bases.\nrename_threshold: how similar two files must to consider one a rename of the other. This is an integer that sets the percentage similarity. The default is 50.\ntarget_limit: the maximum number of files to compare with to look for renames. The default is 200.\nmetric: optional custom function to use to determine the similarity between two files for rename detection.\nrecursion_limit: the upper limit on the number of merges of common ancestors to perform to try to build a new virtual merge base for the merge. The default is no limit. This field is only present on libgit2 versions newer than 0.24.0.\ndefault_driver: the merge driver to use if both sides have changed. This field is only present on libgit2 versions newer than 0.25.0.\nfile_favor: how to handle conflicting file contents for the text driver.\nMERGE_FILE_FAVOR_NORMAL: if both sides of the merge have changes to a section,  make a note of the conflict in the index which git checkout will use to create  a merge file, which the user can then reference to resolve the conflicts. This is  the default.\nMERGE_FILE_FAVOR_OURS: if both sides of the merge have changes to a section,  use the version in the \"ours\" side of the merge in the index.\nMERGE_FILE_FAVOR_THEIRS: if both sides of the merge have changes to a section,  use the version in the \"theirs\" side of the merge in the index.\nMERGE_FILE_FAVOR_UNION: if both sides of the merge have changes to a section,  include each unique line from both sides in the file which is put into the index.\nfile_flags: guidelines for merging files.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.ProxyOptions","location":"stdlib/LibGit2.html#LibGit2.ProxyOptions","category":"type","text":"LibGit2.ProxyOptions\n\nOptions for connecting through a proxy.\n\nMatches the git_proxy_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nproxytype: an enum for the type of proxy to use.  Defined in git_proxy_t.  The corresponding Julia enum is GIT_PROXY and has values:\nPROXY_NONE: do not attempt the connection through a proxy.\nPROXY_AUTO: attempt to figure out the proxy configuration from the git configuration.\nPROXY_SPECIFIED: connect using the URL given in the url field of this struct.\nDefault is to auto-detect the proxy type.\nurl: the URL of the proxy.\ncredential_cb: a pointer to a callback function which will be called if the remote requires authentication to connect.\ncertificate_cb: a pointer to a callback function which will be called if certificate verification fails. This lets the user decide whether or not to keep connecting. If the function returns 1, connecting will be allowed. If it returns 0, the connection will not be allowed. A negative value can be used to return errors.\npayload: the payload to be provided to the two callback functions.\n\nExamples\n\njulia> fo = LibGit2.FetchOptions(\n           proxy_opts = LibGit2.ProxyOptions(url = Cstring(\"https://my_proxy_url.com\")))\n\njulia> fetch(remote, \"master\", options=fo)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.PushOptions","location":"stdlib/LibGit2.html#LibGit2.PushOptions","category":"type","text":"LibGit2.PushOptions\n\nMatches the git_push_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nparallelism: if a pack file must be created, this variable sets the number of worker  threads which will be spawned by the packbuilder. If 0, the packbuilder will auto-set  the number of threads to use. The default is 1.\ncallbacks: the callbacks (e.g. for authentication with the remote) to use for the push.\nproxy_opts: only relevant if the LibGit2 version is greater than or equal to 0.25.0.  Sets options for using a proxy to communicate with a remote. See ProxyOptions  for more information.\ncustom_headers: only relevant if the LibGit2 version is greater than or equal to 0.24.0.  Extra headers needed for the push operation.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.RebaseOperation","location":"stdlib/LibGit2.html#LibGit2.RebaseOperation","category":"type","text":"LibGit2.RebaseOperation\n\nDescribes a single instruction/operation to be performed during the rebase. Matches the git_rebase_operation struct.\n\nThe fields represent:\n\noptype: the type of rebase operation currently being performed. The options are:\nREBASE_OPERATION_PICK: cherry-pick the commit in question.\nREBASE_OPERATION_REWORD: cherry-pick the commit in question, but rewrite its message using the prompt.\nREBASE_OPERATION_EDIT: cherry-pick the commit in question, but allow the user to edit the commit's contents and its message.\nREBASE_OPERATION_SQUASH: squash the commit in question into the previous commit. The commit messages of the two commits will be merged.\nREBASE_OPERATION_FIXUP: squash the commit in question into the previous commit. Only the commit message of the previous commit will be used.\nREBASE_OPERATION_EXEC: do not cherry-pick a commit. Run a command and continue if the command exits successfully.\nid: the GitHash of the commit being worked on during this rebase step.\nexec: in case REBASE_OPERATION_EXEC is used, the command to run during this step (for instance, running the test suite after each commit).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.RebaseOptions","location":"stdlib/LibGit2.html#LibGit2.RebaseOptions","category":"type","text":"LibGit2.RebaseOptions\n\nMatches the git_rebase_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nquiet: inform other git clients helping with/working on the rebase that the rebase should be done \"quietly\". Used for interoperability. The default is 1.\ninmemory: start an in-memory rebase. Callers working on the rebase can go through its steps and commit any changes, but cannot rewind HEAD or update the repository. The workdir will not be modified. Only present on libgit2 versions newer than or equal to 0.24.0.\nrewrite_notes_ref: name of the reference to notes to use to rewrite the commit notes as the rebase is finished.\nmerge_opts: merge options controlling how the trees will be merged at each rebase step.  Only present on libgit2 versions newer than or equal to 0.24.0.\ncheckout_opts: checkout options for writing files when initializing the rebase, stepping through it, and aborting it. See CheckoutOptions for more information.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.RemoteCallbacks","location":"stdlib/LibGit2.html#LibGit2.RemoteCallbacks","category":"type","text":"LibGit2.RemoteCallbacks\n\nCallback settings. Matches the git_remote_callbacks struct.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.SignatureStruct","location":"stdlib/LibGit2.html#LibGit2.SignatureStruct","category":"type","text":"LibGit2.SignatureStruct\n\nAn action signature (e.g. for committers, taggers, etc). Matches the git_signature struct.\n\nThe fields represent:\n\nname: The full name of the committer or author of the commit.\nemail: The email at which the committer/author can be contacted.\nwhen: a TimeStruct indicating when the commit was  authored/committed into the repository.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.StatusEntry","location":"stdlib/LibGit2.html#LibGit2.StatusEntry","category":"type","text":"LibGit2.StatusEntry\n\nProviding the differences between the file as it exists in HEAD and the index, and providing the differences between the index and the working directory. Matches the git_status_entry struct.\n\nThe fields represent:\n\nstatus: contains the status flags for the file, indicating if it is current, or has been changed in some way in the index or work tree.\nhead_to_index: a pointer to a DiffDelta which encapsulates the difference(s) between the file as it exists in HEAD and in the index.\nindex_to_workdir: a pointer to a DiffDelta which encapsulates the difference(s) between the file as it exists in the index and in the workdir.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.StatusOptions","location":"stdlib/LibGit2.html#LibGit2.StatusOptions","category":"type","text":"LibGit2.StatusOptions\n\nOptions to control how git_status_foreach_ext() will issue callbacks. Matches the git_status_opt_t struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nshow: a flag for which files to examine and in which order. The default is Consts.STATUS_SHOW_INDEX_AND_WORKDIR.\nflags: flags for controlling any callbacks used in a status call.\npathspec: an array of paths to use for path-matching. The behavior of the path-matching will vary depending on the values of show and flags.\nThe baseline is the tree to be used for comparison to the working directory and index; defaults to HEAD.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.StrArrayStruct","location":"stdlib/LibGit2.html#LibGit2.StrArrayStruct","category":"type","text":"LibGit2.StrArrayStruct\n\nA LibGit2 representation of an array of strings. Matches the git_strarray struct.\n\nWhen fetching data from LibGit2, a typical usage would look like:\n\nsa_ref = Ref(StrArrayStruct())\n@check ccall(..., (Ptr{StrArrayStruct},), sa_ref)\nres = convert(Vector{String}, sa_ref[])\nfree(sa_ref)\n\nIn particular, note that LibGit2.free should be called afterward on the Ref object.\n\nConversely, when passing a vector of strings to LibGit2, it is generally simplest to rely on implicit conversion:\n\nstrs = String[...]\n@check ccall(..., (Ptr{StrArrayStruct},), strs)\n\nNote that no call to free is required as the data is allocated by Julia.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.TimeStruct","location":"stdlib/LibGit2.html#LibGit2.TimeStruct","category":"type","text":"LibGit2.TimeStruct\n\nTime in a signature. Matches the git_time struct.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.addfile","location":"stdlib/LibGit2.html#LibGit2.addfile","category":"function","text":"addfile(cfg::GitConfig, path::AbstractString,\n        level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_APP,\n        repo::Union{GitRepo, Nothing} = nothing,\n        force::Bool=false)\n\nAdd an existing git configuration file located at path to the current GitConfig cfg. If the file does not exist, it will be created.\n\nlevel sets the git configuration priority level and is determined by\n\nConsts.GIT_CONFIG.\n\nrepo is an optional repository to allow parsing of conditional includes.\nIf force is false and a configuration for the given priority level already exists,\n\naddfile will error. If force is true, the existing configuration will be replaced by the one in the file at path.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.add!","location":"stdlib/LibGit2.html#LibGit2.add!","category":"function","text":"add!(repo::GitRepo, files::AbstractString...; flags::Cuint = Consts.INDEX_ADD_DEFAULT)\nadd!(idx::GitIndex, files::AbstractString...; flags::Cuint = Consts.INDEX_ADD_DEFAULT)\n\nAdd all the files with paths specified by files to the index idx (or the index of the repo). If the file already exists, the index entry will be updated. If the file does not exist already, it will be newly added into the index. files may contain glob patterns which will be expanded and any matching files will be added (unless INDEX_ADD_DISABLE_PATHSPEC_MATCH is set, see below). If a file has been ignored (in .gitignore or in the config), it will not be added, unless it is already being tracked in the index, in which case it will be updated. The keyword argument flags is a set of bit-flags which control the behavior with respect to ignored files:\n\nConsts.INDEX_ADD_DEFAULT - default, described above.\nConsts.INDEX_ADD_FORCE - disregard the existing ignore rules and force addition of the file to the index even if it is already ignored.\nConsts.INDEX_ADD_CHECK_PATHSPEC - cannot be used at the same time as INDEX_ADD_FORCE. Check that each file in files which exists on disk is not in the ignore list. If one of the files is ignored, the function will return EINVALIDSPEC.\nConsts.INDEX_ADD_DISABLE_PATHSPEC_MATCH - turn off glob matching, and only add files to the index which exactly match the paths specified in files.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.add_fetch!","location":"stdlib/LibGit2.html#LibGit2.add_fetch!","category":"function","text":"add_fetch!(repo::GitRepo, rmt::GitRemote, fetch_spec::String)\n\nAdd a fetch refspec for the specified rmt. This refspec will contain information about which branch(es) to fetch from.\n\nExamples\n\njulia> LibGit2.add_fetch!(repo, remote, \"upstream\");\n\njulia> LibGit2.fetch_refspecs(remote)\nString[\"+refs/heads/*:refs/remotes/upstream/*\"]\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.add_push!","location":"stdlib/LibGit2.html#LibGit2.add_push!","category":"function","text":"add_push!(repo::GitRepo, rmt::GitRemote, push_spec::String)\n\nAdd a push refspec for the specified rmt. This refspec will contain information about which branch(es) to push to.\n\nExamples\n\njulia> LibGit2.add_push!(repo, remote, \"refs/heads/master\");\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, branch);\n\njulia> LibGit2.push_refspecs(remote)\nString[\"refs/heads/master\"]\n\nnote: Note\nYou may need to close and reopen the GitRemote in question after updating its push refspecs in order for the change to take effect and for calls to push to work.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.addblob!","location":"stdlib/LibGit2.html#LibGit2.addblob!","category":"function","text":"LibGit2.addblob!(repo::GitRepo, path::AbstractString)\n\nRead the file at path and adds it to the object database of repo as a loose blob. Return the GitHash of the resulting blob.\n\nExamples\n\nhash_str = string(commit_oid)\nblob_file = joinpath(repo_path, \".git\", \"objects\", hash_str[1:2], hash_str[3:end])\nid = LibGit2.addblob!(repo, blob_file)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.author","location":"stdlib/LibGit2.html#LibGit2.author","category":"function","text":"author(c::GitCommit)\n\nReturn the Signature of the author of the commit c. The author is the person who made changes to the relevant file(s). See also committer.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.authors","location":"stdlib/LibGit2.html#LibGit2.authors","category":"function","text":"authors(repo::GitRepo) -> Vector{Signature}\n\nReturn all authors of commits to the repo repository.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nrepo_file = open(joinpath(repo_path, test_file), \"a\")\n\nprintln(repo_file, commit_msg)\nflush(repo_file)\nLibGit2.add!(repo, test_file)\nsig = LibGit2.Signature(\"TEST\", \"TEST@TEST.COM\", round(time(), 0), 0)\ncommit_oid1 = LibGit2.commit(repo, \"commit1\"; author=sig, committer=sig)\nprintln(repo_file, randstring(10))\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid2 = LibGit2.commit(repo, \"commit2\"; author=sig, committer=sig)\n\n# will be a Vector of [sig, sig]\nauths = LibGit2.authors(repo)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.branch","location":"stdlib/LibGit2.html#LibGit2.branch","category":"function","text":"branch(repo::GitRepo)\n\nEquivalent to git branch. Create a new branch from the current HEAD.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.branch!","location":"stdlib/LibGit2.html#LibGit2.branch!","category":"function","text":"branch!(repo::GitRepo, branch_name::AbstractString, commit::AbstractString=\"\"; kwargs...)\n\nCheckout a new git branch in the repo repository. commit is the GitHash, in string form, which will be the start of the new branch. If commit is an empty string, the current HEAD will be used.\n\nThe keyword arguments are:\n\ntrack::AbstractString=\"\": the name of the remote branch this new branch should track, if any. If empty (the default), no remote branch will be tracked.\nforce::Bool=false: if true, branch creation will be forced.\nset_head::Bool=true: if true, after the branch creation finishes the branch head will be set as the HEAD of repo.\n\nEquivalent to git checkout [-b|-B] <branch_name> [<commit>] [--track <track>].\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.branch!(repo, \"new_branch\", set_head=false)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.checkout!","location":"stdlib/LibGit2.html#LibGit2.checkout!","category":"function","text":"checkout!(repo::GitRepo, commit::AbstractString=\"\"; force::Bool=true)\n\nEquivalent to git checkout [-f] --detach <commit>. Checkout the git commit commit (a GitHash in string form) in repo. If force is true, force the checkout and discard any current changes. Note that this detaches the current HEAD.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nopen(joinpath(LibGit2.path(repo), \"file1\"), \"w\") do f\n    write(f, \"111\n\")\nend\nLibGit2.add!(repo, \"file1\")\ncommit_oid = LibGit2.commit(repo, \"add file1\")\nopen(joinpath(LibGit2.path(repo), \"file1\"), \"w\") do f\n    write(f, \"112\n\")\nend\n# would fail without the force=true\n# since there are modifications to the file\nLibGit2.checkout!(repo, string(commit_oid), force=true)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.clone","location":"stdlib/LibGit2.html#LibGit2.clone","category":"function","text":"clone(repo_url::AbstractString, repo_path::AbstractString, clone_opts::CloneOptions)\n\nClone the remote repository at repo_url (which can be a remote URL or a path on the local filesystem) to repo_path (which must be a path on the local filesystem). Options for the clone, such as whether to perform a bare clone or not, are set by CloneOptions.\n\nExamples\n\nrepo_url = \"https://github.com/JuliaLang/Example.jl\"\nrepo = LibGit2.clone(repo_url, \"/home/me/projects/Example\")\n\n\n\n\n\nclone(repo_url::AbstractString, repo_path::AbstractString; kwargs...)\n\nClone a remote repository located at repo_url to the local filesystem location repo_path.\n\nThe keyword arguments are:\n\nbranch::AbstractString=\"\": which branch of the remote to clone, if not the default repository branch (usually master).\nisbare::Bool=false: if true, clone the remote as a bare repository, which will make repo_path itself the git directory instead of repo_path/.git. This means that a working tree cannot be checked out. Plays the role of the git CLI argument --bare.\nremote_cb::Ptr{Cvoid}=C_NULL: a callback which will be used to create the remote before it is cloned. If C_NULL (the default), no attempt will be made to create the remote - it will be assumed to already exist.\ncredentials::Creds=nothing: provides credentials and/or settings when authenticating against a private repository.\ncallbacks::Callbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git clone [-b <branch>] [--bare] <repo_url> <repo_path>.\n\nExamples\n\nrepo_url = \"https://github.com/JuliaLang/Example.jl\"\nrepo1 = LibGit2.clone(repo_url, \"test_path\")\nrepo2 = LibGit2.clone(repo_url, \"test_path\", isbare=true)\njulia_url = \"https://github.com/JuliaLang/julia\"\njulia_repo = LibGit2.clone(julia_url, \"julia_path\", branch=\"release-0.6\")\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.commit","location":"stdlib/LibGit2.html#LibGit2.commit","category":"function","text":"commit(repo::GitRepo, msg::AbstractString; kwargs...) -> GitHash\n\nWrapper around git_commit_create. Create a commit in the repository repo. msg is the commit message. Return the OID of the new commit.\n\nThe keyword arguments are:\n\nrefname::AbstractString=Consts.HEAD_FILE: if not NULL, the name of the reference to update to point to the new commit. For example, \"HEAD\" will update the HEAD of the current branch. If the reference does not yet exist, it will be created.\nauthor::Signature = Signature(repo) is a Signature containing information about the person who authored the commit.\ncommitter::Signature = Signature(repo) is a Signature containing information about the person who committed the commit to the repository. Not necessarily the same as author, for instance if author emailed a patch to committer who committed it.\ntree_id::GitHash = GitHash() is a git tree to use to create the commit, showing its ancestry and relationship with any other history. tree must belong to repo.\nparent_ids::Vector{GitHash}=GitHash[] is a list of commits by GitHash to use as parent commits for the new one, and may be empty. A commit might have multiple parents if it is a merge commit, for example.\n\n\n\n\n\nLibGit2.commit(rb::GitRebase, sig::GitSignature)\n\nCommit the current patch to the rebase rb, using sig as the committer. Is silent if the commit has already been applied.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.committer","location":"stdlib/LibGit2.html#LibGit2.committer","category":"function","text":"committer(c::GitCommit)\n\nReturn the Signature of the committer of the commit c. The committer is the person who committed the changes originally authored by the author, but need not be the same as the author, for example, if the author emailed a patch to a committer who committed it.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.count","location":"stdlib/LibGit2.html#LibGit2.count","category":"function","text":"LibGit2.count(f::Function, walker::GitRevWalker; oid::GitHash=GitHash(), by::Cint=Consts.SORT_NONE, rev::Bool=false)\n\nUsing the GitRevWalker walker to \"walk\" over every commit in the repository's history, find the number of commits which return true when f is applied to them. The keyword arguments are:     * oid: The GitHash of the commit to begin the walk from. The default is to use       push_head! and therefore the HEAD commit and all its ancestors.     * by: The sorting method. The default is not to sort. Other options are to sort by       topology (LibGit2.Consts.SORT_TOPOLOGICAL), to sort forwards in time       (LibGit2.Consts.SORT_TIME, most ancient first) or to sort backwards in time       (LibGit2.Consts.SORT_REVERSE, most recent first).     * rev: Whether to reverse the sorted order (for instance, if topological sorting is used).\n\nExamples\n\ncnt = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.count((oid, repo)->(oid == commit_oid1), walker, oid=commit_oid1, by=LibGit2.Consts.SORT_TIME)\nend\n\nLibGit2.count finds the number of commits along the walk with a certain GitHash commit_oid1, starting the walk from that commit and moving forwards in time from it. Since the GitHash is unique to a commit, cnt will be 1.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.counthunks","location":"stdlib/LibGit2.html#LibGit2.counthunks","category":"function","text":"counthunks(blame::GitBlame)\n\nReturn the number of distinct \"hunks\" with a file. A hunk may contain multiple lines. A hunk is usually a piece of a file that was added/changed/removed together, for example, a function added to a source file or an inner loop that was optimized out of that function later.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.create_branch","location":"stdlib/LibGit2.html#LibGit2.create_branch","category":"function","text":"LibGit2.create_branch(repo::GitRepo, bname::AbstractString, commit_obj::GitCommit; force::Bool=false)\n\nCreate a new branch in the repository repo with name bname, which points to commit commit_obj (which has to be part of repo). If force is true, overwrite an existing branch named bname if it exists. If force is false and a branch already exists named bname, this function will throw an error.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.credentials_callback","location":"stdlib/LibGit2.html#LibGit2.credentials_callback","category":"function","text":"credential_callback(...) -> Cint\n\nA LibGit2 credential callback function which provides different credential acquisition functionality w.r.t. a connection protocol. The payload_ptr is required to contain a LibGit2.CredentialPayload object which will keep track of state and settings.\n\nThe allowed_types contains a bitmask of LibGit2.Consts.GIT_CREDTYPE values specifying which authentication methods should be attempted.\n\nCredential authentication is done in the following order (if supported):\n\nSSH agent\nSSH private/public key pair\nUsername/password plain text\n\nIf a user is presented with a credential prompt they can abort the prompt by typing ^D (pressing the control key together with the d key).\n\nNote: Due to the specifics of the libgit2 authentication procedure, when authentication fails, this function is called again without any indication whether authentication was successful or not. To avoid an infinite loop from repeatedly using the same faulty credentials, we will keep track of state using the payload.\n\nFor addition details see the LibGit2 guide on authenticating against a server.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.credentials_cb","location":"stdlib/LibGit2.html#LibGit2.credentials_cb","category":"function","text":"C function pointer for credentials_callback\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.default_signature","location":"stdlib/LibGit2.html#LibGit2.default_signature","category":"function","text":"Return signature object. Free it after use.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.delete_branch","location":"stdlib/LibGit2.html#LibGit2.delete_branch","category":"function","text":"LibGit2.delete_branch(branch::GitReference)\n\nDelete the branch pointed to by branch.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.diff_files","location":"stdlib/LibGit2.html#LibGit2.diff_files","category":"function","text":"diff_files(repo::GitRepo, branch1::AbstractString, branch2::AbstractString; kwarg...) -> Vector{AbstractString}\n\nShow which files have changed in the git repository repo between branches branch1 and branch2.\n\nThe keyword argument is:\n\nfilter::Set{Consts.DELTA_STATUS}=Set([Consts.DELTA_ADDED, Consts.DELTA_MODIFIED, Consts.DELTA_DELETED])), and it sets options for the diff. The default is to show files added, modified, or deleted.\n\nReturn only the names of the files which have changed, not their contents.\n\nExamples\n\nLibGit2.branch!(repo, \"branch/a\")\nLibGit2.branch!(repo, \"branch/b\")\n# add a file to repo\nopen(joinpath(LibGit2.path(repo),\"file\"),\"w\") do f\n    write(f, \"hello repo\n\")\nend\nLibGit2.add!(repo, \"file\")\nLibGit2.commit(repo, \"add file\")\n# returns [\"file\"]\nfilt = Set([LibGit2.Consts.DELTA_ADDED])\nfiles = LibGit2.diff_files(repo, \"branch/a\", \"branch/b\", filter=filt)\n# returns [] because existing files weren't modified\nfilt = Set([LibGit2.Consts.DELTA_MODIFIED])\nfiles = LibGit2.diff_files(repo, \"branch/a\", \"branch/b\", filter=filt)\n\nEquivalent to git diff --name-only --diff-filter=<filter> <branch1> <branch2>.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.entryid","location":"stdlib/LibGit2.html#LibGit2.entryid","category":"function","text":"entryid(te::GitTreeEntry)\n\nReturn the GitHash of the object to which te refers.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.entrytype","location":"stdlib/LibGit2.html#LibGit2.entrytype","category":"function","text":"entrytype(te::GitTreeEntry)\n\nReturn the type of the object to which te refers. The result will be one of the types which objtype returns, e.g. a GitTree or GitBlob.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.fetch","location":"stdlib/LibGit2.html#LibGit2.fetch","category":"function","text":"fetch(rmt::GitRemote, refspecs; options::FetchOptions=FetchOptions(), msg=\"\")\n\nFetch from the specified rmt remote git repository, using refspecs to determine which remote branch(es) to fetch. The keyword arguments are:\n\noptions: determines the options for the fetch, e.g. whether to prune afterwards. See FetchOptions for more information.\nmsg: a message to insert into the reflogs.\n\n\n\n\n\nfetch(repo::GitRepo; kwargs...)\n\nFetches updates from an upstream of the repository repo.\n\nThe keyword arguments are:\n\nremote::AbstractString=\"origin\": which remote, specified by name, of repo to fetch from. If this is empty, the URL will be used to construct an anonymous remote.\nremoteurl::AbstractString=\"\": the URL of remote. If not specified, will be assumed based on the given name of remote.\nrefspecs=AbstractString[]: determines properties of the fetch.\ncredentials=nothing: provides credentials and/or settings when authenticating against a private remote.\ncallbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git fetch [<remoteurl>|<repo>] [<refspecs>].\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.fetchheads","location":"stdlib/LibGit2.html#LibGit2.fetchheads","category":"function","text":"fetchheads(repo::GitRepo) -> Vector{FetchHead}\n\nReturn the list of all the fetch heads for repo, each represented as a FetchHead, including their names, URLs, and merge statuses.\n\nExamples\n\njulia> fetch_heads = LibGit2.fetchheads(repo);\n\njulia> fetch_heads[1].name\n\"refs/heads/master\"\n\njulia> fetch_heads[1].ismerge\ntrue\n\njulia> fetch_heads[2].name\n\"refs/heads/test_branch\"\n\njulia> fetch_heads[2].ismerge\nfalse\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.fetch_refspecs","location":"stdlib/LibGit2.html#LibGit2.fetch_refspecs","category":"function","text":"fetch_refspecs(rmt::GitRemote) -> Vector{String}\n\nGet the fetch refspecs for the specified rmt. These refspecs contain information about which branch(es) to fetch from.\n\nExamples\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.add_fetch!(repo, remote, \"upstream\");\n\njulia> LibGit2.fetch_refspecs(remote)\nString[\"+refs/heads/*:refs/remotes/upstream/*\"]\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.fetchhead_foreach_cb","location":"stdlib/LibGit2.html#LibGit2.fetchhead_foreach_cb","category":"function","text":"C function pointer for fetchhead_foreach_callback\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.merge_base","location":"stdlib/LibGit2.html#LibGit2.merge_base","category":"function","text":"merge_base(repo::GitRepo, one::AbstractString, two::AbstractString) -> GitHash\n\nFind a merge base (a common ancestor) between the commits one and two. one and two may both be in string form. Return the GitHash of the merge base.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.merge!","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo}","category":"method","text":"merge!(repo::GitRepo; kwargs...) -> Bool\n\nPerform a git merge on the repository repo, merging commits with diverging history into the current branch. Return true if the merge succeeded, false if not.\n\nThe keyword arguments are:\n\ncommittish::AbstractString=\"\": Merge the named commit(s) in committish.\nbranch::AbstractString=\"\": Merge the branch branch and all its commits since it diverged from the current branch.\nfastforward::Bool=false: If fastforward is true, only merge if the merge is a fast-forward (the current branch head is an ancestor of the commits to be merged), otherwise refuse to merge and return false. This is equivalent to the git CLI option --ff-only.\nmerge_opts::MergeOptions=MergeOptions(): merge_opts specifies options for the merge, such as merge strategy in case of conflicts.\ncheckout_opts::CheckoutOptions=CheckoutOptions(): checkout_opts specifies options for the checkout step.\n\nEquivalent to git merge [--ff-only] [<committish> | <branch>].\n\nnote: Note\nIf you specify a branch, this must be done in reference format, since the string will be turned into a GitReference. For example, if you wanted to merge branch branch_a, you would call merge!(repo, branch=\"refs/heads/branch_a\").\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.merge!","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo, Vector{LibGit2.GitAnnotated}}","category":"method","text":"merge!(repo::GitRepo, anns::Vector{GitAnnotated}; kwargs...) -> Bool\n\nMerge changes from the annotated commits (captured as GitAnnotated objects) anns into the HEAD of the repository repo. The keyword arguments are:\n\nmerge_opts::MergeOptions = MergeOptions(): options for how to perform the merge, including whether fastforwarding is allowed. See MergeOptions for more information.\ncheckout_opts::CheckoutOptions = CheckoutOptions(): options for how to perform the checkout. See CheckoutOptions for more information.\n\nanns may refer to remote or local branch heads. Return true if the merge is successful, otherwise return false (for instance, if no merge is possible because the branches have no common ancestor).\n\nExamples\n\nupst_ann = LibGit2.GitAnnotated(repo, \"branch/a\")\n\n# merge the branch in\nLibGit2.merge!(repo, [upst_ann])\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.merge!","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo, Vector{LibGit2.GitAnnotated}, Bool}","category":"method","text":"merge!(repo::GitRepo, anns::Vector{GitAnnotated}, fastforward::Bool; kwargs...) -> Bool\n\nMerge changes from the annotated commits (captured as GitAnnotated objects) anns into the HEAD of the repository repo. If fastforward is true, only a fastforward merge is allowed. In this case, if conflicts occur, the merge will fail. Otherwise, if fastforward is false, the merge may produce a conflict file which the user will need to resolve.\n\nThe keyword arguments are:\n\nmerge_opts::MergeOptions = MergeOptions(): options for how to perform the merge, including whether fastforwarding is allowed. See MergeOptions for more information.\ncheckout_opts::CheckoutOptions = CheckoutOptions(): options for how to perform the checkout. See CheckoutOptions for more information.\n\nanns may refer to remote or local branch heads. Return true if the merge is successful, otherwise return false (for instance, if no merge is possible because the branches have no common ancestor).\n\nExamples\n\nupst_ann_1 = LibGit2.GitAnnotated(repo, \"branch/a\")\n\n# merge the branch in, fastforward\nLibGit2.merge!(repo, [upst_ann_1], true)\n\n# merge conflicts!\nupst_ann_2 = LibGit2.GitAnnotated(repo, \"branch/b\")\n# merge the branch in, try to fastforward\nLibGit2.merge!(repo, [upst_ann_2], true) # will return false\nLibGit2.merge!(repo, [upst_ann_2], false) # will return true\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.ffmerge!","location":"stdlib/LibGit2.html#LibGit2.ffmerge!","category":"function","text":"ffmerge!(repo::GitRepo, ann::GitAnnotated)\n\nFastforward merge changes into current HEAD. This is only possible if the commit referred to by ann is descended from the current HEAD (e.g. if pulling changes from a remote branch which is simply ahead of the local branch tip).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.fullname","location":"stdlib/LibGit2.html#LibGit2.fullname","category":"function","text":"LibGit2.fullname(ref::GitReference)\n\nReturn the name of the reference pointed to by the symbolic reference ref. If ref is not a symbolic reference, return an empty string.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.features","location":"stdlib/LibGit2.html#LibGit2.features","category":"function","text":"features()\n\nReturn a list of git features the current version of libgit2 supports, such as threading or using HTTPS or SSH.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.filename","location":"stdlib/LibGit2.html#LibGit2.filename","category":"function","text":"filename(te::GitTreeEntry)\n\nReturn the filename of the object on disk to which te refers.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.filemode","location":"stdlib/LibGit2.html#LibGit2.filemode","category":"function","text":"filemode(te::GitTreeEntry) -> Cint\n\nReturn the UNIX filemode of the object on disk to which te refers as an integer.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.gitdir","location":"stdlib/LibGit2.html#LibGit2.gitdir","category":"function","text":"LibGit2.gitdir(repo::GitRepo)\n\nReturn the location of the \"git\" files of repo:\n\nfor normal repositories, this is the location of the .git folder.\nfor bare repositories, this is the location of the repository itself.\n\nSee also workdir, path.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.git_url","location":"stdlib/LibGit2.html#LibGit2.git_url","category":"function","text":"LibGit2.git_url(; kwargs...) -> String\n\nCreate a string based upon the URL components provided. When the scheme keyword is not provided the URL produced will use the alternative scp-like syntax.\n\nKeywords\n\nscheme::AbstractString=\"\": the URL scheme which identifies the protocol to be used. For HTTP use \"http\", SSH use \"ssh\", etc. When scheme is not provided the output format will be \"ssh\" but using the scp-like syntax.\nusername::AbstractString=\"\": the username to use in the output if provided.\npassword::AbstractString=\"\": the password to use in the output if provided.\nhost::AbstractString=\"\": the hostname to use in the output. A hostname is required to be specified.\nport::Union{AbstractString,Integer}=\"\": the port number to use in the output if provided. Cannot be specified when using the scp-like syntax.\npath::AbstractString=\"\": the path to use in the output if provided.\n\nwarning: Warning\nAvoid using passwords in URLs. Unlike the credential objects, Julia is not able to securely zero or destroy the sensitive data after use and the password may remain in memory; possibly to be exposed by an uninitialized memory.\n\nExamples\n\njulia> LibGit2.git_url(username=\"git\", host=\"github.com\", path=\"JuliaLang/julia.git\")\n\"git@github.com:JuliaLang/julia.git\"\n\njulia> LibGit2.git_url(scheme=\"https\", host=\"github.com\", path=\"/JuliaLang/julia.git\")\n\"https://github.com/JuliaLang/julia.git\"\n\njulia> LibGit2.git_url(scheme=\"ssh\", username=\"git\", host=\"github.com\", port=2222, path=\"JuliaLang/julia.git\")\n\"ssh://git@github.com:2222/JuliaLang/julia.git\"\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.@githash_str","location":"stdlib/LibGit2.html#LibGit2.@githash_str","category":"macro","text":"@githash_str -> AbstractGitHash\n\nConstruct a git hash object from the given string, returning a GitShortHash if the string is shorter than 40 hexadecimal digits, otherwise a GitHash.\n\nExamples\n\njulia> LibGit2.githash\"d114feb74ce633\"\nGitShortHash(\"d114feb74ce633\")\n\njulia> LibGit2.githash\"d114feb74ce63307afe878a5228ad014e0289a85\"\nGitHash(\"d114feb74ce63307afe878a5228ad014e0289a85\")\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.head","location":"stdlib/LibGit2.html#LibGit2.head","category":"function","text":"LibGit2.head(repo::GitRepo) -> GitReference\n\nReturn a GitReference to the current HEAD of repo.\n\n\n\n\n\nhead(pkg::AbstractString) -> String\n\nReturn current HEAD GitHash of the pkg repo as a string.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.head!","location":"stdlib/LibGit2.html#LibGit2.head!","category":"function","text":"LibGit2.head!(repo::GitRepo, ref::GitReference) -> GitReference\n\nSet the HEAD of repo to the object pointed to by ref.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.head_oid","location":"stdlib/LibGit2.html#LibGit2.head_oid","category":"function","text":"LibGit2.head_oid(repo::GitRepo) -> GitHash\n\nLookup the object id of the current HEAD of git repository repo.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.headname","location":"stdlib/LibGit2.html#LibGit2.headname","category":"function","text":"LibGit2.headname(repo::GitRepo)\n\nLookup the name of the current HEAD of git repository repo. If repo is currently detached, return the name of the HEAD it's detached from.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.init","location":"stdlib/LibGit2.html#LibGit2.init","category":"function","text":"LibGit2.init(path::AbstractString, bare::Bool=false) -> GitRepo\n\nOpen a new git repository at path. If bare is false, the working tree will be created in path/.git. If bare is true, no working directory will be created.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.is_ancestor_of","location":"stdlib/LibGit2.html#LibGit2.is_ancestor_of","category":"function","text":"is_ancestor_of(a::AbstractString, b::AbstractString, repo::GitRepo) -> Bool\n\nReturn true if a, a GitHash in string form, is an ancestor of b, a GitHash in string form.\n\nExamples\n\njulia> repo = GitRepo(repo_path);\n\njulia> LibGit2.add!(repo, test_file1);\n\njulia> commit_oid1 = LibGit2.commit(repo, \"commit1\");\n\njulia> LibGit2.add!(repo, test_file2);\n\njulia> commit_oid2 = LibGit2.commit(repo, \"commit2\");\n\njulia> LibGit2.is_ancestor_of(string(commit_oid1), string(commit_oid2), repo)\ntrue\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isbinary","location":"stdlib/LibGit2.html#LibGit2.isbinary","category":"function","text":"isbinary(blob::GitBlob) -> Bool\n\nUse a heuristic to guess if a file is binary: searching for NULL bytes and looking for a reasonable ratio of printable to non-printable characters among the first 8000 bytes.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.iscommit","location":"stdlib/LibGit2.html#LibGit2.iscommit","category":"function","text":"iscommit(id::AbstractString, repo::GitRepo) -> Bool\n\nCheck if commit id (which is a GitHash in string form) is in the repository.\n\nExamples\n\njulia> repo = GitRepo(repo_path);\n\njulia> LibGit2.add!(repo, test_file);\n\njulia> commit_oid = LibGit2.commit(repo, \"add test_file\");\n\njulia> LibGit2.iscommit(string(commit_oid), repo)\ntrue\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isdiff","location":"stdlib/LibGit2.html#LibGit2.isdiff","category":"function","text":"LibGit2.isdiff(repo::GitRepo, treeish::AbstractString, pathspecs::AbstractString=\"\"; cached::Bool=false)\n\nChecks if there are any differences between the tree specified by treeish and the tracked files in the working tree (if cached=false) or the index (if cached=true). pathspecs are the specifications for options for the diff.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.isdiff(repo, \"HEAD\") # should be false\nopen(joinpath(repo_path, new_file), \"a\") do f\n    println(f, \"here's my cool new file\")\nend\nLibGit2.isdiff(repo, \"HEAD\") # now true\n\nEquivalent to git diff-index <treeish> [-- <pathspecs>].\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isdirty","location":"stdlib/LibGit2.html#LibGit2.isdirty","category":"function","text":"LibGit2.isdirty(repo::GitRepo, pathspecs::AbstractString=\"\"; cached::Bool=false) -> Bool\n\nCheck if there have been any changes to tracked files in the working tree (if cached=false) or the index (if cached=true). pathspecs are the specifications for options for the diff.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.isdirty(repo) # should be false\nopen(joinpath(repo_path, new_file), \"a\") do f\n    println(f, \"here's my cool new file\")\nend\nLibGit2.isdirty(repo) # now true\nLibGit2.isdirty(repo, new_file) # now true\n\nEquivalent to git diff-index HEAD [-- <pathspecs>].\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isorphan","location":"stdlib/LibGit2.html#LibGit2.isorphan","category":"function","text":"LibGit2.isorphan(repo::GitRepo)\n\nCheck if the current branch is an \"orphan\" branch, i.e. has no commits. The first commit to this branch will have no parents.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isset","location":"stdlib/LibGit2.html#LibGit2.isset","category":"function","text":"isset(val::Integer, flag::Integer)\n\nTest whether the bits of val indexed by flag are set (1) or unset (0).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.iszero","location":"stdlib/LibGit2.html#LibGit2.iszero","category":"function","text":"iszero(id::GitHash) -> Bool\n\nDetermine whether all hexadecimal digits of the given GitHash are zero.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.lookup_branch","location":"stdlib/LibGit2.html#LibGit2.lookup_branch","category":"function","text":"lookup_branch(repo::GitRepo, branch_name::AbstractString, remote::Bool=false) -> Union{GitReference, Nothing}\n\nDetermine if the branch specified by branch_name exists in the repository repo. If remote is true, repo is assumed to be a remote git repository. Otherwise, it is part of the local filesystem.\n\nReturn either a GitReference to the requested branch if it exists, or nothing if not.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.map","location":"stdlib/LibGit2.html#LibGit2.map","category":"function","text":"LibGit2.map(f::Function, walker::GitRevWalker; oid::GitHash=GitHash(), range::AbstractString=\"\", by::Cint=Consts.SORT_NONE, rev::Bool=false)\n\nUsing the GitRevWalker walker to \"walk\" over every commit in the repository's history, apply f to each commit in the walk. The keyword arguments are:     * oid: The GitHash of the commit to begin the walk from. The default is to use       push_head! and therefore the HEAD commit and all its ancestors.     * range: A range of GitHashs in the format oid1..oid2. f will be       applied to all commits between the two.     * by: The sorting method. The default is not to sort. Other options are to sort by       topology (LibGit2.Consts.SORT_TOPOLOGICAL), to sort forwards in time       (LibGit2.Consts.SORT_TIME, most ancient first) or to sort backwards in time       (LibGit2.Consts.SORT_REVERSE, most recent first).     * rev: Whether to reverse the sorted order (for instance, if topological sorting is used).\n\nExamples\n\noids = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.map((oid, repo)->string(oid), walker, by=LibGit2.Consts.SORT_TIME)\nend\n\nHere, LibGit2.map visits each commit using the GitRevWalker and finds its GitHash.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.mirror_callback","location":"stdlib/LibGit2.html#LibGit2.mirror_callback","category":"function","text":"Mirror callback function\n\nFunction sets +refs/*:refs/* refspecs and mirror flag for remote reference.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.mirror_cb","location":"stdlib/LibGit2.html#LibGit2.mirror_cb","category":"function","text":"C function pointer for mirror_callback\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.message","location":"stdlib/LibGit2.html#LibGit2.message","category":"function","text":"message(c::GitCommit, raw::Bool=false)\n\nReturn the commit message describing the changes made in commit c. If raw is false, return a slightly \"cleaned up\" message (which has any leading newlines removed). If raw is true, the message is not stripped of any such newlines.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.merge_analysis","location":"stdlib/LibGit2.html#LibGit2.merge_analysis","category":"function","text":"merge_analysis(repo::GitRepo, anns::Vector{GitAnnotated}) -> analysis, preference\n\nRun analysis on the branches pointed to by the annotated branch tips anns and determine under what circumstances they can be merged. For instance, if anns[1] is simply an ancestor of ann[2], then merge_analysis will report that a fast-forward merge is possible.\n\nReturn two outputs, analysis and preference. analysis has several possible values:     * MERGE_ANALYSIS_NONE: it is not possible to merge the elements of anns.     * MERGE_ANALYSIS_NORMAL: a regular merge, when HEAD and the commits that the       user wishes to merge have all diverged from a common ancestor. In this case the       changes have to be resolved and conflicts may occur.     * MERGE_ANALYSIS_UP_TO_DATE: all the input commits the user wishes to merge can       be reached from HEAD, so no merge needs to be performed.     * MERGE_ANALYSIS_FASTFORWARD: the input commit is a descendant of HEAD and so no       merge needs to be performed - instead, the user can simply checkout the       input commit(s).     * MERGE_ANALYSIS_UNBORN: the HEAD of the repository refers to a commit which does not       exist. It is not possible to merge, but it may be possible to checkout the input       commits. preference also has several possible values:     * MERGE_PREFERENCE_NONE: the user has no preference.     * MERGE_PREFERENCE_NO_FASTFORWARD: do not allow any fast-forward merges.     * MERGE_PREFERENCE_FASTFORWARD_ONLY: allow only fast-forward merges and no       other type (which may introduce conflicts). preference can be controlled through the repository or global git configuration.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.name","location":"stdlib/LibGit2.html#LibGit2.name","category":"function","text":"LibGit2.name(ref::GitReference)\n\nReturn the full name of ref.\n\n\n\n\n\nname(rmt::GitRemote)\n\nGet the name of a remote repository, for instance \"origin\". If the remote is anonymous (see GitRemoteAnon) the name will be an empty string \"\".\n\nExamples\n\njulia> repo_url = \"https://github.com/JuliaLang/Example.jl\";\n\njulia> repo = LibGit2.clone(cache_repo, \"test_directory\");\n\njulia> remote = LibGit2.GitRemote(repo, \"origin\", repo_url);\n\njulia> name(remote)\n\"origin\"\n\n\n\n\n\nLibGit2.name(tag::GitTag)\n\nThe name of tag (e.g. \"v0.5\").\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.need_update","location":"stdlib/LibGit2.html#LibGit2.need_update","category":"function","text":"need_update(repo::GitRepo)\n\nEquivalent to git update-index. Return true if repo needs updating.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.objtype","location":"stdlib/LibGit2.html#LibGit2.objtype","category":"function","text":"objtype(obj_type::Consts.OBJECT)\n\nReturn the type corresponding to the enum value.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.path","location":"stdlib/LibGit2.html#LibGit2.path","category":"function","text":"LibGit2.path(repo::GitRepo)\n\nReturn the base file path of the repository repo.\n\nfor normal repositories, this will typically be the parent directory of the \".git\" directory (note: this may be different than the working directory, see workdir for more details).\nfor bare repositories, this is the location of the \"git\" files.\n\nSee also gitdir, workdir.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.peel","location":"stdlib/LibGit2.html#LibGit2.peel","category":"function","text":"peel([T,] ref::GitReference)\n\nRecursively peel ref until an object of type T is obtained. If no T is provided, then ref will be peeled until an object other than a GitTag is obtained.\n\nA GitTag will be peeled to the object it references.\nA GitCommit will be peeled to a GitTree.\n\nnote: Note\nOnly annotated tags can be peeled to GitTag objects. Lightweight tags (the default) are references under refs/tags/ which point directly to GitCommit objects.\n\n\n\n\n\npeel([T,] obj::GitObject)\n\nRecursively peel obj until an object of type T is obtained. If no T is provided, then obj will be peeled until the type changes.\n\nA GitTag will be peeled to the object it references.\nA GitCommit will be peeled to a GitTree.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.posixpath","location":"stdlib/LibGit2.html#LibGit2.posixpath","category":"function","text":"LibGit2.posixpath(path)\n\nStandardise the path string path to use POSIX separators.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.push","location":"stdlib/LibGit2.html#LibGit2.push","category":"function","text":"push(rmt::GitRemote, refspecs; force::Bool=false, options::PushOptions=PushOptions())\n\nPush to the specified rmt remote git repository, using refspecs to determine which remote branch(es) to push to. The keyword arguments are:\n\nforce: if true, a force-push will occur, disregarding conflicts.\noptions: determines the options for the push, e.g. which proxy headers to use. See PushOptions for more information.\n\nnote: Note\nYou can add information about the push refspecs in two other ways: by setting an option in the repository's GitConfig (with push.default as the key) or by calling add_push!. Otherwise you will need to explicitly specify a push refspec in the call to push for it to have any effect, like so: LibGit2.push(repo, refspecs=[\"refs/heads/master\"]).\n\n\n\n\n\npush(repo::GitRepo; kwargs...)\n\nPushes updates to an upstream of repo.\n\nThe keyword arguments are:\n\nremote::AbstractString=\"origin\": the name of the upstream remote to push to.\nremoteurl::AbstractString=\"\": the URL of remote.\nrefspecs=AbstractString[]: determines properties of the push.\nforce::Bool=false: determines if the push will be a force push,  overwriting the remote branch.\ncredentials=nothing: provides credentials and/or settings when authenticating against  a private remote.\ncallbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git push [<remoteurl>|<repo>] [<refspecs>].\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.push!","location":"stdlib/LibGit2.html#LibGit2.push!-Tuple{LibGit2.GitRevWalker, LibGit2.GitHash}","category":"method","text":"LibGit2.push!(w::GitRevWalker, cid::GitHash)\n\nStart the GitRevWalker walker at commit cid. This function can be used to apply a function to all commits since a certain year, by passing the first commit of that year as cid and then passing the resulting w to LibGit2.map.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.push_head!","location":"stdlib/LibGit2.html#LibGit2.push_head!","category":"function","text":"LibGit2.push_head!(w::GitRevWalker)\n\nPush the HEAD commit and its ancestors onto the GitRevWalker w. This ensures that HEAD and all its ancestor commits will be encountered during the walk.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.push_refspecs","location":"stdlib/LibGit2.html#LibGit2.push_refspecs","category":"function","text":"push_refspecs(rmt::GitRemote) -> Vector{String}\n\nGet the push refspecs for the specified rmt. These refspecs contain information about which branch(es) to push to.\n\nExamples\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.add_push!(repo, remote, \"refs/heads/master\");\n\njulia> close(remote);\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.push_refspecs(remote)\nString[\"refs/heads/master\"]\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.raw","location":"stdlib/LibGit2.html#LibGit2.raw","category":"function","text":"raw(id::GitHash) -> Vector{UInt8}\n\nObtain the raw bytes of the GitHash as a vector of length 20.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.read_tree!","location":"stdlib/LibGit2.html#LibGit2.read_tree!","category":"function","text":"LibGit2.read_tree!(idx::GitIndex, tree::GitTree)\nLibGit2.read_tree!(idx::GitIndex, treehash::AbstractGitHash)\n\nRead the tree tree (or the tree pointed to by treehash in the repository owned by idx) into the index idx. The current index contents will be replaced.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.rebase!","location":"stdlib/LibGit2.html#LibGit2.rebase!","category":"function","text":"LibGit2.rebase!(repo::GitRepo, upstream::AbstractString=\"\", newbase::AbstractString=\"\")\n\nAttempt an automatic merge rebase of the current branch, from upstream if provided, or otherwise from the upstream tracking branch. newbase is the branch to rebase onto. By default this is upstream.\n\nIf any conflicts arise which cannot be automatically resolved, the rebase will abort, leaving the repository and working tree in its original state, and the function will throw a GitError. This is roughly equivalent to the following command line statement:\n\ngit rebase --merge [<upstream>]\nif [ -d \".git/rebase-merge\" ]; then\n    git rebase --abort\nfi\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.ref_list","location":"stdlib/LibGit2.html#LibGit2.ref_list","category":"function","text":"LibGit2.ref_list(repo::GitRepo) -> Vector{String}\n\nGet a list of all reference names in the repo repository.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.reftype","location":"stdlib/LibGit2.html#LibGit2.reftype","category":"function","text":"LibGit2.reftype(ref::GitReference) -> Cint\n\nReturn a Cint corresponding to the type of ref:\n\n0 if the reference is invalid\n1 if the reference is an object id\n2 if the reference is symbolic\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.remotes","location":"stdlib/LibGit2.html#LibGit2.remotes","category":"function","text":"LibGit2.remotes(repo::GitRepo)\n\nReturn a vector of the names of the remotes of repo.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.remove!","location":"stdlib/LibGit2.html#LibGit2.remove!","category":"function","text":"remove!(repo::GitRepo, files::AbstractString...)\nremove!(idx::GitIndex, files::AbstractString...)\n\nRemove all the files with paths specified by files in the index idx (or the index of the repo).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.reset","location":"stdlib/LibGit2.html#LibGit2.reset","category":"function","text":"reset(val::Integer, flag::Integer)\n\nUnset the bits of val indexed by flag, returning them to 0.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.reset!","location":"stdlib/LibGit2.html#LibGit2.reset!","category":"function","text":"reset!(payload, [config]) -> CredentialPayload\n\nReset the payload state back to the initial values so that it can be used again within the credential callback. If a config is provided the configuration will also be updated.\n\n\n\n\n\nUpdates some entries, determined by the pathspecs, in the index from the target commit tree.\n\n\n\n\n\nSets the current head to the specified commit oid and optionally resets the index and working tree to match.\n\n\n\n\n\ngit reset [<committish>] [–] <pathspecs>... \n\n\n\n\n\nreset!(repo::GitRepo, id::GitHash, mode::Cint=Consts.RESET_MIXED)\n\nReset the repository repo to its state at id, using one of three modes set by mode:\n\nConsts.RESET_SOFT - move HEAD to id.\nConsts.RESET_MIXED - default, move HEAD to id and reset the index to id.\nConsts.RESET_HARD - move HEAD to id, reset the index to id, and discard all working changes.\n\nExamples\n\n# fetch changes\nLibGit2.fetch(repo)\nisfile(joinpath(repo_path, our_file)) # will be false\n\n# fastforward merge the changes\nLibGit2.merge!(repo, fastforward=true)\n\n# because there was not any file locally, but there is\n# a file remotely, we need to reset the branch\nhead_oid = LibGit2.head_oid(repo)\nnew_head = LibGit2.reset!(repo, head_oid, LibGit2.Consts.RESET_HARD)\n\nIn this example, the remote which is being fetched from does have a file called our_file in its index, which is why we must reset.\n\nEquivalent to git reset [--soft | --mixed | --hard] <id>.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nhead_oid = LibGit2.head_oid(repo)\nopen(joinpath(repo_path, \"file1\"), \"w\") do f\n    write(f, \"111\n\")\nend\nLibGit2.add!(repo, \"file1\")\nmode = LibGit2.Consts.RESET_HARD\n# will discard the changes to file1\n# and unstage it\nnew_head = LibGit2.reset!(repo, head_oid, mode)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.restore","location":"stdlib/LibGit2.html#LibGit2.restore","category":"function","text":"restore(s::State, repo::GitRepo)\n\nReturn a repository repo to a previous State s, for example the HEAD of a branch before a merge attempt. s can be generated using the snapshot function.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.revcount","location":"stdlib/LibGit2.html#LibGit2.revcount","category":"function","text":"LibGit2.revcount(repo::GitRepo, commit1::AbstractString, commit2::AbstractString)\n\nList the number of revisions between commit1 and commit2 (committish OIDs in string form). Since commit1 and commit2 may be on different branches, revcount performs a \"left-right\" revision list (and count), returning a tuple of Ints - the number of left and right commits, respectively. A left (or right) commit refers to which side of a symmetric difference in a tree the commit is reachable from.\n\nEquivalent to git rev-list --left-right --count <commit1> <commit2>.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nrepo_file = open(joinpath(repo_path, test_file), \"a\")\nprintln(repo_file, \"hello world\")\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid1 = LibGit2.commit(repo, \"commit 1\")\nprintln(repo_file, \"hello world again\")\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid2 = LibGit2.commit(repo, \"commit 2\")\nLibGit2.revcount(repo, string(commit_oid1), string(commit_oid2))\n\nThis will return (-1, 0).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.set_remote_url","location":"stdlib/LibGit2.html#LibGit2.set_remote_url","category":"function","text":"set_remote_url(repo::GitRepo, remote_name, url)\nset_remote_url(repo::String, remote_name, url)\n\nSet both the fetch and push url for remote_name for the GitRepo or the git repository located at path. Typically git repos use \"origin\" as the remote name.\n\nExamples\n\nrepo_path = joinpath(tempdir(), \"Example\")\nrepo = LibGit2.init(repo_path)\nLibGit2.set_remote_url(repo, \"upstream\", \"https://github.com/JuliaLang/Example.jl\")\nLibGit2.set_remote_url(repo_path, \"upstream2\", \"https://github.com/JuliaLang/Example2.jl\")\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.shortname","location":"stdlib/LibGit2.html#LibGit2.shortname","category":"function","text":"LibGit2.shortname(ref::GitReference)\n\nReturn a shortened version of the name of ref that's \"human-readable\".\n\njulia> repo = GitRepo(path_to_repo);\n\njulia> branch_ref = LibGit2.head(repo);\n\njulia> LibGit2.name(branch_ref)\n\"refs/heads/master\"\n\njulia> LibGit2.shortname(branch_ref)\n\"master\"\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.snapshot","location":"stdlib/LibGit2.html#LibGit2.snapshot","category":"function","text":"snapshot(repo::GitRepo) -> State\n\nTake a snapshot of the current state of the repository repo, storing the current HEAD, index, and any uncommitted work. The output State can be used later during a call to restore to return the repository to the snapshotted state.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.split_cfg_entry","location":"stdlib/LibGit2.html#LibGit2.split_cfg_entry","category":"function","text":"LibGit2.split_cfg_entry(ce::LibGit2.ConfigEntry) -> Tuple{String,String,String,String}\n\nBreak the ConfigEntry up to the following pieces: section, subsection, name, and value.\n\nExamples\n\nGiven the git configuration file containing:\n\n[credential \"https://example.com\"]\n    username = me\n\nThe ConfigEntry would look like the following:\n\njulia> entry\nConfigEntry(\"credential.https://example.com.username\", \"me\")\n\njulia> LibGit2.split_cfg_entry(entry)\n(\"credential\", \"https://example.com\", \"username\", \"me\")\n\nRefer to the git config syntax documentation for more details.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.status","location":"stdlib/LibGit2.html#LibGit2.status","category":"function","text":"LibGit2.status(repo::GitRepo, path::String) -> Union{Cuint, Cvoid}\n\nLookup the status of the file at path in the git repository repo. For instance, this can be used to check if the file at path has been modified and needs to be staged and committed.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.stage","location":"stdlib/LibGit2.html#LibGit2.stage","category":"function","text":"stage(ie::IndexEntry) -> Cint\n\nGet the stage number of ie. The stage number 0 represents the current state of the working tree, but other numbers can be used in the case of a merge conflict. In such a case, the various stage numbers on an IndexEntry describe which side(s) of the conflict the current state of the file belongs to. Stage 0 is the state before the attempted merge, stage 1 is the changes which have been made locally, stages 2 and larger are for changes from other branches (for instance, in the case of a multi-branch \"octopus\" merge, stages 2, 3, and 4 might be used).\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.tag_create","location":"stdlib/LibGit2.html#LibGit2.tag_create","category":"function","text":"LibGit2.tag_create(repo::GitRepo, tag::AbstractString, commit; kwargs...)\n\nCreate a new git tag tag (e.g. \"v0.5\") in the repository repo, at the commit commit.\n\nThe keyword arguments are:\n\nmsg::AbstractString=\"\": the message for the tag.\nforce::Bool=false: if true, existing references will be overwritten.\nsig::Signature=Signature(repo): the tagger's signature.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.tag_delete","location":"stdlib/LibGit2.html#LibGit2.tag_delete","category":"function","text":"LibGit2.tag_delete(repo::GitRepo, tag::AbstractString)\n\nRemove the git tag tag from the repository repo.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.tag_list","location":"stdlib/LibGit2.html#LibGit2.tag_list","category":"function","text":"LibGit2.tag_list(repo::GitRepo) -> Vector{String}\n\nGet a list of all tags in the git repository repo.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.target","location":"stdlib/LibGit2.html#LibGit2.target","category":"function","text":"LibGit2.target(tag::GitTag)\n\nThe GitHash of the target object of tag.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.toggle","location":"stdlib/LibGit2.html#LibGit2.toggle","category":"function","text":"toggle(val::Integer, flag::Integer)\n\nFlip the bits of val indexed by flag, so that if a bit is 0 it will be 1 after the toggle, and vice-versa.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.transact","location":"stdlib/LibGit2.html#LibGit2.transact","category":"function","text":"transact(f::Function, repo::GitRepo)\n\nApply function f to the git repository repo, taking a snapshot before applying f. If an error occurs within f, repo will be returned to its snapshot state using restore. The error which occurred will be rethrown, but the state of repo will not be corrupted.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.treewalk","location":"stdlib/LibGit2.html#LibGit2.treewalk","category":"function","text":"treewalk(f, tree::GitTree, post::Bool=false)\n\nTraverse the entries in tree and its subtrees in post or pre order. Preorder means beginning at the root and then traversing the leftmost subtree (and recursively on down through that subtree's leftmost subtrees) and moving right through the subtrees. Postorder means beginning at the bottom of the leftmost subtree, traversing upwards through it, then traversing the next right subtree (again beginning at the bottom) and finally visiting the tree root last of all.\n\nThe function parameter f should have following signature:\n\n(String, GitTreeEntry) -> Cint\n\nA negative value returned from f stops the tree walk. A positive value means that the entry will be skipped if post is false.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.upstream","location":"stdlib/LibGit2.html#LibGit2.upstream","category":"function","text":"upstream(ref::GitReference) -> Union{GitReference, Nothing}\n\nDetermine if the branch containing ref has a specified upstream branch.\n\nReturn either a GitReference to the upstream branch if it exists, or nothing if the requested branch does not have an upstream counterpart.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.update!","location":"stdlib/LibGit2.html#LibGit2.update!","category":"function","text":"update!(repo::GitRepo, files::AbstractString...)\nupdate!(idx::GitIndex, files::AbstractString...)\n\nUpdate all the files with paths specified by files in the index idx (or the index of the repo). Match the state of each file in the index with the current state on disk, removing it if it has been removed on disk, or updating its entry in the object database.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.url","location":"stdlib/LibGit2.html#LibGit2.url","category":"function","text":"url(rmt::GitRemote)\n\nGet the fetch URL of a remote git repository.\n\nExamples\n\njulia> repo_url = \"https://github.com/JuliaLang/Example.jl\";\n\njulia> repo = LibGit2.init(mktempdir());\n\njulia> remote = LibGit2.GitRemote(repo, \"origin\", repo_url);\n\njulia> LibGit2.url(remote)\n\"https://github.com/JuliaLang/Example.jl\"\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.version","location":"stdlib/LibGit2.html#LibGit2.version","category":"function","text":"version() -> VersionNumber\n\nReturn the version of libgit2 in use, as a VersionNumber.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.with","location":"stdlib/LibGit2.html#LibGit2.with","category":"function","text":"with(f::Function, obj)\n\nResource management helper function. Applies f to obj, making sure to call close on obj after f successfully returns or throws an error. Ensures that allocated git resources are finalized as soon as they are no longer needed.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.with_warn","location":"stdlib/LibGit2.html#LibGit2.with_warn","category":"function","text":"with_warn(f::Function, ::Type{T}, args...)\n\nResource management helper function. Apply f to args, first constructing an instance of type T from args. Makes sure to call close on the resulting object after f successfully returns or throws an error. Ensures that allocated git resources are finalized as soon as they are no longer needed. If an error is thrown by f, a warning is shown containing the error.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.workdir","location":"stdlib/LibGit2.html#LibGit2.workdir","category":"function","text":"LibGit2.workdir(repo::GitRepo)\n\nReturn the location of the working directory of repo. This will throw an error for bare repositories.\n\nnote: Note\nThis will typically be the parent directory of gitdir(repo), but can be different in some cases: e.g. if either the core.worktree configuration variable or the GIT_WORK_TREE environment variable is set.\n\nSee also gitdir, path.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.GitObject","location":"stdlib/LibGit2.html#LibGit2.GitObject-Tuple{LibGit2.GitTreeEntry}","category":"method","text":"(::Type{T})(te::GitTreeEntry) where T<:GitObject\n\nGet the git object to which te refers and return it as its actual type (the type entrytype would show), for instance a GitBlob or GitTag.\n\nExamples\n\ntree = LibGit2.GitTree(repo, \"HEAD^{tree}\")\ntree_entry = tree[1]\nblob = LibGit2.GitBlob(tree_entry)\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.UserPasswordCredential","location":"stdlib/LibGit2.html#LibGit2.UserPasswordCredential","category":"type","text":"Credential that support only user and password parameters\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.SSHCredential","location":"stdlib/LibGit2.html#LibGit2.SSHCredential","category":"type","text":"SSH credential type\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.isfilled","location":"stdlib/LibGit2.html#LibGit2.isfilled","category":"function","text":"isfilled(cred::AbstractCredential) -> Bool\n\nVerifies that a credential is ready for use in authentication.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.CachedCredentials","location":"stdlib/LibGit2.html#LibGit2.CachedCredentials","category":"type","text":"Caches credential information for re-use\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.CredentialPayload","location":"stdlib/LibGit2.html#LibGit2.CredentialPayload","category":"type","text":"LibGit2.CredentialPayload\n\nRetains the state between multiple calls to the credential callback for the same URL. A CredentialPayload instance is expected to be reset! whenever it will be used with a different URL.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.approve","location":"stdlib/LibGit2.html#LibGit2.approve","category":"function","text":"approve(payload::CredentialPayload; shred::Bool=true) -> Nothing\n\nStore the payload credential for re-use in a future authentication. Should only be called when authentication was successful.\n\nThe shred keyword controls whether sensitive information in the payload credential field should be destroyed. Should only be set to false during testing.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.reject","location":"stdlib/LibGit2.html#LibGit2.reject","category":"function","text":"reject(payload::CredentialPayload; shred::Bool=true) -> Nothing\n\nDiscard the payload credential from begin re-used in future authentication. Should only be called when authentication was unsuccessful.\n\nThe shred keyword controls whether sensitive information in the payload credential field should be destroyed. Should only be set to false during testing.\n\n\n\n\n\n","page":"LibGit2"},{"title":"LibGit2.Consts.GIT_CONFIG","location":"stdlib/LibGit2.html#LibGit2.Consts.GIT_CONFIG","category":"type","text":"Priority level of a config file.\n\nThese priority levels correspond to the natural escalation logic (from higher to lower) when searching for config entries in git.\n\nCONFIG_LEVEL_DEFAULT - Open the global, XDG and system configuration files if any available.\nCONFIG_LEVEL_PROGRAMDATA - System-wide on Windows, for compatibility with portable git\nCONFIG_LEVEL_SYSTEM - System-wide configuration file; /etc/gitconfig on Linux systems\nCONFIG_LEVEL_XDG - XDG compatible configuration file; typically ~/.config/git/config\nCONFIG_LEVEL_GLOBAL - User-specific configuration file (also called Global configuration file); typically ~/.gitconfig\nCONFIG_LEVEL_LOCAL - Repository specific configuration file; $WORK_DIR/.git/config on non-bare repos\nCONFIG_LEVEL_APP - Application specific configuration file; freely defined by applications\nCONFIG_HIGHEST_LEVEL - Represents the highest level available config file (i.e. the most specific config file available that actually is loaded)\n\n\n\n\n\n","page":"LibGit2"},{"title":"Multi-Threading","location":"base/multi-threading.html#lib-multithreading","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.@threads\nBase.Threads.foreach\nBase.Threads.@spawn\nBase.Threads.threadid\nBase.Threads.maxthreadid\nBase.Threads.nthreads\nBase.Threads.threadpool\nBase.Threads.nthreadpools\nBase.Threads.threadpoolsize","page":"Multi-Threading"},{"title":"Base.Threads.@threads","location":"base/multi-threading.html#Base.Threads.@threads","category":"macro","text":"Threads.@threads [schedule] for ... end\n\nA macro to execute a for loop in parallel. The iteration space is distributed to coarse-grained tasks. This policy can be specified by the schedule argument. The execution of the loop waits for the evaluation of all iterations.\n\nSee also: @spawn and pmap in Distributed.\n\nExtended help\n\nSemantics\n\nUnless stronger guarantees are specified by the scheduling option, the loop executed by @threads macro have the following semantics.\n\nThe @threads macro executes the loop body in an unspecified order and potentially concurrently. It does not specify the exact assignments of the tasks and the worker threads. The assignments can be different for each execution. The loop body code (including any code transitively called from it) must not make any assumptions about the distribution of iterations to tasks or the worker thread in which they are executed. The loop body for each iteration must be able to make forward progress independent of other iterations and be free from data races. As such, invalid synchronizations across iterations may deadlock while unsynchronized memory accesses may result in undefined behavior.\n\nFor example, the above conditions imply that:\n\nThe lock taken in an iteration must be released within the same iteration.\nCommunicating between iterations using blocking primitives like Channels is incorrect.\nWrite only to locations not shared across iterations (unless a lock or atomic operation is used).\nUnless the :static schedule is used, the value of threadid() may change even within a single iteration. See Task Migration.\n\nSchedulers\n\nWithout the scheduler argument, the exact scheduling is unspecified and varies across Julia releases. Currently, :dynamic is used when the scheduler is not specified.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n:dynamic (default)\n\n:dynamic scheduler executes iterations dynamically to available worker threads. Current implementation assumes that the workload for each iteration is uniform. However, this assumption may be removed in the future.\n\nThis scheduling option is merely a hint to the underlying execution mechanism. However, a few properties can be expected. The number of Tasks used by :dynamic scheduler is bounded by a small constant multiple of the number of available worker threads (Threads.threadpoolsize()). Each task processes contiguous regions of the iteration space. Thus, @threads :dynamic for x in xs; f(x); end is typically more efficient than @sync for x in xs; @spawn f(x); end if length(xs) is significantly larger than the number of the worker threads and the run-time of f(x) is relatively smaller than the cost of spawning and synchronizing a task (typically less than 10 microseconds).\n\ncompat: Julia 1.8\nThe :dynamic option for the schedule argument is available and the default as of Julia 1.8.\n\n:static\n\n:static scheduler creates one task per thread and divides the iterations equally among them, assigning each task specifically to each thread. In particular, the value of threadid() is guaranteed to be constant within one iteration. Specifying :static is an error if used from inside another @threads loop or from a thread other than 1.\n\nnote: Note\n:static scheduling exists for supporting transition of code written before Julia 1.3. In newly written library functions, :static scheduling is discouraged because the functions using this option cannot be called from arbitrary worker threads.\n\nExample\n\nTo illustrate of the different scheduling strategies, consider the following function busywait containing a non-yielding timed loop that runs for a given number of seconds.\n\njulia> function busywait(seconds)\n            tstart = time_ns()\n            while (time_ns() - tstart) / 1e9 < seconds\n            end\n        end\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :static for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n6.003001 seconds (16.33 k allocations: 899.255 KiB, 0.25% compilation time)\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :dynamic for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n2.012056 seconds (16.05 k allocations: 883.919 KiB, 0.66% compilation time)\n\nThe :dynamic example takes 2 seconds since one of the non-occupied threads is able to run two of the 1-second iterations to complete the for loop.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.foreach","location":"base/multi-threading.html#Base.Threads.foreach","category":"function","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.threadpoolsize())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to f are split across ntasks tasks spawned by Threads.@spawn. This function will wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a manner that enables Julia's scheduler to more freely load-balance work items across threads. This approach generally has higher per-item overhead, but may perform better than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner that incurs lower per-item overhead than FairSchedule, but is less amenable to load-balancing. This approach thus may be more suitable for fine-grained, uniform workloads, but may perform worse than FairSchedule in concurrence with other multithreaded workloads.\n\nExamples\n\njulia> n = 20\n\njulia> c = Channel{Int}(ch -> foreach(i -> put!(ch, i), 1:n), 1)\n\njulia> d = Channel{Int}(n) do ch\n           f = i -> put!(ch, i^2)\n           Threads.foreach(f, c)\n       end\n\njulia> collect(d)\ncollect(d) = [1, 4, 9, 16, 25, 36, 49, 64, 81, 100, 121, 144, 169, 196, 225, 256, 289, 324, 361, 400]\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.@spawn","location":"base/multi-threading.html#Base.Threads.@spawn","category":"macro","text":"Threads.@spawn [:default|:interactive] expr\n\nCreate a Task and schedule it to run on any available thread in the specified threadpool (:default if unspecified). The task is allocated to a thread once one becomes available. To wait for the task to finish, call wait on the result of this macro, or call fetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nnote: Note\nThe thread that the task runs on may change if the task yields, therefore threadid() should not be treated as constant for a task. See Task Migration, and the broader multi-threading manual for further important caveats. See also the chapter on threadpools.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\ncompat: Julia 1.9\nA threadpool may be specified as of Julia 1.9.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadid","location":"base/multi-threading.html#Base.Threads.threadid","category":"function","text":"Threads.threadid() -> Int\n\nGet the ID number of the current thread of execution. The master thread has ID 1.\n\nExamples\n\njulia> Threads.threadid()\n1\n\njulia> Threads.@threads for i in 1:4\n          println(Threads.threadid())\n       end\n4\n2\n5\n4\n\nnote: Note\nThe thread that a task runs on may change if the task yields, which is known as Task Migration. For this reason in most cases it is not safe to use threadid() to index into, say, a vector of buffer or stateful objects.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.maxthreadid","location":"base/multi-threading.html#Base.Threads.maxthreadid","category":"function","text":"Threads.maxthreadid() -> Int\n\nGet a lower bound on the number of threads (across all thread pools) available to the Julia process, with atomic-acquire semantics. The result will always be greater than or equal to threadid() as well as threadid(task) for any task you were able to observe before calling maxthreadid.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.nthreads","location":"base/multi-threading.html#Base.Threads.nthreads","category":"function","text":"Threads.nthreads(:default | :interactive) -> Int\n\nGet the current number of threads within the specified thread pool. The threads in default have id numbers 1:nthreads(:default).\n\nSee also BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library and Threads.maxthreadid().\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadpool","location":"base/multi-threading.html#Base.Threads.threadpool","category":"function","text":"Threads.threadpool(tid = threadid()) -> Symbol\n\nReturns the specified thread's threadpool; either :default or :interactive.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.nthreadpools","location":"base/multi-threading.html#Base.Threads.nthreadpools","category":"function","text":"Threads.nthreadpools() -> Int\n\nReturns the number of threadpools currently configured.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadpoolsize","location":"base/multi-threading.html#Base.Threads.threadpoolsize","category":"function","text":"Threads.threadpoolsize(pool::Symbol = :default) -> Int\n\nGet the number of threads available to the default thread pool (or to the specified thread pool).\n\nSee also: BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"See also Multi-Threading.","page":"Multi-Threading"},{"title":"Atomic operations","location":"base/multi-threading.html#Atomic-operations","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.@atomic\nBase.@atomicswap\nBase.@atomicreplace","page":"Multi-Threading"},{"title":"Base.@atomic","location":"base/multi-threading.html#Base.@atomic","category":"macro","text":"@atomic var\n@atomic order ex\n\nMark var or ex as being performed atomically, if ex is a supported expression. If no order is specified it defaults to :sequentially_consistent.\n\n@atomic a.b.x = new\n@atomic a.b.x += addend\n@atomic :release a.b.x = new\n@atomic :acquire_release a.b.x += addend\n\nPerform the store operation expressed on the right atomically and return the new value.\n\nWith =, this operation translates to a setproperty!(a.b, :x, new) call. With any operator also, this operation translates to a modifyproperty!(a.b, :x, +, addend)[2] call.\n\n@atomic a.b.x max arg2\n@atomic a.b.x + arg2\n@atomic max(a.b.x, arg2)\n@atomic :acquire_release max(a.b.x, arg2)\n@atomic :acquire_release a.b.x + arg2\n@atomic :acquire_release a.b.x max arg2\n\nPerform the binary operation expressed on the right atomically. Store the result into the field in the first argument and return the values (old, new).\n\nThis operation translates to a modifyproperty!(a.b, :x, func, arg2) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomic :sequentially_consistent a.x = 2 # set field x of a, with sequential consistency\n2\n\njulia> @atomic a.x += 1 # increment field x of a, with sequential consistency\n3\n\njulia> @atomic a.x + 1 # increment field x of a, with sequential consistency\n3 => 4\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> @atomic max(a.x, 10) # change field x of a to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic a.x max 5 # again change field x of a to the max value, with sequential consistency\n10 => 10\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomicswap","location":"base/multi-threading.html#Base.@atomicswap","category":"macro","text":"@atomicswap a.b.x = new\n@atomicswap :sequentially_consistent a.b.x = new\n\nStores new into a.b.x and returns the old value of a.b.x.\n\nThis operation translates to a swapproperty!(a.b, :x, new) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicswap a.x = 2+2 # replace field x of a with 4, with sequential consistency\n1\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomicreplace","location":"base/multi-threading.html#Base.@atomicreplace","category":"macro","text":"@atomicreplace a.b.x expected => desired\n@atomicreplace :sequentially_consistent a.b.x expected => desired\n@atomicreplace :sequentially_consistent :monotonic a.b.x expected => desired\n\nPerform the conditional replacement expressed by the pair atomically, returning the values (old, success::Bool). Where success indicates whether the replacement was completed.\n\nThis operation translates to a replaceproperty!(a.b, :x, expected, desired) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n2\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace a.x xchg\n(old = 2, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n0\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"note: Note\nThe following APIs are fairly primitive, and will likely be exposed through an unsafe_*-like wrapper.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Core.Intrinsics.atomic_pointerref(pointer::Ptr{T}, order::Symbol) --> T\nCore.Intrinsics.atomic_pointerset(pointer::Ptr{T}, new::T, order::Symbol) --> pointer\nCore.Intrinsics.atomic_pointerswap(pointer::Ptr{T}, new::T, order::Symbol) --> old\nCore.Intrinsics.atomic_pointermodify(pointer::Ptr{T}, function::(old::T,arg::S)->T, arg::S, order::Symbol) --> old\nCore.Intrinsics.atomic_pointerreplace(pointer::Ptr{T}, expected::Any, new::T, success_order::Symbol, failure_order::Symbol) --> (old, cmp)","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"warning: Warning\nThe following APIs are deprecated, though support for them is likely to remain for several releases.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.Atomic\nBase.Threads.atomic_cas!\nBase.Threads.atomic_xchg!\nBase.Threads.atomic_add!\nBase.Threads.atomic_sub!\nBase.Threads.atomic_and!\nBase.Threads.atomic_nand!\nBase.Threads.atomic_or!\nBase.Threads.atomic_xor!\nBase.Threads.atomic_max!\nBase.Threads.atomic_min!\nBase.Threads.atomic_fence","page":"Multi-Threading"},{"title":"Base.Threads.Atomic","location":"base/multi-threading.html#Base.Threads.Atomic","category":"type","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only accessed atomically, i.e. in a thread-safe manner.\n\nOnly certain \"simple\" types can be used atomically, namely the primitive boolean, integer, and float-point types. These are Bool, Int8...Int128, UInt8...UInt128, and Float16...Float64.\n\nNew atomic objects can be created from a non-atomic values; if none is specified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!, atomic_xchg!, etc.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_cas!","location":"base/multi-threading.html#Base.Threads.atomic_cas!","category":"function","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write newval to x. Otherwise, leaves x unmodified. Returns the old value in x. By comparing the returned value to cmp (via ===) one knows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before the transaction, one records the value in x. After the transaction, the new value is stored only if x has not been modified in the mean time.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xchg!","location":"base/multi-threading.html#Base.Threads.atomic_xchg!","category":"function","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_add!","location":"base/multi-threading.html#Base.Threads.atomic_add!","category":"function","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_sub!","location":"base/multi-threading.html#Base.Threads.atomic_sub!","category":"function","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_and!","location":"base/multi-threading.html#Base.Threads.atomic_and!","category":"function","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_nand!","location":"base/multi-threading.html#Base.Threads.atomic_nand!","category":"function","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_or!","location":"base/multi-threading.html#Base.Threads.atomic_or!","category":"function","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xor!","location":"base/multi-threading.html#Base.Threads.atomic_xor!","category":"function","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_max!","location":"base/multi-threading.html#Base.Threads.atomic_max!","category":"function","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_min!","location":"base/multi-threading.html#Base.Threads.atomic_min!","category":"function","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_fence","location":"base/multi-threading.html#Base.Threads.atomic_fence","category":"function","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering semantics. There are algorithms where this is needed, i.e. where an acquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic operations in Julia already have acquire/release semantics, explicit fences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"ccall using a libuv threadpool (Experimental)","location":"base/multi-threading.html#ccall-using-a-libuv-threadpool-(Experimental)","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.@threadcall","page":"Multi-Threading"},{"title":"Base.@threadcall","location":"base/multi-threading.html#Base.@threadcall","category":"macro","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work in a different thread. This is useful when you want to call a blocking C function without causing the current julia thread to become blocked. Concurrency is limited by size of the libuv thread pool, which defaults to 4 threads but can be increased by setting the UV_THREADPOOL_SIZE environment variable and restarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Low-level synchronization primitives","location":"base/multi-threading.html#Low-level-synchronization-primitives","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"These building blocks are used to create the regular synchronization objects.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.SpinLock","page":"Multi-Threading"},{"title":"Base.Threads.SpinLock","location":"base/multi-threading.html#Base.Threads.SpinLock","category":"type","text":"SpinLock()\n\nCreate a non-reentrant, test-and-test-and-set spin lock. Recursive use will result in a deadlock. This kind of lock should only be used around code that takes little time to execute and does not block (e.g. perform I/O). In general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock. If !islocked(lck::SpinLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\nTest-and-test-and-set spin locks are quickest up to about 30ish contending threads. If you have more contention than that, different synchronization approaches should be considered.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"C Standard Library","location":"base/libc.html#C-Standard-Library","category":"section","text":"","page":"C Standard Library"},{"title":"C Standard Library","location":"base/libc.html","category":"page","text":"Base.Libc.malloc\nBase.Libc.calloc\nBase.Libc.realloc\nBase.Libc.free\nBase.Libc.errno\nBase.Libc.strerror\nBase.Libc.GetLastError\nBase.Libc.FormatMessage\nBase.Libc.time(::Base.Libc.TmStruct)\nBase.Libc.strftime\nBase.Libc.strptime\nBase.Libc.TmStruct\nBase.Libc.flush_cstdio\nBase.Libc.systemsleep","page":"C Standard Library"},{"title":"Base.Libc.malloc","location":"base/libc.html#Base.Libc.malloc","category":"function","text":"malloc(size::Integer) -> Ptr{Cvoid}\n\nCall malloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.calloc","location":"base/libc.html#Base.Libc.calloc","category":"function","text":"calloc(num::Integer, size::Integer) -> Ptr{Cvoid}\n\nCall calloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.realloc","location":"base/libc.html#Base.Libc.realloc","category":"function","text":"realloc(addr::Ptr, size::Integer) -> Ptr{Cvoid}\n\nCall realloc from the C standard library.\n\nSee warning in the documentation for free regarding only using this on memory originally obtained from malloc.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.free","location":"base/libc.html#Base.Libc.free","category":"function","text":"free(addr::Ptr)\n\nCall free from the C standard library. Only use this on memory obtained from malloc, not on pointers retrieved from other C libraries. Ptr objects obtained from C libraries should be freed by the free functions defined in that library, to avoid assertion failures if multiple libc libraries exist on the system.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.errno","location":"base/libc.html#Base.Libc.errno","category":"function","text":"errno([code])\n\nGet the value of the C library's errno. If an argument is specified, it is used to set the value of errno.\n\nThe value of errno is only valid immediately after a ccall to a C library routine that sets it. Specifically, you cannot call errno at the next prompt in a REPL, because lots of code is executed between prompts.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strerror","location":"base/libc.html#Base.Libc.strerror","category":"function","text":"strerror(n=errno())\n\nConvert a system call error code to a descriptive string\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.GetLastError","location":"base/libc.html#Base.Libc.GetLastError","category":"function","text":"GetLastError()\n\nCall the Win32 GetLastError function [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.FormatMessage","location":"base/libc.html#Base.Libc.FormatMessage","category":"function","text":"FormatMessage(n=GetLastError())\n\nConvert a Win32 system call error code to a descriptive string [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.time","location":"base/libc.html#Base.Libc.time-Tuple{Base.Libc.TmStruct}","category":"method","text":"time(t::TmStruct)\n\nConverts a TmStruct struct to a number of seconds since the epoch.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strftime","location":"base/libc.html#Base.Libc.strftime","category":"function","text":"strftime([format], time)\n\nConvert time, given as a number of seconds since the epoch or a TmStruct, to a formatted string using the given format. Supported formats are the same as those in the standard C library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strptime","location":"base/libc.html#Base.Libc.strptime","category":"function","text":"strptime([format], timestr)\n\nParse a formatted time string into a TmStruct giving the seconds, minute, hour, date, etc. Supported formats are the same as those in the standard C library. On some platforms, timezones will not be parsed correctly. If the result of this function will be passed to time to convert it to seconds since the epoch, the isdst field should be filled in manually. Setting it to -1 will tell the C library to use the current system settings to determine the timezone.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.TmStruct","location":"base/libc.html#Base.Libc.TmStruct","category":"type","text":"TmStruct([seconds])\n\nConvert a number of seconds since the epoch to broken-down format, with fields sec, min, hour, mday, month, year, wday, yday, and isdst.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.flush_cstdio","location":"base/libc.html#Base.Libc.flush_cstdio","category":"function","text":"flush_cstdio()\n\nFlushes the C stdout and stderr streams (which may have been written to by external C code).\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.systemsleep","location":"base/libc.html#Base.Libc.systemsleep","category":"function","text":"systemsleep(s::Real)\n\nSuspends execution for s seconds. This function does not yield to Julia's scheduler and therefore blocks the Julia thread that it is running on for the duration of the sleep time.\n\nSee also sleep.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Modules","location":"manual/modules.html#modules","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Modules in Julia help organize code into coherent units. They are delimited syntactically inside module NameOfModule ... end, and have the following features:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Modules are separate namespaces, each introducing a new global scope. This is useful, because it allows the same name to be used for different functions or global variables without conflict, as long as they are in separate modules.\nModules have facilities for detailed namespace management: each defines a set of names it exports, and can import names from other modules with using and import (we explain these below).\nModules can be precompiled for faster loading, and may contain code for runtime initialization.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Typically, in larger Julia packages you will see module code organized into files, eg","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"module SomeModule\n\n# export, using, import statements are usually here; we discuss these below\n\ninclude(\"file1.jl\")\ninclude(\"file2.jl\")\n\nend","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Files and file names are mostly unrelated to modules; modules are associated only with module expressions. One can have multiple files per module, and multiple modules per file. include behaves as if the contents of the source file were evaluated in the global scope of the including module. In this chapter, we use short and simplified examples, so we won't use include.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"The recommended style is not to indent the body of the module, since that would typically lead to whole files being indented. Also, it is common to use UpperCamelCase for module names (just like types), and use the plural form if applicable, especially if the module contains a similarly named identifier, to avoid name clashes. For example,","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"module FastThings\n\nstruct FastThing\n    ...\nend\n\nend","page":"Modules"},{"title":"Namespace management","location":"manual/modules.html#namespace-management","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Namespace management refers to the facilities the language offers for making names in a module available in other modules. We discuss the related concepts and functionality below in detail.","page":"Modules"},{"title":"Qualified names","location":"manual/modules.html#Qualified-names","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Names for functions, variables and types in the global scope like sin, ARGS, and UnitRange always belong to a module, called the parent module, which can be found interactively with parentmodule, for example","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> parentmodule(UnitRange)\nBase","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"One can also refer to these names outside their parent module by prefixing them with their module, eg Base.UnitRange. This is called a qualified name. The parent module may be accessible using a chain of submodules like Base.Math.sin, where Base.Math is called the module path. Due to syntactic ambiguities, qualifying a name that contains only symbols, such as an operator, requires inserting a colon, e.g. Base.:+. A small number of operators additionally require parentheses, e.g. Base.:(==).","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"If a name is qualified, then it is always accessible, and in case of a function, it can also have methods added to it by using the qualified name as the function name.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Within a module, a variable name can be “reserved” without assigning to it by declaring it as global x. This prevents name conflicts for globals initialized after load time. The syntax M.x = y does not work to assign a global in another module; global assignment is always module-local.","page":"Modules"},{"title":"Export lists","location":"manual/modules.html#Export-lists","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Names (referring to functions, types, global variables, and constants) can be added to the export list of a module with export: these are the symbols that are imported when using the module. Typically, they are at or near the top of the module definition so that readers of the source code can find them easily, as in","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> module NiceStuff\n       export nice, DOG\n       struct Dog end      # singleton type, not exported\n       const DOG = Dog()   # named instance, exported\n       nice(x) = \"nice $x\" # function, exported\n       end;\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"but this is just a style suggestion — a module can have multiple export statements in arbitrary locations.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"It is common to export names which form part of the API (application programming interface). In the above code, the export list suggests that users should use nice and DOG. However, since qualified names always make identifiers accessible, this is just an option for organizing APIs: unlike other languages, Julia has no facilities for truly hiding module internals.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Also, some modules don't export names at all. This is usually done if they use common words, such as derivative, in their API, which could easily clash with the export lists of other modules. We will see how to manage name clashes below.","page":"Modules"},{"title":"Standalone using and import","location":"manual/modules.html#Standalone-using-and-import","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Possibly the most common way of loading a module is using ModuleName. This loads the code associated with ModuleName, and brings","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"the module name\nand the elements of the export list into the surrounding global namespace.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Technically, the statement using ModuleName means that a module called ModuleName will be available for resolving names as needed. When a global variable is encountered that has no definition in the current module, the system will search for it among variables exported by ModuleName and use it if it is found there. This means that all uses of that global within the current module will resolve to the definition of that variable in ModuleName.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"To load a module from a package, the statement using ModuleName can be used. To load a module from a locally defined module, a dot needs to be added before the module name like using .ModuleName.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"To continue with our example,","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"would load the above code, making NiceStuff (the module name), DOG and nice available. Dog is not on the export list, but it can be accessed if the name is qualified with the module path (which here is just the module name) as NiceStuff.Dog.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Importantly, using ModuleName is the only form for which export lists matter at all.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"In contrast,","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> import .NiceStuff","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"brings only the module name into scope. Users would need to use NiceStuff.DOG, NiceStuff.Dog, and NiceStuff.nice to access its contents. Usually, import ModuleName is used in contexts when the user wants to keep the namespace clean. As we will see in the next section import .NiceStuff is equivalent to using .NiceStuff: NiceStuff.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"You can combine multiple using and import statements of the same kind in a comma-separated expression, e.g.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using LinearAlgebra, Statistics","page":"Modules"},{"title":"using and import with specific identifiers, and adding methods","location":"manual/modules.html#using-and-import-with-specific-identifiers,-and-adding-methods","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"When using ModuleName: or import ModuleName: is followed by a comma-separated list of names, the module is loaded, but only those specific names are brought into the namespace by the statement. For example,","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff: nice, DOG","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"will import the names nice and DOG.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Importantly, the module name NiceStuff will not be in the namespace. If you want to make it accessible, you have to list it explicitly, as","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff: nice, DOG, NiceStuff","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Julia has two forms for seemingly the same thing because only import ModuleName: f allows adding methods to f without a module path. That is to say, the following example will give an error:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nERROR: error in method definition: function NiceStuff.nice must be explicitly imported to be extended\nStacktrace:\n [1] top-level scope\n   @ none:0\n [2] top-level scope\n   @ none:1\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"This error prevents accidentally adding methods to functions in other modules that you only intended to use.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"There are two ways to deal with this. You can always qualify function names with a module path:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff\n\njulia> struct Cat end\n\njulia> NiceStuff.nice(::Cat) = \"nice 😸\"\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Alternatively, you can import the specific function name:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> import .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nnice (generic function with 2 methods)","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Which one you choose is a matter of style. The first form makes it clear that you are adding a method to a function in another module (remember, that the imports and the method definition may be in separate files), while the second one is shorter, which is especially convenient if you are defining multiple methods.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Once a variable is made visible via using or import, a module may not create its own variable with the same name. Imported variables are read-only; assigning to a global variable always affects a variable owned by the current module, or else raises an error.","page":"Modules"},{"title":"Renaming with as","location":"manual/modules.html#Renaming-with-as","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"An identifier brought into scope by import or using can be renamed with the keyword as. This is useful for working around name conflicts as well as for shortening names. For example, Base exports the function name read, but the CSV.jl package also provides CSV.read. If we are going to invoke CSV reading many times, it would be convenient to drop the CSV. qualifier. But then it is ambiguous whether we are referring to Base.read or CSV.read:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> read;\n\njulia> import CSV: read\nWARNING: ignoring conflicting import of CSV.read into Main","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Renaming provides a solution:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> import CSV: read as rd","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Imported packages themselves can also be renamed:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"import BenchmarkTools as BT","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"as works with using only when a single identifier is brought into scope. For example using CSV: read as rd works, but using CSV as C does not, since it operates on all of the exported names in CSV.","page":"Modules"},{"title":"Mixing multiple using and import statements","location":"manual/modules.html#Mixing-multiple-using-and-import-statements","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"When multiple using or import statements of any of the forms above are used, their effect is combined in the order they appear. For example,","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .NiceStuff         # exported names and the module name\n\njulia> import .NiceStuff: nice  # allows adding methods to unqualified functions\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"would bring all the exported names of NiceStuff and the module name itself into scope, and also allow adding methods to nice without prefixing it with a module name.","page":"Modules"},{"title":"Handling name conflicts","location":"manual/modules.html#Handling-name-conflicts","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Consider the situation where two (or more) packages export the same name, as in","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> module A\n       export f\n       f() = 1\n       end\nA\njulia> module B\n       export f\n       f() = 2\n       end\nB","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"The statement using .A, .B works, but when you try to call f, you get a warning","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> using .A, .B\n\njulia> f\nWARNING: both B and A export \"f\"; uses of it in module Main must be qualified\nERROR: UndefVarError: `f` not defined","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Here, Julia cannot decide which f you are referring to, so you have to make a choice. The following solutions are commonly used:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Simply proceed with qualified names like A.f and B.f. This makes the context clear to the reader of your code, especially if f just happens to coincide but has different meaning in various packages. For example, degree has various uses in mathematics, the natural sciences, and in everyday life, and these meanings should be kept separate.\nUse the as keyword above to rename one or both identifiers, eg\njulia> using .A: f as f\n\njulia> using .B: f as g\n\nwould make B.f available as g. Here, we are assuming that you did not use using A before, which would have brought f into the namespace.\nWhen the names in question do share a meaning, it is common for one module to import it from another, or have a lightweight “base” package with the sole function of defining an interface like this, which can be used by other packages. It is conventional to have such package names end in ...Base (which has nothing to do with Julia's Base module).","page":"Modules"},{"title":"Default top-level definitions and bare modules","location":"manual/modules.html#Default-top-level-definitions-and-bare-modules","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Modules automatically contain using Core, using Base, and definitions of the eval and include functions, which evaluate expressions/files within the global scope of that module.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"If these default definitions are not wanted, modules can be defined using the keyword baremodule instead (note: Core is still imported). In terms of baremodule, a standard module looks like this:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"baremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"If even Core is not wanted, a module that imports nothing and defines no names at all can be defined with Module(:YourNameHere, false, false) and code can be evaluated into it with @eval or Core.eval:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> arithmetic = Module(:arithmetic, false, false)\nMain.arithmetic\n\njulia> @eval arithmetic add(x, y) = $(+)(x, y)\nadd (generic function with 1 method)\n\njulia> arithmetic.add(12, 13)\n25","page":"Modules"},{"title":"Standard modules","location":"manual/modules.html#Standard-modules","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"There are three important standard modules:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Core contains all functionality \"built into\" the language.\nBase contains basic functionality that is useful in almost all cases.\nMain is the top-level module and the current module, when Julia is started.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"note: Standard library modules\nBy default Julia ships with some standard library modules. These behave like regular Julia packages except that you don't need to install them explicitly. For example, if you wanted to perform some unit testing, you could load the Test standard library as follows:using Test","page":"Modules"},{"title":"Submodules and relative paths","location":"manual/modules.html#Submodules-and-relative-paths","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Modules can contain submodules, nesting the same syntax module ... end. They can be used to introduce separate namespaces, which can be helpful for organizing complex codebases. Note that each module introduces its own scope, so submodules do not automatically “inherit” names from their parent.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"It is recommended that submodules refer to other modules within the enclosing parent module (including the latter) using relative module qualifiers in using and import statements. A relative module qualifier starts with a period (.), which corresponds to the current module, and each successive . leads to the parent of the current module. This should be followed by modules if necessary, and eventually the actual name to access, all separated by .s.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Consider the following example, where the submodule SubA defines a function, which is then extended in its “sibling” module:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> module ParentModule\n       module SubA\n       export add_D  # exported interface\n       const D = 3\n       add_D(x) = x + D\n       end\n       using .SubA  # brings `add_D` into the namespace\n       export add_D # export it from ParentModule too\n       module SubB\n       import ..SubA: add_D # relative path for a “sibling” module\n       struct Infinity end\n       add_D(x::Infinity) = x\n       end\n       end;\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"You may see code in packages, which, in a similar situation, uses","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"julia> import .ParentModule.SubA: add_D\n","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"However, this operates through code loading, and thus only works if ParentModule is in a package. It is better to use relative paths.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Note that the order of definitions also matters if you are evaluating values. Consider","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"module TestPackage\n\nexport x, y\n\nx = 0\n\nmodule Sub\nusing ..TestPackage\nz = y # ERROR: UndefVarError: `y` not defined\nend\n\ny = 1\n\nend","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"where Sub is trying to use TestPackage.y before it was defined, so it does not have a value.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"For similar reasons, you cannot use a cyclic ordering:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"module A\n\nmodule B\nusing ..C # ERROR: UndefVarError: `C` not defined\nend\n\nmodule C\nusing ..B\nend\n\nend","page":"Modules"},{"title":"Module initialization and precompilation","location":"manual/modules.html#Module-initialization-and-precompilation","category":"section","text":"","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Large modules can take several seconds to load because executing all of the statements in a module often involves compiling a large amount of code. Julia creates precompiled caches of the module to reduce this time.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Precompiled module files (sometimes called \"cache files\") are created and used automatically when import or using loads a module.  If the cache file(s) do not yet exist, the module will be compiled and saved for future reuse. You can also manually call Base.compilecache(Base.identify_package(\"modulename\")) to create these files without loading the module. The resulting cache files will be stored in the compiled subfolder of DEPOT_PATH[1]. If nothing about your system changes, such cache files will be used when you load the module with import or using.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Precompilation cache files store definitions of modules, types, methods, and constants. They may also store method specializations and the code generated for them, but this typically requires that the developer add explicit precompile directives or execute workloads that force compilation during the package build.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"However, if you update the module's dependencies or change its source code, the module is automatically recompiled upon using or import. Dependencies are modules it imports, the Julia build, files it includes, or explicit dependencies declared by include_dependency(path) in the module file(s).","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"For file dependencies, a change is determined by examining whether the modification time (mtime) of each file loaded by include or added explicitly by include_dependency is unchanged, or equal to the modification time truncated to the nearest second (to accommodate systems that can't copy mtime with sub-second accuracy). It also takes into account whether the path to the file chosen by the search logic in require matches the path that had created the precompile file. It also takes into account the set of dependencies already loaded into the current process and won't recompile those modules, even if their files change or disappear, in order to avoid creating incompatibilities between the running system and the precompile cache. Finally, it takes account of changes in any compile-time preferences.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"If you know that a module is not safe to precompile (for example, for one of the reasons described below), you should put __precompile__(false) in the module file (typically placed at the top). This will cause Base.compilecache to throw an error, and will cause using / import to load it directly into the current process and skip the precompile and caching. This also thereby prevents the module from being imported by any other precompiled module.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"You may need to be aware of certain behaviors inherent in the creation of incremental shared libraries which may require care when writing your module. For example, external state is not preserved. To accommodate this, explicitly separate any initialization steps that must occur at runtime from steps that can occur at compile time. For this purpose, Julia allows you to define an __init__() function in your module that executes any initialization steps that must occur at runtime. This function will not be called during compilation (--output-*). Effectively, you can assume it will be run exactly once in the lifetime of the code. You may, of course, call it manually if necessary, but the default is to assume this function deals with computing state for the local machine, which does not need to be – or even should not be – captured in the compiled image. It will be called after the module is loaded into a process, including if it is being loaded into an incremental compile (--output-incremental=yes), but not if it is being loaded into a full-compilation process.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"In particular, if you define a function __init__() in a module, then Julia will call __init__() immediately after the module is loaded (e.g., by import, using, or require) at runtime for the first time (i.e., __init__ is only called once, and only after all statements in the module have been executed). Because it is called after the module is fully imported, any submodules or other imported modules have their __init__ functions called before the __init__ of the enclosing module.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Two typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries.  For example, suppose that we are calling a C library libfoo that requires us to call a foo_init() initialization function at runtime. Suppose that we also want to define a global constant foo_data_ptr that holds the return value of a void *foo_data() function defined by libfoo – this constant must be initialized at runtime (not at compile time) because the pointer address will change from run to run.  You could accomplish this by defining the following __init__ function in your module:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"const foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Notice that it is perfectly possible to define a global inside a function like __init__; this is one of the advantages of using a dynamic language. But by making it a constant at global scope, we can ensure that the type is known to the compiler and allow it to generate better optimized code. Obviously, any other globals in your module that depends on foo_data_ptr would also have to be initialized in __init__.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Constants involving most Julia objects that are not produced by ccall do not need to be placed in __init__: their definitions can be precompiled and loaded from the cached module image. This includes complicated heap-allocated objects like arrays. However, any routine that returns a raw pointer value must be called at runtime for precompilation to work (Ptr objects will turn into null pointers unless they are hidden inside an isbits object). This includes the return values of the Julia functions @cfunction and pointer.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Dictionary and set types, or in general anything that depends on the output of a hash(key) method, are a trickier case.  In the common case where the keys are numbers, strings, symbols, ranges, Expr, or compositions of these types (via arrays, tuples, sets, pairs, etc.) they are safe to precompile.  However, for a few other key types, such as Function or DataType and generic user-defined types where you haven't defined a hash method, the fallback hash method depends on the memory address of the object (via its objectid) and hence may change from run to run. If you have one of these key types, or if you aren't sure, to be safe you can initialize this dictionary from within your __init__ function. Alternatively, you can use the IdDict dictionary type, which is specially handled by precompilation so that it is safe to initialize at compile-time.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"When using precompilation, it is important to keep a clear sense of the distinction between the compilation phase and the execution phase. In this mode, it will often be much more clearly apparent that Julia is a compiler which allows execution of arbitrary Julia code, not a standalone interpreter that also generates compiled code.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Other known potential failure scenarios include:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Global counters (for example, for attempting to uniquely identify objects). Consider the following code snippet:\nmutable struct UniquedById\n    myid::Int\n    let counter = 0\n        UniquedById() = new(counter += 1)\n    end\nend\nwhile the intent of this code was to give every instance a unique id, the counter value is recorded at the end of compilation. All subsequent usages of this incrementally compiled module will start from that same counter value.\nNote that objectid (which works by hashing the memory pointer) has similar issues (see notes on Dict usage below).\nOne alternative is to use a macro to capture @__MODULE__ and store it alone with the current counter value, however, it may be better to redesign the code to not depend on this global state.\nAssociative collections (such as Dict and Set) need to be re-hashed in __init__. (In the future, a mechanism may be provided to register an initializer function.)\nDepending on compile-time side-effects persisting through load-time. Example include: modifying arrays or other variables in other Julia modules; maintaining handles to open files or devices; storing pointers to other system resources (including memory);\nCreating accidental \"copies\" of global state from another module, by referencing it directly instead of via its lookup path. For example, (in global scope):\n#mystdout = Base.stdout #= will not work correctly, since this will copy Base.stdout into this module =#\n# instead use accessor functions:\ngetstdout() = Base.stdout #= best option =#\n# or move the assignment into the runtime:\n__init__() = global mystdout = Base.stdout #= also works =#","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Several additional restrictions are placed on the operations that can be done while precompiling code to help the user avoid other wrong-behavior situations:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"Calling eval to cause a side-effect in another module. This will also cause a warning to be emitted when the incremental precompile flag is set.\nglobal const statements from local scope after __init__() has been started (see issue #12010 for plans to add an error for this)\nReplacing a module is a runtime error while doing an incremental precompile.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"A few other points to be aware of:","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"No code reload / cache invalidation is performed after changes are made to the source files themselves, (including by Pkg.update), and no cleanup is done after Pkg.rm\nThe memory sharing behavior of a reshaped array is disregarded by precompilation (each view gets its own copy)\nExpecting the filesystem to be unchanged between compile-time and runtime e.g. @__FILE__/source_path() to find resources at runtime, or the BinDeps @checked_lib macro. Sometimes this is unavoidable. However, when possible, it can be good practice to copy resources into the module at compile-time so they won't need to be found at runtime.\nWeakRef objects and finalizers are not currently handled properly by the serializer (this will be fixed in an upcoming release).\nIt is usually best to avoid capturing references to instances of internal metadata objects such as Method, MethodInstance, MethodTable, TypeMapLevel, TypeMapEntry and fields of those objects, as this can confuse the serializer and may not lead to the outcome you desire. It is not necessarily an error to do this, but you simply need to be prepared that the system will try to copy some of these and to create a single unique instance of others.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"It is sometimes helpful during module development to turn off incremental precompilation. The command line flag --compiled-modules={yes|no} enables you to toggle module precompilation on and off. When Julia is started with --compiled-modules=no the serialized modules in the compile cache are ignored when loading modules and module dependencies. More fine-grained control is available with --pkgimages=no, which suppresses only native-code storage during precompilation. Base.compilecache can still be called manually. The state of this command line flag is passed to Pkg.build to disable automatic precompilation triggering when installing, updating, and explicitly building packages.","page":"Modules"},{"title":"Modules","location":"manual/modules.html","category":"page","text":"You can also debug some precompilation failures with environment variables. Setting JULIA_VERBOSE_LINKING=true may help resolve failures in linking shared libraries of compiled native code. See the Developer Documentation part of the Julia manual, where you will find further details in the section documenting Julia's internals under \"Package Images\".","page":"Modules"},{"title":"Distributed Computing","location":"stdlib/Distributed.html#man-distributed","category":"section","text":"","page":"Distributed Computing"},{"title":"Distributed Computing","location":"stdlib/Distributed.html","category":"page","text":"Tools for distributed parallel processing.","page":"Distributed Computing"},{"title":"Distributed Computing","location":"stdlib/Distributed.html","category":"page","text":"Distributed.addprocs\nDistributed.nprocs\nDistributed.nworkers\nDistributed.procs()\nDistributed.procs(::Integer)\nDistributed.workers\nDistributed.rmprocs\nDistributed.interrupt\nDistributed.myid\nDistributed.pmap\nDistributed.RemoteException\nDistributed.ProcessExitedException\nDistributed.Future\nDistributed.RemoteChannel\nDistributed.fetch(::Distributed.Future)\nDistributed.fetch(::RemoteChannel)\nDistributed.remotecall(::Any, ::Integer, ::Any...)\nDistributed.remotecall_wait(::Any, ::Integer, ::Any...)\nDistributed.remotecall_fetch(::Any, ::Integer, ::Any...)\nDistributed.remote_do(::Any, ::Integer, ::Any...)\nDistributed.put!(::RemoteChannel, ::Any...)\nDistributed.put!(::Distributed.Future, ::Any)\nDistributed.take!(::RemoteChannel, ::Any...)\nDistributed.isready(::RemoteChannel, ::Any...)\nDistributed.isready(::Distributed.Future)\nDistributed.AbstractWorkerPool\nDistributed.WorkerPool\nDistributed.CachingPool\nDistributed.default_worker_pool\nDistributed.clear!(::CachingPool)\nDistributed.remote\nDistributed.remotecall(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remotecall_wait(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remotecall_fetch(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remote_do(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.@spawnat\nDistributed.@fetch\nDistributed.@fetchfrom\nDistributed.@distributed\nDistributed.@everywhere\nDistributed.clear!(::Any, ::Any; ::Any)\nDistributed.remoteref_id\nDistributed.channel_from_id\nDistributed.worker_id_from_socket\nDistributed.cluster_cookie()\nDistributed.cluster_cookie(::Any)","page":"Distributed Computing"},{"title":"Distributed.addprocs","location":"stdlib/Distributed.html#Distributed.addprocs","category":"function","text":"addprocs(manager::ClusterManager; kwargs...) -> List of process identifiers\n\nLaunches worker processes via the specified cluster manager.\n\nFor example, Beowulf clusters are supported via a custom cluster manager implemented in the package ClusterManagers.jl.\n\nThe number of seconds a newly launched worker waits for connection establishment from the master can be specified via variable JULIA_WORKER_TIMEOUT in the worker process's environment. Relevant only when using TCP/IP as transport.\n\nTo launch workers without blocking the REPL, or the containing function if launching workers programmatically, execute addprocs in its own task.\n\nExamples\n\n# On busy clusters, call `addprocs` asynchronously\nt = @async addprocs(...)\n\n# Utilize workers as and when they come online\nif nprocs() > 1   # Ensure at least one new worker is available\n   ....   # perform distributed execution\nend\n\n# Retrieve newly launched worker IDs, or any error messages\nif istaskdone(t)   # Check if `addprocs` has completed to ensure `fetch` doesn't block\n    if nworkers() == N\n        new_pids = fetch(t)\n    else\n        fetch(t)\n    end\nend\n\n\n\n\n\naddprocs(machines; tunnel=false, sshflags=``, max_parallel=10, kwargs...) -> List of process identifiers\n\nAdd worker processes on remote machines via SSH. Configuration is done with keyword arguments (see below). In particular, the exename keyword can be used to specify the path to the julia binary on the remote machine(s).\n\nmachines is a vector of \"machine specifications\" which are given as strings of the form [user@]host[:port] [bind_addr[:port]]. user defaults to current user and port to the standard SSH port. If [bind_addr[:port]] is specified, other workers will connect to this worker at the specified bind_addr and port.\n\nIt is possible to launch multiple processes on a remote host by using a tuple in the machines vector or the form (machine_spec, count), where count is the number of workers to be launched on the specified host. Passing :auto as the worker count will launch as many workers as the number of CPU threads on the remote host.\n\nExamples:\n\naddprocs([\n    \"remote1\",               # one worker on 'remote1' logging in with the current username\n    \"user@remote2\",          # one worker on 'remote2' logging in with the 'user' username\n    \"user@remote3:2222\",     # specifying SSH port to '2222' for 'remote3'\n    (\"user@remote4\", 4),     # launch 4 workers on 'remote4'\n    (\"user@remote5\", :auto), # launch as many workers as CPU threads on 'remote5'\n])\n\nKeyword arguments:\n\ntunnel: if true then SSH tunneling will be used to connect to the worker from the master process. Default is false.\nmultiplex: if true then SSH multiplexing is used for SSH tunneling. Default is false.\nssh: the name or path of the SSH client executable used to start the workers. Default is \"ssh\".\nsshflags: specifies additional ssh options, e.g. sshflags=`-i /home/foo/bar.pem`\nmax_parallel: specifies the maximum number of workers connected to in parallel at a host. Defaults to 10.\nshell: specifies the type of shell to which ssh connects on the workers.\nshell=:posix: a POSIX-compatible Unix/Linux shell (sh, ksh, bash, dash, zsh, etc.). The default.\nshell=:csh: a Unix C shell (csh, tcsh).\nshell=:wincmd: Microsoft Windows cmd.exe.\ndir: specifies the working directory on the workers. Defaults to the host's current directory (as found by pwd())\nenable_threaded_blas: if true then  BLAS will run on multiple threads in added processes. Default is false.\nexename: name of the julia executable. Defaults to \"$(Sys.BINDIR)/julia\" or \"$(Sys.BINDIR)/julia-debu