var documenterSearchIndex = {"docs":
[{"title":"LibGit2","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2","category":"section","text":"The LibGit2 module provides bindings to libgit2, a portable C library that implements core functionality for the Git version control system. These bindings are currently used to power Julia's package manager. It is expected that this module will eventually be moved into a separate package."},{"title":"Functionality","page":"LibGit2","location":"stdlib/LibGit2.html#Functionality","category":"section","text":"Some of this documentation assumes some prior knowledge of the libgit2 API. For more information on some of the objects and methods referenced here, consult the upstream libgit2 API reference."},{"title":"LibGit2.Buffer","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.Buffer","category":"type","text":"LibGit2.Buffer\n\nA data buffer for exporting data from libgit2. Matches the git_buf struct.\n\nWhen fetching data from LibGit2, a typical usage would look like:\n\nbuf_ref = Ref(Buffer())\n@check ccall(..., (Ptr{Buffer},), buf_ref)\n# operation on buf_ref\nfree(buf_ref)\n\nIn particular, note that LibGit2.free should be called afterward on the Ref object.\n\n\n\n\n\n"},{"title":"LibGit2.CheckoutOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.CheckoutOptions","category":"type","text":"LibGit2.CheckoutOptions\n\nMatches the git_checkout_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncheckout_strategy: determine how to handle conflicts and whether to force the  checkout/recreate missing files.\ndisable_filters: if nonzero, do not apply filters like CLRF (to convert file newlines between UNIX and DOS).\ndir_mode: read/write/access mode for any directories involved in the checkout. Default is 0755.\nfile_mode: read/write/access mode for any files involved in the checkout.  Default is 0755 or 0644, depending on the blob.\nfile_open_flags: bitflags used to open any files during the checkout.\nnotify_flags: Flags for what sort of conflicts the user should be notified about.\nnotify_cb: An optional callback function to notify the user if a checkout conflict occurs.  If this function returns a non-zero value, the checkout will be cancelled.\nnotify_payload: Payload for the notify callback function.\nprogress_cb: An optional callback function to display checkout progress.\nprogress_payload: Payload for the progress callback.\npaths: If not empty, describes which paths to search during the checkout.  If empty, the checkout will occur over all files in the repository.\nbaseline: Expected content of the workdir, captured in a (pointer to a)  GitTree. Defaults to the state of the tree at HEAD.\nbaseline_index: Expected content of the workdir, captured in a (pointer to a)  GitIndex. Defaults to the state of the index at HEAD.\ntarget_directory: If not empty, checkout to this directory instead of the workdir.\nancestor_label: In case of conflicts, the name of the common ancestor side.\nour_label: In case of conflicts, the name of \"our\" side.\ntheir_label: In case of conflicts, the name of \"their\" side.\nperfdata_cb: An optional callback function to display performance data.\nperfdata_payload: Payload for the performance callback.\n\n\n\n\n\n"},{"title":"LibGit2.CloneOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.CloneOptions","category":"type","text":"LibGit2.CloneOptions\n\nMatches the git_clone_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncheckout_opts: The options for performing the checkout of the remote as part of the clone.\nfetch_opts: The options for performing the pre-checkout fetch of the remote as part of the clone.\nbare: If 0, clone the full remote repository. If non-zero, perform a bare clone, in which  there is no local copy of the source files in the repository and the gitdir and workdir  are the same.\nlocalclone: Flag whether to clone a local object database or do a fetch. The default is to let git decide.  It will not use the git-aware transport for a local clone, but will use it for URLs which begin with file://.\ncheckout_branch: The name of the branch to checkout. If an empty string, the default branch of the  remote will be checked out.\nrepository_cb: An optional callback which will be used to create the new repository into which  the clone is made.\nrepository_cb_payload: The payload for the repository callback.\nremote_cb: An optional callback used to create the GitRemote before making the clone from it.\nremote_cb_payload: The payload for the remote callback.\n\n\n\n\n\n"},{"title":"LibGit2.DescribeOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.DescribeOptions","category":"type","text":"LibGit2.DescribeOptions\n\nMatches the git_describe_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nmax_candidates_tags: consider this many most recent tags in refs/tags to describe a commit.  Defaults to 10 (so that the 10 most recent tags would be examined to see if they describe a commit).\ndescribe_strategy: whether to consider all entries in refs/tags (equivalent to git-describe --tags)  or all entries in refs/ (equivalent to git-describe --all). The default is to only show annotated tags.  If Consts.DESCRIBE_TAGS is passed, all tags, annotated or not, will be considered.  If Consts.DESCRIBE_ALL is passed, any ref in refs/ will be considered.\npattern: only consider tags which match pattern. Supports glob expansion.\nonly_follow_first_parent: when finding the distance from a matching reference to the described  object, only consider the distance from the first parent.\nshow_commit_oid_as_fallback: if no matching reference can be found which describes a commit, show the  commit's GitHash instead of throwing an error (the default behavior).\n\n\n\n\n\n"},{"title":"LibGit2.DescribeFormatOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.DescribeFormatOptions","category":"type","text":"LibGit2.DescribeFormatOptions\n\nMatches the git_describe_format_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nabbreviated_size: lower bound on the size of the abbreviated GitHash to use, defaulting to 7.\nalways_use_long_format: set to 1 to use the long format for strings even if a short format can be used.\ndirty_suffix: if set, this will be appended to the end of the description string if the workdir is dirty.\n\n\n\n\n\n"},{"title":"LibGit2.DiffDelta","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.DiffDelta","category":"type","text":"LibGit2.DiffDelta\n\nDescription of changes to one entry. Matches the git_diff_delta struct.\n\nThe fields represent:\n\nstatus: One of Consts.DELTA_STATUS, indicating whether the file has been added/modified/deleted.\nflags: Flags for the delta and the objects on each side. Determines whether to treat the file(s)  as binary/text, whether they exist on each side of the diff, and whether the object ids are known  to be correct.\nsimilarity: Used to indicate if a file has been renamed or copied.\nnfiles: The number of files in the delta (for instance, if the delta  was run on a submodule commit id, it may contain more than one file).\nold_file: A DiffFile containing information about the file(s) before the changes.\nnew_file: A DiffFile containing information about the file(s) after the changes.\n\n\n\n\n\n"},{"title":"LibGit2.DiffFile","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.DiffFile","category":"type","text":"LibGit2.DiffFile\n\nDescription of one side of a delta. Matches the git_diff_file struct.\n\nThe fields represent:\n\nid: the GitHash of the item in the diff. If the item is empty on this  side of the diff (for instance, if the diff is of the removal of a file), this will  be GitHash(0).\npath: a NULL terminated path to the item relative to the working directory of the repository.\nsize: the size of the item in bytes.\nflags: a combination of the git_diff_flag_t  flags. The ith bit of this integer sets the ith flag.\nmode: the stat mode for the item.\nid_abbrev: only present in LibGit2 versions newer than or equal to 0.25.0.  The length of the id field when converted using string. Usually equal to OID_HEXSZ (40).\n\n\n\n\n\n"},{"title":"LibGit2.DiffOptionsStruct","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.DiffOptionsStruct","category":"type","text":"LibGit2.DiffOptionsStruct\n\nMatches the git_diff_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: flags controlling which files will appear in the diff. Defaults to DIFF_NORMAL.\nignore_submodules: whether to look at files in submodules or not. Defaults to SUBMODULE_IGNORE_UNSPECIFIED, which means the submodule's configuration will control  whether it appears in the diff or not.\npathspec: path to files to include in the diff. Default is to use all files in the repository.\nnotify_cb: optional callback which will notify the user of changes to the diff as file deltas are  added to it.\nprogress_cb: optional callback which will display diff progress. Only relevant on libgit2 versions  at least as new as 0.24.0.\npayload: the payload to pass to notify_cb and progress_cb.\ncontext_lines: the number of unchanged lines used to define the edges of a hunk.  This is also the number of lines which will be shown before/after a hunk to provide  context. Default is 3.\ninterhunk_lines: the maximum number of unchanged lines between two separate  hunks allowed before the hunks will be combined. Default is 0.\nid_abbrev: sets the length of the abbreviated GitHash to print.  Default is 7.\nmax_size: the maximum file size of a blob. Above this size, it will be treated  as a binary blob. The default is 512 MB.\nold_prefix: the virtual file directory in which to place old files on one side  of the diff. Default is \"a\".\nnew_prefix: the virtual file directory in which to place new files on one side  of the diff. Default is \"b\".\n\n\n\n\n\n"},{"title":"LibGit2.FetchHead","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.FetchHead","category":"type","text":"LibGit2.FetchHead\n\nContains the information about HEAD during a fetch, including the name and URL of the branch fetched from, the oid of the HEAD, and whether the fetched HEAD has been merged locally.\n\nThe fields represent:\n\nname: The name in the local reference database of the fetch head, for example,  \"refs/heads/master\".\nurl: The URL of the fetch head.\noid: The GitHash of the tip of the fetch head.\nismerge: Boolean flag indicating whether the changes at the  remote have been merged into the local copy yet or not. If true, the local  copy is up to date with the remote fetch head.\n\n\n\n\n\n"},{"title":"LibGit2.FetchOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.FetchOptions","category":"type","text":"LibGit2.FetchOptions\n\nMatches the git_fetch_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\ncallbacks: remote callbacks to use during the fetch.\nprune: whether to perform a prune after the fetch or not. The default is to  use the setting from the GitConfig.\nupdate_fetchhead: whether to update the FetchHead after the fetch.  The default is to perform the update, which is the normal git behavior.\ndownload_tags: whether to download tags present at the remote or not. The default  is to request the tags for objects which are being downloaded anyway from the server.\nproxy_opts: options for connecting to the remote through a proxy. See ProxyOptions.  Only present on libgit2 versions newer than or equal to 0.25.0.\ncustom_headers: any extra headers needed for the fetch. Only present on libgit2 versions  newer than or equal to 0.24.0.\n\n\n\n\n\n"},{"title":"LibGit2.GitAnnotated","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitAnnotated","category":"type","text":"GitAnnotated(repo::GitRepo, commit_id::GitHash)\nGitAnnotated(repo::GitRepo, ref::GitReference)\nGitAnnotated(repo::GitRepo, fh::FetchHead)\nGitAnnotated(repo::GitRepo, committish::AbstractString)\n\nAn annotated git commit carries with it information about how it was looked up and why, so that rebase or merge operations have more information about the context of the commit. Conflict files contain information about the source/target branches in the merge which are conflicting, for instance. An annotated commit can refer to the tip of a remote branch, for instance when a FetchHead is passed, or to a branch head described using GitReference.\n\n\n\n\n\n"},{"title":"LibGit2.GitBlame","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitBlame","category":"type","text":"GitBlame(repo::GitRepo, path::AbstractString; options::BlameOptions=BlameOptions())\n\nConstruct a GitBlame object for the file at path, using change information gleaned from the history of repo. The GitBlame object records who changed which chunks of the file when, and how. options controls how to separate the contents of the file and which commits to probe - see BlameOptions for more information.\n\n\n\n\n\n"},{"title":"LibGit2.GitBlob","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitBlob","category":"type","text":"GitBlob(repo::GitRepo, hash::AbstractGitHash)\nGitBlob(repo::GitRepo, spec::AbstractString)\n\nReturn a GitBlob object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n"},{"title":"LibGit2.GitCommit","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitCommit","category":"type","text":"GitCommit(repo::GitRepo, hash::AbstractGitHash)\nGitCommit(repo::GitRepo, spec::AbstractString)\n\nReturn a GitCommit object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n"},{"title":"LibGit2.GitConfig","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitConfig","category":"type","text":"GitConfig(path::AbstractString, level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_APP, force::Bool=false)\n\nCreate a new GitConfig by loading configuration information from the file at path. See addfile for more information about the level, repo and force options.\n\n\n\n\n\nGitConfig(repo::GitRepo)\n\nGet the stored configuration for the git repository repo. If repo does not have a specific configuration file set, the default git configuration will be used.\n\n\n\n\n\nGitConfig(level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_DEFAULT)\n\nGet the default git configuration by loading the global and system configuration files into a prioritized configuration. This can be used to access default configuration options outside a specific git repository.\n\n\n\n\n\n"},{"title":"LibGit2.GitHash","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitHash","category":"type","text":"GitHash\n\nA git object identifier, based on the sha-1 hash. It is a 20 byte string (40 hex digits) used to identify a GitObject in a repository.\n\n\n\n\n\n"},{"title":"LibGit2.GitObject","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitObject","category":"type","text":"GitObject(repo::GitRepo, hash::AbstractGitHash)\nGitObject(repo::GitRepo, spec::AbstractString)\n\nReturn the specified object (GitCommit, GitBlob, GitTree or GitTag) from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n"},{"title":"LibGit2.GitRemote","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitRemote","category":"type","text":"GitRemote(repo::GitRepo, rmt_name::AbstractString, rmt_url::AbstractString)::GitRemote\n\nLook up a remote git repository using its name and URL. Uses the default fetch refspec.\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nremote = LibGit2.GitRemote(repo, \"upstream\", repo_url)\n\n\n\n\n\nGitRemote(repo::GitRepo, rmt_name::AbstractString, rmt_url::AbstractString, fetch_spec::AbstractString)::GitRemote\n\nLook up a remote git repository using the repository's name and URL, as well as specifications for how to fetch from the remote (e.g. which remote branch to fetch from).\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nrefspec = \"+refs/heads/mybranch:refs/remotes/origin/mybranch\"\nremote = LibGit2.GitRemote(repo, \"upstream\", repo_url, refspec)\n\n\n\n\n\n"},{"title":"LibGit2.GitRemoteAnon","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitRemoteAnon","category":"function","text":"GitRemoteAnon(repo::GitRepo, url::AbstractString)::GitRemote\n\nLook up a remote git repository using only its URL, not its name.\n\nExamples\n\nrepo = LibGit2.init(repo_path)\nremote = LibGit2.GitRemoteAnon(repo, repo_url)\n\n\n\n\n\n"},{"title":"LibGit2.GitRepo","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitRepo","category":"type","text":"LibGit2.GitRepo(path::AbstractString)\n\nOpen a git repository at path.\n\n\n\n\n\n"},{"title":"LibGit2.GitRepoExt","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitRepoExt","category":"function","text":"LibGit2.GitRepoExt(path::AbstractString, flags::Cuint = Cuint(Consts.REPOSITORY_OPEN_DEFAULT))\n\nOpen a git repository at path with extended controls (for instance, if the current user must be a member of a special access group to read path).\n\n\n\n\n\n"},{"title":"LibGit2.GitRevWalker","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitRevWalker","category":"type","text":"GitRevWalker(repo::GitRepo)\n\nA GitRevWalker walks through the revisions (i.e. commits) of a git repository repo. It is a collection of the commits in the repository, and supports iteration and calls to LibGit2.map and LibGit2.count (for instance, LibGit2.count could be used to determine what percentage of commits in a repository were made by a certain author).\n\ncnt = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.count((oid,repo)->(oid == commit_oid1), walker, oid=commit_oid1, by=LibGit2.Consts.SORT_TIME)\nend\n\nHere, LibGit2.count finds the number of commits along the walk with a certain GitHash. Since the GitHash is unique to a commit, cnt will be 1.\n\n\n\n\n\n"},{"title":"LibGit2.GitShortHash","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitShortHash","category":"type","text":"GitShortHash(hash::GitHash, len::Integer)\n\nA shortened git object identifier, which can be used to identify a git object when it is unique, consisting of the initial len hexadecimal digits of hash (the remaining digits are ignored).\n\n\n\n\n\n"},{"title":"LibGit2.GitSignature","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitSignature","category":"type","text":"LibGit2.GitSignature\n\nThis is a Julia wrapper around a pointer to a git_signature object.\n\n\n\n\n\n"},{"title":"LibGit2.GitStatus","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitStatus","category":"type","text":"LibGit2.GitStatus(repo::GitRepo; status_opts=StatusOptions())\n\nCollect information about the status of each file in the git repository repo (e.g. is the file modified, staged, etc.). status_opts can be used to set various options, for instance whether or not to look at untracked files or whether to include submodules or not. See StatusOptions for more information.\n\n\n\n\n\n"},{"title":"LibGit2.GitTag","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitTag","category":"type","text":"GitTag(repo::GitRepo, hash::AbstractGitHash)\nGitTag(repo::GitRepo, spec::AbstractString)\n\nReturn a GitTag object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n"},{"title":"LibGit2.GitTree","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitTree","category":"type","text":"GitTree(repo::GitRepo, hash::AbstractGitHash)\nGitTree(repo::GitRepo, spec::AbstractString)\n\nReturn a GitTree object from repo specified by hash/spec.\n\nhash is a full (GitHash) or partial (GitShortHash) hash.\nspec is a textual specification: see the git docs for a full list.\n\n\n\n\n\n"},{"title":"LibGit2.IndexEntry","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.IndexEntry","category":"type","text":"LibGit2.IndexEntry\n\nIn-memory representation of a file entry in the index. Matches the git_index_entry struct.\n\n\n\n\n\n"},{"title":"LibGit2.IndexTime","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.IndexTime","category":"type","text":"LibGit2.IndexTime\n\nMatches the git_index_time struct.\n\n\n\n\n\n"},{"title":"LibGit2.BlameOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.BlameOptions","category":"type","text":"LibGit2.BlameOptions\n\nMatches the git_blame_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: one of Consts.BLAME_NORMAL or Consts.BLAME_FIRST_PARENT (the other blame flags  are not yet implemented by libgit2).\nmin_match_characters: the minimum number of alphanumeric characters which much change in a commit in order for the change to be associated with that commit. The default is 20. Only takes effect if one of the Consts.BLAME_*_COPIES flags are used, which libgit2 does not implement yet.\nnewest_commit: the GitHash of the newest commit from which to look at changes.\noldest_commit: the GitHash of the oldest commit from which to look at changes.\nmin_line: the first line of the file from which to starting blaming. The default is 1.\nmax_line: the last line of the file to which to blame. The default is 0, meaning the last line of the file.\n\n\n\n\n\n"},{"title":"LibGit2.MergeOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.MergeOptions","category":"type","text":"LibGit2.MergeOptions\n\nMatches the git_merge_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nflags: an enum for flags describing merge behavior.  Defined in git_merge_flag_t.  The corresponding Julia enum is GIT_MERGE and has values:\nMERGE_FIND_RENAMES: detect if a file has been renamed between the common ancestor and the \"ours\" or \"theirs\" side of the merge. Allows merges where a file has been renamed.\nMERGE_FAIL_ON_CONFLICT: exit immediately if a conflict is found rather than trying to resolve it.\nMERGE_SKIP_REUC: do not write the REUC extension on the index resulting from the merge.\nMERGE_NO_RECURSIVE: if the commits being merged have multiple merge bases, use the first one, rather than trying to recursively merge the bases.\nrename_threshold: how similar two files must to consider one a rename of the other. This is an integer that sets the percentage similarity. The default is 50.\ntarget_limit: the maximum number of files to compare with to look for renames. The default is 200.\nmetric: optional custom function to use to determine the similarity between two files for rename detection.\nrecursion_limit: the upper limit on the number of merges of common ancestors to perform to try to build a new virtual merge base for the merge. The default is no limit. This field is only present on libgit2 versions newer than 0.24.0.\ndefault_driver: the merge driver to use if both sides have changed. This field is only present on libgit2 versions newer than 0.25.0.\nfile_favor: how to handle conflicting file contents for the text driver.\nMERGE_FILE_FAVOR_NORMAL: if both sides of the merge have changes to a section,  make a note of the conflict in the index which git checkout will use to create  a merge file, which the user can then reference to resolve the conflicts. This is  the default.\nMERGE_FILE_FAVOR_OURS: if both sides of the merge have changes to a section,  use the version in the \"ours\" side of the merge in the index.\nMERGE_FILE_FAVOR_THEIRS: if both sides of the merge have changes to a section,  use the version in the \"theirs\" side of the merge in the index.\nMERGE_FILE_FAVOR_UNION: if both sides of the merge have changes to a section,  include each unique line from both sides in the file which is put into the index.\nfile_flags: guidelines for merging files.\n\n\n\n\n\n"},{"title":"LibGit2.ProxyOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.ProxyOptions","category":"type","text":"LibGit2.ProxyOptions\n\nOptions for connecting through a proxy.\n\nMatches the git_proxy_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nproxytype: an enum for the type of proxy to use.  Defined in git_proxy_t.  The corresponding Julia enum is GIT_PROXY and has values:\nPROXY_NONE: do not attempt the connection through a proxy.\nPROXY_AUTO: attempt to figure out the proxy configuration from the git configuration.\nPROXY_SPECIFIED: connect using the URL given in the url field of this struct.\nDefault is to auto-detect the proxy type.\nurl: the URL of the proxy.\ncredential_cb: a pointer to a callback function which will be called if the remote requires authentication to connect.\ncertificate_cb: a pointer to a callback function which will be called if certificate verification fails. This lets the user decide whether or not to keep connecting. If the function returns 1, connecting will be allowed. If it returns 0, the connection will not be allowed. A negative value can be used to return errors.\npayload: the payload to be provided to the two callback functions.\n\nExamples\n\njulia> fo = LibGit2.FetchOptions(\n           proxy_opts = LibGit2.ProxyOptions(url = Cstring(\"https://my_proxy_url.com\")))\n\njulia> fetch(remote, \"master\", options=fo)\n\n\n\n\n\n"},{"title":"LibGit2.PushOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.PushOptions","category":"type","text":"LibGit2.PushOptions\n\nMatches the git_push_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nparallelism: if a pack file must be created, this variable sets the number of worker  threads which will be spawned by the packbuilder. If 0, the packbuilder will auto-set  the number of threads to use. The default is 1.\ncallbacks: the callbacks (e.g. for authentication with the remote) to use for the push.\nproxy_opts: only relevant if the LibGit2 version is greater than or equal to 0.25.0.  Sets options for using a proxy to communicate with a remote. See ProxyOptions  for more information.\ncustom_headers: only relevant if the LibGit2 version is greater than or equal to 0.24.0.  Extra headers needed for the push operation.\nremote_push_options: only relevant if the LibGit2 version is greater than or equal to 1.8.0.  \"Push options\" to deliver to the remote.\n\n\n\n\n\n"},{"title":"LibGit2.RebaseOperation","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.RebaseOperation","category":"type","text":"LibGit2.RebaseOperation\n\nDescribes a single instruction/operation to be performed during the rebase. Matches the git_rebase_operation struct.\n\nThe fields represent:\n\noptype: the type of rebase operation currently being performed. The options are:\nREBASE_OPERATION_PICK: cherry-pick the commit in question.\nREBASE_OPERATION_REWORD: cherry-pick the commit in question, but rewrite its message using the prompt.\nREBASE_OPERATION_EDIT: cherry-pick the commit in question, but allow the user to edit the commit's contents and its message.\nREBASE_OPERATION_SQUASH: squash the commit in question into the previous commit. The commit messages of the two commits will be merged.\nREBASE_OPERATION_FIXUP: squash the commit in question into the previous commit. Only the commit message of the previous commit will be used.\nREBASE_OPERATION_EXEC: do not cherry-pick a commit. Run a command and continue if the command exits successfully.\nid: the GitHash of the commit being worked on during this rebase step.\nexec: in case REBASE_OPERATION_EXEC is used, the command to run during this step (for instance, running the test suite after each commit).\n\n\n\n\n\n"},{"title":"LibGit2.RebaseOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.RebaseOptions","category":"type","text":"LibGit2.RebaseOptions\n\nMatches the git_rebase_options struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nquiet: inform other git clients helping with/working on the rebase that the rebase should be done \"quietly\". Used for interoperability. The default is 1.\ninmemory: start an in-memory rebase. Callers working on the rebase can go through its steps and commit any changes, but cannot rewind HEAD or update the repository. The workdir will not be modified. Only present on libgit2 versions newer than or equal to 0.24.0.\nrewrite_notes_ref: name of the reference to notes to use to rewrite the commit notes as the rebase is finished.\nmerge_opts: merge options controlling how the trees will be merged at each rebase step.  Only present on libgit2 versions newer than or equal to 0.24.0.\ncheckout_opts: checkout options for writing files when initializing the rebase, stepping through it, and aborting it. See CheckoutOptions for more information.\n\n\n\n\n\n"},{"title":"LibGit2.RemoteCallbacks","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.RemoteCallbacks","category":"type","text":"LibGit2.RemoteCallbacks\n\nCallback settings. Matches the git_remote_callbacks struct.\n\n\n\n\n\n"},{"title":"LibGit2.SignatureStruct","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.SignatureStruct","category":"type","text":"LibGit2.SignatureStruct\n\nAn action signature (e.g. for committers, taggers, etc). Matches the git_signature struct.\n\nThe fields represent:\n\nname: The full name of the committer or author of the commit.\nemail: The email at which the committer/author can be contacted.\nwhen: a TimeStruct indicating when the commit was  authored/committed into the repository.\n\n\n\n\n\n"},{"title":"LibGit2.StatusEntry","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.StatusEntry","category":"type","text":"LibGit2.StatusEntry\n\nProviding the differences between the file as it exists in HEAD and the index, and providing the differences between the index and the working directory. Matches the git_status_entry struct.\n\nThe fields represent:\n\nstatus: contains the status flags for the file, indicating if it is current, or has been changed in some way in the index or work tree.\nhead_to_index: a pointer to a DiffDelta which encapsulates the difference(s) between the file as it exists in HEAD and in the index.\nindex_to_workdir: a pointer to a DiffDelta which encapsulates the difference(s) between the file as it exists in the index and in the workdir.\n\n\n\n\n\n"},{"title":"LibGit2.StatusOptions","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.StatusOptions","category":"type","text":"LibGit2.StatusOptions\n\nOptions to control how git_status_foreach_ext() will issue callbacks. Matches the git_status_opt_t struct.\n\nThe fields represent:\n\nversion: version of the struct in use, in case this changes later. For now, always 1.\nshow: a flag for which files to examine and in which order. The default is Consts.STATUS_SHOW_INDEX_AND_WORKDIR.\nflags: flags for controlling any callbacks used in a status call.\npathspec: an array of paths to use for path-matching. The behavior of the path-matching will vary depending on the values of show and flags.\nThe baseline is the tree to be used for comparison to the working directory and index; defaults to HEAD.\n\n\n\n\n\n"},{"title":"LibGit2.StrArrayStruct","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.StrArrayStruct","category":"type","text":"LibGit2.StrArrayStruct\n\nA LibGit2 representation of an array of strings. Matches the git_strarray struct.\n\nWhen fetching data from LibGit2, a typical usage would look like:\n\nsa_ref = Ref(StrArrayStruct())\n@check ccall(..., (Ptr{StrArrayStruct},), sa_ref)\nres = collect(sa_ref[])\nfree(sa_ref)\n\nIn particular, note that LibGit2.free should be called afterward on the Ref object.\n\nConversely, when passing a vector of strings to LibGit2, it is generally simplest to rely on implicit conversion:\n\nstrs = String[...]\n@check ccall(..., (Ptr{StrArrayStruct},), strs)\n\nNote that no call to free is required as the data is allocated by Julia.\n\n\n\n\n\n"},{"title":"LibGit2.TimeStruct","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.TimeStruct","category":"type","text":"LibGit2.TimeStruct\n\nTime in a signature. Matches the git_time struct.\n\n\n\n\n\n"},{"title":"LibGit2.addfile","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.addfile","category":"function","text":"addfile(cfg::GitConfig, path::AbstractString,\n        level::Consts.GIT_CONFIG=Consts.CONFIG_LEVEL_APP,\n        repo::Union{GitRepo, Nothing} = nothing,\n        force::Bool=false)\n\nAdd an existing git configuration file located at path to the current GitConfig cfg. If the file does not exist, it will be created.\n\nlevel sets the git configuration priority level and is determined by\n\nConsts.GIT_CONFIG.\n\nrepo is an optional repository to allow parsing of conditional includes.\nIf force is false and a configuration for the given priority level already exists,\n\naddfile will error. If force is true, the existing configuration will be replaced by the one in the file at path.\n\n\n\n\n\n"},{"title":"LibGit2.add!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.add!","category":"function","text":"add!(repo::GitRepo, files::AbstractString...; flags::Cuint = Consts.INDEX_ADD_DEFAULT)\nadd!(idx::GitIndex, files::AbstractString...; flags::Cuint = Consts.INDEX_ADD_DEFAULT)\n\nAdd all the files with paths specified by files to the index idx (or the index of the repo). If the file already exists, the index entry will be updated. If the file does not exist already, it will be newly added into the index. files may contain glob patterns which will be expanded and any matching files will be added (unless INDEX_ADD_DISABLE_PATHSPEC_MATCH is set, see below). If a file has been ignored (in .gitignore or in the config), it will not be added, unless it is already being tracked in the index, in which case it will be updated. The keyword argument flags is a set of bit-flags which control the behavior with respect to ignored files:\n\nConsts.INDEX_ADD_DEFAULT - default, described above.\nConsts.INDEX_ADD_FORCE - disregard the existing ignore rules and force addition of the file to the index even if it is already ignored.\nConsts.INDEX_ADD_CHECK_PATHSPEC - cannot be used at the same time as INDEX_ADD_FORCE. Check that each file in files which exists on disk is not in the ignore list. If one of the files is ignored, the function will return EINVALIDSPEC.\nConsts.INDEX_ADD_DISABLE_PATHSPEC_MATCH - turn off glob matching, and only add files to the index which exactly match the paths specified in files.\n\n\n\n\n\n"},{"title":"LibGit2.add_fetch!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.add_fetch!","category":"function","text":"add_fetch!(repo::GitRepo, rmt::GitRemote, fetch_spec::String)\n\nAdd a fetch refspec for the specified rmt. This refspec will contain information about which branch(es) to fetch from.\n\nExamples\n\njulia> LibGit2.add_fetch!(repo, remote, \"upstream\");\n\njulia> LibGit2.fetch_refspecs(remote)\nString[\"+refs/heads/*:refs/remotes/upstream/*\"]\n\n\n\n\n\n"},{"title":"LibGit2.add_push!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.add_push!","category":"function","text":"add_push!(repo::GitRepo, rmt::GitRemote, push_spec::String)\n\nAdd a push refspec for the specified rmt. This refspec will contain information about which branch(es) to push to.\n\nExamples\n\njulia> LibGit2.add_push!(repo, remote, \"refs/heads/master\");\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, branch);\n\njulia> LibGit2.push_refspecs(remote)\nString[\"refs/heads/master\"]\n\nnote: Note\nYou may need to close and reopen the GitRemote in question after updating its push refspecs in order for the change to take effect and for calls to push to work.\n\n\n\n\n\n"},{"title":"LibGit2.addblob!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.addblob!","category":"function","text":"LibGit2.addblob!(repo::GitRepo, path::AbstractString)\n\nRead the file at path and adds it to the object database of repo as a loose blob. Return the GitHash of the resulting blob.\n\nExamples\n\nhash_str = string(commit_oid)\nblob_file = joinpath(repo_path, \".git\", \"objects\", hash_str[1:2], hash_str[3:end])\nid = LibGit2.addblob!(repo, blob_file)\n\n\n\n\n\n"},{"title":"LibGit2.author","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.author","category":"function","text":"author(c::GitCommit)\n\nReturn the Signature of the author of the commit c. The author is the person who made changes to the relevant file(s). See also committer.\n\n\n\n\n\n"},{"title":"LibGit2.authors","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.authors","category":"function","text":"authors(repo::GitRepo)::Vector{Signature}\n\nReturn all authors of commits to the repo repository.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nrepo_file = open(joinpath(repo_path, test_file), \"a\")\n\nprintln(repo_file, commit_msg)\nflush(repo_file)\nLibGit2.add!(repo, test_file)\nsig = LibGit2.Signature(\"TEST\", \"TEST@TEST.COM\", round(time(), 0), 0)\ncommit_oid1 = LibGit2.commit(repo, \"commit1\"; author=sig, committer=sig)\nprintln(repo_file, randstring(10))\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid2 = LibGit2.commit(repo, \"commit2\"; author=sig, committer=sig)\n\n# will be a Vector of [sig, sig]\nauths = LibGit2.authors(repo)\n\n\n\n\n\n"},{"title":"LibGit2.branch","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.branch","category":"function","text":"branch(repo::GitRepo)\n\nEquivalent to git branch. Create a new branch from the current HEAD.\n\n\n\n\n\n"},{"title":"LibGit2.branch!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.branch!","category":"function","text":"branch!(repo::GitRepo, branch_name::AbstractString, commit::AbstractString=\"\"; kwargs...)\n\nCheckout a new git branch in the repo repository. commit is the GitHash, in string form, which will be the start of the new branch. If commit is an empty string, the current HEAD will be used.\n\nThe keyword arguments are:\n\ntrack::AbstractString=\"\": the name of the remote branch this new branch should track, if any. If empty (the default), no remote branch will be tracked.\nforce::Bool=false: if true, branch creation will be forced.\nset_head::Bool=true: if true, after the branch creation finishes the branch head will be set as the HEAD of repo.\n\nEquivalent to git checkout [-b|-B] <branch_name> [<commit>] [--track <track>].\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.branch!(repo, \"new_branch\", set_head=false)\n\n\n\n\n\n"},{"title":"LibGit2.checkout!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.checkout!","category":"function","text":"checkout!(repo::GitRepo, commit::AbstractString=\"\"; force::Bool=true)\n\nEquivalent to git checkout [-f] --detach <commit>. Checkout the git commit commit (a GitHash in string form) in repo. If force is true, force the checkout and discard any current changes. Note that this detaches the current HEAD.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nopen(joinpath(LibGit2.path(repo), \"file1\"), \"w\") do f\n    write(f, \"111\n\")\nend\nLibGit2.add!(repo, \"file1\")\ncommit_oid = LibGit2.commit(repo, \"add file1\")\nopen(joinpath(LibGit2.path(repo), \"file1\"), \"w\") do f\n    write(f, \"112\n\")\nend\n# would fail without the force=true\n# since there are modifications to the file\nLibGit2.checkout!(repo, string(commit_oid), force=true)\n\n\n\n\n\n"},{"title":"LibGit2.clone","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.clone","category":"function","text":"clone(repo_url::AbstractString, repo_path::AbstractString, clone_opts::CloneOptions)\n\nClone the remote repository at repo_url (which can be a remote URL or a path on the local filesystem) to repo_path (which must be a path on the local filesystem). Options for the clone, such as whether to perform a bare clone or not, are set by CloneOptions.\n\nExamples\n\nrepo_url = \"https://github.com/JuliaLang/Example.jl\"\nrepo = LibGit2.clone(repo_url, \"/home/me/projects/Example\")\n\n\n\n\n\nclone(repo_url::AbstractString, repo_path::AbstractString; kwargs...)\n\nClone a remote repository located at repo_url to the local filesystem location repo_path.\n\nThe keyword arguments are:\n\nbranch::AbstractString=\"\": which branch of the remote to clone, if not the default repository branch (usually master).\nisbare::Bool=false: if true, clone the remote as a bare repository, which will make repo_path itself the git directory instead of repo_path/.git. This means that a working tree cannot be checked out. Plays the role of the git CLI argument --bare.\nremote_cb::Ptr{Cvoid}=C_NULL: a callback which will be used to create the remote before it is cloned. If C_NULL (the default), no attempt will be made to create the remote - it will be assumed to already exist.\ndepth::Integer=0: create a shallow clone with a history truncated to the specified number of commits. 0 indicates a full clone (the default). Use Consts.FETCH_DEPTH_UNSHALLOW to fetch all missing data from a shallow clone. Note: shallow clones are, at the time of writing, only supported for network protocols (http, https, git, ssh), not for local filesystem paths. (https://github.com/libgit2/libgit2/issues/6634)\ncredentials::Creds=nothing: provides credentials and/or settings when authenticating against a private repository.\ncallbacks::Callbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git clone [-b <branch>] [--bare] [--depth <depth>] <repo_url> <repo_path>.\n\nExamples\n\nrepo_url = \"https://github.com/JuliaLang/Example.jl\"\nrepo1 = LibGit2.clone(repo_url, \"test_path\")\nrepo2 = LibGit2.clone(repo_url, \"test_path\", isbare=true)\njulia_url = \"https://github.com/JuliaLang/julia\"\njulia_repo = LibGit2.clone(julia_url, \"julia_path\", branch=\"release-0.6\")\n# Shallow clone with only the most recent commit\nshallow_repo = LibGit2.clone(repo_url, \"shallow_path\", depth=1)\n\n\n\n\n\n"},{"title":"LibGit2.commit","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.commit","category":"function","text":"commit(repo::GitRepo, msg::AbstractString; kwargs...)::GitHash\n\nWrapper around git_commit_create. Create a commit in the repository repo. msg is the commit message. Return the OID of the new commit.\n\nThe keyword arguments are:\n\nrefname::AbstractString=Consts.HEAD_FILE: if not NULL, the name of the reference to update to point to the new commit. For example, \"HEAD\" will update the HEAD of the current branch. If the reference does not yet exist, it will be created.\nauthor::Signature = Signature(repo) is a Signature containing information about the person who authored the commit.\ncommitter::Signature = Signature(repo) is a Signature containing information about the person who committed the commit to the repository. Not necessarily the same as author, for instance if author emailed a patch to committer who committed it.\ntree_id::GitHash = GitHash() is a git tree to use to create the commit, showing its ancestry and relationship with any other history. tree must belong to repo.\nparent_ids::Vector{GitHash}=GitHash[] is a list of commits by GitHash to use as parent commits for the new one, and may be empty. A commit might have multiple parents if it is a merge commit, for example.\n\n\n\n\n\nLibGit2.commit(rb::GitRebase, sig::GitSignature)\n\nCommit the current patch to the rebase rb, using sig as the committer. Is silent if the commit has already been applied.\n\n\n\n\n\n"},{"title":"LibGit2.committer","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.committer","category":"function","text":"committer(c::GitCommit)\n\nReturn the Signature of the committer of the commit c. The committer is the person who committed the changes originally authored by the author, but need not be the same as the author, for example, if the author emailed a patch to a committer who committed it.\n\n\n\n\n\n"},{"title":"LibGit2.count","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.count","category":"function","text":"LibGit2.count(f::Function, walker::GitRevWalker; oid::GitHash=GitHash(), by::Cint=Consts.SORT_NONE, rev::Bool=false)\n\nUsing the GitRevWalker walker to \"walk\" over every commit in the repository's history, find the number of commits which return true when f is applied to them. The keyword arguments are:     * oid: The GitHash of the commit to begin the walk from. The default is to use       push_head! and therefore the HEAD commit and all its ancestors.     * by: The sorting method. The default is not to sort. Other options are to sort by       topology (LibGit2.Consts.SORT_TOPOLOGICAL), to sort forwards in time       (LibGit2.Consts.SORT_TIME, most ancient first) or to sort backwards in time       (LibGit2.Consts.SORT_REVERSE, most recent first).     * rev: Whether to reverse the sorted order (for instance, if topological sorting is used).\n\nExamples\n\ncnt = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.count((oid, repo)->(oid == commit_oid1), walker, oid=commit_oid1, by=LibGit2.Consts.SORT_TIME)\nend\n\nLibGit2.count finds the number of commits along the walk with a certain GitHash commit_oid1, starting the walk from that commit and moving forwards in time from it. Since the GitHash is unique to a commit, cnt will be 1.\n\n\n\n\n\n"},{"title":"LibGit2.counthunks","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.counthunks","category":"function","text":"counthunks(blame::GitBlame)\n\nReturn the number of distinct \"hunks\" with a file. A hunk may contain multiple lines. A hunk is usually a piece of a file that was added/changed/removed together, for example, a function added to a source file or an inner loop that was optimized out of that function later.\n\n\n\n\n\n"},{"title":"LibGit2.create_branch","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.create_branch","category":"function","text":"LibGit2.create_branch(repo::GitRepo, bname::AbstractString, commit_obj::GitCommit; force::Bool=false)\n\nCreate a new branch in the repository repo with name bname, which points to commit commit_obj (which has to be part of repo). If force is true, overwrite an existing branch named bname if it exists. If force is false and a branch already exists named bname, this function will throw an error.\n\n\n\n\n\n"},{"title":"LibGit2.credentials_callback","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.credentials_callback","category":"function","text":"credential_callback(...)::Cint\n\nA LibGit2 credential callback function which provides different credential acquisition functionality w.r.t. a connection protocol. The payload_ptr is required to contain a LibGit2.CredentialPayload object which will keep track of state and settings.\n\nThe allowed_types contains a bitmask of LibGit2.Consts.GIT_CREDTYPE values specifying which authentication methods should be attempted.\n\nCredential authentication is done in the following order (if supported):\n\nSSH agent\nSSH private/public key pair\nUsername/password plain text\n\nIf a user is presented with a credential prompt they can abort the prompt by typing ^D (pressing the control key together with the d key).\n\nNote: Due to the specifics of the libgit2 authentication procedure, when authentication fails, this function is called again without any indication whether authentication was successful or not. To avoid an infinite loop from repeatedly using the same faulty credentials, we will keep track of state using the payload.\n\nFor addition details see the LibGit2 guide on authenticating against a server.\n\n\n\n\n\n"},{"title":"LibGit2.credentials_cb","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.credentials_cb","category":"function","text":"C function pointer for credentials_callback\n\n\n\n\n\n"},{"title":"LibGit2.default_signature","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.default_signature","category":"function","text":"Return signature object. Free it after use.\n\n\n\n\n\n"},{"title":"LibGit2.delete_branch","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.delete_branch","category":"function","text":"LibGit2.delete_branch(branch::GitReference)\n\nDelete the branch pointed to by branch.\n\n\n\n\n\n"},{"title":"LibGit2.diff_files","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.diff_files","category":"function","text":"diff_files(repo::GitRepo, branch1::AbstractString, branch2::AbstractString; kwarg...)::Vector{AbstractString}\n\nShow which files have changed in the git repository repo between branches branch1 and branch2.\n\nThe keyword argument is:\n\nfilter::Set{Consts.DELTA_STATUS}=Set([Consts.DELTA_ADDED, Consts.DELTA_MODIFIED, Consts.DELTA_DELETED])), and it sets options for the diff. The default is to show files added, modified, or deleted.\n\nReturn only the names of the files which have changed, not their contents.\n\nExamples\n\nLibGit2.branch!(repo, \"branch/a\")\nLibGit2.branch!(repo, \"branch/b\")\n# add a file to repo\nopen(joinpath(LibGit2.path(repo),\"file\"),\"w\") do f\n    write(f, \"hello repo\n\")\nend\nLibGit2.add!(repo, \"file\")\nLibGit2.commit(repo, \"add file\")\n# returns [\"file\"]\nfilt = Set([LibGit2.Consts.DELTA_ADDED])\nfiles = LibGit2.diff_files(repo, \"branch/a\", \"branch/b\", filter=filt)\n# returns [] because existing files weren't modified\nfilt = Set([LibGit2.Consts.DELTA_MODIFIED])\nfiles = LibGit2.diff_files(repo, \"branch/a\", \"branch/b\", filter=filt)\n\nEquivalent to git diff --name-only --diff-filter=<filter> <branch1> <branch2>.\n\n\n\n\n\n"},{"title":"LibGit2.entryid","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.entryid","category":"function","text":"entryid(te::GitTreeEntry)\n\nReturn the GitHash of the object to which te refers.\n\n\n\n\n\n"},{"title":"LibGit2.entrytype","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.entrytype","category":"function","text":"entrytype(te::GitTreeEntry)\n\nReturn the type of the object to which te refers. The result will be one of the types which objtype returns, e.g. a GitTree or GitBlob.\n\n\n\n\n\n"},{"title":"LibGit2.fetch","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.fetch","category":"function","text":"fetch(rmt::GitRemote, refspecs; options::FetchOptions=FetchOptions(), msg=\"\")\n\nFetch from the specified rmt remote git repository, using refspecs to determine which remote branch(es) to fetch. The keyword arguments are:\n\noptions: determines the options for the fetch, e.g. whether to prune afterwards. See FetchOptions for more information.\nmsg: a message to insert into the reflogs.\n\n\n\n\n\nfetch(repo::GitRepo; kwargs...)\n\nFetches updates from an upstream of the repository repo.\n\nThe keyword arguments are:\n\nremote::AbstractString=\"origin\": which remote, specified by name, of repo to fetch from. If this is empty, the URL will be used to construct an anonymous remote.\nremoteurl::AbstractString=\"\": the URL of remote. If not specified, will be assumed based on the given name of remote.\nrefspecs=AbstractString[]: determines properties of the fetch.\ndepth::Integer=0: limit fetching to the specified number of commits from the tip of each remote branch. 0 indicates a full fetch (the default). Use Consts.FETCH_DEPTH_UNSHALLOW to fetch all missing data from a shallow clone. Note: depth is, at the time of writing, only supported for network protocols (http, https, git, ssh), not for local filesystem paths. (https://github.com/libgit2/libgit2/issues/6634)\ncredentials=nothing: provides credentials and/or settings when authenticating against a private remote.\ncallbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git fetch [--depth <depth>] [<remoteurl>|<repo>] [<refspecs>].\n\n\n\n\n\n"},{"title":"LibGit2.fetchheads","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.fetchheads","category":"function","text":"fetchheads(repo::GitRepo)::Vector{FetchHead}\n\nReturn the list of all the fetch heads for repo, each represented as a FetchHead, including their names, URLs, and merge statuses.\n\nExamples\n\njulia> fetch_heads = LibGit2.fetchheads(repo);\n\njulia> fetch_heads[1].name\n\"refs/heads/master\"\n\njulia> fetch_heads[1].ismerge\ntrue\n\njulia> fetch_heads[2].name\n\"refs/heads/test_branch\"\n\njulia> fetch_heads[2].ismerge\nfalse\n\n\n\n\n\n"},{"title":"LibGit2.fetch_refspecs","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.fetch_refspecs","category":"function","text":"fetch_refspecs(rmt::GitRemote)::Vector{String}\n\nGet the fetch refspecs for the specified rmt. These refspecs contain information about which branch(es) to fetch from.\n\nExamples\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.add_fetch!(repo, remote, \"upstream\");\n\njulia> LibGit2.fetch_refspecs(remote)\nString[\"+refs/heads/*:refs/remotes/upstream/*\"]\n\n\n\n\n\n"},{"title":"LibGit2.fetchhead_foreach_cb","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.fetchhead_foreach_cb","category":"function","text":"C function pointer for fetchhead_foreach_callback\n\n\n\n\n\n"},{"title":"LibGit2.merge_base","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.merge_base","category":"function","text":"merge_base(repo::GitRepo, one::AbstractString, two::AbstractString)::GitHash\n\nFind a merge base (a common ancestor) between the commits one and two. one and two may both be in string form. Return the GitHash of the merge base.\n\n\n\n\n\n"},{"title":"LibGit2.merge!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo}","category":"method","text":"merge!(repo::GitRepo; kwargs...)::Bool\n\nPerform a git merge on the repository repo, merging commits with diverging history into the current branch. Return true if the merge succeeded, false if not.\n\nThe keyword arguments are:\n\ncommittish::AbstractString=\"\": Merge the named commit(s) in committish.\nbranch::AbstractString=\"\": Merge the branch branch and all its commits since it diverged from the current branch.\nfastforward::Bool=false: If fastforward is true, only merge if the merge is a fast-forward (the current branch head is an ancestor of the commits to be merged), otherwise refuse to merge and return false. This is equivalent to the git CLI option --ff-only.\nmerge_opts::MergeOptions=MergeOptions(): merge_opts specifies options for the merge, such as merge strategy in case of conflicts.\ncheckout_opts::CheckoutOptions=CheckoutOptions(): checkout_opts specifies options for the checkout step.\n\nEquivalent to git merge [--ff-only] [<committish> | <branch>].\n\nnote: Note\nIf you specify a branch, this must be done in reference format, since the string will be turned into a GitReference. For example, if you wanted to merge branch branch_a, you would call merge!(repo, branch=\"refs/heads/branch_a\").\n\n\n\n\n\n"},{"title":"LibGit2.merge!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo, Vector{LibGit2.GitAnnotated}}","category":"method","text":"merge!(repo::GitRepo, anns::Vector{GitAnnotated}; kwargs...)::Bool\n\nMerge changes from the annotated commits (captured as GitAnnotated objects) anns into the HEAD of the repository repo. The keyword arguments are:\n\nmerge_opts::MergeOptions = MergeOptions(): options for how to perform the merge, including whether fastforwarding is allowed. See MergeOptions for more information.\ncheckout_opts::CheckoutOptions = CheckoutOptions(): options for how to perform the checkout. See CheckoutOptions for more information.\n\nanns may refer to remote or local branch heads. Return true if the merge is successful, otherwise return false (for instance, if no merge is possible because the branches have no common ancestor).\n\nExamples\n\nupst_ann = LibGit2.GitAnnotated(repo, \"branch/a\")\n\n# merge the branch in\nLibGit2.merge!(repo, [upst_ann])\n\n\n\n\n\n"},{"title":"LibGit2.merge!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.merge!-Tuple{GitRepo, Vector{LibGit2.GitAnnotated}, Bool}","category":"method","text":"merge!(repo::GitRepo, anns::Vector{GitAnnotated}, fastforward::Bool; kwargs...)::Bool\n\nMerge changes from the annotated commits (captured as GitAnnotated objects) anns into the HEAD of the repository repo. If fastforward is true, only a fastforward merge is allowed. In this case, if conflicts occur, the merge will fail. Otherwise, if fastforward is false, the merge may produce a conflict file which the user will need to resolve.\n\nThe keyword arguments are:\n\nmerge_opts::MergeOptions = MergeOptions(): options for how to perform the merge, including whether fastforwarding is allowed. See MergeOptions for more information.\ncheckout_opts::CheckoutOptions = CheckoutOptions(): options for how to perform the checkout. See CheckoutOptions for more information.\n\nanns may refer to remote or local branch heads. Return true if the merge is successful, otherwise return false (for instance, if no merge is possible because the branches have no common ancestor).\n\nExamples\n\nupst_ann_1 = LibGit2.GitAnnotated(repo, \"branch/a\")\n\n# merge the branch in, fastforward\nLibGit2.merge!(repo, [upst_ann_1], true)\n\n# merge conflicts!\nupst_ann_2 = LibGit2.GitAnnotated(repo, \"branch/b\")\n# merge the branch in, try to fastforward\nLibGit2.merge!(repo, [upst_ann_2], true) # will return false\nLibGit2.merge!(repo, [upst_ann_2], false) # will return true\n\n\n\n\n\n"},{"title":"LibGit2.ffmerge!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.ffmerge!","category":"function","text":"ffmerge!(repo::GitRepo, ann::GitAnnotated)\n\nFastforward merge changes into current HEAD. This is only possible if the commit referred to by ann is descended from the current HEAD (e.g. if pulling changes from a remote branch which is simply ahead of the local branch tip).\n\n\n\n\n\n"},{"title":"LibGit2.fullname","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.fullname","category":"function","text":"LibGit2.fullname(ref::GitReference)\n\nReturn the name of the reference pointed to by the symbolic reference ref. If ref is not a symbolic reference, return an empty string.\n\n\n\n\n\n"},{"title":"LibGit2.features","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.features","category":"function","text":"features()\n\nReturn a list of git features the current version of libgit2 supports, such as threading or using HTTPS or SSH.\n\n\n\n\n\n"},{"title":"LibGit2.filename","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.filename","category":"function","text":"filename(te::GitTreeEntry)\n\nReturn the filename of the object on disk to which te refers.\n\n\n\n\n\n"},{"title":"LibGit2.filemode","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.filemode","category":"function","text":"filemode(te::GitTreeEntry)::Cint\n\nReturn the UNIX filemode of the object on disk to which te refers as an integer.\n\n\n\n\n\n"},{"title":"LibGit2.gitdir","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.gitdir","category":"function","text":"LibGit2.gitdir(repo::GitRepo)\n\nReturn the location of the \"git\" files of repo:\n\nfor normal repositories, this is the location of the .git folder.\nfor bare repositories, this is the location of the repository itself.\n\nSee also workdir, path.\n\n\n\n\n\n"},{"title":"LibGit2.git_url","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.git_url","category":"function","text":"LibGit2.git_url(; kwargs...)::String\n\nCreate a string based upon the URL components provided. When the scheme keyword is not provided the URL produced will use the alternative scp-like syntax.\n\nKeywords\n\nscheme::AbstractString=\"\": the URL scheme which identifies the protocol to be used. For HTTP use \"http\", SSH use \"ssh\", etc. When scheme is not provided the output format will be \"ssh\" but using the scp-like syntax.\nusername::AbstractString=\"\": the username to use in the output if provided.\npassword::AbstractString=\"\": the password to use in the output if provided.\nhost::AbstractString=\"\": the hostname to use in the output. A hostname is required to be specified.\nport::Union{AbstractString,Integer}=\"\": the port number to use in the output if provided. Cannot be specified when using the scp-like syntax.\npath::AbstractString=\"\": the path to use in the output if provided.\n\nwarning: Warning\nAvoid using passwords in URLs. Unlike the credential objects, Julia is not able to securely zero or destroy the sensitive data after use and the password may remain in memory; possibly to be exposed by an uninitialized memory.\n\nExamples\n\njulia> LibGit2.git_url(username=\"git\", host=\"github.com\", path=\"JuliaLang/julia.git\")\n\"git@github.com:JuliaLang/julia.git\"\n\njulia> LibGit2.git_url(scheme=\"https\", host=\"github.com\", path=\"/JuliaLang/julia.git\")\n\"https://github.com/JuliaLang/julia.git\"\n\njulia> LibGit2.git_url(scheme=\"ssh\", username=\"git\", host=\"github.com\", port=2222, path=\"JuliaLang/julia.git\")\n\"ssh://git@github.com:2222/JuliaLang/julia.git\"\n\n\n\n\n\n"},{"title":"LibGit2.@githash_str","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.@githash_str","category":"macro","text":"@githash_str -> AbstractGitHash\n\nConstruct a git hash object from the given string, returning a GitShortHash if the string is shorter than 40 hexadecimal digits, otherwise a GitHash.\n\nExamples\n\njulia> LibGit2.githash\"d114feb74ce633\"\nGitShortHash(\"d114feb74ce633\")\n\njulia> LibGit2.githash\"d114feb74ce63307afe878a5228ad014e0289a85\"\nGitHash(\"d114feb74ce63307afe878a5228ad014e0289a85\")\n\n\n\n\n\n"},{"title":"LibGit2.head","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.head","category":"function","text":"LibGit2.head(repo::GitRepo)::GitReference\n\nReturn a GitReference to the current HEAD of repo.\n\n\n\n\n\nhead(pkg::AbstractString)::String\n\nReturn current HEAD GitHash of the pkg repo as a string.\n\n\n\n\n\n"},{"title":"LibGit2.head!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.head!","category":"function","text":"LibGit2.head!(repo::GitRepo, ref::GitReference)::GitReference\n\nSet the HEAD of repo to the object pointed to by ref.\n\n\n\n\n\n"},{"title":"LibGit2.head_oid","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.head_oid","category":"function","text":"LibGit2.head_oid(repo::GitRepo)::GitHash\n\nLookup the object id of the current HEAD of git repository repo.\n\n\n\n\n\n"},{"title":"LibGit2.headname","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.headname","category":"function","text":"LibGit2.headname(repo::GitRepo)\n\nLookup the name of the current HEAD of git repository repo. If repo is currently detached, return the name of the HEAD it's detached from.\n\n\n\n\n\n"},{"title":"LibGit2.init","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.init","category":"function","text":"LibGit2.init(path::AbstractString, bare::Bool=false)::GitRepo\n\nOpen a new git repository at path. If bare is false, the working tree will be created in path/.git. If bare is true, no working directory will be created.\n\n\n\n\n\n"},{"title":"LibGit2.is_ancestor_of","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.is_ancestor_of","category":"function","text":"is_ancestor_of(a::AbstractString, b::AbstractString, repo::GitRepo)::Bool\n\nReturn true if a, a GitHash in string form, is an ancestor of b, a GitHash in string form.\n\nExamples\n\njulia> repo = GitRepo(repo_path);\n\njulia> LibGit2.add!(repo, test_file1);\n\njulia> commit_oid1 = LibGit2.commit(repo, \"commit1\");\n\njulia> LibGit2.add!(repo, test_file2);\n\njulia> commit_oid2 = LibGit2.commit(repo, \"commit2\");\n\njulia> LibGit2.is_ancestor_of(string(commit_oid1), string(commit_oid2), repo)\ntrue\n\n\n\n\n\n"},{"title":"LibGit2.isbinary","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isbinary","category":"function","text":"isbinary(blob::GitBlob)::Bool\n\nUse a heuristic to guess if a file is binary: searching for NULL bytes and looking for a reasonable ratio of printable to non-printable characters among the first 8000 bytes.\n\n\n\n\n\n"},{"title":"LibGit2.iscommit","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.iscommit","category":"function","text":"iscommit(id::AbstractString, repo::GitRepo)::Bool\n\nCheck if commit id (which is a GitHash in string form) is in the repository.\n\nExamples\n\njulia> repo = GitRepo(repo_path);\n\njulia> LibGit2.add!(repo, test_file);\n\njulia> commit_oid = LibGit2.commit(repo, \"add test_file\");\n\njulia> LibGit2.iscommit(string(commit_oid), repo)\ntrue\n\n\n\n\n\n"},{"title":"LibGit2.isdiff","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isdiff","category":"function","text":"LibGit2.isdiff(repo::GitRepo, treeish::AbstractString, pathspecs::AbstractString=\"\"; cached::Bool=false)\n\nChecks if there are any differences between the tree specified by treeish and the tracked files in the working tree (if cached=false) or the index (if cached=true). pathspecs are the specifications for options for the diff.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.isdiff(repo, \"HEAD\") # should be false\nopen(joinpath(repo_path, new_file), \"a\") do f\n    println(f, \"here's my cool new file\")\nend\nLibGit2.isdiff(repo, \"HEAD\") # now true\n\nEquivalent to git diff-index <treeish> [-- <pathspecs>].\n\n\n\n\n\n"},{"title":"LibGit2.isdirty","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isdirty","category":"function","text":"LibGit2.isdirty(repo::GitRepo, pathspecs::AbstractString=\"\"; cached::Bool=false)::Bool\n\nCheck if there have been any changes to tracked files in the working tree (if cached=false) or the index (if cached=true). pathspecs are the specifications for options for the diff.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nLibGit2.isdirty(repo) # should be false\nopen(joinpath(repo_path, new_file), \"a\") do f\n    println(f, \"here's my cool new file\")\nend\nLibGit2.isdirty(repo) # now true\nLibGit2.isdirty(repo, new_file) # now true\n\nEquivalent to git diff-index HEAD [-- <pathspecs>].\n\n\n\n\n\n"},{"title":"LibGit2.isorphan","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isorphan","category":"function","text":"LibGit2.isorphan(repo::GitRepo)\n\nCheck if the current branch is an \"orphan\" branch, i.e. has no commits. The first commit to this branch will have no parents.\n\n\n\n\n\n"},{"title":"LibGit2.isset","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isset","category":"function","text":"isset(val::Integer, flag::Integer)\n\nTest whether the bits of val indexed by flag are set (1) or unset (0).\n\n\n\n\n\n"},{"title":"LibGit2.iszero","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.iszero","category":"function","text":"iszero(id::GitHash)::Bool\n\nDetermine whether all hexadecimal digits of the given GitHash are zero.\n\n\n\n\n\n"},{"title":"LibGit2.lookup_branch","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.lookup_branch","category":"function","text":"lookup_branch(repo::GitRepo, branch_name::AbstractString, remote::Bool=false)::Union{GitReference, Nothing}\n\nDetermine if the branch specified by branch_name exists in the repository repo. If remote is true, repo is assumed to be a remote git repository. Otherwise, it is part of the local filesystem.\n\nReturn either a GitReference to the requested branch if it exists, or nothing if not.\n\n\n\n\n\n"},{"title":"LibGit2.map","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.map","category":"function","text":"LibGit2.map(f::Function, walker::GitRevWalker; oid::GitHash=GitHash(), range::AbstractString=\"\", by::Cint=Consts.SORT_NONE, rev::Bool=false)\n\nUsing the GitRevWalker walker to \"walk\" over every commit in the repository's history, apply f to each commit in the walk. The keyword arguments are:     * oid: The GitHash of the commit to begin the walk from. The default is to use       push_head! and therefore the HEAD commit and all its ancestors.     * range: A range of GitHashs in the format oid1..oid2. f will be       applied to all commits between the two.     * by: The sorting method. The default is not to sort. Other options are to sort by       topology (LibGit2.Consts.SORT_TOPOLOGICAL), to sort forwards in time       (LibGit2.Consts.SORT_TIME, most ancient first) or to sort backwards in time       (LibGit2.Consts.SORT_REVERSE, most recent first).     * rev: Whether to reverse the sorted order (for instance, if topological sorting is used).\n\nExamples\n\noids = LibGit2.with(LibGit2.GitRevWalker(repo)) do walker\n    LibGit2.map((oid, repo)->string(oid), walker, by=LibGit2.Consts.SORT_TIME)\nend\n\nHere, LibGit2.map visits each commit using the GitRevWalker and finds its GitHash.\n\n\n\n\n\n"},{"title":"LibGit2.mirror_callback","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.mirror_callback","category":"function","text":"Mirror callback function\n\nFunction sets +refs/*:refs/* refspecs and mirror flag for remote reference.\n\n\n\n\n\n"},{"title":"LibGit2.mirror_cb","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.mirror_cb","category":"function","text":"C function pointer for mirror_callback\n\n\n\n\n\n"},{"title":"LibGit2.message","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.message","category":"function","text":"message(c::GitCommit, raw::Bool=false)\n\nReturn the commit message describing the changes made in commit c. If raw is false, return a slightly \"cleaned up\" message (which has any leading newlines removed). If raw is true, the message is not stripped of any such newlines.\n\n\n\n\n\n"},{"title":"LibGit2.merge_analysis","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.merge_analysis","category":"function","text":"merge_analysis(repo::GitRepo, anns::Vector{GitAnnotated}) -> analysis, preference\n\nRun analysis on the branches pointed to by the annotated branch tips anns and determine under what circumstances they can be merged. For instance, if anns[1] is simply an ancestor of ann[2], then merge_analysis will report that a fast-forward merge is possible.\n\nReturn two outputs, analysis and preference. analysis has several possible values:     * MERGE_ANALYSIS_NONE: it is not possible to merge the elements of anns.     * MERGE_ANALYSIS_NORMAL: a regular merge, when HEAD and the commits that the       user wishes to merge have all diverged from a common ancestor. In this case the       changes have to be resolved and conflicts may occur.     * MERGE_ANALYSIS_UP_TO_DATE: all the input commits the user wishes to merge can       be reached from HEAD, so no merge needs to be performed.     * MERGE_ANALYSIS_FASTFORWARD: the input commit is a descendant of HEAD and so no       merge needs to be performed - instead, the user can simply checkout the       input commit(s).     * MERGE_ANALYSIS_UNBORN: the HEAD of the repository refers to a commit which does not       exist. It is not possible to merge, but it may be possible to checkout the input       commits. preference also has several possible values:     * MERGE_PREFERENCE_NONE: the user has no preference.     * MERGE_PREFERENCE_NO_FASTFORWARD: do not allow any fast-forward merges.     * MERGE_PREFERENCE_FASTFORWARD_ONLY: allow only fast-forward merges and no       other type (which may introduce conflicts). preference can be controlled through the repository or global git configuration.\n\n\n\n\n\n"},{"title":"LibGit2.name","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.name","category":"function","text":"LibGit2.name(ref::GitReference)\n\nReturn the full name of ref.\n\n\n\n\n\nname(rmt::GitRemote)\n\nGet the name of a remote repository, for instance \"origin\". If the remote is anonymous (see GitRemoteAnon) the name will be an empty string \"\".\n\nExamples\n\njulia> repo_url = \"https://github.com/JuliaLang/Example.jl\";\n\njulia> repo = LibGit2.clone(cache_repo, \"test_directory\");\n\njulia> remote = LibGit2.GitRemote(repo, \"origin\", repo_url);\n\njulia> name(remote)\n\"origin\"\n\n\n\n\n\nLibGit2.name(tag::GitTag)\n\nThe name of tag (e.g. \"v0.5\").\n\n\n\n\n\n"},{"title":"LibGit2.need_update","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.need_update","category":"function","text":"need_update(repo::GitRepo)\n\nEquivalent to git update-index. Return true if repo needs updating.\n\n\n\n\n\n"},{"title":"LibGit2.objtype","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.objtype","category":"function","text":"objtype(obj_type::Consts.OBJECT)\n\nReturn the type corresponding to the enum value.\n\n\n\n\n\n"},{"title":"LibGit2.path","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.path","category":"function","text":"LibGit2.path(repo::GitRepo)\n\nReturn the base file path of the repository repo.\n\nfor normal repositories, this will typically be the parent directory of the \".git\" directory (note: this may be different than the working directory, see workdir for more details).\nfor bare repositories, this is the location of the \"git\" files.\n\nSee also gitdir, workdir.\n\n\n\n\n\n"},{"title":"LibGit2.peel","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.peel","category":"function","text":"peel([T,] ref::GitReference)\n\nRecursively peel ref until an object of type T is obtained. If no T is provided, then ref will be peeled until an object other than a GitTag is obtained.\n\nA GitTag will be peeled to the object it references.\nA GitCommit will be peeled to a GitTree.\n\nnote: Note\nOnly annotated tags can be peeled to GitTag objects. Lightweight tags (the default) are references under refs/tags/ which point directly to GitCommit objects.\n\n\n\n\n\npeel([T,] obj::GitObject)\n\nRecursively peel obj until an object of type T is obtained. If no T is provided, then obj will be peeled until the type changes.\n\nA GitTag will be peeled to the object it references.\nA GitCommit will be peeled to a GitTree.\n\n\n\n\n\n"},{"title":"LibGit2.posixpath","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.posixpath","category":"function","text":"LibGit2.posixpath(path)\n\nStandardise the path string path to use POSIX separators.\n\n\n\n\n\n"},{"title":"LibGit2.push","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.push","category":"function","text":"push(rmt::GitRemote, refspecs; force::Bool=false, options::PushOptions=PushOptions())\n\nPush to the specified rmt remote git repository, using refspecs to determine which remote branch(es) to push to. The keyword arguments are:\n\nforce: if true, a force-push will occur, disregarding conflicts.\noptions: determines the options for the push, e.g. which proxy headers to use. See PushOptions for more information.\n\nnote: Note\nYou can add information about the push refspecs in two other ways: by setting an option in the repository's GitConfig (with push.default as the key) or by calling add_push!. Otherwise you will need to explicitly specify a push refspec in the call to push for it to have any effect, like so: LibGit2.push(repo, refspecs=[\"refs/heads/master\"]).\n\n\n\n\n\npush(repo::GitRepo; kwargs...)\n\nPushes updates to an upstream of repo.\n\nThe keyword arguments are:\n\nremote::AbstractString=\"origin\": the name of the upstream remote to push to.\nremoteurl::AbstractString=\"\": the URL of remote.\nrefspecs=AbstractString[]: determines properties of the push.\nforce::Bool=false: determines if the push will be a force push,  overwriting the remote branch.\ncredentials=nothing: provides credentials and/or settings when authenticating against  a private remote.\ncallbacks=Callbacks(): user provided callbacks and payloads.\n\nEquivalent to git push [<remoteurl>|<repo>] [<refspecs>].\n\n\n\n\n\n"},{"title":"LibGit2.push!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.push!-Tuple{LibGit2.GitRevWalker, LibGit2.GitHash}","category":"method","text":"LibGit2.push!(w::GitRevWalker, cid::GitHash)\n\nStart the GitRevWalker walker at commit cid. This function can be used to apply a function to all commits since a certain year, by passing the first commit of that year as cid and then passing the resulting w to LibGit2.map.\n\n\n\n\n\n"},{"title":"LibGit2.push_head!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.push_head!","category":"function","text":"LibGit2.push_head!(w::GitRevWalker)\n\nPush the HEAD commit and its ancestors onto the GitRevWalker w. This ensures that HEAD and all its ancestor commits will be encountered during the walk.\n\n\n\n\n\n"},{"title":"LibGit2.push_refspecs","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.push_refspecs","category":"function","text":"push_refspecs(rmt::GitRemote)::Vector{String}\n\nGet the push refspecs for the specified rmt. These refspecs contain information about which branch(es) to push to.\n\nExamples\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.add_push!(repo, remote, \"refs/heads/master\");\n\njulia> close(remote);\n\njulia> remote = LibGit2.get(LibGit2.GitRemote, repo, \"upstream\");\n\njulia> LibGit2.push_refspecs(remote)\nString[\"refs/heads/master\"]\n\n\n\n\n\n"},{"title":"LibGit2.raw","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.raw","category":"function","text":"raw(id::GitHash)::Vector{UInt8}\n\nObtain the raw bytes of the GitHash as a vector of length 20.\n\n\n\n\n\n"},{"title":"LibGit2.read_tree!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.read_tree!","category":"function","text":"LibGit2.read_tree!(idx::GitIndex, tree::GitTree)\nLibGit2.read_tree!(idx::GitIndex, treehash::AbstractGitHash)\n\nRead the tree tree (or the tree pointed to by treehash in the repository owned by idx) into the index idx. The current index contents will be replaced.\n\n\n\n\n\n"},{"title":"LibGit2.rebase!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.rebase!","category":"function","text":"LibGit2.rebase!(repo::GitRepo, upstream::AbstractString=\"\", newbase::AbstractString=\"\")\n\nAttempt an automatic merge rebase of the current branch, from upstream if provided, or otherwise from the upstream tracking branch. newbase is the branch to rebase onto. By default this is upstream.\n\nIf any conflicts arise which cannot be automatically resolved, the rebase will abort, leaving the repository and working tree in its original state, and the function will throw a GitError. This is roughly equivalent to the following command line statement:\n\ngit rebase --merge [<upstream>]\nif [ -d \".git/rebase-merge\" ]; then\n    git rebase --abort\nfi\n\n\n\n\n\n"},{"title":"LibGit2.ref_list","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.ref_list","category":"function","text":"LibGit2.ref_list(repo::GitRepo)::Vector{String}\n\nGet a list of all reference names in the repo repository.\n\n\n\n\n\n"},{"title":"LibGit2.reftype","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.reftype","category":"function","text":"LibGit2.reftype(ref::GitReference)::Cint\n\nReturn a Cint corresponding to the type of ref:\n\n0 if the reference is invalid\n1 if the reference is an object id\n2 if the reference is symbolic\n\n\n\n\n\n"},{"title":"LibGit2.remotes","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.remotes","category":"function","text":"LibGit2.remotes(repo::GitRepo)\n\nReturn a vector of the names of the remotes of repo.\n\n\n\n\n\n"},{"title":"LibGit2.remove!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.remove!","category":"function","text":"remove!(repo::GitRepo, files::AbstractString...)\nremove!(idx::GitIndex, files::AbstractString...)\n\nRemove all the files with paths specified by files in the index idx (or the index of the repo).\n\n\n\n\n\n"},{"title":"LibGit2.reset","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.reset","category":"function","text":"reset(val::Integer, flag::Integer)\n\nUnset the bits of val indexed by flag, returning them to 0.\n\n\n\n\n\n"},{"title":"LibGit2.reset!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.reset!","category":"function","text":"reset!(payload, [config])::CredentialPayload\n\nReset the payload state back to the initial values so that it can be used again within the credential callback. If a config is provided the configuration will also be updated.\n\n\n\n\n\nUpdates some entries, determined by the pathspecs, in the index from the target commit tree.\n\n\n\n\n\nSets the current head to the specified commit oid and optionally resets the index and working tree to match.\n\n\n\n\n\ngit reset [<committish>] [–] <pathspecs>... \n\n\n\n\n\nreset!(repo::GitRepo, id::GitHash, mode::Cint=Consts.RESET_MIXED)\n\nReset the repository repo to its state at id, using one of three modes set by mode:\n\nConsts.RESET_SOFT - move HEAD to id.\nConsts.RESET_MIXED - default, move HEAD to id and reset the index to id.\nConsts.RESET_HARD - move HEAD to id, reset the index to id, and discard all working changes.\n\nExamples\n\n# fetch changes\nLibGit2.fetch(repo)\nisfile(joinpath(repo_path, our_file)) # will be false\n\n# fastforward merge the changes\nLibGit2.merge!(repo, fastforward=true)\n\n# because there was not any file locally, but there is\n# a file remotely, we need to reset the branch\nhead_oid = LibGit2.head_oid(repo)\nnew_head = LibGit2.reset!(repo, head_oid, LibGit2.Consts.RESET_HARD)\n\nIn this example, the remote which is being fetched from does have a file called our_file in its index, which is why we must reset.\n\nEquivalent to git reset [--soft | --mixed | --hard] <id>.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nhead_oid = LibGit2.head_oid(repo)\nopen(joinpath(repo_path, \"file1\"), \"w\") do f\n    write(f, \"111\n\")\nend\nLibGit2.add!(repo, \"file1\")\nmode = LibGit2.Consts.RESET_HARD\n# will discard the changes to file1\n# and unstage it\nnew_head = LibGit2.reset!(repo, head_oid, mode)\n\n\n\n\n\n"},{"title":"LibGit2.restore","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.restore","category":"function","text":"restore(s::State, repo::GitRepo)\n\nReturn a repository repo to a previous State s, for example the HEAD of a branch before a merge attempt. s can be generated using the snapshot function.\n\n\n\n\n\n"},{"title":"LibGit2.revcount","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.revcount","category":"function","text":"LibGit2.revcount(repo::GitRepo, commit1::AbstractString, commit2::AbstractString)\n\nList the number of revisions between commit1 and commit2 (committish OIDs in string form). Since commit1 and commit2 may be on different branches, revcount performs a \"left-right\" revision list (and count), returning a tuple of Ints - the number of left and right commits, respectively. A left (or right) commit refers to which side of a symmetric difference in a tree the commit is reachable from.\n\nEquivalent to git rev-list --left-right --count <commit1> <commit2>.\n\nExamples\n\nrepo = LibGit2.GitRepo(repo_path)\nrepo_file = open(joinpath(repo_path, test_file), \"a\")\nprintln(repo_file, \"hello world\")\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid1 = LibGit2.commit(repo, \"commit 1\")\nprintln(repo_file, \"hello world again\")\nflush(repo_file)\nLibGit2.add!(repo, test_file)\ncommit_oid2 = LibGit2.commit(repo, \"commit 2\")\nLibGit2.revcount(repo, string(commit_oid1), string(commit_oid2))\n\nThis will return (-1, 0).\n\n\n\n\n\n"},{"title":"LibGit2.set_remote_url","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.set_remote_url","category":"function","text":"set_remote_url(repo::GitRepo, remote_name, url)\nset_remote_url(repo::String, remote_name, url)\n\nSet both the fetch and push url for remote_name for the GitRepo or the git repository located at path. Typically git repos use \"origin\" as the remote name.\n\nExamples\n\nrepo_path = joinpath(tempdir(), \"Example\")\nrepo = LibGit2.init(repo_path)\nLibGit2.set_remote_url(repo, \"upstream\", \"https://github.com/JuliaLang/Example.jl\")\nLibGit2.set_remote_url(repo_path, \"upstream2\", \"https://github.com/JuliaLang/Example2.jl\")\n\n\n\n\n\n"},{"title":"LibGit2.shortname","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.shortname","category":"function","text":"LibGit2.shortname(ref::GitReference)\n\nReturn a shortened version of the name of ref that's \"human-readable\".\n\njulia> repo = GitRepo(path_to_repo);\n\njulia> branch_ref = LibGit2.head(repo);\n\njulia> LibGit2.name(branch_ref)\n\"refs/heads/master\"\n\njulia> LibGit2.shortname(branch_ref)\n\"master\"\n\n\n\n\n\n"},{"title":"LibGit2.snapshot","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.snapshot","category":"function","text":"snapshot(repo::GitRepo)::State\n\nTake a snapshot of the current state of the repository repo, storing the current HEAD, index, and any uncommitted work. The output State can be used later during a call to restore to return the repository to the snapshotted state.\n\n\n\n\n\n"},{"title":"LibGit2.split_cfg_entry","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.split_cfg_entry","category":"function","text":"LibGit2.split_cfg_entry(ce::LibGit2.ConfigEntry)::Tuple{String,String,String,String}\n\nBreak the ConfigEntry up to the following pieces: section, subsection, name, and value.\n\nExamples\n\nGiven the git configuration file containing:\n\n[credential \"https://example.com\"]\n    username = me\n\nThe ConfigEntry would look like the following:\n\njulia> entry\nConfigEntry(\"credential.https://example.com.username\", \"me\")\n\njulia> LibGit2.split_cfg_entry(entry)\n(\"credential\", \"https://example.com\", \"username\", \"me\")\n\nRefer to the git config syntax documentation for more details.\n\n\n\n\n\n"},{"title":"LibGit2.status","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.status","category":"function","text":"LibGit2.status(repo::GitRepo, path::String)::Union{Cuint, Cvoid}\n\nLookup the status of the file at path in the git repository repo. For instance, this can be used to check if the file at path has been modified and needs to be staged and committed.\n\n\n\n\n\n"},{"title":"LibGit2.stage","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.stage","category":"function","text":"stage(ie::IndexEntry)::Cint\n\nGet the stage number of ie. The stage number 0 represents the current state of the working tree, but other numbers can be used in the case of a merge conflict. In such a case, the various stage numbers on an IndexEntry describe which side(s) of the conflict the current state of the file belongs to. Stage 0 is the state before the attempted merge, stage 1 is the changes which have been made locally, stages 2 and larger are for changes from other branches (for instance, in the case of a multi-branch \"octopus\" merge, stages 2, 3, and 4 might be used).\n\n\n\n\n\n"},{"title":"LibGit2.tag_create","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.tag_create","category":"function","text":"LibGit2.tag_create(repo::GitRepo, tag::AbstractString, commit; kwargs...)\n\nCreate a new git tag tag (e.g. \"v0.5\") in the repository repo, at the commit commit.\n\nThe keyword arguments are:\n\nmsg::AbstractString=\"\": the message for the tag.\nforce::Bool=false: if true, existing references will be overwritten.\nsig::Signature=Signature(repo): the tagger's signature.\n\n\n\n\n\n"},{"title":"LibGit2.tag_delete","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.tag_delete","category":"function","text":"LibGit2.tag_delete(repo::GitRepo, tag::AbstractString)\n\nRemove the git tag tag from the repository repo.\n\n\n\n\n\n"},{"title":"LibGit2.tag_list","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.tag_list","category":"function","text":"LibGit2.tag_list(repo::GitRepo)::Vector{String}\n\nGet a list of all tags in the git repository repo.\n\n\n\n\n\n"},{"title":"LibGit2.target","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.target","category":"function","text":"LibGit2.target(tag::GitTag)\n\nThe GitHash of the target object of tag.\n\n\n\n\n\n"},{"title":"LibGit2.toggle","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.toggle","category":"function","text":"toggle(val::Integer, flag::Integer)\n\nFlip the bits of val indexed by flag, so that if a bit is 0 it will be 1 after the toggle, and vice-versa.\n\n\n\n\n\n"},{"title":"LibGit2.transact","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.transact","category":"function","text":"transact(f::Function, repo::GitRepo)\n\nApply function f to the git repository repo, taking a snapshot before applying f. If an error occurs within f, repo will be returned to its snapshot state using restore. The error which occurred will be rethrown, but the state of repo will not be corrupted.\n\n\n\n\n\n"},{"title":"LibGit2.treewalk","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.treewalk","category":"function","text":"treewalk(f, tree::GitTree, post::Bool=false)\n\nTraverse the entries in tree and its subtrees in post or pre order. Preorder means beginning at the root and then traversing the leftmost subtree (and recursively on down through that subtree's leftmost subtrees) and moving right through the subtrees. Postorder means beginning at the bottom of the leftmost subtree, traversing upwards through it, then traversing the next right subtree (again beginning at the bottom) and finally visiting the tree root last of all.\n\nThe function parameter f should have following signature:\n\n(String, GitTreeEntry) -> Cint\n\nA negative value returned from f stops the tree walk. A positive value means that the entry will be skipped if post is false.\n\n\n\n\n\n"},{"title":"LibGit2.upstream","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.upstream","category":"function","text":"upstream(ref::GitReference)::Union{GitReference, Nothing}\n\nDetermine if the branch containing ref has a specified upstream branch.\n\nReturn either a GitReference to the upstream branch if it exists, or nothing if the requested branch does not have an upstream counterpart.\n\n\n\n\n\n"},{"title":"LibGit2.update!","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.update!","category":"function","text":"update!(repo::GitRepo, files::AbstractString...)\nupdate!(idx::GitIndex, files::AbstractString...)\n\nUpdate all the files with paths specified by files in the index idx (or the index of the repo). Match the state of each file in the index with the current state on disk, removing it if it has been removed on disk, or updating its entry in the object database.\n\n\n\n\n\n"},{"title":"LibGit2.url","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.url","category":"function","text":"url(rmt::GitRemote)\n\nGet the fetch URL of a remote git repository.\n\nExamples\n\njulia> repo_url = \"https://github.com/JuliaLang/Example.jl\";\n\njulia> repo = LibGit2.init(mktempdir());\n\njulia> remote = LibGit2.GitRemote(repo, \"origin\", repo_url);\n\njulia> LibGit2.url(remote)\n\"https://github.com/JuliaLang/Example.jl\"\n\n\n\n\n\n"},{"title":"LibGit2.version","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.version","category":"function","text":"version()::VersionNumber\n\nReturn the version of libgit2 in use, as a VersionNumber.\n\n\n\n\n\n"},{"title":"LibGit2.with","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.with","category":"function","text":"with(f::Function, obj)\n\nResource management helper function. Applies f to obj, making sure to call close on obj after f successfully returns or throws an error. Ensures that allocated git resources are finalized as soon as they are no longer needed.\n\n\n\n\n\n"},{"title":"LibGit2.with_warn","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.with_warn","category":"function","text":"with_warn(f::Function, ::Type{T}, args...)\n\nResource management helper function. Apply f to args, first constructing an instance of type T from args. Makes sure to call close on the resulting object after f successfully returns or throws an error. Ensures that allocated git resources are finalized as soon as they are no longer needed. If an error is thrown by f, a warning is shown containing the error.\n\n\n\n\n\n"},{"title":"LibGit2.workdir","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.workdir","category":"function","text":"LibGit2.workdir(repo::GitRepo)\n\nReturn the location of the working directory of repo. This will throw an error for bare repositories.\n\nnote: Note\nThis will typically be the parent directory of gitdir(repo), but can be different in some cases: e.g. if either the core.worktree configuration variable or the GIT_WORK_TREE environment variable is set.\n\nSee also gitdir, path.\n\n\n\n\n\n"},{"title":"LibGit2.GitObject","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.GitObject-Tuple{LibGit2.GitTreeEntry}","category":"method","text":"(::Type{T})(te::GitTreeEntry) where T<:GitObject\n\nGet the git object to which te refers and return it as its actual type (the type entrytype would show), for instance a GitBlob or GitTag.\n\nExamples\n\ntree = LibGit2.GitTree(repo, \"HEAD^{tree}\")\ntree_entry = tree[1]\nblob = LibGit2.GitBlob(tree_entry)\n\n\n\n\n\n"},{"title":"LibGit2.UserPasswordCredential","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.UserPasswordCredential","category":"type","text":"Credential that support only user and password parameters\n\n\n\n\n\n"},{"title":"LibGit2.SSHCredential","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.SSHCredential","category":"type","text":"SSH credential type\n\n\n\n\n\n"},{"title":"LibGit2.isfilled","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.isfilled","category":"function","text":"isfilled(cred::AbstractCredential)::Bool\n\nVerifies that a credential is ready for use in authentication.\n\n\n\n\n\n"},{"title":"LibGit2.CachedCredentials","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.CachedCredentials","category":"type","text":"Caches credential information for re-use\n\n\n\n\n\n"},{"title":"LibGit2.CredentialPayload","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.CredentialPayload","category":"type","text":"LibGit2.CredentialPayload\n\nRetains the state between multiple calls to the credential callback for the same URL. A CredentialPayload instance is expected to be reset! whenever it will be used with a different URL.\n\n\n\n\n\n"},{"title":"LibGit2.approve","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.approve","category":"function","text":"approve(payload::CredentialPayload; shred::Bool=true) -> nothing\n\nStore the payload credential for re-use in a future authentication. Should only be called when authentication was successful.\n\nThe shred keyword controls whether sensitive information in the payload credential field should be destroyed. Should only be set to false during testing.\n\n\n\n\n\n"},{"title":"LibGit2.reject","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.reject","category":"function","text":"reject(payload::CredentialPayload; shred::Bool=true) -> nothing\n\nDiscard the payload credential from begin re-used in future authentication. Should only be called when authentication was unsuccessful.\n\nThe shred keyword controls whether sensitive information in the payload credential field should be destroyed. Should only be set to false during testing.\n\n\n\n\n\n"},{"title":"LibGit2.Consts.GIT_CONFIG","page":"LibGit2","location":"stdlib/LibGit2.html#LibGit2.Consts.GIT_CONFIG","category":"type","text":"Priority level of a config file.\n\nThese priority levels correspond to the natural escalation logic (from higher to lower) when searching for config entries in git.\n\nCONFIG_LEVEL_DEFAULT - Open the global, XDG and system configuration files if any available.\nCONFIG_LEVEL_PROGRAMDATA - System-wide on Windows, for compatibility with portable git\nCONFIG_LEVEL_SYSTEM - System-wide configuration file; /etc/gitconfig on Linux systems\nCONFIG_LEVEL_XDG - XDG compatible configuration file; typically ~/.config/git/config\nCONFIG_LEVEL_GLOBAL - User-specific configuration file (also called Global configuration file); typically ~/.gitconfig\nCONFIG_LEVEL_LOCAL - Repository specific configuration file; $WORK_DIR/.git/config on non-bare repos\nCONFIG_LEVEL_WORKTREE - Worktree specific configuration file; $GIT_DIR/config.worktree\nCONFIG_LEVEL_APP - Application specific configuration file; freely defined by applications\nCONFIG_HIGHEST_LEVEL - Represents the highest level available config file (i.e. the most specific config file available that actually is loaded)\n\n\n\n\n\n"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","category":"section","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions."},{"title":"Arithmetic Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-Operators","category":"section","text":"The following arithmetic operators are supported on all primitive numeric types:\n\nExpression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x, y)\n\nA numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x + y), is treated as a multiplication, except with higher precedence than other binary operations. See Numeric Literal Coefficients for details.\n\nJulia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system.\n\nThe ÷ sign can be conveniently typed by writing \\div<tab> to the REPL or Julia IDE. See the manual section on Unicode input for more information.\n\nHere are some simple examples using arithmetic operators:\n\njulia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5\n\n(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)"},{"title":"Boolean Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Boolean-Operators","category":"section","text":"The following Boolean operators are supported on Bool types:\n\nExpression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or\n\nNegation changes true to false and vice versa. The short-circuiting operations are explained on the linked page."},{"title":"Arithmetic operations with Bool values","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-operations-with-Bool-values","category":"section","text":"Note that Bool is an integer type, such that false is numerically equal to 0 and true is numerically equal to 1. All the usual promotion rules and numeric operators are also defined on it, with a special behavior of arithmetic (non-Boolean) operations when all the arguments are Bool: in those cases, the arguments are promoted to Int instead of keeping their type. Compare e.g. the following equivalent operations with Bool and with a different numeric type (UInt8):\n\njulia> true - true\n0\n\njulia> 0x01 - 0x01\n0x00\n\nAlso, when used in multiplication, false acts as a strong zero:\n\njulia> NaN * false\n0.0\n\njulia> false * Inf\n0.0\n\nThis is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation."},{"title":"Bitwise Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Bitwise-Operators","category":"section","text":"The following bitwise operators are supported on all primitive integer types:\n\nExpression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx ⊼ y bitwise nand (not and)\nx ⊽ y bitwise nor (not or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left\n\nHere are some examples with bitwise operators:\n\njulia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> nand(123, 123)\n-124\n\njulia> 123 ⊼ 123\n-124\n\njulia> nor(123, 124)\n-128\n\njulia> 123 ⊽ 124\n-128\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84"},{"title":"Updating operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Updating-operators","category":"section","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:\n\njulia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4\n\nThe updating versions of all the binary arithmetic and bitwise operators are:\n\n+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=\n\nnote: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64"},{"title":"Vectorized \"dot\" operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#man-dot-operators","category":"section","text":"For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1, 2, 3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" a (non-square) array, but [1, 2, 3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3]. Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise.\n\njulia> [1, 2, 3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27\n\nMore specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element a of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).\n\nFurthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation).\n\nNote the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A, B) = kron(A, B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A, B] .⊗ [C, D] will compute [A⊗C, B⊗D] with no additional coding.\n\nCombining dot operators with numeric literals can be ambiguous. For example, it is not clear whether 1.+x means 1. + x or 1 .+ x. Therefore this syntax is disallowed, and spaces must be used around the operator in such cases."},{"title":"Numeric Comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numeric-Comparisons","category":"section","text":"Standard comparison operations are defined for all the primitive numeric types:\n\nOperator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to\n\nHere are some simple examples:\n\njulia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse\n\nIntegers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:\n\nFinite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.\n\nThe last point is potentially surprising and thus worth noting:\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nand can cause headaches when working with arrays:\n\njulia> [1 NaN] == [1 NaN]\nfalse\n\nJulia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:\n\nFunction Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number\n\nisequal considers NaNs equal to each other:\n\njulia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue\n\nisequal can also be used to distinguish signed zeros:\n\njulia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse\n\nMixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly.\n\nFor other types, isequal defaults to calling ==, so if you want to define equality for your own types then you only need to add a == method. If you define your own equality function, you should probably define a corresponding hash method to ensure that isequal(x,y) implies hash(x) == hash(y)."},{"title":"Chaining comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Chaining-comparisons","category":"section","text":"Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:\n\njulia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue\n\nChaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1.\n\nNote the evaluation behavior of chained comparisons:\n\njulia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse\n\nThe middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation)."},{"title":"Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Elementary-Functions","category":"section","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense.\n\nMoreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A."},{"title":"Operator Precedence and Associativity","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","category":"section","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:\n\nCategory Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - ! ~ ¬ √ ∛ ∜ ⋆ ± ∓ <: >: Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right\n\n[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a)).\n\n[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation.\n\nFor a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scm. Note that some of the operators there are not defined in the Base module but may be given definitions by standard libraries, packages or user code.\n\nYou can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:\n\njulia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)\n\nA symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:\n\njulia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)\n\nNote that symbols such as :sin return precedence 0. This value represents invalid operators and not operators of lowest precedence. Similarly, such operators are assigned associativity :none.\n\nNumeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent.\n\njulia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64\n\nJuxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y)."},{"title":"Numerical Conversions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numerical-Conversions","category":"section","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions.\n\nThe notation T(x) or convert(T, x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x)).\n\nThe following examples show the different forms.\n\njulia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\nSee Conversion and Promotion for how to define your own conversions and promotions."},{"title":"Rounding functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Rounding-functions","category":"section","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T"},{"title":"Division functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Division-functions","category":"section","text":"Function Description\ndiv(x, y), x÷y truncated division; quotient rounded towards zero\nfld(x, y) floored division; quotient rounded towards -Inf\ncld(x, y) ceiling division; quotient rounded towards +Inf\nrem(x, y), x%y remainder; satisfies x == div(x, y)*y + rem(x, y); sign matches x\nmod(x, y) modulus; satisfies x == fld(x, y)*y + mod(x, y); sign matches y\nmod1(x, y) mod with offset 1; returns r∈(0, y] for y>0 or r∈[y, 0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x, y) returns (div(x, y),rem(x, y))\nfldmod(x, y) returns (fld(x, y), mod(x, y))\ngcd(x, y...) greatest positive common divisor of x, y,...\nlcm(x, y...) least positive common multiple of x, y,..."},{"title":"Sign and absolute value functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","category":"section","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x, y) a value with the magnitude of x and the sign of y\nflipsign(x, y) a value with the magnitude of x and the sign of x*y"},{"title":"Powers, logs and roots","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Powers,-logs-and-roots","category":"section","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nfourthroot(x), ∜x fourth root of x\nhypot(x, y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x) - 1 for x near zero\nldexp(x, n) x * 2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b, x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1 + x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x\n\nFor an overview of why functions like hypot, expm1, and log1p are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot."},{"title":"Trigonometric and hyperbolic functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","category":"section","text":"All the standard trigonometric and hyperbolic functions are also defined:\n\nsin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc\n\nThese are all single-argument functions, with atan also accepting two arguments corresponding to a traditional atan2 function.\n\nAdditionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi * x) and cos(pi * x) respectively.\n\nIn order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:\n\nsind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd"},{"title":"Special functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Special-functions","category":"section","text":"Many other special mathematical functions are provided by the package SpecialFunctions.jl."},{"title":"Shared Arrays","page":"Shared Arrays","location":"stdlib/SharedArrays.html#Shared-Arrays","category":"section","text":"SharedArray represents an array, which is shared across multiple processes, on a single machine."},{"title":"SharedArrays.SharedArray","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.SharedArray","category":"type","text":"SharedArray{T}(dims::NTuple; init=false, pids=Int[])\nSharedArray{T,N}(...)\n\nConstruct a SharedArray of a bits type T and size dims across the processes specified by pids - all of which have to be on the same host.  If N is specified by calling SharedArray{T,N}(dims), then N must match the length of dims.\n\nIf pids is left unspecified, the shared array will be mapped across all processes on the current host, including the master. But, localindices and indexpids will only refer to worker processes. This facilitates work distribution code to use workers for actual computation with the master process acting as a driver.\n\nIf an init function of the type initfn(S::SharedArray) is specified, it is called on all the participating workers.\n\nThe shared array is valid as long as a reference to the SharedArray object exists on the node which created the mapping.\n\nSharedArray{T}(filename::AbstractString, dims::NTuple, [offset=0]; mode=nothing, init=false, pids=Int[])\nSharedArray{T,N}(...)\n\nConstruct a SharedArray backed by the file filename, with element type T (must be a bits type) and size dims, across the processes specified by pids - all of which have to be on the same host. This file is mmapped into the host memory, with the following consequences:\n\nThe array data must be represented in binary format (e.g., an ASCII format like CSV cannot be supported)\nAny changes you make to the array values (e.g., A[3] = 0) will also change the values on disk\n\nIf pids is left unspecified, the shared array will be mapped across all processes on the current host, including the master. But, localindices and indexpids will only refer to worker processes. This facilitates work distribution code to use workers for actual computation with the master process acting as a driver.\n\nmode must be one of \"r\", \"r+\", \"w+\", or \"a+\", and defaults to \"r+\" if the file specified by filename already exists, or \"w+\" if not. If an init function of the type initfn(S::SharedArray) is specified, it is called on all the participating workers. You cannot specify an init function if the file is not writable.\n\noffset allows you to skip the specified number of bytes at the beginning of the file.\n\n\n\n\n\n"},{"title":"SharedArrays.SharedVector","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.SharedVector","category":"type","text":"SharedVector\n\nA one-dimensional SharedArray.\n\n\n\n\n\n"},{"title":"SharedArrays.SharedMatrix","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.SharedMatrix","category":"type","text":"SharedMatrix\n\nA two-dimensional SharedArray.\n\n\n\n\n\n"},{"title":"Distributed.procs","page":"Shared Arrays","location":"stdlib/SharedArrays.html#Distributed.procs-Tuple{SharedArray}","category":"method","text":"procs(S::SharedArray)\n\nGet the vector of processes mapping the shared array.\n\n\n\n\n\n"},{"title":"SharedArrays.sdata","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.sdata","category":"function","text":"sdata(S::SharedArray)\n\nReturn the actual Array object backing S.\n\n\n\n\n\n"},{"title":"SharedArrays.indexpids","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.indexpids","category":"function","text":"indexpids(S::SharedArray)\n\nReturn the current worker's index in the list of workers mapping the SharedArray (i.e. in the same list returned by procs(S)), or 0 if the SharedArray is not mapped locally.\n\n\n\n\n\n"},{"title":"SharedArrays.localindices","page":"Shared Arrays","location":"stdlib/SharedArrays.html#SharedArrays.localindices","category":"function","text":"localindices(S::SharedArray)\n\nReturn a range describing the \"default\" indices to be handled by the current process.  This range should be interpreted in the sense of linear indexing, i.e., as a sub-range of 1:length(S).  In multi-process contexts, returns an empty range in the parent process (or any process for which indexpids returns 0).\n\nIt's worth emphasizing that localindices exists purely as a convenience, and you can partition work on the array among workers any way you wish. For a SharedArray, all indices should be equally fast for each worker process.\n\n\n\n\n\n"},{"title":"Essentials","page":"Essentials","location":"base/base.html#Essentials","category":"section","text":""},{"title":"Introduction","page":"Essentials","location":"base/base.html#Introduction","category":"section","text":"Julia Base contains a range of functions and macros appropriate for performing scientific and numerical computing, but is also as broad as those of many general-purpose programming languages. Additional functionality is available from a growing collection of available packages. Functions are grouped by topic below.\n\nSome general notes:\n\nTo use module functions, use import Module to import the module, and Module.fn(x) to use the functions.\nAlternatively, using Module will import all exported Module functions into the current namespace.\nBy convention, function names ending with an exclamation point (!) modify their arguments. Some functions have both modifying (e.g., sort!) and non-modifying (sort) versions.\n\nThe behaviors of Base and standard libraries are stable as defined in SemVer only if they are documented; i.e., included in the Julia documentation and not marked as unstable. See API FAQ for more information."},{"title":"Getting Around","page":"Essentials","location":"base/base.html#Getting-Around","category":"section","text":""},{"title":"Keywords","page":"Essentials","location":"base/base.html#Keywords","category":"section","text":"This is the list of reserved keywords in Julia: baremodule, begin, break, catch, const, continue, do, else, elseif, end, export, false, finally, for, function, global, if, import, let, local, macro, module, quote, return, struct, true, try, using, while. Those keywords are not allowed to be used as variable names.\n\nThe following two-word sequences are reserved: abstract type, mutable struct, primitive type. However, you can create variables with names: abstract, mutable, primitive and type.\n\nFinally: where is parsed as an infix operator for writing parametric method and type definitions; in and isa are parsed as infix operators; public is parsed as a keyword when beginning a toplevel statement; outer is parsed as a keyword when used to modify the scope of a variable in an iteration specification of a for loop; and as is used as a keyword to rename an identifier brought into scope by import or using. Creation of variables named where, in, isa, outer and as is allowed, though."},{"title":"Standard Modules","page":"Essentials","location":"base/base.html#standard-modules","category":"section","text":""},{"title":"Base Submodules","page":"Essentials","location":"base/base.html#base-submodules","category":"section","text":""},{"title":"All Objects","page":"Essentials","location":"base/base.html#All-Objects","category":"section","text":""},{"title":"Properties of Types","page":"Essentials","location":"base/base.html#Properties-of-Types","category":"section","text":""},{"title":"Type relations","page":"Essentials","location":"base/base.html#Type-relations","category":"section","text":""},{"title":"Declared structure","page":"Essentials","location":"base/base.html#Declared-structure","category":"section","text":""},{"title":"Memory layout","page":"Essentials","location":"base/base.html#Memory-layout","category":"section","text":""},{"title":"Special values","page":"Essentials","location":"base/base.html#Special-values","category":"section","text":""},{"title":"Special Types","page":"Essentials","location":"base/base.html#Special-Types","category":"section","text":""},{"title":"Generic Functions","page":"Essentials","location":"base/base.html#Generic-Functions","category":"section","text":""},{"title":"Syntax","page":"Essentials","location":"base/base.html#Syntax","category":"section","text":""},{"title":"Managing deprecations","page":"Essentials","location":"base/base.html#Managing-deprecations","category":"section","text":""},{"title":"Missing Values","page":"Essentials","location":"base/base.html#Missing-Values","category":"section","text":""},{"title":"System","page":"Essentials","location":"base/base.html#System","category":"section","text":""},{"title":"Versioning","page":"Essentials","location":"base/base.html#Versioning","category":"section","text":""},{"title":"Errors","page":"Essentials","location":"base/base.html#Errors","category":"section","text":""},{"title":"Events","page":"Essentials","location":"base/base.html#Events","category":"section","text":""},{"title":"Reflection","page":"Essentials","location":"base/base.html#Reflection","category":"section","text":""},{"title":"Documentation","page":"Essentials","location":"base/base.html#Documentation","category":"section","text":"(See also the documentation chapter.)"},{"title":"Code loading","page":"Essentials","location":"base/base.html#Code-loading","category":"section","text":""},{"title":"Internals","page":"Essentials","location":"base/base.html#Internals","category":"section","text":""},{"title":"Meta","page":"Essentials","location":"base/base.html#Meta","category":"section","text":""},{"title":"Base.exit","page":"Essentials","location":"base/base.html#Base.exit","category":"function","text":"exit(code=0)\n\nStop the program with an exit code. The default exit code is zero, indicating that the program completed successfully. In an interactive session, exit() can be called with the keyboard shortcut ^D.\n\n\n\n\n\n"},{"title":"Base.atexit","page":"Essentials","location":"base/base.html#Base.atexit","category":"function","text":"atexit(f)\n\nRegister a zero- or one-argument function f() to be called at process exit. atexit() hooks are called in last in first out (LIFO) order and run before object finalizers.\n\nIf f has a method defined for one integer argument, it will be called as f(n::Int32), where n is the current exit code, otherwise it will be called as f().\n\ncompat: Julia 1.9\nThe one-argument form requires Julia 1.9\n\nExit hooks are allowed to call exit(n), in which case Julia will exit with exit code n (instead of the original exit code). If more than one exit hook calls exit(n), then Julia will exit with the exit code corresponding to the last called exit hook that calls exit(n). (Because exit hooks are called in LIFO order, \"last called\" is equivalent to \"first registered\".)\n\nNote: Once all exit hooks have been called, no more exit hooks can be registered, and any call to atexit(f) after all hooks have completed will throw an exception. This situation may occur if you are registering exit hooks from background Tasks that may still be executing concurrently during shutdown.\n\n\n\n\n\n"},{"title":"Base.isinteractive","page":"Essentials","location":"base/base.html#Base.isinteractive","category":"function","text":"isinteractive()::Bool\n\nDetermine whether Julia is running an interactive session.\n\n\n\n\n\n"},{"title":"Base.summarysize","page":"Essentials","location":"base/base.html#Base.summarysize","category":"function","text":"Base.summarysize(obj; count = false, exclude=Union{...}, chargeall=Union{...})::Int\n\nCompute all unique objects reachable from the argument and return either their size in memory (in bytes) or the number of allocations they span.\n\nKeyword Arguments\n\ncount: if false, return the total size of the objects in memory. if true, return the number of allocations spanned by the object.\nexclude: specifies the types of objects to exclude from the traversal.\nchargeall: specifies the types of objects to always charge the size of all of their fields, even if those fields would normally be excluded.\n\nSee also sizeof.\n\nExamples\n\njulia> Base.summarysize(1.0)\n8\n\njulia> Base.summarysize(Ref(rand(100)))\n848\n\njulia> sizeof(Ref(rand(100)))\n8\n\njulia> Base.summarysize(Core.svec(1.0, \"testing\", true); count=true)\n4\n\n\n\n\n\n"},{"title":"Base.__precompile__","page":"Essentials","location":"base/base.html#Base.__precompile__","category":"function","text":"__precompile__(isprecompilable::Bool)\n\nSpecify whether the file calling this function is precompilable, defaulting to true. If a module or file is not safely precompilable, it should call __precompile__(false) in order to throw an error if Julia attempts to precompile it.\n\n\n\n\n\n"},{"title":"Base.include","page":"Essentials","location":"base/base.html#Base.include","category":"function","text":"Base.include([mapexpr::Function,] m::Module, path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of module m. Every module (except those defined with baremodule) has its own definition of include omitting the m argument, which evaluates the file in that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n"},{"title":"include","page":"Essentials","location":"base/base.html#include","category":"function","text":"include([mapexpr::Function,] path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of the containing module. Every Module (except those defined with baremodule) has a private 1-argument definition of include, which evaluates the file in that module, for use inside that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files. The argument path is normalized using normpath which will resolve relative path tokens such as .. and convert / to the appropriate path separator.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\nUse Base.include to evaluate a file into another module.\n\nnote: Note\nJulia's syntax lowering recognizes an explicit call to a literal include at top-level and inserts an implicit @Core.latestworld to make any include'd definitions visible to subsequent code. Note however that this recognition is syntactic. I.e. assigning const myinclude = include may require an explicit @Core.latestworld call after myinclude.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n"},{"title":"Base.include_string","page":"Essentials","location":"base/base.html#Base.include_string","category":"function","text":"include_string([mapexpr::Function,] m::Module, code::AbstractString, filename::AbstractString=\"string\")\n\nLike include, except reads code from the given string rather than from a file.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in code, the include_string function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n"},{"title":"Base.include_dependency","page":"Essentials","location":"base/base.html#Base.include_dependency","category":"function","text":"include_dependency(path::AbstractString; track_content::Bool=true)\n\nIn a module, declare that the file, directory, or symbolic link specified by path (relative or absolute) is a dependency for precompilation; that is, if track_content=true the module will need to be recompiled if the content of path changes (if path is a directory the content equals join(readdir(path))). If track_content=false recompilation is triggered when the modification time mtime of path changes.\n\nThis is only needed if your module depends on a path that is not used via include. It has no effect outside of compilation.\n\ncompat: Julia 1.11\nKeyword argument track_content requires at least Julia 1.11. An error is now thrown if path is not readable.\n\n\n\n\n\n"},{"title":"__init__","page":"Essentials","location":"base/base.html#__init__","category":"keyword","text":"__init__\n\nThe __init__() function in a module executes immediately after the module is loaded at runtime for the first time. It is called once, after all other statements in the module have been executed. Because it is called after fully importing the module, __init__ functions of submodules will be executed first. Two typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries. See the manual section about modules for more details.\n\nSee also: OncePerProcess.\n\nExamples\n\nconst foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend\n\n\n\n\n\n"},{"title":"Base.OncePerProcess","page":"Essentials","location":"base/base.html#Base.OncePerProcess","category":"type","text":"OncePerProcess{T}(init::Function)() -> T\n\nCalling a OncePerProcess object returns a value of type T by running the function initializer exactly once per process. All concurrent and future calls in the same process will return exactly the same value. This is useful in code that will be precompiled, as it allows setting up caches or other state which won't get serialized.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const global_state = Base.OncePerProcess{Vector{UInt32}}() do\n           println(\"Making lazy global value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (procstate = global_state()) |> typeof\nMaking lazy global value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> procstate === global_state()\ntrue\n\njulia> procstate === fetch(@async global_state())\ntrue\n\n\n\n\n\n"},{"title":"Base.OncePerTask","page":"Essentials","location":"base/base.html#Base.OncePerTask","category":"type","text":"OncePerTask{T}(init::Function)() -> T\n\nCalling a OncePerTask object returns a value of type T by running the function initializer exactly once per Task. All future calls in the same Task will return exactly the same value.\n\nSee also: task_local_storage.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const task_state = Base.OncePerTask{Vector{UInt32}}() do\n           println(\"Making lazy task value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (taskvec = task_state()) |> typeof\nMaking lazy task value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> taskvec === task_state()\ntrue\n\njulia> taskvec === fetch(@async task_state())\nMaking lazy task value...done.\nfalse\n\n\n\n\n\n"},{"title":"Base.OncePerThread","page":"Essentials","location":"base/base.html#Base.OncePerThread","category":"type","text":"OncePerThread{T}(init::Function)() -> T\n\nCalling a OncePerThread object returns a value of type T by running the function initializer exactly once per thread. All future calls in the same thread, and concurrent or future calls with the same thread id, will return exactly the same value. The object can also be indexed by the threadid for any existing thread, to get (or initialize on this thread) the value stored for that thread. Incorrect usage can lead to data-races or memory corruption so use only if that behavior is correct within your library's threading-safety design.\n\nwarning: Warning\nIt is not necessarily true that a Task only runs on one thread, therefore the value returned here may alias other values or change in the middle of your program. This function may get deprecated in the future. If initializer yields, the thread running the current task after the call might not be the same as the one at the start of the call.\n\nSee also: OncePerTask.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const thread_state = Base.OncePerThread{Vector{UInt32}}() do\n           println(\"Making lazy thread value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (threadvec = thread_state()) |> typeof\nMaking lazy thread value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> threadvec === fetch(@async thread_state())\ntrue\n\njulia> threadvec === thread_state[Threads.threadid()]\ntrue\n\n\n\n\n\n"},{"title":"Base.which","page":"Essentials","location":"base/base.html#Base.which-Tuple{Any, Any}","category":"method","text":"which(f, types)\n\nReturns the method of f (a Method object) that would be called for arguments of the given types.\n\nIf types is an abstract type, then the method that would be called by invoke is returned.\n\nSee also: parentmodule, @which, and @edit.\n\n\n\n\n\n"},{"title":"Base.methods","page":"Essentials","location":"base/base.html#Base.methods","category":"function","text":"methods(f, [types], [module])\n\nReturn the method table for f.\n\nIf types is specified, return an array of methods whose types match. If module is specified, return an array of methods defined in that module. A list of modules can also be specified as an array or set.\n\ncompat: Julia 1.4\nAt least Julia 1.4 is required for specifying a module.\n\nSee also: which, @which and methodswith.\n\n\n\n\n\n"},{"title":"Base.@show","page":"Essentials","location":"base/base.html#Base.@show","category":"macro","text":"@show exs...\n\nPrints one or more expressions, and their results, to stdout, and returns the last result.\n\nSee also: show, @info, println.\n\nExamples\n\njulia> x = @show 1+2\n1 + 2 = 3\n3\n\njulia> @show x^2 x/2;\nx ^ 2 = 9\nx / 2 = 1.5\n\n\n\n\n\n"},{"title":"Base.MainInclude.ans","page":"Essentials","location":"base/base.html#Base.MainInclude.ans","category":"constant","text":"ans\n\nA variable referring to the last computed value, automatically imported to the interactive prompt.\n\n\n\n\n\n"},{"title":"Base.MainInclude.err","page":"Essentials","location":"base/base.html#Base.MainInclude.err","category":"constant","text":"err\n\nA variable referring to the last thrown errors, automatically imported to the interactive prompt. The thrown errors are collected in a stack of exceptions.\n\n\n\n\n\n"},{"title":"Base.active_project","page":"Essentials","location":"base/base.html#Base.active_project","category":"function","text":"active_project()\n\nReturn the path of the active Project.toml file. See also Base.set_active_project.\n\n\n\n\n\n"},{"title":"Base.set_active_project","page":"Essentials","location":"base/base.html#Base.set_active_project","category":"function","text":"set_active_project(projfile::Union{AbstractString,Nothing})\n\nSet the active Project.toml file to projfile. See also Base.active_project.\n\ncompat: Julia 1.8\nThis function requires at least Julia 1.8.\n\n\n\n\n\n"},{"title":"Base.active_manifest","page":"Essentials","location":"base/base.html#Base.active_manifest","category":"function","text":"active_manifest()\nactive_manifest(project_file::AbstractString)\n\nReturn the path of the active manifest file, or the manifest file that would be used for a given project_file.\n\nIn a stacked environment (where multiple environments exist in the load path), this returns the manifest file for the primary (active) environment only, not the manifests from other environments in the stack. See the manual section on Environment stacks for more details on how stacked environments work.\n\nSee Project environments for details on the difference between a project and a manifest, and the naming options and their priority in package loading.\n\nSee also Base.active_project, Base.set_active_project.\n\n\n\n\n\n"},{"title":"module","page":"Essentials","location":"base/base.html#module","category":"keyword","text":"module\n\nmodule declares a Module, which is a separate global variable workspace. Within a module, you can control which names from other modules are visible (via importing), and specify which of your names are intended to be public (via export and public). Modules allow you to create top-level definitions without worrying about name conflicts when your code is used together with somebody else’s. See the manual section about modules for more details.\n\nExamples\n\nmodule Foo\nimport Base.show\nexport MyType, foo\n\nstruct MyType\n    x\nend\n\nbar(x) = 2x\nfoo(a::MyType) = bar(a.x) + 1\nshow(io::IO, a::MyType) = print(io, \"MyType $(a.x)\")\nend\n\n\n\n\n\n"},{"title":"export","page":"Essentials","location":"base/base.html#export","category":"keyword","text":"export\n\nexport is used within modules to tell Julia which names should be made available to the user. For example: export foo makes the name foo available when using the module. See the manual section about modules for details.\n\n\n\n\n\n"},{"title":"public","page":"Essentials","location":"base/base.html#public","category":"keyword","text":"public\n\npublic is used within modules to tell Julia which names are part of the public API of the module. For example: public foo indicates that the name foo is public, without making it available when using the module.\n\nAs export already indicates that a name is public, it is unnecessary and an error to declare a name both as public and as exported. See the manual section about modules for details.\n\ncompat: Julia 1.11\nThe public keyword was added in Julia 1.11. Prior to this the notion of publicness was less explicit.\n\n\n\n\n\n"},{"title":"import","page":"Essentials","location":"base/base.html#import","category":"keyword","text":"import\n\nimport Foo will load the module or package Foo. Names from the imported Foo module can be accessed with dot syntax (e.g. Foo.foo to access the name foo). See the manual section about modules for details.\n\n\n\n\n\n"},{"title":"using","page":"Essentials","location":"base/base.html#using","category":"keyword","text":"using\n\nusing Foo will load the module or package Foo and make its exported names available for direct use. Names can also be used via dot syntax (e.g. Foo.foo to access the name foo), whether they are exported or not. See the manual section about modules for details.\n\nnote: Note\nWhen two or more packages/modules export a name and that name does not refer to the same thing in each of the packages, and the packages are loaded via using without an explicit list of names, it is an error to reference that name without qualification. It is thus recommended that code intended to be forward-compatible with future versions of its dependencies and of Julia, e.g., code in released packages, list the names it uses from each loaded package, e.g., using Foo: Foo, f rather than using Foo.\n\n\n\n\n\n"},{"title":"as","page":"Essentials","location":"base/base.html#as","category":"keyword","text":"as\n\nas is used as a keyword to rename an identifier brought into scope by import or using, for the purpose of working around name conflicts as well as for shortening names.  (Outside of import or using statements, as is not a keyword and can be used as an ordinary identifier.)\n\nimport LinearAlgebra as LA brings the imported LinearAlgebra standard library into scope as LA.\n\nimport LinearAlgebra: eigen as eig, cholesky as chol brings the eigen and cholesky methods from LinearAlgebra into scope as eig and chol respectively.\n\nas works with using only when individual identifiers are brought into scope. For example, using LinearAlgebra: eigen as eig or using LinearAlgebra: eigen as eig, cholesky as chol works, but using LinearAlgebra as LA is invalid syntax, since it is nonsensical to rename all exported names from LinearAlgebra to LA.\n\n\n\n\n\n"},{"title":"baremodule","page":"Essentials","location":"base/base.html#baremodule","category":"keyword","text":"baremodule\n\nbaremodule declares a module that does not contain using Base or local definitions of eval and include. It does still import Core. In other words,\n\nmodule Mod\n\n...\n\nend\n\nis equivalent to\n\nbaremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend\n\n\n\n\n\n"},{"title":"function","page":"Essentials","location":"base/base.html#function","category":"keyword","text":"function\n\nFunctions are defined with the function keyword:\n\nfunction add(a, b)\n    return a + b\nend\n\nOr the short form notation:\n\nadd(a, b) = a + b\n\nThe use of the return keyword is exactly the same as in other languages, but is often optional. A function without an explicit return statement will return the last expression in the function body.\n\n\n\n\n\n"},{"title":"macro","page":"Essentials","location":"base/base.html#macro","category":"keyword","text":"macro\n\nmacro defines a method for inserting generated code into a program. A macro maps a sequence of argument expressions to a returned expression, and the resulting expression is substituted directly into the program at the point where the macro is invoked. Macros are a way to run generated code without calling eval, since the generated code instead simply becomes part of the surrounding program. Macro arguments may include expressions, literal values, and symbols. Macros can be defined for variable number of arguments (varargs), but do not accept keyword arguments. Every macro also implicitly gets passed the arguments __source__, which contains the line number and file name the macro is called from, and __module__, which is the module the macro is expanded in.\n\nSee the manual section on Metaprogramming for more information about how to write a macro.\n\nExamples\n\njulia> macro sayhello(name)\n           return :( println(\"Hello, \", $name, \"!\") )\n       end\n@sayhello (macro with 1 method)\n\njulia> @sayhello \"Charlie\"\nHello, Charlie!\n\njulia> macro saylots(x...)\n           return :( println(\"Say: \", $(x...)) )\n       end\n@saylots (macro with 1 method)\n\njulia> @saylots \"hey \" \"there \" \"friend\"\nSay: hey there friend\n\n\n\n\n\n"},{"title":"return","page":"Essentials","location":"base/base.html#return","category":"keyword","text":"return\n\nreturn x causes the enclosing function to exit early, passing the given value x back to its caller. return by itself with no value is equivalent to return nothing (see nothing).\n\nfunction compare(a, b)\n    a == b && return \"equal to\"\n    a < b ? \"less than\" : \"greater than\"\nend\n\nIn general you can place a return statement anywhere within a function body, including within deeply nested loops or conditionals, but be careful with do blocks. For example:\n\nfunction test1(xs)\n    for x in xs\n        iseven(x) && return 2x\n    end\nend\n\nfunction test2(xs)\n    map(xs) do x\n        iseven(x) && return 2x\n        x\n    end\nend\n\nIn the first example, the return breaks out of test1 as soon as it hits an even number, so test1([5,6,7]) returns 12.\n\nYou might expect the second example to behave the same way, but in fact the return there only breaks out of the inner function (inside the do block) and gives a value back to map. test2([5,6,7]) then returns [5,12,7].\n\nWhen used in a top-level expression (i.e. outside any function), return causes the entire current top-level expression to terminate early.\n\n\n\n\n\n"},{"title":"do","page":"Essentials","location":"base/base.html#do","category":"keyword","text":"do\n\nCreate an anonymous function and pass it as the first argument to a function call. For example:\n\nmap(1:10) do x\n    2x\nend\n\nis equivalent to map(x->2x, 1:10).\n\nUse multiple arguments like so:\n\nmap(1:10, 11:20) do x, y\n    x + y\nend\n\n\n\n\n\n"},{"title":"begin","page":"Essentials","location":"base/base.html#begin","category":"keyword","text":"begin\n\nbegin...end denotes a block of code.\n\nbegin\n    println(\"Hello, \")\n    println(\"World!\")\nend\n\nUsually begin will not be necessary, since keywords such as function and let implicitly begin blocks of code. See also ;.\n\nbegin may also be used when indexing to represent the first index of a collection or the first index of a dimension of an array. For example, a[begin] is the first element of an array a.\n\ncompat: Julia 1.4\nUse of begin as an index requires Julia 1.4 or later.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> A[begin, :]\n2-element Matrix{Int64}:\n 1\n 2\n\n\n\n\n\n"},{"title":"end","page":"Essentials","location":"base/base.html#end","category":"keyword","text":"end\n\nend marks the conclusion of a block of expressions, for example module, struct, mutable struct, begin, let, for etc.\n\nend may also be used when indexing to represent the last index of a collection or the last index of a dimension of an array.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> A[end, :]\n2-element Vector{Int64}:\n 3\n 4\n\n\n\n\n\n"},{"title":"let","page":"Essentials","location":"base/base.html#let","category":"keyword","text":"let\n\nlet blocks create a new hard scope and optionally introduce new local bindings.\n\nJust like the other scope constructs, let blocks define the block of code where newly introduced local variables are accessible. Additionally, the syntax has a special meaning for comma-separated assignments and variable names that may optionally appear on the same line as the let:\n\nlet var1 = value1, var2, var3 = value3\n    code\nend\n\nThe variables introduced on this line are local to the let block and the assignments are evaluated in order, with each right-hand side evaluated in the scope without considering the name on the left-hand side. Therefore it makes sense to write something like let x = x, since the two x variables are distinct with the left-hand side locally shadowing the x from the outer scope. This can even be a useful idiom as new local variables are freshly created each time local scopes are entered, but this is only observable in the case of variables that outlive their scope via closures.  A let variable without an assignment, such as var2 in the example above, declares a new local variable that is not yet bound to a value.\n\nBy contrast, begin blocks also group multiple expressions together but do not introduce scope or have the special assignment syntax.\n\nExamples\n\nIn the function below, there is a single x that is iteratively updated three times by the map. The closures returned all reference that one x at its final value:\n\njulia> function test_outer_x()\n           x = 0\n           map(1:3) do _\n               x += 1\n               return ()->x\n           end\n       end\ntest_outer_x (generic function with 1 method)\n\njulia> [f() for f in test_outer_x()]\n3-element Vector{Int64}:\n 3\n 3\n 3\n\nIf, however, we add a let block that introduces a new local variable we will end up with three distinct variables being captured (one at each iteration) even though we chose to use (shadow) the same name.\n\njulia> function test_let_x()\n           x = 0\n           map(1:3) do _\n               x += 1\n               let x = x\n                   return ()->x\n               end\n           end\n       end\ntest_let_x (generic function with 1 method)\n\njulia> [f() for f in test_let_x()]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\nAll scope constructs that introduce new local variables behave this way when repeatedly run; the distinctive feature of let is its ability to succinctly declare new locals that may shadow outer variables of the same name. For example, directly using the argument of the do function similarly captures three distinct variables:\n\njulia> function test_do_x()\n           map(1:3) do x\n               return ()->x\n           end\n       end\ntest_do_x (generic function with 1 method)\n\njulia> [f() for f in test_do_x()]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"title":"if","page":"Essentials","location":"base/base.html#if","category":"keyword","text":"if/elseif/else\n\nif/elseif/else performs conditional evaluation, which allows portions of code to be evaluated or not evaluated depending on the value of a boolean expression. Here is the anatomy of the if/elseif/else conditional syntax:\n\nif x < y\n    println(\"x is less than y\")\nelseif x > y\n    println(\"x is greater than y\")\nelse\n    println(\"x is equal to y\")\nend\n\nIf the condition expression x < y is true, then the corresponding block is evaluated; otherwise the condition expression x > y is evaluated, and if it is true, the corresponding block is evaluated; if neither expression is true, the else block is evaluated. The elseif and else blocks are optional, and as many elseif blocks as desired can be used.\n\nIn contrast to some other languages conditions must be of type Bool. It does not suffice for conditions to be convertible to Bool.\n\njulia> if 1 end\nERROR: TypeError: non-boolean (Int64) used in boolean context\n\n\n\n\n\n"},{"title":"for","page":"Essentials","location":"base/base.html#for","category":"keyword","text":"for\n\nfor loops repeatedly evaluate a block of statements while iterating over a sequence of values.\n\nThe iteration variable is always a new variable, even if a variable of the same name exists in the enclosing scope. Use outer to reuse an existing local variable for iteration.\n\nExamples\n\njulia> for i in [1, 4, 0]\n           println(i)\n       end\n1\n4\n0\n\n\n\n\n\n"},{"title":"while","page":"Essentials","location":"base/base.html#while","category":"keyword","text":"while\n\nwhile loops repeatedly evaluate a conditional expression, and continue evaluating the body of the while loop as long as the expression remains true. If the condition expression is false when the while loop is first reached, the body is never evaluated.\n\nExamples\n\njulia> i = 1\n1\n\njulia> while i < 5\n           println(i)\n           global i += 1\n       end\n1\n2\n3\n4\n\n\n\n\n\n"},{"title":"break","page":"Essentials","location":"base/base.html#break","category":"keyword","text":"break\n\nBreak out of a loop immediately.\n\nExamples\n\njulia> i = 0\n0\n\njulia> while true\n           global i += 1\n           i > 5 && break\n           println(i)\n       end\n1\n2\n3\n4\n5\n\n\n\n\n\n"},{"title":"continue","page":"Essentials","location":"base/base.html#continue","category":"keyword","text":"continue\n\nSkip the rest of the current loop iteration.\n\nExamples\n\njulia> for i = 1:6\n           iseven(i) && continue\n           println(i)\n       end\n1\n3\n5\n\n\n\n\n\n"},{"title":"try","page":"Essentials","location":"base/base.html#try","category":"keyword","text":"try/catch\n\nA try/catch statement allows intercepting errors (exceptions) thrown by throw so that program execution can continue. For example, the following code attempts to write a file, but warns the user and proceeds instead of terminating execution if the file cannot be written:\n\ntry\n    open(\"/danger\", \"w\") do f\n        println(f, \"Hello\")\n    end\ncatch\n    @warn \"Could not write file.\"\nend\n\nor, when the file cannot be read into a variable:\n\nlines = try\n    open(\"/danger\", \"r\") do f\n        readlines(f)\n    end\ncatch\n    @warn \"File not found.\"\nend\n\nThe syntax catch e (where e is any variable) assigns the thrown exception object to the given variable within the catch block.\n\ntry\n    a_dangerous_operation()\ncatch e\n    if isa(e, EOFError)\n        @warn \"The operation failed - EOF.\"\n    elseif isa(e, OutOfMemoryError)\n        @warn \"The operation failed - OOM.\"\n    else\n        rethrow() # ensure other exceptions can bubble up the call stack\n    end\nend\n\nThe power of the try/catch construct lies in the ability to unwind a deeply nested computation immediately to a much higher level in the stack of calling functions.\n\nA try/catch block can also have an else clause that executes only if no exception occurred:\n\ntry\n    a_dangerous_operation()\ncatch\n    @warn \"The operation failed.\"\nelse\n    @info \"The operation succeeded.\"\nend\n\nA try or try/catch block can also have a finally clause that executes at the end, regardless of whether an exception occurred.  For example, this can be used to guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    operate_on_file(f)\ncatch\n    @warn \"An error occurred!\"\nfinally\n    close(f)\nend\n\n(finally can also be used without a catch block.)\n\ncompat: Julia 1.8\nElse clauses require at least Julia 1.8.\n\n\n\n\n\n"},{"title":"finally","page":"Essentials","location":"base/base.html#finally","category":"keyword","text":"finally\n\nRun some code when a given try block of code exits, regardless of how it exits. For example, here is how we can guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    operate_on_file(f)\nfinally\n    close(f)\nend\n\nWhen control leaves the try block (for example, due to a return, or just finishing normally), close(f) will be executed. If the try block exits due to an exception, the exception will continue propagating. A catch block may be combined with try and finally as well. In this case the finally block will run after catch has handled the error.\n\nWhen evaluating a try/catch/else/finally expression, the value of the entire expression is the value of the last block executed, excluding the finally block. For example:\n\njulia> try\n           1\n       finally\n           2\n       end\n1\n\njulia> try\n           error(\"\")\n       catch\n           1\n       else\n           2\n       finally\n           3\n       end\n1\n\njulia> try\n           0\n       catch\n           1\n       else\n           2\n       finally\n           3\n       end\n2\n\n\n\n\n\n"},{"title":"quote","page":"Essentials","location":"base/base.html#quote","category":"keyword","text":"quote\n\nquote creates multiple expression objects in a block without using the explicit Expr constructor. For example:\n\nex = quote\n    x = 1\n    y = 2\n    x + y\nend\n\nUnlike the other means of quoting, :( ... ), this form introduces QuoteNode elements to the expression tree, which must be considered when directly manipulating the tree. For other purposes, :( ... ) and quote .. end blocks are treated identically.\n\n\n\n\n\n"},{"title":"local","page":"Essentials","location":"base/base.html#local","category":"keyword","text":"local\n\nlocal introduces a new local variable. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> function foo(n)\n           x = 0\n           for i = 1:n\n               local x # introduce a loop-local x\n               x = i\n           end\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> foo(10)\n0\n\n\n\n\n\n"},{"title":"global","page":"Essentials","location":"base/base.html#global","category":"keyword","text":"global\n\nglobal x makes x in the current scope and its inner scopes refer to the global variable of that name. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> z = 3\n3\n\njulia> function foo()\n           global z = 6 # use the z variable defined outside foo\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n6\n\njulia> z\n6\n\n\n\n\n\n"},{"title":"outer","page":"Essentials","location":"base/base.html#outer","category":"keyword","text":"for outer\n\nReuse an existing local variable for iteration in a for loop.\n\nSee the manual section on variable scoping for more information.\n\nSee also for.\n\nExamples\n\njulia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0\n\njulia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3\n\njulia> i = 0 # global variable\n       for outer i = 1:3\n       end\nERROR: syntax: no outer local variable declaration exists for \"for outer\"\n[...]\n\n\n\n\n\n"},{"title":"const","page":"Essentials","location":"base/base.html#const","category":"keyword","text":"const\n\nconst is used to declare global variables whose values will not change. In almost all code (and particularly performance sensitive code) global variables should be declared constant in this way.\n\nconst x = 5\n\nMultiple variables can be declared within a single const:\n\nconst y, z = 7, 11\n\nNote that const only applies to one = operation, therefore const x = y = 1 declares x to be constant but not y. On the other hand, const x = const y = 1 declares both x and y constant.\n\nNote that \"constant-ness\" does not extend into mutable containers; only the association between a variable and its value is constant. If x is an array or dictionary (for example) you can still modify, add, or remove elements.\n\nIn some cases changing the value of a const variable gives a warning instead of an error. However, this can produce unpredictable behavior or corrupt the state of your program, and so should be avoided. This feature is intended only for convenience during interactive use.\n\n\n\n\n\n"},{"title":"struct","page":"Essentials","location":"base/base.html#struct","category":"keyword","text":"struct\n\nThe most commonly used kind of type in Julia is a struct, specified as a name and a set of fields.\n\nstruct Point\n    x\n    y\nend\n\nFields can have type restrictions, which may be parameterized:\n\nstruct Point{X}\n    x::X\n    y::Float64\nend\n\nA struct can also declare an abstract super type via <: syntax:\n\nstruct Point <: AbstractPoint\n    x\n    y\nend\n\nstructs are immutable by default; an instance of one of these types cannot be modified after construction. Use mutable struct instead to declare a type whose instances can be modified.\n\nSee the manual section on Composite Types for more details, such as how to define constructors.\n\n\n\n\n\n"},{"title":"mutable struct","page":"Essentials","location":"base/base.html#mutable struct","category":"keyword","text":"mutable struct\n\nmutable struct is similar to struct, but additionally allows the fields of the type to be set after construction.\n\nIndividual fields of a mutable struct can be marked as const to make them immutable:\n\nmutable struct Baz\n    a::Int\n    const b::Float64\nend\n\ncompat: Julia 1.8\nThe const keyword for fields of mutable structs requires at least Julia 1.8.\n\nSee the manual section on Composite Types for more information.\n\n\n\n\n\n"},{"title":"Base.@kwdef","page":"Essentials","location":"base/base.html#Base.@kwdef","category":"macro","text":"@kwdef typedef\n\nThis is a helper macro that automatically defines a keyword-based constructor for the type declared in the expression typedef, which must be a struct or mutable struct expression. The default argument is supplied by declaring fields of the form field::T = default or field = default. If no default is provided then the keyword argument becomes a required keyword argument in the resulting type constructor.\n\nInner constructors can still be defined, but at least one should accept arguments in the same form as the default inner constructor (i.e. one positional argument per field) in order to function correctly with the keyword outer constructor.\n\ncompat: Julia 1.1\nBase.@kwdef for parametric structs, and structs with supertypes requires at least Julia 1.1.\n\ncompat: Julia 1.9\nThis macro is exported as of Julia 1.9.\n\nExamples\n\njulia> @kwdef struct Foo\n           a::Int = 1         # specified default\n           b::String          # required keyword\n       end\nFoo\n\njulia> Foo(b=\"hi\")\nFoo(1, \"hi\")\n\njulia> Foo()\nERROR: UndefKeywordError: keyword argument `b` not assigned\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"abstract type","page":"Essentials","location":"base/base.html#abstract type","category":"keyword","text":"abstract type\n\nabstract type declares a type that cannot be instantiated, and serves only as a node in the type graph, thereby describing sets of related concrete types: those concrete types which are their descendants. Abstract types form the conceptual hierarchy which makes Julia’s type system more than just a collection of object implementations. For example:\n\nabstract type Number end\nabstract type Real <: Number end\n\nNumber has no supertype, whereas Real is an abstract subtype of Number.\n\n\n\n\n\n"},{"title":"primitive type","page":"Essentials","location":"base/base.html#primitive type","category":"keyword","text":"primitive type\n\nprimitive type declares a concrete type whose data consists only of a series of bits. Classic examples of primitive types are integers and floating-point values. Some example built-in primitive type declarations:\n\nprimitive type Char 32 end\nprimitive type Bool <: Integer 8 end\n\nThe number after the name indicates how many bits of storage the type requires. Currently, only sizes that are multiples of 8 bits are supported. The Bool declaration shows how a primitive type can be optionally declared to be a subtype of some supertype.\n\n\n\n\n\n"},{"title":"where","page":"Essentials","location":"base/base.html#where","category":"keyword","text":"where\n\nThe where keyword creates a UnionAll type, which may be thought of as an iterated union of other types, over all values of some variable. For example Vector{T} where T<:Real includes all Vectors where the element type is some kind of Real number.\n\nThe variable bound defaults to Any if it is omitted:\n\nVector{T} where T    # short for `where T<:Any`\n\nVariables can also have lower bounds:\n\nVector{T} where T>:Int\nVector{T} where Int<:T<:Real\n\nThere is also a concise syntax for nested where expressions. For example, this:\n\nPair{T, S} where S<:Array{T} where T<:Number\n\ncan be shortened to:\n\nPair{T, S} where {T<:Number, S<:Array{T}}\n\nThis form is often found on method signatures.\n\nNote that in this form, the variables are listed outermost-first. This matches the order in which variables are substituted when a type is \"applied\" to parameter values using the syntax T{p1, p2, ...}.\n\n\n\n\n\n"},{"title":"...","page":"Essentials","location":"base/base.html#...","category":"keyword","text":"...\n\nThe \"splat\" operator, ..., represents a sequence of arguments. ... can be used in function definitions, to indicate that the function accepts an arbitrary number of arguments. ... can also be used to apply a function to a sequence of arguments.\n\nExamples\n\njulia> add(xs...) = reduce(+, xs)\nadd (generic function with 1 method)\n\njulia> add(1, 2, 3, 4, 5)\n15\n\njulia> add([1, 2, 3]...)\n6\n\njulia> add(7, 1:100..., 1000:1100...)\n111107\n\n\n\n\n\n"},{"title":";","page":"Essentials","location":"base/base.html#;","category":"keyword","text":";\n\nSemicolons are used as statement separators and mark the beginning of keyword arguments in function declarations or calls.\n\n\n\n\n\n"},{"title":"=","page":"Essentials","location":"base/base.html#=","category":"keyword","text":"=\n\n= is the assignment operator.\n\nFor variable a and expression b, a = b makes a refer to the value of b.\nFor functions f(x), f(x) = x defines a new function constant f, or adds a new method to f if f is already defined; this usage is equivalent to function f(x); x; end.\na[i] = v calls setindex!(a,v,i).\na.b = c calls setproperty!(a,:b,c).\nInside a function call, f(a=b) passes b as the value of keyword argument a.\nInside parentheses with commas, (a=1,) constructs a NamedTuple.\n\nExamples\n\nAssigning a to b does not create a copy of b; instead use copy or deepcopy.\n\njulia> b = [1]; a = b; b[1] = 2; a\n1-element Vector{Int64}:\n 2\n\njulia> b = [1]; a = copy(b); b[1] = 2; a\n1-element Vector{Int64}:\n 1\n\n\nCollections passed to functions are also not copied. Functions can modify (mutate) the contents of the objects their arguments refer to. (The names of functions which do this are conventionally suffixed with '!'.)\n\njulia> function f!(x); x[:] .+= 1; end\nf! (generic function with 1 method)\n\njulia> a = [1]; f!(a); a\n1-element Vector{Int64}:\n 2\n\n\nAssignment can operate on multiple variables in parallel, taking values from an iterable:\n\njulia> a, b = 4, 5\n(4, 5)\n\njulia> a, b = 1:3\n1:3\n\njulia> a, b\n(1, 2)\n\n\nAssignment can operate on multiple variables in series, and will return the value of the right-hand-most expression:\n\njulia> a = [1]; b = [2]; c = [3]; a = b = c\n1-element Vector{Int64}:\n 3\n\njulia> b[1] = 2; a, b, c\n([2], [2], [2])\n\n\nAssignment at out-of-bounds indices does not grow a collection. If the collection is a Vector it can instead be grown with push! or append!.\n\njulia> a = [1, 1]; a[3] = 2\nERROR: BoundsError: attempt to access 2-element Vector{Int64} at index [3]\n[...]\n\njulia> push!(a, 2, 3)\n4-element Vector{Int64}:\n 1\n 1\n 2\n 3\n\n\nAssigning [] does not eliminate elements from a collection; instead use filter!.\n\njulia> a = collect(1:3); a[a .<= 1] = []\nERROR: DimensionMismatch: tried to assign 0 elements to 1 destinations\n[...]\n\njulia> filter!(x -> x > 1, a) # in-place & thus more efficient than a = a[a .> 1]\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\n\n"},{"title":"?:","page":"Essentials","location":"base/base.html#?:","category":"keyword","text":"a ? b : c\n\nShort form for conditionals; read \"if a, evaluate b otherwise evaluate c\". Also known as the ternary operator.\n\nThis syntax is equivalent to if a; b else c end, but is often used to emphasize the value b-or-c which is being used as part of a larger expression, rather than the side effects that evaluating b or c may have.\n\nSee the manual section on control flow for more details.\n\nExamples\n\njulia> x = 1; y = 2;\n\njulia> x > y ? println(\"x is larger\") : println(\"x is not larger\")\nx is not larger\n\njulia> x > y ? \"x is larger\" : x == y ? \"x and y are equal\" : \"y is larger\"\n\"y is larger\"\n\n\n\n\n\n"},{"title":".=","page":"Essentials","location":"base/base.html#.=","category":"keyword","text":".=\n\nPerform broadcasted assignment. The right-side argument is expanded as in broadcast and then assigned into the left-side argument in-place. Fuses with other dotted operators in the same expression; i.e. the whole assignment expression is converted into a single loop.\n\nA .= B is similar to broadcast!(identity, A, B).\n\nExamples\n\njulia> A = zeros(4, 4); B = [1, 2, 3, 4];\n\njulia> A .= B\n4×4 Matrix{Float64}:\n 1.0  1.0  1.0  1.0\n 2.0  2.0  2.0  2.0\n 3.0  3.0  3.0  3.0\n 4.0  4.0  4.0  4.0\n\njulia> A\n4×4 Matrix{Float64}:\n 1.0  1.0  1.0  1.0\n 2.0  2.0  2.0  2.0\n 3.0  3.0  3.0  3.0\n 4.0  4.0  4.0  4.0\n\n\n\n\n\n"},{"title":".","page":"Essentials","location":"base/base.html#.","category":"keyword","text":".\n\nThe dot operator is used to access fields or properties of objects and access variables defined inside modules.\n\nIn general, a.b calls getproperty(a, :b) (see getproperty).\n\nExamples\n\njulia> z = 1 + 2im; z.im\n2\n\njulia> Iterators.product\nproduct (generic function with 1 method)\n\n\n\n\n\n"},{"title":"->","page":"Essentials","location":"base/base.html#->","category":"keyword","text":"x -> y\n\nCreate an anonymous function mapping argument(s) x to the function body y.\n\njulia> f = x -> x^2 + 2x - 1\n#1 (generic function with 1 method)\n\njulia> f(2)\n7\n\nAnonymous functions can also be defined for multiple arguments.\n\njulia> g = (x,y) -> x^2 + y^2\n#2 (generic function with 1 method)\n\njulia> g(2,3)\n13\n\nSee the manual section on anonymous functions for more details.\n\n\n\n\n\n"},{"title":"Base.::","page":"Essentials","location":"base/base.html#Base.::","category":"function","text":":expr\n\nQuote an expression expr, returning the abstract syntax tree (AST) of expr. The AST may be of type Expr, Symbol, or a literal value. The syntax :identifier evaluates to a Symbol.\n\nSee also: Expr, Symbol, Meta.parse\n\nExamples\n\njulia> expr = :(a = b + 2*x)\n:(a = b + 2x)\n\njulia> sym = :some_identifier\n:some_identifier\n\njulia> value = :0xff\n0xff\n\njulia> typeof((expr, sym, value))\nTuple{Expr, Symbol, UInt8}\n\n\n\n\n\n"},{"title":"::","page":"Essentials","location":"base/base.html#::","category":"keyword","text":"::\n\nThe :: operator either asserts that a value has the given type, or declares that a local variable or function return always has the given type.\n\nGiven expression::T, expression is first evaluated. If the result is of type T, the value is simply returned. Otherwise, a TypeError is thrown.\n\nIn local scope, the syntax local x::T or x::T = expression declares that local variable x always has type T. When a value is assigned to the variable, it will be converted to type T by calling convert.\n\nIn a method declaration, the syntax function f(x)::T causes any value returned by the method to be converted to type T.\n\nSee the manual section on Type Declarations.\n\nExamples\n\njulia> (1+2)::AbstractFloat\nERROR: TypeError: typeassert: expected AbstractFloat, got a value of type Int64\n\njulia> (1+2)::Int\n3\n\njulia> let\n           local x::Int\n           x = 2.0\n           x\n       end\n2\n\n\n\n\n\n"},{"title":"[]","page":"Essentials","location":"base/base.html#[]","category":"keyword","text":"[]\n\nSquare brackets are used for indexing (getindex), indexed assignment (setindex!), array literals (Base.vect), array concatenation (vcat, hcat, hvcat, hvncat), and array comprehensions (collect).\n\n\n\n\n\n"},{"title":"Main","page":"Essentials","location":"base/base.html#Main","category":"module","text":"Main\n\nMain is the top-level module, and Julia starts with Main set as the current module.  Variables defined at the prompt go in Main, and varinfo lists variables in Main.\n\njulia> @__MODULE__\nMain\n\n\n\n\n\n"},{"title":"Core","page":"Essentials","location":"base/base.html#Core","category":"module","text":"Core\n\nCore is the module that contains all identifiers considered \"built in\" to the language, i.e. part of the core language and not libraries. Every module implicitly specifies using Core, since you can't do anything without those definitions.\n\n\n\n\n\n"},{"title":"Base","page":"Essentials","location":"base/base.html#Base","category":"module","text":"Base\n\nThe base library of Julia. Base is a module that contains basic functionality (the contents of base/). All modules implicitly contain using Base, since this is needed in the vast majority of cases.\n\n\n\n\n\n"},{"title":"Base.Broadcast","page":"Essentials","location":"base/base.html#Base.Broadcast","category":"module","text":"Base.Broadcast\n\nModule containing the broadcasting implementation.\n\n\n\n\n\n"},{"title":"Base.Docs","page":"Essentials","location":"base/base.html#Base.Docs","category":"module","text":"Docs\n\nThe Docs module provides the @doc macro which can be used to set and retrieve documentation metadata for Julia objects.\n\nPlease see the manual section on documentation for more information.\n\n\n\n\n\n"},{"title":"Base.Iterators","page":"Essentials","location":"base/base.html#Base.Iterators","category":"module","text":"Methods for working with Iterators.\n\n\n\n\n\n"},{"title":"Base.Libc","page":"Essentials","location":"base/base.html#Base.Libc","category":"module","text":"Interface to libc, the C standard library.\n\n\n\n\n\n"},{"title":"Base.Meta","page":"Essentials","location":"base/base.html#Base.Meta","category":"module","text":"Convenience functions for metaprogramming.\n\n\n\n\n\n"},{"title":"Base.StackTraces","page":"Essentials","location":"base/base.html#Base.StackTraces","category":"module","text":"Tools for collecting and manipulating stack traces. Mainly used for building errors.\n\n\n\n\n\n"},{"title":"Base.Sys","page":"Essentials","location":"base/base.html#Base.Sys","category":"module","text":"Provide methods for retrieving information about hardware and the operating system.\n\n\n\n\n\n"},{"title":"Base.Threads","page":"Essentials","location":"base/base.html#Base.Threads","category":"module","text":"Multithreading support.\n\n\n\n\n\n"},{"title":"Base.GC","page":"Essentials","location":"base/base.html#Base.GC","category":"module","text":"Base.GC\n\nModule with garbage collection utilities.\n\n\n\n\n\n"},{"title":"Core.:===","page":"Essentials","location":"base/base.html#Core.:===","category":"function","text":"===(x,y)::Bool\n≡(x,y)::Bool\n\nDetermine whether x and y are identical, in the sense that no program could distinguish them. First the types of x and y are compared. If those are identical, mutable objects are compared by address in memory and immutable objects (such as numbers) are compared by contents at the bit level. This function is sometimes called \"egal\". It always returns a Bool value.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a == b\ntrue\n\njulia> a === b\nfalse\n\njulia> a === a\ntrue\n\n\n\n\n\n"},{"title":"Core.isa","page":"Essentials","location":"base/base.html#Core.isa","category":"function","text":"isa(x, type)::Bool\n\nDetermine whether x is of the given type. Can also be used as an infix operator, e.g. x isa type.\n\nExamples\n\njulia> isa(1, Int)\ntrue\n\njulia> isa(1, Matrix)\nfalse\n\njulia> isa(1, Char)\nfalse\n\njulia> isa(1, Number)\ntrue\n\njulia> 1 isa Number\ntrue\n\n\n\n\n\n"},{"title":"Base.isequal","page":"Essentials","location":"base/base.html#Base.isequal","category":"function","text":"isequal(x)\n\nCreate a function that compares its argument to x using isequal, i.e. a function equivalent to y -> isequal(y, x).\n\nThe returned function is of type Base.Fix2{typeof(isequal)}, which can be used to implement specialized methods.\n\n\n\n\n\nisequal(x, y)::Bool\n\nSimilar to ==, except for the treatment of floating point numbers and of missing values. isequal treats all floating-point NaN values as equal to each other, treats -0.0 as unequal to 0.0, and missing as equal to missing. Always returns a Bool value.\n\nisequal is an equivalence relation - it is reflexive (=== implies isequal), symmetric (isequal(a, b) implies isequal(b, a)) and transitive (isequal(a, b) and isequal(b, c) implies isequal(a, c)).\n\nImplementation\n\nThe default implementation of isequal calls ==, so a type that does not involve floating-point values generally only needs to define ==.\n\nisequal is the comparison function used by hash tables (Dict). isequal(x,y) must imply that hash(x) == hash(y).\n\nThis typically means that types for which a custom == or isequal method exists must implement a corresponding hash method (and vice versa). Collections typically implement isequal by calling isequal recursively on all contents.\n\nFurthermore, isequal is linked with isless, and they work together to define a fixed total ordering, where exactly one of isequal(x, y), isless(x, y), or isless(y, x) must be true (and the other two false).\n\nScalar types generally do not need to implement isequal separate from ==, unless they represent floating-point numbers amenable to a more efficient implementation than that provided as a generic fallback (based on isnan, signbit, and ==).\n\nExamples\n\njulia> isequal([1., NaN], [1., NaN])\ntrue\n\njulia> [1., NaN] == [1., NaN]\nfalse\n\njulia> 0.0 == -0.0\ntrue\n\njulia> isequal(0.0, -0.0)\nfalse\n\njulia> missing == missing\nmissing\n\njulia> isequal(missing, missing)\ntrue\n\n\n\n\n\n"},{"title":"Base.isless","page":"Essentials","location":"base/base.html#Base.isless","category":"function","text":"isless(x, y)\n\nTest whether x is less than y, according to a fixed total order (defined together with isequal). isless is not defined for pairs (x, y) of all types. However, if it is defined, it is expected to satisfy the following:\n\nIf isless(x, y) is defined, then so is isless(y, x) and isequal(x, y), and exactly one of those three yields true.\nThe relation defined by isless is transitive, i.e., isless(x, y) && isless(y, z) implies isless(x, z).\n\nValues that are normally unordered, such as NaN, are ordered after regular values. missing values are ordered last.\n\nThis is the default comparison used by sort!.\n\nImplementation\n\nNon-numeric types with a total order should implement this function. Numeric types only need to implement it if they have special values such as NaN. Types with a partial order should implement <. See the documentation on Alternate Orderings for how to define alternate ordering methods that can be used in sorting and related functions.\n\nExamples\n\njulia> isless(1, 3)\ntrue\n\njulia> isless(\"Red\", \"Blue\")\nfalse\n\n\n\n\n\n"},{"title":"Base.ispositive","page":"Essentials","location":"base/base.html#Base.ispositive","category":"function","text":"ispositive(x)\n\nTest whether x > 0. See also isnegative.\n\ncompat: Julia 1.13\nThis function requires at least Julia 1.13.\n\nExamples\n\njulia> ispositive(-4.0)\nfalse\n\njulia> ispositive(99)\ntrue\n\njulia> ispositive(0.0)\nfalse\n\n\n\n\n\n"},{"title":"Base.isnegative","page":"Essentials","location":"base/base.html#Base.isnegative","category":"function","text":"isnegative(x)\n\nTest whether x < 0. See also ispositive.\n\ncompat: Julia 1.13\nThis function requires at least Julia 1.13.\n\nExamples\n\njulia> isnegative(-4.0)\ntrue\n\njulia> isnegative(99)\nfalse\n\njulia> isnegative(-0.0)\nfalse\n\n\n\n\n\n"},{"title":"Base.isunordered","page":"Essentials","location":"base/base.html#Base.isunordered","category":"function","text":"isunordered(x)\n\nReturn true if x is a value that is not orderable according to <, such as NaN or missing.\n\nThe values that evaluate to true with this predicate may be orderable with respect to other orderings such as isless.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.ifelse","page":"Essentials","location":"base/base.html#Base.ifelse","category":"function","text":"ifelse(condition::Bool, x, y)\n\nReturn x if condition is true, otherwise return y. This differs from ? or if in that it is an ordinary function, so all the arguments are evaluated first. In some cases, using ifelse instead of an if statement can eliminate the branch in generated code and provide higher performance in tight loops.\n\nExamples\n\njulia> ifelse(1 > 2, 1, 2)\n2\n\n\n\n\n\n"},{"title":"Core.typeassert","page":"Essentials","location":"base/base.html#Core.typeassert","category":"function","text":"typeassert(x, type)\n\nThrow a TypeError unless x isa type. The syntax x::type calls this function.\n\nExamples\n\njulia> typeassert(2.5, Int)\nERROR: TypeError: in typeassert, expected Int64, got a value of type Float64\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Core.typeof","page":"Essentials","location":"base/base.html#Core.typeof","category":"function","text":"typeof(x)\n\nGet the concrete type of x.\n\nSee also eltype.\n\nExamples\n\njulia> a = 1//2;\n\njulia> typeof(a)\nRational{Int64}\n\njulia> M = [1 2; 3.5 4];\n\njulia> typeof(M)\nMatrix{Float64} (alias for Array{Float64, 2})\n\n\n\n\n\n"},{"title":"Core.tuple","page":"Essentials","location":"base/base.html#Core.tuple","category":"function","text":"tuple(xs...)\n\nConstruct a tuple of the given objects.\n\nSee also Tuple, ntuple, NamedTuple.\n\nExamples\n\njulia> tuple(1, 'b', pi)\n(1, 'b', π)\n\njulia> ans === (1, 'b', π)\ntrue\n\njulia> Tuple(Real[1, 2, pi])  # takes a collection\n(1, 2, π)\n\n\n\n\n\n"},{"title":"Base.ntuple","page":"Essentials","location":"base/base.html#Base.ntuple","category":"function","text":"ntuple(f, ::Val{N})\n\nCreate a tuple of length N, computing each element as f(i), where i is the index of the element. By taking a Val(N) argument, it is possible that this version of ntuple may generate more efficient code than the version taking the length as an integer. But ntuple(f, N) is preferable to ntuple(f, Val(N)) in cases where N cannot be determined at compile time.\n\nExamples\n\njulia> ntuple(i -> 2*i, Val(4))\n(2, 4, 6, 8)\n\n\n\n\n\nntuple(f, n::Integer)\n\nCreate a tuple of length n, computing each element as f(i), where i is the index of the element.\n\nExamples\n\njulia> ntuple(i -> 2*i, 4)\n(2, 4, 6, 8)\n\n\n\n\n\n"},{"title":"Base.objectid","page":"Essentials","location":"base/base.html#Base.objectid","category":"function","text":"objectid(x)::UInt\n\nGet a hash value for x based on object identity. This value is not unique nor stable between Julia processes or versions.\n\nIf x === y then objectid(x) == objectid(y), and usually when x !== y, objectid(x) != objectid(y).\n\nSee also hash, IdDict.\n\n\n\n\n\n"},{"title":"Base.hash","page":"Essentials","location":"base/base.html#Base.hash","category":"function","text":"hash(x[, h::UInt])::UInt\n\nCompute an integer hash code such that isequal(x,y) implies isequal(hash(x), hash(y)). The optional second argument h is another hash code to be mixed with the result.\n\nNew types should implement the 2-argument form, typically by calling the 2-argument hash method recursively in order to mix hashes of the contents with each other (and with h). Typically, any type that implements hash should also implement its own == (hence isequal) to guarantee the property mentioned above.\n\nThe hash value may change when a new Julia process is started.\n\nwarning: Warning\nWhen implementing the 2-argument form, the second argument h should not be given a default value such h = UInt(0) as this will implicitly create a 1-argument method that is more specific than the fallback (see Note on Optional and keyword Arguments), but potentially with the wrong seed, causing hash inconsistencies.\n\njulia> a = hash(10)\n0x759d18cc5346a65f\n\njulia> hash(10, a) # only use the output of another hash function as the second argument\n0x03158cd61b1b0bd1\n\nSee also: objectid, Dict, Set.\n\n\n\n\n\n"},{"title":"Base.finalizer","page":"Essentials","location":"base/base.html#Base.finalizer","category":"function","text":"finalizer(f, x)\n\nRegister a function f(x) to be called when there are no program-accessible references to x, and return x. The type of x must be a mutable struct, otherwise the function will throw.\n\nf must not cause a task switch, which excludes most I/O operations such as println. Using the @async macro (to defer context switching to outside of the finalizer) or ccall to directly invoke IO functions in C may be helpful for debugging purposes.\n\nNote that there is no guaranteed world age for the execution of f. It may be called in the world age in which the finalizer was registered or any later world age.\n\nExamples\n\nfinalizer(my_mutable_struct) do x\n    @async println(\"Finalizing $x.\")\nend\n\nfinalizer(my_mutable_struct) do x\n    ccall(:jl_safe_printf, Cvoid, (Cstring, Cstring), \"Finalizing %s.\", repr(x))\nend\n\nA finalizer may be registered at object construction. In the following example note that we implicitly rely on the finalizer returning the newly created mutable struct x.\n\nmutable struct MyMutableStruct\n    bar\n    function MyMutableStruct(bar)\n        x = new(bar)\n        f(t) = @async println(\"Finalizing $t.\")\n        finalizer(f, x)\n    end\nend\n\n\n\n\n\n"},{"title":"Base.finalize","page":"Essentials","location":"base/base.html#Base.finalize","category":"function","text":"finalize(x)\n\nImmediately run finalizers registered for object x.\n\n\n\n\n\n"},{"title":"Base.copy","page":"Essentials","location":"base/base.html#Base.copy","category":"function","text":"copy(x)\n\nCreate a shallow copy of x: the outer structure is copied, but not all internal values. For example, copying an array produces a new array with identically-same elements as the original.\n\nSee also copy!, copyto!, deepcopy.\n\n\n\n\n\n"},{"title":"Base.deepcopy","page":"Essentials","location":"base/base.html#Base.deepcopy","category":"function","text":"deepcopy(x)\n\nCreate a deep copy of x: everything is copied recursively, resulting in a fully independent object. For example, deep-copying an array creates deep copies of all the objects it contains and produces a new array with the consistent relationship structure (e.g., if the first two elements are the same object in the original array, the first two elements of the new array will also be the same deepcopyed object). Calling deepcopy on an object should generally have the same effect as serializing and then deserializing it.\n\nWhile it isn't normally necessary, user-defined types can override the default deepcopy behavior by defining a specialized version of the function deepcopy_internal(x::T, dict::IdDict) (which shouldn't otherwise be used), where T is the type to be specialized for, and dict keeps track of objects copied so far within the recursion. Within the definition, deepcopy_internal should be used in place of deepcopy, and the dict variable should be updated as appropriate before returning.\n\nwarning: Warning\nIt is better to avoid this function in favor of custom copy methods or use-case-specific copying functions. deepcopy is slow and can easily copy too many objects, or generate an object that violates invariants, since it does not respect abstraction boundaries.\n\n\n\n\n\n"},{"title":"Base.getproperty","page":"Essentials","location":"base/base.html#Base.getproperty","category":"function","text":"getproperty(value, name::Symbol)\ngetproperty(value, name::Symbol, order::Symbol)\n\nThe syntax a.b calls getproperty(a, :b). The syntax @atomic order a.b calls getproperty(a, :b, :order) and the syntax @atomic a.b calls getproperty(a, :b, :sequentially_consistent).\n\nExamples\n\njulia> struct MyType{T <: Number}\n           x::T\n       end\n\njulia> function Base.getproperty(obj::MyType, sym::Symbol)\n           if sym === :special\n               return obj.x + 1\n           else # fallback to getfield\n               return getfield(obj, sym)\n           end\n       end\n\njulia> obj = MyType(1);\n\njulia> obj.special\n2\n\njulia> obj.x\n1\n\nOne should overload getproperty only when necessary, as it can be confusing if the behavior of the syntax obj.f is unusual. Also note that using methods is often preferable. See also this style guide documentation for more information: Prefer exported methods over direct field access.\n\nSee also getfield, propertynames and setproperty!.\n\n\n\n\n\n"},{"title":"Base.setproperty!","page":"Essentials","location":"base/base.html#Base.setproperty!","category":"function","text":"setproperty!(value, name::Symbol, x)\nsetproperty!(value, name::Symbol, x, order::Symbol)\n\nThe syntax a.b = c calls setproperty!(a, :b, c). The syntax @atomic order a.b = c calls setproperty!(a, :b, c, :order) and the syntax @atomic a.b = c calls setproperty!(a, :b, c, :sequentially_consistent).\n\ncompat: Julia 1.8\nsetproperty! on modules requires at least Julia 1.8.\n\nSee also setfield!, propertynames and getproperty.\n\n\n\n\n\n"},{"title":"Base.replaceproperty!","page":"Essentials","location":"base/base.html#Base.replaceproperty!","category":"function","text":"replaceproperty!(x, f::Symbol, expected, desired, success_order::Symbol=:not_atomic, fail_order::Symbol=success_order)\n\nPerform a compare-and-swap operation on x.f from expected to desired, per egal. The syntax @atomicreplace x.f expected => desired can be used instead of the function call form.\n\nSee also replacefield! setproperty!, setpropertyonce!.\n\n\n\n\n\n"},{"title":"Base.swapproperty!","page":"Essentials","location":"base/base.html#Base.swapproperty!","category":"function","text":"swapproperty!(x, f::Symbol, v, order::Symbol=:not_atomic)\n\nThe syntax @atomic a.b, _ = c, a.b returns (c, swapproperty!(a, :b, c, :sequentially_consistent)), where there must be one getproperty expression common to both sides.\n\nSee also swapfield! and setproperty!.\n\n\n\n\n\n"},{"title":"Base.modifyproperty!","page":"Essentials","location":"base/base.html#Base.modifyproperty!","category":"function","text":"modifyproperty!(x, f::Symbol, op, v, order::Symbol=:not_atomic)\n\nThe syntax @atomic op(x.f, v) (and its equivalent @atomic x.f op v) returns modifyproperty!(x, :f, op, v, :sequentially_consistent), where the first argument must be a getproperty expression and is modified atomically.\n\nInvocation of op(getproperty(x, f), v) must return a value that can be stored in the field f of the object x by default.  In particular, unlike the default behavior of setproperty!, the convert function is not called automatically.\n\nSee also modifyfield! and setproperty!.\n\n\n\n\n\n"},{"title":"Base.setpropertyonce!","page":"Essentials","location":"base/base.html#Base.setpropertyonce!","category":"function","text":"setpropertyonce!(x, f::Symbol, value, success_order::Symbol=:not_atomic, fail_order::Symbol=success_order)\n\nPerform a compare-and-swap operation on x.f to set it to value if previously unset. The syntax @atomiconce x.f = value can be used instead of the function call form.\n\nSee also setfieldonce!, setproperty!, replaceproperty!.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"title":"Base.propertynames","page":"Essentials","location":"base/base.html#Base.propertynames","category":"function","text":"propertynames(x, private=false)\n\nGet a tuple or a vector of the properties (x.property) of an object x. This is typically the same as fieldnames(typeof(x)), but types that overload getproperty should generally overload propertynames as well to get the properties of an instance of the type.\n\npropertynames(x) may return only \"public\" property names that are part of the documented interface of x.   If you want it to also return \"private\" property names intended for internal use, pass true for the optional second argument. REPL tab completion on x. shows only the private=false properties.\n\nSee also: hasproperty, hasfield.\n\n\n\n\n\n"},{"title":"Base.hasproperty","page":"Essentials","location":"base/base.html#Base.hasproperty","category":"function","text":"hasproperty(x, s::Symbol)\n\nReturn a boolean indicating whether the object x has s as one of its own properties.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\nSee also: propertynames, hasfield.\n\n\n\n\n\n"},{"title":"Core.getfield","page":"Essentials","location":"base/base.html#Core.getfield","category":"function","text":"getfield(value, name::Symbol, [order::Symbol], [boundscheck::Bool=true])\ngetfield(value, i::Int, [order::Symbol], [boundscheck::Bool=true])\n\nExtract a field from a composite value by name or position.\n\nOptionally, an ordering can be defined for the operation. If the field was declared @atomic, the specification is strongly recommended to be compatible with the stores to that location. Otherwise, if not declared as @atomic, this parameter must be :not_atomic if specified.\n\nThe bounds check may be disabled, in which case the behavior of this function is undefined if i is out of bounds.\n\nSee also getproperty and fieldnames.\n\nExamples\n\njulia> a = 1//2\n1//2\n\njulia> getfield(a, :num)\n1\n\njulia> a.num\n1\n\njulia> getfield(a, 1)\n1\n\n\n\n\n\n"},{"title":"Core.setfield!","page":"Essentials","location":"base/base.html#Core.setfield!","category":"function","text":"setfield!(value, name::Symbol, x, [order::Symbol])\nsetfield!(value, i::Int, x, [order::Symbol])\n\nAssign x to a named field in value of composite type. The value must be mutable and x must be a subtype of fieldtype(typeof(value), name). Additionally, an ordering can be specified for this operation. If the field was declared @atomic, this specification is mandatory. Otherwise, if not declared as @atomic, it must be :not_atomic if specified. See also setproperty!.\n\nExamples\n\njulia> mutable struct MyMutableStruct\n           field::Int\n       end\n\njulia> a = MyMutableStruct(1);\n\njulia> setfield!(a, :field, 2);\n\njulia> getfield(a, :field)\n2\n\njulia> a = 1//2\n1//2\n\njulia> setfield!(a, :num, 3);\nERROR: setfield!: immutable struct of type Rational cannot be changed\n\n\n\n\n\n"},{"title":"Core.modifyfield!","page":"Essentials","location":"base/base.html#Core.modifyfield!","category":"function","text":"modifyfield!(value, name::Symbol, op, x, [order::Symbol])::Pair\nmodifyfield!(value, i::Int, op, x, [order::Symbol])::Pair\n\nAtomically perform the operations to get and set a field after applying the function op.\n\ny = getfield(value, name)\nz = op(y, x)\nsetfield!(value, name, z)\nreturn y => z\n\nIf supported by the hardware (for example, atomic increment), this may be optimized to the appropriate hardware instruction, otherwise it'll use a loop.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Core.replacefield!","page":"Essentials","location":"base/base.html#Core.replacefield!","category":"function","text":"replacefield!(value, name::Symbol, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\nreplacefield!(value, i::Int, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\n\nAtomically perform the operations to get and conditionally set a field to a given value.\n\ny = getfield(value, name, fail_order)\nok = y === expected\nif ok\n    setfield!(value, name, desired, success_order)\nend\nreturn (; old = y, success = ok)\n\nIf supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise it'll use a loop.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Core.swapfield!","page":"Essentials","location":"base/base.html#Core.swapfield!","category":"function","text":"swapfield!(value, name::Symbol, x, [order::Symbol])\nswapfield!(value, i::Int, x, [order::Symbol])\n\nAtomically perform the operations to simultaneously get and set a field:\n\ny = getfield(value, name)\nsetfield!(value, name, x)\nreturn y\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Core.setfieldonce!","page":"Essentials","location":"base/base.html#Core.setfieldonce!","category":"function","text":"setfieldonce!(value, name::Union{Int,Symbol}, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> success::Bool\n\nAtomically perform the operations to set a field to a given value, only if it was previously not set.\n\nok = !isdefined(value, name, fail_order)\nif ok\n    setfield!(value, name, desired, success_order)\nend\nreturn ok\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"title":"Core.isdefined","page":"Essentials","location":"base/base.html#Core.isdefined","category":"function","text":"isdefined(m::Module, s::Symbol, [order::Symbol])\nisdefined(object, s::Symbol, [order::Symbol])\nisdefined(object, index::Int, [order::Symbol])\n\nTests whether a global variable or object field is defined. The arguments can be a module and a symbol or a composite object and field name (as a symbol) or index. Optionally, an ordering can be defined for the operation. If the field was declared @atomic, the specification is strongly recommended to be compatible with the stores to that location. Otherwise, if not declared as @atomic, this parameter must be :not_atomic if specified.\n\nTo test whether an array element is defined, use isassigned instead.\n\nThe global variable variant is supported for compatibility with older julia releases. For new code, prefer isdefinedglobal.\n\nSee also @isdefined.\n\nExamples\n\njulia> isdefined(Base, :sum)\ntrue\n\njulia> isdefined(Base, :NonExistentMethod)\nfalse\n\njulia> a = 1//2;\n\njulia> isdefined(a, 2)\ntrue\n\njulia> isdefined(a, 3)\nfalse\n\njulia> isdefined(a, :num)\ntrue\n\njulia> isdefined(a, :numerator)\nfalse\n\n\n\n\n\n"},{"title":"Core.isdefinedglobal","page":"Essentials","location":"base/base.html#Core.isdefinedglobal","category":"function","text":"isdefinedglobal(m::Module, s::Symbol, [allow_import::Bool=true, [order::Symbol=:unordered]])\n\nTests whether a global variable s is defined in module m (in the current world age). A variable is considered defined if and only if a value may be read from this global variable and an access will not throw. This includes both constants and global variables that have a value set.\n\nIf allow_import is false, the global variable must be defined inside m and may not be imported from another module.\n\ncompat: Julia 1.12\nThis function requires Julia 1.12 or later.\n\nSee also @isdefined.\n\nExamples\n\njulia> isdefinedglobal(Base, :sum)\ntrue\n\njulia> isdefinedglobal(Base, :NonExistentMethod)\nfalse\n\njulia> isdefinedglobal(Base, :sum, false)\ntrue\n\njulia> isdefinedglobal(Main, :sum, false)\nfalse\n\n\n\n\n\n"},{"title":"Base.@isdefined","page":"Essentials","location":"base/base.html#Base.@isdefined","category":"macro","text":"@isdefined(s)::Bool\n\nTests whether variable s is defined in the current scope.\n\nSee also isdefined for field properties and isassigned for array indexes or haskey for other mappings.\n\nExamples\n\njulia> @isdefined newvar\nfalse\n\njulia> newvar = 1\n1\n\njulia> @isdefined newvar\ntrue\n\njulia> function f()\n           println(@isdefined x)\n           x = 3\n           println(@isdefined x)\n       end\nf (generic function with 1 method)\n\njulia> f()\nfalse\ntrue\n\n\n\n\n\n"},{"title":"Base.convert","page":"Essentials","location":"base/base.html#Base.convert","category":"function","text":"convert(T, x)\n\nConvert x to a value of type T.\n\nIf T is an Integer type, an InexactError will be raised if x is not representable by T, for example if x is not integer-valued, or is outside the range supported by T.\n\nExamples\n\njulia> convert(Int, 3.0)\n3\n\njulia> convert(Int, 3.5)\nERROR: InexactError: Int64(3.5)\nStacktrace:\n[...]\n\nIf T is an AbstractFloat type, then it will return the closest value to x representable by T. Inf is treated as one ulp greater than floatmax(T) for purposes of determining nearest.\n\njulia> x = 1/3\n0.3333333333333333\n\njulia> convert(Float32, x)\n0.33333334f0\n\njulia> convert(BigFloat, x)\n0.333333333333333314829616256247390992939472198486328125\n\nIf T is a collection type and x a collection, the result of convert(T, x) may share memory with all or part of x.\n\njulia> x = Int[1, 2, 3];\n\njulia> y = convert(Vector{Int}, x);\n\njulia> y === x\ntrue\n\nSee also: round, trunc, oftype, reinterpret.\n\n\n\n\n\n"},{"title":"Base.promote","page":"Essentials","location":"base/base.html#Base.promote","category":"function","text":"promote(xs...)\n\nConvert all arguments to a common type, and return them all (as a tuple). If no arguments can be converted, an error is raised.\n\nSee also: promote_type, promote_rule.\n\nExamples\n\njulia> promote(Int8(1), Float16(4.5), Float32(4.1))\n(1.0f0, 4.5f0, 4.1f0)\n\njulia> promote_type(Int8, Float16, Float32)\nFloat32\n\njulia> reduce(Base.promote_typejoin, (Int8, Float16, Float32))\nReal\n\njulia> promote(1, \"x\")\nERROR: promotion of types Int64 and String failed to change any arguments\n[...]\n\njulia> promote_type(Int, String)\nAny\n\n\n\n\n\n"},{"title":"Base.oftype","page":"Essentials","location":"base/base.html#Base.oftype","category":"function","text":"oftype(x, y)\n\nConvert y to the type of x i.e. convert(typeof(x), y).\n\nExamples\n\njulia> x = 4;\n\njulia> y = 3.;\n\njulia> oftype(x, y)\n3\n\njulia> oftype(y, x)\n4.0\n\n\n\n\n\n"},{"title":"Base.widen","page":"Essentials","location":"base/base.html#Base.widen","category":"function","text":"widen(x)\n\nIf x is a type, return a \"larger\" type, defined so that arithmetic operations + and - are guaranteed not to overflow nor lose precision for any combination of values that type x can hold.\n\nFor fixed-size integer types less than 128 bits, widen will return a type with twice the number of bits.\n\nIf x is a value, it is converted to widen(typeof(x)).\n\nExamples\n\njulia> widen(Int32)\nInt64\n\njulia> widen(1.5f0)\n1.5\n\n\n\n\n\n"},{"title":"Base.identity","page":"Essentials","location":"base/base.html#Base.identity","category":"function","text":"identity(x)\n\nThe identity function. Returns its argument.\n\nSee also: one, oneunit, and LinearAlgebra's I.\n\nExamples\n\njulia> identity(\"Well, what did you expect?\")\n\"Well, what did you expect?\"\n\n\n\n\n\n"},{"title":"Core.WeakRef","page":"Essentials","location":"base/base.html#Core.WeakRef","category":"type","text":"WeakRef(x)\n\nw = WeakRef(x) constructs a weak reference to the Julia value x: although w contains a reference to x, it does not prevent x from being garbage collected. w.value is either x (if x has not been garbage-collected yet) or nothing (if x has been garbage-collected).\n\njulia> x = \"a string\"\n\"a string\"\n\njulia> w = WeakRef(x)\nWeakRef(\"a string\")\n\njulia> GC.gc()\n\njulia> w           # a reference is maintained via `x`\nWeakRef(\"a string\")\n\njulia> x = nothing # clear reference\n\njulia> GC.gc()\n\njulia> w\nWeakRef(nothing)\n\n\n\n\n\n"},{"title":"Base.supertype","page":"Essentials","location":"base/base.html#Base.supertype","category":"function","text":"supertype(T::Union{DataType, UnionAll})\n\nReturn the direct supertype of type T. T can be a DataType or a UnionAll type. Does not support type Unions. Also see info on Types.\n\nExamples\n\njulia> supertype(Int32)\nSigned\n\njulia> supertype(Vector)\nDenseVector (alias for DenseArray{T, 1} where T)\n\n\n\n\n\n"},{"title":"Core.Type","page":"Essentials","location":"base/base.html#Core.Type","category":"type","text":"Core.Type{T}\n\nCore.Type is an abstract type which has all type objects as its instances. The only instance of the singleton type Core.Type{T} is the object T.\n\nExamples\n\njulia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\n\n\n\n\n"},{"title":"Core.DataType","page":"Essentials","location":"base/base.html#Core.DataType","category":"type","text":"DataType <: Type{T}\n\nDataType represents explicitly declared types that have names, explicitly declared supertypes, and, optionally, parameters.  Every concrete value in the system is an instance of some DataType.\n\nExamples\n\njulia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataType\n\njulia> struct Point\n           x::Int\n           y\n       end\n\njulia> typeof(Point)\nDataType\n\n\n\n\n\n"},{"title":"Core.:<:","page":"Essentials","location":"base/base.html#Core.:<:","category":"function","text":"<:(T1, T2)::Bool\n\nSubtyping relation, defined between two types. In Julia, a type S is said to be a subtype of a type T if and only if we have S <: T.\n\nFor any type L and any type R, L <: R implies that any value v of type L is also of type R. I.e., (L <: R) && (v isa L) implies v isa R.\n\nThe subtyping relation is a partial order. I.e., <: is:\n\nreflexive: for any type T, T <: T holds\nantisymmetric: for any type A and any type B, (A <: B) && (B <: A) implies A == B\ntransitive: for any type A, any type B and any type C; (A <: B) && (B <: C) implies A <: C\n\nSee also info on Types, Union{}, Any, isa.\n\nExamples\n\njulia> Float64 <: AbstractFloat\ntrue\n\njulia> Vector{Int} <: AbstractArray\ntrue\n\njulia> Matrix{Float64} <: Matrix{AbstractFloat}  # `Matrix` is invariant\nfalse\n\njulia> Tuple{Float64} <: Tuple{AbstractFloat}    # `Tuple` is covariant\ntrue\n\njulia> Union{} <: Int  # The bottom type, `Union{}`, subtypes each type.\ntrue\n\njulia> Union{} <: Float32 <: AbstractFloat <: Real <: Number <: Any  # Operator chaining\ntrue\n\nThe <: keyword also has several syntactic uses which represent the same subtyping relation, but which do not execute the operator or return a Bool:\n\nTo specify the lower bound and the upper bound on a parameter of a UnionAll type in a where statement.\nTo specify the lower bound and the upper bound on a (static) parameter of a method, see Parametric Methods.\nTo define a subtyping relation while declaring a new type, see struct and abstract type.\n\n\n\n\n\n"},{"title":"Core.:>:","page":"Essentials","location":"base/base.html#Core.:>:","category":"function","text":">:(T1, T2)\n\nSupertype operator, equivalent to T2 <: T1.\n\n\n\n\n\n"},{"title":"Base.typejoin","page":"Essentials","location":"base/base.html#Base.typejoin","category":"function","text":"typejoin(T, S, ...)\n\nReturn the closest common ancestor of types T and S, i.e. the narrowest type from which they both inherit. Recurses on additional varargs.\n\nExamples\n\njulia> typejoin(Int, Float64)\nReal\n\njulia> typejoin(Int, Float64, ComplexF32)\nNumber\n\n\n\n\n\n"},{"title":"Base.typeintersect","page":"Essentials","location":"base/base.html#Base.typeintersect","category":"function","text":"typeintersect(T::Type, S::Type)\n\nCompute a type that contains the intersection of T and S. Usually this will be the smallest such type or one close to it.\n\nA special case where exact behavior is guaranteed: when T <: S, typeintersect(S, T) == T == typeintersect(T, S).\n\n\n\n\n\n"},{"title":"Base.promote_type","page":"Essentials","location":"base/base.html#Base.promote_type","category":"function","text":"promote_type(type1, type2, ...)\n\nPromotion refers to converting values of mixed types to a single common type. promote_type represents the default promotion behavior in Julia when operators (usually mathematical) are given arguments of differing types. promote_type generally tries to return a type which can at least approximate most values of either input type without excessively widening.  Some loss is tolerated; for example, promote_type(Int64, Float64) returns Float64 even though strictly, not all Int64 values can be represented exactly as Float64 values.\n\nSee also: promote, promote_typejoin, promote_rule.\n\nExamples\n\njulia> promote_type(Int64, Float64)\nFloat64\n\njulia> promote_type(Int32, Int64)\nInt64\n\njulia> promote_type(Float32, BigInt)\nBigFloat\n\njulia> promote_type(Int16, Float16)\nFloat16\n\njulia> promote_type(Int64, Float16)\nFloat16\n\njulia> promote_type(Int8, UInt16)\nUInt16\n\nwarning: Don't overload this directly\nTo overload promotion for your own types you should overload promote_rule. promote_type calls promote_rule internally to determine the type. Overloading promote_type directly can cause ambiguity errors.\n\n\n\n\n\n"},{"title":"Base.promote_rule","page":"Essentials","location":"base/base.html#Base.promote_rule","category":"function","text":"promote_rule(type1, type2)\n\nSpecifies what type should be used by promote when given values of types type1 and type2. This function should not be called directly, but should have definitions added to it for new types as appropriate.\n\n\n\n\n\n"},{"title":"Base.promote_typejoin","page":"Essentials","location":"base/base.html#Base.promote_typejoin","category":"function","text":"promote_typejoin(T, S)\n\nCompute a type that contains both T and S, which could be either a parent of both types, or a Union if appropriate. Falls back to typejoin.\n\nSee instead promote, promote_type.\n\nExamples\n\njulia> Base.promote_typejoin(Int, Float64)\nReal\n\njulia> Base.promote_type(Int, Float64)\nFloat64\n\n\n\n\n\n"},{"title":"Base.iskindtype","page":"Essentials","location":"base/base.html#Base.iskindtype","category":"function","text":"Base.iskindtype(T)\n\nDetermine whether T is a kind, that is, the type of a Julia type: a DataType, Union, UnionAll, or Core.TypeofBottom.\n\nAll kinds are concrete because types are Julia values.\n\n\n\n\n\n"},{"title":"Base.isdispatchtuple","page":"Essentials","location":"base/base.html#Base.isdispatchtuple","category":"function","text":"isdispatchtuple(T)\n\nDetermine whether type T is a Tuple that could appear as a type signature in dispatch.  For this to be true, every element of the tuple type must be either:\n\nconcrete but not a kind type\na Type{U} with no free type variables in U\n\nnote: Note\nA dispatch tuple is relevant for method dispatch because it has no inhabited subtypes.For example, Tuple{Int, DataType} is concrete, but is not a dispatch tuple because Tuple{Int, Type{Bool}} is an inhabited subtype.Tuple{Tuple{DataType}} is a dispatch tuple because Tuple{DataType} is concrete and not a kind; the subtype Tuple{Tuple{Type{Int}}} is not inhabited.\n\nIf T is not a type, then return false.\n\nExamples\n\njulia> isdispatchtuple(Int)\nfalse\n\njulia> isdispatchtuple(Tuple{Int})\ntrue\n\njulia> isdispatchtuple(Tuple{Number})\nfalse\n\njulia> isdispatchtuple(Tuple{DataType})\nfalse\n\njulia> isdispatchtuple(Tuple{Type{Int}})\ntrue\n\njulia> isdispatchtuple(Tuple{Type})\nfalse\n\n\n\n\n\n"},{"title":"Base.ismutable","page":"Essentials","location":"base/base.html#Base.ismutable","category":"function","text":"ismutable(v)::Bool\n\nReturn true if and only if value v is mutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a DataType, it will tell you that a value of the type is mutable.\n\nnote: Note\nFor technical reasons, ismutable returns true for values of certain special types (for example String and Symbol) even though they cannot be mutated in a permissible way.\n\nSee also isbits, isstructtype.\n\nExamples\n\njulia> ismutable(1)\nfalse\n\njulia> ismutable([1,2])\ntrue\n\ncompat: Julia 1.5\nThis function requires at least Julia 1.5.\n\n\n\n\n\n"},{"title":"Base.isimmutable","page":"Essentials","location":"base/base.html#Base.isimmutable","category":"function","text":"isimmutable(v) -> Bool\n\nwarning: Warning\nConsider using !ismutable(v) instead, as isimmutable(v) will be replaced by !ismutable(v) in a future release. (Since Julia 1.5)\n\nReturn true iff value v is immutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a type, it will tell you that a value of DataType is mutable.\n\nExamples\n\njulia> isimmutable(1)\ntrue\n\njulia> isimmutable([1,2])\nfalse\n\n\n\n\n\n"},{"title":"Base.ismutabletype","page":"Essentials","location":"base/base.html#Base.ismutabletype","category":"function","text":"ismutabletype(T)::Bool\n\nDetermine whether type T was declared as a mutable type (i.e. using mutable struct keyword). If T is not a type, then return false.\n\ncompat: Julia 1.7\nThis function requires at least Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.isabstracttype","page":"Essentials","location":"base/base.html#Base.isabstracttype","category":"function","text":"isabstracttype(T)\n\nDetermine whether type T was declared as an abstract type (i.e. using the abstract type syntax). If T is not a type, then return false.\n\nnote: Note\nWhile abstract types are not concrete and vice versa, types can be neither concrete nor abstract (for example, Vector (a UnionAll)).\n\nSee also: isconcretetype.\n\nExamples\n\njulia> isabstracttype(AbstractArray)\ntrue\n\njulia> isabstracttype(Vector)\nfalse\n\n\n\n\n\n"},{"title":"Base.isprimitivetype","page":"Essentials","location":"base/base.html#Base.isprimitivetype","category":"function","text":"isprimitivetype(T)::Bool\n\nDetermine whether type T was declared as a primitive type (i.e. using the primitive type syntax). If T is not a type, then return false.\n\n\n\n\n\n"},{"title":"Base.issingletontype","page":"Essentials","location":"base/base.html#Base.issingletontype","category":"function","text":"Base.issingletontype(T)\n\nDetermine whether type T has exactly one possible instance; for example, a struct type with no fields except other singleton values. If T is not a concrete type, then return false.\n\n\n\n\n\n"},{"title":"Base.isstructtype","page":"Essentials","location":"base/base.html#Base.isstructtype","category":"function","text":"isstructtype(T)::Bool\n\nDetermine whether type T was declared as a struct type (i.e. using the struct or mutable struct keyword). If T is not a type, then return false.\n\n\n\n\n\n"},{"title":"Base.nameof","page":"Essentials","location":"base/base.html#Base.nameof-Tuple{DataType}","category":"method","text":"nameof(t::DataType)::Symbol\n\nGet the name of a (potentially UnionAll-wrapped) DataType (without its parent module) as a symbol.\n\nExamples\n\njulia> module Foo\n           struct S{T}\n           end\n       end\nFoo\n\njulia> nameof(Foo.S{T} where T)\n:S\n\n\n\n\n\n"},{"title":"Base.fieldnames","page":"Essentials","location":"base/base.html#Base.fieldnames","category":"function","text":"fieldnames(x::DataType)\n\nGet a tuple with the names of the fields of a DataType.\n\nEach name is a Symbol, except when x <: Tuple, in which case each name (actually the index of the field) is an Int.\n\nSee also propertynames, hasfield.\n\nExamples\n\njulia> fieldnames(Rational)\n(:num, :den)\n\njulia> fieldnames(typeof(1+im))\n(:re, :im)\n\njulia> fieldnames(Tuple{String,Int})\n(1, 2)\n\n\n\n\n\n"},{"title":"Base.fieldname","page":"Essentials","location":"base/base.html#Base.fieldname","category":"function","text":"fieldname(x::DataType, i::Integer)\n\nGet the name of field i of a DataType.\n\nThe return type is Symbol, except when x <: Tuple, in which case the index of the field is returned, of type Int.\n\nExamples\n\njulia> fieldname(Rational, 1)\n:num\n\njulia> fieldname(Rational, 2)\n:den\n\njulia> fieldname(Tuple{String,Int}, 2)\n2\n\n\n\n\n\n"},{"title":"Base.fieldindex","page":"Essentials","location":"base/base.html#Base.fieldindex","category":"function","text":"fieldindex(T, name::Symbol, err:Bool=true)\n\nGet the index of a named field, throwing an error if the field does not exist (when err==true) or returning 0 (when err==false).\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldindex(Foo, :y)\n2\n\njulia> fieldindex(Foo, :z)\nERROR: FieldError: type Foo has no field `z`, available fields: `x`, `y`\nStacktrace:\n[...]\n\njulia> fieldindex(Foo, :z, false)\n0\n\ncompat: Julia 1.13\nThis function is exported as of Julia 1.13.\n\n\n\n\n\n"},{"title":"Core.fieldtype","page":"Essentials","location":"base/base.html#Core.fieldtype","category":"function","text":"fieldtype(T, name::Symbol | index::Int)\n\nDetermine the declared type of a field (specified by name or index) in a composite DataType T.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtype(Foo, :x)\nInt64\n\njulia> fieldtype(Foo, 2)\nString\n\n\n\n\n\n"},{"title":"Base.fieldtypes","page":"Essentials","location":"base/base.html#Base.fieldtypes","category":"function","text":"fieldtypes(T::Type)\n\nThe declared types of all fields in a composite DataType T as a tuple.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtypes(Foo)\n(Int64, String)\n\n\n\n\n\n"},{"title":"Base.fieldcount","page":"Essentials","location":"base/base.html#Base.fieldcount","category":"function","text":"fieldcount(t::Type)\n\nGet the number of fields that an instance of the given type would have. An error is thrown if the type is too abstract to determine this.\n\n\n\n\n\n"},{"title":"Base.hasfield","page":"Essentials","location":"base/base.html#Base.hasfield","category":"function","text":"hasfield(T::Type, name::Symbol)\n\nReturn a boolean indicating whether T has name as one of its own fields.\n\nSee also fieldnames, fieldcount, hasproperty.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\nExamples\n\njulia> struct Foo\n            bar::Int\n       end\n\njulia> hasfield(Foo, :bar)\ntrue\n\njulia> hasfield(Foo, :x)\nfalse\n\n\n\n\n\n"},{"title":"Core.nfields","page":"Essentials","location":"base/base.html#Core.nfields","category":"function","text":"nfields(x)::Int\n\nGet the number of fields in the given object.\n\nExamples\n\njulia> a = 1//2;\n\njulia> nfields(a)\n2\n\njulia> b = 1\n1\n\njulia> nfields(b)\n0\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> nfields(ex)\n1\n\nIn these examples, a is a Rational, which has two fields. b is an Int, which is a primitive bitstype with no fields at all. ex is an ErrorException, which has one field.\n\n\n\n\n\n"},{"title":"Base.isconst","page":"Essentials","location":"base/base.html#Base.isconst","category":"function","text":"isconst(t::DataType, s::Union{Int,Symbol})::Bool\n\nDetermine whether a field s is const in a given type t in the sense that a read from said field is consistent for egal objects. Note in particular that out-of-bounds fields are considered const under this definition (because they always throw).\n\n\n\n\n\nisconst(m::Module, s::Symbol)::Bool\nisconst(g::GlobalRef)::Bool\n\nDetermine whether a global is const in a given module m, either because it was declared constant or because it was imported from a constant binding. Note that constant-ness is specific to a particular world age, so the result of this function may not be assumed to hold after a world age update.\n\n\n\n\n\n"},{"title":"Base.isfieldatomic","page":"Essentials","location":"base/base.html#Base.isfieldatomic","category":"function","text":"isfieldatomic(t::DataType, s::Union{Int,Symbol})::Bool\n\nDetermine whether a field s is declared @atomic in a given type t.\n\n\n\n\n\n"},{"title":"Base.sizeof","page":"Essentials","location":"base/base.html#Base.sizeof-Tuple{Type}","category":"method","text":"sizeof(T::DataType)\nsizeof(obj)\n\nSize, in bytes, of the canonical binary representation of the given DataType T, if any. Or the size, in bytes, of object obj if it is not a DataType.\n\nSee also Base.summarysize.\n\nExamples\n\njulia> sizeof(Float32)\n4\n\njulia> sizeof(ComplexF64)\n16\n\njulia> sizeof(1.0)\n8\n\njulia> sizeof(collect(1.0:10.0))\n80\n\njulia> struct StructWithPadding\n           x::Int64\n           flag::Bool\n       end\n\njulia> sizeof(StructWithPadding) # not the sum of `sizeof` of fields due to padding\n16\n\njulia> sizeof(Int64) + sizeof(Bool) # different from above\n9\n\nIf DataType T does not have a specific size, an error is thrown.\n\njulia> sizeof(AbstractArray)\nERROR: Abstract type AbstractArray does not have a definite size.\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.isconcretetype","page":"Essentials","location":"base/base.html#Base.isconcretetype","category":"function","text":"isconcretetype(T)\n\nDetermine whether type T is a concrete type, meaning it could have direct instances (values x such that typeof(x) === T). Note that this is not the negation of isabstracttype(T). If T is not a type, then return false.\n\nnote: Note\nWhile concrete types are not abstract and vice versa, types can be neither concrete nor abstract (for example, Vector (a UnionAll)).\n\nnote: Note\nT must be the exact type that would be returned from typeof.  It is possible for a type U to exist such that T == U, isconcretetype(T), but !isconcretetype(U).\n\nSee also: isbits, isabstracttype, issingletontype.\n\nExamples\n\njulia> isconcretetype(Complex)\nfalse\n\njulia> isconcretetype(Complex{Float32})\ntrue\n\njulia> isconcretetype(Vector)\nfalse\n\njulia> isconcretetype(Vector{Complex})\ntrue\n\njulia> isconcretetype(Vector{Complex{Float32}})\ntrue\n\njulia> isconcretetype(Union{})\nfalse\n\njulia> isconcretetype(Union{Int,String})\nfalse\n\njulia> isconcretetype(Tuple{T} where T<:Int)\nfalse\n\n\n\n\n\n"},{"title":"Base.isbits","page":"Essentials","location":"base/base.html#Base.isbits","category":"function","text":"isbits(x)\n\nReturn true if x is an instance of an isbitstype type.\n\n\n\n\n\n"},{"title":"Base.isbitstype","page":"Essentials","location":"base/base.html#Base.isbitstype","category":"function","text":"isbitstype(T)\n\nReturn true if type T is a \"plain data\" type, meaning it is immutable and contains no references to other values, only primitive types and other isbitstype types. Typical examples are numeric types such as UInt8, Float64, and Complex{Float64}. This category of types is significant since they are valid as type parameters, may not track isdefined / isassigned status, and have a defined layout that is compatible with C. If T is not a type, then return false.\n\nSee also isbits, isprimitivetype, ismutable.\n\nExamples\n\njulia> isbitstype(Complex{Float64})\ntrue\n\njulia> isbitstype(Complex)\nfalse\n\n\n\n\n\n"},{"title":"Base.fieldoffset","page":"Essentials","location":"base/base.html#Base.fieldoffset","category":"function","text":"fieldoffset(type, name::Symbol | i::Integer)\n\nThe byte offset of a field (specified by name or index) of a type relative to its start.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldoffset(Foo, 2)\n0x0000000000000008\n\njulia> fieldoffset(Foo, :x)\n0x0000000000000000\n\nWe can use it to summarize information about a struct:\n\njulia> structinfo(T) = [(fieldoffset(T,i), fieldname(T,i), fieldtype(T,i)) for i = 1:fieldcount(T)];\n\njulia> structinfo(Base.Filesystem.StatStruct)\n14-element Vector{Tuple{UInt64, Symbol, Type}}:\n (0x0000000000000000, :desc, Union{RawFD, String})\n (0x0000000000000008, :device, UInt64)\n (0x0000000000000010, :inode, UInt64)\n (0x0000000000000018, :mode, UInt64)\n (0x0000000000000020, :nlink, Int64)\n (0x0000000000000028, :uid, UInt64)\n (0x0000000000000030, :gid, UInt64)\n (0x0000000000000038, :rdev, UInt64)\n (0x0000000000000040, :size, Int64)\n (0x0000000000000048, :blksize, Int64)\n (0x0000000000000050, :blocks, Int64)\n (0x0000000000000058, :mtime, Float64)\n (0x0000000000000060, :ctime, Float64)\n (0x0000000000000068, :ioerrno, Int32)\n\ncompat: Julia 1.13\nSpecifying the field by name rather than index requires Julia 1.13 or later.\n\n\n\n\n\n"},{"title":"Base.datatype_alignment","page":"Essentials","location":"base/base.html#Base.datatype_alignment","category":"function","text":"Base.datatype_alignment(dt::DataType)::Int\n\nMemory allocation minimum alignment for instances of this type. Can be called on any isconcretetype, although for Memory it will give the alignment of the elements, not the whole object.\n\n\n\n\n\n"},{"title":"Base.datatype_haspadding","page":"Essentials","location":"base/base.html#Base.datatype_haspadding","category":"function","text":"Base.datatype_haspadding(dt::DataType)::Bool\n\nReturn whether the fields of instances of this type are packed in memory, with no intervening padding bits (defined as bits whose value does not impact the semantic value of the instance itself). Can be called on any isconcretetype.\n\n\n\n\n\n"},{"title":"Base.datatype_pointerfree","page":"Essentials","location":"base/base.html#Base.datatype_pointerfree","category":"function","text":"Base.datatype_pointerfree(dt::DataType)::Bool\n\nReturn whether instances of this type can contain references to gc-managed memory. Can be called on any isconcretetype.\n\n\n\n\n\n"},{"title":"Base.typemin","page":"Essentials","location":"base/base.html#Base.typemin","category":"function","text":"typemin(T)\n\nThe lowest value representable by the given (real) numeric DataType T.\n\nSee also: floatmin, maxintfloat, typemax, eps.\n\nExamples\n\njulia> typemin(Int8)\n-128\n\njulia> typemin(UInt32)\n0x00000000\n\njulia> typemin(Float16)\n-Inf16\n\njulia> typemin(Float32)\n-Inf32\n\njulia> floatmin(Float32)  # smallest positive finite Float32 floating point number\n1.1754944f-38\n\njulia> nextfloat(-Inf32) == -floatmax(Float32)  # equivalent ways of getting the lowest finite Float32 floating point number\ntrue\n\n\n\n\n\n"},{"title":"Base.typemax","page":"Essentials","location":"base/base.html#Base.typemax","category":"function","text":"typemax(T)\n\nThe highest value representable by the given (real) numeric DataType.\n\nSee also: floatmax, maxintfloat, typemin, eps.\n\nExamples\n\njulia> typemax(Int8)\n127\n\njulia> typemax(UInt32)\n0xffffffff\n\njulia> typemax(Float64)\nInf\n\njulia> typemax(Float32)\nInf32\n\njulia> floatmax(Float32)  # largest positive finite Float32 floating point number\n3.4028235f38\n\n\n\n\n\n"},{"title":"Base.floatmin","page":"Essentials","location":"base/base.html#Base.floatmin","category":"function","text":"floatmin(T = Float64)\n\nReturn the smallest positive normal number representable by the floating-point type T.\n\nSee also: typemin, maxintfloat, floatmax, eps.\n\nExamples\n\njulia> floatmin(Float16)\nFloat16(6.104e-5)\n\njulia> floatmin(Float32)\n1.1754944f-38\n\njulia> floatmin()\n2.2250738585072014e-308\n\n\n\n\n\n"},{"title":"Base.floatmax","page":"Essentials","location":"base/base.html#Base.floatmax","category":"function","text":"floatmax(T = Float64)\n\nReturn the largest finite number representable by the floating-point type T.\n\nSee also: typemax, maxintfloat, floatmin, eps.\n\nExamples\n\njulia> floatmax(Float16)\nFloat16(6.55e4)\n\njulia> floatmax(Float32)\n3.4028235f38\n\njulia> floatmax()\n1.7976931348623157e308\n\njulia> typemax(Float64)\nInf\n\n\n\n\n\n"},{"title":"Base.maxintfloat","page":"Essentials","location":"base/base.html#Base.maxintfloat","category":"function","text":"maxintfloat(T, S)\n\nThe largest consecutive integer representable in the given floating-point type T that also does not exceed the maximum integer representable by the integer type S.  Equivalently, it is the minimum of maxintfloat(T) and typemax(S).\n\n\n\n\n\nmaxintfloat(T=Float64)\n\nThe largest consecutive integer-valued floating-point number that is exactly represented in the given floating-point type T (which defaults to Float64).\n\nThat is, maxintfloat returns the smallest positive integer-valued floating-point number n such that n+1 is not exactly representable in the type T.\n\nWhen an Integer-type value is needed, use Integer(maxintfloat(T)).\n\nSee also: typemax, floatmax.\n\n\n\n\n\n"},{"title":"Base.eps","page":"Essentials","location":"base/base.html#Base.eps-Tuple{Type{<:AbstractFloat}}","category":"method","text":"eps(::Type{T}) where T<:AbstractFloat\neps()\n\nReturn the machine epsilon of the floating point type T (T = Float64 by default). This is defined as the gap between 1 and the next largest value representable by typeof(one(T)), and is equivalent to eps(one(T)).  (Since eps(T) is a bound on the relative error of T, it is a \"dimensionless\" quantity like one.)\n\nExamples\n\njulia> eps()\n2.220446049250313e-16\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> 1.0 + eps()\n1.0000000000000002\n\njulia> 1.0 + eps()/2\n1.0\n\nMore generally, for any floating-point numeric type, eps corresponds to an upper bound on the distance to the nearest floating-point complex value: if extfl(x) is the closest floating-point value to a number x (e.g. an arbitrary real number), then extfl(x) satisfies x - \textfl(x)  \texteps(x)2, not including overflow cases. This allows the definition of eps to be extended to complex numbers, for which extfl(a + ib) = \textfl(a) + i \textfl(b).\n\n\n\n\n\n"},{"title":"Base.eps","page":"Essentials","location":"base/base.html#Base.eps-Tuple{AbstractFloat}","category":"method","text":"eps(x::AbstractFloat)\n\nReturn the unit in last place (ulp) of x. This is the distance between consecutive representable floating point values at x. In most cases, if the distance on either side of x is different, then the larger of the two is taken, that is\n\neps(x) == max(x-prevfloat(x), nextfloat(x)-x)\n\nThe exceptions to this rule are the smallest and largest finite values (e.g. nextfloat(-Inf) and prevfloat(Inf) for Float64), which round to the smaller of the values.\n\nThe rationale for this behavior is that eps bounds the floating point rounding error. Under the default RoundNearest rounding mode, if y is a real number and x is the nearest floating point number to y, then\n\ny-x leq operatornameeps(x)2\n\nSee also: nextfloat, issubnormal, floatmax.\n\nExamples\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(prevfloat(2.0))\n2.220446049250313e-16\n\njulia> eps(2.0)\n4.440892098500626e-16\n\njulia> x = prevfloat(Inf)      # largest finite Float64\n1.7976931348623157e308\n\njulia> x + eps(x)/2            # rounds up\nInf\n\njulia> x + prevfloat(eps(x)/2) # rounds down\n1.7976931348623157e308\n\n\n\n\n\n"},{"title":"Base.instances","page":"Essentials","location":"base/base.html#Base.instances","category":"function","text":"instances(T::Type)\n\nReturn a collection of all instances of the given type, if applicable. Mostly used for enumerated types (see @enum).\n\nExamples\n\njulia> @enum Color red blue green\n\njulia> instances(Color)\n(red, blue, green)\n\n\n\n\n\n"},{"title":"Core.Any","page":"Essentials","location":"base/base.html#Core.Any","category":"type","text":"Any::DataType\n\nAny is the union of all types. It has the defining property isa(x, Any) == true for any x. Any therefore describes the entire universe of possible values. For example Integer is a subset of Any that includes Int, Int8, and other integer types.\n\n\n\n\n\n"},{"title":"Core.Union","page":"Essentials","location":"base/base.html#Core.Union","category":"type","text":"Union{Types...}\n\nA Union type is an abstract type which includes all instances of any of its argument types. This means that T <: Union{T,S} and S <: Union{T,S}.\n\nLike other abstract types, it cannot be instantiated, even if all of its arguments are non abstract.\n\nExamples\n\njulia> IntOrString = Union{Int,AbstractString}\nUnion{Int64, AbstractString}\n\njulia> 1 isa IntOrString # instance of Int is included in the union\ntrue\n\njulia> \"Hello!\" isa IntOrString # String is also included\ntrue\n\njulia> 1.0 isa IntOrString # Float64 is not included because it is neither Int nor AbstractString\nfalse\n\nExtended Help\n\nUnlike most other parametric types, unions are covariant in their parameters. For example, Union{Real, String} is a subtype of Union{Number, AbstractString}.\n\nThe empty union Union{} is the bottom type of Julia.\n\n\n\n\n\n"},{"title":"Union{}","page":"Essentials","location":"base/base.html#Union{}","category":"keyword","text":"Union{}\n\nUnion{}, the empty Union of types, is the bottom type of the type system. That is, for each T::Type, Union{} <: T. Also see the subtyping operator's documentation: <:.\n\nAs such, Union{} is also an empty/uninhabited type, meaning that it has no values. That is, for each x, isa(x, Union{}) == false.\n\nBase.Bottom is defined as its alias and the type of Union{} is Core.TypeofBottom.\n\nExamples\n\njulia> isa(nothing, Union{})\nfalse\n\njulia> Union{} <: Int\ntrue\n\njulia> typeof(Union{}) === Core.TypeofBottom\ntrue\n\njulia> isa(Union{}, Union)\nfalse\n\n\n\n\n\n"},{"title":"Core.TypeofBottom","page":"Essentials","location":"base/base.html#Core.TypeofBottom","category":"type","text":"Core.TypeofBottom\n\nThe singleton type containing only the value Union{} (which represents the empty type).\n\n\n\n\n\n"},{"title":"Core.UnionAll","page":"Essentials","location":"base/base.html#Core.UnionAll","category":"type","text":"UnionAll\n\nA union of types over all values of a type parameter. UnionAll is used to describe parametric types where the values of some parameters are not known. See the manual section on UnionAll Types.\n\nExamples\n\njulia> typeof(Vector)\nUnionAll\n\njulia> typeof(Vector{Int})\nDataType\n\n\n\n\n\n"},{"title":"Core.Tuple","page":"Essentials","location":"base/base.html#Core.Tuple","category":"type","text":"Tuple{Types...}\n\nA tuple is a fixed-length container that can hold any values of different types, but cannot be modified (it is immutable). The values can be accessed via indexing. Tuple literals are written with commas and parentheses:\n\njulia> (1, 1+1)\n(1, 2)\n\njulia> (1,)\n(1,)\n\njulia> x = (0.0, \"hello\", 6*7)\n(0.0, \"hello\", 42)\n\njulia> x[2]\n\"hello\"\n\njulia> typeof(x)\nTuple{Float64, String, Int64}\n\nA length-1 tuple must be written with a comma, (1,), since (1) would just be a parenthesized value. () represents the empty (length-0) tuple.\n\nA tuple can be constructed from an iterator by using a Tuple type as constructor:\n\njulia> Tuple([\"a\", 1])\n(\"a\", 1)\n\njulia> Tuple{String, Float64}([\"a\", 1])\n(\"a\", 1.0)\n\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore Tuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters are. Tuples do not have field names; fields are only accessed by index. Tuple types may have any number of parameters.\n\nSee the manual section on Tuple Types.\n\nSee also Vararg, NTuple, ntuple, tuple, NamedTuple.\n\n\n\n\n\n"},{"title":"Core.NTuple","page":"Essentials","location":"base/base.html#Core.NTuple","category":"type","text":"NTuple{N, T}\n\nA compact way of representing the type for a tuple of length N where all elements are of type T.\n\nExamples\n\njulia> isa((1, 2, 3, 4, 5, 6), NTuple{6, Int})\ntrue\n\nSee also ntuple.\n\n\n\n\n\n"},{"title":"Core.NamedTuple","page":"Essentials","location":"base/base.html#Core.NamedTuple","category":"type","text":"NamedTuple\n\nNamedTuples are, as their name suggests, named Tuples. That is, they're a tuple-like collection of values, where each entry has a unique name, represented as a Symbol. Like Tuples, NamedTuples are immutable; neither the names nor the values can be modified in place after construction.\n\nA named tuple can be created as a tuple literal with keys, e.g. (a=1, b=2), or as a tuple literal with semicolon after the opening parenthesis, e.g. (; a=1, b=2) (this form also accepts programmatically generated names as described below), or using a NamedTuple type as constructor, e.g. NamedTuple{(:a, :b)}((1,2)).\n\nAccessing the value associated with a name in a named tuple can be done using field access syntax, e.g. x.a, or using getindex, e.g. x[:a] or x[(:a, :b)]. A tuple of the names can be obtained using keys, and a tuple of the values can be obtained using values.\n\nnote: Note\nIteration over NamedTuples produces the values without the names. (See example below.) To iterate over the name-value pairs, use the pairs function.\n\nThe @NamedTuple macro can be used for conveniently declaring NamedTuple types.\n\nExamples\n\njulia> x = (a=1, b=2)\n(a = 1, b = 2)\n\njulia> x.a\n1\n\njulia> x[:a]\n1\n\njulia> x[(:a,)]\n(a = 1,)\n\njulia> keys(x)\n(:a, :b)\n\njulia> values(x)\n(1, 2)\n\njulia> collect(x)\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(pairs(x))\n2-element Vector{Pair{Symbol, Int64}}:\n :a => 1\n :b => 2\n\nIn a similar fashion as to how one can define keyword arguments programmatically, a named tuple can be created by giving pairs name::Symbol => value after a semicolon inside a tuple literal. This and the name=value syntax can be mixed:\n\njulia> (; :a => 1, :b => 2, c=3)\n(a = 1, b = 2, c = 3)\n\nThe name-value pairs can also be provided by splatting a named tuple or any iterator that yields two-value collections holding each a symbol as first value:\n\njulia> keys = (:a, :b, :c); values = (1, 2, 3);\n\njulia> NamedTuple{keys}(values)\n(a = 1, b = 2, c = 3)\n\njulia> (; (keys .=> values)...)\n(a = 1, b = 2, c = 3)\n\njulia> nt1 = (a=1, b=2);\n\njulia> nt2 = (c=3, d=4);\n\njulia> (; nt1..., nt2..., b=20) # the final b overwrites the value from nt1\n(a = 1, b = 20, c = 3, d = 4)\n\njulia> (; zip(keys, values)...) # zip yields tuples such as (:a, 1)\n(a = 1, b = 2, c = 3)\n\nAs in keyword arguments, identifiers and dot expressions imply names:\n\njulia> x = 0\n0\n\njulia> t = (; x)\n(x = 0,)\n\njulia> (; t.x)\n(x = 0,)\n\ncompat: Julia 1.5\nImplicit names from identifiers and dot expressions are available as of Julia 1.5.\n\ncompat: Julia 1.7\nUse of getindex methods with multiple Symbols is available as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.@NamedTuple","page":"Essentials","location":"base/base.html#Base.@NamedTuple","category":"macro","text":"@NamedTuple{key1::Type1, key2::Type2, ...}\n@NamedTuple begin key1::Type1; key2::Type2; ...; end\n\nThis macro gives a more convenient syntax for declaring NamedTuple types. It returns a NamedTuple type with the given keys and types, equivalent to NamedTuple{(:key1, :key2, ...), Tuple{Type1,Type2,...}}. If the ::Type declaration is omitted, it is taken to be Any.   The begin ... end form allows the declarations to be split across multiple lines (similar to a struct declaration), but is otherwise equivalent. The NamedTuple macro is used when printing NamedTuple types to e.g. the REPL.\n\nFor example, the tuple (a=3.1, b=\"hello\") has a type NamedTuple{(:a, :b), Tuple{Float64, String}}, which can also be declared via @NamedTuple as:\n\njulia> @NamedTuple{a::Float64, b::String}\n@NamedTuple{a::Float64, b::String}\n\njulia> @NamedTuple begin\n           a::Float64\n           b::String\n       end\n@NamedTuple{a::Float64, b::String}\n\ncompat: Julia 1.5\nThis macro is available as of Julia 1.5.\n\n\n\n\n\n"},{"title":"Base.@Kwargs","page":"Essentials","location":"base/base.html#Base.@Kwargs","category":"macro","text":"@Kwargs{key1::Type1, key2::Type2, ...}\n\nThis macro gives a convenient way to construct the type representation of keyword arguments from the same syntax as @NamedTuple. For example, when we have a function call like func([positional arguments]; kw1=1.0, kw2=\"2\"), we can use this macro to construct the internal type representation of the keyword arguments as @Kwargs{kw1::Float64, kw2::String}. The macro syntax is specifically designed to simplify the signature type of a keyword method when it is printed in the stack trace view.\n\njulia> @Kwargs{init::Int} # the internal representation of keyword arguments\nBase.Pairs{Symbol, Int64, Nothing, @NamedTuple{init::Int64}}\n\njulia> sum(\"julia\"; init=1)\nERROR: MethodError: no method matching +(::Char, ::Char)\nThe function `+` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  +(::Any, ::Any, ::Any, ::Any...)\n   @ Base operators.jl:585\n  +(::Integer, ::AbstractChar)\n   @ Base char.jl:247\n  +(::T, ::Integer) where T<:AbstractChar\n   @ Base char.jl:237\n\nStacktrace:\n  [1] add_sum(x::Char, y::Char)\n    @ Base ./reduce.jl:24\n  [2] BottomRF\n    @ Base ./reduce.jl:86 [inlined]\n  [3] _foldl_impl(op::Base.BottomRF{typeof(Base.add_sum)}, init::Int64, itr::String)\n    @ Base ./reduce.jl:62\n  [4] foldl_impl(op::Base.BottomRF{typeof(Base.add_sum)}, nt::Int64, itr::String)\n    @ Base ./reduce.jl:48 [inlined]\n  [5] mapfoldl_impl(f::typeof(identity), op::typeof(Base.add_sum), nt::Int64, itr::String)\n    @ Base ./reduce.jl:44 [inlined]\n  [6] mapfoldl(f::typeof(identity), op::typeof(Base.add_sum), itr::String; init::Int64)\n    @ Base ./reduce.jl:175 [inlined]\n  [7] mapreduce(f::typeof(identity), op::typeof(Base.add_sum), itr::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:307 [inlined]\n  [8] sum(f::typeof(identity), a::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:535 [inlined]\n  [9] sum(a::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:564 [inlined]\n [10] top-level scope\n    @ REPL[12]:1\n\ncompat: Julia 1.10\nThis macro is available as of Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.Val","page":"Essentials","location":"base/base.html#Base.Val","category":"type","text":"Val(c)\n\nReturn Val{c}(), which contains no run-time data. Types like this can be used to pass the information between functions through the value c, which must be an isbits value or a Symbol. The intent of this construct is to be able to dispatch on constants directly (at compile time) without having to test the value of the constant at run time.\n\nExamples\n\njulia> f(::Val{true}) = \"Good\"\nf (generic function with 1 method)\n\njulia> f(::Val{false}) = \"Bad\"\nf (generic function with 2 methods)\n\njulia> f(Val(true))\n\"Good\"\n\n\n\n\n\n"},{"title":"Core.Vararg","page":"Essentials","location":"base/base.html#Core.Vararg","category":"constant","text":"Vararg{T,N}\n\nThe last parameter of a tuple type Tuple can be the special value Vararg, which denotes any number of trailing elements. Vararg{T,N} corresponds to exactly N elements of type T. Finally Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are used to represent the arguments accepted by varargs methods (see the section on Varargs Functions in the manual.)\n\nSee also NTuple.\n\nExamples\n\njulia> mytupletype = Tuple{AbstractString, Vararg{Int}}\nTuple{AbstractString, Vararg{Int64}}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalse\n\n\n\n\n\n"},{"title":"Core.Nothing","page":"Essentials","location":"base/base.html#Core.Nothing","category":"type","text":"Nothing\n\nA type with no fields that is the type of nothing.\n\nSee also: isnothing, Some, Missing.\n\n\n\n\n\n"},{"title":"Base.isnothing","page":"Essentials","location":"base/base.html#Base.isnothing","category":"function","text":"isnothing(x)\n\nReturn true if x === nothing, and return false if not.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nSee also something, Base.notnothing, ismissing.\n\n\n\n\n\n"},{"title":"Base.notnothing","page":"Essentials","location":"base/base.html#Base.notnothing","category":"function","text":"notnothing(x)\n\nThrow an error if x === nothing, and return x if not.\n\n\n\n\n\n"},{"title":"Base.Some","page":"Essentials","location":"base/base.html#Base.Some","category":"type","text":"Some{T}\n\nA wrapper type used in Union{Some{T}, Nothing} to distinguish between the absence of a value (nothing) and the presence of a nothing value (i.e. Some(nothing)).\n\nUse something to access the value wrapped by a Some object.\n\n\n\n\n\n"},{"title":"Base.something","page":"Essentials","location":"base/base.html#Base.something","category":"function","text":"something(x...)\n\nReturn the first value in the arguments which is not equal to nothing, if any. Otherwise throw an error. Arguments of type Some are unwrapped.\n\nSee also coalesce, skipmissing, @something.\n\nExamples\n\njulia> something(nothing, 1)\n1\n\njulia> something(Some(1), nothing)\n1\n\njulia> something(Some(nothing), 2) === nothing\ntrue\n\njulia> something(missing, nothing)\nmissing\n\njulia> something(nothing, nothing)\nERROR: ArgumentError: No value arguments present\n\n\n\n\n\n"},{"title":"Base.@something","page":"Essentials","location":"base/base.html#Base.@something","category":"macro","text":"@something(x...)\n\nShort-circuiting version of something.\n\nExamples\n\njulia> f(x) = (println(\"f($x)\"); nothing);\n\njulia> a = 1;\n\njulia> a = @something a f(2) f(3) error(\"Unable to find default for `a`\")\n1\n\njulia> b = nothing;\n\njulia> b = @something b f(2) f(3) error(\"Unable to find default for `b`\")\nf(2)\nf(3)\nERROR: Unable to find default for `b`\n[...]\n\njulia> b = @something b f(2) f(3) Some(nothing)\nf(2)\nf(3)\n\njulia> b === nothing\ntrue\n\ncompat: Julia 1.7\nThis macro is available as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.Enums.Enum","page":"Essentials","location":"base/base.html#Base.Enums.Enum","category":"type","text":"Enum{T<:Integer}\n\nThe abstract supertype of all enumerated types defined with @enum.\n\n\n\n\n\n"},{"title":"Base.Enums.@enum","page":"Essentials","location":"base/base.html#Base.Enums.@enum","category":"macro","text":"@enum EnumName[::BaseType] value1[=x] value2[=y]\n\nCreate an Enum{BaseType} subtype with name EnumName and enum member values of value1 and value2 with optional assigned values of x and y, respectively. EnumName can be used just like other types and enum member values as regular values, such as\n\nExamples\n\njulia> @enum Fruit apple=1 orange=2 kiwi=3\n\njulia> f(x::Fruit) = \"I'm a Fruit with value: $(Int(x))\"\nf (generic function with 1 method)\n\njulia> f(apple)\n\"I'm a Fruit with value: 1\"\n\njulia> Fruit(1)\napple::Fruit = 1\n\nValues can also be specified inside a begin block, e.g.\n\n@enum EnumName begin\n    value1\n    value2\nend\n\nBaseType, which defaults to Int32, must be a primitive subtype of Integer. Member values can be converted between the enum type and BaseType. read and write perform these conversions automatically. In case the enum is created with a non-default BaseType, Integer(value1) will return the integer value1 with the type BaseType.\n\nTo list all the instances of an enum use instances, e.g.\n\njulia> instances(Fruit)\n(apple, orange, kiwi)\n\nIt is possible to construct a symbol from an enum instance:\n\njulia> Symbol(apple)\n:apple\n\n\n\n\n\n"},{"title":"Core.Expr","page":"Essentials","location":"base/base.html#Core.Expr","category":"type","text":"Expr(head::Symbol, args...)\n\nA type representing compound expressions in parsed julia code (ASTs). Each expression consists of a head Symbol identifying which kind of expression it is (e.g. a call, for loop, conditional statement, etc.), and subexpressions (e.g. the arguments of a call). The subexpressions are stored in a Vector{Any} field called args.\n\nSee the manual chapter on Metaprogramming and the developer documentation Julia ASTs.\n\nExamples\n\njulia> Expr(:call, :+, 1, 2)\n:(1 + 2)\n\njulia> dump(:(a ? b : c))\nExpr\n  head: Symbol if\n  args: Array{Any}((3,))\n    1: Symbol a\n    2: Symbol b\n    3: Symbol c\n\n\n\n\n\n"},{"title":"Core.Symbol","page":"Essentials","location":"base/base.html#Core.Symbol","category":"type","text":"Symbol\n\nThe type of object used to represent identifiers in parsed julia code (ASTs). Also often used as a name or label to identify an entity (e.g. as a dictionary key). Symbols can be entered using the : quote operator:\n\njulia> :name\n:name\n\njulia> typeof(:name)\nSymbol\n\njulia> x = 42\n42\n\njulia> eval(:x)\n42\n\nSymbols can also be constructed from strings or other values by calling the constructor Symbol(x...).\n\nSymbols are immutable and their implementation re-uses the same object for all Symbols with the same name.\n\nUnlike strings, Symbols are \"atomic\" or \"scalar\" entities that do not support iteration over characters.\n\n\n\n\n\n"},{"title":"Core.Symbol","page":"Essentials","location":"base/base.html#Core.Symbol-Tuple","category":"method","text":"Symbol(x...)::Symbol\n\nCreate a Symbol by concatenating the string representations of the arguments together.\n\nExamples\n\njulia> Symbol(\"my\", \"name\")\n:myname\n\njulia> Symbol(\"day\", 4)\n:day4\n\n\n\n\n\n"},{"title":"Core.Module","page":"Essentials","location":"base/base.html#Core.Module","category":"type","text":"Module\n\nA Module is a separate global variable workspace. See module and the manual section about modules for details.\n\nModule(name::Symbol=:anonymous, std_imports=true, default_names=true)\n\nReturn a module with the specified name. A baremodule corresponds to Module(:ModuleName, false)\n\nAn empty module containing no names at all can be created with Module(:ModuleName, false, false). This module will not import Base or Core and does not contain a reference to itself.\n\n\n\n\n\n"},{"title":"Core.Function","page":"Essentials","location":"base/base.html#Core.Function","category":"type","text":"Function\n\nAbstract type of all functions.\n\nExamples\n\njulia> isa(+, Function)\ntrue\n\njulia> typeof(sin)\ntypeof(sin) (singleton type of function sin, subtype of Function)\n\njulia> ans <: Function\ntrue\n\n\n\n\n\n"},{"title":"Base.hasmethod","page":"Essentials","location":"base/base.html#Base.hasmethod","category":"function","text":"hasmethod(f, t::Type{<:Tuple}[, kwnames]; world=get_world_counter())::Bool\n\nDetermine whether the given generic function has a method matching the given Tuple of argument types with the upper bound of world age given by world.\n\nIf a tuple of keyword argument names kwnames is provided, this also checks whether the method of f matching t has the given keyword argument names. If the matching method accepts a variable number of keyword arguments, e.g. with kwargs..., any names given in kwnames are considered valid. Otherwise the provided names must be a subset of the method's keyword arguments.\n\nSee also applicable.\n\ncompat: Julia 1.2\nProviding keyword argument names requires Julia 1.2 or later.\n\nExamples\n\njulia> hasmethod(length, Tuple{Array})\ntrue\n\njulia> f(; oranges=0) = oranges;\n\njulia> hasmethod(f, Tuple{}, (:oranges,))\ntrue\n\njulia> hasmethod(f, Tuple{}, (:apples, :bananas))\nfalse\n\njulia> g(; xs...) = 4;\n\njulia> hasmethod(g, Tuple{}, (:a, :b, :c, :d))  # g accepts arbitrary kwargs\ntrue\n\n\n\n\n\n"},{"title":"Core.applicable","page":"Essentials","location":"base/base.html#Core.applicable","category":"function","text":"applicable(f, args...)::Bool\n\nDetermine whether the given generic function has a method applicable to the given arguments.\n\nSee also hasmethod.\n\nExamples\n\njulia> function f(x, y)\n           x + y\n       end;\n\njulia> applicable(f, 1)\nfalse\n\njulia> applicable(f, 1, 2)\ntrue\n\n\n\n\n\n"},{"title":"Base.isambiguous","page":"Essentials","location":"base/base.html#Base.isambiguous","category":"function","text":"Base.isambiguous(m1, m2; ambiguous_bottom=false)::Bool\n\nDetermine whether two methods m1 and m2 may be ambiguous for some call signature. This test is performed in the context of other methods of the same function; in isolation, m1 and m2 might be ambiguous, but if a third method resolving the ambiguity has been defined, this returns false. Alternatively, in isolation m1 and m2 might be ordered, but if a third method cannot be sorted with them, they may cause an ambiguity together.\n\nFor parametric types, the ambiguous_bottom keyword argument controls whether Union{} counts as an ambiguous intersection of type parameters – when true, it is considered ambiguous, when false it is not.\n\nExamples\n\njulia> foo(x::Complex{<:Integer}) = 1\nfoo (generic function with 1 method)\n\njulia> foo(x::Complex{<:Rational}) = 2\nfoo (generic function with 2 methods)\n\njulia> m1, m2 = collect(methods(foo));\n\njulia> typeintersect(m1.sig, m2.sig)\nTuple{typeof(foo), Complex{Union{}}}\n\njulia> Base.isambiguous(m1, m2, ambiguous_bottom=true)\ntrue\n\njulia> Base.isambiguous(m1, m2, ambiguous_bottom=false)\nfalse\n\n\n\n\n\n"},{"title":"Core.invoke","page":"Essentials","location":"base/base.html#Core.invoke","category":"function","text":"invoke(f, argtypes::Type, args...; kwargs...)\ninvoke(f, argtypes::Method, args...; kwargs...)\ninvoke(f, argtypes::CodeInstance, args...; kwargs...)\n\nInvoke a method for the given generic function f matching the specified types argtypes on the specified arguments args and passing the keyword arguments kwargs. The arguments args must conform with the specified types in argtypes, i.e. conversion is not automatically performed. This method allows invoking a method other than the most specific matching method, which is useful when the behavior of a more general definition is explicitly needed (often as part of the implementation of a more specific method of the same function). However, because this means the runtime must do more work, invoke is generally also slower–sometimes significantly so–than doing normal dispatch with a regular call.\n\nBe careful when using invoke for functions that you don't write. What definition is used for given argtypes is an implementation detail unless the function is explicitly states that calling with certain argtypes is a part of public API.  For example, the change between f1 and f2 in the example below is usually considered compatible because the change is invisible by the caller with a normal (non-invoke) call.  However, the change is visible if you use invoke.\n\nPassing a Method instead of a signature\n\nThe argtypes argument may be a Method, in which case the ordinary method table lookup is bypassed entirely and the given method is invoked directly. Needing this feature is uncommon. Note in particular that the specified Method may be entirely unreachable from ordinary dispatch (or ordinary invoke), e.g. because it was replaced or fully covered by more specific methods. If the method is part of the ordinary method table, this call behaves similar to invoke(f, method.sig, args...).\n\ncompat: Julia 1.12\nPassing a Method requires Julia 1.12.\n\nPassing a CodeInstance instead of a signature\n\nThe argtypes argument may be a CodeInstance, bypassing both method lookup and specialization. The semantics of this invocation are similar to a function pointer call of the CodeInstance's invoke pointer. It is an error to invoke a CodeInstance with arguments that do not match its parent MethodInstance or from a world age not included in the min_world/max_world range. It is undefined behavior to invoke a CodeInstance whose behavior does not match the constraints specified in its fields. For some code instances with owner !== nothing (i.e. those generated by external compilers), it may be an error to invoke them after passing through precompilation. This is an advanced interface intended for use with external compiler plugins.\n\ncompat: Julia 1.12\nPassing a CodeInstance requires Julia 1.12.\n\nExamples\n\njulia> f(x::Real) = x^2;\n\njulia> f(x::Integer) = 1 + invoke(f, Tuple{Real}, x);\n\njulia> f(2)\n5\n\njulia> f1(::Integer) = Integer\n       f1(::Real) = Real;\n\njulia> f2(x::Real) = _f2(x)\n       _f2(::Integer) = Integer\n       _f2(_) = Real;\n\njulia> f1(1)\nInteger\n\njulia> f2(1)\nInteger\n\njulia> invoke(f1, Tuple{Real}, 1)\nReal\n\njulia> invoke(f2, Tuple{Real}, 1)\nInteger\n\n\n\n\n\n"},{"title":"Base.@invoke","page":"Essentials","location":"base/base.html#Base.@invoke","category":"macro","text":"@invoke f(arg::T, ...; kwargs...)\n\nProvides a convenient way to call invoke by expanding @invoke f(arg1::T1, arg2::T2; kwargs...) to invoke(f, Tuple{T1,T2}, arg1, arg2; kwargs...). When an argument's type annotation is omitted, it's replaced with Core.Typeof that argument. To invoke a method where an argument is untyped or explicitly typed as Any, annotate the argument with ::Any.\n\nIt also supports the following syntax:\n\n@invoke (x::X).f expands to invoke(getproperty, Tuple{X,Symbol}, x, :f)\n@invoke (x::X).f = v::V expands to invoke(setproperty!, Tuple{X,Symbol,V}, x, :f, v)\n@invoke (xs::Xs)[i::I] expands to invoke(getindex, Tuple{Xs,I}, xs, i)\n@invoke (xs::Xs)[i::I] = v::V expands to invoke(setindex!, Tuple{Xs,V,I}, xs, v, i)\n\nExamples\n\njulia> @macroexpand @invoke f(x::T, y)\n:(Core.invoke(f, Base.Tuple{T, Core.Typeof(y)}, x, y))\n\njulia> @invoke 420::Integer % Unsigned\n0x00000000000001a4\n\njulia> @macroexpand @invoke (x::X).f\n:(Core.invoke(Base.getproperty, Base.Tuple{X, Core.Typeof(:f)}, x, :f))\n\njulia> @macroexpand @invoke (x::X).f = v::V\n:(Core.invoke(Base.setproperty!, Base.Tuple{X, Core.Typeof(:f), V}, x, :f, v))\n\njulia> @macroexpand @invoke (xs::Xs)[i::I]\n:(Core.invoke(Base.getindex, Base.Tuple{Xs, I}, xs, i))\n\njulia> @macroexpand @invoke (xs::Xs)[i::I] = v::V\n:(Core.invoke(Base.setindex!, Base.Tuple{Xs, V, I}, xs, v, i))\n\ncompat: Julia 1.7\nThis macro requires Julia 1.7 or later.\n\ncompat: Julia 1.9\nThis macro is exported as of Julia 1.9.\n\ncompat: Julia 1.10\nThe additional syntax is supported as of Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.invokelatest","page":"Essentials","location":"base/base.html#Base.invokelatest","category":"function","text":"invokelatest(f, args...; kwargs...)\n\nCalls f(args...; kwargs...), but guarantees that the most recent method of f will be executed.   This is useful in specialized circumstances, e.g. long-running event loops or callback functions that may call obsolete versions of a function f. (The drawback is that invokelatest is somewhat slower than calling f directly, and the type of the result cannot be inferred by the compiler.)\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this function was not exported, and was called as Base.invokelatest.\n\n\n\n\n\n"},{"title":"Base.@invokelatest","page":"Essentials","location":"base/base.html#Base.@invokelatest","category":"macro","text":"@invokelatest f(args...; kwargs...)\n\nProvides a convenient way to call invokelatest. @invokelatest f(args...; kwargs...) will simply be expanded into Base.invokelatest(f, args...; kwargs...).\n\nIt also supports the following syntax:\n\n@invokelatest x.f expands to Base.invokelatest(getproperty, x, :f)\n@invokelatest x.f = v expands to Base.invokelatest(setproperty!, x, :f, v)\n@invokelatest xs[i] expands to Base.invokelatest(getindex, xs, i)\n@invokelatest xs[i] = v expands to Base.invokelatest(setindex!, xs, v, i)\n\nnote: Note\nIf f is a global, it will be resolved consistently in the (latest) world as the call target. However, all other arguments (as well as f itself if it is not a literal global) will be evaluated in the current world age.\n\ncompat: Julia 1.7\nThis macro requires Julia 1.7 or later.\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this macro was not exported, and was called as Base.@invokelatest.\n\ncompat: Julia 1.10\nThe additional x.f and xs[i] syntax requires Julia 1.10.\n\n\n\n\n\n"},{"title":"new","page":"Essentials","location":"base/base.html#new","category":"keyword","text":"new, or new{A,B,...}\n\nSpecial function available to inner constructors which creates a new object of the type. The form new{A,B,...} explicitly specifies values of parameters for parametric types. See the manual section on Inner Constructor Methods for more information.\n\n\n\n\n\n"},{"title":"Base.:|>","page":"Essentials","location":"base/base.html#Base.:|>","category":"function","text":"|>(x, f)\n\nInfix operator which applies function f to the argument x. This allows f(g(x)) to be written x |> g |> f. When used with anonymous functions, parentheses are typically required around the definition to get the intended chain.\n\nExamples\n\njulia> 4 |> inv\n0.25\n\njulia> [2, 3, 5] |> sum |> inv\n0.1\n\njulia> [0 1; 2 3] .|> (x -> x^2) |> sum\n14\n\n\n\n\n\n"},{"title":"Base.:∘","page":"Essentials","location":"base/base.html#Base.:∘","category":"function","text":"f ∘ g\n\nCompose functions: i.e. (f ∘ g)(args...; kwargs...) means f(g(args...; kwargs...)). The ∘ symbol can be entered in the Julia REPL (and most editors, appropriately configured) by typing \\circ<tab>.\n\nFunction composition also works in prefix form: ∘(f, g) is the same as f ∘ g. The prefix form supports composition of multiple functions: ∘(f, g, h) = f ∘ g ∘ h and splatting ∘(fs...) for composing an iterable collection of functions. The last argument to ∘ execute first.\n\ncompat: Julia 1.4\nMultiple function composition requires at least Julia 1.4.\n\ncompat: Julia 1.5\nComposition of one function ∘(f) requires at least Julia 1.5.\n\ncompat: Julia 1.7\nUsing keyword arguments requires at least Julia 1.7.\n\nExamples\n\njulia> map(uppercase∘first, [\"apple\", \"banana\", \"carrot\"])\n3-element Vector{Char}:\n 'A': ASCII/Unicode U+0041 (category Lu: Letter, uppercase)\n 'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)\n 'C': ASCII/Unicode U+0043 (category Lu: Letter, uppercase)\n\njulia> (==(6)∘length).([\"apple\", \"banana\", \"carrot\"])\n3-element BitVector:\n 0\n 1\n 1\n\njulia> fs = [\n           x -> 2x\n           x -> x-1\n           x -> x/2\n           x -> x+1\n       ];\n\njulia> ∘(fs...)(3)\n2.0\n\nSee also ComposedFunction, !f::Function.\n\n\n\n\n\n"},{"title":"Base.ComposedFunction","page":"Essentials","location":"base/base.html#Base.ComposedFunction","category":"type","text":"ComposedFunction{Outer,Inner} <: Function\n\nRepresents the composition of two callable objects outer::Outer and inner::Inner. That is\n\nComposedFunction(outer, inner)(args...; kw...) === outer(inner(args...; kw...))\n\nThe preferred way to construct an instance of ComposedFunction is to use the composition operator ∘:\n\njulia> sin ∘ cos === ComposedFunction(sin, cos)\ntrue\n\njulia> typeof(sin∘cos)\nComposedFunction{typeof(sin), typeof(cos)}\n\nThe composed pieces are stored in the fields of ComposedFunction and can be retrieved as follows:\n\njulia> composition = sin ∘ cos\nsin ∘ cos\n\njulia> composition.outer === sin\ntrue\n\njulia> composition.inner === cos\ntrue\n\ncompat: Julia 1.6\nComposedFunction requires at least Julia 1.6. In earlier versions ∘ returns an anonymous function instead.\n\nSee also ∘.\n\n\n\n\n\n"},{"title":"Base.splat","page":"Essentials","location":"base/base.html#Base.splat","category":"function","text":"splat(f)\n\nEquivalent to\n\n    my_splat(f) = args->f(args...)\n\ni.e. given a function returns a new function that takes one argument and splats it into the original function. This is useful as an adaptor to pass a multi-argument function in a context that expects a single argument, but passes a tuple as that single argument.\n\nExamples\n\njulia> map(splat(+), zip(1:3,4:6))\n3-element Vector{Int64}:\n 5\n 7\n 9\n\njulia> my_add = splat(+)\nsplat(+)\n\njulia> my_add((1,2,3))\n6\n\n\n\n\n\n"},{"title":"Base.Fix","page":"Essentials","location":"base/base.html#Base.Fix","category":"type","text":"Fix{N}(f, x)\n\nA type representing a partially-applied version of a function f, with the argument x fixed at position N::Int. In other words, Fix{3}(f, x) behaves similarly to (y1, y2, y3...; kws...) -> f(y1, y2, x, y3...; kws...).\n\ncompat: Julia 1.12\nThis general functionality requires at least Julia 1.12, while Fix1 and Fix2 are available earlier.\n\nnote: Note\nWhen nesting multiple Fix, note that the N in Fix{N} is relative to the current available arguments, rather than an absolute ordering on the target function. For example, Fix{1}(Fix{2}(f, 4), 4) fixes the first and second arg, while Fix{2}(Fix{1}(f, 4), 4) fixes the first and third arg.\n\n\n\n\n\n"},{"title":"Base.Fix1","page":"Essentials","location":"base/base.html#Base.Fix1","category":"type","text":"Alias for Fix{1}. See Fix.\n\n\n\n\n\n"},{"title":"Base.Fix2","page":"Essentials","location":"base/base.html#Base.Fix2","category":"type","text":"Alias for Fix{2}. See Fix.\n\n\n\n\n\n"},{"title":"Base.Returns","page":"Essentials","location":"base/base.html#Base.Returns","category":"type","text":"f = Returns(value)\n\nCreate a callable f such that f(args...; kw...) === value holds.\n\nExamples\n\njulia> f = Returns(42);\n\njulia> f(1)\n42\n\njulia> f(\"hello\", x=32)\n42\n\njulia> f.value\n42\n\ncompat: Julia 1.7\nReturns requires at least Julia 1.7.\n\n\n\n\n\n"},{"title":"Core.eval","page":"Essentials","location":"base/base.html#Core.eval","category":"function","text":"Core.eval(m::Module, expr)\n\nEvaluate an expression in the given module and return the result.\n\n\n\n\n\n"},{"title":"eval","page":"Essentials","location":"base/base.html#eval","category":"function","text":"eval(expr)\n\nEvaluate an expression in the global scope of the containing module. Every Module (except those defined with baremodule) has a private 1-argument definition of eval, which evaluates expressions in that module, for use inside that module.\n\n\n\n\n\n"},{"title":"Base.@eval","page":"Essentials","location":"base/base.html#Base.@eval","category":"macro","text":"@eval [mod,] ex\n\nEvaluate an expression with values interpolated into it using eval. If two arguments are provided, the first is the module to evaluate in.\n\n\n\n\n\n"},{"title":"Base.evalfile","page":"Essentials","location":"base/base.html#Base.evalfile","category":"function","text":"evalfile(path::AbstractString, args::Vector{String}=String[])\n\nLoad the file into an anonymous module using include, evaluate all expressions, and return the value of the last expression. The optional args argument can be used to set the input arguments of the script (i.e. the global ARGS variable). Note that definitions (e.g. methods, globals) are evaluated in the anonymous module and do not affect the current module.\n\nExamples\n\njulia> write(\"testfile.jl\", \"\"\"\n           @show ARGS\n           1 + 1\n       \"\"\");\n\njulia> x = evalfile(\"testfile.jl\", [\"ARG1\", \"ARG2\"]);\nARGS = [\"ARG1\", \"ARG2\"]\n\njulia> x\n2\n\njulia> rm(\"testfile.jl\")\n\n\n\n\n\n"},{"title":"Base.esc","page":"Essentials","location":"base/base.html#Base.esc","category":"function","text":"esc(e)\n\nOnly valid in the context of an Expr returned from a macro. Prevents the macro hygiene pass from turning embedded variables into gensym variables. See the Macros section of the Metaprogramming chapter of the manual for more details and examples.\n\n\n\n\n\n"},{"title":"Base.@inbounds","page":"Essentials","location":"base/base.html#Base.@inbounds","category":"macro","text":"@inbounds(blk)\n\nEliminates array bounds checking within expressions.\n\nIn the example below the in-range check for referencing element i of array A is skipped to improve performance.\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nwarning: Warning\nUsing @inbounds may return incorrect results/crashes/corruption for out-of-bounds indices. The user is responsible for checking it manually. Only use @inbounds when you are certain that all accesses are in bounds (as undefined behavior, e.g. crashes, might occur if this assertion is violated). For example, using 1:length(A) instead of eachindex(A) in a function like the one above is not safely inbounds because the first index of A may not be 1 for all user defined types that subtype AbstractArray.\n\n\n\n\n\n"},{"title":"Base.@boundscheck","page":"Essentials","location":"base/base.html#Base.@boundscheck","category":"macro","text":"@boundscheck(blk)\n\nAnnotates the expression blk as a bounds checking block, allowing it to be elided by @inbounds.\n\nnote: Note\nThe function in which @boundscheck is written must be inlined into its caller in order for @inbounds to have effect.\n\nExamples\n\njulia> @inline function g(A, i)\n           @boundscheck checkbounds(A, i)\n           return \"accessing ($A)[$i]\"\n       end;\n\njulia> f1() = return g(1:2, -1);\n\njulia> f2() = @inbounds return g(1:2, -1);\n\njulia> f1()\nERROR: BoundsError: attempt to access 2-element UnitRange{Int64} at index [-1]\nStacktrace:\n [1] throw_boundserror(::UnitRange{Int64}, ::Tuple{Int64}) at ./abstractarray.jl:455\n [2] checkbounds at ./abstractarray.jl:420 [inlined]\n [3] g at ./none:2 [inlined]\n [4] f1() at ./none:1\n [5] top-level scope\n\njulia> f2()\n\"accessing (1:2)[-1]\"\n\nwarning: Warning\nThe @boundscheck annotation allows you, as a library writer, to opt-in to allowing other code to remove your bounds checks with @inbounds. As noted there, the caller must verify—using information they can access—that their accesses are valid before using @inbounds. For indexing into your AbstractArray subclasses, for example, this involves checking the indices against its axes. Therefore, @boundscheck annotations should only be added to a getindex or setindex! implementation after you are certain its behavior is correct.\n\n\n\n\n\n"},{"title":"Base.@propagate_inbounds","page":"Essentials","location":"base/base.html#Base.@propagate_inbounds","category":"macro","text":"@propagate_inbounds\n\nTells the compiler to inline a function while retaining the caller's inbounds context.\n\n\n\n\n\n"},{"title":"Base.@inline","page":"Essentials","location":"base/base.html#Base.@inline","category":"macro","text":"@inline\n\nGive a hint to the compiler that this function is worth inlining.\n\nSmall functions typically do not need the @inline annotation, as the compiler does it automatically. By using @inline on bigger functions, an extra nudge can be given to the compiler to inline it.\n\n@inline can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\n@inline function longdef(x)\n    ...\nend\n\n# annotate short-form definition\n@inline shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    @inline\n    ...\nend\n\ncompat: Julia 1.8\nThe usage within a function body requires at least Julia 1.8.\n\n\n\n@inline block\n\nGive a hint to the compiler that calls within block are worth inlining.\n\n# The compiler will try to inline `f`\n@inline f(...)\n\n# The compiler will try to inline `f`, `g` and `+`\n@inline f(...) + g(...)\n\nnote: Note\nA callsite annotation always has the precedence over the annotation applied to the definition of the called function:@noinline function explicit_noinline(args...)\n    # body\nend\n\nlet\n    @inline explicit_noinline(args...) # will be inlined\nend\n\nnote: Note\nThe callsite annotation applies to all calls in the block, including function arguments that are themselves calls:# The compiler will not inline `getproperty`, `g` or `f`\n@noinline f(x.inner, g(y))\n\nnote: Note\nWhen there are nested callsite annotations, the innermost annotation has the precedence:@noinline let a0, b0 = ...\n    a = @inline f(a0)  # the compiler will try to inline this call\n    b = f(b0)          # the compiler will NOT try to inline this call\n    return a, b\nend\n\nwarning: Warning\nAlthough a callsite annotation will try to force inlining in regardless of the cost model, there are still chances it can't succeed in it. Especially, recursive calls can not be inlined even if they are annotated as @inlined.\n\ncompat: Julia 1.8\nThe callsite annotation requires at least Julia 1.8.\n\n\n\n\n\n"},{"title":"Base.@noinline","page":"Essentials","location":"base/base.html#Base.@noinline","category":"macro","text":"@noinline\n\nGive a hint to the compiler that it should not inline a function.\n\nSmall functions are typically inlined automatically. By using @noinline on small functions, auto-inlining can be prevented.\n\n@noinline can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\n@noinline function longdef(x)\n    ...\nend\n\n# annotate short-form definition\n@noinline shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    @noinline\n    ...\nend\n\ncompat: Julia 1.8\nThe usage within a function body requires at least Julia 1.8.\n\n\n\n@noinline block\n\nGive a hint to the compiler that it should not inline the calls within block.\n\n# The compiler will try to not inline `f`\n@noinline f(...)\n\n# The compiler will try to not inline `f`, `g` and `+`\n@noinline f(...) + g(...)\n\nnote: Note\nA callsite annotation always has the precedence over the annotation applied to the definition of the called function:@inline function explicit_inline(args...)\n    # body\nend\n\nlet\n    @noinline explicit_inline(args...) # will not be inlined\nend\n\nnote: Note\nWhen there are nested callsite annotations, the innermost annotation has the precedence:@inline let a0, b0 = ...\n    a = @noinline f(a0)  # the compiler will NOT try to inline this call\n    b = f(b0)            # the compiler will try to inline this call\n    return a, b\nend\n\ncompat: Julia 1.8\nThe callsite annotation requires at least Julia 1.8.\n\n\n\nnote: Note\nIf the function is trivial (for example returning a constant) it might get inlined anyway.\n\n\n\n\n\n"},{"title":"Base.@nospecialize","page":"Essentials","location":"base/base.html#Base.@nospecialize","category":"macro","text":"@nospecialize\n\nApplied to a function argument name, hints to the compiler that the method implementation should not be specialized for different types of that argument, but instead use the declared type for that argument. It can be applied to an argument within a formal argument list, or in the function body. When applied to an argument, the macro must wrap the entire argument expression, e.g., @nospecialize(x::Real) or @nospecialize(i::Integer...) rather than wrapping just the argument name. When used in a function body, the macro must occur in statement position and before any code.\n\nWhen used without arguments, it applies to all arguments of the parent scope. In local scope, this means all arguments of the containing function. In global (top-level) scope, this means all methods subsequently defined in the current module.\n\nSpecialization can reset back to the default by using @specialize.\n\nfunction example_function(@nospecialize x)\n    ...\nend\n\nfunction example_function(x, @nospecialize(y = 1))\n    ...\nend\n\nfunction example_function(x, y, z)\n    @nospecialize x y\n    ...\nend\n\n@nospecialize\nf(y) = [x for x in y]\n@specialize\n\nnote: Note\n@nospecialize affects code generation but not inference: it limits the diversity of the resulting native code, but it does not impose any limitations (beyond the standard ones) on type-inference. Use Base.@nospecializeinfer together with @nospecialize to additionally suppress inference.\n\nExamples\n\njulia> f(A::AbstractArray) = g(A)\nf (generic function with 1 method)\n\njulia> @noinline g(@nospecialize(A::AbstractArray)) = A[1]\ng (generic function with 1 method)\n\njulia> @code_typed f([1.0])\nCodeInfo(\n1 ─ %1 =    invoke g(A::AbstractArray)::Float64\n└──      return %1\n) => Float64\n\nHere, the @nospecialize annotation results in the equivalent of\n\nf(A::AbstractArray) = invoke(g, Tuple{AbstractArray}, A)\n\nensuring that only one version of native code will be generated for g, one that is generic for any AbstractArray. However, the specific return type is still inferred for both g and f, and this is still used in optimizing the callers of f and g.\n\n\n\n\n\n"},{"title":"Base.@specialize","page":"Essentials","location":"base/base.html#Base.@specialize","category":"macro","text":"@specialize\n\nReset the specialization hint for an argument back to the default. For details, see @nospecialize.\n\n\n\n\n\n"},{"title":"Base.@nospecializeinfer","page":"Essentials","location":"base/base.html#Base.@nospecializeinfer","category":"macro","text":"Base.@nospecializeinfer function f(args...)\n    @nospecialize ...\n    ...\nend\nBase.@nospecializeinfer f(@nospecialize args...) = ...\n\nTells the compiler to infer f using the declared types of @nospecialized arguments. This can be used to limit the number of compiler-generated specializations during inference.\n\nExamples\n\njulia> f(A::AbstractArray) = g(A)\nf (generic function with 1 method)\n\njulia> @noinline Base.@nospecializeinfer g(@nospecialize(A::AbstractArray)) = A[1]\ng (generic function with 1 method)\n\njulia> @code_typed f([1.0])\nCodeInfo(\n1 ─ %1 =    invoke g(A::AbstractArray)::Any\n└──      return %1\n) => Any\n\nIn this example, f will be inferred for each specific type of A, but g will only be inferred once with the declared argument type A::AbstractArray, meaning that the compiler will not likely see the excessive inference time on it while it can not infer the concrete return type of it. Without the @nospecializeinfer, f([1.0]) would infer the return type of g as Float64, indicating that inference ran for g(::Vector{Float64}) despite the prohibition on specialized code generation.\n\ncompat: Julia 1.10\nUsing Base.@nospecializeinfer requires Julia version 1.10.\n\n\n\n\n\n"},{"title":"Base.@constprop","page":"Essentials","location":"base/base.html#Base.@constprop","category":"macro","text":"Base.@constprop setting [ex]\n\nControl the mode of interprocedural constant propagation for the annotated function.\n\nTwo settings are supported:\n\nBase.@constprop :aggressive [ex]: apply constant propagation aggressively. For a method where the return type depends on the value of the arguments, this can yield improved inference results at the cost of additional compile time.\nBase.@constprop :none [ex]: disable constant propagation. This can reduce compile times for functions that Julia might otherwise deem worthy of constant-propagation. Common cases are for functions with Bool- or Symbol-valued arguments or keyword arguments.\n\nBase.@constprop can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\nBase.@constprop :aggressive function longdef(x)\n    ...\nend\n\n# annotate short-form definition\nBase.@constprop :aggressive shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    Base.@constprop :aggressive\n    ...\nend\n\ncompat: Julia 1.10\nThe usage within a function body requires at least Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.gensym","page":"Essentials","location":"base/base.html#Base.gensym","category":"function","text":"gensym([tag])\n\nGenerates a symbol which will not conflict with other variable names (in the same module).\n\n\n\n\n\n"},{"title":"Base.@gensym","page":"Essentials","location":"base/base.html#Base.@gensym","category":"macro","text":"@gensym\n\nGenerates a gensym symbol for a variable. For example, @gensym x y is transformed into x = gensym(\"x\"); y = gensym(\"y\").\n\n\n\n\n\n"},{"title":"var\"name\"","page":"Essentials","location":"base/base.html#var\"name\"","category":"keyword","text":"var\n\nThe syntax var\"#example#\" refers to a variable named Symbol(\"#example#\"), even though #example# is not a valid Julia identifier name.\n\nThis can be useful for interoperability with programming languages which have different rules for the construction of valid identifiers. For example, to refer to the R variable draw.segments, you can use var\"draw.segments\" in your Julia code.\n\nIt is also used to show julia source code which has gone through macro hygiene or otherwise contains variable names which can't be parsed normally.\n\nNote that this syntax requires parser support so it is expanded directly by the parser rather than being implemented as a normal string macro @var_str.\n\ncompat: Julia 1.3\nThis syntax requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.@goto","page":"Essentials","location":"base/base.html#Base.@goto","category":"macro","text":"@goto name\n\n@goto name unconditionally jumps to the statement at the location @label name.\n\n@label and @goto cannot create jumps to different top-level statements. Attempts cause an error. To still use @goto, enclose the @label and @goto in a block.\n\n\n\n\n\n"},{"title":"Base.@label","page":"Essentials","location":"base/base.html#Base.@label","category":"macro","text":"@label name\n\nLabels a statement with the symbolic label name. The label marks the end-point of an unconditional jump with @goto name.\n\n\n\n\n\n"},{"title":"Base.SimdLoop.@simd","page":"Essentials","location":"base/base.html#Base.SimdLoop.@simd","category":"macro","text":"@simd\n\nAnnotate a for loop to allow the compiler to take extra liberties to allow loop re-ordering\n\nwarning: Warning\nThis feature is experimental and could change or disappear in future versions of Julia. Incorrect use of the @simd macro may cause unexpected results.\n\nThe object iterated over in a @simd for loop should be a one-dimensional range. By using @simd, you are asserting several properties of the loop:\n\nIt is safe to execute iterations in arbitrary or overlapping order, with special consideration for reduction variables.\nFloating-point operations on reduction variables can be reordered or contracted, possibly causing different results than without @simd.\n\nIn many cases, Julia is able to automatically vectorize inner for loops without the use of @simd. Using @simd gives the compiler a little extra leeway to make it possible in more situations. In either case, your inner loop should have the following properties to allow vectorization:\n\nThe loop must be an innermost loop\nThe loop body must be straight-line code. Therefore, @inbounds is   currently needed for all array accesses. The compiler can sometimes turn   short &&, ||, and ?: expressions into straight-line code if it is safe   to evaluate all operands unconditionally. Consider using the ifelse   function instead of ?: in the loop if it is safe to do so.\nAccesses must have a stride pattern and cannot be \"gathers\" (random-index   reads) or \"scatters\" (random-index writes).\nThe stride should be unit stride.\n\nnote: Note\nThe @simd does not assert by default that the loop is completely free of loop-carried memory dependencies, which is an assumption that can easily be violated in generic code. If you are writing non-generic code, you can use @simd ivdep for ... end to also assert that:\n\nThere exists no loop-carried memory dependencies\nNo iteration ever waits on a previous iteration to make forward progress.\n\n\n\n\n\n"},{"title":"Base.@polly","page":"Essentials","location":"base/base.html#Base.@polly","category":"macro","text":"@polly\n\nTells the compiler to apply the polyhedral optimizer Polly to a function.\n\n\n\n\n\n"},{"title":"Base.@generated","page":"Essentials","location":"base/base.html#Base.@generated","category":"macro","text":"@generated f\n\n@generated is used to annotate a function which will be generated. In the body of the generated function, only types of arguments can be read (not the values). The function returns a quoted expression evaluated when the function is called. The @generated macro should not be used on functions mutating the global scope or depending on mutable elements.\n\nSee Metaprogramming for further details.\n\nExamples\n\njulia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"\n\n\n\n\n\n"},{"title":"Base.@assume_effects","page":"Essentials","location":"base/base.html#Base.@assume_effects","category":"macro","text":"Base.@assume_effects setting... [ex]\n\nOverride the compiler's effect modeling. This macro can be used in several contexts:\n\nImmediately before a method definition, to override the entire effect modeling of the applied method.\nWithin a function body without any arguments, to override the entire effect modeling of the enclosing method.\nApplied to a code block, to override the local effect modeling of the applied code block.\n\nExamples\n\njulia> Base.@assume_effects :terminates_locally function fact(x)\n           # usage 1:\n           # this :terminates_locally allows `fact` to be constant-folded\n           res = 1\n           0 ≤ x < 20 || error(\"bad fact\")\n           while x > 1\n               res *= x\n               x -= 1\n           end\n           return res\n       end\nfact (generic function with 1 method)\n\njulia> code_typed() do\n           fact(12)\n       end |> only\nCodeInfo(\n1 ─     return 479001600\n) => Int64\n\njulia> code_typed() do\n           map((2,3,4)) do x\n               # usage 2:\n               # this :terminates_locally allows this anonymous function to be constant-folded\n               Base.@assume_effects :terminates_locally\n               res = 1\n               0 ≤ x < 20 || error(\"bad fact\")\n               while x > 1\n                   res *= x\n                   x -= 1\n               end\n               return res\n           end\n       end |> only\nCodeInfo(\n1 ─     return (2, 6, 24)\n) => Tuple{Int64, Int64, Int64}\n\njulia> code_typed() do\n           map((2,3,4)) do x\n               res = 1\n               0 ≤ x < 20 || error(\"bad fact\")\n               # usage 3:\n               # with this :terminates_locally annotation the compiler skips tainting\n               # `:terminates` effect within this `while` block, allowing the parent\n               # anonymous function to be constant-folded\n               Base.@assume_effects :terminates_locally while x > 1\n                   res *= x\n                   x -= 1\n               end\n               return res\n           end\n       end |> only\nCodeInfo(\n1 ─     return (2, 6, 24)\n) => Tuple{Int64, Int64, Int64}\n\ncompat: Julia 1.8\nUsing Base.@assume_effects requires Julia version 1.8.\n\ncompat: Julia 1.10\nThe usage within a function body requires at least Julia 1.10.\n\ncompat: Julia 1.11\nThe code block annotation requires at least Julia 1.11.\n\nwarning: Warning\nImproper use of this macro causes undefined behavior (including crashes, incorrect answers, or other hard to track bugs). Use with care and only as a last resort if absolutely required. Even in such a case, you SHOULD take all possible steps to minimize the strength of the effect assertion (e.g., do not use :total if :nothrow would have been sufficient).\n\nIn general, each setting value makes an assertion about the behavior of the function, without requiring the compiler to prove that this behavior is indeed true. These assertions are made for all world ages. It is thus advisable to limit the use of generic functions that may later be extended to invalidate the assumption (which would cause undefined behavior).\n\nThe following settings are supported.\n\n:consistent\n:effect_free\n:nothrow\n:terminates_globally\n:terminates_locally\n:notaskstate\n:inaccessiblememonly\n:noub\n:noub_if_noinbounds\n:nortcall\n:foldable\n:removable\n:total\n\nExtended help\n\n\n\n:consistent\n\nThe :consistent setting asserts that for egal (===) inputs:\n\nThe manner of termination (return value, exception, non-termination) will always be the same.\nIf the method returns, the results will always be egal.\n\nnote: Note\nThis in particular implies that the method must not return a freshly allocated mutable object. Multiple allocations of mutable objects (even with identical contents) are not egal.\n\nnote: Note\nThe :consistent-cy assertion is made with respect to a particular world range R. More formally, write fᵢ for the evaluation of f in world-age i, then this setting requires: i  R j  R x y x  y  fᵢ(x)  fⱼ(y)For @assume_effects, the range R is m.primary_world:m.deleted_world of the annotated or containing method.For ordinary code instances, R is ci.min_world:ci.max_world.A further implication is that :consistent functions may not make their return value dependent on the state of the heap or any other global state that is not constant over the given world age range.\n\nnote: Note\nThe :consistent-cy includes all legal rewrites performed by the optimizer. For example, floating-point fastmath operations are not considered :consistent, because the optimizer may rewrite them causing the output to not be :consistent, even for the same world age (e.g. because one ran in the interpreter, while the other was optimized).\n\nnote: Note\nIf :consistent functions terminate by throwing an exception, that exception itself is not required to meet the egality requirement specified above.\n\n\n\n:effect_free\n\nThe :effect_free setting asserts that the method is free of externally semantically visible side effects. The following is an incomplete list of externally semantically visible side effects:\n\nChanging the value of a global variable.\nMutating the heap (e.g. an array or mutable value), except as noted below\nChanging the method table (e.g. through calls to eval)\nFile/Network/etc. I/O\nTask switching\n\nHowever, the following are explicitly not semantically visible, even if they may be observable:\n\nMemory allocations (both mutable and immutable)\nElapsed time\nGarbage collection\nHeap mutations of objects whose lifetime does not exceed the method (i.e. were allocated in the method and do not escape).\nThe returned value (which is externally visible, but not a side effect)\n\nThe rule of thumb here is that an externally visible side effect is anything that would affect the execution of the remainder of the program if the function were not executed.\n\nnote: Note\nThe :effect_free assertion is made both for the method itself and any code that is executed by the method. Keep in mind that the assertion must be valid for all world ages and limit use of this assertion accordingly.\n\n\n\n:nothrow\n\nThe :nothrow settings asserts that this method does not throw an exception (i.e. will either always return a value or never return).\n\nnote: Note\nIt is permissible for :nothrow annotated methods to make use of exception handling internally as long as the exception is not rethrown out of the method itself.\n\nnote: Note\nIf the execution of a method may raise MethodErrors and similar exceptions, then the method is not considered as :nothrow. However, note that environment-dependent errors like StackOverflowError or InterruptException are not modeled by this effect and thus a method that may result in StackOverflowError does not necessarily need to be !:nothrow (although it should usually be !:terminates too).\n\n\n\n:terminates_globally\n\nThe :terminates_globally settings asserts that this method will eventually terminate (either normally or abnormally), i.e. does not loop indefinitely.\n\nnote: Note\nThis :terminates_globally assertion covers any other methods called by the annotated method.\n\nnote: Note\nThe compiler will consider this a strong indication that the method will terminate relatively quickly and may (if otherwise legal) call this method at compile time. I.e. it is a bad idea to annotate this setting on a method that technically, but not practically, terminates.\n\n\n\n:terminates_locally\n\nThe :terminates_locally setting is like :terminates_globally, except that it only applies to syntactic control flow within the annotated method. It is thus a much weaker (and thus safer) assertion that allows for the possibility of non-termination if the method calls some other method that does not terminate.\n\nnote: Note\n:terminates_globally implies :terminates_locally.\n\n\n\n:notaskstate\n\nThe :notaskstate setting asserts that the method does not use or modify the local task state (task local storage, RNG state, etc.) and may thus be safely moved between tasks without observable results.\n\nnote: Note\nThe implementation of exception handling makes use of state stored in the task object. However, this state is currently not considered to be within the scope of :notaskstate and is tracked separately using the :nothrow effect.\n\nnote: Note\nThe :notaskstate assertion concerns the state of the currently running task. If a reference to a Task object is obtained by some other means that does not consider which task is currently running, the :notaskstate effect need not be tainted. This is true, even if said task object happens to be === to the currently running task.\n\nnote: Note\nAccess to task state usually also results in the tainting of other effects, such as :effect_free (if task state is modified) or :consistent (if task state is used in the computation of the result). In particular, code that is not :notaskstate, but is :effect_free and :consistent may still be dead-code-eliminated and thus promoted to :total.\n\n\n\n:inaccessiblememonly\n\nThe :inaccessiblememonly setting asserts that the method does not access or modify externally accessible mutable memory. This means the method can access or modify mutable memory for newly allocated objects that is not accessible by other methods or top-level execution before return from the method, but it can not access or modify any mutable global state or mutable memory pointed to by its arguments.\n\nnote: Note\nBelow is an incomplete list of examples that invalidate this assumption:a global reference or getglobal call to access a mutable global variable\na global assignment or setglobal! call to perform assignment to a non-constant global variable\nsetfield! call that changes a field of a global mutable variable\n\nnote: Note\nThis :inaccessiblememonly assertion covers any other methods called by the annotated method.\n\n\n\n:noub\n\nThe :noub setting asserts that the method will not execute any undefined behavior (for any input). Note that undefined behavior may technically cause the method to violate any other effect assertions (such as :consistent or :effect_free) as well, but we do not model this, and they assume the absence of undefined behavior.\n\n\n\n:nortcall\n\nThe :nortcall setting asserts that the method does not call Core.Compiler.return_type, and that any other methods this method might call also do not call Core.Compiler.return_type.\n\nnote: Note\nTo be precise, this assertion can be used when a call to Core.Compiler.return_type is not made at runtime; that is, when the result of Core.Compiler.return_type is known exactly at compile time and the call is eliminated by the optimizer. However, since whether the result of Core.Compiler.return_type is folded at compile time depends heavily on the compiler's implementation, it is generally risky to assert this if the method in question uses Core.Compiler.return_type in any form.\n\n\n\n:foldable\n\nThis setting is a convenient shortcut for the set of effects that the compiler requires to be guaranteed to constant fold a call at compile time. It is currently equivalent to the following settings:\n\n:consistent\n:effect_free\n:terminates_globally\n:noub\n:nortcall\n\nnote: Note\nThis list in particular does not include :nothrow. The compiler will still attempt constant propagation and note any thrown error at compile time. Note however, that by the :consistent-cy requirements, any such annotated call must consistently throw given the same argument values.\n\nnote: Note\nAn explicit @inbounds annotation inside the function will also disable constant folding and not be overridden by :foldable.\n\n\n\n:removable\n\nThis setting is a convenient shortcut for the set of effects that the compiler requires to be guaranteed to delete a call whose result is unused at compile time. It is currently equivalent to the following settings:\n\n:effect_free\n:nothrow\n:terminates_globally\n\n\n\n:total\n\nThis setting is the maximum possible set of effects. It currently implies the following other settings:\n\n:consistent\n:effect_free\n:nothrow\n:terminates_globally\n:notaskstate\n:inaccessiblememonly\n:noub\n:nortcall\n\nwarning: Warning\n:total is a very strong assertion and will likely gain additional semantics in future versions of Julia (e.g. if additional effects are added and included in the definition of :total). As a result, it should be used with care. Whenever possible, prefer to use the minimum possible set of specific effect assertions required for a particular application. In cases where a large number of effect overrides apply to a set of functions, a custom macro is recommended over the use of :total.\n\n\n\nNegated effects\n\nEffect names may be prefixed by ! to indicate that the effect should be removed from an earlier meta effect. For example, :total !:nothrow indicates that while the call is generally total, it may however throw.\n\n\n\n\n\n"},{"title":"Base.@deprecate","page":"Essentials","location":"base/base.html#Base.@deprecate","category":"macro","text":"@deprecate old new [export_old=true]\n\nDeprecate method old and specify the replacement call new, defining a new method old with the specified signature in the process.\n\nTo prevent old from being exported, set export_old to false.\n\nSee also Base.depwarn().\n\ncompat: Julia 1.5\nAs of Julia 1.5, functions defined by @deprecate do not print warning when julia is run without the --depwarn=yes flag set, as the default value of --depwarn option is no.  The warnings are printed from tests run by Pkg.test().\n\nExamples\n\njulia> @deprecate old_export(x) new(x)\nold_export (generic function with 1 method)\n\njulia> @deprecate old_public(x) new(x) false\nold_public (generic function with 1 method)\n\nCalls to @deprecate without explicit type-annotations will define deprecated methods accepting any number of positional and keyword arguments of type Any.\n\ncompat: Julia 1.9\nKeyword arguments are forwarded when there is no explicit type annotation as of Julia 1.9. For older versions, you can manually forward positional and keyword arguments by doing @deprecate old(args...; kwargs...) new(args...; kwargs...).\n\nTo restrict deprecation to a specific signature, annotate the arguments of old. For example,\n\njulia> new(x::Int) = x;\n\njulia> new(x::Float64) = 2x;\n\njulia> @deprecate old(x::Int) new(x);\n\njulia> methods(old)\n# 1 method for generic function \"old\" from Main:\n [1] old(x::Int64)\n     @ deprecated.jl:94\n\nwill define and deprecate a method old(x::Int) that mirrors new(x::Int) but will not define nor deprecate the method old(x::Float64).\n\n\n\n\n\n"},{"title":"Base.depwarn","page":"Essentials","location":"base/base.html#Base.depwarn","category":"function","text":"Base.depwarn(msg::String, funcsym::Symbol; force=false)\n\nPrint msg as a deprecation warning. The symbol funcsym should be the name of the calling function, which is used to ensure that the deprecation warning is only printed the first time for each call place. Set force=true to force the warning to always be shown, even if Julia was started with --depwarn=no (the default).\n\nSee also @deprecate.\n\nExamples\n\nfunction deprecated_func()\n    Base.depwarn(\"Don't use `deprecated_func()`!\", :deprecated_func)\n\n    1 + 1\nend\n\n\n\n\n\n"},{"title":"Base.Missing","page":"Essentials","location":"base/base.html#Base.Missing","category":"type","text":"Missing\n\nA type with no fields whose singleton instance missing is used to represent missing values.\n\nSee also: skipmissing, nonmissingtype, Nothing.\n\n\n\n\n\n"},{"title":"Base.missing","page":"Essentials","location":"base/base.html#Base.missing","category":"constant","text":"missing\n\nThe singleton instance of type Missing representing a missing value.\n\nSee also: NaN, skipmissing, nonmissingtype.\n\n\n\n\n\n"},{"title":"Base.coalesce","page":"Essentials","location":"base/base.html#Base.coalesce","category":"function","text":"coalesce(x...)\n\nReturn the first value in the arguments which is not equal to missing, if any. Otherwise return missing.\n\nSee also skipmissing, something, @coalesce.\n\nExamples\n\njulia> coalesce(missing, 1)\n1\n\njulia> coalesce(1, missing)\n1\n\njulia> coalesce(nothing, 1)  # returns `nothing`\n\njulia> coalesce(missing, missing)\nmissing\n\n\n\n\n\n"},{"title":"Base.@coalesce","page":"Essentials","location":"base/base.html#Base.@coalesce","category":"macro","text":"@coalesce(x...)\n\nShort-circuiting version of coalesce.\n\nExamples\n\njulia> f(x) = (println(\"f($x)\"); missing);\n\njulia> a = 1;\n\njulia> a = @coalesce a f(2) f(3) error(\"`a` is still missing\")\n1\n\njulia> b = missing;\n\njulia> b = @coalesce b f(2) f(3) error(\"`b` is still missing\")\nf(2)\nf(3)\nERROR: `b` is still missing\n[...]\n\ncompat: Julia 1.7\nThis macro is available as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.ismissing","page":"Essentials","location":"base/base.html#Base.ismissing","category":"function","text":"ismissing(x)\n\nIndicate whether x is missing.\n\nSee also: skipmissing, isnothing, isnan.\n\n\n\n\n\n"},{"title":"Base.skipmissing","page":"Essentials","location":"base/base.html#Base.skipmissing","category":"function","text":"skipmissing(itr)\n\nReturn an iterator over the elements in itr skipping missing values. The returned object can be indexed using indices of itr if the latter is indexable. Indices corresponding to missing values are not valid: they are skipped by keys and eachindex, and a MissingException is thrown when trying to use them.\n\nUse collect to obtain an Array containing the non-missing values in itr. Note that even if itr is a multidimensional array, the result will always be a Vector since it is not possible to remove missings while preserving dimensions of the input.\n\nSee also coalesce, ismissing, something.\n\nExamples\n\njulia> x = skipmissing([1, missing, 2])\nskipmissing(Union{Missing, Int64}[1, missing, 2])\n\njulia> sum(x)\n3\n\njulia> x[1]\n1\n\njulia> x[2]\nERROR: MissingException: the value at index (2,) is missing\n[...]\n\njulia> argmax(x)\n3\n\njulia> collect(keys(x))\n2-element Vector{Int64}:\n 1\n 3\n\njulia> collect(skipmissing([1, missing, 2]))\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(skipmissing([1 missing; 2 missing]))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n"},{"title":"Base.nonmissingtype","page":"Essentials","location":"base/base.html#Base.nonmissingtype","category":"function","text":"nonmissingtype(T::Type)\n\nIf T is a union of types containing Missing, return a new type with Missing removed.\n\nExamples\n\njulia> nonmissingtype(Union{Int64,Missing})\nInt64\n\njulia> nonmissingtype(Any)\nAny\n\ncompat: Julia 1.3\nThis function is exported as of Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.run","page":"Essentials","location":"base/base.html#Base.run","category":"function","text":"run(command, args...; wait::Bool = true)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual). Throws an error if anything goes wrong, including the process exiting with a non-zero status (when wait is true).\n\nThe args... allow you to pass through file descriptors to the command, and are ordered like regular unix file descriptors (eg stdin, stdout, stderr, FD(3), FD(4)...).\n\nIf wait is false, the process runs asynchronously. You can later wait for it and check its exit status by calling success on the returned process object. If the command spawns only a single process, a Process object is returned and the exit code can be retrieved via the exitcode field; see wait for more details.\n\nWhen wait is false, the process' I/O streams are directed to devnull. When wait is true, I/O streams are shared with the parent process. Use pipeline to control I/O redirection.\n\nSee also: Cmd.\n\n\n\n\n\n"},{"title":"Base.devnull","page":"Essentials","location":"base/base.html#Base.devnull","category":"constant","text":"devnull\n\nUsed in a stream redirect to discard all data written to it. Essentially equivalent to /dev/null on Unix or NUL on Windows. Usage:\n\nrun(pipeline(`cat test.txt`, devnull))\n\n\n\n\n\n"},{"title":"Base.success","page":"Essentials","location":"base/base.html#Base.success","category":"function","text":"success(command)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual), and tell whether it was successful (exited with a code of 0). An exception is raised if the process cannot be started.\n\n\n\n\n\n"},{"title":"Base.process_running","page":"Essentials","location":"base/base.html#Base.process_running","category":"function","text":"process_running(p::Process)\n\nDetermine whether a process is currently running.\n\n\n\n\n\n"},{"title":"Base.process_exited","page":"Essentials","location":"base/base.html#Base.process_exited","category":"function","text":"process_exited(p::Process)\n\nDetermine whether a process has exited.\n\n\n\n\n\n"},{"title":"Base.kill","page":"Essentials","location":"base/base.html#Base.kill-Tuple{Base.Process, Integer}","category":"method","text":"kill(p::Process, signum=Base.SIGTERM)\n\nSend a signal to a process. The default is to terminate the process. Returns successfully if the process has already exited, but throws an error if killing the process failed for other reasons (e.g. insufficient permissions).\n\n\n\n\n\n"},{"title":"Base.Sys.set_process_title","page":"Essentials","location":"base/base.html#Base.Sys.set_process_title","category":"function","text":"Sys.set_process_title(title::AbstractString)\n\nSet the process title. No-op on some operating systems.\n\n\n\n\n\n"},{"title":"Base.Sys.get_process_title","page":"Essentials","location":"base/base.html#Base.Sys.get_process_title","category":"function","text":"Sys.get_process_title()\n\nGet the process title. On some systems, will always return an empty string.\n\n\n\n\n\n"},{"title":"Base.ignorestatus","page":"Essentials","location":"base/base.html#Base.ignorestatus","category":"function","text":"ignorestatus(command)\n\nMark a command object so that running it will not throw an error if the result code is non-zero.\n\n\n\n\n\n"},{"title":"Base.detach","page":"Essentials","location":"base/base.html#Base.detach","category":"function","text":"detach(command)\n\nMark a command object so that it will be run in a new process group, allowing it to outlive the julia process, and not have Ctrl-C interrupts passed to it.\n\n\n\n\n\n"},{"title":"Base.Cmd","page":"Essentials","location":"base/base.html#Base.Cmd","category":"type","text":"Cmd(cmd::Cmd; ignorestatus, detach, windows_verbatim, windows_hide, env, dir, uid, gid)\nCmd(exec::Vector{String})\n\nConstruct a new Cmd object, representing an external program and arguments, from cmd, while changing the settings of the optional keyword arguments:\n\nignorestatus::Bool: If true (defaults to false), then the Cmd will not throw an error if the return code is nonzero.\ndetach::Bool: If true (defaults to false), then the Cmd will be run in a new process group, allowing it to outlive the julia process and not have Ctrl-C passed to it.\nwindows_verbatim::Bool: If true (defaults to false), then on Windows the Cmd will send a command-line string to the process with no quoting or escaping of arguments, even arguments containing spaces. (On Windows, arguments are sent to a program as a single \"command-line\" string, and programs are responsible for parsing it into arguments. By default, empty arguments and arguments with spaces or tabs are quoted with double quotes \" in the command line, and \\ or \" are preceded by backslashes. windows_verbatim=true is useful for launching programs that parse their command line in nonstandard ways.) Has no effect on non-Windows systems.\nwindows_hide::Bool: If true (defaults to false), then on Windows no new console window is displayed when the Cmd is executed. This has no effect if a console is already open or on non-Windows systems.\nenv: Set environment variables to use when running the Cmd. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", an array or tuple of \"var\"=>val pairs. In order to modify (rather than replace) the existing environment, initialize env with copy(ENV) and then set env[\"var\"]=val as desired.  To add to an environment block within a Cmd object without replacing all elements, use addenv() which will return a Cmd object with the updated environment.\ndir::AbstractString: Specify a working directory for the command (instead of the current directory).\nuid::Union{Nothing,UInt32}: Set the user ID for the process (Unix only).\ngid::Union{Nothing,UInt32}: Set the group ID for the process (Unix only).\n\nFor any keywords that are not specified, the current settings from cmd are used.\n\nNote that the Cmd(exec) constructor does not create a copy of exec. Any subsequent changes to exec will be reflected in the Cmd object.\n\nThe most common way to construct a Cmd object is with command literals (backticks), e.g.\n\n`ls -l`\n\nThis can then be passed to the Cmd constructor to modify its settings, e.g.\n\nCmd(`echo \"Hello world\"`, ignorestatus=true, detach=false)\n\n\n\n\n\n"},{"title":"Base.setenv","page":"Essentials","location":"base/base.html#Base.setenv","category":"function","text":"setenv(command::Cmd, env; dir)\n\nSet environment variables to use when running the given command. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", or zero or more \"var\"=>val pair arguments. In order to modify (rather than replace) the existing environment, create env through copy(ENV) and then setting env[\"var\"]=val as desired, or use addenv.\n\nThe dir keyword argument can be used to specify a working directory for the command. dir defaults to the currently set dir for command (which is the current working directory if not specified already).\n\nSee also Cmd, addenv, ENV, pwd.\n\n\n\n\n\n"},{"title":"Base.addenv","page":"Essentials","location":"base/base.html#Base.addenv","category":"function","text":"addenv(command::Cmd, env...; inherit::Bool = true)\n\nMerge new environment mappings into the given Cmd object, returning a new Cmd object. Duplicate keys are replaced.  If command does not contain any environment values set already, it inherits the current environment at time of addenv() call if inherit is true. Keys with value nothing are deleted from the env.\n\nSee also Cmd, setenv, ENV.\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.withenv","page":"Essentials","location":"base/base.html#Base.withenv","category":"function","text":"withenv(f, kv::Pair...)\n\nExecute f in an environment that is temporarily modified (not replaced as in setenv) by zero or more \"var\"=>val arguments kv. withenv is generally used via the withenv(kv...) do ... end syntax. A value of nothing can be used to temporarily unset an environment variable (if it is set). When withenv returns, the original environment has been restored.\n\nwarning: Warning\nChanging the environment is not thread-safe. For running external commands with a different environment from the parent process, prefer using addenv over withenv.\n\n\n\n\n\n"},{"title":"Base.shell_escape","page":"Essentials","location":"base/base.html#Base.shell_escape","category":"function","text":"shell_escape(args::Union{Cmd,AbstractString...}; special::AbstractString=\"\")\n\nThe unexported shell_escape function is the inverse of the unexported Base.shell_split() function: it takes a string or command object and escapes any special characters in such a way that calling Base.shell_split() on it would give back the array of words in the original command. The special keyword argument controls what characters in addition to whitespace, backslashes, quotes and dollar signs are considered to be special (default: none).\n\nExamples\n\njulia> Base.shell_escape(\"cat\", \"/foo/bar baz\", \"&&\", \"echo\", \"done\")\n\"cat '/foo/bar baz' && echo done\"\n\njulia> Base.shell_escape(\"echo\", \"this\", \"&&\", \"that\")\n\"echo this && that\"\n\n\n\n\n\n"},{"title":"Base.shell_split","page":"Essentials","location":"base/base.html#Base.shell_split","category":"function","text":"shell_split(command::AbstractString)\n\nSplit a shell command string into its individual components.\n\nExamples\n\njulia> Base.shell_split(\"git commit -m 'Initial commit'\")\n4-element Vector{String}:\n \"git\"\n \"commit\"\n \"-m\"\n \"Initial commit\"\n\n\n\n\n\n"},{"title":"Base.shell_escape_posixly","page":"Essentials","location":"base/base.html#Base.shell_escape_posixly","category":"function","text":"shell_escape_posixly(args::Union{Cmd,AbstractString...})\n\nThe unexported shell_escape_posixly function takes a string or command object and escapes any special characters in such a way that it is safe to pass it as an argument to a posix shell.\n\nSee also: Base.shell_escape()\n\nExamples\n\njulia> Base.shell_escape_posixly(\"cat\", \"/foo/bar baz\", \"&&\", \"echo\", \"done\")\n\"cat '/foo/bar baz' '&&' echo done\"\n\njulia> Base.shell_escape_posixly(\"echo\", \"this\", \"&&\", \"that\")\n\"echo this '&&' that\"\n\n\n\n\n\n"},{"title":"Base.shell_escape_csh","page":"Essentials","location":"base/base.html#Base.shell_escape_csh","category":"function","text":"shell_escape_csh(args::Union{Cmd,AbstractString...})\nshell_escape_csh(io::IO, args::Union{Cmd,AbstractString...})\n\nThis function quotes any metacharacters in the string arguments such that the string returned can be inserted into a command-line for interpretation by the Unix C shell (csh, tcsh), where each string argument will form one word.\n\nIn contrast to a POSIX shell, csh does not support the use of the backslash as a general escape character in double-quoted strings. Therefore, this function wraps strings that might contain metacharacters in single quotes, except for parts that contain single quotes, which it wraps in double quotes instead. It switches between these types of quotes as needed. Linefeed characters are escaped with a backslash.\n\nThis function should also work for a POSIX shell, except if the input string contains a linefeed (\"\\n\") character.\n\nSee also: Base.shell_escape_posixly()\n\n\n\n\n\n"},{"title":"Base.shell_escape_wincmd","page":"Essentials","location":"base/base.html#Base.shell_escape_wincmd","category":"function","text":"shell_escape_wincmd(s::AbstractString)\nshell_escape_wincmd(io::IO, s::AbstractString)\n\nThe unexported shell_escape_wincmd function escapes Windows cmd.exe shell meta characters. It escapes ()!^<>&| by placing a ^ in front. An @ is only escaped at the start of the string. Pairs of \" characters and the strings they enclose are passed through unescaped. Any remaining \" is escaped with ^ to ensure that the number of unescaped \" characters in the result remains even.\n\nSince cmd.exe substitutes variable references (like %USER%) before processing the escape characters ^ and \", this function makes no attempt to escape the percent sign (%), the presence of % in the input may cause severe breakage, depending on where the result is used.\n\nInput strings with ASCII control characters that cannot be escaped (NUL, CR, LF) will cause an ArgumentError exception.\n\nThe result is safe to pass as an argument to a command call being processed by CMD.exe /S /C \" ... \" (with surrounding double-quote pair) and will be received verbatim by the target application if the input does not contain % (else this function will fail with an ArgumentError). The presence of % in the input string may result in command injection vulnerabilities and may invalidate any claim of suitability of the output of this function for use as an argument to cmd (due to the ordering described above), so use caution when assembling a string from various sources.\n\nThis function may be useful in concert with the windows_verbatim flag to Cmd when constructing process pipelines.\n\nwincmd(c::String) =\n   run(Cmd(Cmd([\"cmd.exe\", \"/s /c \\\" $c \\\"\"]);\n           windows_verbatim=true))\nwincmd_echo(s::String) =\n   wincmd(\"echo \" * Base.shell_escape_wincmd(s))\nwincmd_echo(\"hello $(ENV[\"USERNAME\"]) & the \\\"whole\\\" world! (=^I^=)\")\n\nBut take note that if the input string s contains a %, the argument list and echo'ed text may get corrupted, resulting in arbitrary command execution. The argument can alternatively be passed as an environment variable, which avoids the problem with % and the need for the windows_verbatim flag:\n\ncmdargs = Base.shell_escape_wincmd(\"Passing args with %cmdargs% works 100%!\")\nrun(setenv(`cmd /C echo %cmdargs%`, \"cmdargs\" => cmdargs))\n\nwarning: Warning\nThe argument parsing done by CMD when calling batch files (either inside .bat files or as arguments to them) is not fully compatible with the output of this function. In particular, the processing of % is different.\n\nimportant: Important\nDue to a peculiar behavior of the CMD parser/interpreter, each command after a literal | character (indicating a command pipeline) must have shell_escape_wincmd applied twice since it will be parsed twice by CMD. This implies ENV variables would also be expanded twice! For example:to_print = \"All for 1 & 1 for all!\"\nto_print_esc = Base.shell_escape_wincmd(Base.shell_escape_wincmd(to_print))\nrun(Cmd(Cmd([\"cmd\", \"/S /C \\\" break | echo $(to_print_esc) \\\"\"]), windows_verbatim=true))\n\nWith an I/O stream parameter io, the result will be written there, rather than returned as a string.\n\nSee also Base.escape_microsoft_c_args(), Base.shell_escape_posixly().\n\nExamples\n\njulia> Base.shell_escape_wincmd(\"a^\\\"^o\\\"^u\\\"\")\n\"a^^\\\"^o\\\"^^u^\\\"\"\n\n\n\n\n\n"},{"title":"Base.escape_microsoft_c_args","page":"Essentials","location":"base/base.html#Base.escape_microsoft_c_args","category":"function","text":"escape_microsoft_c_args(args::Union{Cmd,AbstractString...})\nescape_microsoft_c_args(io::IO, args::Union{Cmd,AbstractString...})\n\nConvert a collection of string arguments into a string that can be passed to many Windows command-line applications.\n\nMicrosoft Windows passes the entire command line as a single string to the application (unlike POSIX systems, where the shell splits the command line into a list of arguments). Many Windows API applications (including julia.exe), use the conventions of the Microsoft C/C++ runtime to split that command line into a list of strings.\n\nThis function implements an inverse for a parser compatible with these rules. It joins command-line arguments to be passed to a Windows C/C++/Julia application into a command line, escaping or quoting the meta characters space, TAB, double quote and backslash where needed.\n\nSee also Base.shell_escape_wincmd(), Base.escape_raw_string().\n\n\n\n\n\n"},{"title":"Base.setcpuaffinity","page":"Essentials","location":"base/base.html#Base.setcpuaffinity","category":"function","text":"setcpuaffinity(original_command::Cmd, cpus) -> command::Cmd\n\nSet the CPU affinity of the command by a list of CPU IDs (1-based) cpus.  Passing cpus = nothing means to unset the CPU affinity if the original_command has any.\n\nThis function is supported only in Linux and Windows.  It is not supported in macOS because libuv does not support affinity setting.\n\ncompat: Julia 1.8\nThis function requires at least Julia 1.8.\n\nExamples\n\nIn Linux, the taskset command line program can be used to see how setcpuaffinity works.\n\njulia> run(setcpuaffinity(`sh -c 'taskset -p $$'`, [1, 2, 5]));\npid 2273's current affinity mask: 13\n\nNote that the mask value 13 reflects that the first, second, and the fifth bits (counting from the least significant position) are turned on:\n\njulia> 0b010011\n0x13\n\n\n\n\n\n"},{"title":"Base.pipeline","page":"Essentials","location":"base/base.html#Base.pipeline-Tuple{Any, Any, Any, Vararg{Any}}","category":"method","text":"pipeline(from, to, ...)\n\nCreate a pipeline from a data source to a destination. The source and destination can be commands, I/O streams, strings, or results of other pipeline calls. At least one argument must be a command. Strings refer to filenames. When called with more than two arguments, they are chained together from left to right. For example, pipeline(a,b,c) is equivalent to pipeline(pipeline(a,b),c). This provides a more concise way to specify multi-stage pipelines.\n\nExamples:\n\nrun(pipeline(`ls`, `grep xyz`))\nrun(pipeline(`ls`, \"out.txt\"))\nrun(pipeline(\"out.txt\", `grep xyz`))\n\n\n\n\n\n"},{"title":"Base.pipeline","page":"Essentials","location":"base/base.html#Base.pipeline-Tuple{Base.AbstractCmd}","category":"method","text":"pipeline(command; stdin, stdout, stderr, append=false)\n\nRedirect I/O to or from the given command. Keyword arguments specify which of the command's streams should be redirected. append controls whether file output appends to the file. This is a more general version of the 2-argument pipeline function. pipeline(from, to) is equivalent to pipeline(from, stdout=to) when from is a command, and to pipeline(to, stdin=from) when from is another kind of data source.\n\nExamples:\n\nrun(pipeline(`dothings`, stdout=\"out.txt\", stderr=\"errs.txt\"))\nrun(pipeline(`update`, stdout=\"log.txt\", append=true))\n\n\n\n\n\n"},{"title":"Base.Libc.gethostname","page":"Essentials","location":"base/base.html#Base.Libc.gethostname","category":"function","text":"gethostname()::String\n\nGet the local machine's host name.\n\n\n\n\n\n"},{"title":"Base.Libc.getpid","page":"Essentials","location":"base/base.html#Base.Libc.getpid","category":"function","text":"getpid()::Int32\n\nGet Julia's process ID.\n\n\n\n\n\ngetpid(process)::Int32\n\nGet the child process ID, if it still exists.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n"},{"title":"Base.Libc.time","page":"Essentials","location":"base/base.html#Base.Libc.time-Tuple{}","category":"method","text":"time()::Float64\n\nGet the system time in seconds since the epoch, with fairly high (typically, microsecond) resolution.\n\nSee also time_ns.\n\n\n\n\n\n"},{"title":"Base.time_ns","page":"Essentials","location":"base/base.html#Base.time_ns","category":"function","text":"time_ns()::UInt64\n\nGet the time in nanoseconds relative to some machine-specific arbitrary time in the past. The primary use is for measuring elapsed times during program execution. The return value is guaranteed to be monotonic (mod 2⁶⁴) while the system is running, and is unaffected by clock drift or changes to local calendar time, but it may change arbitrarily across system reboots or suspensions.\n\n(Although the returned time is always in nanoseconds, the timing resolution is platform-dependent.)\n\n\n\n\n\n"},{"title":"Base.@time","page":"Essentials","location":"base/base.html#Base.@time","category":"macro","text":"@time expr\n@time \"description\" expr\n\nA macro to execute an expression, printing the time it took to execute, the number of allocations, and the total number of bytes its execution caused to be allocated, before returning the value of the expression. Any time spent garbage collecting (gc), compiling new code, or recompiling invalidated code is shown as a percentage. Any lock conflicts where a ReentrantLock had to wait are shown as a count.\n\nOptionally provide a description string to print before the time report.\n\nIn some cases the system will look inside the @time expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @time @eval ....\n\nSee also @showtime, @timev, @timed, @elapsed, @allocated, and @allocations.\n\nnote: Note\nFor more serious benchmarking, consider the @btime macro from the BenchmarkTools.jl package which among other things evaluates the function multiple times in order to reduce noise.\n\ncompat: Julia 1.8\nThe option to add a description was introduced in Julia 1.8.Recompilation time being shown separately from compilation time was introduced in Julia 1.8\n\ncompat: Julia 1.11\nThe reporting of any lock conflicts was added in Julia 1.11.\n\njulia> x = rand(10,10);\n\njulia> @time x * x;\n  0.606588 seconds (2.19 M allocations: 116.555 MiB, 3.75% gc time, 99.94% compilation time)\n\njulia> @time x * x;\n  0.000009 seconds (1 allocation: 896 bytes)\n\njulia> @time begin\n           sleep(0.3)\n           1+1\n       end\n  0.301395 seconds (8 allocations: 336 bytes)\n2\n\njulia> @time \"A one second sleep\" sleep(1)\nA one second sleep: 1.005750 seconds (5 allocations: 144 bytes)\n\njulia> for loop in 1:3\n            @time loop sleep(1)\n        end\n1: 1.006760 seconds (5 allocations: 144 bytes)\n2: 1.001263 seconds (5 allocations: 144 bytes)\n3: 1.003676 seconds (5 allocations: 144 bytes)\n\n\n\n\n\n"},{"title":"Base.@showtime","page":"Essentials","location":"base/base.html#Base.@showtime","category":"macro","text":"@showtime expr\n\nLike @time but also prints the expression being evaluated for reference.\n\ncompat: Julia 1.8\nThis macro was added in Julia 1.8.\n\nSee also @time.\n\njulia> @showtime sleep(1)\nsleep(1): 1.002164 seconds (4 allocations: 128 bytes)\n\n\n\n\n\n"},{"title":"Base.@timev","page":"Essentials","location":"base/base.html#Base.@timev","category":"macro","text":"@timev expr\n@timev \"description\" expr\n\nThis is a verbose version of the @time macro. It first prints the same information as @time, then any non-zero memory allocation counters, and then returns the value of the expression.\n\nOptionally provide a description string to print before the time report.\n\ncompat: Julia 1.8\nThe option to add a description was introduced in Julia 1.8.\n\nSee also @time, @timed, @elapsed, @allocated, and @allocations.\n\njulia> x = rand(10,10);\n\njulia> @timev x * x;\n  0.546770 seconds (2.20 M allocations: 116.632 MiB, 4.23% gc time, 99.94% compilation time)\nelapsed time (ns): 546769547\ngc time (ns):      23115606\nbytes allocated:   122297811\npool allocs:       2197930\nnon-pool GC allocs:1327\nmalloc() calls:    36\nrealloc() calls:   5\nGC pauses:         3\n\njulia> @timev x * x;\n  0.000010 seconds (1 allocation: 896 bytes)\nelapsed time (ns): 9848\nbytes allocated:   896\npool allocs:       1\n\n\n\n\n\n"},{"title":"Base.@timed","page":"Essentials","location":"base/base.html#Base.@timed","category":"macro","text":"@timed\n\nA macro to execute an expression, and return the value of the expression, elapsed time in seconds, total bytes allocated, garbage collection time, an object with various memory allocation counters, compilation time in seconds, and recompilation time in seconds. Any lock conflicts where a ReentrantLock had to wait are shown as a count.\n\nIn some cases the system will look inside the @timed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @timed @eval ....\n\nSee also @time, @timev, @elapsed, @allocated, @allocations, and @lock_conflicts.\n\njulia> stats = @timed rand(10^6);\n\njulia> stats.time\n0.006634834\n\njulia> stats.bytes\n8000256\n\njulia> stats.gctime\n0.0055765\n\njulia> propertynames(stats.gcstats)\n(:allocd, :malloc, :realloc, :poolalloc, :bigalloc, :freecall, :total_time, :pause, :full_sweep)\n\njulia> stats.gcstats.total_time\n5576500\n\njulia> stats.compile_time\n0.0\n\njulia> stats.recompile_time\n0.0\n\n\ncompat: Julia 1.5\nThe return type of this macro was changed from Tuple to NamedTuple in Julia 1.5.\n\ncompat: Julia 1.11\nThe lock_conflicts, compile_time, and recompile_time fields were added in Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.@elapsed","page":"Essentials","location":"base/base.html#Base.@elapsed","category":"macro","text":"@elapsed\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the number of seconds it took to execute as a floating-point number.\n\nIn some cases the system will look inside the @elapsed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @elapsed @eval ....\n\nSee also @time, @timev, @timed, @allocated, and @allocations.\n\njulia> @elapsed sleep(0.3)\n0.301391426\n\n\n\n\n\n"},{"title":"Base.@allocated","page":"Essentials","location":"base/base.html#Base.@allocated","category":"macro","text":"@allocated\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the total number of bytes allocated during evaluation of the expression.\n\nIf the expression is a function call, an effort is made to measure only allocations from the argument expressions and during the function, excluding any overhead from calling it and not performing constant propagation with the provided argument values. If you want to include those effects, i.e. measuring the call site as well, use the syntax @allocated (()->f(1))().\n\nIt is recommended to measure function calls with only simple argument expressions, e.g. x = []; @allocated f(x) instead of @allocated f([]) to clarify that only f is being measured.\n\nFor more complex expressions, the code is simply run in place and therefore may see allocations due to the surrounding context. For example it is possible for @allocated f(1) and @allocated x = f(1) to give different results.\n\nSee also @allocations, @time, @timev, @timed, and @elapsed.\n\njulia> @allocated rand(10^6)\n8000080\n\n\n\n\n\n"},{"title":"Base.@allocations","page":"Essentials","location":"base/base.html#Base.@allocations","category":"macro","text":"@allocations\n\nA macro to evaluate an expression, discard the resulting value, and instead return the total number of allocations during evaluation of the expression.\n\nSee also @allocated, @time, @timev, @timed, and @elapsed.\n\njulia> @allocations rand(10^6)\n2\n\ncompat: Julia 1.9\nThis macro was added in Julia 1.9.\n\n\n\n\n\n"},{"title":"Base.@lock_conflicts","page":"Essentials","location":"base/base.html#Base.@lock_conflicts","category":"macro","text":"@lock_conflicts\n\nA macro to evaluate an expression, discard the resulting value, and instead return the total number of lock conflicts during evaluation, where a lock attempt on a ReentrantLock resulted in a wait because the lock was already held.\n\nSee also @time, @timev and @timed.\n\njulia> @lock_conflicts begin\n    l = ReentrantLock()\n    Threads.@threads for i in 1:Threads.nthreads()\n        lock(l) do\n        sleep(1)\n        end\n    end\nend\n5\n\ncompat: Julia 1.11\nThis macro was added in Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.TRACE_EVAL","page":"Essentials","location":"base/base.html#Base.TRACE_EVAL","category":"constant","text":"Base.TRACE_EVAL\n\nGlobal control for expression tracing during top-level evaluation. This setting takes priority over the --trace-eval command-line option.\n\nSet to:\n\nnothing - use the command-line --trace-eval setting (default)\n:no - disable expression tracing\n:loc - show only location information during evaluation\n:full - show full expressions being evaluated\n\nExamples\n\n# Enable full expression tracing\nBase.TRACE_EVAL = :full\n\n# Show only locations\nBase.TRACE_EVAL = :loc\n\n# Disable tracing (overrides command-line setting)\nBase.TRACE_EVAL = :no\n\n# Reset to use command-line setting\nBase.TRACE_EVAL = nothing\n\nSee also: Command-line Interface for the --trace-eval option.\n\n\n\n\n\n"},{"title":"Base.EnvDict","page":"Essentials","location":"base/base.html#Base.EnvDict","category":"type","text":"EnvDict()::EnvDict\n\nA singleton of this type provides a hash table interface to environment variables.\n\n\n\n\n\n"},{"title":"Base.ENV","page":"Essentials","location":"base/base.html#Base.ENV","category":"constant","text":"ENV\n\nReference to the singleton EnvDict, providing a dictionary interface to system environment variables.\n\n(On Windows, system environment variables are case-insensitive, and ENV correspondingly converts all keys to uppercase for display, iteration, and copying. Portable code should not rely on the ability to distinguish variables by case, and should beware that setting an ostensibly lowercase variable may result in an uppercase ENV key.)\n\nwarning: Warning\nMutating the environment is not thread-safe.\n\nExamples\n\njulia> ENV\nBase.EnvDict with \"50\" entries:\n  \"SECURITYSESSIONID\"            => \"123\"\n  \"USER\"                         => \"username\"\n  \"MallocNanoZone\"               => \"0\"\n  ⋮                              => ⋮\n\njulia> ENV[\"JULIA_EDITOR\"] = \"vim\"\n\"vim\"\n\njulia> ENV[\"JULIA_EDITOR\"]\n\"vim\"\n\nSee also: withenv, addenv.\n\n\n\n\n\n"},{"title":"Base.Sys.STDLIB","page":"Essentials","location":"base/base.html#Base.Sys.STDLIB","category":"constant","text":"Sys.STDLIB::String\n\nA string containing the full path to the directory containing the stdlib packages.\n\n\n\n\n\n"},{"title":"Base.Sys.isunix","page":"Essentials","location":"base/base.html#Base.Sys.isunix","category":"function","text":"Sys.isunix([os])\n\nPredicate for testing if the OS provides a Unix-like interface. See documentation in Handling Operating System Variation.\n\n\n\n\n\n"},{"title":"Base.Sys.isapple","page":"Essentials","location":"base/base.html#Base.Sys.isapple","category":"function","text":"Sys.isapple([os])\n\nPredicate for testing if the OS is a derivative of Apple Macintosh OS X or Darwin. See documentation in Handling Operating System Variation.\n\n\n\n\n\n"},{"title":"Base.Sys.islinux","page":"Essentials","location":"base/base.html#Base.Sys.islinux","category":"function","text":"Sys.islinux([os])\n\nPredicate for testing if the OS is a derivative of Linux. See documentation in Handling Operating System Variation.\n\n\n\n\n\n"},{"title":"Base.Sys.isbsd","page":"Essentials","location":"base/base.html#Base.Sys.isbsd","category":"function","text":"Sys.isbsd([os])\n\nPredicate for testing if the OS is a derivative of BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nThe Darwin kernel descends from BSD, which means that Sys.isbsd() is true on macOS systems. To exclude macOS from a predicate, use Sys.isbsd() && !Sys.isapple().\n\n\n\n\n\n"},{"title":"Base.Sys.isfreebsd","page":"Essentials","location":"base/base.html#Base.Sys.isfreebsd","category":"function","text":"Sys.isfreebsd([os])\n\nPredicate for testing if the OS is a derivative of FreeBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on FreeBSD but also on other BSD-based systems. Sys.isfreebsd() refers only to FreeBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n"},{"title":"Base.Sys.isopenbsd","page":"Essentials","location":"base/base.html#Base.Sys.isopenbsd","category":"function","text":"Sys.isopenbsd([os])\n\nPredicate for testing if the OS is a derivative of OpenBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on OpenBSD but also on other BSD-based systems. Sys.isopenbsd() refers only to OpenBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n"},{"title":"Base.Sys.isnetbsd","page":"Essentials","location":"base/base.html#Base.Sys.isnetbsd","category":"function","text":"Sys.isnetbsd([os])\n\nPredicate for testing if the OS is a derivative of NetBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on NetBSD but also on other BSD-based systems. Sys.isnetbsd() refers only to NetBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n"},{"title":"Base.Sys.isdragonfly","page":"Essentials","location":"base/base.html#Base.Sys.isdragonfly","category":"function","text":"Sys.isdragonfly([os])\n\nPredicate for testing if the OS is a derivative of DragonFly BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on DragonFly but also on other BSD-based systems. Sys.isdragonfly() refers only to DragonFly.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n"},{"title":"Base.Sys.iswindows","page":"Essentials","location":"base/base.html#Base.Sys.iswindows","category":"function","text":"Sys.iswindows([os])\n\nPredicate for testing if the OS is a derivative of Microsoft Windows NT. See documentation in Handling Operating System Variation.\n\n\n\n\n\n"},{"title":"Base.Sys.windows_version","page":"Essentials","location":"base/base.html#Base.Sys.windows_version","category":"function","text":"Sys.windows_version()\n\nReturn the version number for the Windows NT Kernel as a VersionNumber, i.e. v\"major.minor.build\", or v\"0.0.0\" if this is not running on Windows.\n\n\n\n\n\n"},{"title":"Base.Sys.free_memory","page":"Essentials","location":"base/base.html#Base.Sys.free_memory","category":"function","text":"Sys.free_memory()\n\nGet the total free memory in RAM in bytes.\n\n\n\n\n\n"},{"title":"Base.Sys.total_memory","page":"Essentials","location":"base/base.html#Base.Sys.total_memory","category":"function","text":"Sys.total_memory()\n\nGet the total memory in RAM (including that which is currently used) in bytes. This amount may be constrained, e.g., by Linux control groups. For the unconstrained amount, see Sys.total_physical_memory().\n\n\n\n\n\n"},{"title":"Base.Sys.free_physical_memory","page":"Essentials","location":"base/base.html#Base.Sys.free_physical_memory","category":"function","text":"Sys.free_physical_memory()\n\nGet the free memory of the system in bytes. The entire amount may not be available to the current process; use Sys.free_memory() for the actually available amount.\n\n\n\n\n\n"},{"title":"Base.Sys.total_physical_memory","page":"Essentials","location":"base/base.html#Base.Sys.total_physical_memory","category":"function","text":"Sys.total_physical_memory()\n\nGet the total memory in RAM (including that which is currently used) in bytes. The entire amount may not be available to the current process; see Sys.total_memory().\n\n\n\n\n\n"},{"title":"Base.Sys.uptime","page":"Essentials","location":"base/base.html#Base.Sys.uptime","category":"function","text":"Sys.uptime()\n\nGets the current system uptime in seconds.\n\n\n\n\n\n"},{"title":"Base.Sys.sysimage_target","page":"Essentials","location":"base/base.html#Base.Sys.sysimage_target","category":"function","text":"Sys.sysimage_target()\n\nReturn the CPU target string that was used to build the current system image.\n\nThis function returns the original CPU target specification that was passed to Julia when the system image was compiled. This can be useful for reproducing the same system image or for understanding what CPU features were enabled during compilation.\n\nIf the system image was built with the default settings this will return \"native\".\n\nSee also JULIA_CPU_TARGET.\n\n\n\n\n\n"},{"title":"Base.Sys.isjsvm","page":"Essentials","location":"base/base.html#Base.Sys.isjsvm","category":"function","text":"Sys.isjsvm([os])\n\nPredicate for testing if Julia is running in a JavaScript VM (JSVM), including e.g. a WebAssembly JavaScript embedding in a web browser.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.Sys.loadavg","page":"Essentials","location":"base/base.html#Base.Sys.loadavg","category":"function","text":"Sys.loadavg()\n\nGet the load average. See: https://en.wikipedia.org/wiki/Load_(computing).\n\n\n\n\n\n"},{"title":"Base.Sys.isexecutable","page":"Essentials","location":"base/base.html#Base.Sys.isexecutable","category":"function","text":"isexecutable(path::String)\n\nReturn true if the given path has executable permissions.\n\nnote: Note\nThis permission may change before the user executes path, so it is recommended to execute the file and handle the error if that fails, rather than calling isexecutable first.\n\nnote: Note\nPrior to Julia 1.6, this did not correctly interrogate filesystem ACLs on Windows, therefore it would return true for any file.  From Julia 1.6 on, it correctly determines whether the file is marked as executable or not.\n\nSee also ispath, isreadable, iswritable.\n\n\n\n\n\n"},{"title":"Base.Sys.isreadable","page":"Essentials","location":"base/base.html#Base.Sys.isreadable","category":"function","text":"isreadable(io)::Bool\n\nReturn false if the specified IO object is not readable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           isreadable(io)\n       end\nfalse\n\njulia> open(\"myfile.txt\", \"r\") do io\n           isreadable(io)\n       end\ntrue\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\nisreadable(path::String)\n\nReturn true if the access permissions for the given path permitted reading by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling isreadable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, iswritable.\n\n\n\n\n\n"},{"title":"Base.Sys.iswritable","page":"Essentials","location":"base/base.html#Base.Sys.iswritable","category":"function","text":"iswritable(io)::Bool\n\nReturn false if the specified IO object is not writable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           iswritable(io)\n       end\ntrue\n\njulia> open(\"myfile.txt\", \"r\") do io\n           iswritable(io)\n       end\nfalse\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\niswritable(path::String)\n\nReturn true if the access permissions for the given path permitted writing by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling iswritable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, isreadable.\n\n\n\n\n\n"},{"title":"Base.Sys.which","page":"Essentials","location":"base/base.html#Base.Sys.which","category":"function","text":"Sys.which(program_name::String)\n\nGiven a program name, search the current PATH to find the first binary with the proper executable permissions that can be run and return an absolute path to it, or return nothing if no such program is available. If a path with a directory in it is passed in for program_name, tests that exact path for executable permissions only (with .exe and .com extensions added on Windows platforms); no searching of PATH is performed.\n\n\n\n\n\n"},{"title":"Base.Sys.username","page":"Essentials","location":"base/base.html#Base.Sys.username","category":"function","text":"Sys.username()::String\n\nReturn the username for the current user. If the username cannot be determined or is empty, this function throws an error.\n\nTo retrieve a username that is overridable via an environment variable, e.g., USER, consider using\n\nuser = get(Sys.username, ENV, \"USER\")\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also homedir.\n\n\n\n\n\n"},{"title":"Base.@static","page":"Essentials","location":"base/base.html#Base.@static","category":"macro","text":"@static\n\nPartially evaluate an expression at macro expansion time.\n\nThis is useful in cases where a construct would be invalid in some cases, such as a ccall to an os-dependent function, or macros defined in packages that are not imported.\n\n@static requires a conditional. The conditional can be in an if statement, a ternary operator, or &&||. The conditional is evaluated by recursively expanding macros, lowering and executing the resulting expressions. Then, the matching branch (if any) is returned. All the other branches of the conditional are deleted before they are macro-expanded (and lowered or executed).\n\nExample\n\nSuppose we want to parse an expression expr that is valid only on macOS. We could solve this problem using @static with @static if Sys.isapple() expr end. In case we had expr_apple for macOS and expr_others for the other operating systems, the solution with @static would be @static Sys.isapple() ? expr_apple : expr_others.\n\n\n\n\n\n"},{"title":"Base.VersionNumber","page":"Essentials","location":"base/base.html#Base.VersionNumber","category":"type","text":"VersionNumber\n\nVersion number type which follows the specifications of semantic versioning (semver), composed of major, minor and patch numeric values, followed by pre-release and build alphanumeric annotations.\n\nVersionNumber objects can be compared with all of the standard comparison operators (==, <, <=, etc.), with the result following semver v2.0.0-rc.2 rules.\n\nVersionNumber has the following public fields:\n\nv.major::Integer\nv.minor::Integer\nv.patch::Integer\nv.prerelease::Tuple{Vararg{Union{Integer, AbstractString}}}\nv.build::Tuple{Vararg{Union{Integer, AbstractString}}}\n\nSee also @v_str to efficiently construct VersionNumber objects from semver-format literal strings, VERSION for the VersionNumber of Julia itself, and Version Number Literals in the manual.\n\nExamples\n\njulia> a = VersionNumber(1, 2, 3)\nv\"1.2.3\"\n\njulia> a >= v\"1.2\"\ntrue\n\njulia> b = VersionNumber(\"2.0.1-rc1\")\nv\"2.0.1-rc1\"\n\njulia> b >= v\"2.0.1\"\nfalse\n\n\n\n\n\n"},{"title":"Base.@v_str","page":"Essentials","location":"base/base.html#Base.@v_str","category":"macro","text":"@v_str\n\nString macro used to parse a string to a VersionNumber.\n\nExamples\n\njulia> v\"1.2.3\"\nv\"1.2.3\"\n\njulia> v\"2.0.1-rc1\"\nv\"2.0.1-rc1\"\n\n\n\n\n\n"},{"title":"Base.error","page":"Essentials","location":"base/base.html#Base.error","category":"function","text":"error(msg...)\n\nRaise an ErrorException with a message constructed by string(msg...).\n\n\n\n\n\nerror(message::AbstractString)\n\nRaise an ErrorException with the given message.\n\n\n\n\n\n"},{"title":"Core.throw","page":"Essentials","location":"base/base.html#Core.throw","category":"function","text":"throw(e)\n\nThrow an object as an exception.\n\nSee also: rethrow, error.\n\n\n\n\n\n"},{"title":"Base.rethrow","page":"Essentials","location":"base/base.html#Base.rethrow","category":"function","text":"rethrow()\n\nRethrow the current exception from within a catch block. The rethrown exception will continue propagation as if it had not been caught.\n\nnote: Note\nThe alternative form rethrow(e) allows you to associate an alternative exception object e with the current backtrace. However this misrepresents the program state at the time of the error so you're encouraged to instead throw a new exception using throw(e). In Julia 1.1 and above, using throw(e) will preserve the root cause exception on the stack, as described in current_exceptions.\n\n\n\n\n\n"},{"title":"Base.backtrace","page":"Essentials","location":"base/base.html#Base.backtrace","category":"function","text":"backtrace()\n\nGet a backtrace object for the current program point.\n\n\n\n\n\n"},{"title":"Base.catch_backtrace","page":"Essentials","location":"base/base.html#Base.catch_backtrace","category":"function","text":"catch_backtrace()\n\nGet the backtrace of the current exception, for use within catch blocks.\n\n\n\n\n\n"},{"title":"Base.current_exceptions","page":"Essentials","location":"base/base.html#Base.current_exceptions","category":"function","text":"current_exceptions(task::Task=current_task(); [backtrace::Bool=true])\n\nGet the stack of exceptions currently being handled. For nested catch blocks there may be more than one current exception in which case the most recently thrown exception is last in the stack. The stack is returned as an ExceptionStack which is an AbstractVector of named tuples (exception,backtrace). If backtrace is false, the backtrace in each pair will be set to nothing.\n\nExplicitly passing task will return the current exception stack on an arbitrary task. This is useful for inspecting tasks which have failed due to uncaught exceptions.\n\ncompat: Julia 1.7\nThis function went by the experimental name catch_stack() in Julia 1.1–1.6, and had a plain Vector-of-tuples as a return type.\n\n\n\n\n\n"},{"title":"Base.@assert","page":"Essentials","location":"base/base.html#Base.@assert","category":"macro","text":"@assert cond [text]\n\nThrow an AssertionError if cond is false. This is the preferred syntax for writing assertions, which are conditions that are assumed to be true, but that the user might decide to check anyways, as an aid to debugging if they fail. The optional message text is displayed upon assertion failure.\n\nwarning: Warning\nAn assert might be disabled at some optimization levels. Assert should therefore only be used as a debugging tool and not used for authentication verification (e.g., verifying passwords or checking array bounds). The code must not rely on the side effects of running cond for the correct behavior of a function.\n\nExamples\n\njulia> @assert iseven(3) \"3 is an odd number!\"\nERROR: AssertionError: 3 is an odd number!\n\njulia> @assert isodd(3) \"What even are numbers?\"\n\n\n\n\n\n"},{"title":"Base.Experimental.register_error_hint","page":"Essentials","location":"base/base.html#Base.Experimental.register_error_hint","category":"function","text":"Experimental.register_error_hint(handler, exceptiontype)\n\nRegister a \"hinting\" function handler(io, exception) that can suggest potential ways for users to circumvent errors.  handler should examine exception to see whether the conditions appropriate for a hint are met, and if so generate output to io. Packages should call register_error_hint from within their __init__ function.\n\nFor specific exception types, handler is required to accept additional arguments:\n\nMethodError: provide handler(io, exc::MethodError, argtypes, kwargs), which splits the combined arguments into positional and keyword arguments.\n\nWhen issuing a hint, the output should typically start with \\n.\n\nIf you define custom exception types, your showerror method can support hints by calling Experimental.show_error_hints.\n\nExamples\n\njulia> module Hinter\n\n       only_int(x::Int)      = 1\n       any_number(x::Number) = 2\n\n       function __init__()\n           Base.Experimental.register_error_hint(MethodError) do io, exc, argtypes, kwargs\n               if exc.f == only_int\n                    # Color is not necessary, this is just to show it's possible.\n                    print(io, \"\\nDid you mean to call \")\n                    printstyled(io, \"`any_number`?\", color=:cyan)\n               end\n           end\n       end\n\n       end\n\nThen if you call Hinter.only_int on something that isn't an Int (thereby triggering a MethodError), it issues the hint:\n\njulia> Hinter.only_int(1.0)\nERROR: MethodError: no method matching only_int(::Float64)\nThe function `only_int` exists, but no method is defined for this combination of argument types.\nDid you mean to call `any_number`?\nClosest candidates are:\n    ...\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice. To insulate yourself against changes, consider putting any registrations inside an if isdefined(Base.Experimental, :register_error_hint) ... end block.\n\n\n\n\n\n"},{"title":"Base.Experimental.show_error_hints","page":"Essentials","location":"base/base.html#Base.Experimental.show_error_hints","category":"function","text":"Experimental.show_error_hints(io, ex, args...)\n\nInvoke all handlers from Experimental.register_error_hint for the particular exception type typeof(ex) and all of its supertypes. args must contain any other arguments expected by the handler for that type.\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n"},{"title":"Core.ArgumentError","page":"Essentials","location":"base/base.html#Core.ArgumentError","category":"type","text":"ArgumentError(msg)\n\nThe arguments passed to a function are invalid. msg is a descriptive error message.\n\n\n\n\n\n"},{"title":"Core.AssertionError","page":"Essentials","location":"base/base.html#Core.AssertionError","category":"type","text":"AssertionError([msg])\n\nThe asserted condition did not evaluate to true. Optional argument msg is a descriptive error string.\n\nExamples\n\njulia> @assert false \"this is not true\"\nERROR: AssertionError: this is not true\n\nAssertionError is usually thrown from @assert.\n\n\n\n\n\n"},{"title":"Core.BoundsError","page":"Essentials","location":"base/base.html#Core.BoundsError","category":"type","text":"BoundsError([a],[i])\n\nAn indexing operation into an array, a, tried to access an out-of-bounds element at index i.\n\nExamples\n\njulia> A = fill(1.0, 7);\n\njulia> A[8]\nERROR: BoundsError: attempt to access 7-element Vector{Float64} at index [8]\n\n\njulia> B = fill(1.0, (2,3));\n\njulia> B[2, 4]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [2, 4]\n\n\njulia> B[9]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [9]\n\n\n\n\n\n\n"},{"title":"Base.CompositeException","page":"Essentials","location":"base/base.html#Base.CompositeException","category":"type","text":"CompositeException\n\nWrap a Vector of exceptions thrown by a Task (e.g. generated from a remote worker over a channel or an asynchronously executing local I/O write or a remote worker under pmap) with information about the series of exceptions. For example, if a group of workers are executing several tasks, and multiple workers fail, the resulting CompositeException will contain a \"bundle\" of information from each worker indicating where and why the exception(s) occurred.\n\n\n\n\n\n"},{"title":"Base.DimensionMismatch","page":"Essentials","location":"base/base.html#Base.DimensionMismatch","category":"type","text":"DimensionMismatch([msg])\n\nThe objects called do not have matching dimensionality. Optional argument msg is a descriptive error string.\n\n\n\n\n\n"},{"title":"Core.DivideError","page":"Essentials","location":"base/base.html#Core.DivideError","category":"type","text":"DivideError()\n\nInteger division was attempted with a denominator value of 0.\n\nExamples\n\njulia> 2/0\nInf\n\njulia> div(2, 0)\nERROR: DivideError: integer division error\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Core.DomainError","page":"Essentials","location":"base/base.html#Core.DomainError","category":"type","text":"DomainError(val)\nDomainError(val, msg)\n\nThe argument val to a function or constructor is outside the valid domain.\n\nExamples\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.EOFError","page":"Essentials","location":"base/base.html#Base.EOFError","category":"type","text":"EOFError()\n\nNo more data was available to read from a file or stream.\n\n\n\n\n\n"},{"title":"Core.ErrorException","page":"Essentials","location":"base/base.html#Core.ErrorException","category":"type","text":"ErrorException(msg)\n\nGeneric error type. The error message, in the .msg field, may provide more specific details.\n\nExamples\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> ex.msg\n\"I've done a bad thing\"\n\n\n\n\n\n"},{"title":"Core.FieldError","page":"Essentials","location":"base/base.html#Core.FieldError","category":"type","text":"FieldError(type::DataType, field::Symbol)\n\nAn operation tried to access invalid field on an object of type.\n\ncompat: Julia 1.12\nPrior to Julia 1.12, invalid field access threw an ErrorException\n\nSee getfield\n\nExamples\n\njulia> struct AB\n          a::Float32\n          b::Float64\n       end\n\njulia> ab = AB(1, 3)\nAB(1.0f0, 3.0)\n\njulia> ab.c # field `c` doesn't exist\nERROR: FieldError: type AB has no field `c`, available fields: `a`, `b`\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Core.InexactError","page":"Essentials","location":"base/base.html#Core.InexactError","category":"type","text":"InexactError(name::Symbol, T, val)\n\nCannot exactly convert val to type T in a method of function name.\n\nExamples\n\njulia> convert(Float64, 1+2im)\nERROR: InexactError: Float64(1 + 2im)\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Core.InterruptException","page":"Essentials","location":"base/base.html#Core.InterruptException","category":"type","text":"InterruptException()\n\nThe process was stopped by a terminal interrupt (CTRL+C).\n\nNote that, in Julia script started without -i (interactive) option, InterruptException is not thrown by default.  Calling Base.exit_on_sigint(false) in the script can recover the behavior of the REPL.  Alternatively, a Julia script can be started with\n\njulia -e \"include(popfirst!(ARGS))\" script.jl\n\nto let InterruptException be thrown by CTRL+C during the execution.\n\n\n\n\n\n"},{"title":"Base.KeyError","page":"Essentials","location":"base/base.html#Base.KeyError","category":"type","text":"KeyError(key)\n\nAn indexing operation into an AbstractDict (Dict) or Set like object tried to access or delete a non-existent element.\n\n\n\n\n\n"},{"title":"Core.LoadError","page":"Essentials","location":"base/base.html#Core.LoadError","category":"type","text":"LoadError(file::AbstractString, line::Int, error)\n\nAn error occurred while includeing, requireing, or using a file. The error specifics should be available in the .error field.\n\ncompat: Julia 1.7\nLoadErrors are no longer emitted by @macroexpand, @macroexpand1, and macroexpand as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Core.MethodError","page":"Essentials","location":"base/base.html#Core.MethodError","category":"type","text":"MethodError(f, args)\n\nA method with the required type signature does not exist in the given generic function. Alternatively, there is no unique most-specific method.\n\n\n\n\n\n"},{"title":"Base.MissingException","page":"Essentials","location":"base/base.html#Base.MissingException","category":"type","text":"MissingException(msg)\n\nException thrown when a missing value is encountered in a situation where it is not supported. The error message, in the msg field may provide more specific details.\n\n\n\n\n\n"},{"title":"Core.OutOfMemoryError","page":"Essentials","location":"base/base.html#Core.OutOfMemoryError","category":"type","text":"OutOfMemoryError()\n\nAn operation allocated too much memory for either the system or the garbage collector to handle properly.\n\n\n\n\n\n"},{"title":"Core.ReadOnlyMemoryError","page":"Essentials","location":"base/base.html#Core.ReadOnlyMemoryError","category":"type","text":"ReadOnlyMemoryError()\n\nAn operation tried to write to memory that is read-only.\n\n\n\n\n\n"},{"title":"Core.OverflowError","page":"Essentials","location":"base/base.html#Core.OverflowError","category":"type","text":"OverflowError(msg)\n\nThe result of an expression is too large for the specified type and will cause a wraparound.\n\n\n\n\n\n"},{"title":"Base.ProcessFailedException","page":"Essentials","location":"base/base.html#Base.ProcessFailedException","category":"type","text":"ProcessFailedException\n\nIndicates problematic exit status of a process. When running commands or pipelines, this is thrown to indicate a nonzero exit code was returned (i.e. that the invoked process failed).\n\n\n\n\n\n"},{"title":"Base.TaskFailedException","page":"Essentials","location":"base/base.html#Base.TaskFailedException","category":"type","text":"TaskFailedException\n\nThis exception is thrown by a wait(t) call when task t fails. TaskFailedException wraps the failed task t.\n\n\n\n\n\n"},{"title":"Core.StackOverflowError","page":"Essentials","location":"base/base.html#Core.StackOverflowError","category":"type","text":"StackOverflowError()\n\nThe function call grew beyond the size of the call stack. This usually happens when a call recurses infinitely.\n\n\n\n\n\n"},{"title":"Base.SystemError","page":"Essentials","location":"base/base.html#Base.SystemError","category":"type","text":"SystemError(prefix::AbstractString, [errno::Int32])\n\nA system call failed with an error code (in the errno global variable).\n\n\n\n\n\n"},{"title":"Core.TypeError","page":"Essentials","location":"base/base.html#Core.TypeError","category":"type","text":"TypeError(func::Symbol, context::AbstractString, expected::Type, got)\n\nA type assertion failure, or calling an intrinsic function with an incorrect argument type.\n\n\n\n\n\n"},{"title":"Core.UndefKeywordError","page":"Essentials","location":"base/base.html#Core.UndefKeywordError","category":"type","text":"UndefKeywordError(var::Symbol)\n\nThe required keyword argument var was not assigned in a function call.\n\nExamples\n\njulia> function my_func(;my_arg)\n           return my_arg + 1\n       end\nmy_func (generic function with 1 method)\n\njulia> my_func()\nERROR: UndefKeywordError: keyword argument `my_arg` not assigned\nStacktrace:\n [1] my_func() at ./REPL[1]:2\n [2] top-level scope at REPL[2]:1\n\n\n\n\n\n"},{"title":"Core.UndefRefError","page":"Essentials","location":"base/base.html#Core.UndefRefError","category":"type","text":"UndefRefError()\n\nThe item or field is not defined for the given object.\n\nExamples\n\njulia> struct MyType\n           a::Vector{Int}\n           MyType() = new()\n       end\n\njulia> A = MyType()\nMyType(#undef)\n\njulia> A.a\nERROR: UndefRefError: access to undefined reference\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Core.UndefVarError","page":"Essentials","location":"base/base.html#Core.UndefVarError","category":"type","text":"UndefVarError(var::Symbol, [scope])\n\nA symbol in the current scope is not defined.\n\nExamples\n\njulia> a\nERROR: UndefVarError: `a` not defined in `Main`\n\njulia> a = 1;\n\njulia> a\n1\n\n\n\n\n\n"},{"title":"Base.StringIndexError","page":"Essentials","location":"base/base.html#Base.StringIndexError","category":"type","text":"StringIndexError(str, i)\n\nAn error occurred when trying to access str at index i that is not valid.\n\n\n\n\n\n"},{"title":"Core.InitError","page":"Essentials","location":"base/base.html#Core.InitError","category":"type","text":"InitError(mod::Symbol, error)\n\nAn error occurred when running a module's __init__ function. The actual error thrown is available in the .error field.\n\n\n\n\n\n"},{"title":"Base.retry","page":"Essentials","location":"base/base.html#Base.retry","category":"function","text":"retry(f;  delays=ExponentialBackOff(), check=nothing) -> Function\n\nReturn an anonymous function that calls function f.  If an exception arises, f is repeatedly called again, each time check returns true, after waiting the number of seconds specified in delays.  check should input delays's current state and the Exception.\n\ncompat: Julia 1.2\nBefore Julia 1.2 this signature was restricted to f::Function.\n\nExamples\n\nretry(f, delays=fill(5.0, 3))\nretry(f, delays=rand(5:10, 2))\nretry(f, delays=Base.ExponentialBackOff(n=3, first_delay=5, max_delay=1000))\nretry(http_get, check=(s,e)->e.status == \"503\")(url)\nretry(read, check=(s,e)->isa(e, IOError))(io, 128; all=false)\n\n\n\n\n\n"},{"title":"Base.ExponentialBackOff","page":"Essentials","location":"base/base.html#Base.ExponentialBackOff","category":"type","text":"ExponentialBackOff(; n=1, first_delay=0.05, max_delay=10.0, factor=5.0, jitter=0.1)\n\nA Float64 iterator of length n whose elements exponentially increase at a rate in the interval factor * (1 ± jitter).  The first element is first_delay and all elements are clamped to max_delay.\n\n\n\n\n\n"},{"title":"Base.Timer","page":"Essentials","location":"base/base.html#Base.Timer-Tuple{Function, Real}","category":"method","text":"Timer(callback::Function, delay; interval = 0, spawn::Union{Nothing,Bool}=nothing)\n\nCreate a timer that runs the function callback at each timer expiration.\n\nWaiting tasks are woken and the function callback is called after an initial delay of delay seconds, and then repeating with the given interval in seconds. If interval is equal to 0, the callback is only run once. The function callback is called with a single argument, the timer itself. Stop a timer by calling close. The callback may still be run one final time, if the timer has already expired.\n\nIf spawn is true, the created task will be spawned, meaning that it will be allowed to move thread, which avoids the side-effect of forcing the parent task to get stuck to the thread it is on. If spawn is nothing (default), the task will be spawned if the parent task isn't sticky.\n\ncompat: Julia 1.12\nThe spawn argument was introduced in Julia 1.12.\n\nExamples\n\nHere the first number is printed after a delay of two seconds, then the following numbers are printed quickly.\n\njulia> begin\n           i = 0\n           cb(timer) = (global i += 1; println(i))\n           t = Timer(cb, 2, interval=0.2)\n           wait(t)\n           sleep(0.5)\n           close(t)\n       end\n1\n2\n3\n\n\n\n\n\n"},{"title":"Base.Timer","page":"Essentials","location":"base/base.html#Base.Timer","category":"type","text":"Timer(delay; interval = 0)\n\nCreate a timer that wakes up tasks waiting for it (by calling wait on the timer object).\n\nWaiting tasks are woken after an initial delay of at least delay seconds, and then repeating after at least interval seconds again elapse. If interval is equal to 0, the timer is only triggered once. When the timer is closed (by close) waiting tasks are woken with an error. Use isopen to check whether a timer is still active. An inactive timer will not fire. Use t.timeout and t.interval to read the setup conditions of a Timer t.\n\njulia> t = Timer(1.0; interval=0.5)\nTimer (open, timeout: 1.0 s, interval: 0.5 s) @0x000000010f4e6e90\n\njulia> isopen(t)\ntrue\n\njulia> t.timeout\n1.0\n\njulia> close(t)\n\njulia> isopen(t)\nfalse\n\nnote: Note\ninterval is subject to accumulating time skew. If you need precise events at a particular absolute time, create a new timer at each expiration with the difference to the next time computed.\n\nnote: Note\nA Timer requires yield points to update its state. For instance, isopen(t::Timer) cannot be used to timeout a non-yielding while loop.\n\n!!! compat \"Julia 1.12     The timeout and interval readable properties were added in Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.AsyncCondition","page":"Essentials","location":"base/base.html#Base.AsyncCondition","category":"type","text":"AsyncCondition()\n\nCreate an async condition that wakes up tasks waiting for it (by calling wait on the object) when notified from C by a call to uv_async_send. Waiting tasks are woken with an error when the object is closed (by close). Use isopen to check whether it is still active. A closed condition is inactive and will not wake up tasks.\n\nThis provides an implicit acquire & release memory ordering between the sending and waiting threads.\n\n\n\n\n\n"},{"title":"Base.AsyncCondition","page":"Essentials","location":"base/base.html#Base.AsyncCondition-Tuple{Function}","category":"method","text":"AsyncCondition(callback::Function)\n\nCreate an async condition that calls the given callback function. The callback is passed one argument, the async condition object itself.\n\n\n\n\n\n"},{"title":"Base.nameof","page":"Essentials","location":"base/base.html#Base.nameof-Tuple{Module}","category":"method","text":"nameof(m::Module)::Symbol\n\nGet the name of a Module as a Symbol.\n\nExamples\n\njulia> nameof(Base.Broadcast)\n:Broadcast\n\n\n\n\n\n"},{"title":"Base.parentmodule","page":"Essentials","location":"base/base.html#Base.parentmodule","category":"function","text":"parentmodule(m::Method)::Module\n\nReturn the module in which the given method m is defined.\n\ncompat: Julia 1.9\nPassing a Method as an argument requires Julia 1.9 or later.\n\n\n\n\n\nparentmodule(f::Function, types)::Module\n\nDetermine the module containing the first method of a generic function f matching the specified types.\n\n\n\n\n\nparentmodule(f::Function)::Module\n\nDetermine the module containing the (first) definition of a generic function.\n\n\n\n\n\nparentmodule(t::DataType)::Module\n\nDetermine the module containing the definition of a (potentially UnionAll-wrapped) DataType.\n\nExamples\n\njulia> module Foo\n           struct Int end\n       end\nFoo\n\njulia> parentmodule(Int)\nCore\n\njulia> parentmodule(Foo.Int)\nFoo\n\n\n\n\n\nparentmodule(m::Module)::Module\n\nGet a module's enclosing Module. Main is its own parent.\n\nSee also: names, nameof, fullname, @__MODULE__.\n\nExamples\n\njulia> parentmodule(Main)\nMain\n\njulia> parentmodule(Base.Broadcast)\nBase\n\n\n\n\n\n"},{"title":"Base.pathof","page":"Essentials","location":"base/base.html#Base.pathof-Tuple{Module}","category":"method","text":"pathof(m::Module)\n\nReturn the path of the m.jl file that was used to import module m, or nothing if m was not imported from a package.\n\nUse dirname to get the directory part and basename to get the file name part of the path.\n\nSee also pkgdir.\n\n\n\n\n\n"},{"title":"Base.pkgdir","page":"Essentials","location":"base/base.html#Base.pkgdir-Tuple{Module}","category":"method","text":"pkgdir(m::Module[, paths::String...])\n\nReturn the root directory of the package that declared module m, or nothing if m was not declared in a package. Optionally further path component strings can be provided to construct a path within the package root.\n\nTo get the root directory of the package that implements the current module the form pkgdir(@__MODULE__) can be used.\n\nIf an extension module is given, the root of the parent package is returned.\n\njulia> pkgdir(Foo)\n\"/path/to/Foo.jl\"\n\njulia> pkgdir(Foo, \"src\", \"file.jl\")\n\"/path/to/Foo.jl/src/file.jl\"\n\nSee also pathof.\n\ncompat: Julia 1.7\nThe optional argument paths requires at least Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.pkgversion","page":"Essentials","location":"base/base.html#Base.pkgversion-Tuple{Module}","category":"method","text":"pkgversion(m::Module)\n\nIf the module m belongs to a versioned package, return the version number of that package. Otherwise return nothing.\n\nThe version is read from the package's Project.toml during package load.\n\nTo get the version of the package that imported the current module the form pkgversion(@__MODULE__) can be used.\n\ncompat: Julia 1.9\nThis function was introduced in Julia 1.9.\n\n\n\n\n\n"},{"title":"Base.moduleroot","page":"Essentials","location":"base/base.html#Base.moduleroot","category":"function","text":"moduleroot(m::Module)::Module\n\nFind the root module of a given module. This is the first module in the chain of parent modules of m which is either a registered root module or which is its own parent module.\n\n\n\n\n\n"},{"title":"__module__","page":"Essentials","location":"base/base.html#__module__","category":"keyword","text":"__module__\n\nThe argument __module__ is only visible inside the macro, and it provides information (in the form of a Module object) about the expansion context of the macro invocation. See the manual section on Macro invocation for more information.\n\n\n\n\n\n"},{"title":"__source__","page":"Essentials","location":"base/base.html#__source__","category":"keyword","text":"__source__\n\nThe argument __source__ is only visible inside the macro, and it provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. See the manual section on Macro invocation for more information.\n\n\n\n\n\n"},{"title":"Base.@__MODULE__","page":"Essentials","location":"base/base.html#Base.@__MODULE__","category":"macro","text":"@__MODULE__ -> Module\n\nGet the Module of the toplevel eval, which is the Module code is currently being read from.\n\n\n\n\n\n"},{"title":"Base.@__FUNCTION__","page":"Essentials","location":"base/base.html#Base.@__FUNCTION__","category":"macro","text":"@__FUNCTION__\n\nGet the innermost enclosing function object.\n\nnote: Note\n@__FUNCTION__ has the same scoping behavior as return: when used inside a closure, it refers to the closure and not the outer function. Some macros, including @spawn, @async, etc., wrap their input in closures. When @__FUNCTION__ is used within such code, it will refer to the closure created by the macro rather than the enclosing function.\n\nExamples\n\n@__FUNCTION__ enables recursive anonymous functions:\n\njulia> factorial = (n -> n <= 1 ? 1 : n * (@__FUNCTION__)(n - 1));\n\njulia> factorial(5)\n120\n\n@__FUNCTION__ can be combined with nameof to identify a function's name from within its body:\n\njulia> bar() = nameof(@__FUNCTION__);\n\njulia> bar()\n:bar\n\n\n\n\n\n"},{"title":"Base.@__FILE__","page":"Essentials","location":"base/base.html#Base.@__FILE__","category":"macro","text":"@__FILE__ -> String\n\nExpand to a string with the path to the file containing the macrocall, or an empty string if evaluated by julia -e <expr>. Return nothing if the macro was missing parser source information. Alternatively see PROGRAM_FILE.\n\n\n\n\n\n"},{"title":"Base.@__DIR__","page":"Essentials","location":"base/base.html#Base.@__DIR__","category":"macro","text":"@__DIR__ -> String\n\nMacro to obtain the absolute path of the current directory as a string.\n\nIf in a script, returns the directory of the script containing the @__DIR__ macrocall. If run from a REPL or if evaluated by julia -e <expr>, returns the current working directory.\n\nExamples\n\nThe example illustrates the difference in the behaviors of @__DIR__ and pwd(), by creating a simple script in a different directory than the current working one and executing both commands:\n\njulia> cd(\"/home/JuliaUser\") # working directory\n\njulia> # create script at /home/JuliaUser/Projects\n       open(\"/home/JuliaUser/Projects/test.jl\",\"w\") do io\n           print(io, \"\"\"\n               println(\"@__DIR__ = \", @__DIR__)\n               println(\"pwd() = \", pwd())\n           \"\"\")\n       end\n\njulia> # outputs script directory and current working directory\n       include(\"/home/JuliaUser/Projects/test.jl\")\n@__DIR__ = /home/JuliaUser/Projects\npwd() = /home/JuliaUser\n\n\n\n\n\n"},{"title":"Base.@__LINE__","page":"Essentials","location":"base/base.html#Base.@__LINE__","category":"macro","text":"@__LINE__ -> Int\n\nExpand to the line number of the location of the macrocall. Return 0 if the line number could not be determined.\n\n\n\n\n\n"},{"title":"Base.fullname","page":"Essentials","location":"base/base.html#Base.fullname","category":"function","text":"fullname(m::Module)\n\nGet the fully-qualified name of a module as a tuple of symbols. For example,\n\nExamples\n\njulia> fullname(Base.Iterators)\n(:Base, :Iterators)\n\njulia> fullname(Main)\n(:Main,)\n\n\n\n\n\n"},{"title":"Base.names","page":"Essentials","location":"base/base.html#Base.names","category":"function","text":"names(x::Module; all::Bool=false, imported::Bool=false, usings::Bool=false)::Vector{Symbol}\n\nGet a vector of the public names of a Module, excluding deprecated names. If all is true, then the list also includes non-public names defined in the module, deprecated names, and compiler-generated names. If imported is true, then names explicitly imported from other modules are also included. If usings is true, then names explicitly or implicitly imported via using are also included. Names are returned in sorted order.\n\nAs a special case, all names defined in Main are considered \"public\", since it is not idiomatic to explicitly mark names from Main as public.\n\nnote: Note\nsym ∈ names(SomeModule) does not imply isdefined(SomeModule, sym). names may return symbols marked with public or export, even if they are not defined in the module.\n\nwarning: Warning\nnames may return duplicate names. The duplication happens, e.g. if an imported name conflicts with an already existing identifier.\n\ncompat: Julia 1.12\nThe usings argument requires Julia 1.12 or later.\n\nSee also: Base.isexported, Base.ispublic, Base.@locals, @__MODULE__.\n\n\n\n\n\n"},{"title":"Base.isexported","page":"Essentials","location":"base/base.html#Base.isexported","category":"function","text":"isexported(m::Module, s::Symbol)::Bool\n\nReturns whether a symbol is exported from a module.\n\nSee also: ispublic, names\n\njulia> module Mod\n           export foo\n           public bar\n       end\nMod\n\njulia> Base.isexported(Mod, :foo)\ntrue\n\njulia> Base.isexported(Mod, :bar)\nfalse\n\njulia> Base.isexported(Mod, :baz)\nfalse\n\n\n\n\n\n"},{"title":"Base.ispublic","page":"Essentials","location":"base/base.html#Base.ispublic","category":"function","text":"ispublic(m::Module, s::Symbol)::Bool\n\nReturns whether a symbol is marked as public in a module.\n\nExported symbols are considered public.\n\ncompat: Julia 1.11\nThis function and the notion of publicity were added in Julia 1.11.\n\nSee also: isexported, names\n\njulia> module Mod\n           export foo\n           public bar\n       end\nMod\n\njulia> Base.ispublic(Mod, :foo)\ntrue\n\njulia> Base.ispublic(Mod, :bar)\ntrue\n\njulia> Base.ispublic(Mod, :baz)\nfalse\n\n\n\n\n\n"},{"title":"Base.nameof","page":"Essentials","location":"base/base.html#Base.nameof-Tuple{Function}","category":"method","text":"nameof(f::Function)::Symbol\n\nGet the name of a generic Function as a symbol. For anonymous functions, this is a compiler-generated name. For explicitly-declared subtypes of Function, it is the name of the function's type.\n\n\n\n\n\n"},{"title":"Base.functionloc","page":"Essentials","location":"base/base.html#Base.functionloc-Tuple{Any, Any}","category":"method","text":"functionloc(f::Function, types)\n\nReturn a tuple (filename,line) giving the location of a generic Function definition.\n\n\n\n\n\n"},{"title":"Base.functionloc","page":"Essentials","location":"base/base.html#Base.functionloc-Tuple{Method}","category":"method","text":"functionloc(m::Method)\n\nReturn a tuple (filename,line) giving the location of a Method definition.\n\n\n\n\n\n"},{"title":"Base.@locals","page":"Essentials","location":"base/base.html#Base.@locals","category":"macro","text":"@locals()\n\nConstruct a dictionary of the names (as symbols) and values of all local variables defined as of the call site.\n\ncompat: Julia 1.1\nThis macro requires at least Julia 1.1.\n\nExamples\n\njulia> let x = 1, y = 2\n           Base.@locals\n       end\nDict{Symbol, Any} with 2 entries:\n  :y => 2\n  :x => 1\n\njulia> function f(x)\n           local y\n           show(Base.@locals); println()\n           for i = 1:1\n               show(Base.@locals); println()\n           end\n           y = 2\n           show(Base.@locals); println()\n           nothing\n       end;\n\njulia> f(42)\nDict{Symbol, Any}(:x => 42)\nDict{Symbol, Any}(:i => 1, :x => 42)\nDict{Symbol, Any}(:y => 2, :x => 42)\n\n\n\n\n\n"},{"title":"Core.getglobal","page":"Essentials","location":"base/base.html#Core.getglobal","category":"function","text":"getglobal(module::Module, name::Symbol, [order::Symbol=:monotonic])\n\nRetrieve the value of the binding name from the module module. Optionally, an atomic ordering can be defined for the operation, otherwise it defaults to monotonic.\n\nWhile accessing module bindings using getfield is still supported to maintain compatibility, using getglobal should always be preferred since getglobal allows for control over atomic ordering (getfield is always monotonic) and better signifies the code's intent both to the user as well as the compiler.\n\nMost users should not have to call this function directly – The getproperty function or corresponding syntax (i.e. module.name) should be preferred in all but few very specific use cases.\n\ncompat: Julia 1.9\nThis function requires Julia 1.9 or later.\n\nSee also getproperty and setglobal!.\n\nExamples\n\njulia> a = 1\n1\n\njulia> module M\n       a = 2\n       end;\n\njulia> getglobal(@__MODULE__, :a)\n1\n\njulia> getglobal(M, :a)\n2\n\n\n\n\n\n"},{"title":"Core.setglobal!","page":"Essentials","location":"base/base.html#Core.setglobal!","category":"function","text":"setglobal!(module::Module, name::Symbol, x, [order::Symbol=:monotonic])\n\nSet or change the value of the binding name in the module module to x. No type conversion is performed, so if a type has already been declared for the binding, x must be of appropriate type or an error is thrown.\n\nAdditionally, an atomic ordering can be specified for this operation, otherwise it defaults to monotonic.\n\nUsers will typically access this functionality through the setproperty! function or corresponding syntax (i.e. module.name = x) instead, so this is intended only for very specific use cases.\n\ncompat: Julia 1.9\nThis function requires Julia 1.9 or later.\n\nSee also setproperty! and getglobal\n\nExamples\n\njulia> module M; global a; end;\n\njulia> M.a  # same as `getglobal(M, :a)`\nERROR: UndefVarError: `a` not defined in `M`\nSuggestion: add an appropriate import or assignment. This global was declared but not assigned.\nStacktrace:\n [1] getproperty(x::Module, f::Symbol)\n   @ Base ./Base_compiler.jl:40\n [2] top-level scope\n   @ none:1\n\njulia> setglobal!(M, :a, 1)\n1\n\njulia> M.a\n1\n\n\n\n\n\n"},{"title":"Core.modifyglobal!","page":"Essentials","location":"base/base.html#Core.modifyglobal!","category":"function","text":"modifyglobal!(module::Module, name::Symbol, op, x, [order::Symbol=:monotonic])::Pair\n\nAtomically perform the operations to get and set a global after applying the function op.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also modifyproperty! and setglobal!.\n\n\n\n\n\n"},{"title":"Core.swapglobal!","page":"Essentials","location":"base/base.html#Core.swapglobal!","category":"function","text":"swapglobal!(module::Module, name::Symbol, x, [order::Symbol=:monotonic])\n\nAtomically perform the operations to simultaneously get and set a global.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also swapproperty! and setglobal!.\n\n\n\n\n\n"},{"title":"Core.setglobalonce!","page":"Essentials","location":"base/base.html#Core.setglobalonce!","category":"function","text":"setglobalonce!(module::Module, name::Symbol, value,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> success::Bool\n\nAtomically perform the operations to set a global to a given value, only if it was previously not set.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also setpropertyonce! and setglobal!.\n\n\n\n\n\n"},{"title":"Core.replaceglobal!","page":"Essentials","location":"base/base.html#Core.replaceglobal!","category":"function","text":"replaceglobal!(module::Module, name::Symbol, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\n\nAtomically perform the operations to get and conditionally set a global to a given value.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also replaceproperty! and setglobal!.\n\n\n\n\n\n"},{"title":"Core.declare_const","page":"Essentials","location":"base/base.html#Core.declare_const","category":"function","text":"declare_const(module::Module, name::Symbol, [x])\n\nCreate or replace the constant name in module with the new value x.  When replacing, x does not need to have the same type as the original constant.\n\nWhen x is not given, name becomes an undefined constant; it cannot be read or written to, but can be redefined.\n\nUnlike the syntax const, calling this function does not insert Core.@latestworld to update the world age of the current frame:\n\njulia> begin\n           const x = 1\n           println(x)\n           const x = 2\n           println(x)\n           Core.declare_const(Main, :x, 3)\n           println(x)\n           Core.@latestworld\n           println(x)\n       end\n1\n2\n2\n3\n\ncompat: Julia 1.12\nThis function requires Julia 1.12 or later.  Redefining constants on earlier versions of Julia is unpredictable.\n\nSee also const.\n\n\n\n\n\n"},{"title":"Core.@doc","page":"Essentials","location":"base/base.html#Core.@doc","category":"macro","text":"Documentation\n\nFunctions, methods and types can be documented by placing a string before the definition:\n\n\"\"\"\n    foo(x)\n\nReturn a fooified version of `x`.\n\"\"\"\nfoo(x) = ...\n\nThe @doc macro can be used directly to both set and retrieve documentation / metadata. The macro has special parsing so that the documented object may occur on the next line:\n\n@doc \"blah\"\nfunction foo() ...\n\nBy default, documentation is written as Markdown, but any object can be used as the first argument.\n\nDocumenting objects separately from their definitions\n\nYou can document an object before or after its definition with\n\n@doc \"foo\" function_to_doc\n@doc \"bar\" TypeToDoc\n\nFor macros, the syntax is @doc \"macro doc\" :(Module.@macro) or @doc \"macro doc\" :(string_macro\"\") for string macros. Without the quote :() the expansion of the macro will be documented.\n\nRetrieving Documentation\n\nYou can retrieve docs for functions, macros and other objects as follows:\n\n@doc foo\n@doc @time\n@doc md\"\"\n\ncompat: Julia 1.11\nIn Julia 1.11 and newer, retrieving documentation with the @doc macro requires that the REPL stdlib is loaded.\n\nFunctions & Methods\n\nPlacing documentation before a method definition (e.g. function foo() ... or foo() = ...) will cause that specific method to be documented, as opposed to the whole function. Method docs are concatenated together in the order they were defined to provide docs for the function.\n\n\n\n\n\n"},{"title":"Base.Docs.HTML","page":"Essentials","location":"base/base.html#Base.Docs.HTML","category":"type","text":"HTML(s): Create an object that renders s as html.\n\nHTML(\"<div>foo</div>\")\n\nYou can also use a stream for large amounts of data:\n\nHTML() do io\n  println(io, \"<div>foo</div>\")\nend\n\nwarning: Warning\nHTML is currently exported to maintain backwards compatibility, but this export is deprecated. It is recommended to use this type as Docs.HTML or to explicitly import it from Docs.\n\n\n\n\n\n"},{"title":"Base.Docs.Text","page":"Essentials","location":"base/base.html#Base.Docs.Text","category":"type","text":"Text(s): Create an object that renders s as plain text.\n\nText(\"foo\")\n\nYou can also use a stream for large amounts of data:\n\nText() do io\n  println(io, \"foo\")\nend\n\nwarning: Warning\nText is currently exported to maintain backwards compatibility, but this export is deprecated. It is recommended to use this type as Docs.Text or to explicitly import it from Docs.\n\n\n\n\n\n"},{"title":"Base.Docs.hasdoc","page":"Essentials","location":"base/base.html#Base.Docs.hasdoc","category":"function","text":"Docs.hasdoc(mod::Module, sym::Symbol)::Bool\n\nReturn true if sym in mod has a docstring and false otherwise.\n\n\n\n\n\n"},{"title":"Base.Docs.undocumented_names","page":"Essentials","location":"base/base.html#Base.Docs.undocumented_names","category":"function","text":"undocumented_names(mod::Module; private=false)\n\nReturn a sorted vector of undocumented symbols in module (that is, lacking docstrings). private=false (the default) returns only identifiers declared with public and/or export, whereas private=true returns all symbols in the module (excluding compiler-generated hidden symbols starting with #).\n\nSee also: names, Docs.hasdoc, Base.ispublic.\n\n\n\n\n\n"},{"title":"Base.identify_package","page":"Essentials","location":"base/base.html#Base.identify_package","category":"function","text":"Base.identify_package(name::String)::Union{PkgId, Nothing}\nBase.identify_package(where::Union{Module,PkgId}, name::String)::Union{PkgId, Nothing}\n\nIdentify the package by its name from the current environment stack, returning its PkgId, or nothing if it cannot be found.\n\nIf only the name argument is provided, it searches each environment in the stack and its named direct dependencies.\n\nThe where argument provides the context from where to search for the package: in this case it first checks if the name matches the context itself, otherwise it searches all recursive dependencies (from the resolved manifest of each environment) until it locates the context where, and from there identifies the dependency with the corresponding name.\n\njulia> Base.identify_package(\"Pkg\") # Pkg is a dependency of the default environment\nPkg [44cfe95a-1eb2-52ea-b672-e2afdf69b78f]\n\njulia> using LinearAlgebra\n\njulia> Base.identify_package(LinearAlgebra, \"Pkg\") # Pkg is not a dependency of LinearAlgebra\n\n\n\n\n\n"},{"title":"Base.locate_package","page":"Essentials","location":"base/base.html#Base.locate_package","category":"function","text":"Base.locate_package(pkg::PkgId)::Union{String, Nothing}\n\nThe path to the entry-point file for the package corresponding to the identifier pkg, or nothing if not found. See also identify_package.\n\njulia> pkg = Base.identify_package(\"Pkg\")\nPkg [44cfe95a-1eb2-52ea-b672-e2afdf69b78f]\n\njulia> Base.locate_package(pkg)\n\"/path/to/julia/stdlib/v1.13/Pkg/src/Pkg.jl\"\n\n\n\n\n\n"},{"title":"Base.require","page":"Essentials","location":"base/base.html#Base.require","category":"function","text":"require(into::Module, module::Symbol)\n\nThis function is part of the implementation of using / import, if a module is not already defined in Main. It can also be called directly to force reloading a module, regardless of whether it has been loaded before (for example, when interactively developing libraries).\n\nLoads a source file, in the context of the Main module, on every active node, searching standard locations for files. require is considered a top-level operation, so it sets the current include path but does not use it to search for files (see help for include). This function is typically used to load library code, and is implicitly called by using to load packages.\n\nWhen searching for files, require first looks for package code in the global array LOAD_PATH. require is case-sensitive on all platforms, including those with case-insensitive filesystems like macOS and Windows.\n\nFor more details regarding code loading, see the manual sections on modules and parallel computing.\n\n\n\n\n\n"},{"title":"Base.compilecache","page":"Essentials","location":"base/base.html#Base.compilecache","category":"function","text":"Base.compilecache(module::PkgId)\n\nCreates a precompiled cache file for a module and all of its dependencies. This can be used to reduce package load times. Cache files are stored in DEPOT_PATH[1]/compiled. See Module initialization and precompilation for important notes.\n\n\n\n\n\n"},{"title":"Base.isprecompiled","page":"Essentials","location":"base/base.html#Base.isprecompiled","category":"function","text":"Base.isprecompiled(pkg::PkgId; ignore_loaded::Bool=false)\n\nReturns whether a given PkgId within the active project is precompiled.\n\nBy default this check observes the same approach that code loading takes with respect to when different versions of dependencies are currently loaded to that which is expected. To ignore loaded modules and answer as if in a fresh julia session specify ignore_loaded=true.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.get_extension","page":"Essentials","location":"base/base.html#Base.get_extension","category":"function","text":"get_extension(parent::Module, extension::Symbol)\n\nReturn the module for extension of parent or return nothing if the extension is not loaded.\n\n\n\n\n\n"},{"title":"Base.GC.gc","page":"Essentials","location":"base/base.html#Base.GC.gc","category":"function","text":"GC.gc([full=true])\n\nPerform garbage collection. The argument full determines the kind of collection: a full collection (default) traverses all live objects (i.e. full mark) and should reclaim memory from all unreachable objects. An incremental collection only reclaims memory from young objects which are not reachable.\n\nThe GC may decide to perform a full collection even if an incremental collection was requested.\n\nwarning: Warning\nExcessive use will likely lead to poor performance.\n\n\n\n\n\n"},{"title":"Base.GC.enable","page":"Essentials","location":"base/base.html#Base.GC.enable","category":"function","text":"GC.enable(on::Bool)\n\nControl whether garbage collection is enabled using a boolean argument (true for enabled, false for disabled). Return previous GC state.\n\nwarning: Warning\nDisabling garbage collection should be used only with caution, as it can cause memory use to grow without bound.\n\n\n\n\n\n"},{"title":"Base.GC.@preserve","page":"Essentials","location":"base/base.html#Base.GC.@preserve","category":"macro","text":"GC.@preserve x1 x2 ... xn expr\n\nMark the objects x1, x2, ... as being in use during the evaluation of the expression expr. This is only required in unsafe code where expr implicitly uses memory or other resources owned by one of the xs.\n\nImplicit use of x covers any indirect use of resources logically owned by x which the compiler cannot see. Some examples:\n\nAccessing memory of an object directly via a Ptr\nPassing a pointer to x to ccall\nUsing resources of x which would be cleaned up in the finalizer.\n\n@preserve should generally not have any performance impact in typical use cases where it briefly extends object lifetime. In implementation, @preserve has effects such as protecting dynamically allocated objects from garbage collection.\n\nExamples\n\nWhen loading from a pointer with unsafe_load, the underlying object is implicitly used, for example x is implicitly used by unsafe_load(p) in the following:\n\njulia> let\n           x = Ref{Int}(101)\n           p = Base.unsafe_convert(Ptr{Int}, x)\n           GC.@preserve x unsafe_load(p)\n       end\n101\n\nWhen passing pointers to ccall, the pointed-to object is implicitly used and should be preserved. (Note however that you should normally just pass x directly to ccall which counts as an explicit use.)\n\njulia> let\n           x = \"Hello\"\n           p = pointer(x)\n           Int(GC.@preserve x @ccall strlen(p::Cstring)::Csize_t)\n           # Preferred alternative\n           Int(@ccall strlen(x::Cstring)::Csize_t)\n       end\n5\n\n\n\n\n\n"},{"title":"Base.GC.safepoint","page":"Essentials","location":"base/base.html#Base.GC.safepoint","category":"function","text":"GC.safepoint()\n\nInserts a point in the program where garbage collection may run.\n\nSafepoints are fast and do not themselves trigger garbage collection. However, if another thread has requested the GC to run, reaching a safepoint will cause the current thread to block and wait for the GC.\n\nThis can be useful in rare cases in multi-threaded programs where some tasks are allocating memory (and hence may need to run GC) but other tasks are doing only simple operations (no allocation, task switches, or I/O), which do not yield control to Julia's runtime, and therefore blocks the GC from running. Calling this function periodically in the non-allocating tasks allows garbage collection to run.\n\nNote that even though safepoints are fast (typically around 2 clock cycles), they can still degrade performance if called in a tight loop.\n\ncompat: Julia 1.4\nThis function is available as of Julia 1.4.\n\n\n\n\n\n"},{"title":"Base.GC.enable_logging","page":"Essentials","location":"base/base.html#Base.GC.enable_logging","category":"function","text":"GC.enable_logging(on::Bool)\n\nWhen turned on, print statistics about each GC to stderr.\n\n\n\n\n\n"},{"title":"Base.GC.logging_enabled","page":"Essentials","location":"base/base.html#Base.GC.logging_enabled","category":"function","text":"GC.logging_enabled()\n\nReturn whether GC logging has been enabled via GC.enable_logging.\n\n\n\n\n\n"},{"title":"Base.Meta.lower","page":"Essentials","location":"base/base.html#Base.Meta.lower","category":"function","text":"lower(m, x)\n\nTakes the expression x and returns an equivalent expression in lowered form for executing in module m. See also code_lowered.\n\n\n\n\n\n"},{"title":"Base.Meta.@lower","page":"Essentials","location":"base/base.html#Base.Meta.@lower","category":"macro","text":"@lower [m] x\n\nReturn lowered form of the expression x in module m. By default m is the module in which the macro is called. See also lower.\n\n\n\n\n\n"},{"title":"Base.Meta.parse","page":"Essentials","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString, Int64}","category":"method","text":"parse(str, start; greedy=true, raise=true, depwarn=true, filename=\"none\")\n\nParse the expression string and return an expression (which could later be passed to eval for execution). start is the code unit index into str of the first character to start parsing at (as with all string indexing, these are not character indices). If greedy is true (default), parse will try to consume as much input as it can; otherwise, it will stop as soon as it has parsed a valid expression. Incomplete but otherwise syntactically valid expressions will return Expr(:incomplete, \"(error message)\"). If raise is true (default), syntax errors other than incomplete expressions will raise an error. If raise is false, parse will return an expression that will raise an error upon evaluation. If depwarn is false, deprecation warnings will be suppressed. The filename argument is used to display diagnostics when an error is raised.\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 1) # start of string\n(:((α, β) = (3, 5)), 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 1, greedy=false)\n(:((α, β)), 9)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 16) # end of string\n(nothing, 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 11) # index of 3\n(:((3, 5)), 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 11, greedy=false)\n(3, 13)\n\n\n\n\n\n"},{"title":"Base.Meta.parse","page":"Essentials","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString}","category":"method","text":"parse(str; raise=true, depwarn=true, filename=\"none\")\n\nParse the expression string greedily, returning a single expression. An error is thrown if there are additional characters after the first expression. If raise is true (default), syntax errors will raise an error; otherwise, parse will return an expression that will raise an error upon evaluation.  If depwarn is false, deprecation warnings will be suppressed. The filename argument is used to display diagnostics when an error is raised.\n\njulia> Meta.parse(\"x = 3\")\n:(x = 3)\n\njulia> Meta.parse(\"1.0.2\")\nERROR: ParseError:\n# Error @ none:1:1\n1.0.2\n└──┘ ── invalid numeric constant\n[...]\n\njulia> Meta.parse(\"1.0.2\"; raise = false)\n:($(Expr(:error, \"invalid numeric constant \"1.0.\"\")))\n\njulia> Meta.parse(\"x = \")\n:($(Expr(:incomplete, \"incomplete: premature end of input\")))\n\n\n\n\n\n"},{"title":"Base.Meta.ParseError","page":"Essentials","location":"base/base.html#Base.Meta.ParseError","category":"type","text":"ParseError(msg)\n\nThe expression passed to the parse function could not be interpreted as a valid Julia expression.\n\n\n\n\n\n"},{"title":"Core.QuoteNode","page":"Essentials","location":"base/base.html#Core.QuoteNode","category":"type","text":"QuoteNode\n\nA quoted piece of code, that does not support interpolation. See the manual section about QuoteNodes for details.\n\n\n\n\n\n"},{"title":"Base.macroexpand","page":"Essentials","location":"base/base.html#Base.macroexpand","category":"function","text":"macroexpand(m::Module, x; recursive=true, legacyscope=true)\n\nTake the expression x and return an equivalent expression with all macros removed (expanded) for executing in module m. The recursive keyword controls whether deeper levels of nested macros are also expanded. The legacyscope keyword controls whether legacy macroscope expansion is performed. This is demonstrated in the example below:\n\njulia> module M\n           macro m1()\n               42\n           end\n           macro m2()\n               :(@m1())\n           end\n       end\nM\n\njulia> macroexpand(M, :(@m2()), recursive=true)\n42\n\njulia> macroexpand(M, :(@m2()), recursive=false)\n:(#= REPL[1]:6 =# @m1)\n\n\n\n\n\n"},{"title":"Base.macroexpand!","page":"Essentials","location":"base/base.html#Base.macroexpand!","category":"function","text":"macroexpand!(m::Module, x; recursive=true, legacyscope=false)\n\nTake the expression x and return an equivalent expression with all macros removed (expanded) for executing in module m, modifying x in place without copying. The recursive keyword controls whether deeper levels of nested macros are also expanded. The legacyscope keyword controls whether legacy macroscope expansion is performed.\n\nThis function performs macro expansion without the initial copy step, making it more efficient when the original expression is no longer needed. By default, macroscope expansion is disabled for in-place expansion as it can be called separately if needed.\n\nwarning: Warning\nThis function modifies the input expression x in place. Use macroexpand if you need to preserve the original expression.\n\n\n\n\n\n"},{"title":"Base.@macroexpand","page":"Essentials","location":"base/base.html#Base.@macroexpand","category":"macro","text":"@macroexpand [mod,] ex\n\nReturn equivalent expression with all macros removed (expanded). If two arguments are provided, the first is the module to evaluate in.\n\nThere are differences between @macroexpand and macroexpand.\n\nWhile macroexpand takes a keyword argument recursive, @macroexpand is always recursive. For a non recursive macro version, see @macroexpand1.\nWhile macroexpand has an explicit module argument, @macroexpand always expands with respect to the module in which it is called.\n\nThis is best seen in the following example:\n\njulia> module M\n           macro m()\n               1\n           end\n           function f()\n               (@macroexpand(@m),\n                macroexpand(M, :(@m)),\n                macroexpand(parentmodule(M), :(@m))\n               )\n           end\n       end\nM\n\njulia> macro m()\n           2\n       end\n@m (macro with 1 method)\n\njulia> M.f()\n(1, 1, 2)\n\nWith @macroexpand the expression expands where @macroexpand appears in the code (module M in the example). With macroexpand the expression expands in the module given as the first argument.\n\ncompat: Julia 1.11\nThe two-argument form requires at least Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.@macroexpand1","page":"Essentials","location":"base/base.html#Base.@macroexpand1","category":"macro","text":"@macroexpand1 [mod,] ex\n\nNon recursive version of @macroexpand.\n\n\n\n\n\n"},{"title":"Base.code_lowered","page":"Essentials","location":"base/base.html#Base.code_lowered","category":"function","text":"code_lowered(f, types; generated=true, debuginfo=:default)\n\nReturn an array of the lowered forms (IR) for the methods matching the given generic function and type signature.\n\nIf generated is false, the returned CodeInfo instances will correspond to fallback implementations. An error is thrown if no fallback implementation exists. If generated is true, these CodeInfo instances will correspond to the method bodies yielded by expanding the generators.\n\nThe keyword debuginfo controls the amount of code metadata present in the output.\n\nNote that an error will be thrown if types are not concrete types when generated is true and any of the corresponding methods are an @generated method.\n\n\n\n\n\n"},{"title":"Base.code_typed","page":"Essentials","location":"base/base.html#Base.code_typed","category":"function","text":"code_typed(f, types; kw...)\n\nReturns an array of type-inferred lowered form (IR) for the methods matching the given generic function and type signature.\n\nKeyword Arguments\n\noptimize::Bool = true: optional, controls whether additional optimizations, such as inlining, are also applied.\ndebuginfo::Symbol = :default: optional, controls the amount of code metadata present in the output, possible options are :source or :none.\n\nInternal Keyword Arguments\n\nThis section should be considered internal, and is only for who understands Julia compiler internals.\n\nworld::UInt = Base.get_world_counter(): optional, controls the world age to use when looking up methods, use current world age if not specified.\ninterp::Core.Compiler.AbstractInterpreter = Core.Compiler.NativeInterpreter(world): optional, controls the abstract interpreter to use, use the native interpreter if not specified.\n\nExamples\n\nOne can put the argument types in a tuple to get the corresponding code_typed.\n\njulia> code_typed(+, (Float64, Float64))\n1-element Vector{Any}:\n CodeInfo(\n1 ─ %1 = Base.add_float(x, y)::Float64\n└──      return %1\n) => Float64\n\njulia> code_typed((typeof(-), Float64, Float64))\n1-element Vector{Any}:\n CodeInfo(\n1 ─ %1 = Base.sub_float(x, y)::Float64\n└──      return %1\n) => Float64\n\njulia> code_typed((Type{Int}, UInt8))\n1-element Vector{Any}:\n CodeInfo(\n1 ─ %1 = Core.zext_int(Core.Int64, x)::Int64\n└──      return %1\n) => Int64\n\njulia> code_typed((Returns{Int64},))\n1-element Vector{Any}:\n CodeInfo(\n1 ─ %1 =   builtin Base.getfield(obj, :value)::Int64\n└──      return %1\n) => Int64\n\n\n\n\n\n"},{"title":"Base.precompile","page":"Essentials","location":"base/base.html#Base.precompile","category":"function","text":"precompile(f, argtypes::Tuple{Vararg{Any}}, m::Method)\n\nPrecompile a specific method for the given argument types. This may be used to precompile a different method than the one that would ordinarily be chosen by dispatch, thus mimicking invoke.\n\n\n\n\n\nprecompile(f, argtypes::Tuple{Vararg{Any}})\n\nCompile the given function f for the argument tuple (of types) argtypes, but do not execute it.\n\n\n\n\n\n"},{"title":"Base.jit_total_bytes","page":"Essentials","location":"base/base.html#Base.jit_total_bytes","category":"function","text":"Base.jit_total_bytes()\n\nReturn the total amount (in bytes) allocated by the just-in-time compiler for e.g. native code and data.\n\n\n\n\n\n"},{"title":"Base.Meta.quot","page":"Essentials","location":"base/base.html#Base.Meta.quot","category":"function","text":"Meta.quot(ex)::Expr\n\nQuote expression ex to produce an expression with head quote. This can for instance be used to represent objects of type Expr in the AST. See also the manual section about QuoteNode.\n\nExamples\n\njulia> eval(Meta.quot(:x))\n:x\n\njulia> dump(Meta.quot(:x))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Symbol x\n\njulia> eval(Meta.quot(:(1+2)))\n:(1 + 2)\n\n\n\n\n\n"},{"title":"Base.isexpr","page":"Essentials","location":"base/base.html#Base.isexpr","category":"function","text":"Meta.isexpr(ex, head[, n])::Bool\n\nReturn true if ex is an Expr with the given type head and optionally that the argument list is of length n. head may be a Symbol or collection of Symbols. For example, to check that a macro was passed a function call expression, you might use isexpr(ex, :call).\n\nExamples\n\njulia> ex = :(f(x))\n:(f(x))\n\njulia> Meta.isexpr(ex, :block)\nfalse\n\njulia> Meta.isexpr(ex, :call)\ntrue\n\njulia> Meta.isexpr(ex, [:block, :call]) # multiple possible heads\ntrue\n\njulia> Meta.isexpr(ex, :call, 1)\nfalse\n\njulia> Meta.isexpr(ex, :call, 2)\ntrue\n\n\n\n\n\n"},{"title":"Base.Meta.isidentifier","page":"Essentials","location":"base/base.html#Base.Meta.isidentifier","category":"function","text":" isidentifier(s) -> Bool\n\nReturn whether the symbol or string s contains characters that are parsed as a valid ordinary identifier (not a binary/unary operator) in Julia code; see also Base.isoperator.\n\nInternally Julia allows any sequence of characters in a Symbol (except \\0s), and macros automatically use variable names containing # in order to avoid naming collision with the surrounding code. In order for the parser to recognize a variable, it uses a limited set of characters (greatly extended by Unicode). isidentifier() makes it possible to query the parser directly whether a symbol contains valid characters.\n\nExamples\n\njulia> Meta.isidentifier(:x), Meta.isidentifier(\"1x\")\n(true, false)\n\n\n\n\n\n"},{"title":"Base.Meta.isoperator","page":"Essentials","location":"base/base.html#Base.Meta.isoperator","category":"function","text":"isoperator(s::Symbol)\n\nReturn true if the symbol can be used as an operator, false otherwise.\n\nExamples\n\njulia> Meta.isoperator(:+), Meta.isoperator(:f)\n(true, false)\n\n\n\n\n\n"},{"title":"Base.Meta.isunaryoperator","page":"Essentials","location":"base/base.html#Base.Meta.isunaryoperator","category":"function","text":"isunaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a unary (prefix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isunaryoperator(:-), Meta.isunaryoperator(:√), Meta.isunaryoperator(:f)\n(true, true, false)\n\n\n\n\n\n"},{"title":"Base.Meta.isbinaryoperator","page":"Essentials","location":"base/base.html#Base.Meta.isbinaryoperator","category":"function","text":"isbinaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a binary (infix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isbinaryoperator(:-), Meta.isbinaryoperator(:√), Meta.isbinaryoperator(:f)\n(true, false, false)\n\n\n\n\n\n"},{"title":"Base.Meta.show_sexpr","page":"Essentials","location":"base/base.html#Base.Meta.show_sexpr","category":"function","text":"Meta.show_sexpr([io::IO,], ex)\n\nShow expression ex as a lisp style S-expression.\n\nExamples\n\njulia> Meta.show_sexpr(:(f(x, g(y,z))))\n(:call, :f, :x, (:call, :g, :y, :z))\n\n\n\n\n\n"},{"title":"Building Julia (Detailed)","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia-(Detailed)","category":"section","text":""},{"title":"Downloading the Julia source code","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Downloading-the-Julia-source-code","category":"section","text":"If you are behind a firewall, you may need to use the https protocol instead of the git protocol:\n\ngit config --global url.\"https://\".insteadOf git://\n\nBe sure to also configure your system to use the appropriate proxy settings, e.g. by setting the https_proxy and http_proxy variables."},{"title":"Building Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia","category":"section","text":"When compiled the first time, the build will automatically download pre-built external dependencies. If you prefer to build all the dependencies on your own, or are building on a system that cannot access the network during the build process, add the following in Make.user:\n\nUSE_BINARYBUILDER=0\n\nBuilding Julia requires 5GiB if building all dependencies and approximately 4GiB of virtual memory.\n\nTo perform a parallel build, use make -j N and supply the maximum number of concurrent processes. If the defaults in the build do not work for you, and you need to set specific make parameters, you can save them in Make.user, and place the file in the root of your Julia source. The build will automatically check for the existence of Make.user and use it if it exists.\n\nYou can create out-of-tree builds of Julia by specifying make O=<build-directory> configure on the command line. This will create a directory mirror, with all of the necessary Makefiles to build Julia, in the specified directory. These builds will share the source files in Julia and deps/srccache. Each out-of-tree build directory can have its own Make.user file to override the global Make.user file in the top-level folder. After modifying the Make.user file if necessary, build using: make -C <build-directory>.\n\nIf everything works correctly, there will be a symlink to the julia executable in the build directory which can be run as:\n\n./julia\n\nThe actual executable is in <build-directory>/usr/bin. After running this, you will see a Julia banner and an interactive prompt into which you can enter expressions for evaluation. (Errors related to libraries might be caused by old, incompatible libraries sitting around in your PATH. In this case, try moving the julia directory earlier in the PATH). Note that most of the instructions above apply to unix systems.\n\nTo run julia from anywhere you can:\n\nadd an alias (in bash: echo \"alias julia='<build-directory>/usr/bin/julia'\" >> ~/.bashrc && source ~/.bashrc), or\nadd a soft link to the julia executable in the <build-directory>/usr/bin directory to /usr/local/bin (or any suitable directory already in your path), or\nadd the julia directory to your executable path for this shell session (in bash: export PATH=\"$(pwd):$PATH\" ; in csh or tcsh:\n\nset path= ( $path $cwd ) ), or\n\nadd the julia directory to your executable path permanently (e.g. in .bash_profile), or\nwrite prefix=/path/to/install/folder into Make.user and then run make install. If there is a version of Julia already installed in this folder, you should delete it before running make install.\n\nSome of the options you can set to control the build of Julia are listed and documented at the beginning of the file Make.inc, but you should never edit it for this purpose, use Make.user instead.\n\nJulia's Makefiles define convenient automatic rules called print-<VARNAME> for printing the value of variables, replacing <VARNAME> with the name of the variable to print the value of. For example\n\n$ make print-JULIA_PRECOMPILE\nJULIA_PRECOMPILE=1\n\nThese rules are useful for debugging purposes.\n\nNow you should be able to run Julia like this:\n\njulia\n\nIf you are building a Julia package for distribution on Linux, macOS, or Windows, take a look at the detailed notes in distributing.md."},{"title":"Updating an existing source tree","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Updating-an-existing-source-tree","category":"section","text":"If you have previously downloaded julia using git clone, you can update the existing source tree using git pull rather than starting anew:\n\ncd julia\ngit pull && make\n\nAssuming that you had made no changes to the source tree that will conflict with upstream updates, these commands will trigger a build to update to the latest version."},{"title":"General troubleshooting","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#General-troubleshooting","category":"section","text":"Over time, the base library may accumulate enough changes such that the bootstrapping process in building the system image will fail. If this happens, the build may fail with an error like\n *** This error is usually fixed by running 'make clean'. If the error persists, try 'make cleanall' ***\nAs described, running make clean && make is usually sufficient. Occasionally, the stronger cleanup done by make cleanall is needed.\nNew versions of external dependencies may be introduced which may occasionally cause conflicts with existing builds of older versions.\na. Special make targets exist to help wipe the existing build of a    dependency. For example, make -C deps clean-llvm will clean out the    existing build of llvm so that llvm will be rebuilt from the    downloaded source distribution the next time make is called.    make -C deps distclean-llvm is a stronger wipe which will also delete    the downloaded source distribution, ensuring that a fresh copy of the    source distribution will be downloaded and that any new patches will be    applied the next time make is called.\nb. To delete existing binaries of julia and all its dependencies,    delete the ./usr directory in the source tree.\nIf you've updated macOS recently, be sure to run xcode-select --install to update the command line tools. Otherwise, you could run into errors for missing headers and libraries, such as ld: library not found for -lcrt1.10.6.o.\nIf you've moved the source directory, you might get errors such as  CMake Error: The current CMakeCache.txt directory ... is different than the directory ... where     CMakeCache.txt was created., in which case you may delete the offending dependency under deps\nIn extreme cases, you may wish to reset the source tree to a pristine state. The following git commands may be helpful:\n git reset --hard #Forcibly remove any changes to any files under version control\n git clean -x -f -d #Forcibly remove any file or directory not under version control\nTo avoid losing work, make sure you know what these commands do before you run them. git will not be able to undo these changes!"},{"title":"Platform-Specific Notes","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Platform-Specific-Notes","category":"section","text":"Notes for various operating systems:\n\nLinux\nmacOS\nWindows\nFreeBSD\n\nNotes for various architectures:\n\nARM\nRISC-V"},{"title":"Required Build Tools and External Libraries","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Required-Build-Tools-and-External-Libraries","category":"section","text":"Building Julia requires that the following software be installed:\n\nGNU make                — building dependencies.\ngcc & g++ (>= 7.1) or Clang (>= 5.0, >= 9.3 for Apple Clang) — compiling and linking C, C++.\nlibatomic          — provided by [gcc] and needed to support atomic operations.\npython (>=2.7)          — needed to build LLVM.\ngfortran                — compiling and linking Fortran libraries.\nperl                    — preprocessing of header files of libraries.\nwget, curl, or fetch (FreeBSD) — to automatically download external libraries.\nm4                      — needed to build GMP.\nawk                     — helper tool for Makefiles.\npatch                   — for modifying source code.\ncmake (>= 3.4.3)        — needed to build libgit2.\npkg-config              — needed to build libgit2 correctly, especially for proxy support.\npowershell (>= 3.0)     — necessary only on Windows.\nwhich                   — needed for checking build dependencies.\n\nOn Debian-based distributions (e.g. Ubuntu), you can easily install them with apt-get:\n\nsudo apt-get install build-essential libatomic1 python gfortran perl wget m4 cmake pkg-config curl\n\nOn Red Hat-based distributions (e.g. Fedora, CentOS), you can install them with yum:\n\nsudo dnf install gcc gcc-c++ gcc-gfortran python3 perl wget m4 cmake pkgconfig curl\n\nJulia uses the following external libraries, which are automatically downloaded (or in a few cases, included in the Julia source repository) and then compiled from source the first time you run make. The specific version numbers of these libraries that Julia uses are listed in deps/$(libname).version:\n\nLLVM (15.0 + patches) — compiler infrastructure (see note below).\nFemtoLisp            — packaged with Julia source, and used to implement the compiler front-end.\nlibuv  (custom fork) — portable, high-performance event-based I/O library.\nOpenLibm             — portable libm library containing elementary math functions.\nDSFMT                — fast Mersenne Twister pseudorandom number generator library.\nOpenBLAS             — fast, open, and maintained [basic linear algebra subprograms (BLAS)]\nLAPACK               — library of linear algebra routines for solving systems of simultaneous linear equations, least-squares solutions of linear systems of equations, eigenvalue problems, and singular value problems.\nMKL (optional)       – OpenBLAS and LAPACK may be replaced by Intel's MKL library.\nSuiteSparse          — library of linear algebra routines for sparse matrices.\nPCRE                 — Perl-compatible regular expressions library.\nGMP                  — GNU multiple precision arithmetic library, needed for BigInt support.\nMPFR                 — GNU multiple precision floating point library, needed for arbitrary precision floating point (BigFloat) support.\nlibgit2              — Git linkable library, used by Julia's package manager.\ncurl                 — libcurl provides download and proxy support.\nlibssh2              — library for SSH transport, used by libgit2 for packages with SSH remotes.\nOpenSSL              — library used for cryptography and transport layer security, used by libgit2 and libssh2.\nutf8proc             — a library for processing UTF-8 encoded Unicode strings.\nLLVM libunwind       — LLVM's fork of [libunwind], a library that determines the call-chain of a program.\nITTAPI               — Intel's Instrumentation and Tracing Technology and Just-In-Time API."},{"title":"Build dependencies","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Build-dependencies","category":"section","text":"If you already have one or more of these packages installed on your system, you can prevent Julia from compiling duplicates of these libraries by passing USE_SYSTEM_...=1 to make or adding the line to Make.user. The complete list of possible flags can be found in Make.inc.\n\nPlease be aware that this procedure is not officially supported, as it introduces additional variability into the installation and versioning of the dependencies, and is recommended only for system package maintainers. Unexpected compile errors may result, as the build system will do no further checking to ensure the proper packages are installed."},{"title":"LLVM","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#LLVM","category":"section","text":"The most complicated dependency is LLVM, for which we require additional patches from upstream (LLVM is not backward compatible).\n\nFor packaging Julia with LLVM, we recommend either:\n\nbundling a Julia-only LLVM library inside the Julia package, or\nadding the patches to the LLVM package of the distribution.\nA complete list of patches is available in on Github see the julia-release/18.x branch.\nThe remaining patches are all upstream bug fixes, and have been contributed into upstream LLVM.\n\nUsing an unpatched or different version of LLVM will result in errors and/or poor performance. You can build a different version of LLVM from a remote Git repository with the following options in the Make.user file:\n\n# Force source build of LLVM\nUSE_BINARYBUILDER_LLVM = 0\n# Use Git for fetching LLVM source code\n# this is either `1` to get all of them\nDEPS_GIT = 1\n# or a space-separated list of specific dependencies to download with git\nDEPS_GIT = llvm\n\n# Other useful options:\n#URL of the Git repository you want to obtain LLVM from:\n#  LLVM_GIT_URL = ...\n#Name of the alternate branch to clone from git\n#  LLVM_BRANCH = julia-16.0.6-0\n#SHA hash of the alternate commit to check out automatically\n#  LLVM_SHA1 = $(LLVM_BRANCH)\n#List of LLVM targets to build. It is strongly recommended to keep at least all the\n#default targets listed in `deps/llvm.mk`, even if you don't necessarily need all of them.\n#  LLVM_TARGETS = ...\n#Use ccache for faster recompilation in case you need to restart a build.\n#  USECCACHE = 1\n#  CMAKE_GENERATOR=Ninja\n#  LLVM_ASSERTIONS=1\n#  LLVM_DEBUG=Symbols\n\nThe various build phases are controlled by specific files:\n\ndeps/llvm.version : touch or change to checkout a new version, make get-llvm check-llvm\ndeps/srccache/llvm/source-extracted : result of make extract-llvm\ndeps/llvm/build_Release*/build-configured : result of make configure-llvm\ndeps/llvm/build_Release*/build-configured : result of make compile-llvm\nusr-staging/llvm/build_Release*.tgz : result of make stage-llvm (regenerate with make reinstall-llvm)\nusr/manifest/llvm : result of make install-llvm (regenerate with make uninstall-llvm)\nmake version-check-llvm : runs every time to warn the user if there are local modifications\n\nThough Julia can be built with newer LLVM versions, support for this should be regarded as experimental and not suitable for packaging."},{"title":"libuv","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#libuv","category":"section","text":"Julia uses a custom fork of libuv. It is a small dependency, and can be safely bundled in the same package as Julia, and will not conflict with the system library. Julia builds should not try to use the system libuv."},{"title":"BLAS and LAPACK","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#BLAS-and-LAPACK","category":"section","text":"As a high-performance numerical language, Julia should be linked to a multi-threaded BLAS and LAPACK, such as OpenBLAS or ATLAS, which will provide much better performance than the reference libblas implementations which may be default on some systems."},{"title":"Source distributions of releases","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Source-distributions-of-releases","category":"section","text":"Each pre-release and release of Julia has a \"full\" source distribution and a \"light\" source distribution.\n\nThe full source distribution contains the source code for Julia and all dependencies so that it can be built from source without an internet connection. The light source distribution does not include the source code of dependencies.\n\nFor example, julia-1.0.0.tar.gz is the light source distribution for the v1.0.0 release of Julia, while julia-1.0.0-full.tar.gz is the full source distribution."},{"title":"Building Julia from source with a Git checkout of a stdlib","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia-from-source-with-a-Git-checkout-of-a-stdlib","category":"section","text":"If you need to build Julia from source with a Git checkout of a stdlib, then use make DEPS_GIT=NAME_OF_STDLIB when building Julia.\n\nFor example, if you need to build Julia from source with a Git checkout of Pkg, then use make DEPS_GIT=Pkg when building Julia. The Pkg repo is in stdlib/Pkg, and created initially with a detached HEAD. If you're doing this from a pre-existing Julia repository, you may need to make clean beforehand.\n\nIf you need to build Julia from source with Git checkouts of more than one stdlib, then DEPS_GIT should be a space-separated list of the stdlib names. For example, if you need to build Julia from source with a Git checkout of Pkg, Tar, and Downloads, then use make DEPS_GIT='Pkg Tar Downloads' when building Julia."},{"title":"Building an \"assert build\" of Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-an-\"assert-build\"-of-Julia","category":"section","text":"An \"assert build\" of Julia is a build that was built with both FORCE_ASSERTIONS=1 and LLVM_ASSERTIONS=1. To build an assert build, define both of the following variables in your Make.user file:\n\nFORCE_ASSERTIONS=1\nLLVM_ASSERTIONS=1\n\nPlease note that assert builds of Julia will be slower than regular (non-assert) builds."},{"title":"Building a debug build of Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-a-debug-build-of-Julia","category":"section","text":"A full debug build of Julia can be built with make debug.  This builds a debug version of libjulia and uses it to bootstrap the compiler, before creating a system image with debug symbols enabled.  This can take more than 15 minutes.\n\nAlthough it may result in some differences, a debug build can be built much quicker by bootstrapping from a release build:\n\n$ make julia-src-release julia-sysbase-release\n$ make julia-sysimg-debug CROSS_BOOTSTRAP_JULIA=$PWD/usr/bin/julia CROSS_BOOTSTRAP_SYSBASE=$PWD/usr/lib/julia/sysbase.so"},{"title":"Building 32-bit Julia on a 64-bit machine","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-32-bit-Julia-on-a-64-bit-machine","category":"section","text":"Occasionally, bugs specific to 32-bit architectures may arise, and when this happens it is useful to be able to debug the problem on your local machine. Since most modern 64-bit systems support running programs built for 32-bit ones, if you don't have to recompile Julia from source (e.g. you mainly need to inspect the behavior of a 32-bit Julia without having to touch the C code), you can likely use a 32-bit build of Julia for your system that you can obtain from the official downloads page. However, if you do need to recompile Julia from source one option is to use a Docker container of a 32-bit system. At least for now, building a 32-bit version of Julia is relatively straightforward using ubuntu 32-bit docker images. In brief, after setting up docker here are the required steps:\n\n$ docker pull i386/ubuntu\n$ docker run --platform i386 -i -t i386/ubuntu /bin/bash\n\nAt this point you should be in a 32-bit machine console (note that uname reports the host architecture, so will still say 64-bit, but this will not affect the Julia build). You can add packages and compile code; when you exit, all the changes will be lost, so be sure to finish your analysis in a single session or set up a copy/pastable script you can use to set up your environment.\n\nFrom this point, you should\n\n# apt update\n\n(Note that sudo isn't installed, but neither is it necessary since you are running as root, so you can omit sudo from all commands.)\n\nThen add all the build dependencies, a console-based editor of your choice, git, and anything else you'll need (e.g., gdb, rr, etc). Pick a directory to work in and git clone Julia, check out the branch you wish to debug, and build Julia as usual."},{"title":"Update the version number of a dependency","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Update-the-version-number-of-a-dependency","category":"section","text":"There are two types of builds\n\nBuild everything (deps/ and src/) from source code.  (Add USE_BINARYBUILDER=0 to Make.user, see Building Julia)\nBuild from source (src/) with pre-compiled dependencies (default)\n\nWhen you want to update the version number of a dependency in deps/, you may want to use the following checklist:\n\n### Check list\n\nVersion numbers:\n- [ ] `deps/$(libname).version`: `LIBNAME_VER`, `LIBNAME_BRANCH`, `LIBNAME_SHA1` and `LIBNAME_JLL_VER`\n- [ ] `stdlib/$(LIBNAME_JLL_NAME)_jll/Project.toml`: `version`\n\nChecksum:\n- [ ] `deps/checksums/$(libname)`\n- [ ] `deps/checksums/$(LIBNAME_JLL_NAME)-*/`: `md5` and `sha512`\n\nPatches:\n- [ ] `deps/$(libname).mk`\n- [ ] `deps/patches/$(libname)-*.patch`\n\nNote:\n\nFor specific dependencies, some items in the checklist may not exist.\nFor checksum file, it may be a single file without a suffix, or a folder containing two files."},{"title":"Example: OpenLibm","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Example:-OpenLibm","category":"section","text":"Update Version numbers in deps/openlibm.version\nOPENLIBM_VER := 0.X.Y\nOPENLIBM_BRANCH = v0.X.Y\nOPENLIBM_SHA1 = new-sha1-hash\nUpdate Version number in stdlib/OpenLibm_jll/Project.toml\nversion = \"0.X.Y+0\"\nUpdate checksums in deps/checksums/openlibm\nmake -f contrib/refresh_checksums.mk openlibm\nCheck if the patch files deps/patches/openlibm-*.patch exist\nif patches don't exist, skip.\nif patches exist, check if they have been merged into the new version and need to be removed.   When deleting a patch, remember to modify the corresponding Makefile file (deps/openlibm.mk)."},{"title":"Parallel Computing","page":"Parallel Computing","location":"manual/parallel-computing.html#Parallel-Computing","category":"section","text":"Julia supports these four categories of concurrent and parallel programming:\n\nAsynchronous \"tasks\", or coroutines:\nJulia Tasks allow suspending and resuming computations  for I/O, event handling, producer-consumer processes, and similar patterns.  Tasks can synchronize through operations like wait and fetch, and  communicate via Channels. While strictly not parallel computing by themselves,  Julia lets you schedule Tasks on several threads.\nMulti-threading:\nJulia's multi-threading provides the ability to schedule Tasks  simultaneously on more than one thread or CPU core, sharing memory. This is usually the easiest way  to get parallelism on one's PC or on a single large multi-core server. Julia's multi-threading  is composable. When one multi-threaded function calls another multi-threaded function, Julia  will schedule all the threads globally on available resources, without oversubscribing.\nDistributed computing:\nDistributed computing runs multiple Julia processes with separate memory spaces. These can be on the same  computer or multiple computers. The Distributed standard library provides the capability for remote execution  of a Julia function. With this basic building block, it is possible to build many different kinds of  distributed computing abstractions. Packages like DistributedArrays.jl  are an example of such an abstraction. On the other hand, packages like MPI.jl and  Elemental.jl provide access to the existing MPI ecosystem of libraries.\nGPU computing:\nThe Julia GPU compiler provides the ability to run Julia code natively on GPUs. There  is a rich ecosystem of Julia packages that target GPUs. The JuliaGPU.org  website provides a list of capabilities, supported GPUs, related packages and documentation."},{"title":"CRC32c","page":"CRC32c","location":"stdlib/CRC32c.html#CRC32c","category":"section","text":"Standard library module for computing the CRC-32c checksum."},{"title":"CRC32c.crc32c","page":"CRC32c","location":"stdlib/CRC32c.html#CRC32c.crc32c","category":"function","text":"crc32c(data, crc::UInt32=0x00000000)\n\nCompute the CRC-32c checksum of the given data, which can be an Array{UInt8}, a contiguous subarray thereof, an AbstractVector{UInt8}, or a String. Optionally, you can pass a starting crc integer to be mixed in with the checksum. The crc parameter can be used to compute a checksum on data divided into chunks: performing crc32c(data2, crc32c(data1)) is equivalent to the checksum of [data1; data2]. (Technically, a little-endian checksum is computed.)\n\nThere is also a method crc32c(io, nb, crc) to checksum nb bytes from a stream io, or crc32c(io, crc) to checksum all the remaining bytes. Hence you can do open(crc32c, filename) to checksum an entire file, or crc32c(seekstart(buf)) to checksum an IOBuffer without calling take!.\n\nFor a String, note that the result is specific to the UTF-8 encoding (a different checksum would be obtained from a different Unicode encoding). To checksum an a::AbstractArray of some other bitstype without padding, you can do crc32c(vec(reinterpret(UInt8,a))), but note that the result may be endian-dependent.\n\n\n\n\n\n"},{"title":"CRC32c.crc32c","page":"CRC32c","location":"stdlib/CRC32c.html#CRC32c.crc32c-Tuple{IO, Integer, UInt32}","category":"method","text":"crc32c(io::IO, [nb::Integer,] crc::UInt32=0x00000000)\n\nRead up to nb bytes from io and return the CRC-32c checksum, optionally mixed with a starting crc integer.  If nb is not supplied, then io will be read until the end of the stream.\n\n\n\n\n\n"},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html#macOS","category":"section","text":"You need to have the current Xcode command line utilities installed: run xcode-select --install in the terminal. You will need to rerun this terminal command after each macOS update, otherwise you may run into errors involving missing libraries or headers.\n\nThe dependent libraries are now built with BinaryBuilder and will be automatically downloaded. This is the preferred way to build Julia source. In case you want to build them all on your own, you will need a 64-bit gfortran to compile Julia dependencies.\n\nbrew install gcc\n\nIf you have set LD_LIBRARY_PATH or DYLD_LIBRARY_PATH in your .bashrc or equivalent, Julia may be unable to find various libraries that come bundled with it. These environment variables need to be unset for Julia to work."},{"title":"Reporting and analyzing crashes (segfaults)","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Reporting-and-analyzing-crashes-(segfaults)","category":"section","text":"So you managed to break Julia. Congratulations!  Collected here are some general procedures you can undergo for common symptoms encountered when something goes awry. Including the information from these debugging steps can greatly help the maintainers when tracking down a segfault or trying to figure out why your script is running slower than expected.\n\nIf you've been directed to this page, find the symptom that best matches what you're experiencing and follow the instructions to generate the debugging information requested. Table of symptoms:\n\nSegfaults during bootstrap (sysimg.jl)\nSegfaults when running a script\nErrors during Julia startup\nOther generic segfaults or unreachables reached"},{"title":"Version/Environment info","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#dev-version-info","category":"section","text":"No matter the error, we will always need to know what version of Julia you are running. When Julia first starts up, a header is printed out with a version number and date. Please also include the output of versioninfo() (exported from the InteractiveUtils standard library) in any report you create:\n\nusing InteractiveUtils\nversioninfo()"},{"title":"Segfaults during bootstrap (sysimg.jl)","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Segfaults-during-bootstrap-(sysimg.jl)","category":"section","text":"Segfaults toward the end of the make process of building Julia are a common symptom of something going wrong while Julia is preparsing the corpus of code in the base/ folder. Many factors can contribute toward this process dying unexpectedly, however it is as often as not due to an error in the C-code portion of Julia, and as such must typically be debugged with a debug build inside of gdb. Explicitly:\n\nCreate a debug build of Julia:\n\n$ cd <julia_root>\n$ make debug\n\nNote that this process will likely fail with the same error as a normal make incantation, however this will create a debug executable that will offer gdb the debugging symbols needed to get accurate backtraces. Next, manually run the bootstrap process inside of gdb:\n\n$ cd base/\n$ gdb -x ../contrib/debug_bootstrap.gdb\n\nThis will start gdb, attempt to run the bootstrap process using the debug build of Julia, and print out a backtrace if (when) it segfaults. You may need to hit <enter> a few times to get the full backtrace. Create a gist with the backtrace, the version info, and any other pertinent information you can think of and open a new issue on Github with a link to the gist."},{"title":"Segfaults when running a script","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Segfaults-when-running-a-script","category":"section","text":"The procedure is very similar to Segfaults during bootstrap (sysimg.jl). Create a debug build of Julia, and run your script inside of a debugged Julia process:\n\n$ cd <julia_root>\n$ make debug\n$ gdb --args usr/bin/julia-debug <path_to_your_script>\n\nNote that gdb will sit there, waiting for instructions. Type r to run the process, and bt to generate a backtrace once it segfaults:\n\n(gdb) r\nStarting program: /home/sabae/src/julia/usr/bin/julia-debug ./test.jl\n...\n(gdb) bt\n\nCreate a gist with the backtrace, the version info, and any other pertinent information you can think of and open a new issue on Github with a link to the gist."},{"title":"Errors during Julia startup","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Errors-during-Julia-startup","category":"section","text":"Occasionally errors occur during Julia's startup process (especially when using binary distributions, as opposed to compiling from source) such as the following:\n\n$ julia\nexec: error -5\n\nThese errors typically indicate something is not getting loaded properly very early on in the bootup phase, and our best bet in determining what's going wrong is to use external tools to audit the disk activity of the julia process:\n\nOn Linux, use strace:\n$ strace julia\nOn OSX, use dtruss:\n$ dtruss -f julia\n\nCreate a gist with the strace/ dtruss output, the version info, and any other pertinent information and open a new issue on Github with a link to the gist."},{"title":"Other generic segfaults or unreachables reached","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Other-generic-segfaults-or-unreachables-reached","category":"section","text":"As mentioned elsewhere, julia has good integration with rr for generating traces; this includes, on Linux, the ability to automatically run julia under rr and share the trace after a crash. This can be immensely helpful when debugging such crashes and is strongly encouraged when reporting crash issues to the JuliaLang/julia repo. To run julia under rr automatically, do:\n\njulia --bug-report=rr\n\nTo generate the rr trace locally, but not share, you can do:\n\njulia --bug-report=rr-local\n\nNote that this is only works on Linux. The blog post on Time Travelling Bug Reporting has many more details."},{"title":"Glossary","page":"Reporting and analyzing crashes (segfaults)","location":"devdocs/backtraces.html#Glossary","category":"section","text":"A few terms have been used as shorthand in this guide:\n\n<julia_root> refers to the root directory of the Julia source tree; e.g. it should contain folders such as base, deps, src, test, etc."},{"title":"Running External Programs","page":"Running External Programs","location":"manual/running-external-programs.html#Running-External-Programs","category":"section","text":"Julia borrows backtick notation for commands from the shell, Perl, and Ruby. However, in Julia, writing\n\njulia> `echo hello`\n`echo hello`\n\ndiffers in several aspects from the behavior in various shells, Perl, or Ruby:\n\nInstead of immediately running the command, backticks create a Cmd object to represent the command. You can use this object to connect the command to others via pipes, run it, and read or write to it.\nWhen the command is run, Julia does not capture its output unless you specifically arrange for it to. Instead, the output of the command by default goes to stdout as it would using libc's system call.\nThe command is never run with a shell. Instead, Julia parses the command syntax directly, appropriately interpolating variables and splitting on words as the shell would, respecting shell quoting syntax. The command is run as julia's immediate child process, using fork and exec calls.\n\nnote: Note\nThe following assumes a Posix environment as on Linux or MacOS. On Windows, many similar commands, such as echo and dir, are not external programs and instead are built into the shell cmd.exe itself. One option to run these commands is to invoke cmd.exe, for example cmd /C echo hello. Alternatively Julia can be run inside a Posix environment such as Cygwin.\n\nHere's a simple example of running an external program:\n\njulia> mycommand = `echo hello`\n`echo hello`\n\njulia> typeof(mycommand)\nCmd\n\njulia> run(mycommand);\nhello\n\nThe hello is the output of the echo command, sent to stdout. If the external command fails to run successfully, the run method throws an ProcessFailedException.\n\nIf you want to read the output of the external command, read or readchomp can be used instead:\n\njulia> read(`echo hello`, String)\n\"hello\\n\"\n\njulia> readchomp(`echo hello`)\n\"hello\"\n\nMore generally, you can use open to read from or write to an external command.\n\njulia> open(`less`, \"w\", stdout) do io\n           for i = 1:3\n               println(io, i)\n           end\n       end\n1\n2\n3\n\nThe program name and the individual arguments in a command can be accessed and iterated over as if the command were an array of strings:\n\njulia> collect(`echo \"foo bar\"`)\n2-element Vector{String}:\n \"echo\"\n \"foo bar\"\n\njulia> `echo \"foo bar\"`[2]\n\"foo bar\"\n\nYou can also pass a IOBuffer, and later read from it:\n\njulia> io = PipeBuffer(); # PipeBuffer is a type of IOBuffer\n\njulia> run(`echo world`, devnull, io, stderr);\n\njulia> readlines(io)\n1-element Vector{String}:\n \"world\""},{"title":"Interpolation","page":"Running External Programs","location":"manual/running-external-programs.html#command-interpolation","category":"section","text":"Suppose you want to do something a bit more complicated and use the name of a file in the variable file as an argument to a command. You can use $ for interpolation much as you would in a string literal (see Strings):\n\njulia> file = \"/etc/passwd\"\n\"/etc/passwd\"\n\njulia> `sort $file`\n`sort /etc/passwd`\n\nA common pitfall when running external programs via a shell is that if a file name contains characters that are special to the shell, they may cause undesirable behavior. Suppose, for example, rather than /etc/passwd, we wanted to sort the contents of the file /Volumes/External HD/data.csv. Let's try it:\n\njulia> file = \"/Volumes/External HD/data.csv\"\n\"/Volumes/External HD/data.csv\"\n\njulia> `sort $file`\n`sort '/Volumes/External HD/data.csv'`\n\nHow did the file name get quoted? Julia knows that file is meant to be interpolated as a single argument, so it quotes the word for you. Actually, that is not quite accurate: the value of file is never interpreted by a shell, so there's no need for actual quoting; the quotes are inserted only for presentation to the user. This will even work if you interpolate a value as part of a shell word:\n\njulia> path = \"/Volumes/External HD\"\n\"/Volumes/External HD\"\n\njulia> name = \"data\"\n\"data\"\n\njulia> ext = \"csv\"\n\"csv\"\n\njulia> `sort $path/$name.$ext`\n`sort '/Volumes/External HD/data.csv'`\n\nAs you can see, the space in the path variable is appropriately escaped. But what if you want to interpolate multiple words? In that case, just use an array (or any other iterable container):\n\njulia> files = [\"/etc/passwd\",\"/Volumes/External HD/data.csv\"]\n2-element Vector{String}:\n \"/etc/passwd\"\n \"/Volumes/External HD/data.csv\"\n\njulia> `grep foo $files`\n`grep foo /etc/passwd '/Volumes/External HD/data.csv'`\n\nIf you interpolate an array as part of a shell word, Julia emulates the shell's {a,b,c} argument generation:\n\njulia> names = [\"foo\",\"bar\",\"baz\"]\n3-element Vector{String}:\n \"foo\"\n \"bar\"\n \"baz\"\n\njulia> `grep xylophone $names.txt`\n`grep xylophone foo.txt bar.txt baz.txt`\n\nMoreover, if you interpolate multiple arrays into the same word, the shell's Cartesian product generation behavior is emulated:\n\njulia> names = [\"foo\",\"bar\",\"baz\"]\n3-element Vector{String}:\n \"foo\"\n \"bar\"\n \"baz\"\n\njulia> exts = [\"aux\",\"log\"]\n2-element Vector{String}:\n \"aux\"\n \"log\"\n\njulia> `rm -f $names.$exts`\n`rm -f foo.aux foo.log bar.aux bar.log baz.aux baz.log`\n\nSince you can interpolate literal arrays, you can use this generative functionality without needing to create temporary array objects first:\n\njulia> `rm -rf $[\"foo\",\"bar\",\"baz\",\"qux\"].$[\"aux\",\"log\",\"pdf\"]`\n`rm -rf foo.aux foo.log foo.pdf bar.aux bar.log bar.pdf baz.aux baz.log baz.pdf qux.aux qux.log qux.pdf`"},{"title":"Quoting","page":"Running External Programs","location":"manual/running-external-programs.html#Quoting","category":"section","text":"Inevitably, one wants to write commands that aren't quite so simple, and it becomes necessary to use quotes. Here's a simple example of a Perl one-liner at a shell prompt:\n\nsh$ perl -le '$|=1; for (0..3) { print }'\n0\n1\n2\n3\n\nThe Perl expression needs to be in single quotes for two reasons: so that spaces don't break the expression into multiple shell words, and so that uses of Perl variables like $| (yes, that's the name of a variable in Perl), don't cause interpolation. In other instances, you may want to use double quotes so that interpolation does occur:\n\nsh$ first=\"A\"\nsh$ second=\"B\"\nsh$ perl -le '$|=1; print for @ARGV' \"1: $first\" \"2: $second\"\n1: A\n2: B\n\nIn general, the Julia backtick syntax is carefully designed so that you can just cut-and-paste shell commands as is into backticks and they will work: the escaping, quoting, and interpolation behaviors are the same as the shell's. The only difference is that the interpolation is integrated and aware of Julia's notion of what is a single string value, and what is a container for multiple values. Let's try the above two examples in Julia:\n\njulia> A = `perl -le '$|=1; for (0..3) { print }'`\n`perl -le '$|=1; for (0..3) { print }'`\n\njulia> run(A);\n0\n1\n2\n3\n\njulia> first = \"A\"; second = \"B\";\n\njulia> B = `perl -le 'print for @ARGV' \"1: $first\" \"2: $second\"`\n`perl -le 'print for @ARGV' '1: A' '2: B'`\n\njulia> run(B);\n1: A\n2: B\n\nThe results are identical, and Julia's interpolation behavior mimics the shell's with some improvements due to the fact that Julia supports first-class iterable objects while most shells use strings split on spaces for this, which introduces ambiguities. When trying to port shell commands to Julia, try cut and pasting first. Since Julia shows commands to you before running them, you can easily and safely just examine its interpretation without doing any damage."},{"title":"Pipelines","page":"Running External Programs","location":"manual/running-external-programs.html#Pipelines","category":"section","text":"Shell metacharacters, such as |, &, and >, need to be quoted (or escaped) inside of Julia's backticks:\n\njulia> run(`echo hello '|' sort`);\nhello | sort\n\njulia> run(`echo hello \\| sort`);\nhello | sort\n\nThis expression invokes the echo command with three words as arguments: hello, |, and sort. The result is that a single line is printed: hello | sort. How, then, does one construct a pipeline? Instead of using '|' inside of backticks, one uses pipeline:\n\njulia> run(pipeline(`echo hello`, `sort`));\nhello\n\nThis pipes the output of the echo command to the sort command. Of course, this isn't terribly interesting since there's only one line to sort, but we can certainly do much more interesting things:\n\njulia> run(pipeline(`cut -d: -f3 /etc/passwd`, `sort -n`, `tail -n5`))\n210\n211\n212\n213\n214\n\nThis prints the highest five user IDs on a UNIX system. The cut, sort and tail commands are all spawned as immediate children of the current julia process, with no intervening shell process. Julia itself does the work to setup pipes and connect file descriptors that is normally done by the shell. Since Julia does this itself, it retains better control and can do some things that shells cannot.\n\nJulia can run multiple commands in parallel:\n\njulia> run(`echo hello` & `echo world`);\nworld\nhello\n\nThe order of the output here is non-deterministic because the two echo processes are started nearly simultaneously, and race to make the first write to the stdout descriptor they share with each other and the julia parent process. Julia lets you pipe the output from both of these processes to another program:\n\njulia> run(pipeline(`echo world` & `echo hello`, `sort`));\nhello\nworld\n\nIn terms of UNIX plumbing, what's happening here is that a single UNIX pipe object is created and written to by both echo processes, and the other end of the pipe is read from by the sort command.\n\nIO redirection can be accomplished by passing keyword arguments stdin, stdout, and stderr to the pipeline function:\n\npipeline(`do_work`, stdout=pipeline(`sort`, \"out.txt\"), stderr=\"errs.txt\")"},{"title":"Avoiding Deadlock in Pipelines","page":"Running External Programs","location":"manual/running-external-programs.html#Avoiding-Deadlock-in-Pipelines","category":"section","text":"When reading and writing to both ends of a pipeline from a single process, it is important to avoid forcing the kernel to buffer all of the data.\n\nFor example, when reading all of the output from a command, call read(out, String), not wait(process), since the former will actively consume all of the data written by the process, whereas the latter will attempt to store the data in the kernel's buffers while waiting for a reader to be connected.\n\nAnother common solution is to separate the reader and writer of the pipeline into separate Tasks:\n\nwriter = Threads.@spawn write(process, \"data\")\nreader = Threads.@spawn do_compute(read(process, String))\nwait(writer)\nfetch(reader)\n\n(commonly also, reader is not a separate task, since we immediately fetch it anyways)."},{"title":"Complex Example","page":"Running External Programs","location":"manual/running-external-programs.html#Complex-Example","category":"section","text":"The combination of a high-level programming language, a first-class command abstraction, and automatic setup of pipes between processes is a powerful one. To give some sense of the complex pipelines that can be created easily, here are some more sophisticated examples, with apologies for the excessive use of Perl one-liners:\n\njulia> prefixer(prefix, sleep) = `perl -nle '$|=1; print \"'$prefix' \", $_; sleep '$sleep';'`;\n\njulia> run(pipeline(`perl -le '$|=1; for(0..5){ print; sleep 1 }'`, prefixer(\"A\",2) & prefixer(\"B\",2)));\nB 0\nA 1\nB 2\nA 3\nB 4\nA 5\n\nThis is a classic example of a single producer feeding two concurrent consumers: one perl process generates lines with the numbers 0 through 5 on them, while two parallel processes consume that output, one prefixing lines with the letter \"A\", the other with the letter \"B\". Which consumer gets the first line is non-deterministic, but once that race has been won, the lines are consumed alternately by one process and then the other. (Setting $|=1 in Perl causes each print statement to flush the stdout handle, which is necessary for this example to work. Otherwise all the output is buffered and printed to the pipe at once, to be read by just one consumer process.)\n\nHere is an even more complex multi-stage producer-consumer example:\n\njulia> run(pipeline(`perl -le '$|=1; for(0..5){ print; sleep 1 }'`,\n           prefixer(\"X\",3) & prefixer(\"Y\",3) & prefixer(\"Z\",3),\n           prefixer(\"A\",2) & prefixer(\"B\",2)));\nA X 0\nB Y 1\nA Z 2\nB X 3\nA Y 4\nB Z 5\n\nThis example is similar to the previous one, except there are two stages of consumers, and the stages have different latency so they use a different number of parallel workers, to maintain saturated throughput.\n\nWe strongly encourage you to try all these examples to see how they work."},{"title":"Cmd Objects","page":"Running External Programs","location":"manual/running-external-programs.html#Cmd-Objects","category":"section","text":"The backtick syntax create an object of type Cmd. Such object may also be constructed directly from an existing Cmd or list of arguments:\n\nrun(Cmd(`pwd`, dir=\"..\"))\nrun(Cmd([\"pwd\"], detach=true, ignorestatus=true))\n\nThis allows you to specify several aspects of the Cmd's execution environment via keyword arguments. For example, the dir keyword provides control over the Cmd's working directory:\n\njulia> run(Cmd(`pwd`, dir=\"/\"));\n/\n\nAnd the env keyword allows you to set execution environment variables:\n\njulia> run(Cmd(`sh -c \"echo foo \\$HOWLONG\"`, env=(\"HOWLONG\" => \"ever!\",)));\nfoo ever!\n\nSee Cmd for additional keyword arguments. The setenv and addenv commands provide another means for replacing or adding to the Cmd execution environment variables, respectively:\n\njulia> run(setenv(`sh -c \"echo foo \\$HOWLONG\"`, (\"HOWLONG\" => \"ever!\",)));\nfoo ever!\n\njulia> run(addenv(`sh -c \"echo foo \\$HOWLONG\"`, \"HOWLONG\" => \"ever!\"));\nfoo ever!"},{"title":"Custom LLVM Passes","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#Custom-LLVM-Passes","category":"section","text":"Julia has a number of custom LLVM passes. Broadly, they can be classified into passes that are required to be run to maintain Julia semantics, and passes that take advantage of Julia semantics to optimize LLVM IR."},{"title":"Semantic Passes","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#Semantic-Passes","category":"section","text":"These passes are used to transform LLVM IR into code that is legal to be run on a CPU. Their main purpose is to enable simpler IR to be emitted by codegen, which then enables other LLVM passes to optimize common patterns."},{"title":"CPUFeatures","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#CPUFeatures","category":"section","text":"Filename: llvm-cpufeatures.cpp\nClass Name: CPUFeaturesPass\nOpt Name: module(CPUFeatures)\n\nThis pass lowers the julia.cpu.have_fma.(f32|f64) intrinsic to either true or false, depending on the target architecture and target features present on the function. This intrinsic is often used to determine if using algorithms dependent on fast fused multiply-add operations is better than using standard algorithms not dependent on such instructions."},{"title":"DemoteFloat16","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#DemoteFloat16","category":"section","text":"Filename: llvm-demote-float16.cpp\nClassName: DemoteFloat16Pass\nOpt Name function(DemoteFloat16)\n\nThis pass replaces float16 operations with float32 operations on architectures that do not natively support float16 operations. This is done by inserting fpext and fptrunc instructions around any float16 operation. On architectures that do support native float16 operations, this pass is a no-op."},{"title":"LateGCLowering","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#LateGCLowering","category":"section","text":"Filename: llvm-late-gc-lowering.cpp\nClass Name: LateLowerGCPass\nOpt Name: function(LateLowerGCFrame)\n\nThis pass performs most of the GC rooting work required to track pointers between GC safepoints. It also lowers several intrinsics to their corresponding instruction translation, and is permitted to violate the non-integral invariants previously established (pointer_from_objref is lowered to a ptrtoint instruction here). This pass typically occupies the most time out of all the custom Julia passes, due to its dataflow algorithm to minimize the number of objects live at any safepoint."},{"title":"FinalGCLowering","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#FinalGCLowering","category":"section","text":"Filename: llvm-final-gc-lowering.cpp\nClass Name: FinalLowerGCPass\nOpt Name: module(FinalLowerGC)\n\nThis pass lowers a few last intrinsics to their final form targeting functions in the libjulia library. Separating this from LateGCLowering enables other backends (GPU compilation) to supply their own custom lowerings for these intrinsics, enabling the Julia pipeline to be used on those backends as well."},{"title":"RemoveNI","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#RemoveNI","category":"section","text":"Filename: llvm-remove-ni.cpp\nClass Name: RemoveNIPass\nOpt Name: module(RemoveNI)\n\nThis pass removes the non-integral address spaces from the module's datalayout string. This enables the backend to lower Julia's custom address spaces directly to machine code, without a costly rewrite of every pointer operation to address space 0."},{"title":"SIMDLoop","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#SIMDLoop","category":"section","text":"Filename: llvm-simdloop.cpp\nClass Name: LowerSIMDLoopPass\nOpt Name: loop(LowerSIMDLoop)\n\nThis pass acts as the main driver of the @simd annotation. Codegen inserts a !llvm.loopid marker at the back branch of a loop, which this pass uses to identify loops that were originally marked with @simd. Then, this pass looks for a chain of floating point operations that form a reduce and adds the contract and reassoc fast math flags to allow reassociation (and thus vectorization). This pass does not preserve either loop information nor inference correctness, so it may violate Julia semantics in surprising ways. If the loop was annotated with ivdep as well, then the pass marks the loop as having no loop-carried dependencies (the resulting behavior is undefined if the user annotation was incorrect or gets applied to the wrong loop)."},{"title":"LowerPTLS","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#LowerPTLS","category":"section","text":"Filename: llvm-ptls.cpp\nClass Name: LowerPTLSPass\nOpt Name: module(LowerPTLSPass)\n\nThis pass lowers thread-local Julia intrinsics to assembly instructions. Julia relies on thread-local storage for garbage collection and multithreading task scheduling. When compiling code for system images and package images, this pass replaces calls to intrinsics with loads from global variables that are initialized at load time.\n\nIf codegen produces a function with a swiftself argument and calling convention, this pass assumes the swiftself argument is the pgcstack and will replace the intrinsics with that argument. Doing so provides speedups on architectures that have slow thread local storage accesses."},{"title":"RemoveAddrspaces","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#RemoveAddrspaces","category":"section","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveAddrspacesPass\nOpt Name: module(RemoveAddrspaces)\n\nThis pass renames pointers in one address space to another address space. This is used to remove Julia-specific address spaces from LLVM IR."},{"title":"RemoveJuliaAddrspaces","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#RemoveJuliaAddrspaces","category":"section","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveJuliaAddrspacesPass\nOpt Name: module(RemoveJuliaAddrspaces)\n\nThis pass removes Julia-specific address spaces from LLVM IR. It is mostly used for displaying LLVM IR in a less cluttered format. Internally, it is implemented off the RemoveAddrspaces pass."},{"title":"Multiversioning","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#Multiversioning","category":"section","text":"Filename: llvm-multiversioning.cpp\nClass Name: MultiVersioningPass\nOpt Name: module(JuliaMultiVersioning)\n\nThis pass performs modifications to a module to create functions that are optimized for running on different architectures (see sysimg.md and pkgimg.md for more details). Implementation-wise, it clones functions and applies different target-specific attributes to them to allow the optimizer to use advanced features such as vectorization and instruction scheduling for that platform. It also creates some infrastructure to enable the Julia image loader to select the appropriate version of the function to call based on the architecture the loader is running on. The target-specific attributes are controlled by the julia.mv.specs module flag, which during compilation is derived from the JULIA_CPU_TARGET environment variable. The pass must also be enabled by providing a julia.mv.enable module flag with a value of 1.\n\nwarning: Warning\nUse of llvmcall with multiversioning is dangerous. llvmcall enables access to features not typically exposed by the Julia APIs, and are therefore usually not available on all architectures. If multiversioning is enabled and code generation is requested for a target architecture that does not support the feature required by an llvmcall expression, LLVM will probably error out, likely with an abort and the message LLVM ERROR: Do not know how to split the result of this operator!."},{"title":"GCInvariantVerifier","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#GCInvariantVerifier","category":"section","text":"Filename: llvm-gc-invariant-verifier.cpp\nClass Name: GCInvariantVerifierPass\nOpt Name: module(GCInvariantVerifier)\n\nThis pass is used to verify Julia's invariants about LLVM IR. This includes things such as the nonexistence of ptrtoint in Julia's non-integral address spaces [nislides] and the existence of only blessed addrspacecast instructions (Tracked -> Derived, 0 -> Tracked, etc). It performs no transformations on IR.\n\n[nislides]: https://llvm.org/devmtg/2015-02/slides/chisnall-pointers-not-int.pdf"},{"title":"Optimization Passes","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#Optimization-Passes","category":"section","text":"These passes are used to perform transformations on LLVM IR that LLVM will not perform itself, e.g. fast math flag propagation, escape analysis, and optimizations on Julia-specific internal functions. They use knowledge about Julia's semantics to perform these optimizations."},{"title":"AllocOpt","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#AllocOpt","category":"section","text":"Filename: llvm-alloc-opt.cpp\nClass Name: AllocOptPass\nOpt Name: function(AllocOpt)\n\nJulia does not have the concept of a program stack as a place to allocate mutable objects. However, allocating objects on the stack reduces GC pressure and is critical for GPU compilation. Thus, AllocOpt performs heap to stack conversion of objects that it can prove do not escape the current function. It also performs a number of other optimizations on allocations, such as removing allocations that are never used, optimizing typeof calls to freshly allocated objects, and removing stores to allocations that are immediately overwritten. The escape analysis implementation is located in llvm-alloc-helpers.cpp. Currently, this pass does not use information from EscapeAnalysis.jl, though that may change in the future."},{"title":"PropagateJuliaAddrspaces","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#PropagateJuliaAddrspaces","category":"section","text":"Filename: llvm-propagate-addrspaces.cpp\nClass Name: PropagateJuliaAddrspacesPass\nOpt Name: function(PropagateJuliaAddrspaces)\n\nThis pass is used to propagate Julia-specific address spaces through operations on pointers. LLVM is not allowed to introduce or remove addrspacecast instructions by optimizations, so this pass acts to eliminate redundant addrspace casts by replacing operations with their equivalent in a Julia address space. For more information on Julia's address spaces, see (TODO link to llvm.md)."},{"title":"JuliaLICM","page":"Custom LLVM Passes","location":"devdocs/llvm-passes.html#JuliaLICM","category":"section","text":"Filename: llvm-julia-licm.cpp\nClass Name: JuliaLICMPass\nOpt Name: loop(JuliaLICM)\n\nThis pass is used to hoist Julia-specific intrinsics out of loops. Specifically, it performs the following transformations:\n\nHoist gc_preserve_begin and sink gc_preserve_end out of loops when the preserved objects are loop-invariant.\nSince objects preserved within a loop are likely preserved for the duration of the loop, this transformation can reduce the number of gc_preserve_begin/gc_preserve_end pairs in the IR. This makes it easier for the LateLowerGCPass to identify where particular objects are preserved.\nHoist write barriers with invariant objects\nHere we assume that there are only two generations that an object can be a part of. Given that, a write barrier needs to only execute once for any pair of the same object. Thus, we can hoist write barriers out of loops when the object being written to is loop-invariant.\nHoist allocations out of loops when they do not escape the loop\nWe use a very conservative definition of escape here, the same as the one used in AllocOptPass. This transformation can reduce the number of allocations in the IR, even when an allocation escapes the function altogether.\n\nnote: Note\nThis pass is required to preserve LLVM's MemorySSA (Short Video, Longer Video) and ScalarEvolution (Newer Slides Older Slides) analyses."},{"title":"Modules","page":"Modules","location":"manual/modules.html#modules","category":"section","text":"Modules in Julia help organize code into coherent units. They are delimited syntactically inside module NameOfModule ... end, and have the following features:\n\nModules are separate namespaces, each introducing a new global scope. This is useful, because it allows the same name to be used for different functions or global variables without conflict, as long as they are in separate modules.\nModules have facilities for detailed namespace management: each defines a set of names it exports and marks as public, and can import names from other modules with using and import (we explain these below).\nModules can be precompiled for faster loading, and may contain code for runtime initialization.\n\nTypically, in larger Julia packages you will see module code organized into files, eg\n\nmodule SomeModule\n\n# export, public, using, import statements are usually here; we discuss these below\n\ninclude(\"file1.jl\")\ninclude(\"file2.jl\")\n\nend\n\nFiles and file names are mostly unrelated to modules; modules are associated only with module expressions. One can have multiple files per module, and multiple modules per file. include behaves as if the contents of the source file were evaluated in the global scope of the including module. In this chapter, we use short and simplified examples, so we won't use include.\n\nThe recommended style is not to indent the body of the module, since that would typically lead to whole files being indented. Also, it is common to use UpperCamelCase for module names (just like types), and use the plural form if applicable, especially if the module contains a similarly named identifier, to avoid name clashes. For example,\n\nmodule FastThings\n\nstruct FastThing\n    ...\nend\n\nend"},{"title":"Namespace management","page":"Modules","location":"manual/modules.html#namespace-management","category":"section","text":"Namespace management refers to the facilities the language offers for making names in a module available in other modules. We discuss the related concepts and functionality below in detail."},{"title":"Qualified names","page":"Modules","location":"manual/modules.html#Qualified-names","category":"section","text":"Names for functions, variables and types in the global scope like sin, ARGS, and UnitRange always belong to a module, called the parent module, which can be found interactively with parentmodule, for example\n\njulia> parentmodule(UnitRange)\nBase\n\nOne can also refer to these names outside their parent module by prefixing them with their module, eg Base.UnitRange. This is called a qualified name. The parent module may be accessible using a chain of submodules like Base.Math.sin, where Base.Math is called the module path. Due to syntactic ambiguities, qualifying a name that contains only symbols, such as an operator, requires inserting a colon, e.g. Base.:+. A small number of operators additionally require parentheses, e.g. Base.:(==).\n\nIf a name is qualified, then it is always accessible, and in case of a function, it can also have methods added to it by using the qualified name as the function name.\n\nWithin a module, a variable name can be “reserved” without assigning to it by declaring it as global x. This prevents name conflicts for globals initialized after load time. The syntax M.x = y does not work to assign a global in another module; global assignment is always module-local."},{"title":"Export lists","page":"Modules","location":"manual/modules.html#Export-lists","category":"section","text":"Names (referring to functions, types, global variables, and constants) can be added to the export list of a module with export: these are the symbols that are imported when using the module. Typically, they are at or near the top of the module definition so that readers of the source code can find them easily, as in\n\njulia> module NiceStuff\n       export nice, DOG\n       struct Dog end      # singleton type, not exported\n       const DOG = Dog()   # named instance, exported\n       nice(x) = \"nice $x\" # function, exported\n       end;\n\n\nbut this is just a style suggestion — a module can have multiple export statements in arbitrary locations.\n\nIt is common to export names which form part of the API (application programming interface). In the above code, the export list suggests that users should use nice and DOG. However, since qualified names always make identifiers accessible, this is just an option for organizing APIs: unlike other languages, Julia has no facilities for truly hiding module internals.\n\nAlso, some modules don't export names at all. This is usually done if they use common words, such as derivative, in their API, which could easily clash with the export lists of other modules. We will see how to manage name clashes below.\n\nTo mark a name as public without exporting it into the namespace of folks who call using NiceStuff, one can use public instead of export. This marks the public name(s) as part of the public API, but does not have any namespace implications. The public keyword is only available in Julia 1.11 and above. To maintain compatibility with Julia 1.10 and below, use the @compat macro from the Compat package, or the version-aware construct\n\nVERSION >= v\"1.11.0-DEV.469\" && eval(Meta.parse(\"public a, b, c\"))\n\nexport is a keyword wherever it occurs whereas the public keyword is currently limited to the syntactic top level within a file or module. This limitation exists for compatibility reasons, as public was introduced as a new keyword in Julia 1.11 while export has existed since Julia 1.0. However, this restriction on public may be lifted in future releases, so do not use public as an identifier."},{"title":"Standalone using and import","page":"Modules","location":"manual/modules.html#Standalone-using-and-import","category":"section","text":"For interactive use, the most common way of loading a module is using ModuleName. This loads the code associated with ModuleName, and brings\n\nthe module name\nand the elements of the export list into the surrounding global namespace.\n\nTechnically, the statement using ModuleName means that a module called ModuleName will be available for resolving names as needed. When a global variable is encountered that has no definition in the current module, the system will search for it among variables exported by ModuleName and use it if it is found there. This means that all uses of that global within the current module will resolve to the definition of that variable in ModuleName.\n\nTo load a module from a package, the statement using ModuleName can be used. To load a module from a locally defined module, a dot needs to be added before the module name like using .ModuleName.\n\nTo continue with our example,\n\njulia> using .NiceStuff\n\nwould load the above code, making NiceStuff (the module name), DOG and nice available. Dog is not on the export list, but it can be accessed if the name is qualified with the module path (which here is just the module name) as NiceStuff.Dog.\n\nImportantly, using ModuleName is the only form for which export lists matter at all.\n\nIn contrast,\n\njulia> import .NiceStuff\n\nbrings only the module name into scope. Users would need to use NiceStuff.DOG, NiceStuff.Dog, and NiceStuff.nice to access its contents. As we will see in the next section import .NiceStuff is equivalent to using .NiceStuff: NiceStuff. Usually, import ModuleName or using ModuleName: ModuleName is used in contexts when the user wants to keep the namespace clean.\n\nYou can combine multiple using and import statements of the same kind in a comma-separated expression, e.g.\n\njulia> using LinearAlgebra, Random"},{"title":"using and import with specific identifiers, and adding methods","page":"Modules","location":"manual/modules.html#using-and-import-with-specific-identifiers,-and-adding-methods","category":"section","text":"When using ModuleName: or import ModuleName: is followed by a comma-separated list of names, the module is loaded, but only those specific names are brought into the namespace by the statement. For example,\n\njulia> using .NiceStuff: nice, DOG\n\nwill import the names nice and DOG.\n\nImportantly, the module name NiceStuff will not be in the namespace. If you want to make it accessible, you have to list it explicitly, as\n\njulia> using .NiceStuff: nice, DOG, NiceStuff\n\nWhen two or more packages/modules export a name and that name does not refer to the same thing in each of the packages, and the packages are loaded via using without an explicit list of names, it is an error to reference that name without qualification. It is thus recommended that code intended to be forward-compatible with future versions of its dependencies and of Julia, e.g., code in released packages, list the names it uses from each loaded package, e.g., using Foo: Foo, f rather than using Foo.\n\nJulia has two forms for seemingly the same thing because only import ModuleName: f allows adding methods to f without a module path. That is to say, the following example will give an error:\n\njulia> using .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nERROR: invalid method definition in Main: function NiceStuff.nice must be explicitly imported to be extended\nStacktrace:\n [1] top-level scope\n   @ none:1\n\nThis error prevents accidentally adding methods to functions in other modules that you only intended to use.\n\nThere are two ways to deal with this. You can always qualify function names with a module path:\n\njulia> using .NiceStuff\n\njulia> struct Cat end\n\njulia> NiceStuff.nice(::Cat) = \"nice 😸\"\n\nAlternatively, you can import the specific function name:\n\njulia> import .NiceStuff: nice\n\njulia> struct Mouse end\n\njulia> nice(::Mouse) = \"nice 🐭\"\nnice (generic function with 3 methods)\n\nWhich one you choose is a matter of style. The first form makes it clear that you are adding a method to a function in another module (remember, that the imports and the method definition may be in separate files), while the second one is shorter, which is especially convenient if you are defining multiple methods.\n\nOnce a variable is made visible via using or import, a module may not create its own variable with the same name. Imported variables are read-only; assigning to a global variable always affects a variable owned by the current module, or else raises an error."},{"title":"Renaming with as","page":"Modules","location":"manual/modules.html#Renaming-with-as","category":"section","text":"An identifier brought into scope by import or using can be renamed with the keyword as. This is useful for working around name conflicts as well as for shortening names. For example, Base exports the function name read, but the CSV.jl package also provides CSV.read. If we are going to invoke CSV reading many times, it would be convenient to drop the CSV. qualifier. But then it is ambiguous whether we are referring to Base.read or CSV.read:\n\njulia> read;\n\njulia> import CSV: read\nWARNING: ignoring conflicting import of CSV.read into Main\n\nRenaming provides a solution:\n\njulia> import CSV: read as rd\n\nImported packages themselves can also be renamed:\n\nimport BenchmarkTools as BT\n\nas works with using only when a single identifier is brought into scope. For example using CSV: read as rd works, but using CSV as C does not, since it operates on all of the exported names in CSV."},{"title":"Mixing multiple using and import statements","page":"Modules","location":"manual/modules.html#Mixing-multiple-using-and-import-statements","category":"section","text":"When multiple using or import statements of any of the forms above are used, their effect is combined in the order they appear. For example,\n\njulia> using .NiceStuff         # exported names and the module name\n\njulia> import .NiceStuff: nice  # allows adding methods to unqualified functions\n\n\nwould bring all the exported names of NiceStuff and the module name itself into scope, and also allow adding methods to nice without prefixing it with a module name."},{"title":"Handling name conflicts","page":"Modules","location":"manual/modules.html#Handling-name-conflicts","category":"section","text":"Consider the situation where two (or more) packages export the same name, as in\n\njulia> module A\n       export f\n       f() = 1\n       end\nA\njulia> module B\n       export f\n       f() = 2\n       end\nB\n\nThe statement using .A, .B works, but when you try to call f, you get an error with a hint\n\njulia> using .A, .B\n\njulia> f\nERROR: UndefVarError: `f` not defined in `Main`\nHint: It looks like two or more modules export different bindings with this name, resulting in ambiguity. Try explicitly importing it from a particular module, or qualifying the name with the module it should come from.\n\nHere, Julia cannot decide which f you are referring to, so you have to make a choice. The following solutions are commonly used:\n\nSimply proceed with qualified names like A.f and B.f. This makes the context clear to the reader of your code, especially if f just happens to coincide but has different meaning in various packages. For example, degree has various uses in mathematics, the natural sciences, and in everyday life, and these meanings should be kept separate.\nUse the as keyword above to rename one or both identifiers, eg\njulia> using .A: f as f\n\njulia> using .B: f as g\n\nwould make B.f available as g. Here, we are assuming that you did not use using A before, which would have brought f into the namespace.\nWhen the names in question do share a meaning, it is common for one module to import it from another, or have a lightweight “base” package with the sole function of defining an interface like this, which can be used by other packages. It is conventional to have such package names end in ...Base (which has nothing to do with Julia's Base module)."},{"title":"Precedence order of definitions","page":"Modules","location":"manual/modules.html#Precedence-order-of-definitions","category":"section","text":"There are in general four kinds of binding definitions:\n\nThose provided via implicit import through using M\nThose provided via explicit import (e.g. using M: x, import M: x)\nThose declared implicitly as global (via global x without type specification)\nThose declared explicitly using definition syntax (const, global x::T, struct, etc.)\n\nSyntactically, we divide these into three precedence levels (from weakest to strongest)\n\nImplicit imports\nImplicit declarations\nExplicit declarations and imports\n\nIn general, we permit replacement of weaker bindings by stronger ones:\n\njulia> module M1; const x = 1; export x; end\nMain.M1\n\njulia> using .M1\n\njulia> x # Implicit import from M1\n1\n\njulia> begin; f() = (global x; x = 1) end\n\njulia> x # Implicit declaration\nERROR: UndefVarError: `x` not defined in `Main`\nSuggestion: add an appropriate import or assignment. This global was declared but not assigned.\n\njulia> const x = 2 # Explicit declaration\n2\n\nHowever, within the explicit precedence level, replacement is syntactically disallowed:\n\njulia> module M1; const x = 1; export x; end\nMain.M1\n\njulia> import .M1: x\n\njulia> const x = 2\nERROR: cannot declare Main.x constant; it was already declared as an import\nStacktrace:\n [1] top-level scope\n   @ REPL[3]:1\n\nor ignored:\n\njulia> const y = 2\n2\n\njulia> import .M1: x as y\nWARNING: import of M1.x into Main conflicts with an existing identifier; ignored.\n\nThe resolution of an implicit binding depends on the set of all using'd modules visible in the current world age. See the manual chapter on world age for more details."},{"title":"Default top-level definitions and bare modules","page":"Modules","location":"manual/modules.html#Default-top-level-definitions-and-bare-modules","category":"section","text":"Modules automatically contain using Core, using Base, and definitions of the eval and include functions, which evaluate expressions/files within the global scope of that module.\n\nIf these default definitions are not wanted, modules can be defined using the keyword baremodule instead (note: Core is still imported). In terms of baremodule, a standard module looks like this:\n\nbaremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend\n\nIf even Core is not wanted, a module that imports nothing and defines no names at all can be defined with Module(:YourNameHere, false, false) and code can be evaluated into it with @eval or Core.eval:\n\njulia> arithmetic = Module(:arithmetic, false, false)\nMain.arithmetic\n\njulia> @eval arithmetic add(x, y) = $(+)(x, y)\nadd (generic function with 1 method)\n\njulia> arithmetic.add(12, 13)\n25"},{"title":"Standard modules","page":"Modules","location":"manual/modules.html#Standard-modules","category":"section","text":"There are three important standard modules:\n\nCore contains all functionality \"built into\" the language.\nBase contains basic functionality that is useful in almost all cases.\nMain is the top-level module and the current module, when Julia is started.\n\nnote: Standard library modules\nBy default Julia ships with some standard library modules. These behave like regular Julia packages except that you don't need to install them explicitly. For example, if you wanted to perform some unit testing, you could load the Test standard library as follows:using Test"},{"title":"Submodules and relative paths","page":"Modules","location":"manual/modules.html#Submodules-and-relative-paths","category":"section","text":"Modules can contain submodules, nesting the same syntax module ... end. They can be used to introduce separate namespaces, which can be helpful for organizing complex codebases. Note that each module introduces its own scope, so submodules do not automatically “inherit” names from their parent.\n\nIt is recommended that submodules refer to other modules within the enclosing parent module (including the latter) using relative module qualifiers in using and import statements. A relative module qualifier starts with a period (.), which corresponds to the current module, and each successive . leads to the parent of the current module. This should be followed by modules if necessary, and eventually the actual name to access, all separated by .s. As a special case, however, referring to the module root can be written without ., avoiding the need to count the depth to reach that module.\n\nConsider the following example, where the submodule SubA defines a function, which is then extended in its “sibling” module:\n\njulia> module ParentModule\n       module SubA\n       export add_D  # exported interface\n       const D = 3\n       add_D(x) = x + D\n       end\n       using .SubA  # brings `add_D` into the namespace\n       export add_D # export it from ParentModule too\n       module SubB\n       import ..SubA: add_D # relative path for a “sibling” module\n       # import ParentModule.SubA: add_D # when in a package, such as when this is loaded by using or import, this would be equivalent to the previous import, but not at the REPL\n       struct Infinity end\n       add_D(x::Infinity) = x\n       end\n       end;\n\n\nYou may see code in packages, which, in a similar situation, uses import without the .:\n\njulia> import ParentModule.SubA: add_D\nERROR: ArgumentError: Package ParentModule not found in current path.\n\nHowever, since this operates through code loading, it only works if ParentModule is in a package in a file. If ParentModule was defined at the REPL, it is necessary to use use relative paths:\n\njulia> import .ParentModule.SubA: add_D\n\n\nNote that the order of definitions also matters if you are evaluating values. Consider\n\nmodule TestPackage\n\nexport x, y\n\nx = 0\n\nmodule Sub\nusing ..TestPackage\nz = y # ERROR: UndefVarError: `y` not defined in `Main`\nend\n\ny = 1\n\nend\n\nwhere Sub is trying to use TestPackage.y before it was defined, so it does not have a value.\n\nFor similar reasons, you cannot use a cyclic ordering:\n\nmodule A\n\nmodule B\nusing ..C # ERROR: UndefVarError: `C` not defined in `Main.A`\nend\n\nmodule C\nusing ..B\nend\n\nend"},{"title":"Module initialization and precompilation","page":"Modules","location":"manual/modules.html#Module-initialization-and-precompilation","category":"section","text":"Large modules can take several seconds to load because executing all of the statements in a module often involves compiling a large amount of code. Julia creates precompiled caches of the module to reduce this time.\n\nPrecompiled module files (sometimes called \"cache files\") are created and used automatically when import or using loads a module. If the cache file(s) do not yet exist, the module will be compiled and saved for future reuse. You can also manually call Base.compilecache(Base.identify_package(\"modulename\")) to create these files without loading the module. The resulting cache files will be stored in the compiled subfolder of DEPOT_PATH[1]. If nothing about your system changes, such cache files will be used when you load the module with import or using.\n\nPrecompilation cache files store definitions of modules, types, methods, and constants. They may also store method specializations and the code generated for them, but this typically requires that the developer add explicit precompile directives or execute workloads that force compilation during the package build.\n\nHowever, if you update the module's dependencies or change its source code, the module is automatically recompiled upon using or import. Dependencies are modules it imports, the Julia build, files it includes, or explicit dependencies declared by include_dependency(path) in the module file(s).\n\nFor file dependencies loaded by include, a change is determined by examining whether the file size (fsize) or content (condensed into a hash) is unchanged. For file dependencies loaded by include_dependency a change is determined by examining whether the modification time (mtime) is unchanged, or equal to the modification time truncated to the nearest second (to accommodate systems that can't copy mtime with sub-second accuracy). It also takes into account whether the path to the file chosen by the search logic in require matches the path that had created the precompile file. It also takes into account the set of dependencies already loaded into the current process and won't recompile those modules, even if their files change or disappear, in order to avoid creating incompatibilities between the running system and the precompile cache. Finally, it takes account of changes in any compile-time preferences.\n\nIf you know that a module is not safe to precompile (for example, for one of the reasons described below), you should put __precompile__(false) in the module file (typically placed at the top). This will cause Base.compilecache to throw an error, and will cause using / import to load it directly into the current process and skip the precompile and caching. This also thereby prevents the module from being imported by any other precompiled module.\n\nYou may need to be aware of certain behaviors inherent in the creation of incremental shared libraries which may require care when writing your module. For example, external state is not preserved. To accommodate this, explicitly separate any initialization steps that must occur at runtime from steps that can occur at compile time. For this purpose, Julia allows you to define an __init__() function in your module that executes any initialization steps that must occur at runtime. This function will not be called during compilation (--output-*). Effectively, you can assume it will be run exactly once in the lifetime of the code. You may, of course, call it manually if necessary, but the default is to assume this function deals with computing state for the local machine, which does not need to be – or even should not be – captured in the compiled image. It will be called after the module is loaded into a process, including if it is being loaded into an incremental compile (--output-incremental=yes), but not if it is being loaded into a full-compilation process.\n\nIn particular, if you define a function __init__() in a module, then Julia will call __init__() immediately after the module is loaded (e.g., by import, using, or require) at runtime for the first time (i.e., __init__ is only called once, and only after all statements in the module have been executed). Because it is called after the module is fully imported, any submodules or other imported modules have their __init__ functions called before the __init__ of the enclosing module. This is also synchronized across threads, so that code can safely rely upon this ordering of effects, such that all __init__ will have run, in dependency ordering, before the using result is completed. They may run concurrently with other __init__ methods which are not dependencies however, so be careful when accessing any shared state outside the current module to use locks when needed.\n\nTwo typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries. For example, suppose that we are calling a C library libfoo that requires us to call a foo_init() initialization function at runtime. Suppose that we also want to define a global constant foo_data_ptr that holds the return value of a void *foo_data() function defined by libfoo – this constant must be initialized at runtime (not at compile time) because the pointer address will change from run to run. You could accomplish this by defining the following __init__ function in your module:\n\nconst foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend\n\nNotice that it is perfectly possible to define a global inside a function like __init__; this is one of the advantages of using a dynamic language. But by making it a constant at global scope, we can ensure that the type is known to the compiler and allow it to generate better optimized code. Obviously, any other globals in your module that depends on foo_data_ptr would also have to be initialized in __init__.\n\nConstants involving most Julia objects that are not produced by ccall do not need to be placed in __init__: their definitions can be precompiled and loaded from the cached module image. This includes complicated heap-allocated objects like arrays. However, any routine that returns a raw pointer value must be called at runtime for precompilation to work (Ptr objects will turn into null pointers unless they are hidden inside an isbits object). This includes the return values of the Julia functions @cfunction and pointer.\n\nWhen using precompilation, it is important to keep a clear sense of the distinction between the compilation phase and the execution phase. In this mode, it will often be much more clearly apparent that Julia is a compiler which allows execution of arbitrary Julia code, not a standalone interpreter that also generates compiled code.\n\nOther known potential failure scenarios include:\n\nGlobal counters (for example, for attempting to uniquely identify objects). Consider the following code snippet:\nmutable struct UniquedById\n    myid::Int\n    let counter = 0\n        UniquedById() = new(counter += 1)\n    end\nend\nwhile the intent of this code was to give every instance a unique id, the counter value is recorded at the end of compilation. All subsequent usages of this incrementally compiled module will start from that same counter value.\nNote that objectid (which works by hashing the memory pointer) has similar issues (see notes on Dict usage below).\nOne alternative is to use a macro to capture @__MODULE__ and store it alone with the current counter value, however, it may be better to redesign the code to not depend on this global state.\nAssociative collections (such as Dict and Set) need to be re-hashed in __init__. (In the future, a mechanism may be provided to register an initializer function.)\nDepending on compile-time side-effects persisting through load-time. Example include: modifying arrays or other variables in other Julia modules; maintaining handles to open files or devices; storing pointers to other system resources (including memory);\nCreating accidental \"copies\" of global state from another module, by referencing it directly instead of via its lookup path. For example, (in global scope):\n#mystdout = Base.stdout #= will not work correctly, since this will copy Base.stdout into this module =#\n# instead use accessor functions:\ngetstdout() = Base.stdout #= best option =#\n# or move the assignment into the runtime:\n__init__() = global mystdout = Base.stdout #= also works =#\n\nSeveral additional restrictions are placed on the operations that can be done while precompiling code to help the user avoid other wrong-behavior situations:\n\nCalling eval to cause a side-effect in another module. This will also cause a warning to be emitted when the incremental precompile flag is set.\nglobal const statements from local scope after __init__() has been started (see issue #12010 for plans to add an error for this)\nReplacing a module is a runtime error while doing an incremental precompile.\n\nA few other points to be aware of:\n\nNo code reload / cache invalidation is performed after changes are made to the source files themselves, (including by Pkg.update), and no cleanup is done after Pkg.rm\nThe memory sharing behavior of a reshaped array is disregarded by precompilation (each view gets its own copy)\nExpecting the filesystem to be unchanged between compile-time and runtime e.g. @__FILE__/source_path() to find resources at runtime, or the BinDeps @checked_lib macro. Sometimes this is unavoidable. However, when possible, it can be good practice to copy resources into the module at compile-time so they won't need to be found at runtime.\nWeakRef objects and finalizers are not currently handled properly by the serializer (this will be fixed in an upcoming release).\nIt is usually best to avoid capturing references to instances of internal metadata objects such as Method, MethodInstance, MethodTable, TypeMapLevel, TypeMapEntry and fields of those objects, as this can confuse the serializer and may not lead to the outcome you desire. It is not necessarily an error to do this, but you simply need to be prepared that the system will try to copy some of these and to create a single unique instance of others.\n\nIt is sometimes helpful during module development to turn off incremental precompilation. The command line flag --compiled-modules={yes|no|existing} enables you to toggle module precompilation on and off. When Julia is started with --compiled-modules=no the serialized modules in the compile cache are ignored when loading modules and module dependencies. In some cases, you may want to load existing precompiled modules, but not create new ones. This can be done by starting Julia with --compiled-modules=existing. More fine-grained control is available with --pkgimages={yes|no|existing}, which only affects native-code storage during precompilation. Base.compilecache can still be called manually. The state of this command line flag is passed to Pkg.build to disable automatic precompilation triggering when installing, updating, and explicitly building packages.\n\nYou can also debug some precompilation failures with environment variables. Setting JULIA_VERBOSE_LINKING=true may help resolve failures in linking shared libraries of compiled native code. See the Developer Documentation part of the Julia manual, where you will find further details in the section documenting Julia's internals under \"Package Images\"."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray. This page documents some of the design principles and implementation of SubArrays.\n\nOne of the major design goals is to ensure high performance for views of both IndexLinear and IndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be IndexLinear to the extent that it is possible."},{"title":"Index replacement","page":"SubArrays","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":"Consider making 2d slices of a 3d array:\n\njulia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.967143  0.103929\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.789764  0.806704\n 0.566704  0.962715\n\nview drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2 are two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j]. To extract the value from the parent array A, the natural approach is to replace S1[i,j] with A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]].\n\nThe key feature of the design of SubArrays is that this index replacement can be performed without any runtime overhead."},{"title":"SubArray design","page":"SubArrays","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":""},{"title":"Type parameters and fields","page":"SubArrays","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":"The strategy adopted is first and foremost expressed in the definition of the type:\n\nstruct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend\n\nSubArray has 5 type parameters. The first two are the standard element type and dimensionality.  The next is the type of the parent AbstractArray. The most heavily-used is the fourth parameter, a Tuple of the types of the indices for each dimension. The final one, L, is only provided as a convenience for dispatch; it's a boolean that represents whether the index types support fast linear indexing. More on that later.\n\nIf in our example above A is a Array{Float64, 3}, our S1 case above would be a SubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}. Note in particular the tuple parameter, which stores the types of the indices used to create S1. Likewise,\n\njulia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)\n\nStoring these values allows index replacement, and having the types encoded as parameters allows one to dispatch to efficient algorithms."},{"title":"Index translation","page":"SubArrays","location":"devdocs/subarrays.html#Index-translation","category":"section","text":"Performing index translation requires that you do different things for different concrete SubArray types. For example, for S1, one needs to apply the i,j indices to the first and third dimensions of the parent array, whereas for S2 one needs to apply them to the second and third. The simplest approach to indexing would be to do the type-analysis at runtime:\n\nparentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]\n\nUnfortunately, this would be disastrous in terms of performance: each element access would allocate memory, and involves the running of a lot of poorly-typed code.\n\nThe better approach is to dispatch to specific methods to handle each type of stored index. That's what reindex does: it dispatches on the type of the first stored index and consumes the appropriate number of input indices, and then it recurses on the remaining indices. In the case of S1, this expands to\n\nBase.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])\n\nfor any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below).\n\nThis is the core of a SubArray; indexing methods depend upon reindex to do this index translation. Sometimes, though, we can avoid the indirection and make it even faster."},{"title":"Linear indexing","page":"SubArrays","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that separates successive elements, starting from some offset. This means that we can pre-compute these values and represent linear indexing simply as an addition and multiplication, avoiding the indirection of reindex and (more importantly) the slow computation of the cartesian coordinates entirely.\n\nFor SubArray types, the availability of efficient linear indexing is based purely on the types of the indices, and does not depend on values like the size of the parent array. You can ask whether a given set of indices supports fast linear indexing with the internal Base.viewindexing function:\n\njulia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()\n\nThis is computed during construction of the SubArray and stored in the L type parameter as a boolean that encodes fast linear indexing support. While not strictly necessary, it means that we can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries.\n\nSince this computation doesn't depend on runtime values, it can miss some cases in which the stride happens to be uniform:\n\njulia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2\n\nA view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear indexing indeed could be performed efficiently. However, success in this case depends on the size of the array: if the first dimension instead were odd,\n\njulia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2\n\nthen A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.  Since we have to base this decision based purely on types encoded in the parameters of the SubArray, S = view(A, 2:2:4, :) cannot implement efficient linear indexing."},{"title":"A few details","page":"SubArrays","location":"devdocs/subarrays.html#A-few-details","category":"section","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply determines how and where the stored indices should be reindexed. It not only supports integer indices, but it supports non-scalar indexing, too. This means that views of views don't need two levels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given by the parameter N, is not necessarily equal to the dimensionality of the parent array or the length of the indices tuple. Neither do user-supplied indices necessarily line up with entries in the indices tuple (e.g., the second user-supplied index might correspond to the third dimension of the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal to the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting this dramatically complicates the reindexing process, especially for views of views. Not only do you need to dispatch on the types of the stored indices, but you need to examine whether a given index is the final one and \"merge\" any remaining stored indices together. This is not an easy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly if possible. Consequently, view ensures that the parent array is the appropriate dimensionality for the given indices by reshaping it if needed. The inner SubArray constructor ensures that this invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that reindex simply dispatches on the type of the stored indices in order to determine how many passed indices should be used and where they should go. But with CartesianIndex, there's no longer a one-to-one correspondence between the number of passed arguments and the number of dimensions that they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)), you can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1). It should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across all combinations of dimensionalities in an intractable manner. As such, reindex must never be called with CartesianIndex indices. Fortunately, the scalar case is easily handled by first flattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however, cannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view must ensure that there are no arrays of CartesianIndex in the argument list. If there are, it can simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray with two levels of indirection instead."},{"title":"Mathematics","page":"Mathematics","location":"base/math.html#Mathematics","category":"section","text":""},{"title":"Mathematical Operators","page":"Mathematics","location":"base/math.html#math-ops","category":"section","text":""},{"title":"Mathematical Functions","page":"Mathematics","location":"base/math.html#Mathematical-Functions","category":"section","text":""},{"title":"Customizable binary operators","page":"Mathematics","location":"base/math.html#Customizable-binary-operators","category":"section","text":"Some unicode characters can be used to define new binary operators that support infix notation. For example ⊗(x,y) = kron(x,y) defines the ⊗ (otimes) function to be the Kronecker product, and one can call it as binary operator using infix syntax: C = A ⊗ B as well as with the usual prefix syntax C = ⊗(A,B).\n\nOther characters that support such extensions include \\odot ⊙ and \\oplus ⊕\n\nThe complete list is in the parser code: https://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm\n\nThose that are parsed like * (in terms of precedence) include * / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗ and those that are parsed like + include + - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⟇ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣ There are many others that are related to arrows, comparisons, and powers."},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any}","category":"method","text":"-(x)\n\nUnary minus operator.\n\nSee also: abs, flipsign.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\njulia> -(true)  # promotes to Int\n-1\n\njulia> -(0x003)\n0xfffd\n\n\n\n\n\n"},{"title":"Base.:+","page":"Mathematics","location":"base/math.html#Base.:+","category":"function","text":"+(x, y...)\n\nAddition operator.\n\nInfix x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...), which by default then calls (x+y) + z + ... starting from the left.\n\nNote that overflow is possible for most integer types, including the default Int, when adding large numbers.\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\njulia> [1,2] + [3,4]\n2-element Vector{Int64}:\n 4\n 6\n\njulia> typemax(Int) + 1 < 0\ntrue\n\n\n\n\n\ndt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of the Time are used along with the year, month, and day of the Date to create the new DateTime. Non-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n"},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any, Any}","category":"method","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n"},{"title":"Base.:*","page":"Mathematics","location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any}}","category":"method","text":"*(x, y...)\n\nMultiplication operator.\n\nInfix x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...), which by default then calls (x*y) * z * ... starting from the left.\n\nJuxtaposition such as 2pi also calls *(2, pi). Note that this operation has higher precedence than a literal *. Note also that juxtaposition \"0x...\" (integer zero times a variable whose name starts with x) is forbidden as it clashes with unsigned integer literals: 0x01 isa UInt8.\n\nNote that overflow is possible for most integer types, including the default Int, when multiplying large numbers.\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\njulia> [2 0; 0 3] * [1, 10]  # matrix * vector\n2-element Vector{Int64}:\n  2\n 30\n\njulia> 1/2pi, 1/2*pi  # juxtaposition has higher precedence\n(0.15915494309189535, 1.5707963267948966)\n\njulia> x = [1, 2]; x'x  # adjoint vector * vector\n5\n\n\n\n\n\n"},{"title":"Base.:/","page":"Mathematics","location":"base/math.html#Base.:/","category":"function","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right.\n\nGives floating-point results for integer arguments. See ÷ for integer division, or // for Rational results.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\nA / B\n\nMatrix right-division: A / B is equivalent to (B' \\ A')' where \\ is the left-division operator. For square matrices, the result X is such that A == X*B.\n\nSee also: rdiv!.\n\nExamples\n\njulia> A = Float64[1 4 5; 3 9 2]; B = Float64[1 4 2; 3 4 2; 8 7 1];\n\njulia> X = A / B\n2×3 Matrix{Float64}:\n -0.65   3.75  -1.2\n  3.25  -2.75   1.0\n\njulia> isapprox(A, X*B)\ntrue\n\njulia> isapprox(X, A*pinv(B))\ntrue\n\n\n\n\n\n"},{"title":"Base.:\\","page":"Mathematics","location":"base/math.html#Base.:\\-Tuple{Any, Any}","category":"method","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n"},{"title":"Base.:^","page":"Mathematics","location":"base/math.html#Base.:^-Tuple{Number, Number}","category":"method","text":"^(x, y)\n\nExponentiation operator.\n\nIf x and y are integers, the result may overflow. To enter numbers in scientific notation, use Float64 literals such as 1.2e3 rather than 1.2 * 10^3.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code x^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to enable compile-time specialization on the value of the exponent. (As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y), where usually ^ == Base.^ unless ^ has been defined in the calling namespace.) If y is a negative integer literal, then Base.literal_pow transforms the operation to inv(x)^-y by default, where -y is positive.\n\nSee also exp2, <<.\n\nExamples\n\njulia> 3^5\n243\n\njulia> 3^-1  # uses Base.literal_pow\n0.3333333333333333\n\njulia> p = -1;\n\njulia> 3^p\nERROR: DomainError with -1:\nCannot raise an integer x to a negative power -1.\n[...]\n\njulia> 3.0^p\n0.3333333333333333\n\njulia> 10^19 > 0  # integer overflow\nfalse\n\njulia> big(10)^19 == 1e19\ntrue\n\n\n\n\n\n"},{"title":"Base.fma","page":"Mathematics","location":"base/math.html#Base.fma","category":"function","text":"fma(x, y, z)\n\nComputes x*y+z without rounding the intermediate result x*y. On some systems this is significantly more expensive than x*y+z. fma is used to improve accuracy in certain algorithms. See muladd.\n\n\n\n\n\n"},{"title":"Base.muladd","page":"Mathematics","location":"base/math.html#Base.muladd","category":"function","text":"muladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged with each other or with surrounding operations for performance. For example, this may be implemented as an fma if the hardware supports it efficiently. The result can be different on different machines and can also be different on the same machine due to constant propagation or other optimizations. See fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\nmuladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication. The result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\n"},{"title":"Base.inv","page":"Mathematics","location":"base/math.html#Base.inv-Tuple{Number}","category":"method","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x yields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for some types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div","category":"function","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean (integer) division. Generally equivalent to a mathematical operation x/y without a fractional part.\n\nSee also: cld, fld, rem, divrem.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\njulia> div.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  0  0  1  1  1\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div-Tuple{Any, Any, RoundingMode}","category":"method","text":"div(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient from Euclidean (integer) division. Computes x / y, rounded to an integer according to the rounding mode r. In other words, the quantity\n\nround(x / y, r)\n\nwithout any intermediate rounding.\n\ncompat: Julia 1.4\nThe three-argument method taking a RoundingMode requires Julia 1.4 or later.\n\nSee also fld and cld, which are special cases of this function.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> div(4, 3, RoundToZero) # Matches div(4, 3)\n1\njulia> div(4, 3, RoundDown) # Matches fld(4, 3)\n1\njulia> div(4, 3, RoundUp) # Matches cld(4, 3)\n2\njulia> div(5, 2, RoundNearest)\n2\njulia> div(5, 2, RoundNearestTiesAway)\n3\njulia> div(-5, 2, RoundNearest)\n-2\njulia> div(-5, 2, RoundNearestTiesAway)\n-3\njulia> div(-5, 2, RoundNearestTiesUp)\n-2\njulia> div(4, 3, RoundFromZero)\n2\njulia> div(-4, 3, RoundFromZero)\n-2\n\nBecause div(x, y) implements strictly correct truncated rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> div(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but div(6.0, 0.1, RoundToZero) always truncates the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.fld","page":"Mathematics","location":"base/math.html#Base.fld","category":"function","text":"fld(x, y)\n\nLargest integer less than or equal to x / y. Equivalent to div(x, y, RoundDown).\n\nSee also div, cld, fld1.\n\nExamples\n\njulia> fld(7.3, 5.5)\n1.0\n\njulia> fld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -2  -1  -1  -1  0  0  0  1  1  1\n\nBecause fld(x, y) implements strictly correct floored rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> fld(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but fld(6.0, 0.1) always takes the floor of the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.cld","page":"Mathematics","location":"base/math.html#Base.cld","category":"function","text":"cld(x, y)\n\nSmallest integer larger than or equal to x / y. Equivalent to div(x, y, RoundUp).\n\nSee also div, fld.\n\nExamples\n\njulia> cld(5.5, 2.2)\n3.0\n\njulia> cld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  1  1  1  2  2\n\n\n\n\n\n"},{"title":"Base.mod","page":"Mathematics","location":"base/math.html#Base.mod","category":"function","text":"rem(x::Integer, T::Type{<:Integer})::T\nmod(x::Integer, T::Type{<:Integer})::T\n%(x::Integer, T::Type{<:Integer})::T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable in T, and y is an integer in [typemin(T),typemax(T)]. If T can represent any integer (e.g. T == BigInt), then this operation corresponds to a conversion to T.\n\nExamples\n\njulia> x = 129 % Int8\n-127\n\njulia> typeof(x)\nInt8\n\njulia> x = 129 % BigInt\n129\n\njulia> typeof(x)\nBigInt\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored division by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y if isfinite(y), and magnitude less than abs(y) (with some exceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the type, and so rounding error may occur. In particular, if the exact result is very close to y, then it may be rounded to y.\n\nSee also: rem, div, fld, mod1, invmod.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\njulia> mod.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  0  1  2  0  1  2  0  1  2\n\n\n\n\n\nmod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x ≡ y (mod n), where n = length(r), i.e. y = mod(x - first(r), n) + first(r).\n\nSee also mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))  # mod1(0, 3)\n3\n\njulia> mod(3, 0:2)  # mod(3, 3)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem","category":"function","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in magnitude than y. This value is always exact.\n\nSee also: div, mod, mod1, divrem.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\njulia> rem.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -1  0  -2  -1  0  1  2  0  1  2\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem-Tuple{Any, Any, RoundingMode}","category":"method","text":"rem(x, y, r::RoundingMode=RoundToZero)\n\nCompute the remainder of x after integer division by y, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - y * round(x / y, r)\n\nwithout any intermediate rounding.\n\nif r == RoundNearest, then the result is exact, and in the interval -y  2 y  2. See also RoundNearest.\nif r == RoundToZero (default), then the result is exact, and in the interval 0 y) if x is positive, or (-y 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 y) if y is positive, or (y 0 otherwise. The result may not be exact if x and y have different signs, and abs(x) < abs(y). See also RoundDown.\nif r == RoundUp, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundUp.\nif r == RoundFromZero, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundFromZero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> x = 9; y = 4;\n\njulia> x % y  # same as rem(x, y)\n1\n\njulia> x ÷ y  # same as div(x, y)\n2\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.Math.rem2pi","page":"Mathematics","location":"base/math.html#Base.Math.rem2pi","category":"function","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of 2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally be the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,. or -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π. See also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0. See also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n"},{"title":"Base.Math.mod2pi","page":"Mathematics","location":"base/math.html#Base.Math.mod2pi","category":"function","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by numerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would compute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may be less than 2π. For example, the expression mod2pi(2π) will not return 0, because the intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See rem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n"},{"title":"Base.divrem","page":"Mathematics","location":"base/math.html#Base.divrem","category":"function","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division. Equivalent to (div(x, y, r), rem(x, y, r)). Equivalently, with the default value of r, this call is equivalent to (x ÷ y, x % y).\n\nSee also: fldmod, cld.\n\nExamples\n\njulia> divrem(3, 7)\n(0, 3)\n\njulia> divrem(7, 3)\n(2, 1)\n\n\n\n\n\n"},{"title":"Base.fldmod","page":"Mathematics","location":"base/math.html#Base.fldmod","category":"function","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for divrem(x, y, RoundDown). Equivalent to (fld(x, y), mod(x, y)).\n\nSee also: fld, cld, fldmod1.\n\n\n\n\n\n"},{"title":"Base.fld1","page":"Mathematics","location":"base/math.html#Base.fld1","category":"function","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.mod1","page":"Mathematics","location":"base/math.html#Base.mod1","category":"function","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y) in the range (0 y for positive y and in the range y0) for negative y.\n\nWith integer arguments and positive y, this is equal to mod(x, 1:y), and hence natural for 1-based indexing. By comparison, mod(x, y) == mod(x, 0:y-1) is natural for computations with offsets or strides.\n\nSee also mod, fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  3  1  2  3  1  2  3  1  2\n\njulia> mod1.([-0.1, 0, 0.1, 1, 2, 2.9, 3, 3.1]', 3)\n1×8 Matrix{Float64}:\n 2.9  3.0  0.1  1.0  2.0  2.9  3.0  0.1\n\n\n\n\n\n"},{"title":"Base.fldmod1","page":"Mathematics","location":"base/math.html#Base.fldmod1","category":"function","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also fld1, mod1.\n\n\n\n\n\n"},{"title":"Base.://","page":"Mathematics","location":"base/math.html#Base.://","category":"function","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result. More generally, // can be used for exact rational division of other numeric types with integer or rational components, such as complex numbers with integer components.\n\nNote that floating-point (AbstractFloat) arguments are not permitted by // (even if the values are rational). The arguments must be subtypes of Integer, Rational, or composites thereof.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\njulia> (1+2im) // (3+4im)\n11//25 + 2//25*im\n\njulia> 1.0 // 2\nERROR: MethodError: no method matching //(::Float64, ::Int64)\n[...]\n\n\n\n\n\n"},{"title":"Base.rationalize","page":"Mathematics","location":"base/math.html#Base.rationalize","category":"function","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components of the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n"},{"title":"Base.numerator","page":"Mathematics","location":"base/math.html#Base.numerator","category":"function","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n"},{"title":"Base.denominator","page":"Mathematics","location":"base/math.html#Base.denominator","category":"function","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n"},{"title":"Base.:<<","page":"Mathematics","location":"base/math.html#Base.:<<","category":"function","text":"<<(B::BitVector, n)::BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B with elements shifted n positions backwards, filling with false values. If n < 0, elements are shifted forwards. Equivalent to B >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left by n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0, this is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>, exp2, ldexp.\n\n\n\n\n\n"},{"title":"Base.:>>","page":"Mathematics","location":"base/math.html#Base.:>>","category":"function","text":">>(B::BitVector, n)::BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B with elements shifted n positions forward, filling with false values. If n < 0, elements are shifted backwards. Equivalent to B << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n>>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted right by n bits, filling with 0s if x >= 0, 1s if x < 0, preserving the sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is equivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n"},{"title":"Base.:>>>","page":"Mathematics","location":"base/math.html#Base.:>>>","category":"function","text":">>>(B::BitVector, n)::BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for details and examples.\n\n\n\n\n\n>>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x shifted right by n bits, filling with 0s. For n < 0, this is equivalent to x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For Signed integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this is equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n"},{"title":"Base.bitrotate","page":"Mathematics","location":"base/math.html#Base.bitrotate","category":"function","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation. It returns the value of x with its bits rotated left k times. A negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\nSee also: <<, circshift, BitArray.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n"},{"title":"Base.::","page":"Mathematics","location":"base/math.html#Base.::-Tuple{Any, Any, Any}","category":"method","text":"(:)(start, [step], stop)\n\nRange operator. a:b constructs a range from a to b with a step size equal to +1, which produces:\n\na UnitRange when a and b are integers, or\na StepRange when a and b are characters, or\na StepRangeLen when a and/or b are floating-point.\n\na:s:b is similar but uses a step size of s (a StepRange or StepRangeLen). See also range for more control.\n\nTo create a descending range, use reverse(a:b) or a negative step size, e.g. b:-1:a. Otherwise, when b < a, an empty range will be constructed and normalized to a:a-1.\n\nThe operator : is also used in indexing to select whole dimensions, e.g. in A[:, 1].\n\n: is also used to quote code, e.g. :(x + y) isa Expr and :x isa Symbol. Since :2 isa Int, it does not create a range in indexing: v[:2] == v[2] != v[begin:2].\n\n\n\n\n\n"},{"title":"Base.::","page":"Mathematics","location":"base/math.html#Base.::-Tuple{CartesianIndex, CartesianIndex, CartesianIndex}","category":"method","text":"(:)(start::CartesianIndex, [step::CartesianIndex], stop::CartesianIndex)\n\nConstruct CartesianIndices from two CartesianIndex and an optional step.\n\ncompat: Julia 1.1\nThis method requires at least Julia 1.1.\n\ncompat: Julia 1.6\nThe step range method start:step:stop requires at least Julia 1.6.\n\nExamples\n\njulia> I = CartesianIndex(2,1);\n\njulia> J = CartesianIndex(3,3);\n\njulia> I:J\nCartesianIndices((2:3, 1:3))\n\njulia> I:CartesianIndex(1, 2):J\nCartesianIndices((2:1:3, 1:2:3))\n\n\n\n\n\n"},{"title":"Base.range","page":"Mathematics","location":"base/math.html#Base.range","category":"function","text":"range(start, stop, length)\nrange(start, stop; length, step)\nrange(start; length, stop, step)\nrange(;start, length, stop, step)\n\nConstruct a specialized array with evenly spaced elements and optimized storage (an AbstractRange) from the arguments. Mathematically a range is uniquely determined by any three of start, step, stop and length. Valid invocations of range are:\n\nCall range with any three of start, step, stop, length.\nCall range with two of start, stop, length. In this case step will be assumed to be positive one. If both arguments are Integers, a UnitRange will be returned.\nCall range with one of stop or length. start and step will be assumed to be positive one.\n\nTo construct a descending range, specify a negative step size, e.g. range(5, 1; step = -1) => [5,4,3,2,1]. Otherwise, a stop value less than the start value, with the default step of +1, constructs an empty range. Empty ranges are normalized such that the stop is one less than the start, e.g. range(5, 1) == 5:4.\n\nSee Extended Help for additional details on the returned type. See also logrange for logarithmically spaced points.\n\nExamples\n\njulia> range(1, length=100)\n1:100\n\njulia> range(1, stop=100)\n1:100\n\njulia> range(1, step=5, length=100)\n1:5:496\n\njulia> range(1, step=5, stop=100)\n1:5:96\n\njulia> range(1, 10, length=101)\n1.0:0.09:10.0\n\njulia> range(1, 100, step=5)\n1:5:96\n\njulia> range(stop=10, length=5)\n6:10\n\njulia> range(stop=10, step=1, length=5)\n6:1:10\n\njulia> range(start=1, step=1, stop=10)\n1:1:10\n\njulia> range(; length = 10)\nBase.OneTo(10)\n\njulia> range(; stop = 6)\nBase.OneTo(6)\n\njulia> range(; stop = 6.5)\n1.0:1.0:6.0\n\nIf length is not specified and stop - start is not an integer multiple of step, a range that ends before stop will be produced.\n\njulia> range(1, 3.5, step=2)\n1.0:2.0:3.0\n\nSpecial care is taken to ensure intermediate values are computed rationally. To avoid this induced overhead, see the LinRange constructor.\n\ncompat: Julia 1.1\nstop as a positional argument requires at least Julia 1.1.\n\ncompat: Julia 1.7\nThe versions without keyword arguments and start as a keyword argument require at least Julia 1.7.\n\ncompat: Julia 1.8\nThe versions with stop as a sole keyword argument, or length as a sole keyword argument require at least Julia 1.8.\n\nExtended Help\n\nrange will produce a Base.OneTo when the arguments are Integers and\n\nOnly length is provided\nOnly stop is provided\n\nrange will produce a UnitRange when the arguments are Integers and\n\nOnly start  and stop are provided\nOnly length and stop are provided\n\nA UnitRange is not produced if step is provided even if specified as one.\n\n\n\n\n\n"},{"title":"Base.OneTo","page":"Mathematics","location":"base/math.html#Base.OneTo","category":"type","text":"Base.OneTo(n)\n\nDefine an AbstractUnitRange that behaves like 1:n, with the added distinction that the lower limit is guaranteed (by the type system) to be 1.\n\n\n\n\n\n"},{"title":"Base.StepRangeLen","page":"Mathematics","location":"base/math.html#Base.StepRangeLen","category":"type","text":"StepRangeLen(         ref::R, step::S, len, [offset=1]) where {  R,S}\nStepRangeLen{T,R,S}(  ref::R, step::S, len, [offset=1]) where {T,R,S}\nStepRangeLen{T,R,S,L}(ref::R, step::S, len, [offset=1]) where {T,R,S,L}\n\nA range r where r[i] produces values of type T (in the first form, T is deduced automatically), parameterized by a reference value, a step, and the length. By default ref is the starting value r[1], but alternatively you can supply it as the value of r[offset] for some other index 1 <= offset <= len. The syntax a:b or a:b:c, where any of a, b, or c are floating-point numbers, creates a StepRangeLen.\n\ncompat: Julia 1.7\nThe 4th type parameter L requires at least Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.logrange","page":"Mathematics","location":"base/math.html#Base.logrange","category":"function","text":"logrange(start, stop, length)\nlogrange(start, stop; length)\n\nConstruct a specialized array whose elements are spaced logarithmically between the given endpoints. That is, the ratio of successive elements is a constant, calculated from the length.\n\nThis is similar to geomspace in Python. Unlike PowerRange in Mathematica, you specify the number of elements not the ratio. Unlike logspace in Python and Matlab, the start and stop arguments are always the first and last elements of the result, not powers applied to some base.\n\nExamples\n\njulia> logrange(10, 4000, length=3)\n3-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 10.0, 200.0, 4000.0\n\njulia> ans[2] ≈ sqrt(10 * 4000)  # middle element is the geometric mean\ntrue\n\njulia> range(10, 40, length=3)[2] ≈ (10 + 40)/2  # arithmetic mean\ntrue\n\njulia> logrange(1f0, 32f0, 11)\n11-element Base.LogRange{Float32, Float64}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0, 5.65685, 8.0, 11.3137, 16.0, 22.6274, 32.0\n\njulia> logrange(1, 1000, length=4) ≈ 10 .^ (0:3)\ntrue\n\nSee the LogRange type for further details.\n\nSee also range for linearly spaced points.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.LogRange","page":"Mathematics","location":"base/math.html#Base.LogRange","category":"type","text":"LogRange{T}(start, stop, len) <: AbstractVector{T}\n\nA range whose elements are spaced logarithmically between start and stop, with spacing controlled by len. Returned by logrange.\n\nLike LinRange, the first and last elements will be exactly those provided, but intermediate values may have small floating-point errors. These are calculated using the logs of the endpoints, which are stored on construction, often in higher precision than T.\n\nExamples\n\njulia> logrange(1, 4, length=5)\n5-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0\n\njulia> Base.LogRange{Float16}(1, 4, 5)\n5-element Base.LogRange{Float16, Float64}:\n 1.0, 1.414, 2.0, 2.828, 4.0\n\njulia> logrange(1e-310, 1e-300, 11)[1:2:end]\n6-element Vector{Float64}:\n 1.0e-310\n 9.999999999999974e-309\n 9.999999999999981e-307\n 9.999999999999988e-305\n 9.999999999999994e-303\n 1.0e-300\n\njulia> prevfloat(1e-308, 5) == ans[2]\ntrue\n\nNote that integer eltype T is not allowed. Use for instance round.(Int, xs), or explicit powers of some integer base:\n\njulia> xs = logrange(1, 512, 4)\n4-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 8.0, 64.0, 512.0\n\njulia> 2 .^ (0:3:9) |> println\n[1, 8, 64, 512]\n\ncompat: Julia 1.11\nThis type requires at least Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.:==","page":"Mathematics","location":"base/math.html#Base.:==","category":"function","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===. Should be implemented for all types with a notion of equality, based on the abstract value that an instance represents. For example, all numeric types are compared by numeric value, ignoring type. Strings are compared as sequences of characters, ignoring encoding. Collections of the same type generally compare their key sets, and if those are ==, then compare the values for each of those keys, returning true if all such pairs are ==. Other properties are typically not taken into account (such as the exact type).\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and NaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing, in which case missing is returned (three-valued logic). Collections generally implement three-valued logic akin to all, returning missing if any operands contain missing values and all other pairs are equal. Use isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and handle comparison to other types via promotion rules where possible.\n\nEquality and hashing are intimately related; two values that are considered isequal must have the same hash and by default isequal falls back to ==. If a type customizes the behavior of == and/or isequal, then hash must be similarly implemented to ensure isequal and hash agree. Sets, Dicts, and many other internal implementations assume that this invariant holds.\n\nIf some type defines ==, isequal, and isless then it should also implement < to ensure consistency of comparisons.\n\n\n\n\n\n"},{"title":"Base.:!=","page":"Mathematics","location":"base/math.html#Base.:!=","category":"function","text":"!=(x)\n\nCreate a function that compares its argument to x using !=, i.e. a function equivalent to y -> y != x. The returned function is of type Base.Fix2{typeof(!=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition !=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n"},{"title":"Core.:!==","page":"Mathematics","location":"base/math.html#Core.:!==","category":"function","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n"},{"title":"Base.:<","page":"Mathematics","location":"base/math.html#Base.:<","category":"function","text":"<(x)\n\nCreate a function that compares its argument to x using <, i.e. a function equivalent to y -> y < x. The returned function is of type Base.Fix2{typeof(<)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<(x, y)\n\nLess-than comparison operator. Falls back to isless. Because of the behavior of floating-point NaN values, this operator implements a partial order.\n\nImplementation\n\nNew types with a canonical partial order should implement this function for two arguments of the new type. Types with a canonical total order should implement isless instead.\n\nSee also isunordered.\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:<=","page":"Mathematics","location":"base/math.html#Base.:<=","category":"function","text":"<=(x)\n\nCreate a function that compares its argument to x using <=, i.e. a function equivalent to y -> y <= x. The returned function is of type Base.Fix2{typeof(<=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:>","page":"Mathematics","location":"base/math.html#Base.:>","category":"function","text":">(x)\n\nCreate a function that compares its argument to x using >, i.e. a function equivalent to y -> y > x. The returned function is of type Base.Fix2{typeof(>)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function, and rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n"},{"title":"Base.:>=","page":"Mathematics","location":"base/math.html#Base.:>=","category":"function","text":">=(x)\n\nCreate a function that compares its argument to x using >=, i.e. a function equivalent to y -> y >= x. The returned function is of type Base.Fix2{typeof(>=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nImplementation\n\nNew types should prefer to implement <= instead of this function, and rely on the fallback definition >=(x, y) = y <= x.\n\nFurthermore, in many cases it is enough to implement just < and ==, relying on the fallback definitions of both <= and >=.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n"},{"title":"Base.cmp","page":"Mathematics","location":"base/math.html#Base.cmp","category":"function","text":"cmp(a::AbstractString, b::AbstractString)::Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\n"},{"title":"Base.:~","page":"Mathematics","location":"base/math.html#Base.:~","category":"function","text":"~(x)\n\nBitwise not.\n\nSee also: !, &, |.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n"},{"title":"Base.:&","page":"Mathematics","location":"base/math.html#Base.:&","category":"function","text":"x & y\n\nBitwise and. Implements three-valued logic, returning missing if one operand is missing and the other is true. Add parentheses for function application form: (&)(x, y).\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n"},{"title":"Base.:|","page":"Mathematics","location":"base/math.html#Base.:|","category":"function","text":"x | y\n\nBitwise or. Implements three-valued logic, returning missing if one operand is missing and the other is false.\n\nSee also: &, xor, ||.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n"},{"title":"Base.xor","page":"Mathematics","location":"base/math.html#Base.xor","category":"function","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and ⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n"},{"title":"Base.nand","page":"Mathematics","location":"base/math.html#Base.nand","category":"function","text":"nand(x, y)\n⊼(x, y)\n\nBitwise nand (not and) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊼ b is a synonym for nand(a,b), and ⊼ can be typed by tab-completing \\nand or \\barwedge in the Julia REPL.\n\nExamples\n\njulia> nand(true, false)\ntrue\n\njulia> nand(true, true)\nfalse\n\njulia> nand(true, missing)\nmissing\n\njulia> false ⊼ false\ntrue\n\njulia> [true; true; false] .⊼ [true; false; false]\n3-element BitVector:\n 0\n 1\n 1\n\n\n\n\n\n"},{"title":"Base.nor","page":"Mathematics","location":"base/math.html#Base.nor","category":"function","text":"nor(x, y)\n⊽(x, y)\n\nBitwise nor (not or) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing and the other is not true.\n\nThe infix operation a ⊽ b is a synonym for nor(a,b), and ⊽ can be typed by tab-completing \\nor or \\barvee in the Julia REPL.\n\nExamples\n\njulia> nor(true, false)\nfalse\n\njulia> nor(true, true)\nfalse\n\njulia> nor(true, missing)\nfalse\n\njulia> false ⊽ false\ntrue\n\njulia> false ⊽ missing\nmissing\n\njulia> [true; true; false] .⊽ [true; false; false]\n3-element BitVector:\n 0\n 0\n 1\n\n\n\n\n\n"},{"title":"Base.:!","page":"Mathematics","location":"base/math.html#Base.:!","category":"function","text":"!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a composed function which computes the boolean negation of f.\n\nSee also ∘.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\ncompat: Julia 1.9\nStarting with Julia 1.9, !f returns a ComposedFunction instead of an anonymous function.\n\n\n\n\n\n!(x)\n\nBoolean not. Implements three-valued logic, returning missing if x is missing.\n\nSee also ~ for bitwise not.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n"},{"title":"&&","page":"Mathematics","location":"base/math.html#&&","category":"keyword","text":"x && y\n\nShort-circuiting boolean AND.\n\nThis is equivalent to x ? y : false: it returns false if x is false and the result of evaluating y if x is true. Note that if y is an expression, it is only evaluated when x is true, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) && (statement) can be used as shorthand for if condition; statement; end for an arbitrary statement.\n\nSee also &, the ternary operator ? :, and the manual section on control flow.\n\nExamples\n\njulia> x = 3;\n\njulia> x > 1 && x < 10 && x isa Int\ntrue\n\njulia> x < 0 && error(\"expected positive x\")\nfalse\n\njulia> x > 0 && \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"||","page":"Mathematics","location":"base/math.html#||","category":"keyword","text":"x || y\n\nShort-circuiting boolean OR.\n\nThis is equivalent to x ? true : y: it returns true if x is true and the result of evaluating y if x is false. Note that if y is an expression, it is only evaluated when x is false, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) || (statement) can be used as shorthand for if !(condition); statement; end for an arbitrary statement.\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> pi < 3 || ℯ < 3\ntrue\n\njulia> false || true || println(\"neither is true!\")\ntrue\n\njulia> pi < 3 || \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"Base.isapprox","page":"Mathematics","location":"base/math.html#Base.isapprox","category":"function","text":"isapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\ncompat: Julia 1.5\nThis method requires Julia 1.5 or later.\n\n\n\n\n\nisapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison. Two numbers compare equal if their relative distance or their absolute distance is within tolerance bounds: isapprox returns true if norm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol (absolute tolerance) is zero and the default rtol (relative tolerance) depends on the types of x and y. The keyword argument nans determines whether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to the square root of eps of the type of x or y, whichever is bigger (least precise). This corresponds to requiring equality of about half of the significant digits. Otherwise, e.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for arrays (where an alternative norm choice is sometimes useful). When x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf or NaN), the comparison falls back to checking whether all elements of x and y are approximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y is equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is equivalent to x == 0 since the default atol is 0.  In such cases, you should either supply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g. use x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol automatically because it depends on the overall scaling (the \"units\") of your problem: for example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the radius of the Earth in meters, but an absurdly large tolerance if x is the radius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments requires Julia 1.6 or later.\n\nExamples\n\njulia> isapprox(0.1, 0.15; atol=0.05)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.34)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.33)\nfalse\n\njulia> 0.1 + 1e-10 ≈ 0.1\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0]) # using `norm`\ntrue\n\n\n\n\n\n"},{"title":"Base.sin","page":"Mathematics","location":"base/math.html#Base.sin-Tuple{Number}","category":"method","text":"sin(x::T) where {T <: Number} -> float(T)\n\nCompute sine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, sinpi, sincos, cis, asin.\n\nExamples\n\njulia> round.(sin.(range(0, 2pi, length=9)'), digits=3)\n1×9 Matrix{Float64}:\n 0.0  0.707  1.0  0.707  0.0  -0.707  -1.0  -0.707  -0.0\n\njulia> sind(45)\n0.7071067811865476\n\njulia> sinpi(1/4)\n0.7071067811865475\n\njulia> round.(sincos(pi/6), digits=3)\n(0.5, 0.866)\n\njulia> round(cis(pi/6), digits=3)\n0.866 + 0.5im\n\njulia> round(exp(im*pi/6), digits=3)\n0.866 + 0.5im\n\n\n\n\n\n"},{"title":"Base.cos","page":"Mathematics","location":"base/math.html#Base.cos-Tuple{Number}","category":"method","text":"cos(x::T) where {T <: Number} -> float(T)\n\nCompute cosine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also cosd, cospi, sincos, cis.\n\n\n\n\n\n"},{"title":"Base.Math.sincos","page":"Mathematics","location":"base/math.html#Base.Math.sincos-Tuple{Float64}","category":"method","text":"sincos(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) if isnan(x).\n\nSee also cis, sincospi, sincosd.\n\n\n\n\n\n"},{"title":"Base.tan","page":"Mathematics","location":"base/math.html#Base.tan-Tuple{Number}","category":"method","text":"tan(x::T) where {T <: Number} -> float(T)\n\nCompute tangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also tanh.\n\n\n\n\n\n"},{"title":"Base.Math.sind","page":"Mathematics","location":"base/math.html#Base.Math.sind","category":"function","text":"sind(x::T) where T -> float(T)\n\nCompute sine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.cosd","page":"Mathematics","location":"base/math.html#Base.Math.cosd","category":"function","text":"cosd(x::T) where T -> float(T)\n\nCompute cosine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.tand","page":"Mathematics","location":"base/math.html#Base.Math.tand","category":"function","text":"tand(x::T) where T -> float(T)\n\nCompute tangent of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sincosd","page":"Mathematics","location":"base/math.html#Base.Math.sincosd","category":"function","text":"sincosd(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in degrees, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.Math.sinpi","page":"Mathematics","location":"base/math.html#Base.Math.sinpi","category":"function","text":"sinpi(x::T) where T -> float(T)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.cospi","page":"Mathematics","location":"base/math.html#Base.Math.cospi","category":"function","text":"cospi(x::T) where T -> float(T)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.Math.tanpi","page":"Mathematics","location":"base/math.html#Base.Math.tanpi","category":"function","text":"tanpi(x::T) where T -> float(T)\n\nCompute tan(pi x) more accurately than tan(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also tand, sinpi, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.sincospi","page":"Mathematics","location":"base/math.html#Base.Math.sincospi","category":"function","text":"sincospi(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x, where x is in radians), returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.sinh","page":"Mathematics","location":"base/math.html#Base.sinh-Tuple{Number}","category":"method","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\nSee also sin.\n\n\n\n\n\n"},{"title":"Base.cosh","page":"Mathematics","location":"base/math.html#Base.cosh-Tuple{Number}","category":"method","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\nSee also cos.\n\n\n\n\n\n"},{"title":"Base.tanh","page":"Mathematics","location":"base/math.html#Base.tanh-Tuple{Number}","category":"method","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\nSee also tan, atanh.\n\nExamples\n\njulia> tanh.(-3:3f0)  # Here 3f0 isa Float32\n7-element Vector{Float32}:\n -0.9950548\n -0.9640276\n -0.7615942\n  0.0\n  0.7615942\n  0.9640276\n  0.9950548\n\njulia> tan.(im .* (1:3))\n3-element Vector{ComplexF64}:\n 0.0 + 0.7615941559557649im\n 0.0 + 0.9640275800758169im\n 0.0 + 0.9950547536867306im\n\n\n\n\n\n"},{"title":"Base.asin","page":"Mathematics","location":"base/math.html#Base.asin-Tuple{Number}","category":"method","text":"asin(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse sine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\nSee also asind for output in degrees.\n\nExamples\n\njulia> asin.((0, 1/2, 1))\n(0.0, 0.5235987755982989, 1.5707963267948966)\n\njulia> asind.((0, 1/2, 1))\n(0.0, 30.000000000000004, 90.0)\n\n\n\n\n\n"},{"title":"Base.acos","page":"Mathematics","location":"base/math.html#Base.acos-Tuple{Number}","category":"method","text":"acos(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.atan","page":"Mathematics","location":"base/math.html#Base.atan-Tuple{Number}","category":"method","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one real argument, this is the angle in radians between the positive x-axis and the point (1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the point (x, y), returning a value in the interval -pi pi. This corresponds to a standard atan2 function. Note that by convention atan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\nSee also atand for degrees.\n\nExamples\n\njulia> rad2deg(atan(-1/√3))\n-30.000000000000004\n\njulia> rad2deg(atan(-1, √3))\n-30.000000000000004\n\njulia> rad2deg(atan(1, -√3))\n150.0\n\n\n\n\n\n"},{"title":"Base.Math.asind","page":"Mathematics","location":"base/math.html#Base.Math.asind","category":"function","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acosd","page":"Mathematics","location":"base/math.html#Base.Math.acosd","category":"function","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.atand","page":"Mathematics","location":"base/math.html#Base.Math.atand","category":"function","text":"atand(y::T) where T -> float(T)\natand(y::T, x::S) where {T,S} -> promote_type(T,S)\natand(y::AbstractMatrix{T}) where T -> AbstractMatrix{Complex{float(T)}}\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\nReturn a NaN if isnan(y) or isnan(x). The returned NaN is either a T in the single argument version, or a promote_type(T,S) in the two argument version.\n\ncompat: Julia 1.7\nThe one-argument method supports square matrix arguments as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.Math.sec","page":"Mathematics","location":"base/math.html#Base.Math.sec-Tuple{Number}","category":"method","text":"sec(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csc","page":"Mathematics","location":"base/math.html#Base.Math.csc-Tuple{Number}","category":"method","text":"csc(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cot","page":"Mathematics","location":"base/math.html#Base.Math.cot-Tuple{Number}","category":"method","text":"cot(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.secd","page":"Mathematics","location":"base/math.html#Base.Math.secd","category":"function","text":"secd(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cscd","page":"Mathematics","location":"base/math.html#Base.Math.cscd","category":"function","text":"cscd(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cotd","page":"Mathematics","location":"base/math.html#Base.Math.cotd","category":"function","text":"cotd(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.asec","page":"Mathematics","location":"base/math.html#Base.Math.asec-Tuple{Number}","category":"method","text":"asec(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse secant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acsc","page":"Mathematics","location":"base/math.html#Base.Math.acsc-Tuple{Number}","category":"method","text":"acsc(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosecant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acot","page":"Mathematics","location":"base/math.html#Base.Math.acot-Tuple{Number}","category":"method","text":"acot(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cotangent of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.asecd","page":"Mathematics","location":"base/math.html#Base.Math.asecd","category":"function","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acscd","page":"Mathematics","location":"base/math.html#Base.Math.acscd","category":"function","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acotd","page":"Mathematics","location":"base/math.html#Base.Math.acotd","category":"function","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sech","page":"Mathematics","location":"base/math.html#Base.Math.sech-Tuple{Number}","category":"method","text":"sech(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic secant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csch","page":"Mathematics","location":"base/math.html#Base.Math.csch-Tuple{Number}","category":"method","text":"csch(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cosecant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.coth","page":"Mathematics","location":"base/math.html#Base.Math.coth-Tuple{Number}","category":"method","text":"coth(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cotangent of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.asinh","page":"Mathematics","location":"base/math.html#Base.asinh-Tuple{Number}","category":"method","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n"},{"title":"Base.acosh","page":"Mathematics","location":"base/math.html#Base.acosh-Tuple{Number}","category":"method","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n"},{"title":"Base.atanh","page":"Mathematics","location":"base/math.html#Base.atanh-Tuple{Number}","category":"method","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.asech","page":"Mathematics","location":"base/math.html#Base.Math.asech-Tuple{Number}","category":"method","text":"asech(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic secant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acsch","page":"Mathematics","location":"base/math.html#Base.Math.acsch-Tuple{Number}","category":"method","text":"acsch(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cosecant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acoth","page":"Mathematics","location":"base/math.html#Base.Math.acoth-Tuple{Number}","category":"method","text":"acoth(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cotangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.sinc","page":"Mathematics","location":"base/math.html#Base.Math.sinc","category":"function","text":"sinc(x::T) where {T <: Number} -> float(T)\n\nCompute normalized sinc function operatornamesinc(x) = sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\nReturn a T(NaN) if isnan(x).\n\nSee also cosc, its derivative.\n\n\n\n\n\n"},{"title":"Base.Math.cosc","page":"Mathematics","location":"base/math.html#Base.Math.cosc","category":"function","text":"cosc(x::T) where {T <: Number} -> float(T)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if x = 0. This is the derivative of sinc(x).\n\nReturn a T(NaN) if isnan(x).\n\nSee also sinc.\n\n\n\n\n\n"},{"title":"Base.Math.deg2rad","page":"Mathematics","location":"base/math.html#Base.Math.deg2rad","category":"function","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nSee also rad2deg, sind, pi.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n"},{"title":"Base.Math.rad2deg","page":"Mathematics","location":"base/math.html#Base.Math.rad2deg","category":"function","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nSee also deg2rad.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n"},{"title":"Base.Math.hypot","page":"Mathematics","location":"base/math.html#Base.Math.hypot","category":"function","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in: An Improved Algorithm for hypot(a,b) by Carlos F. Borges The article is available online at arXiv at the link   https://arxiv.org/abs/1904.09481\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nSee also norm in the LinearAlgebra standard library.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\njulia> using LinearAlgebra\n\njulia> norm([a, a, a, a]) == hypot(a, a, a, a)\ntrue\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number}","category":"method","text":"log(x)\n\nCompute the natural logarithm of x.\n\nThrow a DomainError for negative Real arguments. Use Complex arguments to obtain Complex results.\n\nnote: Branch cut\nlog has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also ℯ, log1p, log2, log10.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(-3 + 0im)\n1.0986122886681098 + 3.141592653589793im\n\njulia> log(-3 - 0.0im)\n1.0986122886681098 - 3.141592653589793im\n\njulia> log.(exp.(-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number, Number}","category":"method","text":"log(b,x)\n\nCompute the base b logarithm of x. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will typically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n"},{"title":"Base.log2","page":"Mathematics","location":"base/math.html#Base.log2","category":"function","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throw a DomainError for negative Real arguments.\n\nSee also: exp2, ldexp, ispow2.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nlog2 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log2(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\njulia> log2.(2.0 .^ (-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log10","page":"Mathematics","location":"base/math.html#Base.log10","category":"function","text":"log10(x)\n\nCompute the logarithm of x to base 10. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nlog10 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log10(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.log1p","page":"Mathematics","location":"base/math.html#Base.log1p","category":"function","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throw a DomainError for Real arguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p was called with a real argument < -1 but will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.Math.frexp","page":"Mathematics","location":"base/math.html#Base.Math.frexp","category":"function","text":"frexp(val)\n\nReturn (x,exp) such that x has a magnitude in the interval 12 1) or 0, and val is equal to x times 2^exp.\n\nSee also significand, exponent, ldexp.\n\nExamples\n\njulia> frexp(6.0)\n(0.75, 3)\n\njulia> significand(6.0), exponent(6.0)  # interval [1, 2) instead\n(1.5, 2)\n\njulia> frexp(0.0), frexp(NaN), frexp(-Inf)  # exponent would give an error\n((0.0, 0), (NaN, 0), (-Inf, 0))\n\n\n\n\n\n"},{"title":"Base.exp","page":"Mathematics","location":"base/math.html#Base.exp-Tuple{Float64}","category":"method","text":"exp(x)\n\nCompute the natural base exponential of x, in other words ℯ^x.\n\nSee also exp2, exp10 and cis.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\njulia> exp(im * pi) ≈ cis(pi)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp2","page":"Mathematics","location":"base/math.html#Base.exp2","category":"function","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nSee also ldexp, <<.\n\nExamples\n\njulia> exp2(5)\n32.0\n\njulia> 2^5\n32\n\njulia> exp2(63) > typemax(Int)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp10","page":"Mathematics","location":"base/math.html#Base.exp10","category":"function","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\njulia> 10^2\n100\n\n\n\n\n\n"},{"title":"Base.Math.ldexp","page":"Mathematics","location":"base/math.html#Base.Math.ldexp","category":"function","text":"ldexp(x, n)\n\nCompute x times 2^n.\n\nSee also frexp, exponent.\n\nExamples\n\njulia> ldexp(5.0, 2)\n20.0\n\n\n\n\n\n"},{"title":"Base.Math.modf","page":"Mathematics","location":"base/math.html#Base.Math.modf","category":"function","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts have the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n"},{"title":"Base.expm1","page":"Mathematics","location":"base/math.html#Base.expm1","category":"function","text":"expm1(x)\n\nAccurately compute e^x-1. It avoids the loss of precision involved in the direct evaluation of exp(x) - 1 for small values of x.\n\nExamples\n\njulia> expm1(1e-16)\n1.0e-16\n\njulia> exp(1e-16) - 1\n0.0\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round","category":"function","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type T, or of the same type of x if no T is provided. An InexactError will be thrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits after the decimal place (or before if digits is negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of significant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer. Note that round may give incorrect results if the global rounding mode is changed (see rounding).\n\nWhen rounding to a floating point type, will round to integers representable by that type (and Inf) rather than true integers. Inf is treated as one ulp greater than the floatmax(T) for purposes of determining \"nearest\", similar to convert.\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\njulia> round(Float16, typemax(UInt128))\nInf16\n\njulia> floor(Float16, typemax(UInt128))\nFloat16(6.55e4)\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when operating on binary floating point numbers. For example, the Float64 value represented by 1.15 is actually less than 1.15, yet will be rounded to 1.2. For example:julia> x = 1.15\n1.15\n\njulia> big(1.15)\n1.149999999999999911182158029987476766109466552734375\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundingMode","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundingMode","category":"type","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via rounding/setrounding functions), or as optional arguments for rounding to the nearest integer (via the round function).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero\nRoundUp\nRoundDown\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearest","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearest","category":"constant","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesAway","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesAway","category":"constant","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++ round behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesUp","category":"constant","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript round behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundToZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundToZero","category":"constant","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundFromZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundFromZero","category":"constant","text":"RoundFromZero\n\nRounds away from zero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundUp","category":"constant","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundDown","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundDown","category":"constant","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round-Tuple{Complex{<:AbstractFloat}, RoundingMode, RoundingMode}","category":"method","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=0, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z, breaking ties using the specified RoundingModes. The first RoundingMode is used for rounding the real components while the second is used for rounding the imaginary components.\n\nRoundingModeReal and RoundingModeImaginary default to RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\nExamples\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\njulia> round(3.14 + 4.5im, RoundUp, RoundNearestTiesUp)\n4.0 + 5.0im\n\njulia> round(3.14159 + 4.512im; digits = 1)\n3.1 + 4.5im\n\njulia> round(3.14159 + 4.512im; sigdigits = 3)\n3.14 + 4.51im\n\n\n\n\n\n"},{"title":"Base.ceil","page":"Mathematics","location":"base/math.html#Base.ceil","category":"function","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or equal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the ceiled value is not representable as a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support ceil for a new type, define Base.round(x::NewType, ::RoundingMode{:Up}).\n\n\n\n\n\n"},{"title":"Base.floor","page":"Mathematics","location":"base/math.html#Base.floor","category":"function","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or equal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the floored value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support floor for a new type, define Base.round(x::NewType, ::RoundingMode{:Down}).\n\n\n\n\n\n"},{"title":"Base.trunc","page":"Mathematics","location":"base/math.html#Base.trunc","category":"function","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value is less than or equal to the absolute value of x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the truncated value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support trunc for a new type, define Base.round(x::NewType, ::RoundingMode{:ToZero}).\n\nSee also: %, floor, unsigned, unsafe_trunc.\n\nExamples\n\njulia> trunc(2.22)\n2.0\n\njulia> trunc(-2.22, digits=1)\n-2.2\n\njulia> trunc(Int, -2.22)\n-2\n\n\n\n\n\n"},{"title":"Base.unsafe_trunc","page":"Mathematics","location":"base/math.html#Base.unsafe_trunc","category":"function","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is less than or equal to the absolute value of x. If the value is not representable by T, an arbitrary value will be returned. See also trunc.\n\nExamples\n\njulia> unsafe_trunc(Int, -2.2)\n-2\n\njulia> unsafe_trunc(Int, NaN) isa Int\ntrue\n\n\n\n\n\n"},{"title":"Base.min","page":"Mathematics","location":"base/math.html#Base.min","category":"function","text":"min(x, y, ...)\n\nReturn the minimum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\njulia> min(4, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.max","page":"Mathematics","location":"base/math.html#Base.max","category":"function","text":"max(x, y, ...)\n\nReturn the maximum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\njulia> max(5, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.minmax","page":"Mathematics","location":"base/math.html#Base.minmax","category":"function","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)).\n\nSee also extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n"},{"title":"Base.clamp","page":"Mathematics","location":"base/math.html#Base.clamp","category":"function","text":"clamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nSee also trunc.\n\nExamples\n\njulia> clamp(200, Int8)\n127\n\njulia> clamp(-200, Int8)\n-128\n\njulia> trunc(Int, 4pi^2)\n39\n\n\n\n\n\nclamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments are promoted to a common type.\n\nSee also clamp!, min, max.\n\ncompat: Julia 1.3\nmissing as the first argument requires at least Julia 1.3.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10)], 2.0, 9.0)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11, 8, 5], 10, 6)  # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\n"},{"title":"Base.clamp!","page":"Mathematics","location":"base/math.html#Base.clamp!","category":"function","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place. See also clamp.\n\ncompat: Julia 1.3\nmissing entries in array require at least Julia 1.3.\n\nExamples\n\njulia> row = collect(-4:4)';\n\njulia> clamp!(row, 0, Inf)\n1×9 adjoint(::Vector{Int64}) with eltype Int64:\n 0  0  0  0  0  1  2  3  4\n\njulia> clamp.((-4:4)', 0, Inf)\n1×9 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0  1.0  2.0  3.0  4.0\n\n\n\n\n\n"},{"title":"Base.abs","page":"Mathematics","location":"base/math.html#Base.abs","category":"function","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur, resulting in the return of a negative value. This overflow occurs only when abs is applied to the minimum representable value of a signed integer. That is, when x == typemin(typeof(x)), abs(x) == x < 0, not -x as might be expected.\n\nSee also: abs2, unsigned, sign.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs.(Int8[-128 -127 -126 0 126 127])  # overflow at typemin(Int8)\n1×6 Matrix{Int8}:\n -128  127  126  0  126  127\n\njulia> maximum(abs, [1, -2, 3, -4])\n4\n\n\n\n\n\n"},{"title":"Base.Checked","page":"Mathematics","location":"base/math.html#Base.Checked","category":"module","text":"Checked\n\nThe Checked module provides arithmetic functions for the built-in signed and unsigned Integer types which throw an error when an overflow occurs. They are named like checked_sub, checked_div, etc. In addition, add_with_overflow, sub_with_overflow, mul_with_overflow return both the unchecked results and a boolean value denoting the presence of an overflow.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_abs","page":"Mathematics","location":"base/math.html#Base.Checked.checked_abs","category":"function","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_neg","page":"Mathematics","location":"base/math.html#Base.Checked.checked_neg","category":"function","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_add","page":"Mathematics","location":"base/math.html#Base.Checked.checked_add","category":"function","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_sub","page":"Mathematics","location":"base/math.html#Base.Checked.checked_sub","category":"function","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mul","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mul","category":"function","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_div","page":"Mathematics","location":"base/math.html#Base.Checked.checked_div","category":"function","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_rem","page":"Mathematics","location":"base/math.html#Base.Checked.checked_rem","category":"function","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_fld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_fld","category":"function","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mod","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mod","category":"function","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_cld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_cld","category":"function","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_pow","page":"Mathematics","location":"base/math.html#Base.Checked.checked_pow","category":"function","text":"Base.checked_pow(x, y)\n\nCalculates ^(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.add_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.add_with_overflow","category":"function","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.sub_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.sub_with_overflow","category":"function","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.mul_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.mul_with_overflow","category":"function","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.abs2","page":"Mathematics","location":"base/math.html#Base.abs2","category":"function","text":"abs2(x)\n\nSquared absolute value of x.\n\nThis can be faster than abs(x)^2, especially for complex numbers where abs(x) requires a square root via hypot.\n\nSee also abs, conj, real.\n\nExamples\n\njulia> abs2(-3)\n9\n\njulia> abs2(3.0 + 4.0im)\n25.0\n\njulia> sum(abs2, [1+2im, 3+4im])  # LinearAlgebra.norm(x)^2\n30\n\n\n\n\n\n"},{"title":"Base.copysign","page":"Mathematics","location":"base/math.html#Base.copysign","category":"function","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n"},{"title":"Base.sign","page":"Mathematics","location":"base/math.html#Base.sign","category":"function","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\nSee also signbit, zero, copysign, flipsign.\n\nExamples\n\njulia> sign(-4.0)\n-1.0\n\njulia> sign(99)\n1\n\njulia> sign(-0.0)\n-0.0\n\njulia> sign(0 + im)\n0.0 + 1.0im\n\n\n\n\n\n"},{"title":"Base.signbit","page":"Mathematics","location":"base/math.html#Base.signbit","category":"function","text":"signbit(x)\n\nReturn true if the value of the sign of x is negative, otherwise false.\n\nSee also sign and copysign.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n"},{"title":"Base.flipsign","page":"Mathematics","location":"base/math.html#Base.flipsign","category":"function","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n"},{"title":"Base.sqrt","page":"Mathematics","location":"base/math.html#Base.sqrt-Tuple{Number}","category":"method","text":"sqrt(x)\n\nReturn sqrtx.\n\nThrow a DomainError for negative Real arguments. Use Complex negative arguments instead to obtain a Complex result.\n\nThe prefix operator √ is equivalent to sqrt.\n\nnote: Branch cut\nsqrt has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also: hypot.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\njulia> sqrt(-81 - 0.0im)  # -0.0im is below the branch cut\n0.0 - 9.0im\n\njulia> .√(1:4)\n4-element Vector{Float64}:\n 1.0\n 1.4142135623730951\n 1.7320508075688772\n 2.0\n\n\n\n\n\n"},{"title":"Base.isqrt","page":"Mathematics","location":"base/math.html#Base.isqrt","category":"function","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n"},{"title":"Base.Math.cbrt","page":"Mathematics","location":"base/math.html#Base.Math.cbrt-Tuple{AbstractFloat}","category":"method","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted (returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n"},{"title":"Base.Math.fourthroot","page":"Mathematics","location":"base/math.html#Base.Math.fourthroot-Tuple{Number}","category":"method","text":"fourthroot(x)\n\nReturn the fourth root of x by applying sqrt twice successively.\n\n\n\n\n\n"},{"title":"Base.real","page":"Mathematics","location":"base/math.html#Base.real","category":"function","text":"real(A::AbstractArray)\n\nReturn an array containing the real part of each entry in array A.\n\nEquivalent to real.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> real([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 1\n 0\n 3\n\njulia> real(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n2\n\n\n\n\n\nreal(T::Type)\n\nReturn the type that represents the real part of a value of type T. e.g: for T == Complex{R}, returns R. Equivalent to typeof(real(zero(T))).\n\nExamples\n\njulia> real(Complex{Int})\nInt64\n\njulia> real(Float64)\nFloat64\n\n\n\n\n\nreal(z)\n\nReturn the real part of the complex number z.\n\nSee also: imag, reim, complex, isreal, Real.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\n"},{"title":"Base.imag","page":"Mathematics","location":"base/math.html#Base.imag","category":"function","text":"imag(A::AbstractArray)\n\nReturn an array containing the imaginary part of each entry in array A.\n\nEquivalent to imag.(A), except that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> imag([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 0\n 2\n 4\n\njulia> imag(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n-1\n\n\n\n\n\nimag(z)\n\nReturn the imaginary part of the complex number z.\n\nSee also: conj, reim, adjoint, angle.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\n"},{"title":"Base.reim","page":"Mathematics","location":"base/math.html#Base.reim","category":"function","text":"reim(A::AbstractArray)\n\nReturn a tuple of two arrays containing respectively the real and the imaginary part of each entry in A.\n\nEquivalent to (real.(A), imag.(A)), except that when eltype(A) <: Real A is returned without copying to represent the real part, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> reim([1, 2im, 3 + 4im])\n([1, 0, 3], [0, 2, 4])\n\njulia> reim(fill(2 - im))\n(fill(2), fill(-1))\n\n\n\n\n\nreim(z)\n\nReturn a tuple of the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\n"},{"title":"Base.conj","page":"Mathematics","location":"base/math.html#Base.conj","category":"function","text":"conj(A::AbstractArray)\n\nReturn an array containing the complex conjugate of each entry in array A.\n\nEquivalent to conj.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> conj([1, 2im, 3 + 4im])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 0 - 2im\n 3 - 4im\n\njulia> conj(fill(2 - im))\n0-dimensional Array{Complex{Int64}, 0}:\n2 + 1im\n\n\n\n\n\nconj(z)\n\nCompute the complex conjugate of a complex number z.\n\nSee also: angle, adjoint.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\n"},{"title":"Base.angle","page":"Mathematics","location":"base/math.html#Base.angle","category":"function","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nReturns a number -pi ≤ angle(z) ≤ pi, and is thus discontinuous along the negative real axis.\n\nSee also: atan, cis, rad2deg.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 + 1e-20im))\n180.0\n\njulia> rad2deg(angle(-1 - 1e-20im))\n-180.0\n\n\n\n\n\n"},{"title":"Base.cis","page":"Mathematics","location":"base/math.html#Base.cis","category":"function","text":"cis(x)\n\nMore efficient method for exp(im*x) by using Euler's formula: cos(x) + i sin(x) = exp(i x).\n\nSee also cispi, sincos, exp, angle.\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n"},{"title":"Base.cispi","page":"Mathematics","location":"base/math.html#Base.cispi","category":"function","text":"cispi(x)\n\nMore accurate method for cis(pi*x) (especially for large x).\n\nSee also cis, sincospi, exp, angle.\n\nExamples\n\njulia> cispi(10000)\n1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645954562 + 0.03055685464595456im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.binomial","page":"Mathematics","location":"base/math.html#Base.binomial","category":"function","text":"binomial(x::Number, k::Integer)\n\nThe generalized binomial coefficient, defined for k ≥ 0 by the polynomial\n\nfrac1k prod_j=0^k-1 (x - j)\n\nWhen k < 0 it returns zero.\n\nFor the case of integer x, this is equivalent to the ordinary integer binomial coefficient\n\nbinomnk = fracnk (n-k)\n\nFurther generalizations to non-integer k are mathematically possible, but involve the Gamma function and/or the beta function, which are not provided by the Julia standard library but are available in external packages such as SpecialFunctions.jl.\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\nbinomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in the polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nSee also factorial.\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.factorial","page":"Mathematics","location":"base/math.html#Base.factorial","category":"function","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an integer (promoted to at least 64 bits). Note that this may overflow if n is not small, but you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nSee also binomial.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.gcd","page":"Mathematics","location":"base/math.html#Base.gcd","category":"function","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero). The arguments may be integer and rational numbers.\n\na is a divisor of b if there exists an integer m such that ma=b.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6, 9)\n3\n\njulia> gcd(6, -9)\n3\n\njulia> gcd(6, 0)\n6\n\njulia> gcd(0, 0)\n0\n\njulia> gcd(1//3, 2//3)\n1//3\n\njulia> gcd(1//3, -2//3)\n1//3\n\njulia> gcd(1//3, 2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n"},{"title":"Base.lcm","page":"Mathematics","location":"base/math.html#Base.lcm","category":"function","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero). The arguments may be integer and rational numbers.\n\na is a multiple of b if there exists an integer m such that a=mb.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2, 3)\n6\n\njulia> lcm(-2, 3)\n6\n\njulia> lcm(0, 3)\n0\n\njulia> lcm(0, 0)\n0\n\njulia> lcm(1//3, 2//3)\n2//3\n\njulia> lcm(1//3, -2//3)\n2//3\n\njulia> lcm(1//3, 2)\n2//1\n\njulia> lcm(1, 3, 5, 7)\n105\n\n\n\n\n\n"},{"title":"Base.gcdx","page":"Mathematics","location":"base/math.html#Base.gcdx","category":"function","text":"gcdx(a, b...)\n\nComputes the greatest common (positive) divisor of a and b and their Bézout coefficients, i.e. the integer coefficients u and v that satisfy u*a + v*b = d = gcd(a b). gcdx(a b) returns (d u v).\n\nFor more arguments than two, i.e., gcdx(a, b, c, ...) the Bézout coefficients are computed recursively, returning a solution (d, u, v, w, ...) to u*a + v*b + w*c +  = d = gcd(a b c ).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\ncompat: Julia 1.12\nMore or fewer arguments than two require Julia 1.12 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\njulia> gcdx(15, 12, 20)\n(1, 7, -7, -1)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal Bézout coefficients that are computed by the extended Euclidean algorithm. (Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.) For signed integers, these coefficients u and v are minimal in the sense that u  bd and v  ad. Furthermore, the signs of u and v are chosen so that d is positive. For unsigned integers, the coefficients u and v might be near their typemax, and the identity then holds only via the unsigned integers' modulo arithmetic.\n\n\n\n\n\n"},{"title":"Base.ispow2","page":"Mathematics","location":"base/math.html#Base.ispow2","category":"function","text":"ispow2(n::Number)::Bool\n\nTest whether n is an integer power of two.\n\nSee also count_ones, prevpow, nextpow.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n"},{"title":"Base.nextpow","page":"Mathematics","location":"base/math.html#Base.nextpow","category":"function","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be greater than 1, and x must be greater than 0.\n\nSee also prevpow.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.prevpow","page":"Mathematics","location":"base/math.html#Base.prevpow","category":"function","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer. a must be greater than 1, and x must not be less than 1.\n\nSee also nextpow, isqrt.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.nextprod","page":"Mathematics","location":"base/math.html#Base.nextprod","category":"function","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers p_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.invmod","page":"Mathematics","location":"base/math.html#Base.invmod","category":"function","text":"invmod(n::Integer, T) where {T <: Base.BitInteger}\ninvmod(n::T) where {T <: Base.BitInteger}\n\nCompute the modular inverse of n in the integer ring of type T, i.e. modulo 2^N where N = 8*sizeof(T) (e.g. N = 32 for Int32). In other words, these methods satisfy the following identities:\n\nn * invmod(n) == 1\n(n * invmod(n, T)) % T == 1\n(n % T) * invmod(n, T) == 1\n\nNote that * here is modular multiplication in the integer ring, T.  This will throw an error if n is even, because then it is not relatively prime with 2^N and thus has no such inverse.\n\nSpecifying the modulus implied by an integer type as an explicit value is often inconvenient since the modulus is by definition too big to be represented by the type.\n\nThe modular inverse is computed much more efficiently than the general case using the algorithm described in https://arxiv.org/pdf/2204.04342.pdf.\n\ncompat: Julia 1.11\nThe invmod(n) and invmod(n, T) methods require Julia 1.11 or later.\n\n\n\n\n\ninvmod(n::Integer, m::Integer)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m, and div(ym) = 0. This will throw an error if m = 0, or if gcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2, 5)\n3\n\njulia> invmod(2, 3)\n2\n\njulia> invmod(5, 6)\n5\n\n\n\n\n\n"},{"title":"Base.powermod","page":"Mathematics","location":"base/math.html#Base.powermod","category":"function","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n"},{"title":"Base.ndigits","page":"Mathematics","location":"base/math.html#Base.ndigits","category":"function","text":"ndigits(n::Integer; base::Integer=10, pad::Integer=1)\n\nCompute the number of digits in integer n written in base base (base must not be in [-1, 0, 1]), optionally padded with zeros to a specified size (the result will never be less than pad).\n\nSee also digits, count_ones.\n\nExamples\n\njulia> ndigits(0)\n1\n\njulia> ndigits(12345)\n5\n\njulia> ndigits(1022, base=16)\n3\n\njulia> string(1022, base=16)\n\"3fe\"\n\njulia> ndigits(123, pad=5)\n5\n\njulia> ndigits(-123)\n3\n\n\n\n\n\n"},{"title":"Base.add_sum","page":"Mathematics","location":"base/math.html#Base.add_sum","category":"function","text":"Base.add_sum(x, y)\n\nThe reduction operator used in sum. The main difference from + is that small integers are promoted to Int/UInt.\n\n\n\n\n\n"},{"title":"Base.uabs","page":"Mathematics","location":"base/math.html#Base.uabs","category":"function","text":"Base.uabs(x::Integer)\n\nReturn the absolute value of x, possibly returning a different type should the operation be susceptible to overflow. This typically arises when x is a two's complement signed integer, so that abs(typemin(x)) == typemin(x) < 0, in which case the result of uabs(x) will be an unsigned integer of the same size.\n\n\n\n\n\n"},{"title":"Base.widemul","page":"Mathematics","location":"base/math.html#Base.widemul","category":"function","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nSee also promote, Base.add_sum.\n\nExamples\n\njulia> widemul(Float32(3.0), 4.0) isa BigFloat\ntrue\n\njulia> typemax(Int8) * typemax(Int8)\n1\n\njulia> widemul(typemax(Int8), typemax(Int8))  # == 127^2\n16129\n\n\n\n\n\n"},{"title":"Base.Math.evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.evalpoly","category":"function","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...; that is, the coefficients are given in ascending order by power of x. Loops are unrolled at compile time if the number of coefficients is statically known, i.e. when p is a Tuple. This function generates efficient code using Horner's method if x is real, or using a Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExamples\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n"},{"title":"Base.Math.@evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.@evalpoly","category":"macro","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...; that is, the coefficients are given in ascending order by power of z.  This macro expands to efficient inline code that uses either Horner's method or, for complex z, a more efficient Goertzel-like algorithm.\n\nSee also evalpoly.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n"},{"title":"Base.FastMath.@fastmath","page":"Mathematics","location":"base/math.html#Base.FastMath.@fastmath","category":"macro","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that may violate strict IEEE semantics. This allows the fastest possible operation, but results are undefined – be careful when doing this, as it may change numerical results.\n\nThis sets the LLVM Fast-Math flags, and corresponds to the -ffast-math option in clang. See the notes on performance annotations for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html#scope-of-variables","category":"section","text":"The scope of a variable is the region of code within which a variable is accessible. Variable scoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have arguments called x without the two x's referring to the same thing. Similarly, there are many other cases where different blocks of code can use the same name without referring to the same thing. The rules for when the same variable name does or doesn't refer to the same thing are called scope rules; this section spells them out in detail.\n\nCertain constructs in the language introduce scope blocks, which are regions of code that are eligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary set of source lines; instead, it will always line up with one of these blocks. There are two main types of scopes in Julia, global scope and local scope. The latter can be nested. There is also a distinction in Julia between constructs which introduce a \"hard scope\" and those which only introduce a \"soft scope\", which affects whether shadowing a global variable by the same name is allowed or not.\n\ninfo: Summary\nVariables defined in global scope may be undefined in inner local scopes, depending on where the code is run, in order to balance safety and convenience. The hard and soft local scoping rules define the interplay between global and local variables.However, variables defined only in local scope behave consistently in all contexts. If the variable is already defined, it will be reused. If the variable is not defined, it will be made available to the current and inner scopes (but not outer scopes).\n\ntip: A Common Confusion\nIf you run into an unexpectedly undefined variable,# Print the numbers 1 through 5\ni = 0\nwhile i < 5\n    i += 1     # ERROR: UndefVarError: `i` not defined\n    println(i)\nenda simple fix is to change all global variable definitions into local definitions by wrapping the code in a let block or function.# Print the numbers 1 through 5\nlet i = 0\n    while i < 5\n        i += 1     # Now outer `i` is defined in the inner scope of the while loop\n        println(i)\n    end\nendThis is a common source of confusion when writing procedural scripts, but it becomes a non-issue if code is moved inside functions or executed interactively in the REPL.See also the global and local keywords to explicitly achieve any desired scoping behavior."},{"title":"Scope Constructs","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-scope-table","category":"section","text":"The constructs introducing scope blocks are:\n\nConstruct Scope Type Introduced Scope Types Able to Contain Construct\nmodule, baremodule global global\nstruct local (hard) global\nmacro local (hard) global\nfor, while, try local (soft) global, local\nfunction, do, let, comprehensions, generators local (hard) global, local\n\nNotably missing from this table are begin blocks and if blocks which do not introduce new scopes. The three types of scopes follow somewhat different rules which will be explained below.\n\nJulia uses lexical scoping, meaning that a function's scope does not inherit from its caller's scope, but from the scope in which the function was defined. For example, in the following code the x inside foo refers to the x in the global scope of its module Bar:\n\njulia> module Bar\n           x = 1\n           foo() = x\n       end;\n\nand not a x in the scope where foo is used:\n\njulia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1\n\nThus lexical scope means that what a variable in a particular piece of code refers to can be deduced from the code in which it appears alone and does not depend on how the program executes. A scope nested inside another scope can \"see\" variables in all the outer scopes in which it is contained. Outer scopes, on the other hand, cannot see variables in inner scopes."},{"title":"Global Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Global-Scope","category":"section","text":"Each module introduces a new global scope, separate from the global scope of all other modules—there is no all-encompassing global scope. Modules can introduce variables of other modules into their scope through the using or import statements or through qualified access using the dot-notation, i.e. each module is a so-called namespace as well as a first-class data structure associating names with values.\n\nIf a top-level expression contains a variable declaration with keyword local, then that variable is not accessible outside that expression. The variable inside the expression does not affect global variables of the same name. An example is to declare local x in a begin or if block at the top-level:\n\njulia> x = 1\n       begin\n           local x = 0\n           @show x\n       end\n       @show x;\nx = 0\nx = 1\n\nNote that the interactive prompt (aka REPL) is in the global scope of the module Main."},{"title":"Local Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#local-scope","category":"section","text":"A new local scope is introduced by most code blocks (see above table for a complete list). If such a block is syntactically nested inside of another local scope, the scope it creates is nested inside of all the local scopes that it appears within, which are all ultimately nested inside of the global scope of the module in which the code is evaluated. Variables in outer scopes are visible from any scope they contain — meaning that they can be read and written in inner scopes — unless there is a variable with the same name that \"shadows\" the outer variable of the same name. This is true even if the outer local is declared after (in the sense of textually below) an inner block. When we say that a variable \"exists\" in a given scope, this means that a variable by that name exists in any of the scopes that the current scope is nested inside of, including the current one. If a variable's value is used in a local scope, but nothing with its name exists in this scope, it is assumed to be a global.\n\nSome programming languages require explicitly declaring new variables before using them. Explicit declaration works in Julia too: in any local scope, writing local x declares a new local variable in that scope, regardless of whether there is already a variable named x in an outer scope or not. Declaring each new variable like this is somewhat verbose and tedious, however, so Julia, like many other languages, considers assignment to a variable name that doesn't already exist to implicitly declare that variable. If the current scope is global, the new variable is global; if the current scope is local, the new variable is local to the innermost local scope and will be visible inside of that scope but not outside of it. If you assign to an existing local, it always updates that existing local: you can only shadow a local by explicitly declaring a new local in a nested scope with the local keyword. In particular, this applies to variables assigned in inner functions, which may surprise users coming from Python where assignment in an inner function creates a new local unless the variable is explicitly declared to be non-local.\n\nMostly this is pretty intuitive, but as with many things that behave intuitively, the details are more subtle than one might naïvely imagine.\n\nWhen x = <value> occurs in a local scope, Julia applies the following rules to decide what the expression means based on where the assignment expression occurs and what x already refers to at that location:\n\nExisting local: If x is already a local variable, then the existing local x is assigned;\nHard scope: If x is not already a local variable and assignment occurs inside of any hard scope construct (i.e. within a let block, function, struct or macro body, comprehension, or generator), a new local named x is created in the scope of the assignment;\nSoft scope: If x is not already a local variable and all of the scope constructs containing the assignment are soft scopes (loops, try/catch blocks), the behavior depends on whether the global variable x is defined:\nif global x is undefined, a new local named x is created in the scope of the assignment;\nif global x is defined, the assignment is considered ambiguous:\nin non-interactive contexts (files, eval), an ambiguity warning is printed and a new local is created;\nin interactive contexts (REPL, notebooks), the global variable x is assigned.\n\nYou may note that in non-interactive contexts the hard and soft scope behaviors are identical except that a warning is printed when an implicitly local variable (i.e. not declared with local x) shadows a global. In interactive contexts, the rules follow a more complex heuristic for the sake of convenience. This is covered in depth in examples that follow.\n\nNow that you know the rules, let's look at some examples. Each example is assumed to be evaluated in a fresh REPL session so that the only globals in each snippet are the ones that are assigned in that block of code.\n\nWe'll begin with a nice and clear-cut situation—assignment inside of a hard scope, in this case a function body, when no local variable by that name already exists:\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\nERROR: UndefVarError: `x` not defined in `Main`\n\nInside of the greet function, the assignment x = \"hello\" causes x to be a new local variable in the function's scope. There are two relevant facts: the assignment occurs in local scope and there is no existing local x variable. Since x is local, it doesn't matter if there is a global named x or not. Here for example we define x = 123 before defining and calling greet:\n\njulia> x = 123 # global\n123\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\n123\n\nSince the x in greet is local, the value (or lack thereof) of the global x is unaffected by calling greet. The hard scope rule doesn't care whether a global named x exists or not: assignment to x in a hard scope is local (unless x is declared global).\n\nThe next clear cut situation we'll consider is when there is already a local variable named x, in which case x = <value> always assigns to this existing local x. This is true whether the assignment occurs in the same local scope, an inner local scope in the same function body, or in the body of a function nested inside of another function, also known as a closure.\n\nWe'll use the sum_to function, which computes the sum of integers from one up to n, as an example:\n\nfunction sum_to(n)\n    s = 0 # new local\n    for i = 1:n\n        s = s + i # assign existing local\n    end\n    return s # same local\nend\n\nAs in the previous example, the first assignment to s at the top of sum_to causes s to be a new local variable in the body of the function. The for loop has its own inner local scope within the function scope. At the point where s = s + i occurs, s is already a local variable, so the assignment updates the existing s instead of creating a new local. We can test this out by calling sum_to in the REPL:\n\njulia> function sum_to(n)\n           s = 0 # new local\n           for i = 1:n\n               s = s + i # assign existing local\n           end\n           return s # same local\n       end\nsum_to (generic function with 1 method)\n\njulia> sum_to(10)\n55\n\njulia> s # global\nERROR: UndefVarError: `s` not defined in `Main`\n\nSince s is local to the function sum_to, calling the function has no effect on the global variable s. We can also see that the update s = s + i in the for loop must have updated the same s created by the initialization s = 0 since we get the correct sum of 55 for the integers 1 through 10.\n\nLet's dig into the fact that the for loop body has its own scope for a second by writing a slightly more verbose variation which we'll call sum_to_def, in which we save the sum s + i in a variable t before updating s:\n\njulia> function sum_to_def(n)\n           s = 0 # new local\n           for i = 1:n\n               t = s + i # new local `t`\n               s = t # assign existing local `s`\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def (generic function with 1 method)\n\njulia> sum_to_def(10)\n(55, false)\n\nThis version returns s as before but it also uses the @isdefined macro to return a boolean indicating whether there is a local variable named t defined in the function's outermost local scope. As you can see, there is no t defined outside of the for loop body. This is because of the hard scope rule again: since the assignment to t occurs inside of a function, which introduces a hard scope, the assignment causes t to become a new local variable in the local scope where it appears, i.e. inside of the loop body. Even if there were a global named t, it would make no difference—the hard scope rule isn't affected by anything in global scope.\n\nNote that the local scope of a for loop body is no different from the local scope of an inner function. This means that we could rewrite this example so that the loop body is implemented as a call to an inner helper function and it behaves the same way:\n\njulia> function sum_to_def_closure(n)\n           function loop_body(i)\n               t = s + i # new local `t`\n               s = t # assign same local `s` as below\n           end\n           s = 0 # new local\n           for i = 1:n\n               loop_body(i)\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def_closure (generic function with 1 method)\n\njulia> sum_to_def_closure(10)\n(55, false)\n\nThis example illustrates a couple of key points:\n\nInner function scopes are just like any other nested local scope. In particular, if a variable is already a local outside of an inner function and you assign to it in the inner function, the outer local variable is updated.\nIt doesn't matter if the definition of an outer local happens below where it is updated, the rule remains the same. The entire enclosing local scope is parsed and its locals determined before inner local meanings are resolved.\n\nThis design means that you can generally move code in or out of an inner function without changing its meaning, which facilitates a number of common idioms in the language using closures (see do blocks).\n\nLet's move onto some more ambiguous cases covered by the soft scope rule. We'll explore this by extracting the bodies of the greet and sum_to_def functions into soft scope contexts. First, let's put the body of greet in a for loop—which is soft, rather than hard—and evaluate it in the REPL:\n\njulia> for i = 1:3\n           x = \"hello\" # new local\n           println(x)\n       end\nhello\nhello\nhello\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\nSince the global x is not defined when the for loop is evaluated, the first clause of the soft scope rule applies and x is created as local to the for loop and therefore global x remains undefined after the loop executes. Next, let's consider the body of sum_to_def extracted into global scope, fixing its argument to n = 10\n\ns = 0\nfor i = 1:10\n    t = s + i\n    s = t\nend\ns\n@isdefined(t)\n\nWhat does this code do? Hint: it's a trick question. The answer is \"it depends.\" If this code is entered interactively, it behaves the same way it does in a function body. But if the code appears in a file, it  prints an ambiguity warning and throws an undefined variable error. Let's see it working in the REPL first:\n\njulia> s = 0 # global\n0\n\njulia> for i = 1:10\n           t = s + i # new local `t`\n           s = t # assign global `s`\n       end\n\njulia> s # global\n55\n\njulia> @isdefined(t) # global\nfalse\n\nThe REPL approximates being in the body of a function by deciding whether assignment inside the loop assigns to a global or creates new local based on whether a global variable by that name is defined or not. If a global by the name exists, then the assignment updates it. If no global exists, then the assignment creates a new local variable. In this example we see both cases in action:\n\nThere is no global named t, so t = s + i creates a new t that is local to the for loop;\nThere is a global named s, so s = t assigns to it.\n\nThe second fact is why execution of the loop changes the global value of s and the first fact is why t is still undefined after the loop executes. Now, let's try evaluating this same code as though it were in a file instead:\n\njulia> code = \"\"\"\n       s = 0 # global\n       for i = 1:10\n           t = s + i # new local `t`\n           s = t # new local `s` with warning\n       end\n       s, # global\n       @isdefined(t) # global\n       \"\"\";\n\njulia> include_string(Main, code)\n┌ Warning: Assignment to `s` in soft scope is ambiguous because a global variable by the same name exists: `s` will be treated as a new local. Disambiguate by using `local s` to suppress this warning or `global s` to assign to the existing global variable.\n└ @ string:4\nERROR: LoadError: UndefVarError: `s` not defined in local scope\n\nHere we use include_string, to evaluate code as though it were the contents of a file. We could also save code to a file and then call include on that file—the result would be the same. As you can see, this behaves quite different from evaluating the same code in the REPL. Let's break down what's happening here:\n\nglobal s is defined with the value 0 before the loop is evaluated\nthe assignment s = t occurs in a soft scope—a for loop outside of any function body or other hard scope construct\ntherefore the second clause of the soft scope rule applies, and the assignment is ambiguous so a warning is emitted\nexecution continues, making s local to the for loop body\nsince s is local to the for loop, it is undefined when t = s + i is evaluated, causing an error\nevaluation stops there, but if it got to s and @isdefined(t), it would return 0 and false.\n\nThis demonstrates some important aspects of scope: in a scope, each variable can only have one meaning, and that meaning is determined regardless of the order of expressions. The presence of the expression s = t in the loop causes s to be local to the loop, which means that it is also local when it appears on the right hand side of t = s + i, even though that expression appears first and is evaluated first. One might imagine that the s on the first line of the loop could be global while the s on the second line of the loop is local, but that's not possible since the two lines are in the same scope block and each variable can only mean one thing in a given scope."},{"title":"On Soft Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#on-soft-scope","category":"section","text":"We have now covered all the local scope rules, but before wrapping up this section, perhaps a few words should be said about why the ambiguous soft scope case is handled differently in interactive and non-interactive contexts. There are two obvious questions one could ask:\n\nWhy doesn't it just work like the REPL everywhere?\nWhy doesn't it just work like in files everywhere? And maybe skip the warning?\n\nIn Julia ≤ 0.6, all global scopes did work like the current REPL: when x = <value> occurred in a loop (or try/catch, or struct body) but outside of a function body (or let block or comprehension), it was decided based on whether a global named x was defined or not whether x should be local to the loop. This behavior has the advantage of being intuitive and convenient since it approximates the behavior inside of a function body as closely as possible. In particular, it makes it easy to move code back and forth between a function body and the REPL when trying to debug the behavior of a function. However, it has some downsides. First, it's quite a complex behavior: many people over the years were confused about this behavior and complained that it was complicated and hard both to explain and understand. Fair point. Second, and arguably worse, is that it's bad for programming \"at scale.\" When you see a small piece of code in one place like this, it's quite clear what's going on:\n\ns = 0\nfor i = 1:10\n    s += i\nend\n\nObviously the intention is to modify the existing global variable s. What else could it mean? However, not all real world code is so short or so clear. We found that code like the following often occurs in the wild:\n\nx = 123\n\n# much later\n# maybe in a different file\n\nfor i = 1:10\n    x = \"hello\"\n    println(x)\nend\n\n# much later\n# maybe in yet another file\n# or maybe back in the first one where `x = 123`\n\ny = x + 234\n\nIt's far less clear what should happen here. Since x + \"hello\" is a method error, it seems probable that the intention is for x to be local to the for loop. But runtime values and what methods happen to exist cannot be used to determine the scopes of variables. With the Julia ≤ 0.6 behavior, it's especially concerning that someone might have written the for loop first, had it working just fine, but later when someone else adds a new global far away—possibly in a different file—the code suddenly changes meaning and either breaks noisily or, worse still, silently does the wrong thing. This kind of \"spooky action at a distance\" is something that good programming language designs should prevent.\n\nSo in Julia 1.0, we simplified the rules for scope: in any local scope, assignment to a name that wasn't already a local variable created a new local variable. This eliminated the notion of soft scope entirely as well as removing the potential for spooky action. We uncovered and fixed a significant number of bugs due to the removal of soft scope, vindicating the choice to get rid of it. And there was much rejoicing! Well, no, not really. Because some people were angry that they now had to write:\n\ns = 0\nfor i = 1:10\n    global s += i\nend\n\nDo you see that global annotation in there? Hideous. Obviously this situation could not be tolerated. But seriously, there are two main issues with requiring global for this kind of top-level code:\n\nIt's no longer convenient to copy and paste the code from inside a function body into the REPL to debug it—you have to add global annotations and then remove them again to go back;\nBeginners will write this kind of code without the global and have no idea why their code doesn't work—the error that they get is that s is undefined, which does not seem to enlighten anyone who happens to make this mistake.\n\nAs of Julia 1.5, this code works without the global annotation in interactive contexts like the REPL or Jupyter notebooks (just like Julia 0.6) and in files and other non-interactive contexts, it prints this very direct warning:\n\nAssignment to s in soft scope is ambiguous because a global variable by the same name exists: s will be treated as a new local. Disambiguate by using local s to suppress this warning or global s to assign to the existing global variable.\n\nThis addresses both issues while preserving the \"programming at scale\" benefits of the 1.0 behavior: global variables have no spooky effect on the meaning of code that may be far away; in the REPL copy-and-paste debugging works and beginners don't have any issues; any time someone either forgets a global annotation or accidentally shadows an existing global with a local in a soft scope, which would be confusing anyway, they get a nice clear warning.\n\nAn important property of this design is that any code that executes in a file without a warning will behave the same way in a fresh REPL. And on the flip side, if you take a REPL session and save it to file, if it behaves differently than it did in the REPL, then you will get a warning."},{"title":"Let Blocks","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Let-Blocks","category":"section","text":"let statements create a new hard scope block (see above) and introduce new variable bindings each time they run. The variable need not be immediately assigned:\n\njulia> var1 = let x\n           for i in 1:5\n               (i == 4) && (x = i; break)\n           end\n           x\n       end\n4\n\nWhereas assignments might reassign a new value to an existing value location, let always creates a new location. This difference is usually not important, and is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:\n\njulia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: `z` not defined in local scope\n\nThe assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x since the two x variables are distinct and have separate storage. Here is an example where the behavior of let is needed:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           global i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3\n\nHere we create and store two closures that return variable i. However, it is always the same variable i, so the two closures behave identically. We can use let to create a new binding for i:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           global i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nSince the begin construct does not introduce a new scope, it can be useful to use a zero-argument let to just introduce a new scope block without creating any new bindings immediately:\n\njulia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1\n\nSince let introduces a new scope block, the inner local x is a different variable than the outer local x. This particular example is equivalent to:\n\njulia> let x = 1\n           let x = 2\n           end\n           x\n       end\n1"},{"title":"Loops and Comprehensions","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Loops-and-Comprehensions","category":"section","text":"In loops and comprehensions, new variables introduced in their body scopes are freshly allocated for each loop iteration, as if the loop body were surrounded by a let block, as demonstrated by this example:\n\njulia> Fs = Vector{Any}(undef, 2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nA for loop or comprehension iteration variable is always a new variable:\n\njulia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0\n\nHowever, it is occasionally useful to reuse an existing local variable as the iteration variable. This can be done conveniently by adding the keyword outer:\n\njulia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3"},{"title":"Constants","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Constants","category":"section","text":"A common use of variables is giving names to specific, unchanging values. Such variables are only assigned once. This intent can be conveyed to the compiler using the const keyword:\n\njulia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;\n\nMultiple variables can be declared in a single const statement:\n\njulia> const a, b = 1, 2\n(1, 2)\n\nThe const declaration should only be used in global scope on globals. It is difficult for the compiler to optimize code involving global variables, since their values (or even their types) might change at almost any time. If a global variable will not change, adding a const declaration solves this performance problem.\n\nLocal constants are quite different. The compiler is able to determine automatically when a local variable is constant, so local constant declarations are not necessary, and in fact are currently not supported.\n\nSpecial top-level assignments, such as those performed by the function and struct keywords, are constant by default.\n\nNote that const only affects the variable binding; the variable may be bound to a mutable object (such as an array), and that object may still be modified. Additionally when one tries to assign a value to a variable that is declared constant the following scenarios are possible:\n\nAttempting to replace a constant without the const keyword is disallowed:\njulia> const x = 1.0\n1.0\n\njulia> x = 1\nERROR: invalid assignment to constant x. This redefinition may be permitted using the `const` keyword.\nAll other definitions of constants are permitted, but may cause significant re-compilation:\njulia> const y = 1.0\n1.0\n\njulia> const y = 2.0\n2.0\n\ncompat: Julia 1.12\nPrior to julia 1.12, redefinition of constants was poorly supported. It was restricted to redefinition of constants of the same type and could lead to observably incorrect behavior or crashes. Constant redefinition is highly discouraged in versions of julia prior to 1.12. See the manual for prior julia versions for further information."},{"title":"Typed Globals","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-typed-globals","category":"section","text":"compat: Julia 1.8\nSupport for typed globals was added in Julia 1.8\n\nSimilar to being declared as constants, global bindings can also be declared to always be of a constant type. This can either be done without assigning an actual value using the syntax global x::T or upon assignment as x::T = 123.\n\njulia> x::Float64 = 2.718\n2.718\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> Base.return_types(f)\n1-element Vector{Any}:\n Float64\n\nFor any assignment to a global, Julia will first try to convert it to the appropriate type using convert:\n\njulia> global y::Int\n\njulia> y = 1.0\n1.0\n\njulia> y\n1\n\njulia> y = 3.14\nERROR: InexactError: Int64(3.14)\nStacktrace:\n[...]\n\nThe type does not need to be concrete, but annotations with abstract types typically have little performance benefit.\n\nOnce a global has either been assigned to or its type has been set, the binding type is not allowed to change:\n\njulia> x = 1\n1\n\njulia> global x::Int\nERROR: cannot set type for global x. It already has a value or is already set to a different type.\nStacktrace:\n[...]"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Code-changes","category":"section","text":""},{"title":"Contributing to core functionality or base libraries","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-core-functionality-or-base-libraries","category":"section","text":"By contributing code to Julia, you are agreeing to release it under the MIT License.\n\nThe Julia community uses GitHub issues to track and discuss problems, feature requests, and pull requests (PR).\n\nIssues and pull requests should have self explanatory titles such that they can be understood from the list of PRs and Issues. i.e. Add {feature} and Fix {bug} are good, Fix #12345. Corrects the bug. is bad.\n\nYou can make pull requests for incomplete features to get code review. The convention is to open these as draft PRs and prefix the pull request title with \"WIP:\" for Work In Progress, or \"RFC:\" for Request for Comments when work is completed and ready for merging. This will prevent accidental merging of work that is in progress.\n\nNote: These instructions are for adding to or improving functionality in the base library. Before getting started, it can be helpful to discuss the proposed changes or additions on the Julia Discourse forum or in a GitHub issue—it's possible your proposed change belongs in a package rather than the core language. Also, keep in mind that changing stuff in the base can potentially break a lot of things. Finally, because of the time required to build Julia, note that it's usually faster to develop your code in stand-alone files, get it working, and then migrate it into the base libraries.\n\nAdd new code to Julia's base libraries as follows (this is the \"basic\" approach; see a more efficient approach in the next section):\n\nEdit the appropriate file in the base/ directory, or add new files if necessary. Create tests for your functionality and add them to files in the test/ directory. If you're editing C or Scheme code, most likely it lives in src/ or one of its subdirectories, although some aspects of Julia's REPL initialization live in cli/.\nAdd any new files to sysimg.jl in order to build them into the Julia system image.\nAdd any necessary export symbols in exports.jl.\nInclude your tests in test/Makefile and test/choosetests.jl.\n\nBuild as usual, and do make clean testall to test your contribution. If your contribution includes changes to Makefiles or external dependencies, make sure you can build Julia from a clean tree using git clean -fdx or equivalent (be careful – this command will delete any files lying around that aren't checked into git)."},{"title":"Running specific tests","page":"Code changes","location":"devdocs/contributing/code-changes.html#Running-specific-tests","category":"section","text":"There are make targets for running specific tests:\n\nmake test-bitarray\n\nYou can also use the runtests.jl script, e.g. to run test/bitarray.jl and test/math.jl:\n\n./usr/bin/julia test/runtests.jl bitarray math"},{"title":"Modifying base more efficiently with Revise.jl","page":"Code changes","location":"devdocs/contributing/code-changes.html#Modifying-base-more-efficiently-with-Revise.jl","category":"section","text":"Revise is a package that tracks changes in source files and automatically updates function definitions in your running Julia session. Using it, you can make extensive changes to Base without needing to rebuild in order to test your changes.\n\nHere is the standard procedure:\n\nIf you are planning changes to any types or macros, make those changes and build julia using make. (This is necessary because Revise cannot handle changes to type definitions or macros.) Unless it's required to get Julia to build, you do not have to add any functionality based on the new types, just the type definitions themselves.\nStart a Julia REPL session. Then issue the following commands:\n\nusing Revise    # if you aren't launching it in your `.julia/config/startup.jl`\nRevise.track(Base)\n\nEdit files in base/, save your edits, and test the functionality.\n\nIf you need to restart your Julia session, just start at step 2 above. Revise.track(Base) will note any changes from when Julia was last built and incorporate them automatically. You only need to rebuild Julia if you made code-changes that Revise cannot handle.\n\nFor convenience, there are also test-revise-* targets for every test-* target that use Revise to load any modifications to Base into the current system image before running the corresponding test. This can be useful as a shortcut on the command line (since tests aren't always designed to be run outside the runtest harness)."},{"title":"Contributing to the standard library","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-the-standard-library","category":"section","text":"The standard library (stdlib) packages are baked into the Julia system image. When running the ordinary test workflow on the stdlib packages, the system image version overrides the version you are developing. To test stdlib packages, you can do the following steps:\n\nEdit the UUID field of the Project.toml in the stdlib package\nChange the current directory to the directory of the stdlib you are developing\nStart julia with julia --project=.\nYou can now test the package by running pkg> test in Pkg mode.\n\nBecause you changed the UUID, the package manager treats the stdlib package as different from the one in the system image, and the system image version will not override the package.\n\nBe sure to change the UUID value back before making the pull request."},{"title":"News-worthy changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#News-worthy-changes","category":"section","text":"For new functionality and other substantial changes, add a brief summary to NEWS.md. The news item should cross reference the pull request (PR) parenthetically, in the form ([#pr]). To add the PR reference number, first create the PR, then push an additional commit updating NEWS.md with the PR reference number. We periodically run ./julia doc/NEWS-update.jl from the julia directory to update the cross-reference links, but this should not be done in a typical PR in order to avoid conflicting commits."},{"title":"Annotations for new features, deprecations and behavior changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Annotations-for-new-features,-deprecations-and-behavior-changes","category":"section","text":"API additions and deprecations, and minor behavior changes are allowed in minor version releases. For documented features that are part of the public API, a compatibility note should be added into the manual or the docstring. It should state the Julia minor version that changed the behavior and have a brief message describing the change.\n\nAt the moment, this should always be done with the following compat admonition (so that it would be possible to programmatically find the annotations in the future):\n\n!!! compat \"Julia 1.X\"       This method was added in Julia 1.X."},{"title":"The World Age mechanism","page":"The World Age mechanism","location":"manual/worldage.html#man-world-age","category":"section","text":"note: Note\nWorld age is an advanced concept. For the vast majority of Julia users, the world age mechanism operates invisibly in the background. This documentation is intended for the few users who may encounter world-age related issues or error messages.\n\ncompat: Julia 1.12\nPrior to Julia 1.12, the world age mechanism did not apply to changes to the global binding table. The documentation in this chapter is specific to Julia 1.12+.\n\nwarning: Warning\nThis manual chapter uses internal functions to introspect world age and runtime data structures as an explanatory aid. In general, unless otherwise noted the world age mechanism is not a stable interface and should be interacted with in packages through stable APIs (e.g. invokelatest) only. In particular, do not assume that world ages are always integers or that they have a linear order."},{"title":"World age in general","page":"The World Age mechanism","location":"manual/worldage.html#World-age-in-general","category":"section","text":"The \"world age counter\" is a monotonically increasing counter that is incremented for every change to the global method table or the global binding table (e.g. through method definition, type definition, import/using declaration, creation of (typed) globals or definition of constants).\n\nThe current value of the global world age counter can be retrieved using the (internal) function Base.get_world_counter.\n\njulia> Base.get_world_counter()\n0x0000000000009632\n\njulia> const x = 1\n\njulia> Base.get_world_counter()\n0x0000000000009633\n\nIn addition, each Task stores a local world age that determines which modifications to the global binding and method tables are currently visible to the running task. The world age of the running task will never exceed the global world age counter, but may run arbitrarily behind it. In general the term \"current world age\" refers to the local world age of the currently running task. The current world age may be retrieved using the (internal) function Base.tls_world_age\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           Core.eval(@__MODULE__, :(f() = 1))\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           f()\n       end\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38452, 38452)\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38453, 38452)\nERROR: MethodError: no method matching f()\nThe applicable method may be too new: running in current world age 38452, while global world is 38453.\n\nClosest candidates are:\n  f() (method too new to be called from this world context.)\n   @ Main REPL[2]:3\n\nStacktrace:\n [1] top-level scope\n   @ REPL[2]:5\n\njulia> (f(), Int(Base.tls_world_age()))\n(1, 38453)\n\nHere the definition of the method f raised the global world counter, but the current world age did not change. As a result, the definition of f was not visible in the currently executing task and a MethodError resulted.\n\nnote: Note\nThe method error printing provided additional information that f() is available in a newer world age. This information is added by the error display, not the task that threw the MethodError. The thrown MethodError is identical whether or not a matching definition of f() exists in a newer world age.\n\nHowever, note that the definition of f() was subsequently available at the next REPL prompt, because the current task's world age had been raised. In general, certain syntactic constructs (in particular most definitions) will raise the current task's world age to the latest global world age, thus making all changes (both from the current task and any concurrently executing other tasks) visible. The following statements raise the current world age:\n\nAn explicit invocation of Core.@latestworld\nThe start of every top-level statement\nThe start of every REPL prompt\nAny type or struct definition\nAny method definition\nAny constant declaration\nAny global variable declaration (but not a global variable assignment)\nAny using, import, export or public statement\nCertain other macros like @eval (depends on the macro implementation)\n\nNote, however, that the current task's world age may only ever be permanently incremented at top level. As a general rule, using any of the above statements in non-top-level scope is a syntax error:\n\njulia> f() = Core.@latestworld\nERROR: syntax: World age increment not at top level\nStacktrace:\n [1] top-level scope\n   @ REPL[5]:1\n\nWhen it isn't (for example for @eval), the world age side effect is ignored.\n\nAs a result of these rules, Julia may assume that the world age does not change within the execution of an ordinary function.\n\nfunction my_function()\n    before = Base.tls_world_age()\n    # Any arbitrary code\n    after = Base.tls_world_age()\n    @assert before === after # always true\nend\n\nThis is the key invariant that allows Julia to optimize based on the current state of its global data structures, while still having the well-defined ability to change these data structures."},{"title":"Temporarily raising the world age using invokelatest","page":"The World Age mechanism","location":"manual/worldage.html#Temporarily-raising-the-world-age-using-invokelatest","category":"section","text":"As described above, it is not possible to permanently raise the world age for the remainder of a Task's execution unless the task is executing top-level statements. However, it is possible to temporarily change the world age in a scoped manner using invokelatest:\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           Core.eval(@__MODULE__, :(f() = 1))\n           invokelatest(f)\n       end\n1\n\ninvokelatest will temporarily raise the current task's world age to the latest global world age (at entry to invokelatest) and execute the provided function. Note that the world age will return to its prior value upon exit from invokelatest."},{"title":"World age and const struct redefinitions","page":"The World Age mechanism","location":"manual/worldage.html#World-age-and-const-struct-redefinitions","category":"section","text":"The semantics described above for method redefinition also apply to redefinition of constants:\n\njulia> const x = 1\n1\n\njulia> get_const() = x\nget_const (generic function with 1 method)\n\njulia> begin\n           @show get_const()\n           Core.eval(@__MODULE__, :(const x = 2))\n           @show get_const()\n           Core.@latestworld\n           @show get_const()\n       end\nget_const() = 1\nget_const() = 1\nget_const() = 2\n2\n\nHowever, for the avoidance of doubt, they do not apply to ordinary assignment to global variables, which becomes visible immediately:\n\njulia> global y = 1\n1\n\njulia> get_global() = y\nget_global (generic function with 1 method)\n\njulia> begin\n           @show get_global()\n           Core.eval(@__MODULE__, :(y = 2))\n           @show get_global()\n       end\nget_global() = 1\nget_global() = 2\n2\n\nOne particular special case of constant reassignment is the redefinition of struct types:\n\njulia> struct MyStruct\n           x::Int\n       end\n\njulia> const one_field = MyStruct(1)\nMyStruct(1)\n\njulia> struct MyStruct\n           x::Int\n           y::Float64\n       end\n\njulia> const two_field = MyStruct(1, 2.0)\nMyStruct(1, 2.0)\n\njulia> one_field\n@world(MyStruct, 38452:38455)(1)\n\njulia> two_field\nMyStruct(1, 2.0)\n\nInternally the two definitions of MyStruct are entirely separate types. However, after the new MyStruct type is defined, there is no longer any default binding for the original definition of MyStruct. To nevertheless facilitate access to these types, the special @world macro may be used to access the meaning of a name in a previous world. However, this facility is intended for introspection only and in particular note that world age numbers are not stable across precompilation and should in general be treated opaquely."},{"title":"Binding partition introspection","page":"The World Age mechanism","location":"manual/worldage.html#Binding-partition-introspection","category":"section","text":"In certain cases, it can be helpful to introspect the system's understanding of what a binding means in any particular world age. The default display printing of Core.Binding provides a helpful summary (e.g. on the MyStruct example from above):\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :MyStruct))\nBinding Main.MyStruct\n   38456:∞ - constant binding to MyStruct\n   38452:38455 - constant binding to @world(MyStruct, 38452:38455)\n   38451:38451 - backdated constant binding to @world(MyStruct, 38452:38455)\n   0:38450 - backdated constant binding to @world(MyStruct, 38452:38455)"},{"title":"World age and using/import","page":"The World Age mechanism","location":"manual/worldage.html#World-age-and-using/import","category":"section","text":"Bindings provided via using and import also operate via the world age mechanism. Binding resolution is a stateless function of the import and using definitions visible in the current world age. For example:\n\njulia> module M1; const x = 1; export x; end\n\njulia> module M2; const x = 2; export x; end\n\njulia> using .M1\n\njulia> x\n1\n\njulia> using .M2\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\nHint: It looks like two or more modules export different bindings with this name, resulting in ambiguity. Try explicitly importing it from a particular module, or qualifying the name with the module it should come from.\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :x))\nBinding Main.x\n   38458:∞ - ambiguous binding - guard entry\n   38457:38457 - implicit `using` resolved to constant 1"},{"title":"World age capture","page":"The World Age mechanism","location":"manual/worldage.html#World-age-capture","category":"section","text":"Certain language features capture the current task's world age. Perhaps the most common of these is creation of new tasks. Newly created tasks will inherit the creating task's local world age at creation time and will retain said world age (unless explicitly raised) even if the originating tasks raises its world age:\n\njulia> const x = 1\n\njulia> t = @task (wait(); println(\"Running now\"); x);\n\njulia> const x = 2\n\njulia> schedule(t);\nRunning now\n\njulia> x\n2\n\njulia> fetch(t)\n1\n\nIn addition to tasks, opaque closures also capture their world age at creation. See Base.Experimental.@opaque."},{"title":"Base.@world","page":"The World Age mechanism","location":"manual/worldage.html#Base.@world","category":"macro","text":"@world(sym, world)\n\nResolve the binding sym in world world. See invoke_in_world for running arbitrary code in fixed worlds. world may be UnitRange, in which case the macro will error unless the binding is valid and has the same value across the entire world range.\n\nAs a special case, the world ∞ always refers to the latest world, even if that world is newer than the world currently running.\n\nThe @world macro is primarily used in the printing of bindings that are no longer available in the current world.\n\nExample\n\njulia> struct Foo; a::Int; end\nFoo\n\njulia> fold = Foo(1)\n\njulia> Int(Base.get_world_counter())\n26866\n\njulia> struct Foo; a::Int; b::Int end\nFoo\n\njulia> fold\n@world(Foo, 26866)(1)\n\ncompat: Julia 1.12\nThis functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.get_world_counter","page":"The World Age mechanism","location":"manual/worldage.html#Base.get_world_counter","category":"function","text":"get_world_counter()\n\nReturns the current maximum world-age counter. This counter is monotonically increasing.\n\nwarning: Warning\nThis counter is global and may change at any time between invocations. In general, most reflection functions operate on the current task's world age, rather than the global maximum world age. See tls_world_age as well as the manual chapter of world age.\n\n\n\n\n\n"},{"title":"Base.tls_world_age","page":"The World Age mechanism","location":"manual/worldage.html#Base.tls_world_age","category":"function","text":"tls_world_age()\n\nReturns the world the current_task() is executing within.\n\n\n\n\n\n"},{"title":"Base.invoke_in_world","page":"The World Age mechanism","location":"manual/worldage.html#Base.invoke_in_world","category":"function","text":"invoke_in_world(world, f, args...; kwargs...)\n\nCall f(args...; kwargs...) in a fixed world age, world.\n\nThis is useful for infrastructure running in the user's Julia session which is not part of the user's program. For example, things related to the REPL, editor support libraries, etc. In these cases it can be useful to prevent unwanted method invalidation and recompilation latency, and to prevent the user from breaking supporting infrastructure by mistake.\n\nThe global world age can be queried using Base.get_world_counter() and stored for later use within the lifetime of the current Julia session, or when serializing and reloading the system image.\n\nTechnically, invoke_in_world will prevent any function called by f from being extended by the user during their Julia session. That is, generic function method tables seen by f (and any functions it calls) will be frozen as they existed at the given world age. In a sense, this is like the opposite of invokelatest.\n\nnote: Note\nIt is not valid to store world ages obtained in precompilation for later use. This is because precompilation generates a \"parallel universe\" where the world age refers to system state unrelated to the main Julia session.\n\n\n\n\n\n"},{"title":"Base.Experimental.@opaque","page":"The World Age mechanism","location":"manual/worldage.html#Base.Experimental.@opaque","category":"macro","text":"@opaque ([type, ]args...) -> body\n\nMarks a given closure as \"opaque\". Opaque closures capture the world age of their creation (as opposed to their invocation). This allows for more aggressive optimization of the capture list, but trades off against the ability to inline opaque closures at the call site, if their creation is not statically visible.\n\nAn argument tuple type (type) may optionally be specified, to specify allowed argument types in a more flexible way. In particular, the argument type may be fixed length even if the function is variadic.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n"},{"title":"File Events","page":"File Events","location":"stdlib/FileWatching.html#lib-filewatching","category":"section","text":""},{"title":"Pidfile","page":"File Events","location":"stdlib/FileWatching.html#Pidfile","category":"section","text":"A simple utility tool for creating advisory pidfiles (lock files)."},{"title":"Primary Functions","page":"File Events","location":"stdlib/FileWatching.html#Primary-Functions","category":"section","text":""},{"title":"Helper Functions","page":"File Events","location":"stdlib/FileWatching.html#Helper-Functions","category":"section","text":""},{"title":"FileWatching.poll_fd","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.poll_fd","category":"function","text":"poll_fd(fd, timeout_s::Real=-1; readable=false, writable=false)\n\nMonitor a file descriptor fd for changes in the read or write availability, and with a timeout given by timeout_s seconds.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\nThis is a thin wrapper over calling wait on a FDWatcher, which implements the functionality but requires the user to call close manually when finished with it, or risk serious crashes.\n\n\n\n\n\n"},{"title":"FileWatching.poll_file","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.poll_file","category":"function","text":"poll_file(path::AbstractString, interval_s::Real=5.007, timeout_s::Real=-1) -> (previous::StatStruct, current)\n\nMonitor a file for changes by polling every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds.\n\nReturns a pair of status objects (previous, current) when a change is detected. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (indicating the timeout elapsed), or some other Exception subtype (if the stat operation failed: for example, if the path does not exist).\n\nTo determine when a file was modified, compare !(current isa StatStruct && prev == current) to detect notification of changes to the mtime or inode. However, using watch_file for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\nThis is a thin wrapper over calling wait on a PollingFileWatcher, which implements the functionality, but this function has a small race window between consecutive calls to poll_file where the file might change without being detected.\n\n\n\n\n\n"},{"title":"FileWatching.watch_file","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.watch_file","category":"function","text":"watch_file(path::AbstractString, timeout_s::Real=-1)\n\nWatch file or directory path for changes until a change occurs or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThe returned value is an object with boolean fields renamed, changed, and timedout, giving the result of watching the file.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\nThis is a thin wrapper over calling wait on a FileMonitor. This function has a small race window between consecutive calls to watch_file where the file might change without being detected. To avoid this race, use\n\nfm = FileMonitor(path)\nwait(fm)\n\ndirectly, re-using the same fm each time you wait.\n\n\n\n\n\n"},{"title":"FileWatching.watch_folder","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.watch_folder","category":"function","text":"watch_folder(path::AbstractString, timeout_s::Real=-1)\n\nWatch a file or directory path for changes until a change has occurred or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis will continuing tracking changes for path in the background until unwatch_folder is called on the same path.\n\nThe returned value is an pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed, changed, and timedout, giving the event.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\nThis function is a thin wrapper over calling wait on a FolderMonitor, with added timeout support.\n\n\n\n\n\n"},{"title":"FileWatching.unwatch_folder","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.unwatch_folder","category":"function","text":"unwatch_folder(path::AbstractString)\n\nStop background tracking of changes for path. It is not recommended to do this while another task is waiting for watch_folder to return on the same path, as the result may be unpredictable.\n\n\n\n\n\n"},{"title":"FileWatching.FileMonitor","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.FileMonitor","category":"type","text":"FileMonitor(path::AbstractString)\n\nWatch file or directory path (which must exist) for changes until a change occurs. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nfm = FileMonitor(path) acts like an auto-reset Event, so wait(fm) blocks until there has been at least one event in the file originally at the given path and then returns an object with boolean fields renamed, changed, timedout summarizing all changes that have occurred since the last call to wait returned.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n"},{"title":"FileWatching.FolderMonitor","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.FolderMonitor","category":"type","text":"FolderMonitor(folder::AbstractString)\n\nWatch a file or directory path for changes until a change has occurred. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis acts similar to a Channel, so calling take! (or wait) blocks until some change has occurred. The wait function will return a pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed and changed, giving the event that occurred on it.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n"},{"title":"FileWatching.PollingFileWatcher","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.PollingFileWatcher","category":"type","text":"PollingFileWatcher(path::AbstractString, interval_s::Real=5.007)\n\nMonitor a file for changes by polling stat every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds. Call stat on it to get the most recent, but old, result.\n\nThis acts like an auto-reset Event, so calling wait blocks until the stat result has changed since the previous value captured upon entry to the wait call. The wait function will return a pair of status objects (previous, current) once any stat change is detected since the previous time that wait was called. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (if the wait is canceled by closing this object), or some other Exception subtype (if the stat operation failed: for example, if the path is removed). Note that stat value may be outdated if the file has changed again multiple times.\n\nUsing FileMonitor for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\n\n\n\n\n"},{"title":"FileWatching.FDWatcher","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.FDWatcher","category":"type","text":"FDWatcher(fd::Union{RawFD,WindowsRawSocket}, readable::Bool, writable::Bool)\n\nMonitor a file descriptor fd for changes in the read or write availability.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\nThis acts like a level-set event, so calling wait blocks until one of those conditions is met, but then continues to return without blocking until the condition is cleared (either there is no more to read, or no more space in the write buffer, or both).\n\nwarning: Warning\nYou must call close manually, when finished with this object, before the fd argument is closed. Failure to do so risks serious crashes.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.mkpidlock","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.mkpidlock","category":"function","text":"mkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\nmkpidlock(at::String, proc::Process; kwopts...)\n\nCreate a pidfile lock for the path \"at\" for the current process or the process identified by pid or proc. Can take a function to execute once locked, for usage in do blocks, after which the lock will be automatically closed. If the lock fails and wait is false, then an error is thrown.\n\nThe lock will be released by either close, a finalizer, or shortly after proc exits. Make sure the return value is live through the end of the critical section of your program, so the finalizer does not reclaim it early.\n\nOptional keyword arguments:\n\nmode: file access mode (modified by the process umask). Defaults to world-readable.\npoll_interval: Specify the maximum time to between attempts (if watch_file doesn't work)\nstale_age: Delete an existing pidfile (ignoring the lock) if it is older than this many seconds, based on its mtime.   The file won't be deleted until 5x longer than this if the pid in the file appears that it may be valid.   Or 25x longer if refresh is overridden to 0 to disable lock refreshing.   By default this is disabled (stale_age = 0), but a typical recommended value would be about 3-5x an   estimated normal completion time.\nrefresh: Keeps a lock from becoming stale by updating the mtime every interval of time that passes.   By default, this is set to stale_age/2, which is the recommended value.\nwait: If true, block until we get the lock, if false, raise error if lock fails.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.trymkpidlock","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.trymkpidlock","category":"function","text":"trymkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\ntrymkpidlock(at::String, proc::Process; kwopts...)\n\nLike mkpidlock except returns false instead of waiting if the file is already locked.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.close","page":"File Events","location":"stdlib/FileWatching.html#Base.close-Tuple{FileWatching.Pidfile.LockMonitor}","category":"method","text":"close(lock::LockMonitor)\n\nRelease a pidfile lock.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.open_exclusive","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.open_exclusive","category":"function","text":"open_exclusive(path::String; mode, poll_interval, wait, stale_age, refresh) :: File\n\nCreate a new a file for read-write advisory-exclusive access. If wait is false then error out if the lock files exist otherwise block until we get the lock.\n\nFor a description of the keyword arguments, see mkpidlock.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.tryopen_exclusive","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.tryopen_exclusive","category":"function","text":"tryopen_exclusive(path::String, mode::Integer = 0o444) :: Union{Void, File}\n\nTry to create a new file for read-write advisory-exclusive access, return nothing if it already exists.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.write_pidfile","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.write_pidfile","category":"function","text":"write_pidfile(io, pid)\n\nWrite our pidfile format to an open IO descriptor.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.parse_pidfile","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.parse_pidfile","category":"function","text":"parse_pidfile(file::Union{IO, String}) => (pid, hostname, age)\n\nAttempt to parse our pidfile format, replaced an element with (0, \"\", 0.0), respectively, for any read that failed.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.stale_pidfile","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.stale_pidfile","category":"function","text":"stale_pidfile(path::String, stale_age::Real, refresh::Real) :: Bool\n\nHelper function for open_exclusive for deciding if a pidfile is stale.\n\n\n\n\n\n"},{"title":"FileWatching.Pidfile.isvalidpid","page":"File Events","location":"stdlib/FileWatching.html#FileWatching.Pidfile.isvalidpid","category":"function","text":"isvalidpid(hostname::String, pid::Cuint) :: Bool\n\nAttempt to conservatively estimate whether pid is a valid process id.\n\n\n\n\n\n"},{"title":"Base.Filesystem.touch","page":"File Events","location":"stdlib/FileWatching.html#Base.Filesystem.touch-Tuple{FileWatching.Pidfile.LockMonitor}","category":"method","text":"Base.touch(::Pidfile.LockMonitor)\n\nUpdate the mtime on the lock, to indicate it is still fresh.\n\nSee also the refresh keyword in the mkpidlock constructor.\n\n\n\n\n\n"},{"title":"Calling Conventions","page":"Calling Conventions","location":"devdocs/callconv.html#Calling-Conventions","category":"section","text":"Julia uses three calling conventions for four distinct purposes:\n\nName Prefix Purpose\nNative julia_ Speed via specialized signatures\nJL Call jlcall_ Wrapper for generic calls\nJL Call jl_ Builtins\nC ABI jlcapi_ Wrapper callable from C"},{"title":"Julia Native Calling Convention","page":"Calling Conventions","location":"devdocs/callconv.html#Julia-Native-Calling-Convention","category":"section","text":"The native calling convention is designed for fast non-generic calls. It usually uses a specialized signature.\n\nLLVM ghosts (zero-length types) are omitted.\nLLVM scalars and vectors are passed by value.\nLLVM aggregates (arrays and structs) are passed by reference.\n\nA small return value is returned as LLVM return values. A large return value is returned via the \"structure return\" (sret) convention, where the caller provides a pointer to a return slot.\n\nAn argument or return value that is a homogeneous tuple is sometimes represented as an LLVM vector instead of an LLVM array."},{"title":"JL Call Convention","page":"Calling Conventions","location":"devdocs/callconv.html#JL-Call-Convention","category":"section","text":"The JL Call convention is for builtins and generic dispatch. Hand-written functions using this convention are declared via the macro JL_CALLABLE. The convention uses exactly 3 parameters:\n\nF  - Julia representation of function that is being applied\nargs - pointer to array of pointers to boxes\nnargs - length of the array\n\nThe return value is a pointer to a box."},{"title":"C ABI","page":"Calling Conventions","location":"devdocs/callconv.html#C-ABI","category":"section","text":"C ABI wrappers enable calling Julia from C. The wrapper calls a function using the native calling convention.\n\nTuples are always represented as C arrays."},{"title":"Handling Operating System Variation","page":"Handling Operating System Variation","location":"manual/handling-operating-system-variation.html#Handling-Operating-System-Variation","category":"section","text":"When writing cross-platform applications or libraries, it is often necessary to allow for differences between operating systems. The variable Sys.KERNEL can be used to handle such cases. There are several functions in the Sys module intended to make this easier, such as isunix, islinux, isapple, isbsd, isfreebsd, and iswindows. These may be used as follows:\n\nif Sys.iswindows()\n    windows_specific_thing(a)\nend\n\nNote that islinux, isapple, and isfreebsd are mutually exclusive subsets of isunix. Additionally, there is a macro @static which makes it possible to use these functions to conditionally hide invalid code, as demonstrated in the following examples.\n\nSimple blocks:\n\nccall((@static Sys.iswindows() ? :_fopen : :fopen), ...)\n\nComplex blocks:\n\n@static if Sys.islinux()\n    linux_specific_thing(a)\nelseif Sys.isapple()\n    apple_specific_thing(a)\nelse\n    generic_thing(a)\nend\n\nWhen nesting conditionals, the @static must be repeated for each level (parentheses optional, but recommended for readability):\n\n@static Sys.iswindows() ? :a : (@static Sys.isapple() ? :b : :c)"},{"title":"Dates","page":"Dates","location":"stdlib/Dates.html#Dates","category":"section","text":"The Dates module provides two types for working with dates: Date and DateTime, representing day and millisecond precision, respectively; both are subtypes of the abstract TimeType. The motivation for distinct types is simple: some operations are much simpler, both in terms of code and mental reasoning, when the complexities of greater precision don't have to be dealt with. For example, since the Date type only resolves to the precision of a single date (i.e. no hours, minutes, or seconds), normal considerations for time zones, daylight savings/summer time, and leap seconds are unnecessary and avoided.\n\nBoth Date and DateTime are basically immutable Int64 wrappers. The single instant field of either type is actually a UTInstant{P} type, which represents a continuously increasing machine timeline based on the UT second [1]. The DateTime type is not aware of time zones (naive, in Python parlance), analogous to a LocalDateTime in Java 8. Additional time zone functionality can be added through the TimeZones.jl package, which compiles the IANA time zone database. Both Date and DateTime are based on the ISO 8601 standard, which follows the proleptic Gregorian calendar. One note is that the ISO 8601 standard is particular about BC/BCE dates. In general, the last day of the BC/BCE era, 1-12-31 BC/BCE, was followed by 1-1-1 AD/CE, thus no year zero exists. The ISO standard, however, states that 1 BC/BCE is year zero, so 0000-12-31 is the day before 0001-01-01, and year -0001 (yes, negative one for the year) is 2 BC/BCE, year -0002 is 3 BC/BCE, etc.\n\n[1]: The notion of the UT second is actually quite fundamental. There are basically two different notions of time generally accepted, one based on the physical rotation of the earth (one full rotation = 1 day), the other based on the SI second (a fixed, constant value). These are radically different! Think about it, a \"UT second\", as defined relative to the rotation of the earth, may have a different absolute length depending on the day! Anyway, the fact that Date and DateTime are based on UT seconds is a simplifying, yet honest assumption so that things like leap seconds and all their complexity can be avoided. This basis of time is formally called UT or UT1. Basing types on the UT second basically means that every minute has 60 seconds and every day has 24 hours and leads to more natural calculations when working with calendar dates."},{"title":"Constructors","page":"Dates","location":"stdlib/Dates.html#Constructors","category":"section","text":"Date and DateTime types can be constructed by integer or Period types, by parsing, or through adjusters (more on those later):\n\njulia> DateTime(2013)\n2013-01-01T00:00:00\n\njulia> DateTime(2013,7)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1,12)\n2013-07-01T12:00:00\n\njulia> DateTime(2013,7,1,12,30)\n2013-07-01T12:30:00\n\njulia> DateTime(2013,7,1,12,30,59)\n2013-07-01T12:30:59\n\njulia> DateTime(2013,7,1,12,30,59,1)\n2013-07-01T12:30:59.001\n\njulia> Date(2013)\n2013-01-01\n\njulia> Date(2013,7)\n2013-07-01\n\njulia> Date(2013,7,1)\n2013-07-01\n\njulia> Date(Dates.Year(2013),Dates.Month(7),Dates.Day(1))\n2013-07-01\n\njulia> Date(Dates.Month(7),Dates.Year(2013))\n2013-07-01\n\nDate or DateTime parsing is accomplished by the use of format strings. Format strings work by the notion of defining delimited or fixed-width \"slots\" that contain a period to parse and passing the text to parse and format string to a Date or DateTime constructor, of the form Date(\"2015-01-01\",dateformat\"y-m-d\") or DateTime(\"20150101\",dateformat\"yyyymmdd\").\n\nDelimited slots are marked by specifying the delimiter the parser should expect between two subsequent periods; so \"y-m-d\" lets the parser know that between the first and second slots in a date string like \"2014-07-16\", it should find the - character. The y, m, and d characters let the parser know which periods to parse in each slot.\n\nAs in the case of constructors above such as Date(2013), delimited DateFormats allow for missing parts of dates and times so long as the preceding parts are given. The other parts are given the usual default values. For example, Date(\"1981-03\", dateformat\"y-m-d\") returns 1981-03-01, whilst Date(\"31/12\", dateformat\"d/m/y\") gives 0001-12-31.  (Note that the default year is 1 AD/CE.) An empty string, however, always throws an ArgumentError.\n\nFixed-width slots are specified by repeating the period character the number of times corresponding to the width with no delimiter between characters. So dateformat\"yyyymmdd\" would correspond to a date string like \"20140716\". The parser distinguishes a fixed-width slot by the absence of a delimiter, noting the transition \"yyyymm\" from one period character to the next.\n\nSupport for text-form month parsing is also supported through the u and U characters, for abbreviated and full-length month names, respectively. By default, only English month names are supported, so u corresponds to \"Jan\", \"Feb\", \"Mar\", etc. And U corresponds to \"January\", \"February\", \"March\", etc. Similar to other name=>value mapping functions dayname and monthname, custom locales can be loaded by passing in the locale=>Dict{String,Int} mapping to the MONTHTOVALUEABBR and MONTHTOVALUE dicts for abbreviated and full-name month names, respectively.\n\nThe above examples used the dateformat\"\" string macro. This macro creates a DateFormat object once when the macro is expanded and uses the same DateFormat object even if a code snippet is run multiple times.\n\njulia> for i = 1:10^5\n           Date(\"2015-01-01\", dateformat\"y-m-d\")\n       end\n\nOr you can create the DateFormat object explicitly:\n\njulia> df = DateFormat(\"y-m-d\");\n\njulia> dt = Date(\"2015-01-01\",df)\n2015-01-01\n\njulia> dt2 = Date(\"2015-01-02\",df)\n2015-01-02\n\nAlternatively, use broadcasting:\n\njulia> years = [\"2015\", \"2016\"];\n\njulia> Date.(years, DateFormat(\"yyyy\"))\n2-element Vector{Date}:\n 2015-01-01\n 2016-01-01\n\nFor convenience, you may pass the format string directly (e.g., Date(\"2015-01-01\",\"y-m-d\")), although this form incurs performance costs if you are parsing the same format repeatedly, as it internally creates a new DateFormat object each time.\n\nAs well as via the constructors, a Date or DateTime can be constructed from strings using the parse and tryparse functions, but with an optional third argument of type DateFormat specifying the format; for example, parse(Date, \"06.23.2013\", dateformat\"m.d.y\"), or tryparse(DateTime, \"1999-12-31T23:59:59\") which uses the default format. The notable difference between the functions is that with tryparse, an error is not thrown if the string is empty or in an invalid format; instead nothing is returned.\n\ncompat: Julia 1.9\nBefore Julia 1.9, empty strings could be passed to constructors and parse without error, returning as appropriate DateTime(1), Date(1) or Time(0). Likewise, tryparse did not return nothing.\n\nA full suite of parsing and formatting tests and examples is available in stdlib/Dates/test/io.jl."},{"title":"Durations/Comparisons","page":"Dates","location":"stdlib/Dates.html#Durations/Comparisons","category":"section","text":"Finding the length of time between two Date or DateTime is straightforward given their underlying representation as UTInstant{Day} and UTInstant{Millisecond}, respectively. The difference between Date is returned in the number of Day, and DateTime in the number of Millisecond. Similarly, comparing TimeType is a simple matter of comparing the underlying machine instants (which in turn compares the internal Int64 values).\n\njulia> dt = Date(2012,2,29)\n2012-02-29\n\njulia> dt2 = Date(2000,2,1)\n2000-02-01\n\njulia> dump(dt)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 734562\n\njulia> dump(dt2)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 730151\n\njulia> dt > dt2\ntrue\n\njulia> dt != dt2\ntrue\n\njulia> dt + dt2\nERROR: MethodError: no method matching +(::Date, ::Date)\n[...]\n\njulia> dt * dt2\nERROR: MethodError: no method matching *(::Date, ::Date)\n[...]\n\njulia> dt / dt2\nERROR: MethodError: no method matching /(::Date, ::Date)\n\njulia> dt - dt2\n4411 days\n\njulia> dt2 - dt\n-4411 days\n\njulia> dt = DateTime(2012,2,29)\n2012-02-29T00:00:00\n\njulia> dt2 = DateTime(2000,2,1)\n2000-02-01T00:00:00\n\njulia> dt - dt2\n381110400000 milliseconds"},{"title":"Accessor Functions","page":"Dates","location":"stdlib/Dates.html#Accessor-Functions","category":"section","text":"Because the Date and DateTime types are stored as single Int64 values, date parts or fields can be retrieved through accessor functions. The lowercase accessors return the field as an integer:\n\njulia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.year(t)\n2014\n\njulia> Dates.month(t)\n1\n\njulia> Dates.week(t)\n5\n\njulia> Dates.day(t)\n31\n\nWhile propercase return the same value in the corresponding Period type:\n\njulia> Dates.Year(t)\n2014 years\n\njulia> Dates.Day(t)\n31 days\n\nCompound methods are provided because it is more efficient to access multiple fields at the same time than individually:\n\njulia> Dates.yearmonth(t)\n(2014, 1)\n\njulia> Dates.monthday(t)\n(1, 31)\n\njulia> Dates.yearmonthday(t)\n(2014, 1, 31)\n\nOne may also access the underlying UTInstant or integer value:\n\njulia> dump(t)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 735264\n\njulia> t.instant\nDates.UTInstant{Day}(Day(735264))\n\njulia> Dates.value(t)\n735264"},{"title":"Query Functions","page":"Dates","location":"stdlib/Dates.html#Query-Functions","category":"section","text":"Query functions provide calendrical information about a TimeType. They include information about the day of the week:\n\njulia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.dayofweek(t)\n5\n\njulia> Dates.dayname(t)\n\"Friday\"\n\njulia> Dates.dayofweekofmonth(t) # 5th Friday of January\n5\n\nMonth of the year:\n\njulia> Dates.monthname(t)\n\"January\"\n\njulia> Dates.daysinmonth(t)\n31\n\nAs well as information about the TimeType's year and quarter:\n\njulia> Dates.isleapyear(t)\nfalse\n\njulia> Dates.dayofyear(t)\n31\n\njulia> Dates.quarterofyear(t)\n1\n\njulia> Dates.dayofquarter(t)\n31\n\nThe dayname and monthname methods can also take an optional locale keyword that can be used to return the name of the day or month of the year for other languages/locales. There are also versions of these functions returning the abbreviated names, namely dayabbr and monthabbr. First the mapping is loaded into the LOCALES variable:\n\njulia> french_months = [\"janvier\", \"février\", \"mars\", \"avril\", \"mai\", \"juin\",\n                        \"juillet\", \"août\", \"septembre\", \"octobre\", \"novembre\", \"décembre\"];\n\njulia> french_months_abbrev = [\"janv\",\"févr\",\"mars\",\"avril\",\"mai\",\"juin\",\n                              \"juil\",\"août\",\"sept\",\"oct\",\"nov\",\"déc\"];\n\njulia> french_days = [\"lundi\",\"mardi\",\"mercredi\",\"jeudi\",\"vendredi\",\"samedi\",\"dimanche\"];\n\njulia> Dates.LOCALES[\"french\"] = Dates.DateLocale(french_months, french_months_abbrev, french_days, [\"\"]);\n\nThe above mentioned functions can then be used to perform the queries:\n\njulia> Dates.dayname(t;locale=\"french\")\n\"vendredi\"\n\njulia> Dates.monthname(t;locale=\"french\")\n\"janvier\"\n\njulia> Dates.monthabbr(t;locale=\"french\")\n\"janv\"\n\nSince the abbreviated versions of the days are not loaded, trying to use the function dayabbr will throw an error.\n\njulia> Dates.dayabbr(t;locale=\"french\")\nERROR: BoundsError: attempt to access 1-element Vector{String} at index [5]\nStacktrace:\n[...]"},{"title":"TimeType-Period Arithmetic","page":"Dates","location":"stdlib/Dates.html#TimeType-Period-Arithmetic","category":"section","text":"It's good practice when using any language/date framework to be familiar with how date-period arithmetic is handled as there are some tricky issues to deal with (though much less so for day-precision types).\n\nThe Dates module approach tries to follow the simple principle of trying to change as little as possible when doing Period arithmetic. This approach is also often known as calendrical arithmetic or what you would probably guess if someone were to ask you the same calculation in a conversation. Why all the fuss about this? Let's take a classic example: add 1 month to January 31st, 2014. What's the answer? Javascript will say March 3 (assumes 31 days). PHP says March 2 (assumes 30 days). The fact is, there is no right answer. In the Dates module, it gives the result of February 28th. How does it figure that out? Consider the classic 7-7-7 gambling game in casinos.\n\nNow just imagine that instead of 7-7-7, the slots are Year-Month-Day, or in our example, 2014-01-31. When you ask to add 1 month to this date, the month slot is incremented, so now we have 2014-02-31. Then the day number is checked if it is greater than the last valid day of the new month; if it is (as in the case above), the day number is adjusted down to the last valid day (28). What are the ramifications with this approach? Go ahead and add another month to our date, 2014-02-28 + Month(1) == 2014-03-28. What? Were you expecting the last day of March? Nope, sorry, remember the 7-7-7 slots. As few slots as possible are going to change, so we first increment the month slot by 1, 2014-03-28, and boom, we're done because that's a valid date. On the other hand, if we were to add 2 months to our original date, 2014-01-31, then we end up with 2014-03-31, as expected. The other ramification of this approach is a loss in associativity when a specific ordering is forced (i.e. adding things in different orders results in different outcomes). For example:\n\njulia> (Date(2014,1,29)+Dates.Day(1)) + Dates.Month(1)\n2014-02-28\n\njulia> (Date(2014,1,29)+Dates.Month(1)) + Dates.Day(1)\n2014-03-01\n\nWhat's going on there? In the first line, we're adding 1 day to January 29th, which results in 2014-01-30; then we add 1 month, so we get 2014-02-30, which then adjusts down to 2014-02-28. In the second example, we add 1 month first, where we get 2014-02-29, which adjusts down to 2014-02-28, and then add 1 day, which results in 2014-03-01. One design principle that helps in this case is that, in the presence of multiple Periods, the operations will be ordered by the Periods' types, not their value or positional order; this means Year will always be added first, then Month, then Week, etc. Hence the following does result in associativity and Just Works:\n\njulia> Date(2014,1,29) + Dates.Day(1) + Dates.Month(1)\n2014-03-01\n\njulia> Date(2014,1,29) + Dates.Month(1) + Dates.Day(1)\n2014-03-01\n\nTricky? Perhaps. What is an innocent Dates user to do? The bottom line is to be aware that explicitly forcing a certain associativity, when dealing with months, may lead to some unexpected results, but otherwise, everything should work as expected. Thankfully, that's pretty much the extent of the odd cases in date-period arithmetic when dealing with time in UT (avoiding the \"joys\" of dealing with daylight savings, leap seconds, etc.).\n\nAs a bonus, all period arithmetic objects work directly with ranges:\n\njulia> dr = Date(2014,1,29):Day(1):Date(2014,2,3)\nDate(\"2014-01-29\"):Day(1):Date(\"2014-02-03\")\n\njulia> collect(dr)\n6-element Vector{Date}:\n 2014-01-29\n 2014-01-30\n 2014-01-31\n 2014-02-01\n 2014-02-02\n 2014-02-03\n\njulia> dr = Date(2014,1,29):Dates.Month(1):Date(2014,07,29)\nDate(\"2014-01-29\"):Month(1):Date(\"2014-07-29\")\n\njulia> collect(dr)\n7-element Vector{Date}:\n 2014-01-29\n 2014-02-28\n 2014-03-29\n 2014-04-29\n 2014-05-29\n 2014-06-29\n 2014-07-29"},{"title":"Adjuster Functions","page":"Dates","location":"stdlib/Dates.html#Adjuster-Functions","category":"section","text":"As convenient as date-period arithmetic is, often the kinds of calculations needed on dates take on a calendrical or temporal nature rather than a fixed number of periods. Holidays are a perfect example; most follow rules such as \"Memorial Day = Last Monday of May\", or \"Thanksgiving = 4th Thursday of November\". These kinds of temporal expressions deal with rules relative to the calendar, like first or last of the month, next Tuesday, or the first and third Wednesdays, etc.\n\nThe Dates module provides the adjuster API through several convenient methods that aid in simply and succinctly expressing temporal rules. The first group of adjuster methods deal with the first and last of weeks, months, quarters, and years. They each take a single TimeType as input and return or adjust to the first or last of the desired period relative to the input.\n\njulia> Dates.firstdayofweek(Date(2014,7,16)) # Adjusts the input to the Monday of the input's week\n2014-07-14\n\njulia> Dates.lastdayofmonth(Date(2014,7,16)) # Adjusts to the last day of the input's month\n2014-07-31\n\njulia> Dates.lastdayofquarter(Date(2014,7,16)) # Adjusts to the last day of the input's quarter\n2014-09-30\n\nThe next two higher-order methods, tonext, and toprev, generalize working with temporal expressions by taking a DateFunction as first argument, along with a starting TimeType. A DateFunction is just a function, usually anonymous, that takes a single TimeType as input and returns a Bool, true indicating a satisfied adjustment criterion. For example:\n\njulia> istuesday = x->Dates.dayofweek(x) == Dates.Tuesday; # Returns true if the day of the week of x is Tuesday\n\njulia> Dates.tonext(istuesday, Date(2014,7,13)) # 2014-07-13 is a Sunday\n2014-07-15\n\njulia> Dates.tonext(Date(2014,7,13), Dates.Tuesday) # Convenience method provided for day of the week adjustments\n2014-07-15\n\nThis is useful with the do-block syntax for more complex temporal expressions:\n\njulia> Dates.tonext(Date(2014,7,13)) do x\n           # Return true on the 4th Thursday of November (Thanksgiving)\n           Dates.dayofweek(x) == Dates.Thursday &&\n           Dates.dayofweekofmonth(x) == 4 &&\n           Dates.month(x) == Dates.November\n       end\n2014-11-27\n\nThe Base.filter method can be used to obtain all valid dates/moments in a specified range:\n\n# Pittsburgh street cleaning; Every 2nd Tuesday from April to November\n# Date range from January 1st, 2014 to January 1st, 2015\njulia> dr = Dates.Date(2014):Day(1):Dates.Date(2015);\n\njulia> filter(dr) do x\n           Dates.dayofweek(x) == Dates.Tue &&\n           Dates.April <= Dates.month(x) <= Dates.Nov &&\n           Dates.dayofweekofmonth(x) == 2\n       end\n8-element Vector{Date}:\n 2014-04-08\n 2014-05-13\n 2014-06-10\n 2014-07-08\n 2014-08-12\n 2014-09-09\n 2014-10-14\n 2014-11-11\n\nAdditional examples and tests are available in stdlib/Dates/test/adjusters.jl."},{"title":"Period Types","page":"Dates","location":"stdlib/Dates.html#Period-Types","category":"section","text":"Periods are a human view of discrete, sometimes irregular durations of time. Consider 1 month; it could represent, in days, a value of 28, 29, 30, or 31 depending on the year and month context. Or a year could represent 365 or 366 days in the case of a leap year. Period types are simple Int64 wrappers and are constructed by wrapping any Int64 convertible type, i.e. Year(1) or Month(3.0). Arithmetic between Period of the same type behave like integers, and limited Period-Real arithmetic is available. You can extract the underlying integer with Dates.value.\n\njulia> y1 = Dates.Year(1)\n1 year\n\njulia> y2 = Dates.Year(2)\n2 years\n\njulia> y3 = Dates.Year(10)\n10 years\n\njulia> y1 + y2\n3 years\n\njulia> div(y3,y2)\n5\n\njulia> y3 - y2\n8 years\n\njulia> y3 % y2\n0 years\n\njulia> div(y3,3) # mirrors integer division\n3 years\n\njulia> Dates.value(Dates.Millisecond(10))\n10\n\nRepresenting periods or durations that are not integer multiples of the basic types can be achieved with the Dates.CompoundPeriod type. Compound periods may be constructed manually from simple Period types. Additionally, the canonicalize function can be used to break down a period into a Dates.CompoundPeriod. This is particularly useful to convert a duration, e.g., a difference of two DateTime, into a more convenient representation.\n\njulia> cp = Dates.CompoundPeriod(Day(1),Minute(1))\n1 day, 1 minute\n\njulia> t1 = DateTime(2018,8,8,16,58,00)\n2018-08-08T16:58:00\n\njulia> t2 = DateTime(2021,6,23,10,00,00)\n2021-06-23T10:00:00\n\njulia> canonicalize(t2-t1) # creates a CompoundPeriod\n149 weeks, 6 days, 17 hours, 2 minutes"},{"title":"Rounding","page":"Dates","location":"stdlib/Dates.html#Rounding","category":"section","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round:\n\njulia> floor(Date(1985, 8, 16), Dates.Month)\n1985-08-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Dates.Minute(15))\n2013-02-13T00:45:00\n\njulia> round(DateTime(2016, 8, 6, 20, 15), Dates.Day)\n2016-08-07T00:00:00\n\nUnlike the numeric round method, which breaks ties toward the even number by default, the TimeTyperound method uses the RoundNearestTiesUp rounding mode. (It's difficult to guess what breaking ties to nearest \"even\" TimeType would entail.) Further details on the available RoundingMode s can be found in the API reference.\n\nRounding should generally behave as expected, but there are a few cases in which the expected behaviour is not obvious."},{"title":"Rounding Epoch","page":"Dates","location":"stdlib/Dates.html#Rounding-Epoch","category":"section","text":"In many cases, the resolution specified for rounding (e.g., Dates.Second(30)) divides evenly into the next largest period (in this case, Dates.Minute(1)). But rounding behaviour in cases in which this is not true may lead to confusion. What is the expected result of rounding a DateTime to the nearest 10 hours?\n\njulia> round(DateTime(2016, 7, 17, 11, 55), Dates.Hour(10))\n2016-07-17T12:00:00\n\nThat may seem confusing, given that the hour (12) is not divisible by 10. The reason that 2016-07-17T12:00:00 was chosen is that it is 17,676,660 hours after 0000-01-01T00:00:00, and 17,676,660 is divisible by 10.\n\nAs Julia Date and DateTime values are represented according to the ISO 8601 standard, 0000-01-01T00:00:00 was chosen as base (or \"rounding epoch\") from which to begin the count of days (and milliseconds) used in rounding calculations. (Note that this differs slightly from Julia's internal representation of Date s using Rata Die notation; but since the ISO 8601 standard is most visible to the end user, 0000-01-01T00:00:00 was chosen as the rounding epoch instead of the 0000-12-31T00:00:00 used internally to minimize confusion.)\n\nThe only exception to the use of 0000-01-01T00:00:00 as the rounding epoch is when rounding to weeks. Rounding to the nearest week will always return a Monday (the first day of the week as specified by ISO 8601). For this reason, we use 0000-01-03T00:00:00 (the first day of the first week of year 0000, as defined by ISO 8601) as the base when rounding to a number of weeks.\n\nHere is a related case in which the expected behaviour is not necessarily obvious: What happens when we round to the nearest P(2), where P is a Period type? In some cases (specifically, when P <: Dates.TimePeriod) the answer is clear:\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Hour(2))\n2016-07-17T08:00:00\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Minute(2))\n2016-07-17T08:56:00\n\nThis seems obvious, because two of each of these periods still divides evenly into the next larger order period. But in the case of two months (which still divides evenly into one year), the answer may be surprising:\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Month(2))\n2016-07-01T00:00:00\n\nWhy round to the first day in July, even though it is month 7 (an odd number)? The key is that months are 1-indexed (the first month is assigned 1), unlike hours, minutes, seconds, and milliseconds (the first of which are assigned 0).\n\nThis means that rounding a DateTime to an even multiple of seconds, minutes, hours, or years (because the ISO 8601 specification includes a year zero) will result in a DateTime with an even value in that field, while rounding a DateTime to an even multiple of months will result in the months field having an odd value. Because both months and years may contain an irregular number of days, whether rounding to an even number of days will result in an even value in the days field is uncertain.\n\nSee the API reference for additional information on methods exported from the Dates module."},{"title":"API reference","page":"Dates","location":"stdlib/Dates.html#stdlib-dates-api","category":"section","text":""},{"title":"Dates and Time Types","page":"Dates","location":"stdlib/Dates.html#Dates-and-Time-Types","category":"section","text":""},{"title":"Dates Functions","page":"Dates","location":"stdlib/Dates.html#Dates-Functions","category":"section","text":""},{"title":"Accessor Functions","page":"Dates","location":"stdlib/Dates.html#Accessor-Functions-2","category":"section","text":""},{"title":"Query Functions","page":"Dates","location":"stdlib/Dates.html#Query-Functions-2","category":"section","text":""},{"title":"Adjuster Functions","page":"Dates","location":"stdlib/Dates.html#Adjuster-Functions-2","category":"section","text":""},{"title":"Periods","page":"Dates","location":"stdlib/Dates.html#Periods","category":"section","text":""},{"title":"Rounding Functions","page":"Dates","location":"stdlib/Dates.html#Rounding-Functions","category":"section","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round.\n\nMost Period values can also be rounded to a specified resolution:\n\nThe following functions are not exported:"},{"title":"Conversion Functions","page":"Dates","location":"stdlib/Dates.html#Conversion-Functions","category":"section","text":""},{"title":"Constants","page":"Dates","location":"stdlib/Dates.html#Constants","category":"section","text":"Days of the Week:\n\nVariable Abbr. Value (Int)\nMonday Mon 1\nTuesday Tue 2\nWednesday Wed 3\nThursday Thu 4\nFriday Fri 5\nSaturday Sat 6\nSunday Sun 7\n\nMonths of the Year:\n\nVariable Abbr. Value (Int)\nJanuary Jan 1\nFebruary Feb 2\nMarch Mar 3\nApril Apr 4\nMay May 5\nJune Jun 6\nJuly Jul 7\nAugust Aug 8\nSeptember Sep 9\nOctober Oct 10\nNovember Nov 11\nDecember Dec 12"},{"title":"Common Date Formatters","page":"Dates","location":"stdlib/Dates.html#Common-Date-Formatters","category":"section","text":""},{"title":"Dates.Period","page":"Dates","location":"stdlib/Dates.html#Dates.Period","category":"type","text":"Period\nYear\nQuarter\nMonth\nWeek\nDay\nHour\nMinute\nSecond\nMillisecond\nMicrosecond\nNanosecond\n\nPeriod types represent discrete, human representations of time.\n\n\n\n\n\n"},{"title":"Dates.CompoundPeriod","page":"Dates","location":"stdlib/Dates.html#Dates.CompoundPeriod","category":"type","text":"CompoundPeriod\n\nA CompoundPeriod is useful for expressing time periods that are not a fixed multiple of smaller periods. For example, \"a year and a day\" is not a fixed number of days, but can be expressed using a CompoundPeriod. In fact, a CompoundPeriod is automatically generated by addition of different period types, e.g. Year(1) + Day(1) produces a CompoundPeriod result.\n\n\n\n\n\n"},{"title":"Dates.Instant","page":"Dates","location":"stdlib/Dates.html#Dates.Instant","category":"type","text":"Instant\n\nInstant types represent integer-based, machine representations of time as continuous timelines starting from an epoch.\n\n\n\n\n\n"},{"title":"Dates.UTInstant","page":"Dates","location":"stdlib/Dates.html#Dates.UTInstant","category":"type","text":"UTInstant{T}\n\nThe UTInstant represents a machine timeline based on UT time (1 day = one revolution of the earth). The T is a Period parameter that indicates the resolution or precision of the instant.\n\n\n\n\n\n"},{"title":"Dates.TimeType","page":"Dates","location":"stdlib/Dates.html#Dates.TimeType","category":"type","text":"TimeType\n\nTimeType types wrap Instant machine instances to provide human representations of the machine instant. Time, DateTime and Date are subtypes of TimeType.\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime","category":"type","text":"DateTime\n\nDateTime represents a point in time according to the proleptic Gregorian calendar. The finest resolution of the time is millisecond (i.e., microseconds or nanoseconds cannot be represented by this type). The type supports fixed-point arithmetic, and thus is prone to underflowing (and overflowing). A notable consequence is rounding when adding a Microsecond or a Nanosecond:\n\njulia> dt = DateTime(2023, 8, 19, 17, 45, 32, 900)\n2023-08-19T17:45:32.900\n\njulia> dt + Millisecond(1)\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1000) # 1000us == 1ms\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(999) # 999us rounded to 1000us\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1499) # 1499 rounded to 1000us\n2023-08-19T17:45:32.901\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date","category":"type","text":"Date\n\nDate wraps a UTInstant{Day} and interprets it according to the proleptic Gregorian calendar.\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time","category":"type","text":"Time\n\nTime wraps a Nanosecond and represents a specific moment in a 24-hour day.\n\n\n\n\n\n"},{"title":"Dates.TimeZone","page":"Dates","location":"stdlib/Dates.html#Dates.TimeZone","category":"type","text":"TimeZone\n\nGeographic zone generally based on longitude determining what the time is at a certain location. Some time zones observe daylight savings (eg EST -> EDT). For implementations and more support, see the TimeZones.jl package\n\n\n\n\n\n"},{"title":"Dates.UTC","page":"Dates","location":"stdlib/Dates.html#Dates.UTC","category":"type","text":"UTC\n\nUTC, or Coordinated Universal Time, is the TimeZone from which all others are measured. It is associated with the time at 0° longitude. It is not adjusted for daylight savings.\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-NTuple{7, Int64}","category":"method","text":"DateTime(y, [m, d, h, mi, s, ms])::DateTime\n\nConstruct a DateTime type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-Tuple{Period}","category":"method","text":"DateTime(periods::Period...)::DateTime\n\nConstruct a DateTime type by Period type parts. Arguments may be in any order. DateTime parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-Tuple{Function, Vararg{Any}}","category":"method","text":"DateTime(f::Function, y[, m, d, h, mi, s]; step=Day(1), limit=10000)::DateTime\n\nCreate a DateTime through the adjuster API. The starting point will be constructed from the provided y, m, d... arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied).\n\nExamples\n\njulia> DateTime(dt -> second(dt) == 40, 2010, 10, 20, 10; step = Second(1))\n2010-10-20T10:00:40\n\njulia> DateTime(dt -> hour(dt) == 20, 2010, 10, 20, 10; step = Hour(1), limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-Tuple{TimeType}","category":"method","text":"DateTime(dt::Date)\n\nConvert a Date to a DateTime. The hour, minute, second, and millisecond parts of the new DateTime are assumed to be zero.\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, AbstractString}","category":"method","text":"DateTime(dt::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the format string (see DateFormat  for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> DateTime(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01T00:00:00\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [DateTime(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{DateTime}:\n 2020-01-01T00:00:00\n 2020-01-02T00:00:00\n\n\n\n\n\n"},{"title":"Dates.format","page":"Dates","location":"stdlib/Dates.html#Dates.format-Tuple{TimeType, AbstractString}","category":"method","text":"format(dt::TimeType, format::AbstractString; locale=\"english\")::AbstractString\n\nConstruct a string by using a TimeType object and applying the provided format. The following character codes can be used to construct the format string:\n\nCode Examples Comment\ny 6 Numeric year with a fixed width\nY 1996 Numeric year with a minimum width\nm 1, 12 Numeric month with a minimum width\nu Jan Month name shortened to 3-chars according to the locale\nU January Full month name according to the locale keyword\nd 1, 31 Day of the month with a minimum width\nH 0, 23 Hour (24-hour clock) with a minimum width\nM 0, 59 Minute with a minimum width\nS 0, 59 Second with a minimum width\ns 000, 500 Millisecond with a minimum width of 3\ne Mon, Tue Abbreviated days of the week\nE Monday Full day of week name\n\nThe number of sequential code characters indicate the width of the code. A format of yyyy-mm specifies that the code y should have a width of four while m a width of two. Codes that yield numeric digits have an associated mode: fixed-width or minimum-width. The fixed-width mode left-pads the value with zeros when it is shorter than the specified width and truncates the value when longer. Minimum-width mode works the same as fixed-width except that it does not truncate values longer than the width.\n\nWhen creating a format you can use any non-code characters as a separator. For example to generate the string \"1996-01-15T00:00:00\" you could use format: \"yyyy-mm-ddTHH:MM:SS\". Note that if you need to use a code character as a literal you can use the escape character backslash. The string \"1996y01m\" can be produced with the format raw\"yyyy\\ymm\\m\".\n\n\n\n\n\n"},{"title":"Dates.DateFormat","page":"Dates","location":"stdlib/Dates.html#Dates.DateFormat","category":"type","text":"DateFormat(format::AbstractString, locale=\"english\")\n\nConstruct a date formatting object that can be used for parsing date strings or formatting a date object as a string. The following character codes can be used to construct the format string:\n\nCode Matches Comment\nY 1996, 96 Returns year of 1996, 0096\ny 1996, 96 Same as Y on parse but discards excess digits on format\nm 1, 01 Matches 1 or 2-digit months\nu Jan Matches abbreviated months according to the locale keyword\nU January Matches full month names according to the locale keyword\nd 1, 01 Matches 1 or 2-digit days\nH 00 Matches hours (24-hour clock)\nI 00 For outputting hours with 12-hour clock\nM 00 Matches minutes\nS 00 Matches seconds\ns .500 Matches milliseconds\ne Mon, Tues Matches abbreviated days of the week\nE Monday Matches full name days of the week\np AM Matches AM/PM (case-insensitive)\nyyyymmdd 19960101 Matches fixed-width year, month, and day\n\nCharacters not listed above are normally treated as delimiters between date and time slots. For example a dt string of \"1996-01-15T00:00:00.0\" would have a format string like \"y-m-dTH:M:S.s\". If you need to use a code character as a delimiter you can escape it using backslash. The date \"1995y01m\" would have the format \"y\\ym\\m\".\n\nNote that 12:00AM corresponds 00:00 (midnight), and 12:00PM corresponds to 12:00 (noon). When parsing a time with a p specifier, any hour (either H or I) is interpreted as as a 12-hour clock, so the I code is mainly useful for output.\n\nCreating a DateFormat object is expensive. Whenever possible, create it once and use it many times or try the dateformat\"\" string macro. Using this macro creates the DateFormat object once at macro expansion time and reuses it later. There are also several pre-defined formatters, listed later.\n\nSee DateTime and format for how to use a DateFormat object to parse and write Date strings respectively.\n\n\n\n\n\n"},{"title":"Dates.@dateformat_str","page":"Dates","location":"stdlib/Dates.html#Dates.@dateformat_str","category":"macro","text":"dateformat\"Y-m-d H:M:S\"\n\nCreate a DateFormat object. Similar to DateFormat(\"Y-m-d H:M:S\") but creates the DateFormat object once during macro expansion.\n\nSee DateFormat for details about format specifiers.\n\n\n\n\n\n"},{"title":"Dates.DateTime","page":"Dates","location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, DateFormat}","category":"method","text":"DateTime(dt::AbstractString, df::DateFormat=ISODateTimeFormat)\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\\THH:MM:SS.s\" if omitted.\n\nSimilar to DateTime(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date time strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{Int64, Int64, Int64}","category":"method","text":"Date(y, [m, d])::Date\n\nConstruct a Date type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{Period}","category":"method","text":"Date(period::Period...)::Date\n\nConstruct a Date type by Period type parts. Arguments may be in any order. Date parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{Function, Any, Any, Any}","category":"method","text":"Date(f::Function, y[, m, d]; step=Day(1), limit=10000)::Date\n\nCreate a Date through the adjuster API. The starting point will be constructed from the provided y, m, d arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (given that f::Function is never satisfied).\n\nExamples\n\njulia> Date(date -> week(date) == 20, 2010, 01, 01)\n2010-05-17\n\njulia> Date(date -> year(date) == 2010, 2000, 01, 01)\n2010-01-01\n\njulia> Date(date -> month(date) == 10, 2000, 01, 01; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{TimeType}","category":"method","text":"Date(dt::DateTime)\n\nConvert a DateTime to a Date. The hour, minute, second, and millisecond parts of the DateTime are truncated, so only the year, month and day parts are used in construction.\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, AbstractString}","category":"method","text":"Date(d::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a Date by parsing the d date string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Date(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [Date(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{Date}:\n 2020-01-01\n 2020-01-02\n\n\n\n\n\n"},{"title":"Dates.Date","page":"Dates","location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, DateFormat}","category":"method","text":"Date(d::AbstractString, df::DateFormat=ISODateFormat)\n\nConstruct a Date by parsing the d date string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\" if omitted.\n\nSimilar to Date(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-NTuple{5, Int64}","category":"method","text":"Time(h, [mi, s, ms, us, ns])::Time\n\nConstruct a Time type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-Tuple{TimePeriod}","category":"method","text":"Time(period::TimePeriod...)::Time\n\nConstruct a Time type by Period type parts. Arguments may be in any order. Time parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-Tuple{Function, Vararg{Any}}","category":"method","text":"Time(f::Function, h, mi=0; step::Period=Second(1), limit::Int=10000)\nTime(f::Function, h, mi, s; step::Period=Millisecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms; step::Period=Microsecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms, us; step::Period=Nanosecond(1), limit::Int=10000)\n\nCreate a Time through the adjuster API. The starting point will be constructed from the provided h, mi, s, ms, us arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied). Note that the default step will adjust to allow for greater precision for the given arguments; i.e. if hour, minute, and second arguments are provided, the default step will be Millisecond(1) instead of Second(1).\n\nExamples\n\njulia> Time(t -> minute(t) == 30, 20)\n20:30:00\n\njulia> Time(t -> minute(t) == 0, 20)\n20:00:00\n\njulia> Time(t -> hour(t) == 10, 3; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-Tuple{DateTime}","category":"method","text":"Time(dt::DateTime)\n\nConvert a DateTime to a Time. The hour, minute, second, and millisecond parts of the DateTime are used to create the new Time. Microsecond and nanoseconds are zero by default.\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, AbstractString}","category":"method","text":"Time(t::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a Time by parsing the t time string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Time(\"12:34pm\", \"HH:MMp\")\n12:34:00\n\njulia> a = (\"12:34pm\", \"2:34am\");\n\njulia> [Time(d, dateformat\"HH:MMp\") for d ∈ a] # preferred\n2-element Vector{Time}:\n 12:34:00\n 02:34:00\n\n\n\n\n\n"},{"title":"Dates.Time","page":"Dates","location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, DateFormat}","category":"method","text":"Time(t::AbstractString, df::DateFormat=ISOTimeFormat)\n\nConstruct a Time by parsing the t date time string following the pattern given in the DateFormat object, or dateformat\"HH:MM:SS.s\" if omitted.\n\nSimilar to Time(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted time strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"title":"Dates.now","page":"Dates","location":"stdlib/Dates.html#Dates.now-Tuple{}","category":"method","text":"now()::DateTime\n\nReturn a DateTime corresponding to the user's system time including the system timezone locale.\n\n\n\n\n\n"},{"title":"Dates.now","page":"Dates","location":"stdlib/Dates.html#Dates.now-Tuple{Type{UTC}}","category":"method","text":"now(::Type{UTC})::DateTime\n\nReturn a DateTime corresponding to the user's system time as UTC/GMT. For other time zones, see the TimeZones.jl package.\n\nExamples\n\njulia> now(UTC)\n2023-01-04T10:52:24.864\n\n\n\n\n\n"},{"title":"Base.eps","page":"Dates","location":"stdlib/Dates.html#Base.eps-Tuple{Union{Type{Date}, Type{DateTime}, Type{Time}, TimeType}}","category":"method","text":"eps(::Type{DateTime})::Millisecond\neps(::Type{Date})::Day\neps(::Type{Time})::Nanosecond\neps(::TimeType)::Period\n\nReturn the smallest unit value supported by the TimeType.\n\nExamples\n\njulia> eps(DateTime)\n1 millisecond\n\njulia> eps(Date)\n1 day\n\njulia> eps(Time)\n1 nanosecond\n\n\n\n\n\n"},{"title":"Dates.year","page":"Dates","location":"stdlib/Dates.html#Dates.year","category":"function","text":"year(dt::TimeType)::Int64\n\nThe year of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"title":"Dates.month","page":"Dates","location":"stdlib/Dates.html#Dates.month","category":"function","text":"month(dt::TimeType)::Int64\n\nThe month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"title":"Dates.week","page":"Dates","location":"stdlib/Dates.html#Dates.week","category":"function","text":"week(dt::TimeType)::Int64\n\nReturn the ISO week date of a Date or DateTime as an Int64. Note that the first week of a year is the week that contains the first Thursday of the year, which can result in dates prior to January 4th being in the last week of the previous year. For example, week(Date(2005, 1, 1)) is the 53rd week of 2004.\n\nExamples\n\njulia> week(Date(1989, 6, 22))\n25\n\njulia> week(Date(2005, 1, 1))\n53\n\njulia> week(Date(2004, 12, 31))\n53\n\n\n\n\n\n"},{"title":"Dates.day","page":"Dates","location":"stdlib/Dates.html#Dates.day","category":"function","text":"day(dt::TimeType)::Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"title":"Dates.hour","page":"Dates","location":"stdlib/Dates.html#Dates.hour","category":"function","text":"hour(dt::DateTime)::Int64\n\nThe hour of day of a DateTime as an Int64.\n\n\n\n\n\nhour(t::Time)::Int64\n\nThe hour of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.minute","page":"Dates","location":"stdlib/Dates.html#Dates.minute","category":"function","text":"minute(dt::DateTime)::Int64\n\nThe minute of a DateTime as an Int64.\n\n\n\n\n\nminute(t::Time)::Int64\n\nThe minute of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.second","page":"Dates","location":"stdlib/Dates.html#Dates.second","category":"function","text":"second(dt::DateTime)::Int64\n\nThe second of a DateTime as an Int64.\n\n\n\n\n\nsecond(t::Time)::Int64\n\nThe second of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.millisecond","page":"Dates","location":"stdlib/Dates.html#Dates.millisecond","category":"function","text":"millisecond(dt::DateTime)::Int64\n\nThe millisecond of a DateTime as an Int64.\n\n\n\n\n\nmillisecond(t::Time)::Int64\n\nThe millisecond of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.microsecond","page":"Dates","location":"stdlib/Dates.html#Dates.microsecond","category":"function","text":"microsecond(t::Time)::Int64\n\nThe microsecond of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.nanosecond","page":"Dates","location":"stdlib/Dates.html#Dates.nanosecond","category":"function","text":"nanosecond(t::Time)::Int64\n\nThe nanosecond of a Time as an Int64.\n\n\n\n\n\n"},{"title":"Dates.Year","page":"Dates","location":"stdlib/Dates.html#Dates.Year-Tuple{TimeType}","category":"method","text":"Year(v)\n\nConstruct a Year object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"title":"Dates.Month","page":"Dates","location":"stdlib/Dates.html#Dates.Month-Tuple{TimeType}","category":"method","text":"Month(v)\n\nConstruct a Month object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"title":"Dates.Week","page":"Dates","location":"stdlib/Dates.html#Dates.Week-Tuple{TimeType}","category":"method","text":"Week(v)\n\nConstruct a Week object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"title":"Dates.Day","page":"Dates","location":"stdlib/Dates.html#Dates.Day-Tuple{TimeType}","category":"method","text":"Day(v)\n\nConstruct a Day object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"title":"Dates.Hour","page":"Dates","location":"stdlib/Dates.html#Dates.Hour-Tuple{DateTime}","category":"method","text":"Hour(dt::DateTime)\n\nThe hour part of a DateTime as a Hour.\n\n\n\n\n\n"},{"title":"Dates.Minute","page":"Dates","location":"stdlib/Dates.html#Dates.Minute-Tuple{DateTime}","category":"method","text":"Minute(dt::DateTime)\n\nThe minute part of a DateTime as a Minute.\n\n\n\n\n\n"},{"title":"Dates.Second","page":"Dates","location":"stdlib/Dates.html#Dates.Second-Tuple{DateTime}","category":"method","text":"Second(dt::DateTime)\n\nThe second part of a DateTime as a Second.\n\n\n\n\n\n"},{"title":"Dates.Millisecond","page":"Dates","location":"stdlib/Dates.html#Dates.Millisecond-Tuple{DateTime}","category":"method","text":"Millisecond(dt::DateTime)\n\nThe millisecond part of a DateTime as a Millisecond.\n\n\n\n\n\n"},{"title":"Dates.Microsecond","page":"Dates","location":"stdlib/Dates.html#Dates.Microsecond-Tuple{Time}","category":"method","text":"Microsecond(dt::Time)\n\nThe microsecond part of a Time as a Microsecond.\n\n\n\n\n\n"},{"title":"Dates.Nanosecond","page":"Dates","location":"stdlib/Dates.html#Dates.Nanosecond-Tuple{Time}","category":"method","text":"Nanosecond(dt::Time)\n\nThe nanosecond part of a Time as a Nanosecond.\n\n\n\n\n\n"},{"title":"Dates.yearmonth","page":"Dates","location":"stdlib/Dates.html#Dates.yearmonth","category":"function","text":"yearmonth(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the year and month parts of a Date or DateTime.\n\n\n\n\n\n"},{"title":"Dates.monthday","page":"Dates","location":"stdlib/Dates.html#Dates.monthday","category":"function","text":"monthday(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the month and day parts of a Date or DateTime.\n\n\n\n\n\n"},{"title":"Dates.yearmonthday","page":"Dates","location":"stdlib/Dates.html#Dates.yearmonthday","category":"function","text":"yearmonthday(dt::TimeType) -> (Int64, Int64, Int64)\n\nSimultaneously return the year, month and day parts of a Date or DateTime.\n\n\n\n\n\n"},{"title":"Dates.dayname","page":"Dates","location":"stdlib/Dates.html#Dates.dayname","category":"function","text":"dayname(dt::TimeType; locale=\"english\")::String\ndayname(day::Integer; locale=\"english\")::String\n\nReturn the full day name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayname(Date(\"2000-01-01\"))\n\"Saturday\"\n\njulia> dayname(4)\n\"Thursday\"\n\n\n\n\n\n"},{"title":"Dates.dayabbr","page":"Dates","location":"stdlib/Dates.html#Dates.dayabbr","category":"function","text":"dayabbr(dt::TimeType; locale=\"english\")::String\ndayabbr(day::Integer; locale=\"english\")::String\n\nReturn the abbreviated name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayabbr(Date(\"2000-01-01\"))\n\"Sat\"\n\njulia> dayabbr(3)\n\"Wed\"\n\n\n\n\n\n"},{"title":"Dates.dayofweek","page":"Dates","location":"stdlib/Dates.html#Dates.dayofweek","category":"function","text":"dayofweek(dt::TimeType)::Int64\n\nReturn the day of the week as an Int64 with 1 = Monday, 2 = Tuesday, etc..\n\nExamples\n\njulia> dayofweek(Date(\"2000-01-01\"))\n6\n\n\n\n\n\n"},{"title":"Dates.dayofmonth","page":"Dates","location":"stdlib/Dates.html#Dates.dayofmonth","category":"function","text":"dayofmonth(dt::TimeType)::Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"title":"Dates.dayofweekofmonth","page":"Dates","location":"stdlib/Dates.html#Dates.dayofweekofmonth","category":"function","text":"dayofweekofmonth(dt::TimeType)::Int\n\nFor the day of week of dt, return which number it is in dt's month. So if the day of the week of dt is Monday, then 1 = First Monday of the month, 2 = Second Monday of the month, etc. In the range 1:5.\n\nExamples\n\njulia> dayofweekofmonth(Date(\"2000-02-01\"))\n1\n\njulia> dayofweekofmonth(Date(\"2000-02-08\"))\n2\n\njulia> dayofweekofmonth(Date(\"2000-02-15\"))\n3\n\n\n\n\n\n"},{"title":"Dates.daysofweekinmonth","page":"Dates","location":"stdlib/Dates.html#Dates.daysofweekinmonth","category":"function","text":"daysofweekinmonth(dt::TimeType)::Int\n\nFor the day of week of dt, return the total number of that day of the week in dt's month. Returns 4 or 5. Useful in temporal expressions for specifying the last day of a week in a month by including dayofweekofmonth(dt) == daysofweekinmonth(dt) in the adjuster function.\n\nExamples\n\njulia> daysofweekinmonth(Date(\"2005-01-01\"))\n5\n\njulia> daysofweekinmonth(Date(\"2005-01-04\"))\n4\n\n\n\n\n\n"},{"title":"Dates.monthname","page":"Dates","location":"stdlib/Dates.html#Dates.monthname","category":"function","text":"monthname(dt::TimeType; locale=\"english\")::String\nmonthname(month::Integer, locale=\"english\")::String\n\nReturn the full name of the month of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthname(Date(\"2005-01-04\"))\n\"January\"\n\njulia> monthname(2)\n\"February\"\n\n\n\n\n\n"},{"title":"Dates.monthabbr","page":"Dates","location":"stdlib/Dates.html#Dates.monthabbr","category":"function","text":"monthabbr(dt::TimeType; locale=\"english\")::String\nmonthabbr(month::Integer, locale=\"english\")::String\n\nReturn the abbreviated month name of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthabbr(Date(\"2005-01-04\"))\n\"Jan\"\n\njulia> monthabbr(2)\n\"Feb\"\n\n\n\n\n\n"},{"title":"Dates.daysinmonth","page":"Dates","location":"stdlib/Dates.html#Dates.daysinmonth","category":"function","text":"daysinmonth(dt::TimeType)::Int\n\nReturn the number of days in the month of dt. Value will be 28, 29, 30, or 31.\n\nExamples\n\njulia> daysinmonth(Date(\"2000-01\"))\n31\n\njulia> daysinmonth(Date(\"2001-02\"))\n28\n\njulia> daysinmonth(Date(\"2000-02\"))\n29\n\n\n\n\n\n"},{"title":"Dates.isleapyear","page":"Dates","location":"stdlib/Dates.html#Dates.isleapyear","category":"function","text":"isleapyear(dt::TimeType)::Bool\n\nReturn true if the year of dt is a leap year.\n\nExamples\n\njulia> isleapyear(Date(\"2004\"))\ntrue\n\njulia> isleapyear(Date(\"2005\"))\nfalse\n\n\n\n\n\n"},{"title":"Dates.dayofyear","page":"Dates","location":"stdlib/Dates.html#Dates.dayofyear","category":"function","text":"dayofyear(dt::TimeType)::Int\n\nReturn the day of the year for dt with January 1st being day 1.\n\n\n\n\n\n"},{"title":"Dates.daysinyear","page":"Dates","location":"stdlib/Dates.html#Dates.daysinyear","category":"function","text":"daysinyear(dt::TimeType)::Int\n\nReturn 366 if the year of dt is a leap year, otherwise return 365.\n\nExamples\n\njulia> daysinyear(1999)\n365\n\njulia> daysinyear(2000)\n366\n\n\n\n\n\n"},{"title":"Dates.quarterofyear","page":"Dates","location":"stdlib/Dates.html#Dates.quarterofyear","category":"function","text":"quarterofyear(dt::TimeType)::Int\n\nReturn the quarter that dt resides in. Range of value is 1:4.\n\n\n\n\n\n"},{"title":"Dates.dayofquarter","page":"Dates","location":"stdlib/Dates.html#Dates.dayofquarter","category":"function","text":"dayofquarter(dt::TimeType)::Int\n\nReturn the day of the current quarter of dt. Range of value is 1:92.\n\n\n\n\n\n"},{"title":"Base.trunc","page":"Dates","location":"stdlib/Dates.html#Base.trunc-Tuple{TimeType, Type{Period}}","category":"method","text":"trunc(dt::TimeType, ::Type{Period})::TimeType\n\nTruncates the value of dt according to the provided Period type.\n\nExamples\n\njulia> trunc(DateTime(\"1996-01-01T12:30:00\"), Day)\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"title":"Dates.firstdayofweek","page":"Dates","location":"stdlib/Dates.html#Dates.firstdayofweek","category":"function","text":"firstdayofweek(dt::TimeType)::TimeType\n\nAdjusts dt to the Monday of its week.\n\nExamples\n\njulia> firstdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"title":"Dates.lastdayofweek","page":"Dates","location":"stdlib/Dates.html#Dates.lastdayofweek","category":"function","text":"lastdayofweek(dt::TimeType)::TimeType\n\nAdjusts dt to the Sunday of its week.\n\nExamples\n\njulia> lastdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-07T00:00:00\n\n\n\n\n\n"},{"title":"Dates.firstdayofmonth","page":"Dates","location":"stdlib/Dates.html#Dates.firstdayofmonth","category":"function","text":"firstdayofmonth(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its month.\n\nExamples\n\njulia> firstdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-01T00:00:00\n\n\n\n\n\n"},{"title":"Dates.lastdayofmonth","page":"Dates","location":"stdlib/Dates.html#Dates.lastdayofmonth","category":"function","text":"lastdayofmonth(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its month.\n\nExamples\n\njulia> lastdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-31T00:00:00\n\n\n\n\n\n"},{"title":"Dates.firstdayofyear","page":"Dates","location":"stdlib/Dates.html#Dates.firstdayofyear","category":"function","text":"firstdayofyear(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its year.\n\nExamples\n\njulia> firstdayofyear(DateTime(\"1996-05-20\"))\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"title":"Dates.lastdayofyear","page":"Dates","location":"stdlib/Dates.html#Dates.lastdayofyear","category":"function","text":"lastdayofyear(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its year.\n\nExamples\n\njulia> lastdayofyear(DateTime(\"1996-05-20\"))\n1996-12-31T00:00:00\n\n\n\n\n\n"},{"title":"Dates.firstdayofquarter","page":"Dates","location":"stdlib/Dates.html#Dates.firstdayofquarter","category":"function","text":"firstdayofquarter(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its quarter.\n\nExamples\n\njulia> firstdayofquarter(DateTime(\"1996-05-20\"))\n1996-04-01T00:00:00\n\njulia> firstdayofquarter(DateTime(\"1996-08-20\"))\n1996-07-01T00:00:00\n\n\n\n\n\n"},{"title":"Dates.lastdayofquarter","page":"Dates","location":"stdlib/Dates.html#Dates.lastdayofquarter","category":"function","text":"lastdayofquarter(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its quarter.\n\nExamples\n\njulia> lastdayofquarter(DateTime(\"1996-05-20\"))\n1996-06-30T00:00:00\n\njulia> lastdayofquarter(DateTime(\"1996-08-20\"))\n1996-09-30T00:00:00\n\n\n\n\n\n"},{"title":"Dates.tonext","page":"Dates","location":"stdlib/Dates.html#Dates.tonext-Tuple{TimeType, Int64}","category":"method","text":"tonext(dt::TimeType, dow::Int; same::Bool=false)::TimeType\n\nAdjusts dt to the next day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the next dow, allowing for no adjustment to occur.\n\n\n\n\n\n"},{"title":"Dates.toprev","page":"Dates","location":"stdlib/Dates.html#Dates.toprev-Tuple{TimeType, Int64}","category":"method","text":"toprev(dt::TimeType, dow::Int; same::Bool=false)::TimeType\n\nAdjusts dt to the previous day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the previous dow, allowing for no adjustment to occur.\n\n\n\n\n\n"},{"title":"Dates.tofirst","page":"Dates","location":"stdlib/Dates.html#Dates.tofirst","category":"function","text":"tofirst(dt::TimeType, dow::Int; of=Month)::TimeType\n\nAdjusts dt to the first dow of its month. Alternatively, of=Year will adjust to the first dow of the year.\n\n\n\n\n\n"},{"title":"Dates.tolast","page":"Dates","location":"stdlib/Dates.html#Dates.tolast","category":"function","text":"tolast(dt::TimeType, dow::Int; of=Month)::TimeType\n\nAdjusts dt to the last dow of its month. Alternatively, of=Year will adjust to the last dow of the year.\n\n\n\n\n\n"},{"title":"Dates.tonext","page":"Dates","location":"stdlib/Dates.html#Dates.tonext-Tuple{Function, TimeType}","category":"method","text":"tonext(func::Function, dt::TimeType; step=Day(1), limit=10000, same=false)::TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n"},{"title":"Dates.toprev","page":"Dates","location":"stdlib/Dates.html#Dates.toprev-Tuple{Function, TimeType}","category":"method","text":"toprev(func::Function, dt::TimeType; step=Day(-1), limit=10000, same=false)::TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n"},{"title":"Dates.Period","page":"Dates","location":"stdlib/Dates.html#Dates.Period-Tuple{Any}","category":"method","text":"Year(v)\nQuarter(v)\nMonth(v)\nWeek(v)\nDay(v)\nHour(v)\nMinute(v)\nSecond(v)\nMillisecond(v)\nMicrosecond(v)\nNanosecond(v)\n\nConstruct a Period type with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"title":"Dates.CompoundPeriod","page":"Dates","location":"stdlib/Dates.html#Dates.CompoundPeriod-Tuple{Vector{<:Period}}","category":"method","text":"CompoundPeriod(periods)\n\nConstruct a CompoundPeriod from a Vector of Periods. All Periods of the same type will be added together.\n\nExamples\n\njulia> Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13))\n25 hours\n\njulia> Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1))\n-1 hour, 1 minute\n\njulia> Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2))\n1 month, -2 weeks\n\njulia> Dates.CompoundPeriod(Dates.Minute(50000))\n50000 minutes\n\n\n\n\n\n"},{"title":"Dates.canonicalize","page":"Dates","location":"stdlib/Dates.html#Dates.canonicalize","category":"function","text":"canonicalize(::CompoundPeriod)::CompoundPeriod\n\nReduces the CompoundPeriod into its canonical form by applying the following rules:\n\nAny Period large enough be partially representable by a coarser Period will be broken into multiple Periods (eg. Hour(30) becomes Day(1) + Hour(6))\nPeriods with opposite signs will be combined when possible (eg. Hour(1) - Day(1) becomes -Hour(23))\n\nExamples\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13)))\n1 day, 1 hour\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1)))\n-59 minutes\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2)))\n1 month, -2 weeks\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Minute(50000)))\n4 weeks, 6 days, 17 hours, 20 minutes\n\n\n\n\n\n"},{"title":"Dates.value","page":"Dates","location":"stdlib/Dates.html#Dates.value","category":"function","text":"Dates.value(x::Period)::Int64\n\nFor a given period, return the value associated with that period.  For example, value(Millisecond(10)) returns 10 as an integer.\n\n\n\n\n\n"},{"title":"Dates.default","page":"Dates","location":"stdlib/Dates.html#Dates.default","category":"function","text":"default(p::Period)::Period\n\nReturn a sensible \"default\" value for the input Period by returning T(1) for Year, Month, and Day, and T(0) for Hour, Minute, Second, and Millisecond.\n\n\n\n\n\n"},{"title":"Dates.periods","page":"Dates","location":"stdlib/Dates.html#Dates.periods","category":"function","text":"Dates.periods(::CompoundPeriod)::Vector{Period}\n\nReturn the Vector of Periods that comprise the given CompoundPeriod.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.floor","page":"Dates","location":"stdlib/Dates.html#Base.floor-Tuple{TimeType, Period}","category":"method","text":"floor(dt::TimeType, p::Period)::TimeType\n\nReturn the nearest Date or DateTime less than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: floor(dt, Dates.Hour) is a shortcut for floor(dt, Dates.Hour(1)).\n\njulia> floor(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> floor(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> floor(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-06T00:00:00\n\n\n\n\n\n"},{"title":"Base.ceil","page":"Dates","location":"stdlib/Dates.html#Base.ceil-Tuple{TimeType, Period}","category":"method","text":"ceil(dt::TimeType, p::Period)::TimeType\n\nReturn the nearest Date or DateTime greater than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: ceil(dt, Dates.Hour) is a shortcut for ceil(dt, Dates.Hour(1)).\n\njulia> ceil(Date(1985, 8, 16), Month)\n1985-09-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:45:00\n\njulia> ceil(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\n\n\n\n\n"},{"title":"Base.round","page":"Dates","location":"stdlib/Dates.html#Base.round-Tuple{TimeType, Period, RoundingMode{:NearestTiesUp}}","category":"method","text":"round(dt::TimeType, p::Period, [r::RoundingMode]) -> TimeType\n\nReturn the Date or DateTime nearest to dt at resolution p. By default (RoundNearestTiesUp), ties (e.g., rounding 9:30 to the nearest hour) will be rounded up.\n\nFor convenience, p may be a type instead of a value: round(dt, Dates.Hour) is a shortcut for round(dt, Dates.Hour(1)).\n\njulia> round(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> round(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> round(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\nValid rounding modes for round(::TimeType, ::Period, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\n\n\n\n\n"},{"title":"Base.floor","page":"Dates","location":"stdlib/Dates.html#Base.floor-Union{Tuple{T}, Tuple{Union{Day, Week, TimePeriod}, T}} where T<:Union{Day, Week, TimePeriod}","category":"method","text":"floor(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x down to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: floor(x, Dates.Hour) is a shortcut for floor(x, Dates.Hour(1)).\n\njulia> floor(Day(16), Week)\n2 weeks\n\njulia> floor(Minute(44), Minute(15))\n30 minutes\n\njulia> floor(Hour(36), Day)\n1 day\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"title":"Base.ceil","page":"Dates","location":"stdlib/Dates.html#Base.ceil-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}}","category":"method","text":"ceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x up to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: ceil(x, Dates.Hour) is a shortcut for ceil(x, Dates.Hour(1)).\n\njulia> ceil(Day(16), Week)\n3 weeks\n\njulia> ceil(Minute(44), Minute(15))\n45 minutes\n\njulia> ceil(Hour(36), Day)\n2 days\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"title":"Base.round","page":"Dates","location":"stdlib/Dates.html#Base.round-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}, RoundingMode{:NearestTiesUp}}","category":"method","text":"round(x::Period, precision::T, [r::RoundingMode]) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision. By default (RoundNearestTiesUp), ties (e.g., rounding 90 minutes to the nearest hour) will be rounded up.\n\nFor convenience, precision may be a type instead of a value: round(x, Dates.Hour) is a shortcut for round(x, Dates.Hour(1)).\n\njulia> round(Day(16), Week)\n2 weeks\n\njulia> round(Minute(44), Minute(15))\n45 minutes\n\njulia> round(Hour(36), Day)\n2 days\n\nValid rounding modes for round(::Period, ::T, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"title":"Dates.floorceil","page":"Dates","location":"stdlib/Dates.html#Dates.floorceil","category":"function","text":"floorceil(dt::TimeType, p::Period) -> (TimeType, TimeType)\n\nSimultaneously return the floor and ceil of a Date or DateTime at resolution p. More efficient than calling both floor and ceil individually.\n\n\n\n\n\nfloorceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> (T, T)\n\nSimultaneously return the floor and ceil of Period at resolution p.  More efficient than calling both floor and ceil individually.\n\n\n\n\n\n"},{"title":"Dates.epochdays2date","page":"Dates","location":"stdlib/Dates.html#Dates.epochdays2date","category":"function","text":"epochdays2date(days)::Date\n\nTake the number of days since the rounding epoch (0000-01-01T00:00:00) and return the corresponding Date.\n\n\n\n\n\n"},{"title":"Dates.epochms2datetime","page":"Dates","location":"stdlib/Dates.html#Dates.epochms2datetime","category":"function","text":"epochms2datetime(milliseconds)::DateTime\n\nTake the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) and return the corresponding DateTime.\n\n\n\n\n\n"},{"title":"Dates.date2epochdays","page":"Dates","location":"stdlib/Dates.html#Dates.date2epochdays","category":"function","text":"date2epochdays(dt::Date)::Int64\n\nTake the given Date and return the number of days since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n"},{"title":"Dates.datetime2epochms","page":"Dates","location":"stdlib/Dates.html#Dates.datetime2epochms","category":"function","text":"datetime2epochms(dt::DateTime)::Int64\n\nTake the given DateTime and return the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n"},{"title":"Dates.today","page":"Dates","location":"stdlib/Dates.html#Dates.today","category":"function","text":"today()::Date\n\nReturn the date portion of now().\n\n\n\n\n\n"},{"title":"Dates.unix2datetime","page":"Dates","location":"stdlib/Dates.html#Dates.unix2datetime","category":"function","text":"unix2datetime(x)::DateTime\n\nTake the number of seconds since unix epoch 1970-01-01T00:00:00 and convert to the corresponding DateTime.\n\n\n\n\n\n"},{"title":"Dates.datetime2unix","page":"Dates","location":"stdlib/Dates.html#Dates.datetime2unix","category":"function","text":"datetime2unix(dt::DateTime)::Float64\n\nTake the given DateTime and return the number of seconds since the unix epoch 1970-01-01T00:00:00 as a Float64.\n\n\n\n\n\n"},{"title":"Dates.julian2datetime","page":"Dates","location":"stdlib/Dates.html#Dates.julian2datetime","category":"function","text":"julian2datetime(julian_days)::DateTime\n\nTake the number of Julian calendar days since epoch -4713-11-24T12:00:00 and return the corresponding DateTime.\n\n\n\n\n\n"},{"title":"Dates.datetime2julian","page":"Dates","location":"stdlib/Dates.html#Dates.datetime2julian","category":"function","text":"datetime2julian(dt::DateTime)::Float64\n\nTake the given DateTime and return the number of Julian calendar days since the julian epoch -4713-11-24T12:00:00 as a Float64.\n\n\n\n\n\n"},{"title":"Dates.rata2datetime","page":"Dates","location":"stdlib/Dates.html#Dates.rata2datetime","category":"function","text":"rata2datetime(days)::DateTime\n\nTake the number of Rata Die days since epoch 0000-12-31T00:00:00 and return the corresponding DateTime.\n\n\n\n\n\n"},{"title":"Dates.datetime2rata","page":"Dates","location":"stdlib/Dates.html#Dates.datetime2rata","category":"function","text":"datetime2rata(dt::TimeType)::Int64\n\nReturn the number of Rata Die days since epoch from the given Date or DateTime.\n\n\n\n\n\n"},{"title":"Dates.ISODateTimeFormat","page":"Dates","location":"stdlib/Dates.html#Dates.ISODateTimeFormat","category":"constant","text":"Dates.ISODateTimeFormat\n\nDescribes the ISO8601 formatting for a date and time. This is the default value for Dates.format of a DateTime.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), ISODateTimeFormat)\n\"2018-08-08T12:00:43.001\"\n\n\n\n\n\n"},{"title":"Dates.ISODateFormat","page":"Dates","location":"stdlib/Dates.html#Dates.ISODateFormat","category":"constant","text":"Dates.ISODateFormat\n\nDescribes the ISO8601 formatting for a date. This is the default value for Dates.format of a Date.\n\nExamples\n\njulia> Dates.format(Date(2018, 8, 8), ISODateFormat)\n\"2018-08-08\"\n\n\n\n\n\n"},{"title":"Dates.ISOTimeFormat","page":"Dates","location":"stdlib/Dates.html#Dates.ISOTimeFormat","category":"constant","text":"Dates.ISOTimeFormat\n\nDescribes the ISO8601 formatting for a time. This is the default value for Dates.format of a Time.\n\nExamples\n\njulia> Dates.format(Time(12, 0, 43, 1), ISOTimeFormat)\n\"12:00:43.001\"\n\n\n\n\n\n"},{"title":"Dates.RFC1123Format","page":"Dates","location":"stdlib/Dates.html#Dates.RFC1123Format","category":"constant","text":"Dates.RFC1123Format\n\nDescribes the RFC1123 formatting for a date and time.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), RFC1123Format)\n\"Wed, 08 Aug 2018 12:00:43\"\n\n\n\n\n\n"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#Fixing-precompilation-hangs-due-to-open-tasks-or-IO","category":"section","text":"On Julia 1.10 or higher, you might see the following message:\n\n(Image: Screenshot of precompilation hang)\n\nThis may repeat. If it continues to repeat with no hints that it will resolve itself, you may have a \"precompilation hang\" that requires fixing. Even if it's transient, you might prefer to resolve it so that users will not be bothered by this warning. This page walks you through how to analyze and fix such issues.\n\nIf you follow the advice and hit Ctrl-C, you might see\n\n^C Interrupted: Exiting precompilation...\n\n  1 dependency had warnings during precompilation:\n┌ Test1 [ac89d554-e2ba-40bc-bc5c-de68b658c982]\n│  [pid 2745] waiting for IO to finish:\n│   Handle type        uv_handle_t->data\n│   timer              0x55580decd1e0->0x7f94c3a4c340\n\nThis message conveys two key pieces of information:\n\nthe hang is occurring during precompilation of Test1, a dependency of Test2 (the package we were trying to load with using Test2)\nduring precompilation of Test1, Julia created a Timer object (use ?Timer if you're unfamiliar with Timers) which is still open; until that closes, the process is hung\n\nIf this is enough of a hint for you to figure out how timer = Timer(args...) is being created, one good solution is to add wait(timer) if timer eventually finishes on its own, or close(timer) if you need to force-close it, before the final end of the module.\n\nHowever, there are cases that may not be that straightforward. Usually the best option is to start by determining whether the hang is due to code in Test1 or whether it is due to one of Test1's dependencies:\n\nOption 1: Pkg.add(\"Aqua\") and use Aqua.test_persistent_tasks. This should help you identify which package is causing the problem, after which the instructions below should be followed. If needed, you can create a PkgId as Base.PkgId(UUID(\"...\"), \"Test1\"), where ... comes from the uuid entry in Test1/Project.toml.\nOption 2: manually diagnose the source of the hang.\n\nTo manually diagnose:\n\nPkg.develop(\"Test1\")\nComment out all the code included or defined in Test1, except the using/import statements.\nTry using Test2 (or even using Test1 assuming that hangs too) again\n\nNow we arrive at a fork in the road: either\n\nthe hang persists, indicating it is due to one of your dependencies\nthe hang disappears, indicating that it is due to something in your code."},{"title":"Diagnosing and fixing hangs due to a package dependency","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_deps","category":"section","text":"Use a binary search to identify the problematic dependency: start by commenting out half your dependencies, then when you isolate which half is responsible comment out half of that half, etc. (You don't have to remove them from the project, just comment out the using/import statements.)\n\nOnce you've identified a suspect (here we'll call it ThePackageYouThinkIsCausingTheProblem), first try precompiling that package. If it also hangs during precompilation, continue chasing the problem backwards.\n\nHowever, most likely ThePackageYouThinkIsCausingTheProblem will precompile fine. This suggests it's in the function ThePackageYouThinkIsCausingTheProblem.__init__, which does not run during precompilation of ThePackageYouThinkIsCausingTheProblem but does in any package that loads ThePackageYouThinkIsCausingTheProblem. To test this theory, set up a minimal working example (MWE), something like\n\n(@v1.10) pkg> generate MWE\n  Generating  project MWE:\n    MWE\\Project.toml\n    MWE\\src\\MWE.jl\n\nwhere the source code of MWE.jl is\n\nmodule MWE\nusing ThePackageYouThinkIsCausingTheProblem\nend\n\nand you've added ThePackageYouThinkIsCausingTheProblem to MWE's dependencies.\n\nIf that MWE reproduces the hang, you've found your culprit: ThePackageYouThinkIsCausingTheProblem.__init__ must be creating the Timer object. If the timer object can be safely closed, that's a good option. Otherwise, the most common solution is to avoid creating the timer while any package is being precompiled: add\n\nccall(:jl_generating_output, Cint, ()) == 1 && return nothing\n\nas the first line of ThePackageYouThinkIsCausingTheProblem.__init__, and it will avoid doing any initialization in any Julia process whose purpose is to precompile packages."},{"title":"Fixing package code to avoid hangs","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_fix","category":"section","text":"Search your package for suggestive words (here like \"Timer\") and see if you can identify where the problem is being created. Note that a method definition like\n\nmaketimer() = Timer(timer -> println(\"hi\"), 0; interval=1)\n\nis not problematic in and of itself: it can cause this problem only if maketimer gets called while the module is being defined. This might be happening from a top-level statement such as\n\nconst GLOBAL_TIMER = maketimer()\n\nor it might conceivably occur in a precompile workload.\n\nIf you struggle to identify the causative lines, then consider doing a binary search: comment out sections of your package (or include lines to omit entire files) until you've reduced the problem in scope."},{"title":"Lazy Artifacts","page":"Lazy Artifacts","location":"stdlib/LazyArtifacts.html#Lazy-Artifacts","category":"section","text":"In order for a package to download artifacts lazily, LazyArtifacts must be explicitly listed as a dependency of that package.\n\nFor further information on artifacts, see Artifacts."},{"title":"Initialization of the Julia runtime","page":"Initialization of the Julia runtime","location":"devdocs/init.html#Initialization-of-the-Julia-runtime","category":"section","text":"How does the Julia runtime execute julia -e 'println(\"Hello World!\")' ?"},{"title":"main()","page":"Initialization of the Julia runtime","location":"devdocs/init.html#main()","category":"section","text":"Execution starts at main() in cli/loader_exe.c, which calls jl_load_repl() in cli/loader_lib.c which loads a few libraries, eventually calling jl_repl_entrypoint() in src/jlapi.c.\n\njl_repl_entrypoint() calls libsupport_init() to set the C library locale and to initialize the \"ios\" library (see ios_init_stdstreams() and Legacy ios.c library).\n\nNext jl_parse_opts() is called to process command line options. Note that jl_parse_opts() only deals with options that affect code generation or early initialization. Other options are handled later by exec_options() in base/client.jl.\n\njl_parse_opts() stores command line options in the global jl_options struct."},{"title":"julia_init()","page":"Initialization of the Julia runtime","location":"devdocs/init.html#julia_init()","category":"section","text":"julia_init() in init.c is called by main() and calls _julia_init() in init.c.\n\n_julia_init() begins by calling libsupport_init() again (it does nothing the second time).\n\nrestore_signals() is called to zero the signal handler mask.\n\njl_resolve_sysimg_location() searches configured paths for the base system image. See Building the Julia system image.\n\njl_gc_init() sets up allocation pools and lists for weak refs, preserved values and finalization.\n\njl_init_frontend() loads and initializes a pre-compiled femtolisp image containing the scanner/parser.\n\njl_init_types() creates jl_datatype_t type description objects for the built-in types defined in julia.h. e.g.\n\njl_any_type = jl_new_abstracttype(jl_symbol(\"Any\"), core, NULL, jl_emptysvec);\njl_any_type->super = jl_any_type;\n\njl_type_type = jl_new_abstracttype(jl_symbol(\"Type\"), core, jl_any_type, jl_emptysvec);\n\njl_int32_type = jl_new_primitivetype(jl_symbol(\"Int32\"), core,\n                                     jl_any_type, jl_emptysvec, 32);\n\njl_init_tasks() creates the jl_datatype_t* jl_task_type object; initializes the global jl_root_task struct; and sets jl_current_task to the root task.\n\njl_init_codegen() initializes the LLVM library.\n\njl_init_serializer() initializes 8-bit serialization tags for builtin jl_value_t values.\n\nIf there is no sysimg file (!jl_options.image_file) then the Core and Main modules are created and boot.jl is evaluated:\n\njl_core_module = jl_new_module(jl_symbol(\"Core\"), NULL) creates the Julia Core module.\n\njl_init_intrinsic_functions() creates a new Julia module Intrinsics containing constant jl_intrinsic_type symbols. These define an integer code for each intrinsic function. emit_intrinsic() translates these symbols into LLVM instructions during code generation.\n\njl_init_primitives() hooks C functions up to Julia function symbols. e.g. the symbol Core.:(===)() is bound to C function pointer jl_f_is() by calling add_builtin_func(\"===\", jl_f_is).\n\njl_new_main_module() creates the global \"Main\" module and sets jl_current_task->current_module = jl_main_module.\n\nNote: _julia_init() then sets jl_root_task->current_module = jl_core_module. jl_root_task is an alias of jl_current_task at this point, so the current_module set by jl_new_main_module() above is overwritten.\n\njl_load(\"boot.jl\", sizeof(\"boot.jl\")) calls jl_parse_eval_all which repeatedly calls jl_toplevel_eval_flex() to execute boot.jl. <!– TODO – drill down into eval? –>\n\njl_get_builtin_hooks() initializes global C pointers to Julia globals defined in boot.jl.\n\njl_init_box_caches() pre-allocates global boxed integer value objects for values up to 1024. This speeds up allocation of boxed ints later on. e.g.:\n\njl_value_t *jl_box_uint8(uint32_t x)\n{\n    return boxed_uint8_cache[(uint8_t)x];\n}\n\n_julia_init() iterates over the jl_core_module->bindings.table looking for jl_datatype_t values and sets the type name's module prefix to jl_core_module.\n\njl_add_standard_imports(jl_main_module) does \"using Base\" in the \"Main\" module.\n\nNote: _julia_init() now reverts to jl_root_task->current_module = jl_main_module as it was before being set to jl_core_module above.\n\nPlatform specific signal handlers are initialized for SIGSEGV (OSX, Linux), and SIGFPE (Windows).\n\nOther signals (SIGINFO, SIGBUS, SIGILL, SIGTERM, SIGABRT, SIGQUIT, SIGSYS and SIGPIPE) are hooked up to sigdie_handler() which prints a backtrace.\n\njl_init_restored_module() calls jl_module_run_initializer() for each deserialized module to run the __init__() function.\n\nFinally sigint_handler() is hooked up to SIGINT and calls jl_throw(jl_interrupt_exception).\n\n_julia_init() then returns back to main() in cli/loader_exe.c and main() calls repl_entrypoint(argc, (char**)argv).\n\nsidebar: sysimg\nIf there is a sysimg file, it contains a pre-cooked image of the Core and Main modules (and whatever else is created by boot.jl). See Building the Julia system image.jl_restore_system_image() deserializes the saved sysimg into the current Julia runtime environment and initialization continues after jl_init_box_caches() below...Note: jl_restore_system_image() (and staticdata.c in general) uses the Legacy ios.c library."},{"title":"repl_entrypoint()","page":"Initialization of the Julia runtime","location":"devdocs/init.html#repl_entrypoint()","category":"section","text":"repl_entrypoint() loads the contents of argv[] into Base.ARGS.\n\nIf a .jl \"program\" file was supplied on the command line, then exec_program() calls jl_load(program,len) which calls jl_parse_eval_all which repeatedly calls jl_toplevel_eval_flex() to execute the program.\n\nHowever, in our example (julia -e 'println(\"Hello World!\")'), jl_get_global(jl_base_module, jl_symbol(\"_start\")) looks up Base._start and jl_apply() executes it."},{"title":"Base._start","page":"Initialization of the Julia runtime","location":"devdocs/init.html#Base._start","category":"section","text":"Base._start calls Base.exec_options which calls jl_parse_input_line(\"println(\"Hello World!\")\") to create an expression object and Core.eval(Main, ex) to execute the parsed expression ex in the module context of Main."},{"title":"Core.eval","page":"Initialization of the Julia runtime","location":"devdocs/init.html#Core.eval","category":"section","text":"Core.eval(Main, ex) calls jl_toplevel_eval_in(m, ex), which calls jl_toplevel_eval_flex. jl_toplevel_eval_flex implements a simple heuristic to decide whether to compile a given code thunk or run it by interpreter. When given println(\"Hello World!\"), it would usually decide to run the code by interpreter, in which case it calls jl_interpret_toplevel_thunk, which then calls eval_body.\n\nThe stack dump below shows how the interpreter works its way through various methods of Base.println() and Base.print() before arriving at write(s::IO, a::Array{T}) where T  which does ccall(jl_uv_write()).\n\njl_uv_write() calls uv_write() to write \"Hello World!\" to JL_STDOUT. See Libuv wrappers for stdio.:\n\nHello World!\n\nStack frame Source code Notes\njl_uv_write() jl_uv.c called though ccall\njulia_write_282942 stream.jl function write!(s::IO, a::Array{T}) where T\njulia_print_284639 ascii.jl print(io::IO, s::String) = (write(io, s); nothing)\njlcall_print_284639  \njl_apply() julia.h \njl_trampoline() builtins.c \njl_apply() julia.h \njl_apply_generic() gf.c Base.print(Base.TTY, String)\njl_apply() julia.h \njl_trampoline() builtins.c \njl_apply() julia.h \njl_apply_generic() gf.c Base.print(Base.TTY, String, Char, Char...)\njl_apply() julia.h \njl_f_apply() builtins.c \njl_apply() julia.h \njl_trampoline() builtins.c \njl_apply() julia.h \njl_apply_generic() gf.c Base.println(Base.TTY, String, String...)\njl_apply() julia.h \njl_trampoline() builtins.c \njl_apply() julia.h \njl_apply_generic() gf.c Base.println(String,)\njl_apply() julia.h \ndo_call() interpreter.c \neval_body() interpreter.c \njl_interpret_toplevel_thunk interpreter.c \njl_toplevel_eval_flex toplevel.c \njl_toplevel_eval_in toplevel.c \nCore.eval boot.jl \n\nSince our example has just one function call, which has done its job of printing \"Hello World!\", the stack now rapidly unwinds back to main()."},{"title":"jl_atexit_hook()","page":"Initialization of the Julia runtime","location":"devdocs/init.html#jl_atexit_hook()","category":"section","text":"main() calls jl_atexit_hook(). This calls Base._atexit, then calls jl_gc_run_all_finalizers() and cleans up libuv handles."},{"title":"julia_save()","page":"Initialization of the Julia runtime","location":"devdocs/init.html#julia_save()","category":"section","text":"Finally, main() calls julia_save(), which if requested on the command line, saves the runtime state to a new system image. See jl_compile_all() and jl_save_system_image()."},{"title":"StackTraces","page":"StackTraces","location":"base/stacktraces.html#StackTraces","category":"section","text":"The following methods and types in Base.StackTraces are not exported and need to be called e.g. as StackTraces.lookup(ptr)."},{"title":"Base.StackTraces.StackFrame","page":"StackTraces","location":"base/stacktraces.html#Base.StackTraces.StackFrame","category":"type","text":"StackFrame\n\nStack information representing execution context, with the following fields:\n\nfunc::Symbol\nThe name of the function containing the execution context.\nlinfo::Union{Method, Core.MethodInstance, Core.CodeInstance, Core.CodeInfo, Nothing}\nThe Method, MethodInstance, CodeInstance, or CodeInfo containing the execution context (if it could be found),    or nothing (for example, if the inlining was a result of macro expansion).\nfile::Symbol\nThe path to the file containing the execution context.\nline::Int\nThe line number in the file containing the execution context.\nfrom_c::Bool\nTrue if the code is from C.\ninlined::Bool\nTrue if the code is from an inlined frame.\npointer::UInt64\nRepresentation of the pointer to the execution context as returned by backtrace.\n\n\n\n\n\n"},{"title":"Base.StackTraces.StackTrace","page":"StackTraces","location":"base/stacktraces.html#Base.StackTraces.StackTrace","category":"type","text":"StackTrace\n\nAn alias for Vector{StackFrame} provided for convenience; returned by calls to stacktrace.\n\n\n\n\n\n"},{"title":"Base.StackTraces.stacktrace","page":"StackTraces","location":"base/stacktraces.html#Base.StackTraces.stacktrace","category":"function","text":"stacktrace([trace::Vector{Ptr{Cvoid}},] [c_funcs::Bool=false])::StackTrace\n\nReturn a stack trace in the form of a vector of StackFrames. (By default stacktrace doesn't return C functions, but this can be enabled.) When called without specifying a trace, stacktrace first calls backtrace.\n\n\n\n\n\n"},{"title":"Base.StackTraces.lookup","page":"StackTraces","location":"base/stacktraces.html#Base.StackTraces.lookup","category":"function","text":"lookup(pointer::Ptr{Cvoid})::Vector{StackFrame}\n\nGiven a pointer to an execution context (usually generated by a call to backtrace), looks up stack frame context information. Returns an array of frame information for all functions inlined at that point, innermost function first.\n\n\n\n\n\n"},{"title":"Base.StackTraces.remove_frames!","page":"StackTraces","location":"base/stacktraces.html#Base.StackTraces.remove_frames!","category":"function","text":"remove_frames!(stack::StackTrace, m::Module)\n\nReturn the StackTrace with all StackFrames from the provided Module removed.\n\n\n\n\n\nremove_frames!(stack::StackTrace, name::Symbol)\n\nTakes a StackTrace (a vector of StackFrames) and a function name (a Symbol) and removes the StackFrame specified by the function name from the StackTrace (also removing all frames above the specified function). Primarily used to remove StackTraces functions from the StackTrace prior to returning it.\n\n\n\n\n\n"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html#Julia-ASTs","category":"section","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser (e.g. the Meta.parse function), and manipulated by macros. It is a structured representation of code as it is written, constructed by julia-parser.scm from a character stream. Next there is a lowered form, or IR (intermediate representation), which is used by type inference and code generation. In the lowered form there are fewer types of nodes, all macros are expanded, and all control flow is converted to explicit branches and sequences of statements. The lowered form is constructed by julia-syntax.scm.\n\nFirst we will focus on the AST, since it is needed to write macros."},{"title":"Surface syntax AST","page":"Julia ASTs","location":"devdocs/ast.html#Surface-syntax-AST","category":"section","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers). There is generally a different expression head for each visually distinct syntactic form. Examples will be given in s-expression syntax. Each parenthesized list corresponds to an Expr, where the first element is the head. For example (call f x) corresponds to Expr(:call, :f, :x) in Julia."},{"title":"Calls","page":"Julia ASTs","location":"devdocs/ast.html#Calls","category":"section","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))\n\ndo syntax:\n\nf(x) do a,b\n    body\nend\n\nparses as (do (call f x) (-> (tuple a b) (block body)))."},{"title":"Operators","page":"Julia ASTs","location":"devdocs/ast.html#Operators","category":"section","text":"Most uses of operators are just function calls, so they are parsed with the head call. However some operators are special forms (not necessarily function calls), and in those cases the operator itself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\". Some operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument call. Finally, chains of comparisons have their own special expression structure.\n\nInput AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))"},{"title":"Bracketed forms","page":"Julia ASTs","location":"devdocs/ast.html#Bracketed-forms","category":"section","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)"},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros","category":"section","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)"},{"title":"Strings","page":"Julia ASTs","location":"devdocs/ast.html#Strings","category":"section","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")\n\nDoc string syntax:\n\n\"some docs\"\nf(x) = x\n\nparses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x)))."},{"title":"Imports and such","page":"Julia ASTs","location":"devdocs/ast.html#Imports-and-such","category":"section","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)\npublic a, b (public a b)\n\nusing has the same representation as import, but with expression head :using instead of :import.\n\nTo programmatically create a public statement, you can use Expr(:public, :a, :b) or, closer to regular code, Meta.parse(\"public a, b\"). This approach is necessary due to current limitations on public. The public keyword is only recognized at the syntactic top level within a file (parse_stmts) or module. This restriction was implemented to prevent breaking existing code that used public as an identifier when it was introduced in Julia 1.11."},{"title":"Numbers","page":"Julia ASTs","location":"devdocs/ast.html#Numbers","category":"section","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented directly as scheme numbers in the AST.\n\nInput AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")"},{"title":"Block forms","page":"Julia ASTs","location":"devdocs/ast.html#Block-forms","category":"section","text":"A block of statements is parsed as (block stmt1 stmt2 ...).\n\nIf statement:\n\nif a\n    b\nelseif c\n    d\nelse\n    e\nend\n\nparses as:\n\n(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6) e)))\n\nA while loop parses as (while condition body).\n\nA for loop parses as (for (= var iter) body). If there is more than one iteration specification, they are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body).\n\nbreak and continue are parsed as 0-argument expressions (break) and (continue).\n\nlet is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body), like for loops.\n\nA basic function definition is parsed as (function (call f x) body). A more complex example:\n\nfunction f(x::T; k = 1) where T\n    return x+1\nend\n\nparses as:\n\n(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))\n\nType definition:\n\nmutable struct Foo{T<:S}\n    x::T\nend\n\nparses as:\n\n(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))\n\nThe first argument is a boolean telling whether the type is mutable.\n\ntry blocks parse as (try try_block var catch_block finally_block). If no variable is present after catch, var is #f. If there is no finally clause, then the last argument is not present."},{"title":"Quote expressions","page":"Julia ASTs","location":"devdocs/ast.html#Quote-expressions","category":"section","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $. In Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms. Internally, there is also a need for code quoting without interpolation. In Julia's scheme code, non-interpolating quote is represented with the expression head inert.\n\ninert expressions are converted to Julia QuoteNode objects. These objects wrap a single value of any type, and when evaluated simply return that value.\n\nA quote expression whose argument is an atom also gets converted to a QuoteNode."},{"title":"Line numbers","page":"Julia ASTs","location":"devdocs/ast.html#Line-numbers","category":"section","text":"Source location information is represented as (line line_num file_name) where the third component is optional (and omitted when the current line number, but not file name, changes).\n\nThese expressions are represented as LineNumberNodes in Julia."},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros-2","category":"section","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope. The result of a macro expansion is automatically wrapped in (hygienic-scope block module [lno]), to represent the result of the new scope. The user can insert (escape block) inside to interpolate code from the caller. The lno is the __source__ argument of the macro, if included."},{"title":"Lowered form","page":"Julia ASTs","location":"devdocs/ast.html#Lowered-form","category":"section","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference, optimizations like inlining, and code generation. It is also less obvious to the human, since it results from a significant rearrangement of the input syntax.\n\nIn addition to Symbols and some number types, the following data types exist in lowered form:\n\nExpr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of subexpressions. While almost every part of a surface AST is represented by an Expr, the IR uses only a limited number of Exprs, mostly for calls and some top-level-only forms.\nSlotNumber\nIdentifies arguments and local variables by consecutive numbering. It has an integer-valued id field giving the slot index. The types of these slots can be found in the slottypes field of their CodeInfo object.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the referenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in the code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified by the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function. If the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a QuoteNode whose value field is the symbol a, in order to return the symbol itself instead of evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted by the compiler. The number (id) of an SSAValue is the code array index of the expression whose value it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined."},{"title":"Expr types","page":"Julia ASTs","location":"devdocs/ast.html#Expr-types","category":"section","text":"These symbols appear in the head field of Exprs in lowered form.\n\ncall\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the arguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a SlotNumber or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end. In the 1-argument form, the argument is a symbol. If this symbol already names a function in the current scope, nothing happens. If the symbol is undefined, a new function is created and assigned to the identifier specified by the symbol. If the symbol is defined but names a non-function, an error is raised. The definition of \"names a function\" is that the binding is constant, and refers to an object of singleton type. The rationale for this is that an instance of a singleton type uniquely identifies the type to add the method to. When the type has fields, it wouldn't be clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression first behaves like the 1-argument form above. This argument is ignored from then on. It can be nothing when methods are added strictly by type, (::T)(x) = x, or when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the argument types, and args[2][2] is a SimpleVector of type variables corresponding to the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a method to a function that also has methods defined in different scopes) this is an expression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer, GlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number of fields, or the minimum number of fields called by an inner constructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The arguments are the same as arguments 1, 2, and 4 of struct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4 are the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5 and replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered to this, and the type is always inserted by the compiler. This is very much an internal-only feature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to splat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has already been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception(ct).\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on error. Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a catch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this expression is true or false (true means bounds checks are disabled), it is pushed onto the stack. If the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds, otherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark the inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply copied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the arguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{<:Union{Symbol,Tuple{Symbol,UInt16}, Tuple{Symbol,UInt16,Bool}}: calling convention\nThe calling convention for the call, optionally with effects, and gc_safe (safe to execute concurrently to GC.).\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call. See Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[3] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[4] : constprop\nIndicates whether the opaque closure's identity may be used for constant propagation. The @opaque macro enables this by default, but this will cause additional inference which may be undesirable and prevents the code from running during precompile. If args[4] is a method, the argument is considered skipped.\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7"},{"title":"Method","page":"Julia ASTs","location":"devdocs/ast.html#ast-lowered-method","category":"section","text":"A unique'd container describing the shared metadata for a single method.\n\nname, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness. Uniqueness is required for efficiency, especially for incremental precompile and tracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by compression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, is_for_opaque_closure,\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method."},{"title":"MethodInstance","page":"Julia ASTs","location":"devdocs/ast.html#MethodInstance","category":"section","text":"A unique'd container describing a single callable signature for a Method. See especially Proper maintenance and care of multi-threading locks for important details on how to modify these fields safely.\n\nspecTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a def.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module, if this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes. For the MethodInstance at Method.unspecialized, this is the empty SimpleVector. But for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions. This works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance. Those optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation. Thus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation."},{"title":"CodeInstance","page":"Julia ASTs","location":"devdocs/ast.html#CodeInstance","category":"section","text":"def\nThe MethodInstance that this cache entry is derived from.\nowner\nA token that represents the owner of this CodeInstance. Will use jl_egal to match.\n\nrettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is also the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function, or it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JL_CALLABLE jl_value_t *(*)(jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called. If max_world is the special token value -1, the value is not yet known. It may continue to be used until we encounter a backedge that requires us to reconsider.\nTiming fields\ntime_infer_total: Total cost of computing inferred originally as wall-time from start to finish.\ntime_infer_cache_saved: The cost saved from time_infer_total by having caching. Adding this to time_infer_total should give a stable estimate for comparing the cost of two implementations or one implementation over time. This is generally an over-estimate of the time to infer something, since the cache is frequently effective at handling repeated work.\ntime_infer_self: Self cost of julia inference for inferred (a portion of time_infer_total). This is simply the incremental cost of compiling this one method, if given a fully populated cache of all call targets, even including constant inference results and LimitedAccuracy results, which generally are not in a cache.\ntime_compile: Self cost of llvm JIT compilation (e.g. of computing invoke from inferred). A total cost estimate can be computed by walking all of the edges contents and summing those, while accounting for cycles and duplicates. (This field currently does not include any measured AOT compile times.)"},{"title":"CodeInfo","page":"Julia ASTs","location":"devdocs/ast.html#CodeInfo","category":"section","text":"A (usually temporary) container for holding lowered (and possibly inferred) source code.\n\ncode\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n0x02 - assigned (only false if there are no assignment statements with this var on the left)\n0x08 - used (if there is any read or write of the slot)\n0x10 - statically assigned once\n0x20 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the function (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level 32 bits flags for each expression in the function. See the definition of jl_code_info_t in julia.h for more details.\n\nThese are only populated after inference (or by generated functions in some cases):\n\ndebuginfo\nAn object to retrieve source information for each statements, see How to interpret line numbers in a CodeInfo object.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any. This is mostly present for convenience, as (due to the way OpaqueClosures work) it is not necessarily the rettype used by codegen.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred.\n\nOptional Fields:\n\nslottypes\nAn array of types for the slots.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if necessary during inference.\n\nBoolean properties:\n\npropagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding @boundscheck blocks.\n\nUInt8 settings:\n\nconstprop, inlineable\n0 = use heuristic\n1 = aggressive\n2 = none\npurity Constructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details."},{"title":"How to interpret line numbers in a CodeInfo object","page":"Julia ASTs","location":"devdocs/ast.html#How-to-interpret-line-numbers-in-a-CodeInfo-object","category":"section","text":"There are 2 common forms for this data: one used internally that compresses the data somewhat and one used in the compiler. They contain the same basic info, but the compiler version is all mutable while the version used internally is not.\n\nMany consumers may be able to call Base.IRShow.buildLineInfoNode, Base.IRShow.append_scopes!, or Stacktraces.lookup(::InterpreterIP) to avoid needing to (re-)implement these details specifically.\n\nThe definitions of each of these are:\n\nstruct Core.DebugInfo\n    @noinline\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::SimpleVector{DebugInfo}\n    codelocs::String # compressed data\nend\nmutable struct Core.Compiler.DebugInfoStream\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::Vector{DebugInfo}\n    firstline::Int32 # the starting line for this block (specified by an index of 0)\n    codelocs::Vector{Int32} # for each statement:\n        # index into linetable (if defined), else a line number (in the file represented by def)\n        # then index into edges\n        # then index into edges[linetable]\nend\n\ndef : where this DebugInfo was defined (the Method, MethodInstance, or Symbol of file scope, for example)\nlinetable\nAnother DebugInfo that this was derived from, which contains the actual line numbers, such that this DebugInfo contains only the indexes into it. This avoids making copies, as well as makes it possible to track how each individual statement transformed from source to optimized, not just the separate line numbers. If def is not a Symbol, then that object replaces the current function object for the metadata on what function is conceptually being executed (e.g. think Cassette transforms here). The codelocs values described below also are interpreted as an index into the codelocs in this object, instead of being a line number itself.\nedges : Vector of the unique DebugInfo for every function inlined into this (which recursively have the edges for everything inlined into them).\nfirstline (when uncompressed to DebugInfoStream)\nThe line number associated with the begin statement (or other keyword such as function or quote) that delineates where this code definition \"starts\".\ncodelocs (when uncompressed to DebugInfoStream)\nA vector of indices, with 3 values for each statement in the IR plus one for the starting point of the block, that describe the stacktrace from that point:\nthe integer index into the linetable.codelocs field, giving the original location associated with each statement (including its syntactic edges), or zero indicating no change to the line number from the previously executed statement (which is not necessarily syntactic or lexical prior), or the line number itself if the linetable field is nothing.\nthe integer index into edges, giving the DebugInfo inlined there, or zero if there are no edges.\n(if entry 2 is non-zero) the integer index into edges[].codelocs, to interpret recursively for each function in the inlining stack, or zero indicating to use edges[].firstline as the line number.\nSpecial codes include:\n(zero, zero, *): no change to the line number or edges from the previous statement (you may choose to interpret this either syntactically or lexically). The inlining depth also might have changed, though most callers should ignore that.\n(zero, non-zero, *) : no line number, just edges (usually because of macro-expansion into top-level code)."},{"title":"Performance Tips","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-tips","category":"section","text":"In the following sections, we briefly go through a few techniques that can help make your Julia code run as fast as possible."},{"title":"Table of contents","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-tips-toc","category":"section","text":"Pages = [\"performance-tips.md\"]\nDepth = 3"},{"title":"General advice","page":"Performance Tips","location":"manual/performance-tips.html#General-advice","category":"section","text":""},{"title":"Performance critical code should be inside a function","page":"Performance Tips","location":"manual/performance-tips.html#Performance-critical-code-should-be-inside-a-function","category":"section","text":"Any code that is performance critical should be inside a function. Code inside functions tends to run much faster than top level code, due to how Julia's compiler works.\n\nThe use of functions is not only important for performance: functions are more reusable and testable, and clarify what steps are being done and what their inputs and outputs are, Write functions, not just scripts is also a recommendation of Julia's Styleguide.\n\nThe functions should take arguments, instead of operating directly on global variables, see the next point."},{"title":"Avoid untyped global variables","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-untyped-global-variables","category":"section","text":"The value of an untyped global variable might change at any point, possibly leading to a change of its type. This makes it difficult for the compiler to optimize code using global variables. This also applies to type-valued variables, i.e. type aliases on the global level. Variables should be local, or passed as arguments to functions, whenever possible.\n\nWe find that global names are frequently constants, and declaring them as such greatly improves performance:\n\nconst DEFAULT_VAL = 0\n\nIf a non-constant global is known to always be of the same type, the type should be annotated; const globals need not be annotated because their type is inferred from their initialization value.\n\nUses of untyped globals can be optimized by annotating their types at the point of use:\n\nglobal x = rand(1000)\n\nfunction loop_over_global()\n    s = 0.0\n    for i in x::Vector{Float64}\n        s += i\n    end\n    return s\nend\n\nPassing arguments to functions is better style. It leads to more reusable code and clarifies what the inputs and outputs are.\n\nnote: Note\nAll code in the REPL is evaluated in global scope, so a variable defined and assigned at top level will be a global variable. Variables defined at top level scope inside modules are also global.\n\nIn the following REPL session:\n\njulia> x = 1.0\n1.0\n\nis equivalent to:\n\njulia> global x = 1.0\n1.0\n\nso all the performance issues discussed previously apply."},{"title":"Measure performance with @time and pay attention to memory allocation","page":"Performance Tips","location":"manual/performance-tips.html#Measure-performance-with-[@time](@ref)-and-pay-attention-to-memory-allocation","category":"section","text":"A useful tool for measuring performance is the @time macro. We here repeat the example with the global variable above, but this time with the type annotation removed:\n\njulia> x = rand(1000);\n\njulia> function sum_global()\n           s = 0.0\n           for i in x\n               s += i\n           end\n           return s\n       end;\n\njulia> @time sum_global()\n  0.011539 seconds (9.08 k allocations: 373.386 KiB, 98.69% compilation time)\n523.0007221951678\n\njulia> @time sum_global()\n  0.000091 seconds (3.49 k allocations: 70.156 KiB)\n523.0007221951678\n\nOn the first call (@time sum_global()) the function gets compiled. (If you've not yet used @time in this session, it will also compile functions needed for timing.)  You should not take the results of this run seriously. For the second run, note that in addition to reporting the time, it also indicated that a significant amount of memory was allocated. We are here just computing a sum over all elements in a vector of 64-bit floats so there should be no need to allocate (heap) memory.\n\nWe should clarify that what @time reports is specifically heap allocations, which are typically needed for either mutable objects or for creating/growing variable-sized containers (such as Array or Dict, strings, or \"type-unstable\" objects whose type is only known at runtime). Allocating (or deallocating) such blocks of memory may require an expensive function call to libc (e.g. via malloc in C), and they must be tracked for garbage collection. In contrast, immutable values like numbers (except bignums), tuples, and immutable structs can be stored much more cheaply, e.g. in stack or CPU-register memory, so one doesn’t typically worry about the performance cost of \"allocating\" them.\n\nUnexpected memory allocation is almost always a sign of some problem with your code, usually a problem with type-stability or creating many small temporary arrays. Consequently, in addition to the allocation itself, it's very likely that the code generated for your function is far from optimal. Take such indications seriously and follow the advice below.\n\nFor more information about memory management and garbage collection in Julia, see Memory Management and Garbage Collection.\n\nIn this particular case, the memory allocation is due to the usage of a type-unstable global variable x, so if we instead pass x as an argument to the function it no longer allocates memory (the remaining allocation reported below is due to running the @time macro in global scope) and is significantly faster after the first call:\n\njulia> x = rand(1000);\n\njulia> function sum_arg(x)\n           s = 0.0\n           for i in x\n               s += i\n           end\n           return s\n       end;\n\njulia> @time sum_arg(x)\n  0.007551 seconds (3.98 k allocations: 200.548 KiB, 99.77% compilation time)\n523.0007221951678\n\njulia> @time sum_arg(x)\n  0.000006 seconds (1 allocation: 16 bytes)\n523.0007221951678\n\nThe 1 allocation seen is from running the @time macro itself in global scope. If we instead run the timing in a function, we can see that indeed no allocations are performed:\n\njulia> time_sum(x) = @time sum_arg(x);\n\njulia> time_sum(x)\n  0.000002 seconds\n523.0007221951678\n\nIn some situations, your function may need to allocate memory as part of its operation, and this can complicate the simple picture above. In such cases, consider using one of the tools below to diagnose problems, or write a version of your function that separates allocation from its algorithmic aspects (see Pre-allocating outputs).\n\nnote: Note\nFor more serious benchmarking, consider the BenchmarkTools.jl package which among other things evaluates the function multiple times in order to reduce noise."},{"title":"Break functions into multiple definitions","page":"Performance Tips","location":"manual/performance-tips.html#Break-functions-into-multiple-definitions","category":"section","text":"Writing a function as many small definitions allows the compiler to directly call the most applicable code, or even inline it.\n\nHere is an example of a \"compound function\" that should really be written as multiple definitions:\n\nusing LinearAlgebra\n\nfunction mynorm(A)\n    if isa(A, Vector)\n        return sqrt(real(dot(A,A)))\n    elseif isa(A, Matrix)\n        return maximum(svdvals(A))\n    else\n        error(\"mynorm: invalid argument\")\n    end\nend\n\nThis can be written more concisely and efficiently as:\n\nmynorm(x::Vector) = sqrt(real(dot(x, x)))\nmynorm(A::Matrix) = maximum(svdvals(A))\n\nIt should however be noted that the compiler is quite efficient at optimizing away the dead branches in code written as the mynorm example."},{"title":"Tools","page":"Performance Tips","location":"manual/performance-tips.html#tools","category":"section","text":"Julia and its package ecosystem includes tools that may help you diagnose problems and improve the performance of your code:\n\nProfiling allows you to measure the performance of your running code and identify lines that serve as bottlenecks. For complex projects, the ProfileView package can help you visualize your profiling results.\nThe JET package can help you find common performance problems in your code.\nUnexpectedly-large memory allocations–as reported by @time, @allocated, or the profiler (through calls to the garbage-collection routines)–hint that there might be issues with your code. If you don't see another reason for the allocations, suspect a type problem.  You can also start Julia with the --track-allocation=user option and examine the resulting *.mem files to see information about where those allocations occur. See Memory allocation analysis.\n@code_warntype generates a representation of your code that can be helpful in finding expressions that result in type uncertainty. See @code_warntype below."},{"title":"Type inference","page":"Performance Tips","location":"manual/performance-tips.html#Type-inference","category":"section","text":"In many languages with optional type declarations, adding declarations is the principal way to make code run faster. This is not the case in Julia. In Julia, the compiler generally knows the types of all function arguments, local variables, and expressions. However, there are a few specific instances where declarations are helpful."},{"title":"Avoid containers with abstract type parameters","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-abstract-container","category":"section","text":"When working with parameterized types, including arrays, it is best to avoid parameterizing with abstract types where possible.\n\nConsider the following:\n\njulia> a = Real[]\nReal[]\n\njulia> push!(a, 1); push!(a, 2.0); push!(a, π)\n3-element Vector{Real}:\n 1\n 2.0\n π = 3.1415926535897...\n\nBecause a is an array of abstract type Real, it must be able to hold any Real value. Since Real objects can be of arbitrary size and structure, a must be represented as an array of pointers to individually allocated Real objects. However, if we instead only allow numbers of the same type, e.g. Float64, to be stored in a these can be stored more efficiently:\n\njulia> a = Float64[]\nFloat64[]\n\njulia> push!(a, 1); push!(a, 2.0); push!(a,  π)\n3-element Vector{Float64}:\n 1.0\n 2.0\n 3.141592653589793\n\nAssigning numbers into a will now convert them to Float64 and a will be stored as a contiguous block of 64-bit floating-point values that can be manipulated efficiently.\n\nIf you cannot avoid containers with abstract value types, it is sometimes better to parametrize with Any to avoid runtime type checking. E.g. IdDict{Any, Any} performs better than IdDict{Type, Vector}\n\nSee also the discussion under Parametric Types."},{"title":"Avoid fields with abstract type","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-fields-with-abstract-type","category":"section","text":"Types can be declared without specifying the types of their fields:\n\njulia> struct MyAmbiguousType\n           a\n       end\n\nThis allows a to be of any type. This can often be useful, but it does have a downside: for objects of type MyAmbiguousType, the compiler will not be able to generate high-performance code. The reason is that the compiler uses the types of objects, not their values, to determine how to build code. Unfortunately, very little can be inferred about an object of type MyAmbiguousType:\n\njulia> b = MyAmbiguousType(\"Hello\")\nMyAmbiguousType(\"Hello\")\n\njulia> c = MyAmbiguousType(17)\nMyAmbiguousType(17)\n\njulia> typeof(b)\nMyAmbiguousType\n\njulia> typeof(c)\nMyAmbiguousType\n\nThe values of b and c have the same type, yet their underlying representation of data in memory is very different. Even if you stored just numeric values in field a, the fact that the memory representation of a UInt8 differs from a Float64 also means that the CPU needs to handle them using two different kinds of instructions. Since the required information is not available in the type, such decisions have to be made at run-time. This slows performance.\n\nYou can do better by declaring the type of a. Here, we are focused on the case where a might be any one of several types, in which case the natural solution is to use parameters. For example:\n\njulia> mutable struct MyType{T<:AbstractFloat}\n           a::T\n       end\n\nThis is a better choice than\n\njulia> mutable struct MyStillAmbiguousType\n           a::AbstractFloat\n       end\n\nbecause the first version specifies the type of a from the type of the wrapper object. For example:\n\njulia> m = MyType(3.2)\nMyType{Float64}(3.2)\n\njulia> t = MyStillAmbiguousType(3.2)\nMyStillAmbiguousType(3.2)\n\njulia> typeof(m)\nMyType{Float64}\n\njulia> typeof(t)\nMyStillAmbiguousType\n\nThe type of field a can be readily determined from the type of m, but not from the type of t. Indeed, in t it's possible to change the type of the field a:\n\njulia> typeof(t.a)\nFloat64\n\njulia> t.a = 4.5f0\n4.5f0\n\njulia> typeof(t.a)\nFloat32\n\nIn contrast, once m is constructed, the type of m.a cannot change:\n\njulia> m.a = 4.5f0\n4.5f0\n\njulia> typeof(m.a)\nFloat64\n\nThe fact that the type of m.a is known from m's type—coupled with the fact that its type cannot change mid-function—allows the compiler to generate highly-optimized code for objects like m but not for objects like t.\n\nOf course, all of this is true only if we construct m with a concrete type. We can break this by explicitly constructing it with an abstract type:\n\njulia> m = MyType{AbstractFloat}(3.2)\nMyType{AbstractFloat}(3.2)\n\njulia> typeof(m.a)\nFloat64\n\njulia> m.a = 4.5f0\n4.5f0\n\njulia> typeof(m.a)\nFloat32\n\nFor all practical purposes, such objects behave identically to those of MyStillAmbiguousType.\n\nIt's quite instructive to compare the sheer amount of code generated for a simple function\n\nfunc(m::MyType) = m.a+1\n\nusing\n\ncode_llvm(func, Tuple{MyType{Float64}})\ncode_llvm(func, Tuple{MyType{AbstractFloat}})\n\nFor reasons of length the results are not shown here, but you may wish to try this yourself. Because the type is fully-specified in the first case, the compiler doesn't need to generate any code to resolve the type at run-time. This results in shorter and faster code.\n\nOne should also keep in mind that not-fully-parameterized types behave like abstract types. For example, even though a fully specified Array{T,n} is concrete, Array itself with no parameters given is not concrete:\n\njulia> !isconcretetype(Array), !isabstracttype(Array), isstructtype(Array), !isconcretetype(Array{Int}), isconcretetype(Array{Int,1})\n(true, true, true, true, true)\n\nIn this case, it would be better to avoid declaring MyType with a field a::Array and instead declare the field as a::Array{T,N} or as a::A, where {T,N} or A are parameters of MyType.\n\nThe previous advice is especially useful when the fields of a struct are meant to be functions, or more generally callable objects. It is very tempting to define a struct as follows:\n\nstruct MyCallableWrapper\n    f::Function\nend\n\nBut since Function is an abstract type, every call to wrapper.f will require dynamic dispatch, due to the type instability of accessing the field f. Instead, you should write something like:\n\nstruct MyCallableWrapper{F}\n    f::F\nend\n\nwhich has nearly identical behavior but will be much faster (because the type instability is eliminated). Note that we do not impose F<:Function: this means callable objects which do not subtype Function are also allowed for the field f."},{"title":"Avoid fields with abstract containers","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-fields-with-abstract-containers","category":"section","text":"The same best practices also work for container types:\n\njulia> struct MySimpleContainer{A<:AbstractVector}\n           a::A\n       end\n\njulia> struct MyAmbiguousContainer{T}\n           a::AbstractVector{T}\n       end\n\njulia> struct MyAlsoAmbiguousContainer\n           a::Array\n       end\n\nFor example:\n\njulia> c = MySimpleContainer(1:3);\n\njulia> typeof(c)\nMySimpleContainer{UnitRange{Int64}}\n\njulia> c = MySimpleContainer([1:3;]);\n\njulia> typeof(c)\nMySimpleContainer{Vector{Int64}}\n\njulia> b = MyAmbiguousContainer(1:3);\n\njulia> typeof(b)\nMyAmbiguousContainer{Int64}\n\njulia> b = MyAmbiguousContainer([1:3;]);\n\njulia> typeof(b)\nMyAmbiguousContainer{Int64}\n\njulia> d = MyAlsoAmbiguousContainer(1:3);\n\njulia> typeof(d), typeof(d.a)\n(MyAlsoAmbiguousContainer, Vector{Int64})\n\njulia> d = MyAlsoAmbiguousContainer(1:1.0:3);\n\njulia> typeof(d), typeof(d.a)\n(MyAlsoAmbiguousContainer, Vector{Float64})\n\n\nFor MySimpleContainer, the object is fully-specified by its type and parameters, so the compiler can generate optimized functions. In most instances, this will probably suffice.\n\nWhile the compiler can now do its job perfectly well, there are cases where you might wish that your code could do different things depending on the element type of a. Usually the best way to achieve this is to wrap your specific operation (here, foo) in a separate function:\n\njulia> function sumfoo(c::MySimpleContainer)\n           s = 0\n           for x in c.a\n               s += foo(x)\n           end\n           s\n       end\nsumfoo (generic function with 1 method)\n\njulia> foo(x::Integer) = x\nfoo (generic function with 1 method)\n\njulia> foo(x::AbstractFloat) = round(x)\nfoo (generic function with 2 methods)\n\nThis keeps things simple, while allowing the compiler to generate optimized code in all cases.\n\nHowever, there are cases where you may need to declare different versions of the outer function for different element types or types of the AbstractVector of the field a in MySimpleContainer. You could do it like this:\n\njulia> function myfunc(c::MySimpleContainer{<:AbstractArray{<:Integer}})\n           return c.a[1]+1\n       end\nmyfunc (generic function with 1 method)\n\njulia> function myfunc(c::MySimpleContainer{<:AbstractArray{<:AbstractFloat}})\n           return c.a[1]+2\n       end\nmyfunc (generic function with 2 methods)\n\njulia> function myfunc(c::MySimpleContainer{Vector{T}}) where T <: Integer\n           return c.a[1]+3\n       end\nmyfunc (generic function with 3 methods)\n\njulia> myfunc(MySimpleContainer(1:3))\n2\n\njulia> myfunc(MySimpleContainer(1.0:3))\n3.0\n\njulia> myfunc(MySimpleContainer([1:3;]))\n4"},{"title":"Annotate values taken from untyped locations","page":"Performance Tips","location":"manual/performance-tips.html#Annotate-values-taken-from-untyped-locations","category":"section","text":"It is often convenient to work with data structures that may contain values of any type (arrays of type Array{Any}). But, if you're using one of these structures and happen to know the type of an element, it helps to share this knowledge with the compiler:\n\nfunction foo(a::Array{Any,1})\n    x = a[1]::Int32\n    b = x+1\n    ...\nend\n\nHere, we happened to know that the first element of a would be an Int32. Making an annotation like this has the added benefit that it will raise a run-time error if the value is not of the expected type, potentially catching certain bugs earlier.\n\nIn the case that the type of a[1] is not known precisely, x can be declared via x = convert(Int32, a[1])::Int32. The use of the convert function allows a[1] to be any object convertible to an Int32 (such as UInt8), thus increasing the genericity of the code by loosening the type requirement. Notice that convert itself needs a type annotation in this context in order to achieve type stability. This is because the compiler cannot deduce the type of the return value of a function, even convert, unless the types of all the function's arguments are known.\n\nType annotation will not enhance (and can actually hinder) performance if the type is abstract, or constructed at run-time. This is because the compiler cannot use the annotation to specialize the subsequent code, and the type-check itself takes time. For example, in the code:\n\nfunction nr(a, prec)\n    ctype = prec == 32 ? Float32 : Float64\n    b = Complex{ctype}(a)\n    c = (b + 1.0f0)::Complex{ctype}\n    abs(c)\nend\n\nthe annotation of c harms performance. To write performant code involving types constructed at run-time, use the function-barrier technique discussed below, and ensure that the constructed type appears among the argument types of the kernel function so that the kernel operations are properly specialized by the compiler. For example, in the above snippet, as soon as b is constructed, it can be passed to another function k, the kernel. If, for example, function k declares b as an argument of type Complex{T}, where T is a type parameter, then a type annotation appearing in an assignment statement within k of the form:\n\nc = (b + 1.0f0)::Complex{T}\n\ndoes not hinder performance (but does not help either) since the compiler can determine the type of c at the time k is compiled."},{"title":"Be aware of when Julia avoids specializing","page":"Performance Tips","location":"manual/performance-tips.html#Be-aware-of-when-Julia-avoids-specializing","category":"section","text":"As a heuristic, Julia avoids automatically specializing on argument type parameters in three specific cases: Type, Function, and Vararg. Julia will always specialize when the argument is used within the method, but not if the argument is just passed through to another function. This usually has no performance impact at runtime and improves compiler performance. If you find it does have a performance impact at runtime in your case, you can trigger specialization by adding a type parameter to the method declaration. Here are some examples:\n\nThis will not specialize:\n\nfunction f_type(t)  # or t::Type\n    x = ones(t, 10)\n    return sum(map(sin, x))\nend\n\nbut this will:\n\nfunction g_type(t::Type{T}) where T\n    x = ones(T, 10)\n    return sum(map(sin, x))\nend\n\nThese will not specialize:\n\nf_func(f, num) = ntuple(f, div(num, 2))\ng_func(g::Function, num) = ntuple(g, div(num, 2))\n\nbut this will:\n\nh_func(h::H, num) where {H} = ntuple(h, div(num, 2))\n\nThis will not specialize:\n\nf_vararg(x::Int...) = tuple(x...)\n\nbut this will:\n\ng_vararg(x::Vararg{Int, N}) where {N} = tuple(x...)\n\nOne only needs to introduce a single type parameter to force specialization, even if the other types are unconstrained. For example, this will also specialize, and is useful when the arguments are not all of the same type:\n\nh_vararg(x::Vararg{Any, N}) where {N} = tuple(x...)\n\nNote that @code_typed and friends will always show you specialized code, even if Julia would not normally specialize that method call. You need to check the method internals if you want to see whether specializations are generated when argument types are changed, i.e., if Base.specializations(@which f(...)) contains specializations for the argument in question."},{"title":"Write \"type-stable\" functions","page":"Performance Tips","location":"manual/performance-tips.html#Write-\"type-stable\"-functions","category":"section","text":"When possible, it helps to ensure that a function always returns a value of the same type. Consider the following definition:\n\npos(x) = x < 0 ? 0 : x\n\nAlthough this seems innocent enough, the problem is that 0 is an integer (of type Int) and x might be of any type. Thus, depending on the value of x, this function might return a value of either of two types. This behavior is allowed, and may be desirable in some cases. But it can easily be fixed as follows:\n\npos(x) = x < 0 ? zero(x) : x\n\nThere is also a oneunit function, and a more general oftype(x, y) function, which returns y converted to the type of x."},{"title":"Avoid changing the type of a variable","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-changing-the-type-of-a-variable","category":"section","text":"An analogous \"type-stability\" problem exists for variables used repeatedly within a function:\n\nfunction foo()\n    x = 1\n    for i = 1:10\n        x /= rand()\n    end\n    return x\nend\n\nLocal variable x starts as an integer, and after one loop iteration becomes a floating-point number (the result of / operator). This makes it more difficult for the compiler to optimize the body of the loop. There are several possible fixes:\n\nInitialize x with x = 1.0\nDeclare the type of x explicitly as x::Float64 = 1\nUse an explicit conversion by x = oneunit(Float64)\nInitialize with the first loop iteration, to x = 1 / rand(), then loop for i = 2:10"},{"title":"Separate kernel functions (aka, function barriers)","page":"Performance Tips","location":"manual/performance-tips.html#kernel-functions","category":"section","text":"Many functions follow a pattern of performing some set-up work, and then running many iterations to perform a core computation. Where possible, it is a good idea to put these core computations in separate functions. For example, the following contrived function returns an array of a randomly-chosen type:\n\njulia> function strange_twos(n)\n           a = Vector{rand(Bool) ? Int64 : Float64}(undef, n)\n           for i = 1:n\n               a[i] = 2\n           end\n           return a\n       end;\n\njulia> strange_twos(3)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\nThis should be written as:\n\njulia> function fill_twos!(a)\n           for i = eachindex(a)\n               a[i] = 2\n           end\n       end;\n\njulia> function strange_twos(n)\n           a = Vector{rand(Bool) ? Int64 : Float64}(undef, n)\n           fill_twos!(a)\n           return a\n       end;\n\njulia> strange_twos(3)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\nJulia's compiler specializes code for argument types at function boundaries, so in the original implementation it does not know the type of a during the loop (since it is chosen randomly). Therefore the second version is generally faster since the inner loop can be recompiled as part of fill_twos! for different types of a.\n\nThe second form is also often better style and can lead to more code reuse.\n\nThis pattern is used in several places in Julia Base. For example, see vcat and hcat in abstractarray.jl, or the fill! function, which we could have used instead of writing our own fill_twos!.\n\nFunctions like strange_twos occur when dealing with data of uncertain type, for example data loaded from an input file that might contain either integers, floats, strings, or something else."},{"title":"@code_warntype","page":"Performance Tips","location":"manual/performance-tips.html#man-code-warntype","category":"section","text":"The macro @code_warntype (or its function variant code_warntype) can sometimes be helpful in diagnosing type-related problems. Here's an example:\n\njulia> @noinline pos(x) = x < 0 ? 0 : x;\n\njulia> function f(x)\n           y = pos(x)\n           return sin(y*x + 1)\n       end;\n\njulia> @code_warntype f(3.2)\nMethodInstance for f(::Float64)\n  from f(x) @ Main REPL[9]:1\nArguments\n  #self#::Core.Const(f)\n  x::Float64\nLocals\n  y::Union{Float64, Int64}\nBody::Float64\n1 ─      (y = Main.pos(x))\n│   %2 = (y * x)::Float64\n│   %3 = (%2 + 1)::Float64\n│   %4 = Main.sin(%3)::Float64\n└──      return %4\n\nInterpreting the output of @code_warntype, like that of its cousins @code_lowered, @code_typed, @code_llvm, and @code_native, takes a little practice. Your code is being presented in form that has been heavily digested on its way to generating compiled machine code. Most of the expressions are annotated by a type, indicated by the ::T (where T might be Float64, for example). The most important characteristic of @code_warntype is that non-concrete types are displayed in red; since this document is written in Markdown, which has no color, in this document, red text is denoted by uppercase.\n\nAt the top, the inferred return type of the function is shown as Body::Float64. The next lines represent the body of f in Julia's SSA IR form. The numbered boxes are labels and represent targets for jumps (via goto) in your code. Looking at the body, you can see that the first thing that happens is that pos is called and the return value has been inferred as the Union type Union{Float64, Int64} shown in uppercase since it is a non-concrete type. This means that we cannot know the exact return type of pos based on the input types. However, the result of y*xis a Float64 no matter if y is a Float64 or Int64 The net result is that f(x::Float64) will not be type-unstable in its output, even if some of the intermediate computations are type-unstable.\n\nHow you use this information is up to you. Obviously, it would be far and away best to fix pos to be type-stable: if you did so, all of the variables in f would be concrete, and its performance would be optimal. However, there are circumstances where this kind of ephemeral type instability might not matter too much: for example, if pos is never used in isolation, the fact that f's output is type-stable (for Float64 inputs) will shield later code from the propagating effects of type instability. This is particularly relevant in cases where fixing the type instability is difficult or impossible. In such cases, the tips above (e.g., adding type annotations and/or breaking up functions) are your best tools to contain the \"damage\" from type instability. Also, note that even Julia Base has functions that are type unstable. For example, the function findfirst returns the index into an array where a key is found, or nothing if it is not found, a clear type instability. In order to make it easier to find the type instabilities that are likely to be important, Unions containing either missing or nothing are color highlighted in yellow, instead of red.\n\nThe following examples may help you interpret expressions marked as containing non-concrete types:\n\nFunction body starting with Body::Union{T1,T2})\nInterpretation: function with unstable return type\nSuggestion: make the return value type-stable, even if you have to annotate it\ninvoke Main.g(%%x::Int64)::Union{Float64, Int64}\nInterpretation: call to a type-unstable function g.\nSuggestion: fix the function, or if necessary annotate the return value\ninvoke Base.getindex(%%x::Array{Any,1}, 1::Int64)::Any\nInterpretation: accessing elements of poorly-typed arrays\nSuggestion: use arrays with better-defined types, or if necessary annotate the type of individual element accesses\nBase.getfield(%%x, :(:data))::Array{Float64,N} where N\nInterpretation: getting a field that is of non-concrete type. In this case, the type of x, say ArrayContainer, had a field data::Array{T}. But Array needs the dimension N, too, to be a concrete type.\nSuggestion: use concrete types like Array{T,3} or Array{T,N}, where N is now a parameter of ArrayContainer"},{"title":"Performance of captured variable","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-captured","category":"section","text":"Consider the following example that defines an inner function:\n\nfunction abmult(r::Int)\n    if r < 0\n        r = -r\n    end\n    f = x -> x * r\n    return f\nend\n\nFunction abmult returns a function f that multiplies its argument by the absolute value of r. The inner function assigned to f is called a \"closure\". Inner functions are also used by the language for do-blocks and for generator expressions.\n\nThis style of code presents performance challenges for the language. The parser, when translating it into lower-level instructions, substantially reorganizes the above code by extracting the inner function to a separate code block.  \"Captured\" variables such as r that are shared by inner functions and their enclosing scope are also extracted into a heap-allocated \"box\" accessible to both inner and outer functions because the language specifies that r in the inner scope must be identical to r in the outer scope even after the outer scope (or another inner function) modifies r.\n\nThe discussion in the preceding paragraph referred to the \"parser\", that is, the phase of compilation that takes place when the module containing abmult is first loaded, as opposed to the later phase when it is first invoked. The parser does not \"know\" that Int is a fixed type, or that the statement r = -r transforms an Int to another Int. The magic of type inference takes place in the later phase of compilation.\n\nThus, the parser does not know that r has a fixed type (Int). Nor that r does not change value once the inner function is created (so that the box is unneeded). Therefore, the parser emits code for box that holds an object with an abstract type such as Any, which requires run-time type dispatch for each occurrence of r. This can be verified by applying @code_warntype to the above function. Both the boxing and the run-time type dispatch can cause loss of performance.\n\nIf captured variables are used in a performance-critical section of the code, then the following tips help ensure that their use is performant. First, if it is known that a captured variable does not change its type, then this can be declared explicitly with a type annotation (on the variable, not the right-hand side):\n\nfunction abmult2(r0::Int)\n    r::Int = r0\n    if r < 0\n        r = -r\n    end\n    f = x -> x * r\n    return f\nend\n\nThe type annotation partially recovers lost performance due to capturing because the parser can associate a concrete type to the object in the box. Going further, if the captured variable does not need to be boxed at all (because it will not be reassigned after the closure is created), this can be indicated with let blocks as follows.\n\nfunction abmult3(r::Int)\n    if r < 0\n        r = -r\n    end\n    f = let r = r\n            x -> x * r\n    end\n    return f\nend\n\nThe let block creates a new variable r whose scope is only the inner function. The second technique recovers full language performance in the presence of captured variables. Note that this is a rapidly evolving aspect of the compiler, and it is likely that future releases will not require this degree of programmer annotation to attain performance. In the mean time, some user-contributed packages like FastClosures automate the insertion of let statements as in abmult3."},{"title":"Use @__FUNCTION__ for recursive closures","page":"Performance Tips","location":"manual/performance-tips.html#Use-@__FUNCTION__-for-recursive-closures","category":"section","text":"For recursive closures specifically, the @__FUNCTION__ macro can avoid both type instability and boxing.\n\nFirst, let's see the unoptimized version:\n\nfunction make_fib_unoptimized()\n    fib(n) = n <= 1 ? 1 : fib(n - 1) + fib(n - 2)  # fib is boxed\n    return fib\nend\n\nThe fib function is boxed, meaning the return type is inferred as Any:\n\n@code_warntype make_fib_unoptimized()\n\nNow, to eliminate this type instability, we can instead use @__FUNCTION__ to refer to the concrete function object:\n\nfunction make_fib_optimized()\n    fib(n) = n <= 1 ? 1 : (@__FUNCTION__)(n - 1) + (@__FUNCTION__)(n - 2)\n    return fib\nend\n\nThis gives us a concrete return type:\n\n@code_warntype make_fib_optimized()"},{"title":"Types with values-as-parameters","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-value-type","category":"section","text":"Let's say you want to create an N-dimensional array that has size 3 along each axis. Such arrays can be created like this:\n\njulia> A = fill(5.0, (3, 3))\n3×3 Matrix{Float64}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n\nThis approach works very well: the compiler can figure out that A is an Array{Float64,2} because it knows the type of the fill value (5.0::Float64) and the dimensionality ((3, 3)::NTuple{2,Int}). This implies that the compiler can generate very efficient code for any future usage of A in the same function.\n\nBut now let's say you want to write a function that creates a 3×3×... array in arbitrary dimensions; you might be tempted to write a function\n\njulia> function array3(fillval, N)\n           fill(fillval, ntuple(d->3, N))\n       end\narray3 (generic function with 1 method)\n\njulia> array3(5.0, 2)\n3×3 Matrix{Float64}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n\nThis works, but (as you can verify for yourself using @code_warntype array3(5.0, 2)) the problem is that the output type cannot be inferred: the argument N is a value of type Int, and type-inference does not (and cannot) predict its value in advance. This means that code using the output of this function has to be conservative, checking the type on each access of A; such code will be very slow.\n\nNow, one very good way to solve such problems is by using the function-barrier technique. However, in some cases you might want to eliminate the type-instability altogether. In such cases, one approach is to pass the dimensionality as a parameter, for example through Val{T}() (see \"Value types\"):\n\njulia> function array3(fillval, ::Val{N}) where N\n           fill(fillval, ntuple(d->3, Val(N)))\n       end\narray3 (generic function with 1 method)\n\njulia> array3(5.0, Val(2))\n3×3 Matrix{Float64}:\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n 5.0  5.0  5.0\n\nJulia has a specialized version of ntuple that accepts a Val{::Int} instance as the second parameter; by passing N as a type-parameter, you make its \"value\" known to the compiler. Consequently, this version of array3 allows the compiler to predict the return type.\n\nHowever, making use of such techniques can be surprisingly subtle. For example, it would be of no help if you called array3 from a function like this:\n\nfunction call_array3(fillval, n)\n    A = array3(fillval, Val(n))\nend\n\nHere, you've created the same problem all over again: the compiler can't guess what n is, so it doesn't know the type of Val(n). Attempting to use Val, but doing so incorrectly, can easily make performance worse in many situations. (Only in situations where you're effectively combining Val with the function-barrier trick, to make the kernel function more efficient, should code like the above be used.)\n\nAn example of correct usage of Val would be:\n\nfunction filter3(A::AbstractArray{T,N}) where {T,N}\n    kernel = array3(1, Val(N))\n    filter(A, kernel)\nend\n\nIn this example, N is passed as a parameter, so its \"value\" is known to the compiler. Essentially, Val(T) works only when T is either hard-coded/literal (Val(3)) or already specified in the type-domain."},{"title":"The dangers of abusing multiple dispatch (aka, more on types with values-as-parameters)","page":"Performance Tips","location":"manual/performance-tips.html#The-dangers-of-abusing-multiple-dispatch-(aka,-more-on-types-with-values-as-parameters)","category":"section","text":"Once one learns to appreciate multiple dispatch, there's an understandable tendency to go overboard and try to use it for everything. For example, you might imagine using it to store information, e.g.\n\nstruct Car{Make, Model}\n    year::Int\n    ...more fields...\nend\n\nand then dispatch on objects like Car{:Honda,:Accord}(year, args...).\n\nThis might be worthwhile when either of the following are true:\n\nYou require CPU-intensive processing on each Car, and it becomes vastly more efficient if you know the Make and Model at compile time and the total number of different Make or Model that will be used is not too large.\nYou have homogeneous lists of the same type of Car to process, so that you can store them all in an Array{Car{:Honda,:Accord},N}.\n\nWhen the latter holds, a function processing such a homogeneous array can be productively specialized: Julia knows the type of each element in advance (all objects in the container have the same concrete type), so Julia can \"look up\" the correct method calls when the function is being compiled (obviating the need to check at run-time) and thereby emit efficient code for processing the whole list.\n\nWhen these do not hold, then it's likely that you'll get no benefit; worse, the resulting \"combinatorial explosion of types\" will be counterproductive. If items[i+1] has a different type than item[i], Julia has to look up the type at run-time, search for the appropriate method in method tables, decide (via type intersection) which one matches, determine whether it has been JIT-compiled yet (and do so if not), and then make the call. In essence, you're asking the full type- system and JIT-compilation machinery to basically execute the equivalent of a switch statement or dictionary lookup in your own code.\n\nSome run-time benchmarks comparing (1) type dispatch, (2) dictionary lookup, and (3) a \"switch\" statement can be found on discourse.\n\nPerhaps even worse than the run-time impact is the compile-time impact: Julia will compile specialized functions for each different Car{Make, Model}; if you have hundreds or thousands of such types, then every function that accepts such an object as a parameter (from a custom get_year function you might write yourself, to the generic push! function in Julia Base) will have hundreds or thousands of variants compiled for it. Each of these increases the size of the cache of compiled code, the length of internal lists of methods, etc. Excess enthusiasm for values-as-parameters can easily waste enormous resources."},{"title":"Memory management and arrays","page":"Performance Tips","location":"manual/performance-tips.html#Memory-management-and-arrays","category":"section","text":""},{"title":"Pre-allocate outputs","page":"Performance Tips","location":"manual/performance-tips.html#Pre-allocate-outputs","category":"section","text":"If your function returns an Array or some other complex type, it may have to allocate memory. Unfortunately, oftentimes allocation and its converse, garbage collection, are substantial bottlenecks.\n\nSometimes you can circumvent the need to allocate memory on each function call by preallocating the output. As a trivial example, compare\n\njulia> function xinc(x)\n           return [x + i for i  in 1:3000]\n       end;\n\njulia> function loopinc()\n           y = 0\n           for i = 1:10^5\n               ret = xinc(i)\n               y += ret[2]\n           end\n           return y\n       end;\n\nwith\n\njulia> function xinc!(ret::AbstractVector{T}, x::T) where T\n           for i in 1:3000\n               ret[i] = x+i\n           end\n           nothing\n       end;\n\njulia> function loopinc_prealloc()\n           ret = Vector{Int}(undef, 3000)\n           y = 0\n           for i = 1:10^5\n               xinc!(ret, i)\n               y += ret[2]\n           end\n           return y\n       end;\n\nTiming results:\n\njulia> @time loopinc()\n  0.297454 seconds (200.00 k allocations: 2.239 GiB, 39.80% gc time)\n5000250000\n\njulia> @time loopinc_prealloc()\n  0.009410 seconds (2 allocations: 23.477 KiB)\n5000250000\n\nPreallocation has other advantages, for example by allowing the caller to control the \"output\" type from an algorithm. In the example above, we could have passed a SubArray rather than an Array, had we so desired.\n\nTaken to its extreme, pre-allocation can make your code uglier, so performance measurements and some judgment may be required. However, for \"vectorized\" (element-wise) functions, the convenient syntax x .= f.(y) can be used for in-place operations with fused loops and no temporary arrays (see the dot syntax for vectorizing functions)."},{"title":"Consider using views for slices","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-views","category":"section","text":"In Julia, an array \"slice\" expression like array[1:5, :] creates a copy of that data (except on the left-hand side of an assignment, where array[1:5, :] = ... assigns in-place to that portion of array). If you are doing many operations on the slice, this can be good for performance because it is more efficient to work with a smaller contiguous copy than it would be to index into the original array. On the other hand, if you are just doing a few simple operations on the slice, the cost of the allocation and copy operations can be substantial.\n\nAn alternative is to create a \"view\" of the array, which is an array object (a SubArray) that actually references the data of the original array in-place, without making a copy. (If you write to a view, it modifies the original array's data as well.) This can be done for individual slices by calling view, or more simply for a whole expression or block of code by putting @views in front of that expression. For example:\n\njulia> fcopy(x) = sum(x[2:end-1]);\n\njulia> @views fview(x) = sum(x[2:end-1]);\n\njulia> x = rand(10^6);\n\njulia> @time fcopy(x);\n  0.003051 seconds (3 allocations: 7.629 MB)\n\njulia> @time fview(x);\n  0.001020 seconds (1 allocation: 16 bytes)\n\nNotice both the 3× speedup and the decreased memory allocation of the fview version of the function."},{"title":"Consider StaticArrays.jl for small fixed-size vector/matrix operations","page":"Performance Tips","location":"manual/performance-tips.html#Consider-StaticArrays.jl-for-small-fixed-size-vector/matrix-operations","category":"section","text":"If your application involves many small (< 100 element) arrays of fixed sizes (i.e. the size is known prior to execution), then you might want to consider using the StaticArrays.jl package. This package allows you to represent such arrays in a way that avoids unnecessary heap allocations and allows the compiler to specialize code for the size of the array, e.g. by completely unrolling vector operations (eliminating the loops) and storing elements in CPU registers.\n\nFor example, if you are doing computations with 2d geometries, you might have many computations with 2-component vectors. By using the SVector type from StaticArrays.jl, you can use convenient vector notation and operations like norm(3v - w) on vectors v and w, while allowing the compiler to unroll the code to a minimal computation equivalent to @inbounds hypot(3v[1]-w[1], 3v[2]-w[2])."},{"title":"More dots: Fuse vectorized operations","page":"Performance Tips","location":"manual/performance-tips.html#More-dots:-Fuse-vectorized-operations","category":"section","text":"Julia has a special dot syntax that converts any scalar function into a \"vectorized\" function call, and any operator into a \"vectorized\" operator, with the special property that nested \"dot calls\" are fusing: they are combined at the syntax level into a single loop, without allocating temporary arrays. If you use .= and similar assignment operators, the result can also be stored in-place in a pre-allocated array (see above).\n\nIn a linear-algebra context, this means that even though operations like vector + vector and vector * scalar are defined, it can be advantageous to instead use vector .+ vector and vector .* scalar because the resulting loops can be fused with surrounding computations. For example, consider the two functions:\n\njulia> f(x) = 3x.^2 + 4x + 7x.^3;\n\njulia> fdot(x) = @. 3x^2 + 4x + 7x^3; # equivalent to 3 .* x.^2 .+ 4 .* x .+ 7 .* x.^3\n\nBoth f and fdot compute the same thing. However, fdot (defined with the help of the @. macro) is significantly faster when applied to an array:\n\njulia> x = rand(10^6);\n\njulia> @time f(x);\n  0.019049 seconds (16 allocations: 45.777 MiB, 18.59% gc time)\n\njulia> @time fdot(x);\n  0.002790 seconds (6 allocations: 7.630 MiB)\n\njulia> @time f.(x);\n  0.002626 seconds (8 allocations: 7.630 MiB)\n\nThat is, fdot(x) is ten times faster and allocates 1/6 the memory of f(x), because each * and + operation in f(x) allocates a new temporary array and executes in a separate loop. In this example f.(x) is as fast as fdot(x) but in many contexts it is more convenient to sprinkle some dots in your expressions than to define a separate function for each vectorized operation."},{"title":"Fewer dots: Unfuse certain intermediate broadcasts","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-unfuse","category":"section","text":"The dot loop fusion mentioned above enables concise and idiomatic code to express highly performant operations. However, it is important to remember that the fused operation will be computed at every iteration of the broadcast. This means that in some situations, particularly in the presence of composed or multidimensional broadcasts, an expression with dot calls may be computing a function more times than intended. As an example, say we want to build a random matrix whose rows have Euclidean norm one. We might write something like the following:\n\njulia> x = rand(1000, 1000);\n\njulia> d = sum(abs2, x; dims=2);\n\njulia> @time x ./= sqrt.(d);\n  0.002049 seconds (4 allocations: 96 bytes)\n\nThis will work. However, this expression will actually recompute sqrt(d[i]) for every element in the row x[i, :], meaning that many more square roots are computed than necessary. To see precisely over which indices the broadcast will iterate, we can call Broadcast.combine_axes on the arguments of the fused expression. This will return a tuple of ranges whose entries correspond to the axes of iteration; the product of lengths of these ranges will be the total number of calls to the fused operation.\n\nIt follows that when some components of the broadcast expression are constant along an axis—like the sqrt along the second dimension in the preceding example—there is potential for a performance improvement by forcibly \"unfusing\" those components, i.e. allocating the result of the broadcasted operation in advance and reusing the cached value along its constant axis. Some such potential approaches are to use temporary variables, wrap components of a dot expression in identity, or use an equivalent intrinsically vectorized (but non-fused) function.\n\njulia> @time let s = sqrt.(d); x ./= s end;\n  0.000809 seconds (5 allocations: 8.031 KiB)\n\njulia> @time x ./= identity(sqrt.(d));\n  0.000608 seconds (5 allocations: 8.031 KiB)\n\njulia> @time x ./= map(sqrt, d);\n  0.000611 seconds (4 allocations: 8.016 KiB)\n\nAny of these options yields approximately a three-fold speedup at the cost of an allocation; for large broadcastables this speedup can be asymptotically very large."},{"title":"Access arrays in memory order, along columns","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-column-major","category":"section","text":"Multidimensional arrays in Julia are stored in column-major order. This means that arrays are stacked one column at a time. This can be verified using the vec function or the syntax [:] as shown below (notice that the array is ordered [1 3 2 4], not [1 2 3 4]):\n\njulia> x = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> x[:]\n4-element Vector{Int64}:\n 1\n 3\n 2\n 4\n\nThis convention for ordering arrays is common in many languages like Fortran, Matlab, and R (to name a few). The alternative to column-major ordering is row-major ordering, which is the convention adopted by C and Python (numpy) among other languages. Remembering the ordering of arrays can have significant performance effects when looping over arrays. A rule of thumb to keep in mind is that with column-major arrays, the first index changes most rapidly. Essentially this means that looping will be faster if the inner-most loop index is the first to appear in a slice expression. Keep in mind that indexing an array with : is an implicit loop that iteratively accesses all elements within a particular dimension; it can be faster to extract columns than rows, for example.\n\nConsider the following contrived example. Imagine we wanted to write a function that accepts a Vector and returns a square Matrix with either the rows or the columns filled with copies of the input vector. Assume that it is not important whether rows or columns are filled with these copies (perhaps the rest of the code can be easily adapted accordingly). We could conceivably do this in at least four ways (in addition to the recommended call to the built-in repeat):\n\nfunction copy_cols(x::Vector{T}) where T\n    inds = axes(x, 1)\n    out = similar(Array{T}, inds, inds)\n    for i = inds\n        out[:, i] = x\n    end\n    return out\nend\n\nfunction copy_rows(x::Vector{T}) where T\n    inds = axes(x, 1)\n    out = similar(Array{T}, inds, inds)\n    for i = inds\n        out[i, :] = x\n    end\n    return out\nend\n\nfunction copy_col_row(x::Vector{T}) where T\n    inds = axes(x, 1)\n    out = similar(Array{T}, inds, inds)\n    for col = inds, row = inds\n        out[row, col] = x[row]\n    end\n    return out\nend\n\nfunction copy_row_col(x::Vector{T}) where T\n    inds = axes(x, 1)\n    out = similar(Array{T}, inds, inds)\n    for row = inds, col = inds\n        out[row, col] = x[col]\n    end\n    return out\nend\n\nNow we will time each of these functions using the same random 10000 by 1 input vector:\n\njulia> x = randn(10000);\n\njulia> fmt(f) = println(rpad(string(f)*\": \", 14, ' '), @elapsed f(x))\n\njulia> map(fmt, [copy_cols, copy_rows, copy_col_row, copy_row_col]);\ncopy_cols:    0.331706323\ncopy_rows:    1.799009911\ncopy_col_row: 0.415630047\ncopy_row_col: 1.721531501\n\nNotice that copy_cols is much faster than copy_rows. This is expected because copy_cols respects the column-based memory layout of the Matrix and fills it one column at a time. Additionally, copy_col_row is much faster than copy_row_col because it follows our rule of thumb that the first element to appear in a slice expression should be coupled with the inner-most loop."},{"title":"Copying data is not always bad","page":"Performance Tips","location":"manual/performance-tips.html#Copying-data-is-not-always-bad","category":"section","text":"Arrays are stored contiguously in memory, lending themselves to CPU vectorization and fewer memory accesses due to caching. These are the same reasons that it is recommended to access arrays in column-major order (see above). Irregular access patterns and non-contiguous views can drastically slow down computations on arrays because of non-sequential memory access.\n\nCopying irregularly-accessed data into a contiguous array before repeatedly accessing it can result in a large speedup, such as in the example below. Here, a matrix is being accessed at randomly-shuffled indices before being multiplied. Copying into plain arrays speeds up the multiplication even with the added cost of copying and allocation.\n\njulia> using Random\n\njulia> A = randn(3000, 3000);\n\njulia> x = randn(2000);\n\njulia> inds = shuffle(1:3000)[1:2000];\n\njulia> function iterated_neural_network(A, x, depth)\n           for _ in 1:depth\n               x .= max.(0, A * x)\n           end\n           argmax(x)\n       end\n\njulia> @time iterated_neural_network(view(A, inds, inds), x, 10)\n  0.324903 seconds (12 allocations: 157.562 KiB)\n1569\n\njulia> @time iterated_neural_network(A[inds, inds], x, 10)\n  0.054576 seconds (13 allocations: 30.671 MiB, 13.33% gc time)\n1569\n\nProvided there is enough memory, the cost of copying the view to an array is outweighed by the speed boost from doing the repeated matrix multiplications on a contiguous array."},{"title":"Multithreading and linear algebra","page":"Performance Tips","location":"manual/performance-tips.html#man-multithreading-linear-algebra","category":"section","text":"This section applies to multithreaded Julia code which, in each thread, performs linear algebra operations. Indeed, these linear algebra operations involve BLAS / LAPACK calls, which are themselves multithreaded. In this case, one must ensure that cores aren't oversubscribed due to the two different types of multithreading.\n\nJulia compiles and uses its own copy of OpenBLAS for linear algebra, whose number of threads is controlled by the environment variable OPENBLAS_NUM_THREADS. It can either be set as a command line option when launching Julia, or modified during the Julia session with BLAS.set_num_threads(N) (the submodule BLAS is exported by using LinearAlgebra). Its current value can be accessed with BLAS.get_num_threads().\n\nWhen the user does not specify anything, Julia tries to choose a reasonable value for the number of OpenBLAS threads (e.g. based on the platform, the Julia version, etc.). However, it is generally recommended to check and set the value manually. The OpenBLAS behavior is as follows:\n\nIf OPENBLAS_NUM_THREADS=1, OpenBLAS uses the calling Julia thread(s), i.e. it \"lives in\" the Julia thread that runs the computation.\nIf OPENBLAS_NUM_THREADS=N>1, OpenBLAS creates and manages its own pool of threads (N in total). There is just one OpenBLAS thread pool shared among all Julia threads.\n\nWhen you start Julia in multithreaded mode with JULIA_NUM_THREADS=X, it is generally recommended to set OPENBLAS_NUM_THREADS=1. Given the behavior described above, increasing the number of BLAS threads to N>1 can very easily lead to worse performance, in particular when N<<X. However this is just a rule of thumb, and the best way to set each number of threads is to experiment on your specific application."},{"title":"Alternative linear algebra backends","page":"Performance Tips","location":"manual/performance-tips.html#man-backends-linear-algebra","category":"section","text":"As an alternative to OpenBLAS, there exist several other backends that can help with linear algebra performance. Prominent examples include MKL.jl and AppleAccelerate.jl.\n\nThese are external packages, so we will not discuss them in detail here. Please refer to their respective documentations (especially because they have different behaviors than OpenBLAS with respect to multithreading)."},{"title":"Execution latency, package loading and package precompiling time","page":"Performance Tips","location":"manual/performance-tips.html#Execution-latency,-package-loading-and-package-precompiling-time","category":"section","text":""},{"title":"Reducing time to first plot etc.","page":"Performance Tips","location":"manual/performance-tips.html#Reducing-time-to-first-plot-etc.","category":"section","text":"The first time a julia method is called it (and any methods it calls, or ones that can be statically determined) will be compiled. The @time macro family illustrates this.\n\njulia> foo() = rand(2,2) * rand(2,2)\nfoo (generic function with 1 method)\n\njulia> @time @eval foo();\n  0.252395 seconds (1.12 M allocations: 56.178 MiB, 2.93% gc time, 98.12% compilation time)\n\njulia> @time @eval foo();\n  0.000156 seconds (63 allocations: 2.453 KiB)\n\nNote that @time @eval is better for measuring compilation time because without @eval, some compilation may already be done before timing starts.\n\nWhen developing a package, you may be able to improve the experience of your users with precompilation so that when they use the package, the code they use is already compiled. To precompile package code effectively, it's recommended to use PrecompileTools.jl to run a \"precompile workload\" during precompilation time that is representative of typical package usage, which will cache the native compiled code into the package pkgimage cache, greatly reducing \"time to first execution\" (often referred to as TTFX) for such usage.\n\nNote that PrecompileTools.jl workloads can be disabled and sometimes configured via Preferences if you do not want to spend the extra time precompiling, which may be the case during development of a package."},{"title":"Reducing package loading time","page":"Performance Tips","location":"manual/performance-tips.html#Reducing-package-loading-time","category":"section","text":"Keeping the time taken to load the package down is usually helpful. General good practice for package developers includes:\n\nReduce your dependencies to those you really need. Consider using package extensions to support interoperability with other packages without bloating your essential dependencies.\nAvoid use of __init__() functions unless there is no alternative, especially those which might trigger a lot of compilation, or just take a long time to execute.\nWhere possible, fix invalidations among your dependencies and from your package code.\n\nThe tool @time_imports can be useful in the REPL to review the above factors.\n\njulia> @time @time_imports using Plots\n      0.5 ms  Printf\n     16.4 ms  Dates\n      0.7 ms  Statistics\n               ┌ 23.8 ms SuiteSparse_jll.__init__() 86.11% compilation time (100% recompilation)\n     90.1 ms  SuiteSparse_jll 91.57% compilation time (82% recompilation)\n      0.9 ms  Serialization\n               ┌ 39.8 ms SparseArrays.CHOLMOD.__init__() 99.47% compilation time (100% recompilation)\n    166.9 ms  SparseArrays 23.74% compilation time (100% recompilation)\n      0.4 ms  Statistics → SparseArraysExt\n      0.5 ms  TOML\n      8.0 ms  Preferences\n      0.3 ms  PrecompileTools\n      0.2 ms  Reexport\n... many deps omitted for example ...\n      1.4 ms  Tar\n               ┌ 73.8 ms p7zip_jll.__init__() 99.93% compilation time (100% recompilation)\n     79.4 ms  p7zip_jll 92.91% compilation time (100% recompilation)\n               ┌ 27.7 ms GR.GRPreferences.__init__() 99.77% compilation time (100% recompilation)\n     43.0 ms  GR 64.26% compilation time (100% recompilation)\n               ┌ 2.1 ms Plots.__init__() 91.80% compilation time (100% recompilation)\n    300.9 ms  Plots 0.65% compilation time (100% recompilation)\n  1.795602 seconds (3.33 M allocations: 190.153 MiB, 7.91% gc time, 39.45% compilation time: 97% of which was recompilation)\n\n\nNotice that in this example there are multiple packages loaded, some with __init__() functions, some of which cause compilation of which some is recompilation. Recompilation is caused by earlier packages invalidating methods, then in these cases when the following packages run their __init__() function some hit recompilation before the code can be run.\n\nFurther, note the Statistics extension SparseArraysExt has been activated because SparseArrays is in the dependency tree. i.e. see 0.4 ms  Statistics → SparseArraysExt.\n\nThis report gives a good opportunity to review whether the cost of dependency load time is worth the functionality it brings. Also the Pkg utility why can be used to report why an indirect dependency exists.\n\n(CustomPackage) pkg> why FFMPEG_jll\n  Plots → FFMPEG → FFMPEG_jll\n  Plots → GR → GR_jll → FFMPEG_jll\n\nor to see the indirect dependencies that a package brings in, you can pkg> rm the package, see the deps that are removed from the manifest, then revert the change with pkg> undo.\n\nIf loading time is dominated by slow __init__() methods having compilation, one verbose way to identify what is being compiled is to use the julia args --trace-compile=stderr --trace-compile-timing which will report a precompile statement each time a method is compiled, along with how long compilation took. The InteractiveUtils macro @trace_compile provides a way to enable those args for a specific call. So a call for a complete report report would look like:\n\njulia> @time @time_imports @trace_compile using CustomPackage\n...\n\nNote the --startup-file=no which helps isolate the test from packages you may have in your startup.jl.\n\nMore analysis of the reasons for recompilation can be achieved with the SnoopCompile package."},{"title":"Tracing expression evaluation","page":"Performance Tips","location":"manual/performance-tips.html#Tracing-expression-evaluation","category":"section","text":"If you need to understand what code is being evaluated during test or script execution, you can use the --trace-eval command-line option or the Base.TRACE_EVAL global control to trace the outermost expressions being evaluated (top-level statements). Note this does not individually report the contents of function calls or code blocks:\n\n# Show only location information during evaluation\njulia --trace-eval=loc script.jl\n\n# Show full expressions being evaluated\njulia --trace-eval=full script.jl\n\nYou can also control this programmatically:\n\n# Enable full expression tracing\nBase.TRACE_EVAL = :full\n\n# Show only locations\nBase.TRACE_EVAL = :loc\n\n# Disable tracing\nBase.TRACE_EVAL = :no\n\n# Reset to use command-line setting\nBase.TRACE_EVAL = nothing"},{"title":"Reducing precompilation time","page":"Performance Tips","location":"manual/performance-tips.html#Reducing-precompilation-time","category":"section","text":"If package precompilation is taking a long time, one option is to set the following internal and then precompile.\n\njulia> Base.PRECOMPILE_TRACE_COMPILE[] = \"stderr\"\n\npkg> precompile\n\nThis has the effect of setting --trace-compile=stderr --trace-compile-timing in the precompilation processes themselves, so will show which methods are precompiled and how long they took to precompile.\n\nThere are also profiling options such as using the external profiler Tracy to profile the precompilation process."},{"title":"Miscellaneous","page":"Performance Tips","location":"manual/performance-tips.html#Miscellaneous","category":"section","text":""},{"title":"Tweaks","page":"Performance Tips","location":"manual/performance-tips.html#Tweaks","category":"section","text":"These are some minor points that might help in tight inner loops.\n\nAvoid unnecessary arrays. For example, instead of sum([x,y,z]) use x+y+z.\nUse abs2(z) instead of abs(z)^2 for complex z. In general, try to rewrite code to use abs2 instead of abs for complex arguments.\nUse div(x,y) for truncating division of integers instead of trunc(x/y), fld(x,y) instead of floor(x/y), and cld(x,y) instead of ceil(x/y)."},{"title":"Fix deprecation warnings","page":"Performance Tips","location":"manual/performance-tips.html#Fix-deprecation-warnings","category":"section","text":"A deprecated function internally performs a lookup in order to print a relevant warning only once. This extra lookup can cause a significant slowdown, so all uses of deprecated functions should be modified as suggested by the warnings."},{"title":"Performance Annotations","page":"Performance Tips","location":"manual/performance-tips.html#man-performance-annotations","category":"section","text":"Sometimes you can enable better optimization by promising certain program properties.\n\nUse @inbounds to eliminate array bounds checking within expressions. Be certain before doing this. If the indices are ever out of bounds, you may suffer crashes or silent corruption.\nUse @fastmath to allow floating point optimizations that are correct for real numbers, but lead to differences for IEEE numbers. Be careful when doing this, as this may change numerical results. This corresponds to the -ffast-math option of clang.\nWrite @simd in front of for loops to promise that the iterations are independent and may be reordered. Note that in many cases, Julia can automatically vectorize code without the @simd macro; it is only beneficial in cases where such a transformation would otherwise be illegal, including cases like allowing floating-point re-associativity and ignoring dependent memory accesses (@simd ivdep). Again, be very careful when asserting @simd as erroneously annotating a loop with dependent iterations may result in unexpected results. In particular, note that setindex! on some AbstractArray subtypes is inherently dependent upon iteration order. This feature is experimental and could change or disappear in future versions of Julia.\n\nThe common idiom of using 1:n to index into an AbstractArray is not safe if the Array uses unconventional indexing, and may cause a segmentation fault if bounds checking is turned off. Use LinearIndices(x) or eachindex(x) instead (see also Arrays with custom indices).\n\nnote: Note\nWhile @simd needs to be placed directly in front of an innermost for loop, both @inbounds and @fastmath can be applied to either single expressions or all the expressions that appear within nested blocks of code, e.g., using @inbounds begin or @inbounds for ....\n\nHere is an example with both @inbounds and @simd markup (we here use @noinline to prevent the optimizer from trying to be too clever and defeat our benchmark):\n\n@noinline function inner(x, y)\n    s = zero(eltype(x))\n    for i in eachindex(x, y)\n        @inbounds s += x[i]*y[i]\n    end\n    return s\nend\n\n@noinline function innersimd(x, y)\n    s = zero(eltype(x))\n    @simd for i in eachindex(x, y)\n        @inbounds s += x[i] * y[i]\n    end\n    return s\nend\n\nfunction timeit(n, reps)\n    x = rand(Float32, n)\n    y = rand(Float32, n)\n    s = zero(Float64)\n    time = @elapsed for j in 1:reps\n        s += inner(x, y)\n    end\n    println(\"GFlop/sec        = \", 2n*reps / time*1E-9)\n    time = @elapsed for j in 1:reps\n        s += innersimd(x, y)\n    end\n    println(\"GFlop/sec (SIMD) = \", 2n*reps / time*1E-9)\nend\n\ntimeit(1000, 1000)\n\nOn a computer with a 2.4GHz Intel Core i5 processor, this produces:\n\nGFlop/sec        = 1.9467069505224963\nGFlop/sec (SIMD) = 17.578554163920018\n\n(GFlop/sec measures the performance, and larger numbers are better.)\n\nHere is an example with all three kinds of markup. This program first calculates the finite difference of a one-dimensional array, and then evaluates the L2-norm of the result:\n\nfunction init!(u::Vector)\n    n = length(u)\n    dx = 1.0 / (n-1)\n    @fastmath @inbounds @simd for i in 1:n #by asserting that `u` is a `Vector` we can assume it has 1-based indexing\n        u[i] = sin(2pi*dx*i)\n    end\nend\n\nfunction deriv!(u::Vector, du)\n    n = length(u)\n    dx = 1.0 / (n-1)\n    @fastmath @inbounds du[1] = (u[2] - u[1]) / dx\n    @fastmath @inbounds @simd for i in 2:n-1\n        du[i] = (u[i+1] - u[i-1]) / (2*dx)\n    end\n    @fastmath @inbounds du[n] = (u[n] - u[n-1]) / dx\nend\n\nfunction mynorm(u::Vector)\n    n = length(u)\n    T = eltype(u)\n    s = zero(T)\n    @fastmath @inbounds @simd for i in 1:n\n        s += u[i]^2\n    end\n    @fastmath @inbounds return sqrt(s)\nend\n\nfunction main()\n    n = 2000\n    u = Vector{Float64}(undef, n)\n    init!(u)\n    du = similar(u)\n\n    deriv!(u, du)\n    nu = mynorm(du)\n\n    @time for i in 1:10^6\n        deriv!(u, du)\n        nu = mynorm(du)\n    end\n\n    println(nu)\nend\n\nmain()\n\nOn a computer with a 2.7 GHz Intel Core i7 processor, this produces:\n\n$ julia wave.jl;\n  1.207814709 seconds\n4.443986180758249\n\n$ julia --math-mode=ieee wave.jl;\n  4.487083643 seconds\n4.443986180758249\n\nHere, the option --math-mode=ieee disables the @fastmath macro, so that we can compare results.\n\nIn this case, the speedup due to @fastmath is a factor of about 3.7. This is unusually large – in general, the speedup will be smaller. (In this particular example, the working set of the benchmark is small enough to fit into the L1 cache of the processor, so that memory access latency does not play a role, and computing time is dominated by CPU usage. In many real world programs this is not the case.) Also, in this case this optimization does not change the result – in general, the result will be slightly different. In some cases, especially for numerically unstable algorithms, the result can be very different.\n\nThe annotation @fastmath re-arranges floating point expressions, e.g. changing the order of evaluation, or assuming that certain special cases (inf, nan) cannot occur. In this case (and on this particular computer), the main difference is that the expression 1 / (2*dx) in the function deriv is hoisted out of the loop (i.e. calculated outside the loop), as if one had written idx = 1 / (2*dx). In the loop, the expression ... / (2*dx) then becomes ... * idx, which is much faster to evaluate. Of course, both the actual optimization that is applied by the compiler as well as the resulting speedup depend very much on the hardware. You can examine the change in generated code by using Julia's code_native function.\n\nNote that @fastmath also assumes that NaNs will not occur during the computation, which can lead to surprising behavior:\n\njulia> f(x) = isnan(x);\n\njulia> f(NaN)\ntrue\n\njulia> f_fast(x) = @fastmath isnan(x);\n\njulia> f_fast(NaN)\nfalse"},{"title":"Treat Subnormal Numbers as Zeros","page":"Performance Tips","location":"manual/performance-tips.html#Treat-Subnormal-Numbers-as-Zeros","category":"section","text":"Subnormal numbers, formerly called denormal numbers, are useful in many contexts, but incur a performance penalty on some hardware. A call set_zero_subnormals(true) grants permission for floating-point operations to treat subnormal inputs or outputs as zeros, which may improve performance on some hardware. A call set_zero_subnormals(false) enforces strict IEEE behavior for subnormal numbers.\n\nBelow is an example where subnormals noticeably impact performance on some hardware:\n\nfunction timestep(b::Vector{T}, a::Vector{T}, Δt::T) where T\n    @assert length(a)==length(b)\n    n = length(b)\n    b[1] = 1                            # Boundary condition\n    for i=2:n-1\n        b[i] = a[i] + (a[i-1] - T(2)*a[i] + a[i+1]) * Δt\n    end\n    b[n] = 0                            # Boundary condition\nend\n\nfunction heatflow(a::Vector{T}, nstep::Integer) where T\n    b = similar(a)\n    for t=1:div(nstep,2)                # Assume nstep is even\n        timestep(b,a,T(0.1))\n        timestep(a,b,T(0.1))\n    end\nend\n\nheatflow(zeros(Float32,10),2)           # Force compilation\nfor trial=1:6\n    a = zeros(Float32,1000)\n    set_zero_subnormals(iseven(trial))  # Odd trials use strict IEEE arithmetic\n    @time heatflow(a,1000)\nend\n\nThis gives an output similar to\n\n  0.002202 seconds (1 allocation: 4.063 KiB)\n  0.001502 seconds (1 allocation: 4.063 KiB)\n  0.002139 seconds (1 allocation: 4.063 KiB)\n  0.001454 seconds (1 allocation: 4.063 KiB)\n  0.002115 seconds (1 allocation: 4.063 KiB)\n  0.001455 seconds (1 allocation: 4.063 KiB)\n\nNote how each even iteration is significantly faster.\n\nThis example generates many subnormal numbers because the values in a become an exponentially decreasing curve, which slowly flattens out over time.\n\nTreating subnormals as zeros should be used with caution, because doing so breaks some identities, such as x-y == 0 implies x == y:\n\njulia> x = 3f-38; y = 2f-38;\n\njulia> set_zero_subnormals(true); (x - y, x == y)\n(0.0f0, false)\n\njulia> set_zero_subnormals(false); (x - y, x == y)\n(1.0000001f-38, false)\n\nIn some applications, an alternative to zeroing subnormal numbers is to inject a tiny bit of noise.  For example, instead of initializing a with zeros, initialize it with:\n\na = rand(Float32,1000) * 1.f-9"},{"title":"Avoid string interpolation for I/O","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-string-interpolation-for-I/O","category":"section","text":"When writing data to a file (or other I/O device), forming extra intermediate strings is a source of overhead. Instead of:\n\nprintln(file, \"$a $b\")\n\nuse:\n\nprintln(file, a, \" \", b)\n\nThe first version of the code forms a string, then writes it to the file, while the second version writes values directly to the file. Also notice that in some cases string interpolation can be harder to read. Consider:\n\nprintln(file, \"$(f(a))$(f(b))\")\n\nversus:\n\nprintln(file, f(a), f(b))"},{"title":"Avoid eager string materialization","page":"Performance Tips","location":"manual/performance-tips.html#Avoid-eager-string-materialization","category":"section","text":"In settings where a string representation of an object is only needed conditionally (e.g. in error paths of functions or conditional warnings such as deprecations), it is advisable to avoid the overhead of eagerly materializing the string. Since Julia 1.8, this can be achieved via LazyString and the corresponding string macro @lazy_str.\n\nFor example, instead of:\n\nBase.depwarn(\"`foo` is deprecated for type $(typeof(x))\", :bar)\n\nuse:\n\nBase.depwarn(lazy\"`foo` is deprecated for type $(typeof(x))\", :bar)\n\nor the equivalent macro-free version:\n\nBase.depwarn(LazyString(\"`foo` is deprecated for type \", typeof(x)), :bar)\n\nThrough this approach, the interpolated string will only be constructed when it is actually displayed."},{"title":"Optimize network I/O during parallel execution","page":"Performance Tips","location":"manual/performance-tips.html#Optimize-network-I/O-during-parallel-execution","category":"section","text":"When executing a remote function in parallel:\n\nusing Distributed\n\nresponses = Vector{Any}(undef, nworkers())\n@sync begin\n    for (idx, pid) in enumerate(workers())\n        Threads.@spawn responses[idx] = remotecall_fetch(foo, pid, args...)\n    end\nend\n\nis faster than:\n\nusing Distributed\n\nrefs = Vector{Any}(undef, nworkers())\nfor (idx, pid) in enumerate(workers())\n    refs[idx] = @spawnat pid foo(args...)\nend\nresponses = [fetch(r) for r in refs]\n\nThe former results in a single network round-trip to every worker, while the latter results in two network calls - first by the @spawnat and the second due to the fetch (or even a wait). The fetch/wait is also being executed serially resulting in an overall poorer performance."},{"title":"Use MutableArithmetics for more control over allocation for mutable arithmetic types","page":"Performance Tips","location":"manual/performance-tips.html#man-perftips-mutablearithmetics","category":"section","text":"Some Number subtypes, such as BigInt or BigFloat, may be implemented as mutable struct types, or they may have mutable components. The arithmetic interfaces in Julia Base usually opt for convenience over efficiency in such cases, so using them in a naive manner may result in suboptimal performance. The abstractions of the MutableArithmetics package, on the other hand, make it possible to exploit the mutability of such types for writing fast code that allocates only as much as necessary. MutableArithmetics also makes it possible to copy values of mutable arithmetic types explicitly when necessary. MutableArithmetics is a user package and is not affiliated with the Julia project."},{"title":"C Interface","page":"C Interface","location":"base/c.html#C-Interface","category":"section","text":""},{"title":"LLVM Interface","page":"C Interface","location":"base/c.html#LLVM-Interface","category":"section","text":""},{"title":"Base.@ccall","page":"C Interface","location":"base/c.html#Base.@ccall","category":"macro","text":"@ccall library.function_name(argvalue1::argtype1, ...)::returntype\n@ccall function_name(argvalue1::argtype1, ...)::returntype\n@ccall $function_pointer(argvalue1::argtype1, ...)::returntype\n\nCall a function in a C-exported shared library, specified by library.function_name, where library is a string constant or literal. The library may be omitted, in which case the function_name is resolved in the current process. Alternatively, @ccall may also be used to call a function pointer $function_pointer, such as one returned by dlsym.\n\nEach argvalue to @ccall is converted to the corresponding argtype, by automatic insertion of calls to unsafe_convert(argtype, cconvert(argtype, argvalue)). (See also the documentation for unsafe_convert and cconvert for further details.) In most cases, this simply results in a call to convert(argtype, argvalue).\n\nExamples\n\n@ccall strlen(s::Cstring)::Csize_t\n\nThis calls the C standard library function:\n\nsize_t strlen(char *)\n\nwith a Julia variable named s. See also ccall.\n\nVarargs are supported with the following convention:\n\n@ccall printf(\"%s = %d\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint\n\nThe semicolon is used to separate required arguments (of which there must be at least one) from variadic arguments.\n\nExample using an external library:\n\n# C signature of g_uri_escape_string:\n# char *g_uri_escape_string(const char *unescaped, const char *reserved_chars_allowed, gboolean allow_utf8);\n\nconst glib = \"libglib-2.0\"\n@ccall glib.g_uri_escape_string(my_uri::Cstring, \":/\"::Cstring, true::Cint)::Cstring\n\nThe string literal could also be used directly before the function name, if desired \"libglib-2.0\".g_uri_escape_string(...\n\nIt's possible to declare the ccall as gc_safe by using the gc_safe = true option:\n\n@ccall gc_safe=true strlen(s::Cstring)::Csize_t\n\nThis allows the garbage collector to run concurrently with the ccall, which can be useful whenever the ccall may block outside of julia.\n\nwarning: Warning\nThis option should be used with caution, as it can lead to undefined behavior if the ccall calls back into the julia runtime. (@cfunction/@ccallables are safe however)\n\ncompat: Julia 1.12\nThe gc_safe argument requires Julia 1.12 or higher.\n\n\n\n\n\n"},{"title":"ccall","page":"C Interface","location":"base/c.html#ccall","category":"keyword","text":"ccall((function_name, library), returntype, (argtype1, ...), argvalue1, ...)\nccall(function_name, returntype, (argtype1, ...), argvalue1, ...)\nccall(function_pointer, returntype, (argtype1, ...), argvalue1, ...)\n\nCall a function in a C-exported shared library, specified by the tuple (function_name, library), where each component is either a string or symbol. Instead of specifying a library, one can also use a function_name symbol or string, which is resolved in the current process. Alternatively, ccall may also be used to call a function pointer function_pointer, such as one returned by dlsym.\n\nNote that the argument type tuple must be a literal tuple, and not a tuple-valued variable or expression.\n\nEach argvalue to the ccall will be converted to the corresponding argtype, by automatic insertion of calls to unsafe_convert(argtype, cconvert(argtype, argvalue)). (See also the documentation for unsafe_convert and cconvert for further details.) In most cases, this simply results in a call to convert(argtype, argvalue).\n\n\n\n\n\n"},{"title":"Core.Intrinsics.cglobal","page":"C Interface","location":"base/c.html#Core.Intrinsics.cglobal","category":"function","text":"cglobal((symbol, library) [, type=Cvoid])\n\nObtain a pointer to a global variable in a C-exported shared library, specified exactly as in ccall. Returns a Ptr{Type}, defaulting to Ptr{Cvoid} if no Type argument is supplied. The values can be read or written by unsafe_load or unsafe_store!, respectively.\n\n\n\n\n\n"},{"title":"Base.@cfunction","page":"C Interface","location":"base/c.html#Base.@cfunction","category":"macro","text":"@cfunction(callable, ReturnType, (ArgumentTypes...,)) -> Ptr{Cvoid}\n@cfunction($callable, ReturnType, (ArgumentTypes...,)) -> CFunction\n\nGenerate a C-callable function pointer from the Julia function callable for the given type signature. To pass the return value to a ccall, use the argument type Ptr{Cvoid} in the signature.\n\nNote that the argument type tuple must be a literal tuple, and not a tuple-valued variable or expression (although it can include a splat expression). And that these arguments will be evaluated in global scope during compile-time (not deferred until runtime). Adding a '$' in front of the function argument changes this to instead create a runtime closure over the local variable callable (this is not supported on all architectures).\n\nSee manual section on ccall and cfunction usage.\n\nExamples\n\njulia> function foo(x::Int, y::Int)\n           return x + y\n       end\n\njulia> @cfunction(foo, Int, (Int, Int))\nPtr{Cvoid} @0x000000001b82fcd0\n\n\n\n\n\n"},{"title":"Base.CFunction","page":"C Interface","location":"base/c.html#Base.CFunction","category":"type","text":"CFunction struct\n\nGarbage-collection handle for the return value from @cfunction when the first argument is annotated with '$'. Like all cfunction handles, it should be passed to ccall as a Ptr{Cvoid}, and will be converted automatically at the call site to the appropriate type.\n\nSee @cfunction.\n\n\n\n\n\n"},{"title":"Base.unsafe_convert","page":"C Interface","location":"base/c.html#Base.unsafe_convert","category":"function","text":"unsafe_convert(T, x)\n\nConvert x to a C argument of type T where the input x must be the return value of cconvert(T, ...).\n\nIn cases where convert would need to take a Julia object and turn it into a Ptr, this function should be used to define and perform that conversion.\n\nBe careful to ensure that a Julia reference to x exists as long as the result of this function will be used. Accordingly, the argument x to this function should never be an expression, only a variable name or field reference. For example, x=a.b.c is acceptable, but x=[a,b,c] is not.\n\nThe unsafe prefix on this function indicates that using the result of this function after the x argument to this function is no longer accessible to the program may cause undefined behavior, including program corruption or segfaults, at any later time.\n\nSee also cconvert\n\n\n\n\n\n"},{"title":"Base.cconvert","page":"C Interface","location":"base/c.html#Base.cconvert","category":"function","text":"cconvert(T,x)\n\nConvert x to a value to be passed to C code as type T, typically by calling convert(T, x).\n\nIn cases where x cannot be safely converted to T, unlike convert, cconvert may return an object of a type different from T, which however is suitable for unsafe_convert to handle. The result of this function should be kept valid (for the GC) until the result of unsafe_convert is not needed anymore. This can be used to allocate memory that will be accessed by the ccall. If multiple objects need to be allocated, a tuple of the objects can be used as return value.\n\nNeither convert nor cconvert should take a Julia object and turn it into a Ptr.\n\n\n\n\n\n"},{"title":"Base.unsafe_load","page":"C Interface","location":"base/c.html#Base.unsafe_load","category":"function","text":"unsafe_load(p::Ptr{T}, i::Integer=1)\nunsafe_load(p::Ptr{T}, order::Symbol)\nunsafe_load(p::Ptr{T}, i::Integer, order::Symbol)\n\nLoad a value of type T from the address of the ith element (1-indexed) starting at p. This is equivalent to the C expression p[i-1]. Optionally, an atomic memory ordering can be provided.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program or return garbage answers. Unlike C, dereferencing memory region allocated as different type may be valid provided that the types are compatible.\n\ncompat: Julia 1.10\nThe order argument is available as of Julia 1.10.\n\nSee also: atomic\n\n\n\n\n\n"},{"title":"Base.unsafe_store!","page":"C Interface","location":"base/c.html#Base.unsafe_store!","category":"function","text":"unsafe_store!(p::Ptr{T}, x, i::Integer=1)\nunsafe_store!(p::Ptr{T}, x, order::Symbol)\nunsafe_store!(p::Ptr{T}, x, i::Integer, order::Symbol)\n\nStore a value of type T to the address of the ith element (1-indexed) starting at p. This is equivalent to the C expression p[i-1] = x. Optionally, an atomic memory ordering can be provided.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program. Unlike C, storing memory region allocated as different type may be valid provided that the types are compatible.\n\ncompat: Julia 1.10\nThe order argument is available as of Julia 1.10.\n\nSee also: atomic\n\n\n\n\n\n"},{"title":"Base.unsafe_modify!","page":"C Interface","location":"base/c.html#Base.unsafe_modify!","category":"function","text":"unsafe_modify!(p::Ptr{T}, op, x, [order::Symbol])::Pair\n\nThese atomically perform the operations to get and set a memory address after applying the function op. If supported by the hardware (for example, atomic increment), this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p)\nz = op(y, x)\nunsafe_store!(p, z)\nreturn y => z\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: modifyproperty!, atomic\n\n\n\n\n\n"},{"title":"Base.unsafe_replace!","page":"C Interface","location":"base/c.html#Base.unsafe_replace!","category":"function","text":"unsafe_replace!(p::Ptr{T}, expected, desired,\n               [success_order::Symbol[, fail_order::Symbol=success_order]]) -> (; old, success::Bool)\n\nThese atomically perform the operations to get and conditionally set a memory address to a given value. If supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p, fail_order)\nok = y === expected\nif ok\n    unsafe_store!(p, desired, success_order)\nend\nreturn (; old = y, success = ok)\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: replaceproperty!, atomic\n\n\n\n\n\n"},{"title":"Base.unsafe_swap!","page":"C Interface","location":"base/c.html#Base.unsafe_swap!","category":"function","text":"unsafe_swap!(p::Ptr{T}, x, [order::Symbol])\n\nThese atomically perform the operations to simultaneously get and set a memory address. If supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p)\nunsafe_store!(p, x)\nreturn y\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: swapproperty!, atomic\n\n\n\n\n\n"},{"title":"Base.unsafe_copyto!","page":"C Interface","location":"base/c.html#Base.unsafe_copyto!-Union{Tuple{T}, Tuple{Ptr{T}, Ptr{T}, Any}} where T","category":"method","text":"unsafe_copyto!(dest::Ptr{T}, src::Ptr{T}, N)\n\nCopy N elements from a source pointer to a destination, with no checking. The size of an element is determined by the type of the pointers.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointers dest and src to ensure that they are valid. Incorrect usage may corrupt or segfault your program, in the same manner as C.\n\n\n\n\n\n"},{"title":"Base.unsafe_copyto!","page":"C Interface","location":"base/c.html#Base.unsafe_copyto!-Tuple{Array, Any, Array, Any, Any}","category":"method","text":"unsafe_copyto!(dest::Array, doffs, src::Array, soffs, n)\n\nCopy n elements from a source array to a destination, starting at the linear index soffs in the source and doffs in the destination (1-indexed).\n\nThe unsafe prefix on this function indicates that no validation is performed to ensure that n is inbounds on either array. Incorrect usage may corrupt or segfault your program, in the same manner as C.\n\n\n\n\n\n"},{"title":"Base.copyto!","page":"C Interface","location":"base/c.html#Base.copyto!","category":"function","text":"copyto!(dest, Rdest::CartesianIndices, src, Rsrc::CartesianIndices) -> dest\n\nCopy the block of src in the range of Rsrc to the block of dest in the range of Rdest. The sizes of the two regions must match.\n\nExamples\n\njulia> A = zeros(5, 5);\n\njulia> B = [1 2; 3 4];\n\njulia> Ainds = CartesianIndices((2:3, 2:3));\n\njulia> Binds = CartesianIndices(B);\n\njulia> copyto!(A, Ainds, B, Binds)\n5×5 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0\n 0.0  1.0  2.0  0.0  0.0\n 0.0  3.0  4.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n\n\n\n\n\ncopyto!(dest::AbstractArray, src) -> dest\n\nCopy all elements from collection src to array dest, whose length must be greater than or equal to the length n of src. The first n elements of dest are overwritten, the other elements are left untouched.\n\nSee also copy!, copy.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> x = [1., 0., 3., 0., 5.];\n\njulia> y = zeros(7);\n\njulia> copyto!(y, x);\n\njulia> y\n7-element Vector{Float64}:\n 1.0\n 0.0\n 3.0\n 0.0\n 5.0\n 0.0\n 0.0\n\n\n\n\n\ncopyto!(dest, doffs, src, soffs, n)\n\nCopy n elements from collection src starting at the linear index soffs, to array dest starting at the index doffs. Return dest.\n\n\n\n\n\ncopyto!(B::AbstractMatrix, ir_dest::AbstractUnitRange, jr_dest::AbstractUnitRange,\n        tM::AbstractChar,\n        M::AbstractVecOrMat, ir_src::AbstractUnitRange, jr_src::AbstractUnitRange) -> B\n\nEfficiently copy elements of matrix M to B conditioned on the character parameter tM as follows:\n\ntM Destination Source\n'N' B[ir_dest, jr_dest] M[ir_src, jr_src]\n'T' B[ir_dest, jr_dest] transpose(M)[ir_src, jr_src]\n'C' B[ir_dest, jr_dest] adjoint(M)[ir_src, jr_src]\n\nThe elements B[ir_dest, jr_dest] are overwritten. Furthermore, the index range parameters must satisfy length(ir_dest) == length(ir_src) and length(jr_dest) == length(jr_src).\n\nSee also copy_transpose! and copy_adjoint!.\n\n\n\n\n\ncopyto!(dest::AbstractMatrix, src::UniformScaling)\n\nCopies a UniformScaling onto a matrix.\n\ncompat: Julia 1.1\nIn Julia 1.0 this method only supported a square destination matrix. Julia 1.1. added support for a rectangular matrix.\n\n\n\n\n\n"},{"title":"Base.pointer","page":"C Interface","location":"base/c.html#Base.pointer","category":"function","text":"pointer(array [, index])\n\nGet the native address of an array or string, optionally at a given location index.\n\nThis function is \"unsafe\". Be careful to ensure that a Julia reference to array exists as long as this pointer will be used. The GC.@preserve macro should be used to protect the array argument from garbage collection within a given block of code.\n\nCalling Ref(array[, index]) is generally preferable to this function as it guarantees validity.\n\n\n\n\n\n"},{"title":"Base.unsafe_wrap","page":"C Interface","location":"base/c.html#Base.unsafe_wrap-Union{Tuple{N}, Tuple{T}, Tuple{Union{Type{Array}, Type{Array{T}}, Type{Array{T, N}}}, Ptr{T}, NTuple{N, Int64}}} where {T, N}","category":"method","text":"unsafe_wrap(Array, pointer::Ptr{T}, dims; own = false)\n\nWrap a Julia Array object around the data at the address given by pointer, without making a copy.  The pointer element type T determines the array element type. dims is either an integer (for a 1d array) or a tuple of the array dimensions. own optionally specifies whether Julia should take ownership of the memory, calling free on the pointer when the array is no longer referenced.\n\nThis function is labeled \"unsafe\" because it will crash if pointer is not a valid memory address to data of the requested length. Unlike unsafe_load and unsafe_store!, the programmer is responsible also for ensuring that the underlying data is not accessed through two arrays of different element type, similar to the strict aliasing rule in C.\n\n\n\n\n\n"},{"title":"Base.pointer_from_objref","page":"C Interface","location":"base/c.html#Base.pointer_from_objref","category":"function","text":"pointer_from_objref(x)\n\nGet the memory address of a Julia object as a Ptr. The existence of the resulting Ptr will not protect the object from garbage collection, so you must ensure that the object remains referenced for the whole time that the Ptr will be used.\n\nThis function may not be called on immutable objects, since they do not have stable memory addresses.\n\nSee also unsafe_pointer_to_objref.\n\n\n\n\n\n"},{"title":"Base.unsafe_pointer_to_objref","page":"C Interface","location":"base/c.html#Base.unsafe_pointer_to_objref","category":"function","text":"unsafe_pointer_to_objref(p::Ptr)\n\nConvert a Ptr to an object reference. Assumes the pointer refers to a valid heap-allocated Julia object. If this is not the case, undefined behavior results, hence this function is considered \"unsafe\" and should be used with care.\n\nSee also pointer_from_objref.\n\n\n\n\n\n"},{"title":"Base.disable_sigint","page":"C Interface","location":"base/c.html#Base.disable_sigint","category":"function","text":"disable_sigint(f::Function)\n\nDisable Ctrl-C handler during execution of a function on the current task, for calling external code that may call julia code that is not interrupt safe. Intended to be called using do block syntax as follows:\n\ndisable_sigint() do\n    # interrupt-unsafe code\n    ...\nend\n\nThis is not needed on worker threads (Threads.threadid() != 1) since the InterruptException will only be delivered to the master thread. External functions that do not call julia code or julia runtime automatically disable sigint during their execution.\n\n\n\n\n\n"},{"title":"Base.reenable_sigint","page":"C Interface","location":"base/c.html#Base.reenable_sigint","category":"function","text":"reenable_sigint(f::Function)\n\nRe-enable Ctrl-C handler during execution of a function. Temporarily reverses the effect of disable_sigint.\n\n\n\n\n\n"},{"title":"Base.exit_on_sigint","page":"C Interface","location":"base/c.html#Base.exit_on_sigint","category":"function","text":"exit_on_sigint(on::Bool)\n\nSet exit_on_sigint flag of the julia runtime.  If false, Ctrl-C (SIGINT) is capturable as InterruptException in try block. This is the default behavior in REPL, any code run via -e and -E and in Julia script run with -i option.\n\nIf true, InterruptException is not thrown by Ctrl-C.  Running code upon such event requires atexit.  This is the default behavior in Julia script run without -i option.\n\ncompat: Julia 1.5\nFunction exit_on_sigint requires at least Julia 1.5.\n\n\n\n\n\n"},{"title":"Base.systemerror","page":"C Interface","location":"base/c.html#Base.systemerror","category":"function","text":"systemerror(sysfunc[, errno::Cint=Libc.errno()])\nsystemerror(sysfunc, iftrue::Bool)\n\nRaises a SystemError for errno with the descriptive string sysfunc if iftrue is true\n\n\n\n\n\n"},{"title":"Base.windowserror","page":"C Interface","location":"base/c.html#Base.windowserror","category":"function","text":"windowserror(sysfunc[, code::UInt32=Libc.GetLastError()])\nwindowserror(sysfunc, iftrue::Bool)\n\nLike systemerror, but for Windows API functions that use GetLastError to return an error code instead of setting errno.\n\n\n\n\n\n"},{"title":"Core.Ptr","page":"C Interface","location":"base/c.html#Core.Ptr","category":"type","text":"Ptr{T}\n\nA memory address referring to data of type T.  However, there is no guarantee that the memory is actually valid, or that it actually represents data of the specified type.\n\n\n\n\n\n"},{"title":"Core.Ref","page":"C Interface","location":"base/c.html#Core.Ref","category":"type","text":"Ref{T}\n\nAn object that safely references data of type T. This type is guaranteed to point to valid, Julia-allocated memory of the correct type. The underlying data is protected from freeing by the garbage collector as long as the Ref itself is referenced.\n\nIn Julia, Ref objects are dereferenced (loaded or stored) with [].\n\nCreation of a Ref to a value x of type T is usually written Ref(x). Additionally, for creating interior pointers to containers (such as Array or Ptr), it can be written Ref(a, i) for creating a reference to the i-th element of a.\n\nRef{T}() creates a reference to a value of type T without initialization. For a bitstype T, the value will be whatever currently resides in the memory allocated. For a non-bitstype T, the reference will be undefined and attempting to dereference it will result in an error, \"UndefRefError: access to undefined reference\".\n\nTo check if a Ref is an undefined reference, use isassigned(ref::RefValue). For example, isassigned(Ref{T}()) is false if T is not a bitstype. If T is a bitstype, isassigned(Ref{T}()) will always be true.\n\nWhen passed as a ccall argument (either as a Ptr or Ref type), a Ref object will be converted to a native pointer to the data it references. For most T, or when converted to a Ptr{Cvoid}, this is a pointer to the object data. When T is an isbits type, this value may be safely mutated, otherwise mutation is strictly undefined behavior.\n\nAs a special case, setting T = Any will instead cause the creation of a pointer to the reference itself when converted to a Ptr{Any} (a jl_value_t const* const* if T is immutable, else a jl_value_t *const *). When converted to a Ptr{Cvoid}, it will still return a pointer to the data region as for any other T.\n\nA C_NULL instance of Ptr can be passed to a ccall Ref argument to initialize it.\n\nUse in broadcasting\n\nRef is sometimes used in broadcasting in order to treat the referenced values as a scalar.\n\nExamples\n\njulia> r = Ref(5) # Create a Ref with an initial value\nBase.RefValue{Int64}(5)\n\njulia> r[] # Getting a value from a Ref\n5\n\njulia> r[] = 7 # Storing a new value in a Ref\n7\n\njulia> r # The Ref now contains 7\nBase.RefValue{Int64}(7)\n\njulia> isa.(Ref([1,2,3]), [Array, Dict, Int]) # Treat reference values as scalar during broadcasting\n3-element BitVector:\n 1\n 0\n 0\n\njulia> Ref{Function}()  # Undefined reference to a non-bitstype, Function\nBase.RefValue{Function}(#undef)\n\njulia> try\n           Ref{Function}()[] # Dereferencing an undefined reference will result in an error\n       catch e\n           println(e)\n       end\nUndefRefError()\n\njulia> Ref{Int64}()[]; # A reference to a bitstype refers to an undetermined value if not given\n\njulia> isassigned(Ref{Int64}()) # A reference to a bitstype is always assigned\ntrue\n\n\n\n\n\n"},{"title":"Base.isassigned","page":"C Interface","location":"base/c.html#Base.isassigned-Tuple{Base.RefValue}","category":"method","text":"isassigned(ref::RefValue)::Bool\n\nTest whether the given Ref is associated with a value. This is always true for a Ref of a bitstype object. Return false if the reference is undefined.\n\nExamples\n\njulia> ref = Ref{Function}()\nBase.RefValue{Function}(#undef)\n\njulia> isassigned(ref)\nfalse\n\njulia> ref[] = (foobar(x) = x)\nfoobar (generic function with 1 method)\n\njulia> isassigned(ref)\ntrue\n\njulia> isassigned(Ref{Int}())\ntrue\n\n\n\n\n\n"},{"title":"Base.Cchar","page":"C Interface","location":"base/c.html#Base.Cchar","category":"type","text":"Cchar\n\nEquivalent to the native char c-type.\n\n\n\n\n\n"},{"title":"Base.Cuchar","page":"C Interface","location":"base/c.html#Base.Cuchar","category":"type","text":"Cuchar\n\nEquivalent to the native unsigned char c-type (UInt8).\n\n\n\n\n\n"},{"title":"Base.Cshort","page":"C Interface","location":"base/c.html#Base.Cshort","category":"type","text":"Cshort\n\nEquivalent to the native signed short c-type (Int16).\n\n\n\n\n\n"},{"title":"Base.Cstring","page":"C Interface","location":"base/c.html#Base.Cstring","category":"type","text":"Cstring\n\nA C-style string composed of the native character type Cchars. Cstrings are NUL-terminated. For C-style strings composed of the native wide character type, see Cwstring. For more information about string interoperability with C, see the manual.\n\n\n\n\n\n"},{"title":"Base.Cushort","page":"C Interface","location":"base/c.html#Base.Cushort","category":"type","text":"Cushort\n\nEquivalent to the native unsigned short c-type (UInt16).\n\n\n\n\n\n"},{"title":"Base.Cint","page":"C Interface","location":"base/c.html#Base.Cint","category":"type","text":"Cint\n\nEquivalent to the native signed int c-type (Int32).\n\n\n\n\n\n"},{"title":"Base.Cuint","page":"C Interface","location":"base/c.html#Base.Cuint","category":"type","text":"Cuint\n\nEquivalent to the native unsigned int c-type (UInt32).\n\n\n\n\n\n"},{"title":"Base.Clong","page":"C Interface","location":"base/c.html#Base.Clong","category":"type","text":"Clong\n\nEquivalent to the native signed long c-type.\n\n\n\n\n\n"},{"title":"Base.Culong","page":"C Interface","location":"base/c.html#Base.Culong","category":"type","text":"Culong\n\nEquivalent to the native unsigned long c-type.\n\n\n\n\n\n"},{"title":"Base.Clonglong","page":"C Interface","location":"base/c.html#Base.Clonglong","category":"type","text":"Clonglong\n\nEquivalent to the native signed long long c-type (Int64).\n\n\n\n\n\n"},{"title":"Base.Culonglong","page":"C Interface","location":"base/c.html#Base.Culonglong","category":"type","text":"Culonglong\n\nEquivalent to the native unsigned long long c-type (UInt64).\n\n\n\n\n\n"},{"title":"Base.Cintmax_t","page":"C Interface","location":"base/c.html#Base.Cintmax_t","category":"type","text":"Cintmax_t\n\nEquivalent to the native intmax_t c-type (Int64).\n\n\n\n\n\n"},{"title":"Base.Cuintmax_t","page":"C Interface","location":"base/c.html#Base.Cuintmax_t","category":"type","text":"Cuintmax_t\n\nEquivalent to the native uintmax_t c-type (UInt64).\n\n\n\n\n\n"},{"title":"Base.Csize_t","page":"C Interface","location":"base/c.html#Base.Csize_t","category":"type","text":"Csize_t\n\nEquivalent to the native size_t c-type (UInt).\n\n\n\n\n\n"},{"title":"Base.Cssize_t","page":"C Interface","location":"base/c.html#Base.Cssize_t","category":"type","text":"Cssize_t\n\nEquivalent to the native ssize_t c-type.\n\n\n\n\n\n"},{"title":"Base.Cptrdiff_t","page":"C Interface","location":"base/c.html#Base.Cptrdiff_t","category":"type","text":"Cptrdiff_t\n\nEquivalent to the native ptrdiff_t c-type (Int).\n\n\n\n\n\n"},{"title":"Base.Cwchar_t","page":"C Interface","location":"base/c.html#Base.Cwchar_t","category":"type","text":"Cwchar_t\n\nEquivalent to the native wchar_t c-type (Int32).\n\n\n\n\n\n"},{"title":"Base.Cwstring","page":"C Interface","location":"base/c.html#Base.Cwstring","category":"type","text":"Cwstring\n\nA C-style string composed of the native wide character type Cwchar_ts. Cwstrings are NUL-terminated. For C-style strings composed of the native character type, see Cstring. For more information about string interoperability with C, see the manual.\n\n\n\n\n\n"},{"title":"Base.Cfloat","page":"C Interface","location":"base/c.html#Base.Cfloat","category":"type","text":"Cfloat\n\nEquivalent to the native float c-type (Float32).\n\n\n\n\n\n"},{"title":"Base.Cdouble","page":"C Interface","location":"base/c.html#Base.Cdouble","category":"type","text":"Cdouble\n\nEquivalent to the native double c-type (Float64).\n\n\n\n\n\n"},{"title":"Core.Intrinsics.llvmcall","page":"C Interface","location":"base/c.html#Core.Intrinsics.llvmcall","category":"function","text":"llvmcall(fun_ir::String, returntype, Tuple{argtype1, ...}, argvalue1, ...)\nllvmcall((mod_ir::String, entry_fn::String), returntype, Tuple{argtype1, ...}, argvalue1, ...)\nllvmcall((mod_bc::Vector{UInt8}, entry_fn::String), returntype, Tuple{argtype1, ...}, argvalue1, ...)\n\nCall the LLVM code provided in the first argument. There are several ways to specify this first argument:\n\nas a literal string, representing function-level IR (similar to an LLVM define block), with arguments are available as consecutive unnamed SSA variables (%0, %1, etc.);\nas a 2-element tuple, containing a string of module IR and a string representing the name of the entry-point function to call;\nas a 2-element tuple, but with the module provided as an Vector{UInt8} with bitcode.\n\nNote that contrary to ccall, the argument types must be specified as a tuple type, and not a tuple of types. All types, as well as the LLVM code, should be specified as literals, and not as variables or expressions (it may be necessary to use @eval to generate these literals).\n\nSee test/llvmcall.jl for usage examples.\n\n\n\n\n\n"},{"title":"Proper maintenance and care of multi-threading locks","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Proper-maintenance-and-care-of-multi-threading-locks","category":"section","text":"The following strategies are used to ensure that the code is dead-lock free (generally by addressing the 4th Coffman condition: circular wait).\n\nstructure code such that only one lock will need to be acquired at a time\nalways acquire shared locks in the same order, as given by the table below\navoid constructs that expect to need unrestricted recursion"},{"title":"Types of locks","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Types-of-locks","category":"section","text":"uv_mutex_t (or std::mutex) is a wrapper around platform-specific locks (pthread_mutex_t on Unix, CRITICAL_SECTION on Windows).  It may cause the current OS thread to block, is not reentrant, and is not a safepoint.\n\njl_mutex_t is a reentrant spinlock.  jl_mutex_ts acquired in a try block will be unlocked when we leave the block, either by reaching the end or catching an exception.  JL_LOCK/JL_UNLOCK are safepoints, while JL_LOCK_NOGC/JL_UNLOCK_NOGC are not.  jl_mutex_t must not be held across task switches."},{"title":"Lock hierarchy","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Lock-hierarchy","category":"section","text":"Below are all of the locks that exist in the system and the mechanisms for using them that avoid the potential for deadlocks (no Ostrich algorithm allowed here). Except in the special cases where a rule for avoiding deadlock is given, no lock of a lower level may acquire a lock at a higher level."},{"title":"Level 1","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-1","category":"section","text":"No other lock may be acquired when one of these locks is held.  As a result, the code must not do any allocation or hit any safepoints. Note that there are safepoints when doing allocation, enabling/disabling GC, entering/restoring exception frames, and taking/releasing locks.\n\nsafepoint_lock (uv_mutex_t)\ndanger: Danger\nThis lock is acquired implicitly by JL_LOCK and JL_UNLOCK. Use the _NOGC variants to avoid that for level 1 locks.\nshared_map_lock.mtx (uv_mutex_t)\nfinalizers_lock (jl_mutex_t)\ngc_pages_lock (uv_mutex_t)\ngc_perm_lock (uv_mutex_t)\ngc_queue_observer_lock (uv_mutex_t)\ngc_threads_lock (uv_mutex_t)\nflisp_lock (uv_mutex_t)\nnote: Note\nflisp itself is already threadsafe; this lock only protects the jl_ast_context_list_t pool.  Likewise, the ResourcePool<?>::mutexes just protect the associated resource pool.\nResourcePool<?>.mutex (std::mutex)\nRLST_mutex (std::mutex)\nllvm_printing_mutex (std::mutex)\njl_locked_stream.mutex (std::mutex)\ndebuginfo_asyncsafe (uv_rwlock_t) (can still acquire jl_in_stackwalk (uv_mutex_t, Win32 only))\nprofile_show_peek_cond_lock (jl_mutex_t)\ntrampoline_lock (uv_mutex_t)\nbt_data_prof_lock (uv_mutex_t)\njl_ptls_t.sleep_lock (uv_mutex_t)\ntls_lock (uv_mutex_t)\npage_profile_lock (uv_mutex_t)\nsymtab_lock (uv_mutex_t)\nengine_lock (std::mutex)"},{"title":"Level 2","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-2","category":"section","text":"global_roots_lock\njl_module_t.lock\nnewly_inferred_mutex\nJLDebuginfoPlugin.PluginMutex (std::mutex)\nprecompile_field_replace_lock\nlive_tasks_lock (uv_mutex_t)\nheapsnapshot_lock\njitlock\njl_safepoint_suspend_all_threads and jl_safepoint_resume_all_threads\nnote: Note\nInside a region protected by these functions, all other threads are blocked inside a safepoint.  It is unsafe to take locks that may safepoint in this region."},{"title":"Level 3","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-3","category":"section","text":"jl_method_t.writelock\ntypecache_lock\nlibmap_lock"},{"title":"Level 4","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-4","category":"section","text":"jl_methcache_t.writelock"},{"title":"Level 5","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-5","category":"section","text":"jl_methtable_t.writelock"},{"title":"Level 6","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-6","category":"section","text":"No Julia code may be called while holding a lock above this point.\n\nworld_counter_lock"},{"title":"Level 7","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-7","category":"section","text":"JuliaOJIT::EmissionMutex (std::recursive_mutex)\njl_modules_mutex\njl_uv_mutex (known as iolock from Julia)\ndanger: Danger\nDoing any I/O (for example, printing warning messages or debug information) while holding any other lock listed above may result in pernicious and hard-to-find deadlocks.\nIndividual ThreadSynchronizer locks\ndanger: Danger\nThis may continue to be held after releasing the iolock, or acquired without it, but be very careful to never attempt to acquire the iolock while holding it.\nLibdl.LazyLibrary.lock (ReentrantLock)\norc::ThreadSafeContext\ncfun_lock"},{"title":"Level 8","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Level-8","category":"section","text":"precomp_statement_out_lock\ndispatch_statement_out_lock"},{"title":"Exceptions to the lock hierarchy","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Exceptions-to-the-lock-hierarchy","category":"section","text":"Ordinarily, it is forbidden to acquire locks of equal level to a lock already held.  In these specific cases we use a special protocol for acquiring locks at the same level:\n\njl_method_t.writelock\nInvalidation acquires the lock for every method during its depth-first search for backedges.  To avoid deadlocks, we must already hold world_counter_lock before acquiring multiple jl_method_t.writelocks."},{"title":"Broken locks","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Broken-locks","category":"section","text":"The following locks are broken:\n\nloading.jl: require and register_root_module\nThis file potentially has numerous problems. (fix: needs locks)"},{"title":"Updates to the world counter","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#Updates-to-the-world-counter","category":"section","text":"Thanks to the world age mechanism, Julia can allow the replacement of both methods and bindings, yet remain amenable to optimization. Every compiled CodeInstance has a range of valid world ages; we could conservatively assume all CIs are stale after a world age increment.  However, to avoid spurious recompilation, we track dependencies, called \"edges\", while maintaining the following invariant:\n\nFor every published CodeInstance, either:\n\nmin_world and max_world are finite, and the CI is valid for every world in that range.\nmax_world is ∞ (-1), and this CI is ready for invalidation, meaning for every forward edge:\nIf the edge is a CodeInstance that is invoked or inlined into this CI, the edge's MethodInstance backedge array has an entry pointing back.\nIf the edge is a Binding:\nIf the binding is in another module, it has an entry for this CI in its backedges array.\nIf the binding is in the same module, the Method for this CI is in the module's scanned_methods array.\n\nFor example, the following code replaces a constant in another module, causing a chain of invalidations:\n\nconst c1 = 1\nmodule M const c2 = 2 end\nf() = getfield(M, :c2)\ng() = f() + c1\n\ng()                   # compile g\n\n@eval M const c2 = 3  # invalidate f, g\ng()                   # recompile g\n\nAfter compiling the two versions of g(), the global cache looks like this: (Image: Global cache state after invalidation)\n\nThe maximum world age, jl_world_counter, is protected by the world_counter_lock.  Julia uses a form of optimistic concurrency control to allow type inference without holding world_counter_lock.\n\nPublishing a new method or binding follows these steps:\n\nAcquire world_counter_lock.\nRelaxed-load jl_world_counter and let new_world = jl_world_counter + 1.\nPublish the new binding partitions or method table entries with world range [new_world, ∞).  This step is described in the section on the lock free data structures.\nRelease-store new_world to jl_world_counter.\nRelease world_counter_lock.\n\nType inference proceeds like so:\n\nAcquire-load jl_world_counter (call this validation_world).\nPerform type inference in that world, reading the bindings and method table in that world using the lock-free data structures.\nStore back edges for every inferred CodeInstance:\nFor non-local bindings, this acquires the binding's module's lock.\nFor CIs, this acquires the method's lock.\nAcquire world_counter_lock.\nRelaxed-load jl_world_counter and compare it to validation_world:\nIf it is different, leave the valid world ranges for the inferred CIs unchanged.\nIf it is unchanged, our optimism was rewarded.  We can promote all the inferred CIs valid in validation_world to [validation_world, ∞) and rely on the backedges for invalidation.\nRelease world_counter_lock.\n\n(Image: Two threads doing type inference while another adds a method)\n\nIn the above diagram, threads 1 and 2 are doing type inference (the dotted line), while thread 3 is activating a new method.  The solid boxes represent critical sections where the world_counter_lock is held.  acq, rel, and read, are acquire loads, release stores, and relaxed loads respectively.\n\nT1 promotes its CI in time, but T2 takes too long, blocking on world_counter_lock until T3 has finished publishing the new method and incrementing the world counter.  It reads W+1 and fails to promote its CI, leaving it with a maximum world of W."},{"title":"Lock free data structures","page":"Proper maintenance and care of multi-threading locks","location":"devdocs/locks.html#man-lock-free-data","category":"section","text":"TODO"},{"title":"Single- and multi-dimensional Arrays","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-multi-dim-arrays","category":"section","text":"Julia, like most technical computing languages, provides a first-class array implementation. Most technical computing languages pay a lot of attention to their array implementation at the expense of other containers. Julia does not treat arrays in any special way. The array library is implemented almost completely in Julia itself, and derives its performance from the compiler, just like any other code written in Julia. As such, it's also possible to define custom array types by inheriting from AbstractArray. See the manual section on the AbstractArray interface for more details on implementing a custom array type.\n\nAn array is a collection of objects stored in a multi-dimensional grid. Zero-dimensional arrays are allowed, see this FAQ entry. In the most general case, an array may contain objects of type Any. For most computational purposes, arrays should contain objects of a more specific type, such as Float64 or Int32.\n\nIn general, unlike many other technical computing languages, Julia does not expect programs to be written in a vectorized style for performance. Julia's compiler uses type inference and generates optimized code for scalar array indexing, allowing programs to be written in a style that is convenient and readable, without sacrificing performance, and using less memory at times.\n\nIn Julia, all arguments to functions are passed by sharing (i.e. by pointers). Some technical computing languages pass arrays by value, and while this prevents accidental modification by callees of a value in the caller, it makes avoiding unwanted copying of arrays difficult. By convention, a function name ending with a ! indicates that it will mutate or destroy the value of one or more of its arguments (compare, for example, sort and sort!). Callees must make explicit copies to ensure that they don't modify inputs that they don't intend to change. Many non-mutating functions are implemented by calling a function of the same name with an added ! at the end on an explicit copy of the input, and returning that copy."},{"title":"Basic Functions","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Basic-Functions","category":"section","text":"Function Description\neltype(A) the type of the elements contained in A\nlength(A) the number of elements in A\nndims(A) the number of dimensions of A\nsize(A) a tuple containing the dimensions of A\nsize(A,n) the size of A along dimension n\naxes(A) a tuple containing the valid indices of A\naxes(A,n) a range expressing the valid indices along dimension n\neachindex(A) an efficient iterator for visiting each position in A\nstride(A,k) the stride (linear index distance between adjacent elements) along dimension k\nstrides(A) a tuple of the strides in each dimension"},{"title":"Construction and Initialization","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Construction-and-Initialization","category":"section","text":"Many functions for constructing and initializing arrays are provided. In the following list of such functions, calls with a dims... argument can either take a single tuple of dimension sizes or a series of dimension sizes passed as a variable number of arguments. Most of these functions also accept a first input T, which is the element type of the array. If the type T is omitted it will default to Float64.\n\nFunction Description\nArray{T}(undef, dims...) an uninitialized dense Array\nzeros(T, dims...) an Array of all zeros\nones(T, dims...) an Array of all ones\ntrues(dims...) a BitArray with all values true\nfalses(dims...) a BitArray with all values false\nreshape(A, dims...) an array containing the same data as A, but with different dimensions\ncopy(A) copy A\ndeepcopy(A) copy A, recursively copying its elements\nsimilar(A, T, dims...) an uninitialized array of the same type as A (dense, sparse, etc.), but with the specified element type and dimensions. The second and third arguments are both optional, defaulting to the element type and dimensions of A if omitted.\nreinterpret(T, A) an array with the same binary data as A, but with element type T\nrand(T, dims...) an Array with random, iid [1] and uniformly distributed values. For floating point types T, the values lie in the half-open interval 0 1).\nrandn(T, dims...) an Array with random, iid and standard normally distributed values\nMatrix{T}(I, m, n) m-by-n identity matrix. Requires using LinearAlgebra for I.\nrange(start, stop, n) a range of n linearly spaced elements from start to stop\nfill!(A, x) fill the array A with the value x\nfill(x, dims...) an Array filled with the value x. In particular, fill(x) constructs a zero-dimensional Array containing x.\n\n[1]: iid, independently and identically distributed.\n\nTo see the various ways we can pass dimensions to these functions, consider the following examples:\n\njulia> zeros(Int8, 2, 3)\n2×3 Matrix{Int8}:\n 0  0  0\n 0  0  0\n\njulia> zeros(Int8, (2, 3))\n2×3 Matrix{Int8}:\n 0  0  0\n 0  0  0\n\njulia> zeros((2, 3))\n2×3 Matrix{Float64}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\nHere, (2, 3) is a Tuple and the first argument — the element type — is optional, defaulting to Float64."},{"title":"Array literals","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-array-literals","category":"section","text":"Arrays can also be directly constructed with square braces; the syntax [A, B, C, ...] creates a one-dimensional array (i.e., a vector) containing the comma-separated arguments as its elements. The element type (eltype) of the resulting array is automatically determined by the types of the arguments inside the braces. If all the arguments are the same type, then that is its eltype. If they all have a common promotion type then they get converted to that type using convert and that type is the array's eltype. Otherwise, a heterogeneous array that can hold anything — a Vector{Any} — is constructed; this includes the literal [] where no arguments are given. Array literals can be typed with the syntax T[A, B, C, ...] where T is a type.\n\njulia> [1, 2, 3] # An array of `Int`s\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> promote(1, 2.3, 4//5) # This combination of Int, Float64 and Rational promotes to Float64\n(1.0, 2.3, 0.8)\n\njulia> [1, 2.3, 4//5] # Thus that's the element type of this Array\n3-element Vector{Float64}:\n 1.0\n 2.3\n 0.8\n\njulia> Float32[1, 2.3, 4//5] # Specify element type manually\n3-element Vector{Float32}:\n 1.0\n 2.3\n 0.8\n\njulia> []\nAny[]"},{"title":"Concatenation","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-array-concatenation","category":"section","text":"If the arguments inside the square brackets are separated by single semicolons (;) or newlines instead of commas, then their contents are vertically concatenated together instead of the arguments being used as elements themselves.\n\njulia> [1:2, 4:5] # Has a comma, so no concatenation occurs. The ranges are themselves the elements\n2-element Vector{UnitRange{Int64}}:\n 1:2\n 4:5\n\njulia> [1:2; 4:5]\n4-element Vector{Int64}:\n 1\n 2\n 4\n 5\n\njulia> [1:2\n        4:5\n        6]\n5-element Vector{Int64}:\n 1\n 2\n 4\n 5\n 6\n\nSimilarly, if the arguments are separated by tabs or spaces or double semicolons, then their contents are horizontally concatenated together.\n\njulia> [1:2  4:5  7:8]\n2×3 Matrix{Int64}:\n 1  4  7\n 2  5  8\n\njulia> [[1,2]  [4,5]  [7,8]]\n2×3 Matrix{Int64}:\n 1  4  7\n 2  5  8\n\njulia> [1 2 3] # Numbers can also be horizontally concatenated\n1×3 Matrix{Int64}:\n 1  2  3\n\njulia> [1;; 2;; 3;; 4]\n1×4 Matrix{Int64}:\n 1  2  3  4\n\nSingle semicolons (or newlines) and spaces (or tabs) can be combined to concatenate both horizontally and vertically at the same time.\n\njulia> [1 2\n        3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> [zeros(Int, 2, 2) [1; 2]\n        [3 4]            5]\n3×3 Matrix{Int64}:\n 0  0  1\n 0  0  2\n 3  4  5\n\njulia> [[1 1]; 2 3; [4 4]]\n3×2 Matrix{Int64}:\n 1  1\n 2  3\n 4  4\n\nSpaces (and tabs) have a higher precedence than semicolons, performing any horizontal concatenations first and then concatenating the result. Using double semicolons for the horizontal concatenation, on the other hand, performs any vertical concatenations before horizontally concatenating the result.\n\njulia> [zeros(Int, 2, 2) ; [3 4] ;; [1; 2] ; 5]\n3×3 Matrix{Int64}:\n 0  0  1\n 0  0  2\n 3  4  5\n\njulia> [1:2; 4;; 1; 3:4]\n3×2 Matrix{Int64}:\n 1  1\n 2  3\n 4  4\n\nJust as ; and ;; concatenate in the first and second dimension, using more semicolons extends this same general scheme. The number of semicolons in the separator specifies the particular dimension, so ;;; concatenates in the third dimension, ;;;; in the 4th, and so on. Fewer semicolons take precedence, so the lower dimensions are generally concatenated first.\n\njulia> [1; 2;; 3; 4;; 5; 6;;;\n        7; 8;; 9; 10;; 11; 12]\n2×3×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  3  5\n 2  4  6\n\n[:, :, 2] =\n 7   9  11\n 8  10  12\n\nLike before, spaces (and tabs) for horizontal concatenation have a higher precedence than any number of semicolons. Thus, higher-dimensional arrays can also be written by specifying their rows first, with their elements textually arranged in a manner similar to their layout:\n\njulia> [1 3 5\n        2 4 6;;;\n        7 9 11\n        8 10 12]\n2×3×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  3  5\n 2  4  6\n\n[:, :, 2] =\n 7   9  11\n 8  10  12\n\njulia> [1 2;;; 3 4;;;; 5 6;;; 7 8]\n1×2×2×2 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1  2\n\n[:, :, 2, 1] =\n 3  4\n\n[:, :, 1, 2] =\n 5  6\n\n[:, :, 2, 2] =\n 7  8\n\njulia> [[1 2;;; 3 4];;;; [5 6];;; [7 8]]\n1×2×2×2 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1  2\n\n[:, :, 2, 1] =\n 3  4\n\n[:, :, 1, 2] =\n 5  6\n\n[:, :, 2, 2] =\n 7  8\n\nAlthough they both mean concatenation in the second dimension, spaces (or tabs) and ;; cannot appear in the same array expression unless the double semicolon is simply serving as a \"line continuation\" character. This allows a single horizontal concatenation to span multiple lines (without the line break being interpreted as a vertical concatenation).\n\njulia> [1 2 ;;\n       3 4]\n1×4 Matrix{Int64}:\n 1  2  3  4\n\nTerminating semicolons may also be used to add trailing length 1 dimensions.\n\njulia> [1;;]\n1×1 Matrix{Int64}:\n 1\n\njulia> [2; 3;;;]\n2×1×1 Array{Int64, 3}:\n[:, :, 1] =\n 2\n 3\n\nMore generally, concatenation can be accomplished through the cat function. These syntaxes are shorthands for function calls that themselves are convenience functions:\n\nSyntax Function Description\n cat concatenate input arrays along dimension(s) k\n[A; B; C; ...] vcat shorthand for cat(A...; dims=1)\n[A B C ...] hcat shorthand for cat(A...; dims=2)\n[A B; C D; ...] hvcat simultaneous vertical and horizontal concatenation\n[A; C;; B; D;;; ...] hvncat simultaneous n-dimensional concatenation, where number of semicolons indicate the dimension to concatenate"},{"title":"Typed array literals","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-array-typed-literal","category":"section","text":"An array with a specific element type can be constructed using the syntax T[A, B, C, ...]. This will construct a 1-d array with element type T, initialized to contain elements A, B, C, etc. For example, Any[x, y, z] constructs a heterogeneous array that can contain any values.\n\nConcatenation syntax can similarly be prefixed with a type to specify the element type of the result.\n\njulia> [[1 2] [3 4]]\n1×4 Matrix{Int64}:\n 1  2  3  4\n\njulia> Int8[[1 2] [3 4]]\n1×4 Matrix{Int8}:\n 1  2  3  4"},{"title":"Comprehensions","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-comprehensions","category":"section","text":"Comprehensions provide a general and powerful way to construct arrays. Comprehension syntax is similar to set construction notation in mathematics:\n\nA = [ F(x, y, ...) for x=rx, y=ry, ... ]\n\nThe meaning of this form is that F(x,y,...) is evaluated with the variables x, y, etc. taking on each value in their given list of values. Values can be specified as any iterable object, but will commonly be ranges like 1:n or 2:(n-1), or explicit arrays of values like [1.2, 3.4, 5.7]. The result is an N-d dense array with dimensions that are the concatenation of the dimensions of the variable ranges rx, ry, etc. and each F(x,y,...) evaluation returns a scalar.\n\nThe following example computes a weighted average of the current element and its left and right neighbor along a 1-d grid:\n\njulia> x = [4, 8, 2, 6, 10, 10, 2, 8]\n8-element Vector{Int64}:\n  4\n  8\n  2\n  6\n 10\n 10\n  2\n  8\n\njulia> [ 0.25*x[i-1] + 0.5*x[i] + 0.25*x[i+1] for i=2:length(x)-1 ]\n6-element Vector{Float64}:\n 5.5\n 4.5\n 6.0\n 9.0\n 8.0\n 5.5\n\nThe resulting array type depends on the types of the computed elements just like array literals do. In order to control the type explicitly, a type can be prepended to the comprehension. For example, we could have requested the result in single precision by writing:\n\nFloat32[ 0.25*x[i-1] + 0.5*x[i] + 0.25*x[i+1] for i=2:length(x)-1 ]"},{"title":"Generator Expressions","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-generators","category":"section","text":"Comprehensions can also be written without the enclosing square brackets, producing an object known as a generator. This object can be iterated to produce values on demand, instead of allocating an array and storing them in advance (see Iteration). For example, the following expression sums a series without allocating memory:\n\njulia> sum(1/n^2 for n=1:1000)\n1.6439345666815615\n\nWhen writing a generator expression with multiple dimensions inside an argument list, parentheses are needed to separate the generator from subsequent arguments:\n\njulia> map(tuple, 1/(i+j) for i=1:2, j=1:2, [1:4;])\nERROR: ParseError:\n# Error @ none:1:44\nmap(tuple, 1/(i+j) for i=1:2, j=1:2, [1:4;])\n#                                          └ ── invalid iteration spec: expected one of `=` `in` or `∈`\n\nAll comma-separated expressions after for are interpreted as ranges. Adding parentheses lets us add a third argument to map:\n\njulia> map(tuple, (1/(i+j) for i=1:2, j=1:2), [1 3; 2 4])\n2×2 Matrix{Tuple{Float64, Int64}}:\n (0.5, 1)       (0.333333, 3)\n (0.333333, 2)  (0.25, 4)\n\nGenerators are implemented via inner functions. Just like inner functions used elsewhere in the language, variables from the enclosing scope can be \"captured\" in the inner function. For example, sum(p[i] - q[i] for i=1:n) captures the three variables p, q and n from the enclosing scope. Captured variables can present performance challenges; see performance tips.\n\nRanges in generators and comprehensions can depend on previous ranges by writing multiple for keywords:\n\njulia> [(i, j) for i=1:3 for j=1:i]\n6-element Vector{Tuple{Int64, Int64}}:\n (1, 1)\n (2, 1)\n (2, 2)\n (3, 1)\n (3, 2)\n (3, 3)\n\nIn such cases, the result is always 1-d.\n\nGenerated values can be filtered using the if keyword:\n\njulia> [(i, j) for i=1:3 for j=1:i if i+j == 4]\n2-element Vector{Tuple{Int64, Int64}}:\n (2, 2)\n (3, 1)"},{"title":"Indexing","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-array-indexing","category":"section","text":"The general syntax for indexing into an n-dimensional array A is:\n\nX = A[I_1, I_2, ..., I_n]\n\nwhere each I_k may be a scalar integer, an array of integers, or any other supported index. This includes Colon (:) to select all indices within the entire dimension, ranges of the form a:c or a:b:c to select contiguous or strided subsections, and arrays of booleans to select elements at their true indices.\n\nIf all the indices are scalars, then the result X is a single element from the array A. Otherwise, X is an array with the same number of dimensions as the sum of the dimensionalities of all the indices.\n\nIf all indices I_k are vectors, for example, then the shape of X would be (length(I_1), length(I_2), ..., length(I_n)), with location i_1, i_2, ..., i_n of X containing the value A[I_1[i_1], I_2[i_2], ..., I_n[i_n]].\n\nExample:\n\njulia> A = reshape(collect(1:16), (2, 2, 2, 2))\n2×2×2×2 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1  3\n 2  4\n\n[:, :, 2, 1] =\n 5  7\n 6  8\n\n[:, :, 1, 2] =\n  9  11\n 10  12\n\n[:, :, 2, 2] =\n 13  15\n 14  16\n\njulia> A[1, 2, 1, 1] # all scalar indices\n3\n\njulia> A[[1, 2], [1], [1, 2], [1]] # all vector indices\n2×1×2×1 Array{Int64, 4}:\n[:, :, 1, 1] =\n 1\n 2\n\n[:, :, 2, 1] =\n 5\n 6\n\njulia> A[[1, 2], [1], [1, 2], 1] # a mix of index types\n2×1×2 Array{Int64, 3}:\n[:, :, 1] =\n 1\n 2\n\n[:, :, 2] =\n 5\n 6\n\nNote how the size of the resulting array is different in the last two cases.\n\nIf I_1 is changed to a two-dimensional matrix, then X becomes an n+1-dimensional array of shape (size(I_1, 1), size(I_1, 2), length(I_2), ..., length(I_n)). The matrix adds a dimension.\n\nExample:\n\njulia> A = reshape(collect(1:16), (2, 2, 2, 2));\n\njulia> A[[1 2; 1 2]]\n2×2 Matrix{Int64}:\n 1  2\n 1  2\n\njulia> A[[1 2; 1 2], 1, 2, 1]\n2×2 Matrix{Int64}:\n 5  6\n 5  6\n\nThe location i_1, i_2, i_3, ..., i_{n+1} contains the value at A[I_1[i_1, i_2], I_2[i_3], ..., I_n[i_{n+1}]]. All dimensions indexed with scalars are dropped. For example, if J is an array of indices, then the result of A[2, J, 3] is an array with size size(J). Its jth element is populated by A[2, J[j], 3].\n\nAs a special part of this syntax, the end keyword may be used to represent the last index of each dimension within the indexing brackets, as determined by the size of the innermost array being indexed. Indexing syntax without the end keyword is equivalent to a call to getindex:\n\nX = getindex(A, I_1, I_2, ..., I_n)\n\nExample:\n\njulia> x = reshape(1:16, 4, 4)\n4×4 reshape(::UnitRange{Int64}, 4, 4) with eltype Int64:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> x[2:3, 2:end-1]\n2×2 Matrix{Int64}:\n 6  10\n 7  11\n\njulia> x[1, [2 3; 4 1]]\n2×2 Matrix{Int64}:\n  5  9\n 13  1"},{"title":"Indexed Assignment","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-indexed-assignment","category":"section","text":"The general syntax for assigning values in an n-dimensional array A is:\n\nA[I_1, I_2, ..., I_n] = X\n\nwhere each I_k may be a scalar integer, an array of integers, or any other supported index. This includes Colon (:) to select all indices within the entire dimension, ranges of the form a:c or a:b:c to select contiguous or strided subsections, and arrays of booleans to select elements at their true indices.\n\nIf all indices I_k are integers, then the value in location I_1, I_2, ..., I_n of A is overwritten with the value of X, converting to the eltype of A if necessary.\n\nIf any index I_k is itself an array, then the right hand side X must also be an array with the same shape as the result of indexing A[I_1, I_2, ..., I_n] or a vector with the same number of elements. The value in location I_1[i_1], I_2[i_2], ..., I_n[i_n] of A is overwritten with the value X[i_1, i_2, ..., i_n], converting if necessary. The element-wise assignment operator .= may be used to broadcast X across the selected locations:\n\nA[I_1, I_2, ..., I_n] .= X\n\nJust as in Indexing, the end keyword may be used to represent the last index of each dimension within the indexing brackets, as determined by the size of the array being assigned into. Indexed assignment syntax without the end keyword is equivalent to a call to setindex!:\n\nsetindex!(A, X, I_1, I_2, ..., I_n)\n\nExample:\n\njulia> x = collect(reshape(1:9, 3, 3))\n3×3 Matrix{Int64}:\n 1  4  7\n 2  5  8\n 3  6  9\n\njulia> x[3, 3] = -9;\n\njulia> x[1:2, 1:2] = [-1 -4; -2 -5];\n\njulia> x\n3×3 Matrix{Int64}:\n -1  -4   7\n -2  -5   8\n  3   6  -9"},{"title":"Supported index types","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-supported-index-types","category":"section","text":"In the expression A[I_1, I_2, ..., I_n], each I_k may be a scalar index, an array of scalar indices, or an object that represents an array of scalar indices and can be converted to such by to_indices:\n\nA scalar index. By default this includes:\nNon-boolean integers\nCartesianIndex{N}s, which behave like an N-tuple of integers spanning multiple dimensions (see below for more details)\nAn array of scalar indices. This includes:\nVectors and multidimensional arrays of integers\nEmpty arrays like [], which select no elements e.g. A[[]] (not to be confused with A[])\nRanges like a:c or a:b:c, which select contiguous or strided subsections from a to c (inclusive)\nAny custom array of scalar indices that is a subtype of AbstractArray\nArrays of CartesianIndex{N} (see below for more details)\nAn object that represents an array of scalar indices and can be converted to such by to_indices. By default this includes:\nColon() (:), which represents all indices within an entire dimension or across the entire array\nArrays of booleans, which select elements at their true indices (see below for more details)\n\nSome examples:\n\njulia> A = reshape(collect(1:2:18), (3, 3))\n3×3 Matrix{Int64}:\n 1   7  13\n 3   9  15\n 5  11  17\n\njulia> A[4]\n7\n\njulia> A[[2, 5, 8]]\n3-element Vector{Int64}:\n  3\n  9\n 15\n\njulia> A[[1 4; 3 8]]\n2×2 Matrix{Int64}:\n 1   7\n 5  15\n\njulia> A[[]]\nInt64[]\n\njulia> A[1:2:5]\n3-element Vector{Int64}:\n 1\n 5\n 9\n\njulia> A[2, :]\n3-element Vector{Int64}:\n  3\n  9\n 15\n\njulia> A[:, 3]\n3-element Vector{Int64}:\n 13\n 15\n 17\n\njulia> A[:, 3:3]\n3×1 Matrix{Int64}:\n 13\n 15\n 17"},{"title":"Cartesian indices","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Cartesian-indices","category":"section","text":"The special CartesianIndex{N} object represents a scalar index that behaves like an N-tuple of integers spanning multiple dimensions. For example:\n\njulia> A = reshape(1:32, 4, 4, 2);\n\njulia> A[3, 2, 1]\n7\n\njulia> A[CartesianIndex(3, 2, 1)] == A[3, 2, 1] == 7\ntrue\n\nConsidered alone, this may seem relatively trivial; CartesianIndex simply gathers multiple integers together into one object that represents a single multidimensional index. When combined with other indexing forms and iterators that yield CartesianIndexes, however, this can produce very elegant and efficient code. See Iteration below, and for some more advanced examples, see this blog post on multidimensional algorithms and iteration.\n\nArrays of CartesianIndex{N} are also supported. They represent a collection of scalar indices that each span N dimensions, enabling a form of indexing that is sometimes referred to as pointwise indexing. For example, it enables accessing the diagonal elements from the first \"page\" of A from above:\n\njulia> page = A[:, :, 1]\n4×4 Matrix{Int64}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> page[[CartesianIndex(1, 1),\n             CartesianIndex(2, 2),\n             CartesianIndex(3, 3),\n             CartesianIndex(4, 4)]]\n4-element Vector{Int64}:\n  1\n  6\n 11\n 16\n\nThis can be expressed much more simply with dot broadcasting and by combining it with a normal integer index (instead of extracting the first page from A as a separate step). It can even be combined with a : to extract both diagonals from the two pages at the same time:\n\njulia> A[CartesianIndex.(axes(A, 1), axes(A, 2)), 1]\n4-element Vector{Int64}:\n  1\n  6\n 11\n 16\n\njulia> A[CartesianIndex.(axes(A, 1), axes(A, 2)), :]\n4×2 Matrix{Int64}:\n  1  17\n  6  22\n 11  27\n 16  32\n\nwarning: Warning\nCartesianIndex and arrays of CartesianIndex are not compatible with the end keyword to represent the last index of a dimension. Do not use end in indexing expressions that may contain either CartesianIndex or arrays thereof."},{"title":"Logical indexing","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Logical-indexing","category":"section","text":"Often referred to as logical indexing or indexing with a logical mask, indexing by a boolean array selects elements at the indices where its values are true. Indexing by a boolean vector B is effectively the same as indexing by the vector of integers that is returned by findall(B). Similarly, indexing by a N-dimensional boolean array is effectively the same as indexing by the vector of CartesianIndex{N}s where its values are true. A logical index must be an array of the same shape as the dimension(s) it indexes into, or it must be the only index provided and match the shape of the one-dimensional reshaped view of the array it indexes into. It is generally more efficient to use boolean arrays as indices directly instead of first calling findall.\n\njulia> x = reshape(1:12, 2, 3, 2)\n2×3×2 reshape(::UnitRange{Int64}, 2, 3, 2) with eltype Int64:\n[:, :, 1] =\n 1  3  5\n 2  4  6\n\n[:, :, 2] =\n 7   9  11\n 8  10  12\n\njulia> x[:, [true false; false true; true false]]\n2×3 Matrix{Int64}:\n 1  5   9\n 2  6  10\n\njulia> mask = map(ispow2, x)\n2×3×2 Array{Bool, 3}:\n[:, :, 1] =\n 1  0  0\n 1  1  0\n\n[:, :, 2] =\n 0  0  0\n 1  0  0\n\njulia> x[mask]\n4-element Vector{Int64}:\n 1\n 2\n 4\n 8\n\njulia> x[vec(mask)] == x[mask] # we can also index with a single Boolean vector\ntrue"},{"title":"Number of indices","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Number-of-indices","category":"section","text":""},{"title":"Cartesian indexing","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Cartesian-indexing","category":"section","text":"The ordinary way to index into an N-dimensional array is to use exactly N indices; each index selects the position(s) in its particular dimension. For example, in the three-dimensional array A = rand(4, 3, 2), A[2, 3, 1] will select the number in the second row of the third column in the first \"page\" of the array. This is often referred to as cartesian indexing."},{"title":"Linear indexing","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Linear-indexing","category":"section","text":"When exactly one index i is provided, that index no longer represents a location in a particular dimension of the array. Instead, it selects the ith element using the column-major iteration order that linearly spans the entire array. This is known as linear indexing. It essentially treats the array as though it had been reshaped into a one-dimensional vector with vec.\n\njulia> A = [2 6; 4 7; 3 1]\n3×2 Matrix{Int64}:\n 2  6\n 4  7\n 3  1\n\njulia> A[5]\n7\n\njulia> vec(A)[5]\n7\n\nA linear index into the array A can be converted to a CartesianIndex for cartesian indexing with CartesianIndices(A)[i] (see CartesianIndices), and a set of N cartesian indices can be converted to a linear index with LinearIndices(A)[i_1, i_2, ..., i_N] (see LinearIndices).\n\njulia> CartesianIndices(A)[5]\nCartesianIndex(2, 2)\n\njulia> LinearIndices(A)[2, 2]\n5\n\nIt's important to note that there's a very large asymmetry in the performance of these conversions. Converting a linear index to a set of cartesian indices requires dividing and taking the remainder, whereas going the other way is just multiplies and adds. In modern processors, integer division can be 10-50 times slower than multiplication. While some arrays — like Array itself — are implemented using a linear chunk of memory and directly use a linear index in their implementations, other arrays — like Diagonal — need the full set of cartesian indices to do their lookup (see IndexStyle to introspect which is which).\n\nwarning: Warning\nWhen iterating over all the indices for an array, it is better to iterate over eachindex(A) instead of 1:length(A). Not only will this be faster in cases where A is IndexCartesian, but it will also support arrays with custom indexing, such as OffsetArrays. If only the values are needed, then is better to just iterate the array directly, i.e. for a in A."},{"title":"Omitted and extra indices","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Omitted-and-extra-indices","category":"section","text":"In addition to linear indexing, an N-dimensional array may be indexed with fewer or more than N indices in certain situations.\n\nIndices may be omitted if the trailing dimensions that are not indexed into are all length one. In other words, trailing indices can be omitted only if there is only one possible value that those omitted indices could be for an in-bounds indexing expression. For example, a four-dimensional array with size (3, 4, 2, 1) may be indexed with only three indices as the dimension that gets skipped (the fourth dimension) has length one. Note that linear indexing takes precedence over this rule.\n\njulia> A = reshape(1:24, 3, 4, 2, 1)\n3×4×2×1 reshape(::UnitRange{Int64}, 3, 4, 2, 1) with eltype Int64:\n[:, :, 1, 1] =\n 1  4  7  10\n 2  5  8  11\n 3  6  9  12\n\n[:, :, 2, 1] =\n 13  16  19  22\n 14  17  20  23\n 15  18  21  24\n\njulia> A[1, 3, 2] # Omits the fourth dimension (length 1)\n19\n\njulia> A[1, 3] # Attempts to omit dimensions 3 & 4 (lengths 2 and 1)\nERROR: BoundsError: attempt to access 3×4×2×1 reshape(::UnitRange{Int64}, 3, 4, 2, 1) with eltype Int64 at index [1, 3]\n\njulia> A[19] # Linear indexing\n19\n\nWhen omitting all indices with A[], this semantic provides a simple idiom to retrieve the only element in an array and simultaneously ensure that there was only one element.\n\nSimilarly, more than N indices may be provided if all the indices beyond the dimensionality of the array are 1 (or more generally are the first and only element of axes(A, d) where d is that particular dimension number). This allows vectors to be indexed like one-column matrices, for example:\n\njulia> A = [8, 6, 7]\n3-element Vector{Int64}:\n 8\n 6\n 7\n\njulia> A[2, 1]\n6"},{"title":"Iteration","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Iteration","category":"section","text":"The recommended ways to iterate over a whole array are\n\nfor a in A\n    # Do something with the element a\nend\n\nfor i in eachindex(A)\n    # Do something with i and/or A[i]\nend\n\nThe first construct is used when you need the value, but not index, of each element. In the second construct, i will be an Int if A is an array type with fast linear indexing; otherwise, it will be a CartesianIndex:\n\njulia> A = rand(4, 3);\n\njulia> B = view(A, 1:3, 2:3);\n\njulia> for i in eachindex(B)\n           @show i\n       end\ni = CartesianIndex(1, 1)\ni = CartesianIndex(2, 1)\ni = CartesianIndex(3, 1)\ni = CartesianIndex(1, 2)\ni = CartesianIndex(2, 2)\ni = CartesianIndex(3, 2)\n\nnote: Note\nIn contrast with for i = 1:length(A), iterating with eachindex provides an efficient way to iterate over any array type. Besides, this also supports generic arrays with custom indexing such as OffsetArrays."},{"title":"Array traits","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Array-traits","category":"section","text":"If you write a custom AbstractArray type, you can specify that it has fast linear indexing using\n\nBase.IndexStyle(::Type{<:MyArray}) = IndexLinear()\n\nThis setting will cause eachindex iteration over a MyArray to use integers. If you don't specify this trait, the default value IndexCartesian() is used."},{"title":"Array and Vectorized Operators and Functions","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#man-array-and-vectorized-operators-and-functions","category":"section","text":"The following operators are supported for arrays:\n\nUnary arithmetic – -, +\nBinary arithmetic – -, +, *, /, \\, ^\nComparison – ==, !=, ≈ (isapprox), ≉\n\nTo enable convenient vectorization of mathematical and other operations, Julia provides the dot syntax f.(args...), e.g. sin.(x) or min.(x, y), for elementwise operations over arrays or mixtures of arrays and scalars (a Broadcasting operation); these have the additional advantage of \"fusing\" into a single loop when combined with other dot calls, e.g. sin.(cos.(x)).\n\nAlso, every binary operator supports a dot version that can be applied to arrays (and combinations of arrays and scalars) in such fused broadcasting operations, e.g. z .== sin.(x .* y).\n\nNote that comparisons such as == operate on whole arrays, giving a single boolean answer. Use dot operators like .== for elementwise comparisons. (For comparison operations like <, only the elementwise .< version is applicable to arrays.)\n\nAlso notice the difference between max.(a,b), which broadcasts max elementwise over a and b, and maximum(a), which finds the largest value within a. The same relationship holds for min.(a, b) and minimum(a)."},{"title":"Broadcasting","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Broadcasting","category":"section","text":"It is sometimes useful to perform element-by-element binary operations on arrays of different sizes, such as adding a vector to each column of a matrix. An inefficient way to do this would be to replicate the vector to the size of the matrix:\n\njulia> a = [0.2, 0.5]; A = [1.0 1.6 1.05; 1.07 1.36 1.18];\n\njulia> repeat(a, 1, 3) + A\n2×3 Matrix{Float64}:\n 1.2   1.8   1.25\n 1.57  1.86  1.68\n\nThis is wasteful when dimensions get large, so Julia provides broadcast, which expands singleton dimensions in array arguments to match the corresponding dimension in the other array without using extra memory, and applies the given function elementwise:\n\njulia> broadcast(+, a, A)\n2×3 Matrix{Float64}:\n 1.2   1.8   1.25\n 1.57  1.86  1.68\n\njulia> b = [0.9 0.1]\n1×2 Matrix{Float64}:\n 0.9  0.1\n\njulia> broadcast(+, a, b)\n2×2 Matrix{Float64}:\n 1.1  0.3\n 1.4  0.6\n\nDotted operators such as .+ and .* are equivalent to broadcast calls (except that they fuse, as described above). There is also a broadcast! function to specify an explicit destination (which can also be accessed in a fusing fashion by .= assignment). In fact, f.(args...) is equivalent to broadcast(f, args...), providing a convenient syntax to broadcast any function (dot syntax). Nested \"dot calls\" f.(...) (including calls to .+ etcetera) automatically fuse into a single broadcast call.\n\nAdditionally, broadcast is not limited to arrays (see the function documentation); it also handles scalars, tuples and other collections. By default, only some argument types are considered scalars, including (but not limited to) Numbers, Strings, Symbols, Types, Functions and some common singletons like missing and nothing. All other arguments are iterated over or indexed into elementwise.\n\njulia> convert.(Float32, [1, 2])\n2-element Vector{Float32}:\n 1.0\n 2.0\n\njulia> ceil.(UInt8, [1.2 3.4; 5.6 6.7])\n2×2 Matrix{UInt8}:\n 0x02  0x04\n 0x06  0x07\n\njulia> string.(1:3, \". \", [\"First\", \"Second\", \"Third\"])\n3-element Vector{String}:\n \"1. First\"\n \"2. Second\"\n \"3. Third\"\n\nSometimes, you want a container (like an array) that would normally participate in broadcast to be \"protected\" from broadcast's behavior of iterating over all of its elements. By placing it inside another container (like a single element Tuple) broadcast will treat it as a single value.\n\njulia> ([1, 2, 3], [4, 5, 6]) .+ ([1, 2, 3],)\n([2, 4, 6], [5, 7, 9])\n\njulia> ([1, 2, 3], [4, 5, 6]) .+ tuple([1, 2, 3])\n([2, 4, 6], [5, 7, 9])"},{"title":"Implementation","page":"Single- and multi-dimensional Arrays","location":"manual/arrays.html#Implementation","category":"section","text":"The base array type in Julia is the abstract type AbstractArray{T,N}. It is parameterized by the number of dimensions N and the element type T. AbstractVector and AbstractMatrix are aliases for the 1-d and 2-d cases. Operations on AbstractArray objects are defined using higher level operators and functions, in a way that is independent of the underlying storage. These operations generally work correctly as a fallback for any specific array implementation.\n\nThe AbstractArray type includes anything vaguely array-like, and implementations of it might be quite different from conventional arrays. For example, elements might be computed on request rather than stored. However, any concrete AbstractArray{T,N} type should generally implement at least size(A) (returning an Int tuple), getindex(A, i) and getindex(A, i1, ..., iN); mutable arrays should also implement setindex!. It is recommended that these operations have nearly constant time complexity, as otherwise some array functions may be unexpectedly slow. Concrete types should also typically provide a similar(A, T=eltype(A), dims=size(A)) method, which is used to allocate a similar array for copy and other out-of-place operations. No matter how an AbstractArray{T,N} is represented internally, T is the type of object returned by integer indexing (A[1, ..., 1], when A is not empty) and N should be the length of the tuple returned by size. For more details on defining custom AbstractArray implementations, see the array interface guide in the interfaces chapter.\n\nDenseArray is an abstract subtype of AbstractArray intended to include all arrays where elements are stored contiguously in column-major order (see additional notes in Performance Tips). The Array type is a specific instance of DenseArray;  Vector and Matrix are aliases for the 1-d and 2-d cases. Very few operations are implemented specifically for Array beyond those that are required for all AbstractArrays; much of the array library is implemented in a generic manner that allows all custom arrays to behave similarly.\n\nSubArray is a specialization of AbstractArray that performs indexing by sharing memory with the original array rather than by copying it. A SubArray is created with the view function, which is called the same way as getindex (with an array and a series of index arguments). The result of view looks the same as the result of getindex, except the data is left in place. view stores the input index vectors in a SubArray object, which can later be used to index the original array indirectly. By putting the @views macro in front of an expression or block of code, any array[...] slice in that expression will be converted to create a SubArray view instead.\n\nBitArrays are space-efficient \"packed\" boolean arrays, which store one bit per boolean value. They can be used similarly to Array{Bool} arrays (which store one byte per boolean value), and can be converted to/from the latter via Array(bitarray) and BitArray(array), respectively.\n\nAn array is \"strided\" if it is stored in memory with well-defined spacings (strides) between its elements. A strided array with a supported element type may be passed to an external (non-Julia) library like BLAS or LAPACK by simply passing its pointer and the stride for each dimension. The stride(A, d) is the distance between elements along dimension d. For example, the builtin Array returned by rand(5,7,2) has its elements arranged contiguously in column major order. This means that the stride of the first dimension — the spacing between elements in the same column — is 1:\n\njulia> A = rand(5, 7, 2);\n\njulia> stride(A, 1)\n1\n\nThe stride of the second dimension is the spacing between elements in the same row, skipping as many elements as there are in a single column (5). Similarly, jumping between the two \"pages\" (in the third dimension) requires skipping 5*7 == 35 elements. The strides of this array is the tuple of these three numbers together:\n\njulia> strides(A)\n(1, 5, 35)\n\nIn this particular case, the number of elements skipped in memory matches the number of linear indices skipped. This is only the case for contiguous arrays like Array (and other DenseArray subtypes) and is not true in general. Views with range indices are a good example of non-contiguous strided arrays; consider V = @view A[1:3:4, 2:2:6, 2:-1:1]. This view V refers to the same memory as A but is skipping and re-arranging some of its elements. The stride of the first dimension of V is 3 because we're only selecting every third row from our original array:\n\njulia> V = @view A[1:3:4, 2:2:6, 2:-1:1];\n\njulia> stride(V, 1)\n3\n\nThis view is similarly selecting every other column from our original A — and thus it needs to skip the equivalent of two five-element columns when moving between indices in the second dimension:\n\njulia> stride(V, 2)\n10\n\nThe third dimension is interesting because its order is reversed! Thus to get from the first \"page\" to the second one it must go backwards in memory, and so its stride in this dimension is negative!\n\njulia> stride(V, 3)\n-35\n\nThis means that the pointer for V is actually pointing into the middle of A's memory block, and it refers to elements both backwards and forwards in memory. See the interface guide for strided arrays for more details on defining your own strided arrays. StridedVector and StridedMatrix are convenient aliases for many of the builtin array types that are considered strided arrays, allowing them to dispatch to select specialized implementations that call highly tuned and optimized BLAS and LAPACK functions using just the pointer and strides.\n\nIt is worth emphasizing that strides are about offsets in memory rather than indexing. If you are looking to convert between linear (single-index) indexing and cartesian (multi-index) indexing, see LinearIndices and CartesianIndices."},{"title":"LibCURL","page":"LibCURL","location":"stdlib/LibCURL.html#LibCURL","category":"section","text":"This is a simple Julia wrapper around http://curl.haxx.se/libcurl/ generated using Clang.jl.  Please see the libcurl API documentation for help on how to use this package."},{"title":"Bounds checking","page":"Bounds checking","location":"devdocs/boundscheck.html#Bounds-checking","category":"section","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection."},{"title":"Eliding bounds checks","page":"Bounds checking","location":"devdocs/boundscheck.html#Eliding-bounds-checks","category":"section","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWith a custom array-like type MyArray having:\n\n@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A, i); A.data[to_index(i)])\n\nThen when getindex is inlined into sum, the call to checkbounds(A, i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack."},{"title":"Caution!","page":"Bounds checking","location":"devdocs/boundscheck.html#Caution!","category":"section","text":"It is easy to accidentally expose unsafe operations with @inbounds. You might be tempted to write the above example as\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWhich quietly assumes 1-based indexing and therefore exposes unsafe memory access when used with OffsetArrays:\n\njulia> using OffsetArrays\n\njulia> sum(OffsetArray([1, 2, 3], -10))\n9164911648 # inconsistent results or segfault\n\nWhile the original source of the error here is 1:length(A), the use of @inbounds increases the consequences from a bounds error to a less easily caught and debugged unsafe memory access. It is often difficult or impossible to prove that a method which uses @inbounds is safe, so one must weigh the benefits of performance improvements against the risk of segfaults and silent misbehavior, especially in public facing APIs."},{"title":"Propagating inbounds","page":"Bounds checking","location":"devdocs/boundscheck.html#Propagating-inbounds","category":"section","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).\n\nTo override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining."},{"title":"The bounds checking call hierarchy","page":"Bounds checking","location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","category":"section","text":"The overall hierarchy is:\n\ncheckbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension\n\nHere A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.\n\ncheckbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,\n\ncheckbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)\n\nso checkindex checks a single dimension. All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .\n\nIf you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.\n\nIf you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array. If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.\n\nNote this hierarchy has been designed to reduce the likelihood of method ambiguities. We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument)."},{"title":"Emit bounds checks","page":"Bounds checking","location":"devdocs/boundscheck.html#Emit-bounds-checks","category":"section","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations."},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Using-AI-agents-to-work-on-Julia","category":"section","text":"![WARNING] You are responsible for the code you submit in PRs. Do not submit PRs containing AI-generated code that you do not understand or that does not meet the ordinary quality bar for PRs to julia.\n\nThis page documents best practices for setting up AI agents to work with Julia. If you find additional prompt instructions that work well for common tasks, consider submitting a PR to add these to AGENTS.md."},{"title":"Google Jules","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Google-Jules","category":"section","text":"Use the following for your Initial Setup configuration.\n\ncurl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\n. /home/swebot/.profile\n\nJules has access to the internet, so you can give it links to issues or additional documentation in your prompting."},{"title":"OpenAI Codex","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#OpenAI-Codex","category":"section","text":"Configure the following:\n\nSetup Script\n\napt update\napt install less\ncurl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\nsource /root/.bashrc\nmake -C /workspace/julia/doc alldeps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\"\nmake -C /workspace/julia/test install-revise-deps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\"\n\nEnvironment Variables\n\nJULIA_PKG_OFFLINE=true\n\nCodex does not have internet access after initial setup, so you cannot give it additional information as links - you will need to copy any relevant text into the prompt.\n\nNote that Codex rebuilds the environment after every invocation. This can add significant latency. Codex work best for well-defined tasks that can be solved in a single shot."},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-and-Rational-Numbers","category":"section","text":"Julia includes predefined types for both complex and rational numbers, and supports all the standard Mathematical Operations and Elementary Functions on them. Conversion and Promotion are defined so that operations on any combination of predefined numeric types, whether primitive or composite, behave as expected."},{"title":"Complex Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-Numbers","category":"section","text":"The global constant im is bound to the complex number i, representing the principal square root of -1. (Using mathematicians' i or engineers' j for this global constant was rejected since they are such popular index variable names.) Since Julia allows numeric literals to be juxtaposed with identifiers as coefficients, this binding suffices to provide convenient syntax for complex numbers, similar to the traditional mathematical notation:\n\njulia> 1+2im\n1 + 2im\n\nYou can perform all the standard arithmetic operations with complex numbers:\n\njulia> (1 + 2im)*(2 - 3im)\n8 + 1im\n\njulia> (1 + 2im)/(1 - 2im)\n-0.6 + 0.8im\n\njulia> (1 + 2im) + (1 - 2im)\n2 + 0im\n\njulia> (-3 + 2im) - (5 - 1im)\n-8 + 3im\n\njulia> (-1 + 2im)^2\n-3 - 4im\n\njulia> (-1 + 2im)^2.5\n2.729624464784009 - 6.9606644595719im\n\njulia> (-1 + 2im)^(1 + 1im)\n-0.27910381075826657 + 0.08708053414102428im\n\njulia> 3(2 - 5im)\n6 - 15im\n\njulia> 3(2 - 5im)^2\n-63 - 60im\n\njulia> 3(2 - 5im)^-1.0\n0.20689655172413793 + 0.5172413793103449im\n\nThe promotion mechanism ensures that combinations of operands of different types just work:\n\njulia> 2(1 - 1im)\n2 - 2im\n\njulia> (2 + 3im) - 1\n1 + 3im\n\njulia> (1 + 2im) + 0.5\n1.5 + 2.0im\n\njulia> (2 + 3im) - 0.5im\n2.0 + 2.5im\n\njulia> 0.75(1 + 2im)\n0.75 + 1.5im\n\njulia> (2 + 3im) / 2\n1.0 + 1.5im\n\njulia> (1 - 3im) / (2 + 2im)\n-0.5 - 1.0im\n\njulia> 2im^2\n-2 + 0im\n\njulia> 1 + 3/4im\n1.0 - 0.75im\n\nNote that 3/4im == 3/(4*im) == -(3/4*im), since a literal coefficient binds more tightly than division.\n\nStandard functions to manipulate complex values are provided:\n\njulia> z = 1 + 2im\n1 + 2im\n\njulia> real(1 + 2im) # real part of z\n1\n\njulia> imag(1 + 2im) # imaginary part of z\n2\n\njulia> conj(1 + 2im) # complex conjugate of z\n1 - 2im\n\njulia> abs(1 + 2im) # absolute value of z\n2.23606797749979\n\njulia> abs2(1 + 2im) # squared absolute value\n5\n\njulia> angle(1 + 2im) # phase angle in radians\n1.1071487177940904\n\nAs usual, the absolute value (abs) of a complex number is its distance from zero. abs2 gives the square of the absolute value, and is of particular use for complex numbers since it avoids taking a square root. angle returns the phase angle in radians (also known as the argument or arg function). The full gamut of other Elementary Functions is also defined for complex numbers:\n\njulia> sqrt(1im)\n0.7071067811865476 + 0.7071067811865475im\n\njulia> sqrt(1 + 2im)\n1.272019649514069 + 0.7861513777574233im\n\njulia> cos(1 + 2im)\n2.0327230070196656 - 3.0518977991517997im\n\njulia> exp(1 + 2im)\n-1.1312043837568135 + 2.4717266720048188im\n\njulia> sinh(1 + 2im)\n-0.4890562590412937 + 1.4031192506220405im\n\nNote that mathematical functions typically return real values when applied to real numbers and complex values when applied to complex numbers. For example, sqrt behaves differently when applied to -1 versus -1 + 0im even though -1 == -1 + 0im:\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> sqrt(-1 + 0im)\n0.0 + 1.0im\n\nThe literal numeric coefficient notation does not work when constructing a complex number from variables. Instead, the multiplication must be explicitly written out:\n\njulia> a = 1; b = 2; a + b*im\n1 + 2im\n\nHowever, this is not recommended. Instead, use the more efficient complex function to construct a complex value directly from its real and imaginary parts:\n\njulia> a = 1; b = 2; complex(a, b)\n1 + 2im\n\nThis construction avoids the multiplication and addition operations.\n\nInf and NaN propagate through complex numbers in the real and imaginary parts of a complex number as described in the Special floating-point values section:\n\njulia> 1 + Inf*im\n1.0 + Inf*im\n\njulia> 1 + NaN*im\n1.0 + NaN*im"},{"title":"Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Rational-Numbers","category":"section","text":"Julia has a rational number type to represent exact ratios of integers. Rationals are constructed using the // operator:\n\njulia> 2//3\n2//3\n\nIf the numerator and denominator of a rational have common factors, they are reduced to lowest terms such that the denominator is non-negative:\n\njulia> 6//9\n2//3\n\njulia> -4//8\n-1//2\n\njulia> 5//-15\n-1//3\n\njulia> -4//-12\n1//3\n\nThis normalized form for a ratio of integers is unique, so equality of rational values can be tested by checking for equality of the numerator and denominator. The standardized numerator and denominator of a rational value can be extracted using the numerator and denominator functions:\n\njulia> numerator(2//3)\n2\n\njulia> denominator(2//3)\n3\n\nDirect comparison of the numerator and denominator is generally not necessary, since the standard arithmetic and comparison operations are defined for rational values:\n\njulia> 2//3 == 6//9\ntrue\n\njulia> 2//3 == 9//27\nfalse\n\njulia> 3//7 < 1//2\ntrue\n\njulia> 3//4 > 2//3\ntrue\n\njulia> 2//4 + 1//6\n2//3\n\njulia> 5//12 - 1//4\n1//6\n\njulia> 5//8 * 3//12\n5//32\n\njulia> 6//5 / 10//7\n21//25\n\nRationals can easily be converted to floating-point numbers:\n\njulia> float(3//4)\n0.75\n\nConversion from rational to floating-point respects the following identity for any integral values of a and b, except when a==0 && b <= 0:\n\njulia> a = 1; b = 2;\n\njulia> isequal(float(a//b), a/b)\ntrue\n\njulia> a, b = 0, 0\n(0, 0)\n\njulia> float(a//b)\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]\n\njulia> a/b\nNaN\n\njulia> a, b = 0, -1\n(0, -1)\n\njulia> float(a//b), a/b\n(0.0, -0.0)\n\nConstructing infinite rational values is acceptable:\n\njulia> 5//0\n1//0\n\njulia> x = -3//0\n-1//0\n\njulia> typeof(x)\nRational{Int64}\n\nTrying to construct a NaN rational value, however, is invalid:\n\njulia> 0//0\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]\n\nAs usual, the promotion system makes interactions with other numeric types effortless:\n\njulia> 3//5 + 1\n8//5\n\njulia> 3//5 - 0.5\n0.09999999999999998\n\njulia> 2//7 * (1 + 2im)\n2//7 + 4//7*im\n\njulia> 2//7 * (1.5 + 2im)\n0.42857142857142855 + 0.5714285714285714im\n\njulia> 3//2 / (1 + 2im)\n3//10 - 3//5*im\n\njulia> 1//2 + 2im\n1//2 + 2//1*im\n\njulia> 1 + 2//3im\n1//1 - 2//3*im\n\njulia> 0.5 == 1//2\ntrue\n\njulia> 0.33 == 1//3\nfalse\n\njulia> 0.33 < 1//3\ntrue\n\njulia> 1//3 - 0.33\n0.0033333333333332993"},{"title":"Eval of Julia code","page":"Eval of Julia code","location":"devdocs/eval.html#Eval-of-Julia-code","category":"section","text":"One of the hardest parts about learning how the Julia Language runs code is learning how all of the pieces work together to execute a block of code.\n\nEach chunk of code typically makes a trip through many steps with potentially unfamiliar names, such as (in no particular order): flisp, AST, C++, LLVM, eval, typeinf, macroexpand, sysimg (or system image), bootstrapping, compile, parse, execute, JIT, interpret, box, unbox, intrinsic function, and primitive function, before turning into the desired result (hopefully).\n\nsidebar: Definitions\nREPL\nREPL stands for Read-Eval-Print Loop. It's just what we call the command line environment for short.\nAST\nAbstract Syntax Tree. The AST is the digital representation of the code structure. In this form the code has been tokenized for meaning so that it is more suitable for manipulation and execution.\n\n(Image: Diagram of the compiler flow)"},{"title":"Julia Execution","page":"Eval of Julia code","location":"devdocs/eval.html#Julia-Execution","category":"section","text":"The 10,000 foot view of the whole process is as follows:\n\nThe user starts julia.\nThe C function main() from cli/loader_exe.c gets called. This function processes the command line arguments, filling in the jl_options struct and setting the variable ARGS. It then initializes Julia (by calling julia_init in init.c, which may load a previously compiled sysimg). Finally, it passes off control to Julia by calling Base._start().\nWhen _start() takes over control, the subsequent sequence of commands depends on the command line arguments given. For example, if a filename was supplied, it will proceed to execute that file. Otherwise, it will start an interactive REPL.\nSkipping the details about how the REPL interacts with the user, let's just say the program ends up with a block of code that it wants to run.\nIf the block of code to run is in a file, jl_load(char *filename) gets invoked to load the file and parse it. Each fragment of code is then passed to eval to execute.\nEach fragment of code (or AST), is handed off to eval() to turn into results.\neval() takes each code fragment and tries to run it in jl_toplevel_eval_flex().\njl_toplevel_eval_flex() decides whether the code is a \"toplevel\" action (such as using or module), which would be invalid inside a function. If so, it passes off the code to the toplevel interpreter.\njl_toplevel_eval_flex() then expands the code to eliminate any macros and to \"lower\" the AST to make it simpler to execute.\njl_toplevel_eval_flex() then uses some simple heuristics to decide whether to JIT compile the  AST or to interpret it directly.\nThe bulk of the work to interpret code is handled by eval in interpreter.c.\nIf instead, the code is compiled, the bulk of the work is handled by codegen.cpp. Whenever a  Julia function is called for the first time with a given set of argument types, type inference  will be run on that function. This information is used by the codegen step to generate  faster code.\nEventually, the user quits the REPL, or the end of the program is reached, and the _start()  method returns.\nJust before exiting, main() calls jl_atexit_hook(exit_code).  This calls Base._atexit() (which calls any functions registered to atexit() inside  Julia). Then it calls jl_gc_run_all_finalizers().  Finally, it gracefully cleans up all libuv handles and waits for them to flush and close."},{"title":"Parsing","page":"Eval of Julia code","location":"devdocs/eval.html#dev-parsing","category":"section","text":"By default, Julia uses JuliaSyntax.jl to produce the AST. Historically, it used a small lisp program written in femtolisp, the source-code for which is distributed inside Julia in src/flisp. If the JULIA_USE_FLISP_PARSER environment variable is set to 1, the old parser will be used instead."},{"title":"Macro Expansion","page":"Eval of Julia code","location":"devdocs/eval.html#dev-macro-expansion","category":"section","text":"When eval() encounters a macro, it expands that AST node before attempting to evaluate the expression. Macro expansion involves a handoff from eval() (in Julia), to the parser function jl_macroexpand() (written in flisp) to the Julia macro itself (written in - what else - Julia) via fl_invoke_julia_macro(), and back.\n\nTypically, macro expansion is invoked as a first step during a call to Meta.lower()/Core._lower(), although it can also be invoked directly by a call to macroexpand()/jl_macroexpand()."},{"title":"Type Inference","page":"Eval of Julia code","location":"devdocs/eval.html#dev-type-inference","category":"section","text":"Type inference is implemented in Julia by typeinf() in compiler/typeinfer.jl. Type inference is the process of examining a Julia function and determining bounds for the types of each of its variables, as well as bounds on the type of the return value from the function. This enables many future optimizations, such as unboxing of known immutable values, and compile-time hoisting of various run-time operations such as computing field offsets and function pointers. Type inference may also include other steps such as constant propagation and inlining.\n\nsidebar: More Definitions\nJIT\nJust-In-Time Compilation The process of generating native-machine code into memory right when it is needed.\nLLVM\nLow-Level Virtual Machine (a compiler) The Julia JIT compiler is a program/library called libLLVM. Codegen in Julia refers both to the process of taking a Julia AST and turning it into LLVM instructions, and the process of LLVM optimizing that and turning it into native assembly instructions.\nC++\nThe programming language that LLVM is implemented in, which means that codegen is also implemented in this language. The rest of Julia's library is implemented in C, in part because its smaller feature set makes it more usable as a cross-language interface layer.\nbox\nThis term is used to describe the process of taking a value and allocating a wrapper around the data that is tracked by the garbage collector (gc) and is tagged with the object's type.\nunbox\nThe reverse of boxing a value. This operation enables more efficient manipulation of data when the type of that data is fully known at compile-time (through type inference).\ngeneric function\nA Julia function composed of multiple \"methods\" that are selected for dynamic dispatch based on the argument type-signature\nanonymous function or \"method\"\nA Julia function without a name and without type-dispatch capabilities\nprimitive function\nA f