var documenterSearchIndex = {"docs":
[{"location":"manual/distributed-computing.html#Multi-processing-and-Distributed-Computing","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"An implementation of distributed memory parallel computing is provided by module Distributed as part of the standard library shipped with Julia.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Most modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Julia's implementation of message passing is different from other environments such as MPI[1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Distributed programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Remote references come in two flavors: Future and RemoteChannel.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait on the returned Future, and you can obtain the full value of the result using fetch.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"On the other hand, RemoteChannel s are rewritable. For example, multiple processes can coordinate their processing by referencing the same remote Channel.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Each process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1. As a result, adding 2 or more processes is required to gain benefits from parallel processing methods like pmap. Adding a single process is beneficial if you just wish to do other things in the main process while a long computation is running on the worker.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Let's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU threads (logical cores) on the machine. Note that the -p argument implicitly loads module Distributed.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"$ julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.18526  1.50912\n 1.16296  1.60607","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The first argument to remotecall is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall is considered a low-level interface providing finer control. The second argument to remotecall is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Occasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> remotecall_fetch(r-> fetch(r)[1, 1], 2, r)\n0.18526337335308085","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"This fetches the array on worker 2 and returns the first value. Note, that fetch doesn't move any data in this case, since it's executed on the worker that owns the array. One can also write:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.10824216411304866","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Remember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"To make things easier, the symbol :any can be passed to @spawnat, which picks where to do the operation for you:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> r = @spawnat :any rand(2,2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat :any 1 .+ fetch(r)\nFuture(3, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.38854  1.9098\n 1.20939  1.57158","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Note that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch might be required to move r to the process doing the addition. In this case, @spawnat is smart enough to perform the computation on the process that owns r, so the fetch will be a no-op (no work is done).","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"(It is worth noting that @spawnat is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"An important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"@async is similar to @spawnat, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indices. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. As for v0.7 and beyond, the feeder tasks are able to share state via nextidx because they all run on the same process. Even if Tasks are scheduled cooperatively, locking may still be required in some contexts, as in asynchronous I/O. This means context switches only occur at well-defined points: in this case, when remotecall_fetch is called. This is the current state of implementation and it may change for future Julia versions, as it is intended to make it possible to run up to N Tasks on M Process, aka M:N Threading. Then a lock acquiring\\releasing model for nextidx will be needed, as it is not safe to let multiple processes read-write a resource at the same time.","category":"page"},{"location":"manual/distributed-computing.html#code-availability","page":"Multi-processing and Distributed Computing","title":"Code Availability and Loading Packages","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Your code must be available on any process that runs it. For example, type the following into the Julia prompt:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Array{Float64,2}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawnat :any rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))))\nStacktrace:\n[...]","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Process 1 knew about the function rand2, but process 2 did not.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Most commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"module DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In order to refer to MyType across all processes, DummyModule.jl needs to be loaded on every process.  Calling include(\"DummyModule.jl\") loads it only on a single process.  To load it on every process, use the @everywhere macro (starting Julia with julia -p 2):","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @everywhere include(\"DummyModule.jl\")\nloaded\n      From worker 3:    loaded\n      From worker 2:    loaded","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As usual, this does not bring DummyModule into scope on any of the process, which requires using or import.  Moreover, when DummyModule is brought into scope on one process, it is not on any other:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> using .DummyModule\n\njulia> MyType(7)\nMyType(7)\n\njulia> fetch(@spawnat 2 MyType(7))\nERROR: On worker 2:\nUndefVarError: `MyType` not defined in `Main`\n⋮\n\njulia> fetch(@spawnat 2 DummyModule.MyType(7))\nMyType(7)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"However, it's still possible, for instance, to send a MyType to a process which has loaded DummyModule even if it's not in scope:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> put!(RemoteChannel(2), MyType(7))\nRemoteChannel{Channel{Any}}(2, 1, 13)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A file can also be preloaded on multiple processes at startup with the -L flag, and a driver script can be used to drive the computation:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia -p <n> -L file1.jl -L file2.jl driver.jl","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Finally, if DummyModule.jl is not a standalone file but a package, then using DummyModule will load DummyModule.jl on all processes, but only bring it into scope on the process where using was called.","category":"page"},{"location":"manual/distributed-computing.html#Starting-and-managing-worker-processes","page":"Multi-processing and Distributed Computing","title":"Starting and managing worker processes","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The base Julia installation has in-built support for two types of clusters:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machine-file option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers should use to connect to this worker.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"note: Note\nWhile Julia generally strives for backward compatibility, distribution of code to worker processes relies on Serialization.serialize. As pointed out in the corresponding documentation, this can not be guaranteed to work across different Julia versions, so it is advised that all workers on all machines use the same version.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Functions addprocs, rmprocs, workers, and others are available as a programmatic means of adding, removing and querying the processes in a cluster.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> using Distributed\n\njulia> addprocs(2)\n2-element Array{Int64,1}:\n 2\n 3","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Module Distributed must be explicitly loaded on the master process before invoking addprocs. It is automatically made available on the worker processes.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"note: Note\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes. You may use addprocs(exeflags=\"--project\") to initialize a worker with a particular environment, and then @everywhere using <modulename> or @everywhere include(\"file.jl\").","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Other types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section.","category":"page"},{"location":"manual/distributed-computing.html#Data-Movement","page":"Multi-processing and Distributed Computing","title":"Data Movement","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Sending messages and moving data constitute most of the overhead in a distributed program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various distributed programming constructs.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"fetch can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawnat (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Method 1:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> A = rand(1000,1000);\n\njulia> Bref = @spawnat :any A^2;\n\n[...]\n\njulia> fetch(Bref);","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Method 2:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> Bref = @spawnat :any rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The difference seems trivial, but in fact is quite significant due to the behavior of @spawnat. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawnat and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawnat statement just for this step.","category":"page"},{"location":"manual/distributed-computing.html#Global-variables","page":"Multi-processing and Distributed Computing","title":"Global variables","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Expressions executed remotely via @spawnat, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A = rand(10,10)\nremotecall_fetch(()->sum(A), 2)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In this case sum MUST be defined in the remote process. Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->sum(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"New global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Thus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"For example:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @fetchfrom 2 InteractiveUtils.varinfo()\nname           size summary\n––––––––– ––––––––– ––––––––––––––––––––––\nA         800 bytes 10×10 Array{Float64,2}\nBase                Module\nCore                Module\nMain                Module","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2.","category":"page"},{"location":"manual/distributed-computing.html#Parallel-Map-and-Loops","page":"Multi-processing and Distributed Computing","title":"Parallel Map and Loops","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawnat to flip coins on two processes. First, write the following function in count_heads.jl:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"function count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @everywhere include_string(Main, $(read(\"count_heads.jl\", String)), \"count_heads.jl\")\n\njulia> a = @spawnat :any count_heads(100000000)\nFuture(2, 1, 6, nothing)\n\njulia> b = @spawnat :any count_heads(100000000)\nFuture(3, 1, 7, nothing)\n\njulia> fetch(a)+fetch(b)\n100001564","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"This example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Notice that our use of this pattern with count_heads can be generalized. We used two explicit @spawnat statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, running in distributed memory, which can be written in Julia using @distributed like this:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"nheads = @distributed (+) for i = 1:200000000\n    Int(rand(Bool))\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"This construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Note that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"For example, the following code will not work as intended:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"a = zeros(100000)\n@distributed for i = 1:100000\n    a[i] = i\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"This code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"using SharedArrays\n\na = SharedArray{Float64}(10)\n@distributed for i = 1:10\n    a[i] = i\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Using \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"a = randn(1000)\n@distributed (+) for i = 1:100000\n    f(a[rand(1:end)])\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Here each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @distributed for.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap function. For example, we could compute the singular values of several large random matrices in parallel as follows:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svdvals, M);","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Julia's pmap is designed for the case where each function call does a large amount of work. In contrast, @distributed for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap and @distributed for for the parallel computation. In case of @distributed for, the final reduction is done on the calling process.","category":"page"},{"location":"manual/distributed-computing.html#Remote-References-and-AbstractChannels","page":"Multi-processing and Distributed Computing","title":"Remote References and AbstractChannels","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Remote references always refer to an implementation of an AbstractChannel.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A concrete implementation of an AbstractChannel (like Channel), is required to implement put!, take!, fetch, isready and wait. The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"RemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f is a function executed on pid and it must return an AbstractChannel.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"For example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Methods put!, take!, fetch, isready and wait on a RemoteChannel are proxied onto the backing store on the remote process.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"RemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is the following DictChannel which uses a dictionary as its remote store:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> struct DictChannel{T} <: AbstractChannel{T}\n           d::Dict\n           cond_take::Threads.Condition    # waiting for data to become available\n           DictChannel{T}() where {T} = new(Dict(), Threads.Condition())\n           DictChannel() = DictChannel{Any}()\n       end\n\njulia> begin\n       function Base.put!(D::DictChannel, k, v)\n           @lock D.cond_take begin\n               D.d[k] = v\n               notify(D.cond_take)\n           end\n           return D\n       end\n       function Base.take!(D::DictChannel, k)\n           @lock D.cond_take begin\n               v = fetch(D, k)\n               delete!(D.d, k)\n               return v\n           end\n       end\n       Base.isready(D::DictChannel) = @lock D.cond_take !isempty(D.d)\n       Base.isready(D::DictChannel, k) = @lock D.cond_take haskey(D.d, k)\n       function Base.fetch(D::DictChannel, k)\n           @lock D.cond_take begin\n               wait(D, k)\n               return D.d[k]\n           end\n       end\n       function Base.wait(D::DictChannel, k)\n           @lock D.cond_take begin\n               while !isready(D, k)\n                   wait(D.cond_take)\n               end\n           end\n       end\n       end;\n\njulia> d = DictChannel();\n\njulia> isready(d)\nfalse\n\njulia> put!(d, :k, :v);\n\njulia> isready(d, :k)\ntrue\n\njulia> fetch(d, :k)\n:v\n\njulia> wait(d, :k)\n\njulia> take!(d, :k)\n:v\n\njulia> isready(d, :k)\nfalse","category":"page"},{"location":"manual/distributed-computing.html#Channels-and-RemoteChannels","page":"Multi-processing and Distributed Computing","title":"Channels and RemoteChannels","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The channels example from above can be modified for interprocess communication, as shown below.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"We start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(@async make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, where = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds on worker $where\")\n           global n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741","category":"page"},{"location":"manual/distributed-computing.html#Remote-References-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Remote References and Distributed Garbage Collection","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Objects referred to by remote references can be freed only when all held references in the cluster are deleted.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified. This is implemented in an internal cluster aware serializer. Remote references are only valid in the context of a running cluster. Serializing and deserializing references to and from regular IO objects is not supported.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Since Futures are write-once and cached locally, the act of fetching a Future also updates reference tracking information on the node owning the value.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The node which owns the value frees it once all references to it are cleared.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"With Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"It is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize results in an immediate message sent to the remote node to go ahead and remove its reference to the value.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Once finalized, a reference becomes invalid and cannot be used in any further calls.","category":"page"},{"location":"manual/distributed-computing.html#Local-invocations","page":"Multi-processing and Distributed Computing","title":"Local invocations","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Data is necessarily copied over to the remote node for execution. This is the case for both remotecalls and when data is stored to a RemoteChannel / Future on a different node. As expected, this results in a copy of the serialized objects on the remote node. However, when the destination node is the local node, i.e. the calling process id is the same as the remote node id, it is executed as a local call. It is usually (not always) executed in a different task - but there is no serialization/deserialization of data. Consequently, the call refers to the same object instances as passed - no copies are created. This behavior is highlighted below:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> using Distributed;\n\njulia> rc = RemoteChannel(()->Channel(3));   # RemoteChannel created on local node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i                          # Reusing `v`\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\nArray{Int64,1}[[3], [3], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 1\n\njulia> addprocs(1);\n\njulia> rc = RemoteChannel(()->Channel(3), workers()[1]);   # RemoteChannel created on remote node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\nArray{Int64,1}[[1], [2], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 3","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As can be seen, put! on a locally owned RemoteChannel with the same object v modified between calls results in the same single object instance stored. As opposed to copies of v being created when the node owning rc is a different node.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"It is to be noted that this is generally not an issue. It is something to be factored in only if the object is both being stored locally and modified post the call. In such cases it may be appropriate to store a deepcopy of the object.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"This is also true for remotecalls on the local node as seen in the following example:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> using Distributed; addprocs(1);\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), myid(), v);     # Executed on local node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[1], v2=[1], true\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), workers()[1], v); # Executed on remote node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[0], v2=[1], false","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As can be seen once again, a remote call onto the local node behaves just like a direct invocation. The call modifies local objects passed as arguments. In the remote invocation, it operates on a copy of the arguments.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"To repeat, in general this is not an issue. If the local node is also being used as a compute node, and the arguments used post the call, this behavior needs to be factored in and if required deep copies of arguments must be passed to the call invoked on the local node. Calls on remote nodes will always operate on copies of arguments.","category":"page"},{"location":"manual/distributed-computing.html#man-shared-arrays","page":"Multi-processing and Distributed Computing","title":"Shared Arrays","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Shared Arrays use system shared memory to map the same array across many processes. A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine. Shared Array support is available via the module SharedArrays, which must be explicitly loaded on all participating workers.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A complementary data structure is provided by the external package DistributedArrays.jl in the form of a DArray. While there are some similarities to a SharedArray, the behavior of a DArray is quite different. In a SharedArray, each \"participating\" process has access to the entire array; in contrast, in a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"SharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata. For other AbstractArray types, sdata just returns the object itself, so it's safe to use sdata on any Array-type object.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The constructor for a shared array is of the form:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"SharedArray{T,N}(dims::NTuple; init=false, pids=Int[])","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"which creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host). Note that only elements that are isbits are supported in a SharedArray.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"If an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Here's a brief example:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> using Distributed\n\njulia> addprocs(3)\n3-element Array{Int64,1}:\n 2\n 3\n 4\n\njulia> @everywhere using SharedArrays\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[localindices(S)] = repeat([myid()], length(localindices(S))))\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"SharedArrays.localindices provides disjoint one-dimensional ranges of indices, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = repeat([myid()], length( indexpids(S):length(procs(S)):length(S))))\n3×4 SharedArray{Int64,2}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Since all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"@sync begin\n    for p in procs(S)\n        @async begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"would result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As a more extended and complex example, consider running the following \"kernel\" in parallel:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"q[i,j,t+1] = q[i,j,t] + u[i,j,t]","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indices assigned to this worker:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in range(0, stop=size(q,2), length=nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       end","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Next, define the kernel:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       end","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"We also define a convenience wrapper for a SharedArray implementation","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Now let's compare three different versions, one that runs in a single process:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"one that uses @distributed:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @distributed for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"and one that delegates in chunks:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   @async remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"If we create SharedArrays and time these functions, we get the following results (with julia -p 4):","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Run the functions once to JIT-compile and @time them on the second run:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece.","category":"page"},{"location":"manual/distributed-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Shared Arrays and Distributed Garbage Collection","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner.","category":"page"},{"location":"manual/distributed-computing.html#ClusterManagers","page":"Multi-processing and Distributed Computing","title":"ClusterManagers","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible for","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"launching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transport","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A Julia cluster has the following characteristics:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Connections between workers (using the in-built TCP/IP transport) is established in the following manner:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"addprocs is called on the master process with a ClusterManager object.\naddprocs calls the appropriate launch method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to stdout.\nThe cluster manager captures the stdout of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"While the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Julia provides two in-built cluster managers:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"LocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnames","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"LocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Thus, a minimal cluster manager would need to:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"be a subtype of the abstract ClusterManager\nimplement launch, a method responsible for launching new workers\nimplement manage, which is called at various events during a worker's lifetime (for example, sending an interrupt signal)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"addprocs(manager::FooManager) requires FooManager to implement:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"function launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"struct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The launch method takes the following arguments:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"manager::ClusterManager: the cluster manager that addprocs is called with\nparams::Dict: all the keyword arguments passed to addprocs\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launched","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The launch method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch function MUST exit as soon as all the requested workers have been launched.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Newly launched workers are connected to each other and the master process in an all-to-all manner. Specifying the command line argument --worker[=<cookie>] results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"All workers in a cluster share the same cookie as the master. When the cookie is unspecified, i.e, with the --worker option, the worker tries to read it from its standard input.  LocalManager and SSHManager both pass the cookie to newly launched workers via their  standard inputs.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"By default a worker will listen on a free port at the address returned by a call to getipaddr(). A specific address to listen on may be specified by optional argument --bind-to bind_addr[:port]. This is useful for multi-homed hosts.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"As an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"For every worker launched, the launch method must add a WorkerConfig object (with appropriate fields initialized) to launched","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"mutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Union{IO, Nothing}\n    host::Union{AbstractString, Nothing}\n    port::Union{Integer, Nothing}\n\n    # Used when launching additional workers at a host\n    count::Union{Int, Symbol, Nothing}\n    exename::Union{AbstractString, Cmd, Nothing}\n    exeflags::Union{Cmd, Nothing}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Any\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Union{Bool, Nothing}\n    bind_addr::Union{AbstractString, Nothing}\n    sshflags::Union{Cmd, Nothing}\n    max_parallel::Union{Integer, Nothing}\n\n    # Used by Local/SSH managers\n    connect_at::Any\n\n    [...]\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Most of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"If io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of CPU threads (logical cores) on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"with :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes.","category":"page"},{"location":"manual/distributed-computing.html#Cluster-Managers-with-Custom-Transports","page":"Multi-processing and Distributed Computing","title":"Cluster Managers with Custom Transports","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Each Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Replacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"connect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"connect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The folder clustermanager/0mq in the Examples repository contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"When using custom transports:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Julia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) must be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch is called. The value of this field is passed in all connect callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"kill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The Examples folder clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup.","category":"page"},{"location":"manual/distributed-computing.html#Network-Requirements-for-LocalManager-and-SSHManager","page":"Multi-processing and Distributed Computing","title":"Network Requirements for LocalManager and SSHManager","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on an ephemeral port number assigned by the OS.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-i <keyfile>`.\nIn an all-to-all topology (the default), all workers connect to each other via plain TCP sockets. The security policy on the cluster nodes must thus ensure free connectivity between workers for the ephemeral port range (varies by OS).\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager.\nIf you specify multiplex=true as an option to addprocs, SSH multiplexing is used to create a tunnel between the master and workers. If you have configured SSH multiplexing on your own and the connection has already been established, SSH multiplexing is used regardless of multiplex option. If multiplexing is enabled, forwarding is set by using the existing connection (-O forward option in ssh). This is beneficial if your servers require password authentication; you can avoid authentication in Julia by logging in to the server ahead of addprocs. The control socket will be located at ~/.ssh/julia-%r@%h:%p during the session unless the existing multiplexing connection is used. Note that bandwidth may be limited if you create multiple processes on a node and enable multiplexing, because in that case processes share a single multiplexing TCP connection.","category":"page"},{"location":"manual/distributed-computing.html#man-cluster-cookie","page":"Multi-processing and Distributed Computing","title":"Cluster Cookie","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"cluster_cookie() returns the cookie, while cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie may be passed to the workers at startup via argument --worker=<cookie>. If argument --worker is specified without the cookie, the worker tries to read the cookie from its standard input (stdin). The stdin is closed immediately after the cookie is retrieved.\nClusterManagers can retrieve the cookie on the master by calling cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Note that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument.","category":"page"},{"location":"manual/distributed-computing.html#Specifying-Network-Topology-(Experimental)","page":"Multi-processing and Distributed Computing","title":"Specifying Network Topology (Experimental)","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":":all_to_all, the default: all workers are connected to each other.\n:master_worker: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Keyword argument lazy=true|false only affects topology option :all_to_all. If true, the cluster starts off with the master connected to all workers. Specific worker-worker connections are established at the first remote invocation between two workers. This helps in reducing initial resources allocated for intra-cluster communication. Connections are setup depending on the runtime requirements of a parallel program. Default value for lazy is true.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Currently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases.","category":"page"},{"location":"manual/distributed-computing.html#Noteworthy-external-packages","page":"Multi-processing and Distributed Computing","title":"Noteworthy external packages","text":"","category":"section"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Outside of Julia parallelism there are plenty of external packages that should be mentioned. For example, MPI.jl is a Julia wrapper for the MPI protocol, Dagger.jl provides functionality similar to Python's Dask, and DistributedArrays.jl provides array operations distributed across workers, as outlined above.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"A mention must be made of Julia's GPU programming ecosystem, which includes:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"CUDA.jl wraps the various CUDA libraries and supports compiling Julia kernels for Nvidia GPUs.\noneAPI.jl wraps the oneAPI unified programming model, and supports executing Julia kernels on supported accelerators. Currently only Linux is supported.\nAMDGPU.jl wraps the AMD ROCm libraries and supports compiling Julia kernels for AMD GPUs. Currently only Linux is supported.\nHigh-level libraries like KernelAbstractions.jl, Tullio.jl and ArrayFire.jl.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes by first casting it through distribute() and CuArray().","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Remember when importing DistributedArrays.jl to import it across all processes using @everywhere","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"$ ./julia -p 4\n\njulia> addprocs()\n\njulia> @everywhere using DistributedArrays\n\njulia> using CUDA\n\njulia> B = ones(10_000) ./ 2;\n\njulia> A = ones(10_000) .* π;\n\njulia> C = 2 .* A ./ B;\n\njulia> all(C .≈ 4*π)\ntrue\n\njulia> typeof(C)\nArray{Float64,1}\n\njulia> dB = distribute(B);\n\njulia> dA = distribute(A);\n\njulia> dC = 2 .* dA ./ dB;\n\njulia> all(dC .≈ 4*π)\ntrue\n\njulia> typeof(dC)\nDistributedArrays.DArray{Float64,1,Array{Float64,1}}\n\njulia> cuB = CuArray(B);\n\njulia> cuA = CuArray(A);\n\njulia> cuC = 2 .* cuA ./ cuB;\n\njulia> all(cuC .≈ 4*π);\ntrue\n\njulia> typeof(cuC)\nCuArray{Float64,1}","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"In the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes and call a generic function on it.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"function power_method(M, v)\n    for i in 1:100\n        v = M*v\n        v /= norm(v)\n    end\n\n    return v, norm(M*v) / norm(v)  # or  (M*v) ./ v\nend","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"power_method repeatedly creates a new vector and normalizes it. We have not specified any type signature in function declaration, let's see if it works with the aforementioned datatypes:","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"julia> M = [2. 1; 1 1];\n\njulia> v = rand(2)\n2-element Array{Float64,1}:\n0.40395\n0.445877\n\njulia> power_method(M,v)\n([0.850651, 0.525731], 2.618033988749895)\n\njulia> cuM = CuArray(M);\n\njulia> cuv = CuArray(v);\n\njulia> curesult = power_method(cuM, cuv);\n\njulia> typeof(curesult)\nCuArray{Float64,1}\n\njulia> dM = distribute(M);\n\njulia> dv = distribute(v);\n\njulia> dC = power_method(dM, dv);\n\njulia> typeof(dC)\nTuple{DistributedArrays.DArray{Float64,1,Array{Float64,1}},Float64}","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"To end this short exposure to external packages, we can consider MPI.jl, a Julia wrapper of the MPI protocol. As it would take too long to consider every inner function, it would be better to simply appreciate the approach used to implement the protocol.","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"Consider this toy script which simply calls each subprocess, instantiate its rank and when the master process is reached, performs the ranks' sum","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"import MPI\n\nMPI.Init()\n\ncomm = MPI.COMM_WORLD\nMPI.Barrier(comm)\n\nroot = 0\nr = MPI.Comm_rank(comm)\n\nsr = MPI.Reduce(r, MPI.SUM, root, comm)\n\nif(MPI.Comm_rank(comm) == root)\n   @printf(\"sum of ranks: %s\\n\", sr)\nend\n\nMPI.Finalize()","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"mpirun -np 4 ./julia example.jl","category":"page"},{"location":"manual/distributed-computing.html","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","text":"[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding rma to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see https://mpi-forum.org/docs.","category":"page"},{"location":"manual/modules.html#modules","page":"Modules","title":"Modules","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Modules in Julia help organize code into coherent units. They are delimited syntactically inside module NameOfModule ... end, and have the following features:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Modules are separate namespaces, each introducing a new global scope. This is useful, because it allows the same name to be used for different functions or global variables without conflict, as long as they are in separate modules.\nModules have facilities for detailed namespace management: each defines a set of names it exports and marks as public, and can import names from other modules with using and import (we explain these below).\nModules can be precompiled for faster loading, and may contain code for runtime initialization.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Typically, in larger Julia packages you will see module code organized into files, eg","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"module SomeModule\n\n# export, public, using, import statements are usually here; we discuss these below\n\ninclude(\"file1.jl\")\ninclude(\"file2.jl\")\n\nend","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Files and file names are mostly unrelated to modules; modules are associated only with module expressions. One can have multiple files per module, and multiple modules per file. include behaves as if the contents of the source file were evaluated in the global scope of the including module. In this chapter, we use short and simplified examples, so we won't use include.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"The recommended style is not to indent the body of the module, since that would typically lead to whole files being indented. Also, it is common to use UpperCamelCase for module names (just like types), and use the plural form if applicable, especially if the module contains a similarly named identifier, to avoid name clashes. For example,","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"module FastThings\n\nstruct FastThing\n    ...\nend\n\nend","category":"page"},{"location":"manual/modules.html#namespace-management","page":"Modules","title":"Namespace management","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Namespace management refers to the facilities the language offers for making names in a module available in other modules. We discuss the related concepts and functionality below in detail.","category":"page"},{"location":"manual/modules.html#Qualified-names","page":"Modules","title":"Qualified names","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Names for functions, variables and types in the global scope like sin, ARGS, and UnitRange always belong to a module, called the parent module, which can be found interactively with parentmodule, for example","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> parentmodule(UnitRange)\nBase","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"One can also refer to these names outside their parent module by prefixing them with their module, eg Base.UnitRange. This is called a qualified name. The parent module may be accessible using a chain of submodules like Base.Math.sin, where Base.Math is called the module path. Due to syntactic ambiguities, qualifying a name that contains only symbols, such as an operator, requires inserting a colon, e.g. Base.:+. A small number of operators additionally require parentheses, e.g. Base.:(==).","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"If a name is qualified, then it is always accessible, and in case of a function, it can also have methods added to it by using the qualified name as the function name.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Within a module, a variable name can be “reserved” without assigning to it by declaring it as global x. This prevents name conflicts for globals initialized after load time. The syntax M.x = y does not work to assign a global in another module; global assignment is always module-local.","category":"page"},{"location":"manual/modules.html#Export-lists","page":"Modules","title":"Export lists","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Names (referring to functions, types, global variables, and constants) can be added to the export list of a module with export: these are the symbols that are imported when using the module. Typically, they are at or near the top of the module definition so that readers of the source code can find them easily, as in","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> module NiceStuff\n       export nice, DOG\n       struct Dog end      # singleton type, not exported\n       const DOG = Dog()   # named instance, exported\n       nice(x) = \"nice $x\" # function, exported\n       end;\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"but this is just a style suggestion — a module can have multiple export statements in arbitrary locations.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"It is common to export names which form part of the API (application programming interface). In the above code, the export list suggests that users should use nice and DOG. However, since qualified names always make identifiers accessible, this is just an option for organizing APIs: unlike other languages, Julia has no facilities for truly hiding module internals.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Also, some modules don't export names at all. This is usually done if they use common words, such as derivative, in their API, which could easily clash with the export lists of other modules. We will see how to manage name clashes below.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"To mark a name as public without exporting it into the namespace of folks who call using NiceStuff, one can use public instead of export. This marks the public name(s) as part of the public API, but does not have any namespace implications. The public keyword is only available in Julia 1.11 and above. To maintain compatibility with Julia 1.10 and below, use the @compat macro from the Compat package.","category":"page"},{"location":"manual/modules.html#Standalone-using-and-import","page":"Modules","title":"Standalone using and import","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Possibly the most common way of loading a module is using ModuleName. This loads the code associated with ModuleName, and brings","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"the module name\nand the elements of the export list into the surrounding global namespace.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Technically, the statement using ModuleName means that a module called ModuleName will be available for resolving names as needed. When a global variable is encountered that has no definition in the current module, the system will search for it among variables exported by ModuleName and use it if it is found there. This means that all uses of that global within the current module will resolve to the definition of that variable in ModuleName.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"To load a module from a package, the statement using ModuleName can be used. To load a module from a locally defined module, a dot needs to be added before the module name like using .ModuleName.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"To continue with our example,","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"would load the above code, making NiceStuff (the module name), DOG and nice available. Dog is not on the export list, but it can be accessed if the name is qualified with the module path (which here is just the module name) as NiceStuff.Dog.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Importantly, using ModuleName is the only form for which export lists matter at all.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"In contrast,","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> import .NiceStuff","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"brings only the module name into scope. Users would need to use NiceStuff.DOG, NiceStuff.Dog, and NiceStuff.nice to access its contents. Usually, import ModuleName is used in contexts when the user wants to keep the namespace clean. As we will see in the next section import .NiceStuff is equivalent to using .NiceStuff: NiceStuff.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"You can combine multiple using and import statements of the same kind in a comma-separated expression, e.g.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using LinearAlgebra, Random","category":"page"},{"location":"manual/modules.html#using-and-import-with-specific-identifiers,-and-adding-methods","page":"Modules","title":"using and import with specific identifiers, and adding methods","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"When using ModuleName: or import ModuleName: is followed by a comma-separated list of names, the module is loaded, but only those specific names are brought into the namespace by the statement. For example,","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff: nice, DOG","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"will import the names nice and DOG.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Importantly, the module name NiceStuff will not be in the namespace. If you want to make it accessible, you have to list it explicitly, as","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff: nice, DOG, NiceStuff","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Julia has two forms for seemingly the same thing because only import ModuleName: f allows adding methods to f without a module path. That is to say, the following example will give an error:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nERROR: invalid method definition in Main: function NiceStuff.nice must be explicitly imported to be extended\nStacktrace:\n [1] top-level scope\n   @ none:0\n [2] top-level scope\n   @ none:1\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"This error prevents accidentally adding methods to functions in other modules that you only intended to use.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"There are two ways to deal with this. You can always qualify function names with a module path:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff\n\njulia> struct Cat end\n\njulia> NiceStuff.nice(::Cat) = \"nice 😸\"\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Alternatively, you can import the specific function name:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> import .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nnice (generic function with 2 methods)","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Which one you choose is a matter of style. The first form makes it clear that you are adding a method to a function in another module (remember, that the imports and the method definition may be in separate files), while the second one is shorter, which is especially convenient if you are defining multiple methods.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Once a variable is made visible via using or import, a module may not create its own variable with the same name. Imported variables are read-only; assigning to a global variable always affects a variable owned by the current module, or else raises an error.","category":"page"},{"location":"manual/modules.html#Renaming-with-as","page":"Modules","title":"Renaming with as","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"An identifier brought into scope by import or using can be renamed with the keyword as. This is useful for working around name conflicts as well as for shortening names. For example, Base exports the function name read, but the CSV.jl package also provides CSV.read. If we are going to invoke CSV reading many times, it would be convenient to drop the CSV. qualifier. But then it is ambiguous whether we are referring to Base.read or CSV.read:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> read;\n\njulia> import CSV: read\nWARNING: ignoring conflicting import of CSV.read into Main","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Renaming provides a solution:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> import CSV: read as rd","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Imported packages themselves can also be renamed:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"import BenchmarkTools as BT","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"as works with using only when a single identifier is brought into scope. For example using CSV: read as rd works, but using CSV as C does not, since it operates on all of the exported names in CSV.","category":"page"},{"location":"manual/modules.html#Mixing-multiple-using-and-import-statements","page":"Modules","title":"Mixing multiple using and import statements","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"When multiple using or import statements of any of the forms above are used, their effect is combined in the order they appear. For example,","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .NiceStuff         # exported names and the module name\n\njulia> import .NiceStuff: nice  # allows adding methods to unqualified functions\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"would bring all the exported names of NiceStuff and the module name itself into scope, and also allow adding methods to nice without prefixing it with a module name.","category":"page"},{"location":"manual/modules.html#Handling-name-conflicts","page":"Modules","title":"Handling name conflicts","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Consider the situation where two (or more) packages export the same name, as in","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> module A\n       export f\n       f() = 1\n       end\nA\njulia> module B\n       export f\n       f() = 2\n       end\nB","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"The statement using .A, .B works, but when you try to call f, you get a warning","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> using .A, .B\n\njulia> f\nWARNING: both B and A export \"f\"; uses of it in module Main must be qualified\nERROR: UndefVarError: `f` not defined in `Main`","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Here, Julia cannot decide which f you are referring to, so you have to make a choice. The following solutions are commonly used:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Simply proceed with qualified names like A.f and B.f. This makes the context clear to the reader of your code, especially if f just happens to coincide but has different meaning in various packages. For example, degree has various uses in mathematics, the natural sciences, and in everyday life, and these meanings should be kept separate.\nUse the as keyword above to rename one or both identifiers, eg\njulia> using .A: f as f\n\njulia> using .B: f as g\n\nwould make B.f available as g. Here, we are assuming that you did not use using A before, which would have brought f into the namespace.\nWhen the names in question do share a meaning, it is common for one module to import it from another, or have a lightweight “base” package with the sole function of defining an interface like this, which can be used by other packages. It is conventional to have such package names end in ...Base (which has nothing to do with Julia's Base module).","category":"page"},{"location":"manual/modules.html#Default-top-level-definitions-and-bare-modules","page":"Modules","title":"Default top-level definitions and bare modules","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Modules automatically contain using Core, using Base, and definitions of the eval and include functions, which evaluate expressions/files within the global scope of that module.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"If these default definitions are not wanted, modules can be defined using the keyword baremodule instead (note: Core is still imported). In terms of baremodule, a standard module looks like this:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"baremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"If even Core is not wanted, a module that imports nothing and defines no names at all can be defined with Module(:YourNameHere, false, false) and code can be evaluated into it with @eval or Core.eval:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> arithmetic = Module(:arithmetic, false, false)\nMain.arithmetic\n\njulia> @eval arithmetic add(x, y) = $(+)(x, y)\nadd (generic function with 1 method)\n\njulia> arithmetic.add(12, 13)\n25","category":"page"},{"location":"manual/modules.html#Standard-modules","page":"Modules","title":"Standard modules","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"There are three important standard modules:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Core contains all functionality \"built into\" the language.\nBase contains basic functionality that is useful in almost all cases.\nMain is the top-level module and the current module, when Julia is started.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"note: Standard library modules\nBy default Julia ships with some standard library modules. These behave like regular Julia packages except that you don't need to install them explicitly. For example, if you wanted to perform some unit testing, you could load the Test standard library as follows:using Test","category":"page"},{"location":"manual/modules.html#Submodules-and-relative-paths","page":"Modules","title":"Submodules and relative paths","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Modules can contain submodules, nesting the same syntax module ... end. They can be used to introduce separate namespaces, which can be helpful for organizing complex codebases. Note that each module introduces its own scope, so submodules do not automatically “inherit” names from their parent.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"It is recommended that submodules refer to other modules within the enclosing parent module (including the latter) using relative module qualifiers in using and import statements. A relative module qualifier starts with a period (.), which corresponds to the current module, and each successive . leads to the parent of the current module. This should be followed by modules if necessary, and eventually the actual name to access, all separated by .s.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Consider the following example, where the submodule SubA defines a function, which is then extended in its “sibling” module:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> module ParentModule\n       module SubA\n       export add_D  # exported interface\n       const D = 3\n       add_D(x) = x + D\n       end\n       using .SubA  # brings `add_D` into the namespace\n       export add_D # export it from ParentModule too\n       module SubB\n       import ..SubA: add_D # relative path for a “sibling” module\n       struct Infinity end\n       add_D(x::Infinity) = x\n       end\n       end;\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"You may see code in packages, which, in a similar situation, uses","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"julia> import .ParentModule.SubA: add_D\n","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"However, this operates through code loading, and thus only works if ParentModule is in a package. It is better to use relative paths.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Note that the order of definitions also matters if you are evaluating values. Consider","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"module TestPackage\n\nexport x, y\n\nx = 0\n\nmodule Sub\nusing ..TestPackage\nz = y # ERROR: UndefVarError: `y` not defined in `Main`\nend\n\ny = 1\n\nend","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"where Sub is trying to use TestPackage.y before it was defined, so it does not have a value.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"For similar reasons, you cannot use a cyclic ordering:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"module A\n\nmodule B\nusing ..C # ERROR: UndefVarError: `C` not defined in `Main.A`\nend\n\nmodule C\nusing ..B\nend\n\nend","category":"page"},{"location":"manual/modules.html#Module-initialization-and-precompilation","page":"Modules","title":"Module initialization and precompilation","text":"","category":"section"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Large modules can take several seconds to load because executing all of the statements in a module often involves compiling a large amount of code. Julia creates precompiled caches of the module to reduce this time.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Precompiled module files (sometimes called \"cache files\") are created and used automatically when import or using loads a module.  If the cache file(s) do not yet exist, the module will be compiled and saved for future reuse. You can also manually call Base.compilecache(Base.identify_package(\"modulename\")) to create these files without loading the module. The resulting cache files will be stored in the compiled subfolder of DEPOT_PATH[1]. If nothing about your system changes, such cache files will be used when you load the module with import or using.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Precompilation cache files store definitions of modules, types, methods, and constants. They may also store method specializations and the code generated for them, but this typically requires that the developer add explicit precompile directives or execute workloads that force compilation during the package build.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"However, if you update the module's dependencies or change its source code, the module is automatically recompiled upon using or import. Dependencies are modules it imports, the Julia build, files it includes, or explicit dependencies declared by include_dependency(path) in the module file(s).","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"For file dependencies loaded by include, a change is determined by examining whether the file size (fsize) or content (condensed into a hash) is unchanged. For file dependencies loaded by include_dependency a change is determined by examining whether the modification time (mtime) is unchanged, or equal to the modification time truncated to the nearest second (to accommodate systems that can't copy mtime with sub-second accuracy). It also takes into account whether the path to the file chosen by the search logic in require matches the path that had created the precompile file. It also takes into account the set of dependencies already loaded into the current process and won't recompile those modules, even if their files change or disappear, in order to avoid creating incompatibilities between the running system and the precompile cache. Finally, it takes account of changes in any compile-time preferences.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"If you know that a module is not safe to precompile (for example, for one of the reasons described below), you should put __precompile__(false) in the module file (typically placed at the top). This will cause Base.compilecache to throw an error, and will cause using / import to load it directly into the current process and skip the precompile and caching. This also thereby prevents the module from being imported by any other precompiled module.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"You may need to be aware of certain behaviors inherent in the creation of incremental shared libraries which may require care when writing your module. For example, external state is not preserved. To accommodate this, explicitly separate any initialization steps that must occur at runtime from steps that can occur at compile time. For this purpose, Julia allows you to define an __init__() function in your module that executes any initialization steps that must occur at runtime. This function will not be called during compilation (--output-*). Effectively, you can assume it will be run exactly once in the lifetime of the code. You may, of course, call it manually if necessary, but the default is to assume this function deals with computing state for the local machine, which does not need to be – or even should not be – captured in the compiled image. It will be called after the module is loaded into a process, including if it is being loaded into an incremental compile (--output-incremental=yes), but not if it is being loaded into a full-compilation process.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"In particular, if you define a function __init__() in a module, then Julia will call __init__() immediately after the module is loaded (e.g., by import, using, or require) at runtime for the first time (i.e., __init__ is only called once, and only after all statements in the module have been executed). Because it is called after the module is fully imported, any submodules or other imported modules have their __init__ functions called before the __init__ of the enclosing module.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Two typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries.  For example, suppose that we are calling a C library libfoo that requires us to call a foo_init() initialization function at runtime. Suppose that we also want to define a global constant foo_data_ptr that holds the return value of a void *foo_data() function defined by libfoo – this constant must be initialized at runtime (not at compile time) because the pointer address will change from run to run.  You could accomplish this by defining the following __init__ function in your module:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"const foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Notice that it is perfectly possible to define a global inside a function like __init__; this is one of the advantages of using a dynamic language. But by making it a constant at global scope, we can ensure that the type is known to the compiler and allow it to generate better optimized code. Obviously, any other globals in your module that depends on foo_data_ptr would also have to be initialized in __init__.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Constants involving most Julia objects that are not produced by ccall do not need to be placed in __init__: their definitions can be precompiled and loaded from the cached module image. This includes complicated heap-allocated objects like arrays. However, any routine that returns a raw pointer value must be called at runtime for precompilation to work (Ptr objects will turn into null pointers unless they are hidden inside an isbits object). This includes the return values of the Julia functions @cfunction and pointer.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Dictionary and set types, or in general anything that depends on the output of a hash(key) method, are a trickier case.  In the common case where the keys are numbers, strings, symbols, ranges, Expr, or compositions of these types (via arrays, tuples, sets, pairs, etc.) they are safe to precompile.  However, for a few other key types, such as Function or DataType and generic user-defined types where you haven't defined a hash method, the fallback hash method depends on the memory address of the object (via its objectid) and hence may change from run to run. If you have one of these key types, or if you aren't sure, to be safe you can initialize this dictionary from within your __init__ function. Alternatively, you can use the IdDict dictionary type, which is specially handled by precompilation so that it is safe to initialize at compile-time.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"When using precompilation, it is important to keep a clear sense of the distinction between the compilation phase and the execution phase. In this mode, it will often be much more clearly apparent that Julia is a compiler which allows execution of arbitrary Julia code, not a standalone interpreter that also generates compiled code.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Other known potential failure scenarios include:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Global counters (for example, for attempting to uniquely identify objects). Consider the following code snippet:\nmutable struct UniquedById\n    myid::Int\n    let counter = 0\n        UniquedById() = new(counter += 1)\n    end\nend\nwhile the intent of this code was to give every instance a unique id, the counter value is recorded at the end of compilation. All subsequent usages of this incrementally compiled module will start from that same counter value.\nNote that objectid (which works by hashing the memory pointer) has similar issues (see notes on Dict usage below).\nOne alternative is to use a macro to capture @__MODULE__ and store it alone with the current counter value, however, it may be better to redesign the code to not depend on this global state.\nAssociative collections (such as Dict and Set) need to be re-hashed in __init__. (In the future, a mechanism may be provided to register an initializer function.)\nDepending on compile-time side-effects persisting through load-time. Example include: modifying arrays or other variables in other Julia modules; maintaining handles to open files or devices; storing pointers to other system resources (including memory);\nCreating accidental \"copies\" of global state from another module, by referencing it directly instead of via its lookup path. For example, (in global scope):\n#mystdout = Base.stdout #= will not work correctly, since this will copy Base.stdout into this module =#\n# instead use accessor functions:\ngetstdout() = Base.stdout #= best option =#\n# or move the assignment into the runtime:\n__init__() = global mystdout = Base.stdout #= also works =#","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Several additional restrictions are placed on the operations that can be done while precompiling code to help the user avoid other wrong-behavior situations:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"Calling eval to cause a side-effect in another module. This will also cause a warning to be emitted when the incremental precompile flag is set.\nglobal const statements from local scope after __init__() has been started (see issue #12010 for plans to add an error for this)\nReplacing a module is a runtime error while doing an incremental precompile.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"A few other points to be aware of:","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"No code reload / cache invalidation is performed after changes are made to the source files themselves, (including by Pkg.update), and no cleanup is done after Pkg.rm\nThe memory sharing behavior of a reshaped array is disregarded by precompilation (each view gets its own copy)\nExpecting the filesystem to be unchanged between compile-time and runtime e.g. @__FILE__/source_path() to find resources at runtime, or the BinDeps @checked_lib macro. Sometimes this is unavoidable. However, when possible, it can be good practice to copy resources into the module at compile-time so they won't need to be found at runtime.\nWeakRef objects and finalizers are not currently handled properly by the serializer (this will be fixed in an upcoming release).\nIt is usually best to avoid capturing references to instances of internal metadata objects such as Method, MethodInstance, MethodTable, TypeMapLevel, TypeMapEntry and fields of those objects, as this can confuse the serializer and may not lead to the outcome you desire. It is not necessarily an error to do this, but you simply need to be prepared that the system will try to copy some of these and to create a single unique instance of others.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"It is sometimes helpful during module development to turn off incremental precompilation. The command line flag --compiled-modules={yes|no|existing} enables you to toggle module precompilation on and off. When Julia is started with --compiled-modules=no the serialized modules in the compile cache are ignored when loading modules and module dependencies. In some cases, you may want to load existing precompiled modules, but not create new ones. This can be done by starting Julia with --compiled-modules=existing. More fine-grained control is available with --pkgimages={yes|no|existing}, which only affects native-code storage during precompilation. Base.compilecache can still be called manually. The state of this command line flag is passed to Pkg.build to disable automatic precompilation triggering when installing, updating, and explicitly building packages.","category":"page"},{"location":"manual/modules.html","page":"Modules","title":"Modules","text":"You can also debug some precompilation failures with environment variables. Setting JULIA_VERBOSE_LINKING=true may help resolve failures in linking shared libraries of compiled native code. See the Developer Documentation part of the Julia manual, where you will find further details in the section documenting Julia's internals under \"Package Images\".","category":"page"},{"location":"devdocs/sanitizers.html#Sanitizer-support","page":"Sanitizer support","title":"Sanitizer support","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Sanitizers can be used in custom Julia builds to make it easier to detect certain kinds of errors in Julia's internal C/C++ code.","category":"page"},{"location":"devdocs/sanitizers.html#Address-Sanitizer:-easy-build","page":"Sanitizer support","title":"Address Sanitizer: easy build","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"From a source-checkout of Julia, you should be able to build a version supporting address sanitization in Julia and LLVM as follows:","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"$ mkdir /tmp/julia\n$ contrib/asan/build.sh /tmp/julia/","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Here we've chosen /tmp/julia as a build directory, but you can choose whatever you wish. Once built, run the workload you wish to test with /tmp/julia/julia. Memory bugs will result in errors.","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"If you require customization or further detail, see the documentation below.","category":"page"},{"location":"devdocs/sanitizers.html#General-considerations","page":"Sanitizer support","title":"General considerations","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Using Clang's sanitizers obviously requires you to use Clang (USECLANG=1), but there's another catch: most sanitizers require a run-time library, provided by the host compiler, while the instrumented code generated by Julia's JIT relies on functionality from that library. This implies that the LLVM version of your host compiler must match that of the LLVM library used within Julia.","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"An easy solution is to have a dedicated build folder for providing a matching toolchain, by building with BUILD_LLVM_CLANG=1. You can then refer to this toolchain from another build folder by specifying USECLANG=1 while overriding the CC and CXX variables.","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"The sanitizers error out when they detect a shared library being opened using RTLD_DEEPBIND (ref: google/sanitizers#611). Since libblastrampoline by default uses RTLD_DEEPBIND, we need to set the environment variable LBT_USE_RTLD_DEEPBIND=0 when using a sanitizer.","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"To use one of of the sanitizers set SANITIZE=1 and then the appropriate flag for the sanitizer you want to use.","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"On macOS, this might need some extra flags also to work. Altogether, it might look like this, plus one or more of the SANITIZE_* flags listed below:","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"make -C deps USE_BINARYBUILDER_LLVM=0 LLVM_VER=svn stage-llvm\n\nmake -C src SANITIZE=1 USECLANG=1 \\\n    CC=~+/deps/scratch/llvm-svn/build_Release/bin/clang \\\n    CXX=~+/deps/scratch/llvm-svn/build_Release/bin/clang++ \\\n    CPPFLAGS=\"-isysroot $(xcode-select -p)/Platforms/MacOSX.platform/Developer/SDKs/MacOSX.sdk\" \\\n    CXXFLAGS=\"-isystem $(xcode-select -p)/Toolchains/XcodeDefault.xctoolchain/usr/include/c++/v1\"","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"(or put these into your Make.user, so you don't need to remember them every time).","category":"page"},{"location":"devdocs/sanitizers.html#Address-Sanitizer-(ASAN)","page":"Sanitizer support","title":"Address Sanitizer (ASAN)","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"For detecting or debugging memory bugs, you can use Clang's address sanitizer (ASAN). By compiling with SANITIZE_ADDRESS=1 you enable ASAN for the Julia compiler and its generated code. In addition, you can specify LLVM_SANITIZE=1 to sanitize the LLVM library as well. Note that these options incur a high performance and memory cost. For example, using ASAN for Julia and LLVM makes testall1 take 8-10 times as long while using 20 times as much memory (this can be reduced to respectively a factor of 3 and 4 by using the options described below).","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"By default, Julia sets the allow_user_segv_handler=1 ASAN flag, which is required for signal delivery to work properly. You can define other options using the ASAN_OPTIONS environment flag, in which case you'll need to repeat the default option mentioned before. For example, memory usage can be reduced by specifying fast_unwind_on_malloc=0 and malloc_context_size=2, at the cost of backtrace accuracy. For now, Julia also sets detect_leaks=0, but this should be removed in the future.","category":"page"},{"location":"devdocs/sanitizers.html#Example-setup","page":"Sanitizer support","title":"Example setup","text":"","category":"section"},{"location":"devdocs/sanitizers.html#Step-1:-Install-toolchain","page":"Sanitizer support","title":"Step 1: Install toolchain","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Checkout a Git worktree (or create out-of-tree build directory) at $TOOLCHAIN_WORKTREE and create a config file $TOOLCHAIN_WORKTREE/Make.user with","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"USE_BINARYBUILDER_LLVM=1\nBUILD_LLVM_CLANG=1","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Run:","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"cd $TOOLCHAIN_WORKTREE\nmake -C deps install-llvm install-clang install-llvm-tools","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"to install toolchain binaries in $TOOLCHAIN_WORKTREE/usr/tools","category":"page"},{"location":"devdocs/sanitizers.html#Step-2:-Build-Julia-with-ASAN","page":"Sanitizer support","title":"Step 2: Build Julia with ASAN","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Checkout a Git worktree (or create out-of-tree build directory) at $BUILD_WORKTREE and create a config file $BUILD_WORKTREE/Make.user with","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"TOOLCHAIN=$(TOOLCHAIN_WORKTREE)/usr/tools\n\n# use our new toolchain\nUSECLANG=1\noverride CC=$(TOOLCHAIN)/clang\noverride CXX=$(TOOLCHAIN)/clang++\nexport ASAN_SYMBOLIZER_PATH=$(TOOLCHAIN)/llvm-symbolizer\n\nUSE_BINARYBUILDER_LLVM=1\n\noverride SANITIZE=1\noverride SANITIZE_ADDRESS=1\n\n# make the GC use regular malloc/frees, which are hooked by ASAN\noverride WITH_GC_DEBUG_ENV=1\n\n# default to a debug build for better line number reporting\noverride JULIA_BUILD_MODE=debug\n\n# make ASAN consume less memory\nexport ASAN_OPTIONS=detect_leaks=0:fast_unwind_on_malloc=0:allow_user_segv_handler=1:malloc_context_size=2\n\nJULIA_PRECOMPILE=1\n\n# tell libblastrampoline to not use RTLD_DEEPBIND\nexport LBT_USE_RTLD_DEEPBIND=0","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"Run:","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"cd $BUILD_WORKTREE\nmake debug","category":"page"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"to build julia-debug with ASAN.","category":"page"},{"location":"devdocs/sanitizers.html#Memory-Sanitizer-(MSAN)","page":"Sanitizer support","title":"Memory Sanitizer (MSAN)","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"For detecting use of uninitialized memory, you can use Clang's memory sanitizer (MSAN) by compiling with SANITIZE_MEMORY=1.","category":"page"},{"location":"devdocs/sanitizers.html#Thread-Sanitizer-(TSAN)","page":"Sanitizer support","title":"Thread Sanitizer (TSAN)","text":"","category":"section"},{"location":"devdocs/sanitizers.html","page":"Sanitizer support","title":"Sanitizer support","text":"For debugging data-races and other threading related issues you can use Clang's thread sanitizer (TSAN) by compiling with SANITIZE_THREAD=1.","category":"page"},{"location":"manual/handling-operating-system-variation.html#Handling-Operating-System-Variation","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"","category":"section"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"When writing cross-platform applications or libraries, it is often necessary to allow for differences between operating systems. The variable Sys.KERNEL can be used to handle such cases. There are several functions in the Sys module intended to make this easier, such as isunix, islinux, isapple, isbsd, isfreebsd, and iswindows. These may be used as follows:","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"if Sys.iswindows()\n    windows_specific_thing(a)\nend","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"Note that islinux, isapple, and isfreebsd are mutually exclusive subsets of isunix. Additionally, there is a macro @static which makes it possible to use these functions to conditionally hide invalid code, as demonstrated in the following examples.","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"Simple blocks:","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"ccall((@static Sys.iswindows() ? :_fopen : :fopen), ...)","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"Complex blocks:","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"@static if Sys.islinux()\n    linux_specific_thing(a)\nelseif Sys.isapple()\n    apple_specific_thing(a)\nelse\n    generic_thing(a)\nend","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"When nesting conditionals, the @static must be repeated for each level (parentheses optional, but recommended for readability):","category":"page"},{"location":"manual/handling-operating-system-variation.html","page":"Handling Operating System Variation","title":"Handling Operating System Variation","text":"@static Sys.iswindows() ? :a : (@static Sys.isapple() ? :b : :c)","category":"page"},{"location":"base/multi-threading.html#lib-multithreading","page":"Multi-Threading","title":"Multi-Threading","text":"","category":"section"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base.Threads.@threads\nBase.Threads.foreach\nBase.Threads.@spawn\nBase.Threads.threadid\nBase.Threads.maxthreadid\nBase.Threads.nthreads\nBase.Threads.threadpool\nBase.Threads.nthreadpools\nBase.Threads.threadpoolsize\nBase.Threads.ngcthreads","category":"page"},{"location":"base/multi-threading.html#Base.Threads.@threads","page":"Multi-Threading","title":"Base.Threads.@threads","text":"Threads.@threads [schedule] for ... end\n\nA macro to execute a for loop in parallel. The iteration space is distributed to coarse-grained tasks. This policy can be specified by the schedule argument. The execution of the loop waits for the evaluation of all iterations.\n\nSee also: @spawn and pmap in Distributed.\n\nExtended help\n\nSemantics\n\nUnless stronger guarantees are specified by the scheduling option, the loop executed by @threads macro have the following semantics.\n\nThe @threads macro executes the loop body in an unspecified order and potentially concurrently. It does not specify the exact assignments of the tasks and the worker threads. The assignments can be different for each execution. The loop body code (including any code transitively called from it) must not make any assumptions about the distribution of iterations to tasks or the worker thread in which they are executed. The loop body for each iteration must be able to make forward progress independent of other iterations and be free from data races. As such, invalid synchronizations across iterations may deadlock while unsynchronized memory accesses may result in undefined behavior.\n\nFor example, the above conditions imply that:\n\nA lock taken in an iteration must be released within the same iteration.\nCommunicating between iterations using blocking primitives like Channels is incorrect.\nWrite only to locations not shared across iterations (unless a lock or atomic operation is used).\nUnless the :static schedule is used, the value of threadid() may change even within a single iteration. See Task Migration.\n\nSchedulers\n\nWithout the scheduler argument, the exact scheduling is unspecified and varies across Julia releases. Currently, :dynamic is used when the scheduler is not specified.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n:dynamic (default)\n\n:dynamic scheduler executes iterations dynamically to available worker threads. Current implementation assumes that the workload for each iteration is uniform. However, this assumption may be removed in the future.\n\nThis scheduling option is merely a hint to the underlying execution mechanism. However, a few properties can be expected. The number of Tasks used by :dynamic scheduler is bounded by a small constant multiple of the number of available worker threads (Threads.threadpoolsize()). Each task processes contiguous regions of the iteration space. Thus, @threads :dynamic for x in xs; f(x); end is typically more efficient than @sync for x in xs; @spawn f(x); end if length(xs) is significantly larger than the number of the worker threads and the run-time of f(x) is relatively smaller than the cost of spawning and synchronizing a task (typically less than 10 microseconds).\n\ncompat: Julia 1.8\nThe :dynamic option for the schedule argument is available and the default as of Julia 1.8.\n\n:greedy\n\n:greedy scheduler spawns up to Threads.threadpoolsize() tasks, each greedily working on the given iterated values as they are produced. As soon as one task finishes its work, it takes the next value from the iterator. Work done by any individual task is not necessarily on contiguous values from the iterator. The given iterator may produce values forever, only the iterator interface is required (no indexing).\n\nThis scheduling option is generally a good choice if the workload of individual iterations is not uniform/has a large spread.\n\ncompat: Julia 1.11\nThe :greedy option for the schedule argument is available as of Julia 1.11.\n\n:static\n\n:static scheduler creates one task per thread and divides the iterations equally among them, assigning each task specifically to each thread. In particular, the value of threadid() is guaranteed to be constant within one iteration. Specifying :static is an error if used from inside another @threads loop or from a thread other than 1.\n\nnote: Note\n:static scheduling exists for supporting transition of code written before Julia 1.3. In newly written library functions, :static scheduling is discouraged because the functions using this option cannot be called from arbitrary worker threads.\n\nExamples\n\nTo illustrate of the different scheduling strategies, consider the following function busywait containing a non-yielding timed loop that runs for a given number of seconds.\n\njulia> function busywait(seconds)\n            tstart = time_ns()\n            while (time_ns() - tstart) / 1e9 < seconds\n            end\n        end\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :static for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n6.003001 seconds (16.33 k allocations: 899.255 KiB, 0.25% compilation time)\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :dynamic for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n2.012056 seconds (16.05 k allocations: 883.919 KiB, 0.66% compilation time)\n\nThe :dynamic example takes 2 seconds since one of the non-occupied threads is able to run two of the 1-second iterations to complete the for loop.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Base.Threads.foreach","page":"Multi-Threading","title":"Base.Threads.foreach","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.threadpoolsize())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to f are split across ntasks tasks spawned by Threads.@spawn. This function will wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a manner that enables Julia's scheduler to more freely load-balance work items across threads. This approach generally has higher per-item overhead, but may perform better than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner that incurs lower per-item overhead than FairSchedule, but is less amenable to load-balancing. This approach thus may be more suitable for fine-grained, uniform workloads, but may perform worse than FairSchedule in concurrence with other multithreaded workloads.\n\nExamples\n\njulia> n = 20\n\njulia> c = Channel{Int}(ch -> foreach(i -> put!(ch, i), 1:n), 1)\n\njulia> d = Channel{Int}(n) do ch\n           f = i -> put!(ch, i^2)\n           Threads.foreach(f, c)\n       end\n\njulia> collect(d)\ncollect(d) = [1, 4, 9, 16, 25, 36, 49, 64, 81, 100, 121, 144, 169, 196, 225, 256, 289, 324, 361, 400]\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.@spawn","page":"Multi-Threading","title":"Base.Threads.@spawn","text":"Threads.@spawn [:default|:interactive] expr\n\nCreate a Task and schedule it to run on any available thread in the specified threadpool (:default if unspecified). The task is allocated to a thread once one becomes available. To wait for the task to finish, call wait on the result of this macro, or call fetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nnote: Note\nThe thread that the task runs on may change if the task yields, therefore threadid() should not be treated as constant for a task. See Task Migration, and the broader multi-threading manual for further important caveats. See also the chapter on threadpools.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\ncompat: Julia 1.9\nA threadpool may be specified as of Julia 1.9.\n\nExamples\n\njulia> t() = println(\"Hello from \", Threads.threadid());\n\njulia> tasks = fetch.([Threads.@spawn t() for i in 1:4]);\nHello from 1\nHello from 1\nHello from 3\nHello from 4\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Base.Threads.threadid","page":"Multi-Threading","title":"Base.Threads.threadid","text":"Threads.threadid() -> Int\n\nGet the ID number of the current thread of execution. The master thread has ID 1.\n\nExamples\n\njulia> Threads.threadid()\n1\n\njulia> Threads.@threads for i in 1:4\n          println(Threads.threadid())\n       end\n4\n2\n5\n4\n\nnote: Note\nThe thread that a task runs on may change if the task yields, which is known as Task Migration. For this reason in most cases it is not safe to use threadid() to index into, say, a vector of buffer or stateful objects.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.maxthreadid","page":"Multi-Threading","title":"Base.Threads.maxthreadid","text":"Threads.maxthreadid() -> Int\n\nGet a lower bound on the number of threads (across all thread pools) available to the Julia process, with atomic-acquire semantics. The result will always be greater than or equal to threadid() as well as threadid(task) for any task you were able to observe before calling maxthreadid.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.nthreads","page":"Multi-Threading","title":"Base.Threads.nthreads","text":"Threads.nthreads(:default | :interactive) -> Int\n\nGet the current number of threads within the specified thread pool. The threads in :interactive have id numbers 1:nthreads(:interactive), and the threads in :default have id numbers in nthreads(:interactive) .+ (1:nthreads(:default)).\n\nSee also BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library and Threads.maxthreadid().\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.threadpool","page":"Multi-Threading","title":"Base.Threads.threadpool","text":"Threads.threadpool(tid = threadid()) -> Symbol\n\nReturns the specified thread's threadpool; either :default, :interactive, or :foreign.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.nthreadpools","page":"Multi-Threading","title":"Base.Threads.nthreadpools","text":"Threads.nthreadpools() -> Int\n\nReturns the number of threadpools currently configured.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.threadpoolsize","page":"Multi-Threading","title":"Base.Threads.threadpoolsize","text":"Threads.threadpoolsize(pool::Symbol = :default) -> Int\n\nGet the number of threads available to the default thread pool (or to the specified thread pool).\n\nSee also: BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.ngcthreads","page":"Multi-Threading","title":"Base.Threads.ngcthreads","text":"Threads.ngcthreads() -> Int\n\nReturns the number of GC threads currently configured. This includes both mark threads and concurrent sweep threads.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"See also Multi-Threading.","category":"page"},{"location":"base/multi-threading.html#Atomic-operations","page":"Multi-Threading","title":"Atomic operations","text":"","category":"section"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"atomic","category":"page"},{"location":"base/multi-threading.html#atomic","page":"Multi-Threading","title":"atomic","text":"Unsafe pointer operations are compatible with loading and storing pointers declared with _Atomic and std::atomic type in C11 and C++23 respectively. An error may be thrown if there is not support for atomically loading the Julia type T.\n\nSee also: unsafe_load, unsafe_modify!, unsafe_replace!, unsafe_store!, unsafe_swap!\n\n\n\n\n\n","category":"keyword"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base.@atomic\nBase.@atomicswap\nBase.@atomicreplace\nBase.@atomiconce\nBase.AtomicMemory","category":"page"},{"location":"base/multi-threading.html#Base.@atomic","page":"Multi-Threading","title":"Base.@atomic","text":"@atomic var\n@atomic order ex\n\nMark var or ex as being performed atomically, if ex is a supported expression. If no order is specified it defaults to :sequentially_consistent.\n\n@atomic a.b.x = new\n@atomic a.b.x += addend\n@atomic :release a.b.x = new\n@atomic :acquire_release a.b.x += addend\n\nPerform the store operation expressed on the right atomically and return the new value.\n\nWith =, this operation translates to a setproperty!(a.b, :x, new) call. With any operator also, this operation translates to a modifyproperty!(a.b, :x, +, addend)[2] call.\n\n@atomic a.b.x max arg2\n@atomic a.b.x + arg2\n@atomic max(a.b.x, arg2)\n@atomic :acquire_release max(a.b.x, arg2)\n@atomic :acquire_release a.b.x + arg2\n@atomic :acquire_release a.b.x max arg2\n\nPerform the binary operation expressed on the right atomically. Store the result into the field in the first argument and return the values (old, new).\n\nThis operation translates to a modifyproperty!(a.b, :x, func, arg2) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomic :sequentially_consistent a.x = 2 # set field x of a, with sequential consistency\n2\n\njulia> @atomic a.x += 1 # increment field x of a, with sequential consistency\n3\n\njulia> @atomic a.x + 1 # increment field x of a, with sequential consistency\n3 => 4\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> @atomic max(a.x, 10) # change field x of a to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic a.x max 5 # again change field x of a to the max value, with sequential consistency\n10 => 10\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Base.@atomicswap","page":"Multi-Threading","title":"Base.@atomicswap","text":"@atomicswap a.b.x = new\n@atomicswap :sequentially_consistent a.b.x = new\n\nStores new into a.b.x and returns the old value of a.b.x.\n\nThis operation translates to a swapproperty!(a.b, :x, new) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicswap a.x = 2+2 # replace field x of a with 4, with sequential consistency\n1\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Base.@atomicreplace","page":"Multi-Threading","title":"Base.@atomicreplace","text":"@atomicreplace a.b.x expected => desired\n@atomicreplace :sequentially_consistent a.b.x expected => desired\n@atomicreplace :sequentially_consistent :monotonic a.b.x expected => desired\n\nPerform the conditional replacement expressed by the pair atomically, returning the values (old, success::Bool). Where success indicates whether the replacement was completed.\n\nThis operation translates to a replaceproperty!(a.b, :x, expected, desired) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n2\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace a.x xchg\n(old = 2, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n0\n\ncompat: Julia 1.7\nThis functionality requires at least Julia 1.7.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Base.@atomiconce","page":"Multi-Threading","title":"Base.@atomiconce","text":"@atomiconce a.b.x = value\n@atomiconce :sequentially_consistent a.b.x = value\n@atomiconce :sequentially_consistent :monotonic a.b.x = value\n\nPerform the conditional assignment of value atomically if it was previously unset, returning the value success::Bool. Where success indicates whether the assignment was completed.\n\nThis operation translates to a setpropertyonce!(a.b, :x, value) call.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct AtomicOnce\n           @atomic x\n           AtomicOnce() = new()\n       end\n\njulia> a = AtomicOnce()\nAtomicOnce(#undef)\n\njulia> @atomiconce a.x = 1 # set field x of a to 1, if unset, with sequential consistency\ntrue\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomiconce a.x = 1 # set field x of a to 1, if unset, with sequential consistency\nfalse\n\ncompat: Julia 1.11\nThis functionality requires at least Julia 1.11.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Core.AtomicMemory","page":"Multi-Threading","title":"Core.AtomicMemory","text":"AtomicMemory{T} == GenericMemory{:atomic, T, Core.CPU}\n\nFixed-size DenseVector{T}. Access to its any of its elements is performed atomically (with :monotonic ordering). Setting any of the elements must be accomplished using the @atomic macro and explicitly specifying ordering.\n\nwarning: Warning\nEach element is independently atomic when accessed, and cannot be set non-atomically. Currently the @atomic macro and higher level interface have not been completed, but the building blocks for a future implementation are the internal intrinsics Core.memoryrefget, Core.memoryrefset!, Core.memoryref_isassigned, Core.memoryrefswap!, Core.memoryrefmodify!, and Core.memoryrefreplace!.\n\nFor details, see Atomic Operations\n\ncompat: Julia 1.11\nThis type requires Julia 1.11 or later.\n\n\n\n\n\n","category":"type"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"There are also optional memory ordering parameters for the unsafe set of functions, that select the C/C++-compatible versions of these atomic operations, if that parameter is specified to unsafe_load, unsafe_store!, unsafe_swap!, unsafe_replace!, and unsafe_modify!.","category":"page"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"warning: Warning\nThe following APIs are deprecated, though support for them is likely to remain for several releases.","category":"page"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base.Threads.Atomic\nBase.Threads.atomic_cas!\nBase.Threads.atomic_xchg!\nBase.Threads.atomic_add!\nBase.Threads.atomic_sub!\nBase.Threads.atomic_and!\nBase.Threads.atomic_nand!\nBase.Threads.atomic_or!\nBase.Threads.atomic_xor!\nBase.Threads.atomic_max!\nBase.Threads.atomic_min!\nBase.Threads.atomic_fence","category":"page"},{"location":"base/multi-threading.html#Base.Threads.Atomic","page":"Multi-Threading","title":"Base.Threads.Atomic","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only accessed atomically, i.e. in a thread-safe manner.\n\nOnly certain \"simple\" types can be used atomically, namely the primitive boolean, integer, and float-point types. These are Bool, Int8...Int128, UInt8...UInt128, and Float16...Float64.\n\nNew atomic objects can be created from a non-atomic values; if none is specified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!, atomic_xchg!, etc.\n\n\n\n\n\n","category":"type"},{"location":"base/multi-threading.html#Base.Threads.atomic_cas!","page":"Multi-Threading","title":"Base.Threads.atomic_cas!","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write newval to x. Otherwise, leaves x unmodified. Returns the old value in x. By comparing the returned value to cmp (via ===) one knows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before the transaction, one records the value in x. After the transaction, the new value is stored only if x has not been modified in the mean time.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_xchg!","page":"Multi-Threading","title":"Base.Threads.atomic_xchg!","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_add!","page":"Multi-Threading","title":"Base.Threads.atomic_add!","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_sub!","page":"Multi-Threading","title":"Base.Threads.atomic_sub!","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_and!","page":"Multi-Threading","title":"Base.Threads.atomic_and!","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_nand!","page":"Multi-Threading","title":"Base.Threads.atomic_nand!","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_or!","page":"Multi-Threading","title":"Base.Threads.atomic_or!","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_xor!","page":"Multi-Threading","title":"Base.Threads.atomic_xor!","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_max!","page":"Multi-Threading","title":"Base.Threads.atomic_max!","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_min!","page":"Multi-Threading","title":"Base.Threads.atomic_min!","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#Base.Threads.atomic_fence","page":"Multi-Threading","title":"Base.Threads.atomic_fence","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering semantics. There are algorithms where this is needed, i.e. where an acquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic operations in Julia already have acquire/release semantics, explicit fences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n","category":"function"},{"location":"base/multi-threading.html#ccall-using-a-libuv-threadpool-(Experimental)","page":"Multi-Threading","title":"ccall using a libuv threadpool (Experimental)","text":"","category":"section"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base.@threadcall","category":"page"},{"location":"base/multi-threading.html#Base.@threadcall","page":"Multi-Threading","title":"Base.@threadcall","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work in a different thread. This is useful when you want to call a blocking C function without causing the current julia thread to become blocked. Concurrency is limited by size of the libuv thread pool, which defaults to 4 threads but can be increased by setting the UV_THREADPOOL_SIZE environment variable and restarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n","category":"macro"},{"location":"base/multi-threading.html#Low-level-synchronization-primitives","page":"Multi-Threading","title":"Low-level synchronization primitives","text":"","category":"section"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"These building blocks are used to create the regular synchronization objects.","category":"page"},{"location":"base/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base.Threads.SpinLock","category":"page"},{"location":"base/multi-threading.html#Base.Threads.SpinLock","page":"Multi-Threading","title":"Base.Threads.SpinLock","text":"SpinLock()\n\nCreate a non-reentrant, test-and-test-and-set spin lock. Recursive use will result in a deadlock. This kind of lock should only be used around code that takes little time to execute and does not block (e.g. perform I/O). In general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock. If !islocked(lck::SpinLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\nTest-and-test-and-set spin locks are quickest up to about 30ish contending threads. If you have more contention than that, different synchronization approaches should be considered.\n\n\n\n\n\n","category":"type"},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Optimizations","page":"isbits Union Optimizations","title":"isbits Union Optimizations","text":"","category":"section"},{"location":"devdocs/isbitsunionarrays.html","page":"isbits Union Optimizations","title":"isbits Union Optimizations","text":"In Julia, the Array type holds both \"bits\" values as well as heap-allocated \"boxed\" values. The distinction is whether the value itself is stored inline (in the direct allocated memory of the array), or if the memory of the array is simply a collection of pointers to objects allocated elsewhere. In terms of performance, accessing values inline is clearly an advantage over having to follow a pointer to the actual value. The definition of \"isbits\" generally means any Julia type with a fixed, determinate size, meaning no \"pointer\" fields, see ?isbitstype.","category":"page"},{"location":"devdocs/isbitsunionarrays.html","page":"isbits Union Optimizations","title":"isbits Union Optimizations","text":"Julia also supports Union types, quite literally the union of a set of types. Custom Union type definitions can be extremely handy for applications wishing to \"cut across\" the nominal type system (i.e. explicit subtype relationships) and define methods or functionality on these, otherwise unrelated, set of types. A compiler challenge, however, is in determining how to treat these Union types. The naive approach (and indeed, what Julia itself did pre-0.7), is to simply make a \"box\" and then a pointer in the box to the actual value, similar to the previously mentioned \"boxed\" values. This is unfortunate, however, because of the number of small, primitive \"bits\" types (think UInt8, Int32, Float64, etc.) that would easily fit themselves inline in this \"box\" without needing any indirection for value access. There are two main ways Julia can take advantage of this optimization as of 0.7: isbits Union fields in types, and isbits Union Arrays.","category":"page"},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Structs","page":"isbits Union Optimizations","title":"isbits Union Structs","text":"","category":"section"},{"location":"devdocs/isbitsunionarrays.html","page":"isbits Union Optimizations","title":"isbits Union Optimizations","text":"Julia now includes an optimization wherein \"isbits Union\" fields in types (mutable struct, struct, etc.) will be stored inline. This is accomplished by determining the \"inline size\" of the Union type (e.g. Union{UInt8, Int16} will have a size of two bytes, which represents the size needed of the largest Union type Int16), and in addition, allocating an extra \"type tag byte\" (UInt8), whose value signals the type of the actual value stored inline of the \"Union bytes\". The type tag byte value is the index of the actual value's type in the Union type's order of types. For example, a type tag value of 0x02 for a field with type Union{Nothing, UInt8, Int16} would indicate that an Int16 value is stored in the 16 bits of the field in the structure's memory; a 0x01 value would indicate that a UInt8 value was stored in the first 8 bits of the 16 bits of the field's memory. Lastly, a value of 0x00 signals that the nothing value will be returned for this field, even though, as a singleton type with a single type instance, it technically has a size of 0. The type tag byte for a type's Union field is stored directly after the field's computed Union memory.","category":"page"},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Memory","page":"isbits Union Optimizations","title":"isbits Union Memory","text":"","category":"section"},{"location":"devdocs/isbitsunionarrays.html","page":"isbits Union Optimizations","title":"isbits Union Optimizations","text":"Julia can now also store \"isbits Union\" values inline in a Memory, as opposed to requiring an indirection box. The optimization is accomplished by storing an extra \"type tag memory\" of bytes, one byte per element, alongside the bytes of the actual data. This type tag memory serves the same function as the type field case: its value signals the type of the actual stored Union value. The \"type tag memory\" directly follows the regular data space. So the formula to access an isbits Union Array's type tag bytes is a->data + a->length * a->elsize.","category":"page"},{"location":"base/scopedvalues.html#scoped-values","page":"Scoped Values","title":"Scoped Values","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Scoped values provide an implementation of dynamic scoping in Julia.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"note: Lexical scoping vs dynamic scoping\nLexical scoping is the default behavior in Julia. Under lexical scoping the scope of a variable is determined by the lexical (textual) structure of a program. Under dynamic scoping a variable is bound to the most recent assigned value during the program's execution.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"The state of a scoped value is dependent on the execution path of the program. This means that for a scoped value you may observe multiple different values concurrently.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"compat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"In its simplest form you can create a ScopedValue with a default value and then use with or @with to enter a new dynamic scope. The new scope will inherit all values from the parent scope (and recursively from all outer scopes) with the provided scoped value taking priority over previous definitions.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Let's first look at an example of lexical scope. A let statement begins a new lexical scope within which the outer definition of x is shadowed by it's inner definition.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"x = 1\nlet x = 5\n    @show x # 5\nend\n@show x # 1","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"In the following example, since Julia uses lexical scope, the variable x in the body of f refers to the x defined in the global scope, and entering a let scope does not change the value f observes.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"x = 1\nf() = @show x\nlet x = 5\n    f() # 1\nend\nf() # 1","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Now using a ScopedValue we can use dynamic scoping.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\n\nx = ScopedValue(1)\nf() = @show x[]\nwith(x=>5) do\n    f() # 5\nend\nf() # 1","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Note that the observed value of the ScopedValue is dependent on the execution path of the program.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"It often makes sense to use a const variable to point to a scoped value, and you can set the value of multiple ScopedValues with one call to with.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\n\nf() = @show a[]\ng() = @show b[]\n\nconst a = ScopedValue(1)\nconst b = ScopedValue(2)\n\nf() # a[] = 1\ng() # b[] = 2\n\n# Enter a new dynamic scope and set value.\nwith(a => 3) do\n    f() # a[] = 3\n    g() # b[] = 2\n    with(a => 4, b => 5) do\n        f() # a[] = 4\n        g() # b[] = 5\n    end\n    f() # a[] = 3\n    g() # b[] = 2\nend\n\nf() # a[] = 1\ng() # b[] = 2","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"ScopedValues provides a macro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. @with var=>val expr is equivalent to with(var=>val) do expr end. However, with requires a zero-argument closure or function, which results in an extra call-frame. As an example, consider the following function f:","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nconst a = ScopedValue(1)\nf(x) = a[] + x","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"If you wish to run f in a dynamic scope with a set to 2, then you can use with:","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"with(() -> f(10), a=>2)","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"However, this requires wrapping f in a zero-argument function. If you wish to avoid the extra call-frame, then you can use the @with macro:","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"@with a=>2 f(10)","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"note: Note\nDynamic scopes are inherited by Tasks, at the moment of task creation. Dynamic scopes are not propagated through Distributed.jl operations.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"In the example below we open a new dynamic scope before launching a task. The parent task and the two child tasks observe independent values of the same scoped value at the same time.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst scoped_val = ScopedValue(1)\n@sync begin\n    with(scoped_val => 2)\n        @spawn @show scoped_val[] # 2\n    end\n    with(scoped_val => 3)\n        @spawn @show scoped_val[] # 3\n    end\n    @show scoped_val[] # 1\nend","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Scoped values are constant throughout a scope, but you can store mutable state in a scoped value. Just keep in mind that the usual caveats for global variables apply in the context of concurrent programming.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Care is also required when storing references to mutable state in scoped values. You might want to explicitly unshare mutable state when entering a new dynamic scope.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# Example of using a mutable value wrongly\n@sync begin\n    # `Dict` is not thread-safe the usage below is invalid\n    @spawn (sval_dict[][:a] = 3)\n    @spawn (sval_dict[][:b] = 3)\nend\n\n@sync begin\n    # If we instead pass a unique dictionary to each\n    # task we can access the dictionaries race free.\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:a] = 3)\n    end\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:b] = 3)\n    end\nend","category":"page"},{"location":"base/scopedvalues.html#Example","page":"Scoped Values","title":"Example","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"In the example below we use a scoped value to implement a permission check in a web-application. After determining the permissions of the request, a new dynamic scope is entered and the scoped value LEVEL is set. Other parts of the application can query the scoped value and will receive the appropriate value. Other alternatives like task-local storage and global variables are not well suited for this kind of propagation; our only alternative would have been to thread a value through the entire call-chain.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\n\nconst LEVEL = ScopedValue(:GUEST)\n\nfunction serve(request, response)\n    level = isAdmin(request) ? :ADMIN : :GUEST\n    with(LEVEL => level) do\n        Threads.@spawn handle(request, response)\n    end\nend\n\nfunction open(connection::Database)\n    level = LEVEL[]\n    if level !== :ADMIN\n        error(\"Access disallowed\")\n    end\n    # ... open connection\nend\n\nfunction handle(request, response)\n    # ...\n    open(Database(#=...=#))\n    # ...\nend","category":"page"},{"location":"base/scopedvalues.html#Idioms","page":"Scoped Values","title":"Idioms","text":"","category":"section"},{"location":"base/scopedvalues.html#unshare_mutable_state","page":"Scoped Values","title":"Unshare mutable state","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# If you want to add new values to the dict, instead of replacing\n# it, unshare the values explicitly. In this example we use `merge`\n# to unshare the state of the dictionary in parent scope.\n@sync begin\n    with(sval_dict => merge(sval_dict[], Dict(:a => 10))) do\n        @spawn @show sval_dict[][:a]\n    end\n    @spawn sval_dict[][:a] = 3 # Not a race since they are unshared.\nend","category":"page"},{"location":"base/scopedvalues.html#Scoped-values-as-globals","page":"Scoped Values","title":"Scoped values as globals","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"In order to access the value of a scoped value, the scoped value itself has to be in (lexical) scope. This means most often you likely want to use scoped values as constant globals.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nconst sval = ScopedValue(1)","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Indeed one can think of scoped values as hidden function arguments.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"This does not preclude their use as non-globals.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nfunction main()\n    role = ScopedValue(:client)\n\n    function launch()\n        #...\n        role[]\n    end\n\n    @with role => :server @spawn launch()\n    launch()\nend","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"But it might have been simpler to just directly pass the function argument in these cases.","category":"page"},{"location":"base/scopedvalues.html#Very-many-ScopedValues","page":"Scoped Values","title":"Very many ScopedValues","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"If you find yourself creating many ScopedValue's for one given module, it may be better to use a dedicated struct to hold them.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"using Base.ScopedValues\n\nBase.@kwdef struct Configuration\n    color::Bool = false\n    verbose::Bool = false\nend\n\nconst CONFIG = ScopedValue(Configuration(color=true))\n\n@with CONFIG => Configuration(color=CONFIG[].color, verbose=true) begin\n    @show CONFIG[].color # true\n    @show CONFIG[].verbose # true\nend","category":"page"},{"location":"base/scopedvalues.html#API-docs","page":"Scoped Values","title":"API docs","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Base.ScopedValues.ScopedValue\nBase.ScopedValues.with\nBase.ScopedValues.@with\nBase.isassigned(::Base.ScopedValues.ScopedValue)\nBase.ScopedValues.get","category":"page"},{"location":"base/scopedvalues.html#Base.ScopedValues.ScopedValue","page":"Scoped Values","title":"Base.ScopedValues.ScopedValue","text":"ScopedValue(x)\n\nCreate a container that propagates values across dynamic scopes. Use with to create and enter a new dynamic scope.\n\nValues can only be set when entering a new dynamic scope, and the value referred to will be constant during the execution of a dynamic scope.\n\nDynamic scopes are propagated across tasks.\n\nExamples\n\njulia> using Base.ScopedValues;\n\njulia> const sval = ScopedValue(1);\n\njulia> sval[]\n1\n\njulia> with(sval => 2) do\n           sval[]\n       end\n2\n\njulia> sval[]\n1\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.\n\n\n\n\n\n","category":"type"},{"location":"base/scopedvalues.html#Base.ScopedValues.with","page":"Scoped Values","title":"Base.ScopedValues.with","text":"with(f, (var::ScopedValue{T} => val)...)\n\nExecute f in a new dynamic scope with var set to val. val will be converted to type T.\n\nSee also: ScopedValues.@with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> f(10)\n11\n\njulia> with(a=>2) do\n           f(10)\n       end\n12\n\njulia> f(10)\n11\n\njulia> b = ScopedValue(2);\n\njulia> g(x) = a[] + b[] + x;\n\njulia> with(a=>10, b=>20) do\n           g(30)\n       end\n60\n\njulia> with(() -> a[] * b[], a=>3, b=>4)\n12\n\n\n\n\n\n","category":"function"},{"location":"base/scopedvalues.html#Base.ScopedValues.@with","page":"Scoped Values","title":"Base.ScopedValues.@with","text":"@with (var::ScopedValue{T} => val)... expr\n\nMacro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. val will be converted to type T. @with var=>val expr is equivalent to with(var=>val) do expr end, but @with avoids creating a closure.\n\nSee also: ScopedValues.with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> const a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> @with a=>2 f(10)\n12\n\njulia> @with a=>3 begin\n           x = 100\n           f(x)\n       end\n103\n\n\n\n\n\n","category":"macro"},{"location":"base/scopedvalues.html#Base.isassigned-Tuple{Base.ScopedValues.ScopedValue}","page":"Scoped Values","title":"Base.isassigned","text":"isassigned(val::ScopedValue)\n\nTest whether a ScopedValue has an assigned value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1); b = ScopedValue{Int}();\n\njulia> isassigned(a)\ntrue\n\njulia> isassigned(b)\nfalse\n\n\n\n\n\n","category":"method"},{"location":"base/scopedvalues.html#Base.ScopedValues.get","page":"Scoped Values","title":"Base.ScopedValues.get","text":"get(val::ScopedValue{T})::Union{Nothing, Some{T}}\n\nIf the scoped value isn't set and doesn't have a default value, return nothing. Otherwise returns Some{T} with the current value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.ScopedValue.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(42); b = ScopedValue{Int}();\n\njulia> ScopedValues.get(a)\nSome(42)\n\njulia> isnothing(ScopedValues.get(b))\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/scopedvalues.html#Implementation-notes-and-performance","page":"Scoped Values","title":"Implementation notes and performance","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"Scopes use a persistent dictionary. Lookup and insertion is O(log(32, n)), upon dynamic scope entry a small amount of data is copied and the unchanged data is shared among other scopes.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"The Scope object itself is not user-facing and may be changed in a future version of Julia.","category":"page"},{"location":"base/scopedvalues.html#Design-inspiration","page":"Scoped Values","title":"Design inspiration","text":"","category":"section"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"This design was heavily inspired by JEPS-429, which in turn was inspired by dynamically scoped free variables in many Lisp dialects. In particular Interlisp-D and its deep binding strategy.","category":"page"},{"location":"base/scopedvalues.html","page":"Scoped Values","title":"Scoped Values","text":"A prior design discussed was context variables ala PEPS-567 and implemented in Julia as ContextVariablesX.jl.","category":"page"},{"location":"base/math.html#Mathematics","page":"Mathematics","title":"Mathematics","text":"","category":"section"},{"location":"base/math.html#math-ops","page":"Mathematics","title":"Mathematical Operators","text":"","category":"section"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"Base.:-(::Any)\nBase.:(+)\nBase.:-(::Any, ::Any)\nBase.:*(::Any, ::Any...)\nBase.:(/)\nBase.:\\(::Any, ::Any)\nBase.:^(::Number, ::Number)\nBase.fma\nBase.muladd\nBase.inv(::Number)\nBase.div\nBase.div(::Any, ::Any, ::RoundingMode)\nBase.fld\nBase.cld\nBase.mod\nBase.rem\nBase.rem(::Any, ::Any, ::RoundingMode)\nBase.rem2pi\nBase.Math.mod2pi\nBase.divrem\nBase.fldmod\nBase.fld1\nBase.mod1\nBase.fldmod1\nBase.:(//)\nBase.rationalize\nBase.numerator\nBase.denominator\nBase.:(<<)\nBase.:(>>)\nBase.:(>>>)\nBase.bitrotate\nBase.:(:)\nBase.range\nBase.OneTo\nBase.StepRangeLen\nBase.logrange\nBase.LogRange\nBase.:(==)\nBase.:(!=)\nBase.:(!==)\nBase.:(<)\nBase.:(<=)\nBase.:(>)\nBase.:(>=)\nBase.cmp\nBase.:(~)\nBase.:(&)\nBase.:(|)\nBase.xor\nBase.nand\nBase.nor\nBase.:(!)\n&&\n||","category":"page"},{"location":"base/math.html#Base.:--Tuple{Any}","page":"Mathematics","title":"Base.:-","text":"-(x)\n\nUnary minus operator.\n\nSee also: abs, flipsign.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\njulia> -(true)  # promotes to Int\n-1\n\njulia> -(0x003)\n0xfffd\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.:+","page":"Mathematics","title":"Base.:+","text":"dt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of the Time are used along with the year, month, and day of the Date to create the new DateTime. Non-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n+(x, y...)\n\nAddition operator.\n\nInfix x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...), which by default then calls (x+y) + z + ... starting from the left.\n\nNote that overflow is possible for most integer types, including the default Int, when adding large numbers.\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\njulia> [1,2] + [3,4]\n2-element Vector{Int64}:\n 4\n 6\n\njulia> typemax(Int) + 1 < 0\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:--Tuple{Any, Any}","page":"Mathematics","title":"Base.:-","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any}}","page":"Mathematics","title":"Base.:*","text":"*(x, y...)\n\nMultiplication operator.\n\nInfix x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...), which by default then calls (x*y) * z * ... starting from the left.\n\nJuxtaposition such as 2pi also calls *(2, pi). Note that this operation has higher precedence than a literal *. Note also that juxtaposition \"0x...\" (integer zero times a variable whose name starts with x) is forbidden as it clashes with unsigned integer literals: 0x01 isa UInt8.\n\nNote that overflow is possible for most integer types, including the default Int, when multiplying large numbers.\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\njulia> [2 0; 0 3] * [1, 10]  # matrix * vector\n2-element Vector{Int64}:\n  2\n 30\n\njulia> 1/2pi, 1/2*pi  # juxtaposition has higher precedence\n(0.15915494309189535, 1.5707963267948966)\n\njulia> x = [1, 2]; x'x  # adjoint vector * vector\n5\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.:/","page":"Mathematics","title":"Base.:/","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right.\n\nGives floating-point results for integer arguments. See ÷ for integer division, or // for Rational results.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\nA / B\n\nMatrix right-division: A / B is equivalent to (B' \\ A')' where \\ is the left-division operator. For square matrices, the result X is such that A == X*B.\n\nSee also: rdiv!.\n\nExamples\n\njulia> A = Float64[1 4 5; 3 9 2]; B = Float64[1 4 2; 3 4 2; 8 7 1];\n\njulia> X = A / B\n2×3 Matrix{Float64}:\n -0.65   3.75  -1.2\n  3.25  -2.75   1.0\n\njulia> isapprox(A, X*B)\ntrue\n\njulia> isapprox(X, A*pinv(B))\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:\\-Tuple{Any, Any}","page":"Mathematics","title":"Base.:\\","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.:^-Tuple{Number, Number}","page":"Mathematics","title":"Base.:^","text":"^(x, y)\n\nExponentiation operator.\n\nIf x and y are integers, the result may overflow. To enter numbers in scientific notation, use Float64 literals such as 1.2e3 rather than 1.2 * 10^3.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code x^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to enable compile-time specialization on the value of the exponent. (As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y), where usually ^ == Base.^ unless ^ has been defined in the calling namespace.) If y is a negative integer literal, then Base.literal_pow transforms the operation to inv(x)^-y by default, where -y is positive.\n\nSee also exp2, <<.\n\nExamples\n\njulia> 3^5\n243\n\njulia> 3^-1  # uses Base.literal_pow\n0.3333333333333333\n\njulia> p = -1;\n\njulia> 3^p\nERROR: DomainError with -1:\nCannot raise an integer x to a negative power -1.\n[...]\n\njulia> 3.0^p\n0.3333333333333333\n\njulia> 10^19 > 0  # integer overflow\nfalse\n\njulia> big(10)^19 == 1e19\ntrue\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.fma","page":"Mathematics","title":"Base.fma","text":"fma(x, y, z)\n\nComputes x*y+z without rounding the intermediate result x*y. On some systems this is significantly more expensive than x*y+z. fma is used to improve accuracy in certain algorithms. See muladd.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.muladd","page":"Mathematics","title":"Base.muladd","text":"muladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged with each other or with surrounding operations for performance. For example, this may be implemented as an fma if the hardware supports it efficiently. The result can be different on different machines and can also be different on the same machine due to constant propagation or other optimizations. See fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\nmuladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication. The result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.inv-Tuple{Number}","page":"Mathematics","title":"Base.inv","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x yields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for some types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.div","page":"Mathematics","title":"Base.div","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean (integer) division. Generally equivalent to a mathematical operation x/y without a fractional part.\n\nSee also: cld, fld, rem, divrem.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\njulia> div.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  0  0  1  1  1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.div-Tuple{Any, Any, RoundingMode}","page":"Mathematics","title":"Base.div","text":"div(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient from Euclidean (integer) division. Computes x / y, rounded to an integer according to the rounding mode r. In other words, the quantity\n\nround(x / y, r)\n\nwithout any intermediate rounding.\n\ncompat: Julia 1.4\nThe three-argument method taking a RoundingMode requires Julia 1.4 or later.\n\nSee also fld and cld, which are special cases of this function.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> div(4, 3, RoundToZero) # Matches div(4, 3)\n1\njulia> div(4, 3, RoundDown) # Matches fld(4, 3)\n1\njulia> div(4, 3, RoundUp) # Matches cld(4, 3)\n2\njulia> div(5, 2, RoundNearest)\n2\njulia> div(5, 2, RoundNearestTiesAway)\n3\njulia> div(-5, 2, RoundNearest)\n-2\njulia> div(-5, 2, RoundNearestTiesAway)\n-3\njulia> div(-5, 2, RoundNearestTiesUp)\n-2\njulia> div(4, 3, RoundFromZero)\n2\njulia> div(-4, 3, RoundFromZero)\n-2\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.fld","page":"Mathematics","title":"Base.fld","text":"fld(x, y)\n\nLargest integer less than or equal to x / y. Equivalent to div(x, y, RoundDown).\n\nSee also div, cld, fld1.\n\nExamples\n\njulia> fld(7.3, 5.5)\n1.0\n\njulia> fld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -2  -1  -1  -1  0  0  0  1  1  1\n\nBecause fld(x, y) implements strictly correct floored rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> fld(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but fld(6.0, 0.1) always takes the floor of the true value, so the result is 59.0.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.cld","page":"Mathematics","title":"Base.cld","text":"cld(x, y)\n\nSmallest integer larger than or equal to x / y. Equivalent to div(x, y, RoundUp).\n\nSee also div, fld.\n\nExamples\n\njulia> cld(5.5, 2.2)\n3.0\n\njulia> cld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  1  1  1  2  2\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.mod","page":"Mathematics","title":"Base.mod","text":"mod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x  y (mod n), where n = length(r), i.e. y = mod(x - first(r), n) + first(r).\n\nSee also mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))  # mod1(0, 3)\n3\n\njulia> mod(3, 0:2)  # mod(3, 3)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored division by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y, and magnitude less than abs(y) (with some exceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the type, and so rounding error may occur. In particular, if the exact result is very close to y, then it may be rounded to y.\n\nSee also: rem, div, fld, mod1, invmod.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\njulia> mod.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  0  1  2  0  1  2  0  1  2\n\n\n\n\n\nrem(x::Integer, T::Type{<:Integer}) -> T\nmod(x::Integer, T::Type{<:Integer}) -> T\n%(x::Integer, T::Type{<:Integer}) -> T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable in T, and y is an integer in [typemin(T),typemax(T)]. If T can represent any integer (e.g. T == BigInt), then this operation corresponds to a conversion to T.\n\nExamples\n\njulia> x = 129 % Int8\n-127\n\njulia> typeof(x)\nInt8\n\njulia> x = 129 % BigInt\n129\n\njulia> typeof(x)\nBigInt\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.rem","page":"Mathematics","title":"Base.rem","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in magnitude than y. This value is always exact.\n\nSee also: div, mod, mod1, divrem.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\njulia> rem.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -1  0  -2  -1  0  1  2  0  1  2\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.rem-Tuple{Any, Any, RoundingMode}","page":"Mathematics","title":"Base.rem","text":"rem(x, y, r::RoundingMode=RoundToZero)\n\nCompute the remainder of x after integer division by y, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - y * round(x / y, r)\n\nwithout any intermediate rounding.\n\nif r == RoundNearest, then the result is exact, and in the interval -y  2 y  2. See also RoundNearest.\nif r == RoundToZero (default), then the result is exact, and in the interval 0 y) if x is positive, or (-y 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 y) if y is positive, or (y 0 otherwise. The result may not be exact if x and y have different signs, and abs(x) < abs(y). See also RoundDown.\nif r == RoundUp, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundUp.\nif r == RoundFromZero, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundFromZero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> x = 9; y = 4;\n\njulia> x % y  # same as rem(x, y)\n1\n\njulia> x ÷ y  # same as div(x, y)\n2\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.rem2pi","page":"Mathematics","title":"Base.Math.rem2pi","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of 2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally be the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,. or -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π. See also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0. See also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.mod2pi","page":"Mathematics","title":"Base.Math.mod2pi","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by numerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would compute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may be less than 2π. For example, the expression mod2pi(2π) will not return 0, because the intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See rem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.divrem","page":"Mathematics","title":"Base.divrem","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division. Equivalent to (div(x, y, r), rem(x, y, r)). Equivalently, with the default value of r, this call is equivalent to (x ÷ y, x % y).\n\nSee also: fldmod, cld.\n\nExamples\n\njulia> divrem(3, 7)\n(0, 3)\n\njulia> divrem(7, 3)\n(2, 1)\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.fldmod","page":"Mathematics","title":"Base.fldmod","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for divrem(x, y, RoundDown). Equivalent to (fld(x, y), mod(x, y)).\n\nSee also: fld, cld, fldmod1.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.fld1","page":"Mathematics","title":"Base.fld1","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.mod1","page":"Mathematics","title":"Base.mod1","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y) in the range (0 y for positive y and in the range y0) for negative y.\n\nWith integer arguments and positive y, this is equal to mod(x, 1:y), and hence natural for 1-based indexing. By comparison, mod(x, y) == mod(x, 0:y-1) is natural for computations with offsets or strides.\n\nSee also mod, fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  3  1  2  3  1  2  3  1  2\n\njulia> mod1.([-0.1, 0, 0.1, 1, 2, 2.9, 3, 3.1]', 3)\n1×8 Matrix{Float64}:\n 2.9  3.0  0.1  1.0  2.0  2.9  3.0  0.1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.fldmod1","page":"Mathematics","title":"Base.fldmod1","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also fld1, mod1.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.://","page":"Mathematics","title":"Base.://","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result. More generally, // can be used for exact rational division of other numeric types with integer or rational components, such as complex numbers with integer components.\n\nNote that floating-point (AbstractFloat) arguments are not permitted by // (even if the values are rational). The arguments must be subtypes of Integer, Rational, or composites thereof.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\njulia> (1+2im) // (3+4im)\n11//25 + 2//25*im\n\njulia> 1.0 // 2\nERROR: MethodError: no method matching //(::Float64, ::Int64)\n[...]\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.rationalize","page":"Mathematics","title":"Base.rationalize","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components of the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.numerator","page":"Mathematics","title":"Base.numerator","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.denominator","page":"Mathematics","title":"Base.denominator","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:<<","page":"Mathematics","title":"Base.:<<","text":"<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left by n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0, this is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>, exp2, ldexp.\n\n\n\n\n\n<<(B::BitVector, n) -> BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B with elements shifted n positions backwards, filling with false values. If n < 0, elements are shifted forwards. Equivalent to B >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:>>","page":"Mathematics","title":"Base.:>>","text":">>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted right by n bits, filling with 0s if x >= 0, 1s if x < 0, preserving the sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is equivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n>>(B::BitVector, n) -> BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B with elements shifted n positions forward, filling with false values. If n < 0, elements are shifted backwards. Equivalent to B << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:>>>","page":"Mathematics","title":"Base.:>>>","text":">>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x shifted right by n bits, filling with 0s. For n < 0, this is equivalent to x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For Signed integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this is equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n>>>(B::BitVector, n) -> BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for details and examples.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.bitrotate","page":"Mathematics","title":"Base.bitrotate","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation. It returns the value of x with its bits rotated left k times. A negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\nSee also: <<, circshift, BitArray.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.::","page":"Mathematics","title":"Base.::","text":":expr\n\nQuote an expression expr, returning the abstract syntax tree (AST) of expr. The AST may be of type Expr, Symbol, or a literal value. The syntax :identifier evaluates to a Symbol.\n\nSee also: Expr, Symbol, Meta.parse\n\nExamples\n\njulia> expr = :(a = b + 2*x)\n:(a = b + 2x)\n\njulia> sym = :some_identifier\n:some_identifier\n\njulia> value = :0xff\n0xff\n\njulia> typeof((expr, sym, value))\nTuple{Expr, Symbol, UInt8}\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.range","page":"Mathematics","title":"Base.range","text":"range(start, stop, length)\nrange(start, stop; length, step)\nrange(start; length, stop, step)\nrange(;start, length, stop, step)\n\nConstruct a specialized array with evenly spaced elements and optimized storage (an AbstractRange) from the arguments. Mathematically a range is uniquely determined by any three of start, step, stop and length. Valid invocations of range are:\n\nCall range with any three of start, step, stop, length.\nCall range with two of start, stop, length. In this case step will be assumed to be one. If both arguments are Integers, a UnitRange will be returned.\nCall range with one of stop or length. start and step will be assumed to be one.\n\nSee Extended Help for additional details on the returned type. See also logrange for logarithmically spaced points.\n\nExamples\n\njulia> range(1, length=100)\n1:100\n\njulia> range(1, stop=100)\n1:100\n\njulia> range(1, step=5, length=100)\n1:5:496\n\njulia> range(1, step=5, stop=100)\n1:5:96\n\njulia> range(1, 10, length=101)\n1.0:0.09:10.0\n\njulia> range(1, 100, step=5)\n1:5:96\n\njulia> range(stop=10, length=5)\n6:10\n\njulia> range(stop=10, step=1, length=5)\n6:1:10\n\njulia> range(start=1, step=1, stop=10)\n1:1:10\n\njulia> range(; length = 10)\nBase.OneTo(10)\n\njulia> range(; stop = 6)\nBase.OneTo(6)\n\njulia> range(; stop = 6.5)\n1.0:1.0:6.0\n\nIf length is not specified and stop - start is not an integer multiple of step, a range that ends before stop will be produced.\n\njulia> range(1, 3.5, step=2)\n1.0:2.0:3.0\n\nSpecial care is taken to ensure intermediate values are computed rationally. To avoid this induced overhead, see the LinRange constructor.\n\ncompat: Julia 1.1\nstop as a positional argument requires at least Julia 1.1.\n\ncompat: Julia 1.7\nThe versions without keyword arguments and start as a keyword argument require at least Julia 1.7.\n\ncompat: Julia 1.8\nThe versions with stop as a sole keyword argument, or length as a sole keyword argument require at least Julia 1.8.\n\nExtended Help\n\nrange will produce a Base.OneTo when the arguments are Integers and\n\nOnly length is provided\nOnly stop is provided\n\nrange will produce a UnitRange when the arguments are Integers and\n\nOnly start  and stop are provided\nOnly length and stop are provided\n\nA UnitRange is not produced if step is provided even if specified as one.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.OneTo","page":"Mathematics","title":"Base.OneTo","text":"Base.OneTo(n)\n\nDefine an AbstractUnitRange that behaves like 1:n, with the added distinction that the lower limit is guaranteed (by the type system) to be 1.\n\n\n\n\n\n","category":"type"},{"location":"base/math.html#Base.StepRangeLen","page":"Mathematics","title":"Base.StepRangeLen","text":"StepRangeLen(         ref::R, step::S, len, [offset=1]) where {  R,S}\nStepRangeLen{T,R,S}(  ref::R, step::S, len, [offset=1]) where {T,R,S}\nStepRangeLen{T,R,S,L}(ref::R, step::S, len, [offset=1]) where {T,R,S,L}\n\nA range r where r[i] produces values of type T (in the first form, T is deduced automatically), parameterized by a reference value, a step, and the length. By default ref is the starting value r[1], but alternatively you can supply it as the value of r[offset] for some other index 1 <= offset <= len. The syntax a:b or a:b:c, where any of a, b, or c are floating-point numbers, creates a StepRangeLen.\n\ncompat: Julia 1.7\nThe 4th type parameter L requires at least Julia 1.7.\n\n\n\n\n\n","category":"type"},{"location":"base/math.html#Base.logrange","page":"Mathematics","title":"Base.logrange","text":"logrange(start, stop, length)\nlogrange(start, stop; length)\n\nConstruct a specialized array whose elements are spaced logarithmically between the given endpoints. That is, the ratio of successive elements is a constant, calculated from the length.\n\nThis is similar to geomspace in Python. Unlike PowerRange in Mathematica, you specify the number of elements not the ratio. Unlike logspace in Python and Matlab, the start and stop arguments are always the first and last elements of the result, not powers applied to some base.\n\nExamples\n\njulia> logrange(10, 4000, length=3)\n3-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 10.0, 200.0, 4000.0\n\njulia> ans[2] ≈ sqrt(10 * 4000)  # middle element is the geometric mean\ntrue\n\njulia> range(10, 40, length=3)[2] ≈ (10 + 40)/2  # arithmetic mean\ntrue\n\njulia> logrange(1f0, 32f0, 11)\n11-element Base.LogRange{Float32, Float64}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0, 5.65685, 8.0, 11.3137, 16.0, 22.6274, 32.0\n\njulia> logrange(1, 1000, length=4) ≈ 10 .^ (0:3)\ntrue\n\nSee the LogRange type for further details.\n\nSee also range for linearly spaced points.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.LogRange","page":"Mathematics","title":"Base.LogRange","text":"LogRange{T}(start, stop, len) <: AbstractVector{T}\n\nA range whose elements are spaced logarithmically between start and stop, with spacing controlled by len. Returned by logrange.\n\nLike LinRange, the first and last elements will be exactly those provided, but intermediate values may have small floating-point errors. These are calculated using the logs of the endpoints, which are stored on construction, often in higher precision than T.\n\nExamples\n\njulia> logrange(1, 4, length=5)\n5-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0\n\njulia> Base.LogRange{Float16}(1, 4, 5)\n5-element Base.LogRange{Float16, Float64}:\n 1.0, 1.414, 2.0, 2.828, 4.0\n\njulia> logrange(1e-310, 1e-300, 11)[1:2:end]\n6-element Vector{Float64}:\n 1.0e-310\n 9.999999999999974e-309\n 9.999999999999981e-307\n 9.999999999999988e-305\n 9.999999999999994e-303\n 1.0e-300\n\njulia> prevfloat(1e-308, 5) == ans[2]\ntrue\n\nNote that integer eltype T is not allowed. Use for instance round.(Int, xs), or explicit powers of some integer base:\n\njulia> xs = logrange(1, 512, 4)\n4-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 8.0, 64.0, 512.0\n\njulia> 2 .^ (0:3:9) |> println\n[1, 8, 64, 512]\n\ncompat: Julia 1.11\nThis type requires at least Julia 1.11.\n\n\n\n\n\n","category":"type"},{"location":"base/math.html#Base.:==","page":"Mathematics","title":"Base.:==","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===. Should be implemented for all types with a notion of equality, based on the abstract value that an instance represents. For example, all numeric types are compared by numeric value, ignoring type. Strings are compared as sequences of characters, ignoring encoding. Collections of the same type generally compare their key sets, and if those are ==, then compare the values for each of those keys, returning true if all such pairs are ==. Other properties are typically not taken into account (such as the exact type).\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and NaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing, in which case missing is returned (three-valued logic). Collections generally implement three-valued logic akin to all, returning missing if any operands contain missing values and all other pairs are equal. Use isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and handle comparison to other types via promotion rules where possible.\n\nisequal falls back to ==, so new methods of == will be used by the Dict type to compare keys. If your type will be used as a dictionary key, it should therefore also implement hash.\n\nIf some type defines ==, isequal, and isless then it should also implement < to ensure consistency of comparisons.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:!=","page":"Mathematics","title":"Base.:!=","text":"!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition !=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n!=(x)\n\nCreate a function that compares its argument to x using !=, i.e. a function equivalent to y -> y != x. The returned function is of type Base.Fix2{typeof(!=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:!==","page":"Mathematics","title":"Base.:!==","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:<","page":"Mathematics","title":"Base.:<","text":"<(x, y)\n\nLess-than comparison operator. Falls back to isless. Because of the behavior of floating-point NaN values, this operator implements a partial order.\n\nImplementation\n\nNew types with a canonical partial order should implement this function for two arguments of the new type. Types with a canonical total order should implement isless instead.\n\nSee also isunordered.\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n<(x)\n\nCreate a function that compares its argument to x using <, i.e. a function equivalent to y -> y < x. The returned function is of type Base.Fix2{typeof(<)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:<=","page":"Mathematics","title":"Base.:<=","text":"<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n<=(x)\n\nCreate a function that compares its argument to x using <=, i.e. a function equivalent to y -> y <= x. The returned function is of type Base.Fix2{typeof(<=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:>","page":"Mathematics","title":"Base.:>","text":">(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function, and rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n>(x)\n\nCreate a function that compares its argument to x using >, i.e. a function equivalent to y -> y > x. The returned function is of type Base.Fix2{typeof(>)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:>=","page":"Mathematics","title":"Base.:>=","text":">=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n>=(x)\n\nCreate a function that compares its argument to x using >=, i.e. a function equivalent to y -> y >= x. The returned function is of type Base.Fix2{typeof(>=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.cmp","page":"Mathematics","title":"Base.cmp","text":"cmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(a::AbstractString, b::AbstractString) -> Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:~","page":"Mathematics","title":"Base.:~","text":"~(x)\n\nBitwise not.\n\nSee also: !, &, |.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:&","page":"Mathematics","title":"Base.:&","text":"x & y\n\nBitwise and. Implements three-valued logic, returning missing if one operand is missing and the other is true. Add parentheses for function application form: (&)(x, y).\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:|","page":"Mathematics","title":"Base.:|","text":"x | y\n\nBitwise or. Implements three-valued logic, returning missing if one operand is missing and the other is false.\n\nSee also: &, xor, ||.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.xor","page":"Mathematics","title":"Base.xor","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and ⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.nand","page":"Mathematics","title":"Base.nand","text":"nand(x, y)\n⊼(x, y)\n\nBitwise nand (not and) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊼ b is a synonym for nand(a,b), and ⊼ can be typed by tab-completing \\nand or \\barwedge in the Julia REPL.\n\nExamples\n\njulia> nand(true, false)\ntrue\n\njulia> nand(true, true)\nfalse\n\njulia> nand(true, missing)\nmissing\n\njulia> false ⊼ false\ntrue\n\njulia> [true; true; false] .⊼ [true; false; false]\n3-element BitVector:\n 0\n 1\n 1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.nor","page":"Mathematics","title":"Base.nor","text":"nor(x, y)\n⊽(x, y)\n\nBitwise nor (not or) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing and the other is not true.\n\nThe infix operation a ⊽ b is a synonym for nor(a,b), and ⊽ can be typed by tab-completing \\nor or \\barvee in the Julia REPL.\n\nExamples\n\njulia> nor(true, false)\nfalse\n\njulia> nor(true, true)\nfalse\n\njulia> nor(true, missing)\nfalse\n\njulia> false ⊽ false\ntrue\n\njulia> false ⊽ missing\nmissing\n\njulia> [true; true; false] .⊽ [true; false; false]\n3-element BitVector:\n 0\n 0\n 1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.:!","page":"Mathematics","title":"Base.:!","text":"!(x)\n\nBoolean not. Implements three-valued logic, returning missing if x is missing.\n\nSee also ~ for bitwise not.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a composed function which computes the boolean negation of f.\n\nSee also ∘.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\ncompat: Julia 1.9\nStarting with Julia 1.9, !f returns a ComposedFunction instead of an anonymous function.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#&&","page":"Mathematics","title":"&&","text":"x && y\n\nShort-circuiting boolean AND.\n\nSee also &, the ternary operator ? :, and the manual section on control flow.\n\nExamples\n\njulia> x = 3;\n\njulia> x > 1 && x < 10 && x isa Int\ntrue\n\njulia> x < 0 && error(\"expected positive x\")\nfalse\n\n\n\n\n\n","category":"keyword"},{"location":"base/math.html#||","page":"Mathematics","title":"||","text":"x || y\n\nShort-circuiting boolean OR.\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> pi < 3 || ℯ < 3\ntrue\n\njulia> false || true || println(\"neither is true!\")\ntrue\n\n\n\n\n\n","category":"keyword"},{"location":"base/math.html#Mathematical-Functions","page":"Mathematics","title":"Mathematical Functions","text":"","category":"section"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"Base.isapprox\nBase.sin(::Number)\nBase.cos(::Number)\nBase.sincos(::Float64)\nBase.tan(::Number)\nBase.Math.sind\nBase.Math.cosd\nBase.Math.tand\nBase.Math.sincosd\nBase.Math.sinpi\nBase.Math.cospi\nBase.Math.tanpi\nBase.Math.sincospi\nBase.sinh(::Number)\nBase.cosh(::Number)\nBase.tanh(::Number)\nBase.asin(::Number)\nBase.acos(::Number)\nBase.atan(::Number)\nBase.Math.asind\nBase.Math.acosd\nBase.Math.atand\nBase.Math.sec(::Number)\nBase.Math.csc(::Number)\nBase.Math.cot(::Number)\nBase.Math.secd\nBase.Math.cscd\nBase.Math.cotd\nBase.Math.asec(::Number)\nBase.Math.acsc(::Number)\nBase.Math.acot(::Number)\nBase.Math.asecd\nBase.Math.acscd\nBase.Math.acotd\nBase.Math.sech(::Number)\nBase.Math.csch(::Number)\nBase.Math.coth(::Number)\nBase.asinh(::Number)\nBase.acosh(::Number)\nBase.atanh(::Number)\nBase.Math.asech(::Number)\nBase.Math.acsch(::Number)\nBase.Math.acoth(::Number)\nBase.Math.sinc\nBase.Math.cosc\nBase.Math.deg2rad\nBase.Math.rad2deg\nBase.Math.hypot\nBase.log(::Number)\nBase.log(::Number, ::Number)\nBase.log2\nBase.log10\nBase.log1p\nBase.Math.frexp\nBase.exp(::Float64)\nBase.exp2\nBase.exp10\nBase.Math.ldexp\nBase.Math.modf\nBase.expm1\nBase.round\nBase.Rounding.RoundingMode\nBase.Rounding.RoundNearest\nBase.Rounding.RoundNearestTiesAway\nBase.Rounding.RoundNearestTiesUp\nBase.Rounding.RoundToZero\nBase.Rounding.RoundFromZero\nBase.Rounding.RoundUp\nBase.Rounding.RoundDown\nBase.round(::Complex{<: AbstractFloat}, ::RoundingMode, ::RoundingMode)\nBase.ceil\nBase.floor\nBase.trunc\nBase.unsafe_trunc\nBase.min\nBase.max\nBase.minmax\nBase.Math.clamp\nBase.Math.clamp!\nBase.abs\nBase.Checked\nBase.Checked.checked_abs\nBase.Checked.checked_neg\nBase.Checked.checked_add\nBase.Checked.checked_sub\nBase.Checked.checked_mul\nBase.Checked.checked_div\nBase.Checked.checked_rem\nBase.Checked.checked_fld\nBase.Checked.checked_mod\nBase.Checked.checked_cld\nBase.Checked.checked_pow\nBase.Checked.add_with_overflow\nBase.Checked.sub_with_overflow\nBase.Checked.mul_with_overflow\nBase.abs2\nBase.copysign\nBase.sign\nBase.signbit\nBase.flipsign\nBase.sqrt(::Number)\nBase.isqrt\nBase.Math.cbrt(::AbstractFloat)\nBase.real\nBase.imag\nBase.reim\nBase.conj\nBase.angle\nBase.cis\nBase.cispi\nBase.binomial\nBase.factorial\nBase.gcd\nBase.lcm\nBase.gcdx\nBase.ispow2\nBase.nextpow\nBase.prevpow\nBase.nextprod\nBase.invmod\nBase.powermod\nBase.ndigits\nBase.add_sum\nBase.widemul\nBase.Math.evalpoly\nBase.Math.@evalpoly\nBase.FastMath.@fastmath","category":"page"},{"location":"base/math.html#Base.isapprox","page":"Mathematics","title":"Base.isapprox","text":"isapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison. Two numbers compare equal if their relative distance or their absolute distance is within tolerance bounds: isapprox returns true if norm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol (absolute tolerance) is zero and the default rtol (relative tolerance) depends on the types of x and y. The keyword argument nans determines whether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to the square root of eps of the type of x or y, whichever is bigger (least precise). This corresponds to requiring equality of about half of the significant digits. Otherwise, e.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for arrays (where an alternative norm choice is sometimes useful). When x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf or NaN), the comparison falls back to checking whether all elements of x and y are approximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y is equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is equivalent to x == 0 since the default atol is 0.  In such cases, you should either supply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g. use x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol automatically because it depends on the overall scaling (the \"units\") of your problem: for example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the radius of the Earth in meters, but an absurdly large tolerance if x is the radius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments requires Julia 1.6 or later.\n\nExamples\n\njulia> isapprox(0.1, 0.15; atol=0.05)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.34)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.33)\nfalse\n\njulia> 0.1 + 1e-10 ≈ 0.1\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0]) # using `norm`\ntrue\n\n\n\n\n\nisapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\ncompat: Julia 1.5\nThis method requires Julia 1.5 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.sin-Tuple{Number}","page":"Mathematics","title":"Base.sin","text":"sin(x)\n\nCompute sine of x, where x is in radians.\n\nSee also sind, sinpi, sincos, cis, asin.\n\nExamples\n\njulia> round.(sin.(range(0, 2pi, length=9)'), digits=3)\n1×9 Matrix{Float64}:\n 0.0  0.707  1.0  0.707  0.0  -0.707  -1.0  -0.707  -0.0\n\njulia> sind(45)\n0.7071067811865476\n\njulia> sinpi(1/4)\n0.7071067811865475\n\njulia> round.(sincos(pi/6), digits=3)\n(0.5, 0.866)\n\njulia> round(cis(pi/6), digits=3)\n0.866 + 0.5im\n\njulia> round(exp(im*pi/6), digits=3)\n0.866 + 0.5im\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.cos-Tuple{Number}","page":"Mathematics","title":"Base.cos","text":"cos(x)\n\nCompute cosine of x, where x is in radians.\n\nSee also cosd, cospi, sincos, cis.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.sincos-Tuple{Float64}","page":"Mathematics","title":"Base.Math.sincos","text":"sincos(x)\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning a tuple (sine, cosine).\n\nSee also cis, sincospi, sincosd.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.tan-Tuple{Number}","page":"Mathematics","title":"Base.tan","text":"tan(x)\n\nCompute tangent of x, where x is in radians.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.sind","page":"Mathematics","title":"Base.Math.sind","text":"sind(x)\n\nCompute sine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cosd","page":"Mathematics","title":"Base.Math.cosd","text":"cosd(x)\n\nCompute cosine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.tand","page":"Mathematics","title":"Base.Math.tand","text":"tand(x)\n\nCompute tangent of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.sincosd","page":"Mathematics","title":"Base.Math.sincosd","text":"sincosd(x)\n\nSimultaneously compute the sine and cosine of x, where x is in degrees.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.sinpi","page":"Mathematics","title":"Base.Math.sinpi","text":"sinpi(x)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\nSee also sind, cospi, sincospi.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cospi","page":"Mathematics","title":"Base.Math.cospi","text":"cospi(x)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.tanpi","page":"Mathematics","title":"Base.Math.tanpi","text":"tanpi(x)\n\nCompute tan(pi x) more accurately than tan(pi*x), especially for large x.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also tand, sinpi, cospi, sincospi.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.sincospi","page":"Mathematics","title":"Base.Math.sincospi","text":"sincospi(x)\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x, where x is in radians), returning a tuple (sine, cosine).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.sinh-Tuple{Number}","page":"Mathematics","title":"Base.sinh","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.cosh-Tuple{Number}","page":"Mathematics","title":"Base.cosh","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.tanh-Tuple{Number}","page":"Mathematics","title":"Base.tanh","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\nSee also tan, atanh.\n\nExamples\n\njulia> tanh.(-3:3f0)  # Here 3f0 isa Float32\n7-element Vector{Float32}:\n -0.9950548\n -0.9640276\n -0.7615942\n  0.0\n  0.7615942\n  0.9640276\n  0.9950548\n\njulia> tan.(im .* (1:3))\n3-element Vector{ComplexF64}:\n 0.0 + 0.7615941559557649im\n 0.0 + 0.9640275800758169im\n 0.0 + 0.9950547536867306im\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.asin-Tuple{Number}","page":"Mathematics","title":"Base.asin","text":"asin(x)\n\nCompute the inverse sine of x, where the output is in radians.\n\nSee also asind for output in degrees.\n\nExamples\n\njulia> asin.((0, 1/2, 1))\n(0.0, 0.5235987755982989, 1.5707963267948966)\n\njulia> asind.((0, 1/2, 1))\n(0.0, 30.000000000000004, 90.0)\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.acos-Tuple{Number}","page":"Mathematics","title":"Base.acos","text":"acos(x)\n\nCompute the inverse cosine of x, where the output is in radians\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.atan-Tuple{Number}","page":"Mathematics","title":"Base.atan","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one real argument, this is the angle in radians between the positive x-axis and the point (1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the point (x, y), returning a value in the interval -pi pi. This corresponds to a standard atan2 function. Note that by convention atan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\nSee also atand for degrees.\n\nExamples\n\njulia> rad2deg(atan(-1/√3))\n-30.000000000000004\n\njulia> rad2deg(atan(-1, √3))\n-30.000000000000004\n\njulia> rad2deg(atan(1, -√3))\n150.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.asind","page":"Mathematics","title":"Base.Math.asind","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.acosd","page":"Mathematics","title":"Base.Math.acosd","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.atand","page":"Mathematics","title":"Base.Math.atand","text":"atand(y)\natand(y,x)\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\ncompat: Julia 1.7\nThe one-argument method supports square matrix arguments as of Julia 1.7.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.sec-Tuple{Number}","page":"Mathematics","title":"Base.Math.sec","text":"sec(x)\n\nCompute the secant of x, where x is in radians.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.csc-Tuple{Number}","page":"Mathematics","title":"Base.Math.csc","text":"csc(x)\n\nCompute the cosecant of x, where x is in radians.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.cot-Tuple{Number}","page":"Mathematics","title":"Base.Math.cot","text":"cot(x)\n\nCompute the cotangent of x, where x is in radians.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.secd","page":"Mathematics","title":"Base.Math.secd","text":"secd(x)\n\nCompute the secant of x, where x is in degrees.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cscd","page":"Mathematics","title":"Base.Math.cscd","text":"cscd(x)\n\nCompute the cosecant of x, where x is in degrees.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cotd","page":"Mathematics","title":"Base.Math.cotd","text":"cotd(x)\n\nCompute the cotangent of x, where x is in degrees.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.asec-Tuple{Number}","page":"Mathematics","title":"Base.Math.asec","text":"asec(x)\n\nCompute the inverse secant of x, where the output is in radians. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.acsc-Tuple{Number}","page":"Mathematics","title":"Base.Math.acsc","text":"acsc(x)\n\nCompute the inverse cosecant of x, where the output is in radians. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.acot-Tuple{Number}","page":"Mathematics","title":"Base.Math.acot","text":"acot(x)\n\nCompute the inverse cotangent of x, where the output is in radians. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.asecd","page":"Mathematics","title":"Base.Math.asecd","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.acscd","page":"Mathematics","title":"Base.Math.acscd","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.acotd","page":"Mathematics","title":"Base.Math.acotd","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.sech-Tuple{Number}","page":"Mathematics","title":"Base.Math.sech","text":"sech(x)\n\nCompute the hyperbolic secant of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.csch-Tuple{Number}","page":"Mathematics","title":"Base.Math.csch","text":"csch(x)\n\nCompute the hyperbolic cosecant of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.coth-Tuple{Number}","page":"Mathematics","title":"Base.Math.coth","text":"coth(x)\n\nCompute the hyperbolic cotangent of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.asinh-Tuple{Number}","page":"Mathematics","title":"Base.asinh","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.acosh-Tuple{Number}","page":"Mathematics","title":"Base.acosh","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.atanh-Tuple{Number}","page":"Mathematics","title":"Base.atanh","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.asech-Tuple{Number}","page":"Mathematics","title":"Base.Math.asech","text":"asech(x)\n\nCompute the inverse hyperbolic secant of x. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.acsch-Tuple{Number}","page":"Mathematics","title":"Base.Math.acsch","text":"acsch(x)\n\nCompute the inverse hyperbolic cosecant of x. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.acoth-Tuple{Number}","page":"Mathematics","title":"Base.Math.acoth","text":"acoth(x)\n\nCompute the inverse hyperbolic cotangent of x. \n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.Math.sinc","page":"Mathematics","title":"Base.Math.sinc","text":"sinc(x)\n\nCompute normalized sinc function operatornamesinc(x) = sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\nSee also cosc, its derivative.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cosc","page":"Mathematics","title":"Base.Math.cosc","text":"cosc(x)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if x = 0. This is the derivative of sinc(x).\n\nSee also sinc.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.deg2rad","page":"Mathematics","title":"Base.Math.deg2rad","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nSee also rad2deg, sind, pi.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.rad2deg","page":"Mathematics","title":"Base.Math.rad2deg","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nSee also deg2rad.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.hypot","page":"Mathematics","title":"Base.Math.hypot","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in: An Improved Algorithm for hypot(a,b) by Carlos F. Borges The article is available online at arXiv at the link   https://arxiv.org/abs/1904.09481\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nSee also norm in the LinearAlgebra standard library.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\njulia> using LinearAlgebra\n\njulia> norm([a, a, a, a]) == hypot(a, a, a, a)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.log-Tuple{Number}","page":"Mathematics","title":"Base.log","text":"log(x)\n\nCompute the natural logarithm of x.\n\nThrows DomainError for negative Real arguments. Use complex arguments to obtain complex results. Has a branch cut along the negative real axis, for which -0.0im is taken to be below the axis.\n\nSee also ℯ, log1p, log2, log10.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(-3 + 0im)\n1.0986122886681098 + 3.141592653589793im\n\njulia> log(-3 - 0.0im)\n1.0986122886681098 - 3.141592653589793im\n\njulia> log.(exp.(-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.log-Tuple{Number, Number}","page":"Mathematics","title":"Base.log","text":"log(b,x)\n\nCompute the base b logarithm of x. Throws DomainError for negative Real arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will typically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.log2","page":"Mathematics","title":"Base.log2","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throws DomainError for negative Real arguments.\n\nSee also: exp2, ldexp, ispow2.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nlog2 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log2(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\njulia> log2.(2.0 .^ (-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.log10","page":"Mathematics","title":"Base.log10","text":"log10(x)\n\nCompute the logarithm of x to base 10. Throws DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nlog10 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log10(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.log1p","page":"Mathematics","title":"Base.log1p","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throws DomainError for Real arguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p was called with a real argument < -1 but will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.frexp","page":"Mathematics","title":"Base.Math.frexp","text":"frexp(val)\n\nReturn (x,exp) such that x has a magnitude in the interval 12 1) or 0, and val is equal to x times 2^exp.\n\nSee also significand, exponent, ldexp.\n\nExamples\n\njulia> frexp(6.0)\n(0.75, 3)\n\njulia> significand(6.0), exponent(6.0)  # interval [1, 2) instead\n(1.5, 2)\n\njulia> frexp(0.0), frexp(NaN), frexp(-Inf)  # exponent would give an error\n((0.0, 0), (NaN, 0), (-Inf, 0))\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.exp-Tuple{Float64}","page":"Mathematics","title":"Base.exp","text":"exp(x)\n\nCompute the natural base exponential of x, in other words ℯ^x.\n\nSee also exp2, exp10 and cis.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\njulia> exp(im * pi) ≈ cis(pi)\ntrue\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.exp2","page":"Mathematics","title":"Base.exp2","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nSee also ldexp, <<.\n\nExamples\n\njulia> exp2(5)\n32.0\n\njulia> 2^5\n32\n\njulia> exp2(63) > typemax(Int)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.exp10","page":"Mathematics","title":"Base.exp10","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\njulia> 10^2\n100\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.ldexp","page":"Mathematics","title":"Base.Math.ldexp","text":"ldexp(x, n)\n\nCompute x times 2^n.\n\nSee also frexp, exponent.\n\nExamples\n\njulia> ldexp(5.0, 2)\n20.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.modf","page":"Mathematics","title":"Base.Math.modf","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts have the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.expm1","page":"Mathematics","title":"Base.expm1","text":"expm1(x)\n\nAccurately compute e^x-1. It avoids the loss of precision involved in the direct evaluation of exp(x)-1 for small values of x.\n\nExamples\n\njulia> expm1(1e-16)\n1.0e-16\n\njulia> exp(1e-16) - 1\n0.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.round","page":"Mathematics","title":"Base.round","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type T, or of the same type of x if no T is provided. An InexactError will be thrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits after the decimal place (or before if negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of significant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer. Note that round may give incorrect results if the global rounding mode is changed (see rounding).\n\nWhen rounding to a floating point type, will round to integers representable by that type (and Inf) rather than true integers. Inf is treated as one ulp greater than the floatmax(T) for purposes of determining \"nearest\", similar to convert.\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\njulia> round(Float16, typemax(UInt128))\nInf16\n\njulia> floor(Float16, typemax(UInt128))\nFloat16(6.55e4)\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when operating on binary floating point numbers. For example, the Float64 value represented by 1.15 is actually less than 1.15, yet will be rounded to 1.2. For example:julia> x = 1.15\n1.15\n\njulia> big(1.15)\n1.149999999999999911182158029987476766109466552734375\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Rounding.RoundingMode","page":"Mathematics","title":"Base.Rounding.RoundingMode","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via rounding/setrounding functions), or as optional arguments for rounding to the nearest integer (via the round function).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero\nRoundUp\nRoundDown\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\n\n\n\n\n","category":"type"},{"location":"base/math.html#Base.Rounding.RoundNearest","page":"Mathematics","title":"Base.Rounding.RoundNearest","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundNearestTiesAway","page":"Mathematics","title":"Base.Rounding.RoundNearestTiesAway","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++ round behaviour).\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundNearestTiesUp","page":"Mathematics","title":"Base.Rounding.RoundNearestTiesUp","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript round behaviour).\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundToZero","page":"Mathematics","title":"Base.Rounding.RoundToZero","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundFromZero","page":"Mathematics","title":"Base.Rounding.RoundFromZero","text":"RoundFromZero\n\nRounds away from zero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundUp","page":"Mathematics","title":"Base.Rounding.RoundUp","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.Rounding.RoundDown","page":"Mathematics","title":"Base.Rounding.RoundDown","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n","category":"constant"},{"location":"base/math.html#Base.round-Tuple{Complex{<:AbstractFloat}, RoundingMode, RoundingMode}","page":"Mathematics","title":"Base.round","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=0, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z, breaking ties using the specified RoundingModes. The first RoundingMode is used for rounding the real components while the second is used for rounding the imaginary components.\n\nRoundingModeReal and RoundingModeImaginary default to RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\nExamples\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\njulia> round(3.14 + 4.5im, RoundUp, RoundNearestTiesUp)\n4.0 + 5.0im\n\njulia> round(3.14159 + 4.512im; digits = 1)\n3.1 + 4.5im\n\njulia> round(3.14159 + 4.512im; sigdigits = 3)\n3.14 + 4.51im\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.ceil","page":"Mathematics","title":"Base.ceil","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or equal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the ceiled value is not representable as a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support ceil for a new type, define Base.round(x::NewType, ::RoundingMode{:Up}).\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.floor","page":"Mathematics","title":"Base.floor","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or equal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the floored value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support floor for a new type, define Base.round(x::NewType, ::RoundingMode{:Down}).\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.trunc","page":"Mathematics","title":"Base.trunc","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value is less than or equal to the absolute value of x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the truncated value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support trunc for a new type, define Base.round(x::NewType, ::RoundingMode{:ToZero}).\n\nSee also: %, floor, unsigned, unsafe_trunc.\n\nExamples\n\njulia> trunc(2.22)\n2.0\n\njulia> trunc(-2.22, digits=1)\n-2.2\n\njulia> trunc(Int, -2.22)\n-2\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.unsafe_trunc","page":"Mathematics","title":"Base.unsafe_trunc","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is less than or equal to the absolute value of x. If the value is not representable by T, an arbitrary value will be returned. See also trunc.\n\nExamples\n\njulia> unsafe_trunc(Int, -2.2)\n-2\n\njulia> unsafe_trunc(Int, NaN)\n-9223372036854775808\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.min","page":"Mathematics","title":"Base.min","text":"min(x, y, ...)\n\nReturn the minimum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\njulia> min(4, missing, 6)\nmissing\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.max","page":"Mathematics","title":"Base.max","text":"max(x, y, ...)\n\nReturn the maximum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\njulia> max(5, missing, 6)\nmissing\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.minmax","page":"Mathematics","title":"Base.minmax","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)).\n\nSee also extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.clamp","page":"Mathematics","title":"Base.Math.clamp","text":"clamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments are promoted to a common type.\n\nSee also clamp!, min, max.\n\ncompat: Julia 1.3\nmissing as the first argument requires at least Julia 1.3.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10)], 2.0, 9.0)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11, 8, 5], 10, 6)  # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nSee also trunc.\n\nExamples\n\njulia> clamp(200, Int8)\n127\n\njulia> clamp(-200, Int8)\n-128\n\njulia> trunc(Int, 4pi^2)\n39\n\n\n\n\n\nclamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.clamp!","page":"Mathematics","title":"Base.Math.clamp!","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place. See also clamp.\n\ncompat: Julia 1.3\nmissing entries in array require at least Julia 1.3.\n\nExamples\n\njulia> row = collect(-4:4)';\n\njulia> clamp!(row, 0, Inf)\n1×9 adjoint(::Vector{Int64}) with eltype Int64:\n 0  0  0  0  0  1  2  3  4\n\njulia> clamp.((-4:4)', 0, Inf)\n1×9 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0  1.0  2.0  3.0  4.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.abs","page":"Mathematics","title":"Base.abs","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur, resulting in the return of a negative value. This overflow occurs only when abs is applied to the minimum representable value of a signed integer. That is, when x == typemin(typeof(x)), abs(x) == x < 0, not -x as might be expected.\n\nSee also: abs2, unsigned, sign.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs.(Int8[-128 -127 -126 0 126 127])  # overflow at typemin(Int8)\n1×6 Matrix{Int8}:\n -128  127  126  0  126  127\n\njulia> maximum(abs, [1, -2, 3, -4])\n4\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked","page":"Mathematics","title":"Base.Checked","text":"Checked\n\nThe Checked module provides arithmetic functions for the built-in signed and unsigned Integer types which throw an error when an overflow occurs. They are named like checked_sub, checked_div, etc. In addition, add_with_overflow, sub_with_overflow, mul_with_overflow return both the unchecked results and a boolean value denoting the presence of an overflow.\n\n\n\n\n\n","category":"module"},{"location":"base/math.html#Base.Checked.checked_abs","page":"Mathematics","title":"Base.Checked.checked_abs","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_neg","page":"Mathematics","title":"Base.Checked.checked_neg","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_add","page":"Mathematics","title":"Base.Checked.checked_add","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_sub","page":"Mathematics","title":"Base.Checked.checked_sub","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_mul","page":"Mathematics","title":"Base.Checked.checked_mul","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_div","page":"Mathematics","title":"Base.Checked.checked_div","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_rem","page":"Mathematics","title":"Base.Checked.checked_rem","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_fld","page":"Mathematics","title":"Base.Checked.checked_fld","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_mod","page":"Mathematics","title":"Base.Checked.checked_mod","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_cld","page":"Mathematics","title":"Base.Checked.checked_cld","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.checked_pow","page":"Mathematics","title":"Base.Checked.checked_pow","text":"Base.checked_pow(x, y)\n\nCalculates ^(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.add_with_overflow","page":"Mathematics","title":"Base.Checked.add_with_overflow","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.sub_with_overflow","page":"Mathematics","title":"Base.Checked.sub_with_overflow","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Checked.mul_with_overflow","page":"Mathematics","title":"Base.Checked.mul_with_overflow","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.abs2","page":"Mathematics","title":"Base.abs2","text":"abs2(x)\n\nSquared absolute value of x.\n\nThis can be faster than abs(x)^2, especially for complex numbers where abs(x) requires a square root via hypot.\n\nSee also abs, conj, real.\n\nExamples\n\njulia> abs2(-3)\n9\n\njulia> abs2(3.0 + 4.0im)\n25.0\n\njulia> sum(abs2, [1+2im, 3+4im])  # LinearAlgebra.norm(x)^2\n30\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.copysign","page":"Mathematics","title":"Base.copysign","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.sign","page":"Mathematics","title":"Base.sign","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\nSee also signbit, zero, copysign, flipsign.\n\nExamples\n\njulia> sign(-4.0)\n-1.0\n\njulia> sign(99)\n1\n\njulia> sign(-0.0)\n-0.0\n\njulia> sign(0 + im)\n0.0 + 1.0im\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.signbit","page":"Mathematics","title":"Base.signbit","text":"signbit(x)\n\nReturn true if the value of the sign of x is negative, otherwise false.\n\nSee also sign and copysign.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.flipsign","page":"Mathematics","title":"Base.flipsign","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.sqrt-Tuple{Number}","page":"Mathematics","title":"Base.sqrt","text":"sqrt(x)\n\nReturn sqrtx.\n\nThrows DomainError for negative Real arguments. Use complex negative arguments instead. Note that sqrt has a branch cut along the negative real axis.\n\nThe prefix operator √ is equivalent to sqrt.\n\nSee also: hypot.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\njulia> sqrt(-81 - 0.0im)  # -0.0im is below the branch cut\n0.0 - 9.0im\n\njulia> .√(1:4)\n4-element Vector{Float64}:\n 1.0\n 1.4142135623730951\n 1.7320508075688772\n 2.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.isqrt","page":"Mathematics","title":"Base.isqrt","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.cbrt-Tuple{AbstractFloat}","page":"Mathematics","title":"Base.Math.cbrt","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted (returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n","category":"method"},{"location":"base/math.html#Base.real","page":"Mathematics","title":"Base.real","text":"real(z)\n\nReturn the real part of the complex number z.\n\nSee also: imag, reim, complex, isreal, Real.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\nreal(T::Type)\n\nReturn the type that represents the real part of a value of type T. e.g: for T == Complex{R}, returns R. Equivalent to typeof(real(zero(T))).\n\nExamples\n\njulia> real(Complex{Int})\nInt64\n\njulia> real(Float64)\nFloat64\n\n\n\n\n\nreal(A::AbstractArray)\n\nReturn an array containing the real part of each entry in array A.\n\nEquivalent to real.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> real([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 1\n 0\n 3\n\njulia> real(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n2\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.imag","page":"Mathematics","title":"Base.imag","text":"imag(z)\n\nReturn the imaginary part of the complex number z.\n\nSee also: conj, reim, adjoint, angle.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\nimag(A::AbstractArray)\n\nReturn an array containing the imaginary part of each entry in array A.\n\nEquivalent to imag.(A), except that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> imag([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 0\n 2\n 4\n\njulia> imag(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n-1\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.reim","page":"Mathematics","title":"Base.reim","text":"reim(z)\n\nReturn a tuple of the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\nreim(A::AbstractArray)\n\nReturn a tuple of two arrays containing respectively the real and the imaginary part of each entry in A.\n\nEquivalent to (real.(A), imag.(A)), except that when eltype(A) <: Real A is returned without copying to represent the real part, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> reim([1, 2im, 3 + 4im])\n([1, 0, 3], [0, 2, 4])\n\njulia> reim(fill(2 - im))\n(fill(2), fill(-1))\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.conj","page":"Mathematics","title":"Base.conj","text":"conj(z)\n\nCompute the complex conjugate of a complex number z.\n\nSee also: angle, adjoint.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\nconj(A::AbstractArray)\n\nReturn an array containing the complex conjugate of each entry in array A.\n\nEquivalent to conj.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> conj([1, 2im, 3 + 4im])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 0 - 2im\n 3 - 4im\n\njulia> conj(fill(2 - im))\n0-dimensional Array{Complex{Int64}, 0}:\n2 + 1im\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.angle","page":"Mathematics","title":"Base.angle","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nReturns a number -pi ≤ angle(z) ≤ pi, and is thus discontinuous along the negative real axis.\n\nSee also: atan, cis, rad2deg.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 + 1e-20im))\n180.0\n\njulia> rad2deg(angle(-1 - 1e-20im))\n-180.0\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.cis","page":"Mathematics","title":"Base.cis","text":"cis(x)\n\nMore efficient method for exp(im*x) by using Euler's formula: cos(x) + i sin(x) = exp(i x).\n\nSee also cispi, sincos, exp, angle.\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.cispi","page":"Mathematics","title":"Base.cispi","text":"cispi(x)\n\nMore accurate method for cis(pi*x) (especially for large x).\n\nSee also cis, sincospi, exp, angle.\n\nExamples\n\njulia> cispi(10000)\n1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645954562 + 0.03055685464595456im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.binomial","page":"Mathematics","title":"Base.binomial","text":"binomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in the polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nSee also factorial.\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\nbinomial(x::Number, k::Integer)\n\nThe generalized binomial coefficient, defined for k ≥ 0 by the polynomial\n\nfrac1k prod_j=0^k-1 (x - j)\n\nWhen k < 0 it returns zero.\n\nFor the case of integer x, this is equivalent to the ordinary integer binomial coefficient\n\nbinomnk = fracnk (n-k)\n\nFurther generalizations to non-integer k are mathematically possible, but involve the Gamma function and/or the beta function, which are not provided by the Julia standard library but are available in external packages such as SpecialFunctions.jl.\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.factorial","page":"Mathematics","title":"Base.factorial","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an integer (promoted to at least 64 bits). Note that this may overflow if n is not small, but you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nSee also binomial.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.gcd","page":"Mathematics","title":"Base.gcd","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero). The arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6, 9)\n3\n\njulia> gcd(6, -9)\n3\n\njulia> gcd(6, 0)\n6\n\njulia> gcd(0, 0)\n0\n\njulia> gcd(1//3, 2//3)\n1//3\n\njulia> gcd(1//3, -2//3)\n1//3\n\njulia> gcd(1//3, 2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.lcm","page":"Mathematics","title":"Base.lcm","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero). The arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2, 3)\n6\n\njulia> lcm(-2, 3)\n6\n\njulia> lcm(0, 3)\n0\n\njulia> lcm(0, 0)\n0\n\njulia> lcm(1//3, 2//3)\n2//3\n\njulia> lcm(1//3, -2//3)\n2//3\n\njulia> lcm(1//3, 2)\n2//1\n\njulia> lcm(1, 3, 5, 7)\n105\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.gcdx","page":"Mathematics","title":"Base.gcdx","text":"gcdx(a, b)\n\nComputes the greatest common (positive) divisor of a and b and their Bézout coefficients, i.e. the integer coefficients u and v that satisfy ua+vb = d = gcd(a b). gcdx(a b) returns (d u v).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal Bézout coefficients that are computed by the extended Euclidean algorithm. (Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.) For signed integers, these coefficients u and v are minimal in the sense that u  bd and v  ad. Furthermore, the signs of u and v are chosen so that d is positive. For unsigned integers, the coefficients u and v might be near their typemax, and the identity then holds only via the unsigned integers' modulo arithmetic.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.ispow2","page":"Mathematics","title":"Base.ispow2","text":"ispow2(n::Number) -> Bool\n\nTest whether n is an integer power of two.\n\nSee also count_ones, prevpow, nextpow.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.nextpow","page":"Mathematics","title":"Base.nextpow","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be greater than 1, and x must be greater than 0.\n\nSee also prevpow.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.prevpow","page":"Mathematics","title":"Base.prevpow","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer. a must be greater than 1, and x must not be less than 1.\n\nSee also nextpow, isqrt.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.nextprod","page":"Mathematics","title":"Base.nextprod","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers p_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.invmod","page":"Mathematics","title":"Base.invmod","text":"invmod(n::Integer, m::Integer)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m, and div(ym) = 0. This will throw an error if m = 0, or if gcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2, 5)\n3\n\njulia> invmod(2, 3)\n2\n\njulia> invmod(5, 6)\n5\n\n\n\n\n\ninvmod(n::Integer, T) where {T <: Base.BitInteger}\ninvmod(n::T) where {T <: Base.BitInteger}\n\nCompute the modular inverse of n in the integer ring of type T, i.e. modulo 2^N where N = 8*sizeof(T) (e.g. N = 32 for Int32). In other words these methods satisfy the following identities:\n\nn * invmod(n) == 1\n(n * invmod(n, T)) % T == 1\n(n % T) * invmod(n, T) == 1\n\nNote that * here is modular multiplication in the integer ring, T.\n\nSpecifying the modulus implied by an integer type as an explicit value is often inconvenient since the modulus is by definition too big to be represented by the type.\n\nThe modular inverse is computed much more efficiently than the general case using the algorithm described in https://arxiv.org/pdf/2204.04342.pdf.\n\ncompat: Julia 1.11\nThe invmod(n) and invmod(n, T) methods require Julia 1.11 or later.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.powermod","page":"Mathematics","title":"Base.powermod","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.ndigits","page":"Mathematics","title":"Base.ndigits","text":"ndigits(n::Integer; base::Integer=10, pad::Integer=1)\n\nCompute the number of digits in integer n written in base base (base must not be in [-1, 0, 1]), optionally padded with zeros to a specified size (the result will never be less than pad).\n\nSee also digits, count_ones.\n\nExamples\n\njulia> ndigits(0)\n1\n\njulia> ndigits(12345)\n5\n\njulia> ndigits(1022, base=16)\n3\n\njulia> string(1022, base=16)\n\"3fe\"\n\njulia> ndigits(123, pad=5)\n5\n\njulia> ndigits(-123)\n3\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.add_sum","page":"Mathematics","title":"Base.add_sum","text":"Base.add_sum(x, y)\n\nThe reduction operator used in sum. The main difference from + is that small integers are promoted to Int/UInt.\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.widemul","page":"Mathematics","title":"Base.widemul","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nSee also promote, Base.add_sum.\n\nExamples\n\njulia> widemul(Float32(3.0), 4.0) isa BigFloat\ntrue\n\njulia> typemax(Int8) * typemax(Int8)\n1\n\njulia> widemul(typemax(Int8), typemax(Int8))  # == 127^2\n16129\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.evalpoly","page":"Mathematics","title":"Base.Math.evalpoly","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...; that is, the coefficients are given in ascending order by power of x. Loops are unrolled at compile time if the number of coefficients is statically known, i.e. when p is a Tuple. This function generates efficient code using Horner's method if x is real, or using a Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExamples\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n","category":"function"},{"location":"base/math.html#Base.Math.@evalpoly","page":"Mathematics","title":"Base.Math.@evalpoly","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...; that is, the coefficients are given in ascending order by power of z.  This macro expands to efficient inline code that uses either Horner's method or, for complex z, a more efficient Goertzel-like algorithm.\n\nSee also evalpoly.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n","category":"macro"},{"location":"base/math.html#Base.FastMath.@fastmath","page":"Mathematics","title":"Base.FastMath.@fastmath","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that may violate strict IEEE semantics. This allows the fastest possible operation, but results are undefined – be careful when doing this, as it may change numerical results.\n\nThis sets the LLVM Fast-Math flags, and corresponds to the -ffast-math option in clang. See the notes on performance annotations for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n","category":"macro"},{"location":"base/math.html#Customizable-binary-operators","page":"Mathematics","title":"Customizable binary operators","text":"","category":"section"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"Some unicode characters can be used to define new binary operators that support infix notation. For example ⊗(x,y) = kron(x,y) defines the ⊗ (otimes) function to be the Kronecker product, and one can call it as binary operator using infix syntax: C = A ⊗ B as well as with the usual prefix syntax C = ⊗(A,B).","category":"page"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"Other characters that support such extensions include \\odot ⊙ and \\oplus ⊕","category":"page"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"The complete list is in the parser code: https://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm","category":"page"},{"location":"base/math.html","page":"Mathematics","title":"Mathematics","text":"Those that are parsed like * (in terms of precedence) include * / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗ and those that are parsed like + include + - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⟇ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣ There are many others that are related to arrows, comparisons, and powers.","category":"page"},{"location":"devdocs/precompile_hang.html#Fixing-precompilation-hangs-due-to-open-tasks-or-IO","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"","category":"section"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"On Julia 1.10 or higher, you might see the following message:","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"(Image: Screenshot of precompilation hang)","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"This may repeat. If it continues to repeat with no hints that it will resolve itself, you may have a \"precompilation hang\" that requires fixing. Even if it's transient, you might prefer to resolve it so that users will not be bothered by this warning.  This page walks you through how to analyze and fix such issues.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"If you follow the advice and hit Ctrl-C, you might see","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"^C Interrupted: Exiting precompilation...\n\n  1 dependency had warnings during precompilation:\n┌ Test1 [ac89d554-e2ba-40bc-bc5c-de68b658c982]\n│  [pid 2745] waiting for IO to finish:\n│   Handle type        uv_handle_t->data\n│   timer              0x55580decd1e0->0x7f94c3a4c340","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"This message conveys two key pieces of information:","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"the hang is occurring during precompilation of Test1, a dependency of Test2 (the package we were trying to load with using Test2)\nduring precompilation of Test1, Julia created a Timer object (use ?Timer if you're unfamiliar with Timers) which is still open; until that closes, the process is hung","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"If this is enough of a hint for you to figure out how timer = Timer(args...) is being created, one good solution is to add wait(timer) if timer eventually finishes on its own, or close(timer) if you need to force-close it, before the final end of the module.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"However, there are cases that may not be that straightforward. Usually the best option is to start by determining whether the hang is due to code in Test1 or whether it is due to one of Test1's dependencies:","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Option 1: Pkg.add(\"Aqua\") and use Aqua.test_persistent_tasks. This should help you identify which package is causing the problem, after which the instructions below should be followed. If needed, you can create a PkgId as Base.PkgId(UUID(\"...\"), \"Test1\"), where ... comes from the uuid entry in Test1/Project.toml.\nOption 2: manually diagnose the source of the hang.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"To manually diagnose:","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Pkg.develop(\"Test1\")\nComment out all the code included or defined in Test1, except the using/import statements.\nTry using Test2 (or even using Test1 assuming that hangs too) again","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Now we arrive at a fork in the road: either","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"the hang persists, indicating it is due to one of your dependencies\nthe hang disappears, indicating that it is due to something in your code.","category":"page"},{"location":"devdocs/precompile_hang.html#pchang_deps","page":"Fixing precompilation hangs due to open tasks or IO","title":"Diagnosing and fixing hangs due to a package dependency","text":"","category":"section"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Use a binary search to identify the problematic dependency: start by commenting out half your dependencies, then when you isolate which half is responsible comment out half of that half, etc. (You don't have to remove them from the project, just comment out the using/import statements.)","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Once you've identified a suspect (here we'll call it ThePackageYouThinkIsCausingTheProblem), first try precompiling that package. If it also hangs during precompilation, continue chasing the problem backwards.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"However, most likely ThePackageYouThinkIsCausingTheProblem will precompile fine. This suggests it's in the function ThePackageYouThinkIsCausingTheProblem.__init__, which does not run during precompilation of ThePackageYouThinkIsCausingTheProblem but does in any package that loads ThePackageYouThinkIsCausingTheProblem.  To test this theory, set up a minimal working example (MWE), something like","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"(@v1.10) pkg> generate MWE\n  Generating  project MWE:\n    MWE\\Project.toml\n    MWE\\src\\MWE.jl","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"where the source code of MWE.jl is","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"module MWE\nusing ThePackageYouThinkIsCausingTheProblem\nend","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"and you've added ThePackageYouThinkIsCausingTheProblem to MWE's dependencies.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"If that MWE reproduces the hang, you've found your culprit: ThePackageYouThinkIsCausingTheProblem.__init__ must be creating the Timer object. If the timer object can be safely closed, that's a good option. Otherwise, the most common solution is to avoid creating the timer while any package is being precompiled: add","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"ccall(:jl_generating_output, Cint, ()) == 1 && return nothing","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"as the first line of ThePackageYouThinkIsCausingTheProblem.__init__, and it will avoid doing any initialization in any Julia process whose purpose is to precompile packages.","category":"page"},{"location":"devdocs/precompile_hang.html#pchang_fix","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing package code to avoid hangs","text":"","category":"section"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"Search your package for suggestive words (here like \"Timer\") and see if you can identify where the problem is being created. Note that a method definition like","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"maketimer() = Timer(timer -> println(\"hi\"), 0; interval=1)","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"is not problematic in and of itself: it can cause this problem only if maketimer gets called while the module is being defined. This might be happening from a top-level statement such as","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"const GLOBAL_TIMER = maketimer()","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"or it might conceivably occur in a precompile workload.","category":"page"},{"location":"devdocs/precompile_hang.html","page":"Fixing precompilation hangs due to open tasks or IO","title":"Fixing precompilation hangs due to open tasks or IO","text":"If you struggle to identify the causative lines, then consider doing a binary search: comment out sections of your package (or include lines to omit entire files) until you've reduced the problem in scope.","category":"page"},{"location":"devdocs/build/linux.html#Linux","page":"Linux","title":"Linux","text":"","category":"section"},{"location":"devdocs/build/linux.html","page":"Linux","title":"Linux","text":"GCC version 4.7 or later is required to build Julia.\nTo use external shared libraries not in the system library search path, set USE_SYSTEM_XXX=1 and LDFLAGS=-Wl,-rpath,/path/to/dir/contains/libXXX.so in Make.user.\nInstead of setting LDFLAGS, putting the library directory into the environment variable LD_LIBRARY_PATH (at both compile and run time) also works.\nThe USE_SYSTEM_* flags should be used with caution. These are meant only for troubleshooting, porting, and packaging, where package maintainers work closely with the Julia developers to make sure that Julia is built correctly. Production use cases should use the officially provided binaries. Issues arising from the use of these flags will generally not be accepted.\nSee also the external dependencies.","category":"page"},{"location":"devdocs/build/linux.html#Architecture-Customization","page":"Linux","title":"Architecture Customization","text":"","category":"section"},{"location":"devdocs/build/linux.html","page":"Linux","title":"Linux","text":"Julia can be built for a non-generic architecture by configuring the ARCH Makefile variable in a Make.user file. See the appropriate section of Make.inc for additional customization options, such as MARCH and JULIA_CPU_TARGET.","category":"page"},{"location":"devdocs/build/linux.html","page":"Linux","title":"Linux","text":"For example, to build for Pentium 4, set MARCH=pentium4 and install the necessary system libraries for linking. On Ubuntu, these may include lib32gfortran-6-dev, lib32gcc1, and lib32stdc++6, among others.","category":"page"},{"location":"devdocs/build/linux.html","page":"Linux","title":"Linux","text":"You can also set MARCH=native in Make.user for a maximum-performance build customized for the current machine CPU.","category":"page"},{"location":"devdocs/build/linux.html#Linux-Build-Troubleshooting","page":"Linux","title":"Linux Build Troubleshooting","text":"","category":"section"},{"location":"devdocs/build/linux.html","page":"Linux","title":"Linux","text":"Problem Possible Solution\nOpenBLAS build failure Set one of the following build options in Make.user and build again: <ul><li> OPENBLAS_TARGET_ARCH=BARCELONA (AMD CPUs) or OPENBLAS_TARGET_ARCH=NEHALEM (Intel CPUs)<ul>Set OPENBLAS_DYNAMIC_ARCH = 0 to disable compiling multiple architectures in a single binary.</ul></li><li> OPENBLAS_NO_AVX2 = 1 disables AVX2 instructions, allowing OpenBLAS to compile with OPENBLAS_DYNAMIC_ARCH = 1 using old versions of binutils </li><li> USE_SYSTEM_BLAS=1 uses the system provided libblas <ul><li>Set LIBBLAS=-lopenblas and LIBBLASNAME=libopenblas to force the use of the system provided OpenBLAS when multiple BLAS versions are installed. </li></ul></li></ul><p> If you get an error that looks like ../kernel/x86_64/dgemm_kernel_4x4_haswell.S:1709: Error: no such instruction: `vpermpd $ 0xb1,%ymm0,%ymm0', then you need to set OPENBLAS_DYNAMIC_ARCH = 0 or OPENBLAS_NO_AVX2 = 1, or you need a newer version of binutils (2.18 or newer). (Issue #7653)</p><p> If the linker cannot find gfortran and you get an error like julia /usr/bin/x86_64-linux-gnu-ld: cannot find -lgfortran, check the path with gfortran -print-file-name=libgfortran.so and use the output to export something similar to this: export LDFLAGS=-L/usr/lib/gcc/x86_64-linux-gnu/8/. See Issue #6150.</p>\nIllegal Instruction error Check if your CPU supports AVX while your OS does not (e.g. through virtualization, as described in this issue).","category":"page"},{"location":"manual/multi-threading.html#man-multithreading","page":"Multi-Threading","title":"Multi-Threading","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Visit this blog post for a presentation of Julia multi-threading features.","category":"page"},{"location":"manual/multi-threading.html#Starting-Julia-with-multiple-threads","page":"Multi-Threading","title":"Starting Julia with multiple threads","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"By default, Julia starts up with a single thread of execution. This can be verified by using the command Threads.nthreads():","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> Threads.nthreads()\n1","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The number of execution threads is controlled either by using the -t/--threads command line argument or by using the JULIA_NUM_THREADS environment variable. When both are specified, then -t/--threads takes precedence.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The number of threads can either be specified as an integer (--threads=4) or as auto (--threads=auto), where auto tries to infer a useful default number of threads to use (see Command-line Options for more details).","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"compat: Julia 1.5\nThe -t/--threads command line argument requires at least Julia 1.5. In older versions you must use the environment variable instead.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"compat: Julia 1.7\nUsing auto as value of the environment variable JULIA_NUM_THREADS requires at least Julia 1.7. In older versions, this value is ignored.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Lets start Julia with 4 threads:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"$ julia --threads 4","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Let's verify there are 4 threads at our disposal.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> Threads.nthreads()\n4","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"But we are currently on the master thread. To check, we use the function Threads.threadid","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> Threads.threadid()\n1","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"note: Note\nIf you prefer to use the environment variable you can set it as follows in Bash (Linux/macOS):export JULIA_NUM_THREADS=4C shell on Linux/macOS, CMD on Windows:set JULIA_NUM_THREADS=4Powershell on Windows:$env:JULIA_NUM_THREADS=4Note that this must be done before starting Julia.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"note: Note\nThe number of threads specified with -t/--threads is propagated to worker processes that are spawned using the -p/--procs or --machine-file command line options. For example, julia -p2 -t2 spawns 1 main process with 2 worker processes, and all three processes have 2 threads enabled. For more fine grained control over worker threads use addprocs and pass -t/--threads as exeflags.","category":"page"},{"location":"manual/multi-threading.html#Multiple-GC-Threads","page":"Multi-Threading","title":"Multiple GC Threads","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The Garbage Collector (GC) can use multiple threads. The amount used is either half the number of compute worker threads or configured by either the --gcthreads command line argument or by using the JULIA_NUM_GC_THREADS environment variable.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"compat: Julia 1.10\nThe --gcthreads command line argument requires at least Julia 1.10.","category":"page"},{"location":"manual/multi-threading.html#man-threadpools","page":"Multi-Threading","title":"Threadpools","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"When a program's threads are busy with many tasks to run, tasks may experience delays which may negatively affect the responsiveness and interactivity of the program. To address this, you can specify that a task is interactive when you Threads.@spawn it:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"using Base.Threads\n@spawn :interactive f()","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Interactive tasks should avoid performing high latency operations, and if they are long duration tasks, should yield frequently.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Julia may be started with one or more threads reserved to run interactive tasks:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"$ julia --threads 3,1","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The environment variable JULIA_NUM_THREADS can also be used similarly:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"export JULIA_NUM_THREADS=3,1","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"This starts Julia with 3 threads in the :default threadpool and 1 thread in the :interactive threadpool:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> using Base.Threads\n\njulia> nthreadpools()\n2\n\njulia> threadpool() # the main thread is in the interactive thread pool\n:interactive\n\njulia> nthreads(:default)\n3\n\njulia> nthreads(:interactive)\n1\n\njulia> nthreads()\n3","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"note: Note\nThe zero-argument version of nthreads returns the number of threads in the default pool.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"note: Note\nDepending on whether Julia has been started with interactive threads, the main thread is either in the default or interactive thread pool.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Either or both numbers can be replaced with the word auto, which causes Julia to choose a reasonable default.","category":"page"},{"location":"manual/multi-threading.html#The-@threads-Macro","page":"Multi-Threading","title":"The @threads Macro","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Let's work a simple example using our native threads. Let us create an array of zeros:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> a = zeros(10)\n10-element Vector{Float64}:\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Let us operate on this array simultaneously using 4 threads. We'll have each thread write its thread ID into each location.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Julia supports parallel loops using the Threads.@threads macro. This macro is affixed in front of a for loop to indicate to Julia that the loop is a multi-threaded region:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> Threads.@threads for i = 1:10\n           a[i] = Threads.threadid()\n       end","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The iteration space is split among the threads, after which each thread writes its thread ID to its assigned locations:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> a\n10-element Vector{Float64}:\n 1.0\n 1.0\n 1.0\n 2.0\n 2.0\n 2.0\n 3.0\n 3.0\n 4.0\n 4.0","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Note that Threads.@threads does not have an optional reduction parameter like @distributed.","category":"page"},{"location":"manual/multi-threading.html#Using-@threads-without-data-races","page":"Multi-Threading","title":"Using @threads without data-races","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The concept of a data-race is elaborated on in \"Communication and data races between threads\". For now, just known that a data race can result in incorrect results and dangerous errors.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Lets say we want to make the function sum_single below multithreaded.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> function sum_single(a)\n           s = 0\n           for i in a\n               s += i\n           end\n           s\n       end\nsum_single (generic function with 1 method)\n\njulia> sum_single(1:1_000_000)\n500000500000","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Simply adding @threads exposes a data race with multiple threads reading and writing s at the same time.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> function sum_multi_bad(a)\n           s = 0\n           Threads.@threads for i in a\n               s += i\n           end\n           s\n       end\nsum_multi_bad (generic function with 1 method)\n\njulia> sum_multi_bad(1:1_000_000)\n70140554652","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Note that the result is not 500000500000 as it should be, and will most likely change each evaluation.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"To fix this, buffers that are specific to the task may be used to segment the sum into chunks that are race-free. Here sum_single is reused, with its own internal buffer s. The input vector a is split into nthreads() chunks for parallel work. We then use Threads.@spawn to create tasks that individually sum each chunk. Finally, we sum the results from each task using sum_single again:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> function sum_multi_good(a)\n           chunks = Iterators.partition(a, length(a) ÷ Threads.nthreads())\n           tasks = map(chunks) do chunk\n               Threads.@spawn sum_single(chunk)\n           end\n           chunk_sums = fetch.(tasks)\n           return sum_single(chunk_sums)\n       end\nsum_multi_good (generic function with 1 method)\n\njulia> sum_multi_good(1:1_000_000)\n500000500000","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"note: Note\nBuffers should not be managed based on threadid() i.e. buffers = zeros(Threads.nthreads()) because concurrent tasks can yield, meaning multiple concurrent tasks may use the same buffer on a given thread, introducing risk of data races. Further, when more than one thread is available tasks may change thread at yield points, which is known as task migration.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Another option is the use of atomic operations on variables shared across tasks/threads, which may be more performant depending on the characteristics of the operations.","category":"page"},{"location":"manual/multi-threading.html#man-communication-and-data-races","page":"Multi-Threading","title":"Communication and data-races between threads","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Although Julia's threads can communicate through shared memory, it is notoriously difficult to write correct and data-race free multi-threaded code. Julia's Channels are thread-safe and may be used to communicate safely. There are also sections below that explain how to use locks and atomics to avoid data-races.","category":"page"},{"location":"manual/multi-threading.html#Data-race-freedom","page":"Multi-Threading","title":"Data-race freedom","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"You are entirely responsible for ensuring that your program is data-race free, and nothing promised here can be assumed if you do not observe that requirement. The observed results may be highly unintuitive.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"If data-races are introduced, Julia is not memory safe. Be very careful about reading any data if another thread might write to it, as it could result in segmentation faults or worse. Below are a couple of unsafe ways to access global variables from different threads:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Thread 1:\nglobal b = false\nglobal a = rand()\nglobal b = true\n\nThread 2:\nwhile !b; end\nbad_read1(a) # it is NOT safe to access `a` here!\n\nThread 3:\nwhile !@isdefined(a); end\nbad_read2(a) # it is NOT safe to access `a` here","category":"page"},{"location":"manual/multi-threading.html#man-using-locks","page":"Multi-Threading","title":"Using locks to avoid data-races","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"An important tool to avoid data-races, and thereby write thread-safe code, is the concept of a \"lock\". A lock can be locked and unlocked. If a thread has locked a lock, and not unlocked it, it is said to \"hold\" the lock. If there is only one lock, and we write code the requires holding the lock to access some data, we can ensure that multiple threads will never access the same data simultaneously. Note that the link between a lock and a variable is made by the programmer, and not the program.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"For example, we can create a lock my_lock, and lock it while we mutate a variable my_variable. This is done most simply with the @lock macro:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> my_lock = ReentrantLock();\n\njulia> my_variable = [1, 2, 3];\n\njulia> @lock my_lock my_variable[1] = 100\n100","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"By using a similar pattern with the same lock and variable, but on another thread, the operations are free from data-races.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"We could have performed the operation above with the functional version of lock, in the following two ways:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> lock(my_lock) do\n           my_variable[1] = 100\n       end\n100\n\njulia> begin\n           lock(my_lock)\n           try\n               my_variable[1] = 100\n           finally\n               unlock(my_lock)\n           end\n       end\n100","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"All three options are equivalent. Note how the final version requires an explicit try-block to ensure that the lock is always unlocked, whereas the first two version do this internally. One should always use the lock pattern above when changing data (such as assigning to a global or closure variable) accessed by other threads. Failing to do this could have unforeseen and serious consequences.","category":"page"},{"location":"manual/multi-threading.html#man-atomic-operations","page":"Multi-Threading","title":"Atomic Operations","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Julia supports accessing and modifying values atomically, that is, in a thread-safe way to avoid race conditions. A value (which must be of a primitive type) can be wrapped as Threads.Atomic to indicate it must be accessed in this way. Here we can see an example:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> i = Threads.Atomic{Int}(0);\n\njulia> ids = zeros(4);\n\njulia> old_is = zeros(4);\n\njulia> Threads.@threads for id in 1:4\n           old_is[id] = Threads.atomic_add!(i, id)\n           ids[id] = id\n       end\n\njulia> old_is\n4-element Vector{Float64}:\n 0.0\n 1.0\n 7.0\n 3.0\n\njulia> i[]\n 10\n\njulia> ids\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Had we tried to do the addition without the atomic tag, we might have gotten the wrong answer due to a race condition. An example of what would happen if we didn't avoid the race:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"julia> using Base.Threads\n\njulia> Threads.nthreads()\n4\n\njulia> acc = Ref(0)\nBase.RefValue{Int64}(0)\n\njulia> @threads for i in 1:1000\n          acc[] += 1\n       end\n\njulia> acc[]\n926\n\njulia> acc = Atomic{Int64}(0)\nAtomic{Int64}(0)\n\njulia> @threads for i in 1:1000\n          atomic_add!(acc, 1)\n       end\n\njulia> acc[]\n1000","category":"page"},{"location":"manual/multi-threading.html#man-atomics","page":"Multi-Threading","title":"Per-field atomics","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"We can also use atomics on a more granular level using the @atomic, @atomicswap, @atomicreplace macros, and @atomiconce macros.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Specific details of the memory model and other details of the design are written in the Julia Atomics Manifesto, which will later be published formally.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Any field in a struct declaration can be decorated with @atomic, and then any write must be marked with @atomic also, and must use one of the defined atomic orderings (:monotonic, :acquire, :release, :acquire_release, or :sequentially_consistent). Any read of an atomic field can also be annotated with an atomic ordering constraint, or will be done with monotonic (relaxed) ordering if unspecified.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"compat: Julia 1.7\nPer-field atomics requires at least Julia 1.7.","category":"page"},{"location":"manual/multi-threading.html#Side-effects-and-mutable-function-arguments","page":"Multi-Threading","title":"Side effects and mutable function arguments","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"When using multi-threading we have to be careful when using functions that are not pure as we might get a wrong answer. For instance functions that have a name ending with ! by convention modify their arguments and thus are not pure.","category":"page"},{"location":"manual/multi-threading.html#@threadcall","page":"Multi-Threading","title":"@threadcall","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"External libraries, such as those called via ccall, pose a problem for Julia's task-based I/O mechanism. If a C library performs a blocking operation, that prevents the Julia scheduler from executing any other tasks until the call returns. (Exceptions are calls into custom C code that call back into Julia, which may then yield, or C code that calls jl_yield(), the C equivalent of yield.)","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"The @threadcall macro provides a way to avoid stalling execution in such a scenario. It schedules a C function for execution in a separate thread. A threadpool with a default size of 4 is used for this. The size of the threadpool is controlled via environment variable UV_THREADPOOL_SIZE. While waiting for a free thread, and during function execution once a thread is available, the requesting task (on the main Julia event loop) yields to other tasks. Note that @threadcall does not return until the execution is complete. From a user point of view, it is therefore a blocking call like other Julia APIs.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"It is very important that the called function does not call back into Julia, as it will segfault.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"@threadcall may be removed/changed in future versions of Julia.","category":"page"},{"location":"manual/multi-threading.html#Caveats","page":"Multi-Threading","title":"Caveats","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"At this time, most operations in the Julia runtime and standard libraries can be used in a thread-safe manner, if the user code is data-race free. However, in some areas work on stabilizing thread support is ongoing. Multi-threaded programming has many inherent difficulties, and if a program using threads exhibits unusual or undesirable behavior (e.g. crashes or mysterious results), thread interactions should typically be suspected first.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"There are a few specific limitations and warnings to be aware of when using threads in Julia:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Base collection types require manual locking if used simultaneously by multiple threads where at least one thread modifies the collection (common examples include push! on arrays, or inserting items into a Dict).\nThe schedule used by @spawn is nondeterministic and should not be relied on.\nCompute-bound, non-memory-allocating tasks can prevent garbage collection from running in other threads that are allocating memory. In these cases it may be necessary to insert a manual call to GC.safepoint() to allow GC to run. This limitation will be removed in the future.\nAvoid running top-level operations, e.g. include, or eval of type, method, and module definitions in parallel.\nBe aware that finalizers registered by a library may break if threads are enabled. This may require some transitional work across the ecosystem before threading can be widely adopted with confidence. See the section on the safe use of finalizers for further details.","category":"page"},{"location":"manual/multi-threading.html#man-task-migration","page":"Multi-Threading","title":"Task Migration","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"After a task starts running on a certain thread it may move to a different thread if the task yields.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Such tasks may have been started with @spawn or @threads, although the :static schedule option for @threads does freeze the threadid.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"This means that in most cases threadid() should not be treated as constant within a task, and therefore should not be used to index into a vector of buffers or stateful objects.","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"compat: Julia 1.7\nTask migration was introduced in Julia 1.7. Before this tasks always remained on the same thread that they were started on.","category":"page"},{"location":"manual/multi-threading.html#man-finalizers","page":"Multi-Threading","title":"Safe use of Finalizers","text":"","category":"section"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"Because finalizers can interrupt any code, they must be very careful in how they interact with any global state. Unfortunately, the main reason that finalizers are used is to update global state (a pure function is generally rather pointless as a finalizer). This leads us to a bit of a conundrum. There are a few approaches to dealing with this problem:","category":"page"},{"location":"manual/multi-threading.html","page":"Multi-Threading","title":"Multi-Threading","text":"When single-threaded, code could call the internal jl_gc_enable_finalizers C function to prevent finalizers from being scheduled inside a critical region. Internally, this is used inside some functions (such as our C locks) to prevent recursion when doing certain operations (incremental package loading, codegen, etc.). The combination of a lock and this flag can be used to make finalizers safe.\nA second strategy, employed by Base in a couple places, is to explicitly delay a finalizer until it may be able to acquire its lock non-recursively. The following example demonstrates how this strategy could be applied to Distributed.finalize_ref:\nfunction finalize_ref(r::AbstractRemoteRef)\n    if r.where > 0 # Check if the finalizer is already run\n        if islocked(client_refs) || !trylock(client_refs)\n            # delay finalizer for later if we aren't free to acquire the lock\n            finalizer(finalize_ref, r)\n            return nothing\n        end\n        try # `lock` should always be followed by `try`\n            if r.where > 0 # Must check again here\n                # Do actual cleanup here\n                r.where = 0\n            end\n        finally\n            unlock(client_refs)\n        end\n    end\n    nothing\nend\nA related third strategy is to use a yield-free queue. We don't currently have a lock-free queue implemented in Base, but Base.IntrusiveLinkedListSynchronized{T} is suitable. This can frequently be a good strategy to use for code with event loops. For example, this strategy is employed by Gtk.jl to manage lifetime ref-counting. In this approach, we don't do any explicit work inside the finalizer, and instead add it to a queue to run at a safer time. In fact, Julia's task scheduler already uses this, so defining the finalizer as x -> @spawn do_cleanup(x) is one example of this approach. Note however that this doesn't control which thread do_cleanup runs on, so do_cleanup would still need to acquire a lock. That doesn't need to be true if you implement your own queue, as you can explicitly only drain that queue from your thread.","category":"page"},{"location":"devdocs/build/macos.html#macOS","page":"macOS","title":"macOS","text":"","category":"section"},{"location":"devdocs/build/macos.html","page":"macOS","title":"macOS","text":"You need to have the current Xcode command line utilities installed: run xcode-select --install in the terminal. You will need to rerun this terminal command after each macOS update, otherwise you may run into errors involving missing libraries or headers.","category":"page"},{"location":"devdocs/build/macos.html","page":"macOS","title":"macOS","text":"The dependent libraries are now built with BinaryBuilder and will be automatically downloaded. This is the preferred way to build Julia source. In case you want to build them all on your own, you will need a 64-bit gfortran to compile Julia dependencies.","category":"page"},{"location":"devdocs/build/macos.html","page":"macOS","title":"macOS","text":"brew install gcc","category":"page"},{"location":"devdocs/build/macos.html","page":"macOS","title":"macOS","text":"If you have set LD_LIBRARY_PATH or DYLD_LIBRARY_PATH in your .bashrc or equivalent, Julia may be unable to find various libraries that come bundled with it. These environment variables need to be unset for Julia to work.","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/stdlib/TOML/docs/src/index.md\"","category":"page"},{"location":"stdlib/TOML.html#TOML","page":"TOML","title":"TOML","text":"","category":"section"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"TOML.jl is a Julia standard library for parsing and writing TOML v1.0 files.","category":"page"},{"location":"stdlib/TOML.html#Parsing-TOML-data","page":"TOML","title":"Parsing TOML data","text":"","category":"section"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> data = \"\"\"\n           [database]\n           server = \"192.168.1.1\"\n           ports = [ 8001, 8001, 8002 ]\n       \"\"\";\n\njulia> TOML.parse(data)\nDict{String, Any} with 1 entry:\n  \"database\" => Dict{String, Any}(\"server\"=>\"192.168.1.1\", \"ports\"=>[8001, 8001…","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"To parse a file, use TOML.parsefile. If the file has a syntax error, an exception is thrown:","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> TOML.parse(\"\"\"\n           value = 0.0.0\n       \"\"\")\nERROR: TOML Parser error:\nnone:1:16 error: failed to parse value\n      value = 0.0.0\n                 ^\n[...]","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"There are other versions of the parse functions (TOML.tryparse and TOML.tryparsefile) that instead of throwing exceptions on parser error returns a TOML.ParserError with information:","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> err = TOML.tryparse(\"\"\"\n           value = 0.0.0\n       \"\"\");\n\njulia> err.type\nErrGenericValueError::ErrorType = 14\n\njulia> err.line\n1\n\njulia> err.column\n16","category":"page"},{"location":"stdlib/TOML.html#Exporting-data-to-TOML-file","page":"TOML","title":"Exporting data to TOML file","text":"","category":"section"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"The TOML.print function is used to print (or serialize) data into TOML format.","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> data = Dict(\n          \"names\" => [\"Julia\", \"Julio\"],\n          \"age\" => [10, 20],\n       );\n\njulia> TOML.print(data)\nnames = [\"Julia\", \"Julio\"]\nage = [10, 20]\n\njulia> fname = tempname();\n\njulia> open(fname, \"w\") do io\n           TOML.print(io, data)\n       end\n\njulia> TOML.parsefile(fname)\nDict{String, Any} with 2 entries:\n  \"names\" => [\"Julia\", \"Julio\"]\n  \"age\"   => [10, 20]","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"Keys can be sorted according to some value","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> TOML.print(Dict(\n       \"abc\"  => 1,\n       \"ab\"   => 2,\n       \"abcd\" => 3,\n       ); sorted=true, by=length)\nab = 2\nabc = 1\nabcd = 3","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"For custom structs, pass a function that converts the struct to a supported type","category":"page"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"julia> using TOML\n\njulia> struct MyStruct\n           a::Int\n           b::String\n       end\n\njulia> TOML.print(Dict(\"foo\" => MyStruct(5, \"bar\"))) do x\n           x isa MyStruct && return [x.a, x.b]\n           error(\"unhandled type $(typeof(x))\")\n       end\nfoo = [5, \"bar\"]","category":"page"},{"location":"stdlib/TOML.html#References","page":"TOML","title":"References","text":"","category":"section"},{"location":"stdlib/TOML.html","page":"TOML","title":"TOML","text":"TOML.parse\nTOML.parsefile\nTOML.tryparse\nTOML.tryparsefile\nTOML.print\nTOML.Parser\nTOML.ParserError","category":"page"},{"location":"stdlib/TOML.html#TOML.parse","page":"TOML","title":"TOML.parse","text":"parse(x::Union{AbstractString, IO})\nparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string  or stream x, and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparse.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/TOML.html#TOML.parsefile","page":"TOML","title":"TOML.parsefile","text":"parsefile(f::AbstractString)\nparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparsefile.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/TOML.html#TOML.tryparse","page":"TOML","title":"TOML.tryparse","text":"tryparse(x::Union{AbstractString, IO})\ntryparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string or stream x, and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parse.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/TOML.html#TOML.tryparsefile","page":"TOML","title":"TOML.tryparsefile","text":"tryparsefile(f::AbstractString)\ntryparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parsefile.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/TOML.html#TOML.print","page":"TOML","title":"TOML.print","text":"print([to_toml::Function], io::IO [=stdout], data::AbstractDict; sorted=false, by=identity, inline_tables::IdSet{<:AbstractDict})\n\nWrite data as TOML syntax to the stream io. If the keyword argument sorted is set to true, sort tables according to the function given by the keyword argument by. If the keyword argument inline_tables is given, it should be a set of tables that should be printed \"inline\".\n\nThe following data types are supported: AbstractDict, AbstractVector, AbstractString, Integer, AbstractFloat, Bool, Dates.DateTime, Dates.Time, Dates.Date. Note that the integers and floats need to be convertible to Float64 and Int64 respectively. For other data types, pass the function to_toml that takes the data types and returns a value of a supported type.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/TOML.html#TOML.Parser","page":"TOML","title":"TOML.Parser","text":"Parser()\n\nConstructor for a TOML Parser.  Note that in most cases one does not need to explicitly create a Parser but instead one directly use use TOML.parsefile or TOML.parse.  Using an explicit parser will however reuse some internal data structures which can be beneficial for performance if a larger number of small files are parsed.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/TOML.html#TOML.ParserError","page":"TOML","title":"TOML.ParserError","text":"ParserError\n\nType that is returned from tryparse and tryparsefile when parsing fails. It contains (among others) the following fields:\n\npos, the position in the string when the error happened\ntable, the result that so far was successfully parsed\ntype, an error type, different for different types of errors\n\n\n\n\n\n","category":"type"},{"location":"base/punctuation.html#man-punctuation","page":"Punctuation","title":"Punctuation","text":"","category":"section"},{"location":"base/punctuation.html","page":"Punctuation","title":"Punctuation","text":"Extended documentation for mathematical symbols & functions is here.","category":"page"},{"location":"base/punctuation.html","page":"Punctuation","title":"Punctuation","text":"symbol meaning\n@ the at-sign marks a macro invocation; optionally followed by an argument list\n! an exclamation mark is a prefix operator for logical negation (\"not\")\na! function names that end with an exclamation mark modify one or more of their arguments by convention\n# the number sign (or hash or pound) character begins single line comments\n#= when followed by an equals sign, it begins a multi-line comment (these are nestable)\n=# end a multi-line comment by immediately preceding the number sign with an equals sign\n$ the dollar sign is used for string and expression interpolation\n% the percent symbol is the remainder operator\n^ the caret is the exponentiation operator\n& single ampersand is bitwise and\n&& double ampersands is short-circuiting boolean and\n| single pipe character is bitwise or\n|| double pipe characters is short-circuiting boolean or\n⊻ the unicode xor character is bitwise exclusive or\n~ the tilde is an operator for bitwise not\n' a trailing apostrophe is the adjoint (that is, the complex transpose) operator Aᴴ\n* the asterisk is used for multiplication, including matrix multiplication and string concatenation\n/ forward slash divides the argument on its left by the one on its right\n\\ backslash operator divides the argument on its right by the one on its left, commonly used to solve matrix equations\n() parentheses with no arguments constructs an empty Tuple\n(a,...) parentheses with comma-separated arguments constructs a tuple containing its arguments\n(a=1,...) parentheses with comma-separated assignments constructs a NamedTuple\n(x;y) parentheses can also be used to group one or more semicolon separated expressions\na[] array indexing (calling getindex or setindex!)\n[,] vector literal constructor (calling vect)\n[;] vertical concatenation (calling vcat or hvcat)\n[    ] with space-separated expressions, horizontal concatenation (calling hcat or hvcat)\nT{ } curly braces following a type list that type's parameters\n{} curly braces can also be used to group multiple where expressions in function declarations\n; semicolons separate statements, begin a list of keyword arguments in function declarations or calls, or are used to separate array literals for vertical concatenation\n, commas separate function arguments or tuple or array components\n? the question mark delimits the ternary conditional operator (used like: conditional ? if_true : if_false)\n\" \" the single double-quote character delimits String literals\n\"\"\" \"\"\" three double-quote characters delimits string literals that may contain \" and ignore leading indentation\n' ' the single-quote character delimits Char (that is, character) literals\n` ` the backtick character delimits external process (Cmd) literals\nA... triple periods are a postfix operator that \"splat\" their arguments' contents into many arguments of a function call or declare a varargs function that \"slurps\" up many arguments into a single tuple\na.b single periods access named fields in objects/modules (calling getproperty or setproperty!)\nf.() periods may also prefix parentheses (like f.(...)) or infix operators (like .+) to perform the function element-wise (calling broadcast)\na:b colons (:) used as a binary infix operator construct a range from a to b (inclusive) with fixed step size 1\na:s:b colons (:) used as a ternary infix operator construct a range from a to b (inclusive) with step size s\n: when used by themselves, Colons represent all indices within a dimension, frequently combined with indexing\n:: double-colons represent a type annotation or typeassert, depending on context, frequently used when declaring function arguments\n:( ) quoted expression\n:a Symbol a\n<: subtype operator\n>: supertype operator (reverse of subtype operator)\n= single equals sign is assignment\n== double equals sign is value equality comparison\n=== triple equals sign is programmatically identical equality comparison\n=> right arrow using an equals sign defines a Pair typically used to populate dictionaries\n-> right arrow using a hyphen defines an anonymous function on a single line\n|> pipe operator passes output from the left argument to input of the right argument, usually a function\n∘ function composition operator (typed with \\circ{tab}) combines two functions as though they are a single larger function\n_ underscores may be assigned values which will not be saved, often used to ignore multiple return values or create repetitive comprehensions","category":"page"},{"location":"stdlib/Distributed.html#man-distributed","page":"Distributed Computing","title":"Distributed Computing","text":"","category":"section"},{"location":"stdlib/Distributed.html","page":"Distributed Computing","title":"Distributed Computing","text":"Distributed\nDistributed.addprocs\nDistributed.nprocs\nDistributed.nworkers\nDistributed.procs()\nDistributed.procs(::Integer)\nDistributed.workers\nDistributed.rmprocs\nDistributed.interrupt\nDistributed.myid\nDistributed.pmap\nDistributed.RemoteException\nDistributed.ProcessExitedException\nDistributed.Future\nDistributed.RemoteChannel\nDistributed.fetch(::Distributed.Future)\nDistributed.fetch(::RemoteChannel)\nDistributed.remotecall(::Any, ::Integer, ::Any...)\nDistributed.remotecall_wait(::Any, ::Integer, ::Any...)\nDistributed.remotecall_fetch(::Any, ::Integer, ::Any...)\nDistributed.remote_do(::Any, ::Integer, ::Any...)\nDistributed.put!(::RemoteChannel, ::Any...)\nDistributed.put!(::Distributed.Future, ::Any)\nDistributed.take!(::RemoteChannel, ::Any...)\nDistributed.isready(::RemoteChannel, ::Any...)\nDistributed.isready(::Distributed.Future)\nDistributed.AbstractWorkerPool\nDistributed.WorkerPool\nDistributed.CachingPool\nDistributed.default_worker_pool\nDistributed.clear!\nDistributed.remote\nDistributed.remotecall(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remotecall_wait(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remotecall_fetch(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.remote_do(::Any, ::AbstractWorkerPool, ::Any...)\nDistributed.@spawn\nDistributed.@spawnat\nDistributed.@fetch\nDistributed.@fetchfrom\nDistributed.@distributed\nDistributed.@everywhere\nDistributed.remoteref_id\nDistributed.channel_from_id\nDistributed.worker_id_from_socket\nDistributed.cluster_cookie()\nDistributed.cluster_cookie(::Any)","category":"page"},{"location":"stdlib/Distributed.html#Distributed","page":"Distributed Computing","title":"Distributed","text":"Tools for distributed parallel processing.\n\n\n\n\n\n","category":"module"},{"location":"stdlib/Distributed.html#Distributed.addprocs","page":"Distributed Computing","title":"Distributed.addprocs","text":"addprocs(manager::ClusterManager; kwargs...) -> List of process identifiers\n\nLaunches worker processes via the specified cluster manager.\n\nFor example, Beowulf clusters are supported via a custom cluster manager implemented in the package ClusterManagers.jl.\n\nThe number of seconds a newly launched worker waits for connection establishment from the master can be specified via variable JULIA_WORKER_TIMEOUT in the worker process's environment. Relevant only when using TCP/IP as transport.\n\nTo launch workers without blocking the REPL, or the containing function if launching workers programmatically, execute addprocs in its own task.\n\nExamples\n\n# On busy clusters, call `addprocs` asynchronously\nt = @async addprocs(...)\n\n# Utilize workers as and when they come online\nif nprocs() > 1   # Ensure at least one new worker is available\n   ....   # perform distributed execution\nend\n\n# Retrieve newly launched worker IDs, or any error messages\nif istaskdone(t)   # Check if `addprocs` has completed to ensure `fetch` doesn't block\n    if nworkers() == N\n        new_pids = fetch(t)\n    else\n        fetch(t)\n    end\nend\n\n\n\n\n\naddprocs(machines; tunnel=false, sshflags=``, max_parallel=10, kwargs...) -> List of process identifiers\n\nAdd worker processes on remote machines via SSH. Configuration is done with keyword arguments (see below). In particular, the exename keyword can be used to specify the path to the julia binary on the remote machine(s).\n\nmachines is a vector of \"machine specifications\" which are given as strings of the form [user@]host[:port] [bind_addr[:port]]. user defaults to current user and port to the standard SSH port. If [bind_addr[:port]] is specified, other workers will connect to this worker at the specified bind_addr and port.\n\nIt is possible to launch multiple processes on a remote host by using a tuple in the machines vector or the form (machine_spec, count), where count is the number of workers to be launched on the specified host. Passing :auto as the worker count will launch as many workers as the number of CPU threads on the remote host.\n\nExamples:\n\naddprocs([\n    \"remote1\",               # one worker on 'remote1' logging in with the current username\n    \"user@remote2\",          # one worker on 'remote2' logging in with the 'user' username\n    \"user@remote3:2222\",     # specifying SSH port to '2222' for 'remote3'\n    (\"user@remote4\", 4),     # launch 4 workers on 'remote4'\n    (\"user@remote5\", :auto), # launch as many workers as CPU threads on 'remote5'\n])\n\nKeyword arguments:\n\ntunnel: if true then SSH tunneling will be used to connect to the worker from the master process. Default is false.\nmultiplex: if true then SSH multiplexing is used for SSH tunneling. Default is false.\nssh: the name or path of the SSH client executable used to start the workers. Default is \"ssh\".\nsshflags: specifies additional ssh options, e.g. sshflags=`-i /home/foo/bar.pem`\nmax_parallel: specifies the maximum number of workers connected to in parallel at a host. Defaults to 10.\nshell: specifies the type of shell to which ssh connects on the workers.\nshell=:posix: a POSIX-compatible Unix/Linux shell (sh, ksh, bash, dash, zsh, etc.). The default.\nshell=:csh: a Unix C shell (csh, tcsh).\nshell=:wincmd: Microsoft Windows cmd.exe.\ndir: specifies the working directory on the workers. Defaults to the host's current directory (as found by pwd())\nenable_threaded_blas: if true then  BLAS will run on multiple threads in added processes. Default is false.\nexename: name of the julia executable. Defaults to \"$(Sys.BINDIR)/julia\" or \"$(Sys.BINDIR)/julia-debug\" as the case may be. It is recommended that a common Julia version is used on all remote machines because serialization and code distribution might fail otherwise.\nexeflags: additional flags passed to the worker processes.\ntopology: Specifies how the workers connect to each other. Sending a message between unconnected workers results in an error.\ntopology=:all_to_all: All processes are connected to each other. The default.\ntopology=:master_worker: Only the driver process, i.e. pid 1 connects to the workers. The workers do not connect to each other.\ntopology=:custom: The launch method of the cluster manager specifies the connection topology via fields ident and connect_idents in WorkerConfig. A worker with a cluster manager identity ident will connect to all workers specified in connect_idents.\nlazy: Applicable only with topology=:all_to_all. If true, worker-worker connections are setup lazily, i.e. they are setup at the first instance of a remote call between workers. Default is true.\nenv: provide an array of string pairs such as env=[\"JULIA_DEPOT_PATH\"=>\"/depot\"] to request that environment variables are set on the remote machine. By default only the environment variable JULIA_WORKER_TIMEOUT is passed automatically from the local to the remote environment.\ncmdline_cookie: pass the authentication cookie via the --worker commandline  option. The (more secure) default behaviour of passing the cookie via ssh stdio  may hang with Windows workers that use older (pre-ConPTY) Julia or Windows versions,  in which case cmdline_cookie=true offers a work-around.\n\ncompat: Julia 1.6\nThe keyword arguments ssh, shell, env and cmdline_cookie were added in Julia 1.6.\n\nEnvironment variables:\n\nIf the master process fails to establish a connection with a newly launched worker within 60.0 seconds, the worker treats it as a fatal situation and terminates. This timeout can be controlled via environment variable JULIA_WORKER_TIMEOUT. The value of JULIA_WORKER_TIMEOUT on the master process specifies the number of seconds a newly launched worker waits for connection establishment.\n\n\n\n\n\naddprocs(np::Integer=Sys.CPU_THREADS; restrict=true, kwargs...) -> List of process identifiers\n\nLaunch np workers on the local host using the in-built LocalManager.\n\nLocal workers inherit the current package environment (i.e., active project, LOAD_PATH, and DEPOT_PATH) from the main process.\n\nwarning: Warning\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes.\n\nKeyword arguments:\n\nrestrict::Bool: if true (default) binding is restricted to 127.0.0.1.\ndir, exename, exeflags, env, topology, lazy, enable_threaded_blas: same effect as for SSHManager, see documentation for addprocs(machines::AbstractVector).\n\ncompat: Julia 1.9\nThe inheriting of the package environment and the env keyword argument were added in Julia 1.9.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.nprocs","page":"Distributed Computing","title":"Distributed.nprocs","text":"nprocs()\n\nGet the number of available processes.\n\nExamples\n\njulia> nprocs()\n3\n\njulia> workers()\n2-element Array{Int64,1}:\n 2\n 3\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.nworkers","page":"Distributed Computing","title":"Distributed.nworkers","text":"nworkers()\n\nGet the number of available worker processes. This is one less than nprocs(). Equal to nprocs() if nprocs() == 1.\n\nExamples\n\n$ julia -p 2\n\njulia> nprocs()\n3\n\njulia> nworkers()\n2\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.procs-Tuple{}","page":"Distributed Computing","title":"Distributed.procs","text":"procs()\n\nReturn a list of all process identifiers, including pid 1 (which is not included by workers()).\n\nExamples\n\n$ julia -p 2\n\njulia> procs()\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.procs-Tuple{Integer}","page":"Distributed Computing","title":"Distributed.procs","text":"procs(pid::Integer)\n\nReturn a list of all process identifiers on the same physical node. Specifically all workers bound to the same ip-address as pid are returned.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.workers","page":"Distributed Computing","title":"Distributed.workers","text":"workers()\n\nReturn a list of all worker process identifiers.\n\nExamples\n\n$ julia -p 2\n\njulia> workers()\n2-element Array{Int64,1}:\n 2\n 3\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.rmprocs","page":"Distributed Computing","title":"Distributed.rmprocs","text":"rmprocs(pids...; waitfor=typemax(Int))\n\nRemove the specified workers. Note that only process 1 can add or remove workers.\n\nArgument waitfor specifies how long to wait for the workers to shut down:\n\nIf unspecified, rmprocs will wait until all requested pids are removed.\nAn ErrorException is raised if all workers cannot be terminated before the requested waitfor seconds.\nWith a waitfor value of 0, the call returns immediately with the workers scheduled for removal in a different task. The scheduled Task object is returned. The user should call wait on the task before invoking any other parallel calls.\n\nExamples\n\n$ julia -p 5\n\njulia> t = rmprocs(2, 3, waitfor=0)\nTask (runnable) @0x0000000107c718d0\n\njulia> wait(t)\n\njulia> workers()\n3-element Array{Int64,1}:\n 4\n 5\n 6\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.interrupt","page":"Distributed Computing","title":"Distributed.interrupt","text":"interrupt(pids::Integer...)\n\nInterrupt the current executing task on the specified workers. This is equivalent to pressing Ctrl-C on the local machine. If no arguments are given, all workers are interrupted.\n\n\n\n\n\ninterrupt(pids::AbstractVector=workers())\n\nInterrupt the current executing task on the specified workers. This is equivalent to pressing Ctrl-C on the local machine. If no arguments are given, all workers are interrupted.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.myid","page":"Distributed Computing","title":"Distributed.myid","text":"myid()\n\nGet the id of the current process.\n\nExamples\n\njulia> myid()\n1\n\njulia> remotecall_fetch(() -> myid(), 4)\n4\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.pmap","page":"Distributed Computing","title":"Distributed.pmap","text":"pmap(f, [::AbstractWorkerPool], c...; distributed=true, batch_size=1, on_error=nothing, retry_delays=[], retry_check=nothing) -> collection\n\nTransform collection c by applying f to each element using available workers and tasks.\n\nFor multiple collection arguments, apply f elementwise.\n\nNote that f must be made available to all worker processes; see Code Availability and Loading Packages for details.\n\nIf a worker pool is not specified all available workers will be used via a CachingPool.\n\nBy default, pmap distributes the computation over all specified workers. To use only the local process and distribute over tasks, specify distributed=false. This is equivalent to using asyncmap. For example, pmap(f, c; distributed=false) is equivalent to asyncmap(f,c; ntasks=()->nworkers())\n\npmap can also use a mix of processes and tasks via the batch_size argument. For batch sizes greater than 1, the collection is processed in multiple batches, each of length batch_size or less. A batch is sent as a single request to a free worker, where a local asyncmap processes elements from the batch using multiple concurrent tasks.\n\nAny error stops pmap from processing the remainder of the collection. To override this behavior you can specify an error handling function via argument on_error which takes in a single argument, i.e., the exception. The function can stop the processing by rethrowing the error, or, to continue, return any value which is then returned inline with the results to the caller.\n\nConsider the following two examples. The first one returns the exception object inline, the second a 0 in place of any exception:\n\njulia> pmap(x->iseven(x) ? error(\"foo\") : x, 1:4; on_error=identity)\n4-element Array{Any,1}:\n 1\n  ErrorException(\"foo\")\n 3\n  ErrorException(\"foo\")\n\njulia> pmap(x->iseven(x) ? error(\"foo\") : x, 1:4; on_error=ex->0)\n4-element Array{Int64,1}:\n 1\n 0\n 3\n 0\n\nErrors can also be handled by retrying failed computations. Keyword arguments retry_delays and retry_check are passed through to retry as keyword arguments delays and check respectively. If batching is specified, and an entire batch fails, all items in the batch are retried.\n\nNote that if both on_error and retry_delays are specified, the on_error hook is called before retrying. If on_error does not throw (or rethrow) an exception, the element will not be retried.\n\nExample: On errors, retry f on an element a maximum of 3 times without any delay between retries.\n\npmap(f, c; retry_delays = zeros(3))\n\nExample: Retry f only if the exception is not of type InexactError, with exponentially increasing delays up to 3 times. Return a NaN in place for all InexactError occurrences.\n\npmap(f, c; on_error = e->(isa(e, InexactError) ? NaN : rethrow()), retry_delays = ExponentialBackOff(n = 3))\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.RemoteException","page":"Distributed Computing","title":"Distributed.RemoteException","text":"RemoteException(captured)\n\nExceptions on remote computations are captured and rethrown locally.  A RemoteException wraps the pid of the worker and a captured exception. A CapturedException captures the remote exception and a serializable form of the call stack when the exception was raised.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.ProcessExitedException","page":"Distributed Computing","title":"Distributed.ProcessExitedException","text":"ProcessExitedException(worker_id::Int)\n\nAfter a client Julia process has exited, further attempts to reference the dead child will throw this exception.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.Future","page":"Distributed Computing","title":"Distributed.Future","text":"Future(w::Int, rrid::RRID, v::Union{Some, Nothing}=nothing)\n\nA Future is a placeholder for a single computation of unknown termination status and time. For multiple potential computations, see RemoteChannel. See remoteref_id for identifying an AbstractRemoteRef.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.RemoteChannel","page":"Distributed Computing","title":"Distributed.RemoteChannel","text":"RemoteChannel(pid::Integer=myid())\n\nMake a reference to a Channel{Any}(1) on process pid. The default pid is the current process.\n\nRemoteChannel(f::Function, pid::Integer=myid())\n\nCreate references to remote channels of a specific size and type. f is a function that when executed on pid must return an implementation of an AbstractChannel.\n\nFor example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10 on pid.\n\nThe default pid is the current process.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Base.fetch-Tuple{Distributed.Future}","page":"Distributed Computing","title":"Base.fetch","text":"fetch(x::Future)\n\nWait for and get the value of a Future. The fetched value is cached locally. Further calls to fetch on the same reference return the cached value. If the remote value is an exception, throws a RemoteException which captures the remote exception and backtrace.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.fetch-Tuple{RemoteChannel}","page":"Distributed Computing","title":"Base.fetch","text":"fetch(c::RemoteChannel)\n\nWait for and get a value from a RemoteChannel. Exceptions raised are the same as for a Future. Does not remove the item fetched.\n\n\n\n\n\nfetch(x::Any)\n\nReturn x.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remotecall-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall","text":"remotecall(f, id::Integer, args...; kwargs...) -> Future\n\nCall a function f asynchronously on the given arguments on the specified process. Return a Future. Keyword arguments, if any, are passed through to f.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remotecall_wait-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_wait","text":"remotecall_wait(f, id::Integer, args...; kwargs...)\n\nPerform a faster wait(remotecall(...)) in one message on the Worker specified by worker id id. Keyword arguments, if any, are passed through to f.\n\nSee also wait and remotecall.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remotecall_fetch-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_fetch","text":"remotecall_fetch(f, id::Integer, args...; kwargs...)\n\nPerform fetch(remotecall(...)) in one message. Keyword arguments, if any, are passed through to f. Any remote exceptions are captured in a RemoteException and thrown.\n\nSee also fetch and remotecall.\n\nExamples\n\n$ julia -p 2\n\njulia> remotecall_fetch(sqrt, 2, 4)\n2.0\n\njulia> remotecall_fetch(sqrt, 2, -4)\nERROR: On worker 2:\nDomainError with -4.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\n...\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remote_do-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remote_do","text":"remote_do(f, id::Integer, args...; kwargs...) -> nothing\n\nExecutes f on worker id asynchronously. Unlike remotecall, it does not store the result of computation, nor is there a way to wait for its completion.\n\nA successful invocation indicates that the request has been accepted for execution on the remote node.\n\nWhile consecutive remotecalls to the same worker are serialized in the order they are invoked, the order of executions on the remote worker is undetermined. For example, remote_do(f1, 2); remotecall(f2, 2); remote_do(f3, 2) will serialize the call to f1, followed by f2 and f3 in that order. However, it is not guaranteed that f1 is executed before f3 on worker 2.\n\nAny exceptions thrown by f are printed to stderr on the remote worker.\n\nKeyword arguments, if any, are passed through to f.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.put!-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.put!","text":"put!(rr::RemoteChannel, args...)\n\nStore a set of values to the RemoteChannel. If the channel is full, blocks until space is available. Return the first argument.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.put!-Tuple{Distributed.Future, Any}","page":"Distributed Computing","title":"Base.put!","text":"put!(rr::Future, v)\n\nStore a value to a Future rr. Futures are write-once remote references. A put! on an already set Future throws an Exception. All asynchronous remote calls return Futures and set the value to the return value of the call upon completion.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.take!-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.take!","text":"take!(rr::RemoteChannel, args...)\n\nFetch value(s) from a RemoteChannel rr, removing the value(s) in the process.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.isready-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.isready","text":"isready(rr::RemoteChannel, args...)\n\nDetermine whether a RemoteChannel has a value stored to it. Note that this function can cause race conditions, since by the time you receive its result it may no longer be true. However, it can be safely used on a Future since they are assigned only once.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Base.isready-Tuple{Distributed.Future}","page":"Distributed Computing","title":"Base.isready","text":"isready(rr::Future)\n\nDetermine whether a Future has a value stored to it.\n\nIf the argument Future is owned by a different node, this call will block to wait for the answer. It is recommended to wait for rr in a separate task instead or to use a local Channel as a proxy:\n\np = 1\nf = Future(p)\nerrormonitor(@async put!(f, remotecall_fetch(long_computation, p)))\nisready(f)  # will not block\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.AbstractWorkerPool","page":"Distributed Computing","title":"Distributed.AbstractWorkerPool","text":"AbstractWorkerPool\n\nSupertype for worker pools such as WorkerPool and CachingPool. An AbstractWorkerPool should implement:\n\npush! - add a new worker to the overall pool (available + busy)\nput! - put back a worker to the available pool\ntake! - take a worker from the available pool (to be used for remote function execution)\nlength - number of workers available in the overall pool\nisready - return false if a take! on the pool would block, else true\n\nThe default implementations of the above (on a AbstractWorkerPool) require fields\n\nchannel::Channel{Int}\nworkers::Set{Int}\n\nwhere channel contains free worker pids and workers is the set of all workers associated with this pool.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.WorkerPool","page":"Distributed Computing","title":"Distributed.WorkerPool","text":"WorkerPool(workers::Union{Vector{Int},AbstractRange{Int}})\n\nCreate a WorkerPool from a vector or range of worker ids.\n\nExamples\n\n$ julia -p 3\n\njulia> WorkerPool([2, 3])\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:2), Set([2, 3]), RemoteChannel{Channel{Any}}(1, 1, 6))\n\njulia> WorkerPool(2:4)\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:2), Set([4, 2, 3]), RemoteChannel{Channel{Any}}(1, 1, 7))\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.CachingPool","page":"Distributed Computing","title":"Distributed.CachingPool","text":"CachingPool(workers::Vector{Int})\n\nAn implementation of an AbstractWorkerPool. remote, remotecall_fetch, pmap (and other remote calls which execute functions remotely) benefit from caching the serialized/deserialized functions on the worker nodes, especially closures (which may capture large amounts of data).\n\nThe remote cache is maintained for the lifetime of the returned CachingPool object. To clear the cache earlier, use clear!(pool).\n\nFor global variables, only the bindings are captured in a closure, not the data. let blocks can be used to capture global data.\n\nExamples\n\nconst foo = rand(10^8);\nwp = CachingPool(workers())\nlet foo = foo\n    pmap(i -> sum(foo) + i, wp, 1:100);\nend\n\nThe above would transfer foo only once to each worker.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.default_worker_pool","page":"Distributed Computing","title":"Distributed.default_worker_pool","text":"default_worker_pool()\n\nAbstractWorkerPool containing idle workers - used by remote(f) and pmap (by default). Unless one is explicitly set via default_worker_pool!(pool), the default worker pool is initialized to a WorkerPool.\n\nExamples\n\n$ julia -p 3\n\njulia> default_worker_pool()\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:3), Set([4, 2, 3]), RemoteChannel{Channel{Any}}(1, 1, 4))\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.clear!","page":"Distributed Computing","title":"Distributed.clear!","text":"clear!(syms, pids=workers(); mod=Main)\n\nClears global bindings in modules by initializing them to nothing. syms should be of type Symbol or a collection of Symbols . pids and mod identify the processes and the module in which global variables are to be reinitialized. Only those names found to be defined under mod are cleared.\n\nAn exception is raised if a global constant is requested to be cleared.\n\n\n\n\n\nclear!(pool::CachingPool) -> pool\n\nRemoves all cached functions from all participating workers.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.remote","page":"Distributed Computing","title":"Distributed.remote","text":"remote([p::AbstractWorkerPool], f) -> Function\n\nReturn an anonymous function that executes function f on an available worker (drawn from WorkerPool p if provided) using remotecall_fetch.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.remotecall-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall","text":"remotecall(f, pool::AbstractWorkerPool, args...; kwargs...) -> Future\n\nWorkerPool variant of remotecall(f, pid, ....). Wait for and take a free worker from pool and perform a remotecall on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> f = remotecall(maximum, wp, A)\nFuture(2, 1, 6, nothing)\n\nIn this example, the task ran on pid 2, called from pid 1.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remotecall_wait-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_wait","text":"remotecall_wait(f, pool::AbstractWorkerPool, args...; kwargs...) -> Future\n\nWorkerPool variant of remotecall_wait(f, pid, ....). Wait for and take a free worker from pool and perform a remotecall_wait on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> f = remotecall_wait(maximum, wp, A)\nFuture(3, 1, 9, nothing)\n\njulia> fetch(f)\n0.9995177101692958\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remotecall_fetch-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_fetch","text":"remotecall_fetch(f, pool::AbstractWorkerPool, args...; kwargs...) -> result\n\nWorkerPool variant of remotecall_fetch(f, pid, ....). Waits for and takes a free worker from pool and performs a remotecall_fetch on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> remotecall_fetch(maximum, wp, A)\n0.9995177101692958\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.remote_do-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remote_do","text":"remote_do(f, pool::AbstractWorkerPool, args...; kwargs...) -> nothing\n\nWorkerPool variant of remote_do(f, pid, ....). Wait for and take a free worker from pool and perform a remote_do on it.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.@spawn","page":"Distributed Computing","title":"Distributed.@spawn","text":"@spawn expr\n\nCreate a closure around an expression and run it on an automatically-chosen process, returning a Future to the result. This macro is deprecated; @spawnat :any expr should be used instead.\n\nExamples\n\njulia> addprocs(3);\n\njulia> f = @spawn myid()\nFuture(2, 1, 5, nothing)\n\njulia> fetch(f)\n2\n\njulia> f = @spawn myid()\nFuture(3, 1, 7, nothing)\n\njulia> fetch(f)\n3\n\ncompat: Julia 1.3\nAs of Julia 1.3 this macro is deprecated. Use @spawnat :any instead.\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.@spawnat","page":"Distributed Computing","title":"Distributed.@spawnat","text":"@spawnat p expr\n\nCreate a closure around an expression and run the closure asynchronously on process p. Return a Future to the result. If p is the quoted literal symbol :any, then the system will pick a processor to use automatically.\n\nExamples\n\njulia> addprocs(3);\n\njulia> f = @spawnat 2 myid()\nFuture(2, 1, 3, nothing)\n\njulia> fetch(f)\n2\n\njulia> f = @spawnat :any myid()\nFuture(3, 1, 7, nothing)\n\njulia> fetch(f)\n3\n\ncompat: Julia 1.3\nThe :any argument is available as of Julia 1.3.\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.@fetch","page":"Distributed Computing","title":"Distributed.@fetch","text":"@fetch expr\n\nEquivalent to fetch(@spawnat :any expr). See fetch and @spawnat.\n\nExamples\n\njulia> addprocs(3);\n\njulia> @fetch myid()\n2\n\njulia> @fetch myid()\n3\n\njulia> @fetch myid()\n4\n\njulia> @fetch myid()\n2\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.@fetchfrom","page":"Distributed Computing","title":"Distributed.@fetchfrom","text":"@fetchfrom\n\nEquivalent to fetch(@spawnat p expr). See fetch and @spawnat.\n\nExamples\n\njulia> addprocs(3);\n\njulia> @fetchfrom 2 myid()\n2\n\njulia> @fetchfrom 4 myid()\n4\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.@distributed","page":"Distributed Computing","title":"Distributed.@distributed","text":"@distributed\n\nA distributed memory, parallel for loop of the form :\n\n@distributed [reducer] for var = range\n    body\nend\n\nThe specified range is partitioned and locally executed across all workers. In case an optional reducer function is specified, @distributed performs local reductions on each worker with a final reduction on the calling process.\n\nNote that without a reducer function, @distributed executes asynchronously, i.e. it spawns independent tasks on all available workers and returns immediately without waiting for completion. To wait for completion, prefix the call with @sync, like :\n\n@sync @distributed for var = range\n    body\nend\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.@everywhere","page":"Distributed Computing","title":"Distributed.@everywhere","text":"@everywhere [procs()] expr\n\nExecute an expression under Main on all procs. Errors on any of the processes are collected into a CompositeException and thrown. For example:\n\n@everywhere bar = 1\n\nwill define Main.bar on all current processes. Any processes added later (say with addprocs()) will not have the expression defined.\n\nUnlike @spawnat, @everywhere does not capture any local variables. Instead, local variables can be broadcast using interpolation:\n\nfoo = 1\n@everywhere bar = $foo\n\nThe optional argument procs allows specifying a subset of all processes to have execute the expression.\n\nSimilar to calling remotecall_eval(Main, procs, expr), but with two extra features:\n\n- `using` and `import` statements run on the calling process first, to ensure\n  packages are precompiled.\n- The current source file path used by `include` is propagated to other processes.\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Distributed.html#Distributed.remoteref_id","page":"Distributed Computing","title":"Distributed.remoteref_id","text":"remoteref_id(r::AbstractRemoteRef) -> RRID\n\nFutures and RemoteChannels are identified by fields:\n\nwhere - refers to the node where the underlying object/storage referred to by the reference actually exists.\nwhence - refers to the node the remote reference was created from. Note that this is different from the node where the underlying object referred to actually exists. For example calling RemoteChannel(2) from the master process would result in a where value of 2 and a whence value of 1.\nid is unique across all references created from the worker specified by whence.\n\nTaken together,  whence and id uniquely identify a reference across all workers.\n\nremoteref_id is a low-level API which returns a RRID object that wraps whence and id values of a remote reference.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.channel_from_id","page":"Distributed Computing","title":"Distributed.channel_from_id","text":"channel_from_id(id) -> c\n\nA low-level API which returns the backing AbstractChannel for an id returned by remoteref_id. The call is valid only on the node where the backing channel exists.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.worker_id_from_socket","page":"Distributed Computing","title":"Distributed.worker_id_from_socket","text":"worker_id_from_socket(s) -> pid\n\nA low-level API which, given a IO connection or a Worker, returns the pid of the worker it is connected to. This is useful when writing custom serialize methods for a type, which optimizes the data written out depending on the receiving process id.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.cluster_cookie-Tuple{}","page":"Distributed Computing","title":"Distributed.cluster_cookie","text":"cluster_cookie() -> cookie\n\nReturn the cluster cookie.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.cluster_cookie-Tuple{Any}","page":"Distributed Computing","title":"Distributed.cluster_cookie","text":"cluster_cookie(cookie) -> cookie\n\nSet the passed cookie as the cluster cookie, then returns it.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Cluster-Manager-Interface","page":"Distributed Computing","title":"Cluster Manager Interface","text":"","category":"section"},{"location":"stdlib/Distributed.html","page":"Distributed Computing","title":"Distributed Computing","text":"This interface provides a mechanism to launch and manage Julia workers on different cluster environments. There are two types of managers present in Base: LocalManager, for launching additional workers on the same host, and SSHManager, for launching on remote hosts via ssh. TCP/IP sockets are used to connect and transport messages between processes. It is possible for Cluster Managers to provide a different transport.","category":"page"},{"location":"stdlib/Distributed.html","page":"Distributed Computing","title":"Distributed Computing","text":"Distributed.ClusterManager\nDistributed.WorkerConfig\nDistributed.launch\nDistributed.manage\nDistributed.kill(::ClusterManager, ::Int, ::WorkerConfig)\nDistributed.connect(::ClusterManager, ::Int, ::WorkerConfig)\nDistributed.init_worker\nDistributed.start_worker\nDistributed.process_messages\nDistributed.default_addprocs_params","category":"page"},{"location":"stdlib/Distributed.html#Distributed.ClusterManager","page":"Distributed Computing","title":"Distributed.ClusterManager","text":"ClusterManager\n\nSupertype for cluster managers, which control workers processes as a cluster. Cluster managers implement how workers can be added, removed and communicated with. SSHManager and LocalManager are subtypes of this.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.WorkerConfig","page":"Distributed Computing","title":"Distributed.WorkerConfig","text":"WorkerConfig\n\nType used by ClusterManagers to control workers added to their clusters. Some fields are used by all cluster managers to access a host:\n\nio – the connection used to access the worker (a subtype of IO or Nothing)\nhost – the host address (either a String or Nothing)\nport – the port on the host used to connect to the worker (either an Int or Nothing)\n\nSome are used by the cluster manager to add workers to an already-initialized host:\n\ncount – the number of workers to be launched on the host\nexename – the path to the Julia executable on the host, defaults to \"$(Sys.BINDIR)/julia\" or \"$(Sys.BINDIR)/julia-debug\"\nexeflags – flags to use when launching Julia remotely\n\nThe userdata field is used to store information for each worker by external managers.\n\nSome fields are used by SSHManager and similar managers:\n\ntunnel – true (use tunneling), false (do not use tunneling), or nothing (use default for the manager)\nmultiplex – true (use SSH multiplexing for tunneling) or false\nforward – the forwarding option used for -L option of ssh\nbind_addr – the address on the remote host to bind to\nsshflags – flags to use in establishing the SSH connection\nmax_parallel – the maximum number of workers to connect to in parallel on the host\n\nSome fields are used by both LocalManagers and SSHManagers:\n\nconnect_at – determines whether this is a worker-to-worker or driver-to-worker setup call\nprocess – the process which will be connected (usually the manager will assign this during addprocs)\nospid – the process ID according to the host OS, used to interrupt worker processes\nenviron – private dictionary used to store temporary information by Local/SSH managers\nident – worker as identified by the ClusterManager\nconnect_idents – list of worker ids the worker must connect to if using a custom topology\nenable_threaded_blas – true, false, or nothing, whether to use threaded BLAS or not on the workers\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Distributed.html#Distributed.launch","page":"Distributed Computing","title":"Distributed.launch","text":"launch(manager::ClusterManager, params::Dict, launched::Array, launch_ntfy::Condition)\n\nImplemented by cluster managers. For every Julia worker launched by this function, it should append a WorkerConfig entry to launched and notify launch_ntfy. The function MUST exit once all workers, requested by manager have been launched. params is a dictionary of all keyword arguments addprocs was called with.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.manage","page":"Distributed Computing","title":"Distributed.manage","text":"manage(manager::ClusterManager, id::Integer, config::WorkerConfig. op::Symbol)\n\nImplemented by cluster managers. It is called on the master process, during a worker's lifetime, with appropriate op values:\n\nwith :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Base.kill-Tuple{ClusterManager, Int64, WorkerConfig}","page":"Distributed Computing","title":"Base.kill","text":"kill(manager::ClusterManager, pid::Int, config::WorkerConfig)\n\nImplemented by cluster managers. It is called on the master process, by rmprocs. It should cause the remote worker specified by pid to exit. kill(manager::ClusterManager.....) executes a remote exit() on pid.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Sockets.connect-Tuple{ClusterManager, Int64, WorkerConfig}","page":"Distributed Computing","title":"Sockets.connect","text":"connect(manager::ClusterManager, pid::Int, config::WorkerConfig) -> (instrm::IO, outstrm::IO)\n\nImplemented by cluster managers using custom transports. It should establish a logical connection to worker with id pid, specified by config and return a pair of IO objects. Messages from pid to current process will be read off instrm, while messages to be sent to pid will be written to outstrm. The custom transport implementation must ensure that messages are delivered and received completely and in order. connect(manager::ClusterManager.....) sets up TCP/IP socket connections in-between workers.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Distributed.html#Distributed.init_worker","page":"Distributed Computing","title":"Distributed.init_worker","text":"init_worker(cookie::AbstractString, manager::ClusterManager=DefaultClusterManager())\n\nCalled by cluster managers implementing custom transports. It initializes a newly launched process as a worker. Command line argument --worker[=<cookie>] has the effect of initializing a process as a worker using TCP/IP sockets for transport. cookie is a cluster_cookie.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.start_worker","page":"Distributed Computing","title":"Distributed.start_worker","text":"start_worker([out::IO=stdout], cookie::AbstractString=readline(stdin); close_stdin::Bool=true, stderr_to_stdout::Bool=true)\n\nstart_worker is an internal function which is the default entry point for worker processes connecting via TCP/IP. It sets up the process as a Julia cluster worker.\n\nhost:port information is written to stream out (defaults to stdout).\n\nThe function reads the cookie from stdin if required, and  listens on a free port (or if specified, the port in the --bind-to command line option) and schedules tasks to process incoming TCP connections and requests. It also (optionally) closes stdin and redirects stderr to stdout.\n\nIt does not return.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.process_messages","page":"Distributed Computing","title":"Distributed.process_messages","text":"process_messages(r_stream::IO, w_stream::IO, incoming::Bool=true)\n\nCalled by cluster managers using custom transports. It should be called when the custom transport implementation receives the first message from a remote worker. The custom transport must manage a logical connection to the remote worker and provide two IO objects, one for incoming messages and the other for messages addressed to the remote worker. If incoming is true, the remote peer initiated the connection. Whichever of the pair initiates the connection sends the cluster cookie and its Julia version number to perform the authentication handshake.\n\nSee also cluster_cookie.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Distributed.html#Distributed.default_addprocs_params","page":"Distributed Computing","title":"Distributed.default_addprocs_params","text":"default_addprocs_params(mgr::ClusterManager) -> Dict{Symbol, Any}\n\nImplemented by cluster managers. The default keyword parameters passed when calling addprocs(mgr). The minimal set of options is available by calling default_addprocs_params()\n\n\n\n\n\n","category":"function"},{"location":"devdocs/external_profilers.html#External-Profiler-Support","page":"External Profiler Support","title":"External Profiler Support","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Julia provides explicit support for some external tracing profilers, enabling you to obtain a high-level overview of the runtime's execution behavior.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The currently supported profilers are:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Tracy\nIntel VTune (ITTAPI)","category":"page"},{"location":"devdocs/external_profilers.html#Adding-New-Zones","page":"External Profiler Support","title":"Adding New Zones","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"To add new zones, use the JL_TIMING macro. You can find numerous examples throughout the codebase by searching for JL_TIMING. To add a new type of zone you add it to JL_TIMING_OWNERS (and possibly JL_TIMING_EVENTS).","category":"page"},{"location":"devdocs/external_profilers.html#Dynamically-Enabling-and-Disabling-Zones","page":"External Profiler Support","title":"Dynamically Enabling and Disabling Zones","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The JULIA_TIMING_SUBSYSTEMS environment variable allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE zones.","category":"page"},{"location":"devdocs/external_profilers.html#Tracy-Profiler","page":"External Profiler Support","title":"Tracy Profiler","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Tracy  is a flexible profiler that can be optionally integrated with Julia.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"A typical Tracy session might look like this:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"(Image: Typical Tracy usage)","category":"page"},{"location":"devdocs/external_profilers.html#Building-Julia-with-Tracy","page":"External Profiler Support","title":"Building Julia with Tracy","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"To enable Tracy integration, build Julia with the extra option WITH_TRACY=1 in the Make.user file.","category":"page"},{"location":"devdocs/external_profilers.html#Installing-the-Tracy-Profile-Viewer","page":"External Profiler Support","title":"Installing the Tracy Profile Viewer","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The easiest way to obtain the profile viewer is by adding the TracyProfiler_jll package and launching the profiler with:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"run(TracyProfiler_jll.tracy())","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"note: Note\nOn macOS, you may want to set the TRACY_DPI_SCALE environment variable to 1.0 if the UI elements in the profiler appear excessively large.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"To run a \"headless\" instance that saves the trace to disk, use","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"run(`$(TracyProfiler_jll.capture()) -o mytracefile.tracy`)","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"instead.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"For information on using the Tracy UI, refer to the Tracy manual.","category":"page"},{"location":"devdocs/external_profilers.html#Profiling-Julia-with-Tracy","page":"External Profiler Support","title":"Profiling Julia with Tracy","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"A typical workflow for profiling Julia with Tracy involves starting Julia using:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"JULIA_WAIT_FOR_TRACY=1 ./julia -e '...'","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The environment variable ensures that Julia waits until it has successfully connected to the Tracy profiler before continuing execution. Afterward, use the Tracy profiler UI, click Connect, and Julia execution should resume and profiling should start.","category":"page"},{"location":"devdocs/external_profilers.html#Profiling-package-precompilation-with-Tracy","page":"External Profiler Support","title":"Profiling package precompilation with Tracy","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"To profile a package precompilation process it is easiest to explicitly call into Base.compilecache with the package you want to precompile:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"pkg = Base.identify_package(\"SparseArrays\")\nwithenv(\"JULIA_WAIT_FOR_TRACY\" => 1, \"TRACY_PORT\" => 9001) do\n    Base.compilecache(pkg)\nend","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Here, we use a custom port for tracy which makes it easier to find the correct client in the Tracy UI to connect to.","category":"page"},{"location":"devdocs/external_profilers.html#Adding-metadata-to-zones","page":"External Profiler Support","title":"Adding metadata to zones","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The various jl_timing_show_* and jl_timing_printf functions can be used to attach a string (or strings) to a zone. For example, the trace zone for inference shows the method instance that is being inferred.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"The TracyCZoneColor function can be used to set the color of a certain zone. Search through the codebase to see how it is used.","category":"page"},{"location":"devdocs/external_profilers.html#Viewing-Tracy-files-in-your-browser","page":"External Profiler Support","title":"Viewing Tracy files in your browser","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Visit https://topolarity.github.io/trace-viewer/ for an (experimental) web viewer for Tracy traces.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"You can open a local .tracy file or provide a URL from the web (e.g. a file in a Github repo). If you load a trace file from the web, you can also share the page URL directly with others, enabling them to view the same trace.","category":"page"},{"location":"devdocs/external_profilers.html#Enabling-stack-trace-samples","page":"External Profiler Support","title":"Enabling stack trace samples","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"To enable call stack sampling in Tracy, build Julia with these options in your Make.user file:","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"WITH_TRACY := 1\nWITH_TRACY_CALLSTACKS := 1\nUSE_BINARYBUILDER_LIBTRACYCLIENT := 0","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"You may also need to run make -C deps clean-libtracyclient to force a re-build of Tracy.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"This feature has a significant impact on trace size and profiling overhead, so it is recommended to leave call stack sampling off when possible, especially if you intend to share your trace files online.","category":"page"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"Note that the Julia JIT runtime does not yet have integration for Tracy's symbolification, so Julia functions will typically be unknown in these stack traces.","category":"page"},{"location":"devdocs/external_profilers.html#Intel-VTune-(ITTAPI)-Profiler","page":"External Profiler Support","title":"Intel VTune (ITTAPI) Profiler","text":"","category":"section"},{"location":"devdocs/external_profilers.html","page":"External Profiler Support","title":"External Profiler Support","text":"This section is yet to be written.","category":"page"},{"location":"devdocs/require.html#Module-loading","page":"Module loading","title":"Module loading","text":"","category":"section"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"Base.require is responsible for loading modules and it also manages the precompilation cache. It is the implementation of the import statement.","category":"page"},{"location":"devdocs/require.html#Experimental-features","page":"Module loading","title":"Experimental features","text":"","category":"section"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"The features below are experimental and not part of the stable Julia API. Before building upon them inform yourself about the current thinking and whether they might change soon.","category":"page"},{"location":"devdocs/require.html#Package-loading-callbacks","page":"Module loading","title":"Package loading callbacks","text":"","category":"section"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"It is possible to listen to the packages loaded by Base.require, by registering a callback.","category":"page"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"loaded_packages = Base.PkgId[]\ncallback = (pkg::Base.PkgId) -> push!(loaded_packages, pkg)\npush!(Base.package_callbacks, callback)","category":"page"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"Using this would look something like:","category":"page"},{"location":"devdocs/require.html","page":"Module loading","title":"Module loading","text":"julia> using Example\n\njulia> loaded_packages\n1-element Vector{Base.PkgId}:\n Example [7876af07-990d-54b4-ab0e-23690620f79a]","category":"page"},{"location":"stdlib/JuliaSyntaxHighlighting.html#Julia-Syntax-Highlighting","page":"Julia Syntax Highlighting","title":"Julia Syntax Highlighting","text":"","category":"section"},{"location":"stdlib/JuliaSyntaxHighlighting.html","page":"Julia Syntax Highlighting","title":"Julia Syntax Highlighting","text":"JuliaSyntaxHighlighting.highlight\nJuliaSyntaxHighlighting.highlight!","category":"page"},{"location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight","page":"Julia Syntax Highlighting","title":"JuliaSyntaxHighlighting.highlight","text":"highlight(content::Union{AbstractString, IOBuffer, IOContext{IOBuffer}})\n\nApply syntax highlighting to content using JuliaSyntax.\n\nReturns an AnnotatedString{String}.\n\nExamples\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\") |> Base.annotations\n6-element Vector{Tuple{UnitRange{Int64}, Pair{Symbol, Any}}}:\n (1:3, :face => :julia_funcall)\n (4:4, :face => :julia_rainbow_paren_1)\n (5:5, :face => :julia_number)\n (6:6, :face => :julia_operator)\n (7:7, :face => :julia_number)\n (8:8, :face => :julia_rainbow_paren_1)\n\n\n\n\n\n","category":"function"},{"location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight!","page":"Julia Syntax Highlighting","title":"JuliaSyntaxHighlighting.highlight!","text":"highlight!(content::Union{AnnotatedString, SubString{AnnotatedString}})\n\nModify content by applying syntax highlighting using JuliaSyntax.\n\nExamples\n\njulia> str = Base.AnnotatedString(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight!(str)\n\"sum(1:8)\"\n\njulia> Base.annotations(str)\n6-element Vector{Tuple{UnitRange{Int64}, Pair{Symbol, Any}}}:\n (1:3, :face => :julia_funcall)\n (4:4, :face => :julia_rainbow_paren_1)\n (5:5, :face => :julia_number)\n (6:6, :face => :julia_operator)\n (7:7, :face => :julia_number)\n (8:8, :face => :julia_rainbow_paren_1)\n\n\n\n\n\n","category":"function"},{"location":"base/constants.html#lib-constants","page":"Constants","title":"Constants","text":"","category":"section"},{"location":"base/constants.html","page":"Constants","title":"Constants","text":"Core.nothing\nBase.PROGRAM_FILE\nBase.ARGS\nBase.C_NULL\nBase.VERSION\nBase.DEPOT_PATH\nBase.LOAD_PATH\nBase.Sys.BINDIR\nBase.Sys.CPU_THREADS\nBase.Sys.WORD_SIZE\nBase.Sys.KERNEL\nBase.Sys.ARCH\nBase.Sys.MACHINE","category":"page"},{"location":"base/constants.html#Core.nothing","page":"Constants","title":"Core.nothing","text":"nothing\n\nThe singleton instance of type Nothing, used by convention when there is no value to return (as in a C void function) or when a variable or field holds no value.\n\nSee also: isnothing, something, missing.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.PROGRAM_FILE","page":"Constants","title":"Base.PROGRAM_FILE","text":"PROGRAM_FILE\n\nA string containing the script name passed to Julia from the command line. Note that the script name remains unchanged from within included files. Alternatively see @__FILE__.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.ARGS","page":"Constants","title":"Base.ARGS","text":"ARGS\n\nAn array of the command line arguments passed to Julia, as strings.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.C_NULL","page":"Constants","title":"Base.C_NULL","text":"C_NULL\n\nThe C null pointer constant, sometimes used when calling external code.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.VERSION","page":"Constants","title":"Base.VERSION","text":"VERSION\n\nA VersionNumber object describing which version of Julia is in use. See also Version Number Literals.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.DEPOT_PATH","page":"Constants","title":"Base.DEPOT_PATH","text":"DEPOT_PATH\n\nA stack of \"depot\" locations where the package manager, as well as Julia's code loading mechanisms, look for package registries, installed packages, named environments, repo clones, cached compiled package images, and configuration files. By default it includes:\n\n~/.julia where ~ is the user home as appropriate on the system;\nan architecture-specific shared system directory, e.g. /usr/local/share/julia;\nan architecture-independent shared system directory, e.g. /usr/share/julia.\n\nSo DEPOT_PATH might be:\n\n[joinpath(homedir(), \".julia\"), \"/usr/local/share/julia\", \"/usr/share/julia\"]\n\nThe first entry is the \"user depot\" and should be writable by and owned by the current user. The user depot is where: registries are cloned, new package versions are installed, named environments are created and updated, package repos are cloned, newly compiled package image files are saved, log files are written, development packages are checked out by default, and global configuration data is saved. Later entries in the depot path are treated as read-only and are appropriate for registries, packages, etc. installed and managed by system administrators.\n\nDEPOT_PATH is populated based on the JULIA_DEPOT_PATH environment variable if set.\n\nDEPOT_PATH contents\n\nEach entry in DEPOT_PATH is a path to a directory which contains subdirectories used by Julia for various purposes. Here is an overview of some of the subdirectories that may exist in a depot:\n\nartifacts: Contains content that packages use for which Pkg manages the installation of.\nclones: Contains full clones of package repos. Maintained by Pkg.jl and used as a cache.\nconfig: Contains julia-level configuration such as a startup.jl\ncompiled: Contains precompiled *.ji files for packages. Maintained by Julia.\ndev: Default directory for Pkg.develop. Maintained by Pkg.jl and the user.\nenvironments: Default package environments. For instance the global environment for a specific julia version. Maintained by Pkg.jl.\nlogs: Contains logs of Pkg and REPL operations. Maintained by Pkg.jl and Julia.\npackages: Contains packages, some of which were explicitly installed and some which are implicit dependencies. Maintained by Pkg.jl.\nregistries: Contains package registries. By default only General. Maintained by Pkg.jl.\nscratchspaces: Contains content that a package itself installs via the Scratch.jl package. Pkg.gc() will delete content that is known to be unused.\n\nnote: Note\nPackages that want to store content should use the scratchspaces subdirectory via Scratch.jl instead of creating new subdirectories in the depot root.\n\nSee also JULIA_DEPOT_PATH, and Code Loading.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.LOAD_PATH","page":"Constants","title":"Base.LOAD_PATH","text":"LOAD_PATH\n\nAn array of paths for using and import statements to consider as project environments or package directories when loading code. It is populated based on the JULIA_LOAD_PATH environment variable if set; otherwise it defaults to [\"@\", \"@v#.#\", \"@stdlib\"]. Entries starting with @ have special meanings:\n\n@ refers to the \"current active environment\", the initial value of which is initially determined by the JULIA_PROJECT environment variable or the --project command-line option.\n@stdlib expands to the absolute path of the current Julia installation's standard library directory.\n@name refers to a named environment, which are stored in depots (see JULIA_DEPOT_PATH) under the environments subdirectory. The user's named environments are stored in ~/.julia/environments so @name would refer to the environment in ~/.julia/environments/name if it exists and contains a Project.toml file. If name contains # characters, then they are replaced with the major, minor and patch components of the Julia version number. For example, if you are running Julia 1.2 then @v#.# expands to @v1.2 and will look for an environment by that name, typically at ~/.julia/environments/v1.2.\n\nThe fully expanded value of LOAD_PATH that is searched for projects and packages can be seen by calling the Base.load_path() function.\n\nSee also JULIA_LOAD_PATH, JULIA_PROJECT, JULIA_DEPOT_PATH, and Code Loading.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.BINDIR","page":"Constants","title":"Base.Sys.BINDIR","text":"Sys.BINDIR::String\n\nA string containing the full path to the directory containing the julia executable.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.CPU_THREADS","page":"Constants","title":"Base.Sys.CPU_THREADS","text":"Sys.CPU_THREADS::Int\n\nThe number of logical CPU cores available in the system, i.e. the number of threads that the CPU can run concurrently. Note that this is not necessarily the number of CPU cores, for example, in the presence of hyper-threading.\n\nSee Hwloc.jl or CpuId.jl for extended information, including number of physical cores.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.WORD_SIZE","page":"Constants","title":"Base.Sys.WORD_SIZE","text":"Sys.WORD_SIZE::Int\n\nStandard word size on the current machine, in bits.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.KERNEL","page":"Constants","title":"Base.Sys.KERNEL","text":"Sys.KERNEL::Symbol\n\nA symbol representing the name of the operating system, as returned by uname of the build configuration.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.ARCH","page":"Constants","title":"Base.Sys.ARCH","text":"Sys.ARCH::Symbol\n\nA symbol representing the architecture of the build configuration.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html#Base.Sys.MACHINE","page":"Constants","title":"Base.Sys.MACHINE","text":"Sys.MACHINE::String\n\nA string containing the build triple.\n\n\n\n\n\n","category":"constant"},{"location":"base/constants.html","page":"Constants","title":"Constants","text":"See also:","category":"page"},{"location":"base/constants.html","page":"Constants","title":"Constants","text":"stdin\nstdout\nstderr\nENV\nENDIAN_BOM","category":"page"},{"location":"devdocs/llvm-passes.html#Custom-LLVM-Passes","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Julia has a number of custom LLVM passes. Broadly, they can be classified into passes that are required to be run to maintain Julia semantics, and passes that take advantage of Julia semantics to optimize LLVM IR.","category":"page"},{"location":"devdocs/llvm-passes.html#Semantic-Passes","page":"Custom LLVM Passes","title":"Semantic Passes","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"These passes are used to transform LLVM IR into code that is legal to be run on a CPU. Their main purpose is to enable simpler IR to be emitted by codegen, which then enables other LLVM passes to optimize common patterns.","category":"page"},{"location":"devdocs/llvm-passes.html#CPUFeatures","page":"Custom LLVM Passes","title":"CPUFeatures","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-cpufeatures.cpp\nClass Name: CPUFeaturesPass\nOpt Name: module(CPUFeatures)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass lowers the julia.cpu.have_fma.(f32|f64) intrinsic to either true or false, depending on the target architecture and target features present on the function. This intrinsic is often used to determine if using algorithms dependent on fast fused multiply-add operations is better than using standard algorithms not dependent on such instructions.","category":"page"},{"location":"devdocs/llvm-passes.html#DemoteFloat16","page":"Custom LLVM Passes","title":"DemoteFloat16","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-demote-float16.cpp\nClassName: DemoteFloat16Pass\nOpt Name function(DemoteFloat16)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass replaces float16 operations with float32 operations on architectures that do not natively support float16 operations. This is done by inserting fpext and fptrunc instructions around any float16 operation. On architectures that do support native float16 operations, this pass is a no-op.","category":"page"},{"location":"devdocs/llvm-passes.html#LateGCLowering","page":"Custom LLVM Passes","title":"LateGCLowering","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-late-gc-lowering.cpp\nClass Name: LateLowerGCPass\nOpt Name: function(LateLowerGCFrame)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass performs most of the GC rooting work required to track pointers between GC safepoints. It also lowers several intrinsics to their corresponding instruction translation, and is permitted to violate the non-integral invariants previously established (pointer_from_objref is lowered to a ptrtoint instruction here). This pass typically occupies the most time out of all the custom Julia passes, due to its dataflow algorithm to minimize the number of objects live at any safepoint.","category":"page"},{"location":"devdocs/llvm-passes.html#FinalGCLowering","page":"Custom LLVM Passes","title":"FinalGCLowering","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-final-gc-lowering.cpp\nClass Name: FinalLowerGCPass\nOpt Name: module(FinalLowerGC)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass lowers a few last intrinsics to their final form targeting functions in the libjulia library. Separating this from LateGCLowering enables other backends (GPU compilation) to supply their own custom lowerings for these intrinsics, enabling the Julia pipeline to be used on those backends as well.","category":"page"},{"location":"devdocs/llvm-passes.html#LowerHandlers","page":"Custom LLVM Passes","title":"LowerHandlers","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-lower-handlers.cpp\nClass Name: LowerExcHandlersPass\nOpt Name: function(LowerExcHandlers)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass lowers exception handling intrinsics into calls to runtime functions that are actually called when handling exceptions.","category":"page"},{"location":"devdocs/llvm-passes.html#RemoveNI","page":"Custom LLVM Passes","title":"RemoveNI","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-remove-ni.cpp\nClass Name: RemoveNIPass\nOpt Name: module(RemoveNI)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass removes the non-integral address spaces from the module's datalayout string. This enables the backend to lower Julia's custom address spaces directly to machine code, without a costly rewrite of every pointer operation to address space 0.","category":"page"},{"location":"devdocs/llvm-passes.html#SIMDLoop","page":"Custom LLVM Passes","title":"SIMDLoop","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-simdloop.cpp\nClass Name: LowerSIMDLoopPass\nOpt Name: loop(LowerSIMDLoop)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass acts as the main driver of the @simd annotation. Codegen inserts a !llvm.loopid marker at the back branch of a loop, which this pass uses to identify loops that were originally marked with @simd. Then, this pass looks for a chain of floating point operations that form a reduce and adds the contract and reassoc fast math flags to allow reassociation (and thus vectorization). This pass does not preserve either loop information nor inference correctness, so it may violate Julia semantics in surprising ways. If the loop was annotated with ivdep as well, then the pass marks the loop as having no loop-carried dependencies (the resulting behavior is undefined if the user annotation was incorrect or gets applied to the wrong loop).","category":"page"},{"location":"devdocs/llvm-passes.html#LowerPTLS","page":"Custom LLVM Passes","title":"LowerPTLS","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-ptls.cpp\nClass Name: LowerPTLSPass\nOpt Name: module(LowerPTLSPass)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass lowers thread-local Julia intrinsics to assembly instructions. Julia relies on thread-local storage for garbage collection and multithreading task scheduling. When compiling code for system images and package images, this pass replaces calls to intrinsics with loads from global variables that are initialized at load time.","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"If codegen produces a function with a swiftself argument and calling convention, this pass assumes the swiftself argument is the pgcstack and will replace the intrinsics with that argument. Doing so provides speedups on architectures that have slow thread local storage accesses.","category":"page"},{"location":"devdocs/llvm-passes.html#RemoveAddrspaces","page":"Custom LLVM Passes","title":"RemoveAddrspaces","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveAddrspacesPass\nOpt Name: module(RemoveAddrspaces)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass renames pointers in one address space to another address space. This is used to remove Julia-specific address spaces from LLVM IR.","category":"page"},{"location":"devdocs/llvm-passes.html#RemoveJuliaAddrspaces","page":"Custom LLVM Passes","title":"RemoveJuliaAddrspaces","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveJuliaAddrspacesPass\nOpt Name: module(RemoveJuliaAddrspaces)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass removes Julia-specific address spaces from LLVM IR. It is mostly used for displaying LLVM IR in a less cluttered format. Internally, it is implemented off the RemoveAddrspaces pass.","category":"page"},{"location":"devdocs/llvm-passes.html#Multiversioning","page":"Custom LLVM Passes","title":"Multiversioning","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-multiversioning.cpp\nClass Name: MultiVersioningPass\nOpt Name: module(JuliaMultiVersioning)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass performs modifications to a module to create functions that are optimized for running on different architectures (see sysimg.md and pkgimg.md for more details). Implementation-wise, it clones functions and applies different target-specific attributes to them to allow the optimizer to use advanced features such as vectorization and instruction scheduling for that platform. It also creates some infrastructure to enable the Julia image loader to select the appropriate version of the function to call based on the architecture the loader is running on. The target-specific attributes are controlled by the julia.mv.specs module flag, which during compilation is derived from the JULIA_CPU_TARGET environment variable. The pass must also be enabled by providing a julia.mv.enable module flag with a value of 1.","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"warning: Warning\n","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Use of llvmcall with multiversioning is dangerous. llvmcall enables access to features not typically exposed by the Julia APIs, and are therefore usually not available on all architectures. If multiversioning is enabled and code generation is requested for a target architecture that does not support the feature required by an llvmcall expression, LLVM will probably error out, likely with an abort and the message LLVM ERROR: Do not know how to split the result of this operator!.","category":"page"},{"location":"devdocs/llvm-passes.html#GCInvariantVerifier","page":"Custom LLVM Passes","title":"GCInvariantVerifier","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-gc-invariant-verifier.cpp\nClass Name: GCInvariantVerifierPass\nOpt Name: module(GCInvariantVerifier)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass is used to verify Julia's invariants about LLVM IR. This includes things such as the nonexistence of ptrtoint in Julia's non-integral address spaces [nislides] and the existence of only blessed addrspacecast instructions (Tracked -> Derived, 0 -> Tracked, etc). It performs no transformations on IR.","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"[nislides]: https://llvm.org/devmtg/2015-02/slides/chisnall-pointers-not-int.pdf","category":"page"},{"location":"devdocs/llvm-passes.html#Optimization-Passes","page":"Custom LLVM Passes","title":"Optimization Passes","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"These passes are used to perform transformations on LLVM IR that LLVM will not perform itself, e.g. fast math flag propagation, escape analysis, and optimizations on Julia-specific internal functions. They use knowledge about Julia's semantics to perform these optimizations.","category":"page"},{"location":"devdocs/llvm-passes.html#CombineMulAdd","page":"Custom LLVM Passes","title":"CombineMulAdd","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-muladd.cpp\nClass Name: CombineMulAddPass\nOpt Name: function(CombineMulAdd)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass serves to optimize the particular combination of a regular fmul with a fast fadd into a contract fmul with a fast fadd. This is later optimized by the backend to a fused multiply-add instruction, which can provide significantly faster operations at the cost of more unpredictable semantics.","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"note: Note\nThis optimization only occurs when the fmul has a single use, which is the fast fadd.","category":"page"},{"location":"devdocs/llvm-passes.html#AllocOpt","page":"Custom LLVM Passes","title":"AllocOpt","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-alloc-opt.cpp\nClass Name: AllocOptPass\nOpt Name: function(AllocOpt)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Julia does not have the concept of a program stack as a place to allocate mutable objects. However, allocating objects on the stack reduces GC pressure and is critical for GPU compilation. Thus, AllocOpt performs heap to stack conversion of objects that it can prove do not escape the current function. It also performs a number of other optimizations on allocations, such as removing allocations that are never used, optimizing typeof calls to freshly allocated objects, and removing stores to allocations that are immediately overwritten. The escape analysis implementation is located in llvm-alloc-helpers.cpp. Currently, this pass does not use information from EscapeAnalysis.jl, though that may change in the future.","category":"page"},{"location":"devdocs/llvm-passes.html#PropagateJuliaAddrspaces","page":"Custom LLVM Passes","title":"PropagateJuliaAddrspaces","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-propagate-addrspaces.cpp\nClass Name: PropagateJuliaAddrspacesPass\nOpt Name: function(PropagateJuliaAddrspaces)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass is used to propagate Julia-specific address spaces through operations on pointers. LLVM is not allowed to introduce or remove addrspacecast instructions by optimizations, so this pass acts to eliminate redundant addrspace casts by replacing operations with their equivalent in a Julia address space. For more information on Julia's address spaces, see (TODO link to llvm.md).","category":"page"},{"location":"devdocs/llvm-passes.html#JuliaLICM","page":"Custom LLVM Passes","title":"JuliaLICM","text":"","category":"section"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Filename: llvm-julia-licm.cpp\nClass Name: JuliaLICMPass\nOpt Name: loop(JuliaLICM)","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass is used to hoist Julia-specific intrinsics out of loops. Specifically, it performs the following transformations:","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"Hoist gc_preserve_begin and sink gc_preserve_end out of loops when the preserved objects are loop-invariant.\nSince objects preserved within a loop are likely preserved for the duration of the loop, this transformation can reduce the number of gc_preserve_begin/gc_preserve_end pairs in the IR. This makes it easier for the LateLowerGCPass to identify where particular objects are preserved.\nHoist write barriers with invariant objects\nHere we assume that there are only two generations that an object can be a part of. Given that, a write barrier needs to only execute once for any pair of the same object. Thus, we can hoist write barriers out of loops when the object being written to is loop-invariant.\nHoist allocations out of loops when they do not escape the loop\nWe use a very conservative definition of escape here, the same as the one used in AllocOptPass. This transformation can reduce the number of allocations in the IR, even when an allocation escapes the function altogether.","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"!!!note","category":"page"},{"location":"devdocs/llvm-passes.html","page":"Custom LLVM Passes","title":"Custom LLVM Passes","text":"This pass is required to preserve LLVM's [MemorySSA](https://llvm.org/docs/MemorySSA.html) ([Short Video](https://www.youtube.com/watch?v=bdxWmryoHak), [Longer Video](https://www.youtube.com/watch?v=1e5y6WDbXCQ)) and [ScalarEvolution](https://baziotis.cs.illinois.edu/compilers/introduction-to-scalar-evolution.html) ([Newer Slides](https://llvm.org/devmtg/2018-04/slides/Absar-ScalarEvolution.pdf) [Older Slides](https://llvm.org/devmtg/2009-10/ScalarEvolutionAndLoopOptimization.pdf)) analyses.","category":"page"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/NEWS.md\"","category":"page"},{"location":"NEWS.html#Julia-v1.11-Release-Notes","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"","category":"section"},{"location":"NEWS.html#New-language-features","page":"Julia v1.11 Release Notes","title":"New language features","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"New Memory type that provides a lower-level container as an alternative to Array. Memory has less overhead and a faster constructor, making it a good choice for situations that do not need all the features of Array (e.g. multiple dimensions). Most of the Array type is now implemented in Julia on top of Memory, leading to significant speedups for several functions (e.g. push!) as well as more maintainable code (#51319).\npublic is a new keyword. Symbols marked with public are considered public API. Symbols marked with export are now also treated as public API. The difference between public and export is that public names do not become available when using a package/module (#50105).\nScopedValue implements dynamic scope with inheritance across tasks (#50958).\nManifest.toml files can now be renamed in the format Manifest-v{major}.{minor}.toml to be preferentially picked up by the given julia version. i.e. in the same folder, a Manifest-v1.11.toml would be used by v1.11 and Manifest.toml by every other julia version. This makes managing environments for multiple julia versions at the same time easier (#43845).\nSupport for Unicode 15.1 (#51799).","category":"page"},{"location":"NEWS.html#Language-changes","page":"Julia v1.11 Release Notes","title":"Language changes","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"During precompilation, atexit hooks now run before saving the output file. This allows users to safely tear down background state (such as closing Timers and sending disconnect notifications to heartbeat tasks) and cleanup other resources when the program wants to begin exiting.\nCode coverage and malloc tracking is no longer generated during the package precompilation stage. Further, during these modes pkgimage caches are now used for packages that are not being tracked. This means that coverage testing (the default for julia-actions/julia-runtest) will by default use pkgimage caches for all other packages than the package being tested, likely meaning faster test execution (#52123).\nSpecifying a path in JULIA_DEPOT_PATH now results in the expansion of empty strings to omit the default user depot (#51448).\nPrecompilation cache files are now relocatable and their validity is now verified through a content hash of their source files instead of their mtime (#49866).","category":"page"},{"location":"NEWS.html#Compiler/Runtime-improvements","page":"Julia v1.11 Release Notes","title":"Compiler/Runtime improvements","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Updated GC heuristics to count allocated pages instead of individual objects (#50144).\nAdded support for annotating Base.@assume_effects on code blocks (#52400).","category":"page"},{"location":"NEWS.html#Command-line-option-changes","page":"Julia v1.11 Release Notes","title":"Command-line option changes","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"The entry point for Julia has been standardized to Main.main(args). This must be explicitly opted into using the @main macro (see the docstring for further details). When opted-in, and julia is invoked to run a script or expression (i.e. using julia script.jl or julia -e expr), julia will subsequently run the Main.main function automatically. This is intended to unify script and compilation workflows, where code loading may happen in the compiler and execution of Main.main may happen in the resulting executable. For interactive use, there is no semantic difference between defining a main function and executing the code directly at the end of the script (#50974).\nThe --compiled-modules and --pkgimages flags can now be set to existing, which will cause Julia to consider loading existing cache files, but not to create new ones (#50586, #52573).\nThe --project argument now accepts @script to give a path to a directory with a Project.toml relative to the passed script file. --project=@script/foo for the foo subdirectory. If no path is given after (i.e. --project=@script) then (like --project=@.) the directory and its parents are searched for a Project.toml (#50864 and #53352)","category":"page"},{"location":"NEWS.html#Multi-threading-changes","page":"Julia v1.11 Release Notes","title":"Multi-threading changes","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Threads.@threads now supports the :greedy scheduler, intended for non-uniform workloads (#52096).\nA new public (but unexported) struct Base.Lockable{T, L<:AbstractLock} makes it easy to bundle a resource and its lock together (#52898).","category":"page"},{"location":"NEWS.html#New-library-functions","page":"Julia v1.11 Release Notes","title":"New library functions","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Three new types around the idea of text with \"annotations\" (Pair{Symbol, Any} entries, e.g. :lang => \"en\" or :face => :magenta). These annotations are preserved across operations (e.g. string concatenation with *) when possible.\nAnnotatedString is a new AbstractString type. It wraps an underlying string and allows for annotations to be attached to regions of the string. This type is used extensively in the new StyledStrings standard library to hold styling information.\nAnnotatedChar is a new AbstractChar type. It wraps another char and holds a list of annotations that apply to it.\nAnnotatedIOBuffer is a new IO type that mimics an IOBuffer, but has specialised read/write methods for annotated content. This can be thought of both as a \"string builder\" of sorts and also as glue between annotated and unannotated content.\nin!(x, s::AbstractSet) will return whether x is in s, and insert x in s if not (#45156, #51636).\nThe new Libc.mkfifo function wraps the mkfifo C function on Unix platforms (#34587).\nlogrange(start, stop; length) makes a range of constant ratio, instead of constant step (#39071)\ncopyuntil(out, io, delim) and copyline(out, io) copy data into an out::IO stream (#48273).\neachrsplit(string, pattern) iterates split substrings right to left (#51646).\nSys.username() can be used to return the current user's username (#51897).\nSys.isreadable(), Sys.iswritable() can be used to check if the current user has access permissions that permit reading and writing, respectively. (#53320).\nwrap(Array, m::Union{MemoryRef{T}, Memory{T}}, dims) is the safe counterpart to unsafe_wrap (#52049).\nGC.logging_enabled() can be used to test whether GC logging has been enabled via GC.enable_logging (#51647).\nIdSet is now exported from Base and considered public (#53262).\n@time now reports a count of any lock conflicts where a ReentrantLock had to wait, plus a new macro @lock_conflicts which returns that count (#52883).\nThe new macro Base.Cartesian.@ncallkw is analogous to Base.Cartesian.@ncall, but allows adding keyword arguments to the function call (#51501).\nNew function Docs.hasdoc(module, symbol) tells whether a name has a docstring (#52139).\nNew function Docs.undocumented_names(module) returns a module's undocumented public names (#52413).","category":"page"},{"location":"NEWS.html#New-library-features","page":"Julia v1.11 Release Notes","title":"New library features","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"invmod(n, T) where T is a native integer type now computes the modular inverse of n in the modular integer ring that T defines (#52180).\ninvmod(n) is an abbreviation for invmod(n, typeof(n)) for native integer types (#52180).\nreplace(string, pattern...) now supports an optional IO argument to write the output to a stream rather than returning a string (#48625).\nNew methods allequal(f, itr) and allunique(f, itr) taking a predicate function (#47679).\nsizehint!(s, n) now supports an optional shrink argument to disable shrinking (#51929).\nPassing an IOBuffer as a stdout argument for Process spawn now works as expected, synchronized with wait or success, so a Base.BufferStream is no longer required there for correctness to avoid data races (#52461).\nAfter a process exits, closewrite will no longer be automatically called on the stream passed to it. Call wait on the process instead to ensure the content is fully written, then call closewrite manually to avoid data races, or use the callback form of open to have all that handled automatically (#52461).\n@timed now additionally returns the elapsed compilation and recompilation time (#52889).\nfilter can now act on a NamedTuple (#50795).\nIterators.cycle(iter, n) runs over iter a fixed number of times, instead of forever (#47354).\nzero(::AbstractArray) now applies recursively, so zero([[1,2],[3,4,5]]) now produces the additive identity [[0,0],[0,0,0]] rather than erroring (#38064).\ninclude_dependency(path; track_content=true) allows switching from using mtime to hashing of the precompilation dependency in order to restore relocatability of precompilation caches (#51798).","category":"page"},{"location":"NEWS.html#Standard-library-changes","page":"Julia v1.11 Release Notes","title":"Standard library changes","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"The fallback method write(::IO, ::AbstractArray) used to recursively call write on each element, but now writes the in-memory representation of each value. For example, write(io, 'a':'b') now writes 4 bytes for each character, instead of writing the UTF-8 representation of each character. The new format is compatible with that used by Array, making it possible to use read! to get the data back (#42593).","category":"page"},{"location":"NEWS.html#StyledStrings","page":"Julia v1.11 Release Notes","title":"StyledStrings","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"A new standard library for handling styling in a more comprehensive and structured way (#49586).\nThe new Faces struct serves as a container for text styling information (think typeface, as well as color and decoration), and comes with a framework to provide a convenient, extensible (via addface!), and customisable (with a user's Faces.toml and loadfaces!) approach to styled content (#49586).\nThe new @styled_str string macro provides a convenient way of creating a AnnotatedString with various faces or other attributes applied (#49586).","category":"page"},{"location":"NEWS.html#JuliaSyntaxHighlighting","page":"Julia v1.11 Release Notes","title":"JuliaSyntaxHighlighting","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"A new standard library for applying syntax highlighting to Julia code, this uses JuliaSyntax and StyledStrings to implement a highlight function that creates an AnnotatedString with syntax highlighting applied.","category":"page"},{"location":"NEWS.html#Libdl","page":"Julia v1.11 Release Notes","title":"Libdl","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"A new LazyLibrary type is exported from Libdl for use in building chained lazy library loads, primarily to be used within JLLs (#50074).","category":"page"},{"location":"NEWS.html#LinearAlgebra","page":"Julia v1.11 Release Notes","title":"LinearAlgebra","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"cbrt(::AbstractMatrix{<:Real}) is now defined and returns real-valued matrix cube roots of real-valued matrices (#50661).\neigvals/eigen(A, bunchkaufman(B)) and eigvals/eigen(A, lu(B)), which utilize the Bunchkaufman (LDL) and LU decomposition of B,  respectively, now efficiently compute the generalized eigenvalues (eigen: and eigenvectors) of A and B. Note: The second  argument is the output of bunchkaufman or lu (#50471).\nThere is now a specialized dispatch for eigvals/eigen(::Hermitian{<:Tridiagonal}) which performs a similarity transformation to create a real symmetric tridiagonal matrix, and solve that using the LAPACK routines (#49546).\nStructured matrices now retain either the axes of the parent (for Symmetric/Hermitian/AbstractTriangular/UpperHessenberg), or that of the principal diagonal (for banded matrices) (#52480).\nbunchkaufman and bunchkaufman! now work for any AbstractFloat, Rational and their complex variants. bunchkaufman now supports Integer types, by making an internal conversion to Rational{BigInt}. Added new function inertia that computes the inertia of the diagonal factor given by the BunchKaufman factorization object of a real symmetric or Hermitian matrix. For complex symmetric matrices, inertia only computes the number of zero eigenvalues of the diagonal factor (#51487).\nPackages that specialize matrix-matrix mul! with a method signature of the form mul!(::AbstractMatrix, ::MyMatrix, ::AbstractMatrix, ::Number, ::Number) no longer encounter method ambiguities when interacting with LinearAlgebra. Previously, ambiguities used to arise when multiplying a MyMatrix with a structured matrix type provided by LinearAlgebra, such as AbstractTriangular, which used to necessitate additional methods to resolve such ambiguities. Similar sources of ambiguities have also been removed for matrix-vector mul! operations (#52837).\nlu and issuccess(::LU) now accept an allowsingular keyword argument. When set to true, a valid factorization with rank-deficient U factor will be treated as success instead of throwing an error. Such factorizations are now shown by printing the factors together with a \"rank-deficient\" note rather than printing a \"Failed Factorization\" message (#52957).","category":"page"},{"location":"NEWS.html#Random","page":"Julia v1.11 Release Notes","title":"Random","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"rand now supports sampling over Tuple types (#35856, #50251).\nrand now supports sampling over Pair types (#28705).\nWhen seeding RNGs provided by Random, negative integer seeds can now be used (#51416).\nSeedable random number generators from Random can now be seeded by a string, e.g. seed!(rng, \"a random seed\") (#51527).","category":"page"},{"location":"NEWS.html#REPL","page":"Julia v1.11 Release Notes","title":"REPL","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Tab complete hints now show in lighter text while typing in the repl. To disable set Base.active_repl.options.hint_tab_completes = false interactively, or in startup.jl:\nif VERSION >= v\"1.11.0-0\"\n  atreplinit() do repl\n      repl.options.hint_tab_completes = false\n  end\nend\n(#51229).\nMeta-M with an empty prompt now toggles the contextual module between the previous non-Main contextual module and Main so that switching back and forth is simple (#51616, #52670).","category":"page"},{"location":"NEWS.html#Dates","page":"Julia v1.11 Release Notes","title":"Dates","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"The undocumented function adjust is no longer exported but is now documented (#53092).","category":"page"},{"location":"NEWS.html#Statistics","page":"Julia v1.11 Release Notes","title":"Statistics","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Statistics is now an upgradeable standard library (#46501).","category":"page"},{"location":"NEWS.html#Distributed","page":"Julia v1.11 Release Notes","title":"Distributed","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"pmap now defaults to using a CachingPool (#33892).","category":"page"},{"location":"NEWS.html#Deprecated-or-removed","page":"Julia v1.11 Release Notes","title":"Deprecated or removed","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"Base.map, Iterators.map, and foreach lost their single-argument methods (#52631).","category":"page"},{"location":"NEWS.html#External-dependencies","page":"Julia v1.11 Release Notes","title":"External dependencies","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"The libuv library has been updated from a base of v1.44.2 to v1.48.0 (#49937).\ntput is no longer called to check terminal capabilities; it has been replaced with a pure-Julia terminfo parser (#50797).","category":"page"},{"location":"NEWS.html#Tooling-Improvements","page":"Julia v1.11 Release Notes","title":"Tooling Improvements","text":"","category":"section"},{"location":"NEWS.html","page":"Julia v1.11 Release Notes","title":"Julia v1.11 Release Notes","text":"CI now performs limited automatic typo detection on all PRs. If you merge a PR with a failing typo CI check, then the reported typos will be automatically ignored in future CI runs on PRs that edit those same files (#51704).","category":"page"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings","page":"StyledStrings","title":"StyledStrings","text":"","category":"section"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings-styling","page":"StyledStrings","title":"Styling","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"When working with strings, formatting and styling often appear as a secondary concern.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For instance, when printing to a terminal you might want to sprinkle ANSI escape sequences in the output, when outputting HTML styling constructs (<span style=\"...\">, etc.) serve a similar purpose, and so on. It is possible to simply insert the raw styling constructs into the string next to the content itself, but it quickly becomes apparent that this is not well suited for anything but the most basic use cases. Not all terminals support the same ANSI codes, the styling constructs need to be painstakingly removed when calculating the width of already-styled content, and that's before you even get into handling multiple output formats.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Instead of leaving this headache to be widely experienced downstream, it is tackled head-on by the introduction of a special string type (AnnotatedString). This string type wraps any other AbstractString type and allows for formatting information to be applied to regions (e.g. characters 1 through to 7 are bold and red).","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Regions of a string are styled by applying Faces (think \"typeface\") to them — a structure that holds styling information. As a convenience, faces in the global faces dictionary (e.g. shadow) can just be named instead of giving the Face directly.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Along with these capabilities, we also provide a convenient way for constructing AnnotatedStrings, detailed in Styled String Literals.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"using StyledStrings\nstyled\"{yellow:hello} {blue:there}\"","category":"page"},{"location":"stdlib/StyledStrings.html#Styling-via-[AnnotatedString](@ref-Base.AnnotatedString)s","page":"StyledStrings","title":"Styling via AnnotatedStrings","text":"","category":"section"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings-faces","page":"StyledStrings","title":"Faces","text":"","category":"section"},{"location":"stdlib/StyledStrings.html#The-Face-type","page":"StyledStrings","title":"The Face type","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"A Face specifies details of a typeface that text can be set in. It covers a set of basic attributes that generalize well across different formats, namely:","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"font\nheight\nweight\nslant\nforeground\nbackground\nunderline\nstrikethrough\ninverse\ninherit","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For details on the particular forms these attributes take, see the Face docstring, but of particular interest is inherit as it allows you to inherit attributes from other Faces.","category":"page"},{"location":"stdlib/StyledStrings.html#The-global-faces-dictionary","page":"StyledStrings","title":"The global faces dictionary","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"To make referring to particular styles more convenient, there is a global Dict{Symbol, Face} that allows for Faces to be referred to simply by name. Packages can add faces to this dictionary via the addface! function, and the loaded faces can be easily customized.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"warning: Appropriate face naming\nAny package registering new faces should ensure that they are prefixed by the package name, i.e. follow the format mypackage_myface. This is important for predictability, and to prevent name clashes.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"There is one set of exemptions to the package-prefix rule, the set of basic faces that are part of the default value of the faces dictionary.","category":"page"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings-basic-faces","page":"StyledStrings","title":"Basic faces","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Basic faces are intended to represent a general idea that is widely applicable.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For setting some text with a certain attribute, we have the bold, light, italic, underline, strikethrough, and inverse faces.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"There are also named faces for the 16 terminal colors: black, red, green, yellow, blue, magenta, cyan, white, bright_black/grey/gray, bright_red, bright_green, bright_blue, bright_magenta, bright_cyan, and bright_white.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For shadowed text (i.e. dim but there) there is the shadow face. To indicate a selected region, there is the region face. Similarly for emphasis and highlighting the emphasis and highlight faces are defined. There is also code for code-like text.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For visually indicating the severity of messages, the error, warning, success, info, note, and tip faces are defined.","category":"page"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings-face-toml","page":"StyledStrings","title":"Customisation of faces (Faces.toml)","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"It is good for the name faces in the global face dictionary to be customizable. Theming and aesthetics are nice, and it is important for accessibility reasons too. A TOML file can be parsed into a list of Face specifications that are merged with the pre-existing entry in the face dictionary.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"A Face is represented in TOML like so:","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"[facename]\nattribute = \"value\"\n...\n\n[package.facename]\nattribute = \"value\"","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For example, if the shadow face is too hard to read it can be made brighter like so:","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"[shadow]\nforeground = \"white\"","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"On initialization, the config/faces.toml file under the first Julia depot (usually ~/.julia) is loaded.","category":"page"},{"location":"stdlib/StyledStrings.html#Applying-faces-to-a-AnnotatedString","page":"StyledStrings","title":"Applying faces to a AnnotatedString","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"By convention, the :face attributes of a AnnotatedString hold information on the Faces that currently apply. This can be given in multiple forms, as a single Symbol naming a Faces in the global face dictionary, a Face itself, or a vector of either.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"The show(::IO, ::MIME\"text/plain\", ::AnnotatedString) and show(::IO, ::MIME\"text/html\", ::AnnotatedString) methods both look at the :face attributes and merge them all together when determining the overall styling.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"We can supply :face attributes to a AnnotatedString during construction, add them to the properties list afterwards, or use the convenient Styled String literals.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"str1 = Base.AnnotatedString(\"blue text\", [(1:9, :face => :blue)])\nstr2 = styled\"{blue:blue text}\"\nstr1 == str2\nsprint(print, str1, context = :color => true)\nsprint(show, MIME(\"text/html\"), str1, context = :color => true)","category":"page"},{"location":"stdlib/StyledStrings.html#stdlib-styledstring-literals","page":"StyledStrings","title":"Styled String Literals","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"To ease construction of AnnotatedStrings with Faces applied, the styled\"...\" styled string literal allows for the content and attributes to be easily expressed together via a custom grammar.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Within a styled\"...\" literal, curly braces are considered special characters and must be escaped in normal usage (\\{, \\}). This allows them to be used to express annotations with (nestable) {annotations...:text} constructs.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"The annotations... component is a comma-separated list of three types of annotations.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Face names\nInline Face expressions (key=val,...)\nkey=value pairs","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Interpolation is possible everywhere except for inline face keys.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"For more information on the grammar, see the extended help of the styled\"...\" docstring.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"As an example, we can demonstrate the list of built-in faces mentioned above like so:","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"julia> println(styled\"\nThe basic font-style attributes are {bold:bold}, {light:light}, {italic:italic},\n{underline:underline}, and {strikethrough:strikethrough}.\n\nIn terms of color, we have named faces for the 16 standard terminal colors:\n {black:■} {red:■} {green:■} {yellow:■} {blue:■} {magenta:■} {cyan:■} {white:■}\n {bright_black:■} {bright_red:■} {bright_green:■} {bright_yellow:■} {bright_blue:■} {bright_magenta:■} {bright_cyan:■} {bright_white:■}\n\nSince {code:bright_black} is effectively grey, we define two aliases for it:\n{code:grey} and {code:gray} to allow for regional spelling differences.\n\nTo flip the foreground and background colors of some text, you can use the\n{code:inverse} face, for example: {magenta:some {inverse:inverse} text}.\n\nThe intent-based basic faces are {shadow:shadow} (for dim but visible text),\n{region:region} for selections, {emphasis:emphasis}, and {highlight:highlight}.\nAs above, {code:code} is used for code-like text.\n\nLastly, we have the 'message severity' faces: {error:error}, {warning:warning},\n{success:success}, {info:info}, {note:note}, and {tip:tip}.\n\nRemember that all these faces (and any user or package-defined ones) can\narbitrarily nest and overlap, {region,tip:like {bold,italic:so}}.\")","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"Documenter doesn't properly represent all the styling above, so I've converted it manually to HTML and LaTeX.","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"<pre>\n The basic font-style attributes are <span style=\"font-weight: 700;\">bold</span>, <span style=\"font-weight: 300;\">light</span>, <span style=\"font-style: italic;\">italic</span>,\n <span style=\"text-decoration: underline;\">underline</span>, and <span style=\"text-decoration: line-through\">strikethrough</span>.\n\n In terms of color, we have named faces for the 16 standard terminal colors:\n  <span style=\"color: #1c1a23;\">■</span> <span style=\"color: #a51c2c;\">■</span> <span style=\"color: #25a268;\">■</span> <span style=\"color: #e5a509;\">■</span> <span style=\"color: #195eb3;\">■</span> <span style=\"color: #803d9b;\">■</span> <span style=\"color: #0097a7;\">■</span> <span style=\"color: #dddcd9;\">■</span>\n  <span style=\"color: #76757a;\">■</span> <span style=\"color: #ed333b;\">■</span> <span style=\"color: #33d079;\">■</span> <span style=\"color: #f6d22c;\">■</span> <span style=\"color: #3583e4;\">■</span> <span style=\"color: #bf60ca;\">■</span> <span style=\"color: #26c6da;\">■</span> <span style=\"color: #f6f5f4;\">■</span>\n\n Since <span style=\"color: #0097a7;\">bright_black</span> is effectively grey, we define two aliases for it:\n <span style=\"color: #0097a7;\">grey</span> and <span style=\"color: #0097a7;\">gray</span> to allow for regional spelling differences.\n\n To flip the foreground and background colors of some text, you can use the\n <span style=\"color: #0097a7;\">inverse</span> face, for example: <span style=\"color: #803d9b;\">some </span><span style=\"background-color: #803d9b;\">inverse</span><span style=\"color: #803d9b;\"> text</span>.\n\n The intent-based basic faces are <span style=\"color: #76757a;\">shadow</span> (for dim but visible text),\n <span style=\"background-color: #3a3a3a;\">region</span> for selections, <span style=\"color: #195eb3;\">emphasis</span>, and <span style=\"background-color: #195eb3;\">highlight</span>.\n As above, <span style=\"color: #0097a7;\">code</span> is used for code-like text.\n\n Lastly, we have the 'message severity' faces: <span style=\"color: #ed333b;\">error</span>, <span style=\"color: #e5a509;\">warning</span>,\n <span style=\"color: #25a268;\">success</span>, <span style=\"color: #26c6da;\">info</span>, <span style=\"color: #76757a;\">note</span>, and <span style=\"color: #33d079;\">tip</span>.\n\n Remember that all these faces (and any user or package-defined ones) can\n arbitrarily nest and overlap, <span style=\"color: #33d079;background-color: #3a3a3a;\">like <span style=\"font-weight: 700;font-style: italic;\">so</span></span>.</pre>","category":"page"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"\\begingroup\n\\ttfamily\n\\setlength{\\parindent}{0pt}\n\\setlength{\\parskip}{\\baselineskip}\n\nThe basic font-style attributes are {\\fontseries{b}\\selectfont bold}, {\\fontseries{l}\\selectfont light}, {\\fontshape{it}\\selectfont italic},\\\\\n\\underline{underline}, and {strikethrough}.\n\nIn terms of color, we have named faces for the 16 standard terminal colors:\\\\\n{\\color[HTML]{1c1a23}\\(\\blacksquare\\)} {\\color[HTML]{a51c2c}\\(\\blacksquare\\)} {\\color[HTML]{25a268}\\(\\blacksquare\\)}\n{\\color[HTML]{e5a509}\\(\\blacksquare\\)} {\\color[HTML]{195eb3}\\(\\blacksquare\\)} {\\color[HTML]{803d9b}\\(\\blacksquare\\)}\n{\\color[HTML]{0097a7}\\(\\blacksquare\\)} {\\color[HTML]{dddcd9}\\(\\blacksquare\\)} \\\\\n{\\color[HTML]{76757a}\\(\\blacksquare\\)} {\\color[HTML]{ed333b}\\(\\blacksquare\\)} {\\color[HTML]{33d079}\\(\\blacksquare\\)} {\\color[HTML]{f6d22c}\\(\\blacksquare\\)} {\\color[HTML]{3583e4}\\(\\blacksquare\\)} {\\color[HTML]{bf60ca}\\(\\blacksquare\\)} {\\color[HTML]{26c6da}\\(\\blacksquare\\)} {\\color[HTML]{f6f5f4}\\(\\blacksquare\\)}\n\nSince {\\color[HTML]{0097a7}bright\\_black} is effectively grey, we define two aliases for it:\\\\\n{\\color[HTML]{0097a7}grey} and {\\color[HTML]{0097a7}gray} to allow for regional spelling differences.\n\nTo flip the foreground and background colors of some text, you can use the\\\\\n{\\color[HTML]{0097a7}inverse} face, for example: {\\color[HTML]{803d9b}some \\colorbox[HTML]{803d9b}{\\color[HTML]{000000}inverse} text}.\n\nThe intent-based basic faces are {\\color[HTML]{76757a}shadow} (for dim but visible text),\\\\\n\\colorbox[HTML]{3a3a3a}{region} for selections, {\\color[HTML]{195eb3}emphasis}, and \\colorbox[HTML]{195eb3}{highlight}.\\\\\nAs above, {\\color[HTML]{0097a7}code} is used for code-like text.\n\nLastly, we have the 'message severity' faces: {\\color[HTML]{ed333b}error}, {\\color[HTML]{e5a509}warning},\\\\\n{\\color[HTML]{25a268}success}, {\\color[HTML]{26c6da}info}, {\\color[HTML]{76757a}note}, and {\\color[HTML]{33d079}tip}.\n\nRemember that all these faces (and any user or package-defined ones) can\\\\\narbitrarily nest and overlap, \\colorbox[HTML]{3a3a3a}{\\color[HTML]{33d079}like\n  {\\fontseries{b}\\fontshape{it}\\selectfont so}}.\n\\endgroup","category":"page"},{"location":"stdlib/StyledStrings.html#stdlib-styledstrings-api","page":"StyledStrings","title":"API reference","text":"","category":"section"},{"location":"stdlib/StyledStrings.html","page":"StyledStrings","title":"StyledStrings","text":"StyledStrings.@styled_str\nStyledStrings.styled\nStyledStrings.Face\nStyledStrings.addface!\nStyledStrings.withfaces\nStyledStrings.SimpleColor\nBase.parse(::Type{StyledStrings.SimpleColor}, ::String)\nBase.tryparse(::Type{StyledStrings.SimpleColor}, ::String)\nBase.merge(::StyledStrings.Face, ::StyledStrings.Face)","category":"page"},{"location":"stdlib/StyledStrings.html#StyledStrings.StyledMarkup.@styled_str","page":"StyledStrings","title":"StyledStrings.StyledMarkup.@styled_str","text":"@styled_str -> AnnotatedString\n\nConstruct a styled string. Within the string, {<specs>:<content>} structures apply the formatting to <content>, according to the list of comma-separated specifications <specs>. Each spec can either take the form of a face name, an inline face specification, or a key=value pair. The value must be wrapped by {...} should it contain any of the characters ,=:{}.\n\nString interpolation with $ functions in the same way as regular strings, except quotes need to be escaped. Faces, keys, and values can also be interpolated with $.\n\nExample\n\nstyled\"The {bold:{italic:quick} {(foreground=#cd853f):brown} fox} jumped over the {link={https://en.wikipedia.org/wiki/Laziness}:lazy} dog\"\n\nExtended help\n\nThis macro can be described by the following EBNF grammar:\n\nstyledstring = { styled | interpolated | escaped | plain } ;\n\nspecialchar = '{' | '}' | '$' | '\\\"' ;\nanychar = [\\u0-\\u1fffff] ;\nplain = { anychar - specialchar } ;\nescaped = '\\\\', specialchar ;\n\ninterpolated = '$', ? expr ? | '$(', ? expr ?, ')' ;\n\nstyled = '{', ws, annotations, ':', content, '}' ;\ncontent = { interpolated | plain | escaped | styled } ;\nannotations = annotation | annotations, ws, ',', ws, annotation ;\nannotation = face | inlineface | keyvalue ;\nws = { ' ' | '\\t' | '\\n' } ; (* whitespace *)\n\nface = facename | interpolated ;\nfacename = [A-Za-z0-9_]+ ;\n\ninlineface = '(', ws, [ faceprop ], { ws, ',', faceprop }, ws, ')' ;\nfaceprop = [a-z]+, ws, '=', ws, ( [^,)]+ | interpolated) ;\n\nkeyvalue = key, ws, '=', ws, value ;\nkey = ( [^\\0${}=,:], [^\\0=,:]* ) | interpolated ;\nvalue = simplevalue | curlybraced | interpolated ;\ncurlybraced = '{' { escaped | plain } '}' ;\nsimplevalue = [^${},:], [^,:]* ;\n\nAn extra stipulation not encoded in the above grammar is that plain should be a valid input to unescape_string, with specialchar kept.\n\nThe above grammar for inlineface is simplified, as the actual implementation is a bit more sophisticated. The full behaviour is given below.\n\nfaceprop = ( 'face', ws, '=', ws, ( ? string ? | interpolated ) ) |\n           ( 'height', ws, '=', ws, ( ? number ? | interpolated ) ) |\n           ( 'weight', ws, '=', ws, ( symbol | interpolated ) ) |\n           ( 'slant', ws, '=', ws, ( symbol | interpolated ) ) |\n           ( ( 'foreground' | 'fg' | 'background' | 'bg' ),\n               ws, '=', ws, ( simplecolor | interpolated ) ) |\n           ( 'underline', ws, '=', ws, ( underline | interpolated ) ) |\n           ( 'strikethrough', ws, '=', ws, ( bool | interpolated ) ) |\n           ( 'inverse', ws, '=', ws, ( bool | interpolated ) ) |\n           ( 'inherit', ws, '=', ws, ( inherit | interpolated ) ) ;\n\nnothing = 'nothing' ;\nbool = 'true' | 'false' ;\nsymbol = [^ ,)]+ ;\nhexcolor = ('#' | '0x'), [0-9a-f]{6} ;\nsimplecolor = hexcolor | symbol | nothing ;\n\nunderline = nothing | bool | simplecolor | underlinestyled;\nunderlinestyled = '(', ws, ('' | nothing | simplecolor | interpolated), ws,\n                  ',', ws, ( symbol | interpolated ), ws ')' ;\n\ninherit = ( '[', inheritval, { ',', inheritval }, ']' ) | inheritval;\ninheritval = ws, ':'?, symbol ;\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/StyledStrings.html#StyledStrings.StyledMarkup.styled","page":"StyledStrings","title":"StyledStrings.StyledMarkup.styled","text":"styled(content::AbstractString) -> AnnotatedString\n\nConstruct a styled string. Within the string, {<specs>:<content>} structures apply the formatting to <content>, according to the list of comma-separated specifications <specs>. Each spec can either take the form of a face name, an inline face specification, or a key=value pair. The value must be wrapped by {...} should it contain any of the characters ,=:{}.\n\nThis is a functional equivalent of the @styled_str macro, just without interpolation capabilities.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/StyledStrings.html#StyledStrings.Face","page":"StyledStrings","title":"StyledStrings.Face","text":"A Face is a collection of graphical attributes for displaying text. Faces control how text is displayed in the terminal, and possibly other places too.\n\nMost of the time, a Face will be stored in the global faces dicts as a unique association with a face name Symbol, and will be most often referred to by this name instead of the Face object itself.\n\nAttributes\n\nAll attributes can be set via the keyword constructor, and default to nothing.\n\nheight (an Int or Float64): The height in either deci-pt (when an Int), or as a factor of the base size (when a Float64).\nweight (a Symbol): One of the symbols (from faintest to densest) :thin, :extralight, :light, :semilight, :normal, :medium, :semibold, :bold, :extrabold, or :black. In terminals any weight greater than :normal is displayed as bold, and in terminals that support variable-brightness text, any weight less than :normal is displayed as faint.\nslant (a Symbol): One of the symbols :italic, :oblique, or :normal.\nforeground (a SimpleColor): The text foreground color.\nbackground (a SimpleColor): The text background color.\nunderline, the text underline, which takes one of the following forms:\na Bool: Whether the text should be underlined or not.\n\na SimpleColor: The text should be underlined with this color.\n\na Tuple{Nothing, Symbol}: The text should be underlined using the style set by the Symbol, one of :straight, :double, :curly, :dotted, or :dashed.\n\na Tuple{SimpleColor, Symbol}: The text should be underlined in the specified SimpleColor, and using the style specified by the Symbol, as before.\nstrikethrough (a Bool): Whether the text should be struck through.\ninverse (a Bool): Whether the foreground and background colors should be inverted.\ninherit (a Vector{Symbol}): Names of faces to inherit from, with earlier faces taking priority. All faces inherit from the :default face.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/StyledStrings.html#StyledStrings.addface!","page":"StyledStrings","title":"StyledStrings.addface!","text":"addface!(name::Symbol => default::Face)\n\nCreate a new face by the name name. So long as no face already exists by this name, default is added to both FACES.default and (a copy of) to FACES.current, with the current value returned.\n\nShould the face name already exist, nothing is returned.\n\nExamples\n\njulia> addface!(:mypkg_myface => Face(slant=:italic, underline=true))\nFace (sample)\n         slant: italic\n     underline: true\n\n\n\n\n\n","category":"function"},{"location":"stdlib/StyledStrings.html#StyledStrings.withfaces","page":"StyledStrings","title":"StyledStrings.withfaces","text":"withfaces(f, kv::Pair...)\nwithfaces(f, kvpair_itr)\n\nExecute f with FACES.current temporarily modified by zero or more :name => val arguments kv, or kvpair_itr which produces kv-form values.\n\nwithfaces is generally used via the withfaces(kv...) do ... end syntax. A value of nothing can be used to temporarily unset a face (if it has been set). When withfaces returns, the original FACES.current has been restored.\n\nExamples\n\njulia> withfaces(:yellow => Face(foreground=:red), :green => :blue) do\n           println(styled\"{yellow:red} and {green:blue} mixed make {magenta:purple}\")\n       end\nred and blue mixed make purple\n\n\n\n\n\n","category":"function"},{"location":"stdlib/StyledStrings.html#StyledStrings.SimpleColor","page":"StyledStrings","title":"StyledStrings.SimpleColor","text":"struct SimpleColor\n\nA basic representation of a color, intended for string styling purposes. It can either contain a named color (like :red), or an RGBTuple which is a NamedTuple specifying an r, g, b color with a bit-depth of 8.\n\nConstructors\n\nSimpleColor(name::Symbol)  # e.g. :red\nSimpleColor(rgb::RGBTuple) # e.g. (r=1, b=2, g=3)\nSimpleColor(r::Integer, b::Integer, b::Integer)\nSimpleColor(rgb::UInt32)   # e.g. 0x123456\n\nAlso see tryparse(SimpleColor, rgb::String).\n\n\n\n\n\n","category":"type"},{"location":"stdlib/StyledStrings.html#Base.parse-Tuple{Type{StyledStrings.SimpleColor}, String}","page":"StyledStrings","title":"Base.parse","text":"parse(::Type{SimpleColor}, rgb::String)\n\nAn analogue of tryparse(SimpleColor, rgb::String) (which see), that raises an error instead of returning nothing.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/StyledStrings.html#Base.tryparse-Tuple{Type{StyledStrings.SimpleColor}, String}","page":"StyledStrings","title":"Base.tryparse","text":"tryparse(::Type{SimpleColor}, rgb::String)\n\nAttempt to parse rgb as a SimpleColor. If rgb starts with # and has a length of 7, it is converted into a RGBTuple-backed SimpleColor. If rgb starts with a-z, rgb is interpreted as a color name and converted to a Symbol-backed SimpleColor.\n\nOtherwise, nothing is returned.\n\nExamples\n\njulia> tryparse(SimpleColor, \"blue\")\nSimpleColor(blue)\n\njulia> tryparse(SimpleColor, \"#9558b2\")\nSimpleColor(#9558b2)\n\njulia> tryparse(SimpleColor, \"#nocolor\")\n\n\n\n\n\n","category":"method"},{"location":"stdlib/StyledStrings.html#Base.merge-Tuple{StyledStrings.Face, StyledStrings.Face}","page":"StyledStrings","title":"Base.merge","text":"merge(initial::Face, others::Face...)\n\nMerge the properties of the initial face and others, with later faces taking priority.\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#lib-numbers","page":"Numbers","title":"Numbers","text":"","category":"section"},{"location":"base/numbers.html#Standard-Numeric-Types","page":"Numbers","title":"Standard Numeric Types","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"A type tree for all subtypes of Number in Base is shown below. Abstract types have been marked, the rest are concrete types.","category":"page"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Number  (Abstract Type)\n├─ Complex\n└─ Real  (Abstract Type)\n   ├─ AbstractFloat  (Abstract Type)\n   │  ├─ Float16\n   │  ├─ Float32\n   │  ├─ Float64\n   │  └─ BigFloat\n   ├─ Integer  (Abstract Type)\n   │  ├─ Bool\n   │  ├─ Signed  (Abstract Type)\n   │  │  ├─ Int8\n   │  │  ├─ Int16\n   │  │  ├─ Int32\n   │  │  ├─ Int64\n   │  │  ├─ Int128\n   │  │  └─ BigInt\n   │  └─ Unsigned  (Abstract Type)\n   │     ├─ UInt8\n   │     ├─ UInt16\n   │     ├─ UInt32\n   │     ├─ UInt64\n   │     └─ UInt128\n   ├─ Rational\n   └─ AbstractIrrational  (Abstract Type)\n      └─ Irrational","category":"page"},{"location":"base/numbers.html#Abstract-number-types","page":"Numbers","title":"Abstract number types","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Core.Number\nCore.Real\nCore.AbstractFloat\nCore.Integer\nCore.Signed\nCore.Unsigned\nBase.AbstractIrrational","category":"page"},{"location":"base/numbers.html#Core.Number","page":"Numbers","title":"Core.Number","text":"Number\n\nAbstract supertype for all number types.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Real","page":"Numbers","title":"Core.Real","text":"Real <: Number\n\nAbstract supertype for all real numbers.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.AbstractFloat","page":"Numbers","title":"Core.AbstractFloat","text":"AbstractFloat <: Real\n\nAbstract supertype for all floating point numbers.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Integer","page":"Numbers","title":"Core.Integer","text":"Integer <: Real\n\nAbstract supertype for all integers (e.g. Signed, Unsigned, and Bool).\n\nSee also isinteger, trunc, div.\n\nExamples\n\njulia> 42 isa Integer\ntrue\n\njulia> 1.0 isa Integer\nfalse\n\njulia> isinteger(1.0)\ntrue\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Signed","page":"Numbers","title":"Core.Signed","text":"Signed <: Integer\n\nAbstract supertype for all signed integers.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Unsigned","page":"Numbers","title":"Core.Unsigned","text":"Unsigned <: Integer\n\nAbstract supertype for all unsigned integers.\n\nBuilt-in unsigned integers are printed in hexadecimal, with prefix 0x, and can be entered in the same way.\n\nExamples\n\njulia> typemax(UInt8)\n0xff\n\njulia> Int(0x00d)\n13\n\njulia> unsigned(true)\n0x0000000000000001\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.AbstractIrrational","page":"Numbers","title":"Base.AbstractIrrational","text":"AbstractIrrational <: Real\n\nNumber type representing an exact irrational value, which is automatically rounded to the correct precision in arithmetic operations with other numeric quantities.\n\nSubtypes MyIrrational <: AbstractIrrational should implement at least ==(::MyIrrational, ::MyIrrational), hash(x::MyIrrational, h::UInt), and convert(::Type{F}, x::MyIrrational) where {F <: Union{BigFloat,Float32,Float64}}.\n\nIf a subtype is used to represent values that may occasionally be rational (e.g. a square-root type that represents √n for integers n will give a rational result when n is a perfect square), then it should also implement isinteger, iszero, isone, and == with Real values (since all of these default to false for AbstractIrrational types), as well as defining hash to equal that of the corresponding Rational.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Concrete-number-types","page":"Numbers","title":"Concrete number types","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Core.Float16\nCore.Float32\nCore.Float64\nBase.BigFloat\nCore.Bool\nCore.Int8\nCore.UInt8\nCore.Int16\nCore.UInt16\nCore.Int32\nCore.UInt32\nCore.Int64\nCore.UInt64\nCore.Int128\nCore.UInt128\nBase.Int\nBase.UInt\nBase.BigInt\nBase.Complex\nBase.Rational\nBase.Irrational","category":"page"},{"location":"base/numbers.html#Core.Float16","page":"Numbers","title":"Core.Float16","text":"Float16 <: AbstractFloat <: Real\n\n16-bit floating point number type (IEEE 754 standard). Binary format is 1 sign, 5 exponent, 10 fraction bits.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Float32","page":"Numbers","title":"Core.Float32","text":"Float32 <: AbstractFloat <: Real\n\n32-bit floating point number type (IEEE 754 standard). Binary format is 1 sign, 8 exponent, 23 fraction bits.\n\nThe exponent for scientific notation should be entered as lower-case f, thus 2f3 === 2.0f0 * 10^3 === Float32(2_000). For array literals and comprehensions, the element type can be specified before the square brackets: Float32[1,4,9] == Float32[i^2 for i in 1:3].\n\nSee also Inf32, NaN32, Float16, exponent, frexp.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Float64","page":"Numbers","title":"Core.Float64","text":"Float64 <: AbstractFloat <: Real\n\n64-bit floating point number type (IEEE 754 standard). Binary format is 1 sign, 11 exponent, 52 fraction bits. See bitstring, signbit, exponent, frexp, and significand to access various bits.\n\nThis is the default for floating point literals, 1.0 isa Float64, and for many operations such as 1/2, 2pi, log(2), range(0,90,length=4). Unlike integers, this default does not change with Sys.WORD_SIZE.\n\nThe exponent for scientific notation can be entered as e or E, thus 2e3 === 2.0E3 === 2.0 * 10^3. Doing so is strongly preferred over 10^n because integers overflow, thus 2.0 * 10^19 < 0 but 2e19 > 0.\n\nSee also Inf, NaN, floatmax, Float32, Complex.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.MPFR.BigFloat","page":"Numbers","title":"Base.MPFR.BigFloat","text":"BigFloat <: AbstractFloat\n\nArbitrary precision floating point number type.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Bool","page":"Numbers","title":"Core.Bool","text":"Bool <: Integer\n\nBoolean type, containing the values true and false.\n\nBool is a kind of number: false is numerically equal to 0 and true is numerically equal to 1. Moreover, false acts as a multiplicative \"strong zero\" against NaN and Inf:\n\njulia> [true, false] == [1, 0]\ntrue\n\njulia> 42.0 + true\n43.0\n\njulia> 0 .* (NaN, Inf, -Inf)\n(NaN, NaN, NaN)\n\njulia> false .* (NaN, Inf, -Inf)\n(0.0, 0.0, -0.0)\n\nBranches via if and other conditionals only accept Bool. There are no \"truthy\" values in Julia.\n\nComparisons typically return Bool, and broadcasted comparisons may return BitArray instead of an Array{Bool}.\n\njulia> [1 2 3 4 5] .< pi\n1×5 BitMatrix:\n 1  1  1  0  0\n\njulia> map(>(pi), [1 2 3 4 5])\n1×5 Matrix{Bool}:\n 0  0  0  1  1\n\nSee also trues, falses, ifelse.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int8","page":"Numbers","title":"Core.Int8","text":"Int8 <: Signed <: Integer\n\n8-bit signed integer type.\n\nRepresents numbers n ∈ -128:127. Note that such integers overflow without warning, thus typemax(Int8) + Int8(1) < 0.\n\nSee also Int, widen, BigInt.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt8","page":"Numbers","title":"Core.UInt8","text":"UInt8 <: Unsigned <: Integer\n\n8-bit unsigned integer type.\n\nPrinted in hexadecimal, thus 0x07 == 7.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int16","page":"Numbers","title":"Core.Int16","text":"Int16 <: Signed <: Integer\n\n16-bit signed integer type.\n\nRepresents numbers n ∈ -32768:32767. Note that such integers overflow without warning, thus typemax(Int16) + Int16(1) < 0.\n\nSee also Int, widen, BigInt.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt16","page":"Numbers","title":"Core.UInt16","text":"UInt16 <: Unsigned <: Integer\n\n16-bit unsigned integer type.\n\nPrinted in hexadecimal, thus 0x000f == 15.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int32","page":"Numbers","title":"Core.Int32","text":"Int32 <: Signed <: Integer\n\n32-bit signed integer type.\n\nNote that such integers overflow without warning, thus typemax(Int32) + Int32(1) < 0.\n\nSee also Int, widen, BigInt.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt32","page":"Numbers","title":"Core.UInt32","text":"UInt32 <: Unsigned <: Integer\n\n32-bit unsigned integer type.\n\nPrinted in hexadecimal, thus 0x0000001f == 31.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int64","page":"Numbers","title":"Core.Int64","text":"Int64 <: Signed <: Integer\n\n64-bit signed integer type.\n\nNote that such integers overflow without warning, thus typemax(Int64) + Int64(1) < 0.\n\nSee also Int, widen, BigInt.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt64","page":"Numbers","title":"Core.UInt64","text":"UInt64 <: Unsigned <: Integer\n\n64-bit unsigned integer type.\n\nPrinted in hexadecimal, thus 0x000000000000003f == 63.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int128","page":"Numbers","title":"Core.Int128","text":"Int128 <: Signed <: Integer\n\n128-bit signed integer type.\n\nNote that such integers overflow without warning, thus typemax(Int128) + Int128(1) < 0.\n\nSee also Int, widen, BigInt.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt128","page":"Numbers","title":"Core.UInt128","text":"UInt128 <: Unsigned <: Integer\n\n128-bit unsigned integer type.\n\nPrinted in hexadecimal, thus 0x0000000000000000000000000000007f == 127.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.Int","page":"Numbers","title":"Core.Int","text":"Int\n\nSys.WORD_SIZE-bit signed integer type, Int <: Signed <: Integer <: Real.\n\nThis is the default type of most integer literals and is an alias for either Int32 or Int64, depending on Sys.WORD_SIZE. It is the type returned by functions such as length, and the standard type for indexing arrays.\n\nNote that integers overflow without warning, thus typemax(Int) + 1 < 0 and 10^19 < 0. Overflow can be avoided by using BigInt. Very large integer literals will use a wider type, for instance 10_000_000_000_000_000_000 isa Int128.\n\nInteger division is div alias ÷, whereas / acting on integers returns Float64.\n\nSee also Int64, widen, typemax, bitstring.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Core.UInt","page":"Numbers","title":"Core.UInt","text":"UInt\n\nSys.WORD_SIZE-bit unsigned integer type, UInt <: Unsigned <: Integer.\n\nLike Int, the alias UInt may point to either UInt32 or UInt64, according to the value of Sys.WORD_SIZE on a given computer.\n\nPrinted and parsed in hexadecimal: UInt(15) === 0x000000000000000f.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.GMP.BigInt","page":"Numbers","title":"Base.GMP.BigInt","text":"BigInt <: Signed\n\nArbitrary precision integer type.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.Complex","page":"Numbers","title":"Base.Complex","text":"Complex{T<:Real} <: Number\n\nComplex number type with real and imaginary part of type T.\n\nComplexF16, ComplexF32 and ComplexF64 are aliases for Complex{Float16}, Complex{Float32} and Complex{Float64} respectively.\n\nSee also: Real, complex, real.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.Rational","page":"Numbers","title":"Base.Rational","text":"Rational{T<:Integer} <: Real\n\nRational number type, with numerator and denominator of type T. Rationals are checked for overflow.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Base.Irrational","page":"Numbers","title":"Base.Irrational","text":"Irrational{sym} <: AbstractIrrational\n\nNumber type representing an exact irrational value denoted by the symbol sym, such as π, ℯ and γ.\n\nSee also AbstractIrrational.\n\n\n\n\n\n","category":"type"},{"location":"base/numbers.html#Data-Formats","page":"Numbers","title":"Data Formats","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Base.digits\nBase.digits!\nBase.bitstring\nBase.parse\nBase.tryparse\nBase.big\nBase.signed\nBase.unsigned\nBase.float(::Any)\nBase.Math.significand\nBase.Math.exponent\nBase.complex(::Complex)\nBase.bswap\nBase.hex2bytes\nBase.hex2bytes!\nBase.bytes2hex","category":"page"},{"location":"base/numbers.html#Base.digits","page":"Numbers","title":"Base.digits","text":"digits([T<:Integer], n::Integer; base::T = 10, pad::Integer = 1)\n\nReturn an array with element type T (default Int) of the digits of n in the given base, optionally padded with zeros to a specified size. More significant digits are at higher indices, such that n == sum(digits[k]*base^(k-1) for k=1:length(digits)).\n\nSee also ndigits, digits!, and for base 2 also bitstring, count_ones.\n\nExamples\n\njulia> digits(10)\n2-element Vector{Int64}:\n 0\n 1\n\njulia> digits(10, base = 2)\n4-element Vector{Int64}:\n 0\n 1\n 0\n 1\n\njulia> digits(-256, base = 10, pad = 5)\n5-element Vector{Int64}:\n -6\n -5\n -2\n  0\n  0\n\njulia> n = rand(-999:999);\n\njulia> n == evalpoly(13, digits(n, base = 13))\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.digits!","page":"Numbers","title":"Base.digits!","text":"digits!(array, n::Integer; base::Integer = 10)\n\nFills an array of the digits of n in the given base. More significant digits are at higher indices. If the array length is insufficient, the least significant digits are filled up to the array length. If the array length is excessive, the excess portion is filled with zeros.\n\nExamples\n\njulia> digits!([2, 2, 2, 2], 10, base = 2)\n4-element Vector{Int64}:\n 0\n 1\n 0\n 1\n\njulia> digits!([2, 2, 2, 2, 2, 2], 10, base = 2)\n6-element Vector{Int64}:\n 0\n 1\n 0\n 1\n 0\n 0\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.bitstring","page":"Numbers","title":"Base.bitstring","text":"bitstring(n)\n\nA string giving the literal bit representation of a primitive type.\n\nSee also count_ones, count_zeros, digits.\n\nExamples\n\njulia> bitstring(Int32(4))\n\"00000000000000000000000000000100\"\n\njulia> bitstring(2.2)\n\"0100000000000001100110011001100110011001100110011001100110011010\"\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.parse","page":"Numbers","title":"Base.parse","text":"parse(::Type{SimpleColor}, rgb::String)\n\nAn analogue of tryparse(SimpleColor, rgb::String) (which see), that raises an error instead of returning nothing.\n\n\n\n\n\nparse(::Type{Platform}, triplet::AbstractString)\n\nParses a string platform triplet back into a Platform object.\n\n\n\n\n\nparse(type, str; base)\n\nParse a string as a number. For Integer types, a base can be specified (the default is 10). For floating-point types, the string is parsed as a decimal floating-point number.  Complex types are parsed from decimal strings of the form \"R±Iim\" as a Complex(R,I) of the requested type; \"i\" or \"j\" can also be used instead of \"im\", and \"R\" or \"Iim\" are also permitted. If the string does not contain a valid number, an error is raised.\n\ncompat: Julia 1.1\nparse(Bool, str) requires at least Julia 1.1.\n\nExamples\n\njulia> parse(Int, \"1234\")\n1234\n\njulia> parse(Int, \"1234\", base = 5)\n194\n\njulia> parse(Int, \"afc\", base = 16)\n2812\n\njulia> parse(Float64, \"1.2e-3\")\n0.0012\n\njulia> parse(Complex{Float64}, \"3.2e-1 + 4.5im\")\n0.32 + 4.5im\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.tryparse","page":"Numbers","title":"Base.tryparse","text":"tryparse(::Type{SimpleColor}, rgb::String)\n\nAttempt to parse rgb as a SimpleColor. If rgb starts with # and has a length of 7, it is converted into a RGBTuple-backed SimpleColor. If rgb starts with a-z, rgb is interpreted as a color name and converted to a Symbol-backed SimpleColor.\n\nOtherwise, nothing is returned.\n\nExamples\n\njulia> tryparse(SimpleColor, \"blue\")\nSimpleColor(blue)\n\njulia> tryparse(SimpleColor, \"#9558b2\")\nSimpleColor(#9558b2)\n\njulia> tryparse(SimpleColor, \"#nocolor\")\n\n\n\n\n\ntryparse(type, str; base)\n\nLike parse, but returns either a value of the requested type, or nothing if the string does not contain a valid number.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.big","page":"Numbers","title":"Base.big","text":"big(x)\n\nConvert a number to a maximum precision representation (typically BigInt or BigFloat). See BigFloat for information about some pitfalls with floating-point numbers.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.signed","page":"Numbers","title":"Base.signed","text":"signed(T::Integer)\n\nConvert an integer bitstype to the signed type of the same size.\n\nExamples\n\njulia> signed(UInt16)\nInt16\njulia> signed(UInt64)\nInt64\n\n\n\n\n\nsigned(x)\n\nConvert a number to a signed integer. If the argument is unsigned, it is reinterpreted as signed without checking for overflow.\n\nSee also: unsigned, sign, signbit.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.unsigned","page":"Numbers","title":"Base.unsigned","text":"unsigned(T::Integer)\n\nConvert an integer bitstype to the unsigned type of the same size.\n\nExamples\n\njulia> unsigned(Int16)\nUInt16\njulia> unsigned(UInt64)\nUInt64\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.float-Tuple{Any}","page":"Numbers","title":"Base.float","text":"float(x)\n\nConvert a number or array to a floating point data type.\n\nSee also: complex, oftype, convert.\n\nExamples\n\njulia> float(1:1000)\n1.0:1.0:1000.0\n\njulia> float(typemax(Int32))\n2.147483647e9\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.Math.significand","page":"Numbers","title":"Base.Math.significand","text":"significand(x)\n\nExtract the significand (a.k.a. mantissa) of a floating-point number. If x is a non-zero finite number, then the result will be a number of the same type and sign as x, and whose absolute value is on the interval 12). Otherwise x is returned.\n\nSee also frexp, exponent.\n\nExamples\n\njulia> significand(15.2)\n1.9\n\njulia> significand(-15.2)\n-1.9\n\njulia> significand(-15.2) * 2^3\n-15.2\n\njulia> significand(-Inf), significand(Inf), significand(NaN)\n(-Inf, Inf, NaN)\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.Math.exponent","page":"Numbers","title":"Base.Math.exponent","text":"exponent(x::Real) -> Int\n\nReturns the largest integer y such that 2^y ≤ abs(x).\n\nThrows a DomainError when x is zero, infinite, or NaN. For any other non-subnormal floating-point number x, this corresponds to the exponent bits of x.\n\nSee also signbit, significand, frexp, issubnormal, log2, ldexp.\n\nExamples\n\njulia> exponent(8)\n3\n\njulia> exponent(6.5)\n2\n\njulia> exponent(-1//4)\n-2\n\njulia> exponent(3.142e-4)\n-12\n\njulia> exponent(floatmin(Float32)), exponent(nextfloat(0.0f0))\n(-126, -149)\n\njulia> exponent(0.0)\nERROR: DomainError with 0.0:\nCannot be ±0.0.\n[...]\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.complex-Tuple{Complex}","page":"Numbers","title":"Base.complex","text":"complex(r, [i])\n\nConvert real numbers or arrays to complex. i defaults to zero.\n\nExamples\n\njulia> complex(7)\n7 + 0im\n\njulia> complex([1, 2, 3])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 2 + 0im\n 3 + 0im\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.bswap","page":"Numbers","title":"Base.bswap","text":"bswap(n)\n\nReverse the byte order of n.\n\n(See also ntoh and hton to convert between the current native byte order and big-endian order.)\n\nExamples\n\njulia> a = bswap(0x10203040)\n0x40302010\n\njulia> bswap(a)\n0x10203040\n\njulia> string(1, base = 2)\n\"1\"\n\njulia> string(bswap(1), base = 2)\n\"100000000000000000000000000000000000000000000000000000000\"\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.hex2bytes","page":"Numbers","title":"Base.hex2bytes","text":"hex2bytes(itr)\n\nGiven an iterable itr of ASCII codes for a sequence of hexadecimal digits, returns a Vector{UInt8} of bytes  corresponding to the binary representation: each successive pair of hexadecimal digits in itr gives the value of one byte in the return vector.\n\nThe length of itr must be even, and the returned array has half of the length of itr. See also hex2bytes! for an in-place version, and bytes2hex for the inverse.\n\ncompat: Julia 1.7\nCalling hex2bytes with iterators producing UInt8 values requires Julia 1.7 or later. In earlier versions, you can collect the iterator before calling hex2bytes.\n\nExamples\n\njulia> s = string(12345, base = 16)\n\"3039\"\n\njulia> hex2bytes(s)\n2-element Vector{UInt8}:\n 0x30\n 0x39\n\njulia> a = b\"01abEF\"\n6-element Base.CodeUnits{UInt8, String}:\n 0x30\n 0x31\n 0x61\n 0x62\n 0x45\n 0x46\n\njulia> hex2bytes(a)\n3-element Vector{UInt8}:\n 0x01\n 0xab\n 0xef\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.hex2bytes!","page":"Numbers","title":"Base.hex2bytes!","text":"hex2bytes!(dest::AbstractVector{UInt8}, itr)\n\nConvert an iterable itr of bytes representing a hexadecimal string to its binary representation, similar to hex2bytes except that the output is written in-place to dest. The length of dest must be half the length of itr.\n\ncompat: Julia 1.7\nCalling hex2bytes! with iterators producing UInt8 requires version 1.7. In earlier versions, you can collect the iterable before calling instead.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.bytes2hex","page":"Numbers","title":"Base.bytes2hex","text":"bytes2hex(itr) -> String\nbytes2hex(io::IO, itr)\n\nConvert an iterator itr of bytes to its hexadecimal string representation, either returning a String via bytes2hex(itr) or writing the string to an io stream via bytes2hex(io, itr).  The hexadecimal characters are all lowercase.\n\ncompat: Julia 1.7\nCalling bytes2hex with arbitrary iterators producing UInt8 values requires Julia 1.7 or later. In earlier versions, you can collect the iterator before calling bytes2hex.\n\nExamples\n\njulia> a = string(12345, base = 16)\n\"3039\"\n\njulia> b = hex2bytes(a)\n2-element Vector{UInt8}:\n 0x30\n 0x39\n\njulia> bytes2hex(b)\n\"3039\"\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#General-Number-Functions-and-Constants","page":"Numbers","title":"General Number Functions and Constants","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Base.one\nBase.oneunit\nBase.zero\nBase.im\nBase.MathConstants.pi\nBase.MathConstants.ℯ\nBase.MathConstants.catalan\nBase.MathConstants.eulergamma\nBase.MathConstants.golden\nBase.Inf\nBase.Inf64\nBase.Inf32\nBase.Inf16\nBase.NaN\nBase.NaN64\nBase.NaN32\nBase.NaN16\nBase.issubnormal\nBase.isfinite\nBase.isinf\nBase.isnan\nBase.iszero\nBase.isone\nBase.nextfloat\nBase.prevfloat\nBase.isinteger\nBase.isreal\nCore.Float32(::Any)\nCore.Float64(::Any)\nBase.Rounding.rounding\nBase.Rounding.setrounding(::Type, ::Any)\nBase.Rounding.setrounding(::Function, ::Type, ::RoundingMode)\nBase.Rounding.get_zero_subnormals\nBase.Rounding.set_zero_subnormals","category":"page"},{"location":"base/numbers.html#Base.one","page":"Numbers","title":"Base.one","text":"one(x)\none(T::type)\n\nReturn a multiplicative identity for x: a value such that one(x)*x == x*one(x) == x.  Alternatively one(T) can take a type T, in which case one returns a multiplicative identity for any x of type T.\n\nIf possible, one(x) returns a value of the same type as x, and one(T) returns a value of type T.  However, this may not be the case for types representing dimensionful quantities (e.g. time in days), since the multiplicative identity must be dimensionless.  In that case, one(x) should return an identity value of the same precision (and shape, for matrices) as x.\n\nIf you want a quantity that is of the same type as x, or of type T, even if x is dimensionful, use oneunit instead.\n\nSee also the identity function, and I in LinearAlgebra for the identity matrix.\n\nExamples\n\njulia> one(3.7)\n1.0\n\njulia> one(Int)\n1\n\njulia> import Dates; one(Dates.Day(1))\n1\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.oneunit","page":"Numbers","title":"Base.oneunit","text":"oneunit(x::T)\noneunit(T::Type)\n\nReturn T(one(x)), where T is either the type of the argument or (if a type is passed) the argument.  This differs from one for dimensionful quantities: one is dimensionless (a multiplicative identity) while oneunit is dimensionful (of the same type as x, or of type T).\n\nExamples\n\njulia> oneunit(3.7)\n1.0\n\njulia> import Dates; oneunit(Dates.Day)\n1 day\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.zero","page":"Numbers","title":"Base.zero","text":"zero(x)\nzero(::Type)\n\nGet the additive identity element for the type of x (x can also specify the type itself).\n\nSee also iszero, one, oneunit, oftype.\n\nExamples\n\njulia> zero(1)\n0\n\njulia> zero(big\"2.0\")\n0.0\n\njulia> zero(rand(2,2))\n2×2 Matrix{Float64}:\n 0.0  0.0\n 0.0  0.0\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.im","page":"Numbers","title":"Base.im","text":"im\n\nThe imaginary unit.\n\nSee also: imag, angle, complex.\n\nExamples\n\njulia> im * im\n-1 + 0im\n\njulia> (2.0 + 3im)^2\n-5.0 + 12.0im\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.MathConstants.pi","page":"Numbers","title":"Base.MathConstants.pi","text":"π\npi\n\nThe constant pi.\n\nUnicode π can be typed by writing \\pi then pressing tab in the Julia REPL, and in many editors.\n\nSee also: sinpi, sincospi, deg2rad.\n\nExamples\n\njulia> pi\nπ = 3.1415926535897...\n\njulia> 1/2pi\n0.15915494309189535\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.MathConstants.ℯ","page":"Numbers","title":"Base.MathConstants.ℯ","text":"ℯ\ne\n\nThe constant ℯ.\n\nUnicode ℯ can be typed by writing \\euler and pressing tab in the Julia REPL, and in many editors.\n\nSee also: exp, cis, cispi.\n\nExamples\n\njulia> ℯ\nℯ = 2.7182818284590...\n\njulia> log(ℯ)\n1\n\njulia> ℯ^(im)π ≈ -1\ntrue\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.MathConstants.catalan","page":"Numbers","title":"Base.MathConstants.catalan","text":"catalan\n\nCatalan's constant.\n\nExamples\n\njulia> Base.MathConstants.catalan\ncatalan = 0.9159655941772...\n\njulia> sum(log(x)/(1+x^2) for x in 1:0.01:10^6) * 0.01\n0.9159466120554123\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.MathConstants.eulergamma","page":"Numbers","title":"Base.MathConstants.eulergamma","text":"γ\neulergamma\n\nEuler's constant.\n\nExamples\n\njulia> Base.MathConstants.eulergamma\nγ = 0.5772156649015...\n\njulia> dx = 10^-6;\n\njulia> sum(-exp(-x) * log(x) for x in dx:dx:100) * dx\n0.5772078382499133\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.MathConstants.golden","page":"Numbers","title":"Base.MathConstants.golden","text":"φ\ngolden\n\nThe golden ratio.\n\nExamples\n\njulia> Base.MathConstants.golden\nφ = 1.6180339887498...\n\njulia> (2ans - 1)^2 ≈ 5\ntrue\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.Inf","page":"Numbers","title":"Base.Inf","text":"Inf, Inf64\n\nPositive infinity of type Float64.\n\nSee also: isfinite, typemax, NaN, Inf32.\n\nExamples\n\njulia> π/0\nInf\n\njulia> +1.0 / -0.0\n-Inf\n\njulia> ℯ^-Inf\n0.0\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.Inf64","page":"Numbers","title":"Base.Inf64","text":"Inf, Inf64\n\nPositive infinity of type Float64.\n\nSee also: isfinite, typemax, NaN, Inf32.\n\nExamples\n\njulia> π/0\nInf\n\njulia> +1.0 / -0.0\n-Inf\n\njulia> ℯ^-Inf\n0.0\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.Inf32","page":"Numbers","title":"Base.Inf32","text":"Inf32\n\nPositive infinity of type Float32.\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.Inf16","page":"Numbers","title":"Base.Inf16","text":"Inf16\n\nPositive infinity of type Float16.\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.NaN","page":"Numbers","title":"Base.NaN","text":"NaN, NaN64\n\nA not-a-number value of type Float64.\n\nSee also: isnan, missing, NaN32, Inf.\n\nExamples\n\njulia> 0/0\nNaN\n\njulia> Inf - Inf\nNaN\n\njulia> NaN == NaN, isequal(NaN, NaN), isnan(NaN)\n(false, true, true)\n\nnote: Note\nAlways use isnan or isequal for checking for NaN. Using x === NaN may give unexpected results:julia> reinterpret(UInt32, NaN32)\n0x7fc00000\n\njulia> NaN32p1 = reinterpret(Float32, 0x7fc00001)\nNaN32\n\njulia> NaN32p1 === NaN32, isequal(NaN32p1, NaN32), isnan(NaN32p1)\n(false, true, true)\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.NaN64","page":"Numbers","title":"Base.NaN64","text":"NaN, NaN64\n\nA not-a-number value of type Float64.\n\nSee also: isnan, missing, NaN32, Inf.\n\nExamples\n\njulia> 0/0\nNaN\n\njulia> Inf - Inf\nNaN\n\njulia> NaN == NaN, isequal(NaN, NaN), isnan(NaN)\n(false, true, true)\n\nnote: Note\nAlways use isnan or isequal for checking for NaN. Using x === NaN may give unexpected results:julia> reinterpret(UInt32, NaN32)\n0x7fc00000\n\njulia> NaN32p1 = reinterpret(Float32, 0x7fc00001)\nNaN32\n\njulia> NaN32p1 === NaN32, isequal(NaN32p1, NaN32), isnan(NaN32p1)\n(false, true, true)\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.NaN32","page":"Numbers","title":"Base.NaN32","text":"NaN32\n\nA not-a-number value of type Float32.\n\nSee also: NaN.\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.NaN16","page":"Numbers","title":"Base.NaN16","text":"NaN16\n\nA not-a-number value of type Float16.\n\nSee also: NaN.\n\n\n\n\n\n","category":"constant"},{"location":"base/numbers.html#Base.issubnormal","page":"Numbers","title":"Base.issubnormal","text":"issubnormal(f) -> Bool\n\nTest whether a floating point number is subnormal.\n\nAn IEEE floating point number is subnormal when its exponent bits are zero and its significand is not zero.\n\nExamples\n\njulia> floatmin(Float32)\n1.1754944f-38\n\njulia> issubnormal(1.0f-37)\nfalse\n\njulia> issubnormal(1.0f-38)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isfinite","page":"Numbers","title":"Base.isfinite","text":"isfinite(f) -> Bool\n\nTest whether a number is finite.\n\nExamples\n\njulia> isfinite(5)\ntrue\n\njulia> isfinite(NaN32)\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isinf","page":"Numbers","title":"Base.isinf","text":"isinf(f) -> Bool\n\nTest whether a number is infinite.\n\nSee also: Inf, iszero, isfinite, isnan.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isnan","page":"Numbers","title":"Base.isnan","text":"isnan(f) -> Bool\n\nTest whether a number value is a NaN, an indeterminate value which is neither an infinity nor a finite number (\"not a number\").\n\nSee also: iszero, isone, isinf, ismissing.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.iszero","page":"Numbers","title":"Base.iszero","text":"iszero(x)\n\nReturn true if x == zero(x); if x is an array, this checks whether all of the elements of x are zero.\n\nSee also: isone, isinteger, isfinite, isnan.\n\nExamples\n\njulia> iszero(0.0)\ntrue\n\njulia> iszero([1, 9, 0])\nfalse\n\njulia> iszero([false, 0, 0])\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isone","page":"Numbers","title":"Base.isone","text":"isone(x)\n\nReturn true if x == one(x); if x is an array, this checks whether x is an identity matrix.\n\nExamples\n\njulia> isone(1.0)\ntrue\n\njulia> isone([1 0; 0 2])\nfalse\n\njulia> isone([1 0; 0 true])\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.nextfloat","page":"Numbers","title":"Base.nextfloat","text":"nextfloat(x::AbstractFloat, n::Integer)\n\nThe result of n iterative applications of nextfloat to x if n >= 0, or -n applications of prevfloat if n < 0.\n\n\n\n\n\nnextfloat(x::AbstractFloat)\n\nReturn the smallest floating point number y of the same type as x such x < y. If no such y exists (e.g. if x is Inf or NaN), then return x.\n\nSee also: prevfloat, eps, issubnormal.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.prevfloat","page":"Numbers","title":"Base.prevfloat","text":"prevfloat(x::AbstractFloat, n::Integer)\n\nThe result of n iterative applications of prevfloat to x if n >= 0, or -n applications of nextfloat if n < 0.\n\n\n\n\n\nprevfloat(x::AbstractFloat)\n\nReturn the largest floating point number y of the same type as x such y < x. If no such y exists (e.g. if x is -Inf or NaN), then return x.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isinteger","page":"Numbers","title":"Base.isinteger","text":"isinteger(x) -> Bool\n\nTest whether x is numerically equal to some integer.\n\nExamples\n\njulia> isinteger(4.0)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isreal","page":"Numbers","title":"Base.isreal","text":"isreal(x) -> Bool\n\nTest whether x or all its elements are numerically equal to some real number including infinities and NaNs. isreal(x) is true if isequal(x, real(x)) is true.\n\nExamples\n\njulia> isreal(5.)\ntrue\n\njulia> isreal(1 - 3im)\nfalse\n\njulia> isreal(Inf + 0im)\ntrue\n\njulia> isreal([4.; complex(0,1)])\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Core.Float32-Tuple{Any}","page":"Numbers","title":"Core.Float32","text":"Float32(x [, mode::RoundingMode])\n\nCreate a Float32 from x. If x is not exactly representable then mode determines how x is rounded.\n\nExamples\n\njulia> Float32(1/3, RoundDown)\n0.3333333f0\n\njulia> Float32(1/3, RoundUp)\n0.33333334f0\n\nSee RoundingMode for available rounding modes.\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Core.Float64-Tuple{Any}","page":"Numbers","title":"Core.Float64","text":"Float64(x [, mode::RoundingMode])\n\nCreate a Float64 from x. If x is not exactly representable then mode determines how x is rounded.\n\nExamples\n\njulia> Float64(pi, RoundDown)\n3.141592653589793\n\njulia> Float64(pi, RoundUp)\n3.1415926535897936\n\nSee RoundingMode for available rounding modes.\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.Rounding.rounding","page":"Numbers","title":"Base.Rounding.rounding","text":"rounding(T)\n\nGet the current floating point rounding mode for type T, controlling the rounding of basic arithmetic functions (+, -, *, / and sqrt) and type conversion.\n\nSee RoundingMode for available modes.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.Rounding.setrounding-Tuple{Type, Any}","page":"Numbers","title":"Base.Rounding.setrounding","text":"setrounding(T, mode)\n\nSet the rounding mode of floating point type T, controlling the rounding of basic arithmetic functions (+, -, *, / and sqrt) and type conversion. Other numerical functions may give incorrect or invalid values when using rounding modes other than the default RoundNearest.\n\nNote that this is currently only supported for T == BigFloat.\n\nwarning: Warning\nThis function is not thread-safe. It will affect code running on all threads, but its behavior is undefined if called concurrently with computations that use the setting.\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.Rounding.setrounding-Tuple{Function, Type, RoundingMode}","page":"Numbers","title":"Base.Rounding.setrounding","text":"setrounding(f::Function, T, mode)\n\nChange the rounding mode of floating point type T for the duration of f. It is logically equivalent to:\n\nold = rounding(T)\nsetrounding(T, mode)\nf()\nsetrounding(T, old)\n\nSee RoundingMode for available rounding modes.\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.Rounding.get_zero_subnormals","page":"Numbers","title":"Base.Rounding.get_zero_subnormals","text":"get_zero_subnormals() -> Bool\n\nReturn false if operations on subnormal floating-point values (\"denormals\") obey rules for IEEE arithmetic, and true if they might be converted to zeros.\n\nwarning: Warning\nThis function only affects the current thread.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.Rounding.set_zero_subnormals","page":"Numbers","title":"Base.Rounding.set_zero_subnormals","text":"set_zero_subnormals(yes::Bool) -> Bool\n\nIf yes is false, subsequent floating-point operations follow rules for IEEE arithmetic on subnormal values (\"denormals\"). Otherwise, floating-point operations are permitted (but not required) to convert subnormal inputs or outputs to zero. Returns true unless yes==true but the hardware does not support zeroing of subnormal numbers.\n\nset_zero_subnormals(true) can speed up some computations on some hardware. However, it can break identities such as (x-y==0) == (x==y).\n\nwarning: Warning\nThis function only affects the current thread.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Integers","page":"Numbers","title":"Integers","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Base.count_ones\nBase.count_zeros\nBase.leading_zeros\nBase.leading_ones\nBase.trailing_zeros\nBase.trailing_ones\nBase.isodd\nBase.iseven\nBase.@int128_str\nBase.@uint128_str","category":"page"},{"location":"base/numbers.html#Base.count_ones","page":"Numbers","title":"Base.count_ones","text":"count_ones(x::Integer) -> Integer\n\nNumber of ones in the binary representation of x.\n\nExamples\n\njulia> count_ones(7)\n3\n\njulia> count_ones(Int32(-1))\n32\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.count_zeros","page":"Numbers","title":"Base.count_zeros","text":"count_zeros(x::Integer) -> Integer\n\nNumber of zeros in the binary representation of x.\n\nExamples\n\njulia> count_zeros(Int32(2 ^ 16 - 1))\n16\n\njulia> count_zeros(-1)\n0\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.leading_zeros","page":"Numbers","title":"Base.leading_zeros","text":"leading_zeros(x::Integer) -> Integer\n\nNumber of zeros leading the binary representation of x.\n\nExamples\n\njulia> leading_zeros(Int32(1))\n31\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.leading_ones","page":"Numbers","title":"Base.leading_ones","text":"leading_ones(x::Integer) -> Integer\n\nNumber of ones leading the binary representation of x.\n\nExamples\n\njulia> leading_ones(UInt32(2 ^ 32 - 2))\n31\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.trailing_zeros","page":"Numbers","title":"Base.trailing_zeros","text":"trailing_zeros(x::Integer) -> Integer\n\nNumber of zeros trailing the binary representation of x.\n\nExamples\n\njulia> trailing_zeros(2)\n1\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.trailing_ones","page":"Numbers","title":"Base.trailing_ones","text":"trailing_ones(x::Integer) -> Integer\n\nNumber of ones trailing the binary representation of x.\n\nExamples\n\njulia> trailing_ones(3)\n2\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.isodd","page":"Numbers","title":"Base.isodd","text":"isodd(x::Number) -> Bool\n\nReturn true if x is an odd integer (that is, an integer not divisible by 2), and false otherwise.\n\ncompat: Julia 1.7\nNon-Integer arguments require Julia 1.7 or later.\n\nExamples\n\njulia> isodd(9)\ntrue\n\njulia> isodd(10)\nfalse\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.iseven","page":"Numbers","title":"Base.iseven","text":"iseven(x::Number) -> Bool\n\nReturn true if x is an even integer (that is, an integer divisible by 2), and false otherwise.\n\ncompat: Julia 1.7\nNon-Integer arguments require Julia 1.7 or later.\n\nExamples\n\njulia> iseven(9)\nfalse\n\njulia> iseven(10)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Core.@int128_str","page":"Numbers","title":"Core.@int128_str","text":"@int128_str str\n\nParse str as an Int128. Throw an ArgumentError if the string is not a valid integer.\n\nExamples\n\njulia> int128\"123456789123\"\n123456789123\n\njulia> int128\"123456789123.4\"\nERROR: LoadError: ArgumentError: invalid base 10 digit '.' in \"123456789123.4\"\n[...]\n\n\n\n\n\n","category":"macro"},{"location":"base/numbers.html#Core.@uint128_str","page":"Numbers","title":"Core.@uint128_str","text":"@uint128_str str\n\nParse str as an UInt128. Throw an ArgumentError if the string is not a valid integer.\n\nExamples\n\njulia> uint128\"123456789123\"\n0x00000000000000000000001cbe991a83\n\njulia> uint128\"-123456789123\"\nERROR: LoadError: ArgumentError: invalid base 10 digit '-' in \"-123456789123\"\n[...]\n\n\n\n\n\n","category":"macro"},{"location":"base/numbers.html#BigFloats-and-BigInts","page":"Numbers","title":"BigFloats and BigInts","text":"","category":"section"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"The BigFloat and BigInt types implements arbitrary-precision floating point and integer arithmetic, respectively. For BigFloat the GNU MPFR library is used, and for BigInt the GNU Multiple Precision Arithmetic Library (GMP) is used.","category":"page"},{"location":"base/numbers.html","page":"Numbers","title":"Numbers","text":"Base.MPFR.BigFloat(::Any, rounding::RoundingMode)\nBase.precision\nBase.MPFR.setprecision\nBase.GMP.BigInt(::Any)\nBase.@big_str","category":"page"},{"location":"base/numbers.html#Base.MPFR.BigFloat-Tuple{Any, RoundingMode}","page":"Numbers","title":"Base.MPFR.BigFloat","text":"BigFloat(x::Union{Real, AbstractString} [, rounding::RoundingMode=rounding(BigFloat)]; [precision::Integer=precision(BigFloat)])\n\nCreate an arbitrary precision floating point number from x, with precision precision. The rounding argument specifies the direction in which the result should be rounded if the conversion cannot be done exactly. If not provided, these are set by the current global values.\n\nBigFloat(x::Real) is the same as convert(BigFloat,x), except if x itself is already BigFloat, in which case it will return a value with the precision set to the current global precision; convert will always return x.\n\nBigFloat(x::AbstractString) is identical to parse. This is provided for convenience since decimal literals are converted to Float64 when parsed, so BigFloat(2.1) may not yield what you expect.\n\nSee also:\n\n@big_str\nrounding and setrounding\nprecision and setprecision\n\ncompat: Julia 1.1\nprecision as a keyword argument requires at least Julia 1.1. In Julia 1.0 precision is the second positional argument (BigFloat(x, precision)).\n\nExamples\n\njulia> BigFloat(2.1) # 2.1 here is a Float64\n2.100000000000000088817841970012523233890533447265625\n\njulia> BigFloat(\"2.1\") # the closest BigFloat to 2.1\n2.099999999999999999999999999999999999999999999999999999999999999999999999999986\n\njulia> BigFloat(\"2.1\", RoundUp)\n2.100000000000000000000000000000000000000000000000000000000000000000000000000021\n\njulia> BigFloat(\"2.1\", RoundUp, precision=128)\n2.100000000000000000000000000000000000007\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Base.precision","page":"Numbers","title":"Base.precision","text":"precision(num::AbstractFloat; base::Integer=2)\nprecision(T::Type; base::Integer=2)\n\nGet the precision of a floating point number, as defined by the effective number of bits in the significand, or the precision of a floating-point type T (its current default, if T is a variable-precision type like BigFloat).\n\nIf base is specified, then it returns the maximum corresponding number of significand digits in that base.\n\ncompat: Julia 1.8\nThe base keyword requires at least Julia 1.8.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.MPFR.setprecision","page":"Numbers","title":"Base.MPFR.setprecision","text":"setprecision([T=BigFloat,] precision::Int; base=2)\n\nSet the precision (in bits, by default) to be used for T arithmetic. If base is specified, then the precision is the minimum required to give at least precision digits in the given base.\n\nwarning: Warning\nThis function is not thread-safe. It will affect code running on all threads, but its behavior is undefined if called concurrently with computations that use the setting.\n\ncompat: Julia 1.8\nThe base keyword requires at least Julia 1.8.\n\n\n\n\n\nsetprecision(f::Function, [T=BigFloat,] precision::Integer; base=2)\n\nChange the T arithmetic precision (in the given base) for the duration of f. It is logically equivalent to:\n\nold = precision(BigFloat)\nsetprecision(BigFloat, precision)\nf()\nsetprecision(BigFloat, old)\n\nOften used as setprecision(T, precision) do ... end\n\nNote: nextfloat(), prevfloat() do not use the precision mentioned by setprecision.\n\ncompat: Julia 1.8\nThe base keyword requires at least Julia 1.8.\n\n\n\n\n\n","category":"function"},{"location":"base/numbers.html#Base.GMP.BigInt-Tuple{Any}","page":"Numbers","title":"Base.GMP.BigInt","text":"BigInt(x)\n\nCreate an arbitrary precision integer. x may be an Int (or anything that can be converted to an Int). The usual mathematical operators are defined for this type, and results are promoted to a BigInt.\n\nInstances can be constructed from strings via parse, or using the big string literal.\n\nExamples\n\njulia> parse(BigInt, \"42\")\n42\n\njulia> big\"313\"\n313\n\njulia> BigInt(10)^19\n10000000000000000000\n\n\n\n\n\n","category":"method"},{"location":"base/numbers.html#Core.@big_str","page":"Numbers","title":"Core.@big_str","text":"@big_str str\n\nParse a string into a BigInt or BigFloat, and throw an ArgumentError if the string is not a valid number. For integers _ is allowed in the string as a separator.\n\nExamples\n\njulia> big\"123_456\"\n123456\n\njulia> big\"7891.5\"\n7891.5\n\njulia> big\"_\"\nERROR: ArgumentError: invalid number format _ for BigInt or BigFloat\n[...]\n\nwarning: Warning\nUsing @big_str for constructing BigFloat values may not result in the behavior that might be naively expected: as a macro, @big_str obeys the global precision (setprecision) and rounding mode (setrounding) settings as they are at load time. Thus, a function like () -> precision(big\"0.3\") returns a constant whose value depends on the value of the precision at the point when the function is defined, not at the precision at the time when the function is called.\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/EscapeAnalysis.html#EscapeAnalysis","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Core.Compiler.EscapeAnalysis is a compiler utility module that aims to analyze escape information of Julia's SSA-form IR a.k.a. IRCode.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"This escape analysis aims to:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"leverage Julia's high-level semantics, especially reason about escapes and aliasing via inter-procedural calls\nbe versatile enough to be used for various optimizations including alias-aware SROA, early finalize insertion, copy-free ImmutableArray construction, stack allocation of mutable objects, and so on.\nachieve a simple implementation based on a fully backward data-flow analysis implementation as well as a new lattice design that combines orthogonal lattice properties","category":"page"},{"location":"devdocs/EscapeAnalysis.html#Try-it-out!","page":"EscapeAnalysis","title":"Try it out!","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"You can give a try to the escape analysis by loading the EAUtils.jl utility script that defines the convenience entries code_escapes and @code_escapes for testing and debugging purposes:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"let JULIA_DIR = normpath(Sys.BINDIR, \"..\", \"share\", \"julia\")\n    # load `EscapeAnalysis` module to define the core analysis code\n    include(normpath(JULIA_DIR, \"base\", \"compiler\", \"ssair\", \"EscapeAnalysis\", \"EscapeAnalysis.jl\"))\n    using .EscapeAnalysis\n    # load `EAUtils` module to define the utilities\n    include(normpath(JULIA_DIR, \"test\", \"compiler\", \"EscapeAnalysis\", \"EAUtils.jl\"))\n    using .EAUtils\nend\n\nmutable struct SafeRef{T}\n    x::T\nend\nBase.getindex(x::SafeRef) = x.x;\nBase.setindex!(x::SafeRef, v) = x.x = v;\nBase.isassigned(x::SafeRef) = true;\nget′(x) = isassigned(x) ? x[] : throw(x);\n\nresult = code_escapes((String,String,String,String)) do s1, s2, s3, s4\n    r1 = Ref(s1)\n    r2 = Ref(s2)\n    r3 = SafeRef(s3)\n    try\n        s1 = get′(r1)\n        ret = sizeof(s1)\n    catch err\n        global GV = err # will definitely escape `r1`\n    end\n    s2 = get′(r2)       # still `r2` doesn't escape fully\n    s3 = get′(r3)       # still `r3` doesn't escape fully\n    s4 = sizeof(s4)     # the argument `s4` doesn't escape here\n    return s2, s3, s4\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"The symbols on the side of each call argument and SSA statements represent the following meaning:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"◌ (plain): this value is not analyzed because escape information of it won't be used anyway (when the object is isbitstype for example)\n✓ (green or cyan): this value never escapes (has_no_escape(result.state[x]) holds), colored blue if it has arg escape also (has_arg_escape(result.state[x]) holds)\n↑ (blue or yellow): this value can escape to the caller via return (has_return_escape(result.state[x]) holds), colored yellow if it has unhandled thrown escape also (has_thrown_escape(result.state[x]) holds)\nX (red): this value can escape to somewhere the escape analysis can't reason about like escapes to a global memory (has_all_escape(result.state[x]) holds)\n* (bold): this value's escape state is between the ReturnEscape and AllEscape in the partial order of EscapeInfo, colored yellow if it has unhandled thrown escape also (has_thrown_escape(result.state[x]) holds)\n′: this value has additional object field / array element information in its AliasInfo property","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Escape information of each call argument and SSA value can be inspected programmatically as like:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"result.state[Core.Argument(3)] # get EscapeInfo of `s2`\n\nresult.state[Core.SSAValue(3)] # get EscapeInfo of `r3`","category":"page"},{"location":"devdocs/EscapeAnalysis.html#Analysis-Design","page":"EscapeAnalysis","title":"Analysis Design","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html#Lattice-Design","page":"EscapeAnalysis","title":"Lattice Design","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"EscapeAnalysis is implemented as a data-flow analysis that works on a lattice of x::EscapeInfo, which is composed of the following properties:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"x.Analyzed::Bool: not formally part of the lattice, only indicates x has not been analyzed or not\nx.ReturnEscape::BitSet: records SSA statements where x can escape to the caller via return\nx.ThrownEscape::BitSet: records SSA statements where x can be thrown as exception (used for the exception handling described below)\nx.AliasInfo: maintains all possible values that can be aliased to fields or array elements of x (used for the alias analysis described below)\nx.ArgEscape::Int (not implemented yet): indicates it will escape to the caller through setfield! on argument(s)","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"These attributes can be combined to create a partial lattice that has a finite height, given the invariant that an input program has a finite number of statements, which is assured by Julia's semantics. The clever part of this lattice design is that it enables a simpler implementation of lattice operations by allowing them to handle each lattice property separately[LatticeDesign].","category":"page"},{"location":"devdocs/EscapeAnalysis.html#Backward-Escape-Propagation","page":"EscapeAnalysis","title":"Backward Escape Propagation","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"This escape analysis implementation is based on the data-flow algorithm described in the paper[MM02]. The analysis works on the lattice of EscapeInfo and transitions lattice elements from the bottom to the top until every lattice element gets converged to a fixed point by maintaining a (conceptual) working set that contains program counters corresponding to remaining SSA statements to be analyzed. The analysis manages a single global state that tracks EscapeInfo of each argument and SSA statement, but also note that some flow-sensitivity is encoded as program counters recorded in EscapeInfo's ReturnEscape property, which can be combined with domination analysis later to reason about flow-sensitivity if necessary.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"One distinctive design of this escape analysis is that it is fully backward, i.e. escape information flows from usages to definitions. For example, in the code snippet below, EA first analyzes the statement return %1 and imposes ReturnEscape on %1 (corresponding to obj), and then it analyzes %1 = %new(Base.RefValue{String, _2})) and propagates the ReturnEscape imposed on %1 to the call argument _2 (corresponding to s):","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,)) do s\n    obj = Ref(s)\n    return obj\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"The key observation here is that this backward analysis allows escape information to flow naturally along the use-def chain rather than control-flow[BackandForth]. As a result this scheme enables a simple implementation of escape analysis, e.g. PhiNode for example can be handled simply by propagating escape information imposed on a PhiNode to its predecessor values:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((Bool, String, String)) do cnd, s, t\n    if cnd\n        obj = Ref(s)\n    else\n        obj = Ref(t)\n    end\n    return obj\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html#EA-Alias-Analysis","page":"EscapeAnalysis","title":"Alias Analysis","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"EscapeAnalysis implements a backward field analysis in order to reason about escapes imposed on object fields with certain accuracy, and x::EscapeInfo's x.AliasInfo property exists for this purpose. It records all possible values that can be aliased to fields of x at \"usage\" sites, and then the escape information of that recorded values are propagated to the actual field values later at \"definition\" sites. More specifically, the analysis records a value that may be aliased to a field of object by analyzing getfield call, and then it propagates its escape information to the field when analyzing %new(...) expression or setfield! call[Dynamism].","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,)) do s\n    obj = SafeRef(\"init\")\n    obj[] = s\n    v = obj[]\n    return v\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"In the example above, ReturnEscape imposed on %3 (corresponding to v) is not directly propagated to %1 (corresponding to obj) but rather that ReturnEscape is only propagated to _2 (corresponding to s). Here %3 is recorded in %1's AliasInfo property as it can be aliased to the first field of %1, and then when analyzing Base.setfield!(%1, :x, _2)::String, that escape information is propagated to _2 but not to %1.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"So EscapeAnalysis tracks which IR elements can be aliased across a getfield-%new/setfield! chain in order to analyze escapes of object fields, but actually this alias analysis needs to be generalized to handle other IR elements as well. This is because in Julia IR the same object is sometimes represented by different IR elements and so we should make sure that those different IR elements that actually can represent the same object share the same escape information. IR elements that return the same object as their operand(s), such as PiNode and typeassert, can cause that IR-level aliasing and thus requires escape information imposed on any of such aliased values to be shared between them. More interestingly, it is also needed for correctly reasoning about mutations on PhiNode. Let's consider the following example:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((Bool, String,)) do cond, x\n    if cond\n        ϕ2 = ϕ1 = SafeRef(\"foo\")\n    else\n        ϕ2 = ϕ1 = SafeRef(\"bar\")\n    end\n    ϕ2[] = x\n    y = ϕ1[]\n    return y\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"ϕ1 = %5 and ϕ2 = %6 are aliased and thus ReturnEscape imposed on %8 = Base.getfield(%6, :x)::String (corresponding to y = ϕ1[]) needs to be propagated to Base.setfield!(%5, :x, _3)::String (corresponding to ϕ2[] = x). In order for such escape information to be propagated correctly, the analysis should recognize that the predecessors of ϕ1 and ϕ2 can be aliased as well and equalize their escape information.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"One interesting property of such aliasing information is that it is not known at \"usage\" site but can only be derived at \"definition\" site (as aliasing is conceptually equivalent to assignment), and thus it doesn't naturally fit in a backward analysis. In order to efficiently propagate escape information between related values, EscapeAnalysis.jl uses an approach inspired by the escape analysis algorithm explained in an old JVM paper[JVM05]. That is, in addition to managing escape lattice elements, the analysis also maintains an \"equi\"-alias set, a disjoint set of aliased arguments and SSA statements. The alias set manages values that can be aliased to each other and allows escape information imposed on any of such aliased values to be equalized between them.","category":"page"},{"location":"devdocs/EscapeAnalysis.html#EA-Array-Analysis","page":"EscapeAnalysis","title":"Array Analysis","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"The alias analysis for object fields described above can also be generalized to analyze array operations. EscapeAnalysis implements handlings for various primitive array operations so that it can propagate escapes via arrayref-arrayset use-def chain and does not escape allocated arrays too conservatively:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,)) do s\n    ary = Any[]\n    push!(ary, SafeRef(s))\n    return ary[1], length(ary)\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"In the above example EscapeAnalysis understands that %20 and %2 (corresponding to the allocated object SafeRef(s)) are aliased via the arrayset-arrayref chain and imposes ReturnEscape on them, but not impose it on the allocated array %1 (corresponding to ary). EscapeAnalysis still imposes ThrownEscape on ary since it also needs to account for potential escapes via BoundsError, but also note that such unhandled ThrownEscape can often be ignored when optimizing the ary allocation.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Furthermore, in cases when array index information as well as array dimensions can be known precisely, EscapeAnalysis is able to even reason about \"per-element\" aliasing via arrayref-arrayset chain, as EscapeAnalysis does \"per-field\" alias analysis for objects:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,String)) do s, t\n    ary = Vector{Any}(undef, 2)\n    ary[1] = SafeRef(s)\n    ary[2] = SafeRef(t)\n    return ary[1], length(ary)\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Note that ReturnEscape is only imposed on %2 (corresponding to SafeRef(s)) but not on %4 (corresponding to SafeRef(t)). This is because the allocated array's dimension and indices involved with all arrayref/arrayset operations are available as constant information and EscapeAnalysis can understand that %6 is aliased to %2 but never be aliased to %4. In this kind of case, the succeeding optimization passes will be able to replace Base.arrayref(true, %1, 1)::Any with %2 (a.k.a. \"load-forwarding\") and eventually eliminate the allocation of array %1 entirely (a.k.a. \"scalar-replacement\").","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"When compared to object field analysis, where an access to object field can be analyzed trivially using type information derived by inference, array dimension isn't encoded as type information and so we need an additional analysis to derive that information. EscapeAnalysis at this moment first does an additional simple linear scan to analyze dimensions of allocated arrays before firing up the main analysis routine so that the succeeding escape analysis can precisely analyze operations on those arrays.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"However, such precise \"per-element\" alias analysis is often hard. Essentially, the main difficulty inherit to array is that array dimension and index are often non-constant:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"loop often produces loop-variant, non-constant array indices\n(specific to vectors) array resizing changes array dimension and invalidates its constant-ness","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Let's discuss those difficulties with concrete examples.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"In the following example, EscapeAnalysis fails the precise alias analysis since the index at the Base.arrayset(false, %4, %8, %6)::Vector{Any} is not (trivially) constant. Especially Any[nothing, nothing] forms a loop and calls that arrayset operation in a loop, where %6 is represented as a ϕ-node value (whose value is control-flow dependent). As a result, ReturnEscape ends up imposed on both %23 (corresponding to SafeRef(s)) and %25 (corresponding to SafeRef(t)), although ideally we want it to be imposed only on %23 but not on %25:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,String)) do s, t\n    ary = Any[nothing, nothing]\n    ary[1] = SafeRef(s)\n    ary[2] = SafeRef(t)\n    return ary[1], length(ary)\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"The next example illustrates how vector resizing makes precise alias analysis hard. The essential difficulty is that the dimension of allocated array %1 is first initialized as 0, but it changes by the two :jl_array_grow_end calls afterwards. EscapeAnalysis currently simply gives up precise alias analysis whenever it encounters any array resizing operations and so ReturnEscape is imposed on both %2 (corresponding to SafeRef(s)) and %20 (corresponding to SafeRef(t)):","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"code_escapes((String,String)) do s, t\n    ary = Any[]\n    push!(ary, SafeRef(s))\n    push!(ary, SafeRef(t))\n    ary[1], length(ary)\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"In order to address these difficulties, we need inference to be aware of array dimensions and propagate array dimensions in a flow-sensitive way[ArrayDimension], as well as come up with nice representation of loop-variant values.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"EscapeAnalysis at this moment quickly switches to the more imprecise analysis that doesn't track precise index information in cases when array dimensions or indices are trivially non constant. The switch can naturally be implemented as a lattice join operation of EscapeInfo.AliasInfo property in the data-flow analysis framework.","category":"page"},{"location":"devdocs/EscapeAnalysis.html#EA-Exception-Handling","page":"EscapeAnalysis","title":"Exception Handling","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"It would be also worth noting how EscapeAnalysis handles possible escapes via exceptions. Naively it seems enough to propagate escape information imposed on :the_exception object to all values that may be thrown in a corresponding try block. But there are actually several other ways to access to the exception object in Julia, such as Base.current_exceptions and rethrow. For example, escape analysis needs to account for potential escape of r in the example below:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"const GR = Ref{Any}();\n@noinline function rethrow_escape!()\n    try\n        rethrow()\n    catch err\n        GR[] = err\n    end\nend;\nget′(x) = isassigned(x) ? x[] : throw(x);\n\ncode_escapes() do\n    r = Ref{String}()\n    local t\n    try\n        t = get′(r)\n    catch err\n        t = typeof(err)   # `err` (which `r` aliases to) doesn't escape here\n        rethrow_escape!() # but `r` escapes here\n    end\n    return t\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"It requires a global analysis in order to correctly reason about all possible escapes via existing exception interfaces. For now we always propagate the topmost escape information to all potentially thrown objects conservatively, since such an additional analysis might not be worthwhile to do given that exception handling and error path usually don't need to be very performance sensitive, and also optimizations of error paths might be very ineffective anyway since they are often even \"unoptimized\" intentionally for latency reasons.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"x::EscapeInfo's x.ThrownEscape property records SSA statements where x can be thrown as an exception. Using this information EscapeAnalysis can propagate possible escapes via exceptions limitedly to only those may be thrown in each try region:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"result = code_escapes((String,String)) do s1, s2\n    r1 = Ref(s1)\n    r2 = Ref(s2)\n    local ret\n    try\n        s1 = get′(r1)\n        ret = sizeof(s1)\n    catch err\n        global GV = err # will definitely escape `r1`\n    end\n    s2 = get′(r2)       # still `r2` doesn't escape fully\n    return s2\nend","category":"page"},{"location":"devdocs/EscapeAnalysis.html#Analysis-Usage","page":"EscapeAnalysis","title":"Analysis Usage","text":"","category":"section"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"analyze_escapes is the entry point to analyze escape information of SSA-IR elements.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Most optimizations like SROA (sroa_pass!) are more effective when applied to an optimized source that the inlining pass (ssa_inlining_pass!) has simplified by resolving inter-procedural calls and expanding callee sources. Accordingly, analyze_escapes is also able to analyze post-inlining IR and collect escape information that is useful for certain memory-related optimizations.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"However, since certain optimization passes like inlining can change control flows and eliminate dead code, they can break the inter-procedural validity of escape information. In particularity, in order to collect inter-procedurally valid escape information, we need to analyze a pre-inlining IR.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Because of this reason, analyze_escapes can analyze IRCode at any Julia-level optimization stage, and especially, it is supposed to be used at the following two stages:","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"IPO EA: analyze pre-inlining IR to generate IPO-valid escape information cache\nLocal EA: analyze post-inlining IR to collect locally-valid escape information","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Escape information derived by IPO EA is transformed to the ArgEscapeCache data structure and cached globally. By passing an appropriate get_escape_cache callback to analyze_escapes, the escape analysis can improve analysis accuracy by utilizing cached inter-procedural information of non-inlined callees that has been derived by previous IPO EA. More interestingly, it is also valid to use IPO EA escape information for type inference, e.g., inference accuracy can be improved by forming Const/PartialStruct/MustAlias of mutable object.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"Core.Compiler.EscapeAnalysis.analyze_escapes\nCore.Compiler.EscapeAnalysis.EscapeState\nCore.Compiler.EscapeAnalysis.EscapeInfo","category":"page"},{"location":"devdocs/EscapeAnalysis.html#Core.Compiler.EscapeAnalysis.analyze_escapes","page":"EscapeAnalysis","title":"Core.Compiler.EscapeAnalysis.analyze_escapes","text":"analyze_escapes(ir::IRCode, nargs::Int, get_escape_cache) -> estate::EscapeState\n\nAnalyzes escape information in ir:\n\nnargs: the number of actual arguments of the analyzed call\nget_escape_cache(::MethodInstance) -> Union{Bool,ArgEscapeCache}: retrieves cached argument escape information\n\n\n\n\n\n","category":"function"},{"location":"devdocs/EscapeAnalysis.html#Core.Compiler.EscapeAnalysis.EscapeState","page":"EscapeAnalysis","title":"Core.Compiler.EscapeAnalysis.EscapeState","text":"estate::EscapeState\n\nExtended lattice that maps arguments and SSA values to escape information represented as EscapeInfo. Escape information imposed on SSA IR element x can be retrieved by estate[x].\n\n\n\n\n\n","category":"type"},{"location":"devdocs/EscapeAnalysis.html#Core.Compiler.EscapeAnalysis.EscapeInfo","page":"EscapeAnalysis","title":"Core.Compiler.EscapeAnalysis.EscapeInfo","text":"x::EscapeInfo\n\nA lattice for escape information, which holds the following properties:\n\nx.Analyzed::Bool: not formally part of the lattice, only indicates whether x has been analyzed\nx.ReturnEscape::Bool: indicates x can escape to the caller via return\nx.ThrownEscape::BitSet: records SSA statement numbers where x can be thrown as exception:\nisempty(x.ThrownEscape): x will never be thrown in this call frame (the bottom)\npc ∈ x.ThrownEscape: x may be thrown at the SSA statement at pc\n-1 ∈ x.ThrownEscape: x may be thrown at arbitrary points of this call frame (the top)\nThis information will be used by escape_exception! to propagate potential escapes via exception.\nx.AliasInfo::Union{Bool,IndexableFields,IndexableElements,Unindexable}: maintains all possible values that can be aliased to fields or array elements of x:\nx.AliasInfo === false indicates the fields/elements of x aren't analyzed yet\nx.AliasInfo === true indicates the fields/elements of x can't be analyzed, e.g. the type of x is not known or is not concrete and thus its fields/elements can't be known precisely\nx.AliasInfo::IndexableFields records all the possible values that can be aliased to fields of object x with precise index information\nx.AliasInfo::IndexableElements records all the possible values that can be aliased to elements of array x with precise index information\nx.AliasInfo::Unindexable records all the possible values that can be aliased to fields/elements of x without precise index information\nx.Liveness::BitSet: records SSA statement numbers where x should be live, e.g. to be used as a call argument, to be returned to a caller, or preserved for :foreigncall:\nisempty(x.Liveness): x is never be used in this call frame (the bottom)\n0 ∈ x.Liveness also has the special meaning that it's a call argument of the currently analyzed call frame (and thus it's visible from the caller immediately).\npc ∈ x.Liveness: x may be used at the SSA statement at pc\n-1 ∈ x.Liveness: x may be used at arbitrary points of this call frame (the top)\n\nThere are utility constructors to create common EscapeInfos, e.g.,\n\nNoEscape(): the bottom(-like) element of this lattice, meaning it won't escape to anywhere\nAllEscape(): the topmost element of this lattice, meaning it will escape to everywhere\n\nanalyze_escapes will transition these elements from the bottom to the top, in the same direction as Julia's native type inference routine. An abstract state will be initialized with the bottom(-like) elements:\n\nthe call arguments are initialized as ArgEscape(), whose Liveness property includes 0 to indicate that it is passed as a call argument and visible from a caller immediately\nthe other states are initialized as NotAnalyzed(), which is a special lattice element that is slightly lower than NoEscape, but at the same time doesn't represent any meaning other than it's not analyzed yet (thus it's not formally part of the lattice)\n\n\n\n\n\n","category":"type"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[LatticeDesign]: Our type inference implementation takes the alternative approach, where each lattice property is represented by a special lattice element type object. It turns out that it started to complicate implementations of the lattice operations mainly because it often requires conversion rules between each lattice element type object. And we are working on overhauling our type inference lattice implementation with EscapeInfo-like lattice design.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[MM02]: A Graph-Free approach to Data-Flow Analysis.      Markas Mohnen, 2002, April.      https://api.semanticscholar.org/CorpusID:28519618.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[BackandForth]: Our type inference algorithm in contrast is implemented as a forward analysis, because type information usually flows from \"definition\" to \"usage\" and it is more natural and effective to propagate such information in a forward way.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[Dynamism]: In some cases, however, object fields can't be analyzed precisely. For example, object may escape to somewhere EscapeAnalysis can't account for possible memory effects on it, or fields of the objects simply can't be known because of the lack of type information. In such cases AliasInfo property is raised to the topmost element within its own lattice order, and it causes succeeding field analysis to be conservative and escape information imposed on fields of an unanalyzable object to be propagated to the object itself.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[JVM05]: Escape Analysis in the Context of Dynamic Compilation and Deoptimization.       Thomas Kotzmann and Hanspeter Mössenböck, 2005, June.       https://dl.acm.org/doi/10.1145/1064979.1064996.","category":"page"},{"location":"devdocs/EscapeAnalysis.html","page":"EscapeAnalysis","title":"EscapeAnalysis","text":"[ArrayDimension]: Otherwise we will need yet another forward data-flow analysis on top of the escape analysis.","category":"page"},{"location":"devdocs/debuggingtips.html#gdb-debugging-tips","page":"gdb debugging tips","title":"gdb debugging tips","text":"","category":"section"},{"location":"devdocs/debuggingtips.html#Displaying-Julia-variables","page":"gdb debugging tips","title":"Displaying Julia variables","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Within gdb, any jl_value_t* object obj can be displayed using","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) call jl_(obj)","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"The object will be displayed in the julia session, not in the gdb session. This is a useful way to discover the types and values of objects being manipulated by Julia's C code.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Similarly, if you're debugging some of Julia's internals (e.g., compiler.jl), you can print obj using","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"ccall(:jl_, Cvoid, (Any,), obj)","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"This is a good way to circumvent problems that arise from the order in which julia's output streams are initialized.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Julia's flisp interpreter uses value_t objects; these can be displayed with call fl_print(fl_ctx, ios_stdout, obj).","category":"page"},{"location":"devdocs/debuggingtips.html#Useful-Julia-variables-for-Inspecting","page":"gdb debugging tips","title":"Useful Julia variables for Inspecting","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"While the addresses of many variables, like singletons, can be useful to print for many failures, there are a number of additional variables (see julia.h for a complete list) that are even more useful.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(when in jl_apply_generic) mfunc and jl_uncompress_ast(mfunc->def, mfunc->code) :: for figuring out a bit about the call-stack\njl_lineno and jl_filename :: for figuring out what line in a test to go start debugging from (or figure out how far into a file has been parsed)\n$1 :: not really a variable, but still a useful shorthand for referring to the result of the last gdb command (such as print)\njl_options :: sometimes useful, since it lists all of the command line options that were successfully parsed\njl_uv_stderr :: because who doesn't like to be able to interact with stdio","category":"page"},{"location":"devdocs/debuggingtips.html#Useful-Julia-functions-for-Inspecting-those-variables","page":"gdb debugging tips","title":"Useful Julia functions for Inspecting those variables","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"jl_print_task_backtraces(0) :: Similar to gdb's thread apply all bt or lldb's thread backtrace all. Runs all threads while printing backtraces for all existing tasks.\njl_gdblookup($pc) :: For looking up the current function and line.\njl_gdblookupinfo($pc) :: For looking up the current method instance object.\njl_gdbdumpcode(mi) :: For dumping all of code_typed/code_llvm/code_asm when the REPL is not working right.\njlbacktrace() :: For dumping the current Julia backtrace stack to stderr. Only usable after record_backtrace() has been called.\njl_dump_llvm_value(Value*) :: For invoking Value->dump() in gdb, where it doesn't work natively. For example, f->linfo->functionObject, f->linfo->specFunctionObject, and to_function(f->linfo).\njl_dump_llvm_module(Module*) :: For invoking Module->dump() in gdb, where it doesn't work natively.\nType->dump() :: only works in lldb. Note: add something like ;1 to prevent lldb from printing its prompt over the output\njl_eval_string(\"expr\") :: for invoking side-effects to modify the current state or to lookup symbols\njl_typeof(jl_value_t*) :: for extracting the type tag of a Julia value (in gdb, call macro define jl_typeof jl_typeof first, or pick something short like ty for the first arg to define a shorthand)","category":"page"},{"location":"devdocs/debuggingtips.html#Inserting-breakpoints-for-inspection-from-gdb","page":"gdb debugging tips","title":"Inserting breakpoints for inspection from gdb","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"In your gdb session, set a breakpoint in jl_breakpoint like so:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) break jl_breakpoint","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Then within your Julia code, insert a call to jl_breakpoint by adding","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"ccall(:jl_breakpoint, Cvoid, (Any,), obj)","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"where obj can be any variable or tuple you want to be accessible in the breakpoint.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"It's particularly helpful to back up to the jl_apply frame, from which you can display the arguments to a function using, e.g.,","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) call jl_(args[0])","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Another useful frame is to_function(jl_method_instance_t *li, bool cstyle). The jl_method_instance_t* argument is a struct with a reference to the final AST sent into the compiler. However, the AST at this point will usually be compressed; to view the AST, call jl_uncompress_ast and then pass the result to jl_:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"#2  0x00007ffff7928bf7 in to_function (li=0x2812060, cstyle=false) at codegen.cpp:584\n584          abort();\n(gdb) p jl_(jl_uncompress_ast(li, li->ast))","category":"page"},{"location":"devdocs/debuggingtips.html#Inserting-breakpoints-upon-certain-conditions","page":"gdb debugging tips","title":"Inserting breakpoints upon certain conditions","text":"","category":"section"},{"location":"devdocs/debuggingtips.html#Loading-a-particular-file","page":"gdb debugging tips","title":"Loading a particular file","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Let's say the file is sysimg.jl:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) break jl_load if strcmp(fname, \"sysimg.jl\")==0","category":"page"},{"location":"devdocs/debuggingtips.html#Calling-a-particular-method","page":"gdb debugging tips","title":"Calling a particular method","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) break jl_apply_generic if strcmp((char*)(jl_symbol_name)(jl_gf_mtable(F)->name), \"method_to_break\")==0","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Since this function is used for every call, you will make everything 1000x slower if you do this.","category":"page"},{"location":"devdocs/debuggingtips.html#Dealing-with-signals","page":"gdb debugging tips","title":"Dealing with signals","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Julia requires a few signals to function properly. The profiler uses SIGUSR2 for sampling and the garbage collector uses SIGSEGV for threads synchronization. If you are debugging some code that uses the profiler or multiple threads, you may want to let the debugger ignore these signals since they can be triggered very often during normal operations. The command to do this in GDB is (replace SIGSEGV with SIGUSR2 or other signals you want to ignore):","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) handle SIGSEGV noprint nostop pass","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"The corresponding LLDB command is (after the process is started):","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(lldb) pro hand -p true -s false -n false SIGSEGV","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"If you are debugging a segfault with threaded code, you can set a breakpoint on jl_critical_error (sigdie_handler should also work on Linux and BSD) in order to only catch the actual segfault rather than the GC synchronization points.","category":"page"},{"location":"devdocs/debuggingtips.html#Debugging-during-Julia's-build-process-(bootstrap)","page":"gdb debugging tips","title":"Debugging during Julia's build process (bootstrap)","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Errors that occur during make need special handling. Julia is built in two stages, constructing sys0 and sys.ji. To see what commands are running at the time of failure, use make VERBOSE=1.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"At the time of this writing, you can debug build errors during the sys0 phase from the base directory using:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys0 sysimg.jl","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"You might need to delete all the files in usr/lib/julia/ to get this to work.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"You can debug the sys.ji phase using:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys -J ../usr/lib/julia/sys0.ji sysimg.jl","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"By default, any errors will cause Julia to exit, even under gdb. To catch an error \"in the act\", set a breakpoint in jl_error (there are several other useful spots, for specific kinds of failures, including: jl_too_few_args, jl_too_many_args, and jl_throw).","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Once an error is caught, a useful technique is to walk up the stack and examine the function by inspecting the related call to jl_apply. To take a real-world example:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Breakpoint 1, jl_throw (e=0x7ffdf42de400) at task.c:802\n802 {\n(gdb) p jl_(e)\nErrorException(\"auto_unbox: unable to determine argument type\")\n$2 = void\n(gdb) bt 10\n#0  jl_throw (e=0x7ffdf42de400) at task.c:802\n#1  0x00007ffff65412fe in jl_error (str=0x7ffde56be000 <_j_str267> \"auto_unbox:\n   unable to determine argument type\")\n   at builtins.c:39\n#2  0x00007ffde56bd01a in julia_convert_16886 ()\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n...","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"The most recent jl_apply is at frame #3, so we can go back there and look at the AST for the function julia_convert_16886. This is the uniqued name for some method of convert. f in this frame is a jl_function_t*, so we can look at the type signature, if any, from the specTypes field:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) f 3\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n1281            return f->fptr((jl_value_t*)f, args, nargs);\n(gdb) p f->linfo->specTypes\n$4 = (jl_tupletype_t *) 0x7ffdf39b1030\n(gdb) p jl_( f->linfo->specTypes )\nTuple{Type{Float32}, Float64}           # <-- type signature for julia_convert_16886","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Then, we can look at the AST for this function:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) p jl_( jl_uncompress_ast(f->linfo, f->linfo->ast) )\nExpr(:lambda, Array{Any, 1}[:#s29, :x], Array{Any, 1}[Array{Any, 1}[], Array{Any, 1}[Array{Any, 1}[:#s29, :Any, 0], Array{Any, 1}[:x, :Any, 0]], Array{Any, 1}[], 0], Expr(:body,\nExpr(:line, 90, :float.jl)::Any,\nExpr(:return, Expr(:call, :box, :Float32, Expr(:call, :fptrunc, :Float32, :x)::Any)::Any)::Any)::Any)::Any","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Finally, and perhaps most usefully, we can force the function to be recompiled in order to step through the codegen process. To do this, clear the cached functionObject from the jl_lamdbda_info_t*:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) p f->linfo->functionObject\n$8 = (void *) 0x1289d070\n(gdb) set f->linfo->functionObject = NULL","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Then, set a breakpoint somewhere useful (e.g. emit_function, emit_expr, emit_call, etc.), and run codegen:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) p jl_compile(f)\n... # your breakpoint here","category":"page"},{"location":"devdocs/debuggingtips.html#Debugging-precompilation-errors","page":"gdb debugging tips","title":"Debugging precompilation errors","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Module precompilation spawns a separate Julia process to precompile each module. Setting a breakpoint or catching failures in a precompile worker requires attaching a debugger to the worker. The easiest approach is to set the debugger watch for new process launches matching a given name. For example:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(gdb) attach -w -n julia-debug","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"or:","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"(lldb) process attach -w -n julia-debug","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Then run a script/command to start precompilation. As described earlier, use conditional breakpoints in the parent process to catch specific file-loading events and narrow the debugging window. (some operating systems may require alternative approaches, such as following each fork from the parent process)","category":"page"},{"location":"devdocs/debuggingtips.html#Mozilla's-Record-and-Replay-Framework-(rr)","page":"gdb debugging tips","title":"Mozilla's Record and Replay Framework (rr)","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"Julia now works out of the box with rr, the lightweight recording and deterministic debugging framework from Mozilla. This allows you to replay the trace of an execution deterministically.  The replayed execution's address spaces, register contents, syscall data etc are exactly the same in every run.","category":"page"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"A recent version of rr (3.1.0 or higher) is required.","category":"page"},{"location":"devdocs/debuggingtips.html#Reproducing-concurrency-bugs-with-rr","page":"gdb debugging tips","title":"Reproducing concurrency bugs with rr","text":"","category":"section"},{"location":"devdocs/debuggingtips.html","page":"gdb debugging tips","title":"gdb debugging tips","text":"rr simulates a single-threaded machine by default. In order to debug concurrent code you can use rr record --chaos which will cause rr to simulate between one to eight cores, chosen randomly. You might therefore want to set JULIA_NUM_THREADS=8 and rerun your code under rr until you have caught your bug.","category":"page"},{"location":"stdlib/LibCURL.html#LibCURL","page":"LibCURL","title":"LibCURL","text":"","category":"section"},{"location":"stdlib/LibCURL.html","page":"LibCURL","title":"LibCURL","text":"This is a simple Julia wrapper around http://curl.haxx.se/libcurl/ generated using Clang.jl.  Please see the libcurl API documentation for help on how to use this package.","category":"page"},{"location":"devdocs/stdio.html#printf()-and-stdio-in-the-Julia-runtime","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"","category":"section"},{"location":"devdocs/stdio.html#Libuv-wrappers-for-stdio","page":"printf() and stdio in the Julia runtime","title":"Libuv wrappers for stdio","text":"","category":"section"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"julia.h defines libuv wrappers for the stdio.h streams:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"uv_stream_t *JL_STDIN;\nuv_stream_t *JL_STDOUT;\nuv_stream_t *JL_STDERR;","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"... and corresponding output functions:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"int jl_printf(uv_stream_t *s, const char *format, ...);\nint jl_vprintf(uv_stream_t *s, const char *format, va_list args);","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"These printf functions are used by the .c files in the src/ and cli/ directories wherever stdio is needed to ensure that output buffering is handled in a unified way.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"In special cases, like signal handlers, where the full libuv infrastructure is too heavy, jl_safe_printf() can be used to write(2) directly to STDERR_FILENO:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"void jl_safe_printf(const char *str, ...);","category":"page"},{"location":"devdocs/stdio.html#Interface-between-JL_STD*-and-Julia-code","page":"printf() and stdio in the Julia runtime","title":"Interface between JL_STD* and Julia code","text":"","category":"section"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"Base.stdin, Base.stdout and Base.stderr are bound to the JL_STD* libuv streams defined in the runtime.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"Julia's __init__() function (in base/sysimg.jl) calls reinit_stdio() (in base/stream.jl) to create Julia objects for Base.stdin, Base.stdout and Base.stderr.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"reinit_stdio() uses ccall to retrieve pointers to JL_STD* and calls jl_uv_handle_type() to inspect the type of each stream.  It then creates a Julia Base.IOStream, Base.TTY or Base.PipeEndpoint object to represent each stream, e.g.:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"$ julia -e 'println(typeof((stdin, stdout, stderr)))'\nTuple{Base.TTY,Base.TTY,Base.TTY}\n\n$ julia -e 'println(typeof((stdin, stdout, stderr)))' < /dev/null 2>/dev/null\nTuple{IOStream,Base.TTY,IOStream}\n\n$ echo hello | julia -e 'println(typeof((stdin, stdout, stderr)))' | cat\nTuple{Base.PipeEndpoint,Base.PipeEndpoint,Base.TTY}","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"The Base.read and Base.write methods for these streams use ccall to call libuv wrappers in src/jl_uv.c, e.g.:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"stream.jl: function write(s::IO, p::Ptr, nb::Integer)\n               -> ccall(:jl_uv_write, ...)\n  jl_uv.c:          -> int jl_uv_write(uv_stream_t *stream, ...)\n                        -> uv_write(uvw, stream, buf, ...)","category":"page"},{"location":"devdocs/stdio.html#printf()-during-initialization","page":"printf() and stdio in the Julia runtime","title":"printf() during initialization","text":"","category":"section"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"The libuv streams relied upon by jl_printf() etc., are not available until midway through initialization of the runtime (see init.c, init_stdio()).  Error messages or warnings that need to be printed before this are routed to the standard C library fwrite() function by the following mechanism:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"In sys.c, the JL_STD* stream pointers are statically initialized to integer constants: STD*_FILENO (0, 1 and 2). In jl_uv.c the jl_uv_puts() function checks its uv_stream_t* stream argument and calls fwrite() if stream is set to STDOUT_FILENO or STDERR_FILENO.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"This allows for uniform use of jl_printf() throughout the runtime regardless of whether or not any particular piece of code is reachable before initialization is complete.","category":"page"},{"location":"devdocs/stdio.html#Legacy-ios.c-library","page":"printf() and stdio in the Julia runtime","title":"Legacy ios.c library","text":"","category":"section"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"The src/support/ios.c library is inherited from femtolisp. It provides cross-platform buffered file IO and in-memory temporary buffers.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"ios.c is still used by:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"src/flisp/*.c\nsrc/dump.c – for serialization file IO and for memory buffers.\nsrc/staticdata.c – for serialization file IO and for memory buffers.\nbase/iostream.jl – for file IO (see base/fs.jl for libuv equivalent).","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"Use of ios.c in these modules is mostly self-contained and separated from the libuv I/O system. However, there is one place where femtolisp calls through to jl_printf() with a legacy ios_t stream.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"There is a hack in ios.h that makes the ios_t.bm field line up with the uv_stream_t.type and ensures that the values used for ios_t.bm to not overlap with valid UV_HANDLE_TYPE values.  This allows uv_stream_t pointers to point to ios_t streams.","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"This is needed because jl_printf() caller jl_static_show() is passed an ios_t stream by femtolisp's fl_print() function. Julia's jl_uv_puts() function has special handling for this:","category":"page"},{"location":"devdocs/stdio.html","page":"printf() and stdio in the Julia runtime","title":"printf() and stdio in the Julia runtime","text":"if (stream->type > UV_HANDLE_TYPE_MAX) {\n    return ios_write((ios_t*)stream, str, n);\n}","category":"page"},{"location":"base/iterators.html#Iteration-utilities","page":"Iteration utilities","title":"Iteration utilities","text":"","category":"section"},{"location":"base/iterators.html","page":"Iteration utilities","title":"Iteration utilities","text":"Base.Iterators.Stateful\nBase.Iterators.zip\nBase.Iterators.enumerate\nBase.Iterators.rest\nBase.Iterators.countfrom\nBase.Iterators.take\nBase.Iterators.takewhile\nBase.Iterators.drop\nBase.Iterators.dropwhile\nBase.Iterators.cycle\nBase.Iterators.repeated\nBase.Iterators.product\nBase.Iterators.flatten\nBase.Iterators.flatmap\nBase.Iterators.partition\nBase.Iterators.map\nBase.Iterators.filter\nBase.Iterators.accumulate\nBase.Iterators.reverse\nBase.Iterators.only\nBase.Iterators.peel","category":"page"},{"location":"base/iterators.html#Base.Iterators.Stateful","page":"Iteration utilities","title":"Base.Iterators.Stateful","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n","category":"type"},{"location":"base/iterators.html#Base.Iterators.zip","page":"Iteration utilities","title":"Base.Iterators.zip","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.enumerate","page":"Iteration utilities","title":"Base.Iterators.enumerate","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a different element. This will happen if iter has indices that do not start at 1, and may happen for strings, dictionaries, etc. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.rest","page":"Iteration utilities","title":"Base.Iterators.rest","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state.\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> collect(Iterators.rest([1,2,3,4], 2))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.countfrom","page":"Iteration utilities","title":"Base.Iterators.countfrom","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.take","page":"Iteration utilities","title":"Base.Iterators.take","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.takewhile","page":"Iteration utilities","title":"Base.Iterators.takewhile","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.drop","page":"Iteration utilities","title":"Base.Iterators.drop","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.dropwhile","page":"Iteration utilities","title":"Base.Iterators.dropwhile","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.cycle","page":"Iteration utilities","title":"Base.Iterators.cycle","text":"cycle(iter[, n::Int])\n\nAn iterator that cycles through iter forever. If n is specified, then it cycles through iter that many times. When iter is empty, so are cycle(iter) and cycle(iter, n).\n\nIterators.cycle(iter, n) is the lazy equivalent of Base.repeat(vector, n), while Iterators.repeated(iter, n) is the lazy Base.fill(item, n).\n\ncompat: Julia 1.11\nThe method cycle(iter, n) was added in Julia 1.11.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\njulia> foreach(print, Iterators.cycle(['j', 'u', 'l', 'i', 'a'], 3))\njuliajuliajulia\n\njulia> repeat([1,2,3], 4) == collect(Iterators.cycle([1,2,3], 4))\ntrue\n\njulia> fill([1,2,3], 4) == collect(Iterators.repeated([1,2,3], 4))\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.repeated","page":"Iteration utilities","title":"Base.Iterators.repeated","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nSee also fill, and compare Iterators.cycle.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\njulia> ans == fill([1 2], 4)\ntrue\n\njulia> Iterators.cycle([1 2], 4) |> collect |> println\n[1, 2, 1, 2, 1, 2, 1, 2]\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.product","page":"Iteration utilities","title":"Base.Iterators.product","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.flatten","page":"Iteration utilities","title":"Base.Iterators.flatten","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.flatmap","page":"Iteration utilities","title":"Base.Iterators.flatmap","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.partition","page":"Iteration utilities","title":"Base.Iterators.partition","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.map","page":"Iteration utilities","title":"Base.Iterators.map","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.filter","page":"Iteration utilities","title":"Base.Iterators.filter","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nwarning: Warning\nSubsequent lazy transformations on the iterator returned from filter, such as those performed by Iterators.reverse or cycle, will also delay calls to flt until collecting or iterating over the returned iterable object. If the filter predicate is nondeterministic or its return values depend on the order of iteration over the elements of itr, composition with lazy transformations may result in surprising behavior. If this is undesirable, either ensure that flt is a pure function or collect intermediate filter iterators before further transformations.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.accumulate","page":"Iteration utilities","title":"Base.Iterators.accumulate","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.reverse","page":"Iteration utilities","title":"Base.Iterators.reverse","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order. This iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns an Iterators.Reverse object wrapping itr, which is iterable if the corresponding iterate methods are defined, but some itr types may implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr; more generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.only","page":"Iteration utilities","title":"Base.Iterators.only","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the collection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n","category":"function"},{"location":"base/iterators.html#Base.Iterators.peel","page":"Iteration utilities","title":"Base.Iterators.peel","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates","page":"Dates","title":"Dates","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"DocTestSetup = :(using Dates)","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The Dates module provides two types for working with dates: Date and DateTime, representing day and millisecond precision, respectively; both are subtypes of the abstract TimeType. The motivation for distinct types is simple: some operations are much simpler, both in terms of code and mental reasoning, when the complexities of greater precision don't have to be dealt with. For example, since the Date type only resolves to the precision of a single date (i.e. no hours, minutes, or seconds), normal considerations for time zones, daylight savings/summer time, and leap seconds are unnecessary and avoided.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Both Date and DateTime are basically immutable Int64 wrappers. The single instant field of either type is actually a UTInstant{P} type, which represents a continuously increasing machine timeline based on the UT second [1]. The DateTime type is not aware of time zones (naive, in Python parlance), analogous to a LocalDateTime in Java 8. Additional time zone functionality can be added through the TimeZones.jl package, which compiles the IANA time zone database. Both Date and DateTime are based on the ISO 8601 standard, which follows the proleptic Gregorian calendar. One note is that the ISO 8601 standard is particular about BC/BCE dates. In general, the last day of the BC/BCE era, 1-12-31 BC/BCE, was followed by 1-1-1 AD/CE, thus no year zero exists. The ISO standard, however, states that 1 BC/BCE is year zero, so 0000-12-31 is the day before 0001-01-01, and year -0001 (yes, negative one for the year) is 2 BC/BCE, year -0002 is 3 BC/BCE, etc.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"[1]: The notion of the UT second is actually quite fundamental. There are basically two different notions of time generally accepted, one based on the physical rotation of the earth (one full rotation = 1 day), the other based on the SI second (a fixed, constant value). These are radically different! Think about it, a \"UT second\", as defined relative to the rotation of the earth, may have a different absolute length depending on the day! Anyway, the fact that Date and DateTime are based on UT seconds is a simplifying, yet honest assumption so that things like leap seconds and all their complexity can be avoided. This basis of time is formally called UT or UT1. Basing types on the UT second basically means that every minute has 60 seconds and every day has 24 hours and leads to more natural calculations when working with calendar dates.","category":"page"},{"location":"stdlib/Dates.html#Constructors","page":"Dates","title":"Constructors","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Date and DateTime types can be constructed by integer or Period types, by parsing, or through adjusters (more on those later):","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> DateTime(2013)\n2013-01-01T00:00:00\n\njulia> DateTime(2013,7)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1,12)\n2013-07-01T12:00:00\n\njulia> DateTime(2013,7,1,12,30)\n2013-07-01T12:30:00\n\njulia> DateTime(2013,7,1,12,30,59)\n2013-07-01T12:30:59\n\njulia> DateTime(2013,7,1,12,30,59,1)\n2013-07-01T12:30:59.001\n\njulia> Date(2013)\n2013-01-01\n\njulia> Date(2013,7)\n2013-07-01\n\njulia> Date(2013,7,1)\n2013-07-01\n\njulia> Date(Dates.Year(2013),Dates.Month(7),Dates.Day(1))\n2013-07-01\n\njulia> Date(Dates.Month(7),Dates.Year(2013))\n2013-07-01","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Date or DateTime parsing is accomplished by the use of format strings. Format strings work by the notion of defining delimited or fixed-width \"slots\" that contain a period to parse and passing the text to parse and format string to a Date or DateTime constructor, of the form Date(\"2015-01-01\",dateformat\"y-m-d\") or DateTime(\"20150101\",dateformat\"yyyymmdd\").","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Delimited slots are marked by specifying the delimiter the parser should expect between two subsequent periods; so \"y-m-d\" lets the parser know that between the first and second slots in a date string like \"2014-07-16\", it should find the - character. The y, m, and d characters let the parser know which periods to parse in each slot.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As in the case of constructors above such as Date(2013), delimited DateFormats allow for missing parts of dates and times so long as the preceding parts are given. The other parts are given the usual default values.  For example, Date(\"1981-03\", dateformat\"y-m-d\") returns 1981-03-01, whilst Date(\"31/12\", dateformat\"d/m/y\") gives 0001-12-31.  (Note that the default year is 1 AD/CE.) An empty string, however, always throws an ArgumentError.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Fixed-width slots are specified by repeating the period character the number of times corresponding to the width with no delimiter between characters. So dateformat\"yyyymmdd\" would correspond to a date string like \"20140716\". The parser distinguishes a fixed-width slot by the absence of a delimiter, noting the transition \"yyyymm\" from one period character to the next.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Support for text-form month parsing is also supported through the u and U characters, for abbreviated and full-length month names, respectively. By default, only English month names are supported, so u corresponds to \"Jan\", \"Feb\", \"Mar\", etc. And U corresponds to \"January\", \"February\", \"March\", etc. Similar to other name=>value mapping functions dayname and monthname, custom locales can be loaded by passing in the locale=>Dict{String,Int} mapping to the MONTHTOVALUEABBR and MONTHTOVALUE dicts for abbreviated and full-name month names, respectively.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The above examples used the dateformat\"\" string macro. This macro creates a DateFormat object once when the macro is expanded and uses the same DateFormat object even if a code snippet is run multiple times.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> for i = 1:10^5\n           Date(\"2015-01-01\", dateformat\"y-m-d\")\n       end","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Or you can create the DateFormat object explicitly:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> df = DateFormat(\"y-m-d\");\n\njulia> dt = Date(\"2015-01-01\",df)\n2015-01-01\n\njulia> dt2 = Date(\"2015-01-02\",df)\n2015-01-02","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Alternatively, use broadcasting:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> years = [\"2015\", \"2016\"];\n\njulia> Date.(years, DateFormat(\"yyyy\"))\n2-element Vector{Date}:\n 2015-01-01\n 2016-01-01","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"For convenience, you may pass the format string directly (e.g., Date(\"2015-01-01\",\"y-m-d\")), although this form incurs performance costs if you are parsing the same format repeatedly, as it internally creates a new DateFormat object each time.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As well as via the constructors, a Date or DateTime can be constructed from strings using the parse and tryparse functions, but with an optional third argument of type DateFormat specifying the format; for example, parse(Date, \"06.23.2013\", dateformat\"m.d.y\"), or tryparse(DateTime, \"1999-12-31T23:59:59\") which uses the default format. The notable difference between the functions is that with tryparse, an error is not thrown if the string is empty or in an invalid format; instead nothing is returned.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"compat: Julia 1.9\nBefore Julia 1.9, empty strings could be passed to constructors and parse without error, returning as appropriate DateTime(1), Date(1) or Time(0). Likewise, tryparse did not return nothing.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"A full suite of parsing and formatting tests and examples is available in stdlib/Dates/test/io.jl.","category":"page"},{"location":"stdlib/Dates.html#Durations/Comparisons","page":"Dates","title":"Durations/Comparisons","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Finding the length of time between two Date or DateTime is straightforward given their underlying representation as UTInstant{Day} and UTInstant{Millisecond}, respectively. The difference between Date is returned in the number of Day, and DateTime in the number of Millisecond. Similarly, comparing TimeType is a simple matter of comparing the underlying machine instants (which in turn compares the internal Int64 values).","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> dt = Date(2012,2,29)\n2012-02-29\n\njulia> dt2 = Date(2000,2,1)\n2000-02-01\n\njulia> dump(dt)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 734562\n\njulia> dump(dt2)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 730151\n\njulia> dt > dt2\ntrue\n\njulia> dt != dt2\ntrue\n\njulia> dt + dt2\nERROR: MethodError: no method matching +(::Date, ::Date)\n[...]\n\njulia> dt * dt2\nERROR: MethodError: no method matching *(::Date, ::Date)\n[...]\n\njulia> dt / dt2\nERROR: MethodError: no method matching /(::Date, ::Date)\n\njulia> dt - dt2\n4411 days\n\njulia> dt2 - dt\n-4411 days\n\njulia> dt = DateTime(2012,2,29)\n2012-02-29T00:00:00\n\njulia> dt2 = DateTime(2000,2,1)\n2000-02-01T00:00:00\n\njulia> dt - dt2\n381110400000 milliseconds","category":"page"},{"location":"stdlib/Dates.html#Accessor-Functions","page":"Dates","title":"Accessor Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Because the Date and DateTime types are stored as single Int64 values, date parts or fields can be retrieved through accessor functions. The lowercase accessors return the field as an integer:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.year(t)\n2014\n\njulia> Dates.month(t)\n1\n\njulia> Dates.week(t)\n5\n\njulia> Dates.day(t)\n31","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"While propercase return the same value in the corresponding Period type:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.Year(t)\n2014 years\n\njulia> Dates.Day(t)\n31 days","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Compound methods are provided because it is more efficient to access multiple fields at the same time than individually:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.yearmonth(t)\n(2014, 1)\n\njulia> Dates.monthday(t)\n(1, 31)\n\njulia> Dates.yearmonthday(t)\n(2014, 1, 31)","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"One may also access the underlying UTInstant or integer value:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> dump(t)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 735264\n\njulia> t.instant\nDates.UTInstant{Day}(Day(735264))\n\njulia> Dates.value(t)\n735264","category":"page"},{"location":"stdlib/Dates.html#Query-Functions","page":"Dates","title":"Query Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Query functions provide calendrical information about a TimeType. They include information about the day of the week:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.dayofweek(t)\n5\n\njulia> Dates.dayname(t)\n\"Friday\"\n\njulia> Dates.dayofweekofmonth(t) # 5th Friday of January\n5","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Month of the year:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.monthname(t)\n\"January\"\n\njulia> Dates.daysinmonth(t)\n31","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As well as information about the TimeType's year and quarter:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.isleapyear(t)\nfalse\n\njulia> Dates.dayofyear(t)\n31\n\njulia> Dates.quarterofyear(t)\n1\n\njulia> Dates.dayofquarter(t)\n31","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The dayname and monthname methods can also take an optional locale keyword that can be used to return the name of the day or month of the year for other languages/locales. There are also versions of these functions returning the abbreviated names, namely dayabbr and monthabbr. First the mapping is loaded into the LOCALES variable:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> french_months = [\"janvier\", \"février\", \"mars\", \"avril\", \"mai\", \"juin\",\n                        \"juillet\", \"août\", \"septembre\", \"octobre\", \"novembre\", \"décembre\"];\n\njulia> french_months_abbrev = [\"janv\",\"févr\",\"mars\",\"avril\",\"mai\",\"juin\",\n                              \"juil\",\"août\",\"sept\",\"oct\",\"nov\",\"déc\"];\n\njulia> french_days = [\"lundi\",\"mardi\",\"mercredi\",\"jeudi\",\"vendredi\",\"samedi\",\"dimanche\"];\n\njulia> Dates.LOCALES[\"french\"] = Dates.DateLocale(french_months, french_months_abbrev, french_days, [\"\"]);","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The above mentioned functions can then be used to perform the queries:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.dayname(t;locale=\"french\")\n\"vendredi\"\n\njulia> Dates.monthname(t;locale=\"french\")\n\"janvier\"\n\njulia> Dates.monthabbr(t;locale=\"french\")\n\"janv\"","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Since the abbreviated versions of the days are not loaded, trying to use the function dayabbr will throw an error.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.dayabbr(t;locale=\"french\")\nERROR: BoundsError: attempt to access 1-element Vector{String} at index [5]\nStacktrace:\n[...]","category":"page"},{"location":"stdlib/Dates.html#TimeType-Period-Arithmetic","page":"Dates","title":"TimeType-Period Arithmetic","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"It's good practice when using any language/date framework to be familiar with how date-period arithmetic is handled as there are some tricky issues to deal with (though much less so for day-precision types).","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The Dates module approach tries to follow the simple principle of trying to change as little as possible when doing Period arithmetic. This approach is also often known as calendrical arithmetic or what you would probably guess if someone were to ask you the same calculation in a conversation. Why all the fuss about this? Let's take a classic example: add 1 month to January 31st, 2014. What's the answer? Javascript will say March 3 (assumes 31 days). PHP says March 2 (assumes 30 days). The fact is, there is no right answer. In the Dates module, it gives the result of February 28th. How does it figure that out? Consider the classic 7-7-7 gambling game in casinos.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Now just imagine that instead of 7-7-7, the slots are Year-Month-Day, or in our example, 2014-01-31. When you ask to add 1 month to this date, the month slot is incremented, so now we have 2014-02-31. Then the day number is checked if it is greater than the last valid day of the new month; if it is (as in the case above), the day number is adjusted down to the last valid day (28). What are the ramifications with this approach? Go ahead and add another month to our date, 2014-02-28 + Month(1) == 2014-03-28. What? Were you expecting the last day of March? Nope, sorry, remember the 7-7-7 slots. As few slots as possible are going to change, so we first increment the month slot by 1, 2014-03-28, and boom, we're done because that's a valid date. On the other hand, if we were to add 2 months to our original date, 2014-01-31, then we end up with 2014-03-31, as expected. The other ramification of this approach is a loss in associativity when a specific ordering is forced (i.e. adding things in different orders results in different outcomes). For example:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> (Date(2014,1,29)+Dates.Day(1)) + Dates.Month(1)\n2014-02-28\n\njulia> (Date(2014,1,29)+Dates.Month(1)) + Dates.Day(1)\n2014-03-01","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"What's going on there? In the first line, we're adding 1 day to January 29th, which results in 2014-01-30; then we add 1 month, so we get 2014-02-30, which then adjusts down to 2014-02-28. In the second example, we add 1 month first, where we get 2014-02-29, which adjusts down to 2014-02-28, and then add 1 day, which results in 2014-03-01. One design principle that helps in this case is that, in the presence of multiple Periods, the operations will be ordered by the Periods' types, not their value or positional order; this means Year will always be added first, then Month, then Week, etc. Hence the following does result in associativity and Just Works:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Date(2014,1,29) + Dates.Day(1) + Dates.Month(1)\n2014-03-01\n\njulia> Date(2014,1,29) + Dates.Month(1) + Dates.Day(1)\n2014-03-01","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Tricky? Perhaps. What is an innocent Dates user to do? The bottom line is to be aware that explicitly forcing a certain associativity, when dealing with months, may lead to some unexpected results, but otherwise, everything should work as expected. Thankfully, that's pretty much the extent of the odd cases in date-period arithmetic when dealing with time in UT (avoiding the \"joys\" of dealing with daylight savings, leap seconds, etc.).","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As a bonus, all period arithmetic objects work directly with ranges:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> dr = Date(2014,1,29):Day(1):Date(2014,2,3)\nDate(\"2014-01-29\"):Day(1):Date(\"2014-02-03\")\n\njulia> collect(dr)\n6-element Vector{Date}:\n 2014-01-29\n 2014-01-30\n 2014-01-31\n 2014-02-01\n 2014-02-02\n 2014-02-03\n\njulia> dr = Date(2014,1,29):Dates.Month(1):Date(2014,07,29)\nDate(\"2014-01-29\"):Month(1):Date(\"2014-07-29\")\n\njulia> collect(dr)\n7-element Vector{Date}:\n 2014-01-29\n 2014-02-28\n 2014-03-29\n 2014-04-29\n 2014-05-29\n 2014-06-29\n 2014-07-29","category":"page"},{"location":"stdlib/Dates.html#Adjuster-Functions","page":"Dates","title":"Adjuster Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As convenient as date-period arithmetic is, often the kinds of calculations needed on dates take on a calendrical or temporal nature rather than a fixed number of periods. Holidays are a perfect example; most follow rules such as \"Memorial Day = Last Monday of May\", or \"Thanksgiving = 4th Thursday of November\". These kinds of temporal expressions deal with rules relative to the calendar, like first or last of the month, next Tuesday, or the first and third Wednesdays, etc.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The Dates module provides the adjuster API through several convenient methods that aid in simply and succinctly expressing temporal rules. The first group of adjuster methods deal with the first and last of weeks, months, quarters, and years. They each take a single TimeType as input and return or adjust to the first or last of the desired period relative to the input.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.firstdayofweek(Date(2014,7,16)) # Adjusts the input to the Monday of the input's week\n2014-07-14\n\njulia> Dates.lastdayofmonth(Date(2014,7,16)) # Adjusts to the last day of the input's month\n2014-07-31\n\njulia> Dates.lastdayofquarter(Date(2014,7,16)) # Adjusts to the last day of the input's quarter\n2014-09-30","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The next two higher-order methods, tonext, and toprev, generalize working with temporal expressions by taking a DateFunction as first argument, along with a starting TimeType. A DateFunction is just a function, usually anonymous, that takes a single TimeType as input and returns a Bool, true indicating a satisfied adjustment criterion. For example:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> istuesday = x->Dates.dayofweek(x) == Dates.Tuesday; # Returns true if the day of the week of x is Tuesday\n\njulia> Dates.tonext(istuesday, Date(2014,7,13)) # 2014-07-13 is a Sunday\n2014-07-15\n\njulia> Dates.tonext(Date(2014,7,13), Dates.Tuesday) # Convenience method provided for day of the week adjustments\n2014-07-15","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"This is useful with the do-block syntax for more complex temporal expressions:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> Dates.tonext(Date(2014,7,13)) do x\n           # Return true on the 4th Thursday of November (Thanksgiving)\n           Dates.dayofweek(x) == Dates.Thursday &&\n           Dates.dayofweekofmonth(x) == 4 &&\n           Dates.month(x) == Dates.November\n       end\n2014-11-27","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The Base.filter method can be used to obtain all valid dates/moments in a specified range:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"# Pittsburgh street cleaning; Every 2nd Tuesday from April to November\n# Date range from January 1st, 2014 to January 1st, 2015\njulia> dr = Dates.Date(2014):Day(1):Dates.Date(2015);\n\njulia> filter(dr) do x\n           Dates.dayofweek(x) == Dates.Tue &&\n           Dates.April <= Dates.month(x) <= Dates.Nov &&\n           Dates.dayofweekofmonth(x) == 2\n       end\n8-element Vector{Date}:\n 2014-04-08\n 2014-05-13\n 2014-06-10\n 2014-07-08\n 2014-08-12\n 2014-09-09\n 2014-10-14\n 2014-11-11","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Additional examples and tests are available in stdlib/Dates/test/adjusters.jl.","category":"page"},{"location":"stdlib/Dates.html#Period-Types","page":"Dates","title":"Period Types","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Periods are a human view of discrete, sometimes irregular durations of time. Consider 1 month; it could represent, in days, a value of 28, 29, 30, or 31 depending on the year and month context. Or a year could represent 365 or 366 days in the case of a leap year. Period types are simple Int64 wrappers and are constructed by wrapping any Int64 convertible type, i.e. Year(1) or Month(3.0). Arithmetic between Period of the same type behave like integers, and limited Period-Real arithmetic is available.  You can extract the underlying integer with Dates.value.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> y1 = Dates.Year(1)\n1 year\n\njulia> y2 = Dates.Year(2)\n2 years\n\njulia> y3 = Dates.Year(10)\n10 years\n\njulia> y1 + y2\n3 years\n\njulia> div(y3,y2)\n5\n\njulia> y3 - y2\n8 years\n\njulia> y3 % y2\n0 years\n\njulia> div(y3,3) # mirrors integer division\n3 years\n\njulia> Dates.value(Dates.Millisecond(10))\n10","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Representing periods or durations that are not integer multiples of the basic types can be achieved with the Dates.CompoundPeriod type. Compound periods may be constructed manually from simple Period types. Additionally, the canonicalize function can be used to break down a period into a Dates.CompoundPeriod. This is particularly useful to convert a duration, e.g., a difference of two DateTime, into a more convenient representation.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> cp = Dates.CompoundPeriod(Day(1),Minute(1))\n1 day, 1 minute\n\njulia> t1 = DateTime(2018,8,8,16,58,00)\n2018-08-08T16:58:00\n\njulia> t2 = DateTime(2021,6,23,10,00,00)\n2021-06-23T10:00:00\n\njulia> canonicalize(t2-t1) # creates a CompoundPeriod\n149 weeks, 6 days, 17 hours, 2 minutes","category":"page"},{"location":"stdlib/Dates.html#Rounding","page":"Dates","title":"Rounding","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> floor(Date(1985, 8, 16), Dates.Month)\n1985-08-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Dates.Minute(15))\n2013-02-13T00:45:00\n\njulia> round(DateTime(2016, 8, 6, 20, 15), Dates.Day)\n2016-08-07T00:00:00","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Unlike the numeric round method, which breaks ties toward the even number by default, the TimeTyperound method uses the RoundNearestTiesUp rounding mode. (It's difficult to guess what breaking ties to nearest \"even\" TimeType would entail.) Further details on the available RoundingMode s can be found in the API reference.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Rounding should generally behave as expected, but there are a few cases in which the expected behaviour is not obvious.","category":"page"},{"location":"stdlib/Dates.html#Rounding-Epoch","page":"Dates","title":"Rounding Epoch","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"In many cases, the resolution specified for rounding (e.g., Dates.Second(30)) divides evenly into the next largest period (in this case, Dates.Minute(1)). But rounding behaviour in cases in which this is not true may lead to confusion. What is the expected result of rounding a DateTime to the nearest 10 hours?","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> round(DateTime(2016, 7, 17, 11, 55), Dates.Hour(10))\n2016-07-17T12:00:00","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"That may seem confusing, given that the hour (12) is not divisible by 10. The reason that 2016-07-17T12:00:00 was chosen is that it is 17,676,660 hours after 0000-01-01T00:00:00, and 17,676,660 is divisible by 10.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"As Julia Date and DateTime values are represented according to the ISO 8601 standard, 0000-01-01T00:00:00 was chosen as base (or \"rounding epoch\") from which to begin the count of days (and milliseconds) used in rounding calculations. (Note that this differs slightly from Julia's internal representation of Date s using Rata Die notation; but since the ISO 8601 standard is most visible to the end user, 0000-01-01T00:00:00 was chosen as the rounding epoch instead of the 0000-12-31T00:00:00 used internally to minimize confusion.)","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The only exception to the use of 0000-01-01T00:00:00 as the rounding epoch is when rounding to weeks. Rounding to the nearest week will always return a Monday (the first day of the week as specified by ISO 8601). For this reason, we use 0000-01-03T00:00:00 (the first day of the first week of year 0000, as defined by ISO 8601) as the base when rounding to a number of weeks.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Here is a related case in which the expected behaviour is not necessarily obvious: What happens when we round to the nearest P(2), where P is a Period type? In some cases (specifically, when P <: Dates.TimePeriod) the answer is clear:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Hour(2))\n2016-07-17T08:00:00\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Minute(2))\n2016-07-17T08:56:00","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"This seems obvious, because two of each of these periods still divides evenly into the next larger order period. But in the case of two months (which still divides evenly into one year), the answer may be surprising:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"julia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Month(2))\n2016-07-01T00:00:00","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Why round to the first day in July, even though it is month 7 (an odd number)? The key is that months are 1-indexed (the first month is assigned 1), unlike hours, minutes, seconds, and milliseconds (the first of which are assigned 0).","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"This means that rounding a DateTime to an even multiple of seconds, minutes, hours, or years (because the ISO 8601 specification includes a year zero) will result in a DateTime with an even value in that field, while rounding a DateTime to an even multiple of months will result in the months field having an odd value. Because both months and years may contain an irregular number of days, whether rounding to an even number of days will result in an even value in the days field is uncertain.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"See the API reference for additional information on methods exported from the Dates module.","category":"page"},{"location":"stdlib/Dates.html#stdlib-dates-api","page":"Dates","title":"API reference","text":"","category":"section"},{"location":"stdlib/Dates.html#Dates-and-Time-Types","page":"Dates","title":"Dates and Time Types","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.Period\nDates.CompoundPeriod\nDates.Instant\nDates.UTInstant\nDates.TimeType\nDates.DateTime\nDates.Date\nDates.Time\nDates.TimeZone\nDates.UTC","category":"page"},{"location":"stdlib/Dates.html#Dates.Period","page":"Dates","title":"Dates.Period","text":"Period\nYear\nQuarter\nMonth\nWeek\nDay\nHour\nMinute\nSecond\nMillisecond\nMicrosecond\nNanosecond\n\nPeriod types represent discrete, human representations of time.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.CompoundPeriod","page":"Dates","title":"Dates.CompoundPeriod","text":"CompoundPeriod\n\nA CompoundPeriod is useful for expressing time periods that are not a fixed multiple of smaller periods. For example, \"a year and a  day\" is not a fixed number of days, but can be expressed using a CompoundPeriod. In fact, a CompoundPeriod is automatically generated by addition of different period types, e.g. Year(1) + Day(1) produces a CompoundPeriod result.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.Instant","page":"Dates","title":"Dates.Instant","text":"Instant\n\nInstant types represent integer-based, machine representations of time as continuous timelines starting from an epoch.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.UTInstant","page":"Dates","title":"Dates.UTInstant","text":"UTInstant{T}\n\nThe UTInstant represents a machine timeline based on UT time (1 day = one revolution of the earth). The T is a Period parameter that indicates the resolution or precision of the instant.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.TimeType","page":"Dates","title":"Dates.TimeType","text":"TimeType\n\nTimeType types wrap Instant machine instances to provide human representations of the machine instant. Time, DateTime and Date are subtypes of TimeType.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.DateTime","page":"Dates","title":"Dates.DateTime","text":"DateTime\n\nDateTime represents a point in time according to the proleptic Gregorian calendar. The finest resolution of the time is millisecond (i.e., microseconds or nanoseconds cannot be represented by this type). The type supports fixed-point arithmetic, and thus is prone to underflowing (and overflowing). A notable consequence is rounding when adding a Microsecond or a Nanosecond:\n\njulia> dt = DateTime(2023, 8, 19, 17, 45, 32, 900)\n2023-08-19T17:45:32.900\n\njulia> dt + Millisecond(1)\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1000) # 1000us == 1ms\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(999) # 999us rounded to 1000us\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1499) # 1499 rounded to 1000us\n2023-08-19T17:45:32.901\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.Date","page":"Dates","title":"Dates.Date","text":"Date\n\nDate wraps a UTInstant{Day} and interprets it according to the proleptic Gregorian calendar.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.Time","page":"Dates","title":"Dates.Time","text":"Time\n\nTime wraps a Nanosecond and represents a specific moment in a 24-hour day.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.TimeZone","page":"Dates","title":"Dates.TimeZone","text":"TimeZone\n\nGeographic zone generally based on longitude determining what the time is at a certain location. Some time zones observe daylight savings (eg EST -> EDT). For implementations and more support, see the TimeZones.jl package\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.UTC","page":"Dates","title":"Dates.UTC","text":"UTC\n\nUTC, or Coordinated Universal Time, is the TimeZone from which all others are measured. It is associated with the time at 0° longitude. It is not adjusted for daylight savings.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates-Functions","page":"Dates","title":"Dates Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.DateTime(::Int64, ::Int64, ::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nDates.DateTime(::Dates.Period)\nDates.DateTime(::Function, ::Any...)\nDates.DateTime(::Dates.TimeType)\nDates.DateTime(::AbstractString, ::AbstractString)\nDates.format(::Dates.TimeType, ::AbstractString)\nDates.DateFormat\nDates.@dateformat_str\nDates.DateTime(::AbstractString, ::Dates.DateFormat)\nDates.Date(::Int64, ::Int64, ::Int64)\nDates.Date(::Dates.Period)\nDates.Date(::Function, ::Any, ::Any, ::Any)\nDates.Date(::Dates.TimeType)\nDates.Date(::AbstractString, ::AbstractString)\nDates.Date(::AbstractString, ::Dates.DateFormat)\nDates.Time(::Int64::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nDates.Time(::Dates.TimePeriod)\nDates.Time(::Function, ::Any...)\nDates.Time(::Dates.DateTime)\nDates.Time(::AbstractString, ::AbstractString)\nDates.Time(::AbstractString, ::Dates.DateFormat)\nDates.now()\nDates.now(::Type{Dates.UTC})\nBase.eps(::Union{Type{DateTime}, Type{Date}, Type{Time}, TimeType})","category":"page"},{"location":"stdlib/Dates.html#Dates.DateTime-NTuple{7, Int64}","page":"Dates","title":"Dates.DateTime","text":"DateTime(y, [m, d, h, mi, s, ms]) -> DateTime\n\nConstruct a DateTime type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{Period}","page":"Dates","title":"Dates.DateTime","text":"DateTime(periods::Period...) -> DateTime\n\nConstruct a DateTime type by Period type parts. Arguments may be in any order. DateTime parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{Function, Vararg{Any}}","page":"Dates","title":"Dates.DateTime","text":"DateTime(f::Function, y[, m, d, h, mi, s]; step=Day(1), limit=10000) -> DateTime\n\nCreate a DateTime through the adjuster API. The starting point will be constructed from the provided y, m, d... arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied).\n\nExamples\n\njulia> DateTime(dt -> second(dt) == 40, 2010, 10, 20, 10; step = Second(1))\n2010-10-20T10:00:40\n\njulia> DateTime(dt -> hour(dt) == 20, 2010, 10, 20, 10; step = Hour(1), limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{TimeType}","page":"Dates","title":"Dates.DateTime","text":"DateTime(dt::Date) -> DateTime\n\nConvert a Date to a DateTime. The hour, minute, second, and millisecond parts of the new DateTime are assumed to be zero.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.DateTime","text":"DateTime(dt::AbstractString, format::AbstractString; locale=\"english\") -> DateTime\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the format string (see DateFormat  for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> DateTime(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01T00:00:00\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [DateTime(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{DateTime}:\n 2020-01-01T00:00:00\n 2020-01-02T00:00:00\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.format-Tuple{TimeType, AbstractString}","page":"Dates","title":"Dates.format","text":"format(dt::TimeType, format::AbstractString; locale=\"english\") -> AbstractString\n\nConstruct a string by using a TimeType object and applying the provided format. The following character codes can be used to construct the format string:\n\nCode Examples Comment\ny 6 Numeric year with a fixed width\nY 1996 Numeric year with a minimum width\nm 1, 12 Numeric month with a minimum width\nu Jan Month name shortened to 3-chars according to the locale\nU January Full month name according to the locale keyword\nd 1, 31 Day of the month with a minimum width\nH 0, 23 Hour (24-hour clock) with a minimum width\nM 0, 59 Minute with a minimum width\nS 0, 59 Second with a minimum width\ns 000, 500 Millisecond with a minimum width of 3\ne Mon, Tue Abbreviated days of the week\nE Monday Full day of week name\n\nThe number of sequential code characters indicate the width of the code. A format of yyyy-mm specifies that the code y should have a width of four while m a width of two. Codes that yield numeric digits have an associated mode: fixed-width or minimum-width. The fixed-width mode left-pads the value with zeros when it is shorter than the specified width and truncates the value when longer. Minimum-width mode works the same as fixed-width except that it does not truncate values longer than the width.\n\nWhen creating a format you can use any non-code characters as a separator. For example to generate the string \"1996-01-15T00:00:00\" you could use format: \"yyyy-mm-ddTHH:MM:SS\". Note that if you need to use a code character as a literal you can use the escape character backslash. The string \"1996y01m\" can be produced with the format \"yyyy\\ymm\\m\".\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.DateFormat","page":"Dates","title":"Dates.DateFormat","text":"DateFormat(format::AbstractString, locale=\"english\") -> DateFormat\n\nConstruct a date formatting object that can be used for parsing date strings or formatting a date object as a string. The following character codes can be used to construct the format string:\n\nCode Matches Comment\nY 1996, 96 Returns year of 1996, 0096\ny 1996, 96 Same as Y on parse but discards excess digits on format\nm 1, 01 Matches 1 or 2-digit months\nu Jan Matches abbreviated months according to the locale keyword\nU January Matches full month names according to the locale keyword\nd 1, 01 Matches 1 or 2-digit days\nH 00 Matches hours (24-hour clock)\nI 00 For outputting hours with 12-hour clock\nM 00 Matches minutes\nS 00 Matches seconds\ns .500 Matches milliseconds\ne Mon, Tues Matches abbreviated days of the week\nE Monday Matches full name days of the week\np AM Matches AM/PM (case-insensitive)\nyyyymmdd 19960101 Matches fixed-width year, month, and day\n\nCharacters not listed above are normally treated as delimiters between date and time slots. For example a dt string of \"1996-01-15T00:00:00.0\" would have a format string like \"y-m-dTH:M:S.s\". If you need to use a code character as a delimiter you can escape it using backslash. The date \"1995y01m\" would have the format \"y\\ym\\m\".\n\nNote that 12:00AM corresponds 00:00 (midnight), and 12:00PM corresponds to 12:00 (noon). When parsing a time with a p specifier, any hour (either H or I) is interpreted as as a 12-hour clock, so the I code is mainly useful for output.\n\nCreating a DateFormat object is expensive. Whenever possible, create it once and use it many times or try the dateformat\"\" string macro. Using this macro creates the DateFormat object once at macro expansion time and reuses it later. There are also several pre-defined formatters, listed later.\n\nSee DateTime and format for how to use a DateFormat object to parse and write Date strings respectively.\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Dates.html#Dates.@dateformat_str","page":"Dates","title":"Dates.@dateformat_str","text":"dateformat\"Y-m-d H:M:S\"\n\nCreate a DateFormat object. Similar to DateFormat(\"Y-m-d H:M:S\") but creates the DateFormat object once during macro expansion.\n\nSee DateFormat for details about format specifiers.\n\n\n\n\n\n","category":"macro"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.DateTime","text":"DateTime(dt::AbstractString, df::DateFormat=ISODateTimeFormat) -> DateTime\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\\THH:MM:SS.s\" if omitted.\n\nSimilar to DateTime(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date time strings with a pre-created DateFormat object.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Int64, Int64, Int64}","page":"Dates","title":"Dates.Date","text":"Date(y, [m, d]) -> Date\n\nConstruct a Date type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Period}","page":"Dates","title":"Dates.Date","text":"Date(period::Period...) -> Date\n\nConstruct a Date type by Period type parts. Arguments may be in any order. Date parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Function, Any, Any, Any}","page":"Dates","title":"Dates.Date","text":"Date(f::Function, y[, m, d]; step=Day(1), limit=10000) -> Date\n\nCreate a Date through the adjuster API. The starting point will be constructed from the provided y, m, d arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (given that f::Function is never satisfied).\n\nExamples\n\njulia> Date(date -> week(date) == 20, 2010, 01, 01)\n2010-05-17\n\njulia> Date(date -> year(date) == 2010, 2000, 01, 01)\n2010-01-01\n\njulia> Date(date -> month(date) == 10, 2000, 01, 01; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{TimeType}","page":"Dates","title":"Dates.Date","text":"Date(dt::DateTime) -> Date\n\nConvert a DateTime to a Date. The hour, minute, second, and millisecond parts of the DateTime are truncated, so only the year, month and day parts are used in construction.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.Date","text":"Date(d::AbstractString, format::AbstractString; locale=\"english\") -> Date\n\nConstruct a Date by parsing the d date string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Date(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [Date(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{Date}:\n 2020-01-01\n 2020-01-02\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.Date","text":"Date(d::AbstractString, df::DateFormat=ISODateFormat) -> Date\n\nConstruct a Date by parsing the d date string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\" if omitted.\n\nSimilar to Date(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date strings with a pre-created DateFormat object.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-NTuple{5, Int64}","page":"Dates","title":"Dates.Time","text":"Time(h, [mi, s, ms, us, ns]) -> Time\n\nConstruct a Time type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{TimePeriod}","page":"Dates","title":"Dates.Time","text":"Time(period::TimePeriod...) -> Time\n\nConstruct a Time type by Period type parts. Arguments may be in any order. Time parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{Function, Vararg{Any}}","page":"Dates","title":"Dates.Time","text":"Time(f::Function, h, mi=0; step::Period=Second(1), limit::Int=10000)\nTime(f::Function, h, mi, s; step::Period=Millisecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms; step::Period=Microsecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms, us; step::Period=Nanosecond(1), limit::Int=10000)\n\nCreate a Time through the adjuster API. The starting point will be constructed from the provided h, mi, s, ms, us arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied). Note that the default step will adjust to allow for greater precision for the given arguments; i.e. if hour, minute, and second arguments are provided, the default step will be Millisecond(1) instead of Second(1).\n\nExamples\n\njulia> Time(t -> minute(t) == 30, 20)\n20:30:00\n\njulia> Time(t -> minute(t) == 0, 20)\n20:00:00\n\njulia> Time(t -> hour(t) == 10, 3; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{DateTime}","page":"Dates","title":"Dates.Time","text":"Time(dt::DateTime) -> Time\n\nConvert a DateTime to a Time. The hour, minute, second, and millisecond parts of the DateTime are used to create the new Time. Microsecond and nanoseconds are zero by default.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.Time","text":"Time(t::AbstractString, format::AbstractString; locale=\"english\") -> Time\n\nConstruct a Time by parsing the t time string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Time(\"12:34pm\", \"HH:MMp\")\n12:34:00\n\njulia> a = (\"12:34pm\", \"2:34am\");\n\njulia> [Time(d, dateformat\"HH:MMp\") for d ∈ a] # preferred\n2-element Vector{Time}:\n 12:34:00\n 02:34:00\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.Time","text":"Time(t::AbstractString, df::DateFormat=ISOTimeFormat) -> Time\n\nConstruct a Time by parsing the t date time string following the pattern given in the DateFormat object, or dateformat\"HH:MM:SS.s\" if omitted.\n\nSimilar to Time(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted time strings with a pre-created DateFormat object.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.now-Tuple{}","page":"Dates","title":"Dates.now","text":"now() -> DateTime\n\nReturn a DateTime corresponding to the user's system time including the system timezone locale.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.now-Tuple{Type{UTC}}","page":"Dates","title":"Dates.now","text":"now(::Type{UTC}) -> DateTime\n\nReturn a DateTime corresponding to the user's system time as UTC/GMT. For other time zones, see the TimeZones.jl package.\n\nExamples\n\njulia> now(UTC)\n2023-01-04T10:52:24.864\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Base.eps-Tuple{Union{Type{Date}, Type{DateTime}, Type{Time}, TimeType}}","page":"Dates","title":"Base.eps","text":"eps(::Type{DateTime}) -> Millisecond\neps(::Type{Date}) -> Day\neps(::Type{Time}) -> Nanosecond\neps(::TimeType) -> Period\n\nReturn the smallest unit value supported by the TimeType.\n\nExamples\n\njulia> eps(DateTime)\n1 millisecond\n\njulia> eps(Date)\n1 day\n\njulia> eps(Time)\n1 nanosecond\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Accessor-Functions-2","page":"Dates","title":"Accessor Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.year\nDates.month\nDates.week\nDates.day\nDates.hour\nDates.minute\nDates.second\nDates.millisecond\nDates.microsecond\nDates.nanosecond\nDates.Year(::Dates.TimeType)\nDates.Month(::Dates.TimeType)\nDates.Week(::Dates.TimeType)\nDates.Day(::Dates.TimeType)\nDates.Hour(::DateTime)\nDates.Minute(::DateTime)\nDates.Second(::DateTime)\nDates.Millisecond(::DateTime)\nDates.Microsecond(::Dates.Time)\nDates.Nanosecond(::Dates.Time)\nDates.yearmonth\nDates.monthday\nDates.yearmonthday","category":"page"},{"location":"stdlib/Dates.html#Dates.year","page":"Dates","title":"Dates.year","text":"year(dt::TimeType) -> Int64\n\nThe year of a Date or DateTime as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.month","page":"Dates","title":"Dates.month","text":"month(dt::TimeType) -> Int64\n\nThe month of a Date or DateTime as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.week","page":"Dates","title":"Dates.week","text":"week(dt::TimeType) -> Int64\n\nReturn the ISO week date of a Date or DateTime as an Int64. Note that the first week of a year is the week that contains the first Thursday of the year, which can result in dates prior to January 4th being in the last week of the previous year. For example, week(Date(2005, 1, 1)) is the 53rd week of 2004.\n\nExamples\n\njulia> week(Date(1989, 6, 22))\n25\n\njulia> week(Date(2005, 1, 1))\n53\n\njulia> week(Date(2004, 12, 31))\n53\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.day","page":"Dates","title":"Dates.day","text":"day(dt::TimeType) -> Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.hour","page":"Dates","title":"Dates.hour","text":"hour(dt::DateTime) -> Int64\n\nThe hour of day of a DateTime as an Int64.\n\n\n\n\n\nhour(t::Time) -> Int64\n\nThe hour of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.minute","page":"Dates","title":"Dates.minute","text":"minute(dt::DateTime) -> Int64\n\nThe minute of a DateTime as an Int64.\n\n\n\n\n\nminute(t::Time) -> Int64\n\nThe minute of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.second","page":"Dates","title":"Dates.second","text":"second(dt::DateTime) -> Int64\n\nThe second of a DateTime as an Int64.\n\n\n\n\n\nsecond(t::Time) -> Int64\n\nThe second of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.millisecond","page":"Dates","title":"Dates.millisecond","text":"millisecond(dt::DateTime) -> Int64\n\nThe millisecond of a DateTime as an Int64.\n\n\n\n\n\nmillisecond(t::Time) -> Int64\n\nThe millisecond of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.microsecond","page":"Dates","title":"Dates.microsecond","text":"microsecond(t::Time) -> Int64\n\nThe microsecond of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.nanosecond","page":"Dates","title":"Dates.nanosecond","text":"nanosecond(t::Time) -> Int64\n\nThe nanosecond of a Time as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.Year-Tuple{TimeType}","page":"Dates","title":"Dates.Year","text":"Year(v)\n\nConstruct a Year object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Month-Tuple{TimeType}","page":"Dates","title":"Dates.Month","text":"Month(v)\n\nConstruct a Month object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Week-Tuple{TimeType}","page":"Dates","title":"Dates.Week","text":"Week(v)\n\nConstruct a Week object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Day-Tuple{TimeType}","page":"Dates","title":"Dates.Day","text":"Day(v)\n\nConstruct a Day object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Hour-Tuple{DateTime}","page":"Dates","title":"Dates.Hour","text":"Hour(dt::DateTime) -> Hour\n\nThe hour part of a DateTime as a Hour.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Minute-Tuple{DateTime}","page":"Dates","title":"Dates.Minute","text":"Minute(dt::DateTime) -> Minute\n\nThe minute part of a DateTime as a Minute.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Second-Tuple{DateTime}","page":"Dates","title":"Dates.Second","text":"Second(dt::DateTime) -> Second\n\nThe second part of a DateTime as a Second.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Millisecond-Tuple{DateTime}","page":"Dates","title":"Dates.Millisecond","text":"Millisecond(dt::DateTime) -> Millisecond\n\nThe millisecond part of a DateTime as a Millisecond.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Microsecond-Tuple{Time}","page":"Dates","title":"Dates.Microsecond","text":"Microsecond(dt::Time) -> Microsecond\n\nThe microsecond part of a Time as a Microsecond.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.Nanosecond-Tuple{Time}","page":"Dates","title":"Dates.Nanosecond","text":"Nanosecond(dt::Time) -> Nanosecond\n\nThe nanosecond part of a Time as a Nanosecond.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.yearmonth","page":"Dates","title":"Dates.yearmonth","text":"yearmonth(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the year and month parts of a Date or DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.monthday","page":"Dates","title":"Dates.monthday","text":"monthday(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the month and day parts of a Date or DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.yearmonthday","page":"Dates","title":"Dates.yearmonthday","text":"yearmonthday(dt::TimeType) -> (Int64, Int64, Int64)\n\nSimultaneously return the year, month and day parts of a Date or DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Query-Functions-2","page":"Dates","title":"Query Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.dayname\nDates.dayabbr\nDates.dayofweek\nDates.dayofmonth\nDates.dayofweekofmonth\nDates.daysofweekinmonth\nDates.monthname\nDates.monthabbr\nDates.daysinmonth\nDates.isleapyear\nDates.dayofyear\nDates.daysinyear\nDates.quarterofyear\nDates.dayofquarter","category":"page"},{"location":"stdlib/Dates.html#Dates.dayname","page":"Dates","title":"Dates.dayname","text":"dayname(dt::TimeType; locale=\"english\") -> String\ndayname(day::Integer; locale=\"english\") -> String\n\nReturn the full day name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayname(Date(\"2000-01-01\"))\n\"Saturday\"\n\njulia> dayname(4)\n\"Thursday\"\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayabbr","page":"Dates","title":"Dates.dayabbr","text":"dayabbr(dt::TimeType; locale=\"english\") -> String\ndayabbr(day::Integer; locale=\"english\") -> String\n\nReturn the abbreviated name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayabbr(Date(\"2000-01-01\"))\n\"Sat\"\n\njulia> dayabbr(3)\n\"Wed\"\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayofweek","page":"Dates","title":"Dates.dayofweek","text":"dayofweek(dt::TimeType) -> Int64\n\nReturn the day of the week as an Int64 with 1 = Monday, 2 = Tuesday, etc..\n\nExamples\n\njulia> dayofweek(Date(\"2000-01-01\"))\n6\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayofmonth","page":"Dates","title":"Dates.dayofmonth","text":"dayofmonth(dt::TimeType) -> Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayofweekofmonth","page":"Dates","title":"Dates.dayofweekofmonth","text":"dayofweekofmonth(dt::TimeType) -> Int\n\nFor the day of week of dt, return which number it is in dt's month. So if the day of the week of dt is Monday, then 1 = First Monday of the month, 2 = Second Monday of the month, etc. In the range 1:5.\n\nExamples\n\njulia> dayofweekofmonth(Date(\"2000-02-01\"))\n1\n\njulia> dayofweekofmonth(Date(\"2000-02-08\"))\n2\n\njulia> dayofweekofmonth(Date(\"2000-02-15\"))\n3\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.daysofweekinmonth","page":"Dates","title":"Dates.daysofweekinmonth","text":"daysofweekinmonth(dt::TimeType) -> Int\n\nFor the day of week of dt, return the total number of that day of the week in dt's month. Returns 4 or 5. Useful in temporal expressions for specifying the last day of a week in a month by including dayofweekofmonth(dt) == daysofweekinmonth(dt) in the adjuster function.\n\nExamples\n\njulia> daysofweekinmonth(Date(\"2005-01-01\"))\n5\n\njulia> daysofweekinmonth(Date(\"2005-01-04\"))\n4\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.monthname","page":"Dates","title":"Dates.monthname","text":"monthname(dt::TimeType; locale=\"english\") -> String\nmonthname(month::Integer, locale=\"english\") -> String\n\nReturn the full name of the month of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthname(Date(\"2005-01-04\"))\n\"January\"\n\njulia> monthname(2)\n\"February\"\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.monthabbr","page":"Dates","title":"Dates.monthabbr","text":"monthabbr(dt::TimeType; locale=\"english\") -> String\nmonthabbr(month::Integer, locale=\"english\") -> String\n\nReturn the abbreviated month name of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthabbr(Date(\"2005-01-04\"))\n\"Jan\"\n\njulia> monthabbr(2)\n\"Feb\"\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.daysinmonth","page":"Dates","title":"Dates.daysinmonth","text":"daysinmonth(dt::TimeType) -> Int\n\nReturn the number of days in the month of dt. Value will be 28, 29, 30, or 31.\n\nExamples\n\njulia> daysinmonth(Date(\"2000-01\"))\n31\n\njulia> daysinmonth(Date(\"2001-02\"))\n28\n\njulia> daysinmonth(Date(\"2000-02\"))\n29\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.isleapyear","page":"Dates","title":"Dates.isleapyear","text":"isleapyear(dt::TimeType) -> Bool\n\nReturn true if the year of dt is a leap year.\n\nExamples\n\njulia> isleapyear(Date(\"2004\"))\ntrue\n\njulia> isleapyear(Date(\"2005\"))\nfalse\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayofyear","page":"Dates","title":"Dates.dayofyear","text":"dayofyear(dt::TimeType) -> Int\n\nReturn the day of the year for dt with January 1st being day 1.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.daysinyear","page":"Dates","title":"Dates.daysinyear","text":"daysinyear(dt::TimeType) -> Int\n\nReturn 366 if the year of dt is a leap year, otherwise return 365.\n\nExamples\n\njulia> daysinyear(1999)\n365\n\njulia> daysinyear(2000)\n366\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.quarterofyear","page":"Dates","title":"Dates.quarterofyear","text":"quarterofyear(dt::TimeType) -> Int\n\nReturn the quarter that dt resides in. Range of value is 1:4.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.dayofquarter","page":"Dates","title":"Dates.dayofquarter","text":"dayofquarter(dt::TimeType) -> Int\n\nReturn the day of the current quarter of dt. Range of value is 1:92.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Adjuster-Functions-2","page":"Dates","title":"Adjuster Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Base.trunc(::Dates.TimeType, ::Type{Dates.Period})\nDates.firstdayofweek\nDates.lastdayofweek\nDates.firstdayofmonth\nDates.lastdayofmonth\nDates.firstdayofyear\nDates.lastdayofyear\nDates.firstdayofquarter\nDates.lastdayofquarter\nDates.tonext(::Dates.TimeType, ::Int)\nDates.toprev(::Dates.TimeType, ::Int)\nDates.tofirst\nDates.tolast\nDates.tonext(::Function, ::Dates.TimeType)\nDates.toprev(::Function, ::Dates.TimeType)","category":"page"},{"location":"stdlib/Dates.html#Base.trunc-Tuple{TimeType, Type{Period}}","page":"Dates","title":"Base.trunc","text":"trunc(dt::TimeType, ::Type{Period}) -> TimeType\n\nTruncates the value of dt according to the provided Period type.\n\nExamples\n\njulia> trunc(DateTime(\"1996-01-01T12:30:00\"), Day)\n1996-01-01T00:00:00\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.firstdayofweek","page":"Dates","title":"Dates.firstdayofweek","text":"firstdayofweek(dt::TimeType) -> TimeType\n\nAdjusts dt to the Monday of its week.\n\nExamples\n\njulia> firstdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-01T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.lastdayofweek","page":"Dates","title":"Dates.lastdayofweek","text":"lastdayofweek(dt::TimeType) -> TimeType\n\nAdjusts dt to the Sunday of its week.\n\nExamples\n\njulia> lastdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-07T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.firstdayofmonth","page":"Dates","title":"Dates.firstdayofmonth","text":"firstdayofmonth(dt::TimeType) -> TimeType\n\nAdjusts dt to the first day of its month.\n\nExamples\n\njulia> firstdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-01T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.lastdayofmonth","page":"Dates","title":"Dates.lastdayofmonth","text":"lastdayofmonth(dt::TimeType) -> TimeType\n\nAdjusts dt to the last day of its month.\n\nExamples\n\njulia> lastdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-31T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.firstdayofyear","page":"Dates","title":"Dates.firstdayofyear","text":"firstdayofyear(dt::TimeType) -> TimeType\n\nAdjusts dt to the first day of its year.\n\nExamples\n\njulia> firstdayofyear(DateTime(\"1996-05-20\"))\n1996-01-01T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.lastdayofyear","page":"Dates","title":"Dates.lastdayofyear","text":"lastdayofyear(dt::TimeType) -> TimeType\n\nAdjusts dt to the last day of its year.\n\nExamples\n\njulia> lastdayofyear(DateTime(\"1996-05-20\"))\n1996-12-31T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.firstdayofquarter","page":"Dates","title":"Dates.firstdayofquarter","text":"firstdayofquarter(dt::TimeType) -> TimeType\n\nAdjusts dt to the first day of its quarter.\n\nExamples\n\njulia> firstdayofquarter(DateTime(\"1996-05-20\"))\n1996-04-01T00:00:00\n\njulia> firstdayofquarter(DateTime(\"1996-08-20\"))\n1996-07-01T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.lastdayofquarter","page":"Dates","title":"Dates.lastdayofquarter","text":"lastdayofquarter(dt::TimeType) -> TimeType\n\nAdjusts dt to the last day of its quarter.\n\nExamples\n\njulia> lastdayofquarter(DateTime(\"1996-05-20\"))\n1996-06-30T00:00:00\n\njulia> lastdayofquarter(DateTime(\"1996-08-20\"))\n1996-09-30T00:00:00\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.tonext-Tuple{TimeType, Int64}","page":"Dates","title":"Dates.tonext","text":"tonext(dt::TimeType, dow::Int; same::Bool=false) -> TimeType\n\nAdjusts dt to the next day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the next dow, allowing for no adjustment to occur.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.toprev-Tuple{TimeType, Int64}","page":"Dates","title":"Dates.toprev","text":"toprev(dt::TimeType, dow::Int; same::Bool=false) -> TimeType\n\nAdjusts dt to the previous day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the previous dow, allowing for no adjustment to occur.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.tofirst","page":"Dates","title":"Dates.tofirst","text":"tofirst(dt::TimeType, dow::Int; of=Month) -> TimeType\n\nAdjusts dt to the first dow of its month. Alternatively, of=Year will adjust to the first dow of the year.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.tolast","page":"Dates","title":"Dates.tolast","text":"tolast(dt::TimeType, dow::Int; of=Month) -> TimeType\n\nAdjusts dt to the last dow of its month. Alternatively, of=Year will adjust to the last dow of the year.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.tonext-Tuple{Function, TimeType}","page":"Dates","title":"Dates.tonext","text":"tonext(func::Function, dt::TimeType; step=Day(1), limit=10000, same=false) -> TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.toprev-Tuple{Function, TimeType}","page":"Dates","title":"Dates.toprev","text":"toprev(func::Function, dt::TimeType; step=Day(-1), limit=10000, same=false) -> TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Periods","page":"Dates","title":"Periods","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.Period(::Any)\nDates.CompoundPeriod(::Vector{<:Dates.Period})\nDates.canonicalize\nDates.value\nDates.default\nDates.periods","category":"page"},{"location":"stdlib/Dates.html#Dates.Period-Tuple{Any}","page":"Dates","title":"Dates.Period","text":"Year(v)\nQuarter(v)\nMonth(v)\nWeek(v)\nDay(v)\nHour(v)\nMinute(v)\nSecond(v)\nMillisecond(v)\nMicrosecond(v)\nNanosecond(v)\n\nConstruct a Period type with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.CompoundPeriod-Tuple{Vector{<:Period}}","page":"Dates","title":"Dates.CompoundPeriod","text":"CompoundPeriod(periods) -> CompoundPeriod\n\nConstruct a CompoundPeriod from a Vector of Periods. All Periods of the same type will be added together.\n\nExamples\n\njulia> Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13))\n25 hours\n\njulia> Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1))\n-1 hour, 1 minute\n\njulia> Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2))\n1 month, -2 weeks\n\njulia> Dates.CompoundPeriod(Dates.Minute(50000))\n50000 minutes\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Dates.canonicalize","page":"Dates","title":"Dates.canonicalize","text":"canonicalize(::CompoundPeriod) -> CompoundPeriod\n\nReduces the CompoundPeriod into its canonical form by applying the following rules:\n\nAny Period large enough be partially representable by a coarser Period will be broken into multiple Periods (eg. Hour(30) becomes Day(1) + Hour(6))\nPeriods with opposite signs will be combined when possible (eg. Hour(1) - Day(1) becomes -Hour(23))\n\nExamples\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13)))\n1 day, 1 hour\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1)))\n-59 minutes\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2)))\n1 month, -2 weeks\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Minute(50000)))\n4 weeks, 6 days, 17 hours, 20 minutes\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.value","page":"Dates","title":"Dates.value","text":"Dates.value(x::Period) -> Int64\n\nFor a given period, return the value associated with that period.  For example, value(Millisecond(10)) returns 10 as an integer.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.default","page":"Dates","title":"Dates.default","text":"default(p::Period) -> Period\n\nReturn a sensible \"default\" value for the input Period by returning T(1) for Year, Month, and Day, and T(0) for Hour, Minute, Second, and Millisecond.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.periods","page":"Dates","title":"Dates.periods","text":"Dates.periods(::CompoundPeriod) -> Vector{Period}\n\nReturn the Vector of Periods that comprise the given CompoundPeriod.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Rounding-Functions","page":"Dates","title":"Rounding Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round.","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Base.floor(::Dates.TimeType, ::Dates.Period)\nBase.ceil(::Dates.TimeType, ::Dates.Period)\nBase.round(::Dates.TimeType, ::Dates.Period, ::RoundingMode{:NearestTiesUp})","category":"page"},{"location":"stdlib/Dates.html#Base.floor-Tuple{TimeType, Period}","page":"Dates","title":"Base.floor","text":"floor(dt::TimeType, p::Period) -> TimeType\n\nReturn the nearest Date or DateTime less than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: floor(dt, Dates.Hour) is a shortcut for floor(dt, Dates.Hour(1)).\n\njulia> floor(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> floor(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> floor(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-06T00:00:00\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Base.ceil-Tuple{TimeType, Period}","page":"Dates","title":"Base.ceil","text":"ceil(dt::TimeType, p::Period) -> TimeType\n\nReturn the nearest Date or DateTime greater than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: ceil(dt, Dates.Hour) is a shortcut for ceil(dt, Dates.Hour(1)).\n\njulia> ceil(Date(1985, 8, 16), Month)\n1985-09-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:45:00\n\njulia> ceil(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Base.round-Tuple{TimeType, Period, RoundingMode{:NearestTiesUp}}","page":"Dates","title":"Base.round","text":"round(dt::TimeType, p::Period, [r::RoundingMode]) -> TimeType\n\nReturn the Date or DateTime nearest to dt at resolution p. By default (RoundNearestTiesUp), ties (e.g., rounding 9:30 to the nearest hour) will be rounded up.\n\nFor convenience, p may be a type instead of a value: round(dt, Dates.Hour) is a shortcut for round(dt, Dates.Hour(1)).\n\njulia> round(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> round(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> round(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\nValid rounding modes for round(::TimeType, ::Period, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Most Period values can also be rounded to a specified resolution:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Base.floor(::Dates.ConvertiblePeriod, ::T) where T <: Dates.ConvertiblePeriod\nBase.ceil(::Dates.ConvertiblePeriod, ::Dates.ConvertiblePeriod)\nBase.round(::Dates.ConvertiblePeriod, ::Dates.ConvertiblePeriod, ::RoundingMode{:NearestTiesUp})","category":"page"},{"location":"stdlib/Dates.html#Base.floor-Union{Tuple{T}, Tuple{Union{Day, Week, TimePeriod}, T}} where T<:Union{Day, Week, TimePeriod}","page":"Dates","title":"Base.floor","text":"floor(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x down to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: floor(x, Dates.Hour) is a shortcut for floor(x, Dates.Hour(1)).\n\njulia> floor(Day(16), Week)\n2 weeks\n\njulia> floor(Minute(44), Minute(15))\n30 minutes\n\njulia> floor(Hour(36), Day)\n1 day\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Base.ceil-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}}","page":"Dates","title":"Base.ceil","text":"ceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x up to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: ceil(x, Dates.Hour) is a shortcut for ceil(x, Dates.Hour(1)).\n\njulia> ceil(Day(16), Week)\n3 weeks\n\njulia> ceil(Minute(44), Minute(15))\n45 minutes\n\njulia> ceil(Hour(36), Day)\n2 days\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html#Base.round-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}, RoundingMode{:NearestTiesUp}}","page":"Dates","title":"Base.round","text":"round(x::Period, precision::T, [r::RoundingMode]) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision. By default (RoundNearestTiesUp), ties (e.g., rounding 90 minutes to the nearest hour) will be rounded up.\n\nFor convenience, precision may be a type instead of a value: round(x, Dates.Hour) is a shortcut for round(x, Dates.Hour(1)).\n\njulia> round(Day(16), Week)\n2 weeks\n\njulia> round(Minute(44), Minute(15))\n45 minutes\n\njulia> round(Hour(36), Day)\n2 days\n\nValid rounding modes for round(::Period, ::T, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"The following functions are not exported:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.floorceil\nDates.epochdays2date\nDates.epochms2datetime\nDates.date2epochdays\nDates.datetime2epochms","category":"page"},{"location":"stdlib/Dates.html#Dates.floorceil","page":"Dates","title":"Dates.floorceil","text":"floorceil(dt::TimeType, p::Period) -> (TimeType, TimeType)\n\nSimultaneously return the floor and ceil of a Date or DateTime at resolution p. More efficient than calling both floor and ceil individually.\n\n\n\n\n\nfloorceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> (T, T)\n\nSimultaneously return the floor and ceil of Period at resolution p.  More efficient than calling both floor and ceil individually.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.epochdays2date","page":"Dates","title":"Dates.epochdays2date","text":"epochdays2date(days) -> Date\n\nTake the number of days since the rounding epoch (0000-01-01T00:00:00) and return the corresponding Date.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.epochms2datetime","page":"Dates","title":"Dates.epochms2datetime","text":"epochms2datetime(milliseconds) -> DateTime\n\nTake the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) and return the corresponding DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.date2epochdays","page":"Dates","title":"Dates.date2epochdays","text":"date2epochdays(dt::Date) -> Int64\n\nTake the given Date and return the number of days since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.datetime2epochms","page":"Dates","title":"Dates.datetime2epochms","text":"datetime2epochms(dt::DateTime) -> Int64\n\nTake the given DateTime and return the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Conversion-Functions","page":"Dates","title":"Conversion Functions","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Dates.today\nDates.unix2datetime\nDates.datetime2unix\nDates.julian2datetime\nDates.datetime2julian\nDates.rata2datetime\nDates.datetime2rata","category":"page"},{"location":"stdlib/Dates.html#Dates.today","page":"Dates","title":"Dates.today","text":"today() -> Date\n\nReturn the date portion of now().\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.unix2datetime","page":"Dates","title":"Dates.unix2datetime","text":"unix2datetime(x) -> DateTime\n\nTake the number of seconds since unix epoch 1970-01-01T00:00:00 and convert to the corresponding DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.datetime2unix","page":"Dates","title":"Dates.datetime2unix","text":"datetime2unix(dt::DateTime) -> Float64\n\nTake the given DateTime and return the number of seconds since the unix epoch 1970-01-01T00:00:00 as a Float64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.julian2datetime","page":"Dates","title":"Dates.julian2datetime","text":"julian2datetime(julian_days) -> DateTime\n\nTake the number of Julian calendar days since epoch -4713-11-24T12:00:00 and return the corresponding DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.datetime2julian","page":"Dates","title":"Dates.datetime2julian","text":"datetime2julian(dt::DateTime) -> Float64\n\nTake the given DateTime and return the number of Julian calendar days since the julian epoch -4713-11-24T12:00:00 as a Float64.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.rata2datetime","page":"Dates","title":"Dates.rata2datetime","text":"rata2datetime(days) -> DateTime\n\nTake the number of Rata Die days since epoch 0000-12-31T00:00:00 and return the corresponding DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Dates.datetime2rata","page":"Dates","title":"Dates.datetime2rata","text":"datetime2rata(dt::TimeType) -> Int64\n\nReturn the number of Rata Die days since epoch from the given Date or DateTime.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Dates.html#Constants","page":"Dates","title":"Constants","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Days of the Week:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Variable Abbr. Value (Int)\nMonday Mon 1\nTuesday Tue 2\nWednesday Wed 3\nThursday Thu 4\nFriday Fri 5\nSaturday Sat 6\nSunday Sun 7","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Months of the Year:","category":"page"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"Variable Abbr. Value (Int)\nJanuary Jan 1\nFebruary Feb 2\nMarch Mar 3\nApril Apr 4\nMay May 5\nJune Jun 6\nJuly Jul 7\nAugust Aug 8\nSeptember Sep 9\nOctober Oct 10\nNovember Nov 11\nDecember Dec 12","category":"page"},{"location":"stdlib/Dates.html#Common-Date-Formatters","page":"Dates","title":"Common Date Formatters","text":"","category":"section"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"ISODateTimeFormat\nISODateFormat\nISOTimeFormat\nRFC1123Format","category":"page"},{"location":"stdlib/Dates.html#Dates.ISODateTimeFormat","page":"Dates","title":"Dates.ISODateTimeFormat","text":"Dates.ISODateTimeFormat\n\nDescribes the ISO8601 formatting for a date and time. This is the default value for Dates.format of a DateTime.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), ISODateTimeFormat)\n\"2018-08-08T12:00:43.001\"\n\n\n\n\n\n","category":"constant"},{"location":"stdlib/Dates.html#Dates.ISODateFormat","page":"Dates","title":"Dates.ISODateFormat","text":"Dates.ISODateFormat\n\nDescribes the ISO8601 formatting for a date. This is the default value for Dates.format of a Date.\n\nExamples\n\njulia> Dates.format(Date(2018, 8, 8), ISODateFormat)\n\"2018-08-08\"\n\n\n\n\n\n","category":"constant"},{"location":"stdlib/Dates.html#Dates.ISOTimeFormat","page":"Dates","title":"Dates.ISOTimeFormat","text":"Dates.ISOTimeFormat\n\nDescribes the ISO8601 formatting for a time. This is the default value for Dates.format of a Time.\n\nExamples\n\njulia> Dates.format(Time(12, 0, 43, 1), ISOTimeFormat)\n\"12:00:43.001\"\n\n\n\n\n\n","category":"constant"},{"location":"stdlib/Dates.html#Dates.RFC1123Format","page":"Dates","title":"Dates.RFC1123Format","text":"Dates.RFC1123Format\n\nDescribes the RFC1123 formatting for a date and time.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), RFC1123Format)\n\"Wed, 08 Aug 2018 12:00:43\"\n\n\n\n\n\n","category":"constant"},{"location":"stdlib/Dates.html","page":"Dates","title":"Dates","text":"DocTestSetup = nothing","category":"page"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/stdlib/FileWatching/docs/src/index.md\"","category":"page"},{"location":"stdlib/FileWatching.html#lib-filewatching","page":"File Events","title":"File Events","text":"","category":"section"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"FileWatching.poll_fd\nFileWatching.poll_file\nFileWatching.watch_file\nFileWatching.watch_folder\nFileWatching.unwatch_folder","category":"page"},{"location":"stdlib/FileWatching.html#FileWatching.poll_fd","page":"File Events","title":"FileWatching.poll_fd","text":"poll_fd(fd, timeout_s::Real=-1; readable=false, writable=false)\n\nMonitor a file descriptor fd for changes in the read or write availability, and with a timeout given by timeout_s seconds.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.poll_file","page":"File Events","title":"FileWatching.poll_file","text":"poll_file(path::AbstractString, interval_s::Real=5.007, timeout_s::Real=-1) -> (previous::StatStruct, current)\n\nMonitor a file for changes by polling every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds.\n\nReturns a pair of status objects (previous, current) when a change is detected. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (indicating the timeout elapsed), or some other Exception subtype (if the stat operation failed - for example, if the path does not exist).\n\nTo determine when a file was modified, compare current isa StatStruct && mtime(prev) != mtime(current) to detect notification of changes. However, using watch_file for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.watch_file","page":"File Events","title":"FileWatching.watch_file","text":"watch_file(path::AbstractString, timeout_s::Real=-1)\n\nWatch file or directory path for changes until a change occurs or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThe returned value is an object with boolean fields renamed, changed, and timedout, giving the result of watching the file.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.watch_folder","page":"File Events","title":"FileWatching.watch_folder","text":"watch_folder(path::AbstractString, timeout_s::Real=-1)\n\nWatches a file or directory path for changes until a change has occurred or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis will continuing tracking changes for path in the background until unwatch_folder is called on the same path.\n\nThe returned value is an pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed, changed, and timedout, giving the event.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.unwatch_folder","page":"File Events","title":"FileWatching.unwatch_folder","text":"unwatch_folder(path::AbstractString)\n\nStop background tracking of changes for path. It is not recommended to do this while another task is waiting for watch_folder to return on the same path, as the result may be unpredictable.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#Pidfile","page":"File Events","title":"Pidfile","text":"","category":"section"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"CurrentModule = FileWatching.Pidfile","category":"page"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"A simple utility tool for creating advisory pidfiles (lock files).","category":"page"},{"location":"stdlib/FileWatching.html#Primary-Functions","page":"File Events","title":"Primary Functions","text":"","category":"section"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"mkpidlock\ntrymkpidlock\nclose(lock::LockMonitor)","category":"page"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.mkpidlock","page":"File Events","title":"FileWatching.Pidfile.mkpidlock","text":"mkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\nmkpidlock(at::String, proc::Process; kwopts...)\n\nCreate a pidfile lock for the path \"at\" for the current process or the process identified by pid or proc. Can take a function to execute once locked, for usage in do blocks, after which the lock will be automatically closed. If the lock fails and wait is false, then an error is thrown.\n\nThe lock will be released by either close, a finalizer, or shortly after proc exits. Make sure the return value is live through the end of the critical section of your program, so the finalizer does not reclaim it early.\n\nOptional keyword arguments:\n\nmode: file access mode (modified by the process umask). Defaults to world-readable.\npoll_interval: Specify the maximum time to between attempts (if watch_file doesn't work)\nstale_age: Delete an existing pidfile (ignoring the lock) if it is older than this many seconds, based on its mtime.   The file won't be deleted until 5x longer than this if the pid in the file appears that it may be valid.   Or 25x longer if refresh is overridden to 0 to disable lock refreshing.   By default this is disabled (stale_age = 0), but a typical recommended value would be about 3-5x an   estimated normal completion time.\nrefresh: Keeps a lock from becoming stale by updating the mtime every interval of time that passes.   By default, this is set to stale_age/2, which is the recommended value.\nwait: If true, block until we get the lock, if false, raise error if lock fails.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.trymkpidlock","page":"File Events","title":"FileWatching.Pidfile.trymkpidlock","text":"trymkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\ntrymkpidlock(at::String, proc::Process; kwopts...)\n\nLike mkpidlock except returns false instead of waiting if the file is already locked.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#Base.close-Tuple{FileWatching.Pidfile.LockMonitor}","page":"File Events","title":"Base.close","text":"close(lock::LockMonitor)\n\nRelease a pidfile lock.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/FileWatching.html#Helper-Functions","page":"File Events","title":"Helper Functions","text":"","category":"section"},{"location":"stdlib/FileWatching.html","page":"File Events","title":"File Events","text":"Pidfile.open_exclusive\nPidfile.tryopen_exclusive\nPidfile.write_pidfile\nPidfile.parse_pidfile\nPidfile.stale_pidfile\nPidfile.isvalidpid\nBase.touch(::Pidfile.LockMonitor)","category":"page"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.open_exclusive","page":"File Events","title":"FileWatching.Pidfile.open_exclusive","text":"open_exclusive(path::String; mode, poll_interval, wait, stale_age, refresh) :: File\n\nCreate a new a file for read-write advisory-exclusive access. If wait is false then error out if the lock files exist otherwise block until we get the lock.\n\nFor a description of the keyword arguments, see mkpidlock.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.tryopen_exclusive","page":"File Events","title":"FileWatching.Pidfile.tryopen_exclusive","text":"tryopen_exclusive(path::String, mode::Integer = 0o444) :: Union{Void, File}\n\nTry to create a new file for read-write advisory-exclusive access, return nothing if it already exists.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.write_pidfile","page":"File Events","title":"FileWatching.Pidfile.write_pidfile","text":"write_pidfile(io, pid)\n\nWrite our pidfile format to an open IO descriptor.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.parse_pidfile","page":"File Events","title":"FileWatching.Pidfile.parse_pidfile","text":"parse_pidfile(file::Union{IO, String}) => (pid, hostname, age)\n\nAttempt to parse our pidfile format, replaced an element with (0, \"\", 0.0), respectively, for any read that failed.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.stale_pidfile","page":"File Events","title":"FileWatching.Pidfile.stale_pidfile","text":"stale_pidfile(path::String, stale_age::Real, refresh::Real) :: Bool\n\nHelper function for open_exclusive for deciding if a pidfile is stale.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.isvalidpid","page":"File Events","title":"FileWatching.Pidfile.isvalidpid","text":"isvalidpid(hostname::String, pid::Cuint) :: Bool\n\nAttempt to conservatively estimate whether pid is a valid process id.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/FileWatching.html#Base.Filesystem.touch-Tuple{FileWatching.Pidfile.LockMonitor}","page":"File Events","title":"Base.Filesystem.touch","text":"Base.touch(::Pidfile.LockMonitor)\n\nUpdate the mtime on the lock, to indicate it is still fresh.\n\nSee also the refresh keyword in the mkpidlock constructor.\n\n\n\n\n\n","category":"method"},{"location":"stdlib/UUIDs.html","page":"UUIDs","title":"UUIDs","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/stdlib/UUIDs/docs/src/index.md\"","category":"page"},{"location":"stdlib/UUIDs.html#UUIDs","page":"UUIDs","title":"UUIDs","text":"","category":"section"},{"location":"stdlib/UUIDs.html","page":"UUIDs","title":"UUIDs","text":"UUIDs.uuid1\nUUIDs.uuid4\nUUIDs.uuid5\nUUIDs.uuid_version","category":"page"},{"location":"stdlib/UUIDs.html#UUIDs.uuid1","page":"UUIDs","title":"UUIDs.uuid1","text":"uuid1([rng::AbstractRNG]) -> UUID\n\nGenerates a version 1 (time-based) universally unique identifier (UUID), as specified by RFC 4122. Note that the Node ID is randomly generated (does not identify the host) according to section 4.5 of the RFC.\n\nThe default rng used by uuid1 is not Random.default_rng() and every invocation of uuid1() without an argument should be expected to return a unique identifier. Importantly, the outputs of uuid1 do not repeat even when Random.seed!(seed) is called. Currently (as of Julia 1.6), uuid1 uses Random.RandomDevice as the default rng. However, this is an implementation detail that may change in the future.\n\ncompat: Julia 1.6\nThe output of uuid1 does not depend on Random.default_rng() as of Julia 1.6.\n\nExamples\n\njulia> using Random\n\njulia> rng = MersenneTwister(1234);\n\njulia> uuid1(rng)\nUUID(\"cfc395e8-590f-11e8-1f13-43a2532b2fa8\")\n\n\n\n\n\n","category":"function"},{"location":"stdlib/UUIDs.html#UUIDs.uuid4","page":"UUIDs","title":"UUIDs.uuid4","text":"uuid4([rng::AbstractRNG]) -> UUID\n\nGenerates a version 4 (random or pseudo-random) universally unique identifier (UUID), as specified by RFC 4122.\n\nThe default rng used by uuid4 is not Random.default_rng() and every invocation of uuid4() without an argument should be expected to return a unique identifier. Importantly, the outputs of uuid4 do not repeat even when Random.seed!(seed) is called. Currently (as of Julia 1.6), uuid4 uses Random.RandomDevice as the default rng. However, this is an implementation detail that may change in the future.\n\ncompat: Julia 1.6\nThe output of uuid4 does not depend on Random.default_rng() as of Julia 1.6.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> uuid4(rng)\nUUID(\"856e446e-0c6a-472a-9638-f7b8557cd282\")\n\n\n\n\n\n","category":"function"},{"location":"stdlib/UUIDs.html#UUIDs.uuid5","page":"UUIDs","title":"UUIDs.uuid5","text":"uuid5(ns::UUID, name::String) -> UUID\n\nGenerates a version 5 (namespace and domain-based) universally unique identifier (UUID), as specified by RFC 4122.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> u4 = uuid4(rng)\nUUID(\"856e446e-0c6a-472a-9638-f7b8557cd282\")\n\njulia> u5 = uuid5(u4, \"julia\")\nUUID(\"2df91e3f-da06-5362-a6fe-03772f2e14c9\")\n\n\n\n\n\n","category":"function"},{"location":"stdlib/UUIDs.html#UUIDs.uuid_version","page":"UUIDs","title":"UUIDs.uuid_version","text":"uuid_version(u::UUID) -> Int\n\nInspects the given UUID and returns its version (see RFC 4122).\n\nExamples\n\njulia> uuid_version(uuid4())\n4\n\n\n\n\n\n","category":"function"},{"location":"manual/stacktraces.html#Stack-Traces","page":"Stack Traces","title":"Stack Traces","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"The StackTraces module provides simple stack traces that are both human readable and easy to use programmatically.","category":"page"},{"location":"manual/stacktraces.html#Viewing-a-stack-trace","page":"Stack Traces","title":"Viewing a stack trace","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"The primary function used to obtain a stack trace is stacktrace:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Calling stacktrace() returns a vector of StackTraces.StackFrame s. For ease of use, the alias StackTraces.StackTrace can be used in place of Vector{StackFrame}. (Examples with [...] indicate that output may vary depending on how the code is run.)","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]\n\njulia> @noinline child() = stacktrace()\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> grandparent() = parent()\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\n9-element Array{Base.StackTraces.StackFrame,1}:\n child() at REPL[3]:1\n parent() at REPL[4]:1\n grandparent() at REPL[5]:1\n[...]","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Note that when calling stacktrace() you'll typically see a frame with eval at boot.jl. When calling stacktrace() from the REPL you'll also have a few extra frames in the stack from REPL.jl, usually looking something like this:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92","category":"page"},{"location":"manual/stacktraces.html#Extracting-useful-information","page":"Stack Traces","title":"Extracting useful information","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Each StackTraces.StackFrame contains the function name, file name, line number, lambda info, a flag indicating whether the frame has been inlined, a flag indicating whether it is a C function (by default C functions do not appear in the stack trace), and an integer representation of the pointer returned by backtrace:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> frame = stacktrace()[3]\neval(::Module, ::Expr) at REPL.jl:5\n\njulia> frame.func\n:eval\n\njulia> frame.file\nSymbol(\"~/julia/usr/share/julia/stdlib/v0.7/REPL/src/REPL.jl\")\n\njulia> frame.line\n5\n\njulia> frame.linfo\nMethodInstance for eval(::Module, ::Expr)\n\njulia> frame.inlined\nfalse\n\njulia> frame.from_c\nfalse\n\njulia> frame.pointer\n0x00007f92d6293171","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"This makes stack trace information available programmatically for logging, error handling, and more.","category":"page"},{"location":"manual/stacktraces.html#Error-handling","page":"Stack Traces","title":"Error handling","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"While having easy access to information about the current state of the callstack can be helpful in many places, the most obvious application is in error handling and debugging.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace()\n       end\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[2]:4\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"You may notice that in the example above the first stack frame points at line 4, where stacktrace is called, rather than line 2, where bad_function is called, and bad_function's frame is missing entirely. This is understandable, given that stacktrace is called from the context of the catch. While in this example it's fairly easy to find the actual source of the error, in complex cases tracking down the source of the error becomes nontrivial.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"This can be remedied by passing the result of catch_backtrace to stacktrace. Instead of returning callstack information for the current context, catch_backtrace returns stack information for the context of the most recent exception:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace(catch_backtrace())\n       end\nexample (generic function with 1 method)\n\njulia> example()\n8-element Array{Base.StackTraces.StackFrame,1}:\n bad_function() at REPL[1]:1\n example() at REPL[2]:2\n[...]","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Notice that the stack trace now indicates the appropriate line number and the missing frame.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> @noinline child() = error(\"Whoops!\")\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> @noinline function grandparent()\n           try\n               parent()\n           catch err\n               println(\"ERROR: \", err.msg)\n               stacktrace(catch_backtrace())\n           end\n       end\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\nERROR: Whoops!\n10-element Array{Base.StackTraces.StackFrame,1}:\n error at error.jl:33 [inlined]\n child() at REPL[1]:1\n parent() at REPL[2]:1\n grandparent() at REPL[3]:3\n[...]","category":"page"},{"location":"manual/stacktraces.html#Exception-stacks-and-[current_exceptions](@ref)","page":"Stack Traces","title":"Exception stacks and current_exceptions","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"compat: Julia 1.1\nException stacks requires at least Julia 1.1.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"While handling an exception further exceptions may be thrown. It can be useful to inspect all these exceptions to identify the root cause of a problem. The julia runtime supports this by pushing each exception onto an internal exception stack as it occurs. When the code exits a catch normally, any exceptions which were pushed onto the stack in the associated try are considered to be successfully handled and are removed from the stack.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"The stack of current exceptions can be accessed using the current_exceptions function. For example,","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> try\n           error(\"(A) The root cause\")\n       catch\n           try\n               error(\"(B) An exception while handling the exception\")\n           catch\n               for (exc, bt) in current_exceptions()\n                   showerror(stdout, exc, bt)\n                   println(stdout)\n               end\n           end\n       end\n(A) The root cause\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:2\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259\n(B) An exception while handling the exception\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:5\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"In this example the root cause exception (A) is first on the stack, with a further exception (B) following it. After exiting both catch blocks normally (i.e., without throwing a further exception) all exceptions are removed from the stack and are no longer accessible.","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"The exception stack is stored on the Task where the exceptions occurred. When a task fails with uncaught exceptions, current_exceptions(task) may be used to inspect the exception stack for that task.","category":"page"},{"location":"manual/stacktraces.html#Comparison-with-[backtrace](@ref)","page":"Stack Traces","title":"Comparison with backtrace","text":"","category":"section"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"A call to backtrace returns a vector of Union{Ptr{Nothing}, Base.InterpreterIP}, which may then be passed into stacktrace for translation:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> trace = backtrace()\n18-element Array{Union{Ptr{Nothing}, Base.InterpreterIP},1}:\n Ptr{Nothing} @0x00007fd8734c6209\n Ptr{Nothing} @0x00007fd87362b342\n Ptr{Nothing} @0x00007fd87362c136\n Ptr{Nothing} @0x00007fd87362c986\n Ptr{Nothing} @0x00007fd87362d089\n Base.InterpreterIP(CodeInfo(:(begin\n      Core.SSAValue(0) = backtrace()\n      trace = Core.SSAValue(0)\n      return Core.SSAValue(0)\n  end)), 0x0000000000000000)\n Ptr{Nothing} @0x00007fd87362e4cf\n[...]\n\njulia> stacktrace(trace)\n6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Notice that the vector returned by backtrace had 18 elements, while the vector returned by stacktrace only has 6. This is because, by default, stacktrace removes any lower-level C functions from the stack. If you want to include stack frames from C calls, you can do it like this:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> stacktrace(trace, true)\n21-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n do_call at interpreter.c:324\n eval_value at interpreter.c:416\n eval_body at interpreter.c:559\n jl_interpret_toplevel_thunk_callback at interpreter.c:798\n top-level scope\n jl_interpret_toplevel_thunk at interpreter.c:807\n jl_toplevel_eval_flex at toplevel.c:856\n jl_toplevel_eval_in at builtins.c:624\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n jl_apply_generic at gf.c:2167\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n jl_apply_generic at gf.c:2167\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n jl_fptr_trampoline at gf.c:1838\n jl_apply_generic at gf.c:2167\n jl_apply at julia.h:1540 [inlined]\n start_task at task.c:268\n ip:0xffffffffffffffff","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"Individual pointers returned by backtrace can be translated into StackTraces.StackFrame s by passing them into StackTraces.lookup:","category":"page"},{"location":"manual/stacktraces.html","page":"Stack Traces","title":"Stack Traces","text":"julia> pointer = backtrace()[1];\n\njulia> frame = StackTraces.lookup(pointer)\n1-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n\njulia> println(\"The top frame is from $(frame[1].func)!\")\nThe top frame is from jl_apply_generic!","category":"page"},{"location":"stdlib/Base64.html","page":"Base64","title":"Base64","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/stdlib/Base64/docs/src/index.md\"","category":"page"},{"location":"stdlib/Base64.html#Base64","page":"Base64","title":"Base64","text":"","category":"section"},{"location":"stdlib/Base64.html","page":"Base64","title":"Base64","text":"Base64.Base64\nBase64.Base64EncodePipe\nBase64.base64encode\nBase64.Base64DecodePipe\nBase64.base64decode\nBase64.stringmime","category":"page"},{"location":"stdlib/Base64.html#Base64.Base64","page":"Base64","title":"Base64.Base64","text":"Base64\n\nFunctionality for base64 encoding and decoding, a method to represent binary data using text, common on the web.\n\n\n\n\n\n","category":"module"},{"location":"stdlib/Base64.html#Base64.Base64EncodePipe","page":"Base64","title":"Base64.Base64EncodePipe","text":"Base64EncodePipe(ostream)\n\nReturn a new write-only I/O stream, which converts any bytes written to it into base64-encoded ASCII bytes written to ostream.  Calling close on the Base64EncodePipe stream is necessary to complete the encoding (but does not close ostream).\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> iob64_encode = Base64EncodePipe(io);\n\njulia> write(iob64_encode, \"Hello!\")\n6\n\njulia> close(iob64_encode);\n\njulia> str = String(take!(io))\n\"SGVsbG8h\"\n\njulia> String(base64decode(str))\n\"Hello!\"\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Base64.html#Base64.base64encode","page":"Base64","title":"Base64.base64encode","text":"base64encode(writefunc, args...; context=nothing)\nbase64encode(args...; context=nothing)\n\nGiven a write-like function writefunc, which takes an I/O stream as its first argument, base64encode(writefunc, args...) calls writefunc to write args... to a base64-encoded string, and returns the string. base64encode(args...) is equivalent to base64encode(write, args...): it converts its arguments into bytes using the standard write functions and returns the base64-encoded string.\n\nThe optional keyword argument context can be set to :key=>value pair or an IO or IOContext object whose attributes are used for the I/O stream passed to writefunc or write.\n\nSee also base64decode.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Base64.html#Base64.Base64DecodePipe","page":"Base64","title":"Base64.Base64DecodePipe","text":"Base64DecodePipe(istream)\n\nReturn a new read-only I/O stream, which decodes base64-encoded data read from istream.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> iob64_decode = Base64DecodePipe(io);\n\njulia> write(io, \"SGVsbG8h\")\n8\n\njulia> seekstart(io);\n\njulia> String(read(iob64_decode))\n\"Hello!\"\n\n\n\n\n\n","category":"type"},{"location":"stdlib/Base64.html#Base64.base64decode","page":"Base64","title":"Base64.base64decode","text":"base64decode(string)\n\nDecode the base64-encoded string and returns a Vector{UInt8} of the decoded bytes.\n\nSee also base64encode.\n\nExamples\n\njulia> b = base64decode(\"SGVsbG8h\")\n6-element Vector{UInt8}:\n 0x48\n 0x65\n 0x6c\n 0x6c\n 0x6f\n 0x21\n\njulia> String(b)\n\"Hello!\"\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Base64.html#Base64.stringmime","page":"Base64","title":"Base64.stringmime","text":"stringmime(mime, x; context=nothing)\n\nReturn an AbstractString containing the representation of x in the requested mime type. This is similar to repr(mime, x) except that binary data is base64-encoded as an ASCII string.\n\nThe optional keyword argument context can be set to :key=>value pair or an IO or IOContext object whose attributes are used for the I/O stream passed to show.\n\n\n\n\n\n","category":"function"},{"location":"devdocs/offset-arrays.html#man-custom-indices","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Conventionally, Julia's arrays are indexed starting at 1, whereas some other languages start numbering at 0, and yet others (e.g., Fortran) allow you to specify arbitrary starting indices.  While there is much merit in picking a standard (i.e., 1 for Julia), there are some algorithms which simplify considerably if you can index outside the range 1:size(A,d) (and not just 0:size(A,d)-1, either). To facilitate such computations, Julia supports arrays with arbitrary indices.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"The purpose of this page is to address the question, \"what do I have to do to support such arrays in my own code?\"  First, let's address the simplest case: if you know that your code will never need to handle arrays with unconventional indexing, hopefully the answer is \"nothing.\" Old code, on conventional arrays, should function essentially without alteration as long as it was using the exported interfaces of Julia. If you find it more convenient to just force your users to supply traditional arrays where indexing starts at one, you can add","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Base.require_one_based_indexing(arrays...)","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"where arrays... is a list of the array objects that you wish to check for anything that violates 1-based indexing.","category":"page"},{"location":"devdocs/offset-arrays.html#Generalizing-existing-code","page":"Arrays with custom indices","title":"Generalizing existing code","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"As an overview, the steps are:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"replace many uses of size with axes\nreplace 1:length(A) with eachindex(A), or in some cases LinearIndices(A)\nreplace explicit allocations like Array{Int}(undef, size(B)) with similar(Array{Int}, axes(B))","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"These are described in more detail below.","category":"page"},{"location":"devdocs/offset-arrays.html#Things-to-watch-out-for","page":"Arrays with custom indices","title":"Things to watch out for","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Because unconventional indexing breaks many people's assumptions that all arrays start indexing with 1, there is always the chance that using such arrays will trigger errors. The most frustrating bugs would be incorrect results or segfaults (total crashes of Julia). For example, consider the following function:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"function mycopy!(dest::AbstractVector, src::AbstractVector)\n    length(dest) == length(src) || throw(DimensionMismatch(\"vectors must match\"))\n    # OK, now we're safe to use @inbounds, right? (not anymore!)\n    for i = 1:length(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"This code implicitly assumes that vectors are indexed from 1; if dest starts at a different index than src, there is a chance that this code would trigger a segfault. (If you do get segfaults, to help locate the cause try running julia with the option --check-bounds=yes.)","category":"page"},{"location":"devdocs/offset-arrays.html#Using-axes-for-bounds-checks-and-loop-iteration","page":"Arrays with custom indices","title":"Using axes for bounds checks and loop iteration","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"axes(A) (reminiscent of size(A)) returns a tuple of AbstractUnitRange{<:Integer} objects, specifying the range of valid indices along each dimension of A.  When A has unconventional indexing, the ranges may not start at 1.  If you just want the range for a particular dimension d, there is axes(A, d).","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Base implements a custom range type, OneTo, where OneTo(n) means the same thing as 1:n but in a form that guarantees (via the type system) that the lower index is 1. For any new AbstractArray type, this is the default returned by axes, and it indicates that this array type uses \"conventional\" 1-based indexing.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"For bounds checking, note that there are dedicated functions checkbounds and checkindex which can sometimes simplify such tests.","category":"page"},{"location":"devdocs/offset-arrays.html#Linear-indexing-(LinearIndices)","page":"Arrays with custom indices","title":"Linear indexing (LinearIndices)","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Some algorithms are most conveniently (or efficiently) written in terms of a single linear index, A[i] even if A is multi-dimensional. Regardless of the array's native indices, linear indices always range from 1:length(A). However, this raises an ambiguity for one-dimensional arrays (a.k.a., AbstractVector): does v[i] mean linear indexing , or Cartesian indexing with the array's native indices?","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"For this reason, your best option may be to iterate over the array with eachindex(A), or, if you require the indices to be sequential integers, to get the index range by calling LinearIndices(A). This will return axes(A, 1) if A is an AbstractVector, and the equivalent of 1:length(A) otherwise.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"By this definition, 1-dimensional arrays always use Cartesian indexing with the array's native indices. To help enforce this, it's worth noting that the index conversion functions will throw an error if shape indicates a 1-dimensional array with unconventional indexing (i.e., is a Tuple{UnitRange} rather than a tuple of OneTo). For arrays with conventional indexing, these functions continue to work the same as always.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Using axes and LinearIndices, here is one way you could rewrite mycopy!:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"function mycopy!(dest::AbstractVector, src::AbstractVector)\n    axes(dest) == axes(src) || throw(DimensionMismatch(\"vectors must match\"))\n    for i in LinearIndices(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend","category":"page"},{"location":"devdocs/offset-arrays.html#Allocating-storage-using-generalizations-of-similar","page":"Arrays with custom indices","title":"Allocating storage using generalizations of similar","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Storage is often allocated with Array{Int}(undef, dims) or similar(A, args...). When the result needs to match the indices of some other array, this may not always suffice. The generic replacement for such patterns is to use similar(storagetype, shape).  storagetype indicates the kind of underlying \"conventional\" behavior you'd like, e.g., Array{Int} or BitArray or even dims->zeros(Float32, dims) (which would allocate an all-zeros array). shape is a tuple of Integer or AbstractUnitRange values, specifying the indices that you want the result to use. Note that a convenient way of producing an all-zeros array that matches the indices of A is simply zeros(A).","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Let's walk through a couple of explicit examples. First, if A has conventional indices, then similar(Array{Int}, axes(A)) would end up calling Array{Int}(undef, size(A)), and thus return an array.  If A is an AbstractArray type with unconventional indexing, then similar(Array{Int}, axes(A)) should return something that \"behaves like\" an Array{Int} but with a shape (including indices) that matches A.  (The most obvious implementation is to allocate an Array{Int}(undef, size(A)) and then \"wrap\" it in a type that shifts the indices.)","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Note also that similar(Array{Int}, (axes(A, 2),)) would allocate an AbstractVector{Int} (i.e., 1-dimensional array) that matches the indices of the columns of A.","category":"page"},{"location":"devdocs/offset-arrays.html#Writing-custom-array-types-with-non-1-indexing","page":"Arrays with custom indices","title":"Writing custom array types with non-1 indexing","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Most of the methods you'll need to define are standard for any AbstractArray type, see Abstract Arrays. This page focuses on the steps needed to define unconventional indexing.","category":"page"},{"location":"devdocs/offset-arrays.html#Custom-AbstractUnitRange-types","page":"Arrays with custom indices","title":"Custom AbstractUnitRange types","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"If you're writing a non-1 indexed array type, you will want to specialize axes so it returns a UnitRange, or (perhaps better) a custom AbstractUnitRange.  The advantage of a custom type is that it \"signals\" the allocation type for functions like similar. If we're writing an array type for which indexing will start at 0, we likely want to begin by creating a new AbstractUnitRange, ZeroRange, where ZeroRange(n) is equivalent to 0:n-1.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"In general, you should probably not export ZeroRange from your package: there may be other packages that implement their own ZeroRange, and having multiple distinct ZeroRange types is (perhaps counterintuitively) an advantage: ModuleA.ZeroRange indicates that similar should create a ModuleA.ZeroArray, whereas ModuleB.ZeroRange indicates a ModuleB.ZeroArray type.  This design allows peaceful coexistence among many different custom array types.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Note that the Julia package CustomUnitRanges.jl can sometimes be used to avoid the need to write your own ZeroRange type.","category":"page"},{"location":"devdocs/offset-arrays.html#Specializing-axes","page":"Arrays with custom indices","title":"Specializing axes","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Once you have your AbstractUnitRange type, then specialize axes:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Base.axes(A::ZeroArray) = map(n->ZeroRange(n), A.size)","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"where here we imagine that ZeroArray has a field called size (there would be other ways to implement this).","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"In some cases, the fallback definition for axes(A, d):","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"axes(A::AbstractArray{T,N}, d) where {T,N} = d <= N ? axes(A)[d] : OneTo(1)","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"may not be what you want: you may need to specialize it to return something other than OneTo(1) when d > ndims(A).  Likewise, in Base there is a dedicated function axes1 which is equivalent to axes(A, 1) but which avoids checking (at runtime) whether ndims(A) > 0. (This is purely a performance optimization.)  It is defined as:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"axes1(A::AbstractArray{T,0}) where {T} = OneTo(1)\naxes1(A::AbstractArray) = axes(A)[1]","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"If the first of these (the zero-dimensional case) is problematic for your custom array type, be sure to specialize it appropriately.","category":"page"},{"location":"devdocs/offset-arrays.html#Specializing-similar","page":"Arrays with custom indices","title":"Specializing similar","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Given your custom ZeroRange type, then you should also add the following two specializations for similar:","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"function Base.similar(A::AbstractArray, T::Type, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nfunction Base.similar(f::Union{Function,DataType}, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Both of these should allocate your custom array type.","category":"page"},{"location":"devdocs/offset-arrays.html#Specializing-reshape","page":"Arrays with custom indices","title":"Specializing reshape","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Optionally, define a method","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Base.reshape(A::AbstractArray, shape::Tuple{ZeroRange,Vararg{ZeroRange}}) = ...","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"and you can reshape an array so that the result has custom indices.","category":"page"},{"location":"devdocs/offset-arrays.html#For-objects-that-mimic-AbstractArray-but-are-not-subtypes","page":"Arrays with custom indices","title":"For objects that mimic AbstractArray but are not subtypes","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"has_offset_axes depends on having axes defined for the objects you call it on. If there is some reason you don't have an axes method defined for your object, consider defining a method","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"Base.has_offset_axes(obj::MyNon1IndexedArraylikeObject) = true","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"This will allow code that assumes 1-based indexing to detect a problem and throw a helpful error, rather than returning incorrect results or segfaulting julia.","category":"page"},{"location":"devdocs/offset-arrays.html#Catching-errors","page":"Arrays with custom indices","title":"Catching errors","text":"","category":"section"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"If your new array type triggers errors in other code, one helpful debugging step can be to comment out @boundscheck in your getindex and setindex! implementation. This will ensure that every element access checks bounds. Or, restart julia with --check-bounds=yes.","category":"page"},{"location":"devdocs/offset-arrays.html","page":"Arrays with custom indices","title":"Arrays with custom indices","text":"In some cases it may also be helpful to temporarily disable size and length for your new array type, since code that makes incorrect assumptions frequently uses these functions.","category":"page"},{"location":"devdocs/locks.html#Proper-maintenance-and-care-of-multi-threading-locks","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"","category":"section"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following strategies are used to ensure that the code is dead-lock free (generally by addressing the 4th Coffman condition: circular wait).","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"structure code such that only one lock will need to be acquired at a time\nalways acquire shared locks in the same order, as given by the table below\navoid constructs that expect to need unrestricted recursion","category":"page"},{"location":"devdocs/locks.html#Locks","page":"Proper maintenance and care of multi-threading locks","title":"Locks","text":"","category":"section"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Below are all of the locks that exist in the system and the mechanisms for using them that avoid the potential for deadlocks (no Ostrich algorithm allowed here):","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following are definitely leaf locks (level 1), and must not try to acquire any other lock:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"safepoint\nNote that this lock is acquired implicitly by JL_LOCK and JL_UNLOCK. use the _NOGC variants to avoid that for level 1 locks.While holding this lock, the code must not do any allocation or hit any safepoints. Note that there are safepoints when doing allocation, enabling / disabling GC, entering / restoring exception frames, and taking / releasing locks.\nshared_map\nfinalizers\npagealloc\ngcpermlock\nflisp\njlinstackwalk (Win32)\nResourcePool<?>::mutex\nRLST_mutex\nllvmprintingmutex\njllockedstream::mutex\ndebuginfo_asyncsafe\ninferencetimingmutex\nExecutionEngine::SessionLock\nflisp itself is already threadsafe, this lock only protects the jl_ast_context_list_t pool likewise, the ResourcePool<?>::mutexes just protect the associated resource pool","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is a leaf lock (level 2), and only acquires level 1 locks (safepoint) internally:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"globalrootslock\nModule->lock\nJLDebuginfoPlugin::PluginMutex\nnewlyinferredmutex","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is a level 3 lock, which can only acquire level 1 or level 2 locks internally:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Method->writelock\ntypecache","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is a level 4 lock, which can only recurse to acquire level 1, 2, or 3 locks:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"MethodTable->writelock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"No Julia code may be called while holding a lock above this point.","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"orc::ThreadSafeContext (TSCtx) locks occupy a special spot in the locking hierarchy. They are used to protect LLVM's global non-threadsafe state, but there may be an arbitrary number of them. By default, all of these locks may be treated as level 5 locks for the purposes of comparing with the rest of the hierarchy. Acquiring a TSCtx should only be done from the JIT's pool of TSCtx's, and all locks on that TSCtx should be released prior to returning it to the pool. If multiple TSCtx locks must be acquired at the same time (due to recursive compilation), then locks should be acquired in the order that the TSCtxs were borrowed from the pool.","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is a level 5 lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"JuliaOJIT::EmissionMutex","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following are a level 6 lock, which can only recurse to acquire locks at lower levels:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"codegen\njlmodulesmutex","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is an almost root lock (level end-1), meaning only the root look may be held when trying to acquire it:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"typeinf\nthis one is perhaps one of the most tricky ones, since type-inference can be invoked from many pointscurrently the lock is merged with the codegen lock, since they call each other recursively","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following lock synchronizes IO operation. Be aware that doing any I/O (for example, printing warning messages or debug information) while holding any other lock listed above may result in pernicious and hard-to-find deadlocks. BE VERY CAREFUL!","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"iolock\nIndividual ThreadSynchronizers locks\nthis may continue to be held after releasing the iolock, or acquired without it, but be very careful to never attempt to acquire the iolock while holding it\nLibdl.LazyLibrary lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following is the root lock, meaning no other lock shall be held when trying to acquire it:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"toplevel\nthis should be held while attempting a top-level action (such as making a new type or defining a new method): trying to obtain this lock inside a staged function will cause a deadlock condition!additionally, it's unclear if any code can safely run in parallel with an arbitrary toplevel expression, so it may require all threads to get to a safepoint first","category":"page"},{"location":"devdocs/locks.html#Broken-Locks","page":"Proper maintenance and care of multi-threading locks","title":"Broken Locks","text":"","category":"section"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"The following locks are broken:","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"toplevel\ndoesn't exist right nowfix: create it\nModule->lock\nThis is vulnerable to deadlocks since it can't be certain it is acquired in sequence. Some operations (such as import_module) are missing a lock.fix: replace with jl_modules_mutex?\nloading.jl: require and register_root_module\nThis file potentially has numerous problems.fix: needs locks","category":"page"},{"location":"devdocs/locks.html#Shared-Global-Data-Structures","page":"Proper maintenance and care of multi-threading locks","title":"Shared Global Data Structures","text":"","category":"section"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"These data structures each need locks due to being shared mutable global state. It is the inverse list for the above lock priority list. This list does not include level 1 leaf resources due to their simplicity.","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"MethodTable modifications (def, cache) : MethodTable->writelock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Type declarations : toplevel lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Type application : typecache lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Global variable tables : Module->lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Module serializer : toplevel lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"JIT & type-inference : codegen lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"MethodInstance/CodeInstance updates : Method->writelock, codegen lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"These are set at construction and immutable:\nspecTypes\nsparam_vals\ndef\nowner","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"These are set by jl_type_infer (while holding codegen lock):\ncache\nrettype\ninferred","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"    * valid ages","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"inInference flag:\noptimization to quickly avoid recurring into jl_type_infer while it is already running\nactual state (of setting inferred, then fptr) is protected by codegen lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Function pointers:\nthese transition once, from NULL to a value, while the codegen lock is held\nCode-generator cache (the contents of functionObjectsDecls):\nthese can transition multiple times, but only while the codegen lock is held\nit is valid to use old version of this, or block for new versions of this, so races are benign, as long as the code is careful not to reference other data in the method instance (such as rettype) and assume it is coordinated, unless also holding the codegen lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"LLVMContext : codegen lock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"Method : Method->writelock","category":"page"},{"location":"devdocs/locks.html","page":"Proper maintenance and care of multi-threading locks","title":"Proper maintenance and care of multi-threading locks","text":"roots array (serializer and codegen)\ninvoke / specializations / tfunc modifications","category":"page"},{"location":"manual/unicode-input.html#Unicode-Input","page":"Unicode Input","title":"Unicode Input","text":"","category":"section"},{"location":"manual/unicode-input.html","page":"Unicode Input","title":"Unicode Input","text":"The following table lists Unicode characters that can be entered via tab completion of LaTeX-like abbreviations in the Julia REPL (and in various other editing environments).  You can also get information on how to type a symbol by entering it in the REPL help, i.e. by typing ? and then entering the symbol in the REPL (e.g., by copy-paste from somewhere you saw the symbol).","category":"page"},{"location":"manual/unicode-input.html","page":"Unicode Input","title":"Unicode Input","text":"warning: Warning\nThis table may appear to contain missing characters in the second column, or even show characters that are inconsistent with the characters as they are rendered in the Julia REPL. In these cases, users are strongly advised to check their choice of fonts in their browser and REPL environment, as there are known issues with glyphs in many fonts.","category":"page"},{"location":"manual/unicode-input.html","page":"Unicode Input","title":"Unicode Input","text":"#\n# Generate a table containing all LaTeX and Emoji tab completions available in the REPL.\n#\nimport REPL, Markdown\nconst NBSP = '\\u00A0'\n\nfunction tab_completions(symbols...)\n    completions = Dict{String, Vector{String}}()\n    for each in symbols, (k, v) in each\n        completions[v] = push!(get!(completions, v, String[]), k)\n    end\n    return completions\nend\n\nfunction unicode_data()\n    file = normpath(@__DIR__, \"..\", \"..\", \"..\", \"..\", \"..\", \"doc\", \"UnicodeData.txt\")\n    names = Dict{UInt32, String}()\n    open(file) do unidata\n        for line in readlines(unidata)\n            id, name, desc = split(line, \";\")[[1, 2, 11]]\n            codepoint = parse(UInt32, \"0x$id\")\n            names[codepoint] = titlecase(lowercase(\n                name == \"\" ? desc : desc == \"\" ? name : \"$name / $desc\"))\n        end\n    end\n    return names\nend\n\n# Surround combining characters with no-break spaces (i.e '\\u00A0'). Follows the same format\n# for how unicode is displayed on the unicode.org website:\n# https://util.unicode.org/UnicodeJsps/character.jsp?a=0300\nfunction fix_combining_chars(char)\n    cat = Base.Unicode.category_code(char)\n    return cat == 6 || cat == 8 ? \"$NBSP$char$NBSP\" : \"$char\"\nend\n\nfunction table_entries(completions, unicode_dict)\n    entries = Any[Any[\n        [\"Code point(s)\"],\n        [\"Character(s)\"],\n        [\"Tab completion sequence(s)\"],\n        [\"Unicode name(s)\"],\n    ]]\n    for (chars, inputs) in sort!(collect(completions), by = first)\n        code_points, unicode_names, characters = String[], String[], String[]\n        for char in chars\n            push!(code_points, \"U+$(uppercase(string(UInt32(char), base = 16, pad = 5)))\")\n            push!(unicode_names, get(unicode_dict, UInt32(char), \"(No Unicode name)\"))\n            push!(characters, isempty(characters) ? fix_combining_chars(char) : \"$char\")\n        end\n        inputs_md = []\n        for (i, input) in enumerate(inputs)\n            i > 1 && push!(inputs_md, \", \")\n            push!(inputs_md, Markdown.Code(\"\", input))\n        end\n        push!(entries, [\n            [join(code_points, \" + \")],\n            [join(characters)],\n            inputs_md,\n            [join(unicode_names, \" + \")],\n        ])\n    end\n    table = Markdown.Table(entries, [:l, :c, :l, :l])\n    # We also need to wrap the Table in a Markdown.MD \"document\"\n    return Markdown.MD([table])\nend\n\ntable_entries(\n    tab_completions(\n        REPL.REPLCompletions.latex_symbols,\n        REPL.REPLCompletions.emoji_symbols\n    ),\n    unicode_data()\n)","category":"page"},{"location":"manual/types.html#man-types","page":"Types","title":"Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type systems have traditionally fallen into two quite different camps: static type systems, where every program expression must have a type computable before the execution of the program, and dynamic type systems, where nothing is known about types until run time, when the actual values manipulated by the program are available. Object orientation allows some flexibility in statically typed languages by letting code be written without the precise types of values being known at compile time. The ability to write code that can operate on different types is called polymorphism. All code in classic dynamically typed languages is polymorphic: only by explicitly checking types, or when objects fail to support operations at run-time, are the types of any values ever restricted.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Julia's type system is dynamic, but gains some of the advantages of static type systems by making it possible to indicate that certain values are of specific types. This can be of great assistance in generating efficient code, but even more significantly, it allows method dispatch on the types of function arguments to be deeply integrated with the language. Method dispatch is explored in detail in Methods, but is rooted in the type system presented here.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The default behavior in Julia when types are omitted is to allow values to be of any type. Thus, one can write many useful Julia functions without ever explicitly using types. When additional expressiveness is needed, however, it is easy to gradually introduce explicit type annotations into previously \"untyped\" code. Adding annotations serves three primary purposes: to take advantage of Julia's powerful multiple-dispatch mechanism,  to improve human readability, and to catch programmer errors.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Describing Julia in the lingo of type systems, it is: dynamic, nominative and parametric. Generic types can be parameterized, and the hierarchical relationships between types are explicitly declared, rather than implied by compatible structure. One particularly distinctive feature of Julia's type system is that concrete types may not subtype each other: all concrete types are final and may only have abstract types as their supertypes. While this might at first seem unduly restrictive, it has many beneficial consequences with surprisingly few drawbacks. It turns out that being able to inherit behavior is much more important than being able to inherit structure, and inheriting both causes significant difficulties in traditional object-oriented languages. Other high-level aspects of Julia's type system that should be mentioned up front are:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"There is no division between object and non-object values: all values in Julia are true objects having a type that belongs to a single, fully connected type graph, all nodes of which are equally first-class as types.\nThere is no meaningful concept of a \"compile-time type\": the only type a value has is its actual type when the program is running. This is called a \"run-time type\" in object-oriented languages where the combination of static compilation with polymorphism makes this distinction significant.\nOnly values, not variables, have types – variables are simply names bound to values, although for simplicity we may say \"type of a variable\" as shorthand for \"type of the value to which a variable refers\".\nBoth abstract and concrete types can be parameterized by other types. They can also be parameterized by symbols, by values of any type for which isbits returns true (essentially, things like numbers and bools that are stored like C types or structs with no pointers to other objects), and also by tuples thereof. Type parameters may be omitted when they do not need to be referenced or restricted.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Julia's type system is designed to be powerful and expressive, yet clear, intuitive and unobtrusive. Many Julia programmers may never feel the need to write code that explicitly uses types. Some kinds of programming, however, become clearer, simpler, faster and more robust with declared types.","category":"page"},{"location":"manual/types.html#Type-Declarations","page":"Types","title":"Type Declarations","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The :: operator can be used to attach type annotations to expressions and variables in programs. There are two primary reasons to do this:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"As an assertion to help confirm that your program works the way you expect, and\nTo provide extra type information to the compiler, which can then improve performance in some cases.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"When appended to an expression computing a value, the :: operator is read as \"is an instance of\". It can be used anywhere to assert that the value of the expression on the left is an instance of the type on the right. When the type on the right is concrete, the value on the left must have that type as its implementation – recall that all concrete types are final, so no implementation is a subtype of any other. When the type is abstract, it suffices for the value to be implemented by a concrete type that is a subtype of the abstract type. If the type assertion is not true, an exception is thrown, otherwise, the left-hand value is returned:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> (1+2)::AbstractFloat\nERROR: TypeError: in typeassert, expected AbstractFloat, got a value of type Int64\n\njulia> (1+2)::Int\n3","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This allows a type assertion to be attached to any expression in-place.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"When appended to a variable on the left-hand side of an assignment, or as part of a local declaration, the :: operator means something a bit different: it declares the variable to always have the specified type, like a type declaration in a statically-typed language such as C. Every value assigned to the variable will be converted to the declared type using convert:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> function foo()\n           x::Int8 = 100\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> x = foo()\n100\n\njulia> typeof(x)\nInt8","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This feature is useful for avoiding performance \"gotchas\" that could occur if one of the assignments to a variable changed its type unexpectedly.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This \"declaration\" behavior only occurs in specific contexts:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"local x::Int8  # in a local declaration\nx::Int8 = 10   # as the left-hand side of an assignment","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"and applies to the whole current scope, even before the declaration.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"As of Julia 1.8, type declarations can now be used in global scope i.e. type annotations can be added to global variables to make accessing them type stable.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> x::Int = 10\n10\n\njulia> x = 3.5\nERROR: InexactError: Int64(3.5)\n\njulia> function foo(y)\n           global x = 15.8    # throws an error when foo is called\n           return x + y\n       end\nfoo (generic function with 1 method)\n\njulia> foo(10)\nERROR: InexactError: Int64(15.8)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Declarations can also be attached to function definitions:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"function sinc(x)::Float64\n    if x == 0\n        return 1\n    end\n    return sin(pi*x)/(pi*x)\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Returning from this function behaves just like an assignment to a variable with a declared type: the value is always converted to Float64.","category":"page"},{"location":"manual/types.html#man-abstract-types","page":"Types","title":"Abstract Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Abstract types cannot be instantiated, and serve only as nodes in the type graph, thereby describing sets of related concrete types: those concrete types which are their descendants. We begin with abstract types even though they have no instantiation because they are the backbone of the type system: they form the conceptual hierarchy which makes Julia's type system more than just a collection of object implementations.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Recall that in Integers and Floating-Point Numbers, we introduced a variety of concrete types of numeric values: Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, Float16, Float32, and Float64. Although they have different representation sizes, Int8, Int16, Int32, Int64 and Int128 all have in common that they are signed integer types. Likewise UInt8, UInt16, UInt32, UInt64 and UInt128 are all unsigned integer types, while Float16, Float32 and Float64 are distinct in being floating-point types rather than integers. It is common for a piece of code to make sense, for example, only if its arguments are some kind of integer, but not really depend on what particular kind of integer. For example, the greatest common denominator algorithm works for all kinds of integers, but will not work for floating-point numbers. Abstract types allow the construction of a hierarchy of types, providing a context into which concrete types can fit. This allows you, for example, to easily program to any type that is an integer, without restricting an algorithm to a specific type of integer.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Abstract types are declared using the abstract type keyword. The general syntaxes for declaring an abstract type are:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"abstract type «name» end\nabstract type «name» <: «supertype» end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The abstract type keyword introduces a new abstract type, whose name is given by «name». This name can be optionally followed by <: and an already-existing type, indicating that the newly declared abstract type is a subtype of this \"parent\" type.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"When no supertype is given, the default supertype is Any – a predefined abstract type that all objects are instances of and all types are subtypes of. In type theory, Any is commonly called \"top\" because it is at the apex of the type graph. Julia also has a predefined abstract \"bottom\" type, at the nadir of the type graph, which is written as Union{}. It is the exact opposite of Any: no object is an instance of Union{} and all types are supertypes of Union{}.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Let's consider some of the abstract types that make up Julia's numerical hierarchy:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"abstract type Number end\nabstract type Real          <: Number end\nabstract type AbstractFloat <: Real end\nabstract type Integer       <: Real end\nabstract type Signed        <: Integer end\nabstract type Unsigned      <: Integer end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The Number type is a direct child type of Any, and Real is its child. In turn, Real has two children (it has more, but only two are shown here; we'll get to the others later): Integer and AbstractFloat, separating the world into representations of integers and representations of real numbers. Representations of real numbers include floating-point types, but also include other types, such as rationals. AbstractFloat includes only floating-point representations of real numbers. Integers are further subdivided into Signed and Unsigned varieties.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The <: operator in general means \"is a subtype of\", and, used in declarations like those above, declares the right-hand type to be an immediate supertype of the newly declared type. It can also be used in expressions as a subtype operator which returns true when its left operand is a subtype of its right operand:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Integer <: Number\ntrue\n\njulia> Integer <: AbstractFloat\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An important use of abstract types is to provide default implementations for concrete types. To give a simple example, consider:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"function myplus(x,y)\n    x+y\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The first thing to note is that the above argument declarations are equivalent to x::Any and y::Any. When this function is invoked, say as myplus(2,5), the dispatcher chooses the most specific method named myplus that matches the given arguments. (See Methods for more information on multiple dispatch.)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Assuming no method more specific than the above is found, Julia next internally defines and compiles a method called myplus specifically for two Int arguments based on the generic function given above, i.e., it implicitly defines and compiles:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"function myplus(x::Int,y::Int)\n    x+y\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"and finally, it invokes this specific method.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Thus, abstract types allow programmers to write generic functions that can later be used as the default method by many combinations of concrete types. Thanks to multiple dispatch, the programmer has full control over whether the default or more specific method is used.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An important point to note is that there is no loss in performance if the programmer relies on a function whose arguments are abstract types, because it is recompiled for each tuple of concrete argument types with which it is invoked. (There may be a performance issue, however, in the case of function arguments that are containers of abstract types; see Performance Tips.)","category":"page"},{"location":"manual/types.html#Primitive-Types","page":"Types","title":"Primitive Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"warning: Warning\nIt is almost always preferable to wrap an existing primitive type in a new composite type than to define your own primitive type.This functionality exists to allow Julia to bootstrap the standard primitive types that LLVM supports. Once they are defined, there is very little reason to define more.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A primitive type is a concrete type whose data consists of plain old bits. Classic examples of primitive types are integers and floating-point values. Unlike most languages, Julia lets you declare your own primitive types, rather than providing only a fixed set of built-in ones. In fact, the standard primitive types are all defined in the language itself:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"primitive type Float16 <: AbstractFloat 16 end\nprimitive type Float32 <: AbstractFloat 32 end\nprimitive type Float64 <: AbstractFloat 64 end\n\nprimitive type Bool <: Integer 8 end\nprimitive type Char <: AbstractChar 32 end\n\nprimitive type Int8    <: Signed   8 end\nprimitive type UInt8   <: Unsigned 8 end\nprimitive type Int16   <: Signed   16 end\nprimitive type UInt16  <: Unsigned 16 end\nprimitive type Int32   <: Signed   32 end\nprimitive type UInt32  <: Unsigned 32 end\nprimitive type Int64   <: Signed   64 end\nprimitive type UInt64  <: Unsigned 64 end\nprimitive type Int128  <: Signed   128 end\nprimitive type UInt128 <: Unsigned 128 end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The general syntaxes for declaring a primitive type are:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"primitive type «name» «bits» end\nprimitive type «name» <: «supertype» «bits» end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The number of bits indicates how much storage the type requires and the name gives the new type a name. A primitive type can optionally be declared to be a subtype of some supertype. If a supertype is omitted, then the type defaults to having Any as its immediate supertype. The declaration of Bool above therefore means that a boolean value takes eight bits to store, and has Integer as its immediate supertype. Currently, only sizes that are multiples of 8 bits are supported and you are likely to experience LLVM bugs with sizes other than those used above. Therefore, boolean values, although they really need just a single bit, cannot be declared to be any smaller than eight bits.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The types Bool, Int8 and UInt8 all have identical representations: they are eight-bit chunks of memory. Since Julia's type system is nominative, however, they are not interchangeable despite having identical structure. A fundamental difference between them is that they have different supertypes: Bool's direct supertype is Integer, Int8's is Signed, and UInt8's is Unsigned. All other differences between Bool, Int8, and UInt8 are matters of behavior – the way functions are defined to act when given objects of these types as arguments. This is why a nominative type system is necessary: if structure determined type, which in turn dictates behavior, then it would be impossible to make Bool behave any differently than Int8 or UInt8.","category":"page"},{"location":"manual/types.html#Composite-Types","page":"Types","title":"Composite Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Composite types are called records, structs, or objects in various languages. A composite type is a collection of named fields, an instance of which can be treated as a single value. In many languages, composite types are the only kind of user-definable type, and they are by far the most commonly used user-defined type in Julia as well.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In mainstream object oriented languages, such as C++, Java, Python and Ruby, composite types also have named functions associated with them, and the combination is called an \"object\". In purer object-oriented languages, such as Ruby or Smalltalk, all values are objects whether they are composites or not. In less pure object oriented languages, including C++ and Java, some values, such as integers and floating-point values, are not objects, while instances of user-defined composite types are true objects with associated methods. In Julia, all values are objects, but functions are not bundled with the objects they operate on. This is necessary since Julia chooses which method of a function to use by multiple dispatch, meaning that the types of all of a function's arguments are considered when selecting a method, rather than just the first one (see Methods for more information on methods and dispatch). Thus, it would be inappropriate for functions to \"belong\" to only their first argument. Organizing methods into function objects rather than having named bags of methods \"inside\" each object ends up being a highly beneficial aspect of the language design.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Composite types are introduced with the struct keyword followed by a block of field names, optionally annotated with types using the :: operator:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct Foo\n           bar\n           baz::Int\n           qux::Float64\n       end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Fields with no type annotation default to Any, and can accordingly hold any type of value.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"New objects of type Foo are created by applying the Foo type object like a function to values for its fields:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> foo = Foo(\"Hello, world.\", 23, 1.5)\nFoo(\"Hello, world.\", 23, 1.5)\n\njulia> typeof(foo)\nFoo","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"When a type is applied like a function it is called a constructor. Two constructors are generated automatically (these are called default constructors). One accepts any arguments and calls convert to convert them to the types of the fields, and the other accepts arguments that match the field types exactly. The reason both of these are generated is that this makes it easier to add new definitions without inadvertently replacing a default constructor.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since the bar field is unconstrained in type, any value will do. However, the value for baz must be convertible to Int:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Foo((), 23.5, 1)\nERROR: InexactError: Int64(23.5)\nStacktrace:\n[...]","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"You may find a list of field names using the fieldnames function.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> fieldnames(Foo)\n(:bar, :baz, :qux)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"You can access the field values of a composite object using the traditional foo.bar notation:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> foo.bar\n\"Hello, world.\"\n\njulia> foo.baz\n23\n\njulia> foo.qux\n1.5","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Composite objects declared with struct are immutable; they cannot be modified after construction. This may seem odd at first, but it has several advantages:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"It can be more efficient. Some structs can be packed efficiently into arrays, and in some cases the compiler is able to avoid allocating immutable objects entirely.\nIt is not possible to violate the invariants provided by the type's constructors.\nCode using immutable objects can be easier to reason about.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An immutable object might contain mutable objects, such as arrays, as fields. Those contained objects will remain mutable; only the fields of the immutable object itself cannot be changed to point to different objects.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Where required, mutable composite objects can be declared with the keyword mutable struct, to be discussed in the next section.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"If all the fields of an immutable structure are indistinguishable (===) then two immutable values containing those fields are also indistinguishable:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct X\n           a::Int\n           b::Float64\n       end\n\njulia> X(1, 2) === X(1, 2)\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"There is much more to say about how instances of composite types are created, but that discussion depends on both Parametric Types and on Methods, and is sufficiently important to be addressed in its own section: Constructors.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"For many user-defined types X, you may want to define a method Base.broadcastable(x::X) = Ref(x) so that instances of that type act as 0-dimensional \"scalars\" for broadcasting.","category":"page"},{"location":"manual/types.html#Mutable-Composite-Types","page":"Types","title":"Mutable Composite Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"If a composite type is declared with mutable struct instead of struct, then instances of it can be modified:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> mutable struct Bar\n           baz\n           qux::Float64\n       end\n\njulia> bar = Bar(\"Hello\", 1.5);\n\njulia> bar.qux = 2.0\n2.0\n\njulia> bar.baz = 1//2\n1//2","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An extra interface between the fields and the user can be provided through Instance Properties. This grants more control on what can be accessed and modified using the bar.baz notation.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In order to support mutation, such objects are generally allocated on the heap, and have stable memory addresses. A mutable object is like a little container that might hold different values over time, and so can only be reliably identified with its address. In contrast, an instance of an immutable type is associated with specific field values –- the field values alone tell you everything about the object. In deciding whether to make a type mutable, ask whether two instances with the same field values would be considered identical, or if they might need to change independently over time. If they would be considered identical, the type should probably be immutable.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"To recap, two essential properties define immutability in Julia:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"It is not permitted to modify the value of an immutable type.\nFor bits types this means that the bit pattern of a value once set will never change and that value is the identity of a bits type.\nFor composite  types, this means that the identity of the values of its fields will never change. When the fields are bits types, that means their bits will never change, for fields whose values are mutable types like arrays, that means the fields will always refer to the same mutable value even though that mutable value's content may itself be modified.\nAn object with an immutable type may be copied freely by the compiler since its immutability makes it impossible to programmatically distinguish between the original object and a copy.\nIn particular, this means that small enough immutable values like integers and floats are typically passed to functions in registers (or stack allocated).\nMutable values, on the other hand are heap-allocated and passed to functions as pointers to heap-allocated values except in cases where the compiler is sure that there's no way to tell that this is not what is happening.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In cases where one or more fields of an otherwise mutable struct is known to be immutable, one can declare these fields as such using const as shown below. This enables some, but not all of the optimizations of immutable structs, and can be used to enforce invariants on the particular fields marked as const.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"compat: Julia 1.8\nconst annotating fields of mutable structs requires at least Julia 1.8.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> mutable struct Baz\n           a::Int\n           const b::Float64\n       end\n\njulia> baz = Baz(1, 1.5);\n\njulia> baz.a = 2\n2\n\njulia> baz.b = 2.0\nERROR: setfield!: const field .b of type Baz cannot be changed\n[...]","category":"page"},{"location":"manual/types.html#man-declared-types","page":"Types","title":"Declared Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The three kinds of types (abstract, primitive, composite) discussed in the previous sections are actually all closely related. They share the same key properties:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"They are explicitly declared.\nThey have names.\nThey have explicitly declared supertypes.\nThey may have parameters.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Because of these shared properties, these types are internally represented as instances of the same concept, DataType, which is the type of any of these types:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataType","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A DataType may be abstract or concrete. If it is concrete, it has a specified size, storage layout, and (optionally) field names. Thus a primitive type is a DataType with nonzero size, but no field names. A composite type is a DataType that has field names or is empty (zero size).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Every concrete value in the system is an instance of some DataType.","category":"page"},{"location":"manual/types.html#Type-Unions","page":"Types","title":"Type Unions","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A type union is a special abstract type which includes as objects all instances of any of its argument types, constructed using the special Union keyword:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> IntOrString = Union{Int,AbstractString}\nUnion{Int64, AbstractString}\n\njulia> 1 :: IntOrString\n1\n\njulia> \"Hello!\" :: IntOrString\n\"Hello!\"\n\njulia> 1.0 :: IntOrString\nERROR: TypeError: in typeassert, expected Union{Int64, AbstractString}, got a value of type Float64","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The compilers for many languages have an internal union construct for reasoning about types; Julia simply exposes it to the programmer. The Julia compiler is able to generate efficient code in the presence of Union types with a small number of types [1], by generating specialized code in separate branches for each possible type.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A particularly useful case of a Union type is Union{T, Nothing}, where T can be any type and Nothing is the singleton type whose only instance is the object nothing. This pattern is the Julia equivalent of Nullable, Option or Maybe types in other languages. Declaring a function argument or a field as Union{T, Nothing} allows setting it either to a value of type T, or to nothing to indicate that there is no value. See this FAQ entry for more information.","category":"page"},{"location":"manual/types.html#Parametric-Types","page":"Types","title":"Parametric Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An important and powerful feature of Julia's type system is that it is parametric: types can take parameters, so that type declarations actually introduce a whole family of new types – one for each possible combination of parameter values. There are many languages that support some version of generic programming, wherein data structures and algorithms to manipulate them may be specified without specifying the exact types involved. For example, some form of generic programming exists in ML, Haskell, Ada, Eiffel, C++, Java, C#, F#, and Scala, just to name a few. Some of these languages support true parametric polymorphism (e.g. ML, Haskell, Scala), while others support ad-hoc, template-based styles of generic programming (e.g. C++, Java). With so many different varieties of generic programming and parametric types in various languages, we won't even attempt to compare Julia's parametric types to other languages, but will instead focus on explaining Julia's system in its own right. We will note, however, that because Julia is a dynamically typed language and doesn't need to make all type decisions at compile time, many traditional difficulties encountered in static parametric type systems can be relatively easily handled.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"All declared types (the DataType variety) can be parameterized, with the same syntax in each case. We will discuss them in the following order: first, parametric composite types, then parametric abstract types, and finally parametric primitive types.","category":"page"},{"location":"manual/types.html#man-parametric-composite-types","page":"Types","title":"Parametric Composite Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type parameters are introduced immediately after the type name, surrounded by curly braces:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct Point{T}\n           x::T\n           y::T\n       end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This declaration defines a new parametric type, Point{T}, holding two \"coordinates\" of type T. What, one may ask, is T? Well, that's precisely the point of parametric types: it can be any type at all (or a value of any bits type, actually, although here it's clearly used as a type). Point{Float64} is a concrete type equivalent to the type defined by replacing T in the definition of Point with Float64. Thus, this single declaration actually declares an unlimited number of types: Point{Float64}, Point{AbstractString}, Point{Int64}, etc. Each of these is now a usable concrete type:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64}\nPoint{Float64}\n\njulia> Point{AbstractString}\nPoint{AbstractString}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The type Point{Float64} is a point whose coordinates are 64-bit floating-point values, while the type Point{AbstractString} is a \"point\" whose \"coordinates\" are string objects (see Strings).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Point itself is also a valid type object, containing all instances Point{Float64}, Point{AbstractString}, etc. as subtypes:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64} <: Point\ntrue\n\njulia> Point{AbstractString} <: Point\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Other types, of course, are not subtypes of it:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Float64 <: Point\nfalse\n\njulia> AbstractString <: Point\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Concrete Point types with different values of T are never subtypes of each other:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64} <: Point{Int64}\nfalse\n\njulia> Point{Float64} <: Point{Real}\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"warning: Warning\nThis last point is very important: even though Float64 <: Real we DO NOT have Point{Float64} <: Point{Real}.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In other words, in the parlance of type theory, Julia's type parameters are invariant, rather than being covariant (or even contravariant). This is for practical reasons: while any instance of Point{Float64} may conceptually be like an instance of Point{Real} as well, the two types have different representations in memory:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"An instance of Point{Float64} can be represented compactly and efficiently as an immediate pair of 64-bit values;\nAn instance of Point{Real} must be able to hold any pair of instances of Real. Since objects that are instances of Real can be of arbitrary size and structure, in practice an instance of Point{Real} must be represented as a pair of pointers to individually allocated Real objects.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The efficiency gained by being able to store Point{Float64} objects with immediate values is magnified enormously in the case of arrays: an Array{Float64} can be stored as a contiguous memory block of 64-bit floating-point values, whereas an Array{Real} must be an array of pointers to individually allocated Real objects – which may well be boxed 64-bit floating-point values, but also might be arbitrarily large, complex objects, which are declared to be implementations of the Real abstract type.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since Point{Float64} is not a subtype of Point{Real}, the following method can't be applied to arguments of type Point{Float64}:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"function norm(p::Point{Real})\n    sqrt(p.x^2 + p.y^2)\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A correct way to define a method that accepts all arguments of type Point{T} where T is a subtype of Real is:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"function norm(p::Point{<:Real})\n    sqrt(p.x^2 + p.y^2)\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"(Equivalently, one could define function norm(p::Point{T} where T<:Real) or function norm(p::Point{T}) where T<:Real; see UnionAll Types.)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"More examples will be discussed later in Methods.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"How does one construct a Point object? It is possible to define custom constructors for composite types, which will be discussed in detail in Constructors, but in the absence of any special constructor declarations, there are two default ways of creating new composite objects, one in which the type parameters are explicitly given and the other in which they are implied by the arguments to the object constructor.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since the type Point{Float64} is a concrete type equivalent to Point declared with Float64 in place of T, it can be applied as a constructor accordingly:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> p = Point{Float64}(1.0, 2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(p)\nPoint{Float64}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"For the default constructor, exactly one argument must be supplied for each field:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64}(1.0)\nERROR: MethodError: no method matching Point{Float64}(::Float64)\nThe type `Point{Float64}` exists, but no method is defined for this combination of argument types when trying to construct it.\n[...]\n\njulia> Point{Float64}(1.0, 2.0, 3.0)\nERROR: MethodError: no method matching Point{Float64}(::Float64, ::Float64, ::Float64)\nThe type `Point{Float64}` exists, but no method is defined for this combination of argument types when trying to construct it.\n[...]","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Only one default constructor is generated for parametric types, since overriding it is not possible. This constructor accepts any arguments and converts them to the field types.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In many cases, it is redundant to provide the type of Point object one wants to construct, since the types of arguments to the constructor call already implicitly provide type information. For that reason, you can also apply Point itself as a constructor, provided that the implied value of the parameter type T is unambiguous:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> p1 = Point(1.0,2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(p1)\nPoint{Float64}\n\njulia> p2 = Point(1,2)\nPoint{Int64}(1, 2)\n\njulia> typeof(p2)\nPoint{Int64}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In the case of Point, the type of T is unambiguously implied if and only if the two arguments to Point have the same type. When this isn't the case, the constructor will fail with a MethodError:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point(1,2.5)\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T\n   @ Main none:2\n\nStacktrace:\n[...]","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Constructor methods to appropriately handle such mixed cases can be defined, but that will not be discussed until later on in Constructors.","category":"page"},{"location":"manual/types.html#Parametric-Abstract-Types","page":"Types","title":"Parametric Abstract Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Parametric abstract type declarations declare a collection of abstract types, in much the same way:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> abstract type Pointy{T} end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"With this declaration, Pointy{T} is a distinct abstract type for each type or integer value of T. As with parametric composite types, each such instance is a subtype of Pointy:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Pointy{Int64} <: Pointy\ntrue\n\njulia> Pointy{1} <: Pointy\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Parametric abstract types are invariant, much as parametric composite types are:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Pointy{Float64} <: Pointy{Real}\nfalse\n\njulia> Pointy{Real} <: Pointy{Float64}\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The notation Pointy{<:Real} can be used to express the Julia analogue of a covariant type, while Pointy{>:Int} the analogue of a contravariant type, but technically these represent sets of types (see UnionAll Types).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Pointy{Float64} <: Pointy{<:Real}\ntrue\n\njulia> Pointy{Real} <: Pointy{>:Int}\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Much as plain old abstract types serve to create a useful hierarchy of types over concrete types, parametric abstract types serve the same purpose with respect to parametric composite types. We could, for example, have declared Point{T} to be a subtype of Pointy{T} as follows:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct Point{T} <: Pointy{T}\n           x::T\n           y::T\n       end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Given such a declaration, for each choice of T, we have Point{T} as a subtype of Pointy{T}:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64} <: Pointy{Float64}\ntrue\n\njulia> Point{Real} <: Pointy{Real}\ntrue\n\njulia> Point{AbstractString} <: Pointy{AbstractString}\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This relationship is also invariant:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Point{Float64} <: Pointy{Real}\nfalse\n\njulia> Point{Float64} <: Pointy{<:Real}\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"What purpose do parametric abstract types like Pointy serve? Consider if we create a point-like implementation that only requires a single coordinate because the point is on the diagonal line x = y:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct DiagPoint{T} <: Pointy{T}\n           x::T\n       end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Now both Point{Float64} and DiagPoint{Float64} are implementations of the Pointy{Float64} abstraction, and similarly for every other possible choice of type T. This allows programming to a common interface shared by all Pointy objects, implemented for both Point and DiagPoint. This cannot be fully demonstrated, however, until we have introduced methods and dispatch in the next section, Methods.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"There are situations where it may not make sense for type parameters to range freely over all possible types. In such situations, one can constrain the range of T like so:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> abstract type Pointy{T<:Real} end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"With such a declaration, it is acceptable to use any type that is a subtype of Real in place of T, but not types that are not subtypes of Real:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Pointy{Float64}\nPointy{Float64}\n\njulia> Pointy{Real}\nPointy{Real}\n\njulia> Pointy{AbstractString}\nERROR: TypeError: in Pointy, in T, expected T<:Real, got Type{AbstractString}\n\njulia> Pointy{1}\nERROR: TypeError: in Pointy, in T, expected T<:Real, got a value of type Int64","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type parameters for parametric composite types can be restricted in the same manner:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"struct Point{T<:Real} <: Pointy{T}\n    x::T\n    y::T\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"To give a real-world example of how all this parametric type machinery can be useful, here is the actual definition of Julia's Rational immutable type (except that we omit the constructor here for simplicity), representing an exact ratio of integers:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"struct Rational{T<:Integer} <: Real\n    num::T\n    den::T\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"It only makes sense to take ratios of integer values, so the parameter type T is restricted to being a subtype of Integer, and a ratio of integers represents a value on the real number line, so any Rational is an instance of the Real abstraction.","category":"page"},{"location":"manual/types.html#Tuple-Types","page":"Types","title":"Tuple Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Tuples are an abstraction of the arguments of a function – without the function itself. The salient aspects of a function's arguments are their order and their types. Therefore a tuple type is similar to a parameterized immutable type where each parameter is the type of one field. For example, a 2-element tuple type resembles the following immutable type:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"struct Tuple2{A,B}\n    a::A\n    b::B\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"However, there are three key differences:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Tuple types may have any number of parameters.\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore Tuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters are.\nTuples do not have field names; fields are only accessed by index.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Tuple values are written with parentheses and commas. When a tuple is constructed, an appropriate tuple type is generated on demand:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof((1,\"foo\",2.5))\nTuple{Int64, String, Float64}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Note the implications of covariance:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Tuple{Int,AbstractString} <: Tuple{Real,Any}\ntrue\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,Real}\nfalse\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,}\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Intuitively, this corresponds to the type of a function's arguments being a subtype of the function's signature (when the signature matches).","category":"page"},{"location":"manual/types.html#Vararg-Tuple-Types","page":"Types","title":"Vararg Tuple Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The last parameter of a tuple type can be the special value Vararg, which denotes any number of trailing elements:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> mytupletype = Tuple{AbstractString,Vararg{Int}}\nTuple{AbstractString, Vararg{Int64}}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Moreover Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are used to represent the arguments accepted by varargs methods (see Varargs Functions).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The special value Vararg{T,N} (when used as the last parameter of a tuple type) corresponds to exactly N elements of type T.  NTuple{N,T} is a convenient alias for Tuple{Vararg{T,N}}, i.e. a tuple type containing exactly N elements of type T.","category":"page"},{"location":"manual/types.html#Named-Tuple-Types","page":"Types","title":"Named Tuple Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Named tuples are instances of the NamedTuple type, which has two parameters: a tuple of symbols giving the field names, and a tuple type giving the field types. For convenience, NamedTuple types are printed using the @NamedTuple macro which provides a convenient struct-like syntax for declaring these types via key::Type declarations, where an omitted ::Type corresponds to ::Any.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof((a=1,b=\"hello\")) # prints in macro form\n@NamedTuple{a::Int64, b::String}\n\njulia> NamedTuple{(:a, :b), Tuple{Int64, String}} # long form of the type\n@NamedTuple{a::Int64, b::String}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The begin ... end form of the @NamedTuple macro allows the declarations to be split across multiple lines (similar to a struct declaration), but is otherwise equivalent:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> @NamedTuple begin\n           a::Int\n           b::String\n       end\n@NamedTuple{a::Int64, b::String}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A NamedTuple type can be used as a constructor, accepting a single tuple argument. The constructed NamedTuple type can be either a concrete type, with both parameters specified, or a type that specifies only field names:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> @NamedTuple{a::Float32,b::String}((1, \"\"))\n(a = 1.0f0, b = \"\")\n\njulia> NamedTuple{(:a, :b)}((1, \"\"))\n(a = 1, b = \"\")","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"If field types are specified, the arguments are converted. Otherwise the types of the arguments are used directly.","category":"page"},{"location":"manual/types.html#Parametric-Primitive-Types","page":"Types","title":"Parametric Primitive Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Primitive types can also be declared parametrically. For example, pointers are represented as primitive types which would be declared in Julia like this:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"# 32-bit system:\nprimitive type Ptr{T} 32 end\n\n# 64-bit system:\nprimitive type Ptr{T} 64 end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The slightly odd feature of these declarations as compared to typical parametric composite types, is that the type parameter T is not used in the definition of the type itself – it is just an abstract tag, essentially defining an entire family of types with identical structure, differentiated only by their type parameter. Thus, Ptr{Float64} and Ptr{Int64} are distinct types, even though they have identical representations. And of course, all specific pointer types are subtypes of the umbrella Ptr type:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Ptr{Float64} <: Ptr\ntrue\n\njulia> Ptr{Int64} <: Ptr\ntrue","category":"page"},{"location":"manual/types.html#UnionAll-Types","page":"Types","title":"UnionAll Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"We have said that a parametric type like Ptr acts as a supertype of all its instances (Ptr{Int64} etc.). How does this work? Ptr itself cannot be a normal data type, since without knowing the type of the referenced data the type clearly cannot be used for memory operations. The answer is that Ptr (or other parametric types like Array) is a different kind of type called a UnionAll type. Such a type expresses the iterated union of types for all values of some parameter.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"UnionAll types are usually written using the keyword where. For example Ptr could be more accurately written as Ptr{T} where T, meaning all values whose type is Ptr{T} for some value of T. In this context, the parameter T is also often called a \"type variable\" since it is like a variable that ranges over types. Each where introduces a single type variable, so these expressions are nested for types with multiple parameters, for example Array{T,N} where N where T.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The type application syntax A{B,C} requires A to be a UnionAll type, and first substitutes B for the outermost type variable in A. The result is expected to be another UnionAll type, into which C is then substituted. So A{B,C} is equivalent to A{B}{C}. This explains why it is possible to partially instantiate a type, as in Array{Float64}: the first parameter value has been fixed, but the second still ranges over all possible values. Using explicit where syntax, any subset of parameters can be fixed. For example, the type of all 1-dimensional arrays can be written as Array{T,1} where T.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type variables can be restricted with subtype relations. Array{T} where T<:Integer refers to all arrays whose element type is some kind of Integer. The syntax Array{<:Integer} is a convenient shorthand for Array{T} where T<:Integer. Type variables can have both lower and upper bounds. Array{T} where Int<:T<:Number refers to all arrays of Numbers that are able to contain Ints (since T must be at least as big as Int). The syntax where T>:Int also works to specify only the lower bound of a type variable, and Array{>:Int} is equivalent to Array{T} where T>:Int.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since where expressions nest, type variable bounds can refer to outer type variables. For example Tuple{T,Array{S}} where S<:AbstractArray{T} where T<:Real refers to 2-tuples whose first element is some Real, and whose second element is an Array of any kind of array whose element type contains the type of the first tuple element.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The where keyword itself can be nested inside a more complex declaration. For example, consider the two types created by the following declarations:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> const T1 = Array{Array{T, 1} where T, 1}\nVector{Vector} (alias for Array{Array{T, 1} where T, 1})\n\njulia> const T2 = Array{Array{T, 1}, 1} where T\nArray{Vector{T}, 1} where T","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type T1 defines a 1-dimensional array of 1-dimensional arrays; each of the inner arrays consists of objects of the same type, but this type may vary from one inner array to the next. On the other hand, type T2 defines a 1-dimensional array of 1-dimensional arrays all of whose inner arrays must have the same type.  Note that T2 is an abstract type, e.g., Array{Array{Int,1},1} <: T2, whereas T1 is a concrete type. As a consequence, T1 can be constructed with a zero-argument constructor a=T1() but T2 cannot.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"There is a convenient syntax for naming such types, similar to the short form of function definition syntax:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Vector{T} = Array{T, 1}","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This is equivalent to const Vector = Array{T,1} where T. Writing Vector{Float64} is equivalent to writing Array{Float64,1}, and the umbrella type Vector has as instances all Array objects where the second parameter – the number of array dimensions – is 1, regardless of what the element type is. In languages where parametric types must always be specified in full, this is not especially helpful, but in Julia, this allows one to write just Vector for the abstract type including all one-dimensional dense arrays of any element type.","category":"page"},{"location":"manual/types.html#man-singleton-types","page":"Types","title":"Singleton types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Immutable composite types with no fields are called singletons. Formally, if","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"T is an immutable composite type (i.e. defined with struct),\na isa T && b isa T implies a === b,","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"then T is a singleton type.[2] Base.issingletontype can be used to check if a type is a singleton type. Abstract types cannot be singleton types by construction.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"From the definition, it follows that there can be only one instance of such types:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct NoFields\n       end\n\njulia> NoFields() === NoFields()\ntrue\n\njulia> Base.issingletontype(NoFields)\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The === function confirms that the constructed instances of NoFields are actually one and the same.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Parametric types can be singleton types when the above condition holds. For example,","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct NoFieldsParam{T}\n       end\n\njulia> Base.issingletontype(NoFieldsParam) # Can't be a singleton type ...\nfalse\n\njulia> NoFieldsParam{Int}() isa NoFieldsParam # ... because it has ...\ntrue\n\njulia> NoFieldsParam{Bool}() isa NoFieldsParam # ... multiple instances.\ntrue\n\njulia> Base.issingletontype(NoFieldsParam{Int}) # Parametrized, it is a singleton.\ntrue\n\njulia> NoFieldsParam{Int}() === NoFieldsParam{Int}()\ntrue","category":"page"},{"location":"manual/types.html#Types-of-functions","page":"Types","title":"Types of functions","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Each function has its own type, which is a subtype of Function.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> foo41(x) = x + 1\nfoo41 (generic function with 1 method)\n\njulia> typeof(foo41)\ntypeof(foo41) (singleton type of function foo41, subtype of Function)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Note how typeof(foo41) prints as itself. This is merely a convention for printing, as it is a first-class object that can be used like any other value:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> T = typeof(foo41)\ntypeof(foo41) (singleton type of function foo41, subtype of Function)\n\njulia> T <: Function\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Types of functions defined at top-level are singletons. When necessary, you can compare them with ===.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Closures also have their own type, which is usually printed with names that end in #<number>. Names and types for functions defined at different locations are distinct, but not guaranteed to be printed the same way across sessions.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof(x -> x + 1)\nvar\"#9#10\"","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Types of closures are not necessarily singletons.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> addy(y) = x -> x + y\naddy (generic function with 1 method)\n\njulia> typeof(addy(1)) === typeof(addy(2))\ntrue\n\njulia> addy(1) === addy(2)\nfalse\n\njulia> Base.issingletontype(typeof(addy(1)))\nfalse","category":"page"},{"location":"manual/types.html#man-typet-type","page":"Types","title":"Type{T} type selectors","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"For each type T, Type{T} is an abstract parametric type whose only instance is the object T. Until we discuss Parametric Methods and conversions, it is difficult to explain the utility of this construct, but in short, it allows one to specialize function behavior on specific types as values. This is useful for writing methods (especially parametric ones) whose behavior depends on a type that is given as an explicit argument rather than implied by the type of one of its arguments.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since the definition is a little difficult to parse, let's look at some examples:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> isa(Float64, Type{Float64})\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\njulia> isa(Float64, Type{Real})\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In other words, isa(A, Type{B}) is true if and only if A and B are the same object and that object is a type.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In particular, since parametric types are invariant, we have","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct TypeParamExample{T}\n           x::T\n       end\n\njulia> TypeParamExample isa Type{TypeParamExample}\ntrue\n\njulia> TypeParamExample{Int} isa Type{TypeParamExample}\nfalse\n\njulia> TypeParamExample{Int} isa Type{TypeParamExample{Int}}\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Without the parameter, Type is simply an abstract type which has all type objects as its instances:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type)\ntrue","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Any object that is not a type is not an instance of Type:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> isa(1, Type)\nfalse\n\njulia> isa(\"foo\", Type)\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"While Type is part of Julia's type hierarchy like any other abstract parametric type, it is not commonly used outside method signatures except in some special cases. Another important use case for Type is sharpening field types which would otherwise be captured less precisely, e.g. as DataType in the example below where the default constructor could lead to performance problems in code relying on the precise wrapped type (similarly to abstract type parameters).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct WrapType{T}\n       value::T\n       end\n\njulia> WrapType(Float64) # default constructor, note DataType\nWrapType{DataType}(Float64)\n\njulia> WrapType(::Type{T}) where T = WrapType{Type{T}}(T)\nWrapType\n\njulia> WrapType(Float64) # sharpened constructor, note more precise Type{Float64}\nWrapType{Type{Float64}}(Float64)","category":"page"},{"location":"manual/types.html#Type-Aliases","page":"Types","title":"Type Aliases","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Sometimes it is convenient to introduce a new name for an already expressible type. This can be done with a simple assignment statement. For example, UInt is aliased to either UInt32 or UInt64 as is appropriate for the size of pointers on the system:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"# 32-bit system:\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> UInt\nUInt64","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This is accomplished via the following code in base/boot.jl:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"if Int === Int64\n    const UInt = UInt64\nelse\n    const UInt = UInt32\nend","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Of course, this depends on what Int is aliased to – but that is predefined to be the correct type – either Int32 or Int64.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"(Note that unlike Int, Float does not exist as a type alias for a specific sized AbstractFloat. Unlike with integer registers, where the size of Int reflects the size of a native pointer on that machine, the floating point register sizes are specified by the IEEE-754 standard.)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Type aliases may be parametrized:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> const Family{T} = Set{T}\nSet\n\njulia> Family{Char} === Set{Char}\ntrue","category":"page"},{"location":"manual/types.html#Operations-on-Types","page":"Types","title":"Operations on Types","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Since types in Julia are themselves objects, ordinary functions can operate on them. Some functions that are particularly useful for working with or exploring types have already been introduced, such as the <: operator, which indicates whether its left hand operand is a subtype of its right hand operand.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The isa function tests if an object is of a given type and returns true or false:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> isa(1, Int)\ntrue\n\njulia> isa(1, AbstractFloat)\nfalse","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The typeof function, already used throughout the manual in examples, returns the type of its argument. Since, as noted above, types are objects, they also have types, and we can ask what their types are:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof(Rational{Int})\nDataType\n\njulia> typeof(Union{Real,String})\nUnion","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"What if we repeat the process? What is the type of a type of a type? As it happens, types are all composite values and thus all have a type of DataType:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> typeof(DataType)\nDataType\n\njulia> typeof(Union)\nDataType","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"DataType is its own type.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Another operation that applies to some types is supertype, which reveals a type's supertype. Only declared types (DataType) have unambiguous supertypes:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> supertype(Float64)\nAbstractFloat\n\njulia> supertype(Number)\nAny\n\njulia> supertype(AbstractString)\nAny\n\njulia> supertype(Any)\nAny","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"If you apply supertype to other type objects (or non-type objects), a MethodError is raised:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> supertype(Union{Float64,Int64})\nERROR: MethodError: no method matching supertype(::Type{Union{Float64, Int64}})\nThe function `supertype` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n[...]","category":"page"},{"location":"manual/types.html#man-custom-pretty-printing","page":"Types","title":"Custom pretty-printing","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Often, one wants to customize how instances of a type are displayed.  This is accomplished by overloading the show function.  For example, suppose we define a type to represent complex numbers in polar form:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct Polar{T<:Real} <: Number\n           r::T\n           Θ::T\n       end\n\njulia> Polar(r::Real,Θ::Real) = Polar(promote(r,Θ)...)\nPolar","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Here, we've added a custom constructor function so that it can take arguments of different Real types and promote them to a common type (see Constructors and Conversion and Promotion). (Of course, we would have to define lots of other methods, too, to make it act like a Number, e.g. +, *, one, zero, promotion rules and so on.) By default, instances of this type display rather simply, with information about the type name and the field values, as e.g. Polar{Float64}(3.0,4.0).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"If we want it to display instead as 3.0 * exp(4.0im), we would define the following method to print the object to a given output object io (representing a file, terminal, buffer, etcetera; see Networking and Streams):","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Base.show(io::IO, z::Polar) = print(io, z.r, \" * exp(\", z.Θ, \"im)\")","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"More fine-grained control over display of Polar objects is possible. In particular, sometimes one wants both a verbose multi-line printing format, used for displaying a single object in the REPL and other interactive environments, and also a more compact single-line format used for print or for displaying the object as part of another object (e.g. in an array). Although by default the show(io, z) function is called in both cases, you can define a different multi-line format for displaying an object by overloading a three-argument form of show that takes the text/plain MIME type as its second argument (see Multimedia I/O), for example:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Base.show(io::IO, ::MIME\"text/plain\", z::Polar{T}) where{T} =\n           print(io, \"Polar{$T} complex number:\\n   \", z)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"(Note that print(..., z) here will call the 2-argument show(io, z) method.) This results in:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Polar(3, 4.0)\nPolar{Float64} complex number:\n   3.0 * exp(4.0im)\n\njulia> [Polar(3, 4.0), Polar(4.0,5.3)]\n2-element Vector{Polar{Float64}}:\n 3.0 * exp(4.0im)\n 4.0 * exp(5.3im)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"where the single-line show(io, z) form is still used for an array of Polar values.   Technically, the REPL calls display(z) to display the result of executing a line, which defaults to show(stdout, MIME(\"text/plain\"), z), which in turn defaults to show(stdout, z), but you should not define new display methods unless you are defining a new multimedia display handler (see Multimedia I/O).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Moreover, you can also define show methods for other MIME types in order to enable richer display (HTML, images, etcetera) of objects in environments that support this (e.g. IJulia).   For example, we can define formatted HTML display of Polar objects, with superscripts and italics, via:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> Base.show(io::IO, ::MIME\"text/html\", z::Polar{T}) where {T} =\n           println(io, \"<code>Polar{$T}</code> complex number: \",\n                   z.r, \" <i>e</i><sup>\", z.Θ, \" <i>i</i></sup>\")","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"A Polar object will then display automatically using HTML in an environment that supports HTML display, but you can call show manually to get HTML output if you want:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> show(stdout, \"text/html\", Polar(3.0,4.0))\n<code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup>","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"<p>An HTML renderer would display this as: <code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup></p>","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"As a rule of thumb, the single-line show method should print a valid Julia expression for creating the shown object.  When this show method contains infix operators, such as the multiplication operator (*) in our single-line show method for Polar above, it may not parse correctly when printed as part of another object.  To see this, consider the expression object (see Program representation) which takes the square of a specific instance of our Polar type:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> a = Polar(3, 4.0)\nPolar{Float64} complex number:\n   3.0 * exp(4.0im)\n\njulia> print(:($a^2))\n3.0 * exp(4.0im) ^ 2","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Because the operator ^ has higher precedence than * (see Operator Precedence and Associativity), this output does not faithfully represent the expression a ^ 2 which should be equal to (3.0 * exp(4.0im)) ^ 2.  To solve this issue, we must make a custom method for Base.show_unquoted(io::IO, z::Polar, indent::Int, precedence::Int), which is called internally by the expression object when printing:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> function Base.show_unquoted(io::IO, z::Polar, ::Int, precedence::Int)\n           if Base.operator_precedence(:*) <= precedence\n               print(io, \"(\")\n               show(io, z)\n               print(io, \")\")\n           else\n               show(io, z)\n           end\n       end\n\njulia> :($a^2)\n:((3.0 * exp(4.0im)) ^ 2)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"The method defined above adds parentheses around the call to show when the precedence of the calling operator is higher than or equal to the precedence of multiplication.  This check allows expressions which parse correctly without the parentheses (such as :($a + 2) and :($a == 2)) to omit them when printing:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> :($a + 2)\n:(3.0 * exp(4.0im) + 2)\n\njulia> :($a == 2)\n:(3.0 * exp(4.0im) == 2)","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In some cases, it is useful to adjust the behavior of show methods depending on the context. This can be achieved via the IOContext type, which allows passing contextual properties together with a wrapped IO stream. For example, we can build a shorter representation in our show method when the :compact property is set to true, falling back to the long representation if the property is false or absent:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> function Base.show(io::IO, z::Polar)\n           if get(io, :compact, false)::Bool\n               print(io, z.r, \"ℯ\", z.Θ, \"im\")\n           else\n               print(io, z.r, \" * exp(\", z.Θ, \"im)\")\n           end\n       end","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"This new compact representation will be used when the passed IO stream is an IOContext object with the :compact property set. In particular, this is the case when printing arrays with multiple columns (where horizontal space is limited):","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> show(IOContext(stdout, :compact=>true), Polar(3, 4.0))\n3.0ℯ4.0im\n\njulia> [Polar(3, 4.0) Polar(4.0,5.3)]\n1×2 Matrix{Polar{Float64}}:\n 3.0ℯ4.0im  4.0ℯ5.3im","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"See the IOContext documentation for a list of common properties which can be used to adjust printing.","category":"page"},{"location":"manual/types.html#\"Value-types\"","page":"Types","title":"\"Value types\"","text":"","category":"section"},{"location":"manual/types.html","page":"Types","title":"Types","text":"In Julia, you can't dispatch on a value such as true or false. However, you can dispatch on parametric types, and Julia allows you to include \"plain bits\" values (Types, Symbols, Integers, floating-point numbers, tuples, etc.) as type parameters.  A common example is the dimensionality parameter in Array{T,N}, where T is a type (e.g., Float64) but N is just an Int.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"You can create your own custom types that take values as parameters, and use them to control dispatch of custom types. By way of illustration of this idea, let's introduce the parametric type Val{x}, and its constructor Val(x) = Val{x}(), which serves as a customary way to exploit this technique for cases where you don't need a more elaborate hierarchy.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"Val is defined as:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> struct Val{x}\n       end\n\njulia> Val(x) = Val{x}()\nVal","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"There is no more to the implementation of Val than this.  Some functions in Julia's standard library accept Val instances as arguments, and you can also use it to write your own functions.  For example:","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"julia> firstlast(::Val{true}) = \"First\"\nfirstlast (generic function with 1 method)\n\njulia> firstlast(::Val{false}) = \"Last\"\nfirstlast (generic function with 2 methods)\n\njulia> firstlast(Val(true))\n\"First\"\n\njulia> firstlast(Val(false))\n\"Last\"","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"For consistency across Julia, the call site should always pass a Val instance rather than using a type, i.e., use foo(Val(:bar)) rather than foo(Val{:bar}).","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"It's worth noting that it's extremely easy to mis-use parametric \"value\" types, including Val; in unfavorable cases, you can easily end up making the performance of your code much worse.  In particular, you would never want to write actual code as illustrated above.  For more information about the proper (and improper) uses of Val, please read the more extensive discussion in the performance tips.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"[1]: \"Small\" is defined by the max_union_splitting configuration, which currently defaults to 4.","category":"page"},{"location":"manual/types.html","page":"Types","title":"Types","text":"[2]: A few popular languages have singleton types, including Haskell, Scala and Ruby.","category":"page"},{"location":"stdlib/Unicode.html","page":"Unicode","title":"Unicode","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/stdlib/Unicode/docs/src/index.md\"","category":"page"},{"location":"stdlib/Unicode.html#Unicode","page":"Unicode","title":"Unicode","text":"","category":"section"},{"location":"stdlib/Unicode.html","page":"Unicode","title":"Unicode","text":"The Unicode module provides essential functionality for managing Unicode characters and strings. It includes validation, category determination, normalization, case transformation, and grapheme segmentation, enabling effective Unicode data handling.","category":"page"},{"location":"stdlib/Unicode.html","page":"Unicode","title":"Unicode","text":"Unicode\nUnicode.julia_chartransform\nUnicode.isassigned\nUnicode.isequal_normalized\nUnicode.normalize\nUnicode.graphemes","category":"page"},{"location":"stdlib/Unicode.html#Unicode","page":"Unicode","title":"Unicode","text":"The Unicode module provides essential functionality for managing Unicode characters and strings. It includes validation, category determination, normalization, case transformation, and grapheme segmentation, enabling effective Unicode data handling.\n\n\n\n\n\n","category":"module"},{"location":"stdlib/Unicode.html#Unicode.julia_chartransform","page":"Unicode","title":"Unicode.julia_chartransform","text":"Unicode.julia_chartransform(c::Union{Char,Integer})\n\nMap the Unicode character (Char) or codepoint (Integer) c to the corresponding \"equivalent\" character or codepoint, respectively, according to the custom equivalence used within the Julia parser (in addition to NFC normalization).\n\nFor example, 'µ' (U+00B5 micro) is treated as equivalent to 'μ' (U+03BC mu) by Julia's parser, so julia_chartransform performs this transformation while leaving other characters unchanged:\n\njulia> Unicode.julia_chartransform('µ')\n'μ': Unicode U+03BC (category Ll: Letter, lowercase)\n\njulia> Unicode.julia_chartransform('x')\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia_chartransform is mainly useful for passing to the Unicode.normalize function in order to mimic the normalization used by the Julia parser:\n\njulia> s = \"µö\"\n\"µö\"\n\njulia> s2 = Unicode.normalize(s, compose=true, stable=true, chartransform=Unicode.julia_chartransform)\n\"μö\"\n\njulia> collect(s2)\n2-element Vector{Char}:\n 'μ': Unicode U+03BC (category Ll: Letter, lowercase)\n 'ö': Unicode U+00F6 (category Ll: Letter, lowercase)\n\njulia> s2 == string(Meta.parse(s))\ntrue\n\ncompat: Julia 1.8\nThis function was introduced in Julia 1.8.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Unicode.html#Unicode.isassigned","page":"Unicode","title":"Unicode.isassigned","text":"Unicode.isassigned(c) -> Bool\n\nReturn true if the given char or integer is an assigned Unicode code point.\n\nExamples\n\njulia> Unicode.isassigned(101)\ntrue\n\njulia> Unicode.isassigned('\\x01')\ntrue\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Unicode.html#Unicode.isequal_normalized","page":"Unicode","title":"Unicode.isequal_normalized","text":"isequal_normalized(s1::AbstractString, s2::AbstractString; casefold=false, stripmark=false, chartransform=identity)\n\nReturn whether s1 and s2 are canonically equivalent Unicode strings.   If casefold=true, ignores case (performs Unicode case-folding); if stripmark=true, strips diacritical marks and other combining characters.\n\nAs with Unicode.normalize, you can also pass an arbitrary function via the chartransform keyword (mapping Integer codepoints to codepoints) to perform custom normalizations, such as Unicode.julia_chartransform.\n\ncompat: Julia 1.8\nThe isequal_normalized function was added in Julia 1.8.\n\nExamples\n\nFor example, the string \"noël\" can be constructed in two canonically equivalent ways in Unicode, depending on whether \"ë\" is formed from a single codepoint U+00EB or from the ASCII character 'e' followed by the U+0308 combining-diaeresis character.\n\njulia> s1 = \"noël\"\n\"noël\"\n\njulia> s2 = \"noël\"\n\"noël\"\n\njulia> s1 == s2\nfalse\n\njulia> isequal_normalized(s1, s2)\ntrue\n\njulia> isequal_normalized(s1, \"noel\", stripmark=true)\ntrue\n\njulia> isequal_normalized(s1, \"NOËL\", casefold=true)\ntrue\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Unicode.html#Unicode.normalize","page":"Unicode","title":"Unicode.normalize","text":"Unicode.normalize(s::AbstractString; keywords...)\nUnicode.normalize(s::AbstractString, normalform::Symbol)\n\nNormalize the string s. By default, canonical composition (compose=true) is performed without ensuring Unicode versioning stability (compat=false), which produces the shortest possible equivalent string but may introduce composition characters not present in earlier Unicode versions.\n\nAlternatively, one of the four \"normal forms\" of the Unicode standard can be specified: normalform can be :NFC, :NFD, :NFKC, or :NFKD.  Normal forms C (canonical composition) and D (canonical decomposition) convert different visually identical representations of the same abstract string into a single canonical form, with form C being more compact.  Normal forms KC and KD additionally canonicalize \"compatibility equivalents\": they convert characters that are abstractly similar but visually distinct into a single canonical choice (e.g. they expand ligatures into the individual characters), with form KC being more compact.\n\nAlternatively, finer control and additional transformations may be obtained by calling Unicode.normalize(s; keywords...), where any number of the following boolean keywords options (which all default to false except for compose) are specified:\n\ncompose=false: do not perform canonical composition\ndecompose=true: do canonical decomposition instead of canonical composition (compose=true is ignored if present)\ncompat=true: compatibility equivalents are canonicalized\ncasefold=true: perform Unicode case folding, e.g. for case-insensitive string comparison\nnewline2lf=true, newline2ls=true, or newline2ps=true: convert various newline sequences (LF, CRLF, CR, NEL) into a linefeed (LF), line-separation (LS), or paragraph-separation (PS) character, respectively\nstripmark=true: strip diacritical marks (e.g. accents)\nstripignore=true: strip Unicode's \"default ignorable\" characters (e.g. the soft hyphen or the left-to-right marker)\nstripcc=true: strip control characters; horizontal tabs and form feeds are converted to spaces; newlines are also converted to spaces unless a newline-conversion flag was specified\nrejectna=true: throw an error if unassigned code points are found\nstable=true: enforce Unicode versioning stability (never introduce characters missing from earlier Unicode versions)\n\nYou can also use the chartransform keyword (which defaults to identity) to pass an arbitrary function mapping Integer codepoints to codepoints, which is called on each character in s as it is processed, in order to perform arbitrary additional normalizations. For example, by passing chartransform=Unicode.julia_chartransform, you can apply a few Julia-specific character normalizations that are performed by Julia when parsing identifiers (in addition to NFC normalization: compose=true, stable=true).\n\nFor example, NFKC corresponds to the options compose=true, compat=true, stable=true.\n\nExamples\n\njulia> \"é\" == Unicode.normalize(\"é\") #LHS: Unicode U+00e9, RHS: U+0065 & U+0301\ntrue\n\njulia> \"μ\" == Unicode.normalize(\"µ\", compat=true) #LHS: Unicode U+03bc, RHS: Unicode U+00b5\ntrue\n\njulia> Unicode.normalize(\"JuLiA\", casefold=true)\n\"julia\"\n\njulia> Unicode.normalize(\"JúLiA\", stripmark=true)\n\"JuLiA\"\n\ncompat: Julia 1.8\nThe chartransform keyword argument requires Julia 1.8.\n\n\n\n\n\n","category":"function"},{"location":"stdlib/Unicode.html#Unicode.graphemes","page":"Unicode","title":"Unicode.graphemes","text":"graphemes(s::AbstractString) -> GraphemeIterator\n\nReturn an iterator over substrings of s that correspond to the extended graphemes in the string, as defined by Unicode UAX #29. (Roughly, these are what users would perceive as single characters, even though they may contain more than one codepoint; for example a letter combined with an accent mark is a single grapheme.)\n\n\n\n\n\ngraphemes(s::AbstractString, m:n) -> SubString\n\nReturns a SubString of s consisting of the m-th through n-th graphemes of the string s, where the second argument m:n is an integer-valued AbstractUnitRange.\n\nLoosely speaking, this corresponds to the m:n-th user-perceived \"characters\" in the string.  For example:\n\njulia> s = graphemes(\"exposé\", 3:6)\n\"posé\"\n\njulia> collect(s)\n5-element Vector{Char}:\n 'p': ASCII/Unicode U+0070 (category Ll: Letter, lowercase)\n 'o': ASCII/Unicode U+006F (category Ll: Letter, lowercase)\n 's': ASCII/Unicode U+0073 (category Ll: Letter, lowercase)\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n '́': Unicode U+0301 (category Mn: Mark, nonspacing)\n\nThis consists of the 3rd to 7th codepoints (Chars) in \"exposé\", because the grapheme \"é\" is actually two Unicode codepoints (an 'e' followed by an acute-accent combining character U+0301).\n\nBecause finding grapheme boundaries requires iteration over the string contents, the graphemes(s, m:n) function requires time proportional to the length of the string (number of codepoints) before the end of the substring.\n\ncompat: Julia 1.9\nThe m:n argument of graphemes requires Julia 1.9.\n\n\n\n\n\n","category":"function"},{"location":"devdocs/cartesian.html#Base.Cartesian","page":"Base.Cartesian","title":"Base.Cartesian","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"The (non-exported) Cartesian module provides macros that facilitate writing multidimensional algorithms. Most often you can write such algorithms with straightforward techniques; however, there are a few cases where Base.Cartesian is still useful or necessary.","category":"page"},{"location":"devdocs/cartesian.html#Principles-of-usage","page":"Base.Cartesian","title":"Principles of usage","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"A simple example of usage is:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"@nloops 3 i A begin\n    s += @nref 3 A i\nend","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"which generates the following code:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"for i_3 = axes(A, 3)\n    for i_2 = axes(A, 2)\n        for i_1 = axes(A, 1)\n            s += A[i_1, i_2, i_3]\n        end\n    end\nend","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"In general, Cartesian allows you to write generic code that contains repetitive elements, like the nested loops in this example.  Other applications include repeated expressions (e.g., loop unwinding) or creating function calls with variable numbers of arguments without using the \"splat\" construct (i...).","category":"page"},{"location":"devdocs/cartesian.html#Basic-syntax","page":"Base.Cartesian","title":"Basic syntax","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"The (basic) syntax of @nloops is as follows:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"The first argument must be an integer (not a variable) specifying the number of loops.\nThe second argument is the symbol-prefix used for the iterator variable. Here we used i, and variables i_1, i_2, i_3 were generated.\nThe third argument specifies the range for each iterator variable. If you use a variable (symbol) here, it's taken as axes(A, dim). More flexibly, you can use the anonymous-function expression syntax described below.\nThe last argument is the body of the loop. Here, that's what appears between the begin...end.","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"There are some additional features of @nloops described in the reference section.","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"@nref follows a similar pattern, generating A[i_1,i_2,i_3] from @nref 3 A i. The general practice is to read from left to right, which is why @nloops is @nloops 3 i A expr (as in for i_2 = axes(A, 2), where i_2 is to the left and the range is to the right) whereas @nref is @nref 3 A i (as in A[i_1,i_2,i_3], where the array comes first).","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"If you're developing code with Cartesian, you may find that debugging is easier when you examine the generated code, using @macroexpand:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"DocTestSetup = quote\n    import Base.Cartesian: @nref\nend","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"julia> @macroexpand @nref 2 A i\n:(A[i_1, i_2])","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"DocTestSetup = nothing","category":"page"},{"location":"devdocs/cartesian.html#Supplying-the-number-of-expressions","page":"Base.Cartesian","title":"Supplying the number of expressions","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"The first argument to both of these macros is the number of expressions, which must be an integer. When you're writing a function that you intend to work in multiple dimensions, this may not be something you want to hard-code. The recommended approach is to use a @generated function.  Here's an example:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"@generated function mysum(A::Array{T,N}) where {T,N}\n    quote\n        s = zero(T)\n        @nloops $N i A begin\n            s += @nref $N A i\n        end\n        s\n    end\nend","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"Naturally, you can also prepare expressions or perform calculations before the quote block.","category":"page"},{"location":"devdocs/cartesian.html#Anonymous-function-expressions-as-macro-arguments","page":"Base.Cartesian","title":"Anonymous-function expressions as macro arguments","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"Perhaps the single most powerful feature in Cartesian is the ability to supply anonymous-function expressions that get evaluated at parsing time.  Let's consider a simple example:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"@nexprs 2 j->(i_j = 1)","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"@nexprs generates n expressions that follow a pattern. This code would generate the following statements:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"i_1 = 1\ni_2 = 1","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"In each generated statement, an \"isolated\" j (the variable of the anonymous function) gets replaced by values in the range 1:2. Generally speaking, Cartesian employs a LaTeX-like syntax.  This allows you to do math on the index j.  Here's an example computing the strides of an array:","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"s_1 = 1\n@nexprs 3 j->(s_{j+1} = s_j * size(A, j))","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"would generate expressions","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"s_1 = 1\ns_2 = s_1 * size(A, 1)\ns_3 = s_2 * size(A, 2)\ns_4 = s_3 * size(A, 3)","category":"page"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"Anonymous-function expressions have many uses in practice.","category":"page"},{"location":"devdocs/cartesian.html#dev-cartesian-reference","page":"Base.Cartesian","title":"Macro reference","text":"","category":"section"},{"location":"devdocs/cartesian.html","page":"Base.Cartesian","title":"Base.Cartesian","text":"Base.Cartesian.@nloops\nBase.Cartesian.@nref\nBase.Cartesian.@nextract\nBase.Cartesian.@nexprs\nBase.Cartesian.@ncall\nBase.Cartesian.@ncallkw\nBase.Cartesian.@ntuple\nBase.Cartesian.@nall\nBase.Cartesian.@nany\nBase.Cartesian.@nif","category":"page"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nloops","page":"Base.Cartesian","title":"Base.Cartesian.@nloops","text":"@nloops N itersym rangeexpr bodyexpr\n@nloops N itersym rangeexpr preexpr bodyexpr\n@nloops N itersym rangeexpr preexpr postexpr bodyexpr\n\nGenerate N nested loops, using itersym as the prefix for the iteration variables. rangeexpr may be an anonymous-function expression, or a simple symbol var in which case the range is axes(var, d) for dimension d.\n\nOptionally, you can provide \"pre\" and \"post\" expressions. These get executed first and last, respectively, in the body of each loop. For example:\n\n@nloops 2 i A d -> j_d = min(i_d, 5) begin\n    s += @nref 2 A j\nend\n\nwould generate:\n\nfor i_2 = axes(A, 2)\n    j_2 = min(i_2, 5)\n    for i_1 = axes(A, 1)\n        j_1 = min(i_1, 5)\n        s += A[j_1, j_2]\n    end\nend\n\nIf you want just a post-expression, supply nothing for the pre-expression. Using parentheses and semicolons, you can supply multi-statement expressions.\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nref","page":"Base.Cartesian","title":"Base.Cartesian.@nref","text":"@nref N A indexexpr\n\nGenerate expressions like A[i_1, i_2, ...]. indexexpr can either be an iteration-symbol prefix, or an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nref 3 A i\n:(A[i_1, i_2, i_3])\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nextract","page":"Base.Cartesian","title":"Base.Cartesian.@nextract","text":"@nextract N esym isym\n\nGenerate N variables esym_1, esym_2, ..., esym_N to extract values from isym. isym can be either a Symbol or anonymous-function expression.\n\n@nextract 2 x y would generate\n\nx_1 = y[1]\nx_2 = y[2]\n\nwhile @nextract 3 x d->y[2d-1] yields\n\nx_1 = y[1]\nx_2 = y[3]\nx_3 = y[5]\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nexprs","page":"Base.Cartesian","title":"Base.Cartesian.@nexprs","text":"@nexprs N expr\n\nGenerate N expressions. expr should be an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nexprs 4 i -> y[i] = A[i+j]\nquote\n    y[1] = A[1 + j]\n    y[2] = A[2 + j]\n    y[3] = A[3 + j]\n    y[4] = A[4 + j]\nend\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ncall","page":"Base.Cartesian","title":"Base.Cartesian.@ncall","text":"@ncall N f sym...\n\nGenerate a function call expression. sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nFor example, @ncall 3 func a generates\n\nfunc(a_1, a_2, a_3)\n\nwhile @ncall 2 func a b i->c[i] yields\n\nfunc(a, b, c[1], c[2])\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ncallkw","page":"Base.Cartesian","title":"Base.Cartesian.@ncallkw","text":"@ncallkw N f kw sym...\n\nGenerate a function call expression with keyword arguments kw.... As in the case of @ncall, sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nExamples\n\njulia> using Base.Cartesian\n\njulia> f(x...; a, b = 1, c = 2, d = 3) = +(x..., a, b, c, d);\n\njulia> x_1, x_2 = (-1, -2); b = 0; kw = (c = 0, d = 0);\n\njulia> @ncallkw 2 f (; a = 0, b, kw...) x\n-3\n\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ntuple","page":"Base.Cartesian","title":"Base.Cartesian.@ntuple","text":"@ntuple N expr\n\nGenerates an N-tuple. @ntuple 2 i would generate (i_1, i_2), and @ntuple 2 k->k+1 would generate (2,3).\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nall","page":"Base.Cartesian","title":"Base.Cartesian.@nall","text":"@nall N expr\n\nCheck whether all of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nall 3 d->(i_d > 1) would generate the expression (i_1 > 1 && i_2 > 1 && i_3 > 1). This can be convenient for bounds-checking.\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nany","page":"Base.Cartesian","title":"Base.Cartesian.@nany","text":"@nany N expr\n\nCheck whether any of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nany 3 d->(i_d > 1) would generate the expression (i_1 > 1 || i_2 > 1 || i_3 > 1).\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nif","page":"Base.Cartesian","title":"Base.Cartesian.@nif","text":"@nif N conditionexpr expr\n@nif N conditionexpr expr elseexpr\n\nGenerates a sequence of if ... elseif ... else ... end statements. For example:\n\n@nif 3 d->(i_d >= size(A,d)) d->(error(\"Dimension \", d, \" too big\")) d->println(\"All OK\")\n\nwould generate:\n\nif i_1 > size(A, 1)\n    error(\"Dimension \", 1, \" too big\")\nelseif i_2 > size(A, 2)\n    error(\"Dimension \", 2, \" too big\")\nelse\n    println(\"All OK\")\nend\n\n\n\n\n\n","category":"macro"},{"location":"devdocs/inference.html#Inference","page":"Inference","title":"Inference","text":"","category":"section"},{"location":"devdocs/inference.html#How-inference-works","page":"Inference","title":"How inference works","text":"","category":"section"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"In Julia compiler, \"type inference\" refers to the process of deducing the types of later values from the types of input values. Julia's approach to inference has been described in the blog posts below:","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"Shows a simplified implementation of the data-flow analysis algorithm, that Julia's type inference routine is based on.\nGives a high level view of inference with a focus on its inter-procedural convergence guarantee.\nExplains a refinement on the algorithm introduced in 2.","category":"page"},{"location":"devdocs/inference.html#Debugging-compiler.jl","page":"Inference","title":"Debugging compiler.jl","text":"","category":"section"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"You can start a Julia session, edit compiler/*.jl (for example to insert print statements), and then replace Core.Compiler in your running session by navigating to base and executing include(\"compiler/compiler.jl\"). This trick typically leads to much faster development than if you rebuild Julia for each change.","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"Alternatively, you can use the Revise.jl package to track the compiler changes by using the command Revise.track(Core.Compiler) at the beginning of your Julia session. As explained in the Revise documentation, the modifications to the compiler will be reflected when the modified files are saved.","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"A convenient entry point into inference is typeinf_code. Here's a demo running inference on convert(Int, UInt(1)):","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"# Get the method\natypes = Tuple{Type{Int}, UInt}  # argument types\nmths = methods(convert, atypes)  # worth checking that there is only one\nm = first(mths)\n\n# Create variables needed to call `typeinf_code`\ninterp = Core.Compiler.NativeInterpreter()\nsparams = Core.svec()      # this particular method doesn't have type-parameters\nrun_optimizer = true       # run all inference optimizations\ntypes = Tuple{typeof(convert), atypes.parameters...} # Tuple{typeof(convert), Type{Int}, UInt}\nCore.Compiler.typeinf_code(interp, m, types, sparams, run_optimizer)","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"If your debugging adventures require a MethodInstance, you can look it up by calling Core.Compiler.specialize_method using many of the variables above. A CodeInfo object may be obtained with","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"# Returns the CodeInfo object for `convert(Int, ::UInt)`:\nci = (@code_typed convert(Int, UInt(1)))[1]","category":"page"},{"location":"devdocs/inference.html#The-inlining-algorithm-(inline_worthy)","page":"Inference","title":"The inlining algorithm (inline_worthy)","text":"","category":"section"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"Much of the hardest work for inlining runs in ssa_inlining_pass!. However, if your question is \"why didn't my function inline?\" then you will most likely be interested in inline_worthy, which makes a decision to inline the function call or not.","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"inline_worthy implements a cost-model, where \"cheap\" functions get inlined; more specifically, we inline functions if their anticipated run-time is not large compared to the time it would take to issue a call to them if they were not inlined. The cost-model is extremely simple and ignores many important details: for example, all for loops are analyzed as if they will be executed once, and the cost of an if...else...end includes the summed cost of all branches. It's also worth acknowledging that we currently lack a suite of functions suitable for testing how well the cost model predicts the actual run-time cost, although BaseBenchmarks provides a great deal of indirect information about the successes and failures of any modification to the inlining algorithm.","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"The foundation of the cost-model is a lookup table, implemented in add_tfunc and its callers, that assigns an estimated cost (measured in CPU cycles) to each of Julia's intrinsic functions. These costs are based on standard ranges for common architectures (see Agner Fog's analysis for more detail).","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"We supplement this low-level lookup table with a number of special cases. For example, an :invoke expression (a call for which all input and output types were inferred in advance) is assigned a fixed cost (currently 20 cycles). In contrast, a :call expression, for functions other than intrinsics/builtins, indicates that the call will require dynamic dispatch, in which case we assign a cost set by Params.inline_nonleaf_penalty (currently set at 1000). Note that this is not a \"first-principles\" estimate of the raw cost of dynamic dispatch, but a mere heuristic indicating that dynamic dispatch is extremely expensive.","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"Each statement gets analyzed for its total cost in a function called statement_cost. You can display the cost associated with each statement as follows:","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"julia> Base.print_statement_costs(stdout, map, (typeof(sqrt), Tuple{Int},)) # map(sqrt, (2,))\nmap(f, t::Tuple{Any}) @ Base tuple.jl:281\n  0 1 ─ %1  = $(Expr(:boundscheck, true))::Bool\n  0 │   %2  = Base.getfield(_3, 1, %1)::Int64\n  1 │   %3  = Base.sitofp(Float64, %2)::Float64\n  0 │   %4  = Base.lt_float(%3, 0.0)::Bool\n  0 └──       goto #3 if not %4\n  0 2 ─       invoke Base.Math.throw_complex_domainerror(:sqrt::Symbol, %3::Float64)::Union{}\n  0 └──       unreachable\n 20 3 ─ %8  = Base.Math.sqrt_llvm(%3)::Float64\n  0 └──       goto #4\n  0 4 ─       goto #5\n  0 5 ─ %11 = Core.tuple(%8)::Tuple{Float64}\n  0 └──       return %11\n","category":"page"},{"location":"devdocs/inference.html","page":"Inference","title":"Inference","text":"The line costs are in the left column. This includes the consequences of inlining and other forms of optimization.","category":"page"},{"location":"manual/asynchronous-programming.html#man-asynchronous","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"When a program needs to interact with the outside world, for example communicating with another machine over the internet, operations in the program may need to happen in an unpredictable order. Say your program needs to download a file. We would like to initiate the download operation, perform other operations while we wait for it to complete, and then resume the code that needs the downloaded file when it is available. This sort of scenario falls in the domain of asynchronous programming, sometimes also referred to as concurrent programming (since, conceptually, multiple things are happening at once).","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"To address these scenarios, Julia provides Tasks (also known by several other names, such as symmetric coroutines, lightweight threads, cooperative multitasking, or one-shot continuations). When a piece of computing work (in practice, executing a particular function) is designated as a Task, it becomes possible to interrupt it by switching to another Task. The original Task can later be resumed, at which point it will pick up right where it left off. At first, this may seem similar to a function call. However there are two key differences. First, switching tasks does not use any space, so any number of task switches can occur without consuming the call stack. Second, switching among tasks can occur in any order, unlike function calls, where the called function must finish executing before control returns to the calling function.","category":"page"},{"location":"manual/asynchronous-programming.html#Basic-Task-operations","page":"Asynchronous Programming","title":"Basic Task operations","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"You can think of a Task as a handle to a unit of computational work to be performed. It has a create-start-run-finish lifecycle. Tasks are created by calling the Task constructor on a 0-argument function to run, or using the @task macro:","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> t = @task begin; sleep(5); println(\"done\"); end\nTask (runnable) @0x00007f13a40c0eb0","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"@task x is equivalent to Task(()->x).","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"This task will wait for five seconds, and then print done. However, it has not started running yet. We can run it whenever we're ready by calling schedule:","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> schedule(t);","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"If you try this in the REPL, you will see that schedule returns immediately. That is because it simply adds t to an internal queue of tasks to run. Then, the REPL will print the next prompt and wait for more input. Waiting for keyboard input provides an opportunity for other tasks to run, so at that point t will start. t calls sleep, which sets a timer and stops execution. If other tasks have been scheduled, they could run then. After five seconds, the timer fires and restarts t, and you will see done printed. t is then finished.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"The wait function blocks the calling task until some other task finishes. So for example if you type","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> schedule(t); wait(t)","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"instead of only calling schedule, you will see a five second pause before the next input prompt appears. That is because the REPL is waiting for t to finish before proceeding.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"It is common to want to create a task and schedule it right away, so the macro @async is provided for that purpose –- @async x is equivalent to schedule(@task x).","category":"page"},{"location":"manual/asynchronous-programming.html#Communicating-with-Channels","page":"Asynchronous Programming","title":"Communicating with Channels","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"In some problems, the various pieces of required work are not naturally related by function calls; there is no obvious \"caller\" or \"callee\" among the jobs that need to be done. An example is the producer-consumer problem, where one complex procedure is generating values and another complex procedure is consuming them. The consumer cannot simply call a producer function to get a value, because the producer may have more values to generate and so might not yet be ready to return. With tasks, the producer and consumer can both run as long as they need to, passing values back and forth as necessary.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Julia provides a Channel mechanism for solving this problem. A Channel is a waitable first-in first-out queue which can have multiple tasks reading from and writing to it.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Let's define a producer task, which produces values via the put! call. To consume values, we need to schedule the producer to run in a new task. A special Channel constructor which accepts a 1-arg function as an argument can be used to run a task bound to a channel. We can then take! values repeatedly from the channel object:","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> function producer(c::Channel)\n           put!(c, \"start\")\n           for n=1:4\n               put!(c, 2n)\n           end\n           put!(c, \"stop\")\n       end;\n\njulia> chnl = Channel(producer);\n\njulia> take!(chnl)\n\"start\"\n\njulia> take!(chnl)\n2\n\njulia> take!(chnl)\n4\n\njulia> take!(chnl)\n6\n\njulia> take!(chnl)\n8\n\njulia> take!(chnl)\n\"stop\"","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"One way to think of this behavior is that producer was able to return multiple times. Between calls to put!, the producer's execution is suspended and the consumer has control.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"The returned Channel can be used as an iterable object in a for loop, in which case the loop variable takes on all the produced values. The loop is terminated when the channel is closed.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> for x in Channel(producer)\n           println(x)\n       end\nstart\n2\n4\n6\n8\nstop","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Note that we did not have to explicitly close the channel in the producer. This is because the act of binding a Channel to a Task associates the open lifetime of a channel with that of the bound task. The channel object is closed automatically when the task terminates. Multiple channels can be bound to a task, and vice-versa.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"While the Task constructor expects a 0-argument function, the Channel method that creates a task-bound channel expects a function that accepts a single argument of type Channel. A common pattern is for the producer to be parameterized, in which case a partial function application is needed to create a 0 or 1 argument anonymous function.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"For Task objects this can be done either directly or by use of a convenience macro:","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"function mytask(myarg)\n    ...\nend\n\ntaskHdl = Task(() -> mytask(7))\n# or, equivalently\ntaskHdl = @task mytask(7)","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"To orchestrate more advanced work distribution patterns, bind and schedule can be used in conjunction with Task and Channel constructors to explicitly link a set of channels with a set of producer/consumer tasks.","category":"page"},{"location":"manual/asynchronous-programming.html#More-on-Channels","page":"Asynchronous Programming","title":"More on Channels","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"A channel can be visualized as a pipe, i.e., it has a write end and a read end :","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Multiple writers in different tasks can write to the same channel concurrently via put! calls.\nMultiple readers in different tasks can read data concurrently via take! calls.\nAs an example:\n# Given Channels c1 and c2,\nc1 = Channel(32)\nc2 = Channel(32)\n\n# and a function `foo` which reads items from c1, processes the item read\n# and writes a result to c2,\nfunction foo()\n    while true\n        data = take!(c1)\n        [...]               # process data\n        put!(c2, result)    # write out result\n    end\nend\n\n# we can schedule `n` instances of `foo` to be active concurrently.\nfor _ in 1:n\n    errormonitor(@async foo())\nend\nChannels are created via the Channel{T}(sz) constructor. The channel will only hold objects of type T. If the type is not specified, the channel can hold objects of any type. sz refers to the maximum number of elements that can be held in the channel at any time. For example, Channel(32) creates a channel that can hold a maximum of 32 objects of any type. A Channel{MyType}(64) can hold up to 64 objects of MyType at any time.\nIf a Channel is empty, readers (on a take! call) will block until data is available.\nIf a Channel is full, writers (on a put! call) will block until space becomes available.\nisready tests for the presence of any object in the channel, while wait waits for an object to become available.\nA Channel is in an open state initially. This means that it can be read from and written to freely via take! and put! calls. close closes a Channel. On a closed Channel, put! will fail. For example:\njulia> c = Channel(2);\n\njulia> put!(c, 1) # `put!` on an open channel succeeds\n1\n\njulia> close(c);\n\njulia> put!(c, 2) # `put!` on a closed channel throws an exception.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]\ntake! and fetch (which retrieves but does not remove the value) on a closed channel successfully return any existing values until it is emptied. Continuing the above example:\njulia> fetch(c) # Any number of `fetch` calls succeed.\n1\n\njulia> fetch(c)\n1\n\njulia> take!(c) # The first `take!` removes the value.\n1\n\njulia> take!(c) # No more data available on a closed channel.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Consider a simple example using channels for inter-task communication. We start 4 tasks to process data from a single jobs channel. Jobs, identified by an id (job_id), are written to the channel. Each task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id and the simulated time to the results channel. Finally all the results are printed out.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"julia> const jobs = Channel{Int}(32);\n\njulia> const results = Channel{Tuple}(32);\n\njulia> function do_work()\n           for job_id in jobs\n               exec_time = rand()\n               sleep(exec_time)                # simulates elapsed time doing actual work\n                                               # typically performed externally.\n               put!(results, (job_id, exec_time))\n           end\n       end;\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(@async make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for i in 1:4 # start 4 tasks to process requests in parallel\n           errormonitor(@async do_work())\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds\")\n           global n = n - 1\n       end\n4 finished in 0.22 seconds\n3 finished in 0.45 seconds\n1 finished in 0.5 seconds\n7 finished in 0.14 seconds\n2 finished in 0.78 seconds\n5 finished in 0.9 seconds\n9 finished in 0.36 seconds\n6 finished in 0.87 seconds\n8 finished in 0.79 seconds\n10 finished in 0.64 seconds\n12 finished in 0.5 seconds\n11 finished in 0.97 seconds\n0.029772311","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Instead of errormonitor(t), a more robust solution may be to use bind(results, t), as that will not only log any unexpected failures, but also force the associated resources to close and propagate the exception everywhere.","category":"page"},{"location":"manual/asynchronous-programming.html#More-task-operations","page":"Asynchronous Programming","title":"More task operations","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Task operations are built on a low-level primitive called yieldto. yieldto(task, value) suspends the current task, switches to the specified task, and causes that task's last yieldto call to return the specified value. Notice that yieldto is the only operation required to use task-style control flow; instead of calling and returning we are always just switching to a different task. This is why this feature is also called \"symmetric coroutines\"; each task is switched to and from using the same mechanism.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"yieldto is powerful, but most uses of tasks do not invoke it directly. Consider why this might be. If you switch away from the current task, you will probably want to switch back to it at some point, but knowing when to switch back, and knowing which task has the responsibility of switching back, can require considerable coordination. For example, put! and take! are blocking operations, which, when used in the context of channels maintain state to remember who the consumers are. Not needing to manually keep track of the consuming task is what makes put! easier to use than the low-level yieldto.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"In addition to yieldto, a few other basic functions are needed to use tasks effectively.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"current_task gets a reference to the currently-running task.\nistaskdone queries whether a task has exited.\nistaskstarted queries whether a task has run yet.\ntask_local_storage manipulates a key-value store specific to the current task.","category":"page"},{"location":"manual/asynchronous-programming.html#Tasks-and-events","page":"Asynchronous Programming","title":"Tasks and events","text":"","category":"section"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"Most task switches occur as a result of waiting for events such as I/O requests, and are performed by a scheduler included in Julia Base. The scheduler maintains a queue of runnable tasks, and executes an event loop that restarts tasks based on external events such as message arrival.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"The basic function for waiting for an event is wait. Several objects implement wait; for example, given a Process object, wait will wait for it to exit. wait is often implicit; for example, a wait can happen inside a call to read to wait for data to be available.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"In all of these cases, wait ultimately operates on a Condition object, which is in charge of queueing and restarting tasks. When a task calls wait on a Condition, the task is marked as non-runnable, added to the condition's queue, and switches to the scheduler. The scheduler will then pick another task to run, or block waiting for external events. If all goes well, eventually an event handler will call notify on the condition, which causes tasks waiting for that condition to become runnable again.","category":"page"},{"location":"manual/asynchronous-programming.html","page":"Asynchronous Programming","title":"Asynchronous Programming","text":"A task created explicitly by calling Task is initially not known to the scheduler. This allows you to manage tasks manually using yieldto if you wish. However, when such a task waits for an event, it still gets restarted automatically when the event happens, as you would expect.","category":"page"},{"location":"manual/variables-and-scoping.html#scope-of-variables","page":"Scope of Variables","title":"Scope of Variables","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The scope of a variable is the region of code within which a variable is accessible. Variable scoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have arguments called x without the two x's referring to the same thing. Similarly, there are many other cases where different blocks of code can use the same name without referring to the same thing. The rules for when the same variable name does or doesn't refer to the same thing are called scope rules; this section spells them out in detail.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Certain constructs in the language introduce scope blocks, which are regions of code that are eligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary set of source lines; instead, it will always line up with one of these blocks. There are two main types of scopes in Julia, global scope and local scope. The latter can be nested. There is also a distinction in Julia between constructs which introduce a \"hard scope\" and those which only introduce a \"soft scope\", which affects whether shadowing a global variable by the same name is allowed or not.","category":"page"},{"location":"manual/variables-and-scoping.html#man-scope-table","page":"Scope of Variables","title":"Scope constructs","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The constructs introducing scope blocks are:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Construct Scope type Allowed within\nmodule, baremodule global global\nstruct local (soft) global\nfor, while, try local (soft) global, local\nmacro local (hard) global\nfunctions, do blocks, let blocks, comprehensions, generators local (hard) global, local","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Notably missing from this table are begin blocks and if blocks which do not introduce new scopes. The three types of scopes follow somewhat different rules which will be explained below.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Julia uses lexical scoping, meaning that a function's scope does not inherit from its caller's scope, but from the scope in which the function was defined. For example, in the following code the x inside foo refers to the x in the global scope of its module Bar:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> module Bar\n           x = 1\n           foo() = x\n       end;","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"and not a x in the scope where foo is used:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Thus lexical scope means that what a variable in a particular piece of code refers to can be deduced from the code in which it appears alone and does not depend on how the program executes. A scope nested inside another scope can \"see\" variables in all the outer scopes in which it is contained. Outer scopes, on the other hand, cannot see variables in inner scopes.","category":"page"},{"location":"manual/variables-and-scoping.html#Global-Scope","page":"Scope of Variables","title":"Global Scope","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Each module introduces a new global scope, separate from the global scope of all other modules—there is no all-encompassing global scope. Modules can introduce variables of other modules into their scope through the using or import statements or through qualified access using the dot-notation, i.e. each module is a so-called namespace as well as a first-class data structure associating names with values.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"If a top-level expression contains a variable declaration with keyword local, then that variable is not accessible outside that expression. The variable inside the expression does not affect global variables of the same name. An example is to declare local x in a begin or if block at the top-level:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> x = 1\n       begin\n           local x = 0\n           @show x\n       end\n       @show x;\nx = 0\nx = 1","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Note that the interactive prompt (aka REPL) is in the global scope of the module Main.","category":"page"},{"location":"manual/variables-and-scoping.html#local-scope","page":"Scope of Variables","title":"Local Scope","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"A new local scope is introduced by most code blocks (see above table for a complete list). If such a block is syntactically nested inside of another local scope, the scope it creates is nested inside of all the local scopes that it appears within, which are all ultimately nested inside of the global scope of the module in which the code is evaluated. Variables in outer scopes are visible from any scope they contain — meaning that they can be read and written in inner scopes — unless there is a local variable with the same name that \"shadows\" the outer variable of the same name. This is true even if the outer local is declared after (in the sense of textually below) an inner block. When we say that a variable \"exists\" in a given scope, this means that a variable by that name exists in any of the scopes that the current scope is nested inside of, including the current one.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Some programming languages require explicitly declaring new variables before using them. Explicit declaration works in Julia too: in any local scope, writing local x declares a new local variable in that scope, regardless of whether there is already a variable named x in an outer scope or not. Declaring each new variable like this is somewhat verbose and tedious, however, so Julia, like many other languages, considers assignment to a variable name that doesn't already exist to implicitly declare that variable. If the current scope is global, the new variable is global; if the current scope is local, the new variable is local to the innermost local scope and will be visible inside of that scope but not outside of it. If you assign to an existing local, it always updates that existing local: you can only shadow a local by explicitly declaring a new local in a nested scope with the local keyword. In particular, this applies to variables assigned in inner functions, which may surprise users coming from Python where assignment in an inner function creates a new local unless the variable is explicitly declared to be non-local.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Mostly this is pretty intuitive, but as with many things that behave intuitively, the details are more subtle than one might naïvely imagine.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"When x = <value> occurs in a local scope, Julia applies the following rules to decide what the expression means based on where the assignment expression occurs and what x already refers to at that location:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Existing local: If x is already a local variable, then the existing local x is assigned;\nHard scope: If x is not already a local variable and assignment occurs inside of any hard scope construct (i.e. within a let block, function or macro body, comprehension, or generator), a new local named x is created in the scope of the assignment;\nSoft scope: If x is not already a local variable and all of the scope constructs containing the assignment are soft scopes (loops, try/catch blocks, or struct blocks), the behavior depends on whether the global variable x is defined:\nif global x is undefined, a new local named x is created in the scope of the assignment;\nif global x is defined, the assignment is considered ambiguous:\nin non-interactive contexts (files, eval), an ambiguity warning is printed and a new local is created;\nin interactive contexts (REPL, notebooks), the global variable x is assigned.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"You may note that in non-interactive contexts the hard and soft scope behaviors are identical except that a warning is printed when an implicitly local variable (i.e. not declared with local x) shadows a global. In interactive contexts, the rules follow a more complex heuristic for the sake of convenience. This is covered in depth in examples that follow.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Now that you know the rules, let's look at some examples. Each example is assumed to be evaluated in a fresh REPL session so that the only globals in each snippet are the ones that are assigned in that block of code.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"We'll begin with a nice and clear-cut situation—assignment inside of a hard scope, in this case a function body, when no local variable by that name already exists:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\nERROR: UndefVarError: `x` not defined in `Main`","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Inside of the greet function, the assignment x = \"hello\" causes x to be a new local variable in the function's scope. There are two relevant facts: the assignment occurs in local scope and there is no existing local x variable. Since x is local, it doesn't matter if there is a global named x or not. Here for example we define x = 123 before defining and calling greet:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> x = 123 # global\n123\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\n123","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Since the x in greet is local, the value (or lack thereof) of the global x is unaffected by calling greet. The hard scope rule doesn't care whether a global named x exists or not: assignment to x in a hard scope is local (unless x is declared global).","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The next clear cut situation we'll consider is when there is already a local variable named x, in which case x = <value> always assigns to this existing local x. This is true whether the assignment occurs in the same local scope, an inner local scope in the same function body, or in the body of a function nested inside of another function, also known as a closure.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"We'll use the sum_to function, which computes the sum of integers from one up to n, as an example:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"function sum_to(n)\n    s = 0 # new local\n    for i = 1:n\n        s = s + i # assign existing local\n    end\n    return s # same local\nend","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"As in the previous example, the first assignment to s at the top of sum_to causes s to be a new local variable in the body of the function. The for loop has its own inner local scope within the function scope. At the point where s = s + i occurs, s is already a local variable, so the assignment updates the existing s instead of creating a new local. We can test this out by calling sum_to in the REPL:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function sum_to(n)\n           s = 0 # new local\n           for i = 1:n\n               s = s + i # assign existing local\n           end\n           return s # same local\n       end\nsum_to (generic function with 1 method)\n\njulia> sum_to(10)\n55\n\njulia> s # global\nERROR: UndefVarError: `s` not defined in `Main`","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Since s is local to the function sum_to, calling the function has no effect on the global variable s. We can also see that the update s = s + i in the for loop must have updated the same s created by the initialization s = 0 since we get the correct sum of 55 for the integers 1 through 10.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Let's dig into the fact that the for loop body has its own scope for a second by writing a slightly more verbose variation which we'll call sum_to_def, in which we save the sum s + i in a variable t before updating s:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function sum_to_def(n)\n           s = 0 # new local\n           for i = 1:n\n               t = s + i # new local `t`\n               s = t # assign existing local `s`\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def (generic function with 1 method)\n\njulia> sum_to_def(10)\n(55, false)","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"This version returns s as before but it also uses the @isdefined macro to return a boolean indicating whether there is a local variable named t defined in the function's outermost local scope. As you can see, there is no t defined outside of the for loop body. This is because of the hard scope rule again: since the assignment to t occurs inside of a function, which introduces a hard scope, the assignment causes t to become a new local variable in the local scope where it appears, i.e. inside of the loop body. Even if there were a global named t, it would make no difference—the hard scope rule isn't affected by anything in global scope.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Note that the local scope of a for loop body is no different from the local scope of an inner function. This means that we could rewrite this example so that the loop body is implemented as a call to an inner helper function and it behaves the same way:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function sum_to_def_closure(n)\n           function loop_body(i)\n               t = s + i # new local `t`\n               s = t # assign same local `s` as below\n           end\n           s = 0 # new local\n           for i = 1:n\n               loop_body(i)\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def_closure (generic function with 1 method)\n\njulia> sum_to_def_closure(10)\n(55, false)","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"This example illustrates a couple of key points:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Inner function scopes are just like any other nested local scope. In particular, if a variable is already a local outside of an inner function and you assign to it in the inner function, the outer local variable is updated.\nIt doesn't matter if the definition of an outer local happens below where it is updated, the rule remains the same. The entire enclosing local scope is parsed and its locals determined before inner local meanings are resolved.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"This design means that you can generally move code in or out of an inner function without changing its meaning, which facilitates a number of common idioms in the language using closures (see do blocks).","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Let's move onto some more ambiguous cases covered by the soft scope rule. We'll explore this by extracting the bodies of the greet and sum_to_def functions into soft scope contexts. First, let's put the body of greet in a for loop—which is soft, rather than hard—and evaluate it in the REPL:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> for i = 1:3\n           x = \"hello\" # new local\n           println(x)\n       end\nhello\nhello\nhello\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Since the global x is not defined when the for loop is evaluated, the first clause of the soft scope rule applies and x is created as local to the for loop and therefore global x remains undefined after the loop executes. Next, let's consider the body of sum_to_def extracted into global scope, fixing its argument to n = 10","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"s = 0\nfor i = 1:10\n    t = s + i\n    s = t\nend\ns\n@isdefined(t)","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"What does this code do? Hint: it's a trick question. The answer is \"it depends.\" If this code is entered interactively, it behaves the same way it does in a function body. But if the code appears in a file, it  prints an ambiguity warning and throws an undefined variable error. Let's see it working in the REPL first:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> s = 0 # global\n0\n\njulia> for i = 1:10\n           t = s + i # new local `t`\n           s = t # assign global `s`\n       end\n\njulia> s # global\n55\n\njulia> @isdefined(t) # global\nfalse","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The REPL approximates being in the body of a function by deciding whether assignment inside the loop assigns to a global or creates new local based on whether a global variable by that name is defined or not. If a global by the name exists, then the assignment updates it. If no global exists, then the assignment creates a new local variable. In this example we see both cases in action:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"There is no global named t, so t = s + i creates a new t that is local to the for loop;\nThere is a global named s, so s = t assigns to it.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The second fact is why execution of the loop changes the global value of s and the first fact is why t is still undefined after the loop executes. Now, let's try evaluating this same code as though it were in a file instead:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> code = \"\"\"\n       s = 0 # global\n       for i = 1:10\n           t = s + i # new local `t`\n           s = t # new local `s` with warning\n       end\n       s, # global\n       @isdefined(t) # global\n       \"\"\";\n\njulia> include_string(Main, code)\n┌ Warning: Assignment to `s` in soft scope is ambiguous because a global variable by the same name exists: `s` will be treated as a new local. Disambiguate by using `local s` to suppress this warning or `global s` to assign to the existing global variable.\n└ @ string:4\nERROR: LoadError: UndefVarError: `s` not defined in local scope","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Here we use include_string, to evaluate code as though it were the contents of a file. We could also save code to a file and then call include on that file—the result would be the same. As you can see, this behaves quite different from evaluating the same code in the REPL. Let's break down what's happening here:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"global s is defined with the value 0 before the loop is evaluated\nthe assignment s = t occurs in a soft scope—a for loop outside of any function body or other hard scope construct\ntherefore the second clause of the soft scope rule applies, and the assignment is ambiguous so a warning is emitted\nexecution continues, making s local to the for loop body\nsince s is local to the for loop, it is undefined when t = s + i is evaluated, causing an error\nevaluation stops there, but if it got to s and @isdefined(t), it would return 0 and false.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"This demonstrates some important aspects of scope: in a scope, each variable can only have one meaning, and that meaning is determined regardless of the order of expressions. The presence of the expression s = t in the loop causes s to be local to the loop, which means that it is also local when it appears on the right hand side of t = s + i, even though that expression appears first and is evaluated first. One might imagine that the s on the first line of the loop could be global while the s on the second line of the loop is local, but that's not possible since the two lines are in the same scope block and each variable can only mean one thing in a given scope.","category":"page"},{"location":"manual/variables-and-scoping.html#on-soft-scope","page":"Scope of Variables","title":"On Soft Scope","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"We have now covered all the local scope rules, but before wrapping up this section, perhaps a few words should be said about why the ambiguous soft scope case is handled differently in interactive and non-interactive contexts. There are two obvious questions one could ask:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Why doesn't it just work like the REPL everywhere?\nWhy doesn't it just work like in files everywhere? And maybe skip the warning?","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"In Julia ≤ 0.6, all global scopes did work like the current REPL: when x = <value> occurred in a loop (or try/catch, or struct body) but outside of a function body (or let block or comprehension), it was decided based on whether a global named x was defined or not whether x should be local to the loop. This behavior has the advantage of being intuitive and convenient since it approximates the behavior inside of a function body as closely as possible. In particular, it makes it easy to move code back and forth between a function body and the REPL when trying to debug the behavior of a function. However, it has some downsides. First, it's quite a complex behavior: many people over the years were confused about this behavior and complained that it was complicated and hard both to explain and understand. Fair point. Second, and arguably worse, is that it's bad for programming \"at scale.\" When you see a small piece of code in one place like this, it's quite clear what's going on:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"s = 0\nfor i = 1:10\n    s += i\nend","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Obviously the intention is to modify the existing global variable s. What else could it mean? However, not all real world code is so short or so clear. We found that code like the following often occurs in the wild:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"x = 123\n\n# much later\n# maybe in a different file\n\nfor i = 1:10\n    x = \"hello\"\n    println(x)\nend\n\n# much later\n# maybe in yet another file\n# or maybe back in the first one where `x = 123`\n\ny = x + 234","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"It's far less clear what should happen here. Since x + \"hello\" is a method error, it seems probable that the intention is for x to be local to the for loop. But runtime values and what methods happen to exist cannot be used to determine the scopes of variables. With the Julia ≤ 0.6 behavior, it's especially concerning that someone might have written the for loop first, had it working just fine, but later when someone else adds a new global far away—possibly in a different file—the code suddenly changes meaning and either breaks noisily or, worse still, silently does the wrong thing. This kind of \"spooky action at a distance\" is something that good programming language designs should prevent.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"So in Julia 1.0, we simplified the rules for scope: in any local scope, assignment to a name that wasn't already a local variable created a new local variable. This eliminated the notion of soft scope entirely as well as removing the potential for spooky action. We uncovered and fixed a significant number of bugs due to the removal of soft scope, vindicating the choice to get rid of it. And there was much rejoicing! Well, no, not really. Because some people were angry that they now had to write:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"s = 0\nfor i = 1:10\n    global s += i\nend","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Do you see that global annotation in there? Hideous. Obviously this situation could not be tolerated. But seriously, there are two main issues with requiring global for this kind of top-level code:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"It's no longer convenient to copy and paste the code from inside a function body into the REPL to debug it—you have to add global annotations and then remove them again to go back;\nBeginners will write this kind of code without the global and have no idea why their code doesn't work—the error that they get is that s is undefined, which does not seem to enlighten anyone who happens to make this mistake.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"As of Julia 1.5, this code works without the global annotation in interactive contexts like the REPL or Jupyter notebooks (just like Julia 0.6) and in files and other non-interactive contexts, it prints this very direct warning:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Assignment to s in soft scope is ambiguous because a global variable by the same name exists: s will be treated as a new local. Disambiguate by using local s to suppress this warning or global s to assign to the existing global variable.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"This addresses both issues while preserving the \"programming at scale\" benefits of the 1.0 behavior: global variables have no spooky effect on the meaning of code that may be far away; in the REPL copy-and-paste debugging works and beginners don't have any issues; any time someone either forgets a global annotation or accidentally shadows an existing global with a local in a soft scope, which would be confusing anyway, they get a nice clear warning.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"An important property of this design is that any code that executes in a file without a warning will behave the same way in a fresh REPL. And on the flip side, if you take a REPL session and save it to file, if it behaves differently than it did in the REPL, then you will get a warning.","category":"page"},{"location":"manual/variables-and-scoping.html#Let-Blocks","page":"Scope of Variables","title":"Let Blocks","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"let statements create a new hard scope block (see above) and introduce new variable bindings each time they run. The variable need not be immediately assigned:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> var1 = let x\n           for i in 1:5\n               (i == 4) && (x = i; break)\n           end\n           x\n       end\n4","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Whereas assignments might reassign a new value to an existing value location, let always creates a new location. This difference is usually not important, and is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: `z` not defined in local scope","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x since the two x variables are distinct and have separate storage. Here is an example where the behavior of let is needed:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           global i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Here we create and store two closures that return variable i. However, it is always the same variable i, so the two closures behave identically. We can use let to create a new binding for i:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           global i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Since the begin construct does not introduce a new scope, it can be useful to use a zero-argument let to just introduce a new scope block without creating any new bindings immediately:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Since let introduces a new scope block, the inner local x is a different variable than the outer local x. This particular example is equivalent to:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> let x = 1\n           let x = 2\n           end\n           x\n       end\n1","category":"page"},{"location":"manual/variables-and-scoping.html#Loops-and-Comprehensions","page":"Scope of Variables","title":"Loops and Comprehensions","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"In loops and comprehensions, new variables introduced in their body scopes are freshly allocated for each loop iteration, as if the loop body were surrounded by a let block, as demonstrated by this example:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> Fs = Vector{Any}(undef, 2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"A for loop or comprehension iteration variable is always a new variable:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"However, it is occasionally useful to reuse an existing local variable as the iteration variable. This can be done conveniently by adding the keyword outer:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3","category":"page"},{"location":"manual/variables-and-scoping.html#Constants","page":"Scope of Variables","title":"Constants","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"A common use of variables is giving names to specific, unchanging values. Such variables are only assigned once. This intent can be conveyed to the compiler using the const keyword:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Multiple variables can be declared in a single const statement:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const a, b = 1, 2\n(1, 2)","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The const declaration should only be used in global scope on globals. It is difficult for the compiler to optimize code involving global variables, since their values (or even their types) might change at almost any time. If a global variable will not change, adding a const declaration solves this performance problem.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Local constants are quite different. The compiler is able to determine automatically when a local variable is constant, so local constant declarations are not necessary, and in fact are currently not supported.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Special top-level assignments, such as those performed by the function and struct keywords, are constant by default.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Note that const only affects the variable binding; the variable may be bound to a mutable object (such as an array), and that object may still be modified. Additionally when one tries to assign a value to a variable that is declared constant the following scenarios are possible:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"if a new value has a different type than the type of the constant then an error is thrown:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const x = 1.0\n1.0\n\njulia> x = 1\nERROR: invalid redefinition of constant x","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"if a new value has the same type as the constant then a warning is printed:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const y = 1.0\n1.0\n\njulia> y = 2.0\nWARNING: redefinition of constant y. This may fail, cause incorrect answers, or produce other errors.\n2.0","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"if an assignment would not result in the change of variable value no message is given:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const z = 100\n100\n\njulia> z = 100\n100","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The last rule applies to immutable objects even if the variable binding would change, e.g.:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const s1 = \"1\"\n\"1\"\n\njulia> s2 = \"1\"\n\"1\"\n\njulia> pointer.([s1, s2], 1)\n2-element Array{Ptr{UInt8},1}:\n Ptr{UInt8} @0x00000000132c9638\n Ptr{UInt8} @0x0000000013dd3d18\n\njulia> s1 = s2\n\"1\"\n\njulia> pointer.([s1, s2], 1)\n2-element Array{Ptr{UInt8},1}:\n Ptr{UInt8} @0x0000000013dd3d18\n Ptr{UInt8} @0x0000000013dd3d18","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"However, for mutable objects the warning is printed as expected:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const a = [1]\n1-element Vector{Int64}:\n 1\n\njulia> a = [1]\nWARNING: redefinition of constant a. This may fail, cause incorrect answers, or produce other errors.\n1-element Vector{Int64}:\n 1","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Note that although sometimes possible, changing the value of a const variable is strongly discouraged, and is intended only for convenience during interactive use. Changing constants can cause various problems or unexpected behaviors. For instance, if a method references a constant and is already compiled before the constant is changed, then it might keep using the old value:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> const x = 1\n1\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> f()\n1\n\njulia> x = 2\nWARNING: redefinition of constant x. This may fail, cause incorrect answers, or produce other errors.\n2\n\njulia> f()\n1","category":"page"},{"location":"manual/variables-and-scoping.html#man-typed-globals","page":"Scope of Variables","title":"Typed Globals","text":"","category":"section"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"compat: Julia 1.8\nSupport for typed globals was added in Julia 1.8","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Similar to being declared as constants, global bindings can also be declared to always be of a constant type. This can either be done without assigning an actual value using the syntax global x::T or upon assignment as x::T = 123.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> x::Float64 = 2.718\n2.718\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> Base.return_types(f)\n1-element Vector{Any}:\n Float64","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"For any assignment to a global, Julia will first try to convert it to the appropriate type using convert:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> global y::Int\n\njulia> y = 1.0\n1.0\n\njulia> y\n1\n\njulia> y = 3.14\nERROR: InexactError: Int64(3.14)\nStacktrace:\n[...]","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"The type does not need to be concrete, but annotations with abstract types typically have little performance benefit.","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"Once a global has either been assigned to or its type has been set, the binding type is not allowed to change:","category":"page"},{"location":"manual/variables-and-scoping.html","page":"Scope of Variables","title":"Scope of Variables","text":"julia> x = 1\n1\n\njulia> global x::Int\nERROR: cannot set type for global x. It already has a value or is already set to a different type.\nStacktrace:\n[...]","category":"page"},{"location":"manual/noteworthy-differences.html#Noteworthy-Differences-from-other-Languages","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"","category":"section"},{"location":"manual/noteworthy-differences.html#Noteworthy-differences-from-MATLAB","page":"Noteworthy Differences from other Languages","title":"Noteworthy differences from MATLAB","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Although MATLAB users may find Julia's syntax familiar, Julia is not a MATLAB clone. There are major syntactic and functional differences. The following are some noteworthy differences that may trip up Julia users accustomed to MATLAB:","category":"page"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Julia arrays are indexed with square brackets, A[i,j].\nJulia arrays are not copied when assigned to another variable. After A = B, changing elements of B will modify A as well. To avoid this, use A = copy(B).\nJulia values are not copied when passed to a function. If a function modifies an array, the changes will be visible in the caller.\nJulia does not automatically grow arrays in an assignment statement. Whereas in MATLAB a(4) = 3.2 can create the array a = [0 0 0 3.2] and a(5) = 7 can grow it into a = [0 0 0 3.2 7], the corresponding Julia statement a[5] = 7 throws an error if the length of a is less than 5 or if this statement is the first use of the identifier a. Julia has push! and append!, which grow Vectors much more efficiently than MATLAB's a(end+1) = val.\nThe imaginary unit sqrt(-1) is represented in Julia as im, not i or j as in MATLAB.\nIn Julia, literal numbers without a decimal point (such as 42) create integers instead of floating point numbers. As a result, some operations can throw a domain error if they expect a float; for example, julia> a = -1; 2^a throws a domain error, as the result is not an integer (see the FAQ entry on domain errors for details).\nIn Julia, multiple values are returned and assigned as tuples, e.g. (a, b) = (1, 2) or a, b = 1, 2. MATLAB's nargout, which is often used in MATLAB to do optional work based on the number of returned values, does not exist in Julia. Instead, users can use optional and keyword arguments to achieve similar capabilities.\nJulia has true one-dimensional arrays. Column vectors are of size N, not Nx1. For example, rand(N) makes a 1-dimensional array.\nIn Julia, [x,y,z] will always construct a 3-element array containing x, y and z.\nTo concatenate in the first (\"vertical\") dimension use either vcat(x,y,z) or separate with semicolons ([x; y; z]).\nTo concatenate in the second (\"horizontal\") dimension use either hcat(x,y,z) or separate with spaces ([x y z]).\nTo construct block matrices (concatenating in the first two dimensions), use either hvcat or combine spaces and semicolons ([a b; c d]).\nIn Julia, a:b and a:b:c construct AbstractRange objects. To construct a full vector like in MATLAB, use collect(a:b). Generally, there is no need to call collect though. An AbstractRange object will act like a normal array in most cases but is more efficient because it lazily computes its values. This pattern of creating specialized objects instead of full arrays is used frequently, and is also seen in functions such as range, or with iterators such as enumerate, and zip. The special objects can mostly be used as if they were normal arrays.\nFunctions in Julia return values from their last expression or the return keyword instead of listing the names of variables to return in the function definition (see The return Keyword for details).\nA Julia script may contain any number of functions, and all definitions will be externally visible when the file is loaded. Function definitions can be loaded from files outside the current working directory.\nIn Julia, reductions such as sum, prod, and maximum are performed over every element of an array when called with a single argument, as in sum(A), even if A has more than one dimension.\nIn Julia, parentheses must be used to call a function with zero arguments, like in rand().\nJulia discourages the use of semicolons to end statements. The results of statements are not automatically printed (except at the interactive prompt), and lines of code do not need to end with semicolons. println or @printf can be used to print specific output.\nIn Julia, if A and B are arrays, logical comparison operations like A == B do not return an array of booleans. Instead, use A .== B, and similarly for the other boolean operators like <, >.\nIn Julia, the operators &, |, and ⊻ (xor) perform the bitwise operations equivalent to and, or, and xor respectively in MATLAB, and have precedence similar to Python's bitwise operators (unlike C). They can operate on scalars or element-wise across arrays and can be used to combine logical arrays, but note the difference in order of operations: parentheses may be required (e.g., to select elements of A equal to 1 or 2 use (A .== 1) .| (A .== 2)).\nIn Julia, the elements of a collection can be passed as arguments to a function using the splat operator ..., as in xs=[1,2]; f(xs...).\nJulia's svd returns singular values as a vector instead of as a dense diagonal matrix.\nIn Julia, ... is not used to continue lines of code. Instead, incomplete expressions automatically continue onto the next line.\nIn both Julia and MATLAB, the variable ans is set to the value of the last expression issued in an interactive session. In Julia, unlike MATLAB, ans is not set when Julia code is run in non-interactive mode.\nJulia's structs do not support dynamically adding fields at runtime, unlike MATLAB's classes. Instead, use a Dict. Dict in Julia isn't ordered.\nIn Julia each module has its own global scope/namespace, whereas in MATLAB there is just one global scope.\nIn MATLAB, an idiomatic way to remove unwanted values is to use logical indexing, like in the expression x(x>3) or in the statement x(x>3) = [] to modify x in-place. In contrast, Julia provides the higher order functions filter and filter!, allowing users to write filter(z->z>3, x) and filter!(z->z>3, x) as alternatives to the corresponding transliterations x[x.>3] and x = x[x.>3]. Using filter! reduces the use of temporary arrays.\nThe analogue of extracting (or \"dereferencing\") all elements of a cell array, e.g. in vertcat(A{:}) in MATLAB, is written using the splat operator in Julia, e.g. as vcat(A...).\nIn Julia, the adjoint function performs conjugate transposition; in MATLAB, adjoint provides the \"adjugate\" or classical adjoint, which is the transpose of the matrix of cofactors.\nIn Julia, a^b^c is evaluated a^(b^c) while in MATLAB it's (a^b)^c.","category":"page"},{"location":"manual/noteworthy-differences.html#Noteworthy-differences-from-R","page":"Noteworthy Differences from other Languages","title":"Noteworthy differences from R","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"One of Julia's goals is to provide an effective language for data analysis and statistical programming. For users coming to Julia from R, these are some noteworthy differences:","category":"page"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Julia's single quotes enclose characters, not strings.\nJulia can create substrings by indexing into strings. In R, strings must be converted into character vectors before creating substrings.\nIn Julia, like Python but unlike R, strings can be created with triple quotes \"\"\" ... \"\"\". This syntax is convenient for constructing strings that contain line breaks.\nIn Julia, varargs are specified using the splat operator ..., which always follows the name of a specific variable, unlike R, for which ... can occur in isolation.\nIn Julia, modulus is mod(a, b), not a %% b. % in Julia is the remainder operator.\nJulia constructs vectors using brackets. Julia's [1, 2, 3] is the equivalent of R's c(1, 2, 3).\nIn Julia, not all data structures support logical indexing. Furthermore, logical indexing in Julia is supported only with vectors of length equal to the object being indexed. For example:\nIn R, c(1, 2, 3, 4)[c(TRUE, FALSE)] is equivalent to c(1, 3).\nIn R, c(1, 2, 3, 4)[c(TRUE, FALSE, TRUE, FALSE)] is equivalent to c(1, 3).\nIn Julia, [1, 2, 3, 4][[true, false]] throws a BoundsError.\nIn Julia, [1, 2, 3, 4][[true, false, true, false]] produces [1, 3].\nLike many languages, Julia does not always allow operations on vectors of different lengths, unlike R where the vectors only need to share a common index range.  For example, c(1, 2, 3, 4) + c(1, 2) is valid R but the equivalent [1, 2, 3, 4] + [1, 2] will throw an error in Julia.\nJulia allows an optional trailing comma when that comma does not change the meaning of code. This can cause confusion among R users when indexing into arrays. For example, x[1,] in R would return the first row of a matrix; in Julia, however, the comma is ignored, so x[1,] == x[1], and will return the first element. To extract a row, be sure to use :, as in x[1,:].\nJulia's map takes the function first, then its arguments, unlike lapply(<structure>, function, ...) in R. Similarly Julia's equivalent of apply(X, MARGIN, FUN, ...) in R is mapslices where the function is the first argument.\nMultivariate apply in R, e.g. mapply(choose, 11:13, 1:3), can be written as broadcast(binomial, 11:13, 1:3) in Julia. Equivalently Julia offers a shorter dot syntax for vectorizing functions binomial.(11:13, 1:3).\nJulia uses end to denote the end of conditional blocks, like if, loop blocks, like while/ for, and functions. In lieu of the one-line if ( cond ) statement, Julia allows statements of the form if cond; statement; end, cond && statement and !cond || statement. Assignment statements in the latter two syntaxes must be explicitly wrapped in parentheses, e.g. cond && (x = value).\nIn Julia, <-, <<- and -> are not assignment operators.\nJulia's -> creates an anonymous function.\nJulia's * operator can perform matrix multiplication, unlike in R. If A and B are matrices, then A * B denotes a matrix multiplication in Julia, equivalent to R's A %*% B. In R, this same notation would perform an element-wise (Hadamard) product. To get the element-wise multiplication operation, you need to write A .* B in Julia.\nJulia performs matrix transposition using the transpose function and conjugated transposition using the ' operator or the adjoint function. Julia's transpose(A) is therefore equivalent to R's t(A). Additionally a non-recursive transpose in Julia is provided by the permutedims function.\nJulia does not require parentheses when writing if statements or for/while loops: use for i in [1, 2, 3] instead of for (i in c(1, 2, 3)) and if i == 1 instead of if (i == 1).\nJulia does not treat the numbers 0 and 1 as Booleans. You cannot write if (1) in Julia, because if statements accept only booleans. Instead, you can write if true, if Bool(1), or if 1==1.\nJulia does not provide nrow and ncol. Instead, use size(M, 1) for nrow(M) and size(M, 2) for ncol(M).\nJulia is careful to distinguish scalars, vectors and matrices.  In R, 1 and c(1) are the same. In Julia, they cannot be used interchangeably.\nJulia's diag and diagm are not like R's.\nJulia cannot assign to the results of function calls on the left hand side of an assignment operation: you cannot write diag(M) = fill(1, n).\nJulia discourages populating the main namespace with functions. Most statistical functionality for Julia is found in packages under the JuliaStats organization. For example:\nFunctions pertaining to probability distributions are provided by the Distributions package.\nThe DataFrames package provides data frames.\nGeneralized linear models are provided by the GLM package.\nJulia provides tuples and real hash tables, but not R-style lists. When returning multiple items, you should typically use a tuple or a named tuple: instead of list(a = 1, b = 2), use (1, 2) or (a=1, b=2).\nJulia encourages users to write their own types, which are easier to use than S3 or S4 objects in R. Julia's multiple dispatch system means that table(x::TypeA) and table(x::TypeB) act like R's table.TypeA(x) and table.TypeB(x).\nIn Julia, values are not copied when assigned or passed to a function. If a function modifies an array, the changes will be visible in the caller. This is very different from R and allows new functions to operate on large data structures much more efficiently.\nIn Julia, vectors and matrices are concatenated using hcat, vcat and hvcat, not c, rbind and cbind like in R.\nIn Julia, a range like a:b is not shorthand for a vector like in R, but is a specialized AbstractRange object that is used for iteration. To convert a range into a vector, use collect(a:b).\nThe : operator has a different precedence in R and Julia. In particular, in Julia arithmetic operators have higher precedence than the : operator, whereas the reverse is true in R. For example, 1:n-1 in Julia is equivalent to 1:(n-1) in R.\nJulia's max and min are the equivalent of pmax and pmin respectively in R, but both arguments need to have the same dimensions.  While maximum and minimum replace max and min in R, there are important differences.\nJulia's sum, prod, maximum, and minimum are different from their counterparts in R. They all accept an optional keyword argument dims, which indicates the dimensions, over which the operation is carried out.  For instance, let A = [1 2; 3 4] in Julia and B <- rbind(c(1,2),c(3,4)) be the same matrix in R.  Then sum(A) gives the same result as sum(B), but sum(A, dims=1) is a row vector containing the sum over each column and sum(A, dims=2) is a column vector containing the sum over each row. This contrasts to the behavior of R, where separate colSums(B) and rowSums(B) functions provide these functionalities. If the dims keyword argument is a vector, then it specifies all the dimensions over which the sum is performed, while retaining the dimensions of the summed array, e.g. sum(A, dims=(1,2)) == hcat(10). It should be noted that there is no error checking regarding the second argument.\nJulia has several functions that can mutate their arguments. For example, it has both sort and sort!.\nIn R, performance requires vectorization. In Julia, almost the opposite is true: the best performing code is often achieved by using devectorized loops.\nJulia is eagerly evaluated and does not support R-style lazy evaluation. For most users, this means that there are very few unquoted expressions or column names.\nJulia does not support the NULL type. The closest equivalent is nothing, but it behaves like a scalar value rather than like a list. Use x === nothing instead of is.null(x).\nIn Julia, missing values are represented by the missing object rather than by NA. Use ismissing(x) (or ismissing.(x) for element-wise operation on vectors) instead of is.na(x). The skipmissing function is generally used instead of na.rm=TRUE (though in some particular cases functions take a skipmissing argument).\nJulia lacks the equivalent of R's assign or get.\nIn Julia, return does not require parentheses.\nIn R, an idiomatic way to remove unwanted values is to use logical indexing, like in the expression x[x>3] or in the statement x = x[x>3] to modify x in-place. In contrast, Julia provides the higher order functions filter and filter!, allowing users to write filter(z->z>3, x) and filter!(z->z>3, x) as alternatives to the corresponding transliterations x[x.>3] and x = x[x.>3]. Using filter! reduces the use of temporary arrays.","category":"page"},{"location":"manual/noteworthy-differences.html#Noteworthy-differences-from-Python","page":"Noteworthy Differences from other Languages","title":"Noteworthy differences from Python","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Julia's for, if, while, etc. blocks are terminated by the end keyword. Indentation level is not significant as it is in Python. Unlike Python, Julia has no pass keyword.\nStrings are denoted by double quotation marks (\"text\") in Julia (with three double quotation marks for multi-line strings), whereas in Python they can be denoted either by single ('text') or double quotation marks (\"text\"). Single quotation marks are used for characters in Julia ('c').\nString concatenation is done with * in Julia, not + like in Python. Analogously, string repetition is done with ^, not *. Implicit string concatenation of string literals like in Python (e.g. 'ab' 'cd' == 'abcd') is not done in Julia.\nPython Lists—flexible but slow—correspond to the Julia Vector{Any} type or more generally Vector{T} where T is some non-concrete element type. \"Fast\" arrays like NumPy arrays that store elements in-place (i.e., dtype is np.float64, [('f1', np.uint64), ('f2', np.int32)], etc.) can be represented by Array{T} where T is a concrete, immutable element type. This includes built-in types like Float64, Int32, Int64 but also more complex types like Tuple{UInt64,Float64} and many user-defined types as well.\nIn Julia, indexing of arrays, strings, etc. is 1-based not 0-based.\nJulia's slice indexing includes the last element, unlike in Python. a[2:3] in Julia is a[1:3] in Python.\nUnlike Python, Julia allows AbstractArrays with arbitrary indexes. Python's special interpretation of negative indexing, a[-1] and a[-2], should be written a[end] and a[end-1] in Julia.\nJulia requires end for indexing until the last element. x[1:] in Python is equivalent to x[2:end] in Julia.\nIn Julia, : before any object creates a Symbol or quotes an expression; so, x[:5] is same as x[5]. If you want to get the first n elements of an array, then use range indexing.\nJulia's range indexing has the format of x[start:step:stop], whereas Python's format is x[start:(stop+1):step]. Hence, x[0:10:2] in Python is equivalent to x[1:2:10] in Julia. Similarly, x[::-1] in Python, which refers to the reversed array, is equivalent to x[end:-1:1] in Julia.\nIn Julia, ranges can be constructed independently as start:step:stop, the same syntax it uses in array-indexing.  The range function is also supported.\nIn Julia, indexing a matrix with arrays like X[[1,2], [1,3]] refers to a sub-matrix that contains the intersections of the first and second rows with the first and third columns. In Python, X[[1,2], [1,3]] refers to a vector that contains the values of cell [1,1] and [2,3] in the matrix. X[[1,2], [1,3]] in Julia is equivalent with X[np.ix_([0,1],[0,2])] in Python. X[[0,1], [0,2]] in Python is equivalent with X[[CartesianIndex(1,1), CartesianIndex(2,3)]] in Julia.\nJulia has no line continuation syntax: if, at the end of a line, the input so far is a complete expression, it is considered done; otherwise the input continues. One way to force an expression to continue is to wrap it in parentheses.\nJulia arrays are column-major (Fortran-ordered) whereas NumPy arrays are row-major (C-ordered) by default. To get optimal performance when looping over arrays, the order of the loops should be reversed in Julia relative to NumPy (see relevant section of Performance Tips).\nJulia's updating operators (e.g. +=, -=, ...) are not in-place whereas NumPy's are. This means A = [1, 1]; B = A; B += [3, 3] doesn't change values in A, it rather rebinds the name B to the result of the right-hand side B = B + 3, which is a new array. For in-place operation, use B .+= 3 (see also dot operators), explicit loops, or InplaceOps.jl.\nJulia evaluates default values of function arguments every time the method is invoked, unlike in Python where the default values are evaluated only once when the function is defined. For example, the function f(x=rand()) = x returns a new random number every time it is invoked without argument. On the other hand, the function g(x=[1,2]) = push!(x,3) returns [1,2,3] every time it is called as g().\nIn Julia, keyword arguments must be passed using keywords, unlike Python in which it is usually possible to pass them positionally. Attempting to pass a keyword argument positionally alters the method signature leading to a MethodError or calling of the wrong method.\nIn Julia % is the remainder operator, whereas in Python it is the modulus.\nIn Julia, the commonly used Int type corresponds to the machine integer type (Int32 or Int64), unlike in Python, where int is an arbitrary length integer. This means in Julia the Int type will overflow, such that 2^64 == 0. If you need larger values use another appropriate type, such as Int128, BigInt or a floating point type like Float64.\nThe imaginary unit sqrt(-1) is represented in Julia as im, not j as in Python.\nIn Julia, the exponentiation operator is ^, not ** as in Python.\nJulia uses nothing of type Nothing to represent a null value, whereas Python uses None of type NoneType.\nIn Julia, the standard operators over a matrix type are matrix operations, whereas, in Python, the standard operators are element-wise operations. When both A and B are matrices, A * B in Julia performs matrix multiplication, not element-wise multiplication as in Python. A * B in Julia is equivalent with A @ B in Python, whereas A * B in Python is equivalent with A .* B in Julia.\nThe adjoint operator ' in Julia returns an adjoint of a vector (a lazy representation of row vector), whereas the transpose operator .T over a vector in Python returns the original vector (non-op).\nIn Julia, a function may contain multiple concrete implementations (called methods), which are selected via multiple dispatch based on the types of all arguments to the call, as compared to functions in Python, which have a single implementation and no polymorphism (as opposed to Python method calls which use a different syntax and allows dispatch on the receiver of the method).\nThere are no classes in Julia. Instead there are structures (mutable or immutable), containing data but no methods.\nCalling a method of a class instance in Python (x = MyClass(*args); x.f(y)) corresponds to a function call in Julia, e.g. x = MyType(args...); f(x, y). In general, multiple dispatch is more flexible and powerful than the Python class system.\nJulia structures may have exactly one abstract supertype, whereas Python classes can inherit from one or more (abstract or concrete) superclasses.\nThe logical Julia program structure (Packages and Modules) is independent of the file structure, whereas the Python code structure is defined by directories (Packages) and files (Modules).\nIn Julia, it is idiomatic to split the text of large modules into multiple files, without introducing a new module per file. The code is reassembled inside a single module in a main file via include. While the Python equivalent (exec) is not typical for this use (it will silently clobber prior definitions), Julia programs are defined as a unit at the module level with using or import, which will only get executed once when first needed–like include in Python. Within those modules, the individual files that make up that module are loaded with include by listing them once in the intended order.\nThe ternary operator x > 0 ? 1 : -1 in Julia corresponds to a conditional expression in Python 1 if x > 0 else -1.\nIn Julia the @ symbol refers to a macro, whereas in Python it refers to a decorator.\nException handling in Julia is done using try — catch — finally, instead of try — except — finally. In contrast to Python, it is not recommended to use exception handling as part of the normal workflow in Julia (compared with Python, Julia is faster at ordinary control flow but slower at exception-catching).\nIn Julia loops are fast, there is no need to write \"vectorized\" code for performance reasons.\nBe careful with non-constant global variables in Julia, especially in tight loops. Since you can write close-to-metal code in Julia (unlike Python), the effect of globals can be drastic (see Performance Tips).\nIn Julia, rounding and truncation are explicit. Python's int(3.7) should be floor(Int, 3.7) or Int(floor(3.7)) and is distinguished from round(Int, 3.7). floor(x) and round(x) on their own return an integer value of the same type as x rather than always returning Int.\nIn Julia, parsing is explicit. Python's float(\"3.7\") would be parse(Float64, \"3.7\") in Julia.\nIn Python, the majority of values can be used in logical contexts (e.g. if \"a\": means the following block is executed, and if \"\": means it is not). In Julia, you need explicit conversion to Bool (e.g. if \"a\" throws an exception). If you want to test for a non-empty string in Julia, you would explicitly write if !isempty(\"\").  Perhaps surprisingly, in Python if \"False\" and bool(\"False\") both evaluate to True (because \"False\" is a non-empty string); in Julia, parse(Bool, \"false\") returns false.\nIn Julia, a new local scope is introduced by most code blocks, including loops and try — catch — finally. Note that comprehensions (list, generator, etc.) introduce a new local scope both in Python and Julia, whereas if blocks do not introduce a new local scope in both languages.","category":"page"},{"location":"manual/noteworthy-differences.html#Noteworthy-differences-from-C/C","page":"Noteworthy Differences from other Languages","title":"Noteworthy differences from C/C++","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Julia arrays are indexed with square brackets, and can have more than one dimension A[i,j]. This syntax is not just syntactic sugar for a reference to a pointer or address as in C/C++. See the manual entry about array construction.\nIn Julia, indexing of arrays, strings, etc. is 1-based not 0-based.\nJulia arrays are not copied when assigned to another variable. After A = B, changing elements of B will modify A as well. Updating operators like += do not operate in-place, they are equivalent to A = A + B which rebinds the left-hand side to the result of the right-hand side expression.\nJulia arrays are column major (Fortran ordered) whereas C/C++ arrays are row major ordered by default. To get optimal performance when looping over arrays, the order of the loops should be reversed in Julia relative to C/C++ (see relevant section of Performance Tips).\nJulia values are not copied when assigned or passed to a function. If a function modifies an array, the changes will be visible in the caller.\nIn Julia, whitespace is significant, unlike C/C++, so care must be taken when adding/removing whitespace from a Julia program.\nIn Julia, literal numbers without a decimal point (such as 42) create signed integers, of type Int, but literals too large to fit in the machine word size will automatically be promoted to a larger size type, such as Int64 (if Int is Int32), Int128, or the arbitrarily large BigInt type. There are no numeric literal suffixes, such as L, LL, U, UL, ULL to indicate unsigned and/or signed vs. unsigned. Decimal literals are always signed, and hexadecimal literals (which start with 0x like C/C++), are unsigned, unless when they encode more than 128 bits, in which case they are of type BigInt. Hexadecimal literals also, unlike C/C++/Java and unlike decimal literals in Julia, have a type based on the length of the literal, including leading 0s. For example, 0x0 and 0x00 have type UInt8, 0x000 and 0x0000 have type UInt16, then literals with 5 to 8 hex digits have type UInt32, 9 to 16 hex digits type UInt64, 17 to 32 hex digits type UInt128, and more that 32 hex digits type BigInt. This needs to be taken into account when defining hexadecimal masks, for example ~0xf == 0xf0 is very different from ~0x000f == 0xfff0. 64 bit Float64 and 32 bit Float32 bit literals are expressed as 1.0 and 1.0f0 respectively. Floating point literals are rounded (and not promoted to the BigFloat type) if they can not be exactly represented.  Floating point literals are closer in behavior to C/C++. Octal (prefixed with 0o) and binary (prefixed with 0b) literals are also treated as unsigned (or BigInt for more than 128 bits).\nIn Julia, the division operator / returns a floating point number when both operands are of integer type.  To perform integer division, use div or ÷.\nIndexing an Array with floating point types is generally an error in Julia. The Julia equivalent of the C expression a[i / 2] is a[i ÷ 2 + 1], where i is of integer type.\nString literals can be delimited with either \"  or \"\"\", \"\"\" delimited literals can contain \" characters without quoting it like \"\\\"\". String literals can have values of other variables or expressions interpolated into them, indicated by $variablename or $(expression), which evaluates the variable name or the expression in the context of the function.\n// indicates a Rational number, and not a single-line comment (which is # in Julia)\n#= indicates the start of a multiline comment, and =# ends it.\nFunctions in Julia return values from their last expression(s) or the return keyword.  Multiple values can be returned from functions and assigned as tuples, e.g. (a, b) = myfunction() or a, b = myfunction(), instead of having to pass pointers to values as one would have to do in C/C++ (i.e. a = myfunction(&b).\nJulia does not require the use of semicolons to end statements. The results of expressions are not automatically printed (except at the interactive prompt, i.e. the REPL), and lines of code do not need to end with semicolons. println or @printf can be used to print specific output. In the REPL, ; can be used to suppress output. ; also has a different meaning within [ ], something to watch out for. ; can be used to separate expressions on a single line, but are not strictly necessary in many cases, and are more an aid to readability.\nIn Julia, the operator ⊻ (xor) performs the bitwise XOR operation, i.e. ^ in C/C++.  Also, the bitwise operators do not have the same precedence as C/C++, so parenthesis may be required.\nJulia's ^ is exponentiation (pow), not bitwise XOR as in C/C++ (use ⊻, or xor, in Julia)\nJulia has two right-shift operators, >> and >>>.  >> performs an arithmetic shift, >>> always performs a logical shift, unlike C/C++, where the meaning of >> depends on the type of the value being shifted.\nJulia's -> creates an anonymous function, it does not access a member via a pointer.\nJulia does not require parentheses when writing if statements or for/while loops: use for i in [1, 2, 3] instead of for (int i=1; i <= 3; i++) and if i == 1 instead of if (i == 1).\nJulia does not treat the numbers 0 and 1 as Booleans. You cannot write if (1) in Julia, because if statements accept only booleans. Instead, you can write if true, if Bool(1), or if 1==1.\nJulia uses end to denote the end of conditional blocks, like if, loop blocks, like while/ for, and functions. In lieu of the one-line if ( cond ) statement, Julia allows statements of the form if cond; statement; end, cond && statement and !cond || statement. Assignment statements in the latter two syntaxes must be explicitly wrapped in parentheses, e.g. cond && (x = value), because of the operator precedence.\nJulia has no line continuation syntax: if, at the end of a line, the input so far is a complete expression, it is considered done; otherwise the input continues. One way to force an expression to continue is to wrap it in parentheses.\nJulia macros operate on parsed expressions, rather than the text of the program, which allows them to perform sophisticated transformations of Julia code. Macro names start with the @ character, and have both a function-like syntax, @mymacro(arg1, arg2, arg3), and a statement-like syntax, @mymacro arg1 arg2 arg3. The forms are interchangeable; the function-like form is particularly useful if the macro appears within another expression, and is often clearest. The statement-like form is often used to annotate blocks, as in the distributed for construct: @distributed for i in 1:n; #= body =#; end. Where the end of the macro construct may be unclear, use the function-like form.\nJulia has an enumeration type, expressed using the macro @enum(name, value1, value2, ...) For example: @enum(Fruit, banana=1, apple, pear)\nBy convention, functions that modify their arguments have a ! at the end of the name, for example push!.\nIn C++, by default, you have static dispatch, i.e. you need to annotate a function as virtual, in order to have dynamic dispatch. On the other hand, in Julia every method is \"virtual\" (although it's more general than that since methods are dispatched on every argument type, not only this, using the most-specific-declaration rule).","category":"page"},{"location":"manual/noteworthy-differences.html#Julia-C/C:-Namespaces","page":"Noteworthy Differences from other Languages","title":"Julia ⇔ C/C++: Namespaces","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"C/C++ namespaces correspond roughly to Julia modules.\nThere are no private globals or fields in Julia.  Everything is publicly accessible through fully qualified paths (or relative paths, if desired).\nusing MyNamespace::myfun (C++) corresponds roughly to import MyModule: myfun (Julia).\nusing namespace MyNamespace (C++) corresponds roughly to using MyModule (Julia)\nIn Julia, only exported symbols are made available to the calling module.\nIn C++, only elements found in the included (public) header files are made available.\nCaveat: import/using keywords (Julia) also load modules (see below).\nCaveat: import/using (Julia) works only at the global scope level (modules)\nIn C++, using namespace X works within arbitrary scopes (ex: function scope).","category":"page"},{"location":"manual/noteworthy-differences.html#Julia-C/C:-Module-loading","page":"Noteworthy Differences from other Languages","title":"Julia ⇔ C/C++: Module loading","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"When you think of a C/C++ \"library\", you are likely looking for a Julia \"package\".\nCaveat: C/C++ libraries often house multiple \"software modules\" whereas Julia \"packages\" typically house one.\nReminder: Julia modules are global scopes (not necessarily \"software modules\").\nInstead of build/make scripts, Julia uses \"Project Environments\" (sometimes called either \"Project\" or \"Environment\").\nBuild scripts are only needed for more complex applications (like those needing to compile or download C/C++ executables).\nTo develop application or project in Julia, you can initialize its root directory as a \"Project Environment\", and house application-specific code/packages there. This provides good control over project dependencies, and future reproducibility.\nAvailable packages are added to a \"Project Environment\" with the Pkg.add() function or Pkg REPL mode. (This does not load said package, however).\nThe list of available packages (direct dependencies) for a \"Project Environment\" are saved in its Project.toml file.\nThe full dependency information for a \"Project Environment\" is auto-generated & saved in its Manifest.toml file by Pkg.resolve().\nPackages (\"software modules\") available to the \"Project Environment\" are loaded with import or using.\nIn C/C++, you #include <moduleheader> to get object/function declarations, and link in libraries when you build the executable.\nIn Julia, calling using/import again just brings the existing module into scope, but does not load it again (similar to adding the non-standard #pragma once to C/C++).\nDirectory-based package repositories (Julia) can be made available by adding repository paths to the Base.LOAD_PATH array.\nPackages from directory-based repositories do not require the Pkg.add() tool prior to being loaded with import or using. They are simply available to the project.\nDirectory-based package repositories are the quickest solution to developing local libraries of \"software modules\".","category":"page"},{"location":"manual/noteworthy-differences.html#Julia-C/C:-Assembling-modules","page":"Noteworthy Differences from other Languages","title":"Julia ⇔ C/C++: Assembling modules","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"In C/C++, .c/.cpp files are compiled & added to a library with build/make scripts.\nIn Julia, import [PkgName]/using [PkgName] statements load [PkgName].jl located in a package's [PkgName]/src/ subdirectory.\nIn turn, [PkgName].jl typically loads associated source files with calls to include \"[someotherfile].jl\".\ninclude \"./path/to/somefile.jl\" (Julia) is very similar to #include \"./path/to/somefile.jl\" (C/C++).\nHowever include \"...\" (Julia) is not used to include header files (not required).\nDo not use include \"...\" (Julia) to load code from other \"software modules\" (use import/using instead).\ninclude \"path/to/some/module.jl\" (Julia) would instantiate multiple versions of the same code in different modules (creating distinct types (etc.) with the same names).\ninclude \"somefile.jl\" is typically used to assemble multiple files within the same Julia package (\"software module\"). It is therefore relatively straightforward to ensure file are included only once (No #ifdef confusion).","category":"page"},{"location":"manual/noteworthy-differences.html#Julia-C/C:-Module-interface","page":"Noteworthy Differences from other Languages","title":"Julia ⇔ C/C++: Module interface","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"C++ exposes interfaces using \"public\" .h/.hpp files whereas Julia modules mark specific symbols that are intended for their users as publicor exported.\nOften, Julia modules simply add functionality by generating new \"methods\" to existing functions (ex: Base.push!).\nDevelopers of Julia packages therefore cannot rely on header files for interface documentation.\nInterfaces for Julia packages are typically described using docstrings, README.md, static web pages, ...\nSome developers choose not to export all symbols required to use their package/module, but should still mark unexported user facing symbols as public.\nUsers might be expected to access these components by qualifying functions/structs/... with the package/module name (ex: MyModule.run_this_task(...)).","category":"page"},{"location":"manual/noteworthy-differences.html#Julia-C/C:-Quick-reference","page":"Noteworthy Differences from other Languages","title":"Julia ⇔ C/C++: Quick reference","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Software Concept Julia C/C++\nunnamed scope begin ... end { ... }\nfunction scope function x() ... end int x() { ... }\nglobal scope module MyMod ... end namespace MyNS { ... }\nsoftware module A Julia \"package\" .h/.hpp files<br>+compiled somelib.a\nassembling<br>software modules SomePkg.jl: ...<br>import(\"subfile1.jl\")<br>import(\"subfile2.jl\")<br>... $(AR) *.o &rArr; somelib.a\nimport<br>software module import SomePkg #include <somelib><br>+link in somelib.a\nmodule library LOAD_PATH[], *Git repository,<br>**custom package registry more .h/.hpp files<br>+bigger compiled somebiglib.a","category":"page"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"* The Julia package manager supports registering multiple packages from a single Git repository.<br> * This allows users to house a library of related packages in a single repository.<br> ** Julia registries are primarily designed to provide versioning \\& distribution of packages.<br> ** Custom package registries can be used to create a type of module library.","category":"page"},{"location":"manual/noteworthy-differences.html#Noteworthy-differences-from-Common-Lisp","page":"Noteworthy Differences from other Languages","title":"Noteworthy differences from Common Lisp","text":"","category":"section"},{"location":"manual/noteworthy-differences.html","page":"Noteworthy Differences from other Languages","title":"Noteworthy Differences from other Languages","text":"Julia uses 1-based indexing for arrays by default, and it can also handle arbitrary index offsets.\nFunctions and variables share the same namespace (“Lisp-1”).\nThere is a Pair type, but it is not meant to be used as a COMMON-LISP:CONS. Various iterable collections can be used interchangeably in most parts of the language (eg splatting, tuples, etc). Tuples are the closest to Common Lisp lists for short collections of heterogeneous elements. Use NamedTuples in place of alists. For larger collections of homogeneous types, Arrays and Dicts should be used.\nThe typical Julia workflow for prototyping also uses continuous manipulation of the image, implemented with the Revise.jl package.\nFor performance, Julia prefers that operations have type stability. Where Common Lisp abstracts away from the underlying machine operations, Julia cleaves closer to them. For example:\nInteger division using / always returns a floating-point result, even if the computation is exact.\n// always returns a rational result\n÷ always returns a (truncated) integer result\nBignums are supported, but conversion is not automatic; ordinary integers overflow.\nComplex numbers are supported, but to get complex results, you need complex inputs.\nThere are multiple Complex and Rational types, with different component types.\nModules (namespaces) can be hierarchical. import and using have a dual role: they load the code and make it available in the namespace. import for only the module name is possible (roughly equivalent to ASDF:LOAD-OP). Slot names don't need to be exported separately. Global variables can't be assigned to from outside the module (except with eval(mod, :(var = val)) as an escape hatch).\nMacros start with @, and are not as seamlessly integrated into the language as Commo