import path from 'path';
import ts from 'typescript';
import { createFilter } from 'rollup-pluginutils';
import { Linter, Configuration } from 'tslint';

function normalizePath (id) {
  return path
    .relative(process.cwd(), id)
    .split(path.sep)
    .join('/')
}

function isString (value) {
  return Object.prototype.toString.call(value) === '[object String]'
}

function tslint (options) {
  if ( options === void 0 ) options = {};

  var linter;

  var filter = createFilter(
    options.include,
    options.exclude || 'node_modules/**'
  );

  // formatter: "stylish"
  // rulesDirectory: null,
  // formattersDirectory: "customFormatters/"

  var tsConfigSearchPath = options.tsConfigSearchPath || process.cwd();
  var tsConfigFile = ts.findConfigFile(tsConfigSearchPath, ts.sys.fileExists);

  var config = {
    fix: options.fix || false,
    formatter: options.formatter || 'stylish',
    formattersDirectory: options.formattersDirectory || null,
    rulesDirectory: options.rulesDirectory || null
  };

  return {
    name: 'tslint',
    sourceMap: false,

    options: function options () {
      var program = Linter.createProgram(tsConfigFile);

      linter = new Linter(config, program);
    },

    transform: function transform (code, id) {
      var fileName = normalizePath(id);
      if (!filter(id)) {
        return null
      }

      var configuration =
        options.configuration === null ||
        options.configuration === undefined ||
        isString(options.configuration)
          ? Configuration.findConfiguration(
            options.configuration || null,
            fileName
          ).results
          : Configuration.parseConfigFile(options.configuration, process.cwd());

      linter.lint(id, code, configuration);
      var result = linter.getResult();

      // Clear all results for current file from tslint
      linter.failures = [];
      linter.fixes = [];

      var hasWarnings = options.throwOnWarning && result.warningCount !== 0;
      var hasErrors = (options.throwOnError || options.throwError) && result.errorCount !== 0;

      if (result.errorCount || result.warningCount) {
        console.log(result.output);
      }

      if (hasWarnings && hasErrors) {
        throw Error('Warnings or errors were found')
      }

      if (hasWarnings) {
        throw Error('Warnings were found')
      }

      if (hasErrors) {
        throw Error('Errors were found')
      }
    }
  }
}

export default tslint;
//# sourceMappingURL=rollup-plugin-tslint.es.js.map
