// Copyright (c) Huawei Technologies Co., Ltd. 2025. All rights reserved.
// This source file is part of the Cangjie project, licensed under Apache-2.0
// with Runtime Library Exception.
//
// See https://cangjie-lang.cn/pages/LICENSE for license information.


#ifndef MRT_RUNTIME_H
#define MRT_RUNTIME_H

// The general runtime interface/Base to provide a necessary
//  top-level access point for sub-module interaction, since
// runtime holds the instance of sub-modules (managers)
#include "Base/Log.h"

namespace MapleRuntime {
extern "C" {
// C interfaces for scan stack debug.
void MRT_FlushGCInfo();
} // extern "C"

// The manager to coordinate other runtime components

class HeapManager;
class ObjectManager;
class ExceptionManager;
class StackManager;
class SignalManager;
class LogManager;
class LoaderManager;
class ThreadManager;
class MutatorManager;
class ConcurrencyModel;
class TypeInfoManager;
#ifdef _WIN64
class WinModuleManager;
#endif

class Runtime {
public:
    // return singleton of runtime instance.
    static inline Runtime& Current()
    {
        CHECK(runtime != nullptr);
        return *runtime;
    }

    static inline Runtime* CurrentRef() { return runtime; }

    inline SignalManager& GetSignalManager()
    {
        CHECK(signalManager != nullptr);
        return *signalManager;
    }

    inline HeapManager& GetHeapManager()
    {
        CHECK(heapManager != nullptr);
        return *heapManager;
    }

    inline ObjectManager& GetObjectManager()
    {
        CHECK(objectManager != nullptr);
        return *objectManager;
    }

    inline ExceptionManager& GetExceptionManager()
    {
        CHECK(exceptionManager != nullptr);
        return *exceptionManager;
    }

    inline StackManager& GetStackManager()
    {
        CHECK(stackManager != nullptr);
        return *stackManager;
    }

    inline MutatorManager& GetMutatorManager()
    {
        CHECK(mutatorManager != nullptr);
        return *mutatorManager;
    }

    inline ConcurrencyModel& GetConcurrencyModel()
    {
        CHECK(concurrencyModel != nullptr);
        return *concurrencyModel;
    }

    inline LoaderManager& GetLoaderManager()
    {
        CHECK(loaderManager != nullptr);
        return *loaderManager;
    }

#ifdef _WIN64
    inline WinModuleManager& GetWinModuleManager()
    {
        CHECK(winModuleManager != nullptr);
        return *winModuleManager;
    }
#endif

    virtual RuntimeParam GetRuntimeParam() const = 0;
    virtual void SetGCThreshold(uint64_t threshold) = 0;

protected:
    static Runtime* runtime;

    Runtime() = default;
    virtual ~Runtime() = default;

    ConcurrencyModel* concurrencyModel = nullptr; // managed threads
    SignalManager* signalManager = nullptr;       // managed signal handling
    HeapManager* heapManager = nullptr;           // managed heap
    ObjectManager* objectManager = nullptr;       // object model
    ExceptionManager* exceptionManager = nullptr; // managed exception
    StackManager* stackManager = nullptr;         // managed stack
    LogManager* logManager = nullptr;             // log
    LoaderManager* loaderManager = nullptr;       // loader module
    MutatorManager* mutatorManager = nullptr;     // manage all mutators registered
    TypeInfoManager* typeInfoManager = nullptr;   // manage TypeInfo generated by generics
#ifdef _WIN64
    WinModuleManager* winModuleManager = nullptr; // manage windows module data
#endif
};
} // namespace MapleRuntime
#endif // MRT_RUNTIME_H
