// Copyright (c) Huawei Technologies Co., Ltd. 2025. All rights reserved.
// This source file is part of the Cangjie project, licensed under Apache-2.0
// with Runtime Library Exception.
//
// See https://cangjie-lang.cn/pages/LICENSE for license information.

#include "StdioTransport.h"
#include "../languageserver/capabilities/shutdown/Shutdown.h"
#include "../languageserver/logger/Logger.h"

namespace ark {
nlohmann::json EncodeError(const MessageErrorDetail &errorInfo)
{
    nlohmann::json root;
    root["message"] = errorInfo.message;
    root["code"] = static_cast<int64_t>(errorInfo.code);
    return root;
}

void StdioTransport::SetIO(std::FILE *in, std::FILE *out)
{
    pFileIn = in;
    pFileOut = out;
}

void StdioTransport::Notify(std::string method, ValueOrError params)
{
    nlohmann::json root;
    root["jsonrpc"] = "2.0";
    root["method"] = method;
    if (params.type == ValueOrErrorCheck::VALUE) {
        root["params"] = std::move(params.jsonValue);
    } else {
        root["error"] = EncodeError(params.errorInfo);
    }
    SendMsg(root);
}

void StdioTransport::Reply(nlohmann::json id, ValueOrError result)
{
    nlohmann::json root;
    root["jsonrpc"] = "2.0";
    root["id"] = std::move(id);
    if (result.type == ValueOrErrorCheck::VALUE) {
        root["result"] = std::move(result.jsonValue);
    } else {
        root["error"] = EncodeError(result.errorInfo);
    }
    SendMsg(root);
}

LSPRet StdioTransport::Loop(MessageHandler &handler)
{
    Logger &logger = Logger::Instance();
    std::stringstream log;
    while (!feof(pFileIn)) {
        auto json = ReadRawMessage();
        if (!json.empty()) {
            CleanAndLog(log, "receive message body:" + json);
            logger.LogMessage(MessageType::MSG_INFO, log.str());
            logger.CollectMessageInfo(logger.LogInfo(MessageType::MSG_INFO, json));
            nlohmann::json doc;
            try {
                doc = nlohmann::json::parse(json);
            } catch (nlohmann::detail::parse_error &errs) {
                CleanAndLog(log, errs.what());
                logger.LogMessage(MessageType::MSG_WARNING, log.str());
                continue;
            }
            LSPRet ret = HandleMessage(std::move(doc), handler);
            if (ret == LSPRet::NORMAL_EXIT || ret == LSPRet::ABNORMAL_EXIT) {
                return ret;
            }
        }
    }
    return LSPRet::ERR_IO;
}

void StdioTransport::SendMsg(const nlohmann::json &message)
{
    std::stringstream log;
    std::ostringstream os;
    // "-1", no indentation format
    os << message.dump(-1, ' ', false, nlohmann::json::error_handler_t::ignore);
    (void)fprintf(pFileOut, "Content-Length:%d%s%s",
                   static_cast<int>(os.str().size()), MessageHeaderEndOfLine::GetEol().c_str(), os.str().c_str());
    (void)fflush(pFileOut);
    CleanAndLog(log, "send message body:" + os.str());
    Logger::Instance().LogMessage(MessageType::MSG_INFO, log.str());
}

std::string StdioTransport::ReadRawMessage()
{
    return ReadStandardMessage();
}

bool ReadLine(std::FILE *in, std::string &out)
{
    static constexpr int bufSize = 1024;
    size_t size = 0;
    out.clear();
    for (;;) {
        out.resize(size + bufSize);

        if (!RetryAfterSignalUnlessShutdown(nullptr, [&] { return std::fgets(&out[size], bufSize, in); })) {
            return false;
        }
        clearerr(in);

        size_t read = std::strlen(&out[size]);
        if (read > 0 && out[size + read - 1] == '\n') {
            out.resize(size + read);
            return true;
        }
        size += read;
    }
}

void Trim(std::string &s)
{
    if (s.empty()) {
        return;
    }
    (void)s.erase(0, s.find_first_not_of(' '));
    (void)s.erase(s.find_last_not_of(' ') + 1);
}

std::string StdioTransport::ReadStandardMessage()
{
    unsigned long long contentLength = 0;
    std::string line;
    Logger &logger = Logger::Instance();
    for (;;) {
        if (feof(pFileIn) || ferror(pFileIn) || !ReadLine(pFileIn, line)) { return ""; }

        if (line.front() == '#') { continue; }
        Trim(line);
        auto found = line.find("Content-Length:");
        if (found == std::string::npos && strlen(line.c_str())) {
            break;
        }
        if (found != std::string::npos) {
            if (contentLength != 0) {
                logger.LogMessage(MessageType::MSG_WARNING, "Duplicate Content-Length header received.");
            }
            std::stringstream stream;
            stream << line.substr(found + strlen("Content-Length:"));
            stream >> contentLength;
            stream.clear();
        }
    }
    if (contentLength > 1 << MAX_MESSAGE_LENGTH) {
        logger.LogMessage(MessageType::MSG_WARNING, "Refusing to read message with too long Content-Length.");
        return "";
    }
    if (contentLength == 0) {
        logger.LogMessage(MessageType::MSG_WARNING, "Missing Content-Length header, or zero-length message.");
        return "";
    }

    std::string json(contentLength, '\0');
    size_t read;
    size_t pos = 0;
    while (pos < contentLength) {
        read = RetryAfterSignalUnlessShutdown(0, [&json, &pos, &contentLength, this] {
            return std::fread(&json[pos], 1, contentLength - pos, pFileIn);
        });
        if (read == 0) {
            logger.LogMessage(MessageType::MSG_WARNING, "Input was aborted.");
            return "";
        }

        clearerr(pFileIn);
        pos += read;
    }
    return std::move(json);
}

MessageErrorDetail DecodeError(const nlohmann::json &o)
{
    std::string msg = o.contains("message") ? o.value("message", "") : "Unspecified error";
    ErrorCode code = o.contains("code") ? ErrorCode(o.value("code", -1)) : ErrorCode::UNKNOWN_ERROR_CODE;
    return {msg, code};
}

LSPRet StdioTransport::HandleMessage(nlohmann::json message, MessageHandler &handler)
{
    Logger &logger = Logger::Instance();
    // Message must be an object with "jsonrpc":"2.0".
    if (message["jsonrpc"].is_null() || message.value("jsonrpc", "") != "2.0") {
        logger.LogMessage(MessageType::MSG_WARNING, "jsonrpc is null or not 2.0.");
        return LSPRet::ERR_JSON;
    }
    nlohmann::json id = nullptr;
    if (!message["id"].is_null()) {
        id = std::move(message["id"]);
    }

    // call or notify
    if (!message["method"].is_null()) {
        std::string method = message.value("method", "");
        if (method == "exit") {
            if (ShutdownRequested()) {
                return LSPRet::NORMAL_EXIT;
            }
            return LSPRet::ABNORMAL_EXIT;
        }
        nlohmann::json params = nullptr;
        if (message["params"].is_object()) {
            params = std::move(message["params"]);
        }
        if (id.is_null()) {
            return handler.OnNotify(method, std::move(params));
        }
        return handler.OnCall(method, std::move(params), std::move(id));
    }
    if (id.is_null()) {
        std::stringstream log;
        log << "No method and no response ID, message: " << message;
        logger.LogMessage(MessageType::MSG_WARNING, log.str());
        return LSPRet::ERR_JSON;
    }
    // reply
    if (!message["error"].is_null()) {
        return handler.OnReply(std::move(id),
                               ValueOrError(ValueOrErrorCheck::ERR, DecodeError(message["error"])));
    }
    nlohmann::json jsonValue = nullptr;
    if (!message["result"].is_null()) {
        jsonValue = std::move(message["result"]);
    }
    return handler.OnReply(std::move(id), ValueOrError(ValueOrErrorCheck::VALUE, jsonValue));
}
}
