(ns carousel.core
  (:require [reagent.core :as r]))

(defn prev-idx [idx-model]
  (swap! idx-model #(max (dec (or % 0)) 0)))

(defn next-idx [idx-model idx-max]
  (swap! idx-model #(min (inc (or % 0)) idx-max)))

(defn -carousel-base
  [{:keys [item prev-fn next-fn slide?
           arrow-style arrow-left arrow-right]
      :or {arrow-style :guillemets
           arrow-left  (case arrow-style
                         :guillemets [:span.guillemets "‹"]
                         :zmdi       [:span.zmdi.zmdi-chevron-left])
           arrow-right (case arrow-style
                         :guillemets [:span.guillemets "›"]
                         :zmdi       [:span.zmdi.zmdi-chevron-right])}}]
  (r/with-let [next-transition (r/atom nil)]
    [:div.carousel
      [:div.carousel-arrow.carousel-prev
        {:on-click (comp prev-fn #(reset! next-transition "back"))}
        arrow-left]
      [:div.carousel-content-container {:class @next-transition}
        (cond slide?
              [:> js/React.addons.CSSTransitionGroup
                {:transition-name "contentslide"
                 :transition-enter-timeout 1000
                 :transition-leave-timeout 1000}
                (with-meta [:div.carousel-content item]
                           (meta item))]
              :else
              [:div.carousel-content item])]
      [:div.carousel-arrow.carousel-next
        {:on-click (comp next-fn #(reset! next-transition "forward"))}
        arrow-right]]))

(defn -carousel-idx
  [{:as args :keys [idx-model item-fn item-cnt]}]
  (r/with-let [idx-model (or idx-model (r/atom 0))]
    [-carousel-base
      (merge args
        {:item    (item-fn @idx-model)
         :prev-fn #(prev-idx idx-model)
         :next-fn #(next-idx idx-model (dec item-cnt))})]))

(defn -carousel-items
  [{:as args :keys [idx-model item-seq]}]
  [-carousel-idx
    (merge args
      {:item-fn  #(with-meta (nth item-seq %) {:key (str "item-" %)})
       :item-cnt (count item-seq)})])

(defn carousel [& {:as args :keys [item-seq item-fn item]}]
  (cond item-seq (-carousel-items args)
        item-fn  (-carousel-idx   args)
        item     (-carousel-base  args)))
