(ns zolodeck.demonic.helper
  (:use [datomic.api :only [q db tempid] :as db]
        [zolodeck.utils.clojure :only [defrunonce random-guid diff]]
        [zolodeck.utils.maps :only [select-keys-if] :as maps]
        [zolodeck.utils.debug]
        [zolodeck.utils.clojure])
  (:require [clojure.tools.logging :as logger]))

(def CONN)
(def ^:dynamic TX-DATA)
(def ^:dynamic DATOMIC-DB)
(def ^:dynamic DATOMIC-TEST false)

(defn setup-schema [schema-txs]
  @(db/transact CONN (vec schema-txs)))

(defn setup-connection [datomic-db-name] 
  (def CONN (db/connect datomic-db-name)))

(defn start-it-up- [datomic-db-name datomic-schema]
  (db/create-database datomic-db-name)
  (setup-connection datomic-db-name)
  (setup-schema datomic-schema))

(defrunonce initialize-datomic [datomic-db-name datomic-schema]
  (start-it-up- datomic-db-name datomic-schema))

(defn next-temp-id []
  (- (System/currentTimeMillis)))

(defn temp-db-id? [eid]
  (map? eid))

(defn load-from-db [eid]
  (if (and eid (not (temp-db-id? eid)))
    (db/entity @DATOMIC-DB eid)))

(defn retract-entity-txn [entity]
  [:db.fn/retractEntity (:db/id entity)])

(defn append-ref-txn [entity attrib value-entities]
  [:db/add (:db/id entity) attrib (map :db/id value-entities)])

(defn- datomic-transact [txns]
  (let [tf (db/transact-async CONN txns)]
    @tf))

(defn run-transaction [tx-data]
  (swap! TX-DATA conj tx-data)
  (swap! DATOMIC-DB db/with tx-data))

(defn commit-pending-transactions []
  (when-not DATOMIC-TEST
    (when-not (empty? @TX-DATA)
      (doseq [t @TX-DATA]
        (when-not (empty? t)
          (datomic-transact (doall t)))))))

(defn run-in-demarcation [thunk]
  (binding [TX-DATA (atom [])
            DATOMIC-DB (atom (db/db CONN))]
    (let [res (thunk)]
      (commit-pending-transactions)
      res)))

;; creating new datomic transaction ready maps

(defn is-entity-map? [v]
  (instance? datomic.query.EntityMap v))

(defn entity->map [e]
  (-> (select-keys e (keys e))
      (assoc :db/id (:db/id e))))

(defn- non-db-keys [a-map]
  (remove #(= :db/id %) (keys a-map)))

(defn- guid-key [a-map]
  (-> a-map non-db-keys first .getNamespace (str "/guid") keyword))

(defn assoc-demonic-attributes [entity-or-map]
  (when entity-or-map
    (let [gk (guid-key entity-or-map)]
      (-> entity-or-map
          entity->map          
          (assoc gk (or (gk entity-or-map) (random-guid)))
          (assoc :db/id (or (:db/id entity-or-map) (db/tempid :db.part/user)))))))

