(ns zilti.boot-midje
  {:boot/export-tasks true}
  (:refer-clojure :exclude [test])
  (:require [boot.core :as core]
            [boot.pod :as pod]
            [boot.task.built-in :as built-in]
            [clojure.set :as set]
            [clojure.java.io :as io]
            midje.util.ecosystem
            midje.config
            [midje.repl :as repl]
            [midje.data.project-state :as project-state]))

(def pod-deps '[])

(defn init [fresh-pod]
  (pod/with-eval-in fresh-pod
    (require 'midje.repl
             '[midje.data.project-state :as project-state]
             '[boot.core :as core]
             '[clojure.string :as str]
             'clojure.test)
    (def init? (atom true))
    (defn warn-notifier [result]
      (try
        (let [result (str/split result #"\n")
              result (nth result (- (count result) 2))]
          (when (re-matches #".*FAILURE.*" result)
            (let [failure-count (second (str/split result #"\s"))]
              (Integer/parseInt failure-count))))
        (catch Exception ex (do
                              (println "WARNING: Couldn't analyze" result)
                              (.printStackTrace ex)))))
    
    (defn run-tests [test-options]
      (let [result
            (with-out-str (binding [clojure.test/*test-out* *out*]
                            (if @init?
                              (do
                                (reset! init? false)
                                (project-state/load-everything test-options))
                              ((project-state/mkfn:react-to-changes test-options)))))]
        (println result)
        (warn-notifier result)))))

(defn update-fileset [fileset test-path]
  (let [fileset (loop [test-path test-path
                       fileset fileset]
                  (if-not (empty? test-path)
                    (recur (rest test-path)
                           (core/add-source fileset (io/file (first test-path))))
                    fileset))]
    (core/commit! fileset)
    
    (alter-var-root #'midje.util.ecosystem/leiningen-paths-var
                    (constantly (map str (core/input-dirs fileset))))
    fileset))

(defn do-test [worker-pod {:keys [namespace dirs filter level] :as options}]
  (println "Running tests...")
  (let [test-options {:interval 500
                      :fact-filters (if-not (nil? filter) filter [])
                      :files (into [] dirs)
                      :on-require-failure #'repl/on-require-failure
                      :namespace-stream-checker #'repl/namespace-stream-checker}
        warnings      (pod/with-eval-in worker-pod
                        (run-tests ~test-options))]
    (when-not (nil? warnings)
      (swap! core/*warnings* + warnings))))

(defn update-fileset [fileset test-paths]
  (loop [fileset fileset
         paths test-paths]
    (if-not (empty? paths)
      (recur (core/add-resource fileset (io/file (first paths))) (rest paths))
      (core/commit! fileset))))

(core/deftask midje
  "Run midje tests in boot."
  [t test-paths TESTPATH #{str} "additional paths where the test files reside (analogous to :source-paths)."
   n namespaces NAMESPACE #{sym} "symbols of the namespaces to run tests in.
                              A partial namespace ending in a '*' will load all sub-namespaces.
                              Example: `(load-facts 'midje.ideas.*)`
`"
   f filters FILTER #{str} "midje filters. Only facts matching one or more of the arguments are loaded. The filter arguments are:

                              :keyword      -- Does the metadata have a truthy value for the keyword?
                              \"string\"      -- Does the fact's name contain the given string? 
                              #\"regex\"      -- Does any part of the fact's name match the regex?
                              a function    -- Does the function return a truthy value when given the fact's metadata?
`"
   c config CONFIG #{str} "list of midje config files."
   l level LEVEL int "Set Midje's verbosity level using one of the following options:

                              :print-normally    (0) -- failures and a summary.
                              :print-no-summary (-1) -- failures only.
                              :print-nothing    (-2) -- nothing is printed.
                                                     -- (but return value can be checked)
                              :print-namespaces  (1) -- print the namespace for each group of facts.
                              :print-facts       (2) -- print fact descriptions in addition to namespaces.

                             "]
  (let [worker-pod (pod/make-pod (update-in (core/get-env) [:dependencies] into pod-deps))
        res-paths (set/union test-paths (:resource-paths (core/get-env)) (:source-paths (core/get-env)))]
    (core/set-env! :resource-paths res-paths)
    (println "Starting pod...")
    (init worker-pod)
    (core/with-pre-wrap fileset
      (let [fileset (update-fileset fileset res-paths)]
        (when (seq config)
          (midje.util.ecosystem/set-config-files! config))
        (do-test worker-pod {:namespace namespaces :dirs res-paths :filter filters :level level})
        (core/commit! fileset)))))
