(ns spirit.mail.remote
  (:require [spirit.mail.interop.message :as message]
            [spirit.mail.interop.flag :as flag]
            [spirit.protocol.imail :as mail]
            [hara.component :as component])
  (:import java.util.Properties
           javax.mail.Session
           javax.mail.Folder)
  (:refer-clojure :exclude [send]))

(def ^:dynamic *default-ports*
  {:smtp {:none 25
          :ssl 465}
   :pop3 {:none 110
          :ssl 995}
   :imap {:none 143
          :ssl 993}})

(def access-map {:read-only  Folder/READ_ONLY
                 :read-write Folder/READ_WRITE})

(def ^:dynamic *default-folder* "INBOX")

(def ^:dynamic *default-access* :read-write)

(defmethod print-method javax.mail.Service
  [v w]
  (.write w (str "#service " "\"" (str (.getURLName v)) "\"")))

(defmethod print-method javax.mail.Folder
  [v w]
  (.write w (str "#folder " {:count (.getMessageCount v)
                             :name (.getFullName v)})))

(defn get-transport
  "gets the transport for a particular session
 
   (get-transport (Session/getInstance (Properties.))
                  \"smtp\")
   ;;=> #service \"smtps://\"
   "
  {:added "0.8"}
  [session protocol]
  (.getTransport session protocol))

(defn get-store
  "gets the mailbox store for a particular session
 
   (get-store (Session/getInstance (Properties.))
              \"imap\")
   ;;=> #service \"imap://\"
   "
  {:added "0.8"}
  [session protocol]
  (.getStore session protocol))

(defn connect
  "creates a service given method and settings
   
   (connect {:type :smtp
             :host \"smtp.webfaction.com\"
             :security :ssl
             :auth {:username \"zcaudate_spirit\"
                    :password \"spirit\"}}
            get-transport)
   ;;=> #service \"smtps://zcaudate@smtp.webfaction.com:465\"
  "
  {:added "0.8"}
  [{:keys [type host port security auth params]} get-service-fn]
  (let [security (or security :none)
        session  (Session/getDefaultInstance (or params
                                                 (Properties.)))
        protocol (str (name type)
                      (if (= :none security) "" "s"))
        service (get-service-fn session protocol)
        port    (or port (-> *default-ports* type security))]
    (if auth
      (.connect service host port (:username auth) (:password auth))
      (.connect service host port))
    service))

(defn send
  "creates a service given method and settings
   
   (send {:type :smtp
          :host \"smtp.webfaction.com\"
          :security :ssl
         :auth {:username \"zcaudate_spirit\"
                 :password \"spirit\"}}
         {:from \"spirit@caudate.me\"
          :to   [\"spirit@caudate.me\"]
          :subject \"Test\"
          :body \"Hello There\"})"
  {:added "0.8"}
  [mailer message]
  (let [conn (connect mailer get-transport)
        m (message/message message)]
    (.sendMessage conn m (.getAllRecipients m))))

(defn send-bulk
  "creates a service given method and settings
   
   (send-bulk {:type :smtp
               :host \"smtp.webfaction.com\"
               :security :ssl
              :auth {:username \"zcaudate_spirit\"
                      :password \"spirit\"}}
              [{:from \"spirit@caudate.me\"
                :to   [\"a@a.com\"]
                :subject \"Test A\"
                :body \"Hello There A\"}
               {:from \"spirit@caudate.me\"
                :to   [\"b@b.com\"]
                :subject \"Test B\"
                :body \"Hello There B\"}])"
  {:added "0.8"}
  [mailer messages]
  (let [conn (connect mailer get-transport)]
    (doseq [message messages]
      (let [m (message/message message)]
        (.sendMessage conn m (.getAllRecipients m))))))

(defn remote-folder
  "returns the folder for message retrieval
 
   (remote-folder (connect {:type :imap
                            :host \"mail.webfaction.com\"
                            :auth {:username \"zcaudate_spirit\"
                                   :password \"spirit\"}}
                           get-store)
                 \"INBOX\"
                  :read-write)
   ;;=> #folder {:count 1, :name \"INBOX\"}
   "
  {:added "0.8"}
  ([connection]
   (remote-folder connection nil nil))
  ([connection name access]
   (let [folder (.getFolder connection (or name
                                           *default-folder*))
         _ (.open folder (access-map  (or access
                                          *default-access*)))]
     folder)))

(defn remote-state
  "returns both the storage and folder connection
 
   (remote-state {:type :pop3
                  :host \"mail.webfaction.com\"
                  :folder   \"INBOX\"    ;; default
                  :access   :read-only ;; default :read-write
                  :security :none      ;; default
                  :auth {:username \"zcaudate_spirit\"
                         :password \"spirit\"}})
  ;; => {:connection #service \"smtps://zcaudate@smtp.webfaction.com:465\",
   ;;     :folder #folder {:count 1, :name \"INBOX\"}}
   "
  {:added "0.8"}
  [mailbox]
  (let [connection (connect mailbox get-store)
        folder (remote-folder connection
                              (:folder mailbox)
                              (:access mailbox))]
    {:connection connection :folder folder}))

(defn mailbox-folder
  "returns the folder for message retrieval, contains connection state
 
   (def box (-> (mail/create-mailbox {:type :pop3
                                      :host \"mail.webfaction.com\"
                                      :auth {:username \"zcaudate_spirit\"
                                             :password \"spirit\"}})
                (component/start)))
 
   (time (mailbox-folder box))
   ;;=> #folder {:count 1, :name \"INBOX\"}
 
   (time (mailbox-folder box))
   ;;=> faster than before
   "
  {:added "0.8"}
  [{:keys [state] :as mailbox}]
  (let [{:keys [folder connection]} @state]
    (cond (or (nil? connection)
              (not (.isConnected connection)))
          (let [rs (remote-state mailbox)]
            (-> (reset! state rs)
                :folder))

          (and folder (.isOpen folder))
          folder

          (.isConnected connection)
          (let [folder (remote-folder connection
                                      (:folder mailbox)
                                      (:access mailbox))]
            (-> (swap! state assoc :folder folder)
                :folder)))))

(defrecord RemoteMailer []
  Object
  (toString [v]
    (str "#mailer" (into {} (dissoc v :state))))

  mail/IMailer
  (-send-mail [mailer message]
    (send mailer message))

  (-send-bulk [mailer messages]
    (send-bulk mailer messages)))

(defmethod print-method RemoteMailer
  [v w]
  (.write w (str v)))

(defmethod mail/create-mailer :smtp
  [m]
  (map->RemoteMailer m))

(defrecord RemoteMailbox []
  Object
  (toString [v]
    (str "#mailbox" (into {} (dissoc v :state))))

  mail/IMailbox
  #_(-has-new-mail [mailbox]
    (.hasNewMessages (mailbox-folder mailbox)))
  (-count-mail  [mailbox]
    (.getMessageCount (mailbox-folder mailbox)))
  #_(-update-mail [mailbox i add delete])
  (-get-mail    [mailbox i]
    (.getMessage (mailbox-folder mailbox)))
  (-list-mail   [mailbox]
    (.getMessages (mailbox-folder mailbox)))
  (-clear-mail  [mailbox])

  component/IComponent
  (-start [mailbox]
    (assoc mailbox :state (atom {})))

  (-stop [mailbox]
    (dissoc mailbox :state)))

(defmethod print-method RemoteMailbox
  [v w]
  (.write w (str v)))

(defmethod mail/create-mailbox :imap
  [m]
  (map->RemoteMailbox m))

(defmethod mail/create-mailbox :pop3
  [m]
  (map->RemoteMailbox m))
