(ns spirit.mail.local
  (:require [clojure.string :as string]
            [hara.object :as object]
            [spirit.mail.interop.message :as message]
            [spirit.mail.common :as common]
            [spirit.protocol.imail :as mail])
  (:refer-clojure :exclude [send]))

(defn sendmail-cmd
  "finds path to the sendmail program
 
   (sendmail-cmd)
   ;;=> \"/usr/sbin/sendmail\"
   "
  {:added "0.8"}
  []
  (-> (ProcessBuilder. ["which" "sendmail"])
      (.start)
      (.getInputStream)
      (slurp)
      (string/trim-newline)))

(defn send
  "sends an email using local smtp provider
 
   (send {:from \"z@z.com\"
         :to [\"z@z.com\"]
          :subject (str \"Hello There - \" (rand))
          :body [{:type :inline
                  :text \"Here is the attachment\\n\\n\"}
                 {:type :attachment
                  :description \"hello there\"
                  :file \"project.clj\"}]})"
  {:added "0.8"}
  [{:keys [from to subject body] :as message}]
  (let [m   (message/message message)
        all (object/to-data (object/get m :all-recipients))
        p   (.start (ProcessBuilder. (concat [(sendmail-cmd) "-f" from]
                                             all)))
        out (.getOutputStream p)]
    (.writeTo m out)
    (.close out)
    (.waitFor p)
    (.exitValue p)))

(defrecord LocalMailer []
  Object
  (toString [_]
    (str "#mailer.local" {}))
  
  mail/IMailer
  (-send-mail [_ message]
    (send message)))

(defmethod print-method LocalMailer
  [v w]
  (.write w (str v)))

(defmethod mail/create-mailer :local
  [_]
  (LocalMailer.))
