(ns spirit.mail.interop.message
  (:require [hara.object :as object]
            [hara.reflect :as reflect]
            [spirit.mail.interop
             [address :as address]
             [body :as body]
             [header :as header]
             [flag :as flag]
             [recipient :as recipient]
             [wrap :as wrap]])
  (:import [javax.mail Message Session]
           [javax.mail.internet MimeMessage MimeMultipart]))

(defn empty-message
  "creates an empty MimeMessage
 
   (object/to-data (empty-message))
   => {:all-headers [],
       :all-header-lines [],
       :size -1,
       :content-type \"text/plain\",
       :flags [],
      :line-count -1}"
  {:added "0.8"}
  ([]
   (empty-message (Session/getInstance (java.util.Properties.))))
  ([session]
   (MimeMessage. session)))

(defn get-recipients
  "gets recipients of a particular field
   
   (-> (doto (empty-message)
         (.addRecipient (recipient/list :to)
                        (InternetAddress. \"z@z.com\")))
       ((get-recipients :to))
       (seq))
   ;;=> (#address \"z@z.com\")
   "
  {:added "0.8"}
  [k]
  (fn [m] (.getRecipients m (recipient/list k))))

(defn set-recipients
  "sets recipients of a particular field
 
   (-> (doto (empty-message)
         ((set-recipients :to)  [\"a@a.com\"])
         ((set-recipients :bcc) [\"b@b.com\"])
         ((set-recipients :cc)  [\"c@c.com\"]))
       (object/to-data)
       :all-recipients)
  => [\"a@a.com\" \"c@c.com\" \"b@b.com\"]"
  {:added "0.8"}
  [k]
  (fn [m l]
    (doto m (.addRecipients (recipient/list k)
                            (address/write-addresses l)))))

(defn set-body
  "sets the body of a given message
 
   (-> (set-body (empty-message) \"Hello World\")
       (object/get :body))
   => \"Hello World\""
  {:added "0.8"}
  [m v]
  (cond (string? v)
        (.setText m v)
        
        (or (instance? MimeMultipart v) (vector? v))
        (.setContent m (body/body v))

        :else
        (throw (Exception. "Entry either text or a multipart entry")))
  m)

(def read-mime-message
  (let [methods (-> (object/read-getters MimeMessage)
                    (dissoc :content :data-handler :input-stream :raw-input-stream :content-stream)
                    (update-in [:from] wrap/wrap-first)
                    (merge {:body (wrap/wrap-suppress (fn [m] (.getContent m)))
                            :to   (get-recipients :to)
                            :cc   (get-recipients :cc)
                            :bcc  (get-recipients :bcc)}))]
    methods))

(def write-mime-message
  (let [{:keys [content text] :as methods} (object/write-all-setters MimeMessage)]
    (merge (dissoc methods :content :text)
           {:body {:type Object
                   :fn set-body}
            :to  {:type java.util.List
                  :fn (set-recipients :to)}
            :cc  {:type java.util.List
                  :fn (set-recipients :cc)}
            :bcc {:type java.util.List
                  :fn (set-recipients :bcc)}})))

(object/map-like
 MimeMessage
 {:tag "message"
  :read  {:methods read-mime-message}
  :write {:methods write-mime-message
          :empty (fn [_] (empty-message))}})

(defn message
  "constructs a MimeMessage from a map
 
   (-> (message {:from \"z@z.com\"
                 :cc  [\"a@a.com\"]
                 :bcc [\"b@b.com\" \"c@c.com\"]
                 :reply-to [\"y@y.com\"]
                 :subject \"Hello There\"
                 :body \"This is Cool\"})
       (object/to-data)
       (select-keys [:all-headers :subject :body]))
  => {:all-headers [[\"From\" \"z@z.com\"]
                     [\"Reply-To\" \"y@y.com\"]
                     [\"Cc\" \"a@a.com\"]
                     [\"Bcc\" \"b@b.com, c@c.com\"]
                     [\"Subject\" \"Hello There\"]]
       :subject \"Hello There\"
       :body \"This is Cool\"}
 
   "
  {:added "0.8"}
  [{:keys [from to cc bcc subject content text reply-to] :as m}]
  (object/from-data m MimeMessage))
