(ns lucid.unit
  (:require [lucid.core
             [code :as code]]
            [lucid.unit
             [zipper :as zipper]
             [scaffold :as scaffold]]
            [lucid.query :as query]
            [hara.io
             [file :as fs]
             [project :as project]
             [watch :as watch]]
            [clojure.string :as string]
            [clojure.set :as set]
            [rewrite-clj.zip :as source])
  (:refer-clojure :exclude [import]))

(def ^:dynamic *watchers* {})

(defn exclude
  "helper function for excluding certain namespaces
 
   (exclude '{lucid.legacy.analyzer :a
              lucid.legacy :a
              lucid.aether :b}
            [\"lucid.legacy\"])
   => '{lucid.aether :b}"
  {:added "1.2"}
  [lookup exclusions]
  (reduce-kv (fn [out ns v]
               (let [nss (str ns)
                     exclude? (->> exclusions
                                   (map (fn [ex]
                                          (.startsWith nss ex)))
                                   (some true?))]
                 (if exclude?
                   out
                   (assoc out ns v))))
             {}
             lookup))

(defn import
  "imports unit tests as docstrings
          
   ;; import docstrings for the current namespace
   (import)
 
   ;; import docstrings for a given namespace
   (import 'lucid.unit)"
  {:added "1.2"}
  ([] (import *ns*))
  ([ns] (import ns (project/project)))
  ([ns project] (import ns (project/file-lookup project) project))
  ([ns lookup project]
   (let [ns (code/source-namespace ns)
         src-file  (lookup ns)
         test-file (lookup (symbol (str ns "-test")))
         import-fn (fn [nsp refers]
                     (fn [zloc]
                       (zipper/insert zloc nsp refers)))
         refers (if test-file
                  (code/analyse-file :test test-file)
                  {})]
     (if src-file
       (zipper/walk-file src-file nil refers import-fn)))))

(defn import-project
  "imports all unit tests int the project as docstrings
   
   ;; import docstrings for the entire project
   (import-project)"
  {:added "1.2"}
  ([] (import-project (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         lookup   (project/file-lookup project)
         nss (-> (project/all-files (:source-paths project) {} project)
                 (exclude exclusions)
                 (keys)
                 (sort))
         len (count nss)]
     (println "IMPORTING UNIT TESTS:")
     (doall (map-indexed (fn [i ns]
                           (println (format "%s (%d of %d)" ns (inc i) len))
                           (try (import ns lookup project)
                                (catch Exception e
                                  (println ".... ERRORED:" ns))))
                         nss))
     nil)))

(defn purge
  "purge docstrings and meta from file
          
   ;; removes docstrings for the current namespace  
   (purge)
 
   ;; removes docstrings for a given namespace
   (purge 'lucid.unit)
 
   ;; removes docstrings for the entire project
   (purge :all)"
  {:added "1.2"}
  ([] (purge *ns*))
  ([ns] (purge ns (project/project)))
  ([ns project] (purge ns (project/file-lookup project) project))
  ([ns lookup project]
   (cond (= ns :all)
         (doseq [ns (keys (project/all-files (:source-paths project) {} project))]
           (purge ns lookup project))

         :else
         (let [ns (code/source-namespace ns)
               src-file (lookup ns)
               purge-fn (fn [nsp references] identity)]
           (if src-file
             (zipper/walk-file src-file nil nil purge-fn))))))

(defn missing
  "checks functions that are missing in a given namespace
 
   ;; lists missing tests for current namespace
   (missing)
   
   ;; lists missing tests for specific namespace
   (missing 'lucid.unit)"
  {:added "1.2"}
  ([] (missing *ns*))
  ([ns] (missing ns (project/project)))
  ([ns project] (missing ns (project/file-lookup project) project))
  ([ns lookup project]
   (let [ns (code/source-namespace ns)
         src-file (lookup ns)
         test-file (lookup (symbol (str ns "-test")))
         src-vars (if src-file (code/all-source-vars src-file))
         test-vars (if test-file (code/all-test-vars test-file))]
     {ns (set/difference (set src-vars) (set test-vars))})))

(defn all-missing
  "finds functions with missing tests in the project
          
   (all-missing)"
  {:added "1.4"}
  ([] (all-missing (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         lookup (project/file-lookup project)
         sources (-> (project/all-files (:source-paths project) {} project)
                     (exclude exclusions))]
     (->> (map #(missing % lookup project) (keys sources))
          (apply merge)
          (remove (fn [[_ v]] (empty? v)))
          (sort)))))

(defn all-functions
  "finds all the functions in the project
          
   (all-functions)"
  {:added "1.4"}
  ([] (all-functions (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         sources (-> (project/all-files (:source-paths project) {} project)
                     (exclude exclusions))]
     (reduce (fn [out [ns file]]
               (assoc out ns (code/all-source-vars file)))
             {}
             sources))))

(defn all-tests
  "finds all unit tests in the project
          
   (all-tests)"
  {:added "1.4"}
  ([] (all-tests (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         lookup     (project/file-lookup project)
         sources    (-> (project/all-files (:source-paths project) {} project)
                        (exclude exclusions))]
     (reduce (fn [out [ns _]]
               (if-let [test-file (lookup (symbol (str ns "-test")))]
                 (assoc out ns (code/all-test-vars test-file))
                 out))
             {}
             sources))))

(defn orphaned
  "finds all unit tests that do not have functions
   
   ;; lists orphaned tests for current namespace
   (orphaned)
   
   ;; lists orphaned tests for specific namespace
   (orphaned 'lucid.unit)"
  {:added "1.2"}
  ([] (orphaned *ns*))
  ([ns] (orphaned ns (project/project)))
  ([ns project] (orphaned ns (project/file-lookup project) project))
  ([ns lookup project]
   (let [ns (code/source-namespace ns)
         full-sym (fn [sym] (symbol (str ns) (name sym)))
         src-file (lookup ns)
         test-file (lookup (symbol (str ns "-test")))
         src-vars (if src-file (map full-sym (code/all-source-vars src-file)))
         test-vars (if test-file (code/all-test-vars test-file {:full true}))]
     {ns (set/difference (set test-vars) (set src-vars))})))

(defn all-orphaned
  "finds tests with missing functions in the project
 
   (all-orphaned)"
  {:added "1.4"}
  ([] (all-orphaned (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         lookup (project/file-lookup project)
         sources (-> (project/all-files (:source-paths project) {} project)
                     (exclude exclusions))]
     (->> (map #(orphaned % lookup project) (keys sources))
          (apply merge)
          (remove (fn [[_ v]] (empty? v)))
          (sort)))))

(defn scaffold
  "builds the unit test scaffolding for the source
 
   ;; generates test scaffolding for current namespace
   (scaffold)
   
   ;; generates test scaffolding for specific namespace
   (scaffold 'lucid.unit)"
  {:added "1.2"}
  ([] (scaffold *ns*))
  ([ns] (scaffold ns (project/project)))
  ([ns project] (scaffold/scaffold ns (project/file-lookup project) project)))

(defn scaffold-project
  "scaffolds the extire project
 
   ;; generates test scaffolding for entire project
   (scaffold-project)"
  {:added "1.2"}
  ([] (scaffold-project (project/project)))
  ([project]
   (let [missing (map first (all-missing project))]
     (doseq [ns missing]
       (println "Scaffolding:" ns)
       (scaffold ns project)))))

(defn todos
  "finds all todos of a particular namespace
 
   (todos 'lucid.unit-test)"
  {:added "1.2"}
  ([] (todos *ns*))
  ([ns]
   (let [project (project/project)
         lookup  (project/all-files (:test-paths project) {} project)]
     (todos ns lookup project)))
  ([ns lookup project]
   (->> (read-string (str "[" (slurp (lookup ns)) "]"))
        (filter (fn [form]
                  (and (list? form)
                       (= '(fact "TODO")
                          (take 2 form)))))
        (mapv (comp symbol name :refer meta)))))

(defn all-todos
  "finds all todos of a particular namespace
 
   (all-todos)"
  {:added "1.2"}
  ([] (all-todos (project/project)))
  ([project]
   (let [lookup  (project/all-files (:test-paths project) {} project)
         ls      (->> (map (fn [ns]
                             [ns (todos ns lookup project)])
                           (sort (keys lookup)))
                      (remove (comp empty? second)))]
     ls
     #_{:count (reduce (fn [out [_ v]]
                       (+ out (count v)))
                     0
                     ls)
      :forms ls})))

(defn in-order?
  "checks vars in the test file is in correct order
   
   ;; checks ordering for current namespace
   (in-order?)
   
   ;; checks ordering for specific namespace
   (in-order? 'lucid.unit)"
  {:added "1.2"}
  ([] (in-order? *ns*))
  ([ns] (in-order? ns (project/project)))
  ([ns project] (scaffold/in-order? ns (project/file-lookup project) project)))

(defn arrange
  "arranges tests so that vars are in correct order
   
   ;; arranges tests for current namespace
   (arrange)
   
   ;; arranges tests for specific namespace
   (arrange 'lucid.unit)
   
   (arrange 'lucid.query.match.fn)"
  {:added "1.2"}
  ([] (arrange *ns*))
  ([ns] (arrange ns (project/project)))
  ([ns project] (scaffold/arrange ns (project/file-lookup project) project)))

(defn arrange-project
  "imports all unit tests in the project as docstrings
   
   ;; arranges all the unit tests in order of the functions
   (arrange-project)"
  {:added "1.2"}
  ([] (arrange-project (project/project)))
  ([project]
   (let [exclusions (-> project :unit :exclude)
         lookup   (project/file-lookup project)
         nss (-> (project/all-files (:source-paths project) {} project)
                 (exclude exclusions)
                 (keys)
                 (sort))
         len (count nss)]
     (println "ARRANGING UNIT TESTS:")
     (doall (map-indexed (fn [i ns]
                           (println (format "%s (%d of %d)" ns (inc i) len))
                           (try (scaffold/arrange ns lookup project)
                                (catch Exception e
                                  (println ".... ERRORED:" ns))))
                         nss))
     nil)))

(defn unwatch
  "removes the automatic watching and importing of tests
 
   (unwatch)"
  {:added "1.2"}
  ([] (unwatch (project/project)))
  ([{:keys [root] :as project}]
   (when-let [watcher (get *watchers* root)]
     (alter-var-root #'*watchers* dissoc root)
     (watch/stop-watcher watcher))))

(defn watch
  "automatic imports tests when files change
 
   (watch)"
  {:added "1.2"}
  ([] (watch {} (project/project)))
  ([opts {:keys [root test-paths] :as project}]
   (unwatch project)
   (let [lookup    (project/file-lookup project)
         ns-lookup (zipmap (vals lookup) (keys lookup))
         watcher   (-> (watch/watcher test-paths
                                      (fn [_ file]
                                        (let [ns (ns-lookup (str file))]
                                          (import ns lookup project)))
                                      {:recursive true
                                       :filter ["*.clj"]})
                       (watch/start-watcher))]
     (alter-var-root #'*watchers* assoc root watcher)
     watcher)))

(defn project-stats
  "automatic imports tests when files change
 
   (project-stats (all-missing)
                 {:levels 2 :ns true})"
  {:added "1.4"}
  [functions {:keys [levels ns] :or {levels 2}}]
  (reduce (fn [out [namespace methods]]
            (let [len (count methods)
                  arr (if levels (take levels (string/split (str namespace) #"\.")))]
              (cond-> out
                (and ns
                     (pos? len)) (update-in [:ns] (fnil #(conj % namespace) []))
                levels   (update-in [:label (symbol (string/join "." arr))] (fnil + 0) len)
                :then (update-in [:total :method] (fnil + 0) len)
                :then (update-in [:total :ns] (fnil inc 0)))))
          {}
          functions))

(def project-map {:functions all-functions
                  :tests all-tests
                  :missing all-missing
                  :orphaned all-orphaned
                  :todos all-todos})

(defn stats
  "automatic imports tests when files change
 
   (stats [:missing :orphaned] {:levels false})
   ;; => {:missing {:total {:method 195, :ns 47}},
   ;;     :orphaned {:total {:method 27, :ns 10}}}
   "
  {:added "1.4"}
  ([]
   [:functions :tests :missing :orphaned :todos])
  ([selection]
   (stats selection {}))
  ([selection opts]
   (stats selection (project/project) opts))
  ([selection project opts]
   (cond (keyword? selection)
         (project-stats ((project-map selection) project) opts)

         (vector? selection)
         (reduce (fn [out k]
                   (assoc out k (stats k project opts)))
                 {}
                 selection)

         :else
         (stats))))
