(ns lucid.system.jvm
  (:require [hara.object :as object])
  (:import [java.lang.management ManagementFactory ClassLoadingMXBean CompilationMXBean
            GarbageCollectorMXBean MemoryMXBean MemoryPoolMXBean MemoryManagerMXBean 
            OperatingSystemMXBean RuntimeMXBean ThreadMXBean MemoryUsage]))

(object/map-like
 
 ClassLoadingMXBean
 {:tag "class-loading"}
 CompilationMXBean
 {:tag "compilation"}
 GarbageCollectorMXBean
 {:tag "gc"}
 MemoryMXBean
 {:tag "memory"}
 MemoryPoolMXBean
 {:tag "memory-pool"
  :exclude [:collection-usage-threshold-exceeded?
            :collection-usage-threshold-count
            :collection-usage-threshold
            :collection-usage
            :usage-threshold
            :usage-threshold-count
            :usage-threshold-exceeded?]}
 MemoryManagerMXBean
 {:tag "memory-manager"}
 OperatingSystemMXBean
 {:tag "os"}
 RuntimeMXBean
 {:tag "runtime"}
 ThreadMXBean
 {:tag "thread"}
 MemoryUsage
 {:tag "usage"})

(defn class-loading-bean
  "gives information about class loading
 
   (object/to-map (class-loading-bean))
   => (contains {:loaded-class-count number?
                 :total-loaded-class-count number?
                 :unloaded-class-count number?
                 :verbose? checks/boolean?})"
  {:added "1.2"}
  []
  (ManagementFactory/getClassLoadingMXBean))

(defn compilation-bean
  "gives information about compilation
 
   (object/to-map (compilation-bean))
   => (contains {:name string?
                 :total-compilation-time number?
                 :compilation-time-monitoring-supported? checks/boolean?})"
  {:added "1.2"}
  []
  (ManagementFactory/getCompilationMXBean))

(defn gc-bean
  "gives information about the garbage collector
   
   (object/to-map (first (gc-bean)))
   => (contains {:collection-count number?
                 :collection-time number?})"
  {:added "1.2"}
  []
  (ManagementFactory/getGarbageCollectorMXBeans))

(defn memory-bean
  "gives information about the system memory
 
   (object/to-map (memory-bean))
   => (contains-in {:heap-memory-usage
                    {:committed number?
                     :init number?
                     :max number?
                     :used number?}
                   :non-heap-memory-usage
                    {:committed  number?
                     :init number?
                     :max number?
                     :used number?}
                    :object-pending-finalization-count number?
                    :verbose? checks/boolean?})"
  {:added "1.2"}
  []
  (ManagementFactory/getMemoryMXBean))

(defn memory-manager-bean
  "gives information about the memory manager
 
   (object/to-map (first (memory-manager-bean)))
   => (contains-in {:memory-pool-names [string?],
                    :name string?
                    :valid? checks/boolean?})"
  {:added "1.2"}
  []
  (ManagementFactory/getMemoryManagerMXBeans))

(defn memory-pool-bean
  "gives information about the memory pool
 
   (object/to-map (first (memory-pool-bean)))"
  {:added "1.2"}
  []
  (ManagementFactory/getMemoryPoolMXBeans))

(defn os-bean
  "gives information about the operating system
 
   (object/to-map (os-bean))
   => (contains {:arch string?
                 :available-processors number?
                 :name string?
                 :system-load-average number?
                 :version string?})"
  {:added "1.2"}
  []
  (ManagementFactory/getOperatingSystemMXBean))

(defn runtime-bean
  "gives information about the system runtime
 
   (object/to-map (runtime-bean))"
  {:added "1.2"}
  []
  (ManagementFactory/getRuntimeMXBean))

(defn thread-bean
  "gives information about the current thread executor
 
   (object/to-map (thread-bean))"
  {:added "1.2"}
  []
  (ManagementFactory/getThreadMXBean))

(def jvm-map
  {:class-loading class-loading-bean
   :compilation compilation-bean
   :gc gc-bean
   :memory memory-bean
   :memory-manager memory-manager-bean
   :memory-pool memory-pool-bean
   :os os-bean
   :runtime runtime-bean
   :thread thread-bean})
