(ns lucid.query.match.pattern
  (:require  [hara.common.checks :refer [hash-map? lazy-seq?]]
             [lucid.query.match regex fn
              [actual :as actual]
              [eval :as eval]
              [optional :as optional]]
             [lucid.query.common :as common]
             [lucid.legacy.match :as match]
             [clojure.walk :as walk]))

(defn transform-pattern
  "converts an input into an actual matchable pattern
 
   (transform-pattern ^:& #{:a :b})
   
   (transform-pattern ^:% (list 'symbol \"_\"))
   => '(((quote symbol) \"_\") :seq)
 
   (transform-pattern #{:a :b})
   => '(:or :b :a)
 
   (transform-pattern [:a :b])
   => [:a :b]"
  {:added "1.2"}
  [template]
  (cond (:& (meta template))       (actual/actual-pattern template)
        (:% (meta template))       (eval/eval-pattern template)
        (#{'(quote &)
           '(quote _)} template)    template
        (or (lazy-seq? template)
            (list? template))      (cond (empty? template)
                                         (actual/actual-pattern template)

                                         :else
                                         (list (apply list (map transform-pattern template)) :seq))    
        (#{'& '_} template)        template
        (vector? template)         (vec (map transform-pattern template))
        (set? template)            (let [pats (map transform-pattern template)
                                         pats (if (empty? pats) [#'common/none] pats)]
                                     (apply list :or pats))
        (hash-map? template)  (->> (map (fn [[k v]]
                                      [(transform-pattern k) (transform-pattern v)]) template)
                                   (into {}))
        (symbol? template)    (list 'quote template)
        :else template))

(defn pattern-form
  "constructs the form that evaluates the the function
 
   (pattern-form 'a '(a))"
  {:added "1.2"}
  [sym template]
  (let [clauses [[(transform-pattern template)] true :else false]]
    (match/clj-form [sym] clauses)))

(defn pattern-single-fn
  "creates a function based on template
 
   ((pattern-single-fn '(a)) '(a))
   => true
 
   ((pattern-single-fn '(a)) '(b))
   => false"
  {:added "1.2"}
  [template]
  (let [sym        (gensym)
        match-form (pattern-form sym template)
        all-fn    `(fn [form#]
                     (let [~sym form#]
                       ~match-form))]
    (eval all-fn)))

(defn pattern-matches
  "pattern matches for a given template
   ((pattern-matches ()) ())
   => '(())
 
   ((pattern-matches []) ())
   => ()
   
   ((pattern-matches '(^:% symbol? ^:? (+ 1 _ ^:? _))) '(+ (+ 1 2 3)))
   => '((^{:% true} symbol? ^{:? 0} (+ 1 _ ^{:? 1} _)))"
  {:added "1.2"}
  [template]
  (let [all-fns (->> template
                     (optional/pattern-seq)
                     (mapv (juxt identity pattern-single-fn)))]
    (fn [form]
      (or (mapcat (fn [[template f]]
                    (if (f form) [template])) all-fns)
          []))))

(defn pattern-fn
  "make sure that functions are working properly
   ((pattern-fn vector?) [])
   => throws
 
   ((pattern-fn #'vector?) [])
   => true
 
   ((pattern-fn '^:% vector?) [])
   => true
   
   ((pattern-fn '^:% symbol?) [])
   => false
 
   ((pattern-fn '[^:% vector?]) [[]])
   => true"
  {:added "1.2"}
  [template]
  (fn [value]
    (-> ((pattern-matches template) value)
        empty?
        not)))
